// **************************************************************
// *** TAPHeadLabel Version 3.0 for Delphi 4.0                ***
// *** Copyright (c) 1997-99 a priori computer solutions gmbh ***
// *** coded by Marc Hoffmann - All rights reserved.          ***
// *** Status: Freeware                                       ***
// **************************************************************
//
// This component is based on Netscape's gradient labels. It
// allows you to define separate captions for each alignment.
// The captions supports automatically clipping and can be 
// accompany with different gradient fillings.
//
// Try to use this label as a form-titlebar. Just set the
// property "FormMove" to "Yes" and "Align" to "alTop".
// (Look Demo)
//
// History
// =============================
// 3.0: New Delphi 4.0-Support
// 2.9: First official release (based on the Delphi 1.0-Version
//      distributed 1997. Fixed the system-crash.
// 2.x: Some internal releases (just for testing purposes)
// 1.0: First release under Delphi 1.0
// ----------------------------------------------------------------
// 
// Contact
// =============================
// Internet: http://www.apriori.de/index.htm?delphi
// eMail:    mhoffmann@apriori.de
// ----------------------------------------------------------------
unit APHeadLabel;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Dialogs, Forms,
  StdCtrls, ExtCtrls, DsgnIntf;

type
  TGlyphMode      =   (gmNormal, gmOverlayed);
  TGradientStyle  =   (gsNone, gsBottom, gsLeft, gsRight, gsTop);
  TTextAlignmt    =   (taCenter, taLeft, taRight);
  TColorSteps     =   1..255;
  TNumGlyphs      =   1..4;
  TGradientArray  =   Array[1..255] of TColor;
  TBoundLines     =   Set of (blLeft, blTop, blRight, blBottom);

  // forward-declarations
  TAPHeadLabel = class;

  // declare property editors
  TAboutProperty = class(TPropertyEditor)
  public
    { Public-Deklarationen }
    procedure         Edit; override;
    function          GetAttributes: TPropertyAttributes; override;
    function          GetValue: String; override;
  end;

  // declare gradient-property-object
  TGradient = class(TPersistent)
  private
    { Private-Deklarationen }
    Parent:           TAPHeadLabel;
    FColorSteps:      TColorSteps;
    FEndColor:        TColor;
    FStartColor:      TColor;
    FGradientStyle:   TGradientStyle;

    procedure         SetColorSteps(Value: TColorSteps);
    procedure         SetEndColor(Value: TColor);
    procedure         SetGradientStyle(Value: TGradientStyle);
    procedure         SetStartColor(Value: TColor);
  published
    { Published-Deklarationen }
    property          ColorSteps: TColorSteps read FColorSteps write SetColorSteps default 64;
    property          EndColor: TColor read FEndColor write SetEndColor default clSilver;
    property          StartColor: TColor read FStartColor write SetStartColor default clGray;
    property          Style: TGradientStyle read FGradientStyle write SetGradientStyle default gsLeft;
  end;

  // declare subcaption-property-object
  TSubCaption = class(TPersistent)
  private
    { Private-Deklarationen }
    Parent:           TAPHeadLabel;
    FCaption:         String;
    FColor:           TColor;
    FStyle:           TFontStyles;
    FSize:            Byte;
    FEllipsis:        Boolean;
    FMargin:          Byte;

    procedure         SetMargin(Value: Byte);
    procedure         SetSubCaption(Value: String);
    procedure         SetSubColor(Value: TColor);
    procedure         SetSubStyle(Value: TFontStyles);
    procedure         SetSubSize(Value: Byte);
    procedure         SetSubEllipsis(Value: Boolean);
  published
    { Published-Deklarationen }
    property          Caption: String read FCaption write SetSubCaption;
    property          Color: TColor read FColor write SetSubColor default clWindowText;
    property          Ellipsis: Boolean read FEllipsis write SetSubEllipsis;
    property          Size: Byte read FSize write SetSubSize default 8;
    property          Margin: Byte read FMargin write SetMargin default 8;
    property          Style: TFontStyles read FStyle write SetSubStyle;
  end;

  // define master-component
  TAPHeadLabel     =  class(TCustomLabel)
  private
    { Private-Deklarationen }
    GradientBand:     TGradientArray;

    FAbout:           TAboutProperty;
    FAutoBounds:      Boolean;
    FAlign:           TAlign;
    FBoundLines:      TBoundLines;
    FBoundColor:      TColor;
    FFont:            TFont;
    FFormMove:        Boolean;
    FGradient:        TGradient;
    FGlyph:           TBitmap;
    FGlyphMode:       TGlyphMode;
    FMargin:          Byte;
    FNumGlyphs:       TNumGlyphs;
    FSpacing:         Byte;
    FSubCaption:      TSubCaption;
    FTextAlignmt:     TTextAlignmt;

    FOnClick:         TNotifyEvent;
    FOnDblClick:      TNotifyEvent;
    FOnGlyphClick:    TNotifyEvent;
    FOnGlyphDblClick: TNotifyEvent;

    procedure         CMTextChanged(var Msg: TMessage); message CM_TEXTCHANGED;
    procedure         CMFontChanged(var Msg: TMessage); message CM_FONTCHANGED;
    procedure         WMLButtonDown(var Msg: TWMLBUTTONDOWN); message WM_LBUTTONDOWN;
    procedure         WMLButtonDblClk(var Msg: TWMLBUTTONDBLCLK); message WM_LBUTTONDBLCLK;

    procedure         DrawGradientBand;
    procedure         SetAutoBounds(Value: Boolean);
    procedure         SetAlign(Value: TAlign);
    procedure         SetBoundLines(Value: TBoundLines);
    procedure         SetBoundColor(Value: TColor);
    procedure         SetFormMove(Value: Boolean);
    procedure         SetGlyph(Value: TBitmap);
    procedure         SetGlyphMode(Value: TGlyphMode);
    procedure         SetMargin(Value: Byte);
    procedure         SetNumGlyphs(Value: TNumGlyphs);
    procedure         SetSpacing(Value: Byte);
    procedure         SetTextAlignmt(Value: TTextAlignmt);

    procedure         CalcNumGlyphs;
    procedure         ResizeHeadLabel;

    function          GetClippedCaption(Offset: Integer): String;
    function          GetSubCaption: String;
    function          MouseInGlyph: Boolean;
  protected
    { Protected-Deklarationen }
    procedure         Paint; override;
    procedure         CalcGradientBand(StartCol, EndCol: TColor; Steps: Byte);
  public
    { Public-Deklarationen }
    constructor       Create(AOwner: TComponent); override;
    destructor        Destroy; override;
  published
    { Published-Deklarationen }
    property          About: TAboutProperty read FAbout write FAbout;
    property          Alignment: TTextAlignmt read FTextAlignmt write SetTextAlignmt default taLeft;
    property          Align: TAlign read FAlign write SetAlign default alNone;
    property          AutoBounds: Boolean read FAutoBounds write SetAutoBounds default True;
    property          BoundLines: TBoundLines read FBoundLines write SetBoundLines;
    property          BoundColor: TColor read FBoundColor write SetBoundColor default clGray;
    property          FormMove: Boolean read FFormMove write SetFormMove default False;
    property          Glyph: TBitmap read FGlyph write SetGlyph;
    property          GlyphMode: TGlyphMode read FGlyphMode write SetGlyphMode default gmNormal;
    property          Gradient: TGradient read FGradient write FGradient;
    property          Margin: Byte read FMargin write SetMargin default 5;
    property          NumGlyphs: TNumGlyphs read FNumGlyphs write SetNumGlyphs default 1;
    property          Spacing: Byte read FSpacing write SetSpacing default 5;
    property          SubCaption: TSubCaption read FSubCaption write FSubCaption;

    property          Anchors;
    property          BiDiMode;
    property          Caption;
    property          Constraints;
    property          Enabled;
    property          FocusControl;
    property          Font;
    property          ParentBiDiMode;
    property          ParentFont default True;
    property          ParentShowHint;
    property          PopupMenu;
    property          ShowHint;

    property          OnClick: TNotifyEvent read FOnClick write FOnClick;
    property          OnDblClick: TNotifyEvent read FOnDblClick write FOnDblClick;
    property          OnGlyphClick: TNotifyEvent read FOnGlyphClick write FOnGlyphClick;
    property          OnGlyphDblClick: TNotifyEvent read FOnGlyphDblClick write FOnGlyphDblClick;

    property          OnDragDrop;
    property          OnDragOver;
    property          OnEndDrag;
    property          OnMouseDown;
    property          OnMouseUp;
    property          OnMouseMove;
    property          OnStartDrag;
  end;

  // declare component-editor
  TAPHeadEditor = class(TDefaultEditor)
  public
    procedure       ExecuteVerb(Index: Integer); override;

    function        GetVerb(Index: Integer): string; override;
    function        GetVerbCount: Integer; override;
  end;

procedure Register;

implementation

const
  SC_DRAGMOVE = $F012;

procedure DisplayAbout;
begin
  Application.MessageBox('TAPHeadLabel for Delphi 4.0 - Graphic Component' + #13 +
                         'Copyright (C)1997-99 a priori computer solutions GmbH'+ #13#13 +
                         '- FREEWARE -' + #13#13 +
                         'eMail: mhoffmann@apriori.de'+ #13 +
                         'Internet: http://www.apriori.de',
                         'TAPHeadLabel Version 3.0', Mb_OK + MB_ICONINFORMATION);
end;

// *** TAboutProperty ******************************************************
// Inspector Field-Attributes
function TAboutProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paDialog, paReadOnly]
end;

// Inspector Field-Value
function TAboutProperty.GetValue: String;
begin
  GetValue := '(About)';
end;

// Display Component-Info
procedure TAboutProperty.Edit;
begin
  DisplayAbout;
end;

// *** TGradient ***********************************************************
procedure TGradient.SetColorSteps(Value: TColorSteps);
begin
  if Value <> FColorSteps then begin
    FColorSteps := Value;
    Parent.Invalidate;
  end;
end;

// set gradient start-color
procedure TGradient.SetStartColor(Value: TColor);
begin
  if Value <> FStartColor then begin
    FStartColor := Value;
    Parent.Invalidate;
  end;
end;

// set gradient end-color
procedure TGradient.SetEndColor(Value: TColor);
begin
  if Value <> FEndColor then begin
    FEndColor := Value;
    Parent.Invalidate;
  end;
end;

// set gradient-style
procedure TGradient.SetGradientStyle(Value: TGradientStyle);
begin
  if Value <> FGradientStyle then begin
    FGradientStyle := Value;
    Parent.Invalidate;
  end;
end;

// *** TSubCaption ************************************************************
// set right margin
procedure TSubCaption.SetMargin(Value: Byte);
begin
  if Value <> FMargin then begin
    FMargin := Value;
    Parent.Invalidate;
    if Parent.FAutoBounds then Parent.ResizeHeadLabel;
  end;
end;

// set subcolor
procedure TSubCaption.SetSubColor(Value: TColor);
begin
  if Value <> FColor then begin
    FColor := Value;
    Parent.Invalidate;
  end;
end;

// set subcaption
procedure TSubCaption.SetSubCaption(Value: String);
begin
  if Value <> FCaption then begin
    FCaption := Value;
    Parent.Alignment := taLeft;
    if Parent.FAutoBounds then Parent.ResizeHeadLabel;
  end;
end;

// set substyle
procedure TSubCaption.SetSubStyle(Value: TFontStyles);
begin
  if Value <> FStyle then begin
    FStyle := Value;
    Parent.Invalidate;
  end;
end;

// set subsize
procedure TSubCaption.SetSubSize(Value: Byte);
begin
  if Value <> FSize then begin
    FSize := Value;
    Parent.Invalidate;
  end;
end;

// set subellipsis
procedure TSubCaption.SetSubEllipsis(Value: Boolean);
begin
  if Value <> FEllipsis then begin
    if ((Value) and (FCaption <> '')) or Not(Value) then begin
      FEllipsis := Value;
      Parent.Invalidate;
      if Parent.FAutoBounds then Parent.ResizeHeadLabel;
    end;
  end;
end;

// *** TAPHeadLabel *************************************************************
// create new component-instance & set defaults
constructor TAPHeadLabel.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  { set default font }
  FTextAlignmt             := taLeft;
  FFont                    := TFont.Create;
  FFont.Name               := 'Arial';
  FFont.Size               := 10;
  FFont.Style              := [fsBold];
  Font.Assign(FFont);

  { set default glyph }
  FGlyph                   := TBitmap.Create;
  FGlyphMode               := gmNormal;
  FNumGlyphs               := 1;
  FGlyph.Assign(nil);

  { set display glyph }
  FGlyph             := TBitmap.Create;
  FGlyph.Assign(nil);

  { set default gradient }
  FGradient                := TGradient.Create;
  FGradient.Parent         := Self;
  FGradient.FColorSteps    := 64;
  FGradient.FEndColor      := clSilver;
  FGradient.FStartColor    := clGray;
  FGradient.FGradientStyle := gsLeft;

  { set default sub-caption }
  FSubCaption              := TSubCaption.Create;
  FSubCaption.Parent       := Self;
  FSubCaption.FMargin      := 8;
  FSubCaption.FColor       := clWindowText;
  FSubCaption.FEllipsis    := False;
  FSubCaption.FSize        := 8;

  { set default options }
  AutoSize                 := False;
  FAutoBounds              := True;
  FAlign                   := alNone;
  FBoundColor              := clGray;
  FMargin                  := 5;
  FSpacing                 := 5;
  FFormMove                := False;
  ParentFont               := True;
end;

// destroy new component & free memory allocations (used by the glyph)
destructor TAPHeadLabel.Destroy;
begin
  FFont.Free;
  FGlyph.Free;
  FGradient.Free;
  FSubCaption.Free;
  inherited Destroy;
end;

// fontchanged
procedure TAPHeadLabel.CMFontChanged(var Msg: TMessage);
begin
  inherited;
  FFont.Assign(Font);
end;

// caption-changed
procedure TAPHeadLabel.CMTextChanged(var Msg: TMessage);
begin
  inherited;
  if FAutoBounds then ResizeHeadLabel;
end;

// single-click-event
procedure TAPHeadLabel.WMLButtonDown(var Msg: TWMLBUTTONDOWN);
begin
  inherited;
  if FFormMove then begin
    ReleaseCapture;
    TWinControl(Owner).Perform(WM_SYSCOMMAND, SC_DRAGMOVE, 0);
  end;
  if (MouseInGlyph) and (Assigned(FOnGlyphClick)) then
    FOnGlyphClick(Self)
  else begin
    if Assigned(FOnClick) then FOnClick(Self);
  end;
end;

// double-click-event
procedure TAPHeadLabel.WMLButtonDblClk(var Msg: TWMLBUTTONDBLCLK);
begin
  inherited;
  if (MouseInGlyph) and (Assigned(FOnGlyphDblClick)) then
    FOnGlyphDblClick(Self)
  else if Assigned(FOnDblClick) then FOnDblClick(Self);
end;

// calculate new gradient-colorband
procedure TAPHeadLabel.CalcGradientBand(StartCol, EndCol: TColor; Steps: Byte);
var
  i, SR, SG, SB, ER, EG, EB: Integer;
  SPR, SPG, SPB:             Real;
begin
  SR  := GetRValue(StartCol); ER := GetRValue(EndCol);
  SG  := GetGValue(StartCol); EG := GetGValue(EndCol);
  SB  := GetBValue(StartCol); EB := GetBValue(EndCol);
  SPR := (ER - SR) / Steps;
  SPG := (EG - SG) / Steps;
  SPB := (EB - SB) / Steps;
  for i:=0 to Steps - 1 do
    GradientBand[i + 1] := RGB(Trunc(SR + i * SPR), Trunc(SG + i * SPG), Trunc(SB + i * SPB));
end;

// set label-align
procedure TAPHeadLabel.SetAlign(Value: TAlign);
begin
  if Value <> FAlign then begin
    FAlign := Value;
    if (Value = alClient) and (FAutoBounds) then FAutoBounds := False;
    inherited Align:=Value;
  end;
end;

// set auto-bounds
procedure TAPHeadLabel.SetAutoBounds(Value: Boolean);
begin
  if Value <> FAutoBounds then begin
    if ((Value) and (Align <> alClient)) or Not(Value) then begin
      FAutoBounds := Value;
      if Value then ResizeHeadLabel;
    end;
  end;
end;

// set bound-lines
procedure TAPHeadLabel.SetBoundLines(Value: TBoundLines);
begin
  if Value <> FBoundLines then begin
    FBoundLines := Value;
    Invalidate;
  end;
end;

// set form move
procedure TAPHeadLabel.SetFormMove(Value: Boolean);
begin
  if Value <> FFormMove then begin
    FFormMove := Value;
    if (Value) and (Align <> alTop) and (csDesigning in ComponentState) and (MessageDlg('Set Align to top?', mtConfirmation, [mbYes, mbNo], 0) = mrYES) then
      SetAlign(alTop);
  end;
end;

// set bound-color
procedure TAPHeadLabel.SetBoundColor(Value: TColor);
begin
  if Value <> FBoundColor then begin
    FBoundColor := Value;
    Invalidate;
  end;
end;

// set glyph
procedure TAPHeadLabel.SetGlyph(Value: TBitmap);
begin
  if Value <> FGlyph then begin
    FGlyph.Assign(Value);
    CalcNumGlyphs;
    if FAutoBounds then ResizeHeadLabel;
  end;
end;

// calculate num-glyphs
procedure TAPHeadLabel.CalcNumGlyphs;
begin
  if Not(FGlyph.Empty) and (FGlyph.Width mod FGlyph.Height = 0) then
    SetNumGlyphs(FGlyph.Width div FGlyph.Height)
  else SetNumGlyphs(1);
end;

// set num-glyphs
procedure TAPHeadLabel.SetNumGlyphs(Value: TNumGlyphs);
begin
  FNumGlyphs := Value;
  Invalidate;
end;

// set gylph-mode
procedure TAPHeadLabel.SetGlyphMode(Value: TGlyphMode);
begin
  if Value <> FGlyphMode then begin
    FGlyphMode := Value;
    Invalidate;
  end;
end;

// set left margin
procedure TAPHeadLabel.SetMargin(Value: Byte);
begin
  if Value <> FMargin then begin
    FMargin := Value;
    Invalidate;
    if FAutoBounds then ResizeHeadLabel;
  end;
end;

// set glyph-spacing
procedure TAPHeadLabel.SetSpacing(Value: Byte);
begin
  if Value <> FSpacing then begin
    FSpacing := Value;
    Invalidate;
    if FAutoBounds then ResizeHeadLabel;
  end;
end;

// set main-caption text alignment
procedure TAPHeadLabel.SetTextAlignmt(Value: TTextAlignmt);
begin
  if (FSubCaption.Caption = '') or ((FSubCaption.Caption <> '') and (Value = taLeft)) then begin
    FTextAlignmt := Value;
    Invalidate;
  end;
end;

// draw gradient-colorband
procedure TAPHeadLabel.DrawGradientBand;
var
  GStyle:    TGradientStyle;
  ColorRect: TRect;
  i:         Integer;

begin
  GStyle:=FGradient.FGradientStyle;
  case GStyle of
    gsLeft,  gsTop:    CalcGradientBand(FGradient.FStartColor, FGradient.FEndColor, FGradient.FColorSteps);
    gsRight, gsBottom: CalcGradientBand(FGradient.FEndColor, FGradient.FStartColor, FGradient.FColorSteps);
  end;

  // fade from left to right / right to left
  if (GStyle = gsLeft) or (GStyle = gsRight) then begin
    ColorRect.Top    := 0;
    ColorRect.Bottom := Height;

    { draw band }
    with Canvas do for i := 0 to FGradient.FColorSteps - 1 do begin
      ColorRect.Left  := MulDiv(i, Width, FGradient.FColorSteps);
      ColorRect.Right := MulDiv(i + 1, Width, FGradient.FColorSteps);
      Brush.Color     := GradientBand[i + 1];
      FillRect(ColorRect);
    end;
  end else begin

    // fade from top to bottom / bottom to top
    ColorRect.Left  := 0;
    ColorRect.Right := Width;
    with Canvas do for i := 0 to FGradient.FColorSteps - 1 do begin
      ColorRect.Top    := MulDiv(i, Height, FGradient.FColorSteps);
      ColorRect.Bottom := MulDiv(i + 1, Height, FGradient.FColorSteps);
      Brush.Color      := GradientBand[i + 1];
      FillRect(ColorRect);
    end;
  end;
end;

// paint master-component
procedure TAPHeadLabel.Paint;
const
  TextAlign: Array[TTextAlignmt] of Word = (DT_CENTER, DT_LEFT, DT_RIGHT);
var
  GOffset,
  GWidth, GHeight: Integer;
  Rect:            TRect;
  Bitmap:          TBitmap;
begin
  with Canvas do begin

    // draw gradient-colorband
    if FGradient.FGradientStyle <> gsNone then
      DrawGradientBand
    else begin
      Brush.Color:=FGradient.FStartColor;
      FillRect(GetClientRect);
    end;

    // draw bound-lines
    Pen.Color:=FBoundColor;
    if blLeft in FBoundLines then begin
      MoveTo(0, 0);
      LineTo(0, Height - 1);
    end;
    if blTop in FBoundLines then begin
      MoveTo(0, 0);
      LineTo(Width - 1, 0);
    end;
    if blRight in FBoundLines then begin
      MoveTo(Width - 1, 0);
      LineTo(Width - 1, Height - 1);
    end;
    if blBottom in FBoundLines then begin
      MoveTo(0, Height - 1);
      LineTo(Width - 1, Height - 1);
    end;

    // display (scaled) transparent-glyph
    GOffset := FMargin;
    if not(FGlyph.Empty) then begin
      Bitmap := TBitmap.Create;
      try
        Bitmap.Assign(FGlyph);
        Bitmap.Transparent := True;
        Bitmap.Width := Bitmap.Width div FNumGlyphs;
        GWidth       := Bitmap.Width;
        GHeight      := Bitmap.Height;
        if GWidth > Width then begin
          GWidth := Width - 2;
          Dec(GHeight, (Bitmap.Width - GWidth) * 4);
        end;
        if GHeight > Height then begin
          GHeight := Height - 2;
          Dec(GWidth, (Bitmap.Height - GHeight) div 4);
        end;
        StretchDraw(Bounds(FMargin, (Height - GHeight) div 2, GWidth, GHeight), Bitmap);
        if FGlyphMode = gmNormal then Inc(GOffset, GWidth + FSpacing);
      finally
        Bitmap.Free;
      end;
    end;

    // display main-caption (left alignment)
    SetBkMode(Canvas.Handle, 0);
    Font.Assign(FFont);
    Rect := ClientRect;
    Inc(Rect.Left, GOffset);
    DrawText(Handle, PChar(GetClippedCaption(GOffset)), -1, Rect, DT_LEFT or DT_SINGLELINE or DT_VCENTER or TextAlign[FTextAlignmt]);

    // display sub-caption (right alignment)
    if FSubCaption.FCaption <> '' then begin
      Font.Color := FSubCaption.FColor;
      Font.Size  := FSubCaption.FSize;
      Font.Style := FSubCaption.FStyle;
      Rect       := ClientRect;
      Dec(Rect.Right, FSubCaption.FMargin);
      if Width > Canvas.TextWidth(GetSubCaption) + 8 then
        DrawText(Handle, PChar(GetSubCaption), -1, Rect, DT_RIGHT or DT_SINGLELINE or DT_VCENTER);
    end;
  end;
end;

// returns clipped main-caption
function TAPHeadLabel.GetClippedCaption(Offset: Integer): String;
var
  NewCaption: String;
  NewWidth:   Integer;

  procedure KillLastChar;
  begin
    if Length(NewCaption) > 0 then begin
      Delete(NewCaption, Length(NewCaption), 1);
      if Canvas.TextWidth(NewCaption) > NewWidth then KillLastChar;
    end;
  end;
begin
  NewCaption := Caption;
  NewWidth   := Width - (Offset + Canvas.TextWidth(GetSubCaption) + FSubCaption.FMargin + Canvas.TextWidth('... '));
  if Canvas.TextWidth(NewCaption) > NewWidth then KillLastChar;
  if (Length(Caption) > Length(NewCaption)) and (NewCaption <> '') then
    Result := Concat(NewCaption, '...')
  else Result := NewCaption;
end;

// return sub-caption (with optinal ellipsis...)
function TAPHeadLabel.GetSubCaption: String;
var
  EllCap: String;
begin
  EllCap := FSubCaption.Caption;
  if FSubCaption.FEllipsis then EllCap := Concat(EllCap, '...');
  Result := EllCap;
end;

// return if mouse-in-glyph
function TAPHeadLabel.MouseInGlyph: Boolean;
var
  CPos: TPoint;
begin
  Result:=False;
  if (Not(FGlyph.Empty)) and (GetCursorPos(CPos)) then begin
    CPos   := ScreenToClient(CPos);
    Result := (CPos.X >= FMargin) and (CPos.X <= FMargin + FGlyph.Width);
  end;
end;

// resize Head-Label
procedure TAPHeadLabel.ResizeHeadLabel;
var
  GXOff, GYOff, WOff, HOff, NewHeight:  Integer;
begin
  if Align <> alClient then begin
    GXOff := 0; GYOff := 0; WOff := 0; HOff := 0;
    if Not(FGlyph.Empty) then begin
      GXOff := FGlyph.Width;
      GYOff := FGlyph.Height;
    end;

    NewHeight := Canvas.TextHeight(Caption);
    if Canvas.TextHeight(GetSubCaption) > NewHeight then NewHeight := Canvas.TextHeight(GetSubCaption);
    if GYOff > NewHeight then NewHeight := GYOff;

    if blLeft in FBoundLines   then Inc(WOff);
    if blRight in FBoundLines  then Inc(WOff);
    if blTop in FBoundLines    then Inc(HOff);
    if blBottom in FBoundLines then Inc(HOff);

    if Align in [alNone, alLeft, alRight] then Width  := FMargin + GXOff + FSpacing + Canvas.TextWidth(Caption) + Canvas.TextWidth(GetSubCaption) + FSubCaption.FMargin + Canvas.TextWidth('... ') + WOff;
    if Align in [alNone, alTop, alBottom] then Height := NewHeight + HOff;
  end;
end;

// *** TAPHeadEditor *******************************************************
function TAPHeadEditor.GetVerbCount: Integer;
begin
  Result := 2;
end;

// set Context-MenuItems
function TAPHeadEditor.GetVerb(Index: Integer): string;
begin
  case Index of
    0: Result := 'About...';
    1: Result := 'Minimize Bounds...';
  end;
end;

// execute verbs
procedure TAPHeadEditor.ExecuteVerb(Index: Integer);
begin
  case Index of
    0: DisplayAbout;
    1: TAPHeadLabel(Component).ResizeHeadLabel;
  end;
end;

// register new component (- editors)
procedure Register;
begin
  RegisterComponents('a priori', [TAPHeadLabel]);
  RegisterPropertyEditor(TypeInfo(TAboutProperty), TAPHeadLabel, 'About', TAboutProperty);
  RegisterComponentEditor(TAPHeadLabel, TAPHeadEditor);
end;

end.
