unit RTS;

// RTS         - This unit (real time shadowing) is responsible
//               for the creation of shadows from a given Scene.
//               In order to avoid unnecessary version checks, it's assumed
//               OpenGL version 1.1+ is installed.
// Version     - 0.0.11
// Last Change - 11. February 1998
// for more information see help file

// This unit is based on the description of real time shadows from Mark Kilgard.
// Original copyright notice:

    // Copyright (c) Mark J. Kilgard, 1997.

    // This program is freely distributable without licensing fees and is
    // provided without guarantee or warrantee expressed or implied.  This
    // program is -not- in the public domain.

    // Real-time Shadowing library, Version 0.8

    // XXX This is library is not fully implemented yet, but still quite
    // functional.

interface

uses Geometry, OpenGL;

const UniquePassThroughValue : GLFLoat = 34567;
      MAX_CONTEXTS           = 1;

type T2DFltVector     = array[0..1] of GLfloat;

     TRTSWarning      = (RTS_ERROR_OUT_OF_MEMORY,
                         RTS_WARNING_EYE_IN_SHADOW,
                         RTS_WARNING_LIGHT_TOO_CLOSE);
     TRTSLightState   = (RTS_OFF,RTS_SHINING,RTS_SHINING_AND_CASTING);
     TRTSObjectState  = (RTS_NOT_SHADOWING,RTS_SHADOWING);
     TRTSMode         = (RTS_NO_SHADOWS,RTS_USE_SHADOWS,RTS_USE_SHADOWS_NO_OVERLAP);

     PVertexHolder2D = ^TVertexHolder2D;
     TVertexHolder2D = record
                         Next : PVertexHolder2D;
                         V    : T2DFltVector;
                       end;

     PVertexHolder3D = ^TVertexHolder3D;
     TVertexHolder3D = record
                         Next : PVertexHolder3D;
                         V    : TAffineVector;
                       end;

     PTessellationContext = ^TTessellationContext;
     TTessellationContext = record
                              RefCnt           : Integer;
                              Tess             : PGLUTesselator;
                              CombineList      : PFloatVector;
                              CombineListSize,
                              CombineNext      : Integer;
                              ExcessList2D     : PVertexHolder2D;
                              SaveFirst        : Boolean;
                              FirstVertex      : PFloatVector;
                              FeedbackBuffer   : PGLFloat;
                              FeedbackBufferSize,
                              FeedbackBufferReturned : Integer;
                              ShadowProjectionDistance,
                              ExtentScale      : GLFLoat;
                              SilhouetteSize   : Integer;
                              Silhouette       : PFloatVector;
                              NextVertex       : Integer;
                              Header           : PAffineIntVector;
                              ExcessList3D     : PVertexHolder3D;
                            end;

     PRTSObject           = ^TRTSObject;
     PRTSLight            = ^TRTSLight;
     PRTSScene            = ^TRTSScene;
     PShadowVolumeState   = ^TShadowVolumeState;
     TSceneList           = array[0..0] of PRTSScene;
     TLightList           = array[0..0] of PRTSLight;
     TObjectList          = array[0..0] of PRTSObject;
     TSVSList             = array[0..0] of PShadowVolumeState;
     TByte32              = array[0..31] of Byte;

     TRenderSceneFunc     = procedure(CastingLight: GLenum; SceneData: Pointer; Scene: PRTSScene);

     TRTSScene            = record
                              EyePos            : TAffineVector;
                              UsableStencilBits : GLbitfield;
                              NumStencilBits    : Integer;
                              BitList           : TByte32;
                              RenderSceneFunc   : TRenderSceneFunc;
                              SceneData         : Pointer;
                              NumContexts       : Integer;
                              Context           : array[0..MAX_CONTEXTS-1] of PTessellationContext;
                              ViewScale         : GLFLoat;
                              StencilBits       : GLInt;
                              StencilValidateNeeded : Boolean;
                              SceneAmbient      : TVector;
                              LightListSize     : Integer;
                              LightList         : ^TLightList;
                            end;


     TShadowVolumeState   = record
                              LightSerNum,
                              ObjectSerNum,
                              SilhouetteSize : Integer;
                              Silhouette     : PFloatVector;
                              Phi, Theta     : GLFLoat;
                              Axis           : TAffineVector;
                              TopScale       : GLfloat;
                            end;

     TRTSLight            = record
                              RefCnt,
                              SerNum           : Integer;
                              glLight          : GLEnum;
                              LightPos         : TAffineVector;
                              Radius           : GLFLoat;
                              State            : TRTSLightState;
                              SceneListSize    : Integer;
                              SceneList        : ^TSceneList;
                              ObjectListSize   : Integer;
                              ObjectList       : ^TObjectList;
                              ShadowVolumeList : ^TSVSList;
                            end;

     TRenderObjectFunc    = procedure(ObjectData: Pointer);

     TRTSObject           = record
                              RefCnt,
                              SerNum        : Integer;
                              ObjectPos     : TAffineVector;
                              MaxRadius     : GLFLoat;
                              RenderObject  : TRenderObjectFunc;
                              ObjectData    : Pointer;
                              State         : TRTSObjectState;
                              FeedbackBufferSizeGuess,
                              LightListSize : Integer;
                              LightList     : ^TLightList;
                            end;

function RTSCreateScene(EyePos: TAffineVector; UsableStencilBits: GLBitField;
                        RenderSceneFunc: TRenderSceneFunc; SceneData: Pointer) : PRTSScene;
function RTSCreateLight(glLight: GLEnum; LightPos: TVector; Radius: GLFloat) : PRTSlight;
function RTSCreateObject(ObjectPos: TAffineFltVector; MaxRadius: GLFLoat;
                         RenderObject: TRenderObjectFunc; ObjectData: Pointer;
                         FeedbackBufferSizeGuess: Integer) : PRTSobject;
procedure RTSAddLightToScene(Scene: PRTSScene; Light: PRTSlight);
procedure RTSAddObjectToLight(Light: PRTSlight; AObject: PRTSobject);
procedure RTSFreeLight(var Light: PRTSLight);
procedure RTSFreeObject(var AObject: PRTSObject);
procedure RTSFreeScene(var Scene: PRTSScene);
procedure RTSSetLightState(Light: PRTSlight; State: TRTSlightState);
procedure RTSSetObjectState(AObject: PRTSobject; State: TRTSobjectState);
procedure RTSUpdateEyePos(Scene: PRTSScene; EyePos: TAffineVector);
procedure RTSUpdateUsableStencilBits(Scene: PRTSScene; UsableStencilBits: GLbitfield);
procedure RTSUpdateLightPos(Light: PRTSlight; LightPos: TVector);
procedure RTSUpdateLightRadius(Light: PRTSlight; Radius: GLfloat);
procedure RTSUpdateObjectPos(AObject: PRTSobject; ObjectPos: TAffineVector);
procedure RTSUpdateObjectShape(AObject: PRTSobject);
procedure RTSUpdateObjectMaxRadius(AObject: PRTSobject; MaxRadius: GLfloat);
procedure RTSRenderScene(Scene: PRTSScene; Mode: TRTSMode);
procedure RTSRenderSilhouette(Scene: PRTSScene; Light: PRTSlight; AObject: PRTSobject);

//------------------------------------------------------------------------------

implementation

uses Dialogs, Math, SysUtils;

const X : Integer = 0;
      Y : Integer = 1;
      Z : Integer = 2;
      
var HasBlendSubtract : Boolean;

{$R- - switch off range checking, since some c-like typecasts must be made}

//------------------------------------------------------------------------------

function ExtensionSupported(AExt: String) : Boolean;

var Extensions : PChar;

begin
  Extensions:=PChar(glGetString(GL_EXTENSIONS));
  if Pos(AExt,Extensions) > 0 then Result:=True
                              else Result:=False;
end;

//------------------------------------------------------------------------------

function NextVertexHolder3D(Context: PTessellationContext): PFloatVector;

var Holder : PVertexHolder3D;

begin
  if Context.NextVertex >= Context.SilhouetteSize then
  begin
    New(Holder);
    Holder.Next:=Context.ExcessList3D;
    Context.ExcessList3D:=Holder;
    Result:=@Holder.V;
  end
  else Result:=@Context.Silhouette[Context.NextVertex*3];
  Inc(Context.NextVertex);
end;

//----------------- tesselation callbacks --------------------------------------

procedure StartCallback(AType: GLEnum; PolyData: Pointer); stdcall;

var Context : PTessellationContext;
    New     : PFloatVector;

begin
  Context:=PolyData;
  Assert(AType = GL_LINE_LOOP);
  Context.SaveFirst:=True;
  New:=NextVertexHolder3D(Context);
  Context.Header:=Pointer(New);
  Context.Header[X]:=Context.NextVertex;
  Context.Header[Y]:=$deadbabe;  // Aid assertion testing.
  Context.Header[Z]:=$deadbeef;  // Non-terminator token.
  New:=NextVertexHolder3D(Context);
  New[X]:=0;
  New[Y]:=0;
  New[Z]:=0;
end;

//------------------------------------------------------------------------------

procedure VertexCallback(Data, PolyData: Pointer); stdcall;

var Context : PTessellationContext;
    V       : PFloatVector;
    New     : PFloatVector;

begin
  Context:=PolyData;
  V:=Data;
  New:=NextVertexHolder3D(Context);
  New[X]:=Context.ExtentScale*V[X];
  New[Y]:=Context.ExtentScale*V[Y];
  New[Z]:=Context.ShadowProjectionDistance;
  if Context.SaveFirst then
  begin
    Context.FirstVertex:=New;
    Context.SaveFirst:=False;
  end
end;

//------------------------------------------------------------------------------

procedure StopCallBack(PolyData: Pointer); stdcall;

var Context : PTessellationContext;
    New     : PFloatVector;

begin
  Context:=PolyData;
  New:=NextVertexHolder3D(Context);
  New[X]:=Context.FirstVertex[X];
  New[Y]:=Context.FirstVertex[Y];
  New[Z]:=Context.FirstVertex[Z];

  Assert(Context.FirstVertex[Z] = Context.ShadowProjectionDistance);
  Assert(Context.Header[1] = $deadbabe);
  Assert(Context.Header[2] = $deadbeef);
  Context.Header[1]:=Context.NextVertex-Context.Header[0];
end;

//------------------------------------------------------------------------------

procedure FreeExcessList(Context: PTessellationContext);

var Holder, Next : PVertexHolder2D;

begin
  Holder:=Context.ExcessList2D;
  while assigned(Holder) do
  begin
    Next:=Holder.Next;
    Dispose(Holder);
    Holder:=Next;
  end;
  Context.ExcessList2D:=nil;
end;

//------------------------------------------------------------------------------

procedure CombineCallback(Coords: TAffineDblVector; D: THomogeneousPtrVector;
                          W: TVector; DataOut : PPointer; PolyData: Pointer); stdcall;

var Context : PTessellationContext;
    Holder  : PVertexHolder2D;
    V       : PFloatVector;

begin
  Context:=PolyData;
  if Context.CombineNext >= Context.CombineListSize then
  begin
    New(Holder);
    Holder.Next:=Context.ExcessList2D;
    Context.ExcessList2D:=Holder;
    V:=@Holder.V;
  end
  else V:=@Context.CombineList[Context.CombineNext*2];

  V[X]:=Coords[0];
  V[Y]:=Coords[1];
  DataOut^:=V;
  Inc(Context.CombineNext);
end;

//------------------------------------------------------------------------------

procedure ErrorCallback(ErrNo: GLenum); stdcall;

begin
  ShowMessage('RTS ErrorCallback: '+gluErrorString(ErrNo));
end;

//------------------------------------------------------------------------------

procedure GenerateSilhouette(SVS: PShadowVolumeState; Context: PTessellationContext);

var BufferEnd,
    Loc,
    EyeLoc           : PGLFloat;
    V                : TAffineDblVector;
    WatchingForEyePos : Boolean;
    Token,
    NVertices,
    OldSize,
    I,J              : Integer;
    PassThroughToken : GLFloat;
    Holder, Next     : PVertexHolder3D;

begin
  Assert(Context.ExcessList2D = nil);
  Assert(Context.ExcessList3D = nil);

  Context.NextVertex:=0;
  Context.SilhouetteSize:=SVS.SilhouetteSize;
  Context.Silhouette:=SVS.Silhouette;

  WatchingForEyePos:=False;
  EyeLoc:=nil;

  gluTessBeginPolygon(Context.Tess,Context);
  Loc:=Context.FeedbackBuffer;
  BufferEnd:=Loc;
  Inc(BufferEnd,Context.FeedbackBufferReturned);
  while Longint(Loc) < Longint(BufferEnd) do
  begin
    Token:=Round(Loc^);
    Inc(Loc);                          
    case Token of
      GL_POLYGON_TOKEN : begin
                           NVertices:=Round(Loc^);
                           Inc(Loc);
                           Assert(NVertices >= 3);
                           gluTessBeginContour(Context.Tess);
                           for I:=0 to NVertices-1 do
                           begin
                             V[X]:=PFloatVector(Loc)[X];
                             V[Y]:=PFloatVector(Loc)[Y];
                             V[Z]:=0;
                             gluTessVertex(Context.Tess,V,Loc);
                             Inc(Loc,2);
                           end;
                           gluTessEndContour(Context.Tess);
                         end;
      GL_PASS_THROUGH_TOKEN : begin
                                PassThroughToken:=Loc^;
                                if PassThroughToken = UniquePassThroughValue
                                  then WatchingForEyePos:=not WatchingForEyePos
                                  else
                                    // ignore everything else
                                    ShowMessage(Format('WARNING: Unexpected feedback token $%x (%d).',[Token, Token]));
                                Inc(Loc);
                              end;
      GL_POINT_TOKEN : begin
                         if WatchingForEyePos then
                         begin
                           Showmessage('WARNING: Eye point possibly within the shadow volume.'#13+
                                       '         Program should be improved to handle this.');
                            { XXX Write code to handle this case.  You would need to determine
                               if the point was instead any of the returned boundary polyons.
                               Once you found that you were really in the clipping volume, then I
                               haven't quite thought about what you do.}
                            EyeLoc:=Loc;
                            Inc(Loc,3);
                            WatchingForEyePos:=False;
                         end
                         else
                           // ignore everything else
                           ShowMessage(Format('WARNING: Unexpected feedback token $%x (%d).',[Token, Token]));
                           Inc(Loc);
                       end
    else
      // ignore everything else
      ShowMessage(Format('WARNING: Unexpected feedback token $%x (%d).',[Token, Token]));
    end;
  end;
  gluTessEndPolygon(Context.Tess);

  // Free any memory that got allocated due to CombineCallback during
  // tessellation and then enlarge the combineList so we hopefully don't need
  // the CombineCallback list next time.
  if Context.CombineNext > Context.CombineListSize then
  begin
    FreeExcessList(Context);  
    Context.CombineListSize:=Context.CombineNext;
    ReallocMem(Context.CombineList,2*SizeOf(GLfloat)*Context.CombineListSize);
  end;
  Context.CombineNext:=0;
  Context.Header[2]:=$cafecafe;  // terminating token

  if assigned(Context.ExcessList3D) then
  begin
    OldSize:=Context.SilhouetteSize;
    Assert(Context.NextVertex > Context.SilhouetteSize);
    Context.SilhouetteSize:=Context.NextVertex;
    ReallocMem(Context.Silhouette,3*Context.SilhouetteSize*SizeOf(GLFloat));
    if Context.Silhouette = nil then
      raise Exception.Create('RTS: GenerateSilhouette: out of memory');
    Holder:=Context.ExcessList3D;
    while assigned(Holder) do
    begin
      Dec(Context.NextVertex);
      Context.Silhouette[Context.NextVertex*3]:=Holder.V[0];
      Context.Silhouette[Context.NextVertex*3+1]:=Holder.V[1];
      Context.Silhouette[Context.NextVertex*3+2]:=Holder.V[2];
      Next:=Holder.Next;
      Dispose(Holder);
      Holder:=Next;
    end;
    Assert(Context.NextVertex = OldSize);
    Context.ExcessList3D:=nil;
  end;
  SVS.SilhouetteSize:=Context.SilhouetteSize;
  SVS.Silhouette:=Context.Silhouette;
end;

//------------------------------------------------------------------------------

function ListBits(UsableStencilBits: GLbitfield; var BitList: TByte32) : Integer;

var Num, Bit : Integer;

begin
  Num:=0; Bit:=0;
  FillChar(BitList,SizeOf(TByte32),0);
  while UsableStencilBits > 0 do
  begin
    if (UsableStencilBits and 1) > 0 then
    begin
      BitList[Num]:=Bit;
      Inc(Num);
    end;
    Inc(Bit);
    UsableStencilBits:=UsableStencilBits shr 1;
  end;
  Result:=Num;
end;

//------------------------------------------------------------------------------

function GetViewScale(Scene: PRTSScene) : GLFLoat;

var MaxViewSize : T2DFltVector;

begin
  if Scene.ViewScale = 0 then
  begin
    glGetFloatv(GL_MAX_VIEWPORT_DIMS,@MaxViewSize);
    Scene.ViewScale:=MinValue([MaxViewSize[0],MaxViewSize[1]])/2;

    // Other stuff piggy backs on ViewScale to ensure initialization.
    glGetIntegerv(GL_STENCIL_BITS,@Scene.StencilBits);
    HasBlendSubtract:=ExtensionSupported('GL_EXT_blend_subtract');
  end;
  Result:=Scene.ViewScale;
end;

//------------------------------------------------------------------------------

procedure CaptureLightView(Scene: PRTSScene; Light: PRTSlight; AObject: PRTSobject;
                           SVS: PShadowVolumeState; Context: PTessellationContext);

var FeedbackBufferSizeGuess : Integer;
    Up, Temp,
    LightDelta, EyeDelta    : TAffineVector;
    NearExtent,
    FarExtent,
    LightDistance,
    EyeDistance,
    FieldOfViewRatio,
    FieldOfViewAngle        : GLFLoat;
    ViewScale               : Integer;
    Returned                : GLInt;

begin
  ViewScale:=Round(GetViewScale(Scene));

  glMatrixMode(GL_PROJECTION);
  glPushMatrix;
  glLoadIdentity;

  // Calculate the Light's distance from the AObject being shadowed.
  LightDelta:=VectorAffineSubtract(AObject.ObjectPos,Light.LightPos);
  //LightDelta:=VectorAffineSubtract(Light.LightPos,AObject.ObjectPos);
  LightDistance:=VectorLength(LightDelta);

  // Determine the appropriate field of view.  We want to use as narrow a
  // field of view as possible to not waste resolution, but not narrower than
  // the AObject.  Add 5% extra slop.
  FieldOfViewRatio:=AObject.MaxRadius/LightDistance;
  if FieldOfViewRatio > 0.99 then
  begin
    ShowMessage(Format('WARNING: Clamping FOV to 164 degrees for determining shadow.'#13+
                       '         Light distance:=%g, AObject maxmium Radius:=%g',
                       [LightDistance, AObject.MaxRadius]));

    // 2*asin(0.99) ~= 164 degrees.
    FieldOfViewRatio:=0.99;
  end;
  // Pre-compute scaling factors for the near and far extent of the shadow volume.
  Context.ExtentScale:=Light.Radius*FieldOfViewRatio/ViewScale;
  Context.ShadowProjectionDistance:=Light.Radius;

  NearExtent:=0.5*(LightDistance-AObject.MaxRadius);
  if NearExtent < 0.0001 then
  begin
    ShowMessage(Format('WARNING: Clamping near clip plane to 0.0001 because Light source too near.'#13+
                       '         Light distance:=%g, AObject maxmium Radius:=%g',
                       [LightDistance,AObject.MaxRadius]));
    NearExtent:=0.0001;
  end;
  FarExtent:=2*(LightDistance+AObject.MaxRadius);

  EyeDelta:=VectorAffineSubtract(Scene.EyePos,Light.LightPos);
  EyeDistance:=1.05*VectorLength(EyeDelta);
  if EyeDistance > FarExtent then FarExtent:=EyeDistance;
  FieldOfViewAngle:=2*RadToDeg(arcsin(FieldOfViewRatio));
  gluPerspective(FieldOfViewAngle,1,NearExtent,FarExtent);

  SVS.Axis:=VectorCrossProduct(YVector,MakeAffineVector([LightDelta[X],0,LightDelta[Z]]));
  SVS.TopScale:=(LightDistance+AObject.MaxRadius)/Light.Radius;
  SVS.Phi:=RadToDeg(arccos(VectorAngle(ZVector,MakeAffineVector([LightDelta[X],0,LightDelta[Z]]))));
  if LightDelta[X] < 0 then SVS.Phi:=-SVS.Phi;
  if LightDelta[X]+LightDelta[Y] = 0
    then SVS.Theta:=0
    else SVS.Theta:=RadToDeg(arccos(VectorAngle(LightDelta,YVector)))-90;

  glMatrixMode(GL_MODELVIEW);
  glPushMatrix;
  glLoadIdentity;
  gluLookAt(Light.LightPos[X],Light.LightPos[Y],Light.LightPos[Z],
            AObject.ObjectPos[X],AObject.ObjectPos[Y],AObject.ObjectPos[Z],
            0,1,0);
  glPushAttrib(GL_VIEWPORT_BIT);
  glViewport(-ViewScale,-ViewScale,2*ViewScale,2*ViewScale);

  FeedbackBufferSizeGuess:=AObject.FeedbackBufferSizeGuess;

  repeat
    if FeedbackBufferSizeGuess > Context.FeedbackBufferSize then
    begin
      Context.FeedbackBufferSize:=FeedbackBufferSizeGuess;
      // XXX Add 32 words of slop (an extra cache line) to end for buggy
      // hardware that uses DMA to return feedback results but that sometimes
      // overrun the buffer.  Yuck.
      ReallocMem(Context.FeedbackBuffer,Context.FeedbackBufferSize*SizeOf(GLfloat)+32*4);
    end;
    glFeedbackBuffer(Context.FeedbackBufferSize,GL_2D,Context.FeedbackBuffer);
    glRenderMode(GL_FEEDBACK);

    // Render the eye position.  The eye position is 'bracketed' by unique pass
    // through tokens.  These bracketing pass through tokens let us determine if
    // the eye position was clipped or not.  This helps us determine whether  the
    // eye position is possibly within the shadow volume or not.  If the point is
    // clipped, the eye position is not in the shadow volume.  If the point is
    // not clipped, a more complicated test is necessary to determine if the eye
    // position is really in the shadow volume or not.  See GenerateSilhouette.
    glPassThrough(UniquePassThroughValue);
    glBegin(GL_POINTS);
    glVertex3fv(@Scene.EyePos);
    glEnd;
    glPassThrough(UniquePassThroughValue);
    AObject.RenderObject(AObject.ObjectData);

    Returned:=glRenderMode(GL_RENDER);
    Assert(Returned <= Context.FeedbackBufferSize);
    if Returned = -1 then FeedbackBufferSizeGuess:=Context.FeedbackBufferSize+(Context.FeedbackBufferSize shr 1);
  until Returned <> -1; // try again with larger feedback buffer
  glPopAttrib;          // restore viewport
  glMatrixMode(GL_MODELVIEW);
  glPopMatrix;
  glMatrixMode(GL_PROJECTION);
  glPopMatrix;
  
  Context.SilhouetteSize:=SVS.SilhouetteSize;
  Context.Silhouette:=SVS.Silhouette;
  Context.FeedbackBufferReturned:=Returned;
end;

//------------------------------------------------------------------------------

function CreateTessellationContext : PTessellationContext;

var Context : PTessellationContext;
    ATess   : PGLUTesselator;

begin
  Result:=nil;
  New(Context);
  if Context = nil then Exit;
  ATess:=gluNewTess;
  if ATess = nil then
  begin
    Dispose(Context);
    Exit;
  end;
  gluTessProperty(ATess,GLU_TESS_BOUNDARY_ONLY,1);
  gluTessProperty(ATess,GLU_TESS_WINDING_RULE,GLU_TESS_WINDING_NONZERO);
  gluTessCallback(ATess,GLU_TESS_BEGIN_DATA,@StartCallback);
  gluTessCallback(ATess,GLU_TESS_VERTEX_DATA,@VertexCallback);
  gluTessCallback(ATess,GLU_TESS_COMBINE_DATA,@CombineCallback);
  gluTessCallback(ATess,GLU_TESS_END_DATA,@StopCallBack);
  gluTessCallback(ATess,GLU_TESS_ERROR,@ErrorCallback);
  with Context^ do
  begin
    Tess:=ATess;
    RefCnt:=0;
    CombineListSize:=0;
    CombineList:=nil;
    CombineNext:=0;
    ExcessList2D:=nil;
    FeedbackBufferSize:=0;
    FeedbackBuffer:=nil;
    ExcessList3D:=nil;
    Header:=nil;
  end;
  Result:=Context;
end;

//------------------------------------------------------------------------------

function RTSCreateScene(EyePos: TAffineVector; UsableStencilBits: GLBitField;
                        RenderSceneFunc: TRenderSceneFunc; SceneData: Pointer) : PRTSScene;

var Scene : PRTSScene;

begin
  Result:=nil;
  New(Scene);
  if Scene = nil then Exit;
  
  Scene.EyePos[X]:=EyePos[X];
  Scene.EyePos[Y]:=EyePos[Y];
  Scene.EyePos[Z]:=EyePos[Z];
  Scene.UsableStencilBits:=UsableStencilBits;
  Scene.RenderSceneFunc:=RenderSceneFunc;
  Scene.SceneData:=SceneData;

  Scene.NumContexts:=1;
  Scene.Context[0]:=CreateTessellationContext;

  Scene.ViewScale:=0;  // 0 means 'to be determined'.
  Scene.StencilValidateNeeded:=True;

  Scene.SceneAmbient[0]:=0.2;
  Scene.SceneAmbient[1]:=0.2;
  Scene.SceneAmbient[2]:=0.2;
  Scene.SceneAmbient[3]:=1.0;

  Scene.LightListSize:=0;
  Scene.LightList:=nil;

  Result:=Scene;
end;

//------------------------------------------------------------------------------

function RTSCreateLight(glLight: GLEnum; LightPos: TVector; Radius: GLFloat) : PRTSlight;

var Light : PRTSlight;

begin
  Result:=nil;
  New(Light);
  if Light = nil then Exit;
  
  Light.RefCnt:=1;
  Light.SerNum:=1;

  Light.glLight:=glLight;
  Light.LightPos[X]:=LightPos[X];
  Light.LightPos[Y]:=LightPos[Y];
  Light.LightPos[Z]:=LightPos[Z];
  Light.Radius:=Radius;
  Light.State:=RTS_SHINING_AND_CASTING;

  Light.SceneListSize:=0;
  Light.SceneList:=nil;

  Light.ObjectListSize:=0;
  Light.ObjectList:=nil;
  Light.ShadowVolumeList:=nil;

  Result:=Light;
end;

//------------------------------------------------------------------------------

function RTSCreateObject(ObjectPos: TAffineVector; MaxRadius: GLFLoat;
                         RenderObject: TRenderObjectFunc; ObjectData: Pointer;
                         FeedbackBufferSizeGuess: Integer) : PRTSobject;

var AObject : PRTSobject;

begin
  Result:=nil;
  New(AObject);
  if AObject = nil then Exit;
  AObject.RefCnt:=1;
  AObject.SerNum:=1;

  AObject.ObjectPos[X]:=ObjectPos[X];
  AObject.ObjectPos[Y]:=ObjectPos[Y];
  AObject.ObjectPos[Z]:=ObjectPos[Z];
  AObject.MaxRadius:=MaxRadius;
  AObject.RenderObject:=RenderObject;
  AObject.ObjectData:=ObjectData;
  AObject.FeedbackBufferSizeGuess:=FeedbackBufferSizeGuess;
  AObject.State:=RTS_SHADOWING;
  AObject.LightListSize:=0;
  AObject.LightList:=nil;

  Result:=AObject;
end;

//------------------------------------------------------------------------------

procedure RTSAddLightToScene(Scene: PRTSScene; Light: PRTSlight);

var I : Integer;

begin
  I:=0; // in case the loop never starts
  for I:=0 to Light.SceneListSize-1 do
  begin
    // scene already in the list?
    if Light.SceneList[I] = Scene then Exit;
    // found an empty entry?
    if Light.SceneList[I] = nil then Break;
  end;
  // nothing found, so adjust list size and insert new entry
  if I = Light.SceneListSize then
  begin
    Inc(Light.SceneListSize);
    ReallocMem(Light.SceneList,Light.SceneListSize*SizeOf(PRTSScene));
    if (Light.SceneList = nil) then raise Exception.Create('RTSAddLightToScene: out of memory.');
  end;

  Light.SceneList[I]:=Scene;

  I:=0; // in case the loop never starts
  for I:=0 to Scene.LightListSize-1 do
  begin
    if Scene.LightList[I] = Light then raise Exception.Create('RTSAddLightToScene: inconsistent lists.');
    if Scene.LightList[I] = nil then break;
  end;
  // nothing found, so adjust list size and insert new entry
  if I = Scene.LightListSize then
  begin
    Inc(Scene.LightListSize);
    ReallocMem(Scene.LightList,Scene.LightListSize*SizeOf(PRTSlight));
    if (Scene.LightList = nil) then raise Exception.Create('RTSAddLightToScene: out of memory.');
  end;

  Scene.LightList[I]:=Light;
  Inc(Light.RefCnt);
end;

//------------------------------------------------------------------------------

procedure InitShadowVolumeState(SVS: PShadowVolumeState);

begin
  SVS.SilhouetteSize:=0;
  SVS.Silhouette:=nil;
  SVS.LightSerNum:=0;
  SVS.ObjectSerNum:=0;
end;

//------------------------------------------------------------------------------

procedure FreeShadowVolumeState(var SVS: PShadowVolumeState);

// free allocated memory for the given SVS

begin
  if assigned(SVS) then
  with SVS^ do
  begin
    FreeMem(Silhouette,3*SilhouetteSize*SizeOf(GLfloat));
    Dispose(SVS);
    SVS:=nil;
  end;
end;

//------------------------------------------------------------------------------

procedure RTSFreeLight(var Light: PRTSLight);

// free allocated memory for the given light

var I : Integer;

begin
  if assigned(Light) then
  with Light^ do
  begin
    if assigned(ObjectList) then
    begin
      FreeMem(ObjectList,ObjectListSize*SizeOf(PRTSObject));
      for I:=0 to ObjectListSize-1 do FreeShadowVolumeState(ShadowVolumeList[I]);
      FreeMem(ShadowVolumeList,ObjectListSize*SizeOf(PShadowVolumeState));
    end;
    Dispose(Light);
    Light:=nil;
  end;
end;

//------------------------------------------------------------------------------

procedure RTSFreeObject(var AObject: PRTSObject);

// free allocated memory for the given object

begin
  if assigned(AObject) then
  with AObject^ do
  begin
    if assigned(LightList) then FreeMem(LightList,LightListSize*SizeOf(PRTSLight));
    Dispose(AObject);
    AObject:=nil;
  end;
end;

//------------------------------------------------------------------------------

procedure FreeTessellationContext(var Context: PTessellationContext);

// free allocated memory for the given context

begin
  if assigned(Context) then
  with Context^ do
  begin
    if assigned(Tess) then gluDeleteTess(Tess);
    if assigned(CombineList) then FreeMem(CombineList,2*SizeOf(GLfloat)*CombineListSize);
    if assigned(FeedbackBuffer) then FreeMem(FeedbackBuffer,FeedbackBufferSize*SizeOf(GLfloat)+32*4);
    // Silhouette will be freed in 'FreeShadowVolumeState'
    Dispose(Context);
    Context:=nil;
  end;
end;

//------------------------------------------------------------------------------

procedure RTSFreeScene(var Scene: PRTSScene);

// free allocated memory for the given scene

var I : Integer;

begin
  if assigned(Scene) then
  with Scene^ do
  begin
    for I:=0 to MAX_CONTEXTS-1 do FreeTessellationContext(Context[I]);
    if assigned(LightList) then FreeMem(LightList,LightListSize*SizeOf(PRTSLight));
    Dispose(Scene);
    Scene:=nil;
  end;
end;

//------------------------------------------------------------------------------

procedure RTSAddObjectToLight(Light: PRTSlight; AObject: PRTSobject);

var I : Integer;

begin
  I:=0; // in case the loop never starts
  for I:=0 to AObject.LightListSize-1 do
  begin
    if AObject.LightList[I] = Light then Exit;
    if AObject.LightList[I] = nil then BreaK;
  end;
  if I = AObject.LightListSize then
  begin
    Inc(AObject.LightListSize);
    ReallocMem(AObject.LightList,AObject.LightListSize*SizeOf(PRTSlight));
    if AObject.LightList = nil then raise Exception.Create('RTSAddObjectToLight: out of memory');
  end;
  AObject.LightList[I]:=Light;

  I:=0; // in case the loop never starts
  for I:=0 to Light.ObjectListSize-1 do
  begin
    if Light.ObjectList[I] = AObject then raise Exception.Create('RTSAddObjectToLight: inconsistent lists');
    if Light.ObjectList[I] = nil then Break;
  end;

  if I = Light.ObjectListSize then
  begin
    // Extend object list.
    Inc(Light.ObjectListSize);
    ReallocMem(Light.ObjectList,Light.ObjectListSize*SizeOf(PRTSObject));
    if (Light.ObjectList = nil) then raise Exception.Create('RTSAddObjectToLight: out of memory');
    // Extend shadow volume list.
    ReallocMem(Light.ShadowVolumeList,Light.ObjectListSize*SizeOf(PShadowVolumeState));
    if (Light.ShadowVolumeList = nil) then raise Exception.Create('RTSAddObjectToLight: out of memory');
  end;

  New(Light.ShadowVolumeList[I]);
  if (Light.ShadowVolumeList[I] = nil) then raise Exception.Create('RTSAddObjectToLight: out of memory');
  InitShadowVolumeState(Light.ShadowVolumeList[I]);
  Light.ObjectList[I]:=AObject;
  Inc(Light.RefCnt);
  Inc(AObject.RefCnt);
end;

//------------------------------------------------------------------------------

procedure RTSSetLightState(Light: PRTSlight; State: TRTSlightState);

begin
  Light.State:=State;
end;

//------------------------------------------------------------------------------

procedure RTSSetObjectState(AObject: PRTSobject; State: TRTSobjectState);

begin
  AObject.State:=State;
end;

//------------------------------------------------------------------------------

procedure RTSUpdateEyePos(Scene: PRTSScene; EyePos: TAffineVector);

begin
  Scene.EyePos:=EyePos;
end;

//------------------------------------------------------------------------------

procedure RTSUpdateUsableStencilBits(Scene: PRTSScene; UsableStencilBits: GLbitfield);

begin
  Scene.UsableStencilBits:=UsableStencilBits;
  Scene.StencilValidateNeeded:=True;
end;

//------------------------------------------------------------------------------

procedure RTSUpdateLightPos(Light: PRTSlight; LightPos: TVector);

begin
  Light.LightPos[X]:=LightPos[X];
  Light.LightPos[Y]:=LightPos[Y];
  Light.LightPos[Z]:=LightPos[Z];
  Inc(Light.SerNum);
end;

//------------------------------------------------------------------------------

procedure RTSUpdateLightRadius(Light: PRTSlight; Radius: GLfloat);

begin
  Light.Radius:=Radius;
  Inc(Light.SerNum);
end;

//------------------------------------------------------------------------------

procedure RTSUpdateObjectPos(AObject: PRTSobject; ObjectPos: TAffineVector);

begin
  AObject.ObjectPos:=ObjectPos;
  Inc(AObject.SerNum);
end;

//------------------------------------------------------------------------------

procedure RTSUpdateObjectShape(AObject: PRTSobject);

begin
  Inc(AObject.SerNum);
end;

//------------------------------------------------------------------------------

procedure RTSUpdateObjectMaxRadius(AObject: PRTSobject; MaxRadius: GLfloat);

begin
  AObject.MaxRadius:=MaxRadius;
  Inc(AObject.SerNum);
end;

//------------------------------------------------------------------------------

procedure RenderSilhouette(SVS: PShadowVolumeState);

var Info : PInteger;

begin
  if not assigned(SVS.Silhouette) then Exit;
  Info:=Pointer(SVS.Silhouette);
  Assert(SizeOf(GLfloat) = SizeOf(GLint));

  glInterleavedArrays(GL_V2F,3*SizeOf(GLfloat),SVS.Silhouette);
  while True do
  begin
    Assert((PAffineIntVector(Info)[2] = $deadbeef) or
           (PAffineIntVector(Info)[2] = $cafecafe));
    // Two fewer vertices get rendered in the RenderSilhouette case (compared
    // to RenderShadowVolumeBase) because a line loop does not need
    // the initial fan center or the final repeated first vertex.
    glDrawArrays(GL_LINE_LOOP,PAffineIntVector(Info)[0]+1,PAffineIntVector(Info)[1]-2);
    if PAffineIntVector(Info)[2] = $cafecafe then Exit;
    Inc(Info,3*(PAffineIntVector(Info)[1]+1));
  end;
end;

//------------------------------------------------------------------------------

procedure RenderShadowVolumeBase(SVS: PShadowVolumeState);

var Info : PInteger;

begin
  if not assigned(SVS.Silhouette) then Exit;
  Info:=Pointer(SVS.Silhouette);
  Assert(SizeOf(GLfloat) = SizeOf(GLint));
  glRotatef(SVS.Theta,SVS.Axis[X],SVS.Axis[Y],SVS.Axis[Z]);
  glRotatef(SVS.Phi,0,1,0);
  while True do
  begin
    Assert((PAffineIntVector(Info)[2] = $deadbeef) or
           (PAffineIntVector(Info)[2] = $cafecafe));
    // Note: assumes that glInterleavedArrays has already been called.
    glDrawArrays(GL_TRIANGLE_FAN,PAffineIntVector(Info)[0],PAffineIntVector(Info)[1]);
    if PAffineIntVector(Info)[2] = $cafecafe then Exit;
    Inc(Info,3*(PAffineIntVector(Info)[1]+1));
  end;
end;

//------------------------------------------------------------------------------

procedure RenderShadowVolume(SVS: PShadowVolumeState; LightPos: TAffineVector);

begin
  glPushMatrix;
  glTranslatef(LightPos[X],LightPos[Y],LightPos[Z]);
  RenderShadowVolumeBase(SVS);
  glPopMatrix;
end;

//------------------------------------------------------------------------------

procedure RenderShadowVolumeTop(SVS: PShadowVolumeState; LightPos: TAffineVector);

begin
  glPushMatrix;
  glTranslatef(LightPos[X],LightPos[Y],LightPos[Z]);
  glScalef(SVS.TopScale, SVS.TopScale, SVS.TopScale);
  RenderShadowVolumeBase(SVS);
  glPopMatrix;
end;

//------------------------------------------------------------------------------

procedure ValidateShadowVolume(Scene: PRTSScene; Light: PRTSlight;
                               AObject: PRTSobject; SVS: PShadowVolumeState);

begin
  // Serial number mismatch indicates light or object has changed since last
  // shadow volume generation. If mismatch, regenerate the shadow volume.
  if (Light.SerNum <> SVS.LightSerNum)    or
     (AObject.SerNum <> SVS.ObjectSerNum) then
  begin
    CaptureLightView(Scene,Light,AObject,SVS,Scene.Context[0]);
    GenerateSilhouette(SVS,Scene.Context[0]);
    SVS.LightSerNum:=Light.SerNum;
    SVS.ObjectSerNum:=AObject.SerNum;
  end;
end;

//------------------------------------------------------------------------------

procedure RTSRenderScene(Scene: PRTSScene; Mode: TRTSMode);

const TotalDarkness : TVector = (0,0,0,0);

var I, Obj, Bit,
    NumStencilBits,
    NumCastingLights,
    NumShadowingObjects,
    ReservedStencilBit  : Integer;
    FirstLight,
    PrevLight,
    ALight              : PRTSlight;
    AObject             : PRTSobject;
    FullStencilMask     : GLbitfield;
    SVS                 : PShadowVolumeState;
    ShadowStencilBits   : GLbitfield;

begin
  // Expect application (caller) to do the glClear (including stencil).
  // Expect application (caller) to enable depth testing.
  if Mode <> RTS_NO_SHADOWS then
  begin
    // Validate shadow volumes, count casting lights, and stash the first light.
    NumCastingLights:=0;
    FirstLight:=nil;
    for I:=0 to Scene.LightListSize-1 do
    begin
      ALight:=Scene.LightList[I];
      if assigned(ALight) then
      begin
        if ALight.State <> RTS_OFF then
        begin
          if ALight.State = RTS_SHINING_AND_CASTING then
          begin
            if NumCastingLights = 0 then
            begin
              // Count number of shadowing objects.
              NumShadowingObjects:=0;
              for Obj:=0 to ALight.ObjectListSize-1 do
                if ALight.ObjectList[Obj].State = RTS_SHADOWING then Inc(NumShadowingObjects);
              if NumShadowingObjects = 0 then Continue; // Not casting on any object; skip it.
              Assert(FirstLight = nil);
              FirstLight:=ALight;
            end;
            Inc(NumCastingLights);
            if (NumCastingLights = 1) or hasBlendSubtract
              then glEnable(ALight.glLight)
              else glDisable(ALight.glLight);

            for Obj:=0 to ALight.ObjectListSize-1 do
            begin
              AObject:=ALight.ObjectList[Obj];
              if (AObject.State = RTS_SHADOWING) then
              begin
                SVS:=ALight.ShadowVolumeList[Obj];
                ValidateShadowVolume(Scene,ALight,AObject,SVS);
              end;
            end;
          end
          else glEnable(ALight.glLight);
        end
        else glDisable(ALight.glLight);
      end;
    end;
  end;
  glEnable(GL_LIGHTING);
  Scene.RenderSceneFunc(GL_NONE,Scene.SceneData,Scene);

  if (Mode = RTS_NO_SHADOWS) or
     (NumCastingLights = 0)  then Exit;

  Assert(assigned(FirstLight));
  Assert(NumShadowingObjects > 0);

  // Determine exactly which stencil bits usable for shadowing.
  if Scene.StencilValidateNeeded then
  begin
    ShadowStencilBits:=Scene.UsableStencilBits and ((1 shl Scene.StencilBits)-1);
    Scene.NumStencilBits:=ListBits(ShadowStencilBits,Scene.BitList);
    if Scene.NumStencilBits = 0 then
      ShowMessage(Format('WARNING: No stencil bits available for shadowing, expect bad results.'#13+
                         '         Frame buffer stencil bits = %d, usable stencil bits = $%x.',
                         [Scene.StencilBits, Scene.UsableStencilBits]));
    Scene.StencilValidateNeeded:=False;
  end;
  NumStencilBits:=Scene.NumStencilBits;

  // The first light is easier than the rest since we need subtractive
  // blending for two or more lights. Do the first light the fast way.

  Bit:=0;
  Assert(not Scene.StencilValidateNeeded);

  glDisable(FirstLight.glLight);
  glEnable(GL_CULL_FACE);
  glEnable(GL_STENCIL_TEST);
  glDepthMask(GL_FALSE);
  Obj:=0;
  while (FirstLight.ObjectList[Obj].State = RTS_NOT_SHADOWING) do Inc(Obj);

  repeat
    Assert(Bit < NumStencilBits);
    Assert(FirstLight.ObjectList[Obj].State = RTS_SHADOWING);
    Assert(Obj < FirstLight.ObjectListSize);

    FullStencilMask:=0;
    repeat
      glInterleavedArrays(GL_V3F,0,FirstLight.ShadowVolumeList[Obj].Silhouette);
      glDisable(GL_LIGHTING);
      glColorMask(GL_FALSE,GL_FALSE,GL_FALSE,GL_FALSE);
      glColor3f(1,0,0);
      glStencilFunc(GL_ALWAYS,0,0);
      glCullFace(GL_FRONT);
      FullStencilMask:=FullStencilMask or (1 shl Scene.BitList[Bit]);
      glStencilMask(1 shl Scene.BitList[Bit]);
      glStencilOp(GL_KEEP,GL_KEEP,GL_INVERT);
      RenderShadowVolume(FirstLight.ShadowVolumeList[Obj],FirstLight.LightPos);
      glCullFace(GL_BACK);
      glStencilOp(GL_KEEP,GL_KEEP,GL_INVERT);
      RenderShadowVolume(FirstLight.ShadowVolumeList[Obj],FirstLight.LightPos);
      glColor3f(0, 1, 0);

      glDisable(GL_CULL_FACE);
      glStencilOp(GL_KEEP,GL_KEEP,GL_ZERO);
      RenderShadowVolumeTop(FirstLight.ShadowVolumeList[Obj],FirstLight.LightPos);
      glEnable(GL_CULL_FACE);

      Inc(Bit);
      // skip unused objects
      repeat
        Inc(Obj);
      until (Obj >= FirstLight.ObjectListSize) or
            (FirstLight.ObjectList[Obj].State <> RTS_NOT_SHADOWING);

    until not ((Bit < NumStencilBits) and (Obj < FirstLight.ObjectListSize));

    glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_TRUE);
    glDepthFunc(GL_EQUAL);
    glStencilOp(GL_KEEP,GL_KEEP,GL_KEEP);
    glStencilFunc(GL_NOTEQUAL,0,FullStencilMask);
    glEnable(GL_LIGHTING);
    Scene.RenderSceneFunc(FirstLight.glLight,Scene.SceneData,Scene);
    if Obj < FirstLight.ObjectListSize then
    begin
      glStencilMask($FFFF);
      glClear(GL_STENCIL_BUFFER_BIT);
      glDepthFunc(GL_LESS);  // XXX needed?
      Bit:=0;
    end;
  until not (Obj < FirstLight.ObjectListSize);

  if NumCastingLights = 1 then
  begin
    glStencilMask($FFFF);
    glCullFace(GL_BACK);  // XXX Needed?
    glDepthMask(GL_TRUE);
    glDepthFunc(GL_LESS);
    glDisable(GL_STENCIL_TEST);
    glDisableClientState(GL_VERTEX_ARRAY);
    Exit;
  end;
  // Get ready to subtract out the particular contribution for each light
  // source in regions shadowed by the light source's shadowing objects.
  glLightModelfv(GL_LIGHT_MODEL_AMBIENT,@TotalDarkness);
  glDepthFunc(GL_LESS);
  {$IFDEF GL_EXT_blend_subtract}
  if hasBlendSubtract then glBlendEquationEXT(GL_FUNC_REVERSE_SUBTRACT_EXT)
                      else {$ENDIF} glBlendFunc(GL_ONE,GL_ONE);
  glEnable(GL_BLEND);

  PrevLight:=FirstLight;

  for I:=1 to Scene.LightListSize-1 do
  begin
    ALight:=Scene.LightList[I];
    if assigned(ALight) then
    begin
      if (ALight.State = RTS_SHINING_AND_CASTING) then
      begin
        // Count number of shadowing objects.
        NumShadowingObjects:=0;
        for Obj:=0 to ALight.ObjectListSize-1 do
          if ALight.ObjectList[Obj].State = RTS_SHADOWING then Inc(NumShadowingObjects);

        if NumShadowingObjects > 0 then
        begin
          Assert(not Scene.StencilValidateNeeded);

          // Switch off the last light; switch on the current light (all
          // other lights should be disabled).
          glDisable(PrevLight.glLight);
          glEnable(ALight.glLight);

          // Complicated logic to try to figure out the stencil clear
          // strategy.  Tries hard to conserve stencil bit planes and Scene
          // re-renders.
          if NumStencilBits < NumShadowingObjects then
          begin
            if NumStencilBits = 1 then
            begin
              ShowMessage('WARNING: 1 bit of stencil not enough to reserve a bit.'#13+
                          '         Skipping lights beyond the first.');
              Continue;
            end;
            // Going to require one or more stencil clears; this requires
            // reserving a bit of stencil to avoid double subtracts.
            ReservedStencilBit:=1 shl Scene.BitList[0];
            Bit:=1;
            glStencilMask($FFFF);
            glClear(GL_STENCIL_BUFFER_BIT);
            glDepthFunc(GL_LESS);  // XXX Needed?
          end
          else
          begin
            // Faster cases.  All the objects can be rendered each to a
            // distinct available stencil plane.  No need to reserve a
            // stencil bit to avoid double blending since only one Scene
            // render required.
            ReservedStencilBit:=0;
            if NumShadowingObjects <= (NumStencilBits-Bit) then
              // Best case:  Enough stencil bits available to not even
              // require a stencil clear for this light.  Keep 'bit' as is. 
            else
            begin
              // Not enough left over bitplanes to subtract out this light
              // with what's currently available, so clear the stencil buffer
              // to get enough.
              glStencilMask($FFFF);
              glClear(GL_STENCIL_BUFFER_BIT);
              Bit:=0;
            end;
          end;

          Obj:=0;
          while (ALight.ObjectList[Obj].State = RTS_NOT_SHADOWING) do Inc(Obj);

          repeat
            Assert(Bit < NumStencilBits);
            Assert(ALight.ObjectList[Obj].State = RTS_SHADOWING);
            Assert(Obj < ALight.ObjectListSize);

            FullStencilMask:=ReservedStencilBit;

            repeat
              glInterleavedArrays(GL_V3F,0,ALight.ShadowVolumeList[Obj].Silhouette);
              glDisable(GL_LIGHTING);
              glColorMask(GL_FALSE,GL_FALSE,GL_FALSE,GL_FALSE);
              glStencilFunc(GL_ALWAYS,0,0);
              glCullFace(GL_FRONT);
              FullStencilMask:=FullStencilMask or (1 shl Scene.BitList[Bit]);
              glStencilMask(1 shl Scene.BitList[Bit]);
              glStencilOp(GL_KEEP,GL_KEEP,GL_INVERT);
              RenderShadowVolume(ALight.ShadowVolumeList[Obj],ALight.LightPos);

              glCullFace(GL_BACK);
              glStencilOp(GL_KEEP,GL_KEEP,GL_INVERT);
              RenderShadowVolume(ALight.ShadowVolumeList[Obj],ALight.LightPos);
              glDisable(GL_CULL_FACE);
              glStencilOp(GL_KEEP,GL_KEEP,GL_ZERO);
              RenderShadowVolumeTop(ALight.ShadowVolumeList[Obj],ALight.LightPos);
              glEnable(GL_CULL_FACE);

              Inc(Bit);
              repeat
                Inc(Obj);
              until not ((Obj < ALight.ObjectListSize) and
                         (ALight.ObjectList[Obj].State = RTS_NOT_SHADOWING));

            until not ((Bit < Scene.NumStencilBits) and
                       (Obj < ALight.objectListSize));

            glColorMask(GL_TRUE,GL_TRUE,GL_TRUE,GL_TRUE);
            glDepthFunc(GL_EQUAL);
            if ReservedStencilBit <> 0 then
            begin
              glStencilMask(ReservedStencilBit);
              glStencilOp(GL_KEEP,GL_KEEP,GL_ONE);
              if hasBlendSubtract
                then
                  // Subtract lighting contribution inside of shadow; prevent
                  // double drawing via stencil
                  glStencilFunc(GL_GREATER,ReservedStencilBit,FullStencilMask)
                else
                  // Add lighting contribution outside of shadow; prevent
                  // double drawing via stencil.
                  glStencilFunc(GL_EQUAL, 0, fullStencilMask);
            end
            else
              if hasBlendSubtract then
              begin
                glStencilOp(GL_KEEP,GL_KEEP,GL_ZERO);
                glStencilFunc(GL_NOTEQUAL,0,FullStencilMask);
              end
              else
              begin
                glStencilOp(GL_KEEP, GL_KEEP, GL_REPLACE);
                glStencilFunc(GL_EQUAL, 0, fullStencilMask);
              end;
            glEnable(GL_LIGHTING);
            Scene.RenderSceneFunc(ALight.glLight,Scene.SceneData,Scene);

            if Obj < ALight.ObjectListSize then
            begin
              Assert(ReservedStencilBit <> 0);
              glStencilMask($FFFF);
              glClear(GL_STENCIL_BUFFER_BIT);
              glDepthFunc(GL_LESS);  // XXX Needed?
              Bit:=1;
            end;
          until not (Obj < ALight.ObjectListSize);

          PrevLight:=ALight;
        end;
      end;
    end;
  end;

  glStencilMask($FFFF);
  glCullFace(GL_BACK);  // XXX needed?
  glDepthMask(GL_TRUE);
  glDepthFunc(GL_LESS);
  glDisable(GL_STENCIL_TEST);
  glDisable(GL_BLEND);
  glDisableClientState(GL_VERTEX_ARRAY);
  glLightModelfv(GL_LIGHT_MODEL_AMBIENT,@Scene.SceneAmbient);
end;

//------------------------------------------------------------------------------

procedure RTSRenderSilhouette(Scene: PRTSScene; Light: PRTSlight; AObject: PRTSobject);

var LightDelta : TAffineVector;
    LightDistance,
    ViewScale,
    FieldOfViewRatio,
    ExtentScale       : GLFLoat;
    SVSRec            : TShadowVolumeState;
    SVS               : PShadowVolumeState;
    Obj               : Integer;
    AnonymousShadowVolumeState : Boolean;

begin
  // Calculate the light's distance from the object being shadowed.
  LightDelta[X]:=AObject.ObjectPos[X]-Light.LightPos[X];
  LightDelta[Y]:=AObject.ObjectPos[Y]-Light.LightPos[Y];
  LightDelta[Z]:=AObject.ObjectPos[Z]-Light.LightPos[Z];
  LightDistance:=Sqrt(LightDelta[X]*LightDelta[X]+
                      LightDelta[Y]*LightDelta[Y]+
                      LightDelta[Z]*LightDelta[Z]);

  ViewScale:=GetViewScale(Scene);
  FieldOfViewRatio:=AObject.MaxRadius/LightDistance;
  ExtentScale:=Light.Radius*FieldOfViewRatio/ViewScale;

  for Obj:=0 to Light.ObjectListSize-1do
  begin
    if Light.ObjectList[Obj] = AObject then
    begin
      SVS:=Light.ShadowVolumeList[Obj];
      AnonymousShadowVolumeState:=False;
      Break;
    end;
  end;

  // It probably makes sense to have the object on the light's object list
  // already since then we would have a ShadowVolumeState structure ready to
  // use and likely to have a reasonably sized silhouette VertexCallback array. Plus,
  // we'd validate the light and object's shadow volume.
  // Anyway, RTSRenderSilhouette will still handle the case where the object
  // is not already added to the specified light for generality (but not
  // economy).  Use an 'anonymous' ShadowVolumeState data structure that only
  // lives during this routine.

  if Obj = Light.ObjectListSize then
  begin
    SVS:=@SVSRec;
    AnonymousShadowVolumeState:=True;
    InitShadowVolumeState(SVS);
  end;

  ValidateShadowVolume(Scene,Light,AObject,SVS);

  glPushAttrib(GL_ENABLE_BIT);
  // Disable a few things likely to screw up the rendering of the silhouette.
  glDisable(GL_LIGHTING);
  glDisable(GL_DEPTH_TEST);
  glDisable(GL_STENCIL_TEST);
  glDisable(GL_ALPHA_TEST);
  glDisable(GL_BLEND);

  glMatrixMode(GL_PROJECTION);
  glPushMatrix;
  glLoadIdentity;
  gluOrtho2D(-ViewScale,ViewScale,-ViewScale,ViewScale);
  glMatrixMode(GL_MODELVIEW);
  glPushMatrix;
  glLoadIdentity;
  glScalef(1/ExtentScale,1/ExtentScale,1/ExtentScale);

  RenderSilhouette(SVS);

  glMatrixMode(GL_MODELVIEW);
  glPopMatrix;
  glMatrixMode(GL_PROJECTION);
  glPopMatrix;
  glPopAttrib;

  if AnonymousShadowVolumeState then
    // Deallocate 'anonymous' ShadowVolumeState's silhouette VertexCallback array.
    FreeMem(SVS.Silhouette);
end;

//------------------------------------------------------------------------------

end.

