unit GLTexture;

// GLTexture   - This unit handles all the color and texture stuff.
// Version     - 0.5.1
// 07-JAN-2000 ml: minor changes in TGLColor
// 04-JAN-2000 ml: minor changes in TGLColor
// 30-DEC-99 ml: bug fixes

interface

uses
  Windows, Classes, OpenGL12, GraphicEx, Graphics, Geometry, SysUtils, JPEG;

type
  PColorVector = ^TColorVector;
  TColorVector = TVector;

const // color definitions

      // Window's colors (must be filled at program
      // startup, since they depend on the desktop scheme)

      {$J+ - allow change of the following typed constants}

      clrScrollBar           : TColorVector = (0,0,0,1);
      clrBackground          : TColorVector = (0,0,0,1);
      clrActiveCaption       : TColorVector = (0,0,0,1);
      clrInactiveCaption     : TColorVector = (0,0,0,1);
      clrMenu                : TColorVector = (0,0,0,1);
      clrWindow              : TColorVector = (0,0,0,1);
      clrWindowFrame         : TColorVector = (0,0,0,1);
      clrMenuText            : TColorVector = (0,0,0,1);
      clrWindowText          : TColorVector = (0,0,0,1);
      clrCaptionText         : TColorVector = (0,0,0,1);
      clrActiveBorder        : TColorVector = (0,0,0,1);
      clrInactiveBorder      : TColorVector = (0,0,0,1);
      clrAppWorkSpace        : TColorVector = (0,0,0,1);
      clrHighlight           : TColorVector = (0,0,0,1);
      clrHighlightText       : TColorVector = (0,0,0,1);
      clrBtnFace             : TColorVector = (0,0,0,1);
      clrBtnShadow           : TColorVector = (0,0,0,1);
      clrGrayText            : TColorVector = (0,0,0,1);
      clrBtnText             : TColorVector = (0,0,0,1);
      clrInactiveCaptionText : TColorVector = (0,0,0,1);
      clrBtnHighlight        : TColorVector = (0,0,0,1);
      clr3DDkShadow          : TColorVector = (0,0,0,1);
      clr3DLight             : TColorVector = (0,0,0,1);
      clrInfoText            : TColorVector = (0,0,0,1);
      clrInfoBk              : TColorVector = (0,0,0,1);
      
      {$J- - disable change of other typed constants}

      // 'static' color definitions
      // sort of grays
      clrBlack               : TColorVector = (0,    0,    0,    1);
      clrGray05              : TColorVector = (0.05, 0.05, 0.05, 1);
      clrGray10              : TColorVector = (0.10, 0.10, 0.10, 1);
      clrGray15              : TColorVector = (0.15, 0.15, 0.15, 1);
      clrGray20              : TColorVector = (0.20, 0.20, 0.20, 1);
      clrGray25              : TColorVector = (0.25, 0.25, 0.25, 1);
      clrGray30              : TColorVector = (0.30, 0.30, 0.30, 1);
      clrGray35              : TColorVector = (0.35, 0.35, 0.35, 1);
      clrGray40              : TColorVector = (0.40, 0.40, 0.40, 1);
      clrGray45              : TColorVector = (0.45, 0.45, 0.45, 1);
      clrGray50              : TColorVector = (0.50, 0.50, 0.50, 1);
      clrGray55              : TColorVector = (0.55, 0.55, 0.55, 1);
      clrGray60              : TColorVector = (0.60, 0.60, 0.60, 1);
      clrGray65              : TColorVector = (0.65, 0.65, 0.65, 1);
      clrGray70              : TColorVector = (0.70, 0.70, 0.70, 1);
      clrGray75              : TColorVector = (0.75, 0.75, 0.75, 1);
      clrGray80              : TColorVector = (0.80, 0.80, 0.80, 1);
      clrGray85              : TColorVector = (0.85, 0.85, 0.85, 1);
      clrGray90              : TColorVector = (0.90, 0.90, 0.90, 1);
      clrGray95              : TColorVector = (0.95, 0.95, 0.95, 1);
      clrWhite               : TColorVector = (1,    1,    1,    1);

      // other grays
      clrDimGray             : TColorVector = (0.329412, 0.329412, 0.329412, 1);
      clrGray                : TColorVector = (0.752941, 0.752941, 0.752941, 1);
      clrLightGray           : TColorVector = (0.658824, 0.658824, 0.658824, 1);

      // colors en masse
      clrAquamarine          : TColorVector = (0.439216, 0.858824, 0.576471, 1);
      clrBlueViolet          : TColorVector = (0.62352,  0.372549, 0.623529, 1);
      clrBrown               : TColorVector = (0.647059, 0.164706, 0.164706, 1);
      clrCadetBlue           : TColorVector = (0.372549, 0.623529, 0.623529, 1);
      clrCoral               : TColorVector = (1,        0.498039, 0.0,      1);
      clrCornflowerBlue      : TColorVector = (0.258824, 0.258824, 0.435294, 1);
      clrDarkGreen           : TColorVector = (0.184314, 0.309804, 0.184314, 1);
      clrDarkOliveGreen      : TColorVector = (0.309804, 0.309804, 0.184314, 1);
      clrDarkOrchid          : TColorVector = (0.6,      0.196078, 0.8,      1);
      clrDarkSlateBlue       : TColorVector = (0.419608, 0.137255, 0.556863, 1);
      clrDarkSlateGray       : TColorVector = (0.184314, 0.309804, 0.309804, 1);
      clrDarkSlateGrey       : TColorVector = (0.184314, 0.309804, 0.309804, 1);
      clrDarkTurquoise       : TColorVector = (0.439216, 0.576471, 0.858824, 1);
      clrFirebrick           : TColorVector = (0.556863, 0.137255, 0.137255, 1);
      clrForestGreen         : TColorVector = (0.137255, 0.556863, 0.137255, 1);
      clrGold                : TColorVector = (0.8,      0.498039, 0.196078, 1);
      clrGoldenrod           : TColorVector = (0.858824, 0.858824, 0.439216, 1);
      clrGreenYellow         : TColorVector = (0.576471, 0.858824, 0.439216, 1);
      clrIndian              : TColorVector = (0.309804, 0.184314, 0.184314, 1);
      clrKhaki               : TColorVector = (0.623529, 0.623529, 0.372549, 1);
      clrLightBlue           : TColorVector = (0.74902,  0.847059, 0.847059, 1);
      clrLightSteelBlue      : TColorVector = (0.560784, 0.560784, 0.737255, 1);
      clrLimeGreen           : TColorVector = (0.196078, 0.8,      0.196078, 1);
      clrMaroon              : TColorVector = (0.556863, 0.137255, 0.419608, 1);
      clrMediumAquamarine    : TColorVector = (0.196078, 0.8,      0.6,      1);
      clrMediumBlue          : TColorVector = (0.196078, 0.196078, 0.8,      1);
      clrMediumForestGreen   : TColorVector = (0.419608, 0.556863, 0.137255, 1);
      clrMediumGoldenrod     : TColorVector = (0.917647, 0.917647, 0.678431, 1);
      clrMediumOrchid        : TColorVector = (0.576471, 0.439216, 0.858824, 1);
      clrMediumSeaGreen      : TColorVector = (0.258824, 0.435294, 0.258824, 1);
      clrMediumSlateBlue     : TColorVector = (0.498039, 0,        1,        1);
      clrMediumSpringGreen   : TColorVector = (0.498039, 1,        0,        1);
      clrMediumTurquoise     : TColorVector = (0.439216, 0.858824, 0.858824, 1);
      clrMediumViolet        : TColorVector = (0.858824, 0.439216, 0.576471, 1);
      clrMidnightBlue        : TColorVector = (0.184314, 0.184314, 0.309804, 1);
      clrNavy                : TColorVector = (0.137255, 0.137255, 0.556863, 1);
      clrNavyBlue            : TColorVector = (0.137255, 0.137255, 0.556863, 1);
      clrOrange              : TColorVector = (1,        0.5,      0.0,      1);
      clrOrangeRed           : TColorVector = (1,        0.25,     0,        1);
      clrOrchid              : TColorVector = (0.858824, 0.439216, 0.858824, 1);
      clrPaleGreen           : TColorVector = (0.560784, 0.737255, 0.560784, 1);
      clrPink                : TColorVector = (0.737255, 0.560784, 0.560784, 1);
      clrPlum                : TColorVector = (0.917647, 0.678431, 0.917647, 1);
      clrSalmon              : TColorVector = (0.435294, 0.258824, 0.258824, 1);
      clrSeaGreen            : TColorVector = (0.137255, 0.556863, 0.419608, 1);
      clrSienna              : TColorVector = (0.556863, 0.419608, 0.137255, 1);
      clrSkyBlue             : TColorVector = (0.196078, 0.6,      0.8,      1);
      clrSlateBlue           : TColorVector = (0,        0.498039, 1,        1);
      clrSpringGreen         : TColorVector = (0,        1,        0.498039, 1);
      clrSteelBlue           : TColorVector = (0.137255, 0.419608, 0.556863, 1);
      clrTan                 : TColorVector = (0.858824, 0.576471, 0.439216, 1);
      clrThistle             : TColorVector = (0.847059, 0.74902,  0.847059, 1);
      clrTurquoise           : TColorVector = (0.678431, 0.917647, 0.917647, 1);
      clrViolet              : TColorVector = (0.309804, 0.184314, 0.309804, 1);
      clrVioletRed           : TColorVector = (0.8,      0.196078, 0.6,      1);
      clrWheat               : TColorVector = (0.847059, 0.847059, 0.74902,  1);
      clrYellowGreen         : TColorVector = (0.6,      0.8,      0.196078, 1);
      clrSummerSky           : TColorVector = (0.22,     0.69,     0.87,     1);
      clrRichBlue            : TColorVector = (0.35,     0.35,     0.67,     1);
      clrBrass               : TColorVector = (0.71,     0.65,     0.26,     1);
      clrCopper              : TColorVector = (0.72,     0.45,     0.20,     1);
      clrBronze              : TColorVector = (0.55,     0.47,     0.14,     1);
      clrBronze2             : TColorVector = (0.65,     0.49,     0.24,     1);
      clrSilver              : TColorVector = (0.90,     0.91,     0.98,     1);
      clrBrightGold          : TColorVector = (0.85,     0.85,     0.10,     1);
      clrOldGold             : TColorVector = (0.81,     0.71,     0.23,     1);
      clrFeldspar            : TColorVector = (0.82,     0.57,     0.46,     1);
      clrQuartz              : TColorVector = (0.85,     0.85,     0.95,     1);
      clrNeonPink            : TColorVector = (1.00,     0.43,     0.78,     1);
      clrDarkPurple          : TColorVector = (0.53,     0.12,     0.47,     1);
      clrNeonBlue            : TColorVector = (0.30,     0.30,     1.00,     1);
      clrCoolCopper          : TColorVector = (0.85,     0.53,     0.10,     1);
      clrMandarinOrange      : TColorVector = (0.89,     0.47,     0.20,     1);
      clrLightWood           : TColorVector = (0.91,     0.76,     0.65,     1);
      clrMediumWood          : TColorVector = (0.65,     0.50,     0.39,     1);
      clrDarkWood            : TColorVector = (0.52,     0.37,     0.26,     1);
      clrSpicyPink           : TColorVector = (1.00,     0.11,     0.68,     1);
      clrSemiSweetChoc       : TColorVector = (0.42,     0.26,     0.15,     1);
      clrBakersChoc          : TColorVector = (0.36,     0.20,     0.09,     1);
      clrFlesh               : TColorVector = (0.96,     0.80,     0.69,     1);
      clrNewTan              : TColorVector = (0.92,     0.78,     0.62,     1);
      clrNewMidnightBlue     : TColorVector = (0.00,     0.00,     0.61,     1);
      clrVeryDarkBrown       : TColorVector = (0.35,     0.16,     0.14,     1);
      clrDarkBrown           : TColorVector = (0.36,     0.25,     0.20,     1);
      clrDarkTan             : TColorVector = (0.59,     0.41,     0.31,     1);
      clrGreenCopper         : TColorVector = (0.32,     0.49,     0.46,     1);
      clrDkGreenCopper       : TColorVector = (0.29,     0.46,     0.43,     1);
      clrDustyRose           : TColorVector = (0.52,     0.39,     0.39,     1);
      clrHuntersGreen        : TColorVector = (0.13,     0.37,     0.31,     1);
      clrScarlet             : TColorVector = (0.55,     0.09,     0.09,     1);
      clrMediumPurple        : TColorVector = (0.73,     0.16,     0.96,     1);
      clrLightPurple         : TColorVector = (0.87,     0.58,     0.98,     1);
      clrVeryLightPurple     : TColorVector = (0.94,     0.81,     0.99,     1);
      clrGreen               : TColorVector = (0,        1,        0,        1);
      clrOlive               : TColorVector = (0,        1,        1,        1);
      clrPurple              : TColorVector = (1,        0,        1,        1);
      clrTeal                : TColorVector = (0,        1,        1,        1);
      clrRed                 : TColorVector = (1,        0,        0,        1);
      clrLime                : TColorVector = (0,        1,        0,        1);
      clrYellow              : TColorVector = (1,        1,        0,        1);
      clrBlue                : TColorVector = (0,        0,        1,        1);
      clrFuchsia             : TColorVector = (1,        0,        1,        1);
      clrAqua                : TColorVector = (0,        1,        1,        1);

type PRGBColor    = ^TRGBColor;
     TRGBColor    = TAffineByteVector;

     PTexPoint    = ^TTexPoint;
     TTexPoint    = record
                      S,T : TGLFLoat;
                    end;
     PTexPointArray = ^TTexPointArray;
     TTexPointArray = array[0..0] of TTexPoint;

     TTextureMode = (tmDecal,tmModulate,tmBlend,tmReplace);
     TTextureWrap = (twBoth,twNone,twVertical,twHorizontal);
     TMinFilter   = (miNearest,miLinear,miNearestMipmapNearest,miLinearMipmapNearest,
                     miNearestMipmapLinear,miLinearMipmapLinear);
     TMagFilter   = (maNearest,maLinear);

     TFaceProperties = class;
     TTexture        = class;
     TMaterial       = class;

     TTexBaseClass = class(TPersistent)
     private
       FInherited : Boolean;
       FOwner     : TPersistent;
       procedure SetInherited(AValue: Boolean);
     public
       constructor Create(AOwner: TPersistent); virtual;
       procedure Update; virtual; abstract;
     published
       property IsInherited: Boolean read FInherited write SetInherited;
     end;

     TGLColor = class(TTexBaseClass)
     private
       FColor : TColorVector;
       procedure SetColor(AColor: TColorVector);
       procedure SetColorComponent(Index: Integer; Value: TGLFloat);
     public
       constructor Create(AOwner: TPersistent); override;
       procedure Assign(Source: TPersistent); override;
       procedure Update; override;
       property Color: TColorVector read FColor write SetColor;
       function AsAddress: PGLFloat;
     published
       property Red:   TGLFloat index 0 read FColor[0] write SetColorComponent;
       property Green: TGLFloat index 1 read FColor[1] write SetColorComponent;
       property Blue:  TGLFloat index 2 read FColor[2] write SetColorComponent;
       property Alpha: TGLFloat index 3 read FColor[3] write SetColorComponent;
     end;

     TTextureChange  = (tcImage,tcParams);
     TTextureChanges = set of TTextureChange;

     TImageSource    = (isPersistent,isPicFile,
                        isRessource,isScanner,isCapture);

     TImagePath      = TFileName;

     TTextureImage = class(TPersistent)
     private
       FOwner  : TTexture;
       FPath   : TImagePath;
       FValid  : Boolean;
       FReloading : Boolean;
       procedure SetPath(AValue: TImagePath);
     protected
       function GetBitmap: HBitmap; virtual; abstract;
       function GetHeight: Integer; virtual; abstract;
       function GetWidth: Integer; virtual; abstract;
       procedure SetHeight(AValue: Integer); virtual; abstract;
       procedure SetWidth(AValue: Integer); virtual; abstract;
       procedure Validate;
     public
       constructor Create(AOwner: TTexture); virtual;
       destructor Destroy; override;
       procedure DataNeeded; virtual;
       procedure Invalidate;
       procedure LoadImage(AName: TFileName); virtual;
       procedure ReleaseData; virtual;
       procedure Update; virtual;
       property Bitmap: HBitmap read GetBitmap;
       property Height: Integer read GetHeight write SetHeight;
       property Owner: TTexture read FOwner;
       property Valid: Boolean read FValid;
       property Width: Integer read GetWidth write SetWidth;
     published
       property Path: TImagePath read FPath write SetPath;
     end;

     TPersistentImage = class(TTextureImage)
     private
       FPicture    : TPicture;
     protected
       function GetBitmap: HBitmap; override;
       function GetHeight: Integer; override;
       function GetWidth: Integer; override;
       procedure PictureChanged(Sender: TObject);
       procedure SetHeight(AValue: Integer); override;
       procedure SetWidth(AValue: Integer); override;
     public
       constructor Create(AOwner: TTexture); override;
       destructor Destroy; override;
       procedure LoadImage(AName: TFileName); override;
     published
       property Picture: TPicture read FPicture write FPicture;
     end;

     TPicFileImage = class(TTextureImage)
     private
       FPicture : TPicture;
     protected
       function GetBitmap: HBitmap; override;
       function GetHeight: Integer; override;
       function GetWidth: Integer; override;
       procedure SetHeight(AValue: Integer); override;
       procedure SetWidth(AValue: Integer); override;
     public
       procedure DataNeeded; override;
       procedure ReleaseData; override;
     published
       property Picture: TPicture read FPicture write FPicture;
     end;

     TCaptureImage = class(TTextureImage)
     private
       FBitmap    : TBitmap;
       FLeft,
       FTop       : Integer;
     protected
       function GetBitmap: HBitmap; override;
       function GetHeight: Integer; override;
       function GetWidth: Integer; override;
       procedure PictureChanged(Sender: TObject);
       procedure SetHeight(AValue: Integer); override;
       procedure SetLeft(AValue: Integer);
       procedure SetTop(AValue: Integer);
       procedure SetWidth(AValue: Integer); override;
     public
       constructor Create(AOwner: TTexture); override;
       destructor Destroy; override;
       procedure DataNeeded; override;
     published
       property Left: Integer read FLeft write SetLeft default 32;
       property Top: Integer read FTop write SetTop default 32;
     end;

     TTexture = class(TTexBaseClass)
     private
       FHandle      : TGLuint;
       FTextureMode : TTextureMode;
       FTextureWrap : TTextureWrap;
       FMinFilter   : TMinFilter;
       FMagFilter   : TMagFilter;
       FChanges     : TTextureChanges;
       FDisabled    : Boolean;
       FImage       : TTextureImage;
       FImageSource : TImageSource;
       procedure SetImage(AValue: TTextureImage);
       procedure SetImageSource(AValue: TImageSource);
       procedure SetMagFilter(AValue: TMagFilter);
       procedure SetMinFilter(AValue: TMinFilter);
       procedure SetTextureMode(AValue: TTextureMode);
       procedure SetTextureWrap(AValue: TTextureWrap);
       procedure SetDisabled(AValue: Boolean);
     protected
       function GetHandle: TGLuint; virtual;
       function IsPowerOf2(Value: Integer): Boolean;
       procedure PrepareImage; virtual;
       procedure PrepareParams; virtual;
     public
       constructor Create(AOwner: TPersistent); override;
       destructor  Destroy; override;
       procedure Apply;
       procedure Assign(Source: TPersistent); override;
       procedure DestroyHandle;
       procedure DisableAutoTexture;
       procedure InitAutoTexture(TexRep: PTexPoint);
       procedure ReloadImage; 
       function RoundDownToPowerOf2(Value: Integer): Integer;
       function RoundUpToPowerOf2(Value: Integer): Integer;
       procedure Update; override;

       property  Handle: TGLuint read GetHandle;
     published
       // The property 'ImageSource' must be the first one in this list,
       // in order to make the streaming system set it first, before any
       // subproperty of it is accessed!
       property ImageSource: TImageSource read FImageSource write SetImageSource default isPersistent;
       property Image: TTextureImage read FImage write SetImage;
       property MagFilter: TMagFilter read FMagFilter write SetMagFilter;
       property MinFilter: TMinFilter read FMinFilter write SetMinFilter;
       property TextureMode: TTexturemode read FTextureMode write SetTextureMode;
       property TextureWrap: TTextureWrap read FTextureWrap write SetTextureWrap;
       property Disabled: Boolean read FDisabled write SetDisabled default True;
     end;

     TShininess = 0..128;

     TFaceProperties = class(TTexBaseClass)
     private
       FAmbient,
       FDiffuse,
       FSpecular,
       FEmission  : TGLColor;
       FShininess : TShininess;
       procedure SetAmbient(AValue: TGLColor);
       procedure SetDiffuse(AValue: TGLColor);
       procedure SetEmission(AValue: TGLColor);
       procedure SetSpecular(AValue: TGLColor);
       procedure SetShininess(AValue: TShininess);
     public
       constructor Create(AOwner: TPersistent); override;
       destructor Destroy; override;
       procedure Apply(AFace: TGLEnum);
       procedure Assign(Source: TPersistent); override;
       procedure Update; override;
     published
       property Ambient: TGLColor read FAmbient write SetAmbient;
       property Diffuse: TGLColor read FDiffuse write SetDiffuse;
       property Emission: TGLColor read FEmission write SetEmission;
       property Shininess: TShininess read FShininess write SetShininess;
       property Specular: TGLColor read FSpecular write SetSpecular;
     end;

     TMaterial = class(TTexBaseClass)
     private
       FOwner           : TPersistent;
       FFrontProperties,
       FBackProperties  : TFaceProperties;
       FTexture         : TTexture;
       procedure SetBackProperties(Values: TFaceProperties);
       procedure SetFrontProperties(Values: TFaceProperties);
       procedure SetTexture(ATexture: TTexture);
     public
       constructor Create(AOwner: TPersistent); override;
       destructor Destroy; override;
       procedure Apply;
       procedure Assign(Source: TPersistent); override;
       procedure Update; override;
     published
       property BackProperties: TFaceProperties read FBackProperties write SetBackProperties;
       property FrontProperties: TFaceProperties read FFrontProperties write SetFrontProperties;
       property Texture: TTexture read FTexture write SetTexture;
     end;

     PColorEntry = ^TColorEntry;
     TColorEntry = record
                     Name  : String[31];
                     Color : TColorVector;
                   end;

     TColorManager = class(TList)
     public
       destructor Destroy; override;
       procedure AddColor(AName: String; AColor: TColorVector);
       procedure EnumColors(Proc: TGetStrProc);
       function  FindColor(AName: String): TColorVector;
       function  GetColor(AName: String): TColorVector;
       function  GetColorName(AColor: TColorVector): String;
       procedure RegisterDefaultColors;
       procedure RemoveColor(AName: String);
     end;

var
  ColorManager: TColorManager;

function ConvertColorVector(AColor: TColorVector): TColor;
function ConvertRGBColor(AColor: array of Byte): TColorVector;
function ConvertWinColor(AColor: TColor): TColorVector;
procedure RegisterColor(AName: String; AColor: TColorVector);
procedure UnregisterColor(AName: String);

//------------------------------------------------------------------------------

implementation

uses Dialogs, GLScene, GLScreen, GLStrings, Math;

//---------------------- TTexBaseClass -----------------------------------------

constructor TTexBaseClass.Create(AOwner: TPersistent);

begin
  inherited Create;
  FOwner:=AOwner;
  FInherited:=True;
end;

//------------------------------------------------------------------------------

procedure TTexBaseClass.SetInherited(AValue: Boolean);

begin
  if FInherited <> AValue then
  begin
    FInherited := AValue;
    if assigned(FOwner) and
       (FOwner is TTexBaseClass) and
       not AValue then TTexBaseClass(FOwner).IsInherited := AValue;
  end;
end;

//---------------------- TGLColor ----------------------------------------------

constructor TGLColor.Create(AOwner: TPersistent);

begin
  inherited Create(AOwner);
  FColor := clrBlack;
end;

//------------------------------------------------------------------------------

procedure TGLColor.SetColor(AColor: TColorVector);

begin
  FColor:=AColor;
  IsInherited:=False;
  Update;
end;

//------------------------------------------------------------------------------

procedure TGLColor.SetColorComponent(Index: Integer; Value: TGLFloat);

begin
  if FColor[Index] <> Value then
  begin
    FColor[Index] := Value;
    IsInherited := False;
  Update;
  end;
end;

//------------------------------------------------------------------------------

procedure TGLColor.Assign(Source: TPersistent);

begin
  if assigned(Source) and (Source is TGLColor) then
  begin
    FColor:=TGLColor(Source).FColor;
    IsInherited:=False;
  Update;
  end;
  inherited Assign(Source);
end;

//------------------------------------------------------------------------------

procedure TGLColor.Update;

begin
  if assigned(FOwner) then
    if FOwner is TTexBaseClass then TTexBaseClass(FOwner).Update
                               else TBaseSceneObject(FOwner).Update
end;

//------------------------------------------------------------------------------

function TGLColor.AsAddress: PGLFloat;

begin
  Result := @FColor;
end;

//----------------- TFaceProperties --------------------------------------------

constructor TFaceProperties.Create(AOwner: TPersistent);

begin
  inherited Create(Aowner);
  FAmbient:=TGLColor.Create(Self);
  FDiffuse:=TGLColor.Create(Self);
  FEmission:=TGLColor.Create(Self);
  FSpecular:=TGLColor.Create(Self);
end;

//------------------------------------------------------------------------------

destructor TFaceProperties.Destroy;

begin
  FAmbient.Free;
  FDiffuse.Free;
  FEmission.Free;
  FSpecular.Free;
  inherited Destroy;
end;

//------------------------------------------------------------------------------

procedure TFaceProperties.Apply(AFace: TGLEnum);

begin
  if not IsInherited then
  begin
    glMateriali(AFace,GL_SHININESS,FShininess);
    with Emission do
      if not IsInherited then glMaterialfv(AFace,GL_EMISSION,AsAddress);
    with Ambient do
      if not IsInherited then glMaterialfv(AFace,GL_AMBIENT,AsAddress);
    with Diffuse do
      if not IsInherited then glMaterialfv(AFace,GL_DIFFUSE,AsAddress);
    with Specular do
      if not IsInherited then glMaterialfv(AFace,GL_SPECULAR,AsAddress);
  end;
end;

//------------------------------------------------------------------------------

procedure TFaceProperties.Assign(Source: TPersistent);

begin
  if assigned(Source) and (Source is TFaceProperties) then
  begin
    FAmbient.FColor:=TFaceProperties(Source).FAmbient.FColor;
    FDiffuse.FColor:=TFaceProperties(Source).FDiffuse.FColor;
    FSpecular.FColor:=TFaceProperties(Source).FSpecular.FColor;
    FShininess:=TFaceProperties(Source).FShininess;
    FEmission.FColor:=TFaceProperties(Source).FEmission.FColor;
    IsInherited:=False;
  end
  else inherited Assign(Source);
end;

//------------------------------------------------------------------------------

procedure TFaceProperties.Update;

begin
  if assigned(FOwner) then TTexBaseClass(FOwner).Update;
end;

//------------------------------------------------------------------------------

procedure TFaceProperties.SetAmbient(AValue: TGLColor);

begin
  FAmbient.FColor:=AValue.FColor;
  IsInherited:=False;
end;

//------------------------------------------------------------------------------

procedure TFaceProperties.SetDiffuse(AValue: TGLColor);

begin
  FDiffuse.FColor:=AValue.FColor;
  IsInherited:=False;
end;

//------------------------------------------------------------------------------

procedure TFaceProperties.SetEmission(AValue: TGLColor);

begin
  FEmission.FColor:=AValue.FColor;
  IsInherited:=False;
end;

//------------------------------------------------------------------------------

procedure TFaceProperties.SetSpecular(AValue: TGLColor);

begin
  FSpecular.FColor:=AValue.FColor;
  IsInherited:=False;
end;

//------------------------------------------------------------------------------

procedure TFaceProperties.SetShininess(AValue: TShininess);

begin
  if FShininess <> AValue then
  begin
    FShininess:=AValue;
    IsInherited:=False;
  end;
end;

//----------------- TTextureImage ----------------------------------------------

constructor TTextureImage.Create(AOwner: TTexture);

begin
  inherited Create;
  FOwner:=AOwner;
  FValid:=False;
end;

//------------------------------------------------------------------------------

destructor TTextureImage.Destroy;

begin
  inherited Destroy;
end;

//------------------------------------------------------------------------------

procedure TTextureImage.DataNeeded;

begin
  FReloading:=False;
end;

//------------------------------------------------------------------------------

procedure TTextureImage.Invalidate;

// Mark the texture image as invalid, so it'll be (re)loaded
// the next time it is used.

begin
  FValid:=False;
  Include(Owner.FChanges,tcImage);
end;

//------------------------------------------------------------------------------

procedure TTextureImage.LoadImage(AName: TFileName);
// assign a new image path
begin
  Invalidate;
  FOwner.FImage.FPath:=AName;
  if Length(AName) > 0 then
  begin
    FOwner.IsInherited:=False;
    Update;
  end;
end;

//------------------------------------------------------------------------------

procedure TTextureImage.ReleaseData;

begin
end;

//------------------------------------------------------------------------------

procedure TTextureImage.SetPath(AValue: TImagePath);

// load new image as texture

begin
  if CompareText(AValue,FPath) <> 0 then
    LoadImage(AValue);
end;

//------------------------------------------------------------------------------

procedure TTextureImage.Update;
begin
  Include(Owner.FChanges,tcImage);
  Owner.Update;
end;

//------------------------------------------------------------------------------

procedure TTextureImage.Validate;

// Mark the texture image as valid, showing it can be used.

begin
  FValid:=True;
end;

//----------------- TPersistentImage -------------------------------------------

constructor TPersistentImage.Create(AOwner: TTexture);

begin
  inherited Create(AOwner);
  FPicture:=TPicture.Create;
  FPicture.OnChange:=PictureChanged;
end;

//------------------------------------------------------------------------------

destructor TPersistentImage.Destroy;

begin
  FPicture.Free;
  inherited Destroy;
end;

//------------------------------------------------------------------------------

function TPersistentImage.GetHeight: Integer;

begin
  Result:=FPicture.Height;
end;

//------------------------------------------------------------------------------

function TPersistentImage.GetWidth: Integer;

begin
  Result:=FPicture.Width;
end;

//------------------------------------------------------------------------------

function TPersistentImage.GetBitmap: HBitmap;

begin
  Result:=FPicture.Bitmap.Handle;
end;

//------------------------------------------------------------------------------

procedure TPersistentImage.LoadImage(AName: TFileName);
// load new image
begin
  FOwner.FImage.FPath:=AName;
  Picture.LoadFromFile(AName);
end;

//------------------------------------------------------------------------------

procedure TPersistentImage.PictureChanged(Sender: TObject);
begin
  FOwner.IsInherited:=False;
  Update;
end;

//------------------------------------------------------------------------------

procedure TPersistentImage.SetHeight(AValue: Integer);

begin
  FPicture.Bitmap.Height := AValue;
end;

//------------------------------------------------------------------------------

procedure TPersistentImage.SetWidth(AValue: Integer);

begin
  FPicture.Bitmap.Width := AValue;
end;

//----------------- TPicFileImage ----------------------------------------------

function TPicFileImage.GetHeight: Integer;

begin
  Result:=FPicture.Height;
end;

//------------------------------------------------------------------------------

function TPicFileImage.GetWidth: Integer;

begin
  Result:=FPicture.Width;
end;

//------------------------------------------------------------------------------

function TPicFileImage.GetBitmap: HBitmap;
begin
  if not Valid then
    raise Exception.Create(glsImageInvalid);
  Result:=FPicture.Bitmap.Handle;
end;

//------------------------------------------------------------------------------

procedure TPicFileImage.DataNeeded;

begin
  if (not Valid) and (Length(FPath) > 0) then
  begin
    FPicture:=TPicture.Create;
    try
      FPicture.LoadFromFile(FPath);
      Validate;
    except
      FPicture.Free;
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure TPicFileImage.ReleaseData;

begin
  if Valid then
  begin
    FPicture.Free;
    FPicture:=nil;
    Invalidate;
  end;
end;

//------------------------------------------------------------------------------

procedure TPicFileImage.SetHeight(AValue: Integer);

begin
  FPicture.Bitmap.Height := AValue;
end;

//------------------------------------------------------------------------------

procedure TPicFileImage.SetWidth(AValue: Integer);

begin
  FPicture.Bitmap.Width := AValue;
end;

//----------------- TCaptureImage ----------------------------------------------

constructor TCaptureImage.Create(AOwner: TTexture);

begin
  inherited Create(AOwner);
  FBitmap:=TBitmap.Create;
  FBitmap.Width:=32;
  FBitmap.Height:=32;
  //FBitmap.OnChange:=PictureChanged;
end;

//------------------------------------------------------------------------------

destructor TCaptureImage.Destroy;

begin
  FBitmap.Free;
  inherited Destroy;
end;

//------------------------------------------------------------------------------

function TCaptureImage.GetHeight: Integer;

begin
  Result:=FBitmap.Height;
end;

//------------------------------------------------------------------------------

function TCaptureImage.GetWidth: Integer;

begin
  Result:=FBitmap.Width;
end;

//------------------------------------------------------------------------------

function TCaptureImage.GetBitmap: HBitmap;
begin
  if not Valid then
    raise Exception.Create(glsImageInvalid);
  Result:=FBitmap.Handle;
end;

//------------------------------------------------------------------------------

procedure TCaptureImage.DataNeeded;

var Rect : TRectangle;

begin
  if not Valid then
  begin
    Rect.Left:=FLeft;
    Rect.Top:=FTop;
    Rect.Width:=FBitmap.Width;
    Rect.Height:=FBitmap.Height;
    ReadScreenImage(FBitmap.Canvas.Handle,0,0,Rect);
    Validate;
  end;
end;

//------------------------------------------------------------------------------

procedure TCaptureImage.PictureChanged(Sender: TObject);
begin
  if Valid then Update;
end;

//------------------------------------------------------------------------------

procedure TCaptureImage.SetHeight(AValue: Integer);

begin
  if FBitmap.Height <> AValue then
  begin
    FBitmap.Height:=AValue;
    Invalidate;
  end;
end;

//------------------------------------------------------------------------------

procedure TCaptureImage.SetLeft(AValue: Integer);

begin
  if FLeft <> AValue then
  begin
    FLeft:=AValue;
    Invalidate;
  end;
end;

//------------------------------------------------------------------------------

procedure TCaptureImage.SetTop(AValue: Integer);

begin
  if FTop <> AValue then
  begin
    FTop:=AValue;
    Invalidate;
  end;
end;

//------------------------------------------------------------------------------

procedure TCaptureImage.SetWidth(AValue: Integer);

begin
  if FBitmap.Width <> AValue then
  begin
    FBitmap.Width:=AValue;
    Invalidate;
  end;
end;

//----------------- TTexture ---------------------------------------------------

constructor TTexture.Create(AOwner: TPersistent);

begin
  inherited Create(AOwner);
  FDisabled:=True;
  FChanges:=[tcImage,tcParams];
  FImageSource:=isPersistent;
  FImage:=TPersistentImage.Create(Self);
end;

//------------------------------------------------------------------------------

destructor TTexture.Destroy;

begin
  DestroyHandle;
  FImage.Free;
  inherited Destroy;
end;

//------------------------------------------------------------------------------

procedure TTexture.SetImage(AValue: TTextureImage);

begin
  FImage.Assign(AValue);
  IsInherited:=False;
end;

//------------------------------------------------------------------------------

procedure TTexture.SetImageSource(AValue: TImageSource);

var NewImage : TTextureImage;

begin
  if FImageSource <> AValue then
  begin
    case AValue of
      isPersistent : NewImage:=TPersistentImage.Create(Self);
      isPicFile    : NewImage:=TPicFileImage.Create(Self);
      isCapture    : NewImage:=TCaptureImage.Create(Self);
      isRessource,
      isScanner    : raise Exception.Create('not yet supported');
    else
      NewImage:=nil;
    end;
    FImage.Free;
    FImage:=NewImage;
    FImageSource:=AValue;
    FImage.FPath:='';
  end;
end;

//------------------------------------------------------------------------------

procedure TTexture.SetMagFilter(AValue: TMagFilter);

begin
  if AValue <> FMagFilter then
  begin
    FMagFilter:=AValue;
    //Include(FChanges,tcParams);
    ReloadImage;
    IsInherited:=False;
  end;
end;

//------------------------------------------------------------------------------

procedure TTexture.SetMinFilter(AValue: TMinFilter);

begin
  if AValue <> FMinFilter then
  begin
    FMinFilter:=AValue;
    //Include(FChanges,tcParams);
    ReloadImage;
    IsInherited:=False;
  end;
end;

//------------------------------------------------------------------------------

procedure TTexture.SetTextureMode(AValue: TTextureMode);

begin
  if AValue <> FTextureMode then
  begin
    FTextureMode:=AValue;
    Include(FChanges,tcParams);
    IsInherited:=False;
  end;
end;

//------------------------------------------------------------------------------

procedure TTexture.SetDisabled(AValue: Boolean);

begin
  if AValue <> FDisabled then
  begin
    FDisabled:=AValue;
    Update;
  end;
end;

//------------------------------------------------------------------------------

procedure TTexture.SetTextureWrap(AValue: TTextureWrap);

begin
  if AValue <> FTextureWrap then
  begin
    FTextureWrap:=AValue;
    Include(FChanges,tcParams);
    IsInherited:=False;
  end;
end;

//------------------------------------------------------------------------------

procedure TTexture.Apply;

begin
  if not IsInherited then
  begin
    glBindTexture(GL_TEXTURE_2D,Handle);
  end;
end;

//------------------------------------------------------------------------------

function TTexture.GetHandle: TGLuint;

begin
  if (FHandle = 0) or (FChanges <> []) then
  begin
    if FHandle = 0 then
    begin
      glGenTextures(1,@FHandle);
      if FHandle = 0 then ShowError(glsNoNewTexture);
    end;
    glBindtexture(GL_TEXTURE_2D, FHandle);
    if tcImage in FChanges then
      PrepareImage;
    if tcParams in FChanges then
      PrepareParams;
    FChanges:=[];
  end;
  Result:=FHandle;
end;

//------------------------------------------------------------------------------

procedure TTexture.PrepareImage;

// load texture to OpenGL subsystem

type PPixelArray  = ^TByteArray;

var Data        : PPixelArray;
    BMInfo      : TBitmapInfo;
    I,ImageSize : Integer;
    Temp        : Byte;
    MemDC       : HDC;

    Buffer: PPixelArray;
begin
  with BMinfo.bmiHeader do
  begin
    // make image data available
    Image.DataNeeded;
    // create description of the required image format
    FillChar(BMInfo,SizeOf(BMInfo),0);
    biSize:=sizeof(TBitmapInfoHeader);
    biBitCount:=24;
    biWidth := Image.Width;
    biHeight := -Image.Height;
    ImageSize:=Abs(biWidth*biHeight);
    biPlanes:=1;
    biCompression:=BI_RGB;
    MemDC:=CreateCompatibleDC(0);
    Getmem(Data,ImageSize*3);
    Buffer := nil;  // shut up compiler
    try
      // get the actual bits of the image
      GetDIBits(MemDC,Image.Bitmap,0,Abs(biHeight),Data,BMInfo,DIB_RGB_COLORS);
      biHeight := Abs(biHeight);

      GetMem(Buffer, RoundUpToPowerOf2(biHeight) * RoundUpToPowerOf2(biWidth) * 3);

      gluScaleImage(GL_RGB, biWidth, biHeight, GL_UNSIGNED_BYTE, Data,
        RoundUpToPowerOf2(biWidth), RoundUpToPowerOf2(biHeight),
        GL_UNSIGNED_BYTE, Buffer);

      if GL_EXT_bgra then
        if (FMinFilter = miNearest) or (FMinFilter = miLinear) then
          glTexImage2d(GL_TEXTURE_2D,0,3,RoundUpToPowerOf2(biWidth),
            RoundUpToPowerOf2(biHeight),0,GL_BGR_EXT,GL_UNSIGNED_BYTE,Buffer)
        else
          gluBuild2DMipmaps(GL_TEXTURE_2D,3,
            RoundUpToPowerOf2(biWidth),RoundUpToPowerOf2(biHeight),
            GL_BGR_EXT, GL_UNSIGNED_BYTE,Buffer)
      else
      begin
        {$IFOPT R+} {$DEFINE RangeCheck} {$R-} {$ENDIF}
        for I:=0 TO ImageSize-1 do //swap blue with red to go from bgr to rgb
        begin
          Temp:=Buffer[I*3];
          Buffer[I*3]:=Buffer[I*3+2];
          Buffer[I*3+2]:=Temp;
        end;
        {$IFDEF RangeCheck} {$UNDEF RangeCheck} {$R+} {$ENDIF}
        if (FMinFilter = miNearest) OR (FMinFilter = miLinear) then
          glTexImage2d(GL_TEXTURE_2D,0,3,RoundUpToPowerOf2(biWidth),
            RoundUpToPowerOf2(biHeight),0,GL_RGB,GL_UNSIGNED_BYTE,Buffer)
        else
          gluBuild2DMipmaps(GL_TEXTURE_2D,3,RoundUpToPowerOf2(biWidth),
            RoundUpToPowerOf2(biHeight),GL_RGB, GL_UNSIGNED_BYTE,Buffer);
        end;


      (*
      // Now set the bits depending on the features supported by OpenGL.
      if GL_EXT_bgra then
        // BGR extension avoids color component swapping
        if (FMinFilter = miNearest) OR (FMinFilter = miLinear) then
          glTexImage2d(GL_TEXTURE_2D,0,3,biWidth,biHeight,0,GL_BGR_EXT,GL_UNSIGNED_BYTE,Data)
        else
          gluBuild2DMipmaps(GL_TEXTURE_2D,3,biWidth,biHeight,GL_BGR_EXT, GL_UNSIGNED_BYTE,Data)
      else
      begin
        // No BGR support, so we must swap the color components by hand.
        // switch off range check for color swapping, make sure we restore the original state
        {$IFOPT R+} {$DEFINE RangeCheck} {$R-} {$ENDIF}
        for I:=0 TO ImageSize-1 do //swap blue with red to go from bgr to rgb
        begin
          Temp:=Data[I*3];
          Data[I*3]:=Data[I*3+2];
          Data[I*3+2]:=Temp;
        end;
        // restore range check, if previously activated
        {$IFDEF RangeCheck} {$UNDEF RangeCheck} {$R+} {$ENDIF}
        if (FMinFilter = miNearest) OR (FMinFilter = miLinear) then
          glTexImage2d(GL_TEXTURE_2D,0,3,biWidth,biHeight,0,GL_RGB,GL_UNSIGNED_BYTE,Data)
        else
          gluBuild2DMipmaps(GL_TEXTURE_2D,3,biWidth,biHeight,GL_RGB, GL_UNSIGNED_BYTE,Data);
        end;
      *)
    finally
      FreeMem(Data);
      FreeMem(Buffer);
      DeleteDC(MemDC);
      Image.ReleaseData;
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure TTexture.PrepareParams;

begin
  glHint(GL_PERSPECTIVE_CORRECTION_HINT,GL_NICEST);
  glPixelStorei(GL_UNPACK_ALIGNMENT,4);
  glPixelStorei(GL_UNPACK_ROW_LENGTH,0);
  glPixelStorei(GL_UNPACK_SKIP_ROWS,0);
  glPixelStorei(GL_UNPACK_SKIP_PIXELS,0);

  case FTextureWrap of
    twBoth       : begin
                     glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_REPEAT);
                     glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_REPEAT);
                   end;
    twNone       : begin
                     glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_CLAMP);
                     glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_CLAMP);
                   end;
    twHorizontal : begin
                     glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_REPEAT);
                     glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_CLAMP);
                   end;
    twVertical   : begin
                     glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_CLAMP);
                     glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_REPEAT);
                   end;
  end;

  case FMinFilter of
    miNearest              : glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_NEAREST);
    miLinear               : glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR);
    miNearestMipmapNearest : glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_NEAREST_MIPMAP_NEAREST);
    miLinearMipmapNearest  : glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR_MIPMAP_NEAREST);
    miNearestMipmapLinear  : glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_NEAREST_MIPMAP_LINEAR);
    miLinearMipmapLinear   : glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR_MIPMAP_LINEAR);
  end;

  case FMagFilter of
    maNearest : glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_NEAREST);
    maLinear  : glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
  end;

  case FTextureMode of
    tmDecal    : glTexEnvi(GL_TEXTURE_ENV,GL_TEXTURE_ENV_MODE,GL_DECAL);
    tmModulate : glTexEnvi(GL_TEXTURE_ENV,GL_TEXTURE_ENV_MODE,GL_MODULATE);
    tmBlend    : glTexEnvi(GL_TEXTURE_ENV,GL_TEXTURE_ENV_MODE,GL_BLEND);
    tmReplace  : glTexEnvi(GL_TEXTURE_ENV,GL_TEXTURE_ENV_MODE,GL_REPLACE);
  end;
end;

//------------------------------------------------------------------------------

procedure TTexture.Assign(Source: TPersistent);

begin
  if assigned(Source) and (Source is TTexture) then
  begin
    FImage.Assign(TTexture(Source).FImage);
    FTextureMode:=TTexture(Source).FTextureMode;
    FTextureWrap:=TTexture(Source).FTextureWrap;
    FMinFilter:=TTexture(Source).FMinFilter;
    FMagFilter:=TTexture(Source).FMagFilter;
    FDisabled:=TTexture(Source).FDisabled;
    FChanges:=[tcParams,tcImage];
    IsInherited:=False;
  end
  else inherited Assign(Source);
end;

//------------------------------------------------------------------------------

procedure TTexture.ReloadImage;

begin
  FChanges:=[tcImage,tcParams];
end;

//------------------------------------------------------------------------------

function TTexture.RoundDownToPowerOf2(Value: Integer): Integer;
var
  LogTwo : Extended;
begin
  LogTwo:=log2(Value);
  if Trunc(LogTwo) < LogTwo then
    Result:=Trunc(Power(2,Trunc(LogTwo)))
  else
    Result:=Value;
end;

//------------------------------------------------------------------------------

function TTexture.RoundUpToPowerOf2(Value: Integer): Integer;
var
  LogTwo : Extended;
begin
  LogTwo:=log2(Value);
  if Trunc(LogTwo) < LogTwo then
    Result:=Trunc(Power(2,Trunc(LogTwo)+1))
  else
    Result:=value;
end;

//------------------------------------------------------------------------------

procedure TTexture.DestroyHandle;

begin
  if FHandle <> 0 then
  begin
    glDeleteTextures(1,@FHandle);
    FHandle:=0;
    FChanges:=[tcParams,tcImage];
  end;
end;

//------------------------------------------------------------------------------

procedure TTexture.DisableAutoTexture;

begin
  glDisable(GL_TEXTURE_GEN_S);
  glDisable(GL_TEXTURE_GEN_T);
end;

//------------------------------------------------------------------------------

procedure TTexture.InitAutoTexture(TexRep: PTexPoint);

const EmptyVector : TVector = (0,0,0,0);

var SGenParams,
    TGenParams  : TVector;

begin
  SGenParams:=MakeVector([1,0,0,0]);
  TGenParams:=MakeVector([0,1,0,0]);

  glTexGeni(GL_S,GL_TEXTURE_GEN_MODE,GL_OBJECT_LINEAR);
  if assigned(TexRep) then SGenparams[0]:=TexRep.S;
  glTexGenfv(GL_S,GL_OBJECT_PLANE,@SGenParams);
  glTexGeni(GL_T, GL_TEXTURE_GEN_MODE, GL_OBJECT_LINEAR);
  if assigned(TexRep) then TGenparams[1]:=TexRep.T;
  glTexGenfv(GL_T,GL_OBJECT_PLANE,@TGenparams);

  glEnable(GL_TEXTURE_GEN_S);
  glEnable(GL_TEXTURE_GEN_T);
end;

//------------------------------------------------------------------------------

function TTexture.IsPowerOf2(Value: Integer): Boolean;

begin
  Result:=Trunc(log2(Value)) = log2(Value);
end;

//------------------------------------------------------------------------------

procedure TTexture.Update;
begin
  if assigned(FOwner) then TTexBaseClass(FOwner).Update;
end;

//----------------- TMaterial --------------------------------------------------

constructor TMaterial.Create(AOwner: TPersistent);

begin
  inherited Create(nil);
  FOwner:=AOwner;
  FBackProperties:=TFaceProperties.Create(Self);
  FFrontProperties:=TFaceProperties.Create(Self);
  FTexture:=TTexture.Create(Self);
end;

//------------------------------------------------------------------------------

destructor TMaterial.Destroy;

begin
  FBackProperties.Free;
  FFrontProperties.Free;
  FTexture.Free;
  inherited Destroy;
end;

//------------------------------------------------------------------------------

procedure TMaterial.SetBackProperties(Values: TFaceProperties);

begin
  FBackProperties.Assign(Values);
end;

//------------------------------------------------------------------------------

procedure TMaterial.SetFrontProperties(Values: TFaceProperties);

begin
  FFrontProperties.Assign(Values);
end;

//------------------------------------------------------------------------------

procedure TMaterial.SetTexture(ATexture: TTexture);

begin
  FTexture.Assign(ATexture);
end;

//------------------------------------------------------------------------------

procedure TMaterial.Apply;

begin
  if not IsInherited then
  begin
    FFrontProperties.Apply(GL_FRONT);
    FBackProperties.Apply(GL_BACK);
    if not FTexture.Disabled then
    begin
      TBaseSceneObject(FOwner).Scene.CurrentViewer.RequestedState([stTexture2D]);
      FTexture.Apply;
    end
    else TBaseSceneObject(FOwner).Scene.CurrentViewer.UnnecessaryState([stTexture2D]);
  end;
end;

//------------------------------------------------------------------------------

procedure TMaterial.Assign(Source: TPersistent);

begin
  if assigned(Source) and (Source is TMaterial) then
  begin
    FBackProperties.Assign(TMaterial(Source).FBackProperties);
    FFrontProperties.Assign(TMaterial(Source).FFrontProperties);
    FTexture.Assign(TMaterial(Source).FTexture);
    IsInherited:=False;
  end;
  inherited Assign(Source);
end;

//------------------------------------------------------------------------------

procedure TMaterial.Update;
begin
  if assigned(FOwner) then
    TSceneObject(FOwner).StructureChanged;
end;

//----------------- color manager ----------------------------------------------

function TColorManager.FindColor(AName: String): TColorVector;

var I : Integer;

begin
  Result:=clrBlack;
  for I:=0 to Count-1 do
    if CompareText(TColorEntry(Items[I]^).Name,AName) = 0 then
    begin
      Result:=TColorEntry(Items[I]^).Color;
      Break;
    end;
end;

//------------------------------------------------------------------------------

function TColorManager.GetColor(AName: String): TColorVector;

// convert a clrxxxx color name or a '<red green blue alpha>'
// description into a color vector

var WorkCopy  : String;
    Delimiter : Integer;

begin
  WorkCopy:=Trim(AName);
  if AName[1] in ['(','[','<'] then WorkCopy:=Copy(WorkCopy,2,Length(AName)-2);
  if CompareText(Copy(WorkCopy,1,3),'clr') = 0 then Result:=FindColor(WorkCopy)
                                               else
  try
    // initialize result
    Result:=clrBlack;
    WorkCopy:=Trim(WorkCopy);
    Delimiter:=Pos(' ',WorkCopy);
    if (Length(WorkCopy) > 0) and (Delimiter > 0) then
    begin
      Result[0]:=StrToFloat(Copy(WorkCopy,1,Delimiter-1));
      System.Delete(WorkCopy,1,Delimiter);
      WorkCopy:=TrimLeft(WorkCopy);
      Delimiter:=Pos(' ',WorkCopy);
      if (Length(WorkCopy) > 0) and (Delimiter > 0) then
      begin
        Result[1]:=StrToFloat(Copy(WorkCopy,1,Delimiter-1));
        System.Delete(WorkCopy,1,Delimiter);
        WorkCopy:=TrimLeft(WorkCopy);
        Delimiter:=Pos(' ',WorkCopy);
        if (Length(WorkCopy) > 0) and (Delimiter > 0) then
        begin
          Result[2]:=StrToFloat(Copy(WorkCopy,1,Delimiter-1));
          System.Delete(WorkCopy,1,Delimiter);
          WorkCopy:=TrimLeft(WorkCopy);
          Result[3]:=StrToFloat(WorkCopy);
        end
        else Result[2]:=StrToFloat(WorkCopy);
      end
      else Result[1]:=StrToFloat(WorkCopy);
    end
    else Result[0]:=StrToFloat(WorkCopy);
  except
    ShowMessage('Wrong vector format. Use: ''<red green blue alpha>''!');
    Abort;
  end;
end;

//------------------------------------------------------------------------------

function TColorManager.GetColorName(AColor: TColorVector): String;

const MinDiff = 1e-6;

var I : Integer;

begin
  for I:=0 to Count-1 do
    with TColorEntry(Items[I]^) do
      if (Abs(Color[0]-AColor[0]) < MinDiff) and
         (Abs(Color[1]-AColor[1]) < MinDiff) and
         (Abs(Color[2]-AColor[2]) < MinDiff) and
         (Abs(Color[3]-AColor[3]) < MinDiff) then Break;
  if I < Count then Result:=TColorEntry(Items[I]^).Name
               else
      Result:=Format('<%.3f %.3f %.3f %.3f>',[AColor[0],AColor[1],AColor[2],AColor[3]]);
end;

//------------------------------------------------------------------------------

destructor TColorManager.Destroy;

var I : Integer;

begin
  for I:=0 to Count-1 do FreeMem(Items[I],SizeOf(TColorEntry));
  inherited Destroy;
end;

//------------------------------------------------------------------------------

procedure TColorManager.AddColor(AName: String; AColor: TColorVector);

var NewEntry : PColorEntry;

begin
  New(NewEntry);
  if NewEntry = nil then raise Exception.Create('Could not allocate memory for color registration!');
  with NewEntry^ do
  begin
    Name:=AName;
    Color:=AColor;
  end;
  Add(NewEntry);
end;

//------------------------------------------------------------------------------

procedure TColorManager.EnumColors(Proc: TGetStrProc);

var I : Integer;

begin
  for I:=0 to Count-1 do Proc(TColorEntry(Items[I]^).Name);
end;

//------------------------------------------------------------------------------

procedure TColorManager.RegisterDefaultColors;

begin
  Capacity:=150;
  AddColor('clrBlack',clrBlack);
  AddColor('clrGray05',clrGray05);
  AddColor('clrGray10',clrGray10);
  AddColor('clrGray15',clrGray15);
  AddColor('clrGray20',clrGray20);
  AddColor('clrGray25',clrGray25);
  AddColor('clrGray30',clrGray30);
  AddColor('clrGray35',clrGray35);
  AddColor('clrGray40',clrGray40);
  AddColor('clrGray45',clrGray45);
  AddColor('clrGray50',clrGray50);
  AddColor('clrGray55',clrGray55);
  AddColor('clrGray60',clrGray60);
  AddColor('clrGray65',clrGray65);
  AddColor('clrGray70',clrGray70);
  AddColor('clrGray75',clrGray75);
  AddColor('clrGray80',clrGray80);
  AddColor('clrGray85',clrGray85);
  AddColor('clrGray90',clrGray90);
  AddColor('clrGray95',clrGray95);
  AddColor('clrWhite',clrWhite);
  AddColor('clrDimGray',clrDimGray);
  AddColor('clrGray',clrGray);
  AddColor('clrLightGray',clrLightGray);
  AddColor('clrAquamarine',clrAquamarine);
  AddColor('clrBakersChoc',clrBakersChoc);
  AddColor('clrBlueViolet',clrBlueViolet);
  AddColor('clrBrass',clrBrass);
  AddColor('clrBrightGold',clrBrightGold);
  AddColor('clrBronze',clrBronze);
  AddColor('clrBronze2',clrBronze2);
  AddColor('clrBrown',clrBrown);
  AddColor('clrCadetBlue',clrCadetBlue);
  AddColor('clrCoolCopper',clrCoolCopper);
  AddColor('clrCopper',clrCopper);
  AddColor('clrCoral',clrCoral);
  AddColor('clrCornflowerBlue',clrCornflowerBlue);
  AddColor('clrDarkBrown',clrDarkBrown);
  AddColor('clrDarkGreen',clrDarkGreen);
  AddColor('clrDarkOliveGreen',clrDarkOliveGreen);
  AddColor('clrDarkOrchid',clrDarkOrchid);
  AddColor('clrDarkPurple',clrDarkPurple);
  AddColor('clrDarkSlateBlue',clrDarkSlateBlue);
  AddColor('clrDarkSlateGray',clrDarkSlateGray);
  AddColor('clrDarkSlateGrey',clrDarkSlateGrey);
  AddColor('clrDarkTan',clrDarkTan);
  AddColor('clrDarkTurquoise',clrDarkTurquoise);
  AddColor('clrDarkWood',clrDarkWood);
  AddColor('clrDkGreenCopper',clrDkGreenCopper);
  AddColor('clrDustyRose',clrDustyRose);
  AddColor('clrFeldspar',clrFeldspar);
  AddColor('clrFirebrick',clrFirebrick);
  AddColor('clrFlesh',clrFlesh);
  AddColor('clrForestGreen',clrForestGreen);
  AddColor('clrGold',clrGold);
  AddColor('clrGoldenrod',clrGoldenrod);
  AddColor('clrGreenCopper',clrGreenCopper);
  AddColor('clrGreenYellow',clrGreenYellow);
  AddColor('clrHuntersGreen',clrHuntersGreen);
  AddColor('clrIndian',clrIndian);
  AddColor('clrKhaki',clrKhaki);
  AddColor('clrLightBlue',clrLightBlue);
  AddColor('clrLightPurple',clrLightPurple);
  AddColor('clrLightSteelBlue',clrLightSteelBlue);
  AddColor('clrLightWood',clrLightWood);
  AddColor('clrLimeGreen',clrLimeGreen);
  AddColor('clrMandarinOrange',clrMandarinOrange);
  AddColor('clrMaroon',clrMaroon);
  AddColor('clrMediumAquamarine',clrMediumAquamarine);
  AddColor('clrMediumBlue',clrMediumBlue);
  AddColor('clrMediumForestGreen',clrMediumForestGreen);
  AddColor('clrMediumGoldenrod',clrMediumGoldenrod);
  AddColor('clrMediumOrchid',clrMediumOrchid);
  AddColor('clrMediumPurple',clrMediumPurple);
  AddColor('clrMediumSeaGreen',clrMediumSeaGreen);
  AddColor('clrMediumSlateBlue',clrMediumSlateBlue);
  AddColor('clrMediumSpringGreen',clrMediumSpringGreen);
  AddColor('clrMediumTurquoise',clrMediumTurquoise);
  AddColor('clrMediumViolet',clrMediumViolet);
  AddColor('clrMediumWood',clrMediumWood);
  AddColor('clrMidnightBlue',clrMidnightBlue);
  AddColor('clrNavy',clrNavy);
  AddColor('clrNavyBlue',clrNavyBlue);
  AddColor('clrNeonBlue',clrNeonBlue);
  AddColor('clrNeonPink',clrNeonPink);
  AddColor('clrNewMidnightBlue',clrNewMidnightBlue);
  AddColor('clrNewTan',clrNewTan);
  AddColor('clrOldGold',clrOldGold);
  AddColor('clrOrange',clrOrange);
  AddColor('clrOrangeRed',clrOrangeRed);
  AddColor('clrOrchid',clrOrchid);
  AddColor('clrPaleGreen',clrPaleGreen);
  AddColor('clrPink',clrPink);
  AddColor('clrPlum',clrPlum);
  AddColor('clrQuartz',clrQuartz);
  AddColor('clrRichBlue',clrRichBlue);
  AddColor('clrSalmon',clrSalmon);
  AddColor('clrScarlet',clrScarlet);
  AddColor('clrSeaGreen',clrSeaGreen);
  AddColor('clrSemiSweetChoc',clrSemiSweetChoc);
  AddColor('clrSienna',clrSienna);
  AddColor('clrSilver',clrSilver);
  AddColor('clrSkyBlue',clrSkyBlue);
  AddColor('clrSlateBlue',clrSlateBlue);
  AddColor('clrSpicyPink',clrSpicyPink);
  AddColor('clrSpringGreen',clrSpringGreen);
  AddColor('clrSteelBlue',clrSteelBlue);
  AddColor('clrSummerSky',clrSummerSky);
  AddColor('clrTan',clrTan);
  AddColor('clrThistle',clrThistle);
  AddColor('clrTurquoise',clrTurquoise);
  AddColor('clrViolet',clrViolet);
  AddColor('clrVioletRed',clrVioletRed);
  AddColor('clrVeryDarkBrown',clrVeryDarkBrown);
  AddColor('clrVeryLightPurple',clrVeryLightPurple);
  AddColor('clrWheat',clrWheat);
  AddColor('clrYellowGreen',clrYellowGreen);
  AddColor('clrGreen',clrGreen);
  AddColor('clrOlive',clrOlive);
  AddColor('clrPurple',clrPurple);
  AddColor('clrTeal',clrTeal);
  AddColor('clrRed',clrRed);
  AddColor('clrLime',clrLime);
  AddColor('clrYellow',clrYellow);
  AddColor('clrBlue',clrBlue);
  AddColor('clrFuchsia',clrFuchsia);
  AddColor('clrAqua',clrAqua);

  AddColor('clrScrollBar',clrScrollBar);
  AddColor('clrBackground',clrBackground);
  AddColor('clrActiveCaption',clrActiveCaption);
  AddColor('clrInactiveCaption',clrInactiveCaption);
  AddColor('clrMenu',clrMenu);
  AddColor('clrWindow',clrWindow);
  AddColor('clrWindowFrame',clrWindowFrame);
  AddColor('clrMenuText',clrMenuText);
  AddColor('clrWindowText',clrWindowText);
  AddColor('clrCaptionText',clrCaptionText);
  AddColor('clrActiveBorder',clrActiveBorder);
  AddColor('clrInactiveBorder',clrInactiveBorder);
  AddColor('clrAppWorkSpace',clrAppWorkSpace);
  AddColor('clrHighlight',clrHighlight);
  AddColor('clrHighlightText',clrHighlightText);
  AddColor('clrBtnFace',clrBtnFace);
  AddColor('clrBtnShadow',clrBtnShadow);
  AddColor('clrGrayText',clrGrayText);
  AddColor('clrBtnText',clrBtnText);
  AddColor('clrInactiveCaptionText',clrInactiveCaptionText);
  AddColor('clrBtnHighlight',clrBtnHighlight);
  AddColor('clr3DDkShadow',clr3DDkShadow);
  AddColor('clr3DLight',clr3DLight);
  AddColor('clrInfoText',clrInfoText);
  AddColor('clrInfoBk',clrInfoBk);
end;

//------------------------------------------------------------------------------

procedure TColorManager.RemoveColor(AName: String);
var
  I : Integer;
begin
  for I:=0 to Count-1 do
    if CompareText(TColorEntry(Items[I]^).Name,AName) = 0 then
    begin
      Delete(I);
      Break;
    end;
end;

//------------------------------------------------------------------------------

function ConvertWinColor(AColor: TColor): TColorVector;

// converts a delphi color into its RGB fragments and correct range

var WinColor : Longint;

begin
  // Delphi color to Windows color
  WinColor:=ColorToRGB(AColor);
  // convert 0..255 range into 0..1 range
  Result:=ConvertRGBColor([WinColor and $FF,
                           (WinColor shr 8) and $FF,
                           (WinColor shr 16) and $FF]);
end;

//------------------------------------------------------------------------------

function ConvertColorVector(AColor: TColorVector): TColor;

// converts a color vector (containing float values)

begin
  Result := RGB(Round(255 * AColor[0]), Round(255 * AColor[1]), Round(255 * AColor[2]));
end;

//------------------------------------------------------------------------------

function ConvertRGBColor(AColor: array of Byte): TColorVector;

// converts RGB components into a color vector with correct range

begin
  // convert 0..255 range into 0..1 range
  Result[0] := AColor[0] / 255;
  if High(AColor) > 0 then Result[1] := AColor[1] / 255
                      else Result[1] := 0;
  if High(AColor) > 1 then Result[2] := AColor[2] / 255
                      else Result[2] := 0;
  if High(AColor) > 2 then Result[3] := AColor[3] / 255
                      else Result[3] := 1;
end;

//------------------------------------------------------------------------------

procedure InitWinColors;

begin
  clrScrollBar:=ConvertWinColor(clScrollBar);
  clrBackground:=ConvertWinColor(clBackground);
  clrActiveCaption:=ConvertWinColor(clActiveCaption);
  clrInactiveCaption:=ConvertWinColor(clInactiveCaption);
  clrMenu:=ConvertWinColor(clMenu);
  clrWindow:=ConvertWinColor(clWindow);
  clrWindowFrame:=ConvertWinColor(clWindowFrame);
  clrMenuText:=ConvertWinColor(clMenuText);
  clrWindowText:=ConvertWinColor(clWindowText);
  clrCaptionText:=ConvertWinColor(clCaptionText);
  clrActiveBorder:=ConvertWinColor(clActiveBorder);
  clrInactiveBorder:=ConvertWinColor(clInactiveBorder);
  clrAppWorkSpace:=ConvertWinColor(clAppWorkSpace);
  clrHighlight:=ConvertWinColor(clHighlight);
  clrHighlightText:=ConvertWinColor(clHighlightText);
  clrBtnFace:=ConvertWinColor(clBtnFace);
  clrBtnShadow:=ConvertWinColor(clBtnShadow);
  clrGrayText:=ConvertWinColor(clGrayText);
  clrBtnText:=ConvertWinColor(clBtnText);
  clrInactiveCaptionText:=ConvertWinColor(clInactiveCaptionText);
  clrBtnHighlight:=ConvertWinColor(clBtnHighlight);
  clr3DDkShadow:=ConvertWinColor(cl3DDkShadow);
  clr3DLight:=ConvertWinColor(cl3DLight);
  clrInfoText:=ConvertWinColor(clInfoText);
  clrInfoBk:=ConvertWinColor(clInfoBk);
end;

//------------------------------------------------------------------------------

procedure RegisterColor(AName: String; AColor: TColorVector);

begin
  ColorManager.AddColor(AName,AColor);
end;

//------------------------------------------------------------------------------

procedure UnregisterColor(AName: String);

begin
  ColorManager.RemoveColor(AName);
end;

//------------------------------------------------------------------------------

initialization
  InitWinColors;
  ColorManager:=TColorManager.Create;
  ColorManager.RegisterDefaultColors;
finalization
  ColorManager.Free;
end.
