unit GLSceneRegister;

// Registration unit for GLScene library
// 30-DEC-99 ml: scene editor added, structural changes

interface

{$I DFS.inc}

procedure Register;

//----------------------------------------------------------------------------------------------------------------------

implementation

uses
  Windows, Classes, DsgnIntf, GLScene, GLScreen, GLObjects, GLTexture, PlugInManager,
  SysUtils, Dialogs, ExtDlgs, Forms, GLSceneEdit, Graphics;

type
  TGLSceneViewerEditor = class(TComponentEditor)
  public
    procedure ExecuteVerb(Index: Integer); override;
    function GetVerb(Index: Integer): String; override;
    function GetVerbCount: Integer; override;
  end;

  TGLSceneEditor = class(TComponentEditor)
  public
    constructor Create(AComponent: TComponent; ADesigner: IFormDesigner); override;

    procedure Edit; override;
  end;

  TPlugInProperty = class(TPropertyEditor)
  public
    procedure Edit; override;
    function GetAttributes: TPropertyAttributes; override;
    function GetValue: String; override;
  end;

  TResolutionProperty = class(TPropertyEditor)
    function GetAttributes: TPropertyAttributes; override;
    function GetValue : String; override;
    procedure GetValues(Proc: TGetStrProc); override;
    procedure SetValue(const Value: String); override;
  end;

  TTextureProperty = class(TClassProperty)
  protected
    function GetAttributes: TPropertyAttributes; override;
  end;

  TTextureImageProperty = class(TClassProperty)
  protected
    function GetAttributes: TPropertyAttributes; override;
    procedure Edit; override;
  end;

  TGLColorProperty = class(TClassProperty)
  private
  protected
    function GetAttributes: TPropertyAttributes; override;
    procedure GetValues(Proc: TGetStrProc); override;
  public
    {$ifdef DFS_COMPILER_5_UP}
    procedure ListDrawValue(const Value: string; ACanvas: TCanvas; const ARect: TRect; ASelected: Boolean); override;
    procedure PropDrawValue(ACanvas: TCanvas; const ARect: TRect; ASelected: Boolean); override;
    {$endif}
    function GetValue: String; override;
    procedure SetValue(const Value: string); override;
  end;

  TVectorFileProperty = class(TClassProperty)
  protected
    function GetAttributes: TPropertyAttributes; override;
    function GetValue: String; override;
    procedure Edit; override;
    procedure SetValue(const Value: string); override;
  end;


//----------------------------------------------------------------------------------------------------------------------

procedure TGLSceneViewerEditor.ExecuteVerb(Index: Integer);

var
  Source: TSceneViewer;

begin
  Source := Component as TSceneViewer;
  case Index of
    0:
      Source.ShowInfo;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGLSceneViewerEditor.GetVerb(Index: Integer): string;

begin
  case Index of
    0:
      Result := 'Show context info';
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGLSceneViewerEditor.GetVerbCount: Integer;

begin
  Result := 1;
end;

//----------------- TGLSceneEditor -------------------------------------------------------------------------------------

constructor TGLSceneEditor.Create(AComponent: TComponent; ADesigner: IFormDesigner);

// The editor is created automatically on selection of a scene viewer so we prepare it
// by creating the edit form (remains hidden until Edit is called).

begin
  inherited;
  if GLSceneEditorForm = nil then GLSceneEditorForm := TGLSceneEditorForm.Create(Application);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGLSceneEditor.Edit;

begin
  if GLSceneEditorForm = nil then GLSceneEditorForm := TGLSceneEditorForm.Create(Application);
  if Assigned(GLSceneEditorForm) then
  begin
    GLSceneEditorForm.SetScene(Component as TGLScene, Designer);
    GLSceneEditorForm.Show;
  end;
end;

//----------------- TPlugInProperty ------------------------------------------------------------------------------------

procedure TPlugInProperty.Edit;

var
  Manager: TPlugInManager;

begin
  Manager := TPlugInList(GetOrdValue).Owner;
  Manager.EditPlugInList;
end;

//----------------------------------------------------------------------------------------------------------------------

function TPlugInProperty.GetAttributes: TPropertyAttributes;

begin
  Result:=[paDialog];
end;


//----------------------------------------------------------------------------------------------------------------------

function TPlugInProperty.GetValue: String;

begin
  Result := 'registered : ' + IntToStr(TStringList(GetOrdValue).Count);
end;

//----------------- TResolutionProperty --------------------------------------------------------------------------------

function TResolutionProperty.GetAttributes: TPropertyAttributes;

begin
  Result:=[paValueList];
end;

//----------------------------------------------------------------------------------------------------------------------

function TResolutionProperty.GetValue : String;

begin
  Result:=VideoModes[GetOrdValue].Description;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TResolutionProperty.GetValues(Proc: TGetStrProc);

var
  I: Integer;

begin
  for I:=0 to NumberVideoModes-1 do Proc(VideoModes[I].Description);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TResolutionProperty.SetValue(const Value: String);

const Nums = ['0'..'9'];

var XRes,YRes,BPP : Integer;
    Pos, SLength  : Integer;
    TempStr       : String;

begin
  if CompareText(Value,'default') <> 0 then
  begin
    // initialize scanning
    TempStr:=Trim(Value)+'|'; // ensure at least one delimiter
    SLength:=Length(TempStr);
    XRes:=0; YRes:=0; BPP:=0;
    // contains the string something?
    if SLength > 1 then
    begin
      // determine first number
      for Pos:=1 to SLength do
        if not (TempStr[Pos] in Nums) then Break;
      if Pos <= SLength then
      begin
        // found a number?
        XRes:=StrToInt(Copy(TempStr,1,Pos-1));
        // search for following non-numerics
        for Pos:=Pos to SLength do
          if TempStr[Pos] in Nums then Break;
        Delete(TempStr,1,Pos-1); // take it out of the String
        SLength:=Length(TempStr); // rest length of String
        if SLength > 1 then // something to scan?
        begin
          // determine second number
          for Pos:=1 to SLength do
            if not (TempStr[Pos] in Nums) then Break;
          if Pos <= SLength then
          begin
            YRes:=StrToInt(Copy(TempStr,1,Pos-1));
            // search for following non-numerics
            for Pos:=Pos to SLength do
              if TempStr[Pos] in Nums then Break;
            Delete(TempStr,1,Pos-1); // take it out of the String
            SLength:=Length(TempStr); // rest length of String
            if SLength > 1 then
            begin
              for Pos:=1 to SLength do
                if not (TempStr[Pos] in Nums) then Break;
              if Pos <= SLength then BPP:=StrToInt(Copy(TempStr,1,Pos-1));
            end;
          end;
        end;
      end;
    end;
    SetOrdValue(GetIndexFromResolution(XRes,YRes,BPP));
  end
  else SetOrdValue(0);
end;
                           
//----------------- TTextureProperty -----------------------------------------------------------------------------------

function TTextureProperty.GetAttributes: TPropertyAttributes;

begin
  Result:=[paSubProperties];
end;

//----------------------------------------------------------------------------------------------------------------------

function TTextureImageProperty.GetAttributes: TPropertyAttributes;

begin
  Result:=[paDialog];
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TTextureImageProperty.Edit;
var
  Owner : TTexture;
  OPD   : TOpenPictureDialog;
begin
  Owner:=TTextureImage(GetOrdValue).Owner;
  case Owner.ImageSource of
    isPersistent,
    isPicFile     : begin
                      OPD:=TOpenPictureDialog.Create(nil);
                      try
                        if Length(Owner.Image.Path) > 0 then
                          //OPD.InitialDir:=ExtractFilePath(Owner.Image.Path);
                          OPD.FileName :=Owner.Image.Path;
                        if OPD.Execute then
                        begin
                          Owner.Image.Invalidate;
                          Owner.Image.Path:=OPD.FileName;
                          if Owner.ImageSource = isPersistent then
                            TPersistentImage(Owner.Image).Picture.LoadFromFile(OPD.FileName);
                        end;
                      finally
                        OPD.Free;
                      end;
                    end;
  end;
  Owner.Image.Update;
  Designer.Modified;
end;

//----------------- TGLColorproperty -----------------------------------------------------------------------------------

function TGLColorProperty.GetAttributes: TPropertyAttributes;

begin
  Result := [paSubProperties, paValueList, paMultiSelect];
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGLColorProperty.GetValues(Proc: TGetStrProc);

begin
  ColorManager.EnumColors(Proc);
end;

//----------------------------------------------------------------------------------------------------------------------

function TGLColorProperty.GetValue: String;

begin
  Result := ColorManager.GetColorName(TGLColor(GetOrdValue).Color);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGLColorProperty.SetValue(const Value: string);

begin
  TGLColor(GetOrdValue).Color := ColorManager.GetColor(Value);
  Modified;
end;

//----------------------------------------------------------------------------------------------------------------------

{$ifdef DFS_COMPILER_5_UP}
// Owner draw color values, only available in D5 and higher

procedure TGLColorProperty.ListDrawValue(const Value: string; ACanvas: TCanvas; const ARect: TRect; ASelected: Boolean);

// draws for each color in the list of colors a small rectangle

  //--------------- local function --------------------------------------------

  function ColorToBorderColor(AColor: TColorVector): TColor;

  begin
    if (AColor[0] > 0.75) or
       (AColor[1] > 0.75) or
       (AColor[2] > 0.75) then Result := clBlack
                          else
      if ASelected then Result := clWhite
                   else Result := ConvertColorVector(AColor);
  end;

  //---------------------------------------------------------------------------
  
var
  vRight: Integer;
  vOldPenColor,
  vOldBrushColor: TColor;
  Color: TColorVector;

begin
  vRight := (ARect.Bottom - ARect.Top) + ARect.Left;
  with ACanvas do
  try
    // save off things
    vOldPenColor := Pen.Color;
    vOldBrushColor := Brush.Color;

    // frame things
    Pen.Color := Brush.Color;
    Rectangle(ARect.Left, ARect.Top, vRight, ARect.Bottom);

    // set things up and do the work
    Color := ColorManager.GetColor(Value);
    Brush.Color := ConvertColorVector(Color);
    Pen.Color := ColorToBorderColor(Color);

    Rectangle(ARect.Left + 1, ARect.Top + 1, vRight - 1, ARect.Bottom - 1);

    // restore the things we twiddled with
    Brush.Color := vOldBrushColor;
    Pen.Color := vOldPenColor;
  finally
    inherited ListDrawValue(Value, ACanvas, Rect(vRight, ARect.Top, ARect.Right, ARect.Bottom), ASelected);
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGLColorProperty.PropDrawValue(ACanvas: TCanvas; const ARect: TRect; ASelected: Boolean);

// draws the small color rectangle in the object inspector

begin
  if GetVisualValue <> '' then ListDrawValue(GetVisualValue, ACanvas, ARect, True)
                          else inherited PropDrawValue(ACanvas, ARect, ASelected);
end;
{$endif}

//----------------- TVectorFileProperty --------------------------------------------------------------------------------

function TVectorFileProperty.GetAttributes: TPropertyAttributes;

begin
  Result := [paDialog];
end;

//----------------------------------------------------------------------------------------------------------------------

function TVectorFileProperty.GetValue: String;

begin
  Result := GetStrValue;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TVectorFileProperty.Edit;

var ODialog   : TOpenDialog;
    Component : TFreeForm;
    Desc, F    : String;

begin
  Component := GetComponent(0) as TFreeForm;
  ODialog := TOpenDialog.Create(nil);
  try
    GetVectorFileFormats.BuildFilterStrings(TVectorFile, Desc, F);
    ODialog.Filter := Desc;
    if ODialog.Execute then
    begin
      Component.LoadFromFile(ODialog.FileName);
      Modified;
    end;
  finally
    ODialog.Free;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TVectorFileProperty.SetValue(const Value: string);

begin
  SetStrValue(Value);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure Register;

begin
  RegisterComponents('OpenGL', [TGLScene, TSceneViewer, TPlugInManager]);
  RegisterComponentEditor(TSceneViewer, TGLSceneViewerEditor);
  RegisterComponentEditor(TGLScene, TGLSceneEditor);
  RegisterPropertyEditor(TypeInfo(TPlugInList), TPlugInManager, 'PlugIns', TPlugInProperty);
  RegisterPropertyEditor(TypeInfo(TResolution), nil, '', TResolutionProperty);
  RegisterPropertyEditor(TypeInfo(TGLColor), nil, '', TGLColorProperty);
  RegisterPropertyEditor(TypeInfo(TTexture), TMaterial, '', TTextureProperty);
  RegisterPropertyEditor(TypeInfo(TTextureImage), TTexture, '', TTextureImageProperty);
  RegisterPropertyEditor(TypeInfo(TFileName),  TFreeForm, 'FileName', TVectorFileProperty);
  // this registration is solely to make those classes working with the object inspector
  RegisterNoIcon([TLightsource, TCamera, TSphere, TCube, TCylinder, TCone, TTorus,
                  TSpaceText, TMesh, TFreeForm, TTeapot, TDodecahedron, TDisk, TPlane,
                  TActor]);
end;

//----------------------------------------------------------------------------------------------------------------------

end.
