unit GLSceneEdit;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ComCtrls, GLScene, Menus, DsgnIntf;

type
  TGLSceneEditorForm = class(TForm)
    Tree: TTreeView;
    PopupMenu1: TPopupMenu;
    AddCameraMenuItem: TMenuItem;
    AddObjectMenuItem: TMenuItem;
    N1: TMenuItem;
    DelObjectMenuItem: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure AddCameraMenuItemClick(Sender: TObject);
    procedure DelObjectMenuItemClick(Sender: TObject);
    procedure TreeEditing(Sender: TObject; Node: TTreeNode; var AllowEdit: Boolean);
    procedure PopupMenu1Popup(Sender: TObject);
    procedure TreeDragOver(Sender, Source: TObject; X, Y: Integer; State: TDragState; var Accept: Boolean);
    procedure TreeDragDrop(Sender, Source: TObject; X, Y: Integer);
    procedure TreeChange(Sender: TObject; Node: TTreeNode);
    procedure TreeEdited(Sender: TObject; Node: TTreeNode; var S: String);
    procedure TreeMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure TreeEnter(Sender: TObject);
  private
    FScene: TGLScene;
    FObjectNode,
    FCameraNode: TTreeNode;
    FCurrentDesigner: IFormDesigner;
    procedure ReadScene;
    procedure ResetTree;
    function AddNodes(ANode: TTreeNode; AObject: TBaseSceneObject): TTreeNode;
    procedure AddObjectClick(Sender: TObject);
  protected
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
  public
    procedure SetScene(Scene: TGLScene; Designer: IFormDesigner);
  end;

var
  GLSceneEditorForm: TGLSceneEditorForm;

//----------------------------------------------------------------------------------------------------------------------

implementation

uses
  GLObjects, GLStrings;
  
{$R *.DFM}

//----------------- TGLSceneEditorForm ---------------------------------------------------------------------------------

procedure TGLSceneEditorForm.SetScene(Scene: TGLScene; Designer: IFormDesigner);

begin
  FScene := Scene;
  FCurrentDesigner := Designer;
  ResetTree;
  if Assigned(FScene) then
  begin
    ReadScene;
    Caption := 'GLScene editor: ' + FScene.Name;
  end
  else
  begin
    Caption := 'GLScene editor';
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGLSceneEditorForm.FormCreate(Sender: TObject);

var
  CurrentNode: TTreeNode;

begin
  PopupMenu1.Images := ObjectManager.ObjectIcons;
  Tree.Images := ObjectManager.ObjectIcons;
  Tree.Indent := ObjectManager.ObjectIcons.Width;
  with Tree.Items do
  begin
    // first add the scene root
    CurrentNode := Add(nil, glsSceneRoot);
    with CurrentNode do
    begin
      ImageIndex := ObjectManager.SceneRootIndex;
      SelectedIndex := ObjectManager.SceneRootIndex;
    end;
    // next the root for all cameras
    FCameraNode := AddChild(CurrentNode, glsCameraRoot);
    with FCameraNode do
    begin
      ImageIndex := ObjectManager.CameraRootIndex;
      SelectedIndex := ObjectManager.CameraRootIndex;
    end;

    // and the root for all objects
    FObjectNode := AddChild(CurrentNode, glsObjectRoot);
    with FObjectNode do
    begin
      ImageIndex:=ObjectManager.ObjectRootIndex;
      SelectedIndex:=ObjectManager.ObjectRootIndex;
    end;
  end
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGLSceneEditorForm.ReadScene;

var
  I: Integer;

begin
  Tree.Items.BeginUpdate;
  with FScene do
  begin
    if assigned(Cameras) then
    begin
      FCameraNode.Data := Cameras;
      for I := 0 to Cameras.Count - 1 do AddNodes(FCameraNode, Cameras[I]);
    end;

    if assigned(Objects) then
    begin
      FObjectNode.Data := Objects;
      with Objects do
        for I := 0 to Count - 1 do AddNodes(FObjectNode, Children[I]);
    end;
  end;
  Tree.Items.EndUpdate;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGLSceneEditorForm.ResetTree;

// delete all subtrees (empty tree)

begin
  Tree.Items.BeginUpdate;
  FCameraNode.DeleteChildren;
  FCameraNode.Data:=nil;
  FObjectNode.DeleteChildren;
  FObjectNode.Data:=nil;
  Tree.Items.EndUpdate;
end;

//----------------------------------------------------------------------------------------------------------------------

function TGLSceneEditorForm.AddNodes(ANode: TTreeNode; AObject: TBaseSceneObject): TTreeNode;

// adds the given scene object as well as its children to the tree structure and returns
// the last add node (e.g. for selection)

var
  I: Integer;
  CurrentNode: TTreeNode;

begin
  Result := Tree.Items.AddChildObject(ANode, AObject.Name, AObject);
  Result.ImageIndex := ObjectManager.GetImageIndex(TSceneObjectClass(AObject.ClassType));
  Result.SelectedIndex := Result.ImageIndex;
  CurrentNode := Result;
  for I := 0 to AObject.Count - 1 do Result := AddNodes(CurrentNode, AObject[I]);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGLSceneEditorForm.FormClose(Sender: TObject; var Action: TCloseAction);

begin
  Action := caFree;
  GLSceneEditorForm := nil;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGLSceneEditorForm.AddCameraMenuItemClick(Sender: TObject);

var
  AObject: TBaseSceneObject;
  Node: TTreeNode;
  
begin
  AObject := FScene.Cameras.AddNewChild(TCamera);
  AObject.Name := FCurrentDesigner.UniqueName(AObject.Classname);
  Node := AddNodes(FCameraNode, AObject);
  Node.Selected := True;
  FCurrentDesigner.Modified;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGLSceneEditorForm.DelObjectMenuItemClick(Sender: TObject);

var
  AObject: TBaseSceneObject;
  Allowed,
  KeepChildren: Boolean;
  ConfirmMsg: String;
  Buttons: TMsgDlgButtons;

begin
  if assigned(Tree.Selected) and (Tree.Selected.Level > 1) then
  begin
    AObject := TBaseSceneObject(Tree.Selected.Data);
    // ask for confirming
    if AObject.Name <> '' then ConfirmMsg := 'Delete ' + AObject.Name
                          else ConfirmMsg := 'Delete the marked object';
    Buttons := [mbOK, mbCancel];
    // are there children to care for?
    if AObject.Count > 0 then
    begin
      ConfirmMsg := ConfirmMsg + ' and all its children?';
      Buttons := [mbAll] + Buttons;
    end
    else ConfirmMsg := ConfirmMsg + '?';

    case MessageDlg(ConfirmMsg, mtConfirmation, Buttons, 0) of
      mrAll:
        begin
          KeepChildren := False;
          Allowed := True;
        end;
      mrOK:
        begin
          KeepChildren := True;
          Allowed := True;
        end;
      mrCancel:
        begin
          Allowed := False;
          KeepChildren := True;
        end;
    else
      Allowed := False;
      KeepChildren := True;
    end;

    // deletion allowed?
    if allowed then
    begin
      AObject.Parent.Remove(AObject, KeepChildren);
      AObject.Free;
      Tree.Selected.Free;
    end
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGLSceneEditorForm.TreeEditing(Sender: TObject; Node: TTreeNode; var AllowEdit: Boolean);

begin
  AllowEdit := Node.Level > 1; 
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGLSceneEditorForm.PopupMenu1Popup(Sender: TObject);

var
  ObjectList: TStringList;
  I: Integer;
  Item: TMenuItem;

begin
  ObjectList := TStringList.Create;
  with Tree do
  try
    if Selected = FCameraNode then AddCameraMenuItem.Enabled := True
                              else AddCameraMenuItem.Enabled := False;
    if (Selected = FObjectNode) or
       Selected.HasAsParent(FObjectNode) then AddObjectMenuItem.Enabled := True
                                         else AddObjectMenuItem.Enabled := False;
    if Selected.Level > 1 then DelObjectMenuItem.Enabled := True
                          else DelObjectMenuItem.Enabled := False;
                          
    // before popup create a new list of all registered scene objects
    if AddObjectMenuItem.Enabled then
    begin
      ObjectManager.GetRegisteredSceneObjects(ObjectList);
      while AddObjectMenuItem.Count > 0 do AddObjectMenuItem.Delete(0);
      for I := 0 to ObjectList.Count - 1 do
      begin
        Item := NewItem(ObjectList[I], 0, False, True, AddObjectClick, 0, '');
        Item.ImageIndex := ObjectManager.GetImageIndex(ObjectManager.GetClassFromIndex(I));
        AddObjectMenuItem.Add(Item);
      end;
    end;
  finally
    ObjectList.Free;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGLSceneEditorForm.AddObjectClick(Sender: TObject);

var
  AParent,
  AObject: TBaseSceneObject;
  Node: TTreeNode;

begin
  with Tree do
    if assigned(Selected) and (Selected.Level > 0) then
    begin
      AParent := TBaseSceneObject(Selected.Data);
      AObject := AParent.AddNewChild(ObjectManager.GetClassFromIndex(TMenuItem(Sender).MenuIndex));
      AObject.Name := FCurrentDesigner.UniqueName(AObject.Classname);
      Node := AddNodes(Selected, AObject);
      Node.Selected := True;
      FCurrentDesigner.Modified;
    end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGLSceneEditorForm.TreeDragOver(Sender, Source: TObject; X, Y: Integer; State: TDragState;
  var Accept: Boolean);

var
  Target: TTreeNode;
  
begin
  Accept := False;
  if Source = Tree then
  begin
    with Source as TTreeView do
    begin
      Target := GetNodeAt(X, Y);
      Accept := Assigned(Target) and
                (Selected <> Target) and
                Assigned(DropTarget.Data) and
                not Target.HasAsParent(Selected);
    end;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGLSceneEditorForm.TreeDragDrop(Sender, Source: TObject; X, Y: Integer);

var
  SourceNode,
  DestinationNode: TTreeNode;
  SourceObject,
  DestinationObject: TBaseSceneObject;

begin
  DestinationNode := Tree.DropTarget;
  if assigned(DestinationNode) and
     (Source = Tree) then
  begin
    SourceNode := TTreeView(Source).Selected;
    SourceObject := SourceNode.Data;
    DestinationObject := DestinationNode.Data;
    DestinationObject.Insert(0, SourceObject);
    SourceNode.MoveTo(DestinationNode, naAddChildFirst);
    FCurrentDesigner.Modified;
  end;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGLSceneEditorForm.Notification(AComponent: TComponent; Operation: TOperation);

begin
  if (FScene = AComponent) and (Operation = opRemove) then
  begin
    FScene := nil;
    ResetTree;
  end;
  inherited;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGLSceneEditorForm.TreeChange(Sender: TObject; Node: TTreeNode);

begin
  if Assigned(Node.Data) then FCurrentDesigner.SelectComponent(Node.Data)
                         else FCurrentDesigner.SelectComponent(FScene);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGLSceneEditorForm.TreeEdited(Sender: TObject; Node: TTreeNode; var S: String);

// renaming a node means renaming a scene object

begin
  TBaseSceneObject(Node.Data).Name := S;
  FCurrentDesigner.Modified;
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGLSceneEditorForm.TreeMouseDown(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);

var
  Node: TTreeNode;
  HI: THitTests;

begin
  Node := Tree.GetNodeAt(X, Y);
  HI := Tree.GetHitTestInfoAt(X, Y);
  if Button = mbLeft then
  begin
    if Assigned(Node) and
       (htOnLabel in HI) and
       (Node.Level > 1) then Tree.BeginDrag(False);
  end
  else
    if (Button = mbRight) and
       Assigned(Node) and
       (htOnLabel in HI) then Tree.Selected := Node;

  if Assigned(Tree.Selected) then FCurrentDesigner.SelectComponent(Tree.Selected.Data);
end;

//----------------------------------------------------------------------------------------------------------------------

procedure TGLSceneEditorForm.TreeEnter(Sender: TObject);

begin
  if Assigned(Tree.Selected) then FCurrentDesigner.SelectComponent(Tree.Selected.Data);
end;

//----------------------------------------------------------------------------------------------------------------------

end.
