unit GLScene;

// TGLScene    - An encapsulation of the OpenGL API
// Version     - 0.5.8
// 30-DEC-99 ml: adjustments for Delphi 5

interface

{$R-}

uses Windows, Classes, Controls, GLScreen, SysUtils,
     GLTexture, Graphics, Messages, OpenGL12, Geometry, MMSystem;

type
  TObjectHandle = TGLUInt; // a display list name or GL_LIGHTx constant

  TNormalDirection = (ndInside, ndOutside);
  TPolygonMode = (pmFill, pmLines, pmPoints);
  TTransformationMode = (tmLocal, tmParentNoPos, tmParentWithPos);

  // used to reflect all relevant (binary) states of OpenGL subsystem
  TGLState = (stAlphaTest, stAutoNormal,
              stBlend, stColorMaterial, stCullFace, stDepthTest, stDither, 
              stFog, stLighting, stLineSmooth, stLineStipple, 
              stLogicOp, stNormalize, stPointSmooth, stPolygonSmooth, 
              stPolygonStipple, stScissorTest, stStencilTest, 
              stTexture1D, stTexture2D);
  TGLStates = set of TGLState;

  // used to decribe only the changes in an object, which have to be reflected in the scene
  TObjectChange = (ocSpot, ocAttenuation, ocTransformation, ocStructure);
  TObjectChanges = set of TObjectChange;

  // flags for design notification
  TSceneOperation = (soAdd, soRemove, soMove, soRename, soSelect, soBeginUpdate, soEndUpdate);

  // flags for allocated buffers
  TBuffer = (buColor, buDepth, buStencil, buAccum);
  TBuffers = set of TBuffer;

  // options for the rendering context
  TContextOption = (roDoubleBuffer, roRenderToWindow, roTwoSideLighting);
  TContextOptions = set of TContextOption;

  // IDs for limit determination
  TLimitType = (limClipPlanes, limEvalOrder, limLights, limListNesting,
                limModelViewStack, limNameStack, limPixelMapTable, limProjectionStack,
                limTextureSize, limTextureStack, limViewportDims, limAccumAlphaBits,
                limAccumBlueBits, limAccumGreenBits, limAccumRedBits, limAlphaBits,
                limAuxBuffers, limBlueBits, limGreenBits, limRedBits, limIndexBits,
                limStereo, limDoubleBuffer, limSubpixelBits, limDepthBits, limStencilBits);

  TBoundingBox = record
    LeftLowerBack,
    RightUpperFront: TAffineVector;
  end;

  TBaseSceneObject = class;
  TSceneObjectClass = class of TBaseSceneObject;
  TCustomSceneObject = class;
  TGLScene = class;

  TGLCoordinates = class(TPersistent)
  private
    FOwner: TBaseSceneObject;
    FCoords: TVector;
    procedure SetAsVector(Value: TVector);
    procedure SetCoordinate(Index: Integer; AValue: TGLFloat);
  protected
    procedure DefineProperties(Filer: TFiler); override;
    procedure ReadData(Stream: TStream);
    procedure WriteData(Stream: TStream);
  public
    constructor Create(AOwner: TBaseSceneObject); virtual;

    procedure Update;

    property AsVector: TVector read FCoords write SetAsVector;
    property W: TGLFloat index 3 read FCoords[3] write SetCoordinate;
  published
    property X: TGLFloat index 0 read FCoords[0] write SetCoordinate stored False;
    property Y: TGLFloat index 1 read FCoords[1] write SetCoordinate stored False;
    property Z: TGLFloat index 2 read FCoords[2] write SetCoordinate stored False;
  end;

  // base class for all scene objects
  TBaseSceneObject = class(TComponent)
  private
    FPosition: TGLCoordinates;
    FDirection, 
    FUp: TGLCoordinates;
    FScaling: TAffineVector;
    FChanges: TObjectChanges;
    FParent: TBaseSceneObject;
    FScene: TGLScene;
    FChildren: TList;
    FVisible: Boolean;
    FLocalMatrix, 
    FGlobalMatrix: TMatrix;
    FMatrixDirty: Boolean;
    FUpdateCount: Integer;
    FShowAxes: Boolean;
    FRollAngle, 
    FPitchAngle, 
    FTurnAngle: Single;                  // absolute rotation in degrees
    FIsCalculating: Boolean;
    FTransMode: TTransformationMode;

    FBoundingBox: TBoundingBox;
    function Get(Index: Integer): TBaseSceneObject;
    function GetCount: Integer;
    function GetIndex: Integer;
    function GetMatrix: TMatrix;
    procedure SetIndex(AValue: Integer);
    procedure SetDirection(AVector: TGLCoordinates);
    procedure SetUp(AVector: TGLCoordinates);
    procedure SetMatrix(AValue: TMatrix);
    procedure SetPosition(APosition: TGLCoordinates);
    procedure SetPitchAngle(AValue: Single);
    procedure SetRollAngle(AValue: Single);
    procedure SetShowAxes(AValue: Boolean);
    procedure SetScaleX(AValue: TGLFloat);
    procedure SetScaleY(AValue: TGLFloat);
    procedure SetScaleZ(AValue: TGLFloat);
    procedure SetTurnAngle(AValue: Single);
    procedure SetVisible(AValue: Boolean);
  protected                                            
//    procedure GetBoundingBox; virtual; abstract;
    procedure DrawAxes(Pattern: TGLushort);
    procedure GetChildren(AProc: TGetChildProc; Root: TComponent); override;
    function GetHandle: TObjectHandle; virtual;
    procedure GetOrientationVectors(var Up, Direction: TAffineVector);
    procedure RebuildMatrix;
    procedure SetName(const NewName: TComponentName); override;
    procedure SetParentComponent(Value: TComponent); override;

    property PitchAngle: Single read FPitchAngle write SetPitchAngle;
    property RollAngle: Single read FRollAngle write SetRollAngle;
    property TransformationMode: TTransformationMode read FTransMode write FTransMode;
    property TurnAngle: Single read FTurnAngle write SetTurnAngle;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    procedure RestoreMatrix;
    function AddNewChild(AChild: TSceneObjectClass): TBaseSceneObject; virtual;
    procedure AddChild(AChild: TBaseSceneObject); virtual;
    procedure Assign(Source: TPersistent); override;
    procedure BeginUpdate; virtual;
    procedure BuildList; virtual;
    procedure CoordinateChanged(Sender: TGLCoordinates); virtual;
    procedure DeleteChildren; virtual;
    procedure DestroyList; virtual;
    procedure EndUpdate; virtual;
    procedure FinishObject; virtual;
    function GetParentComponent: TComponent; override;
    function HasParent: Boolean; override;
    procedure Insert(AIndex: Integer; AChild: TBaseSceneObject); virtual;
    function  IsUpdating: Boolean;
    procedure Lift(ADistance: Single);
    procedure Loaded; override;
    procedure Move(ADistance: Single);
    procedure MoveTo(NewParent: TBaseSceneObject); virtual;
    procedure Pitch(Angle: Single);
    procedure PrepareObject; virtual;
    procedure Remove(AChild: TBaseSceneObject; KeepChildren: Boolean); virtual;
    procedure Render; virtual;
    procedure Roll(Angle: Single);
    procedure Scale(Sx, Sy, Sz: TGLFloat);
    property ShowAxes: Boolean read FShowAxes write SetShowAxes;
    procedure Slide(ADistance: Single);
    procedure StructureChanged;
    procedure Translate(Tx, Ty, Tz: TGLFloat); virtual;
    procedure TransformationChanged;
    procedure Turn(Angle: Single);
    procedure Update;
    procedure ValidateTransformation; virtual;

    property BoundingBox: TBoundingBox read FBoundingBox;
    property Changes: TObjectChanges read FChanges;
    property Children[Index: Integer]: TBaseSceneObject read Get; default;
    property Count: Integer read GetCount;
    property Handle: TObjectHandle read GetHandle;
    property Index: Integer read GetIndex write SetIndex;
    property Matrix: TMatrix  read GetMatrix write SetMatrix;
    property Parent: TBaseSceneObject read FParent write MoveTo;
    property Position: TGLCoordinates read FPosition write SetPosition;
    property Direction: TGLCoordinates read FDirection write SetDirection;
    property Up: TGLCoordinates read FUp write SetUp;
    property ScaleX: TGLFloat read FScaling[0] write SetScaleX;
    property ScaleY: TGLFloat read FScaling[1] write SetScaleY;
    property ScaleZ: TGLFloat read FScaling[2] write SetScaleZ;
    property Scene: TGLScene read FScene;
    property Visible: Boolean read FVisible write SetVisible default True;
  end;

  TRenderOptions = class(TPersistent)
  private
    FOwner: TCustomSceneObject;
    FFrontPolygonMode, 
    FBackPolygonMode: TPolygonMode;
    procedure SetBackPolygonMode(AValue: TPolygonMode);
    procedure SetFrontPolygonMode(AValue: TPolygonMode);
  protected
    constructor Create(AOwner: TCustomSceneObject); virtual;
    procedure Update; virtual;
  public
    procedure Assign(Source: TPersistent); override;
    procedure Apply; virtual;
  published
    property BackPolygonMode: TPolygonMode read FBackPolygonMode write SetBackPolygonMode default pmFill;
    property FrontPolygonMode: TPolygonMode read FFrontPolygonMode write SetFrontPolygonMode default pmFill;
  end;

  // extended base class with material and rendering options
  TCustomSceneObject = class(TBaseSceneObject)
  private
    FHandle: TObjectHandle;
    FBoundingBox: TBoundingBox;
    FOptions: TRenderOptions;
    FMaterial: TMaterial;
    procedure SetMaterial(AValue: TMaterial);
    procedure SetOptions(AValue: TRenderOptions);
  protected
    function GetHandle: TObjectHandle; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    procedure Assign(Source: TPersistent); override;
    procedure BuildList; override;
    procedure DestroyList; override;
    procedure ReloadTexture;
    procedure Render; override;

    property BoundingBox: TBoundingBox read FBoundingBox;
    property Handle: TObjectHandle read GetHandle;
    property Material: TMaterial read FMaterial write SetMaterial;
    property Options: TRenderOptions read FOptions write SetOptions;
    property Visible: Boolean read FVisible write SetVisible default True;
  end;

  TSceneObject = class(TCustomSceneObject)
  published
    property Direction;
    property Options;
    property Material;
    property PitchAngle;
    property Position;
    property RollAngle;
    property ScaleX;
    property ScaleY;
    property ScaleZ;
    property ShowAxes;
    property TransformationMode;
    property TurnAngle;
    property Up;
    property Visible;
  end;

  TProxyObject = class(TSceneObject)
  private
    FObjectLink: TCustomSceneObject;
  public
    property ObjectLink: TCustomSceneObject read FObjectLink write FObjectLink;
  end;

  TExternalObject = class(TSceneObject)
  private
    FTypeName: String;
  public
    property TypeName: String read FTypeName write FTypeName;
  end;

  TLightSource = class(TBaseSceneObject)
  private
    FLightID: TObjectHandle;
    FSpotDirection: TGLCoordinates;
    FSpotExponent,
    FSpotCutOff, 
    FConstAttenuation, 
    FLinearAttenuation, 
    FQuadraticAttenuation: TGLFloat;
    FShining: Boolean;
    FAmbient, 
    FDiffuse, 
    FSpecular: TGLColor;
    procedure SetAmbient(AValue: TGLColor);
    procedure SetDiffuse(AValue: TGLColor);
    procedure SetSpecular(AValue: TGLColor);
    procedure SetConstAttenuation(AValue: TGLFloat);
    procedure SetLinearAttenuation(AValue: TGLFloat);
    procedure SetQuadraticAttenuation(AValue: TGLFloat);
    procedure SetShining(AValue: Boolean);
    procedure SetSpotDirection(AVector: TGLCoordinates);
    procedure SetSpotExponent(AValue: TGLFloat);
    procedure SetSpotCutOff(AValue: TGLFloat);
  protected
    function GetHandle: TObjectHandle; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure DestroyList; override;
    procedure CoordinateChanged(Sender: TGLCoordinates); override;
    procedure RenderLensFlares(from, at: TAffineVector; near_clip: TGLFloat);
    procedure ValidateTransformation; override;
  published
    property Ambient: TGLColor read FAmbient write SetAmbient;
    property ConstAttenuation: TGLFloat read FConstAttenuation write SetConstAttenuation;
    property Diffuse: TGLColor read FDiffuse write SetDiffuse;
    property LinearAttenuation: TGLFloat read FLinearAttenuation write SetLinearAttenuation;
    property QuadraticAttenuation: TGLFloat read FQuadraticAttenuation write SetQuadraticAttenuation;
    property Position;
    property Shining: Boolean read FShining write SetShining default True;
    property Specular: TGLColor read FSpecular write SetSpecular;
    property SpotCutOff: TGLFloat read FSpotCutOff write SetSpotCutOff;
    property SpotDirection: TGLCoordinates read FSpotDirection write SetSpotDirection;
    property SpotExponent: TGLFloat read FSpotExponent write SetSpotExponent;
  end;

  TCamera = class(TBaseSceneObject)
  private
    FFocalLength: Single;
    FDepthOfView: Single;
    FNearPlane: Single;                  // nearest distance to the camera
    FModified: Boolean;
    function GetModified: Boolean;
    procedure SetDepthOfView(AValue: Single);
    procedure SetFocalLength(AValue: Single);
  protected
    procedure Changed(Sender: TGLCoordinates);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;

    procedure Apply;
    procedure ApplyPerspective(Viewport: TRectangle; Width, Height: Integer; DPI: Integer);
    procedure AutoLeveling(Factor: Single);
    procedure CoordinateChanged(Sender: TGLCoordinates); override;
    procedure Reset;
    procedure ZoomAll;

    property Modified: Boolean read GetModified write FModified;
  published
    property DepthOfView: Single read FDepthOfView write SetDepthOfView;
    property FocalLength: Single read FFocalLength write SetFocalLength;
    property Position;
    property Direction;
    property Up;
  end;

  TSceneViewer = class;

  TGLScene = class(TComponent)
  private
    FUpdateCount: Integer;
    FObjects: TCustomSceneObject;
    FCameras: TBaseSceneObject;
    FBaseContext: HGLRC;
    FLights, 
    FViewers: TList;
    FLastCamera, 
    FCurrentCamera: TCamera;
    FCurrentViewer: TSceneViewer;
  protected
    procedure AddLight(ALight: TLightSource);
    procedure SetupLights(Maximum: Integer);
    procedure DoAfterRender;
    procedure GetChildren(AProc: TGetChildProc; Root: TComponent); override;
    procedure Loaded; override;
    procedure RemoveLight(ALight: TLightSource);
    procedure SetChildOrder(AChild: TComponent; Order: Integer); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure AddViewer(AViewer: TSceneViewer);
    procedure BeginUpdate;
    procedure RenderScene(AViewer: TSceneViewer);
    procedure EndUpdate;
    function  IsUpdating: Boolean;
    procedure NotifyChange;
    procedure RemoveViewer(AViewer: TSceneViewer);
    procedure ValidateTransformation(ACamera: TCamera);
    property Cameras: TBaseSceneObject read FCameras;
    property CurrentCamera: TCamera read FCurrentCamera;
    property Lights: TList read FLights;
    property Objects: TCustomSceneObject read FObjects;
    property CurrentViewer: TSceneViewer read FCurrentViewer;
  end;

  TJoystickButton = (jbButton1, jbButton2, jbButton3, jbButton4);
  TJoystickButtons = set of TJoystickButton;

  TJoystickID = (NoJoystick, Joystick1, Joystick2);
  TActiveMode = (inactive, active);
  TJoyPos = (jpMin, jpCenter, jpMax);
  TJoyAxis = (jaX, jaY, jaZ, jaR, jaU, jaV);

  TJoystickEvent = procedure(Sender: TObject; JoyID: TJoystickID; Buttons: TJoystickButtons; XDeflection, YDeflection: Integer) of Object;

  // a simple class to handle incoming joystick messages of the scene viewer
  TGLJoystick = class(TPersistent)
  private
    FOwner: TSceneViewer;
    FNumButtons, 
    FLastX, 
    FLastY, 
    FLastZ: Integer;
    FThreshold, 
    FInterval: Cardinal;
    FCapture: Boolean;
    FJoystickID: TJoystickID;
    FDesignMode: TActiveMode;
    FMinMaxInfo: array[TJoyAxis, TJoyPos] of Integer;
    FXPosInfo, 
    FYPosInfo: array[0..4] of Integer;
    procedure SetCapture(AValue: Boolean);
    procedure SetInterval(AValue: Cardinal);
    procedure SetJoystickID(AValue: TJoystickID);
    procedure SetThreshold(AValue: Cardinal);
  protected
    procedure DoJoystickCapture(AHandle: HWND; AJoystick: TJoystickID);
    procedure DoJoystickRelease(AJoystick: TJoystickID);
    procedure DoXYMove(Buttons: Word; XPos, YPos: Integer);
    procedure DoZMove(Buttons: Word; ZPos: Integer);
    procedure ReapplyCapture(AJoystick: TJoystickID);
  public
    constructor Create(AOwner: TSceneViewer); virtual;
    procedure Assign(Source: TPersistent); override;
    procedure HandleMessage(AMessage: TMessage);
  published
    property Capture: Boolean read FCapture write SetCapture;
    property DesignMode: TActiveMode read FDesignMode write FDesignMode;
    property Interval: Cardinal read FInterval write SetInterval default 100;
    property JoystickID: TJoystickID read FJoystickID write SetJoystickID default NoJoystick;
    property Threshold: Cardinal read FThreshold write SetThreshold default 100;
  end;

  PPickRecord = ^TPickRecord;
  TPickRecord = record
    AObject: TBaseSceneObject;
    ZMin, ZMax: Single;
  end;

  TPickSortType = (psDefault, psName, psMinDepth, psMaxDepth);

  // list class for object picking
  TPickList =  class(TList)
  private
    function GetFar(AValue: Integer): Single;
    function GetHit(AValue: Integer): TBaseSceneObject;
    function GetNear(AValue: Integer): Single;
  protected
  public
    constructor Create(SortType: TPickSortType);
    destructor Destroy; override;
    procedure AddHit(Obj: TBaseSceneObject; ZMin, ZMax: Single);
    procedure Clear; override;
    function FindObject(AObject: TBaseSceneObject): Integer;
    property FarDistance[Index: Integer]: Single read GetFar;
    property Hit[Index: Integer]: TBaseSceneObject read GetHit; default;
    property NearDistance[Index: Integer]: Single read GetNear;
  end;

  TDrawState = (dsNone, dsRendering, dsPicking, dsPrinting);

  TFogMode = (fmLinear, fmExp, fmExp2);
  TFogEnvironment = class(TPersistent)
  private
    FSceneViewer: TSceneViewer;
    FFogColor: TGLColor;       // alpha value means the fog density
    FFogStart, 
    FFogEnd: TGLfloat;
    FFogMode: TFogMode;
    FChanged: Boolean;
    procedure SetFogColor(Value: TGLColor);
    procedure SetFogStart(Value: TGLfloat);
    procedure SetFogEnd(Value: TGLfloat);
    procedure SetFogMode(Value: TFogMode);
    procedure Change;
  public
    constructor Create(Viewer: TSceneViewer); virtual;
    destructor Destroy; override;

    procedure ApplyFog;
    procedure Assign(Source: TPersistent); override;
  published
    property FogColor: TGLColor read FFogColor write SetFogColor;
    property FogStart: TGLfloat read FFogStart write SetFogStart;
    property FogEnd: TGLfloat read FFogEnd write SetFogEnd;
    property FogMode: TFogMode read FFogMode write SetFogMode; 
  end;

  TSpecial = (spLensFlares, spLandScape);
  TSpecials = set of TSpecial;

  TSceneViewer = class(TWinControl)
  private
    // handle
    FRenderingContext: HGLRC;

    // OpenGL properties
    FMaxLightSources: Integer;
    FDoubleBuffered,
    FDepthTest,
    FFaceCulling,
    FFogEnable,
    FLighting: Boolean;
    FCurrentStates,
    FSaveStates: TGLStates;
    FBackgroundColor: TColor;
    FBackground: TTexture;

    // private variables
    FCanvas: TCanvas;
    FFrames: Longint;              // used to perform monitoring
    FTicks: TLargeInteger;        // used to perform monitoring
    FState: TDrawState;
    FMonitor: Boolean;
    FFramesPerSecond: TGLFloat;
    FViewPort: TRectangle;
    FBuffers: TBuffers;
    FDisplayOptions: TDisplayOptions;
    FContextOptions: TContextOptions;
    FOwnRefresh: Boolean;
    FCamera: TCamera;
    FJoystick: TGLJoystick;
    FSpecials: TSpecials;
    FFogEnviroment: TFogEnvironment;

    FBeforeRender: TNotifyEvent;
    FAfterRender: TNotifyEvent;
    FOnJoystickButtonChange, 
    FOnJoystickMove: TJoystickEvent;
    procedure SetBackgroundColor(AColor: TColor);
    function  GetLimit(Which: TLimitType): Integer;
    procedure SetCamera(ACamera: TCamera);
    procedure SetContextOptions(Options: TContextOptions);
    procedure SetDepthTest(AValue: Boolean);
    procedure SetFaceCulling(AValue: Boolean);
    procedure SetJoystick(AValue: TGLJoystick);
    procedure SetLighting(AValue: Boolean);
    procedure SetFogEnable(AValue: Boolean);
    procedure SetFogEnvironment(AValue: TFogEnvironment);
    procedure SetSpecials(Value: TSpecials);

    procedure WMEraseBkgnd(var Message: TWMEraseBkgnd); Message WM_ERASEBKGND;
    procedure WMPaint(var Message: TWMPaint); Message WM_PAINT;
    procedure WMSize(var Message: TWMSize); Message WM_SIZE;

  protected
    procedure CreateParams(var Params: TCreateParams); override;
    procedure CreateWnd; override;
    procedure ClearBuffers;
    procedure DestroyWnd; override;
    procedure Loaded; override;
    function ObjectInScene(Obj: TBaseSceneObject): Boolean;
    procedure ReadContextProperties;
    procedure SetupRenderingContext;
  public
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy; override;
    
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    procedure PickObjects(var Rect: TRect; PickList: TPickList; ObjectCountGuess: Integer);
    procedure Render;
    procedure RenderToBitmap(ABitmap: TBitmap; DPI: Integer);
    procedure RequestedState(States: TGLStates);
    procedure RenderToFile(const AFile: TFileName; DPI: Integer);
    procedure Invalidate; override;
    procedure SetViewPort(X, Y, W, H: Integer);
    procedure ShowInfo;
    procedure UnnecessaryState(States: TGLStates);
    function GetLandsHeight(X, Y: TGLfloat): TGLfloat;

    property Buffers: TBuffers read FBuffers;
    property Canvas: TCanvas read FCanvas;
    property CurrentStates: TGLStates read FCurrentStates;
    property DoubleBuffered: Boolean read FDoubleBuffered;
    property FramesPerSecond: TGLFloat read FFramesPerSecond;
    property LimitOf[Which: TLimitType]: Integer read GetLimit;
    property MaxLightSources: Integer read FMaxLightSources;
    property RenderingContext: HGLRC read FRenderingContext;
    property State: TDrawState read FState;
  published
    property FogEnvironment: TFogEnvironment read FFogEnviroment write SeTFogEnvironment;
    property Align;
    property Anchors;
    property BackgroundColor: TColor read FBackgroundColor write SetBackgroundColor;
    property Camera: TCamera read FCamera write SetCamera;
    property Constraints;
    property ContextOptions: TContextOptions read FContextOptions write SetContextOptions default [roDoubleBuffer, roRenderToWindow];
    property DepthTest: Boolean read FDepthTest write SetDepthTest default True;
    property DisplayOptions: TDisplayOptions read FDisplayOptions write FDisplayOptions;
    property DragCursor;
    property DragMode;
    property Enabled;
    property FaceCulling: Boolean read FFaceCulling write SetFaceCulling default True;
    property HelpContext;
    property Hint;
    property Joystick: TGLJoystick read FJoystick write SetJoystick;
    property FogEnable: Boolean read FFogEnable write SetFogEnable;
    property Lighting: Boolean read FLighting write SetLighting default True;
    property Monitor: Boolean read FMonitor write FMonitor;
    property PopupMenu;
    property Specials: TSpecials read FSpecials write SetSpecials;
    property Visible;

    // events
    property AfterRender: TNotifyEvent read FAfterRender write FAfterRender;
    property BeforeRender: TNotifyEvent read FBeforeRender write FBeforeRender;
    property OnClick;
    property OnDragDrop;
    property OnDragOver;
    property OnJoystickButtonChange: TJoystickEvent read FOnJoystickButtonChange
                                                    write FOnJoystickButtonChange;
    property OnJoystickMove: TJoystickEvent read FOnJoystickMove write FOnJoystickMove;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

procedure CheckOpenGLError;
procedure ShowError(Error: String);
procedure ShowErrorFormatted(Error: String; const Args: array of const);

//------------------------------------------------------------------------------

implementation

uses
  Consts, Dialogs, ExtDlgs, Forms, GLObjects, GLSpecials, GLStrings, Info, Math;

const
  GLAllStates = [stAlphaTest..stStencilTest];

type
  OpenGLError = class(Exception);

var
  CounterFrequency: TLargeInteger;

//------------------ external global routines ----------------------------------

procedure CheckOpenGLError;

// Gets the oldest error from OpenGL engine and tries to clear the error queue.
// Because under some circumstances reading the error code creates a new error
// and thus hanging up the thread, we limit the loop to 6 reads.

var GLError: TGLEnum;
    Count: Word;

begin
  GLError := glGetError;
  if GLError <> GL_NO_ERROR then
  begin
    Count := 0;
    while (glGetError <> GL_NO_ERROR) and (Count < 6) do Inc(Count);
    raise OpenGLError.Create(gluErrorString(GLError));
  end;
end;

//------------------------------------------------------------------------------

procedure ShowError(Error: String);

begin
  MessageBeep(MB_ICONHAND);
  raise OpenGLError.Create(Error);
end;

//------------------------------------------------------------------------------

procedure ShowErrorFormatted(Error: String; const Args: array of const);

begin
  MessageBeep(MB_ICONHAND);
  raise OpenGLError.CreateFmt(Error, Args);
end;

//------------------ internal global routines ----------------------------------

function CreateSceneObject(AScene: TGLScene; AObject: TSceneObjectClass): TBaseSceneObject;

// Creates a new component with unique name. This function is internally used by functions
// which need to create new scene objects (AddNewChild...).

var AOwner: TCustomForm;

begin
  // get the owner of the parent
  AOwner := AScene.Owner as TCustomForm;
  // create the object with the resulting form as owner (at design time, this object
  // will then appear in the source code)
  Result := AObject.Create(AOwner);
  // create an unique name
  {if csDesigning in AScene.ComponentState then
    Result.Name := IFormDesigner(AOwner.Designer).UniqueName(AObject.ClassName);}
end;

//----------------- TPickList -------------------------------------------------

var
  SortFlag: TPickSortType;
                                                       
constructor TPickList.Create(SortType: TPickSortType);

begin
  SortFlag := SortType;
  inherited Create;
end;

//------------------------------------------------------------------------------

destructor TPickList.Destroy;

begin
  Clear;
  inherited Destroy;
end;

//------------------------------------------------------------------------------

function CompareFunction(Item1, Item2: Pointer): Integer;

var
  Diff: Single;

begin
  Result := 0;
  case SortFlag of
    psName:
      Result := CompareText(PPickRecord(Item1).AObject.Name, PPickRecord(Item2).AObject.Name);
    psMinDepth:
      begin
        Diff := PPickRecord(Item1).ZMin - PPickRecord(Item2).ZMin;
        if Diff < 0 then Result := -1
                    else
          if Diff > 0 then Result := 1
                      else Result := 0;
      end;
    psMaxDepth:
      begin
        Diff := Round(PPickRecord(Item1).ZMax - PPickRecord(Item2).ZMax);
        if Diff < 0 then Result := -1
                    else
          if Diff > 0 then Result := 1
                      else Result := 0;
      end;
  end;
end;

//------------------------------------------------------------------------------

procedure TPickList.AddHit(Obj: TBaseSceneObject; ZMin, ZMax: Single);

var
  NewRecord: PPickRecord;

begin
  New(NewRecord);
  try
    NewRecord.AObject := Obj;
    NewRecord.ZMin := ZMin;
    NewRecord.ZMax := ZMax;
    Add(NewRecord);
    if SortFlag <> psDefault then Sort(CompareFunction);
  except
    Dispose(NewRecord);
    raise;
  end;
end;

//------------------------------------------------------------------------------

procedure TPickList.Clear;

var
  I: Integer;

begin
  for I := 0 to Count-1 do
    Dispose(PPickRecord(Items[I]));
  inherited Clear;
end;

//------------------------------------------------------------------------------

function TPickList.FindObject(AObject: TBaseSceneObject): Integer;

var
  I: Integer;

begin
  Result := -1;
  if assigned(AObject) then
    for I := 0 to Count - 1 do
      if Hit[I] = AObject then
      begin
        Result := I;
        Break;
      end;
end;

//------------------------------------------------------------------------------

function TPickList.GetFar(AValue: Integer): Single;

begin
  Result := PPickRecord(Items[AValue]).ZMax;
end;

//------------------------------------------------------------------------------

function TPickList.GetHit(AValue: Integer): TBaseSceneObject;

begin
  Result := PPickRecord(Items[AValue]).AObject;
end;

//------------------------------------------------------------------------------

function TPickList.GetNear(AValue: Integer): Single;

begin
  Result := PPickRecord(Items[AValue]).ZMin;
end;

//----------------- TGLCoordinates ---------------------------------------------

constructor TGLCoordinates.Create(AOwner: TBaseSceneObject);

begin
  inherited Create;
  FOwner := AOwner;
end;

//------------------------------------------------------------------------------

procedure TGLCoordinates.DefineProperties(Filer: TFiler);

begin
  inherited;
  Filer.DefineBinaryProperty('Coordinates', ReadData, WriteData, True);
end;

//------------------------------------------------------------------------------

procedure TGLCoordinates.ReadData(Stream: TStream);

begin
  Stream.Read(FCoords, SizeOf(FCoords));
end;

//------------------------------------------------------------------------------

procedure TGLCoordinates.WriteData(Stream: TStream);

begin
  Stream.Write(FCoords, SizeOf(FCoords));
end;

//------------------------------------------------------------------------------

procedure TGLCoordinates.SetAsVector(Value: TVector);

begin
  FCoords := Value;
  FOwner.CoordinateChanged(Self);
end;

//------------------------------------------------------------------------------

procedure TGLCoordinates.SetCoordinate(Index: Integer; AValue: TGLFloat);

begin
  FCoords[Index] := AValue;
  FOwner.CoordinateChanged(Self);
end;

//------------------------------------------------------------------------------

procedure TGLCoordinates.Update;

begin
  FOwner.Update;
end;

//------------------ TRenderOptions --------------------------------------------

constructor TRenderOptions.Create(AOwner: TCustomSceneObject);

begin
  inherited Create;
  FOwner := AOwner;
  FBackPolygonMode := pmFill;
  FFrontPolygonMode := pmFill;
end;

//------------------------------------------------------------------------------

procedure TRenderOptions.Assign(Source: TPersistent);

begin
  if Source is TRenderOptions then
  begin
    FBackPolygonMode := TrenderOptions(Source).FBackPolygonMode;
    FFrontPolygonMode := TrenderOptions(Source).FFrontPolygonMode;
  end
  else inherited Assign(Source);
end;

//------------------------------------------------------------------------------

procedure TRenderOptions.SetBackPolygonMode(AValue: TPolygonMode);

begin
  if AValue <> FBackPolygonMode then
  begin
    FBackPolygonMode := AValue;
    Update;
  end;
end;

//------------------------------------------------------------------------------

procedure TRenderOptions.SetFrontPolygonMode(AValue: TPolygonMode);

begin
  if AValue <> FFrontPolygonMode then
  begin
    FFrontPolygonMode := AValue;
    Update;
  end;
end;

//------------------------------------------------------------------------------

procedure TRenderOptions.Update;

begin
  if assigned(FOwner) then
    FOwner.StructureChanged;
end;

//------------------------------------------------------------------------------

procedure TRenderOptions.Apply;

begin
  case FFrontPolygonMode of
    pmFill:
      glPolygonMode(GL_FRONT, GL_FILL);
    pmLines:
      glPolygonMode(GL_FRONT, GL_LINE);
    pmPoints:
      glPolygonMode(GL_FRONT, GL_POINT);
  end;

  case FBackPolygonMode of
    pmFill:
      glPolygonMode(GL_BACK, GL_FILL);
    pmLines:
      glPolygonMode(GL_BACK, GL_LINE);
    pmPoints:
      glPolygonMode(GL_BACK, GL_POINT);
  end;
end;

//------------------ TBaseSceneObject ----------------------------------------------

constructor TBaseSceneObject.Create(AOwner: TComponent);

begin
  inherited Create(AOwner);
  FChanges := [ocTransformation, ocStructure];
  FPosition := TGLCoordinates.Create(Self);
  FPosition.FCoords := MakeVector([0, 0, 0, 1]);
  FDirection := TGLCoordinates.Create(self);
  FDirection.FCoords := MakeVector([0, 0, 1, 0]);
  FUp := TGLCoordinates.Create(self);
  FUp.FCoords := MakeVector([0, 1, 0, 0]);
  FScaling := MakeAffineVector([1, 1, 1]);
  FGlobalMatrix := IdentityMatrix;
  FLocalMatrix := IdentityMatrix;
  FChildren := TList.Create;
  FVisible := True;
  FMatrixDirty := True;
end;

//------------------------------------------------------------------------------

destructor TBaseSceneObject.Destroy;

begin
  DestroyList;
  FPosition.Free;
  FDirection.Free;
  FUp.Free;
  if FChildren.Count > 0 then DeleteChildren;
  FChildren.Free;
  if assigned(FParent) then FParent.Remove(Self, False);
  if assigned(FScene) then FScene.NotifyChange;
  inherited Destroy;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.DestroyList;

var I: Integer;

begin
  Include(FChanges, ocStructure);
  for I := 0 to FChildren.Count-1 do
    TBaseSceneObject(FChildren[I]).DestroyList;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.BeginUpdate;

begin
  Inc(FUpdateCount);
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.BuildList;

begin
  glListBase(0);
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.DeleteChildren;

begin
  // children remove themself from child list
  while FChildren.Count > 0 do TBaseSceneObject(FChildren.Items[0]).Free;
  FChildren.Clear;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.DrawAxes(Pattern: TGLushort);

var
  AxisLen: TGLFloat;

begin
  AxisLen := FScene.CurrentViewer.FCamera.FDepthOfView;
  glPushAttrib(GL_ENABLE_BIT or GL_CURRENT_BIT or GL_LIGHTING_BIT or GL_LINE_BIT or GL_COLOR_BUFFER_BIT);
  glDisable(GL_LIGHTING);
  glEnable(GL_LINE_STIPPLE);
  glEnable(GL_LINE_SMOOTH);
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glLineWidth(1);
  glLineStipple(1, Pattern);
  glBegin(GL_LINES);
    glColor3f(0, 0, 0); glVertex3f(0, 0, 0); glVertex3f(-AxisLen, 0, 0);
    glColor3f(1, 0, 0); glVertex3f(0, 0, 0); glVertex3f(AxisLen, 0, 0);
    glColor3f(0, 0, 0); glVertex3f(0, 0, 0); glVertex3f(0, -AxisLen, 0);
    glColor3f(0, 1, 0); glVertex3f(0, 0, 0); glVertex3f(0, AxisLen, 0);
    glColor3f(0, 0, 0); glVertex3f(0, 0, 0); glVertex3f(0, 0, -AxisLen);
    glColor3f(0, 0, 1); glVertex3f(0, 0, 0); glVertex3f(0, 0, AxisLen);
  glEnd;
  glPopAttrib;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.GetChildren(AProc: TGetChildProc; Root: TComponent);

var
  I: Integer;

begin
  for I := 0 to Count - 1 do AProc(FChildren[I]);
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.RebuildMatrix;

var
  LeftVector: TAffineVector;

begin
  if FMatrixDirty then
  begin
    LeftVector := VectorCrossProduct(MakeAffineVector(FUp.FCoords), MakeAffineVector(FDirection.FCoords));
    VectorNormalize(LeftVector);
    FLocalMatrix[0] := MakeVector(LeftVector); VectorScale(FLocalMatrix[0], FScaling[0]);
    FLocalMatrix[1] := FUp.FCoords;            VectorScale(FLocalMatrix[1], FScaling[1]);
    FLocalMatrix[2] := FDirection.FCoords;     VectorScale(FLocalMatrix[2], FScaling[2]);
    FLocalMatrix[3] := FPosition.FCoords;
    FMatrixDirty := False;
  end;
end;

//------------------------------------------------------------------------------

function TBaseSceneObject.Get(Index: Integer): TBaseSceneObject;

begin
  Result := FChildren[Index];
end;

//------------------------------------------------------------------------------

function TBaseSceneObject.GetCount: Integer;

begin
  Result := FChildren.Count;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.AddChild(AChild: TBaseSceneObject);

begin
  if assigned(FScene) and (AChild is TLightSource) then FScene.AddLight(TLightSource(AChild));
  FChildren.Add(AChild);
  AChild.FParent := Self;
  AChild.FScene := FScene;
  TransformationChanged;
end;

//------------------------------------------------------------------------------

function TBaseSceneObject.AddNewChild(AChild: TSceneObjectClass): TBaseSceneObject;

// create a new scene object and add it to this object as new child

begin
  Result := CreateSceneObject(FScene, AChild);
  AddChild(Result);
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.RestoreMatrix;

begin
  glLoadMatrixf(@FGlobalMatrix);
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.Assign(Source: TPersistent);

var
  I: Integer;

begin
  if Source is TBaseSceneObject then
  begin
    FPosition.FCoords := TBaseSceneObject(Source).FPosition.FCoords;
    FChanges := [ocTransformation, ocStructure];
    FVisible := TBaseSceneObject(Source).FVisible;
    FGlobalmatrix := TBaseSceneObject(Source).FGLobalMatrix;
    SetMatrix(TCustomSceneObject(Source).FLocalMatrix);
    DeleteChildren;
    Parent := TBaseSceneObject(Source).Parent; // implies necessary updates
    if assigned(Scene) then Scene.BeginUpdate;
    for I := 0 to TBaseSceneObject(Source).Count - 1 do AddChild(TBaseSceneObject(Source)[I]);
    if assigned(Scene) then Scene.EndUpdate;
  end
  else inherited Assign(Source);
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.Insert(AIndex: Integer; AChild: TBaseSceneObject);

begin
  with FChildren do
  begin
    if assigned(AChild.FParent) then AChild.FParent.Remove(AChild, False);
    Insert(AIndex, AChild);
  end;
  AChild.FParent := Self;
  if AChild.FScene <> FScene then AChild.DestroyList;
  AChild.FScene := FScene;
  if assigned(FScene) then
  begin
    if AChild is TLightSource then FScene.AddLight(TLightSource(AChild));
  end;
  TransformationChanged;
end;

//------------------------------------------------------------------------------

function TBaseSceneObject.IsUpdating: Boolean;

begin
  Result := (FUpdateCount <> 0) or (csReading in ComponentState);
end;

//------------------------------------------------------------------------------

function TBaseSceneObject.GetIndex: Integer;

begin
  Result := -1;
  if assigned(FParent) then Result := FParent.FChildren.IndexOf(Self);
end;

//------------------------------------------------------------------------------

function TBaseSceneObject.GetHandle: TObjectHandle;

begin
  Result := 0;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.GetOrientationVectors(var Up, Direction: TAffineVector);

// returns Up and Direction vectors depending on the transformation mode

begin
  if (FTransMode <> tmLocal) and assigned(FParent) then
  begin
    Up := MakeAffineVector(FParent.FUp.FCoords);
    Direction := MakeAffineVector(FParent.FDirection.FCoords);
  end
  else
  begin
    Up := MakeAffineVector(FUp.FCoords);
    Direction := MakeAffineVector(FDirection.FCoords);
  end;
end;

//------------------------------------------------------------------------------

function TBaseSceneObject.GetParentComponent: TComponent;

begin
  Result := FParent;
end;

//------------------------------------------------------------------------------

function TBaseSceneObject.HasParent: Boolean;

begin
  Result := assigned(FParent);
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.Lift(ADistance: Single);

// moves object along the Up vector (move up/down)

var
  Up, Dir: TAffineVector;

begin
  if FTransMode = tmParentWithPos then
  begin
    GetOrientationVectors(Up, Dir);
    FPosition.X := FPosition.X + ADistance * Up[0];
    FPosition.Y := FPosition.Y + ADistance * Up[1];
    FPosition.Z := FPosition.Z + ADistance * Up[2];
  end
  else
  begin
    FPosition.X := FPosition.X + ADistance * FUp.FCoords[0];
    FPosition.Y := FPosition.Y + ADistance * FUp.FCoords[1];
    FPosition.Z := FPosition.Z + ADistance * FUp.FCoords[2];
  end;
  TransformationChanged;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.Loaded;

begin
  inherited;
  if FPosition.W = 0 then FPosition.W := 1;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.Move(ADistance: Single);

// moves object along the direction vector

var
  Len: Single;
  Up, Dir: TAffineVector;

begin
  if FTransMode = tmParentWithPos then
  begin
    GetOrientationVectors(Up, Dir);
    Len := 1 / Vectorlength(Dir);
    FPosition.AsVector := MakeVector([FPosition.X + ADistance * Dir[0] * Len, 
                                      FPosition.Y + ADistance * Dir[1] * Len,
                                      FPosition.Z + ADistance * Dir[2] * Len, 1]);
  end
  else
  begin
    Len := 1 / Vectorlength(FDirection.FCoords);
    FPosition.AsVector := MakeVector([FPosition.X + ADistance * FDirection.X * Len,
                                      FPosition.Y + ADistance * FDirection.Y * Len,
                                      FPosition.Z + ADistance * FDirection.Z * Len, 1]);
  end;
  TransformationChanged;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.Pitch(Angle: Single);

var
  RightVector: TAffineVector;
  Up, Dir: TAffineVector;

begin
  GetOrientationVectors(Up, Dir);
  RightVector := VectorCrossProduct(Dir, Up);
  Angle := DegToRad(Angle);
  VectorRotate(FUp.FCoords, RightVector, -Angle);
  VectorNormalize(FUp.FCoords);
  VectorRotate(FDirection.FCoords, RightVector, -Angle);
  VectorNormalize(FDirection.FCoords);
  if FTransMode = tmParentWithPos then VectorRotate(FPosition.FCoords, RightVector, Angle);
  // Direction is automatically updated

  FPitchAngle := -RadToDeg(arctan2(FDirection.Y, Sqrt(Sqr(FDirection.X) + Sqr(FDirection.Z))));
  if FDirection.X < 0 then
    if FDirection.Y < 0 then FPitchAngle := 180 - FPitchAngle
                        else FPitchAngle := -180 - FPitchAngle;
  TransformationChanged;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.SetPitchAngle(AValue: Single);

var
  RightVector: TAffineVector;
  Up, Dir: TAffineVector;
  Diff: Extended;

begin
  if AValue <> FPitchAngle then
  begin
    GetOrientationVectors(Up, Dir);
    if not (csLoading in ComponentState) then
    begin
      Diff := DegToRad(FPitchAngle - AValue);
      RightVector := VectorCrossProduct(MakeAffineVector(Dir), MakeAffineVector(Up));
      VectorRotate(FUp.FCoords, RightVector, Diff);
      VectorNormalize(FUp.FCoords);
      VectorRotate(FDirection.FCoords, RightVector, Diff);
      VectorNormalize(FDirection.FCoords);
      if FTransMode = tmParentWithPos then VectorRotate(FPosition.FCoords, RightVector, Diff);
      TransformationChanged;
    end;
    FPitchAngle := AValue;
    while FPitchAngle > 180 do FPitchAngle := FPitchAngle - 360;
    while FPitchAngle < -180 do FPitchAngle := FPitchAngle + 360;
  end;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.Roll(Angle: Single);

var
  RightVector: TAffineVector;
  Up, Dir: TAffineVector;

begin
  GetOrientationVectors(Up, Dir);
  Angle := DegToRad(Angle);
  VectorRotate(FUp.FCoords, Dir, Angle);
  VectorNormalize(FUp.FCoords);
  VectorRotate(FDirection.FCoords, Dir, Angle);
  VectorNormalize(FDirection.FCoords);
  if FTransMode = tmParentWithPos then VectorRotate(FPosition.FCoords, Dir, Angle);

  // calculate new rotation angle from vectors
  RightVector := VectorCrossProduct(MakeAffineVector(FDirection.FCoords), MakeAffineVector(FUp.FCoords));
  FRollAngle := -RadToDeg(arctan2(Rightvector[1], Sqrt(Sqr(RightVector[0]) + Sqr(RightVector[2]))));
  if RightVector[0] < 0 then
    if RightVector[1] < 0 then FRollAngle := 180 - FRollAngle
                          else FRollAngle := -180 - FRollAngle;
  TransformationChanged;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.SetRollAngle(AValue: Single);

var
  Up, Dir: TAffineVector;
  Diff: Extended;

begin
  if AValue <> FRollAngle then
  begin
    GetOrientationVectors(Up, Dir);
    if not (csLoading in ComponentState) then
    begin
      Diff := DegToRad(FRollAngle - AValue);
      VectorRotate(FUp.FCoords, Dir, Diff);
      VectorNormalize(FUp.FCoords);
      VectorRotate(FDirection.FCoords, Dir, Diff);
      VectorNormalize(FDirection.FCoords);
      if FTransMode = tmParentWithPos then VectorRotate(FPosition.FCoords, Dir, Diff);
      TransformationChanged;
    end;
    FRollAngle := AValue;
    while FRollAngle > 180 do FRollAngle := FRollAngle - 360;
    while FRollAngle < -180 do FRollAngle := FRollAngle + 360;
  end;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.Scale(Sx, Sy, Sz: TGLFloat);

begin
  FScaling := MakeAffineVector([Sx, Sy, Sz]);
  TransformationChanged;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.Slide(ADistance: Single);

// moves camera along the right vector (move left and right)

var
  RightVector: TAffineVector;
  Up, Dir: TAffineVector;

begin
  if FTransMode = tmParentWithPos then GetOrientationVectors(Up, Dir)
                                  else
  begin
    Up := MakeAffineVector(FUp.FCoords);
    Dir := MakeAffineVector(FDirection.FCoords);
  end;
  RightVector := VectorCrossProduct(MakeAffineVector(Dir), MakeAffineVector(Up));
  VectorNormalize(RightVector);
  FPosition.FCoords := MakeVector([FPosition.X + ADistance * RightVector[0], 
                                   FPosition.Y + ADistance * RightVector[1],
                                   FPosition.Z + ADistance * RightVector[2], 1]);
  TransformationChanged;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.Turn(Angle: Single);

var
  Up, Dir: TAffineVector;

begin
  GetOrientationVectors(Up, Dir);
  Angle := DegToRad(Angle);
  VectorRotate(FUp.FCoords, Up, Angle);
  VectorNormalize(FUp.FCoords);
  VectorRotate(FDirection.FCoords, Up, Angle);
  VectorNormalize(FDirection.FCoords);
  if FTransMode = tmParentWithPos then VectorRotate(FPosition.FCoords, Up, Angle);
  FTurnAngle := -RadToDeg(arctan2(FDirection.X, Sqrt(Sqr(FDirection.Y) + Sqr(FDirection.Z))));
  if FDirection.X < 0 then
    if FDirection.Y < 0 then FTurnAngle := 180 - FTurnAngle
                        else FTurnAngle := -180 - FTurnAngle;
  TransformationChanged;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.SetTurnAngle(AValue: Single);

var
  Up, Dir: TAffineVector;
  Diff: Extended;

begin
  if AValue <> FTurnAngle then
  begin
    GetOrientationVectors(Up, Dir);
    if not (csLoading in ComponentState) then
    begin
      Diff := DegToRad(FTurnAngle - AValue);
      VectorRotate(FUp.FCoords, Up, Diff);
      VectorNormalize(FUp.FCoords);
      VectorRotate(FDirection.FCoords, Up, Diff);
      VectorNormalize(FDirection.FCoords);
      if FTransMode = tmParentWithPos then VectorRotate(FPosition.FCoords, Up, Diff);
      TransformationChanged;
    end;
    FTurnAngle := AValue;
    while FTurnAngle > 180 do FTurnAngle := FTurnAngle - 360;
    while FTurnAngle < -180 do FTurnAngle := FTurnAngle + 360;
  end;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.SetShowAxes(AValue: Boolean);

begin
  if FShowAxes <> AValue then
  begin
    FShowAxes := AValue;
    Update;
  end;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.SetScaleX(AValue: TGLFloat);

begin
  if AValue <> FScaling[0] then
  begin
    FScaling[0] := AValue;
    TransformationChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.SetScaleY(AValue: TGLFloat);

begin
  if AValue <> FScaling[1] then
  begin
    FScaling[1] := AValue;
    TransformationChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.SetScaleZ(AValue: TGLFloat);

begin
  if AValue <> FScaling[2] then
  begin
    FScaling[2] := AValue;
    TransformationChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.SetName(const NewName: TComponentName);

begin
  if Name <> NewName then
  begin
    inherited SetName(NewName);
  end;
end;

//------------------------------------------------------------------------------

function TBaseSceneObject.GetMatrix: TMatrix;

begin
  // update local matrix if necessary
  RebuildMatrix;
  Result := FLocalMatrix;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.SetIndex(AValue: Integer);

var
  Count: Integer;
  AParent: TBaseSceneObject;

begin
  if assigned(FParent) then
  begin
    Count := FParent.Count;
    AParent := FParent;
    if AValue < 0 then AValue :=  0;
    if AValue >= Count then AValue :=  Count - 1;
    if AValue <> Index then
    begin
      if assigned(FScene) then FScene.BeginUpdate;
      FParent.Remove(Self, False);
      AParent.Insert(AValue, Self);
      if assigned(FScene) then FScene.EndUpdate;
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.SetParentComponent(Value: TComponent);

begin
  if assigned(FParent) then
  begin
    FParent.Remove(Self, False);
    FParent := nil;
  end;
  // first level object?
  if Value is TGLScene then
      if Self is TCamera then TGLScene(Value).Cameras.AddChild(Self)
                         else TGLScene(Value).Objects.AddChild(Self)
                       else TBaseSceneObject(Value).AddChild(Self);  // normal parent-child relation
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.StructureChanged;

begin
  Include(FChanges, ocStructure);
  Update;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.TransformationChanged;

begin
  FMatrixDirty := True;
  Include(FChanges, ocTransformation);
  Update;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.MoveTo(NewParent: TBaseSceneObject);

begin
  if assigned(FParent) then
  begin
    FParent.Remove(Self, False);
    FParent := nil;
  end;
  if assigned(NewParent) then NewParent.AddChild(Self)
                         else FScene := nil;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.EndUpdate;

begin
 if FUpdateCount > 0 then Dec(FUpdateCount);
 if FUpdateCount = 0 then Update;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.FinishObject;

begin
  Exclude(FChanges, ocTransformation);
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.CoordinateChanged(Sender: TGLCoordinates); 

// recalculate an orthonormal system

var
  RightVector, Temp: TAffineVector;

begin
  if FIsCalculating then Exit;
  FIsCalculating := True;
  try
    if Sender = FDirection then
    begin
      if VectorLength(FUp.FCoords) = 0 then FUp.FCoords := MakeVector([0, 1]);
      VectorNormalize(FDirection.FCoords);
      // adjust up vector
      RightVector := VectorCrossProduct(MakeAffineVector(FDirection.FCoords), MakeAffineVector(FUp.FCoords));
      // Rightvector is zero if direction changed exactly by 90 degrees, 
      // in this case assume a default vector
      if VectorLength(RightVector) = 0 then
      begin
        RightVector := MakeAffineVector([FDirection.X + 1, FDirection.Y + 1, FDirection.Y + 1]);
        VectorNormalize(RightVector);
      end;
      Temp := VectorCrossProduct(RightVector, MakeAffineVector(FDirection.FCoords));
      FUp.FCoords := MakeVector(Temp);
      VectorNormalize(FUp.FCoords);
    end
    else
    begin
      if VectorLength(FDirection.FCoords) = 0 then FDirection.FCoords := MakeVector([0, 0, 1]);
      VectorNormalize(FUp.FCoords);
      // adjust up vector
      RightVector := VectorCrossProduct(MakeAffineVector(FDirection.FCoords), MakeAffineVector(FUp.FCoords));
      // Rightvector is zero if direction changed exactly by 90 degrees, 
      // in this case assume a default vector
      if VectorLength(RightVector) = 0 then
      begin
        RightVector := MakeAffineVector([FUp.X + 1, FUp.Y + 1, FUp.Y + 1]);
        VectorNormalize(RightVector);
      end;
      Temp := VectorCrossProduct(MakeAffineVector(FUp.FCoords), RightVector);
      FDirection.FCoords := MakeVector(Temp);
      VectorNormalize(FDirection.FCoords);
    end;
    TransformationChanged;
  finally
    FIsCalculating := False;
  end;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.PrepareObject;

begin
  glLoadMatrixf(@FGLobalMatrix);
  if FScene.FCurrentViewer.State = dsPicking then glLoadName(Integer(Self));
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.Remove(AChild: TBaseSceneObject; KeepChildren: Boolean);

// Takes a scene object out of the child list, but doesn't destroy it. If 'KeepChildren'
// is true its children will be kept as new children in this
// scene object.

begin
  if assigned(FScene) and (AChild is TLightSource) then FScene.RemoveLight(TLightSource(AChild));
  FChildren.Remove(AChild);
  AChild.FParent := nil;
  if KeepChildren then
  begin
    BeginUpdate;
    with AChild do
      while Count > 0 do Children[0].MoveTo(Self);
    EndUpdate;
  end
  else Update;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.Render;

var
  I: Integer;

begin
  PrepareObject;
  if (FVisible) then
    //if FScene.FCurrentViewer.ObjectInScene(Self) then
      glCallList(Handle);
  if FShowAxes then
    DrawAxes($CCCC);
  for I := 0 to Count-1 do
    Self[I].Render;
  FinishObject;
 {$ifdef DEBUG} CheckOpenGLError; {$endif}
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.Update;

begin
  if assigned(FScene) and not IsUpdating then FScene.NotifyChange;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.ValidateTransformation;

// calculate own global matrix and let the children do the same with their's

var
  I: Integer;

begin
  // determine predecessor in transformation pipeline
  if (FParent = nil) then
  begin
    if (Scene.FLastCamera <> Scene.FCurrentCamera) or (ocTransformation in Scene.FCurrentCamera.FChanges) then
    begin
      FGlobalMatrix := Scene.CurrentCamera.FGLobalMatrix;
      Include(FChanges, ocTransformation);
    end;
  end
  else
    if ocTransformation in FChanges + FParent.FChanges then
    begin
      if ocTransformation in FChanges then RebuildMatrix;
      FGlobalMatrix := MatrixMultiply(FLocalMatrix, FParent.FGlobalMatrix);
      Include(FChanges, ocTransformation);
    end;

  for I := 0 to Count - 1 do Self[I].ValidateTransformation;
  Exclude(FChanges, ocTransformation);
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.SetMatrix(AValue: TMatrix);

var
  Temp: TAffineVector;

begin
  FLocalMatrix := AValue;
  FDirection.FCoords := FLocalMatrix[2];
  FUp.FCoords := FLocalMatrix[1];
  Temp := MakeAffineVector(FLocalMatrix[0]);
  FScaling[0] := VectorLength(Temp);
  FScaling[1] := VectorLength(FUp.FCoords);
  FScaling[2] := VectorLength(FDirection.FCoords);
  FPosition.FCoords := FLocalMatrix[3];
  FMatrixDirty := False;
  Include(FChanges, ocTransformation);
  Update;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.SetPosition(APosition: TGLCoordinates);

begin
  FPosition.FCoords := APosition.FCoords;
  TransformationChanged;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.SetDirection(AVector: TGLCoordinates);

begin
  FDirection.FCoords := AVector.FCoords;
  TransformationChanged;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.SetUp(AVector: TGLCoordinates);

begin
  FUp.FCoords := AVector.FCoords;
  TransformationChanged;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.SetVisible(AValue: Boolean);

begin
  if FVisible <> AValue then
  begin
    FVisible := AValue;
    Update;
  end;
end;

//------------------------------------------------------------------------------

procedure TBaseSceneObject.Translate(Tx, Ty, Tz: TGLFloat);

begin
  FPosition.FCoords := MakeVector([Tx, Ty, Tz, FPosition.W]);
end;

//------------------ TCustomSceneObject ----------------------------------------------

constructor TCustomSceneObject.Create(AOwner: TComponent);

begin
  inherited Create(AOwner);
  FHandle := 0;
  FMaterial := TMaterial.Create(Self);
  FOptions := TRenderOptions.Create(Self);
end;

//------------------------------------------------------------------------------

destructor TCustomSceneObject.Destroy;

begin
  FMaterial.Free;
  inherited Destroy;
end;

//------------------------------------------------------------------------------

procedure TCustomSceneObject.DestroyList;

begin
  if FHandle > 0 then
  begin
    glDeleteLists(FHandle, 1);
    FHandle := 0;
  end;
  inherited DestroyList;
end;

//------------------------------------------------------------------------------

procedure TCustomSceneObject.Assign(Source: TPersistent);

begin
  if Source is TCustomSceneObject then
  begin
    FBoundingBox := TCustomSceneObject(Source).FBoundingBox;
    FOptions.Assign(TCustomSceneObject(Source).FOptions);
    FMaterial.Assign(TCustomSceneObject(Source).FMaterial);
  end;
  inherited Assign(Source);
end;

//------------------------------------------------------------------------------

procedure TCustomSceneObject.BuildList;

begin
  FOptions.Apply;
  inherited BuildList;
end;

//------------------------------------------------------------------------------

function TCustomSceneObject.GetHandle: TObjectHandle;

begin
  if ocStructure in FChanges then
  begin
    if FHandle = 0 then
    begin
      FHandle := glGenLists(1);
      if FHandle = 0 then ShowErrorFormatted(glsDisplayList, [Name]);
    end;
    glNewList(FHandle, GL_COMPILE);
    BuildList;
    glEndList;
    Exclude(FChanges, ocStructure);
  end;
  Result := FHandle;
end;

//------------------------------------------------------------------------------

procedure TCustomSceneObject.SetMaterial(AValue: TMaterial);

begin
  FMaterial.Assign(AValue);
  Update;
end;

//------------------------------------------------------------------------------

procedure TCustomSceneObject.SetOptions(AValue: TRenderOptions);

begin
  FOptions.Assign(AValue);
  StructureChanged;
end;

//------------------------------------------------------------------------------

procedure TCustomSceneObject.ReloadTexture;

var
  I: Integer;

begin
  FMaterial.Texture.ReloadImage;
  for I := 0 to Count - 1 do
    if Self[I] is TCustomSceneObject then TCustomSceneObject(Self[I]).ReloadTexture;
end;

//------------------------------------------------------------------------------

procedure TCustomSceneObject.Render;

begin
  FMaterial.Apply;
  inherited Render;
end;

//----------------- TCamera ----------------------------------------------------

constructor TCamera.Create(AOwner: TCOmponent);

begin
  inherited;
  FFocalLength := 50;
  FDepthOfView := 100;
  FDirection.FCoords := MakeVector([0, 0, -1, 0]);
end;

//------------------------------------------------------------------------------

destructor TCamera.Destroy;

begin
  inherited;
end;

//------------------------------------------------------------------------------

function TCamera.GetModified: Boolean;

begin
  Result := FModified;
  FModified := False;
end;

//------------------------------------------------------------------------------

procedure TCamera.Apply;

// apply camera transformation

begin
  if ocTransformation in FChanges then
  begin
    gluLookAt(FPosition.X, FPosition.Y, FPosition.Z, 
              FPosition.X + FDirection.X, 
              FPosition.Y + FDirection.Y, 
              FPosition.Z + FDirection.Z, 
              FUp.X, FUp.Y, FUp.Z);
    glGetFloatv(GL_MODELVIEW_MATRIX, @FGLobalMatrix);
  end;
end;

//------------------------------------------------------------------------------

procedure TCamera.ApplyPerspective(Viewport: TRectangle; Width, Height: Integer; DPI: Integer);

var
  Left, Right,
  Top, Bottom,
  zFar, MaxDim,
  Ratio: Double;

begin
  // determine biggest dimension and resolution (height or width)
  MaxDim := Width;
  if Height > MaxDim then
    MaxDim := Height;

  // calculate near plane distance and extensions;
  // Scene ratio is determined by the window ratio. The viewport is just a
  // specific part of the entire window and has therefore no influence on the
  // scene ratio. What we need to know, though, is the ratio between the window
  // borders (left, top, right and bottom) and the viewport borders.
  // Note: viewport.top is actually bottom, because the window (and viewport) origin
  // in OGL is the lower left corner

  // calculate window/viewport ratio for right extent
  Ratio := (2 * Viewport.Width + 2 * Viewport.Left - Width) / Width;
  // calculate aspect ratio correct right value of the view frustum and take
  // the window/viewport ratio also into account
  Right := Ratio * Width / (2 * MaxDim);

  // the same goes here for the other three extents
  // left extent:
  Ratio := (Width - 2 * Viewport.Left) / Width;
  Left := -Ratio * Width / (2 * MaxDim);

  // top extent (keep in mind the origin is left lower corner):
  Ratio := (2 * Viewport.Height + 2 * Viewport.Top - Height) / Height;
  Top := Ratio * Height / (2 * MaxDim);

  // bottom extent:
  Ratio := (Height - 2 * Viewport.Top) / Height;
  Bottom := -Ratio * Height / (2 * MaxDim);

  FNearPlane := FFocalLength * 2 * DPI / (25.4 * MaxDim);
  zFar := FNearPlane + FDepthOfView;

  // finally create view frustum
  glFrustum(Left, Right, Bottom, Top, FNearPlane, zFar);
end;

//------------------------------------------------------------------------------

procedure TCamera.AutoLeveling(Factor: Single);

var
  RightVector,
  RotAxis: TAffineVector;
  Angle: Single;
  ACopy: TAffineVector;

begin
  Angle := RadToDeg(arccos(VectorAffineDotProduct(MakeAffineVector(FUp.FCoords), YVector)));
  RotAxis := VectorCrossProduct(YVector, MakeAffineVector(FUp.FCoords));
  if (Angle > 1) and (VectorLength(RotAxis) > 0) then
  begin
    RightVector := VectorCrossProduct(MakeAffineVector(FDirection.FCoords), MakeAffineVector(FUp.FCoords));
    VectorRotate(FUp.FCoords, RotAxis, Angle / 10 / Factor);
    VectorNormalize(FUp.FCoords);
    // adjust local coordinates
    ACopy := VectorCrossProduct(MakeAffineVector(FUp.FCoords), RightVector);
    FDirection.FCoords := MakeVector(ACopy);
    FRollAngle := -RadToDeg(arctan2(Rightvector[1], Sqrt(Sqr(RightVector[0]) + Sqr(RightVector[2]))));
    FModified := True;
  end;  
end;

//------------------------------------------------------------------------------

procedure TCamera.Changed(Sender: TGLCoordinates);

begin
  FModified := True;
end;

//------------------------------------------------------------------------------

procedure TCamera.CoordinateChanged(Sender: TGLCoordinates);

begin
  inherited;
  Changed(Sender);
end;

//------------------------------------------------------------------------------

procedure TCamera.Reset;

var Extent: Single;

begin
  FRollAngle := 0;
  FFocalLength := 50;
  with FScene.CurrentViewer do
  begin
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity;
    ApplyPerspective(FViewport, Width, Height, GetDeviceCaps(Canvas.Handle, LOGPIXELSX));
    FUp.FCoords := MakeVector([0, 1]);
    Extent := MinIntValue([FViewport.Height, FViewport.Width]) / 4;
  end;
  FPosition.FCoords := MakeVector([0, 0, FNearPlane * Extent, 1]);
  FDirection.FCoords := MakeVector([0, 0, -1]);
end;

//------------------------------------------------------------------------------

procedure TCamera.ZoomAll;

// position the camera so that the whole scene can be seen

var Extent: Single;

begin
  with Scene.CurrentViewer do
  begin
    Extent := MinIntValue([FViewport.Height, FViewport.Width]) / 4;
    FPosition.FCoords := MakeVector([0, 0, 0, 1]);
    Move(-FNearPlane * Extent);
    // let the camera look at the scene center
    FDirection.FCoords := MakeVector([-FPosition.X, -FPosition.Y, -FPosition.Z]);
  end;
end;

//------------------------------------------------------------------------------

procedure TCamera.SetDepthOfView(AValue: Single);

begin
  if FDepthOfView <> AValue then
  begin
    FDepthOfView := AValue;
    if not (csLoading in Scene.ComponentState) then
    begin
      FModified := True;
      Update;
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure TCamera.SetFocalLength(AValue: Single);

begin
  if AValue < 1 then AValue := 1;
  if FFocalLength <> AValue  then
  begin
    FFocalLength := AValue;
    if not (csLoading in Scene.ComponentState) then
    begin
      FModified := True;
      Update;
    end;
  end;
end;

//------------------ TLightSource ----------------------------------------------

constructor TLightSource.Create(AOwner: TComponent);

begin
  inherited Create(AOwner);
  FShining := True;
  FSpotDirection := TGLCoordinates.Create(Self);
  FSpotDirection.FCoords := MakeVector([0, 0, -1]);
  FConstAttenuation := 1;
  FLinearAttenuation := 0;
  FQuadraticAttenuation := 0;
  FSpotCutOff := 180;
  FSpotExponent := 0;
  FAmbient := TGLColor.Create(Self);
  FDiffuse := TGLColor.Create(Self);
  FSpecular := TGLColor.Create(Self);
  FVisible := False;
  FChanges := [];
end;

//------------------------------------------------------------------------------

destructor TLightSource.Destroy;

begin
  FSpotDirection.Free;
  FAmbient.Free;
  FDiffuse.Free;
  FSpecular.Free;
  inherited Destroy;
end;

//------------------------------------------------------------------------------

procedure TLightSource.DestroyList;

begin
end;

//------------------------------------------------------------------------------

procedure TLightSource.CoordinateChanged(Sender: TGLCoordinates);

begin
  if Sender = FSpotDirection then Include(FChanges, ocSpot);
  TransformationChanged;
end;

//------------------------------------------------------------------------------

function TLightSource.GetHandle: TObjectHandle;

// light sources have different handle types than normal scene objects

begin
  Result := 0;
end;

//------------------------------------------------------------------------------

procedure TLightSource.SetShining(AValue: Boolean);

begin
  if AValue <> FShining then
  begin
    FShining := AValue;
    Update;
  end;
end;

//------------------------------------------------------------------------------

procedure TLightSource.SetSpotDirection(AVector: TGLCoordinates);

begin
  FSpotDirection.FCoords := AVector.FCoords;
  Include(FChanges, ocSpot);
  Update;
end;

//------------------------------------------------------------------------------

procedure TLightSource.SetSpotExponent(AValue: TGLFloat);

begin
  if FSpotExponent <> AValue then
  begin
    FSpotExponent := AValue;
    Include(FChanges, ocSpot);
    Update;
  end;
end;

//------------------------------------------------------------------------------

procedure TLightSource.SetSpotCutOff(AValue: TGLFloat);

begin
  if FSpotCutOff <> AValue then
  begin
    FSpotCutOff := AValue;
    Include(FChanges, ocSpot);
    Update;
  end;  
end;

//------------------------------------------------------------------------------

procedure TLightSource.SetAmbient(AValue: TGLColor);

begin
  FAmbient.Color := AValue.Color;
  Update;
end;

//------------------------------------------------------------------------------

procedure TLightSource.SetDiffuse(AValue: TGLColor);

begin
  FDiffuse.Color := AValue.Color;
  Update;
end;

//------------------------------------------------------------------------------

procedure TLightSource.SetSpecular(AValue: TGLColor);

begin
  FSpecular.Color := AValue.Color;
  Update;
end;

//------------------------------------------------------------------------------

procedure TLightSource.SetConstAttenuation(AValue: TGLFloat);

begin
  if FConstAttenuation <> AValue then
  begin
    FConstAttenuation := AValue;
    Include(FChanges, ocAttenuation);
    Update;
  end;
end;

//------------------------------------------------------------------------------

procedure TLightSource.SetLinearAttenuation(AValue: TGLFloat);

begin
  if FLinearAttenuation <> AValue then
  begin
    FLinearAttenuation := AValue;
    Include(FChanges, ocAttenuation);
    Update;
  end;  
end;

//------------------------------------------------------------------------------

procedure TLightSource.SetQuadraticAttenuation(AValue: TGLFloat);

begin
  if FQuadraticAttenuation <> AValue then
  begin
    FQuadraticAttenuation := AValue;
    Include(FChanges, ocAttenuation);
    Update;
  end;
end;

//------------------------------------------------------------------------------

procedure TLightSource.RenderLensFlares(from, at: TAffineVector; near_clip: TGLFloat);

const
  global_scale = 0.5;
  MinDot = 1e-20;

var
  view_dir, tmp,
  light_dir,
  pos, LightPos,
  dx, dy, center,
  axis, sx, sy: TAffineVector;
  dot: Extended;
  I: Integer;
  NewFrom, NewAt: TAffineVector;
  LightColor: TAffineVector;

begin
  // determine current light position
  LightPos := MakeAffineVector([FGLobalMatrix[3, 0], FGLobalMatrix[3, 1], FGLobalMatrix[3, 2]]);
  // take out camera influence
  Newat := VectorAffineSubtract(at, from);
  Newfrom := MakeAffineVector([0, 0, 0]);

  // view_dir = normalize(at-from)
  view_dir := VectorAffineSubtract(Newat, NewFrom);
  VectorNormalize(view_dir);

  // center = from + near_clip * view_dir
  tmp := view_dir;
  VectorScale(tmp, near_clip);
  center := VectorAffineAdd(Newfrom, tmp);

  // light_dir = normalize(light-from)
  light_dir := VectorAffineSubtract(LightPos, Newfrom);
  VectorNormalize(light_dir);

  // light = from + dot(light, view_dir) * near_clip * light_dir
  dot := VectorAffineDotProduct(light_dir, view_dir);
  tmp := light_dir;
  if Abs(Dot) < MinDot then
    if Dot < 0 then Dot := -MinDot
               else Dot := MinDot;
  VectorScale(tmp, near_clip / dot);
  LightPos := VectorAffineAdd(Newfrom, tmp);

  // axis = light - center
  axis := VectorAffineSubtract(LightPos, center);
  dx := axis;

  // dx = normalize(axis)
  VectorNormalize(dx);

  // dy = cross(dx, view_dir)
  dy := VectorCrossProduct(dx, view_dir);

  //glPushAttrib(GL_ENABLE_BIT or GL_CURRENT_BIT or GL_LIGHTING_BIT or GL_LINE_BIT
  //  or GL_COLOR_BUFFER_BIT or GL_TEXTURE_BIT);
  glPushAttrib(GL_ALL_ATTRIB_BITS);
  Scene.CurrentViewer.UnnecessaryState([stDepthTest, stLighting]);
  Scene.CurrentViewer.RequestedState([stBlend, stTexture2D]);
  glBlendFunc(GL_ONE, GL_ONE);
  glLoadIdentity;

  for I := 0 to LensFlares.Count - 1 do
    with LensFlares do
    begin
      sx := dx;
      VectorScale(sx, Flare[I].Scale * global_scale);
      sy := dy;
      VectorScale(sy, Flare[I].Scale * global_scale);

      //glColor3fv(LensFlares.Flare[I].ColorAddr);
      LightColor := MakeAffineVector(Diffuse.Color);
      LightColor := VectorAffineAdd(LightColor, LensFlares.Flare[I].Color);;
      VectorScale(LightColor, 0.5);
      glColor3fv(@LightColor);
      if Flare[I].FlareType < 0 then
      begin
        glBindTexture(GL_TEXTURE_2D, ShineTexture[FlareTic]);
        FlareTic := (FlareTic + 1) mod 10;
      end
      else
        glBindTexture(GL_TEXTURE_2D, FlareTexture[Flare[I].FlareType]);

      // position = center + flare[i].loc * axis
      tmp := axis;
      VectorScale(tmp, Flare[I].Location);
      Pos := VectorAffineAdd(center, tmp);

      glBegin(GL_QUADS);
        glTexCoord2f(0, 0);
        tmp := VectorAffineAdd(Pos, sx);
        tmp := VectorAffineAdd(tmp, sy);
        glVertex3fv(@tmp);

        glTexCoord2f(1, 0);
        tmp := VectorAffineSubtract(Pos, sx);
        tmp := VectorAffineAdd(tmp, sy);
        glVertex3fv(@tmp);

        glTexCoord2f(1, 1);
        tmp := VectorAffineSubtract(Pos, sx);
        tmp := VectorAffineSubtract(tmp, sy);
        glVertex3fv(@tmp);

        glTexCoord2f(0, 1);
        tmp := VectorAffineAdd(Pos, sx);
        tmp := VectorAffineSubtract(tmp, sy);
        glVertex3fv(@tmp);
      glEnd;
    end;
  Scene.CurrentViewer.RequestedState([stDepthTest, stLighting]);
  Scene.CurrentViewer.UnnecessaryState([stBlend, stTexture2D]);
  glPopAttrib;
end;

//------------------------------------------------------------------------------

procedure TLightSource.ValidateTransformation;

// calculate own global matrix and let the children do the same with their's

begin
  // check the predecessor and its transformation state
  if assigned(FParent) then
    // has the object or its parent a new local/global matrix?
    if ocTransformation in FChanges + FParent.FChanges then
    begin
      FGlobalMatrix := MatrixMultiply(CreateTranslationMatrix(FPosition.FCoords), FParent.FGlobalMatrix);
      Include(FChanges, ocTransformation);
    end else
  else
  begin
    FGlobalMatrix := CreateTranslationMatrix(FPosition.FCoords);
    Include(FChanges, ocTransformation);
  end;
  // now let the children validate their matrices
  inherited ValidateTransformation;
end;

//------------------ TGLScene --------------------------------------------------

constructor TGLScene.Create(AOwner: TComponent);
                                                             
begin
  inherited Create(AOwner);
  // root creation
  FObjects := TCustomSceneObject.Create(Self);
  FObjects.Name := 'ObjectRoot';
  FObjects.FScene := Self;
  FCameras := TBaseSceneObject.Create(Self);
  FCameras.Name := 'CameraRoot';
  FCameras.FScene := Self;
  FLights := TList.Create;
  // actual maximum number of lights is stored in TSceneViewer
  FLights.Count := 8;
end;

//------------------------------------------------------------------------------

destructor TGLScene.Destroy;

begin
  FCameras.Free;
  FLights.Free;
  FObjects.Free;
  inherited Destroy;
end;

//------------------------------------------------------------------------------

procedure TGLScene.AddLight(ALight: TLightSource);

var
  I: Integer;

begin
  for I := 0 to FLights.Count - 1 do
    if FLights[I] = nil then
    begin
      FLights[I] := ALight;
      ALight.FLightID := GL_LIGHT0 + I;
      Break;
    end;
end;

//------------------------------------------------------------------------------

procedure TGLScene.AddViewer(AViewer: TSceneViewer);

begin
  if FViewers = nil then FViewers := TList.Create;
  if FViewers.IndexOf(AViewer) < 0 then FViewers.Add(AViewer);
  if FBaseContext = 0 then
  begin
    FBaseContext := TSceneViewer(FViewers[0]).RenderingContext;
    // the following initialization should be conditionally made to avoid adding
    // unnecessary overhead to the application
    if spLensFlares in TSceneViewer(FViewers[0]).FSpecials then InitLensFlares;
    if spLandScape in TSceneViewer(FViewers[0]).FSpecials then InitLandScape;
  end;
  if FViewers.Count > 1 then
    wglShareLists(FBaseContext, AViewer.RenderingContext);
end;

//------------------------------------------------------------------------------

procedure TGLScene.GetChildren(AProc: TGetChildProc; Root: TComponent);

begin
  FObjects.GetChildren(AProc, Root);
  FCameras.GetChildren(AProc, Root);
end;

//------------------------------------------------------------------------------

procedure TGLScene.RemoveLight(ALight: TLightSource);

var LIndex: Integer;

begin
  LIndex := FLights.IndexOf(ALight);
  if LIndex > -1 then FLights[LIndex] := nil;
end;

//------------------------------------------------------------------------------

procedure TGLScene.SetChildOrder(AChild: TComponent; Order: Integer);

begin
  (AChild as TBaseSceneObject).Index := Order;
end;

//------------------------------------------------------------------------------

procedure TGLScene.Loaded;

begin
  inherited Loaded;
end;

//------------------------------------------------------------------------------

function TGLScene.IsUpdating: Boolean;

begin
  Result := (FUpdateCount <> 0) or (csLoading in ComponentState) or (csDestroying in ComponentState);
end;

//------------------------------------------------------------------------------

procedure TGLScene.BeginUpdate;

begin
  Inc(FUpdateCount);
end;

//------------------------------------------------------------------------------

procedure TGLScene.RenderScene(AViewer: TSceneViewer);

begin
  FCurrentViewer := AViewer;
  FObjects.Render;
end;

//------------------------------------------------------------------------------

procedure TGLScene.EndUpdate;

begin
  if FUpdateCount > 0 then Dec(FUpdateCount);
  if FUpdateCount = 0 then NotifyChange;
end;

//------------------------------------------------------------------------------

procedure TGLScene.RemoveViewer(AViewer: TSceneViewer);

begin
  if assigned(FViewers) then
  begin
    // we need a new base context to share with if the previous base context
    // is about to be destroyed
    if (FViewers.IndexOf(AViewer) = 0) and (FViewers.Count > 1) then
       FBaseContext := TSceneViewer(FViewers[1]).RenderingContext;
    // if AViewer is the last one in the list then remove other
    // shared stuff before (!) the viewer is deleted
    if FViewers.Count = 1 then
    begin
      LensFlares.Free;
      LandScape.Free;
    end;
    FViewers.Remove(AViewer);
    if FViewers.Count = 0 then
    begin
      FViewers.Free;
      FViewers := nil;
      FBaseContext := 0;
      if not (csDestroying in ComponentState) then
      begin
        FObjects.DestroyList;
        FObjects.ReloadTexture;
      end;  
    end;  
  end;
end;

//------------------------------------------------------------------------------

procedure TGLScene.ValidateTransformation(ACamera: TCamera);

begin
  FCurrentCamera := ACamera;
  ACamera.Apply;
  FObjects.ValidateTransformation;
  Exclude(ACamera.FChanges, ocTransformation);
  FLastCamera := FCurrentCamera;
end;

//------------------------------------------------------------------------------

procedure TGLScene.NotifyChange;

var
  I: Integer;

begin
  if (not IsUpdating) and assigned(FViewers) then
    for I := 0 to FViewers.Count - 1 do TSceneViewer(FViewers[I]).Invalidate;
end;

//------------------------------------------------------------------------------

procedure TGLScene.SetupLights(Maximum: Integer);

{$J+}
const
  NullPosition: TVector = (0, 0, 0, 0);
{$J-}

var
  I: Integer;
  LS: TLightSource;
  Max: Integer;

begin
  // start searching through all light sources
  if Maximum < FLights.Count then Max := Maximum
                             else Max := FLights.Count;
  for I := 0 to Max - 1 do
  begin
    LS := TLightSource(FLights[I]);
    if assigned(LS) and LS.Shining then
    with LS do
    begin
      glEnable(FLightID);
      glLoadMatrixf(@FGlobalMatrix);
      NullPosition[3] := Position.W;
      glLightfv(FLightID, GL_POSITION, @NullPosition);
      with FAmbient  do if not IsInherited then glLightfv(FLightID, GL_AMBIENT, AsAddress);
      with FDiffuse  do if not IsInherited then glLightfv(FLightID, GL_DIFFUSE, AsAddress);
      with FSpecular do if not IsInherited then glLightfv(FLightID, GL_SPECULAR, AsAddress);
      if ocSpot in FChanges then
      begin
        glLightfv(FLightID, GL_SPOT_DIRECTION, @FSpotDirection.FCoords);
        glLightfv(FLightID, GL_SPOT_EXPONENT, @FSpotExponent);
        glLightfv(FLightID, GL_SPOT_CUTOFF, @FSpotCutOff);
        Exclude(FChanges, ocSpot);
      end;
      if ocAttenuation in FChanges then
      begin
        glLightfv(FLightID, GL_CONSTANT_ATTENUATION, @FConstAttenuation);
        glLightfv(FLightID, GL_LINEAR_ATTENUATION, @FLinearAttenuation);
        glLightfv(FLightID, GL_QUADRATIC_ATTENUATION, @FQuadraticAttenuation);
        Exclude(FChanges, ocAttenuation);
      end;
    end
    else glDisable(GL_LIGHT0 + I);
  end;
end;

//------------------------------------------------------------------------------

procedure TGLScene.DoAfterRender;

var
  I: Integer;

begin
  for I := 0 to FLights.Count-1 do
    if assigned(FLights[I]) and TLightSource(FLights[I]).Shining then
      TLightSource(FLights[I]).RenderLensFlares(MakeAffineVector(CurrentCamera.Position.FCoords), 
                                                MakeAffineVector(CurrentCamera.FDirection.FCoords), 
                                                CurrentViewer.FCamera.FNearPlane);
end;

//------------------ TGLJoystick ---------------------------------------------------

const
  JoystickIDToNative: array[NoJoystick..Joystick2] of Byte = (9, JOYSTICKID1, JOYSTICKID2);

constructor TGLJoystick.Create(AOwner: TSceneViewer);

begin
  FOwner := AOwner;
  FInterval := 100;
  FThreshold := 100;
  FJoystickID := NoJoystick;
  FLastX := -1;
  FLastY := -1;
  FLastZ := -1;
end;

//------------------------------------------------------------------------------

procedure TGLJoystick.Assign(Source: TPersistent);

begin
  if Source is TGLJoystick then
  begin
    FInterval := TGLJoystick(Source).FInterval;
    FThreshold := TGLJoystick(Source).FThreshold;
    FCapture := TGLJoystick(Source).FCapture;
    FJoystickID := TGLJoystick(Source).FJoystickID;
    try
      ReapplyCapture(FJoystickID);
    except
      FJoystickID := NoJoystick;
      FCapture := False;
      raise;
    end;
  end
  else inherited Assign(Source);
end;

//------------------------------------------------------------------------------

function DoScale(AValue: Integer): Integer;

begin
  Result := Round(AValue/1);
end;

//------------------------------------------------------------------------------

procedure TGLJoystick.ReapplyCapture(AJoystick: TJoystickID);

var JC: TJoyCaps;

begin
  DoJoystickRelease(AJoystick);
  if FCapture then
    with JC do
    begin
      joyGetDevCaps(JoystickIDToNative[FJoystickID], @JC, SizeOf(JC));
      FNumButtons := wNumButtons;
      FMinMaxInfo[jaX, jpMin] := DoScale(wXMin);
      FMinMaxInfo[jaX, jpCenter] := DoScale((wXMin + wXMax) div 2); FMinMaxInfo[jaX, jpMax] := DoScale(wXMax);
      FMinMaxInfo[jaY, jpMin] := DoScale(wYMin); FMinMaxInfo[jaY, jpCenter] := DoScale((wYMin + wYMax) div 2); FMinMaxInfo[jaY, jpMax] := DoScale(wYMax);
      FMinMaxInfo[jaZ, jpMin] := DoScale(wZMin); FMinMaxInfo[jaZ, jpCenter] := DoScale((wZMin + wZMax) div 2); FMinMaxInfo[jaZ, jpMax] := DoScale(wZMax);
      FMinMaxInfo[jaR, jpMin] := DoScale(wRMin); FMinMaxInfo[jaR, jpCenter] := DoScale((wRMin + wRMax) div 2); FMinMaxInfo[jaR, jpMax] := DoScale(wRMax);
      FMinMaxInfo[jaU, jpMin] := DoScale(wUMin); FMinMaxInfo[jaU, jpCenter] := DoScale((wUMin + wUMax) div 2); FMinMaxInfo[jaU, jpMax] := DoScale(wUMax);
      FMinMaxInfo[jaV, jpMin] := DoScale(wVMin); FMinMaxInfo[jaV, jpCenter] := DoScale((wVMin + wVMax) div 2); FMinMaxInfo[jaV, jpMax] := DoScale(wVMax);
      DoJoystickCapture(FOwner.Handle, AJoystick)
    end;
end;

//------------------------------------------------------------------------------

procedure TGLJoystick.DoJoystickCapture(AHandle: HWND; AJoystick: TJoystickID);

begin
  try
    case joySetCapture(AHandle, JoystickIDToNative[AJoystick], FInterval, False) of
      MMSYSERR_NODRIVER:
        ShowError(glsNoJoystickDriver);
      JOYERR_UNPLUGGED:
        ShowMessage(glsConnectJoystick);
      JOYERR_NOCANDO:
        ShowError(glsJoystickError);
      JOYERR_NOERROR:
        joySetThreshold(JoystickIDToNative[AJoystick], FThreshold);
    else ShowError(glsJoystickError);
    end
  except
    on E: Exception do
    begin
      FCapture := False;
      Application.ShowException(E);
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure TGLJoystick.DoJoystickRelease(AJoystick: TJoystickID);

begin
  if AJoystick <> NoJoystick then
    joyReleaseCapture(JoystickIDToNative[AJoystick]);
end;

//------------------------------------------------------------------------------

procedure TGLJoystick.SetCapture(AValue: Boolean);

begin
  if FCapture <> AValue then
  begin
    FCapture := AValue;
    if not (csReading in FOwner.ComponentState) then
    try
      ReapplyCapture(FJoystickID);
    except
      FCapture := False;
      raise;
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure TGLJoystick.SetInterval(AValue: Cardinal);

begin
  if FInterval <> AValue then
  begin
    FInterval := AValue;
    if not (csReading in FOwner.ComponentState) then ReapplyCapture(FJoystickID);
  end;
end;

//------------------------------------------------------------------------------

procedure TGLJoystick.SetJoystickID(AValue: TJoystickID);

begin
  if FJoystickID <> AValue then
  try
    if not (csReading in FOwner.ComponentState) then ReapplyCapture(AValue);
    FJoystickID := AValue;
  except
    on E: Exception do
    begin
      ReapplyCapture(FJoystickID);
      Application.ShowException(E);
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure TGLJoystick.SetThreshold(AValue: Cardinal);

begin
  if FThreshold <> AValue then
  begin
    FThreshold := AValue;
    if not (csReading in FOwner.ComponentState) then ReapplyCapture(FJoystickID);
  end;
end;

//------------------------------------------------------------------------------

function MakeJoyButtons(Param: UINT): TJoystickButtons;

begin
  Result := [];
  if (Param and JOY_BUTTON1) > 0 then Include(Result, jbButton1);
  if (Param and JOY_BUTTON2) > 0 then Include(Result, jbButton2);
  if (Param and JOY_BUTTON3) > 0 then Include(Result, jbButton3);
  if (Param and JOY_BUTTON4) > 0 then Include(Result, jbButton4);
end;

//------------------------------------------------------------------------------

function Approximation(Data: array of Integer): Integer;

// calculate a better estimation of the last value in the given data, depending
// on the other values (used to approximate a smoother joystick movement)
//
// based on Gauss' principle of smallest squares in Maximum-Likelihood and
// linear normal equations

var
  SumX, SumY, SumXX, SumYX: Double;
  I, Comps: Integer;
  a0, a1: Double;

begin
  SumX := 0;
  SumY := 0;
  SumXX := 0;
  SumYX := 0;
  Comps := High(Data) + 1;
  for I := 0 to High(Data) do
  begin
    SumX := SumX + I;
    SumY := SumY + Data[I];
    SumXX := SumXX + I * I;
    SumYX := SumYX + I * Data[I];
  end;
  a0 := (SumY * SumXX - SumX * SumYX) / (Comps * SumXX - SumX * SumX);
  a1 := (Comps * SumYX - SumY * SumX) / (Comps * SumXX - SumX * SumX);
  Result := Round(a0 + a1 * High(Data));
end;

//------------------------------------------------------------------------------

procedure TGLJoystick.DoXYMove(Buttons: Word; XPos, YPos: Integer);

var
  JoyButtons: TJoystickButtons;
  I: Integer;
  dX, dY: Integer;

begin
  XPos := DoScale(XPos);
  YPos := DoScale(YPos);
  if (FLastX = -1) or (FLastY = -1) then
  begin
    FLastX := XPos;
    FLastY := YPos;
    for I := 0 to 4 do
    begin
      FXPosInfo[I] := XPos;
      FYPosInfo[I] := YPos;
    end;
  end
  else
  begin
    Move(FXPosInfo[1], FXPosInfo[0], 16);
    FXPosInfo[4] := XPos;
    XPos := Approximation(FXPosInfo);
    Move(FYPosInfo[1], FYPosInfo[0], 16);
    FYPosInfo[4] := YPos;
    YPos := Approximation(FYPosInfo);
    JoyButtons := MakeJoyButtons(Buttons);
    dX := Round((XPos-FMinMaxInfo[jaX, jpCenter]) * 100 / FMinMaxInfo[jaX, jpCenter]);
    dY := Round((YPos-FMinMaxInfo[jaY, jpCenter]) * 100 / FMinMaxInfo[jaY, jpCenter]);
    if assigned(FOwner.FOnJoystickMove) then FOwner.FOnJoystickMove(Self, FJoystickID, JoyButtons, dX, dY);
    FLastX := XPos;
    FLastY := YPos;
  end;
end;

//------------------------------------------------------------------------------

procedure TGLJoystick.DoZMove(Buttons: Word; ZPos: Integer);

begin
  if FLastZ = -1 then FLastZ := Round(ZPos * FOwner.FCamera.FDepthOfView / 65536)
                 else
  begin
  end;
end;

//------------------------------------------------------------------------------

procedure TGLJoystick.HandleMessage(AMessage: TMessage);

begin
  case AMessage.Msg of
    MM_JOY1MOVE, 
    MM_JOY2MOVE:
      DoXYMove(AMessage.wParam, AMessage.lParamLo, AMessage.lParamHi);
    MM_JOY1ZMOVE,
    MM_JOY2ZMOVE:
      DoZMove(AMessage.wParam, AMessage.lParamLo);
  end;
end;

//------------------ TFogEnvironment ------------------------------------------------

// Note: The fog implementation is not conformal with the rest of the scene management
//       because it is viewer bound not scene bound. 

constructor TFogEnvironment.Create(Viewer: TSceneViewer);

begin
  inherited Create;
  FSceneViewer :=  Viewer;
  FFogColor :=  TGLColor.Create(Self);
  FFogColor.Color :=  clrBlack;
  FFogColor.Alpha :=  0.8;
  FFogMode :=  fmLinear;
  FFogStart :=  10;
  FFogEnd :=  1000;
end;

//------------------------------------------------------------------------------

destructor TFogEnvironment.Destroy;

begin
  FFogColor.Free;
  inherited Destroy;
end;

//------------------------------------------------------------------------------

procedure TFogEnvironment.Change;

begin
  FChanged := True;
  FSceneViewer.Invalidate;
end;

//------------------------------------------------------------------------------

procedure TFogEnvironment.SetFogColor(Value: TGLColor);

begin
  if Assigned(Value) then
  begin
    FFogColor.Assign(Value);
    Change;
  end;
end;

//------------------------------------------------------------------------------

procedure TFogEnvironment.SetFogStart(Value: TGLfloat);

begin
  if Value <> FFogStart then
  begin
    FFogStart :=  Value;
    Change;
  end;
end;

//------------------------------------------------------------------------------

procedure TFogEnvironment.SetFogEnd(Value: TGLfloat);

begin
  if Value <> FFogEnd then
  begin
    FFogEnd :=  Value;
    Change;
  end;
end;

//------------------------------------------------------------------------------

procedure TFogEnvironment.Assign(Source: TPersistent);

begin
  if Source is TFogEnvironment then
  begin
    FFogColor.Assign(TFogEnvironment(Source).FFogColor);
    FFogStart :=  TFogEnvironment(Source).FFogStart;
    FFogEnd :=  TFogEnvironment(Source).FFogEnd;
    FFogMode :=  TFogEnvironment(Source).FFogMode;
    FChanged := True;
  end
  else
    inherited Assign(Source);
end;

//------------------------------------------------------------------------------

procedure TFogEnvironment.SetFogMode(Value: TFogMode);

begin
  if Value <> FFogMode then
  begin
    FFogMode :=  Value;
    Change;
  end;
end;

//------------------------------------------------------------------------------

procedure TFogEnvironment.ApplyFog;

begin
  if FChanged then
    with FSceneViewer do
    begin
      if FRenderingContext > 0 then
      begin
        ActivateRenderingContext(FCanvas.Handle, FRenderingContext);
        case FFogMode of
          fmLinear:
            glFogi(GL_FOG_MODE, GL_LINEAR);
          fmExp:
            glFogi(GL_FOG_MODE, GL_EXP);
          fmExp2:
            glFogi(GL_FOG_MODE, GL_EXP2);
        end;
        glFogfv(GL_FOG_COLOR, FFogColor.AsAddress);
        glFogf(GL_FOG_DENSITY, FFogColor.Alpha);
        glFogf(GL_FOG_START, FFogStart);
        glFogf(GL_FOG_END, FFogEnd);
        FChanged :=  False;
        DeactivateRenderingContext;
      end;
    end;
end;

//------------------ TSceneViewer --------------------------------------------------

constructor TSceneViewer.Create(AOwner: TComponent);

begin
  inherited Create(AOwner);
  ControlStyle := [csClickEvents, csDoubleClicks, csOpaque];
  if csDesigning in ComponentState then ControlStyle := ControlStyle + [csFramed];
  FCanvas := TCanvas.Create;
  FDisplayOptions := TDisplayOptions.Create;
  FBackground := TTexture.Create(nil);
  FBackground.ImageSource := isCapture;
  FBackground.IsInherited := False;

  // initialize private state variables
  FFogEnviroment :=  TFogEnvironment.Create(Self);
  FBackgroundColor :=  clBtnFace;
  FDepthTest :=  True;
  FFaceCulling :=  True;
  FLighting :=  True;
  FFogEnable :=  False;

  FContextOptions := [roDoubleBuffer, roRenderToWindow];
  // performance check off
  FMonitor := False;
  FFramesPerSecond := 0;
  FFrames := 0;
  FTicks := 0;
  FJoystick := TGLJoystick.Create(Self);
  FState := dsNone;
end;

//------------------------------------------------------------------------------
 
destructor TSceneViewer.Destroy;

begin
  // clean up and terminate
  if assigned (FCamera) and assigned (FCamera.FScene) then FCamera.FScene.RemoveViewer(Self);
  FCamera := nil;
  FBackground.Free;
  FDisplayOptions.Free;
  FJoystick.Free;
  DestroyHandle;
  FFogEnviroment.free;
  //FLandScapeOption.free;
  FCanvas.Free;
  inherited Destroy;
end;

//------------------------------------------------------------------------------

procedure TSceneViewer.CreateParams(var Params: TCreateParams);

begin
  inherited CreateParams(Params);
  with Params do
  begin
    if (not (csDesigning in ComponentState) and (woDesktop in FDisplayOptions.WindowAttributes)) or
       (not assigned(Parent) and (ParentWindow = 0)) then
    begin
      WndParent := 0;
      Style := WS_POPUP or WS_VISIBLE;
    end;
    //else Style := WS_CHILD;
    Style := Style or WS_CLIPCHILDREN or WS_CLIPSIBLINGS;
    if woTransparent in FDisplayOptions.WindowAttributes then ExStyle := ExStyle or WS_EX_TRANSPARENT;
    WindowClass.Style :=  CS_VREDRAW or CS_HREDRAW; // or CS_OWNDC;
  end;
end;

//------------------------------------------------------------------------------

function TSceneViewer.ObjectInScene(Obj: TBaseSceneObject): Boolean;

var
  ModelMatrix: THomogeneousDblMatrix;
  ProjectMatrix: THomogeneousDblMatrix;
  VP: THomogeneousIntVector;
  WinX, WinY, WinZ: Double;
  R: TRect;
  P: TPoint;
  
begin
  Result :=  True;
  glGetDoublev(GL_MODELVIEW_MATRIX, @ModelMatrix);
  glGetDoublev(GL_PROJECTION_MATRIX, @ProjectMatrix);
  glGetIntegerv(GL_VIEWPORT, @VP);
  gluProject(Obj.Position.X, Obj.Position.Y, Obj.Position.Z, ModelMatrix, ProjectMatrix, VP, @WinX, @WinY, @WinZ);
  R :=  Rect(Vp[0], Vp[1], Vp[2], Vp[3]);
  P.x :=  Round(WinX);
  P.y :=  Round(WinY);
  if (not PtInRect(R, P)) then
    Result :=  False;
end;

//------------------------------------------------------------------------------

procedure TSceneViewer.ReadContextProperties;

begin
  FMaxLightSources := LimitOf[limLights];
  FDoubleBuffered := LimitOf[limDoubleBuffer] > 0;
  if glIsEnabled(GL_DEPTH_TEST) > 0 then Include(FCurrentStates, stDepthTest);
  if glIsEnabled(GL_CULL_FACE) > 0 then Include(FCurrentStates, stCullFace);
  if glIsEnabled(GL_LIGHTING) > 0 then Include(FCurrentStates, stLighting);
  if glIsEnabled(GL_FOG) > 0 then Include(FCurrentStates, stFog);
end;

//------------------------------------------------------------------------------

procedure TSceneViewer.SetupRenderingContext;

var
  ColorDepth: Cardinal;
  NewStates: TGLStates;

begin
  ColorDepth := GetDeviceCaps(Canvas.Handle, BITSPIXEL) * GetDeviceCaps(Canvas.Handle, PLANES);
  if roTwoSideLighting in FContextOptions then glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE)
                                          else glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE);
  NewStates := [stNormalize];
  if DepthTest then Include(NewStates, stDepthTest);
  if FaceCulling then Include(NewStates, stCullFace);
  if Lighting then Include(NewStates, stLighting);
  if FogEnable then Include(NewStates, stFog);
  if ColorDepth < 24 then Include(NewStates, stDither)
                     else Exclude(NewStates, stDither);
  RequestedState(NewStates);
  glDepthFunc(GL_LESS);
  glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
end;

//------------------------------------------------------------------------------

function TSceneViewer.GetLimit(Which: TLimitType): Integer;

// determine the limit of the given kind from OpenGL implementation

var
  VP: array[0..1] of Double;

begin
  case Which of
    limClipPlanes:
      glGetIntegerv(GL_MAX_CLIP_PLANES, @Result);
    limEvalOrder:
      glGetIntegerv(GL_MAX_EVAL_ORDER, @Result);
    limLights:
      glGetIntegerv(GL_MAX_LIGHTS, @Result);
    limListNesting:
      glGetIntegerv(GL_MAX_LIST_NESTING, @Result);
    limModelViewStack:
      glGetIntegerv(GL_MAX_MODELVIEW_STACK_DEPTH, @Result);
    limNameStack:
      glGetIntegerv(GL_MAX_NAME_STACK_DEPTH, @Result);
    limPixelMapTable:
      glGetIntegerv(GL_MAX_PIXEL_MAP_TABLE, @Result);
    limProjectionStack:
      glGetIntegerv(GL_MAX_PROJECTION_STACK_DEPTH, @Result);
    limTextureSize:
      glGetIntegerv(GL_MAX_TEXTURE_SIZE, @Result);
    limTextureStack:
      glGetIntegerv(GL_MAX_TEXTURE_STACK_DEPTH, @Result);
    limViewportDims:
      begin
        glGetDoublev(GL_MAX_VIEWPORT_DIMS, @VP);
        Result := Round(MaxValue(VP));
      end;
    limAccumAlphaBits:
      glGetIntegerv(GL_ACCUM_ALPHA_BITS, @Result);
    limAccumBlueBits:
      glGetIntegerv(GL_ACCUM_BLUE_BITS, @Result);
    limAccumGreenBits:
      glGetIntegerv(GL_ACCUM_GREEN_BITS, @Result);
    limAccumRedBits:
      glGetIntegerv(GL_ACCUM_RED_BITS, @Result);
    limAlphaBits:
      glGetIntegerv(GL_ALPHA_BITS, @Result);
    limAuxBuffers:
      glGetIntegerv(GL_AUX_BUFFERS, @Result);
    limDepthBits:
      glGetIntegerv(GL_DEPTH_BITS, @Result);
    limStencilBits:
      glGetIntegerv(GL_STENCIL_BITS, @Result);
    limBlueBits:
      glGetIntegerv(GL_BLUE_BITS, @Result);
    limGreenBits:
      glGetIntegerv(GL_GREEN_BITS, @Result);
    limRedBits:
      glGetIntegerv(GL_RED_BITS, @Result);
    limIndexBits:
      glGetIntegerv(GL_INDEX_BITS, @Result);
    limStereo:
      glGetIntegerv(GL_STEREO, @Result);
    limDoubleBuffer:
      glGetIntegerv(GL_DOUBLEBUFFER, @Result);
    limSubpixelBits:
      glGetIntegerv(GL_SUBPIXEL_BITS, @Result);
  else
    Result := 0;
  end;
end;

//------------------------------------------------------------------------------

procedure TSceneViewer.Loaded;

var
  NewMode: Integer;

begin
  inherited Loaded;
  if not (csDesigning in ComponentState) then
    // set display mode depending on the different screen options
    // full screen requested?
    if DisplayOptions.FullScreen then
    begin
      // full screen mode, so check for window fitting
      case DisplayOptions.WindowFitting of
        wfFitWindowToScreen: // set screen to the specified size
          begin
            NewMode := DisplayOptions.ScreenResolution;
            if NewMode <> 0 then SetFullScreenMode(NewMode);
          end;
        wfFitScreenToWindow: // adjust screen size to window size
          begin
            NewMode := GetIndexFromResolution(Width, Height, VideoModes[0].ColorDepth);
            SetFullScreenMode(NewMode);
          end;
      end;
      Left := 0;
      Top := 0;
      ShowWindow(Handle, SW_SHOWMAXIMIZED);
    end
    else
      // no full screen mode for the application, but perhaps
      // a specific resolution or color depth?
      if DisplayOptions.ScreenResolution <> 0 then
        SetFullScreenMode(DisplayOptions.ScreenResolution);
  // initiate first draw
  FOwnRefresh := False;
  // initiate window creation
  HandleNeeded;
  if assigned(FCamera) and assigned(FCamera.FScene) then FCamera.FScene.AddViewer(Self);
end;

//------------------------------------------------------------------------------

procedure TSceneViewer.WMEraseBkgnd(var Message: TWMEraseBkgnd);

begin
  Message.Result := 1;
end;

//------------------------------------------------------------------------------

procedure TSceneViewer.WMSize(var Message: TWMSize);

var
  APoint: TPoint;

begin
  inherited;
  if (woTransparent in DisplayOptions.WindowAttributes) then
  begin
    APoint.X := Left;
    APoint.Y := Top;
    if not (woDesktop in DisplayOptions.WindowAttributes) or
       (csDesigning in ComponentState)                    then
       APoint := Parent.ClientToScreen(APoint);

    TCaptureImage(FBackground.Image).Left := APoint.X;
    TCaptureImage(FBackground.Image).Top := APoint.Y;
    FBackground.Image.Width := FBackground.RoundUpToPowerOf2(Message.Width);
    FBackground.Image.Height := FBackground.RoundUpToPowerOf2(Message.Height);
  end;
  // define viewport
  if FRenderingContext <> 0 then
  begin
    ActivateRenderingContext(FCanvas.Handle, FRenderingContext);
    with FViewPort do
    begin
      Width := Message.Width;
      Height := Message.Height;
      if Height = 0 then Height := 1;
      glViewport(0, 0, Width, Height);
    end;
    DeactivateRenderingContext;
  end;
end;

//------------------------------------------------------------------------------

procedure TSceneViewer.WMPaint(var Message: TWMPaint);

var
  PS: TPaintStruct;

begin
  BeginPaint(Handle, PS);
  Render;
  EndPaint(Handle, PS);
  Message.Result := 0;
end;

//------------------------------------------------------------------------------

procedure TSceneViewer.RenderToFile(const AFile: TFileName; DPI: Integer);

var
  ABitmap: TBitmap;
  SaveDialog: TSavePictureDialog;
  SaveAllowed: Boolean;
  FName: String;

begin
  if FState = dsNone then
  begin
    SaveDialog := nil;
    ABitmap := TBitmap.Create;
    try
      ABitmap.Width := Width;
      ABitmap.Height := Height;
      ABitmap.PixelFormat := pf24Bit;
      RenderToBitmap(ABitmap, DPI);
      FState := dsRendering;
      FName := AFile;
      SaveAllowed := True;
      if FName = '' then
      begin
        SaveDialog := TSavePictureDialog.Create(Application);
        with SaveDialog do
        begin
          Options := [ofHideReadOnly, ofNoReadOnlyReturn];
          SaveAllowed := Execute;
        end;
      end;
      if SaveAllowed then
      begin
        if FName = '' then
        begin
          FName := SaveDialog.FileName;
          if (FileExists(SaveDialog.FileName)) then
            SaveAllowed := MessageDlg(Format('Overwrite file %s?', [SaveDialog.FileName]), mtConfirmation, [mbYes, mbNo], 0) = mrYes;
        end;
        if SaveAllowed then ABitmap.SaveToFile(FName);
      end;
    finally
      SaveDialog.Free;
      ABitmap.Free;
      FState := dsNone;
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure TSceneViewer.Invalidate;

begin
  FOwnRefresh := True;
  inherited Invalidate;
end;

//------------------------------------------------------------------------------

procedure TSceneViewer.SetViewPort(X, Y, W, H: Integer);

begin
  with FViewPort do
  begin
    Left := X;
    Top := Y;
    Width := W;
    Height := H;
  end;
  Perform(WM_SIZE, SIZE_RESTORED, MakeLong(Width, Height));
  if not (csReading in ComponentState) then Invalidate;
end;

//------------------------------------------------------------------------------

procedure SetupStates(NeededStates: TGLStates);

begin
  if stAlphaTest      in NeededStates then
    glEnable(GL_ALPHA_TEST);
  if stAutoNormal     in NeededStates then
    glEnable(GL_AUTO_NORMAL);
  if stBlend          in NeededStates then
    glEnable(GL_BLEND);
  if stColorMaterial  in NeededStates then
    glEnable(GL_COLOR_MATERIAL);
  if stCullFace       in NeededStates then
    glEnable(GL_CULL_FACE);
  if stDepthTest      in NeededStates then
    glEnable(GL_DEPTH_TEST);
  if stDither         in NeededStates then
    glEnable(GL_DITHER);
  if stFog            in NeededStates then
    glEnable(GL_FOG);
  if stLighting       in NeededStates then
    glEnable(GL_LIGHTING);
  if stLineSmooth     in NeededStates then
    glEnable(GL_LINE_SMOOTH);
  if stLineStipple    in NeededStates then
    glEnable(GL_LINE_STIPPLE);
  if stLogicOp        in NeededStates then
    glEnable(GL_LOGIC_OP);
  if stNormalize      in NeededStates then
    glEnable(GL_NORMALIZE);
  if stPointSmooth    in NeededStates then
    glEnable(GL_POINT_SMOOTH);
  if stPolygonSmooth  in NeededStates then
    glEnable(GL_POLYGON_SMOOTH);
  if stPolygonStipple in NeededStates then
    glEnable(GL_POLYGON_STIPPLE);
  if stScissorTest    in NeededStates then
    glEnable(GL_SCISSOR_TEST);
  if stStencilTest    in NeededStates then
    glEnable(GL_STENCIL_TEST);
  if stTexture1D      in NeededStates then
    glEnable(GL_TEXTURE_1D);
  if stTexture2D      in NeededStates then
    glEnable(GL_TEXTURE_2D);
end;

//------------------------------------------------------------------------------

procedure TSceneViewer.RenderToBitmap(ABitmap: TBitmap; DPI: Integer);

var
  BitmapContext: HGLRC;
  BackColor: TColorVector;
  ColorBits: Integer;
  Viewport: TRectangle;
  LastStates: TGLStates;
  Resolution: Integer;

begin
  if FState = dsNone then
  begin
    FState := dsPrinting;
    case ABitmap.PixelFormat of
      pfCustom,  // use current color depth
      pfDevice:
        ColorBits := VideoModes[CurrentVideoMode].ColorDepth;
      pf1bit,    // OpenGL needs at least 4 bits
      pf4bit:
        ColorBits := 4;
      pf8bit:
        ColorBits := 8;
      pf15bit:
        ColorBits := 15;
      pf16bit:
        ColorBits := 16;
      pf24bit:
        ColorBits := 24;
      pf32bit:
        ColorBits := 32;
    else
      ColorBits := 24;
    end;
    BitmapContext := CreateRenderingContext(ABitmap.Canvas.Handle, [], ColorBits, 0, 0, 0, 0);
    if BitmapContext = 0 then ShowError(glsWrongBitmapCanvas);
    try
      ActivateRenderingContext(ABitmap.Canvas.Handle, BitmapContext);
      // save current window context states
      LastStates := FCurrentStates;
      SetupStates(FCurrentStates);
      if roTwoSideLighting in FContextOptions then glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE)
                                              else glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE);
      glEnable(GL_NORMALIZE);
      BackColor := ConvertWinColor(FBackgroundColor);
      glClearColor(BackColor[0], BackColor[1], BackColor[2], BackColor[3]);

      if (woTransparent in DisplayOptions.WindowAttributes) then FBackground.ReloadImage;
      // set the desired viewport and limit output to this rectangle
      with Viewport do
      begin
        Left := 0;
        Top := 0;
        Width := ABitmap.Width;
        Height := ABitmap.Height;
        glViewport(Left, Top, Width, Height);
      end;

      ClearBuffers;

      glMatrixMode(GL_PROJECTION);
      glLoadIdentity;
      Resolution := DPI;
      if Resolution = 0 then Resolution := GetDeviceCaps(ABitmap.Canvas.Handle, LOGPIXELSX);
      FCamera.ApplyPerspective(Viewport, ABitmap.Width, ABitmap.Height, Resolution);

      glMatrixMode(GL_MODELVIEW);
      glLoadIdentity;
      if assigned(FBeforeRender) then FBeforeRender(Self);

      if assigned(FCamera) and assigned(FCamera.FScene) then
      with FCamera.FScene do
      begin
        Camera.Scene.ValidateTransformation(Camera);
        SetupLights(FMaxLightSources);
        with Camera.Scene do
        begin
          Objects.ReloadTexture;
          Objects.DestroyList;
          FogEnvironment.ApplyFog;
          RenderScene(Self);
          Objects.DestroyList;
        end;
      end;

      if assigned(FAfterRender) then
        FAfterRender(Self);
      glFinish;
    finally
      DestroyRenderingContext(BitmapContext);
      FCurrentStates := LastStates;
      FState := dsNone;
    end;
  end;  
end;

//------------------------------------------------------------------------------

procedure TSceneViewer.RequestedState(States: TGLStates);

var
  NeededStates: TGLStates;

begin
  // create window and rendering context if not yet done
  HandleNeeded;

  // get all states, which are requested but not yet set
  NeededStates := States - FCurrentStates;
  if NeededStates <> [] then
  begin
    SetupStates(NeededStates);
    FCurrentStates := FCurrentStates + NeededStates;
  end;
end;

//------------------------------------------------------------------------------

procedure TSceneViewer.ShowInfo;

begin
  // most info is available with active context only
  Application.CreateForm(TInfoForm, InfoForm);
  try
    ActivateRenderingContext(Canvas.Handle, FRenderingContext);
    InfoForm.GetInfoFrom(Self);
    DeactivateRenderingContext;
    InfoForm.ShowModal;
  finally
    InfoForm.Free;
  end;
end;

//------------------------------------------------------------------------------

procedure TSceneViewer.UnnecessaryState(States: TGLStates);

var
  TakeOutStates: TGLStates;

begin
  // create window and rendering context if not yet done
  HandleNeeded;

  // get all states, which are to be taken out, but still set
  TakeOutStates := States * FCurrentStates;
  if TakeOutStates <> [] then
  begin
    // now reset all these states
    if stAlphaTest      in TakeOutStates then
      glDisable(GL_ALPHA_TEST);
    if stAutoNormal     in TakeOutStates then
      glDisable(GL_AUTO_NORMAL);
    if stBlend          in TakeOutStates then
      glDisable(GL_BLEND);
    if stColorMaterial  in TakeOutStates then
      glDisable(GL_COLOR_MATERIAL);
    if stCullFace       in TakeOutStates then
      glDisable(GL_CULL_FACE);
    if stDepthTest      in TakeOutStates then
      glDisable(GL_DEPTH_TEST);
    if stDither         in TakeOutStates then
      glDisable(GL_DITHER);
    if stFog            in TakeOutStates then
      glDisable(GL_FOG);
    if stLighting       in TakeOutStates then
      glDisable(GL_LIGHTING);
    if stLineSmooth     in TakeOutStates then
      glDisable(GL_LINE_SMOOTH);
    if stLineStipple    in TakeOutStates then
      glDisable(GL_LINE_STIPPLE);
    if stLogicOp        in TakeOutStates then
      glDisable(GL_LOGIC_OP);
    if stNormalize      in TakeOutStates then
      glDisable(GL_NORMALIZE);
    if stPointSmooth    in TakeOutStates then
      glDisable(GL_POINT_SMOOTH);
    if stPolygonSmooth  in TakeOutStates then
      glDisable(GL_POLYGON_SMOOTH);
    if stPolygonStipple in TakeOutStates then
      glDisable(GL_POLYGON_STIPPLE);
    if stScissorTest    in TakeOutStates then
      glDisable(GL_SCISSOR_TEST);
    if stStencilTest    in TakeOutStates then
      glDisable(GL_STENCIL_TEST);
    if stTexture1D      in TakeOutStates then
      glDisable(GL_TEXTURE_1D);
    if stTexture2D      in TakeOutStates then
      glDisable(GL_TEXTURE_2D);

    FCurrentStates := FCurrentStates - TakeOutStates;
  end;
end;

//------------------------------------------------------------------------------

procedure TSceneViewer.ClearBuffers;

// Clear all allocated OpenGL buffers. The color buffer is a special case, because
// transparency must be simulated if required.

type
  PPixelArray  = ^TByteVector;

var
  BufferBits: TGLBitfield;

begin
  // handle transparency simulation
  if (woTransparent in DisplayOptions.WindowAttributes) then
  begin
    glPushAttrib(GL_ENABLE_BIT);
    glEnable(GL_TEXTURE_2D);
    glDisable(GL_LIGHTING);
    glDisable(GL_DITHER);
    glDisable(GL_DEPTH_TEST);
    glDisable(GL_BLEND);
    // Invalidate initiated by the scene itself?
//    if not FOwnRefresh then FBackground.Image.Invalidate;
    FBackground.DisableAutoTexture;
    FBackground.Apply;
    glMatrixMode(GL_MODELVIEW);
    glPushMatrix;
    glLoadIdentity;
    glMatrixMode(GL_PROJECTION);
    glPushMatrix;
    glLoadIdentity;
    glOrtho(0, Width - 1, 0, Height - 1, 0, 100);
    glFrontFace(GL_CCW);
    glBegin(GL_QUADS);
      glTexCoord2f(0, 1 - Height / FBackground.Image.Height);
      glVertex3f(0, 0, 0);

      glTexCoord2f(Width / FBackground.Image.Width, 1 - Height / FBackground.Image.Height);
      glVertex3f(Width - 1, 0, 0);

      glTexCoord2f(Width/FBackground.Image.Width, 1);
      glVertex3f(Width - 1, Height - 1, 0);

      glTexCoord2f(0, 1);
      glVertex3f(0, Height - 1, 0);
    glEnd;
    glMatrixMode(GL_MODELVIEW);
    glPopMatrix;
    glMatrixMode(GL_PROJECTION);
    glPopMatrix;
    glPopAttrib;
  end;
  FOwnRefresh := False;

  BufferBits := 0;
  if (buColor in Buffers) and not (woTransparent in DisplayOptions.WindowAttributes) then
    BufferBits := GL_COLOR_BUFFER_BIT;
  if buDepth in Buffers then BufferBits :=  BufferBits or GL_DEPTH_BUFFER_BIT;
  if buStencil in Buffers then BufferBits :=  BufferBits or GL_STENCIL_BUFFER_BIT;
  if buAccum   in Buffers then BufferBits := BufferBits or GL_ACCUM_BUFFER_BIT;
  if BufferBits <> 0 then glClear(BufferBits);
end;

//------------------------------------------------------------------------------

procedure TSceneViewer.Notification(AComponent: TComponent; Operation: TOperation);

begin
  inherited;
  if (Operation = opRemove) and (AComponent = FCamera) then Camera := nil;
end;

//------------------------------------------------------------------------------

procedure TSceneViewer.PickObjects(var Rect: TRect; PickList: TPickList; ObjectCountGuess: Integer);

type
  PCardinalVector = ^TCardinalVector;
  TCardinalVector = array[0..0] of Cardinal;

var
  Buffer: PCardinalVector;
  BufferSize,
  BufferSizeGuess: Integer;
  Hits: Integer;
  I: Integer;
  Current,
  Next: Cardinal;
  szmin, szmax: Single;

begin
  if (FState = dsNone) and assigned(PickList) then
  try
    FState := dsPicking;
    ActivateRenderingContext(FCanvas.Handle, FRenderingContext);
    glMatrixMode(GL_PROJECTION);
    glPushMatrix;
    glLoadIdentity;
    gluPickMatrix(Rect.Left, Height - Rect.Top, Abs(Rect.Right - Rect.Left), Abs(Rect.Bottom - Rect.Top), TVector4i(FViewport));
    FCamera.ApplyPerspective(FViewport, Width, Height, GetDeviceCaps(Canvas.Handle, LOGPIXELSX));
    Buffer := nil;
    BufferSize := 0;
    BufferSizeGuess := Abs(ObjectCountGuess) * 4; // assume 4 entries per hit
    repeat
      if BufferSizeGuess > BufferSize then
      begin
        BufferSize := BufferSizeGuess;
        // Add 32 integers of slop (an extra cache line) to end for buggy
        // hardware that uses DMA to return select results but that sometimes
        // overrun the buffer.  Yuck.
        ReallocMem(Buffer, BufferSize * SizeOf(Integer) + 32 * 4);
      end;

      glSelectBuffer(BufferSize, @Buffer^);
      glRenderMode(GL_SELECT);
      glInitNames;
      glPushName(0);

      glMatrixMode(GL_MODELVIEW);
      glLoadIdentity;

      if assigned(FCamera) and assigned(FCamera.FScene) then
      with FCamera.FScene do
      begin
        Camera.Scene.ValidateTransformation(Camera);
        Camera.Scene.RenderScene(Self);
      end;
      glFlush;
      Hits := glRenderMode(GL_RENDER);
      if Hits < 0 then BufferSizeGuess := BufferSize + (BufferSize shr 1);
    until Hits > -1; // try again with larger selection buffer

    Next := 0;
    PickList.Clear;
    PickList.Capacity := Hits;
    for I := 0 to Hits-1 do
    begin
      Current := Next;
      Next := Current + Buffer[Current] + 3;
      szmin := (Buffer[current + 1] shr 1) / MaxInt;
      szmax := (Buffer[current + 2] shr 1) / MaxInt;
      PickList.AddHit(TCustomSceneObject(Buffer[Current + 3]), szmin, szmax);
    end;
  finally
    FreeMem(Buffer);
    glMatrixMode(GL_PROJECTION);
    glPopMatrix;
    DeactivateRenderingContext;
    FState := dsNone;
  end;
end;

//------------------------------------------------------------------------------

function TSceneViewer.GetLandsHeight(X, Y: TGLfloat): TGLfloat;

begin
  Result :=  LandScape.Get_Height(X, Y);
end;

//------------------------------------------------------------------------------

procedure TSceneViewer.Render;

var
  Counter1,
  Counter2: TLargeInteger;

begin
  if (FState <> dsNone) or not visible then Exit;
  ActivateRenderingContext(FCanvas.Handle, FRenderingContext);
  FState := dsRendering;
  try
    // performance data demanded?
    if FMonitor then QueryPerformanceCounter(Counter1);

    // clear the buffers
    ClearBuffers;

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity;
    if assigned(FCamera) then
      FCamera.ApplyPerspective(FViewport, Width, Height, GetDeviceCaps(Canvas.Handle, LOGPIXELSX));

    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity;
    if assigned(FBeforeRender) then
      FBeforeRender(Self);
    if assigned(FCamera) and assigned(FCamera.FScene) then
    with FCamera.FScene do
    begin
      FCamera.Scene.ValidateTransformation(FCamera);
      if spLandScape in FSpecials then
        LandScape.RenderLandScape(FCamera);
      SetupLights(FMaxLightSources);
      FogEnvironment.ApplyFog;
      FCamera.Scene.RenderScene(Self);
    end;

    if assigned(FAfterRender) then
      FAfterRender(Self);
    if spLensFlares in FSpecials then
      if assigned(FCamera) and assigned(FCamera.FScene) then
        Camera.Scene.DoAfterRender;
    glFlush;
    if FDoubleBuffered then SwapBuffers(Canvas.Handle);
    {$ifdef DEBUG} CheckOpenGLError; {$endif}

    // performance data demanded?
    if FMonitor then
    begin
      // yes, calculate average frames per second...
      Inc(FFrames);
      if FFrames > 1 then // ...but leave out the very first frame
      begin
        QueryPerformanceCounter(Counter2);
        // in second run take an 'avarage' value for the first run into account
        // by simply using twice the time from this run
        if FFrames = 2 then FTicks := FTicks + 2 * (Counter2 - Counter1)
                       else FTicks := FTicks + Counter2 - Counter1;
        if FTicks > 0 then  FFramesPerSecond := FFrames * CounterFrequency / FTicks;
      end;
    end;
  finally
    DeactivateRenderingContext;
    FState := dsNone;
  end;
end;

//------------------------------------------------------------------------------

procedure TSceneViewer.CreateWnd;

var
  BackColor: TColorVector;
  Options: TRCOptions;
  
begin
  inherited CreateWnd;
  FCanvas.Handle := GetDC(Handle);

  // initialize and activate the OpenGL rendering context
  // need to do this only once per window creation as we have a private DC
  FState := dsRendering;
  try
    Options := [];
    if roDoubleBuffer in ContextOptions then Include(Options, opDoubleBuffered);

    FRenderingContext := CreateRenderingContext(Canvas.Handle, Options, 32, 0, 0, 0, 0);
    // still no rendering context? -> must be something wrong
    if FRenderingContext = 0 then ShowError(glsNoRenderingContext);

    ActivateRenderingContext(Canvas.Handle, FRenderingContext);
    if not GL_VERSION_1_1 then
    begin
      DestroyRenderingContext(FRenderingContext);
      ShowError(glsWrongVersion);
    end;
    FBuffers := [buColor, buDepth];

    // define viewport, this is necessary because the first WM_SIZE message is posted before
    // the rendering context has been created
    with FViewPort do
    begin
      Left := 0;
      Top := 0;
      Width := Self.Width;
      Height := Self.Height;
      glViewport(0, 0, Width, Height);
    end;

    // set up initial context states
    if FSaveStates <> [] then
    begin
      // might be a recreated window, so reset all states which where activated when
      // the window was destroyed
      SetupStates(FSaveStates);
      FCurrentStates := FSaveStates;
      FSaveStates := [];
    end
    else
    begin
      ReadContextProperties;
      SetupRenderingContext;
    end;

    BackColor := ConvertWinColor(FBackgroundColor);
    glClearColor(BackColor[0], BackColor[1], BackColor[2], BackColor[3]);
    DeactivateRenderingContext;
    if woStayOnTop in DisplayOptions.WindowAttributes then
      SetWindowPos(Handle, HWND_TOPMOST, 0, 0, 0, 0, SWP_NOCOPYBITS or SWP_NOMOVE or SWP_NOSIZE);
    FOwnRefresh := False;
  finally
    FState := dsNone;
  end;
end;

//------------------------------------------------------------------------------

procedure TSceneViewer.DestroyWnd;

begin
  FSaveStates := FCurrentStates;
  FBackground.DestroyHandle;
  DeactivateRenderingContext;
  // free the rendering context
  DestroyRenderingContext(FRenderingContext);
  FRenderingContext := 0;
  inherited DestroyWnd;
end;

//------------------------------------------------------------------------------

procedure TSceneViewer.SetBackgroundColor(AColor: TColor);

var
  BackColor: TColorVector;

begin
  if FBackgroundColor <> AColor then
  begin
    FBackgroundColor := AColor;
    if not (csReading in ComponentState) then
    begin
      BackColor := ConvertWinColor(FBackgroundColor);
      ActivateRenderingContext(FCanvas.Handle, FRenderingContext);
      glClearColor(BackColor[0], BackColor[1], BackColor[2], BackColor[3]);
      DeactivateRenderingContext;
      Invalidate;
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure TSceneViewer.SetCamera(ACamera: TCamera);

begin
  if FCamera <> ACamera then
  begin
    if assigned(FCamera) then
    begin
      if assigned(FCamera.FScene) then
        FCamera.FScene.RemoveViewer(Self);
      FCamera := nil;
    end;
    if assigned(ACamera) and assigned(ACamera.FScene) then
    begin
      FCamera := ACamera;
      Include(FCamera.FChanges, ocTransformation);
      if not (csLoading in ComponentState) then
      begin
        RecreateWnd;
        HandleNeeded;
        ACamera.FScene.AddViewer(Self);
      end;
    end;
    Invalidate;
  end;
end;

//------------------------------------------------------------------------------

procedure TSceneViewer.SetContextOptions(Options: TContextOptions);

var
  Temp: TCamera;

begin
  if FContextOptions <> Options then
  begin
    Temp := Camera;
    Camera := nil;
    FContextOptions := Options;
    Camera := Temp;
  end;
end;

//------------------------------------------------------------------------------

procedure TSceneViewer.SetDepthTest(AValue: Boolean);

begin
  if FDepthTest <> AValue then
  begin
    FDepthTest := AValue;
    if not (csReading in ComponentState) then
    begin
      ActivateRenderingContext(FCanvas.Handle, FRenderingContext);
      if AValue then RequestedState([stDepthTest])
                else UnnecessaryState([stDepthTest]);
      DeactivateRenderingContext;
      Invalidate;
    end;  
  end;
end;

//------------------------------------------------------------------------------

procedure TSceneViewer.SetFaceCulling(AValue: Boolean);

begin
  if FFaceCulling <> AValue then
  begin
    FFaceCulling := AValue;
    if not (csReading in ComponentState) then
    begin
      ActivateRenderingContext(FCanvas.Handle, FRenderingContext);
      if AValue then RequestedState([stCullFace])
                else UnnecessaryState([stCullFace]);
      DeactivateRenderingContext;
      Invalidate;
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure TSceneViewer.SetJoystick(AValue: TGLJoystick);

begin
  FJoystick.Assign(AValue);
end;

//------------------------------------------------------------------------------

procedure TSceneViewer.SetLighting(AValue: Boolean);

begin
  if FLighting <> AValue then
  begin
    FLighting := AValue;
    if not (csReading in ComponentState) then
    begin
      ActivateRenderingContext(FCanvas.Handle, FRenderingContext);
      if AValue then RequestedState([stLighting])
                else UnnecessaryState([stLighting]);
      DeactivateRenderingContext;
      Invalidate;
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure TSceneViewer.SetFogEnable(AValue: Boolean);

begin
  if FFogEnable <> AValue then
  begin
    FFogEnable := AValue;
    if not (csReading in ComponentState) then
    begin
      ActivateRenderingContext(FCanvas.Handle, FRenderingContext);
      FFogEnviroment.Change;
      if AValue then RequestedState([stFog])
                else UnnecessaryState([stFog]);
      DeactivateRenderingContext;
      Invalidate;
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure TSceneViewer.SetSpecials(Value: TSpecials);

begin
  if FSpecials <> Value then
  begin
    FSpecials :=  Value;
    Invalidate;
  end;
end;

//------------------------------------------------------------------------------

procedure TSceneViewer.SetFogEnvironment(AValue: TFogEnvironment);

begin
  ActivateRenderingContext(FCanvas.Handle, FRenderingContext);
  FFogEnviroment.Assign(AValue);
  FFogEnviroment.Change;
  DeactivateRenderingContext;
  Invalidate;
end;

//------------------------------------------------------------------------------


initialization
  // preparation for high resolution timer
  if not QueryPerformanceFrequency(CounterFrequency) then CounterFrequency := 0;
  InitOpenGL;
finalization
end.
