unit GLObjects;

// GLObjects   - implementation of scene objects plus some management routines
// version     - 0.5.8
// 05-JAN-2000 ml: adjustment of loader routine for 3DS files
// 04-JAN-2000 ml: included new 3DS classes

{$R-}

interface

uses Windows, Classes, Controls, Geometry, GLScene, GLTexture,
     Graphics, OpenGL12, PlugInManager, SysUtils, AsyncTimer, extctrls,
     // 3DS Support
     File3DS, Types3DS,
     // MD2 Support
     FileMD2, TypeMD2;

type  PNormal    = ^TNormal;
      TNormal    = TAffineVector;

      PVertex    = ^TVertex;
      TVertex    = TAffineVector;

      // used to describe what kind of winding has a front face
      TFaceWinding       = (fwCounterClockWise, fwClockWise);

      TPlane = class (TSceneObject)
      private
        FXOffset,
        FYOffset,
        FWidth,
        FHeight: TGLFloat;
        FXTiles,
        FYTiles: Cardinal;
        procedure SetHeight(AValue: TGLFloat);
        procedure SetWidth(AValue: TGLFloat);
        procedure SetXOffset(const Value: TGLFloat);
        procedure SetXTiles(const Value: Cardinal);
        procedure SetYOffset(const Value: TGLFloat);
        procedure SetYTiles(const Value: Cardinal);
      public
        constructor Create(AOwner: TComponent); override;
        procedure BuildList; override;

        procedure Assign(Source: TPersistent); override;
      published
        property Height: TGLFloat read FHeight write SetHeight;
        property Width: TGLFloat read FWidth write SetWidth;
        property XOffset: TGLFloat read FXOffset write SetXOffset;
        property XTiles: Cardinal read FXTiles write SetXTiles default 1;
        property YOffset: TGLFloat read FYOffset write SetYOffset;
        property YTiles: Cardinal read FYTiles write SetYTiles default 1;
      end;

      TCubePart  = (cpTop, cpBottom, cpFront, cpBack, cpLeft, cpRight);
      TCubeParts = set of TCubePart;

      TCube = class (TSceneObject)
      private
        FCubeWidth, 
        FCubeHeight, 
        FCubeDepth  : TGLFloat;
        FParts      : TCubeParts;
        FNormalDirection : TNormalDirection;
        procedure SetCubeWidth(AValue: TGLFloat);
        procedure SetCubeHeight(AValue: TGLFloat);
        procedure SetCubeDepth(AValue: TGLFloat);
        procedure SetParts(AValue: TCubeParts);
        procedure SetNormalDirection(AValue: TNormalDirection);
      public
        constructor Create(AOwner: TComponent); override;
        procedure BuildList; override;
        procedure Assign(Source: TPersistent); override;
      published
        property CubeWidth: TGLFloat read FCubeWidth write SetCubeWidth;
        property CubeHeight: TGLFloat read FCubeHeight write SetCubeHeight;
        property CubeDepth: TGLFloat read FCubeDepth write SetCubeDepth;
        property NormalDirection: TNormalDirection read FNormalDirection write SetNormalDirection default ndOutside;
        property Parts: TCubeParts read FParts write SetParts;
      end;

      TQuadricNormal = (qnFlat, qnSmooth, qnNone);

      TQuadricObject = class(TSceneObject)
      private
        FNormals : TQuadricNormal;
        FNormalDirection : TNormalDirection;
        procedure SetNormals(AValue: TQuadricNormal);
        procedure SetNormalDirection(AValue: TNormalDirection);
        procedure SetupQuadricParams(Quadric: PGLUquadricObj);
      public
        constructor Create(AOwner: TComponent);override;
        procedure Assign(Source:TPersistent);override;
      published
        property Normals:TQuadricNormal read FNormals write SetNormals default qnSmooth;
        property NormalDirection:TNormalDirection read FNormalDirection write SetNormalDirection default ndOutside;
      end;

      TAngleLimit1 = -90..90;
      TAngleLimit2 = 0..360;
      TCapType = (ctNone, ctCenter, ctFlat);

      TSphere = class (TQuadricObject)
      private
        FRadius  : TGLFloat;
        FSlices, 
        FStacks  : TGLInt;
        FTop     : TAngleLimit1;
        FBottom  : TAngleLimit1;
        FStart   : TAngleLimit2;
        FStop    : TAngleLimit2;
        FTopCap, 
        FBottomCap : TCapType;
        procedure SetBottom(AValue: TAngleLimit1);
        procedure SetBottomCap(AValue: TCapType);
        procedure SetRadius(AValue: TGLFloat);
        procedure SetSlices(AValue: TGLInt);
        procedure SetStart(AValue: TAngleLimit2);
        procedure SetStop(AValue: TAngleLimit2);
        procedure SetStacks(AValue: TGLInt);
        procedure SetTop(AValue: TAngleLimit1);
        procedure SetTopCap(AValue: TCapType);
      public
        constructor Create(AOwner:TComponent); override;
        procedure BuildList; override;
        procedure Assign(Source:TPersistent); override;
      published
        property Bottom: TAngleLimit1 read FBottom write SetBottom default -90;
        property BottomCap: TCapType read FBottomCap write SetBottomCap;
        property Radius: TGLFloat read FRadius write SetRadius;
        property Slices: TGLInt read FSlices write SetSlices default 16;
        property Stacks: TGLInt read FStacks write SetStacks default 16;
        property Start: TAngleLimit2 read FStart write SetStart default 0;
        property Stop: TAngleLimit2 read FStop write SetStop default 360;
        property Top: TAngleLimit1 read FTop write SetTop default 90;
        property TopCap: TCapType read FTopCap write SetTopCap;
      end;

      TDisk = class(TQuadricObject)
      private
        FStartAngle, 
        FSweepAngle, 
        FOuterRadius, 
        FInnerRadius : TGLFloat;
        FSlices,
        FLoops       : TGLInt;
        procedure SetOuterRadius(AValue: TGLFloat);
        procedure SetInnerRadius(AValue: TGLFloat);
        procedure SetSlices(AValue: TGLInt);
        procedure SetLoops(AValue: TGLInt);
        procedure SetStartAngle(AValue: TGLFloat);
        procedure SetSweepAngle(AValue: TGLFloat);
      public
        constructor Create(AOwner: TComponent); override;
        procedure BuildList; override;
        procedure Assign(Source: TPersistent); override;
      published
        property InnerRadius: TGLFloat read FInnerRadius write SetInnerRadius;
        property Loops: TGLInt read FLoops write SetLoops default 16;
        property OuterRadius: TGLFloat read FOuterRadius write SetOuterRadius;
        property Slices: TGLInt read FSlices write SetSlices default 16;
        property StartAngle: TGLFloat read FStartAngle write SetStartAngle;
        property SweepAngle: TGLFloat read FSweepAngle write SetSweepAngle;
      end;

      TCylinderBase = class(TQuadricObject)
      private
        FBottomRadius : TGLFloat;
        FSlices, 
        FStacks  : TGLInt;
        FHeight  : TGLFloat;
        procedure SetBottomRadius(AValue: TGLFloat);
        procedure SetHeight(AValue: TGLFloat);
        procedure SetSlices(AValue: TGLInt);
        procedure SetStacks(AValue: TGLInt);
      public
        constructor Create(AOwner: TComponent); override;
        procedure Assign(Source: TPersistent); override;
      published
        property BottomRadius: TGLFloat read FBottomRadius write SetBottomRadius;
        property Height: TGLFloat read FHeight write SetHeight;
        property Slices: TGLInt read FSlices write SetSlices default 16;
        property Stacks: TGLInt read FStacks write SetStacks default 16;
      end;

      TConePart  = (coSides, coBottom);
      TConeParts = set of TConePart;

      TCone = class(TCylinderBase)
      private
        FParts : TConeParts;
        procedure SetParts(AValue: TConeParts);
      public
        constructor Create(AOwner: TComponent); override;
        procedure BuildList; override;
        procedure Assign(Source: TPersistent); override;
      published
        property Parts:TConeParts read FParts Write SetParts;
      end;

      TCylinderPart = (cySides, cyBottom, cyTop);
      TCylinderParts = set of TCylinderPart;

      TCylinder = class(TCylinderBase)
      private
        FParts     : TCylinderparts;
        FTopRadius : TGLFloat;
        procedure SetTopRadius(AValue: TGLFloat);
        procedure SetParts(AValue: TCylinderParts);
      public
        constructor Create(AOwner: TComponent); override;
        procedure BuildList; override;
        procedure Assign(Source: TPersistent); override;
      published
        property TopRadius: TGLFloat read FTopRadius write SetTopRadius;
        property Parts: TCylinderParts read FParts Write SetParts;
      end;

      TVertexList = class(TPersistent)
      private
        FValues : PFloatVector;
        FSize   : Integer;
        FOwner  : TSceneObject;
        FEntrySize : Integer;
        function GetCount: Integer;
        function GetFirstEntry: PGLFloat;
        function GetFirstColor: PGLFLoat;
        function GetFirstNormal: PGLFLoat;
        function GetFirstVertex: PGLFLoat;
        procedure ReadItems(Reader: TReader);
        procedure WriteItems(Writer: TWriter);
      protected
        procedure DefineProperties(Filer: TFiler); override;
      public
        constructor Create(AOwner: TSceneObject);
        destructor Destroy; override;
        procedure AddVertex(Vertex: PVertex; Normal: PNormal; Color: PColorVector; TexPoint: PTexPoint);
        procedure Assign(Source: TPersistent); override;
        procedure Clear;
        procedure Update;
        property Count: Integer read GetCount;
        property EntrySize: Integer read FEntrySize;
        property FirstColor: PGLFloat read GetFirstColor;
        property FirstEntry: PGLFLoat read GetFirstEntry;
        property FirstNormal: PGLFloat read GetFirstNormal;
        property FirstVertex: PGLFloat read GetFirstVertex;
        property Size: Integer read FSize;
      end;

      TMeshMode = (mmTriangleStrip, mmTriangleFan, mmTriangles, 
                   mmQuadStrip, mmQuads, mmPolygon);
      TVertexMode = (vmV, vmVN, vmVNC, vmVNCT);

      TMesh = class(TSceneObject)
      private
        FVertices   : TVertexList;
        FMode       : TMeshMode;
        FVertexMode : TVertexMode;
        procedure SetMode(AValue: TMeshMode);
        procedure SetVertices(AValue: TVertexList);
        procedure SetVertexMode(AValue: TVertexMode);
        function  CalcPlaneNormal(x1, y1, z1, x2, y2, z2, x3, y3, z3: TGLFloat): TAffineFltVector;
      public
        constructor Create(AOwner: TComponent); override;
        destructor Destroy; override;
        procedure Assign(Source: TPersistent); override;
        procedure BuildList; override;
        procedure CalcNormals(Frontface: TFaceWinding);
        property Vertices: TVertexList read FVertices write SetVertices;
      published
        property Mode: TMeshMode read FMode write SetMode;
        property VertexMode: TVertexMode read FVertexMode write SetVertexMode default vmVNCT;
      end;

      TFreeForm = class;

      // TVectorFile is an abstract base class for different vector file
      // formats like 3DS, DXF. The actual implementation for these
      // files must be done seperately. The concept for TVectorFile
      // is very similar to TGraphic (see there)
      TVectorFile = class(TPersistent)
      private
        FOwner : TFreeForm;
      public
        constructor Create(AOwner: TFreeForm); virtual;

        procedure BuildMeshObjects(Version: TReleaseLevel; Materials: TMaterialList; Objects: TObjectList); virtual; abstract;
        procedure LoadFromFile(const FileName: String); virtual; abstract;
        procedure LoadFromStream(Stream: TStream); virtual; abstract;

        property Owner: TFreeForm read FOwner;
      end;

      TVectorFileClass = class of TVectorFile;

      // currently there is one actual implementation of a vector file reader class:
      TGL3DSVectorFile = class(TVectorFile)
      public
        procedure BuildMeshObjects(Version: TReleaseLevel; Materials: TMaterialList; Objects: TObjectList); override;
        procedure LoadFromFile(const FileName: String); override;
        procedure LoadFromStream(Stream: TStream); override;
      end;

      TVertexStorage = (vsCopy, vsReference, vsHandOver);

      PFaceGroup = ^TFaceGroup;
      TFaceGroup = record
                    FaceProperties : TFaceProperties;
                    IndexCount     : Cardinal;
                    Indices        : PIntegerArray;
                  end;

      TFaceGroups = class(TList)
      private
        function GetFaceGroup(Index: Integer): PFaceGroup;
      public
        procedure Clear; override;
        property Items[Index: Integer]: PFaceGroup read GetFaceGroup; default;
      end;

      PMeshObject = ^TMeshObject;
      TMeshObject = record
        Mode: TMeshMode;      // currently only mmTriangles are supported
        Vertices: PVectorArray;   // comprises all vertices of the object in no particular order
        Normals: PVectorArray;
        TexCoords: PTexPointArray;
        FaceGroups: TFaceGroups;    // a list of face groups, each comprising a material description and
                                    // a list of indices into the VerticesArray array
      end;

      // The md2 vector file:
      TGLMD2VectorFile = class(TVectorFile)
      public
        function ConvertMD2Structure(MD2File: TFileMD2; frame: Integer): PMeshObject;
        procedure LoadFromFile(const FileName: String); override;
        //procedure LoadFromStream(Stream: TStream); override;
      end;

      TFreeForm = class(TSceneObject)
      private
        FObjects: TList;     // a list of mesh objects
        FWinding: TGLEnum;
        FCenter: TAffineVector; // used when autocentering externally loaded data
      public
        constructor Create(AOwner: TComponent); override;
        destructor Destroy; override;

        function AddMeshObject(AObject: PMeshObject): Integer; virtual;
        procedure Assign(Source: TPersistent); override;
        procedure BuildList; override;
        procedure Clear;
        procedure LoadFromFile(const Filename: string);

        property Winding: TGLEnum read FWinding;
      end;

      TActor = class(TFreeForm)
      private
        FNumberFrame: Integer;
        FStartFrame,
        FEndFrame: Integer;
        FCurrentFrame: Integer;

        FTimer: TAsyncTimer;
        FAction: Boolean;
        FInterval: Integer;

        procedure SetInterval(Value: Integer);
        procedure SetAction(Value: Boolean);

        procedure ActionOnTimer(Sender: TObject);

        procedure SetCurrentFrame(Value: Integer);
        procedure SetStartFrame(Value: Integer);
        procedure SetEndFrame(Value: Integer);
      public
        constructor Create(AOwner: TComponent); override;
        destructor Destroy; override;

        procedure Assign(Source: TPersistent); override;
        procedure BuildList; override;
        procedure LoadFromFile(const Filename: string);

        procedure NextFrame(N: Integer);
        procedure PrevFrame(N: Integer);

        property Winding: TGLEnum read FWinding;
      published
        property NumberFrame: Integer read FNumberFrame;
        property StartFrame: Integer read FStartFrame write SetStartFrame;
        property EndFrame: Integer read FEndFrame write SetEndFrame;
        property CurrentFrame: Integer read FCurrentFrame write SetCurrentFrame;

        property Action: Boolean read FAction write SetAction;
        property Interval: Integer read FInterval write SetInterval;
      end;


      TTorus = class(TSceneObject)
      private
        FRings,
        FSides        : Cardinal;
        FMinorRadius, 
        FMajorRadius  : Single;
        procedure SetMajorRadius(AValue: Single);
        procedure SetMinorRadius(AValue: Single);
        procedure SetRings(AValue: Cardinal);
        procedure SetSides(AValue: Cardinal);
      public
        constructor Create(AOwner: TComponent); override;
        procedure BuildList; override;
      published
        property MajorRadius: Single read FMajorRadius write SetMajorRadius;
        property MinorRadius: Single read FMinorRadius write SetMinorRadius;
        property Rings: Cardinal read FRings write SetRings;
        property Sides: Cardinal read FSides write SetSides;
      end;

      TSpaceText = class(TSceneObject)
      private
        FFont       : TFont;
        FText       : String;
        FExtrusion  : Single;
        procedure SetExtrusion(AValue: Single);
        procedure SetFont(AFont: TFont);
        procedure SetText(AText: String);
      protected
        BaseList    : TGLuint;
        FontChanged : Boolean;
      public
        constructor Create(AOwner: TComponent); override;
        destructor Destroy; override;
        procedure BuildList; override;
        procedure PrepareObject; override;
      published
        property Extrusion: Single read FExtrusion write SetExtrusion;
        property Font: TFont read FFont write SetFont;
        property Text: String read FText write SetText;
      end;

      TTeapot = class(TSceneObject)
      private
        FGrid : Cardinal;
      public
        constructor Create(AOwner: TComponent); override;
        procedure BuildList; override;
      end;

      TDodecahedron = class(TSceneObject)
      public
        procedure BuildList; override;
      end;

      TRotationSolid = class(TSceneObject)
      public
        procedure BuildList; override;
      end;

      PSceneObjectEntry = ^TSceneObjectEntry;
      // holds a relation between an scene object class, its global identification, 
      // its location in the object stock and its icon reference
      TSceneObjectEntry = record
                             ObjectClass : TSceneObjectClass;
                             Name : String[32];    // type name of the object
                             Index,                // index into "FObjectStock"
                             ImageIndex : Integer; // index into "FObjectIcons"
                           end;

      TObjectManager = class(TResourceManager)
      private
        FObjectStock : TSceneObject;     // a list of objects, which can be used for scene building
        FObjectIcons : TImageList;       // a list of icons for scene objects
        FOverlayIndex,                   // indices into the object icon list
        FSceneRootIndex, 
        FCameraRootIndex, 
        FLightsourceRootIndex, 
        FObjectRootIndex, 
        FStockObjectRootIndex : Integer;
      protected
        procedure CreateDefaultObjectIcons;
        procedure DestroySceneObjectList;
        function FindSceneObjectClass(AObjectClass: TSceneObjectClass; ASceneObject: String) : PSceneObjectEntry;
      public
        constructor Create(Aowner: TComponent); override;
        destructor Destroy; override;
        function GetClassFromIndex(Index: Integer): TSceneObjectClass;
        function GetImageIndex(ASceneObject: TSceneObjectClass) : Integer;
        procedure GetRegisteredSceneObjects(ObjectList: TStringList);
        procedure RegisterSceneObject(ASceneObject: TSceneObjectClass; AName: String; AImage: HBitmap);
        procedure UnRegisterSceneObject(ASceneObject: TSceneObjectClass; AName: String);
        procedure Notify(Sender: TPlugInManager; Operation: TOperation; PlugIn: Integer); override;

        property ObjectIcons: TImageList read FObjectIcons;
        property SceneRootIndex: Integer read FSceneRootIndex;
        property LightsourceRootIndex: Integer read FLightsourceRootIndex;
        property CameraRootIndex: Integer read FCameraRootIndex;
        property ObjectRootIndex: Integer read FObjectRootIndex;
      end;

     PVectorFileFormat = ^TVectorFileFormat;
     TVectorFileFormat = record
       VectorFileClass : TVectorFileClass;
       Extension       : String;
       Description     : String;
       DescResID       : Integer;
     end;

     TVectorFileFormatsList = class(TList)
     public
       destructor Destroy; override;
       procedure Add(const Ext, Desc: String; DescID: Integer; AClass: TVectorFileClass);
       function FindExt(Ext: string): TVectorFileClass;
       procedure Remove(AClass: TVectorFileClass);
       procedure BuildFilterStrings(VectorFileClass: TVectorFileClass; var Descriptions, Filters: string);
     end;
     
function GetVectorFileFormats: TVectorFileFormatsList;      
procedure RegisterVectorFileFormat(const AExtension, ADescription: String; AClass: TVectorFileClass);
procedure UnregisterVectorFileClass(AClass: TVectorFileClass);

var ObjectManager : TObjectManager;

//------------------------------------------------------------------------------

implementation

uses Consts, Dialogs, Forms, GLMisc, GLStrings, Math, PlugInIntf;

{$R GLObjects.Res}

type // holds an entry in the font manager list (used in TSpaceText)
     PFontEntry        = ^TFontEntry;
     TFontEntry        = record
                           Name      : String;
                           Styles    : TFontStyles;
                           Extrusion : Single;
                           Base      : TGLuint;
                           RefCount  : Integer;
                         end;

     // manages a list of fonts for which display lists were created
     TFontManager = class(TList)
     public
       destructor Destroy; override;
       function FindFont(AName: String; FStyles: TFontStyles; FExtrusion: Single) : PFontEntry;
       function FindFontByList(AList: TGLuint): PFontEntry;
       function GetFontBase(AName: String; FStyles: TFontStyles; FExtrusion: Single) : TGLuint;
       procedure Release(List: TGLuint);
     end;

     EInvalidVectorFile = class(Exception);

var SceneObjectList   : TList;
    FontManager       : TFontManager;
    VectorFileFormats : TVectorFileFormatsList;

//----------------- vector format support --------------------------------------

destructor TVectorFileFormatsList.Destroy;

var I: Integer;

begin
  for I := 0 to Count - 1 do Dispose(PVectorFileFormat(Items[I]));
  inherited Destroy;
end;

//------------------------------------------------------------------------------

procedure TVectorFileFormatsList.Add(const Ext, Desc: String; DescID: Integer;
                                     AClass: TVectorFileClass);

var NewRec: PVectorFileFormat;

begin
  New(NewRec);
  with NewRec^ do
  begin
    Extension := AnsiLowerCase(Ext);
    VectorFileClass := AClass;
    Description := Desc;
    DescResID := DescID;
  end;
  inherited Add(NewRec);
end;

//------------------------------------------------------------------------------

function TVectorFileFormatsList.FindExt(Ext: string): TVectorFileClass;

var I: Integer;

begin
  Ext := AnsiLowerCase(Ext);
  for I := Count-1 downto 0 do
    with PVectorFileFormat(Items[I])^ do
      if Extension = Ext then
      begin
        Result := VectorFileClass;
        Exit;
      end;
  Result := nil;
end;

//------------------------------------------------------------------------------

procedure TVectorFileFormatsList.Remove(AClass: TVectorFileClass);

var I : Integer;
    P : PVectorFileFormat;

begin
  for I := Count-1 downto 0 do
  begin
    P := PVectorFileFormat(Items[I]);
    if P^.VectorFileClass.InheritsFrom(AClass) then
    begin
      Dispose(P);
      Delete(I);
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure TVectorFileFormatsList.BuildFilterStrings(VectorFileClass: TVectorFileClass;
                                                    var Descriptions, Filters: string);

var C, I : Integer;
    P    : PVectorFileFormat;
    
begin
  Descriptions := '';
  Filters := '';
  C := 0;
  for I := Count-1 downto 0 do
  begin
    P := PVectorFileFormat(Items[I]);
    if P^.VectorFileClass.InheritsFrom(VectorFileClass) and (P^.Extension <> '') then
      with P^ do
      begin
        if C <> 0 then
        begin
          Descriptions := Descriptions+'|';
          Filters := Filters+';';
        end;
        if (Description = '') and (DescResID <> 0) then Description := LoadStr(DescResID);
        FmtStr(Descriptions, '%s%s (*.%s)|*.%2:s', [Descriptions, Description, Extension]);
        FmtStr(Filters, '%s*.%s', [Filters, Extension]);
        Inc(C);
      end;
  end;
  if C > 1 then FmtStr(Descriptions, '%s (%s)|%1:s|%s', [sAllFilter, Filters, Descriptions]);
end;

//----------------- TVectorFile ------------------------------------------------

constructor TVectorFile.Create(AOwner: TFreeForm);

begin
  FOwner := AOwner;
  inherited Create;
end;

//----------------- TGL3DSVectorFile -------------------------------------------

procedure TGL3DSVectorFile.BuildMeshObjects(Version: TReleaseLevel; Materials: TMaterialList; Objects: TObjectList);

type
  TSmoothIndexEntry = array[0..31] of Cardinal;

  PSmoothIndexArray = ^TSmoothIndexArray;
  TSmoothIndexArray = array[0..0] of TSmoothIndexEntry;

var
  Marker: PByteArray;
  CurrentVertexCount: Cardinal;
  SmoothIndices: PSmoothIndexArray;
  Mesh: PMeshObject;

  //--------------- local functions -------------------------------------------

  function FacePropsFromMaterial(const Name: String): TFaceProperties;

  var
    Material: PMaterial3DS;

  begin
    Material := Materials.MaterialByName[Name];
    if Material = nil then ShowErrorFormatted(glsMaterialNotFound, [Name]);
    Result := TFaceProperties.Create(nil);
    with Result do
    begin
      Ambient.Color := MakeVector([Material.Ambient.R, Material.Ambient.G, Material.Ambient.B, 1]);
      Diffuse.Color := MakeVector([Material.Diffuse.R, Material.Diffuse.G, Material.Diffuse.B, 1]);
      Specular.Color := MakeVector([Material.Specular.R, Material.Specular.G, Material.Specular.B, 1]);
      VectorScale(Specular.Color, 1 - Material.Shininess);
      Shininess := Round((1 - Material.ShinStrength) * 128);
    end;
  end;

  //----------------------------------------------------------------------

  function IsVertexMarked(P: Pointer; Index: Integer): Boolean; assembler;

  // tests the Index-th bit, returns True if set else False

  asm
                     BT [EAX], EDX
                     SETC AL
  end;

  //---------------------------------------------------------------------------

  function MarkVertex(P: Pointer; Index: Integer): Boolean; assembler;

  // sets the Index-th bit and return True if it was already set else False

  asm
                     BTS [EAX], EDX
                     SETC AL
  end;

  //---------------------------------------------------------------------------

  procedure StoreSmoothIndex(ThisIndex, SmoothingGroup, NewIndex: Cardinal; P: Pointer);

  // Stores new vertex index (NewIndex) into the smooth index array of vertex ThisIndex
  // using field SmoothingGroup, which must not be 0.
  // For each vertex in the vertex array (also for duplicated vertices) an array of 32 cardinals
  // is maintained (each for one possible smoothing group. If a vertex must be duplicated because
  // it has no smoothing group or a different one then the index of the newly created vertex is
  // stored in the SmoothIndices to avoid loosing the conjunction between not yet processed vertices
  // and duplicated vertices.
  // Note: Only one smoothing must be assigned per vertex. Some available models break this rule and
  //       have more than one group assigned to a face. To make the code fail safe the group ID
  //       is scanned for the lowest bit set.

  asm
                   PUSH EBX
                   BSF EBX, EDX                  // determine smoothing group index (convert flag into an index)
                   MOV EDX, [P]                  // get address of index array
                   SHL EAX, 7                    // ThisIndex * SizeOf(TSmoothIndexEntry)
                   ADD EAX, EDX
                   LEA EDX, [4 * EBX + EAX]      // Address of array + vertex index + smoothing group index
                   MOV [EDX], ECX
                   POP EBX
  end;

  //---------------------------------------------------------------------------

  function GetSmoothIndex(ThisIndex, SmoothingGroup: Cardinal; P: Pointer): Cardinal;

  // Retrieves the vertex index for the given index and smoothing group.
  // This redirection is necessary because a vertex might have been duplicated.

  asm
                   PUSH EBX
                   BSF EBX, EDX                  // determine smoothing group index
                   SHL EAX, 7                    // ThisIndex * SizeOf(TSmoothIndexEntry)
                   ADD EAX, ECX
                   LEA ECX, [4 * EBX + EAX]      // Address of array + vertex index + smoothing group index
                   MOV EAX, [ECX]
                   POP EBX
  end;
  
  //---------------------------------------------------------------------------

  procedure DuplicateVertex(Index: Integer);

  // extends the vector and normal array by one entry and duplicates the vertex data given by Index
  // the marker and texture arrays will be extended too, if necessary

  begin
    // enhance vertex array
    ReallocMem(Mesh.Vertices, (CurrentVertexCount + 1) * SizeOf(TVector3f));
    Mesh.Vertices[CurrentVertexCount] := Mesh.Vertices[Index];

    // enhance normal array
    ReallocMem(Mesh.Normals, (CurrentVertexCount + 1) * SizeOf(TVector3f));
    Mesh.Normals[CurrentVertexCount] := NullVector;

    // enhance smooth index array
    ReallocMem(SmoothIndices, (CurrentVertexCount + 1) * SizeOf(TSmoothIndexEntry));
    FillChar(SmoothIndices[CurrentVertexCount], SizeOf(TSmoothIndexEntry), $FF);

    // enhance marker array
    if (CurrentVertexCount div 8) <> ((CurrentVertexCount + 1) div 8) then
    begin
      ReallocMem(Marker, ((CurrentVertexCount + 1) div 8) + 1);
      Marker[(CurrentVertexCount div 8) + 1] := 0;
    end;

    // enhance texture coordinates array
    if assigned(Mesh.TexCoords) then
    begin
      ReallocMem(Mesh.TexCoords, (CurrentVertexCount + 1) * SizeOf(TTexVert3DS));
      Mesh.TexCoords[CurrentVertexCount] := Mesh.TexCoords[Index];
    end;

    Inc(CurrentVertexCount);
  end;

  //--------------- end local functions ---------------------------------------

var
  Size: Cardinal;
  Material,
  I, J: Integer;
  FaceGroup: PFaceGroup;
  V1, V2: TAffineVector;
  Face,
  SubFace,
  Vertex,
  TargetVertex: Integer;
  SmoothingGroup: Cardinal;
  CurrentIndex: Word;
  Vector1,
  Vector2,
  Normal: TAffineVector;
  TotalCount: Cardinal;

begin
  // determine front face winding
  if Version >= rlRelease3 then FOwner.FWinding := GL_CCW
                           else FOwner.FWinding := GL_CW;
  TotalCount := 0;

  for I := 0 to Objects.MeshCount - 1 do
  with PMesh3DS(Objects.Mesh[I])^ do
  begin
    if IsHidden or (NVertices < 3) then Continue;

    // New() just calls GetMem, but I want the memory cleared
    Mesh := AllocMem(SizeOf(TMeshObject));
    Mesh.Mode := mmTriangles;
    // make a copy of the vertex data, this must always be available
    CurrentVertexCount := NVertices;
    Size := NVertices * SizeOf(TPoint3DS);
    // allocate memory, we do not need to clear it
    GetMem(Mesh.Vertices, Size);
    // we don't need to consider the local mesh matrix here, since all vertices are already
    // transformed into their final positions
    Move(VertexArray^, Mesh.Vertices^, Size);

    // texturing data available (optional)?
    if NTextVerts > 0 then
    begin
      Size := NTextVerts * SizeOf(TTexVert3DS);
      GetMem(Mesh.TexCoords, Size);
      Move(TextArray^, Mesh.TexCoords^, Size);
    end;

    // allocate memory for the face normal array, the final normal array and the marker array
    Mesh.Normals := AllocMem(NVertices * SizeOf(TVector3f));
    Marker := AllocMem((NVertices div 8) + 1); // one bit for each vertex
    GetMem(SmoothIndices, NVertices * SizeOf(TSmoothIndexEntry));

    if SmoothArray = nil then
    begin
      // no smoothing groups to consider
      for Face := 0 to NFaces - 1 do
      with FaceArray[Face], Mesh^ do
      begin
        // normal vector for the face
        Vector1 := VectorAffineSubtract(Vertices[V1], Vertices[V2]);
        Vector2 := VectorAffineSubtract(Vertices[V3], Vertices[V2]);
        if Version < rlRelease3 then Normal := VectorCrossProduct(Vector1, Vector2)
                                 else Normal := VectorCrossProduct(Vector2, Vector1);
        // go for each vertex in the current face
        for Vertex := 0 to 2 do
        begin
          // copy current index for faster access
          CurrentIndex := FaceRec[Vertex];
          // already been touched?
          if IsVertexMarked(Marker, CurrentIndex) then
          begin
            // already touched vertex must be duplicated
            DuplicateVertex(CurrentIndex);
            FaceRec[Vertex] := CurrentVertexCount - 1;
            Normals[CurrentVertexCount - 1] := Normal;
          end
          else
          begin
            // not yet touched, so just store the normal
            Normals[CurrentIndex] := Normal;
            MarkVertex(Marker, CurrentIndex);
          end;
        end;
      end;
    end
    else
    begin
      // smoothing groups are to be considered
      for Face := 0 to NFaces - 1 do
      with FaceArray[Face], Mesh^ do
      begin
        // normal vector for the face
        Vector1 := VectorAffineSubtract(Vertices[V1], Vertices[V2]);
        Vector2 := VectorAffineSubtract(Vertices[V3], Vertices[V2]);
        if Version < rlRelease3 then Normal := VectorCrossProduct(Vector1, Vector2)
                                else Normal := VectorCrossProduct(Vector2, Vector1);
        SmoothingGroup := SmoothArray[Face];
        
        // go for each vertex in the current face
        for Vertex := 0 to 2 do
        begin
          // copy current index for faster access
          CurrentIndex := FaceRec[Vertex];
          // Has vertex already been touched?
          if IsVertexMarked(Marker, CurrentIndex) then
          begin
            // check smoothing group
            if SmoothingGroup = 0 then
            begin
              // no smoothing then just duplicate vertex
              DuplicateVertex(CurrentIndex);
              FaceRec[Vertex] := CurrentVertexCount - 1;
              Normals[CurrentVertexCount - 1] := Normal;
              // mark new vertex also as touched
              MarkVertex(Marker, CurrentVertexCount - 1);
            end
            else
            begin
              // this vertex must be smoothed, check if there's already a (duplicated) vertex
              // for this smoothing group
              TargetVertex := GetSmoothIndex(CurrentIndex, SmoothingGroup, SmoothIndices);
              if TargetVertex < 0 then
              begin
                // vertex has not yet been duplicated for this smoothing group, so do it now
                DuplicateVertex(CurrentIndex);
                FaceRec[Vertex] := CurrentVertexCount - 1;
                Normals[CurrentVertexCount - 1] := Normal;
                StoreSmoothIndex(CurrentIndex, SmoothingGroup, CurrentVertexCount - 1, SmoothIndices);
                StoreSmoothIndex(CurrentVertexCount - 1, SmoothingGroup, CurrentVertexCount - 1, SmoothIndices);
                // mark new vertex also as touched
                MarkVertex(Marker, CurrentVertexCount - 1);
              end
              else
              begin
                // vertex has already been duplicated, so just add normal vector to other vertex...
                Normals[TargetVertex] := VectorAffineAdd(Normals[TargetVertex], Normal);
                // ...and tell which new vertex has to be used from now on
                FaceRec[Vertex] := TargetVertex;
              end;
            end;
          end
          else
          begin
            // vertex not yet touched, so just store the normal
            Normals[CurrentIndex] := Normal;
            // initialize smooth indices for this vertex
            FillChar(SmoothIndices[CurrentIndex], SizeOf(TSmoothIndexEntry), $FF);
            if SmoothingGroup <> 0 then StoreSmoothIndex(CurrentIndex, SmoothingGroup, CurrentIndex, SmoothIndices);
            MarkVertex(Marker, CurrentIndex);
          end;
        end;
      end;
    end;
    FreeMem(Marker);
    FreeMem(SmoothIndices);

    // sum up all vertices to do auto centering
    Inc(TotalCount, CurrentVertexCount);
    with Mesh^, FOwner do
    begin
      for J := 0 to CurrentVertexCount - 1 do
        FCenter := VectorAffineAdd(FCenter, Vertices[J]);
    end;

    // finally normalize the Normals array
    with Mesh^ do
      for J := 0 to CurrentVertexCount - 1 do VectorNormalize(Normals[J]);

    // now go for each material group
    Mesh.FaceGroups := TFaceGroups.Create;
    // if there's no face to material assignment then just copy the
    // face definitions and rely on the default texture of the scene object
    if NMats = 0 then
    begin
      New(FaceGroup);
      with FaceGroup^ do
      begin
        FaceProperties := TFaceProperties.Create(nil);
        FaceProperties.IsInherited := True;
        IndexCount := 3 * NFaces;
        GetMem(Indices, IndexCount * SizeOf(Integer));
        // copy the face list
        for J := 0 to NFaces - 1 do
        begin
          Indices[3 * J + 0] := FaceArray[J].V1;
          Indices[3 * J + 1] := FaceArray[J].V2;
          Indices[3 * J + 2] := FaceArray[J].V3;
        end;
      end;
      Mesh.FaceGroups.Add(FaceGroup);
    end
    else
    begin
      for Material := 0 to NMats - 1 do
      begin
        New(FaceGroup);
        with FaceGroup^ do
        begin
          FaceProperties := FacePropsFromMaterial(MatArray[Material].Name);
          IndexCount := 3 * MatArray[Material].NFaces;
          GetMem(Indices, IndexCount * SizeOf(Integer));
          // copy all vertices belonging to the current face into our index array, 
          // there won't be redundant vertices since this would mean a face has more than one
          // material
          with MatArray[Material] do
            for J := 0 to NFaces - 1 do // NFaces is the one from FaceGroup
            begin
              Indices[3 * J + 0] := FaceArray[FaceIndex[J]].V1;
              Indices[3 * J + 1] := FaceArray[FaceIndex[J]].V2;
              Indices[3 * J + 2] := FaceArray[FaceIndex[J]].V3;
            end;
        end;
        Mesh.FaceGroups.Add(FaceGroup);
      end;
    end;
    FOwner.AddMeshObject(Mesh);
  end;
  if TotalCount > 0 then VectorScale(FOwner.FCenter, 1 / TotalCount);
end;

//------------------------------------------------------------------------------

procedure TGL3DSVectorFile.LoadFromFile(const FileName: String);

begin
  with TFile3DS.CreateFromFile(FileName) do
  try
    BuildMeshObjects(DatabaseRelease, Materials, Objects);
  finally
    Free;
  end;
end;

//------------------------------------------------------------------------------

procedure TGL3DSVectorFile.LoadFromStream(Stream: TStream);

begin
  with TFile3DS.Create do
  try
    LoadFromStream(Stream);
    BuildMeshObjects(DatabaseRelease, Materials, Objects);
  finally
    Free;
  end;
end;

//------------------------------------------------------------------------------

function GetVectorFileFormats: TVectorFileFormatsList;

begin
  if VectorFileFormats = nil then VectorFileFormats := TVectorFileFormatsList.Create;
  Result := VectorFileFormats;
end;

//------------------------------------------------------------------------------

procedure RegisterVectorFileFormat(const AExtension, ADescription: String; AClass: TVectorFileClass);

begin
  GetVectorFileFormats.Add(AExtension, ADescription, 0, AClass);
end;

//------------------------------------------------------------------------------

procedure UnregisterVectorFileClass(AClass: TVectorFileClass);

begin
  if VectorFileFormats <> nil then VectorFileFormats.Remove(AClass);
end;

//----------------- TFontManager -----------------------------------------------

destructor TFontManager.Destroy;

var I : Integer;

begin
  for I := 0 to Count-1 do
  begin
    if TFontEntry(Items[I]^).Base <> 0 then glDeleteLists(TFontEntry(Items[I]^).Base, 255);
    FreeMem(Items[I], SizeOf(TFontEntry));
  end;
  inherited Destroy;
end;

//------------------------------------------------------------------------------

function TFontManager.FindFont(AName: String; FStyles: TFontStyles; FExtrusion: Single) : PFontEntry;

var I : Integer;

begin
  Result := nil;
  // try to find an entry with the required attributes
  for I  := 0 to Count-1 do
    with TFontEntry(Items[I]^) do
      if (CompareText(Name, AName) = 0) and
         (Styles = FStyles)            and
         (Extrusion = FExtrusion)      then
      begin
        // entry found
        Result := Items[I];
        Exit;
      end;
end;

//------------------------------------------------------------------------------

function TFontManager.FindFontByList(AList: TGLuint): PFontEntry;

var I : Integer;

begin
  Result := nil;
  // try to find an entry with the required attributes
  for I  := 0 to Count-1 do
    with TFontEntry(Items[I]^) do
      if Base = AList then
      begin // entry found
        Result := Items[I];
        Exit;
      end;
end;

//------------------------------------------------------------------------------

function TFontManager.GetFontBase(AName: String; FStyles: TFontStyles; FExtrusion: Single) : TGLuint;

var NewEntry : PFontEntry;
    MemDC    : HDC;
    AFont    : TFont;

begin
  NewEntry := FindFont(AName, FStyles, FExtrusion);
  if assigned(NewEntry) then
  begin
    Inc(NewEntry^.RefCount);
    Result := NewEntry^.Base;
    Exit;
  end;  
  // no entry found, so create one
  New(NewEntry);
  try
    NewEntry^.Name := AName;
    NewEntry^.Styles := FStyles;
    NewEntry^.Extrusion := FExtrusion;
    NewEntry^.RefCount := 1;
    // create a font to be used while display list creation
    AFont := TFont.Create;
    MemDC := CreateCompatibleDC(0);
    try
      AFont.Name := AName;
      AFont.Style := FStyles;
      SelectObject(MemDC, AFont.Handle);
      NewEntry^.Base := glGenLists(255);
      if NewEntry^.Base = 0 then raise Exception.Create('FontManager: no more display lists available');
      if not wglUseFontOutlines(MemDC, 0, 255, NewEntry^.Base, 0, FExtrusion, WGL_FONT_POLYGONS, nil) then
        raise Exception.Create('FontManager: font creation failed');
    finally
      AFont.Free;
      DeleteDC(MemDC);
    end;
    Add(NewEntry);
    Result := NewEntry^.Base;
  except
    if NewEntry^.Base <> 0 then glDeleteLists(NewEntry^.Base, 255);
    Dispose(NewEntry);
    raise;
  end;
end;

//------------------------------------------------------------------------------

procedure TFontManager.Release(List: TGLuint);

var Entry : PFontEntry;

begin
  Entry := FindFontByList(List);
  if assigned(Entry) then
  begin
    Dec(Entry^.RefCount);
    if Entry^.RefCount = 0 then
    begin
      glDeleteLists(Entry^.Base, 255);
      Remove(Entry);
    end;
  end;
end;

//----------------- TPlane -----------------------------------------------------

constructor TPlane.Create(AOwner:Tcomponent);

begin
  inherited Create(AOwner);
  FWidth := 1;
  FHeight := 1;
  FXTiles := 1;
  FXOffset := 0;
  FYTiles := 1;
  FYOffset := 0;
end;

//------------------------------------------------------------------------------

procedure TPlane.BuildList;

var hw, hh : TGLFloat;

begin
  inherited BuildList;
  hw :=  FWidth/2;
  hh :=  FHeight/2;

  glBegin(GL_QUADS);
  glNormal3f(  0,  0, 1);
  glTexCoord2f(FXTiles+FXOffset, FYTiles+FYOffset);
  glVertex2f( hw, hh);
  glTexCoord2f(0, FYTiles+FYOffset);
  glVertex2f(-hw, hh);
  glTexCoord2f(0, 0);
  glVertex2f(-hw, -hh);
  glTexCoord2f(FXTiles+FXOffset, 0);
  glVertex2f( hw, -hh);
  glEnd;
end;

//------------------------------------------------------------------------------

procedure TPlane.SetWidth(AValue : TGLFloat);

begin
  if AValue <> FWidth then
  begin
    FWidth := AValue;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TPlane.SetHeight(AValue:TGLFloat);

begin
  if AValue <> FHeight then
  begin
    FHeight := AValue;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TPlane.SetXOffset(const Value: TGLFloat);

begin
  if Value <> FXOffset then
  begin
    FXOffset := Value;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TPlane.SetXTiles(const Value: Cardinal);

begin
  if Value <> FXTiles then
  begin
    FXTiles := Value;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TPlane.SetYOffset(const Value: TGLFloat);

begin
  if Value <> FYOffset then
  begin
    FYOffset := Value;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TPlane.SetYTiles(const Value: Cardinal);

begin
  if Value <> FYTiles then
  begin
    FYTiles := Value;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TPlane.Assign(Source: TPersistent);

begin
  if assigned(Source) and (Source is TCube) then
  begin
    FWidth := TPlane(Source).FWidth;
    FHeight := TPlane(Source).FHeight;
  end;
  inherited Assign(Source);
end;

//----------------- TCube ------------------------------------------------------

constructor TCube.Create(AOwner:Tcomponent);

begin
  inherited Create(AOwner);
  FCubeWidth := 1;
  FCubeHeight := 1;
  FCubeDepth := 1;
  FParts := [cpTop, cpBottom, cpFront, cpBack, cpLeft, cpRight];
  FNormalDirection := ndOutside;
end;

//------------------------------------------------------------------------------

procedure TCube.BuildList;

var hw, hh, 
    hd, nd  : TGLFloat;

begin
  inherited BuildList;
  if FNormalDirection = ndInside then nd := -1
                                 else nd := 1;
  hw :=  FCubeWidth/2;
  hh :=  FCubeHeight/2;
  hd :=  FCubeDepth/2;

  if cpFront in FParts then
  begin
    glBegin(GL_QUADS);
    glNormal3f(  0,  0, nd);
    glTexCoord2f(1, 1);
    glVertex3f( hw, hh, hd);
    glTexCoord2f(0, 1);
    glVertex3f(-hw, hh, hd);
    glTexCoord2f(0, 0);
    glVertex3f(-hw, -hh, hd);
    glTexCoord2f(1, 0);
    glVertex3f( hw, -hh, hd);
    glEnd;
  end;
  if cpBack in FParts then
  begin
    glBegin(GL_QUADS);
    glNormal3f(  0,  0, -nd);
    glTexCoord2f(0, 1);
    glVertex3f( hw, hh, -hd);
    glTexCoord2f(0, 0);
    glVertex3f( hw, -hh, -hd);
    glTexCoord2f(1, 0);
    glVertex3f(-hw, -hh, -hd);
    glTexCoord2f(1, 1);
    glVertex3f(-hw, hh, -hd);
    glEnd;
  end;
  if cpLeft in FParts then
  begin
    glBegin(GL_QUADS);
    glNormal3f(-nd,  0,  0);
    glTexCoord2f(1, 1);
    glVertex3f(-hw, hh, hd);
    glTexCoord2f(0, 1);
    glVertex3f(-hw, hh, -hd);
    glTexCoord2f(0, 0);
    glVertex3f(-hw, -hh, -hd);
    glTexCoord2f(1, 0);
    glVertex3f(-hw, -hh, hd);
    glEnd;
  end;
  if cpRight in FParts then
  begin  
    glBegin(GL_QUADS);
    glNormal3f(nd,  0,  0);
    glTexCoord2f(0, 1);
    glVertex3f(hw, hh, hd);
    glTexCoord2f(0, 0);
    glVertex3f(hw, -hh, hd);
    glTexCoord2f(1, 0);
    glVertex3f(hw, -hh, -hd);
    glTexCoord2f(1, 1);
    glVertex3f(hw, hh, -hd);
    glEnd;
  end;
  if cpTop in FParts then
  begin
    glBegin(GL_QUADS);
    glNormal3f(  0, nd,  0);
    glTexCoord2f(0, 1);
    glVertex3f(-hw, hh, -hd);
    glTexCoord2f(0, 0);
    glVertex3f(-hw, hh, hd);
    glTexCoord2f(1, 0);
    glVertex3f( hw, hh, hd);
    glTexCoord2f(1, 1);
    glVertex3f( hw, hh, -hd);
    glEnd;
  end;
  if cpBottom in FParts then
  begin
    glBegin(GL_QUADS);
    glNormal3f(  0, -nd,  0);
    glTexCoord2f(0, 0);
    glVertex3f(-hw, -hh, -hd);
    glTexCoord2f(1, 0);
    glVertex3f( hw, -hh, -hd);
    glTexCoord2f(1, 1);
    glVertex3f( hw, -hh, hd);
    glTexCoord2f(0, 1);
    glVertex3f(-hw, -hh, hd);
    glEnd;
  end;
end;

//------------------------------------------------------------------------------

procedure TCube.SetCubeWidth(AValue : TGLFloat);

begin
  if AValue <> FCubeWidth then
  begin
    FCubeWidth := AValue;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TCube.SetCubeHeight(AValue:TGLFloat);

begin
  if AValue <> FCubeHeight then
  begin
    FCubeHeight := AValue;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TCube.SetCubeDepth(AValue: TGLFloat);

begin
  if AValue <> FCubeDepth then
  begin
    FCubeDepth := AValue;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TCube.SetParts(AValue:TCubeParts);

begin
 if AValue <> FParts then
 begin
   FParts := AValue;
   StructureChanged;
 end;
end;

//------------------------------------------------------------------------------

procedure TCube.SetNormalDirection(AValue: TNormalDirection);

begin
  if AValue <> FNormalDirection then
  begin
    FNormalDirection := AValue;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TCube.Assign(Source: TPersistent);

begin
  if assigned(Source) and (Source is TCube) then
  begin
    FCubeWidth := TCube(Source).FCubewidth;
    FCubeHeight := TCube(Source).FCubeHeight;
    FCubeDepth := TCube(Source).FCubeDepth;
    FParts := TCube(Source).FParts;
    FNormalDirection := TCube(Source).FNormalDirection;
  end;
  inherited Assign(Source);
end;

//----------------- TQuadricObject ---------------------------------------------

constructor TQuadricObject.Create(AOwner:TComponent);

begin
  inherited Create(AOwner);
  FNormals := qnSmooth;
  FNormalDirection := ndOutside;
end;

//------------------------------------------------------------------------------

procedure TQuadricObject.SetNormals(AValue:TQuadricNormal);

begin
  if AValue <> FNormals then
  begin
    FNormals := AValue;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TQuadricObject.SetNormalDirection(AValue:TNormalDirection);

begin
  if AValue <> FNormalDirection then
  begin
    FNormalDirection := AValue;
    StructureChanged;
  end;
end;


//------------------------------------------------------------------------------

procedure TQuadricObject.SetupQuadricParams(Quadric:PGLUquadricObj);

var WithTexture : Boolean;

begin
  gluQuadricDrawStyle(Quadric, GLU_FILL);
  gluQuadricNormals(Quadric, GLU_SMOOTH);
  gluQuadricOrientation(Quadric, GLU_OUTSIDE);
  {case Options.FrontPolygonMode of
    pmFill   : gluQuadricDrawStyle(Quadric, GLU_FILL);
    pmLines  : gluQuadricDrawStyle(Quadric, GLU_LINE);
    pmPoints : gluQuadricDrawStyle(Quadric, GLU_POINT);
  end;

  case FNormalDirection of
    ndInside  : gluQuadricOrientation(Quadric, GLU_INSIDE);
    ndOutside : gluQuadricOrientation(Quadric, GLU_OUTSIDE);
  end;


  case FNormals of
    qnSmooth : gluQuadricNormals(Quadric, GLU_SMOOTH);
    qnFlat   : gluQuadricNormals(Quadric, GLU_FLAT);
    qnNone   : gluQuadricNormals(Quadric, GLU_NONE);
  end;}
  {WithTexture := (stTexture1D in Scene.CurrentStates) or
               (stTexture2D in Scene.CurrentStates);}
  WithTexture := True;
  gluQuadricTexture(Quadric, Ord(WithTexture));
end;

//------------------------------------------------------------------------------

procedure TQuadricObject.Assign(Source:TPersistent);

begin
  if assigned(Source) and (Source is TQuadricObject) then
  begin
    FNormals := TQuadricObject(Source).FNormals;
    FNormalDirection := TQuadricObject(Source).FNormalDirection;
  end;
  inherited Assign(Source);
end;

//----------------- TSphere ----------------------------------------------------

constructor TSphere.Create(AOwner:TComponent);

begin
  inherited Create(AOwner);
  FRadius := 0.5;
  FSlices := 16;
  FStacks := 16;
  FTop := 90;
  FBottom := -90;
  FStart := 0;
  FStop := 360;
end;

//------------------------------------------------------------------------------

procedure TSphere.BuildList;

var
  V1, V2, N1 : TAffineVector;
  AngTop, AngBottom,
  AngStart, AngStop,
  StepV, StepH,
  SinP, CosP,
  SinP2, CosP2,
  SinT, CosT,
  Phi, Phi2,
  Theta: Extended;
  I, J: Integer;
  DoReverse: Boolean;

begin
  inherited BuildList;
  DoReverse := FNormalDirection = ndInside;
  if DoReverse then glFrontFace(GL_CW)
               else glFrontFace(GL_CCW);

  // common settings
  AngTop := DegToRad(FTop);
  AngBottom := DegToRad(FBottom);
  AngStart := DegToRad(FStart);
  AngStop := DegToRad(FStop);
  StepH := (AngStop - AngStart) / FSlices;
  StepV := (AngTop - AngBottom) / FStacks;
  glScalef(Radius, Radius, Radius);

  // top cap
  if (FTop < 90) and (FTopCap in [ctCenter, ctFlat]) then
  begin
    glBegin(GL_TRIANGLE_FAN);
    SinCos(AngTop, SinP, CosP);
    glTexCoord2f(0.5, 0.5);
    if DoReverse then glNormal3f(0, -1, 0)
                 else glNormal3f(0, 1, 0);
    if FTopCap = ctCenter then glVertex3f(0, 0, 0)
                          else
    begin
      glVertex3f(0, SinP, 0);
      if DoReverse then N1 := MakeAffineVector([0, -1, 0])
                   else N1 := MakeAffineVector([0, 1, 0]);
    end;
    V1[1] := SinP;
    Theta := AngStart;
    for I := 0 to FSlices do
    begin
      SinCos(Theta, SinT, CosT);
      V1[0] := CosP * SinT;
      V1[2] := CosP * CosT;
      if FTopCap = ctCenter then
      begin
        N1 := VectorPerpendicular(MakeAffineVector([0, 1, 0]), V1);
        if DoReverse then VectorNegate(N1);
      end;
      glTexCoord2f(SinT / 2 + 0.5, CosT / 2 + 0.5);
      glNormal3fv(@N1);
      glVertex3fv(@V1);
      Theta := Theta + StepH;
    end;
    glEnd;
  end;

  // main body
  Phi := AngTop;
  Phi2 := Phi - StepV;
  for J := 0 to FStacks - 1 do
  begin
    Theta := AngStart;
    SinCos(Phi, SinP, CosP);
    SinCos(Phi2, SinP2, CosP2);
    V1[1] := SinP;
    V2[1] := SinP2;
    glBegin(GL_TRIANGLE_STRIP);
    for I := 0 to FSlices do
    begin
      SinCos(Theta, SinT, CosT);
      V1[0] := CosP * SinT;
      V2[0] := CosP2 * SinT;
      V1[2] := CosP * CosT;
      V2[2] := CosP2 * CosT;
      glTexCoord2f(I / FSlices, J / (FStacks - 1));
      if DoReverse then
      begin
        N1 := V1;
        VectorNegate(N1);
        glNormal3fv(@N1);
      end
      else glNormal3fv(@V1);
      glVertex3fv(@V1);

      glTexCoord2f(I / FSlices, (J + 1) / (FStacks - 1));
            if DoReverse then
      begin
        N1 := V2;
        VectorNegate(N1);
        glNormal3fv(@N1);
      end
      else glNormal3fv(@V2);
      glVertex3fv(@V2);

      Theta := Theta+StepH;
    end;
    glEnd;
    Phi := Phi2;
    Phi2 := Phi2 - StepV;
  end;

  // bottom cap
  if (FBottom > -90) and (FBottomCap in [ctCenter, ctFlat]) then
  begin
    glBegin(GL_TRIANGLE_FAN);
    SinCos(AngBottom, SinP, CosP);
    glTexCoord2f(0.5, 0.5);
    if DoReverse then glNormal3f(0, 1, 0)
                 else glNormal3f(0, -1, 0);
    if FBottomCap = ctCenter then glVertex3f(0, 0, 0)
                             else
    begin
      glVertex3f(0, SinP, 0);
      if DoReverse then N1 := MakeAffineVector([0, -1, 0])
                   else N1 := MakeAffineVector([0, 1, 0]);
    end;
    V1[1] := SinP;
    Theta := AngStop;
    for I := 0 to FSlices do
    begin
      SinCos(Theta, SinT, CosT);
      V1[0] := CosP * SinT;
      V1[2] := CosP * CosT;
      if FTopCap = ctCenter then
      begin
        N1 := VectorPerpendicular(MakeAffineVector([0, -1, 0]), V1);
        if DoReverse then VectorNegate(N1);
      end;
      glTexCoord2f(SinT / 2 + 0.5, CosT / 2 + 0.5);
      glNormal3fv(@N1);
      glVertex3fv(@V1);
      Theta := Theta - StepH;
    end;
    // restore face winding
    glEnd;
  end;
  glFrontFace(GL_CCW);
end;

//------------------------------------------------------------------------------

procedure TSphere.SetBottom(AValue: TAngleLimit1);

begin
  if FBottom <> AValue then
  begin
    if FTop < AValue then ShowError(glsSphereTopBottom);
    FBottom := AValue;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TSphere.SetBottomCap(AValue: TCapType);

begin
  if FBottomCap <> AValue then
  begin
    FBottomCap := AValue;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TSphere.SetRadius(AValue:TGLFloat);

begin
  if AValue <> FRadius then
  begin
    FRadius := AValue;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TSphere.SetSlices(AValue:TGLInt);

begin
  if AValue <> FSlices then
  begin
    FSlices := AValue;
    if FSlices = 0 then FSlices := 1;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TSphere.SetStacks(AValue:TGLInt);

begin
  if AValue <> FStacks then
  begin
    FStacks := AValue;
    if FStacks = 0 then FStacks := 1;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TSphere.SetStart(AValue: TAngleLimit2);

begin
  if FStart <> AValue then
  begin
    if AValue > FStop then ShowError(glsSphereStartStop);
    FStart := AValue;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TSphere.SetStop(AValue: TAngleLimit2);

begin
  if FStop <> AValue then
  begin
    if AValue < FStart then ShowError(glsSphereStartStop);
    FStop := AValue;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TSphere.SetTop(AValue: TAngleLimit1);

begin
  if FTop <> AValue then
  begin
    if AValue < FBottom then ShowError(glsSphereTopBottom);
    FTop := AValue;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TSphere.SetTopCap(AValue: TCapType);

begin
  if FTopCap <> AValue then
  begin
    FTopCap := AValue;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TSphere.Assign(Source:TPersistent);

begin
  if assigned(Source) and (Source is TSphere) then
  begin
    FRadius := TSphere(Source).FRadius;
    FSlices := TSphere(Source).FSlices;
    FStacks := TSphere(Source).FStacks;
    FBottom := TSphere(Source).FBottom;
    FTop := TSphere(Source).FTop;
    FStart := TSphere(Source).FStart;
    FStop := TSphere(Source).FStop;
  end;
  inherited Assign(Source);
end;

//----------------- TDisk ------------------------------------------------------

constructor TDisk.Create(AOwner:TComponent);

begin
  inherited Create(AOwner);
  FOuterRadius := 0.5;
  FInnerRadius := 0;
  FSlices := 16;
  FLoops := 16;
  FStartAngle := 0;
  FSweepAngle := 360;
end;

//------------------------------------------------------------------------------

procedure TDisk.BuildList;

var Quadric: PGLUquadricObj;

begin
  inherited BuildList;
  Quadric := gluNewQuadric();
  SetupQuadricParams(Quadric);
  gluPartialDisk(Quadric, FInnerRadius, FOuterRadius, FSlices, FLoops, FStartAngle, FSweepAngle);
  gluDeleteQuadric(Quadric);
end;

//------------------------------------------------------------------------------

procedure TDisk.SetOuterRadius(AValue:TGLFloat);

begin
  if AValue <> FOuterRadius then
  begin
    FOuterRadius := AValue;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TDisk.SetInnerRadius(AValue:TGLFloat);

begin
  if AValue <> FInnerRadius then
  begin
    FInnerRadius := AValue;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TDisk.SetSlices(AValue:TGLInt);

begin
  if AValue <> FSlices then
  begin
    FSlices := AValue;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TDisk.SetLoops(AValue:TGLInt);

begin
  if AValue <> FLoops then
  begin
    FLoops := AValue;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TDisk.SetStartAngle(AValue:TGLFloat);

begin
  if AValue <> FStartAngle then
  begin
    FStartAngle := AValue;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TDisk.SetSweepAngle(AValue:TGLFloat);

begin
  if AValue <> FSweepAngle then
  begin
    FSweepAngle := AValue;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TDisk.Assign(Source:TPersistent);

begin
  if assigned(Source) and (Source is TDisk) then
  begin
    FOuterRadius := TDisk(Source).FOuterRadius;
    FInnerRadius := TDisk(Source).FInnerRadius;
    FSlices := TDisk(Source).FSlices;
    FLoops := TDisk(Source).FLoops;
    FStartAngle := TDisk(Source).FStartAngle;
    FSweepAngle := TDisk(Source).FSweepAngle;
  end;
  inherited Assign(Source);
end;

//----------------- TCylinderBase ----------------------------------------------

constructor TCylinderBase.Create(AOwner:TComponent);

begin
  inherited Create(AOwner);
  FBottomRadius := 0.5;
  FHeight := 1;
  FSlices := 16;
  FStacks := 16;
end;

//------------------------------------------------------------------------------

procedure TCylinderBase.SetBottomRadius(AValue:TGLFloat);

begin
  if AValue <> FBottomRadius then
  begin
    FBottomRadius := AValue;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TCylinderBase.SetHeight(AValue:TGLFloat);

begin
  if AValue <> FHeight then
  begin
    FHeight := AValue;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TCylinderBase.SetSlices(AValue:TGLInt);

begin
  if AValue <> FSlices then
  begin
    FSlices := AValue;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TCylinderBase.SetStacks(AValue:TGLInt);

begin
  if AValue <> FStacks then
  begin
    FStacks := AValue;
    StructureChanged;
  end;
end;


//------------------------------------------------------------------------------

procedure TCylinderBase.Assign(Source:TPersistent);

begin
  if assigned(Source) and (Source is TCylinderBase) then
  begin
    FBottomRadius := TCylinderBase(Source).FBottomRadius;
    FSlices := TCylinderBase(Source).FSlices;
    FStacks := TCylinderBase(Source).FStacks;
    FHeight := TCylinderBase(Source).FHeight;
  end;
  inherited Assign(Source);
end;

//----------------- TCone ------------------------------------------------------

constructor TCone.Create(AOwner:TComponent);

begin
  inherited Create(AOwner);
  FParts := [coSides, coBottom];
end;

//------------------------------------------------------------------------------

procedure TCone.BuildList;

var Quadric: PGLUquadricObj;

begin
  inherited BuildList;
  Quadric := gluNewQuadric();
  SetupQuadricParams(Quadric);
  glRotated(-90, 1, 0, 0);
  glTranslated(0, 0, -FHeight/2);
  if coSides in FParts then gluCylinder(Quadric, BottomRadius, 0, Height, Slices, Stacks);
  if coBottom in FParts then
  begin
    case FNormalDirection of
      ndInside  : gluQuadricOrientation(Quadric, GLU_OUTSIDE);  //swap orientation because top of a disk is defined as outside
      ndOutside : gluQuadricOrientation(Quadric, GLU_INSIDE);
    end;
    gluDisk(Quadric, 0, BottomRadius, Slices, Round(FStacks/FHeight*FBottomRadius));
  end;
  gluDeleteQuadric(Quadric);
end;

//------------------------------------------------------------------------------

procedure TCone.SetParts(AValue:TConeParts);

begin
  if AValue <> FParts then
  begin
    FParts := AValue;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TCone.Assign(Source: TPersistent);

begin
  if assigned(Source) and (Source is TCone) then
  begin
    FParts := TCone(Source).FParts;
  end;
  inherited Assign(Source);
end;

//----------------- TCylinder --------------------------------------------------

constructor TCylinder.Create(AOwner:TComponent);

begin
  inherited Create(AOwner);
  FTopRadius := 0.5;
  FParts := [cySides, cyBottom, cyTop];
end;

//------------------------------------------------------------------------------

procedure TCylinder.BuildList;

var Quadric : PGLUquadricObj;

begin
  inherited BuildList;
  Quadric := gluNewQuadric;
  SetupQuadricParams(Quadric);
  glRotatef(-90, 1, 0, 0);
  glTranslated(0, 0, -FHeight/2);
  if cySides in FParts then gluCylinder(Quadric, FBottomRadius, FTopRadius, FHeight, FSlices, FStacks);
  if cyTop in FParts then
  begin
    glPushMatrix;
    glTranslatef(0, 0, FHeight);
    gluDisk(Quadric, 0, FTopRadius, FSlices, Round(FStacks/FHeight*FTopRadius));
    glPopMatrix;
  end;
  if cyBottom in FParts then
  begin
    //swap orientation because top of a disk is defined as outside
    case FNormalDirection of
      ndInside  : gluQuadricOrientation(Quadric, GLU_OUTSIDE);
      ndOutside : gluQuadricOrientation(Quadric, GLU_INSIDE);
    end;
    gluDisk(Quadric, 0, FBottomRadius, FSlices, Round(FStacks/FHeight*FBottomRadius));
  end;
  gluDeleteQuadric(Quadric);
end;

//------------------------------------------------------------------------------

procedure TCylinder.SetTopRadius(AValue: TGLFloat);

begin
  if AValue <> FTopRadius then
  begin
    FTopRadius := AValue;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TCylinder.SetParts(AValue: TCylinderParts);

begin
  if AValue <> FParts then
  begin
    FParts := AValue;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TCylinder.Assign(Source: TPersistent);

begin
  if assigned(SOurce) and (Source is TCylinder) then
  begin
    FParts := TCylinder(Source).FParts;
    FTopRadius := TCylinder(Source).FTopRadius;
  end;
  inherited Assign(Source);
end;

//----------------- TVertexList ------------------------------------------------

constructor TVertexList.Create(AOwner: TSceneObject);

begin
  inherited Create;
  FOwner := AOwner;
  FValues := nil;
  FSize := 0;
  // precalculate size of a complete vertex entry
  FEntrySize := SizeOf(TVertex)+SizeOf(TNormal)+SizeOf(TColorVector)+SizeOf(TTexPoint);
end;

//------------------------------------------------------------------------------

destructor TVertexList.Destroy;

begin
  FreeMem(FValues, FSize);
  inherited destroy;
end;

//------------------------------------------------------------------------------

function TVertexList.GetCount: Integer;

begin
  Result := FSize div FEntrySize;
end;

//------------------------------------------------------------------------------

function TVertexList.GetFirstColor: PGLFLoat;

const Entry : Integer = 2;

begin
  Result := @FValues[Entry];
end;

//------------------------------------------------------------------------------

function TVertexList.GetFirstEntry: PGLFloat;

begin
  Result := Pointer(FValues);
end;

//------------------------------------------------------------------------------

function TVertexList.GetFirstNormal: PGLFLoat;

const Entry : Integer = 6;

begin
  Result := @FValues[Entry];
end;

//------------------------------------------------------------------------------

function TVertexList.GetFirstVertex: PGLFLoat;

const Entry : Integer = 9;

begin
  Result := @FValues[Entry];
end;

//------------------------------------------------------------------------------

procedure TVertexList.Update;

begin
  if assigned(FOWner) then FOwner.Update;
end;

//------------------------------------------------------------------------------

procedure TVertexList.DefineProperties(Filer:TFiler);

begin
  Filer.DefineProperty('Items', ReadItems, WriteItems, Count>0);
end;

//------------------------------------------------------------------------------

procedure TVertexList.ReadItems(Reader: TReader);

begin
  Clear;
  Reader.ReadListBegin;
  //while not Reader.EndOfValues do FValues.Add(Pointer(Reader.ReadInteger));
  Reader.ReadListEnd;
end;

//------------------------------------------------------------------------------

procedure  TVertexList.WriteItems(Writer: TWriter);

//var I : Integer;

begin
  Writer.WriteListBegin;
  //for i := 0 to FValues.Count-1 do Writer.WriteInteger(LongInt(FValues[i]));
  Writer.WriteListEnd;
end;

//------------------------------------------------------------------------------

procedure TVertexList.AddVertex(Vertex: PVertex; Normal: PNormal; Color: PColorVector; TexPoint: PTexPoint);

const Empty : TVector = (0, 0, 0, 0);

var Dest : ^Byte;

begin
  if Vertex = nil then raise Exception.Create('Cannot add an empty vertex.');
  // extend memory space
  ReallocMem(FValues, (FSize+FEntrySize));
  // calculate destination address for new vertex data
  Dest := Pointer(FValues);
  Inc(Dest, FSize);         // inc in Byte units
  Inc(FSize, FEntrySize);
  // store texture coordinates if given
  if assigned(TexPoint) then Move(TexPoint^, Dest^, SizeOf(TTexPoint))
                        else Move(Empty, Dest^, SizeOf(TTexPoint));
  Inc(Dest, SizeOf(TTexPoint));
  // store color components if given
  if assigned(Color) then Move(Color^, Dest^, SizeOf(TColorVector))
                     else Move(Empty, Dest^, SizeOf(TColorVector));
  Inc(Dest, SizeOf(TColorVector));
  // store normal vector if given
  if assigned(Normal) then Move(Normal^, Dest^, SizeOf(TNormal))
                      else Move(Empty, Dest^, SizeOf(TNormal));
  Inc(Dest, SizeOf(TNormal));
  // store vertex
  Move(Vertex^, Dest^, SizeOf(TVertex));
  FOwner.StructureChanged;
end;

//------------------------------------------------------------------------------

procedure TVertexList.Clear;

begin
  FreeMem(FValues, FSize);
  FSize := 0;
  FValues := nil;
  FOwner.StructureChanged;
end;

//------------------------------------------------------------------------------

procedure TVertexList.Assign(Source: TPersistent);

begin
  if assigned(Source) and (Source is TVertexList) then
  begin
    ReallocMem(FValues, TVertexList(Source).FSize);
    FSize := TVertexList(Source).FSize;
    Move(TVertexList(Source).FValues^, FValues^, FSize);
  end
  else inherited Assign(Source);
end;

//----------------- TMesh ------------------------------------------------------

constructor TMesh.Create(AOwner:TComponent);

begin
  inherited Create(AOwner);
  FVertices := TVertexList.Create(Self);
  FVertices.AddVertex(@XVector, @ZVector, nil, nil);
  FVertices.AddVertex(@ZVector, @ZVector, nil, nil);
  FVertices.AddVertex(@YVector, @ZVector, nil, nil);
  FVertexmode := vmVNCT;         //should change this later to default to vmVN. But need to
end;                           //change GLMeshPropform so that it greys out unused vertex info

//------------------------------------------------------------------------------

destructor TMesh.Destroy;

begin
  FVertices.Free;
  inherited Destroy;
end;

//------------------------------------------------------------------------------

procedure TMesh.BuildList;

var VertexCount : Longint;

begin
  inherited BuildList;
  glPushAttrib(GL_POLYGON_BIT);
  glFrontFace(GL_CW);
  case FVertexMode of
    vmV    : glInterleavedArrays(GL_V3F, FVertices.EntrySize, FVertices.FirstVertex);
    vmVN   : glInterleavedArrays(GL_N3F_V3F, FVertices.EntrySize, FVertices.FirstNormal);
    vmVNC  : glInterleavedArrays(GL_C4F_N3F_V3F, FVertices.EntrySize, FVertices.FirstColor);
    vmVNCT : glInterleavedArrays(GL_T2F_C4F_N3F_V3F, 0, FVertices.FirstEntry);
  end;
  VertexCount := FVertices.Count;
  case FMode of
    mmTriangleStrip : glDrawArrays(GL_TRIANGLE_STRIP, 0, VertexCount);
    mmTriangleFan   : glDrawArrays(GL_TRIANGLE_FAN, 0, VertexCount);
    mmTriangles     : glDrawArrays(GL_TRIANGLES, 0, VertexCount);
    mmQuadStrip     : glDrawArrays(GL_QUAD_STRIP, 0, VertexCount);
    mmQuads         : glDrawArrays(GL_QUADS, 0, VertexCount);
    mmPolygon       : glDrawArrays(GL_POLYGON, 0, VertexCount);
  end;
  glPopAttrib;
end;

//------------------------------------------------------------------------------

procedure TMesh.SetMode(AValue: TMeshMode);

begin
  if AValue <> FMode then
  begin
    FMode := AValue;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TMesh.SetVertices(AValue: TVertexList);

begin
  if AValue <> FVertices then
  begin
    FVertices.Assign(AValue);
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TMesh.SetVertexMode(AValue: TVertexMode);

begin
  if AValue<>FVertexMode then
  begin
    FVertexMode := AValue;
    StructureChanged;
  end;
end;


//------------------------------------------------------------------------------

function TMesh.CalcPlaneNormal(x1, y1, z1, x2, y2, z2, x3, y3, z3: TGLFloat): TAffineVector;

var V1, V2 : TAffineVector;

begin
  V1[0] := x2-x1; V1[1] := y2-y1; V1[2] := z2-z1;
  V2[0] := x3-x1; V2[1] := y3-y1; V2[2] := z3-z1;
  Result := VectorCrossProduct(V1, V2);
  VectorNormalize(Result);
end;

//------------------------------------------------------------------------------

procedure TMesh.CalcNormals(Frontface: TFaceWinding);

var x1, y1, z1, 
    x2, y2, z2, 
    x3, y3, z3  : TGLFloat;
    Vn        : TAffineFltVector;
    I         : Integer;

begin
  case FMode of
    mmTriangles:
      with Vertices do
      begin
        for I := 0 to (Size-1) div (36*SizeOf(TGLFloat)) do
        begin
          x1 := FValues[I*36+ 9]; y1 := FValues[I*36+10]; z1 := FValues[I*36+11];
          x2 := FValues[I*36+21]; y2 := FValues[I*36+22]; z2 := FValues[I*36+23];
          x3 := FValues[I*36+33]; y3 := FValues[I*36+34]; z3 := FValues[I*36+35];
          if FrontFace=fwCounterClockWise then
            Vn := CalcPlaneNormal(x1, y1, z1, x2, y2, z2, x3, y3, z3)
          else
            Vn := CalcPlaneNormal(x3, y3, z3, x2, y2, z2, x1, y1, z1);
          FValues[I*36+ 6] := Vn[0]; FValues[I*36+ 7] := Vn[1]; FValues[I*36+ 8] := Vn[2];
          FValues[I*36+18] := Vn[0]; FValues[I*36+19] := Vn[1]; FValues[I*36+20] := Vn[2];
          FValues[I*36+30] := Vn[0]; FValues[I*36+31] := Vn[1]; FValues[I*36+32] := Vn[2];
        end;
      end;
    mmQuads:
      with Vertices do
      begin
        for I := 0 to (Size-1) div (48*SizeOf(TGLFloat))  do
        begin
          x1 := FValues[I*48+ 9]; y1 := FValues[I*48+10]; z1 := FValues[I*48+11];
          x2 := FValues[I*48+21]; y2 := FValues[I*48+22]; z2 := FValues[I*48+23];
          x3 := FValues[I*48+33]; y3 := FValues[I*48+34]; z3 := FValues[I*48+35];
          if FrontFace=fwCounterClockWise then
            Vn := CalcPlaneNormal(x1, y1, z1, x2, y2, z2, x3, y3, z3)
          else
            Vn := CalcPlaneNormal(x3, y3, z3, x2, y2, z2, x1, y1, z1);
          FValues[I*48+ 6] := Vn[0]; FValues[I*48+ 7] := Vn[1]; FValues[I*48+ 8] := Vn[2];
          FValues[I*48+18] := Vn[0]; FValues[I*48+19] := Vn[1]; FValues[I*48+20] := Vn[2];
          FValues[I*48+30] := Vn[0]; FValues[I*48+31] := Vn[1]; FValues[I*48+32] := Vn[2];
          FValues[I*48+42] := Vn[0]; FValues[I*48+43] := Vn[1]; FValues[I*48+44] := Vn[2];
        end;
      end;
    mmPolygon:
      with Vertices do
      begin
        I := 0;
        x1 := FValues[I+ 9]; y1 := FValues[I+10]; z1 := FValues[I+11];
        x2 := FValues[I+21]; y2 := FValues[I+22]; z2 := FValues[I+23];
        x3 := FValues[I+33]; y3 := FValues[I+34]; z3 := FValues[I+35];
        if FrontFace=fwCounterClockWise then
          Vn := CalcPlaneNormal(x1, y1, z1, x2, y2, z2, x3, y3, z3)
        else
          Vn := CalcPlaneNormal(x3, y3, z3, x2, y2, z2, x1, y1, z1);
        for I := 0 to (Size-1) div (12*SizeOf(TGLFloat)) do
        begin
          FValues[I*12+6] := Vn[0];
          FValues[I*12+7] := Vn[1];
          FValues[I*12+8] := Vn[2];
        end;
      end;
  else ShowMessage('Sorry. Calculating normals only supported for mmTriangles, mmQuads and mmPolygon at present');
  end;
end;

//------------------------------------------------------------------------------

procedure TMesh.Assign(Source:TPersistent);

begin
  if assigned(Source) and (Source is TMesh) then
  begin
    FVertices.Assign(TMesh(Source).Vertices);
    FMode := TMesh(Source).FMode;
  end
  else inherited Assign(Source);
end;

//----------------- TFaceGroups -------------------------------------------------

procedure TFaceGroups.Clear;

var I : Integer;
    FaceGroup : PFaceGroup;

begin
  for I := 0 to Count-1 do
  begin
    FaceGroup := Get(I);
    FaceGroup.FaceProperties.Free;
    FreeMem(FaceGroup.Indices);
    Dispose(FaceGroup);
  end;
  inherited;
end;

//------------------------------------------------------------------------------

function TFaceGroups.GetFaceGroup(Index: Integer): PFaceGroup;
begin
  Result := Get(Index);
end;

//----------------- TFreeForm --------------------------------------------------

constructor TFreeForm.Create(AOwner:TComponent);

begin
  inherited Create(AOwner);
  FObjects := TList.Create;
end;

//------------------------------------------------------------------------------

destructor TFreeForm.Destroy;

begin
  Clear;
  FObjects.Free;
  inherited Destroy;
end;

//------------------------------------------------------------------------------

procedure TFreeForm.BuildList;

var I, J : Integer;
    Mesh : PMeshObject;

begin
  inherited BuildList;
  glPushAttrib(GL_POLYGON_BIT);
  glFrontFace(FWinding);
  // move scene so that its geometric center is at origin
  glTranslatef(-FCenter[0], -FCenter[1], -FCenter[2]);

  // go through all loaded meshs
  for I := 0 to FObjects.Count-1 do
  begin
    Mesh := FObjects[I];

    // there must always be a list of vertices
    glVertexPointer(3, GL_FLOAT, 0, Mesh.Vertices);
    glEnableClientState(GL_VERTEX_ARRAY);

    // enable the normal array if available
    if assigned(Mesh.Normals) then
    begin
      glNormalPointer(GL_FLOAT, 0, Mesh.Normals);
      glEnableClientState(GL_NORMAL_ARRAY);
    end
    else glDisableCLientState(GL_NORMAL_ARRAY);

    // enable the texture coordinates array if available
    if assigned(Mesh.TexCoords) then
    begin
      glTexCoordPointer(2, GL_FLOAT, 0, Mesh.TexCoords);
      glEnableClientState(GL_TEXTURE_COORD_ARRAY);
      {Scene.CurrentViewer.RequestedState([stTexture2D]);
      Material.Texture.Apply;}
    end
    else
    begin
      glDisableCLientState(GL_TEXTURE_COORD_ARRAY);
      //Scene.CurrentViewer.UnnecessaryState([stTexture2D]);
    end;

    for J := 0 to Mesh.FaceGroups.Count - 1 do
      with Mesh.FaceGroups[J]^ do
      begin
        FaceProperties.Apply(GL_FRONT);
        case Mesh.Mode of
          mmTriangleStrip:
            glDrawElements(GL_TRIANGLE_STRIP, IndexCount, GL_UNSIGNED_INT, Indices);
          mmTriangleFan:
            glDrawElements(GL_TRIANGLE_FAN, IndexCount, GL_UNSIGNED_INT, Indices);
          mmTriangles:
            glDrawElements(GL_TRIANGLES, IndexCount, GL_UNSIGNED_INT, Indices);
          mmQuadStrip:
            glDrawElements(GL_QUAD_STRIP, IndexCount, GL_UNSIGNED_INT, Indices);
          mmQuads:
            glDrawElements(GL_QUADS, IndexCount, GL_UNSIGNED_INT, Indices);
          mmPolygon:
            glDrawElements(GL_POLYGON, IndexCount, GL_UNSIGNED_INT, Indices);
        end;
      end;
  end;
  glPopAttrib;
end;

//------------------------------------------------------------------------------

procedure TFreeForm.Clear;

var Mesh : PMeshObject;
    I    : Integer;

begin
  for I := 0 to FObjects.Count - 1 do
  begin
    Mesh := FObjects[I];
    if assigned(Mesh.Vertices) then FreeMem(Mesh.Vertices);
    if assigned(Mesh.Normals) then FreeMem(Mesh.Normals);
    if assigned(Mesh.TexCoords) then FreeMem(Mesh.TexCoords);
    Mesh.FaceGroups.Free;
    Dispose(Mesh);
  end;
  FObjects.Clear;
  FCenter := NullVector;
end;

//------------------------------------------------------------------------------

procedure TFreeForm.LoadFromFile(const Filename: String);

var Ext           : String;
    NewVectorFile : TVectorFile;
    VectorFileClass : TVectorFileClass;

begin
  if FileName <> '' then
  begin
    Clear;
    Ext := ExtractFileExt(Filename);
    Delete(Ext, 1, 1);
    VectorFileClass := GetVectorFileFormats.FindExt(Ext);
    if VectorFileClass = nil then raise EInvalidVectorFile.CreateFmt(SUnknownExtension, [Ext]);
    NewVectorFile := VectorFileClass.Create(Self);
    try
      if assigned(Scene) then Scene.BeginUpdate;
      NewVectorFile.LoadFromFile(Filename);
      if assigned(Scene) then Scene.EndUpdate;
    except
      NewVectorFile.Free;
      raise;
    end;
  end;
  StructureChanged;
end;

//------------------------------------------------------------------------------

function TFreeForm.AddMeshObject(AObject: PMeshObject): Integer;

begin
  Result := FObjects.Add(AObject);
end;

//------------------------------------------------------------------------------

procedure TFreeForm.Assign(Source:TPersistent);

begin
  if assigned(Source) and (Source is TMesh) then
  begin
{    FVertices.Assign(TMesh(Source).Vertices);
    FMode := TMesh(Source).FMode;}
  end;
  inherited Assign(Source);
end;

//----------------- TTorus -----------------------------------------------------

constructor TTorus.Create(AOwner:TComponent);

begin
  inherited Create(AOwner);
  FRings := 25;
  FSides := 15;
  FMinorRadius := 0.1;
  FMajorRadius := 0.4;
end;

//------------------------------------------------------------------------------

procedure TTorus.BuildList;

var I, J         : Integer;
    Theta, Phi, 
    Theta1, 
    cosPhi, sinPhi, dist : TGLFloat;
    cosTheta, sinTheta: TGLFloat;
    cosTheta1, sinTheta1: TGLFloat;
    ringDelta, sideDelta: TGLFloat;

begin
  inherited BuildList;
  // handle texture generation
  Material.Texture.InitAutoTexture(nil);

  ringDelta := 2*Pi/FRings;
  sideDelta := 2*Pi/FSides;
  theta := 0;
  cosTheta := 1;
  sinTheta := 0;
  for I := FRings-1 downto 0 do
  begin
    theta1 := theta+ringDelta;
    cosTheta1 := cos(theta1);
    sinTheta1 := sin(theta1);
    glBegin(GL_QUAD_STRIP);
    phi := 0;
    for J := FSides downto 0 do
    begin
      phi := phi+sideDelta;
      cosPhi := cos(phi);
      sinPhi := sin(phi);
      dist := FMajorRadius+FMinorRadius*cosPhi;

      glNormal3f(cosTheta1*cosPhi, -sinTheta1*cosPhi, sinPhi);
      glVertex3f(cosTheta1*dist, -sinTheta1*dist, FMinorRadius*sinPhi);
      glNormal3f(cosTheta*cosPhi, -sinTheta*cosPhi, sinPhi);
      glVertex3f(cosTheta*dist, -sinTheta*dist, FMinorRadius*sinPhi);
    end;
    glEnd;
    theta := theta1;
    cosTheta := cosTheta1;
    sinTheta := sinTheta1;
  end;
  Material.Texture.DisableAutoTexture;
end;

//------------------------------------------------------------------------------

procedure TTorus.SetMajorRadius(AValue: Single);

begin
  if FMajorRadius <> AValue then
  begin
    FMajorRadius := AValue;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TTorus.SetMinorRadius(AValue: Single);

begin
  if FMinorRadius <> AValue then
  begin
    FMinorRadius := AValue;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TTorus.SetRings(AValue: Cardinal);

begin
  if FRings <> AValue then
  begin
    FRings := AValue;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TTorus.SetSides(AValue: Cardinal);

begin
  if FSides <> AValue then
  begin
    FSides := AValue;
    StructureChanged;
  end;
end;

//----------------- TTeapot ----------------------------------------------------

constructor TTeapot.Create(AOwner: TComponent);

begin
  inherited Create(AOwner);
  FGrid := 5;
end;

//------------------------------------------------------------------------------

procedure TTeapot.BuildList;

const PatchData : array[0..9, 0..15] of Integer =
      ((102, 103, 104, 105,  4,  5,  6,  7,  8,  9, 10, 11, 12, 13, 14, 15), // rim
       ( 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27), // body
       ( 24, 25, 26, 27, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40), // body
       ( 96, 96, 96, 96, 97, 98, 99, 100, 101, 101, 101, 101,  0,  1,  2,  3), // lid
       (  0,  1,  2,  3, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117), // lid
       (118, 118, 118, 118, 124, 122, 119, 121, 123, 126, 125, 120, 40, 39, 38, 37), // bottom
       ( 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54, 55, 56), // handle
       ( 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 28, 65, 66, 67), // handle
       ( 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83), // spout
       ( 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95));// spout

      CPData : array[0..126, 0..2] of TGLFloat =
      ((0.2, 0, 2.7), (0.2, -0.112, 2.7), (0.112, -0.2, 2.7), (0, -0.2, 2.7), (1.3375, 0, 2.53125), 
       (1.3375, -0.749, 2.53125), (0.749, -1.3375, 2.53125), (0, -1.3375, 2.53125), 
       (1.4375, 0, 2.53125), (1.4375, -0.805, 2.53125), (0.805, -1.4375, 2.53125), 
       (0, -1.4375, 2.53125), (1.5, 0, 2.4), (1.5, -0.84, 2.4), (0.84, -1.5, 2.4), (0, -1.5, 2.4), 
       (1.75, 0, 1.875), (1.75, -0.98, 1.875), (0.98, -1.75, 1.875), (0, -1.75, 1.875), (2, 0, 1.35), 
       (2, -1.12, 1.35), (1.12, -2, 1.35), (0, -2, 1.35), (2, 0, 0.9), (2, -1.12, 0.9), (1.12, -2, 0.9), 
       (0, -2, 0.9), (-2, 0, 0.9), (2, 0, 0.45), (2, -1.12, 0.45), (1.12, -2, 0.45), (0, -2, 0.45), 
       (1.5, 0, 0.225), (1.5, -0.84, 0.225), (0.84, -1.5, 0.225), (0, -1.5, 0.225), (1.5, 0, 0.15), 
       (1.5, -0.84, 0.15), (0.84, -1.5, 0.15), (0, -1.5, 0.15), (-1.6, 0, 2.025), (-1.6, -0.3, 2.025), 
       (-1.5, -0.3, 2.25), (-1.5, 0, 2.25), (-2.3, 0, 2.025), (-2.3, -0.3, 2.025), (-2.5, -0.3, 2.25), 
       (-2.5, 0, 2.25), (-2.7, 0, 2.025), (-2.7, -0.3, 2.025), (-3, -0.3, 2.25), (-3, 0, 2.25), 
       (-2.7, 0, 1.8), (-2.7, -0.3, 1.8), (-3, -0.3, 1.8), (-3, 0, 1.8), (-2.7, 0, 1.575), 
       (-2.7, -0.3, 1.575), (-3, -0.3, 1.35), (-3, 0, 1.35), (-2.5, 0, 1.125), (-2.5, -0.3, 1.125), 
       (-2.65, -0.3, 0.9375), (-2.65, 0, 0.9375), (-2, -0.3, 0.9), (-1.9, -0.3, 0.6), (-1.9, 0, 0.6), 
       (1.7, 0, 1.425), (1.7, -0.66, 1.425), (1.7, -0.66, 0.6), (1.7, 0, 0.6), (2.6, 0, 1.425), 
       (2.6, -0.66, 1.425), (3.1, -0.66, 0.825), (3.1, 0, 0.825), (2.3, 0, 2.1), (2.3, -0.25, 2.1), 
       (2.4, -0.25, 2.025), (2.4, 0, 2.025), (2.7, 0, 2.4), (2.7, -0.25, 2.4), (3.3, -0.25, 2.4), 
       (3.3, 0, 2.4), (2.8, 0, 2.475), (2.8, -0.25, 2.475), (3.525, -0.25, 2.49375), 
       (3.525, 0, 2.49375), (2.9, 0, 2.475), (2.9, -0.15, 2.475), (3.45, -0.15, 2.5125), 
       (3.45, 0, 2.5125), (2.8, 0, 2.4), (2.8, -0.15, 2.4), (3.2, 0.15, 2.4), (3.2, 0, 2.4), 
       (0, 0, 3.15), (0.8, 0, 3.15), (0.8, -0.45, 3.15), (0.45, -0.8, 3.15), (0, -0.8, 3.15), 
       (0, 0, 2.85), (1.4, 0, 2.4), (1.4, -0.784, 2.4), (0.784, -1.4, 2.4), (0, -1.4, 2.4), 
       (0.4, 0, 2.55), (0.4, -0.224, 2.55), (0.224, -0.4, 2.55), (0, -0.4, 2.55), (1.3, 0, 2.55), 
       (1.3, -0.728, 2.55), (0.728, -1.3, 2.55), (0, -1.3, 2.55), (1.3, 0, 2.4), (1.3, -0.728, 2.4), 
       (0.728, -1.3, 2.4), (0, -1.3, 2.4), (0, 0, 0), (1.425, -0.798, 0), (1.5, 0, 0.075), (1.425, 0, 0), 
       (0.798, -1.425, 0), (0, -1.5, 0.075), (0, -1.425, 0), (1.5, -0.84, 0.075), (0.84, -1.5, 0.075));

      Tex : array[0..1, 0..1, 0..1] of TGLFloat = (((0, 0), (1, 0)), ((0, 1), (1, 1)));

var P, Q, R, S  : array[0..3, 0..3, 0..2] of TGLFloat;
    I, J, K, L, 
    GRD      : Integer;

begin
  inherited BuildList;

  if FGrid < 2 then FGrid := 2;
  GRD := FGrid;
  glPushMatrix;
  glTranslatef(0, -0.25, 0);
  glRotatef(-90, 1, 0, 0);
  glScalef(0.15, 0.15, 0.15);
  glPushAttrib(GL_POLYGON_BIT or GL_ENABLE_BIT or GL_EVAL_BIT);
  glFrontFace(GL_CW);
  glEnable(GL_AUTO_NORMAL);
  glEnable(GL_MAP2_VERTEX_3);
  glEnable(GL_MAP2_TEXTURE_COORD_2);
  for I := 0 to 9 do
  begin
    for J := 0 to 3 do
    begin
      for K := 0 to 3 do
      begin
        for L := 0 to 2 do
        begin
          P[J, K, L] := CPData[PatchData[I, J*4+K], L];
          Q[J, K, L] := CPData[PatchData[I, J*4+(3-K)], L];
          if L = 1 then Q[J, K, L] := -Q[J, K, L];
          if I < 6 then
          begin
            R[J, K, L] := CPData[PatchData[I, J*4+(3-K)], L];
            if L = 0 then R[J, K, L] := -R[J, K, L];
            S[J, K, L] := CPData[PatchData[I, J*4+K], L];
            if L < 2 then S[J, K, L] := -S[J, K, L];
          end;
        end;
      end;
    end;
    glMapGrid2f(GRD, 0, 1, GRD, 0, 1);
    glMap2f(GL_MAP2_TEXTURE_COORD_2, 0, 1, 2, 2, 0, 1, 4, 2, @Tex);
    glMap2f(GL_MAP2_VERTEX_3, 0, 1, 3, 4, 0, 1, 12, 4, @P);
    glEvalMesh2(GL_FILL, 0, GRD, 0, GRD);
    glMap2f(GL_MAP2_VERTEX_3, 0, 1, 3, 4, 0, 1, 12, 4, @Q);
    glEvalMesh2(GL_FILL, 0, GRD, 0, GRD);
    if I < 6 then
    begin
      glMap2f(GL_MAP2_VERTEX_3, 0, 1, 3, 4, 0, 1, 12, 4, @R);
      glEvalMesh2(GL_FILL, 0, GRD, 0, GRD);
      glMap2f(GL_MAP2_VERTEX_3, 0, 1, 3, 4, 0, 1, 12, 4, @S);
      glEvalMesh2(GL_FILL, 0, GRD, 0, GRD);
    end;
  end;
  glPopAttrib;
  glPopMatrix;
end;

//----------------- TSpaceText ----------------------------------------------------

constructor TSpaceText.Create(AOwner:TComponent);

begin
  inherited Create(AOwner);
  FFont := TFont.Create;
  FFont.Name := 'Arial';
  FontChanged := True;
  FExtrusion := 0;
end;

//------------------------------------------------------------------------------

destructor TSpaceText.Destroy;

begin
  FFont.Free;
  FontManager.Release(BaseList);
  inherited Destroy;
end;

//------------------------------------------------------------------------------

procedure TSpaceText.BuildList;

begin
  inherited BuildList;
  if Length(FText) > 0 then
  begin
    // create texture coordinates if necessary
    if not (Material.Texture.Disabled)    and
       not (Material.Texture.IsInherited) then Material.Texture.InitAutoTexture(nil);
    glPushAttrib(GL_POLYGON_BIT);
    glListBase(BaseList);
    glCallLists(Length(FText), GL_UNSIGNED_BYTE, PChar(FText));
    glPopAttrib;
    //Material.Texture.DisableAutoTexture;
  end;
end;

//------------------------------------------------------------------------------

procedure TSpaceText.PrepareObject;

begin
  if FontChanged and (Length(FText) > 0) then
  with FFont do
  begin
    FontManager.Release(BaseList);
    BaseList := FontManager.GetFontBase(Name, Style, FExtrusion);
    FontChanged := False;
  end;
  inherited PrepareObject;
end;

//------------------------------------------------------------------------------

procedure TSpaceText.SetExtrusion(AValue: Single);

begin
  if FExtrusion <> AValue then
  begin
    FExtrusion := AValue;
    FontChanged := True;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------

procedure TSpaceText.SetFont(AFont: TFont);

begin
  FFont.Assign(AFont);
  FontChanged := True;
  StructureChanged;
end;

//------------------------------------------------------------------------------

procedure TSpaceText.SetText(AText: String);

begin
  if FText <> AText then
  begin
    FText := AText;
    StructureChanged;
  end;
end;

//----------------- TDodecahedron ----------------------------------------------

procedure TDodecahedron.BuildList;

const A = 1.61803398875; // (Sqrt(5)+1)/2
      B = 0.61803398875; // (Sqrt(5)-1)/2
      C = 1;

const Vertices : array[0..19, 0..2] of TGLFloat =
      ((-A, 0, B), (-A, 0, -B), (A, 0, -B), (A, 0, B), 
       (B, -A, 0), (-B, -A, 0), (-B, A, 0), (B, A, 0), 
       (0, B, -A), (0, -B, -A), (0, -B, A), (0, B, A), 
       (-C, -C, C), (-C, -C, -C), (C, -C, -C), (C, -C, C), 
       (-C, C, C), (-C, C, -C), (C, C, -C), (C, C, C));

      Polygons : array[0..11, 0..4] of TGLInt =
      (( 0, 12, 10, 11, 16), 
       ( 1, 17, 8, 9, 13), 
       ( 2, 14, 9, 8, 18), 
       ( 3, 19, 11, 10, 15), 
       ( 4, 14, 2, 3, 15), 
       ( 5, 12, 0, 1, 13), 
       ( 6, 17, 1, 0, 16), 
       ( 7, 19, 3, 2, 18), 
       ( 8, 17, 6, 7, 18), 
       ( 9, 14, 4, 5, 13), 
       (10, 12, 5, 4, 15), 
       (11, 19, 7, 6, 16));

var I     : Integer;
    U, V, N : TAffineVector;

begin
  inherited BuildList;

  glScalef(0.3, 0.3, 0.3);
  for I := 0 to 11 do
  begin
    U[0] := Vertices[Polygons[I, 2], 0]-Vertices[Polygons[I, 1], 0];
    U[1] := Vertices[Polygons[I, 2], 1]-Vertices[Polygons[I, 1], 1];
    U[2] := Vertices[Polygons[I, 2], 2]-Vertices[Polygons[I, 1], 2];

    V[0] := Vertices[Polygons[I, 0], 0]-Vertices[Polygons[I, 1], 0];
    V[1] := Vertices[Polygons[I, 0], 1]-Vertices[Polygons[I, 1], 1];
    V[2] := Vertices[Polygons[I, 0], 2]-Vertices[Polygons[I, 1], 2];

    N := VectorCrossProduct(U, V);
    VectorNormalize(N);

    glBegin(GL_TRIANGLE_FAN);
    glNormal3fv(@N);
    glVertex3fv(@Vertices[Polygons[I, 0], 0]);
    glVertex3fv(@Vertices[Polygons[I, 1], 0]);
    glVertex3fv(@Vertices[Polygons[I, 2], 0]);
    glVertex3fv(@Vertices[Polygons[I, 3], 0]);
    glVertex3fv(@Vertices[Polygons[I, 4], 0]);
    glEnd;
  end;
end;

//----------------- TRotationSolid ---------------------------------------------

procedure TRotationSolid.BuildList;

begin
  inherited BuildList;
end;

//----------------- end of predefined objects ----------------------------------

//----------------- TObjectManager -------------------------------------------------------------

constructor TObjectManager.Create(AOwner: TComponent);

begin
  inherited Create(AOwner);
  SceneObjectList := TList.Create;
  FObjectStock := TSceneObject.Create(nil);
  FObjectStock.Name := 'FObjectStock';
  CreateDefaultObjectIcons;
end;

//------------------------------------------------------------------------------

destructor TObjectManager.Destroy;
 
begin
  DestroySceneObjectList;
  FObjectStock.Free;  // scene object instances will be freed by the destructor
  FObjectIcons.Free;
  inherited Destroy;
end;

//------------------------------------------------------------------------------

procedure TObjectManager.Notify(Sender: TPlugInManager; Operation: TOperation; PlugIn: Integer);

begin
end;

//------------------------------------------------------------------------------

function TObjectManager.FindSceneObjectClass(AObjectClass: TSceneObjectClass; ASceneObject: String) : PSceneObjectEntry;

var I     : Integer;
    Found : Boolean;
    
begin
  Result := nil;
  Found := False;
  if AObjectClass = nil then AObjectClass := TExternalObject;
  with SceneObjectList do
  begin
    for I := 0 to Count-1 do
      with TSceneObjectEntry(Items[I]^) do
        if (ObjectClass = AObjectClass) and
           (Length(ASceneObject) = 0)   OR
           (CompareText(Name, ASceneObject) = 0) then
        begin
          Found := True;
          Break;
        end;
    if Found then Result := Items[I];
  end;
end;

//------------------------------------------------------------------------------

function TObjectManager.GetClassFromIndex(Index: Integer): TSceneObjectClass;

begin
  if Index < 0 then Index := 0;
  if Index > SceneObjectList.Count-1 then Index := SceneObjectList.Count-1;
  Result := TSceneObjectEntry(SceneObjectList.Items[Index+1]^).ObjectClass;
end;

//------------------------------------------------------------------------------

function TObjectManager.GetImageIndex(ASceneObject: TSceneObjectClass) : Integer;

var AClassEntry : PSceneObjectEntry;
    AName       : String;

begin
  if ASceneObject = TExternalObject then AName := TExternalObject(ASceneObject).TypeName
                                    else AName := '';
  AClassEntry := FindSceneObjectClass(ASceneObject, AName);
  if assigned(AClassEntry) then Result := AClassEntry^.ImageIndex
                           else Result := 0;
end;

//------------------------------------------------------------------------------

procedure TObjectManager.GetRegisteredSceneObjects(ObjectList: TStringList);

var I : Integer;

begin
  if ObjectList = nil then Exit;
  with ObjectList do
  begin
    Clear;
    for I := 1 to SceneObjectList.Count-1 do
      with TSceneObjectEntry(SceneObjectList.Items[I]^) do Add(Name);
  end;
end;

//------------------------------------------------------------------------------

procedure TObjectManager.RegisterSceneObject(ASceneObject: TSceneObjectClass; AName: String; AImage: HBitmap);

// registers a stock object and adds it to the stock object list

var NewEntry  : PSceneObjectEntry;
    AInstance : TBaseSceneObject;
    Pic       : TPicture;

begin
  if ASceneObject = nil then ASceneObject := TExternalObject;
  with SceneObjectList do
  begin
    // make sure no class is registered twice
    if assigned(FindSceneObjectClass(ASceneObject, AName)) then Exit;
    New(NewEntry);
    Pic := TPicture.Create;
    try
      with NewEntry^ do
      begin
        // object stock stuff
        AInstance := ASceneObject.Create(FObjectStock);
        AInstance.Name := AName;
        FObjectStock.AddChild(AInstance);
        // registered objects list stuff
        ObjectClass := ASceneObject;
        NewEntry^.Name := AName;
        Index := AInstance.Index;
        if AImage <> 0 then
        begin
          Pic.Bitmap.Handle := AImage;
          FObjectIcons.AddMasked(Pic.Bitmap, Pic.Bitmap.Canvas.Pixels[0, 0]);
          ImageIndex := FObjectIcons.Count-1;
        end
        else ImageIndex := 0;
      end;
      Add(NewEntry);
    finally
      Pic.Free;
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure TObjectManager.UnRegisterSceneObject(ASceneObject: TSceneObjectClass; AName: String);

// unregisters a stock object and removes it from the stock object list

var OldEntry : PSceneObjectEntry;
    AObject  : TBaseSceneObject;

begin
  // find the class in the scene object list
  OldEntry := FindSceneObjectClass(ASceneObject, AName);
  // found?
  if assigned(OldEntry) then
  begin
    // yes, so get its instance in "FObjectStock"
    AObject := FObjectStock[OldEntry^.Index];
    // remove its entry from the list of registered objects
    SceneObjectList.Remove(OldEntry);
    // free the instance
    FObjectStock[OldEntry.Index].Free;
    // remove the instance entry from object stock
    FObjectStock.Remove(AObject, False);
    // finally free the memory for the entry 
    Dispose(OldEntry);
  end;
end;

//------------------------------------------------------------------------------

procedure TObjectManager.CreateDefaultObjectIcons;

var Pic : TPicture;

begin
  Pic := TPicture.Create;
  // load first pic to get size
  Pic.Bitmap.Handle := LoadBitmap(HInstance, 'GLS_CROSS_16');
  FObjectIcons := TImageList.CreateSize(Pic.Width, Pic.height);
  with FObjectIcons, Pic.Bitmap.Canvas do
  try
    // There's a more direct way for loading images into the image list, but
    // the image quality suffers too much
    AddMasked(Pic.Bitmap, Pixels[0, 0]); FOverlayIndex := Count-1;
    Overlay(FOverlayIndex, 0); // used as indicator for disabled objects
    Pic.Bitmap.Handle := LoadBitmap(HInstance, 'GLS_UNIVERSE2_16');
    AddMasked(Pic.Bitmap, Pixels[0, 0]); FSceneRootIndex := Count-1;
    Pic.Bitmap.Handle := LoadBitmap(HInstance, 'GLS_CAMERA2_16');
    AddMasked(Pic.Bitmap, Pixels[0, 0]); FCameraRootIndex := Count-1;
    Pic.Bitmap.Handle := LoadBitmap(HInstance, 'GLS_LAMPS2_16');
    AddMasked(Pic.Bitmap, Pixels[0, 0]); FLightsourceRootIndex := Count-1;
    Pic.Bitmap.Handle := LoadBitmap(HInstance, 'GLS_OBJECTS2_16');
    AddMasked(Pic.Bitmap, Pixels[0, 0]); FObjectRootIndex := Count-1;
    AddMasked(Pic.Bitmap, Pixels[0, 0]); FStockObjectRootIndex := Count-1;
  finally
    Pic.Free;
  end;
end;

//------------------------------------------------------------------------------

procedure TObjectManager.DestroySceneObjectList;

var I : Integer;

begin
  with SceneObjectList do
  begin
    for I := 0 to Count-1 do FreeMem(Items[I]);
    Free;
  end;
end;


//----------------- TActor -----------------------------------------------------

constructor TActor.Create(AOwner:TComponent);

begin
  inherited Create(AOwner);
  FNumberFrame := 0;
  FStartFrame := 0;
  FEndFrame := 0;
  FCurrentFrame := 0;

  // 20 frames per second
  FInterval := 100;
  FTimer := TAsyncTimer.Create(Self);
  FTimer.Enabled := False;
  FTimer.Interval := FInterval;
  FTimer.OnTimer := ActionOnTimer;
  //First, we should keep the scene move smoothly, not the actor
  FTimer.TimerThreadPriority := tpNormal;
  FTimer.TakerThreadPriority := tpIdle;
  FAction := False;
end;

//------------------------------------------------------------------------------

destructor TActor.Destroy;
begin
  FTimer.Enabled := False;
  FTimer.free;
  inherited Destroy;
end;

//------------------------------------------------------------------------------

procedure TActor.SetInterval(Value: Integer);
begin
  if Value<>FInterval then
    FInterval := Value;
  if FInterval=0 then
  begin
    FTimer.Enabled := False;
    FTimer.Interval := 0;
    FAction := False;
  end;
end;

//------------------------------------------------------------------------------

procedure TActor.SetAction(Value: Boolean);
begin
  if FInterval=0 then
    exit;
  if Value<>FAction then
  begin
    FAction := Value;
    FTimer.Enabled := FAction;
  end;
end;

//------------------------------------------------------------------------------

procedure TActor.ActionOnTimer(Sender: TObject);
begin
  if (FNumberFrame>1) and (FStartFrame<>FEndFrame) then
    NextFrame(1);
end;


//------------------------------------------------------------------------------
procedure TActor.SetCurrentFrame(Value: Integer);
begin
  if Value<>FCurrentFrame then
  begin
    if Value>NumberFrame-1 then
      FCurrentFrame := NumberFrame-1
    else
      if Value<0 then
        FCurrentFrame := 0
      else
        FCurrentFrame := Value;
    StructureChanged;
  end;
end;

//------------------------------------------------------------------------------
procedure TActor.SetStartFrame(Value: Integer);
begin
  if (Value>=0) and (Value<NumberFrame) and (Value<>FStartFrame) then
    FStartFrame := Value;
  if FEndFrame<FStartFrame then
    FEndFrame := FStartFrame;
  if FCurrentFrame<FStartFrame then
    CurrentFrame := FStartFrame;
end;

//------------------------------------------------------------------------------
procedure TActor.SetEndFrame(Value: Integer);
begin
  if (Value>=0) and (Value<NumberFrame) and (Value<>FEndFrame) then
    FEndFrame := Value;
  if FCurrentFrame>FEndFrame then
    CurrentFrame := FEndFrame;
end;

//------------------------------------------------------------------------------
procedure TActor.NextFrame(N: Integer);
var
  Value: Integer;
begin
  Value := FCurrentFrame + N;
  if Value>FEndFrame then
  begin
    Value := FStartFrame + (Value - FEndFrame);
    if Value>FEndFrame then
      Value := FEndFrame;
  end;
  CurrentFrame := Value;
end;

//------------------------------------------------------------------------------
procedure TActor.PrevFrame(N: Integer);
var
  Value: Integer;
begin
  Value := FCurrentFrame - N;
  if Value<FStartFrame then
  begin
    Value := FEndFrame - (FStartFrame - Value);
    if Value<FStartFrame then
      Value := FStartFrame;
  end;
  CurrentFrame := Value;
end;


//------------------------------------------------------------------------------
procedure TActor.BuildList;

var
  Mesh : PMeshObject;
  I: Integer;
  FaceGroup: PFaceGroup;

begin
  // Build List can not inherited from TFreeForm, so apply it direct
  Options.Apply;
  glListBase(0);

  // go through all loaded meshs
  if FObjects.Count=0 then
  begin
//    glPopAttrib;
    exit;
  end;


  glFrontFace(FWinding);

  Mesh := FObjects[FCurrentFrame];
  glShadeModel(GL_SMOOTH);

  glBegin(GL_TRIANGLES);
  FaceGroup := Mesh.FaceGroups[0];
  for I:=0 to (FaceGroup.IndexCount div 3)-1 do
  begin
    glTexCoord2f(Mesh.TexCoords[3*I+0].s, Mesh.TexCoords[3*I+0].t);
    glVertex3f(Mesh.Vertices[FaceGroup.Indices[3*I+0]][0],
      Mesh.Vertices[FaceGroup.Indices[3*I+0]][1],
      Mesh.Vertices[FaceGroup.Indices[3*I+0]][2]);
    glTexCoord2f(Mesh.TexCoords[3*I+1].s, Mesh.TexCoords[3*I+1].t);
    glVertex3f(Mesh.Vertices[FaceGroup.Indices[3*I+1]][0],
      Mesh.Vertices[FaceGroup.Indices[3*I+1]][1],
      Mesh.Vertices[FaceGroup.Indices[3*I+1]][2]);
    glTexCoord2f(Mesh.TexCoords[3*I+2].s, Mesh.TexCoords[3*I+2].t);
    glVertex3f(Mesh.Vertices[FaceGroup.Indices[3*I+2]][0],
      Mesh.Vertices[FaceGroup.Indices[3*I+2]][1],
      Mesh.Vertices[FaceGroup.Indices[3*I+2]][2]);
  end;
  {
  for I:=0 to (Mesh.FaceGroups[0].IndexCount div 3)-1 do
  begin
    glTexCoord2f(Mesh.TexCoords[3*I+0].s, Mesh.TexCoords[3*I+0].t);
    glVertex3f(Mesh.Vertices[Mesh.FaceGroups[0].Indices[3*I+0]][0],
      Mesh.Vertices[Mesh.FaceGroups[0].Indices[3*I+0]][1],
      Mesh.Vertices[Mesh.FaceGroups[0].Indices[3*I+0]][2]);
    glTexCoord2f(Mesh.TexCoords[3*I+1].s, Mesh.TexCoords[3*I+1].t);
    glVertex3f(Mesh.Vertices[Mesh.FaceGroups[0].Indices[3*I+1]][0],
      Mesh.Vertices[Mesh.FaceGroups[0].Indices[3*I+1]][1],
      Mesh.Vertices[Mesh.FaceGroups[0].Indices[3*I+1]][2]);
    glTexCoord2f(Mesh.TexCoords[3*I+2].s, Mesh.TexCoords[3*I+2].t);
    glVertex3f(Mesh.Vertices[Mesh.FaceGroups[0].Indices[3*I+2]][0],
      Mesh.Vertices[Mesh.FaceGroups[0].Indices[3*I+2]][1],
      Mesh.Vertices[Mesh.FaceGroups[0].Indices[3*I+2]][2]);
  end;
  }
  glEnd;

//  glPopAttrib;
end;

//------------------------------------------------------------------------------

procedure TActor.LoadFromFile(const Filename: String);

var Ext           : String;
    NewVectorFile : TVectorFile;
    VectorFileClass : TVectorFileClass;

begin
  if FileName <> '' then
  begin
    FStartFrame := 0;
    FEndFrame := 0;
    FCurrentFrame := 0;
    FNumberFrame := 0;
    Clear;
    Ext := ExtractFileExt(Filename);
    Delete(Ext, 1, 1);
    VectorFileClass := GetVectorFileFormats.FindExt(Ext);
    if VectorFileClass = nil then
      raise EInvalidVectorFile.CreateFmt(SUnknownExtension, [Ext]);
    NewVectorFile := VectorFileClass.Create(Self);
    try
      if assigned(Scene) then Scene.BeginUpdate;
      NewVectorFile.LoadFromFile(Filename);
      if assigned(Scene) then Scene.EndUpdate;
    except
      NewVectorFile.Free;
      raise;
    end;
  end;
  FNumberFrame := FObjects.Count;
  FStartFrame := 0;
  FEndFrame := FObjects.Count-1;
  FCurrentFrame := 0;
  StructureChanged;
end;

//------------------------------------------------------------------------------

procedure TActor.Assign(Source:TPersistent);

begin
  if assigned(Source) and (Source is TActor) then
  begin
{    FVertices.Assign(TMesh(Source).Vertices);
    FMode := TMesh(Source).FMode;}
  end;
  inherited Assign(Source);
end;


//----------------- TGLMD2VectorFile -------------------------------------------

function TGLMD2VectorFile.ConvertMD2Structure(MD2File: TFileMD2; frame: Integer): PMeshObject;
var
  I: Integer;
  Mesh: PMeshObject;
  FaceGroup: PFaceGroup;
begin
  New(Mesh);
  FillChar(Mesh^, SizeOf(Mesh^), 0);
  Mesh^.Mode := mmTriangles;
  with Mesh^, MD2File do
  begin
    GetMem(Vertices, m_iVertices * sizeof(TAffineVector));
    GetMem(TexCoords, 3* m_iTriangles * sizeof(TTexPoint));

    FaceGroups := TFaceGroups.Create;
    with FaceGroups do
    begin
      New(FaceGroup);
      with FaceGroup^ do
      begin
        FaceProperties := TFaceProperties.Create(nil);
        FaceProperties.IsInherited := True;
        IndexCount := 3 * m_iTriangles;
        GetMem(Indices, IndexCount * SizeOf(Integer));
        // copy the face list
        for I := 0 to m_iTriangles - 1 do
        begin
          TexCoords[I*3].S := IndexList(m_index_list)[i].a_s;
          TexCoords[I*3].t := IndexList(m_index_list)[i].a_t;
          Indices[I*3] := IndexList(m_index_list)[i].a;

          TexCoords[I*3+1].S := IndexList(m_index_list)[i].b_s;
          TexCoords[I*3+1].t := IndexList(m_index_list)[i].b_t;
          Indices[I*3+1] := IndexList(m_index_list)[i].b;

          TexCoords[I*3+2].S := IndexList(m_index_list)[i].c_s;
          TexCoords[I*3+2].t := IndexList(m_index_list)[i].c_t;
          Indices[I*3+2] := IndexList(m_index_list)[i].c;
        end;
      end;
      Add(FaceGroup);
    end;

    for I:=0 to m_iVertices-1 do
    begin
      Vertices[I][0] := VertList(frameList(m_frame_list)[frame].vertex)[I].x;
      Vertices[I][1] := VertList(frameList(m_frame_list)[frame].vertex)[I].y;
      Vertices[I][2] := VertList(frameList(m_frame_list)[frame].vertex)[I].z;
    end;
  end;
  Result := Mesh;
end;

//------------------------------------------------------------------------------

procedure TGLMD2VectorFile.LoadFromFile(const FileName: String);
var
  I: Integer;
  MD2File: TFileMD2;
  Mesh: PMeshObject;
begin
  MD2File := TFileMD2.Create;
  MD2File.LoadFromFile(FileName);
  try
    for I:=0 to MD2File.m_iFrames-1 do
    begin
      Mesh := ConvertMD2Structure(MD2File, I);
      FOwner.AddMeshObject(Mesh);
    end;
  finally
    MD2File.Free;
  end;
end;


//------------------------------------------------------------------------------

initialization
  RegisterVectorFileFormat('md2', 'Quake II model files', TGLMD2VectorFile);
  RegisterVectorFileFormat('3ds', '3D Studio files', TGL3DSVectorFile);
  RegisterVectorFileFormat('prj', '3D Studio project files', TGL3DSVectorFile);
  FontManager := TFontManager.Create;
  ObjectManager := TObjectManager.Create(nil);
  with ObjectManager do
  begin
    RegisterSceneObject(TCamera, 'Camera', LoadBitmap(HInstance, 'GLS_CAMERA_16'));
    RegisterSceneObject(TLightSource, 'Lightsource', LoadBitmap(HInstance, 'GLS_LAMP_16'));
    RegisterSceneObject(TPlane, 'Plane', LoadBitmap(HInstance, 'GLS_PLANE_16'));
    RegisterSceneObject(TCube, 'Cube', LoadBitmap(HInstance, 'GLS_CUBE_16'));
    RegisterSceneObject(TDisk, 'Disk', LoadBitmap(HInstance, 'GLS_DISK_16'));
    RegisterSceneObject(TSphere, 'Sphere', LoadBitmap(HInstance, 'GLS_SPHERE_16'));
    RegisterSceneObject(TCylinder, 'Cylinder', LoadBitmap(HInstance, 'GLS_CYLINDER_16'));
    RegisterSceneObject(TCone, 'Cone', LoadBitmap(HInstance, 'GLS_CONE_16'));
    RegisterSceneObject(TTorus, 'Torus', LoadBitmap(HInstance, 'GLS_TORUS_16'));
    RegisterSceneObject(TSpaceText, 'SpaceText', LoadBitmap(HInstance, 'GLS_TEXT_16'));
    RegisterSceneObject(TMesh, 'Mesh', LoadBitmap(HInstance, 'GLS_FREEFORM_16'));
    RegisterSceneObject(TFreeForm, 'FreeForm', LoadBitmap(HInstance, 'GLS_FREEFORM_16'));
    RegisterSceneObject(TActor, 'Actor', LoadBitmap(HInstance, 'GLS_FREEFORM_16'));
    RegisterSceneObject(TTeapot, 'Teapot', LoadBitmap(HInstance, 'GLS_TEAPOT_16'));
    RegisterSceneObject(TDodecahedron, 'Dodecahedron', LoadBitmap(HInstance, 'GLS_DODECAHEDRON_16'));
    //RegisterSceneObject(TRotationSolid, 'RotationSolid', 0);
  end;
  // need to explicitly register these classes
  RegisterClasses([TLightsource, TCamera, TSphere, TCube, TCylinder, TCone, TTorus,
                   TSpaceText, TMesh, TFreeForm, TTeapot, TDodecahedron, TDisk, TPlane,
                   TActor]);
finalization
  ObjectManager.Free;
  FontManager.Free;
  VectorFileFormats.Free;
  UnregisterVectorFileClass(TGL3DSVectorFile);
end.
