//==========================================================================================
// FMOD Main header file. Copyright (c), FireLight Multimedia 2000.
//==========================================================================================

unit FMOD;

interface
uses windows;

//===============================================================================================
//= FMOD DEFINED TYPES
//===============================================================================================

type
	pFSOUND_VECTOR = ^FSOUND_VECTOR;
	FSOUND_VECTOR = record
		x : single;
		y : single;
		z : single;
	end;

	pFSOUND_SAMPLE = ^FSOUND_SAMPLE;
	FSOUND_SAMPLE = record
		Thing : byte;
	end;
	pFSOUND_STREAM = ^FSOUND_STREAM;
	FSOUND_STREAM = record
		Thing : byte;
	end;
	pFSOUND_DSPUNIT = ^FSOUND_DSPUNIT;
	FSOUND_DSPUNIT = record
		Thing : byte;
	end;
	pFSOUND_MATERIAL = ^FSOUND_MATERIAL;
	FSOUND_MATERIAL = record
		Thing : byte;
	end;
	pFSOUND_GEOMLIST = ^FSOUND_GEOMLIST;
	FSOUND_GEOMLIST = record
		Thing : byte;
	end;
	pFMUSIC_MODULE = ^FMUSIC_MODULE;
	FMUSIC_MODULE = record
		Thing : byte;
	end;

        pPointer=^Pointer;
        pDWORD=^DWORD;
        pUlong=^Ulong;
        pSingle=^Single;

	// callback types
    FSOUND_STREAMCALLBACK	= Procedure (stream:pFSOUND_STREAM; buff : pointer; length,param : DWORD); cdecl;
	FSOUND_DSPCALLBACK		= Function (originalbuffer : pointer; newbuffer : pointer; length,param : DWORD):pointer; cdecl;
	FMUSIC_CALLBACK			= Procedure (mdl:pFMUSIC_MODULE; param : byte); cdecl;

    FSOUND_OPENCALLBACK		= Function  (name : pChar) : ULong; cdecl;
    FSOUND_CLOSECALLBACK	= Procedure (handle : ULong); cdecl;
    FSOUND_READCALLBACK		= Function  (buffer : pointer; size : DWORD; handle : ULong) : DWORD; cdecl;
    FSOUND_SEEKCALLBACK		= Procedure (handle : ULong; pos : DWORD; mode : ByteBool); cdecl;
    FSOUND_TELLCALLBACK		= Function  (handle : ULong) : DWORD; cdecl;


	FMOD_ERRORS =
	(
		FMOD_ERR_NONE,			   // No errors
		FMOD_ERR_BUSY,             // Cannot call this command after FSOUND_Init.  Call FSOUND_Close first.
		FMOD_ERR_UNINITIALIZED,	   // This command failed because FSOUND_Init was not called
		FMOD_ERR_INIT,			   // Error initializing output device.
		FMOD_ERR_ALLOCATED,		   // Error initializing output device, but more specifically, the output device is already in use and cannot be reused.
		FMOD_ERR_PLAY,			   // Playing the sound failed.
		FMOD_ERR_OUTPUT_FORMAT,	   // Soundcard does not support the features needed for this soundsystem (16bit stereo output)
		FMOD_ERR_COOPERATIVELEVEL, // Error setting cooperative level for hardware.
		FMOD_ERR_CREATEBUFFER,	   // Error creating hardware sound buffer.
		FMOD_ERR_FILE_NOTFOUND,	   // File not found
		FMOD_ERR_FILE_FORMAT,	   // Unknown file format
		FMOD_ERR_FILE_BAD,		   // Error loading file
		FMOD_ERR_MEMORY,           // Not enough memory 
		FMOD_ERR_VERSION,		   // The version number of this file format is not supported
		FMOD_ERR_INVALID_MIXER,    // Incorrect mixer selected
		FMOD_ERR_INVALID_PARAM,	   // An invalid parameter was passed to this function
		FMOD_ERR_NO_EAX,		   // Tried to use an EAX command on a non EAX enabled channel or output.
		FMOD_ERR_NO_EAX2,		   // Tried to use an advanced EAX2 command on a non EAX2 enabled channel or output.
		FMOD_ERR_CHANNEL_ALLOC,	   // Failed to allocate a new channel
		FMOD_ERR_RECORD			   // Recording is not supported on this machine
	);
	
	
	//	These output types are used with FSOUND_SetOutput, to choose which output driver to use.
	//	
	//	FSOUND_OUTPUT_A3D will cause FSOUND_Init to FAIL if you have not got a vortex 
	//	based A3D card.  The suggestion for this is to immediately try and reinitialize FMOD with
	//	FSOUND_OUTPUT_DSOUND, and if this fails, try initializing FMOD with	FSOUND_OUTPUT_WAVEOUT.
	//	
	//	FSOUND_OUTPUT_DSOUND will not support hardware 3d acceleration if the sound card driver 
	//	does not support DirectX 6 Voice Manager Extensions.
	FSOUND_OUTPUTTYPES =
	(
		FSOUND_OUTPUT_NOSOUND,		// NoSound driver, all calls to this succeed but do nothing.
		FSOUND_OUTPUT_WINMM,		// Windows Multimedia driver.
		FSOUND_OUTPUT_DSOUND,		// DirectSound driver.  You need this to get EAX or EAX2 support.
		FSOUND_OUTPUT_A3D			// A3D driver.  You need this to get geometry and EAX reverb support.
	);
	
	
	//	These mixer types are used with FSOUND_SetMixer, to choose which mixer to use, or to act 
	//	upon for other reasons using FSOUND_GetMixer.
	FSOUND_MIXERTYPES =
	(
		FSOUND_MIXER_AUTODETECT,	// This enables autodetection of the fastest mixer based on your cpu.
		FSOUND_MIXER_BLENDMODE,		// This enables the standard non mmx, blendmode mixer.
		FSOUND_MIXER_MMXP5,			// This enables the mmx, pentium optimized blendmode mixer.
		FSOUND_MIXER_MMXP6,			// This enables the mmx, ppro/p2/p3 optimized mixer.
	
		FSOUND_MIXER_QUALITY_AUTODETECT,// This enables autodetection of the fastest quality mixer based on your cpu.
		FSOUND_MIXER_QUALITY_FPU,		// This enables the interpolating FPU mixer. 
		FSOUND_MIXER_QUALITY_MMXP5,		// This enables the interpolating p5 MMX mixer. 
		FSOUND_MIXER_QUALITY_MMXP6,		// This enables the interpolating ppro/p2/p3 MMX mixer.
		FSOUND_MIXER_QUALITY_FPU_VOLUMERAMP// Enables the interpolating 'de-clicking' FPU mixer.
	);

	//	These definitions describe the type of song being played.
	FMUSIC_TYPES =
	(
		FMUSIC_TYPE_NONE,		
		FMUSIC_TYPE_MOD,		// Protracker / Fasttracker
		FMUSIC_TYPE_S3M,		// ScreamTracker 3
		FMUSIC_TYPE_XM,			// FastTracker 2
		FMUSIC_TYPE_IT			// Impulse Tracker.
	);

	// These are environment types defined for use with the FSOUND_Reverb API.
	FSOUND_REVERB_ENVIRONMENTS =
	(
		FSOUND_ENVIRONMENT_GENERIC,
		FSOUND_ENVIRONMENT_PADDEDCELL,
		FSOUND_ENVIRONMENT_ROOM,
		FSOUND_ENVIRONMENT_BATHROOM,
		FSOUND_ENVIRONMENT_LIVINGROOM,
		FSOUND_ENVIRONMENT_STONEROOM,
		FSOUND_ENVIRONMENT_AUDITORIUM,
		FSOUND_ENVIRONMENT_CONCERTHALL,
		FSOUND_ENVIRONMENT_CAVE,
		FSOUND_ENVIRONMENT_ARENA,
		FSOUND_ENVIRONMENT_HANGAR,
		FSOUND_ENVIRONMENT_CARPETEDHALLWAY,
		FSOUND_ENVIRONMENT_HALLWAY,
		FSOUND_ENVIRONMENT_STONECORRIDOR,
		FSOUND_ENVIRONMENT_ALLEY,
		FSOUND_ENVIRONMENT_FOREST,
		FSOUND_ENVIRONMENT_CITY,
		FSOUND_ENVIRONMENT_MOUNTAINS,
		FSOUND_ENVIRONMENT_QUARRY,
		FSOUND_ENVIRONMENT_PLAIN,
		FSOUND_ENVIRONMENT_PARKINGLOT,
		FSOUND_ENVIRONMENT_SEWERPIPE,
		FSOUND_ENVIRONMENT_UNDERWATER,
		FSOUND_ENVIRONMENT_DRUGGED,
		FSOUND_ENVIRONMENT_DIZZY,
		FSOUND_ENVIRONMENT_PSYCHOTIC,

		FSOUND_ENVIRONMENT_COUNT
	);

const
	FMOD_VERSION	=	3.20;

	// DSP unit priorities for system units.  
	// Use FSOUND_DSP_DEFAULTPRIORITY_USER to put DSP effects in.
	FSOUND_DSP_DEFAULTPRIORITY_CLEARUNIT		= 0;	// DSP CLEAR unit - done first
	FSOUND_DSP_DEFAULTPRIORITY_SFXUNIT			= 100;	// DSP SFX unit - done second
	FSOUND_DSP_DEFAULTPRIORITY_MUSICUNIT		= 200;	// DSP MUSIC unit - done third
	FSOUND_DSP_DEFAULTPRIORITY_USER				= 300;	// User priority, use this as reference
	FSOUND_DSP_DEFAULTPRIORITY_CLIPANDCOPYUNIT	= 1000;	// DSP CLIP AND COPY unit - last

	//	Driver description bitfields.  Use FSOUND_Driver_GetCaps to determine if a driver enumerated
	//	has the settings you are after.  The enumerated driver depends on the output mode, see
	//	FSOUND_OUTPUTTYPES
	FSOUND_CAPS_HARDWARE				= $1;	// This driver supports hardware accelerated 3d sound.
	FSOUND_CAPS_EAX						= $2;	// This driver supports EAX reverb
	FSOUND_CAPS_GEOMETRY_OCCLUSIONS		= $4;	// This driver supports (A3D) geometry occlusions
	FSOUND_CAPS_GEOMETRY_REFLECTIONS	= $8;	// This driver supports (A3D) geometry reflections
	FSOUND_CAPS_EAX2					= $10;	// This driver supports EAX2/A3D3 reverb


	//	Sample description bitfields, OR them together for loading and describing samples.
	FSOUND_LOOP_OFF		= $01;		// For non looping samples.
	FSOUND_LOOP_NORMAL	= $02;		// For forward looping samples.
	FSOUND_LOOP_BIDI	= $04;		// For bidirectional looping samples.  (no effect if in hardware).
	FSOUND_8BITS		= $08;		// For 8 bit samples.
	FSOUND_16BITS		= $10;		// For 16 bit samples.
	FSOUND_MONO			= $20;		// For mono samples.
	FSOUND_STEREO		= $40;		// For stereo samples.
	FSOUND_UNSIGNED		= $80;		// For source data containing unsigned samples.
	FSOUND_SIGNED		= $100;		// For source data containing signed data.
	FSOUND_DELTA		= $200;		// For source data stored as delta values.
	FSOUND_IT214		= $400;		// For source data stored using IT214 compression.
	FSOUND_IT215		= $800;		// For source data stored using IT215 compression.
	FSOUND_HW3D			= $1000;	// Attempts to make samples use 3d hardware acceleration. (if the card supports it)
	FSOUND_2D			= $2000;	// Ignores any 3d processing.  overrides FSOUND_HW3D.  Located in software.
	FSOUND_STREAMABLE	= $4000;	// For realtime streamable samples.  If you dont supply this sound may come out corrupted.

	FSOUND_NORMAL		= $29;		// (FSOUND_LOOP_OFF + FSOUND_8BITS + FSOUND_MONO)		

	// Playback method for a CD Audio track, using FSOUND_CD_Play
	FSOUND_CD_PLAYCONTINUOUS	= 0;	// Starts from the current track and plays to end of CD.
	FSOUND_CD_PLAYONCE			= 1;	// Plays the specified track then stops.
	FSOUND_CD_PLAYLOOPED		= 2;	// Plays the specified track looped, forever until stopped manually.
	FSOUND_CD_PLAYRANDOM		= 3;	// Plays tracks in random order

	// Misccellaneous values for FMOD functions.
    FSOUND_FREE			= $FFFFFFFF;	// value to play on any free channel, or to allocate a sample in a free sample slot.
	FSOUND_UNMANAGED	= $FFFFFFFE;	// value to allocate a sample that is NOT managed by FSOUND or placed in a sample slot.
	FSOUND_STEREOPAN	= $FFFFFFFF;	// value for FSOUND_SetPan so that stereo sounds are not played at half volume.  See FSOUND_SetPan for more on this.
	FSOUND_REVERBMIX_USEDISTANCE =	-1.0;	// used with FSOUND_Reverb_SetMix to attenuate reverb by distance automatically
	FSOUND_REVERB_IGNOREPARAM = $FF676981;	// used with FSOUND_Reverb_SetEnvironment or FSOUND_Reverb_SetEnvironmentAdvanced to ignore parameters.

	// A set of predefined environment PARAMETERS, created by Creative Labs
	// These can be placed directly into the FSOUND_Reverb_SetEnvironment call (in the C version)

	{FSOUND_PRESET_GENERIC         = FSOUND_ENVIRONMENT_GENERIC,0.5,   1.493,  0.5;
	FSOUND_PRESET_PADDEDCELL      = FSOUND_ENVIRONMENT_PADDEDCELL,0.25,  0.1,    0.0;
	FSOUND_PRESET_ROOM            = FSOUND_ENVIRONMENT_ROOM,0.417, 0.4,    0.666;
	FSOUND_PRESET_BATHROOM        = FSOUND_ENVIRONMENT_BATHROOM,0.653, 1.499,  0.166;
	FSOUND_PRESET_LIVINGROOM      = FSOUND_ENVIRONMENT_LIVINGROOM,0.208, 0.478,  0.0;
	FSOUND_PRESET_STONEROOM       = FSOUND_ENVIRONMENT_STONEROOM,0.5,   2.309,  0.888;
	FSOUND_PRESET_AUDITORIUM      = FSOUND_ENVIRONMENT_AUDITORIUM,0.403, 4.279,  0.5;
	FSOUND_PRESET_CONCERTHALL     = FSOUND_ENVIRONMENT_CONCERTHALL,0.5,   3.961,  0.5;
	FSOUND_PRESET_CAVE            = FSOUND_ENVIRONMENT_CAVE,0.5,   2.886,  1.304;
	FSOUND_PRESET_ARENA           = FSOUND_ENVIRONMENT_ARENA,0.361, 7.284,  0.332;
	FSOUND_PRESET_HANGAR          = FSOUND_ENVIRONMENT_HANGAR,0.5,   10.0,   0.3;
	FSOUND_PRESET_CARPETEDHALLWAY = FSOUND_ENVIRONMENT_CARPETEDHALLWAY,0.153, 0.259,  2.0;
	FSOUND_PRESET_HALLWAY         = FSOUND_ENVIRONMENT_HALLWAY,0.361, 1.493,  0.0;
	FSOUND_PRESET_STONECORRIDOR   = FSOUND_ENVIRONMENT_STONECORRIDOR,0.444, 2.697,  0.638;
	FSOUND_PRESET_ALLEY           = FSOUND_ENVIRONMENT_ALLEY,0.25,  1.752,  0.776;
	FSOUND_PRESET_FOREST          = FSOUND_ENVIRONMENT_FOREST,0.111, 3.145,  0.472;
	FSOUND_PRESET_CITY            = FSOUND_ENVIRONMENT_CITY,0.111, 2.767,  0.224;
	FSOUND_PRESET_MOUNTAINS       = FSOUND_ENVIRONMENT_MOUNTAINS,0.194, 7.841,  0.472;
	FSOUND_PRESET_QUARRY          = FSOUND_ENVIRONMENT_QUARRY,1.0,   1.499,  0.5;
	FSOUND_PRESET_PLAIN           = FSOUND_ENVIRONMENT_PLAIN,0.097, 2.767,  0.224;
	FSOUND_PRESET_PARKINGLOT      = FSOUND_ENVIRONMENT_PARKINGLOT,0.208, 1.652,  1.5;
	FSOUND_PRESET_SEWERPIPE       = FSOUND_ENVIRONMENT_SEWERPIPE,0.652, 2.886,  0.25;
	FSOUND_PRESET_UNDERWATER      = FSOUND_ENVIRONMENT_UNDERWATER,1.0,   1.499,  0.0;
	FSOUND_PRESET_DRUGGED         = FSOUND_ENVIRONMENT_DRUGGED,0.875, 8.392,  1.388;
	FSOUND_PRESET_DIZZY           = FSOUND_ENVIRONMENT_DIZZY,0.139, 17.234, 0.666;
	FSOUND_PRESET_PSYCHOTIC       = FSOUND_ENVIRONMENT_PSYCHOTIC,0.486, 7.563,  0.806;
         }
	// Geometry flags, used as the mode flag in FSOUND_Geometry_AddPolygon
	FSOUND_GEOMETRY_NORMAL				= $0;	// Default geometry type.  Occluding polygon
	FSOUND_GEOMETRY_REFLECTIVE			= $01;	// This polygon is reflective
	FSOUND_GEOMETRY_OPENING				= $02;	// Overlays a transparency over the previous polygon.  The 'openingfactor' value supplied is copied internally.
	FSOUND_GEOMETRY_OPENING_REFERENCE	= $04;	// Overlays a transparency over the previous polygon.  The 'openingfactor' supplied is pointed to (for access when building a list)


//===============================================================================================
//= FUNCTION PROTOTYPES
//===============================================================================================


// ==================================
// Initialization / Global functions.
// ==================================

// Pre FSOUND_Init functions. These can't be called after FSOUND_Init is called (they will fail)
Function	FSOUND_SetOutput(outputtype : FSOUND_OUTPUTTYPES) : ByteBool; stdcall;
Function	FSOUND_SetDriver(driver : DWORD) : ByteBool; stdcall;
Function	FSOUND_SetMixer(mixer : FSOUND_MIXERTYPES) : ByteBool; stdcall;
Function	FSOUND_SetBufferSize(len_ms : DWORD) : ByteBool; stdcall;
Function	FSOUND_SetHWND(hwnd : pointer) : ByteBool; stdcall;
Function	FSOUND_SetMinHardwareChannels(min : DWORD) : ByteBool; stdcall;
Function	FSOUND_SetMaxHardwareChannels(max : DWORD) : ByteBool; stdcall;

// Main initialization / closedown functions
Function	FSOUND_Init(mixrate, maxchannels, vcmmode : DWORD) : ByteBool; stdcall;
Procedure	FSOUND_Close; stdcall;

// Runtime 
Procedure 	FSOUND_SetSFXMasterVolume(volume : DWORD); stdcall;
Procedure 	FSOUND_SetPanSeperation(pansep : single); stdcall;

// Error functions
Function	FSOUND_GetError : FMOD_ERRORS; stdcall;


// ===================================
// Sample management / load functions.
// ===================================

// File functions
Function	FSOUND_Sample_LoadWav(index : DWORD; filename : pChar; mode : ULong) : pFSOUND_SAMPLE; stdcall;
Function	FSOUND_Sample_LoadMpeg(index : DWORD; filename : pChar; mode : ULong) : pFSOUND_SAMPLE; stdcall;
Function	FSOUND_Sample_LoadRaw(index : DWORD; filename : pChar; mode : ULong) : pFSOUND_SAMPLE; stdcall;
Function	FSOUND_Sample_LoadWavMemory(index : DWORD; data : pointer; mode : ULong; length : DWORD) : pFSOUND_SAMPLE; stdcall;
Function	FSOUND_Sample_LoadMpegMemory(index : DWORD; data : pointer; mode : ULong; length : DWORD) : pFSOUND_SAMPLE; stdcall;

// Sample management functions
Function	FSOUND_Sample_Alloc(index,length : DWORD; mode : ULong; deffreq, defvol, defpan, defpri : DWORD) : pFSOUND_SAMPLE; stdcall;
Procedure 	FSOUND_Sample_Free(sptr : pFSOUND_SAMPLE); stdcall;
Function	FSOUND_Sample_Upload(sptr : pFSOUND_SAMPLE; srcdata : pointer; mode : ULong) : ByteBool; stdcall;
Function	FSOUND_Sample_Lock(sptr : pFSOUND_SAMPLE; offset, length : DWORD; ptr1,ptr2 : pPointer; len1, len2 : pULong) : ByteBool; stdcall;
Function	FSOUND_Sample_Unlock(sptr : pFSOUND_SAMPLE; ptr1,ptr2 : pointer; len1, len2 : DWORD) : ByteBool; stdcall;

// Sample control functions
Function	FSOUND_Sample_SetLoopMode(sptr : pFSOUND_SAMPLE; loopmode : ULong) : ByteBool; stdcall;
Function	FSOUND_Sample_SetLoopPoints(sptr : pFSOUND_SAMPLE; loopstart, loopend : DWORD) : ByteBool; stdcall;
Function	FSOUND_Sample_SetDefaults(sptr : pFSOUND_SAMPLE; deffreq, defvol, defpan, defpri : DWORD) : ByteBool; stdcall;
Function	FSOUND_Sample_SetMinMaxDistance(sptr : pFSOUND_SAMPLE; min, max : single) : ByteBool; stdcall;

  
// ============================
// Channel control functions.
// ============================

// Playing and stopping sounds.
Function	FSOUND_PlaySound(channel : DWORD; sptr : pFSOUND_SAMPLE) : DWORD; stdcall;
Function	FSOUND_PlaySoundAttrib(channel : DWORD; sptr : pFSOUND_SAMPLE; freq, vol, pan : DWORD) : DWORD; stdcall;
Function	FSOUND_3D_PlaySound(channel : DWORD; sptr : pFSOUND_SAMPLE; pos:pFSOUND_VECTOR; vel:pFSOUND_VECTOR) : DWORD; stdcall;
Function	FSOUND_StopSound(channel : DWORD) : ByteBool; stdcall;
Procedure 	FSOUND_StopAllChannels; stdcall;
 
// Functions to control playback of a channel.
Function	FSOUND_SetFrequency(channel : DWORD; freq : DWORD) : ByteBool; stdcall;
Function	FSOUND_SetVolume(channel : DWORD; vol : DWORD) : ByteBool; stdcall;
Function	FSOUND_SetVolumeAbsolute(channel : DWORD; vol : DWORD) : ByteBool; stdcall;
Function	FSOUND_SetPan(channel : DWORD; pan : DWORD) : ByteBool; stdcall;
Function	FSOUND_SetSurround(channel : DWORD; surround : ByteBool) : ByteBool; stdcall;
Function	FSOUND_SetMute(channel : DWORD; mute : ByteBool) : ByteBool; stdcall;
Function	FSOUND_SetPriority(channel : DWORD; priority : DWORD) : ByteBool; stdcall;
Function	FSOUND_SetReserved(channel : DWORD; reserved : ByteBool) : ByteBool; stdcall;
Function	FSOUND_SetPaused(channel : DWORD; paused : ByteBool) : ByteBool; stdcall;
Function	FSOUND_MixBuffers(destbuffer, srcbuffer : pointer; len, freq, vol, pan : DWORD; mode : ULong) : ByteBool; stdcall;


// ================================
// Information retrieval functions.
// ================================

// System information
Function	FSOUND_GetVersion : single; stdcall;
Function	FSOUND_GetOutput : FSOUND_OUTPUTTYPES; stdcall;
Function	FSOUND_GetDriver : DWORD; stdcall;
Function	FSOUND_GetMixer : FSOUND_MIXERTYPES; stdcall;
Function	FSOUND_GetNumDrivers : DWORD; stdcall;
Function	FSOUND_GetDriverName(id : DWORD) : pChar; stdcall;
Function	FSOUND_GetDriverCaps(id : DWORD; caps : pULong) : ByteBool; stdcall;

Function	FSOUND_GetOutputRate : DWORD; stdcall;
Function	FSOUND_GetMaxChannels : DWORD; stdcall;
Function	FSOUND_GetMaxSamples : DWORD; stdcall;
Function	FSOUND_GetSFXMasterVolume : DWORD; stdcall;
Function	FSOUND_GetNumHardwareChannels : DWORD; stdcall;
Function	FSOUND_GetChannelsPlaying : DWORD; stdcall;
Function	FSOUND_GetCPUUsage : single; stdcall;

// Channel information
Function	FSOUND_IsPlaying(channel : DWORD) : ByteBool; stdcall;
Function	FSOUND_GetFrequency(channel : DWORD) : DWORD; stdcall;
Function	FSOUND_GetVolume(channel : DWORD) : DWORD; stdcall;
Function	FSOUND_GetPan(channel : DWORD) : DWORD; stdcall;
Function	FSOUND_GetSurround(channel : DWORD) : ByteBool; stdcall;
Function	FSOUND_GetMute(channel : DWORD) : ByteBool; stdcall;
Function	FSOUND_GetPriority(channel : DWORD) : DWORD; stdcall;
Function	FSOUND_GetReserved(channel : DWORD) : ByteBool; stdcall;
Function	FSOUND_GetPaused(channel : DWORD) : ByteBool; stdcall;
Function	FSOUND_GetCurrentPosition(channel : DWORD) : ULong; stdcall;
Function	FSOUND_GetCurrentSample(channel : DWORD) : pFSOUND_SAMPLE; stdcall;
Function	FSOUND_GetCurrentVU(channel : DWORD) : single; stdcall;

// Sample information 
Function	FSOUND_Sample_Get(sampno : DWORD) : pFSOUND_SAMPLE; stdcall;
Function	FSOUND_Sample_GetLength(sptr : pFSOUND_SAMPLE) : ULong; stdcall;
Function	FSOUND_Sample_GetLoopPoints(sptr : pFSOUND_SAMPLE; loopstart, loopend : pDWORD) : ByteBool; stdcall;
Function	FSOUND_Sample_GetDefaults(sptr : pFSOUND_SAMPLE; deffreq, defvol, defpan, defpri : pDWORD) : ByteBool; stdcall;
Function	FSOUND_Sample_GetMode(sptr : pFSOUND_SAMPLE) : ULong; stdcall;

 
// ===================
// 3D sound functions.
// ===================
// see also FSOUND_3D_PlaySound (above)
// see also FSOUND_Sample_SetMinMaxDistance (above)
Procedure 	FSOUND_3D_Update; stdcall;
Function	FSOUND_3D_SetAttributes(channel : DWORD; pos, vel : pFSOUND_VECTOR) : ByteBool; stdcall; 
Function	FSOUND_3D_GetAttributes(channel : DWORD; pos, vel : pFSOUND_VECTOR) : ByteBool; stdcall;
Procedure 	FSOUND_3D_Listener_SetAttributes(pos, vel : pFSOUND_VECTOR; fx, fy, fz, tx, ty, tz : single); stdcall;
Procedure 	FSOUND_3D_Listener_GetAttributes(pos, vel : pFSOUND_VECTOR; fx, fy, fz, tx, ty, tz : psingle); stdcall;
Procedure 	FSOUND_3D_Listener_SetDopplerFactor(scale : single); stdcall;
Procedure 	FSOUND_3D_Listener_SetDistanceFactor(scale : single); stdcall;
Procedure 	FSOUND_3D_Listener_SetRolloffFactor(scale : single); stdcall;


// ===================
// Geometry functions.
// ===================

// scene/polygon functions
Function	FSOUND_Geometry_AddPolygon(p1,p2,p3,p4,normal : pFSOUND_VECTOR; mode : ULong; openingfactor : psingle) : ByteBool; stdcall;
Function	FSOUND_Geometry_AddList(geomlist : pFSOUND_GEOMLIST) : DWORD; stdcall;

// polygon list functions
Function	FSOUND_Geometry_List_Create(boundingvolume : ByteBool) : pFSOUND_GEOMLIST; stdcall;
Function	FSOUND_Geometry_List_Free(geomlist : pFSOUND_GEOMLIST) : ByteBool; stdcall;
Function	FSOUND_Geometry_List_Begin(geomlist : pFSOUND_GEOMLIST) : ByteBool; stdcall;
Function	FSOUND_Geometry_List_End(geomlist : pFSOUND_GEOMLIST) : ByteBool; stdcall;
Function	FSOUND_Geometry_List_Add(geomlist : pFSOUND_GEOMLIST) : ByteBool; stdcall;

// material functions
Function	FSOUND_Geometry_Material_Create : pFSOUND_MATERIAL; stdcall;
Function	FSOUND_Geometry_Material_Free(material : pFSOUND_MATERIAL) : ByteBool; stdcall;
Function	FSOUND_Geometry_Material_SetAttributes(material : pFSOUND_MATERIAL; reflectancegain, reflectancefreq, transmittancegain, transmittancefreq : single) : ByteBool; stdcall;
Function	FSOUND_Geometry_Material_GetAttributes(material : pFSOUND_MATERIAL; reflectancegain, reflectancefreq, transmittancegain, transmittancefreq : psingle) : ByteBool; stdcall;
Function	FSOUND_Geometry_Material_Set(material : pFSOUND_MATERIAL) : ByteBool; stdcall;

// occlusion (supported in software, eax, eax2, a3d3)
Function	FSOUND_Geometry_SetOcclusionFactor(factor : single) : ByteBool; stdcall;

// ==============================================
// Reverb functions. (eax, eax2, a3d 3.0 reverb)
// ==============================================

// eax1, eax2, a3d 3.0 (use FSOUND_REVERB_PRESETS if you like), (eax2 support through emulation/parameter conversion)
Function	FSOUND_Reverb_SetEnvironment(env : FSOUND_REVERB_ENVIRONMENTS; vol, decay, damp : single) : ByteBool; stdcall;
// eax2, a3d 3.0 only, does not work on eax1
Function	FSOUND_Reverb_SetEnvironmentAdvanced(
				env : FSOUND_REVERB_ENVIRONMENTS;
				Room : DWORD;					// [-10000, 0]     default: -10000 mB
				RoomHF : DWORD;					// [-10000, 0]     default: 0 mB
				RoomRolloffFactor : single;		// [0.0, 10.0]     default: 0.0
				DecayTime : single;				// [0.1, 20.0]     default: 1.0 s
				DecayHFRatio : single;			// [0.1, 2.0]      default: 0.5
				Reflections : DWORD;			// [-10000, 1000]  default: -10000 mB
				ReflectionsDelay : single;		// [0.0, 0.3]      default: 0.02 s
				Reverb : DWORD;					// [-10000, 2000]  default: -10000 mB
				ReverbDelay : single;			// [0.0, 0.1]      default: 0.04 s
				EnvironmentSize : single;		// [0.0, 100.0]    default: 100.0 %
				EnvironmentDiffusion : single;	// [0.0, 100.0]    default: 100.0 %
				AirAbsorptionHF : single) :		// [20.0, 20000.0] default: 5000.0 Hz
					FSOUND_REVERB_ENVIRONMENTS; stdcall;	

Function	FSOUND_Reverb_SetMix(channel : DWORD; mix : single) : ByteBool; stdcall;

// information functions
Function	FSOUND_Reverb_GetEnvironment(env : pDWORD; vol, decay, damp : psingle) : ByteBool; stdcall;
Function	FSOUND_Reverb_GetEnvironmentAdvanced(
					env : pDWORD;
					Room : pDWORD;
					RoomHF : pDWORD;
					RoomRolloffFactor : psingle;
					DecayTime : psingle;
					DecayHFRatio : psingle;
					Reflections : pDWORD;
					ReflectionsDelay : psingle;
					Reverb : pDWORD;
					ReverbDelay : psingle;
					EnvironmentSize : psingle;
					EnvironmentDiffusion : psingle;
					AirAbsorptionHF : psingle) : ByteBool; stdcall;
Function	FSOUND_Reverb_GetMix(channel : DWORD; mix : psingle) : ByteBool; stdcall;

// =========================
// File Streaming functions.
// =========================

Function	FSOUND_Stream_Create(callback:FSOUND_STREAMCALLBACK; length : DWORD; mode : ULong; samplerate, userdata : DWORD) : pFSOUND_STREAM; stdcall;
Function	FSOUND_Stream_Open(filename : pChar; mode : ULong; samplerate : DWORD) : pFSOUND_STREAM; stdcall;
Function	FSOUND_Stream_OpenWav(filename : pChar; mode : ULong) : pFSOUND_STREAM; stdcall;
Function	FSOUND_Stream_OpenMpeg(filename : pChar; mode : ULong) : pFSOUND_STREAM; stdcall;
Function	FSOUND_Stream_Play(channel : DWORD; stream : pFSOUND_STREAM) : DWORD; stdcall;
Function	FSOUND_Stream_Stop(stream : pFSOUND_STREAM) : ByteBool; stdcall;
Function	FSOUND_Stream_Close(stream : pFSOUND_STREAM) : ByteBool; stdcall;

Function	FSOUND_Stream_SetPaused(stream : pFSOUND_STREAM; paused : ByteBool) : ByteBool; stdcall;
Function	FSOUND_Stream_GetPaused(stream : pFSOUND_STREAM) : ByteBool; stdcall;
Function	FSOUND_Stream_SetPosition(stream : pFSOUND_STREAM; position : DWORD) : ByteBool; stdcall;
Function	FSOUND_Stream_GetPosition(stream : pFSOUND_STREAM): DWORD; stdcall;
Function	FSOUND_Stream_GetTime(stream : pFSOUND_STREAM) : DWORD; stdcall;
Function	FSOUND_Stream_GetLength(stream : pFSOUND_STREAM) : DWORD; stdcall;

// ===================
// CD audio functions.
// ===================

Function	FSOUND_CD_Play(track : DWORD) : ByteBool; stdcall;
Procedure 	FSOUND_CD_SetPlayMode(mode : ByteBool) stdcall;
Function	FSOUND_CD_Stop : ByteBool; stdcall;
Function	FSOUND_CD_SetPaused(paused : ByteBool) : ByteBool; stdcall;
Function	FSOUND_CD_GetPaused : ByteBool; stdcall;
Function	FSOUND_CD_GetTrack : DWORD; stdcall;
Function	FSOUND_CD_GetNumTracks : DWORD; stdcall;
Function	FSOUND_CD_Eject : ByteBool; stdcall;


// ==============
// DSP functions.
// ==============

// DSP Unit control and information functions.
Function	FSOUND_DSP_Create(callback:FSOUND_DSPCALLBACK; priority, param : DWORD) : PFSOUND_DSPUNIT; stdcall;
Procedure 	FSOUND_DSP_Free(unt : pFSOUND_DSPUNIT); stdcall;

Procedure 	FSOUND_DSP_SetPriority(unt : pFSOUND_DSPUNIT; priority : DWORD); stdcall;
Function	FSOUND_DSP_GetPriority(unt : pFSOUND_DSPUNIT) : DWORD; stdcall;

Procedure 	FSOUND_DSP_SetActive(unt : pFSOUND_DSPUNIT; active : ByteBool); stdcall;
Function	FSOUND_DSP_GetActive(unt : pFSOUND_DSPUNIT) : ByteBool; stdcall;

// Functions to get hold of FSOUND 'system DSP unit' handles.
Function	FSOUND_DSP_GetClearUnit : pFSOUND_DSPUNIT; stdcall;
Function	FSOUND_DSP_GetSFXUnit : pFSOUND_DSPUNIT; stdcall;
Function	FSOUND_DSP_GetMusicUnit : pFSOUND_DSPUNIT; stdcall;
Function	FSOUND_DSP_GetClipAndCopyUnit : pFSOUND_DSPUNIT; stdcall;

// misc DSP functions
Procedure 	FSOUND_DSP_ClearMixBuffer; stdcall;
Function	FSOUND_DSP_GetBufferLength : DWORD; stdcall;


// =========================
// Recording functions
// =========================

// recording initialization
Function	FSOUND_Record_SetDriver(outputtype : DWORD) : ByteBool; stdcall;

// information functions.
Function	FSOUND_Record_GetCaps(modes : pULong) : ByteBool; stdcall;
Function	FSOUND_Record_GetNumDrivers : DWORD; stdcall;
Function	FSOUND_Record_GetDriverName(id : DWORD) : pChar; stdcall;
Function	FSOUND_Record_GetDriver : DWORD; stdcall;

// recording functionality.  Only one recording session will work at a time
Function	FSOUND_Record_StartSample(sptr : pFSOUND_SAMPLE; loop : ByteBool) : ByteBool; stdcall;
Function	FSOUND_Record_Stop : ByteBool; stdcall;
Function	FSOUND_Record_GetPosition : DWORD; stdcall;

// =========================
// File system override
// =========================

Procedure FSOUND_File_SetCallbacks(OpenCallback  : FSOUND_OPENCALLBACK;
                                   CloseCallback : FSOUND_CLOSECALLBACK;
                                   ReadCallback  : FSOUND_READCALLBACK;
                                   SeekCallback  : FSOUND_SEEKCALLBACK;
                                   TellCallback  : FSOUND_TELLCALLBACK); stdcall;

// =============================================================================================
// FMUSIC API
// =============================================================================================

// Song management / playback functions.
// =====================================
Function	FMUSIC_LoadSong(name : pChar) : pFMUSIC_MODULE; stdcall;
Function	FMUSIC_LoadSongMemory(data : pointer; length : DWORD) : pFMUSIC_MODULE; stdcall;
Function	FMUSIC_FreeSong(mdl : pFMUSIC_MODULE) : ByteBool; stdcall;
Function	FMUSIC_PlaySong(mdl : pFMUSIC_MODULE) : ByteBool; stdcall;
Function	FMUSIC_StopSong(mdl : pFMUSIC_MODULE) : ByteBool; stdcall;
Procedure 	FMUSIC_StopAllSongs; stdcall;
Function	FMUSIC_SetZxxCallback(mdl : pFMUSIC_MODULE; callback : FMUSIC_CALLBACK) : ByteBool; stdcall;
Function	FMUSIC_SetRowCallback(mdl : pFMUSIC_MODULE; callback : FMUSIC_CALLBACK; rowstep:DWORD) : ByteBool; stdcall;
Function	FMUSIC_SetOrderCallback(mdl : pFMUSIC_MODULE; callback : FMUSIC_CALLBACK; orderstep:DWORD) : ByteBool; stdcall;
Function	FMUSIC_OptimizeChannels(mdl : pFMUSIC_MODULE; maxchannels, minvolume : DWORD) : ByteBool; stdcall;


// Runtime song functions.
// =======================
Function	FMUSIC_SetOrder(mdl : pFMUSIC_MODULE; order : DWORD) : ByteBool; stdcall;
Function	FMUSIC_SetPaused(mdl : pFMUSIC_MODULE; pause : ByteBool) : ByteBool; stdcall;
Function	FMUSIC_SetMasterVolume(mdl : pFMUSIC_MODULE; volume : DWORD) : ByteBool; stdcall;
Function	FMUSIC_SetPanSeperation(mdl : pFMUSIC_MODULE; pansep : single) : ByteBool; stdcall;

// Static song information functions.
// ==================================
Function	FMUSIC_GetName(mdl : pFMUSIC_MODULE) : pChar; stdcall;
Function	FMUSIC_GetType(mdl : pFMUSIC_MODULE) : FMUSIC_TYPES; stdcall;
Function	FMUSIC_UsesLinearFrequencies(mdl : pFMUSIC_MODULE) : ByteBool; stdcall;
Function	FMUSIC_GetNumOrders(mdl : pFMUSIC_MODULE) : DWORD; stdcall;
Function	FMUSIC_GetNumPatterns(mdl : pFMUSIC_MODULE) : DWORD; stdcall;
Function	FMUSIC_GetNumInstruments(mdl : pFMUSIC_MODULE) : DWORD; stdcall;
Function	FMUSIC_GetNumSamples(mdl : pFMUSIC_MODULE) : DWORD; stdcall;
Function	FMUSIC_GetNumChannels(mdl : pFMUSIC_MODULE) : DWORD; stdcall;
Function	FMUSIC_GetSample(mdl : pFMUSIC_MODULE; sampno : DWORD) : pFSOUND_SAMPLE; stdcall;
Function	FMUSIC_GetPatternLength(mdl : pFMUSIC_MODULE; orderno : DWORD) : DWORD; stdcall; 
 
// Runtime song information.
// =========================
Function	FMUSIC_IsFinished(mdl : pFMUSIC_MODULE) : ByteBool; stdcall;
Function	FMUSIC_IsPlaying(mdl : pFMUSIC_MODULE) : ByteBool; stdcall;
Function	FMUSIC_GetMasterVolume(mdl : pFMUSIC_MODULE) : DWORD; stdcall;
Function	FMUSIC_GetGlobalVolume(mdl : pFMUSIC_MODULE) : DWORD; stdcall;
Function	FMUSIC_GetOrder(mdl : pFMUSIC_MODULE) : DWORD; stdcall;
Function	FMUSIC_GetPattern(mdl : pFMUSIC_MODULE) : DWORD; stdcall;
Function	FMUSIC_GetSpeed(mdl : pFMUSIC_MODULE) : DWORD; stdcall;
Function	FMUSIC_GetBPM(mdl : pFMUSIC_MODULE) : DWORD; stdcall;
Function	FMUSIC_GetRow(mdl : pFMUSIC_MODULE) : DWORD; stdcall;
Function	FMUSIC_GetPaused(mdl : pFMUSIC_MODULE) : ByteBool; stdcall;
Function	FMUSIC_GetTime(mdl : pFMUSIC_MODULE) : ULong; stdcall;
  


















































implementation




// ==================================
// Initialization / Global functions.
// ==================================

// Pre FSOUND_Init functions. These can't be called after FSOUND_Init is called (they will fail)
Function	FSOUND_SetOutput(outputtype : FSOUND_OUTPUTTYPES) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_SetOutput@4';
Function	FSOUND_SetDriver(driver : DWORD) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_SetDriver@4';
Function	FSOUND_SetMixer(mixer : FSOUND_MIXERTYPES) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_SetMixer@4';
Function	FSOUND_SetBufferSize(len_ms : DWORD) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_SetBufferSize@4';
Function	FSOUND_SetHWND(hwnd : pointer) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_SetHWND@4';
Function	FSOUND_SetMinHardwareChannels(min : DWORD) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_SetMinHardwareChannels@4'
Function	FSOUND_SetMaxHardwareChannels(max : DWORD) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_SetMaxHardwareChannels@4'

// Main initialization / closedown functions
Function	FSOUND_Init(mixrate, maxchannels, vcmmode : DWORD) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Init@12';
Procedure	FSOUND_Close; stdcall; external 'fmod.dll' name '_FSOUND_Close@0';

// Runtime 
Procedure 	FSOUND_SetSFXMasterVolume(volume : DWORD); stdcall; external 'fmod.dll' name '_FSOUND_SetSFXMasterVolume@4';
Procedure 	FSOUND_SetPanSeperation(pansep : single); stdcall; external 'fmod.dll' name '_FSOUND_SetPanSeperation@4';

// Error functions
Function	FSOUND_GetError : FMOD_ERRORS; stdcall; external 'fmod.dll' name '_FSOUND_GetError@0';


// ===================================
// Sample management / load functions.
// ===================================

// File functions
Function	FSOUND_Sample_LoadWav(index : DWORD; filename : pChar; mode : ULong) : pFSOUND_SAMPLE; stdcall; external 'fmod.dll' name '_FSOUND_Sample_LoadWav@12';
Function	FSOUND_Sample_LoadMpeg(index : DWORD; filename : pChar; mode : ULong) : pFSOUND_SAMPLE; stdcall; external 'fmod.dll' name '_FSOUND_Sample_LoadMpeg@12';
Function	FSOUND_Sample_LoadRaw(index : DWORD; filename : pChar; mode : ULong) : pFSOUND_SAMPLE; stdcall; external 'fmod.dll' name '_FSOUND_Sample_LoadRaw@12';
Function	FSOUND_Sample_LoadWavMemory(index : DWORD; data : pointer; mode : ULong; length : DWORD) : pFSOUND_SAMPLE; stdcall; external 'fmod.dll' name '_FSOUND_Sample_LoadWavMemory@16';
Function	FSOUND_Sample_LoadMpegMemory(index : DWORD; data : pointer; mode : ULong; length : DWORD) : pFSOUND_SAMPLE; stdcall; external 'fmod.dll' name '_FSOUND_Sample_LoadMpegMemory@16';

// Sample management functions
Function	FSOUND_Sample_Alloc(index,length : DWORD; mode : ULong; deffreq, defvol, defpan, defpri : DWORD) : pFSOUND_SAMPLE; stdcall; external 'fmod.dll' name '_FSOUND_Sample_Alloc@28';
Procedure 	FSOUND_Sample_Free(sptr : pFSOUND_SAMPLE); stdcall; external 'fmod.dll' name '_FSOUND_Sample_Free@4';
Function	FSOUND_Sample_Upload(sptr : pFSOUND_SAMPLE; srcdata : pointer; mode : ULong) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Sample_Upload@12';
Function	FSOUND_Sample_Lock(sptr : pFSOUND_SAMPLE; offset, length : DWORD; ptr1,ptr2 : pPointer; len1, len2 : pULong) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Sample_Lock@28';
Function	FSOUND_Sample_Unlock(sptr : pFSOUND_SAMPLE; ptr1,ptr2 : pointer; len1, len2 : DWORD) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Sample_Unlock@20';

// Sample control functions
Function	FSOUND_Sample_SetLoopMode(sptr : pFSOUND_SAMPLE; loopmode : ULong) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Sample_SetLoopMode@8';
Function	FSOUND_Sample_SetLoopPoints(sptr : pFSOUND_SAMPLE; loopstart, loopend : DWORD) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Sample_SetLoopPoints@12';
Function	FSOUND_Sample_SetDefaults(sptr : pFSOUND_SAMPLE; deffreq, defvol, defpan, defpri : DWORD) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Sample_SetDefaults@20';
//Procedure 	FSOUND_Sample_UseMulaw(mulaw : ByteBool); stdcall; external 'fmod.dll' name '_FSOUND_Sample_UseMulaw@4';
Function	FSOUND_Sample_SetMinMaxDistance(sptr : pFSOUND_SAMPLE; min, max : single) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Sample_SetMinMaxDistance@12';

  
// ============================
// Channel control functions.
// ============================

// Playing and stopping sounds.
Function	FSOUND_PlaySound(channel : DWORD; sptr : pFSOUND_SAMPLE) : DWORD; stdcall; external 'fmod.dll' name '_FSOUND_PlaySound@8';
Function	FSOUND_PlaySoundAttrib(channel : DWORD; sptr : pFSOUND_SAMPLE; freq, vol, pan : DWORD) : DWORD; stdcall; external 'fmod.dll' name '_FSOUND_PlaySoundAttrib@20';
Function	FSOUND_3D_PlaySound(channel : DWORD; sptr : pFSOUND_SAMPLE; pos:pFSOUND_VECTOR; vel:pFSOUND_VECTOR) : DWORD; stdcall; external 'fmod.dll' name '_FSOUND_3D_PlaySound@16';
Function	FSOUND_StopSound(channel : DWORD) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_StopSound@4';
Procedure 	FSOUND_StopAllChannels; stdcall; external 'fmod.dll' name '_FSOUND_StopAllChannels@0';
 
// Functions to control playback of a channel.
Function	FSOUND_SetFrequency(channel : DWORD; freq : DWORD) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_SetFrequency@8';
Function	FSOUND_SetVolume(channel : DWORD; vol : DWORD) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_SetVolume@8';
Function	FSOUND_SetVolumeAbsolute(channel : DWORD; vol : DWORD) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_SetVolumeAbsolute@8';
Function	FSOUND_SetPan(channel : DWORD; pan : DWORD) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_SetPan@8';
Function	FSOUND_SetSurround(channel : DWORD; surround : ByteBool) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_SetSurround@8';
Function	FSOUND_SetMute(channel : DWORD; mute : ByteBool) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_SetMute@8';
Function	FSOUND_SetPriority(channel : DWORD; priority : DWORD) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_SetPriority@8';
Function	FSOUND_SetReserved(channel : DWORD; reserved : ByteBool) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_SetReserved@8';
Function	FSOUND_SetPaused(channel : DWORD; paused : ByteBool) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_SetPaused@8';
Function	FSOUND_MixBuffers(destbuffer, srcbuffer : pointer; len, freq, vol, pan : DWORD; mode : ULong) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_MixBuffers@28';


// ================================
// Information retrieval functions.
// ================================

// System information
Function	FSOUND_GetVersion : single; stdcall; external 'fmod.dll' name '_FSOUND_GetVersion@0';
Function	FSOUND_GetOutput : FSOUND_OUTPUTTYPES; stdcall; external 'fmod.dll' name '_FSOUND_GetOutput@0';
Function	FSOUND_GetDriver : DWORD; stdcall; external 'fmod.dll' name '_FSOUND_GetDriver@0';
Function	FSOUND_GetMixer	: FSOUND_MIXERTYPES; stdcall; external 'fmod.dll' name '_FSOUND_GetMixer@0';
Function	FSOUND_GetNumDrivers : DWORD; stdcall; external 'fmod.dll' name '_FSOUND_GetNumDrivers@0';
Function	FSOUND_GetDriverName(id : DWORD) : pChar; stdcall; external 'fmod.dll' name '_FSOUND_GetDriverName@4';
Function	FSOUND_GetDriverCaps(id : DWORD; caps : pULong) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_GetDriverCaps@8';

Function	FSOUND_GetOutputRate : DWORD; stdcall; external 'fmod.dll' name '_FSOUND_GetOutputRate@0';
Function	FSOUND_GetMaxChannels : DWORD; stdcall; external 'fmod.dll' name '_FSOUND_GetMaxChannels@0';
Function	FSOUND_GetMaxSamples : DWORD; stdcall; external 'fmod.dll' name '_FSOUND_GetMaxSamples@0';
Function	FSOUND_GetSFXMasterVolume : DWORD; stdcall; external 'fmod.dll' name '_FSOUND_GetSFXMasterVolume@0';
Function	FSOUND_GetNumHardwareChannels : DWORD; stdcall; external 'fmod.dll' name '_FSOUND_GetNumHardwareChannels@0';
Function	FSOUND_GetChannelsPlaying : DWORD; stdcall; external 'fmod.dll' name '_FSOUND_GetChannelsPlaying@0';
Function	FSOUND_GetCPUUsage : single; stdcall; external 'fmod.dll' name '_FSOUND_GetCPUUsage@0';

// Channel information
Function	FSOUND_IsPlaying(channel : DWORD) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_IsPlaying@4';
Function	FSOUND_GetFrequency(channel : DWORD) : DWORD; stdcall; external 'fmod.dll' name '_FSOUND_GetFrequency@4';
Function	FSOUND_GetVolume(channel : DWORD) : DWORD; stdcall; external 'fmod.dll' name '_FSOUND_GetVolume@4';
Function	FSOUND_GetPan(channel : DWORD) : DWORD; stdcall; external 'fmod.dll' name '_FSOUND_GetPan@4';
Function	FSOUND_GetSurround(channel : DWORD) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_GetSurround@4';
Function	FSOUND_GetMute(channel : DWORD) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_GetMute@4';
Function	FSOUND_GetPriority(channel : DWORD) : DWORD; stdcall; external 'fmod.dll' name '_FSOUND_GetPriority@4';
Function	FSOUND_GetReserved(channel : DWORD) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_GetReserved@4';
Function	FSOUND_GetPaused(channel : DWORD) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_GetPaused@4';
Function	FSOUND_GetCurrentPosition(channel : DWORD) : ULong; stdcall; external 'fmod.dll' name '_FSOUND_GetCurrentPosition@4';
Function	FSOUND_GetCurrentSample(channel : DWORD) : pFSOUND_SAMPLE; stdcall; external 'fmod.dll' name '_FSOUND_GetCurrentSample@4';
Function	FSOUND_GetCurrentVU(channel : DWORD) : single; stdcall; external 'fmod.dll' name '_FSOUND_GetCurrentVU@4';

// Sample information 
Function	FSOUND_Sample_Get(sampno : DWORD) : pFSOUND_SAMPLE; stdcall; external 'fmod.dll' name '_FSOUND_Sample_Get@4';
Function	FSOUND_Sample_GetLength(sptr : pFSOUND_SAMPLE) : ULong; stdcall; external 'fmod.dll' name '_FSOUND_Sample_GetLength@4';
Function	FSOUND_Sample_GetLoopPoints(sptr : pFSOUND_SAMPLE; loopstart, loopend : pDWORD) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Sample_GetLoopPoints@12';
Function	FSOUND_Sample_GetDefaults(sptr : pFSOUND_SAMPLE; deffreq, defvol, defpan, defpri : pDWORD) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Sample_GetDefaults@20';
Function	FSOUND_Sample_GetMode(sptr : pFSOUND_SAMPLE) : ULong; stdcall; external 'fmod.dll' name '_FSOUND_Sample_GetMode@4';
//Function	FSOUND_Sample_IsUsingMulaw : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Sample_IsUsingMulaw@0';

 
// ===================
// 3D sound functions.
// ===================
// see also FSOUND_3D_PlaySound (above)
// see also FSOUND_Sample_SetMinMaxDistance (above)
Procedure 	FSOUND_3D_Update; stdcall; external 'fmod.dll' name '_FSOUND_3D_Update@0';
Function	FSOUND_3D_SetAttributes(channel : DWORD; pos, vel : pFSOUND_VECTOR) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_3D_SetAttributes@12'; 
Function	FSOUND_3D_GetAttributes(channel : DWORD; pos, vel : pFSOUND_VECTOR) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_3D_GetAttributes@12';
Procedure 	FSOUND_3D_Listener_SetAttributes(pos, vel : pFSOUND_VECTOR; fx, fy, fz, tx, ty, tz : single); stdcall; external 'fmod.dll' name '_FSOUND_3D_Listener_SetAttributes@32';
Procedure 	FSOUND_3D_Listener_GetAttributes(pos, vel : pFSOUND_VECTOR; fx, fy, fz, tx, ty, tz : psingle); stdcall; external 'fmod.dll' name '_FSOUND_3D_Listener_GetAttributes@32';
Procedure 	FSOUND_3D_Listener_SetDopplerFactor(scale : single); stdcall; external 'fmod.dll' name '_FSOUND_3D_Listener_SetDopplerFactor@4';
Procedure 	FSOUND_3D_Listener_SetDistanceFactor(scale : single); stdcall; external 'fmod.dll' name '_FSOUND_3D_Listener_SetDistanceFactor@4';
Procedure 	FSOUND_3D_Listener_SetRolloffFactor(scale : single); stdcall; external 'fmod.dll' name '_FSOUND_3D_Listener_SetRolloffFactor@4';


// ===================
// Geometry functions.
// ===================

// scene/polygon functions
Function	FSOUND_Geometry_AddPolygon(p1,p2,p3,p4,normal : pFSOUND_VECTOR; mode : ULong; openingfactor : psingle) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Geometry_AddPolygon@28';
Function	FSOUND_Geometry_AddList(geomlist : pFSOUND_GEOMLIST) : DWORD; stdcall; external 'fmod.dll' name '_FSOUND_Geometry_AddList@4';

// polygon list functions
Function	FSOUND_Geometry_List_Create(boundingvolume : ByteBool) : pFSOUND_GEOMLIST; stdcall; external 'fmod.dll' name '_FSOUND_Geometry_List_Create@4';
Function	FSOUND_Geometry_List_Free(geomlist : pFSOUND_GEOMLIST) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Geometry_List_Free@4';
Function	FSOUND_Geometry_List_Begin(geomlist : pFSOUND_GEOMLIST) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Geometry_List_Begin@4';
Function	FSOUND_Geometry_List_End(geomlist : pFSOUND_GEOMLIST) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Geometry_List_End@4';
Function	FSOUND_Geometry_List_Add(geomlist : pFSOUND_GEOMLIST) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Geometry_List_Add@4';

// material functions
Function	FSOUND_Geometry_Material_Create : pFSOUND_MATERIAL; stdcall; external 'fmod.dll' name '_FSOUND_Geometry_Material_Create@0';
Function	FSOUND_Geometry_Material_Free(material : pFSOUND_MATERIAL) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Geometry_Material_Free@4';
Function	FSOUND_Geometry_Material_SetAttributes(material : pFSOUND_MATERIAL; reflectancegain, reflectancefreq, transmittancegain, transmittancefreq : single) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Geometry_Material_SetAttributes@20';
Function	FSOUND_Geometry_Material_GetAttributes(material : pFSOUND_MATERIAL; reflectancegain, reflectancefreq, transmittancegain, transmittancefreq : psingle) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Geometry_Material_GetAttributes@20';
Function	FSOUND_Geometry_Material_Set(material : pFSOUND_MATERIAL) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Geometry_Material_Set@4';

// occlusion (supported in software, eax, eax2, a3d3)
Function	FSOUND_Geometry_SetOcclusionFactor(factor : single) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Geometry_SetOcclusionFactor@4';

// ==============================================
// Reverb functions. (eax, eax2, a3d 3.0 reverb)
// ==============================================

// eax1, eax2, a3d 3.0 (use FSOUND_REVERB_PRESETS if you like), (eax2 support through emulation/parameter conversion)
Function	FSOUND_Reverb_SetEnvironment(env : FSOUND_REVERB_ENVIRONMENTS; vol, decay, damp : single) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Reverb_SetEnvironment@16';
// eax2, a3d 3.0 only, does not work on eax1
Function	FSOUND_Reverb_SetEnvironmentAdvanced(
				env : FSOUND_REVERB_ENVIRONMENTS;
				Room : DWORD;					// [-10000, 0]     default: -10000 mB
				RoomHF : DWORD;					// [-10000, 0]     default: 0 mB
				RoomRolloffFactor : single;		// [0.0, 10.0]     default: 0.0
				DecayTime : single;				// [0.1, 20.0]     default: 1.0 s
				DecayHFRatio : single;			// [0.1, 2.0]      default: 0.5
				Reflections : DWORD;			// [-10000, 1000]  default: -10000 mB
				ReflectionsDelay : single;		// [0.0, 0.3]      default: 0.02 s
				Reverb : DWORD;					// [-10000, 2000]  default: -10000 mB
				ReverbDelay : single;			// [0.0, 0.1]      default: 0.04 s
				EnvironmentSize : single;		// [0.0, 100.0]    default: 100.0 %
				EnvironmentDiffusion : single;	// [0.0, 100.0]    default: 100.0 %
				AirAbsorptionHF : single) :		// [20.0, 20000.0] default: 5000.0 Hz
					FSOUND_REVERB_ENVIRONMENTS; stdcall; external 'fmod.dll' name '_FSOUND_Reverb_SetEnvironmentAdvanced@52';

Function	FSOUND_Reverb_SetMix(channel : DWORD; mix : single) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Reverb_SetMix@8';

// information functions
Function	FSOUND_Reverb_GetEnvironment(env : pDWORD; vol, decay, damp : psingle) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Reverb_GetEnvironment@16';
Function	FSOUND_Reverb_GetEnvironmentAdvanced(
					env : pDWORD;
					Room : pDWORD;
					RoomHF : pDWORD;
					RoomRolloffFactor : psingle;
					DecayTime : psingle;
					DecayHFRatio : psingle;
					Reflections : pDWORD;
					ReflectionsDelay : psingle;
					Reverb : pDWORD;
					ReverbDelay : psingle;
					EnvironmentSize : psingle;
					EnvironmentDiffusion : psingle;
					AirAbsorptionHF : psingle) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Reverb_GetEnvironmentAdvanced@52';
Function	FSOUND_Reverb_GetMix(channel : DWORD; mix : psingle) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Reverb_GetMix@8';
 
// =========================
// File Streaming functions.
// =========================

Function	FSOUND_Stream_Create(callback:FSOUND_STREAMCALLBACK; length : DWORD; mode : ULong; samplerate, userdata : DWORD) : pFSOUND_STREAM; stdcall; external 'fmod.dll' name '_FSOUND_Stream_Create@20';
Function	FSOUND_Stream_Open(filename : pChar; mode : ULong; samplerate : DWORD) : pFSOUND_STREAM; stdcall; external 'fmod.dll' name '_FSOUND_Stream_Open@12';
Function	FSOUND_Stream_OpenWav(filename : pChar; mode : ULong) : pFSOUND_STREAM; stdcall; external 'fmod.dll' name '_FSOUND_Stream_OpenWav@8';
Function	FSOUND_Stream_OpenMpeg(filename : pChar; mode : ULong) : pFSOUND_STREAM; stdcall; external 'fmod.dll' name '_FSOUND_Stream_OpenMpeg@8';
Function	FSOUND_Stream_Play(channel : DWORD; stream : pFSOUND_STREAM) : DWORD; stdcall; external 'fmod.dll' name '_FSOUND_Stream_Play@8';
Function	FSOUND_Stream_Stop(stream : pFSOUND_STREAM) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Stream_Stop@4';
Function	FSOUND_Stream_Close(stream : pFSOUND_STREAM) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Stream_Close@4';

Function	FSOUND_Stream_SetPaused(stream : pFSOUND_STREAM; paused : ByteBool) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Stream_SetPaused@8';
Function	FSOUND_Stream_GetPaused(stream : pFSOUND_STREAM) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Stream_GetPaused@4';
Function	FSOUND_Stream_SetPosition(stream : pFSOUND_STREAM; position : DWORD) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Stream_SetPosition@8';
Function	FSOUND_Stream_GetPosition(stream : pFSOUND_STREAM): DWORD; stdcall; external 'fmod.dll' name '_FSOUND_Stream_GetPosition@4';
Function	FSOUND_Stream_GetTime(stream : pFSOUND_STREAM) : DWORD; stdcall; external 'fmod.dll' name '_FSOUND_Stream_GetTime@4';
Function	FSOUND_Stream_GetLength(stream : pFSOUND_STREAM) : DWORD; stdcall; external 'fmod.dll' name '_FSOUND_Stream_GetLength@4';

// ===================
// CD audio functions.
// ===================

Function	FSOUND_CD_Play(track : DWORD) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_CD_Play@4';
Procedure 	FSOUND_CD_SetPlayMode(mode : ByteBool) stdcall; external 'fmod.dll' name '_FSOUND_CD_SetPlayMode@4';
Function	FSOUND_CD_Stop : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_CD_Stop@0';
Function	FSOUND_CD_SetPaused(paused : ByteBool) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_CD_SetPaused@4';
Function	FSOUND_CD_GetPaused : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_CD_GetPaused@0';
Function	FSOUND_CD_GetTrack : DWORD; stdcall; external 'fmod.dll' name '_FSOUND_CD_GetTrack@0';
Function	FSOUND_CD_GetNumTracks : DWORD; stdcall; external 'fmod.dll' name '_FSOUND_CD_GetNumTracks@0';
Function	FSOUND_CD_Eject : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_CD_Eject@0';


// ==============
// DSP functions.
// ==============

// DSP Unit control and information functions.
Function	FSOUND_DSP_Create(callback:FSOUND_DSPCALLBACK; priority, param : DWORD) : PFSOUND_DSPUNIT; stdcall; external 'fmod.dll' name '_FSOUND_DSP_Create@12';
Procedure 	FSOUND_DSP_Free(unt : pFSOUND_DSPUNIT); stdcall; external 'fmod.dll' name '_FSOUND_DSP_Free@4';

Procedure 	FSOUND_DSP_SetPriority(unt : pFSOUND_DSPUNIT; priority : DWORD); stdcall; external 'fmod.dll' name '_FSOUND_DSP_SetPriority@8';
Function	FSOUND_DSP_GetPriority(unt : pFSOUND_DSPUNIT) : DWORD; stdcall; external 'fmod.dll' name '_FSOUND_DSP_GetPriority@4';

Procedure 	FSOUND_DSP_SetActive(unt : pFSOUND_DSPUNIT; active : ByteBool); stdcall; external 'fmod.dll' name '_FSOUND_DSP_SetActive@8';
Function	FSOUND_DSP_GetActive(unt : pFSOUND_DSPUNIT) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_DSP_GetActive@4';

// Functions to get hold of FSOUND 'system DSP unit' handles.
Function	FSOUND_DSP_GetClearUnit : pFSOUND_DSPUNIT; stdcall; external 'fmod.dll' name '_FSOUND_DSP_GetClearUnit@0';
Function	FSOUND_DSP_GetSFXUnit : pFSOUND_DSPUNIT; stdcall; external 'fmod.dll' name '_FSOUND_DSP_GetSFXUnit@0';
Function	FSOUND_DSP_GetMusicUnit : pFSOUND_DSPUNIT; stdcall; external 'fmod.dll' name '_FSOUND_DSP_GetMusicUnit@0';
Function	FSOUND_DSP_GetClipAndCopyUnit : pFSOUND_DSPUNIT; stdcall; external 'fmod.dll' name '_FSOUND_DSP_GetClipAndCopyUnit@0';

// misc DSP functions
Procedure 	FSOUND_DSP_ClearMixBuffer; stdcall; external 'fmod.dll' name '_FSOUND_DSP_ClearMixBuffer@0';
Function	FSOUND_DSP_GetBufferLength : DWORD; stdcall; external 'fmod.dll' name '_FSOUND_DSP_GetBufferLength@0';


// =========================
// Recording functions
// =========================

// recording initialization
Function	FSOUND_Record_SetDriver(outputtype : DWORD) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Record_SetDriver@4';

// information functions.
Function	FSOUND_Record_GetCaps(modes : pULong) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Record_GetCaps@4';
Function	FSOUND_Record_GetNumDrivers : DWORD; stdcall; external 'fmod.dll' name '_FSOUND_Record_GetNumDrivers@0';
Function	FSOUND_Record_GetDriverName(id : DWORD) : pChar; stdcall; external 'fmod.dll' name '_FSOUND_Record_GetDriverName@4';
Function	FSOUND_Record_GetDriver : DWORD; stdcall; external 'fmod.dll' name '_FSOUND_Record_GetDriver@0';

// recording functionality.  Only one recording session will work at a time
Function	FSOUND_Record_StartSample(sptr : pFSOUND_SAMPLE; loop : ByteBool) : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Record_StartSample@8';
Function	FSOUND_Record_Stop : ByteBool; stdcall; external 'fmod.dll' name '_FSOUND_Record_Stop@0';
Function	FSOUND_Record_GetPosition : DWORD; stdcall; external 'fmod.dll' name '_FSOUND_Record_GetPosition@0';

// =========================
// File system override
// =========================

Procedure FSOUND_File_SetCallbacks(OpenCallback  : FSOUND_OPENCALLBACK;
                                   CloseCallback : FSOUND_CLOSECALLBACK;
                                   ReadCallback  : FSOUND_READCALLBACK;
                                   SeekCallback  : FSOUND_SEEKCALLBACK;
                                   TellCallback  : FSOUND_TELLCALLBACK); stdcall; external 'fmod.dll' name '_FSOUND_File_SetCallbacks@20';



// =============================================================================================
// FMUSIC API
// =============================================================================================

// Song management / playback functions.
// =====================================
Function	FMUSIC_LoadSong(name : pChar) : pFMUSIC_MODULE; stdcall; external 'fmod.dll' name '_FMUSIC_LoadSong@4';
Function	FMUSIC_LoadSongMemory(data : pointer; length : DWORD) : pFMUSIC_MODULE; stdcall; external 'fmod.dll' name '_FMUSIC_LoadSongMemory@8';
Function	FMUSIC_FreeSong(mdl : pFMUSIC_MODULE) : ByteBool; stdcall; external 'fmod.dll' name '_FMUSIC_FreeSong@4';
Function	FMUSIC_PlaySong(mdl : pFMUSIC_MODULE) : ByteBool; stdcall; external 'fmod.dll' name '_FMUSIC_PlaySong@4';
Function	FMUSIC_StopSong(mdl : pFMUSIC_MODULE) : ByteBool; stdcall; external 'fmod.dll' name '_FMUSIC_StopSong@4';
Procedure 	FMUSIC_StopAllSongs; stdcall; external 'fmod.dll' name '_FMUSIC_StopAllSongs@0';
Function	FMUSIC_SetZxxCallback(mdl : pFMUSIC_MODULE; callback : FMUSIC_CALLBACK) : ByteBool; stdcall; external 'fmod.dll' name '_FMUSIC_SetZxxCallback@8';
Function	FMUSIC_SetRowCallback(mdl : pFMUSIC_MODULE; callback : FMUSIC_CALLBACK; rowstep:DWORD) : ByteBool; stdcall; external 'fmod.dll' name '_FMUSIC_SetRowCallback@12';
Function	FMUSIC_SetOrderCallback(mdl : pFMUSIC_MODULE; callback : FMUSIC_CALLBACK; orderstep:DWORD) : ByteBool; stdcall; external 'fmod.dll' name '_FMUSIC_SetOrderCallback@12';

Function	FMUSIC_OptimizeChannels(mdl : pFMUSIC_MODULE; maxchannels, minvolume : DWORD) : ByteBool; stdcall; external 'fmod.dll' name '_FMUSIC_OptimizeChannels@12';


// Runtime song functions.
// =======================
Function	FMUSIC_SetOrder(mdl : pFMUSIC_MODULE; order : DWORD) : ByteBool; stdcall; external 'fmod.dll' name '_FMUSIC_SetOrder@8';
Function	FMUSIC_SetPaused(mdl : pFMUSIC_MODULE; pause : ByteBool) : ByteBool; stdcall; external 'fmod.dll' name '_FMUSIC_SetPaused@8';
Function	FMUSIC_SetMasterVolume(mdl : pFMUSIC_MODULE; volume : DWORD) : ByteBool; stdcall; external 'fmod.dll' name '_FMUSIC_SetMasterVolume@8';
Function	FMUSIC_SetPanSeperation(mdl : pFMUSIC_MODULE; pansep : single) : ByteBool; stdcall; external 'fmod.dll' name '_FMUSIC_SetPanSeperation@8';
 
// Static song information functions.
// ==================================
Function	FMUSIC_GetName(mdl : pFMUSIC_MODULE) : pChar; stdcall; external 'fmod.dll' name '_FMUSIC_GetName@4';
Function	FMUSIC_GetType(mdl : pFMUSIC_MODULE) : FMUSIC_TYPES; stdcall; external 'fmod.dll' name '_FMUSIC_GetType@4';
Function	FMUSIC_UsesLinearFrequencies(mdl : pFMUSIC_MODULE) : ByteBool; stdcall; external 'fmod.dll' name '_FMUSIC_UsesLinearFrequencies@4';
//Function	FMUSIC_UsesMulaw(mdl : pFMUSIC_MODULE) : ByteBool; stdcall; external 'fmod.dll' name '_FMUSIC_UsesMulaw@4';
Function	FMUSIC_GetNumOrders(mdl : pFMUSIC_MODULE) : DWORD; stdcall; external 'fmod.dll' name '_FMUSIC_GetNumOrders@4';
Function	FMUSIC_GetNumPatterns(mdl : pFMUSIC_MODULE) : DWORD; stdcall; external 'fmod.dll' name '_FMUSIC_GetNumPatterns@4';
Function	FMUSIC_GetNumInstruments(mdl : pFMUSIC_MODULE) : DWORD; stdcall; external 'fmod.dll' name '_FMUSIC_GetNumInstruments@4';
Function	FMUSIC_GetNumSamples(mdl : pFMUSIC_MODULE) : DWORD; stdcall; external 'fmod.dll' name '_FMUSIC_GetNumSamples@4';
Function	FMUSIC_GetNumChannels(mdl : pFMUSIC_MODULE) : DWORD; stdcall; external 'fmod.dll' name '_FMUSIC_GetNumChannels@4';
Function	FMUSIC_GetSample(mdl : pFMUSIC_MODULE; sampno : DWORD) : pFSOUND_SAMPLE; stdcall; external 'fmod.dll' name '_FMUSIC_GetSample@8';
Function	FMUSIC_GetPatternLength(mdl : pFMUSIC_MODULE; orderno : DWORD) : DWORD; stdcall; external 'fmod.dll' name '_FMUSIC_GetPatternLength@8';

// Runtime song information.
// =========================
Function	FMUSIC_IsFinished(mdl : pFMUSIC_MODULE) : ByteBool; stdcall; external 'fmod.dll' name '_FMUSIC_IsFinished@4';
Function	FMUSIC_IsPlaying(mdl : pFMUSIC_MODULE) : ByteBool; stdcall; external 'fmod.dll' name '_FMUSIC_IsPlaying@4';
Function	FMUSIC_GetMasterVolume(mdl : pFMUSIC_MODULE) : DWORD; stdcall; external 'fmod.dll' name '_FMUSIC_GetMasterVolume@4';
Function	FMUSIC_GetGlobalVolume(mdl : pFMUSIC_MODULE) : DWORD; stdcall; external 'fmod.dll' name '_FMUSIC_GetGlobalVolume@4';
Function	FMUSIC_GetOrder(mdl : pFMUSIC_MODULE) : DWORD; stdcall; external 'fmod.dll' name '_FMUSIC_GetOrder@4';
Function	FMUSIC_GetPattern(mdl : pFMUSIC_MODULE) : DWORD; stdcall; external 'fmod.dll' name '_FMUSIC_GetPattern@4';
Function	FMUSIC_GetSpeed(mdl : pFMUSIC_MODULE) : DWORD; stdcall; external 'fmod.dll' name '_FMUSIC_GetSpeed@4';
Function	FMUSIC_GetBPM(mdl : pFMUSIC_MODULE) : DWORD; stdcall; external 'fmod.dll' name '_FMUSIC_GetBPM@4';
Function	FMUSIC_GetRow(mdl : pFMUSIC_MODULE) : DWORD; stdcall; external 'fmod.dll' name '_FMUSIC_GetRow@4';
Function	FMUSIC_GetPaused(mdl : pFMUSIC_MODULE) : ByteBool; stdcall; external 'fmod.dll' name '_FMUSIC_GetPaused@4';
Function	FMUSIC_GetTime(mdl : pFMUSIC_MODULE) : ULong; stdcall; external 'fmod.dll' name '_FMUSIC_GetTime@4';

end.
