{: GLVectorFileObjects

	Vector File related objects for GLScene<p>

	<b>Historique : </b><font size=-1><ul>
      <li>28/06/00 - Egg - Support for "ObjectStyle"
      <li>23/06/00 - Egg - Reversed "t" texture coord for MD2,
                           TActorAnimations can now load/save
      <li>21/06/00 - Egg - Added frame change events to TActor,
                           Added TActorAnimations collection
      <li>19/06/00 - Egg - Completed smooth movement interpolation for TActor
      <li>07/06/00 - Egg - TVectorFile now longers assumes a TFreeForm as Owner,
                           Added generic TVectorFile.LoadFromFile
      <li>26/05/00 - Egg - Removed dependency to GLObjects,
                           TFreeForm now may use InterleavedArrays instead of
                           IndexedArrays (better BuildList compatibility)
      <li>22/04/00 - Egg - Fixed Material handlings in TFreeForm, inverted CCW/CW
                           convention for 3DS Release3
		<li>11/04/00 - Egg - Removed unnecessary code in finalization (thanks Uwe)
	   <li>09/02/00 - Egg - Creation from split of GLObjects,
                           fixed class registrations and formats unregistration
	</ul></font>
}
unit GLVectorFileObjects;

interface

uses Classes, GLScene, OpenGL12, Geometry, SysUtils, GLMisc, GLTexture,
     // MD2 Support
	  FileMD2, TypeMD2
     ;

type

   TFreeForm = class;

   PFaceGroup = ^TFaceGroup;
   TFaceGroup = record
                 FaceProperties : TGLFaceProperties;
                 IndexCount     : Cardinal;
                 Indices        : PIntegerArray;
                 InterLeavedArray : PFloatArray; // interleaved array, calc'ed from the others
               end;

   TFaceGroups = class(TList)
   private
     function GetFaceGroup(Index: Integer): PFaceGroup;
   public
     procedure Clear; override;
     property Items[Index: Integer]: PFaceGroup read GetFaceGroup; default;
   end;

   PMeshObject = ^TMeshObject;
   TMeshObject = record
     Mode: TMeshMode;              // mmTriangles and mmTriangleStrip are supported
     Vertices: PAffineVectorArray; // comprises all vertices of the object in no particular order
     VertexCount : Integer;
     Normals: PAffineVectorArray;
     TexCoords: PTexPointArray;
     FaceGroups: TFaceGroups;    // a list of face groups, each comprising a material description and
                                 // a list of indices into the VerticesArray array
   end;

   // TMeshNormalsOrientation
   //
   {: Determines how normals orientation is defined in a mesh.<p>
      - mnoDefault : uses default orientation<br>
      - mnoInvert : inverse of default orientation<br>
      - mnoAutoSolid : autocalculate to make the mesh globally solid<br>
      - mnoAutoHollow : autocalculate to make the mesh globally hollow<br> }
   TMeshNormalsOrientation = (mnoDefault, mnoInvert); //, mnoAutoSolid, mnoAutoHollow);

   // TVectorFile
   //
   {: Abstract base class for different vector file formats.<p>
      The actual implementation for these files (3DS, DXF..) must be done
      seperately. The concept for TVectorFile is very similar to TGraphic
      (see Delphi Help). }
   TVectorFile = class (TPersistent)
      private
         { Private Declarations }
         FOwner : TPersistent;
         FNormalsOrientation : TMeshNormalsOrientation;

      protected
         { Protected Declarations }
         function GetOwner : TPersistent; override;
         procedure SetNormalsOrientation(const val : TMeshNormalsOrientation); virtual;

      public
         { Public Declarations }
         constructor Create(AOwner: TPersistent); virtual;

         {: Creates a stream from the file and loads the stream. }
         procedure LoadFromFile(const aFileName: String); dynamic;
         procedure LoadFromStream(aStream: TStream); dynamic; abstract;

         property NormalsOrientation : TMeshNormalsOrientation read FNormalsOrientation write SetNormalsOrientation;
   end;

   TVectorFileClass = class of TVectorFile;

   // TGL3DSVectorFile
   //
   //: The 3DStudio vector file
   TGL3DSVectorFile = class(TVectorFile)
      public
         { Public Declarations }
         procedure LoadFromStream(aStream: TStream); override;
   end;

   // TGLMD2VectorFile
   //
   //: The md2 vector file
   TGLMD2VectorFile = class(TVectorFile)
      protected
         { Protected Declarations }
         function ConvertMD2Structure(MD2File: TFileMD2; frame: Integer): PMeshObject;
      public
         { Public Declarations }
         procedure LoadFromStream(aStream: TStream); override;
   end;

   // TBaseMesh
   //
   {: Base class for mesh objects. }
   TBaseMesh = class(TGLSceneObject)
      private
         { Private Declarations }
         FObjects: TList;     // a list of mesh objects

      protected
         { Protected Declarations }

      public
         { Public Declarations }
         constructor Create(AOwner: TComponent); override;
         destructor Destroy; override;

         function AddMeshObject(AObject: PMeshObject): Integer; virtual;

         procedure Clear;
         {: Loads a vector file.<p>
            A vector files (for instance a ".3DS") stores the definition of
            a mesh as well as materials property.<p>
            Loading a file replaces the current one (if any). }
         procedure LoadFromFile(const Filename : String);
         {: Loads a vector file from a stream.<p>
            See LoadFromFile.. }
         procedure LoadFromStream(const Filename : String; aStream : TStream);

   end;

   // TFreeFormMode
   //
   {: Chooses how FreeForm vertex data is passed to OpenGL.<p>
      - ffmInterleavedArrays : each mesh vertices is passed to OpenGL with its
         data (normal, texture coord...) in sequence. Use this mode if
         indexed arrays induce rendering glitches (rare problem).<br>
      - ffmIndexedArrays : all vertices are passed first, then indexes (to
         define triangles) are passed to OpenGL (default).<p>
      ffmIndexedArrays is theoretically the best way to describe a mesh (for
      instance a vertex used in 2 triangles will only be passed once, when
      it will be passed twice in interleaved mode), however it has been rarely
      optimized in hardware drivers, leading to performances similar to those
      of ffmInterleavedArrays. }
   TFreeFormMode = (ffmInterleavedArrays, ffmIndexedArrays);

   // TFreeForm
   //
   {: Container objects for a vector file mesh.<p>
      FreeForms allows loading and rendering vector files (like 3DStudio
      ".3DS" file) in GLScene. Meshes can be loaded with the LoadFromFile
      method.<p>
      A FreeForm may contain more than one mesh, but they will all be handled
      as a single object in a scene. }
   TFreeForm = class(TBaseMesh)
      private
         { Private Declarations }
         FUseMeshMaterials : Boolean;
         FMode : TFreeFormMode;
         FNormalsOrientation : TMeshNormalsOrientation;

      protected
         { Protected Declarations }
         procedure SetUseMeshMaterials(const val : Boolean);
         procedure SetNormalsOrientation(const val : TMeshNormalsOrientation);

      public
         { Public Declarations }
         constructor Create(AOwner: TComponent); override;
         destructor Destroy; override;

         procedure BuildList; override;

      published
         { Published Declarations }
         {: Defines wether materials declared in the vector file mesh are used. }
         property UseMeshMaterials : Boolean read FUseMeshMaterials write SetUseMeshMaterials default True;
         {: Adjusts how mesh data is passed to OpenGL.<p>
            See TFreeFormMode for details. }
         property Mode : TFreeFormMode read FMode write FMode default ffmIndexedArrays;
         {: Normals orientation for owned mesh.<p> }
         property NormalsOrientation : TMeshNormalsOrientation read FNormalsOrientation write SetNormalsOrientation default mnoDefault;
   end;

	// TActorAnimation
	//
	TActorAnimation = class (TCollectionItem)
	   private
	      { Private Declarations }
         FName : String;
         FStartFrame : Integer;
         FEndFrame : Integer;

	   protected
	      { Protected Declarations }
         function GetDisplayName : String; override;
         procedure SetStartFrame(const val : Integer);
         procedure SetEndFrame(const val : Integer);
         procedure SetAsString(const val : String);
         function GetAsString : String;

      public
	      { Public Declarations }
	      constructor Create(Collection : TCollection); override;
	      destructor Destroy; override;
	      procedure Assign(Source: TPersistent); override;

         property AsString : String read GetAsString write SetAsString;

	   published
	      { Published Declarations }
         property Name : String read FName write FName;
         property StartFrame : Integer read FStartFrame write SetStartFrame;
         property EndFrame : Integer read FEndFrame write SetEndFrame;
	end;

   TActor = class;

	// TActorAnimations
	//
	TActorAnimations = class (TCollection)
	   private
	      { Private Declarations }
	      owner : TActor;

	   protected
	      { Protected Declarations }
	      function GetOwner: TPersistent; override;
         procedure SetItems(index : Integer; const val : TActorAnimation);
	      function GetItems(index : Integer) : TActorAnimation;

      public
	      { Public Declarations }
	      constructor Create(AOwner : TActor);
         function Add: TActorAnimation;
	      function FindItemID(ID: Integer): TActorAnimation;
	      function FindName(const aName : String) : TActorAnimation;

	      procedure SetToStrings(aStrings : TStrings);
         procedure SaveToStream(aStream : TStream);
         procedure LoadFromStream(aStream : TStream);
         procedure SaveToFile(const fileName : String);
         procedure LoadFromFile(const fileName : String);

	      property Items[index : Integer] : TActorAnimation read GetItems write SetItems; default;
   end;

   // TActorFrameInterpolation
   //
   {: Actor frame-interpolation mode.<p>
      - afpNone : no interpolation, display CurrentFrame only<br>
      - afpLinear : perform linear interpolation between current and next frame }
   TActorFrameInterpolation = (afpNone, afpLinear);

   // TActorActionMode
   //
   {: Defines how an actor plays between its StartFrame and EndFrame.<p>
      <ul>
      <li>aamNone : no animation is performed
      <li>aamPlayOnce : play from current frame to EndFrame, once end frame has
         been reached, switches to aamNone
      <li>aamLoop : play from current frame to EndFrame, once end frame has
         been reached, sets CurrentFrame to StartFrame
      <li>aamBounceForward : play from current frame to EndFrame, once end frame
         has been reached, switches to aamBounceBackward
      <li>aamBounceBackward : play from current frame to StartFrame, once start
         frame has been reached, switches to aamBounceForward
      </ul> }
   TActorAnimationMode = (aamNone, aamPlayOnce, aamLoop, aamBounceForward,
                          aamBounceBackward);

   // TActor
   //
   TActor = class(TBaseMesh)
      private
         { Private Declarations }
         FFrameCount : Integer;
         FStartFrame, FEndFrame : Integer;
         FCurrentFrame : Integer;
         FCurrentFrameDelta : Single;
         FFrameInterpolation : TActorFrameInterpolation;
         FInterval : Integer;
         FAnimationMode : TActorAnimationMode;
         FOnFrameChanged : TNotifyEvent;
         FOnEndFrameReached, FOnStartFrameReached : TNotifyEvent;
         FAnimations : TActorAnimations;

      protected
         { Protected Declarations }
         procedure SetCurrentFrame(Value: Integer);
         procedure SetStartFrame(Value: Integer);
         procedure SetEndFrame(Value: Integer);
         procedure SetAnimations(const val : TActorAnimations);

      public
         { Public Declarations }
         constructor Create(AOwner: TComponent); override;
         destructor Destroy; override;

         procedure BuildList; override;
         procedure LoadFromFile(const Filename: string);

			procedure DoProgress(const deltaTime, newTime : Double); override;

	      procedure SwitchToAnimation(anAnimation : TActorAnimation); overload;
	      procedure SwitchToAnimation(const animationName : String); overload;
	      procedure SwitchToAnimation(animationIndex : Integer); overload;

         {: Synchronize self animation with an other actor.<p>
            Copies Start/Current/End Frame values, CurrentFrameDelta and
            AnimationMode. }
         procedure Synchronize(referenceActor : TActor);

         function  NextFrameIndex : Integer;
         procedure NextFrame(n : Integer = 1);
         procedure PrevFrame(n : Integer = 1);

         property FrameCount: Integer read FFrameCount;

      published
         { Published Declarations }
         property StartFrame: Integer read FStartFrame write SetStartFrame;
         property EndFrame: Integer read FEndFrame write SetEndFrame;

         property CurrentFrame: Integer read FCurrentFrame write SetCurrentFrame;
         {: Value in the [0; 1] range expressing the delta to the next frame.<p> }
         property CurrentFrameDelta : Single read FCurrentFrameDelta write FCurrentFrameDelta;
         property FrameInterpolation : TActorFrameInterpolation read FFrameInterpolation write FFrameInterpolation default afpLinear;

         {: See TActorAnimationMode.<p> }
         property AnimationMode : TActorAnimationMode read FAnimationMode write FAnimationMode default aamNone;
         {: Interval between frames, in milliseconds. }
         property Interval : Integer read FInterval write FInterval;

         {: Triggered after each CurrentFrame change. } 
         property OnFrameChanged : TNotifyEvent read FOnFrameChanged write FOnFrameChanged;
         {: Triggered after EndFrame has been reached by progression or "nextframe" }
         property OnEndFrameReached : TNotifyEvent read FOnEndFrameReached write FOnEndFrameReached;
         {: Triggered after StartFrame has been reached by progression or "nextframe" }
         property OnStartFrameReached : TNotifyEvent read FOnStartFrameReached write FOnStartFrameReached;

         property Animations : TActorAnimations read FAnimations write SetAnimations;
   end;


   PVectorFileFormat = ^TVectorFileFormat;
   TVectorFileFormat = record
      VectorFileClass : TVectorFileClass;
      Extension       : String;
      Description     : String;
      DescResID       : Integer;
   end;

   TVectorFileFormatsList = class(TList)
   public
     destructor Destroy; override;
     procedure Add(const Ext, Desc: String; DescID: Integer; AClass: TVectorFileClass);
     function FindExt(Ext: string): TVectorFileClass;
     procedure Remove(AClass: TVectorFileClass);
     procedure BuildFilterStrings(VectorFileClass: TVectorFileClass; var Descriptions, Filters: string);
   end;

   EInvalidVectorFile = class(Exception);


function GetVectorFileFormats : TVectorFileFormatsList;
procedure RegisterVectorFileFormat(const aExtension, aDescription: String;
                                   aClass : TVectorFileClass);
procedure UnregisterVectorFileClass(aClass : TVectorFileClass);

// ------------------------------------------------------------------
// ------------------------------------------------------------------
// ------------------------------------------------------------------
implementation
// ------------------------------------------------------------------
// ------------------------------------------------------------------
// ------------------------------------------------------------------

uses GLStrings, consts,
     // 3DS Support
	  File3DS, Types3DS;

var
   vVectorFileFormats : TVectorFileFormatsList;

const
   cAAFHeader = 'AAF';

function GetVectorFileFormats: TVectorFileFormatsList;
begin
   if not Assigned(vVectorFileFormats)then
      vVectorFileFormats := TVectorFileFormatsList.Create;
   Result := vVectorFileFormats;
end;

procedure RegisterVectorFileFormat(const AExtension, ADescription: String; AClass: TVectorFileClass);
begin
   RegisterClass(AClass);
	GetVectorFileFormats.Add(AExtension, ADescription, 0, AClass);
end;

procedure UnregisterVectorFileClass(AClass: TVectorFileClass);
begin
	if Assigned(vVectorFileFormats) then
		vVectorFileFormats.Remove(AClass);
end;

//----------------- vector format support --------------------------------------

destructor TVectorFileFormatsList.Destroy;

var I: Integer;

begin
  for I := 0 to Count - 1 do Dispose(PVectorFileFormat(Items[I]));
  inherited Destroy;
end;

//------------------------------------------------------------------------------

procedure TVectorFileFormatsList.Add(const Ext, Desc: String; DescID: Integer;
                                     AClass: TVectorFileClass);

var NewRec: PVectorFileFormat;

begin
  New(NewRec);
  with NewRec^ do
  begin
    Extension := AnsiLowerCase(Ext);
    VectorFileClass := AClass;
    Description := Desc;
    DescResID := DescID;
  end;
  inherited Add(NewRec);
end;

//------------------------------------------------------------------------------

function TVectorFileFormatsList.FindExt(Ext: string): TVectorFileClass;

var I: Integer;

begin
  Ext := AnsiLowerCase(Ext);
  for I := Count-1 downto 0 do
    with PVectorFileFormat(Items[I])^ do
      if Extension = Ext then
      begin
        Result := VectorFileClass;
        Exit;
      end;
  Result := nil;
end;

//------------------------------------------------------------------------------

procedure TVectorFileFormatsList.Remove(AClass: TVectorFileClass);

var I : Integer;
    P : PVectorFileFormat;

begin
  for I := Count-1 downto 0 do
  begin
    P := PVectorFileFormat(Items[I]);
    if P^.VectorFileClass.InheritsFrom(AClass) then
    begin
      Dispose(P);
      Delete(I);
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure TVectorFileFormatsList.BuildFilterStrings(VectorFileClass: TVectorFileClass;
                                                    var Descriptions, Filters: string);

var C, I : Integer;
    P    : PVectorFileFormat;

begin
  Descriptions := '';
  Filters := '';
  C := 0;
  for I := Count-1 downto 0 do
  begin
    P := PVectorFileFormat(Items[I]);
    if P^.VectorFileClass.InheritsFrom(VectorFileClass) and (P^.Extension <> '') then
      with P^ do
      begin
        if C <> 0 then
        begin
          Descriptions := Descriptions+'|';
          Filters := Filters+';';
        end;
        if (Description = '') and (DescResID <> 0) then Description := LoadStr(DescResID);
        FmtStr(Descriptions, '%s%s (*.%s)|*.%2:s', [Descriptions, Description, Extension]);
        FmtStr(Filters, '%s*.%s', [Filters, Extension]);
        Inc(C);
      end;
  end;
  if C > 1 then FmtStr(Descriptions, '%s (%s)|%1:s|%s', [sAllFilter, Filters, Descriptions]);
end;

//----------------- TFaceGroups -------------------------------------------------

procedure TFaceGroups.Clear;

var I : Integer;
    FaceGroup : PFaceGroup;

begin
  for I := 0 to Count-1 do
  begin
    FaceGroup := Get(I);
    FaceGroup.FaceProperties.Free;
    FreeMem(FaceGroup.Indices);
    FreeMem(FaceGroup.InterleavedArray);
    Dispose(FaceGroup);
  end;
  inherited;
end;

// GetFaceGroup
//
function TFaceGroups.GetFaceGroup(Index: Integer): PFaceGroup;
begin
  Result := Get(Index);
end;

// ------------------
// ------------------ TVectorFile ------------------
// ------------------

// Create
//
constructor TVectorFile.Create(AOwner: TPersistent);
begin
   inherited Create;
   FOwner := AOwner;
end;

// LoadFromFile
//
procedure TVectorFile.LoadFromFile(const aFileName: String);
var
   fs : TFileStream;
begin
   fs:=TFileStream.Create(aFileName, fmOpenRead+fmShareDenyNone);
   try
      LoadFromStream(fs);
   finally
      fs.Free;
   end;
end;

// SetNormalsOrientation
//
procedure TVectorFile.SetNormalsOrientation(const val : TMeshNormalsOrientation);
begin
   FNormalsOrientation:=val;
end;

// GetOwner
//
function TVectorFile.GetOwner : TPersistent;
begin
   Result:=FOwner;
end;

// ------------------
// ------------------ TGL3DSVectorFile ------------------
// ------------------

// LoadFromStream
//
procedure TGL3DSVectorFile.LoadFromStream(aStream: TStream);
type
   TSmoothIndexEntry = array[0..31] of Cardinal;
   PSmoothIndexArray = ^TSmoothIndexArray;
   TSmoothIndexArray = array[0..0] of TSmoothIndexEntry;
var
   Marker: PByteArray;
   CurrentVertexCount: Cardinal;
   SmoothIndices: PSmoothIndexArray;
   Mesh: PMeshObject;

   //--------------- local functions -------------------------------------------

   function FacePropsFromMaterial(Materials : TMaterialList; const Name: String): TGLFaceProperties;
   var
      material: PMaterial3DS;
      specColor : TVector;
   begin
      material := Materials.MaterialByName[Name];
      Assert(Assigned(material));
      Result := TGLFaceProperties.Create(nil);
      with Result do begin
         Ambient.Color := VectorMake(Material.Ambient.R, Material.Ambient.G, Material.Ambient.B, 1);
         Diffuse.Color := VectorMake(Material.Diffuse.R, Material.Diffuse.G, Material.Diffuse.B, 1);
         specColor := VectorMake(Material.Specular.R, Material.Specular.G, Material.Specular.B, 1);
         ScaleVector(specColor, 1 - Material.Shininess);
         Specular.Color := specColor;
         Shininess := Round((1 - Material.ShinStrength) * 128);
      end;
   end;

   //----------------------------------------------------------------------

   function IsVertexMarked(P: Pointer; Index: Integer): Boolean; assembler;
      // tests the Index-th bit, returns True if set else False
   asm
                     BT [EAX], EDX
                     SETC AL
   end;

   //---------------------------------------------------------------------------

   function MarkVertex(P: Pointer; Index: Integer): Boolean; assembler;
      // sets the Index-th bit and return True if it was already set else False
   asm
                     BTS [EAX], EDX
                     SETC AL
   end;

   //---------------------------------------------------------------------------

   procedure StoreSmoothIndex(ThisIndex, SmoothingGroup, NewIndex: Cardinal; P: Pointer);
      // Stores new vertex index (NewIndex) into the smooth index array of vertex ThisIndex
      // using field SmoothingGroup, which must not be 0.
      // For each vertex in the vertex array (also for duplicated vertices) an array of 32 cardinals
      // is maintained (each for one possible smoothing group. If a vertex must be duplicated because
      // it has no smoothing group or a different one then the index of the newly created vertex is
      // stored in the SmoothIndices to avoid loosing the conjunction between not yet processed vertices
      // and duplicated vertices.
      // Note: Only one smoothing must be assigned per vertex. Some available models break this rule and
      //       have more than one group assigned to a face. To make the code fail safe the group ID
      //       is scanned for the lowest bit set.
   asm
                   PUSH EBX
                   BSF EBX, EDX                  // determine smoothing group index (convert flag into an index)
                   MOV EDX, [P]                  // get address of index array
                   SHL EAX, 7                    // ThisIndex * SizeOf(TSmoothIndexEntry)
                   ADD EAX, EDX
                   LEA EDX, [4 * EBX + EAX]      // Address of array + vertex index + smoothing group index
                   MOV [EDX], ECX
                   POP EBX
   end;

   //---------------------------------------------------------------------------

   function GetSmoothIndex(ThisIndex, SmoothingGroup: Cardinal; P: Pointer): Cardinal;
      // Retrieves the vertex index for the given index and smoothing group.
      // This redirection is necessary because a vertex might have been duplicated.
   asm
                   PUSH EBX
                   BSF EBX, EDX                  // determine smoothing group index
                   SHL EAX, 7                    // ThisIndex * SizeOf(TSmoothIndexEntry)
                   ADD EAX, ECX
                   LEA ECX, [4 * EBX + EAX]      // Address of array + vertex index + smoothing group index
                   MOV EAX, [ECX]
                   POP EBX
   end;

   //---------------------------------------------------------------------------

   procedure DuplicateVertex(Index: Integer);
      // extends the vector and normal array by one entry and duplicates the vertex data given by Index
      // the marker and texture arrays will be extended too, if necessary
   begin
      // enhance vertex array
      ReallocMem(Mesh.Vertices, (CurrentVertexCount + 1) * SizeOf(TVector3f));
      Mesh.Vertices[CurrentVertexCount] := Mesh.Vertices[Index];
      // enhance normal array
      ReallocMem(Mesh.Normals, (CurrentVertexCount + 1) * SizeOf(TVector3f));
      Mesh.Normals[CurrentVertexCount] := NullVector;
      // enhance smooth index array
      ReallocMem(SmoothIndices, (CurrentVertexCount + 1) * SizeOf(TSmoothIndexEntry));
      FillChar(SmoothIndices[CurrentVertexCount], SizeOf(TSmoothIndexEntry), $FF);
      // enhance marker array
      if (CurrentVertexCount div 8) <> ((CurrentVertexCount + 1) div 8) then begin
         ReallocMem(Marker, ((CurrentVertexCount + 1) div 8) + 1);
         Marker[(CurrentVertexCount div 8) + 1] := 0;
      end;
      // enhance texture coordinates array
      if assigned(Mesh.TexCoords) then begin
         ReallocMem(Mesh.TexCoords, (CurrentVertexCount + 1) * SizeOf(TTexVert3DS));
         Mesh.TexCoords[CurrentVertexCount] := Mesh.TexCoords[Index];
      end;
      Inc(CurrentVertexCount);
   end;

   //--------------- end local functions ---------------------------------------

var
  Size: Cardinal;
  iMaterial, I, J: Integer;
  FaceGroup: PFaceGroup;
  V1, V2: TAffineVector;
  Face, SubFace, Vertex, TargetVertex: Integer;
  SmoothingGroup: Cardinal;
  CurrentIndex: Word;
  Vector1, Vector2, Normal, center : TAffineVector;
  TotalCount: Cardinal;
begin
   center:=NullVector;
   with TFile3DS.Create do try
      LoadFromStream(aStream);
      // determine front face winding
      { TODO : better face winding }
   { if Version<rlRelease3 then
         FOwner.FWinding:=GL_CCW
      else FOwner.FWinding:=GL_CW; }
      TotalCount:=0;
      for i:=0 to Objects.MeshCount-1 do with PMesh3DS(Objects.Mesh[I])^ do begin
         if IsHidden or (NVertices<3) then Continue;

         // New() just calls GetMem, but I want the memory cleared
         mesh:=AllocMem(SizeOf(TMeshObject));
         mesh.Mode:=mmTriangles;
         // make a copy of the vertex data, this must always be available
         CurrentVertexCount:=NVertices;
         Size := NVertices*SizeOf(TPoint3DS);
         // allocate memory, we do not need to clear it
         GetMem(Mesh.Vertices, Size);
         // we don't need to consider the local mesh matrix here, since all vertices are already
         // transformed into their final positions
         Move(VertexArray^, Mesh.Vertices^, Size);

         // texturing data available (optional)?
         if NTextVerts > 0 then begin
            Size := NTextVerts * SizeOf(TTexVert3DS);
            GetMem(Mesh.TexCoords, Size);
            Move(TextArray^, Mesh.TexCoords^, Size);
         end;

         // allocate memory for the face normal array, the final normal array and the marker array
         Mesh.Normals := AllocMem(NVertices * SizeOf(TVector3f));
         Marker := AllocMem((NVertices div 8) + 1); // one bit for each vertex
         GetMem(SmoothIndices, NVertices * SizeOf(TSmoothIndexEntry));

         if SmoothArray=nil then begin
            // no smoothing groups to consider
            for face:=0 to NFaces-1 do with FaceArray[Face], Mesh^ do begin
               // normal vector for the face
               Vector1:=VectorSubstract(Vertices[V1], Vertices[V2]);
               Vector2:=VectorSubstract(Vertices[V3], Vertices[V2]);
               if DatabaseRelease>=rlRelease3 then
                  Normal:=VectorCrossProduct(Vector1, Vector2)
               else Normal:=VectorCrossProduct(Vector2, Vector1);
               // go for each vertex in the current face
               for Vertex:=0 to 2 do begin
                  // copy current index for faster access
                  CurrentIndex:=FaceRec[Vertex];
                  // already been touched?
                  if IsVertexMarked(Marker, CurrentIndex) then begin
                     // already touched vertex must be duplicated
                     DuplicateVertex(CurrentIndex);
                     FaceRec[Vertex]:=CurrentVertexCount-1;
                     Normals[CurrentVertexCount-1]:=Normal;
                  end else begin
                     // not yet touched, so just store the normal
                     Normals[CurrentIndex] := Normal;
                     MarkVertex(Marker, CurrentIndex);
                  end;
               end;
            end;
         end else begin
            // smoothing groups are to be considered
            for Face:=0 to NFaces-1 do with FaceArray[Face], Mesh^ do begin
               // normal vector for the face
               Vector1:=VectorSubstract(Vertices[V1], Vertices[V2]);
               Vector2:=VectorSubstract(Vertices[V3], Vertices[V2]);
               if DatabaseRelease>=rlRelease3 then
                  Normal:=VectorCrossProduct(Vector1, Vector2)
               else Normal:=VectorCrossProduct(Vector2, Vector1);
               SmoothingGroup:=SmoothArray[Face];
               // go for each vertex in the current face
               for Vertex:=0 to 2 do begin
                  // copy current index for faster access
                  currentIndex:=FaceRec[Vertex];
                  // Has vertex already been touched?
                  if IsVertexMarked(Marker, currentIndex) then begin
                     // check smoothing group
                     if SmoothingGroup = 0 then begin
                        // no smoothing then just duplicate vertex
                        DuplicateVertex(CurrentIndex);
                        FaceRec[Vertex] := CurrentVertexCount - 1;
                        Normals[CurrentVertexCount - 1] := Normal;
                        // mark new vertex also as touched
                        MarkVertex(Marker, CurrentVertexCount - 1);
                     end else begin
                        // this vertex must be smoothed, check if there's already
                        // a (duplicated) vertex for this smoothing group
                        TargetVertex := GetSmoothIndex(CurrentIndex, SmoothingGroup, SmoothIndices);
                        if TargetVertex < 0 then begin
                           // vertex has not yet been duplicated for this smoothing
                           // group, so do it now
                           DuplicateVertex(CurrentIndex);
                           FaceRec[Vertex] := CurrentVertexCount - 1;
                           Normals[CurrentVertexCount - 1] := Normal;
                           StoreSmoothIndex(CurrentIndex, SmoothingGroup, CurrentVertexCount - 1, SmoothIndices);
                           StoreSmoothIndex(CurrentVertexCount - 1, SmoothingGroup, CurrentVertexCount - 1, SmoothIndices);
                           // mark new vertex also as touched
                           MarkVertex(Marker, CurrentVertexCount - 1);
                        end else begin
                           // vertex has already been duplicated,
                           // so just add normal vector to other vertex...
                           Normals[TargetVertex] := VectorAdd(Normals[TargetVertex], Normal);
                           // ...and tell which new vertex has to be used from now on
                           FaceRec[Vertex] := TargetVertex;
                        end;
                     end;
                  end else begin
                     // vertex not yet touched, so just store the normal
                     Normals[CurrentIndex] := Normal;
                     // initialize smooth indices for this vertex
                     FillChar(SmoothIndices[CurrentIndex], SizeOf(TSmoothIndexEntry), $FF);
                     if SmoothingGroup <> 0 then
                        StoreSmoothIndex(CurrentIndex, SmoothingGroup, CurrentIndex, SmoothIndices);
                     MarkVertex(Marker, CurrentIndex);
                  end;
               end;
            end;
         end;
         FreeMem(Marker);
         FreeMem(SmoothIndices);

         Mesh^.VertexCount:=CurrentVertexCount;

         // sum up all vertices to do auto centering
         Inc(TotalCount, CurrentVertexCount);
         with Mesh^ do
            for j:=0 to CurrentVertexCount-1 do
               center:=VectorAdd(center, Vertices[J]);

         // and normalize the Normals array
         with Mesh^ do for j:=0 to CurrentVertexCount-1 do
            NormalizeVector(Normals[j]);

         // now go for each material group
         Mesh.FaceGroups := TFaceGroups.Create;
         // if there's no face to material assignment then just copy the
         // face definitions and rely on the default texture of the scene object
         if NMats = 0 then begin
            New(FaceGroup);
            with FaceGroup^ do begin
               FaceProperties := TGLFaceProperties.Create(nil);
               InterLeavedArray:=nil;
               IndexCount:=3*NFaces;
               GetMem(Indices, IndexCount * SizeOf(Integer));
               // copy the face list
               for j:=0 to NFaces-1 do begin
                  Indices[3*J+0]:=FaceArray[J].V1;
                  Indices[3*J+1]:=FaceArray[J].V2;
                  Indices[3*J+2]:=FaceArray[J].V3;
               end;
            end;
            Mesh.FaceGroups.Add(FaceGroup);
         end else begin
            for iMaterial := 0 to NMats - 1 do begin
               New(FaceGroup);
               with FaceGroup^ do begin
                  FaceProperties := FacePropsFromMaterial(Materials, MatArray[iMaterial].Name);
                  InterLeavedArray:=nil;
                  IndexCount := 3 * MatArray[iMaterial].NFaces;
                  GetMem(Indices, IndexCount * SizeOf(Integer));
                  // copy all vertices belonging to the current face into our index array,
                  // there won't be redundant vertices since this would mean a face has more than one
                  // material
                  // NFaces is the one from FaceGroup
                  with MatArray[iMaterial] do for J := 0 to NFaces - 1 do begin
                     Indices[3 * J + 0] := FaceArray[FaceIndex[J]].V1;
                     Indices[3 * J + 1] := FaceArray[FaceIndex[J]].V2;
                     Indices[3 * J + 2] := FaceArray[FaceIndex[J]].V3;
                  end;
               end;
               Mesh.FaceGroups.Add(FaceGroup);
            end;
         end;
         if FOwner is TBaseMesh then
            TBaseMesh(FOwner).AddMeshObject(Mesh);
      end;
      if (TotalCount>0) and (FOwner is TBaseMesh) then begin
         ScaleVector(center, 1/TotalCount);
         for i:=0 to TBaseMesh(FOwner).FObjects.Count-1 do begin
            Mesh := TBaseMesh(FOwner).FObjects[I];
            with Mesh^ do for j:=0 to VertexCount-1 do
               SubstractVector(Vertices[J], center);
         end;
      end;
   finally
      Free;
   end;
end;

//----------------- TBaseMesh --------------------------------------------------

// Create
//
constructor TBaseMesh.Create(AOwner:TComponent);
begin
   inherited Create(AOwner);
   ObjectStyle:=ObjectStyle+[osDoesTemperWithColorsOrFaceWinding];
   FObjects := TList.Create;
end;

// Destroy
//
destructor TBaseMesh.Destroy;
begin
   Clear;
   FObjects.Free;
   inherited Destroy;
end;

// Clear
//
procedure TBaseMesh.Clear;
var
   Mesh : PMeshObject;
   I    : Integer;
begin
   for I := 0 to FObjects.Count - 1 do begin
      Mesh := FObjects[I];
      if assigned(Mesh.Vertices) then FreeMem(Mesh.Vertices);
      if assigned(Mesh.Normals) then FreeMem(Mesh.Normals);
      if assigned(Mesh.TexCoords) then FreeMem(Mesh.TexCoords);
      Mesh.FaceGroups.Free;
      Dispose(Mesh);
   end;
   FObjects.Clear;
end;

// LoadFromFile
//
procedure TBaseMesh.LoadFromFile(const Filename: String);
var
   Ext           : String;
   newVectorFile : TVectorFile;
   vectorFileClass : TVectorFileClass;
begin
   if FileName <> '' then begin
      Clear;
      Ext := ExtractFileExt(Filename);
      Delete(Ext, 1, 1);
      vectorFileClass := GetVectorFileFormats.FindExt(Ext);
      if not Assigned(vectorFileClass) then
         raise EInvalidVectorFile.CreateFmt(SUnknownExtension, [Ext]);
      newVectorFile := VectorFileClass.Create(Self);
      try
         if Assigned(Scene) then Scene.BeginUpdate;
         newVectorFile.LoadFromFile(Filename);
         if Assigned(Scene) then Scene.EndUpdate;
      finally
         NewVectorFile.Free;
      end;
   end;
   StructureChanged;
end;

// LoadFromStream
//
procedure TBaseMesh.LoadFromStream(const filename : String; aStream : TStream);
var
   Ext           : String;
   newVectorFile : TVectorFile;
   vectorFileClass : TVectorFileClass;
begin
   if FileName <> '' then begin
      Clear;
      Ext := ExtractFileExt(Filename);
      Delete(Ext, 1, 1);
      vectorFileClass := GetVectorFileFormats.FindExt(Ext);
      if not Assigned(vectorFileClass) then
         raise EInvalidVectorFile.CreateFmt(SUnknownExtension, [Ext]);
      newVectorFile := VectorFileClass.Create(Self);
      try
         if Assigned(Scene) then Scene.BeginUpdate;
         newVectorFile.LoadFromStream(aStream);
         if Assigned(Scene) then Scene.EndUpdate;
      finally
         NewVectorFile.Free;
      end;
   end;
   StructureChanged;
end;

// AddMeshObject
//
function TBaseMesh.AddMeshObject(AObject: PMeshObject): Integer;
begin
  Result := FObjects.Add(AObject);
end;


// ------------------
// ------------------ TFreeForm ------------------
// ------------------

// Create
//
constructor TFreeForm.Create(AOwner:TComponent);
begin
   inherited Create(AOwner);
   ObjectStyle:=ObjectStyle+[osDoesTemperWithColorsOrFaceWinding];
   FUseMeshMaterials:=True;
   FMode:=ffmIndexedArrays;
end;

// Destroy
//
destructor TFreeForm.Destroy;
begin
   inherited Destroy;
end;

// SetUseMeshMaterials
//
procedure TFreeForm.SetUseMeshMaterials(const val : Boolean);
begin
   if val<>FUseMeshMaterials then begin
      FUseMeshMaterials:=val;
      StructureChanged;
   end;
end;

// SetNormalsOrientation
//
procedure TFreeForm.SetNormalsOrientation(const val : TMeshNormalsOrientation);
begin
   if val<>FNormalsOrientation then begin
      FNormalsOrientation:=val;
      { TODO rewrite of TVectorFile and proper use in FreeForm }
   end;
end;

// BuildList
//
procedure TFreeForm.BuildList;
var
   I, J, k, n, idx, p : Integer;
   Mesh : PMeshObject;
   vm : TVertexMode;
begin
   inherited BuildList;
   ResetGLPolygonMode;
   ResetGLMaterialColors;

   // move scene so that its geometric center is at origin
   glPushMatrix;

   // set winding
   glPushAttrib(GL_POLYGON_BIT);

   case FNormalsOrientation of
      mnoDefault : glFrontFace(GL_CW);
      mnoInvert : glFrontFace(GL_CCW);
   end;

   case Mode of
      ffmInterleavedArrays : begin
         // go through all loaded meshs
         for I := 0 to FObjects.Count-1 do begin
            Mesh := FObjects[I];
            if Assigned(Mesh.Normals) then begin
               n:=6; vm:=vmVN;
            end else begin
               n:=3; vm:=vmV;
            end;
            if Assigned(Mesh.TexCoords) then begin
               Inc(n, 2);
               if vm=vmV then vm:=vmVT else vm:=vmVNT;
            end;
            for j:=0 to Mesh.FaceGroups.Count - 1 do with Mesh.FaceGroups[J]^ do begin
               // apply material
               if FUseMeshMaterials then
                  FaceProperties.Apply(GL_FRONT);
               // (re)build interleaved array
               if Assigned(InterLeavedArray) then
                  ReallocMem(InterLeavedArray, n * Integer(IndexCount) * SizeOf(TGLFloat))
               else GetMem(InterLeavedArray, n * Integer(IndexCount) * SizeOf(TGLFloat));
               case vm of
                  vmVN : for k:=0 to IndexCount-1 do begin
                     idx:=Indices[k];  p:=k*n;
                     System.Move(Mesh.Normals[idx][0],   InterleavedArray[p+0], 3*SizeOf(TGLFloat));
                     System.Move(Mesh.Vertices[idx][0],  InterleavedArray[p+3], 3*SizeOf(TGLFloat));
                  end;
                  vmVT : for k:=0 to IndexCount-1 do begin
                     idx:=Indices[k];  p:=k*n;
                     System.Move(Mesh.TexCoords[idx],    InterleavedArray[p+0], 2*SizeOf(TGLFloat));
                     System.Move(Mesh.Vertices[idx][0],  InterleavedArray[p+2], 3*SizeOf(TGLFloat));
                  end;
                  vmVNT : for k:=0 to IndexCount-1 do begin
                     idx:=Indices[k];  p:=k*n;
                     System.Move(Mesh.TexCoords[idx],    InterleavedArray[p+0], 2*SizeOf(TGLFloat));
                     System.Move(Mesh.Normals[idx][0],   InterleavedArray[p+2], 3*SizeOf(TGLFloat));
                     System.Move(Mesh.Vertices[idx][0],  InterleavedArray[p+5], 3*SizeOf(TGLFloat));
                  end;
               else
                  vm:=vmV;
                  System.Move(Mesh.Vertices[0][0], InterLeavedArray[0],
                              3*Integer(IndexCount)*SizeOf(TGLFloat));
               end;
               glInterleavedArrays(cVertexModeToGLEnum[vm], 0, InterLeavedArray);
               glDrawArrays(cMeshModeToGLEnum[Mesh.Mode], 0, IndexCount);
            end;
         end;
      end;
      ffmIndexedArrays : begin
         glEnableClientState(GL_VERTEX_ARRAY);
         // go through all loaded meshs
         for I := 0 to FObjects.Count-1 do begin
            Mesh := FObjects[I];
            glVertexPointer(3, GL_FLOAT, 0, Mesh.Vertices);
            // enable the normal array if available
            if Assigned(Mesh.Normals) then begin
               glNormalPointer(GL_FLOAT, 0, Mesh.Normals);
               glEnableClientState(GL_NORMAL_ARRAY);
            end else glDisableCLientState(GL_NORMAL_ARRAY);
            // enable the texture coordinates array if available
            if Assigned(Mesh.TexCoords) then begin
               glTexCoordPointer(2, GL_FLOAT, 0, Mesh.TexCoords);
               glEnableClientState(GL_TEXTURE_COORD_ARRAY);
            end else glDisableCLientState(GL_TEXTURE_COORD_ARRAY);
            for j:=0 to Mesh.FaceGroups.Count - 1 do with Mesh.FaceGroups[J]^ do begin
               // apply material
               if FUseMeshMaterials then
                  FaceProperties.Apply(GL_FRONT);
               glDrawElements(cMeshModeToGLEnum[Mesh.Mode], IndexCount, GL_UNSIGNED_INT, Indices);
            end;
         end;
      end;
   else
      Assert(False);
   end;
   glPopAttrib;
   glPopMatrix;
end;

// ------------------
// ------------------ TActorAnimation ------------------
// ------------------

// Create
//
constructor TActorAnimation.Create(Collection : TCollection);
begin
	inherited Create(Collection);
end;

destructor TActorAnimation.Destroy;
begin
	inherited Destroy;
end;

procedure TActorAnimation.Assign(Source: TPersistent);
begin
	if Source is TActorAnimation then begin
      FName:=TActorAnimation(Source).FName;
      FStartFrame:=TActorAnimation(Source).FStartFrame;
      FEndFrame:=TActorAnimation(Source).FEndFrame;
	end;
	inherited Destroy;
end;

// GetDisplayName
//
function TActorAnimation.GetDisplayName : String;
begin
	Result:=Format('%d - %s [%d - %d]', [Index, Name, StartFrame, EndFrame]);
end;

// SetStartFrame
//
procedure TActorAnimation.SetStartFrame(const val : Integer);
begin
   if val<0 then
      FStartFrame:=0
   else if val>=TActorAnimations(Collection).Owner.FrameCount then
      FStartFrame:=TActorAnimations(Collection).Owner.FrameCount-1;
   if FStartFrame>FEndFrame then
      FEndFrame:=FStartFrame;
end;

// SetEndFrame
//
procedure TActorAnimation.SetEndFrame(const val : Integer);
begin
   if val<0 then
      FEndFrame:=0
   else if val>=TActorAnimations(Collection).Owner.FrameCount then
      FEndFrame:=TActorAnimations(Collection).Owner.FrameCount-1;
   if FStartFrame>FEndFrame then
      FStartFrame:=FEndFrame;
end;

// SetAsString
//
procedure TActorAnimation.SetAsString(const val : String);
var
   sl : TStringList;
begin
   sl:=TStringList.Create;
   try
      sl.CommaText:=val;
      Assert(sl.Count=3);
      FName:=sl[0];
      FStartFrame:=StrToInt(sl[1]);
      FEndFrame:=StrToInt(sl[2]);
   finally
      sl.Free;
   end;
end;

// GetAsString
//
function TActorAnimation.GetAsString : String;
begin
   Result:=Format('%s,%d,%d', [FName, FStartFrame, FEndFrame]);
end;

// ------------------
// ------------------ TActorAnimations ------------------
// ------------------

// Create
//
constructor TActorAnimations.Create(AOwner : TActor);
begin
	Owner:=AOwner;
	inherited Create(TActorAnimation);
end;

// GetOwner
//
function TActorAnimations.GetOwner: TPersistent;
begin
	Result:=Owner;
end;

// SetItems
//
procedure TActorAnimations.SetItems(index : Integer; const val : TActorAnimation);
begin
	inherited Items[index]:=val;
end;

// GetItems
//
function TActorAnimations.GetItems(index : Integer) : TActorAnimation;
begin
	Result:=TActorAnimation(inherited Items[index]);
end;

// Add
//
function TActorAnimations.Add: TActorAnimation;
begin
	Result:=(inherited Add) as TActorAnimation;
end;

// FindItemID
//
function TActorAnimations.FindItemID(ID: Integer): TActorAnimation;
begin
	Result:=(inherited FindItemID(ID)) as TActorAnimation;
end;

// FindName
//
function TActorAnimations.FindName(const aName : String) : TActorAnimation;
var
   i : Integer;
begin
	Result:=nil;
   for i:=0 to Count-1 do if CompareText(Items[i].Name, aName)=0 then begin
      Result:=Items[i];
      Break;
   end;
end;

// SetToStrings
//
procedure TActorAnimations.SetToStrings(aStrings : TStrings);
var
   i : Integer;
begin
   with aStrings do begin
      BeginUpdate;
      Clear;
      for i:=0 to Self.Count-1 do
         Add(Self.Items[i].Name);
      EndUpdate;
   end;
end;

// SaveToStream
//
procedure TActorAnimations.SaveToStream(aStream : TStream);
var
   i : Integer;
begin
   WriteCRLFString(aStream, cAAFHeader);
   WriteCRLFString(aStream, IntToStr(Count));
   for i:=0 to Count-1 do
      WriteCRLFString(aStream, Items[i].AsString);
end;

// LoadFromStream
//
procedure TActorAnimations.LoadFromStream(aStream : TStream);
var
   i, n : Integer;
begin
   Clear;
   Assert(ReadCRLFString(aStream)=cAAFHeader);
   n:=StrToInt(ReadCRLFString(aStream));
   for i:=0 to n-1 do
      Add.AsString:=ReadCRLFString(aStream);
end;

// SaveToFile
//
procedure TActorAnimations.SaveToFile(const fileName : String);
var
   fs : TFileStream;
begin
   fs:=TFileStream.Create(fileName, fmCreate);
   try
      SaveToStream(fs);
   finally
      fs.Free;
   end;
end;

// LoadFromFile
//
procedure TActorAnimations.LoadFromFile(const fileName : String);
var
   fs : TFileStream;
begin
   fs:=TFileStream.Create(fileName, fmOpenRead+fmShareDenyWrite);
   try
      LoadFromStream(fs);
   finally
      fs.Free;
   end;
end;

// ------------------
// ------------------ TActor ------------------
// ------------------

// Create
//
constructor TActor.Create(AOwner:TComponent);
begin
   inherited Create(AOwner);
   ObjectStyle:=ObjectStyle+[osDirectDraw];
   FFrameInterpolation:=afpLinear;
   FAnimationMode:=aamNone;
   FInterval:=100; // 10 animation frames per second
   FAnimations:=TActorAnimations.Create(Self);
end;

// Destroy
//
destructor TActor.Destroy;
begin
   FAnimations.Free;
   inherited Destroy;
end;

// SetCurrentFrame
//
procedure TActor.SetCurrentFrame(Value: Integer);
begin
   if Value<>CurrentFrame then begin
      if Value>FrameCount-1 then
         FCurrentFrame:=FrameCount-1
      else if Value<0 then
         FCurrentFrame:=0
      else FCurrentFrame:=Value;
      FCurrentFrameDelta:=0;
      case AnimationMode of
         aamPlayOnce :
            if CurrentFrame=EndFrame then FAnimationMode:=aamNone;
         aamBounceForward :
            if CurrentFrame=EndFrame then FAnimationMode:=aamBounceBackward;
         aamBounceBackward :
            if CurrentFrame=StartFrame then FAnimationMode:=aamBounceForward;
      end;
      StructureChanged;
      if Assigned(FOnFrameChanged) then FOnFrameChanged(Self);
   end;
end;

// SetStartFrame
//
procedure TActor.SetStartFrame(Value: Integer);
begin
   if (Value>=0) and (Value<FrameCount) and (Value<>StartFrame) then
      FStartFrame := Value;
   if EndFrame<StartFrame then
      FEndFrame := FStartFrame;
   if CurrentFrame<StartFrame then
      CurrentFrame := FStartFrame;
end;

// SetEndFrame
//
procedure TActor.SetEndFrame(Value: Integer);
begin
   if (Value>=0) and (Value<FrameCount) and (Value<>EndFrame) then
      FEndFrame := Value;
   if CurrentFrame>EndFrame then
      CurrentFrame := FEndFrame;
end;

// SetAnimations
//
procedure TActor.SetAnimations(const val : TActorAnimations);
begin
   FAnimations.Assign(val);
end;

// NextFrameIndex
//
function TActor.NextFrameIndex : Integer;
begin
   case AnimationMode of
      aamNone, aamPlayOnce, aamLoop, aamBounceForward : begin
         Result:=CurrentFrame+1;
         if Result>EndFrame then begin
            Result:=StartFrame+(Result-EndFrame-1);
            if Result>EndFrame then
               Result:=EndFrame;
         end;
      end;
      aamBounceBackward : begin
         Result:=CurrentFrame-1;
         if Result<StartFrame then begin
            Result:=EndFrame-(StartFrame-Result-1);
            if Result<StartFrame then
               Result:=StartFrame;
         end;
      end;
   else
      Result:=CurrentFrame;
      Assert(False);
   end;
end;

// NextFrame
//
procedure TActor.NextFrame(n : Integer = 1);
begin
   while n>0 do begin
      CurrentFrame:=NextFrameIndex;
      Dec(n);
      if Assigned(FOnEndFrameReached) and (CurrentFrame=EndFrame) then
         FOnEndFrameReached(Self);
      if Assigned(FOnStartFrameReached) and (CurrentFrame=StartFrame) then
         FOnStartFrameReached(Self);
   end;
end;

// PrevFrame
//
procedure TActor.PrevFrame(n : Integer = 1);
var
   value: Integer;
begin
   value:=FCurrentFrame-N;
   if value<FStartFrame then begin
      Value := FEndFrame-(FStartFrame - Value);
      if Value<FStartFrame then
         Value:=FStartFrame;
   end;
   CurrentFrame:=Value;
end;

// BuildList
//
procedure TActor.BuildList;
var
   i, k : Integer;
   mesh1, mesh2 : PMeshObject;
   faceGroup : PFaceGroup;
   vertexPool : PAffineVectorArray;
   lerpFactor : Single;
begin
   inherited BuildList;
   if FObjects.Count=0 then Exit;
   glPushAttrib(GL_ENABLE_BIT);
   glDisable(GL_NORMALIZE);
   case FrameInterpolation of
      afpLinear : begin // Linear frame interpolation
         mesh1:=FObjects[CurrentFrame];
         mesh2:=FObjects[NextFrameIndex];
         Assert((mesh1.VertexCount=mesh2.VertexCount));
         lerpFactor:=CurrentFrameDelta;
         // alloc a vertex pool
         vertexPool:=AllocMem(mesh1.VertexCount*SizeOf(TAffineVector));
         // interpolate vertices
         for i:=0 to mesh1.VertexCount-1 do begin
            VectorLerp(mesh1.Vertices[i], mesh2.Vertices[i], lerpFactor, vertexPool[i]);
         end;
      end;
   else
      // afpNone
      mesh1:=PMeshObject(FObjects[FCurrentFrame]);
      vertexPool:=mesh1.Vertices;
   end;
   // draw
   glBegin(GL_TRIANGLES);
   with mesh1^ do begin
      faceGroup := FaceGroups[0];
      k:=0; while k<Integer(faceGroup.IndexCount) do begin
         glTexCoord2fv(@TexCoords[k+0]);
         glVertex3fv(@vertexPool[faceGroup.Indices[k+0]]);
         glTexCoord2fv(@TexCoords[k+1]);
         glVertex3fv(@vertexPool[faceGroup.Indices[k+1]]);
         glTexCoord2fv(@TexCoords[k+2]);
         glVertex3fv(@vertexPool[faceGroup.Indices[k+2]]);
         Inc(k, 3);
      end;
   end;
   glEnd;
   if FrameInterpolation<>afpNone then begin
      FreeMem(vertexPool);
   end;
   glPopAttrib;
end;

// LoadFromFile
//
procedure TActor.LoadFromFile(const Filename: String);
begin
   inherited LoadFromFile(fileName);
   FFrameCount := FObjects.Count;
   FStartFrame := 0;
   FEndFrame := FFrameCount-1;
   FCurrentFrame := 0;
   if Assigned(FOnFrameChanged) then FOnFrameChanged(Self);
end;

// DoProgress
//
procedure TActor.DoProgress(const deltaTime, newTime : Double);
var
   fDelta : Single;
begin
   inherited;
   if (AnimationMode<>aamNone) and (StartFrame<>EndFrame) and (FrameCount>1) and (Interval>0) then begin
      FCurrentFrameDelta:=FCurrentFrameDelta+(deltaTime*1000)/FInterval;
      if FCurrentFrameDelta>1 then begin
         // we need to step on
         fDelta:=Frac(FCurrentFrameDelta);
         NextFrame(Trunc(FCurrentFrameDelta));
         FCurrentFrameDelta:=fDelta;
      end else if FrameInterpolation<>afpNone then
         StructureChanged;
   end;
end;

// SwitchToAnimation
//
procedure TActor.SwitchToAnimation(const animationName : String);
begin
   SwitchToAnimation(Animations.FindName(animationName));
end;

// SwitchToAnimation
//
procedure TActor.SwitchToAnimation(animationIndex : Integer);
begin
   if (animationIndex>=0) and (animationIndex<Animations.Count) then
      SwitchToAnimation(Animations[animationIndex]);
end;

// SwitchToAnimation
//
procedure TActor.SwitchToAnimation(anAnimation : TActorAnimation);
begin
   if Assigned(anAnimation) then begin
      StartFrame:=anAnimation.StartFrame;
      EndFrame:=anAnimation.EndFrame;
      CurrentFrame:=StartFrame;
   end;
end;

// Synchronize
//
procedure TActor.Synchronize(referenceActor : TActor);
begin
   if Assigned(referenceActor) then begin
      FStartFrame:=referenceActor.StartFrame;
      FEndFrame:=referenceActor.EndFrame;
      CurrentFrame:=referenceActor.CurrentFrame;
      CurrentFrameDelta:=referenceActor.CurrentFrameDelta;
   end;
end;

// ------------------
// ------------------ TGLMD2VectorFile ------------------
// ------------------

// ConvertMD2Structure
//
function TGLMD2VectorFile.ConvertMD2Structure(MD2File: TFileMD2; frame: Integer): PMeshObject;
var
  I: Integer;
  Mesh: PMeshObject;
  FaceGroup: PFaceGroup;
begin
  New(Mesh);
  FillChar(Mesh^, SizeOf(Mesh^), 0);
  Mesh^.Mode := mmTriangles;
  with Mesh^, MD2File do
  begin
    GetMem(Vertices, m_iVertices * sizeof(TAffineVector));
    GetMem(TexCoords, 3* m_iTriangles * sizeof(TTexPoint));
    Mesh.VertexCount:=m_iVertices;

    FaceGroups := TFaceGroups.Create;
    with FaceGroups do begin
      New(FaceGroup);
      with FaceGroup^ do begin
        FaceProperties := TGLFaceProperties.Create(nil);
        InterLeavedArray:=nil;
        IndexCount := 3 * m_iTriangles;
        GetMem(Indices, IndexCount * SizeOf(Integer));
        // copy the face list
        for I := 0 to m_iTriangles - 1 do begin
          TexCoords[I*3].S := IndexList(m_index_list)[i].a_s;
          TexCoords[I*3].t := -IndexList(m_index_list)[i].a_t;
          Indices[I*3] := IndexList(m_index_list)[i].a;

          TexCoords[I*3+1].S := IndexList(m_index_list)[i].b_s;
          TexCoords[I*3+1].t := -IndexList(m_index_list)[i].b_t;
          Indices[I*3+1] := IndexList(m_index_list)[i].b;

          TexCoords[I*3+2].S := IndexList(m_index_list)[i].c_s;
          TexCoords[I*3+2].t := -IndexList(m_index_list)[i].c_t;
          Indices[I*3+2] := IndexList(m_index_list)[i].c;
        end;
      end;
      Add(FaceGroup);
    end;

    for I:=0 to m_iVertices-1 do
    begin
      Vertices[I][0] := VertList(frameList(m_frame_list)[frame].vertex)[I].x;
      Vertices[I][1] := VertList(frameList(m_frame_list)[frame].vertex)[I].y;
      Vertices[I][2] := VertList(frameList(m_frame_list)[frame].vertex)[I].z;
    end;
  end;
  Result := Mesh;
end;

// LoadFromStream
//
procedure TGLMD2VectorFile.LoadFromStream(aStream : TStream);
var
   i : Integer;
   MD2File : TFileMD2;
   Mesh : PMeshObject;
begin
   MD2File:=TFileMD2.Create;
   MD2File.LoadFromStream(aStream);
   try
      for I:=0 to MD2File.m_iFrames-1 do begin
         Mesh := ConvertMD2Structure(MD2File, I);
         TBaseMesh(FOwner).AddMeshObject(Mesh);
      end;
      if FOwner is TActor then with TActor(FOwner).Animations do begin
         Clear;
         with MD2File do for i:=0 to frameNames.Count-1 do with Add do begin
            Name:=frameNames[i];
            FStartFrame:=Integer(frameNames.Objects[i]);
            if i<frameNames.Count-1 then
               FEndFrame:=Integer(frameNames.Objects[i+1])-1
            else FEndFrame:=m_iFrames-1;
         end;
      end;
   finally
      MD2File.Free;
   end;
end;

// ------------------------------------------------------------------
// ------------------------------------------------------------------
// ------------------------------------------------------------------
initialization
// ------------------------------------------------------------------
// ------------------------------------------------------------------
// ------------------------------------------------------------------

   RegisterVectorFileFormat('md2', 'Quake II model files', TGLMD2VectorFile);
   RegisterVectorFileFormat('3ds', '3D Studio files', TGL3DSVectorFile);
   RegisterVectorFileFormat('prj', '3D Studio project files', TGL3DSVectorFile);
   RegisterClasses([TFreeForm, TActor, TVectorFile]);

finalization

   vVectorFileFormats.Free;

end.

