{: GLScene<p>

	Base classes and structures for GLScene.<p>

	<b>Historique : </b><font size=-1><ul>
      <li>28/06/00 - Egg - Added ObjectStyle to TGLBaseSceneObject, various
                           changes to the list/handle mechanism
      <li>22/06/00 - Egg - Added TLightStyle (suggestion by Roger Cao)
      <li>19/06/00 - Egg - Optimized SetXxxAngle
      <li>09/06/00 - Egg - First row of Geometry-related upgrades
      <li>07/06/00 - Egg - Removed dependency to 'Math',
                           RenderToFile <-> Bitmap Overload (Aaron Hochwimmer)
      <li>28/05/00 - Egg - AxesBuildList now available as a procedure,
                           Un-re-fixed TGLLightSource.DestroyList,
                           Fixed RenderToBitmap
      <li>26/05/00 - Egg - Slightly changed DrawAxes to avoid a bug in nVidia OpenGL
      <li>23/05/00 - Egg - Added first set of collision-detection methods
      <li>22/05/00 - Egg - SetXxxxAngle now properly assigns to FXxxxAngle
      <li>08/05/00 - Egg - Added Absolute?Vector funcs to TGLBaseSceneObject
      <li>08/05/00 - RoC - Fixes in TGLScene.LoadFromFile, TGLLightSource.DestroyList
      <li>26/04/00 - Egg - TagFloat now available in TGLBaseSceneObject,
									Added TGLProxyObject
      <li>18/04/00 - Egg - Added TGLObjectEffect structures,
                           TGLCoordinates.CreateInitialized
		<li>17/04/00 - Egg - Fixed BaseSceneObject.Assign (wasn't duping children),
									Removed CreateSceneObject,
                           Optimized TGLSceneViewer.Invalidate
      <li>16/04/00 - Egg - Splitted Render to Render + RenderChildren
		<li>11/04/00 - Egg - Added TGLBaseSceneObject.SetScene (thanks Uwe)
									and fixed various funcs accordingly
      <li>10/04/00 - Egg - Improved persistence logic for behaviours,
									Added RegisterGLBehaviourNameChangeEvent
		<li>06/04/00 - Egg - RebuildMatrix should be slightly faster now
		<li>05/04/00 - Egg - Added TGLBehaviour stuff,
									Angles are now public stuff in TGLBaseSceneObject
      <li>26/03/00 - Egg - Added TagFloat to TGLCustomSceneObject,
                           Parent is now longer copied in "Assign"
		<li>22/03/00 - Egg - TGLStates moved to GLMisc,
                           Removed TGLCamera.FModified stuff,
                           Fixed position bug in TGLScene.SetupLights
		<li>20/03/00 - Egg - PickObjects now uses "const" and has helper funcs,
                           Dissolved TGLRenderOptions into material and face props (RIP),
                           Joystick stuff moved to a separate unit and component
      <li>19/03/00 - Egg - Added DoProgress method and event
		<li>18/03/00 - Egg - Fixed a few "Assign" I forgot to update after adding props,
									Added bmAdditive blending mode
		<li>14/03/00 - Egg - Added RegisterGLBaseSceneObjectNameChangeEvent,
									Added BarycenterXxx and SqrDistance funcs,
									Fixed (?) AbsolutePosition,
                           Added ResetPerformanceMonitor
		<li>14/03/00 - Egg - Added SaveToFile, LoadFromFile to GLScene,
		<li>03/03/00 - Egg - Disabled woTransparent handling
		<li>12/02/00 - Egg - Added Material Library
		<li>10/02/00 - Egg - Added Initialize to TGLCoordinates
      <li>09/02/00 - Egg - All GLScene objects now begin with 'TGL',
                           OpenGL now initialized upon first create of a TGLSceneViewer
		<li>07/02/00 - Egg - Added ImmaterialSceneObject,
									Added Camera handling funcs : MoveAroundTarget,
									AdjustDistanceToTarget, DistanceToTarget,
									ScreenDeltaToVector, TGLCoordinates.Translate,
                           Deactivated "specials" (ain't working yet),
                           Scaling now a TGLCoordinates
      <li>06/02/00 - Egg - balanced & secured all context activations,
									added Assert & try..finally & default galore,
                           OpenGLError renamed to EOpenGLError,
                           ShowErrorXxx funcs renamed to RaiseOpenGLError,
                           fixed CreateSceneObject (was wrongly requiring a TCustomForm),
                           fixed DoJoystickCapture error handling,
                           added TGLUpdateAbleObject
		<li>05/02/00 - Egg - Javadocisation, fixes and enhancements :<br>
                           TGLSceneViewer.SetContextOptions,
									TActiveMode -> TJoystickDesignMode,
                           TGLCamera.TargetObject and TGLCamera.AutoLeveling,
                           TGLBaseSceneObject.CoordinateChanged
	</ul></font>
}
unit GLScene;
// TGLScene    - An encapsulation of the OpenGL API
// Version     - 0.5.8
// 30-DEC-99 ml: adjustments for Delphi 5

interface

{$R-}

{$I DFS.INC}

uses Windows, Classes, Controls, GLScreen, GLMisc, GLTexture, SysUtils,
	Graphics, Messages, OpenGL12, Geometry, XCollection;

type

   // TGLProxyObjectOption
   //
   {: Defines which features are taken from the master object. }
   TGLProxyObjectOption = (pooEffects, pooBuildList, pooChildren);
   TGLProxyObjectOptions = set of TGLProxyObjectOption;

const
   cDefaultProxyOptions = [pooEffects, pooBuildList, pooChildren];

type
  TObjectHandle = TGLUInt; // a display list name or GL_LIGHTx constant

  TNormalDirection = (ndInside, ndOutside);
  TTransformationMode = (tmLocal, tmParentNoPos, tmParentWithPos);

  // used to decribe only the changes in an object, which have to be reflected in the scene
  TObjectChange = (ocSpot, ocAttenuation, ocTransformation, ocStructure);
  TObjectChanges = set of TObjectChange;

  // flags for design notification
  TSceneOperation = (soAdd, soRemove, soMove, soRename, soSelect, soBeginUpdate, soEndUpdate);

  // flags for allocated buffers
  TBuffer = (buColor, buDepth, buStencil, buAccum);
  TBuffers = set of TBuffer;

  // options for the rendering context
  TContextOption = (roDoubleBuffer, roRenderToWindow, roTwoSideLighting);
  TContextOptions = set of TContextOption;

  // IDs for limit determination
  TLimitType = (limClipPlanes, limEvalOrder, limLights, limListNesting,
                limModelViewStack, limNameStack, limPixelMapTable, limProjectionStack,
                limTextureSize, limTextureStack, limViewportDims, limAccumAlphaBits,
                limAccumBlueBits, limAccumGreenBits, limAccumRedBits, limAlphaBits,
                limAuxBuffers, limBlueBits, limGreenBits, limRedBits, limIndexBits,
                limStereo, limDoubleBuffer, limSubpixelBits, limDepthBits, limStencilBits);

	TGLBaseSceneObject = class;
	TGLSceneObjectClass = class of TGLBaseSceneObject;
	TGLCustomSceneObject = class;
	TGLScene = class;
	TGLBehaviours = class;
   TGLObjectEffects = class;
   TGLSceneViewer = class;

	// TGLCoordinates
	//
	{: Stores and homogenous vector.<p>
		This class is basicly a container for a TVector, allowing proper use of
		delphi property editors and editing in the IDE. Vector/Coordinates
		manipulation methods are only minimal.<br>
		Handles dynamic default values to save resource file space.<p>
		Note : only affine components are published. }
	TGLCoordinates = class(TGLUpdateAbleObject)
		private
			{ Private Declarations }
			FCoords, FDefaultCoords: TVector;
			procedure SetAsVector(const value: TVector);
			procedure SetCoordinate(Index: Integer; AValue: TGLFloat);

		protected
			{ Protected Declarations }
			procedure DefineProperties(Filer: TFiler); override;
			procedure ReadData(Stream: TStream);
			procedure WriteData(Stream: TStream);

		public
			{ Public Declarations }
         constructor CreateInitialized(aOwner : TPersistent; const value : TVector);

			procedure Assign(Source: TPersistent); override;
         procedure WriteToFiler(writer : TWriter);
         procedure ReadFromFiler(reader : TReader);

         procedure Initialize(const value : TVector);
			procedure NotifyChange; override;

			procedure Translate(const translationVector : TVector);
			procedure AddScaledVector(const factor : Single; const translationVector : TVector);
         procedure Invert;
         function VectorLength : TGLFloat;
         procedure SetVector(const x, y, z, w : Single);
         procedure SetPoint(const x, y, z : Single);

         function AsAddress : PGLFloat;

			property AsVector: TVector read FCoords write SetAsVector;
			property W: TGLFloat index 3 read FCoords[3] write SetCoordinate;

		published
			{ Published Declarations }
			property X: TGLFloat index 0 read FCoords[0] write SetCoordinate stored False;
			property Y: TGLFloat index 1 read FCoords[1] write SetCoordinate stored False;
			property Z: TGLFloat index 2 read FCoords[2] write SetCoordinate stored False;

  	end;

   // TGLObjectsSorting
   //
   {: Determines if objects are sorted, and how.<p>
      Sorting is done level by level (and not for all entities), values are :<br>
      osInherited : use inherited sorting mode, defaults to osRenderFarthestFirst<br>
      osNone : do not sort objects.<br>
		osRenderFarthestFirst : render objects whose Position is the farthest from
			the camera first.<br>
      osRenderBlendedLast : opaque objects are not sorted and rendered
         first, blended ones are rendered afterwards and depth sorted. }
   TGLObjectsSorting = (osInherited, osNone, osRenderFarthestFirst, osRenderBlendedLast);

   // TGLObjectStyle
   //
   {: Possible styles/options for a GLScene object.<p>
   - osDirectDraw : object shall not make use of compiled call lists, but issue
      direct calls each time a render should be performed.<br>
   - osDoesTemperWithColorsOrFaceWinding : object is not "GLScene compatible" for
       color/face winding. "GLScene compatible" objects must use GLMisc functions
       for color or face winding setting (to avoid redundant OpenGL calls), for
       objects that don't comply, the internal cache must be flushed.
   }
   TGLObjectStyle = (osDirectDraw, osDoesTemperWithColorsOrFaceWinding);
   TGLObjectStyles = set of TGLObjectStyle;

	// TGLProgressEvent
	//
   {: Progression event for time-base animations/simulations.<p>
      deltaTime is the time delta since last progress and newTime is the new
      time after the progress event is completed. }
   TGLProgressEvent = procedure (Sender : TObject; const deltaTime, newTime : Double) of object;

	// TGLBaseSceneObject
   //
   {: Base class for all scene objects.<p> }
	TGLBaseSceneObject = class (TGLUpdateAbleComponent)
		private
			{ Private Declarations }
         FObjectStyle : TGLObjectStyles;
			FHandle: TObjectHandle;
         FPosition : TGLCoordinates;
         FDirection, FUp : TGLCoordinates;
         FScaling : TGLCoordinates;
			FChanges : TObjectChanges;
			FParent : TGLBaseSceneObject;
         FScene : TGLScene;
         FChildren : TList;
         FVisible : Boolean;
         FLocalMatrix, FGlobalMatrix: TMatrix;
         FMatrixDirty : Boolean;
         FUpdateCount : Integer;
			FShowAxes : Boolean;
         FRollAngle, FPitchAngle, FTurnAngle: Single; // absolute rotation in degrees
			FIsCalculating: Boolean;
         FTransMode: TTransformationMode;
			FObjectsSorting : TGLObjectsSorting;
			FOnProgress : TGLProgressEvent;
			FBehaviours : TGLBehaviours;
         FObjectEffects : TGLObjectEffects;
			FTagFloat: TGLFloat;

			function Get(Index: Integer): TGLBaseSceneObject;
			function GetCount: Integer;
         function GetIndex: Integer;
         function GetMatrix: TMatrix;
         procedure SetParent(const val : TGLBaseSceneObject);
         procedure SetIndex(AValue: Integer);
         procedure SetDirection(AVector: TGLCoordinates);
         procedure SetUp(AVector: TGLCoordinates);
			procedure SetMatrix(AValue: TMatrix);
         procedure SetPosition(APosition: TGLCoordinates);
         procedure SetPitchAngle(AValue: Single);
         procedure SetRollAngle(AValue: Single);
         procedure SetShowAxes(AValue: Boolean);
			procedure SetScaling(AValue: TGLCoordinates);
			procedure SetTurnAngle(AValue: Single);
			procedure SetVisible(AValue: Boolean);
			procedure SetObjectsSorting(const val : TGLObjectsSorting);
			procedure SetBehaviours(const val : TGLBehaviours);
			procedure SetEffects(const val : TGLObjectEffects);
			procedure SetScene(const value : TGLScene);

		protected
			{ Protected Declarations }
         procedure Loaded; override;
			procedure DefineProperties(Filer: TFiler); override;
			procedure WriteBehaviours(stream : TStream);
         procedure ReadBehaviours(stream : TStream);
			procedure DrawAxes(Pattern: TGLushort);
			procedure GetChildren(AProc: TGetChildProc; Root: TComponent); override;
			function GetHandle: TObjectHandle; virtual;
         //: Returns Up and Direction vectors depending on the transformation mode
			procedure GetOrientationVectors(var up, direction: TAffineVector);
			procedure RebuildMatrix;
			procedure SetName(const NewName: TComponentName); override;
			procedure SetParentComponent(Value: TComponent); override;
			//: Recalculate an orthonormal system
			procedure CoordinateChanged(Sender: TGLCoordinates); virtual;

         property ObjectStyle : TGLObjectStyles read FObjectStyle write FObjectStyle;

			property LocalMatrix : TMatrix read FLocalMatrix;
			property GlobalMatrix : TMatrix read FGlobalMatrix;

		public
			{ Public Declarations }
			constructor Create(AOwner: TComponent); override;
			destructor Destroy; override;
			procedure Assign(Source: TPersistent); override;

			procedure RestoreMatrix;
			{: Calculates the object's absolute matrix by composing all local matrices.<p>
            Multiplying a local coordinate with this matrix gives an absolute coordinate.<p>
				The current implem is probably buggy and slow... }
         function AbsoluteMatrix : TMatrix;
			{: Calculates the object's absolute inverse matrix.<p>
            Multiplying an absolute coordinate with this matrix gives a local coordinate.<p>
				The current implem uses transposition(AbsoluteMatrix), which is true
            unless you're using some scaling... }
         function InvAbsoluteMatrix : TMatrix;
			{: Calculates the object's absolute coordinates.<p>
				The current implem is probably buggy and slow... }
			function AbsolutePosition : TVector;
         {: Returns the Absolute X Vector expressed in local coordinates. }
         function AbsoluteXVector : TVector;
         {: Returns the Absolute Y Vector expressed in local coordinates. }
         function AbsoluteYVector : TVector;
         {: Returns the Absolute Z Vector expressed in local coordinates. }
         function AbsoluteZVector : TVector;

			{: Calculates the object's square distance to a point.<p>
				AbsolutePosition is considered as being the object position. }
			function SqrDistanceTo(const pt : TVector) : Single;
			{: Calculates the object's barycenter in absolute coordinates.<p>
				Default behaviour is to consider Barycenter=AbsolutePosition
				(whatever the number of children).<br>
				SubClasses where AbsolutePosition is not the barycenter should
				override this method as it is used for distance calculation, during
				rendering for instance, and may lead to visual inconsistencies. }
			function BarycenterAbsolutePosition : TVector; virtual;
			{: Calculates the object's barycenter distance to a point.<p> }
			function BarycenterSqrDistanceTo(const pt : TVector) : Single;

         {: Shall returns the object's axis aligned extensions.<p>
            The dimensions are measured from object center and are expressed
            <i>with</i> scale accounted for, in the object's coordinates
            (not in absolute coordinates).<p>
            Default value is the object's Scale. }
         function AxisAlignedDimensions : TVector; virtual;
         {: Indicates if a point is within an object.<p>
            Given coordinate is an absolute coordinate.<br>
            Linear or surfacic objects shall always return False.<p>
            Default value is based on AxisAlignedDimension and a cube bounding. }
         function PointInObject(const point : TVector) : Boolean; virtual;
         {: Request to determine an intersection with a casted ray.<p>
            Given coordinates & vector are in absolute coordinates.<bR>
            rayStart may be a point inside the object, allowing retrieval of
            the multiple intersects of the ray.<p>
            When intersectXXX parameters are nil (default) implementation should
            take advantage of this to optimize calculus, if not, and an intersect
            is found, non nil parameters should be defined.<p>
            Default value is based on AxisAlignedDimension and a cube bounding. }
         function RayCastIntersect(const rayStart, rayVector : TAffineVector;
                                   intersectPoint : PAffineVector = nil;
                                   intersectNormal : PAffineVector = nil) : Boolean; virtual;

			//: Create a new scene object and add it to this object as new child
			function AddNewChild(AChild: TGLSceneObjectClass): TGLBaseSceneObject; virtual;
         procedure AddChild(AChild: TGLBaseSceneObject); virtual;
			procedure DeleteChildren; dynamic;
			procedure Insert(AIndex: Integer; AChild: TGLBaseSceneObject); dynamic;
			{: Takes a scene object out of the child list, but doesn't destroy it.<p>
				If 'KeepChildren' is true its children will be kept as new children
				in this scene object. }
			procedure Remove(AChild: TGLBaseSceneObject; KeepChildren: Boolean); virtual;
			function IndexOfChild(AChild: TGLBaseSceneObject) : Integer;
         procedure MoveChildUp(anIndex : Integer);
         procedure MoveChildDown(anIndex : Integer);

			procedure DoProgress(const deltaTime, newTime : Double); override;
         procedure MoveTo(newParent : TGLBaseSceneObject); dynamic;
			procedure MoveUp;
			procedure MoveDown;
			procedure BeginUpdate; virtual;
			procedure EndUpdate; virtual;
			{: Make object-specific geometry description here.<p>
				Subclasses should MAINTAIN OpenGL states (restore the states if
				they were altered). }
			procedure BuildList; virtual;
			procedure DestroyList; virtual;
         procedure FinishObject; virtual;
			function GetParentComponent: TComponent; override;
         function HasParent: Boolean; override;
         function  IsUpdating: Boolean;
         procedure Lift(ADistance: Single);
         //: Moves object along the direction vector
			procedure Move(ADistance: Single);
         procedure Pitch(Angle: Single);
			procedure PrepareObject; virtual;
			procedure Render(objectsSorting : TGLObjectsSorting;
								  const cameraPosition : TVector;
								  var currentStates : TGLStates); virtual;
			procedure RenderChildren(firstChildIndex, lastChildIndex : Integer;
                          objectsSorting : TGLObjectsSorting;
								  const cameraPosition : TVector;
								  var currentStates : TGLStates); virtual;
			procedure Roll(Angle: Single);
         //: Moves camera along the right vector (move left and right)
         procedure Slide(ADistance: Single);
         procedure StructureChanged;
         procedure Translate(tx, ty, tz : TGLFloat); virtual;
			procedure TransformationChanged;
			procedure Turn(Angle: Single);
			procedure NotifyChange; override;
			//: Calculate own global matrix and let the children do the same with their's
			procedure ValidateTransformation; virtual;

         property PitchAngle: Single read FPitchAngle write SetPitchAngle;
         property RollAngle: Single read FRollAngle write SetRollAngle;
         property TransformationMode: TTransformationMode read FTransMode write FTransMode default tmLocal;
			property TurnAngle: Single read FTurnAngle write SetTurnAngle;

			property ShowAxes: Boolean read FShowAxes write SetShowAxes default False;
         property Changes: TObjectChanges read FChanges;
         property Children[Index: Integer]: TGLBaseSceneObject read Get; default;
         property Count: Integer read GetCount;
			property Handle: TObjectHandle read GetHandle;
         property Index: Integer read GetIndex write SetIndex;
			property Matrix: TMatrix  read GetMatrix write SetMatrix;
			property Parent: TGLBaseSceneObject read FParent write SetParent;
			property Position: TGLCoordinates read FPosition write SetPosition;
			property Direction: TGLCoordinates read FDirection write SetDirection;
			property Up: TGLCoordinates read FUp write SetUp;
			property Scale : TGLCoordinates read FScaling write SetScaling;
			property Scene : TGLScene read FScene;
			property Visible : Boolean read FVisible write SetVisible default True;
			property ObjectsSorting : TGLObjectsSorting read FObjectsSorting write SetObjectsSorting default osInherited;
			property OnProgress : TGLProgressEvent read FOnProgress write FOnProgress;
			property Behaviours : TGLBehaviours read FBehaviours write SetBehaviours stored False;
         property Effects : TGLObjectEffects read FObjectEffects write SetEffects stored False;

		published
			{ Published Declarations }
			property TagFloat: TGLFLoat read FTagFloat write FTagFloat;

	end;

	// TGLBaseBehaviour
	//
	{: Base class for implementing behaviours in TGLScene.<p>
		Behaviours are regrouped in a collection attached to a TGLBaseSceneObject,
		and are part of the "Progress" chain of events. Behaviours allows clean
		application of time-based alterations to objects (movements, shape or
		texture changes...).<p>
		Since behaviours are implemented as classes, there are basicly two kinds
		of strategies for subclasses :<ul>
		<li>stand-alone : the subclass does it all, and holds all necessary data
			(covers animation, inertia etc.)
		<li>proxy : the subclass is an interface to and external, shared operator
			(like gravity, force-field effects etc.)
		</ul><br>
		Some behaviours may be cooperative (like force-fields affects inertia)
		or unique (e.g. only one inertia behaviour per object).<p>
		NOTES :<ul>
		<li>Don't forget to override the ReadFromFiler/WriteToFiler persistence
			methods if you add data in a subclass !
		<li>Subclasses must be registered using the RegisterXCollectionItemClass
         function
		</ul> }
	TGLBaseBehaviour = class (TXCollectionItem)
		protected
			{ Protected Declarations }
         procedure SetName(const val : String); override;

         {: Override this function to write subclass data. }
         procedure WriteToFiler(writer : TWriter); override;
         {: Override this function to read subclass data. }
         procedure ReadFromFiler(reader : TReader); override;

			{: Returns the TGLBaseSceneObject on which the behaviour should be applied.<p>
				Does NOT check for nil owners. }
			function OwnerBaseSceneObject : TGLBaseSceneObject;

		public
			{ Public Declarations }
			constructor Create(aOwner : TXCollection); override;
			destructor Destroy; override;

			procedure DoProgress(const deltaTime, newTime : Double); virtual;
	end;

   // TGLBehaviour
   //
   {: Ancestor for non-rendering behaviours.<p>
		This class shall never receive any properties, it's just here to differentiate
      rendereing and non-rendering behaviours. Rendereing behaviours are named
      "TGLObjectEffect", non-rendering effects (like inertia) are simply named
      "TGLBehaviour". }
   TGLBehaviour = class (TGLBaseBehaviour)
   end;

	TGLBehaviourClass = class of TGLBehaviour;

	// TGLBehaviours
	//
	{: Holds a list of TGLBehaviour objects.<p>
		This object expects itself to be owned by a TGLBaseSceneObject.<p> }
	TGLBehaviours = class (TXCollection)
		protected
			{ Protected Declarations }
			function GetBehaviour(index : Integer) : TGLBehaviour;

		public
			{ Public Declarations }
			constructor Create(aOwner : TPersistent); override;

         class function ItemsClass : TXCollectionItemClass; override;

			property Behaviour[index : Integer] : TGLBehaviour read GetBehaviour; default;

			procedure DoProgress(const deltaTime, newTime : Double);
	end;

   // TGLObjectEffect
   //
   {: A rendering effect that can be applied to SceneObjects.<p>
		ObjectEffect is a subclass of behaviour that gets a chance to Render
      an object-related special effect.<p>
      TGLObjectEffect should not be used as base class for custom effects,
      instead you should use the following base classes :<ul>
      <li>TGLObjectPreEffect is rendered before owner object render
      <li>TGLObjectPostEffect is rendered after the owner object render
      <li>TGLObjectAfterEffect is rendered at the end of the scene rendering
      </ul><br>NOTES :<ul>
		<li>Don't forget to override the ReadFromFiler/WriteToFiler persistence
			methods if you add data in a subclass !
		<li>Subclasses must be registered using the RegisterXCollectionItemClass
         function
		</ul> }
   TGLObjectEffect = class (TGLBehaviour)
		protected
			{ Protected Declarations }
			{: Override this function to write subclass data. }
         procedure WriteToFiler(writer : TWriter); override;
         {: Override this function to read subclass data. }
         procedure ReadFromFiler(reader : TReader); override;

		public
			{ Public Declarations }
         procedure Render(sceneViewer : TGLSceneViewer;
								  const cameraPosition : TVector;
								  var currentStates : TGLStates); virtual;
   end;

   // TGLObjectPreEffect
   //
   {: An object effect that gets rendered before owner object's render.<p>
      The current OpenGL matrices and material are that of the owner object. }
	TGLObjectPreEffect = class (TGLObjectEffect)
   end;

   // TGLObjectPostEffect
   //
   {: An object effect that gets rendered after owner object's render.<p>
      The current OpenGL matrices and material are that of the owner object. }
   TGLObjectPostEffect = class (TGLObjectEffect)
   end;

   // TGLObjectAfterEffect
   //
   {: An object effect that gets rendered at scene's end.<p>
      No particular OpenGL matrices or material should be assumed. }
   TGLObjectAfterEffect = class (TGLObjectEffect)
   end;

	// TGLObjectEffects
	//
	{: Holds a list of object effects.<p>
		This object expects itself to be owned by a TGLBaseSceneObject.<p> }
	TGLObjectEffects = class (TXCollection)
		protected
			{ Protected Declarations }
			function GetEffect(index : Integer) : TGLObjectEffect;

		public
			{ Public Declarations }
			constructor Create(aOwner : TPersistent); override;

         class function ItemsClass : TXCollectionItemClass; override;

			property ObjectEffect[index : Integer] : TGLObjectEffect read GetEffect; default;

			procedure DoProgress(const deltaTime, newTime : Double);
         procedure RenderPreEffects(sceneViewer : TGLSceneViewer;
                                    const cameraPosition : TVector;
                                    var currentStates : TGLStates);
         {: Also take care of registering after effects with the GLSceneViewer. }
         procedure RenderPostEffects(sceneViewer : TGLSceneViewer;
                                     const cameraPosition : TVector;
                                     var currentStates : TGLStates);
	end;

	// TGLCustomSceneObject
	//
	{: Extended base class with material and rendering options. }
	TGLCustomSceneObject = class(TGLBaseSceneObject)
		private
			FMaterial: TGLMaterial;
			procedure SetGLMaterial(AValue: TGLMaterial);

		protected
			{ Protected Declarations }

		public
			{ Public Declarations }
			constructor Create(AOwner: TComponent); override;
			destructor Destroy; override;

			procedure Assign(Source: TPersistent); override;
			procedure ReloadTexture;
			procedure Render(objectsSorting : TGLObjectsSorting;
								  const cameraPosition : TVector;
								  var currentStates : TGLStates); override;

			property Handle: TObjectHandle read GetHandle;
			property Material: TGLMaterial read FMaterial write SeTGLMaterial;
			property Visible: Boolean read FVisible write SetVisible default True;

   end;

   // TGLSceneRootObject
   //
   {: This class shall be used only as a hierarchy root.<p>
      It exists only as a container and shall never be rotated/scaled etc. as
      the class type is used in parenting optimizations.<p>
      Shall never implement or add any functionality. }
   TGLSceneRootObject = class (TGLCustomSceneObject)
   ;

	// TGLImmaterialSceneObject
	//
	{: Base class for objects that do not have a published "material".<p>
		Note that the material is available in public properties.<br>
		Subclassing should be reserved to structural objects. }
	TGLImmaterialSceneObject = class(TGLCustomSceneObject)
		published
			property ObjectsSorting;
			property Direction;
			property PitchAngle;
			property Position;
			property RollAngle;
			property Scale;
			property ShowAxes;
			property TransformationMode;
			property TurnAngle;
			property Up;
			property OnProgress;
			property Behaviours;
	end;

	// TGLSceneObject
	//
	{: Base class for standard scene objects. }
	TGLSceneObject = class(TGLImmaterialSceneObject)
		published
			{ Published Declarations }
			property Material;
			property Visible;
	end;

	// TGLProxyObject
	//
	{: A full proxy object.<p>
		This object literally uses another object's Render method to do its own
		rendering, however, it has a coordinate system and a life of its own.<br>
		Use it for duplicates of an object. }
	TGLProxyObject = class (TGLBaseSceneObject)
		private
			{ Private Declarations }
			FMasterObject : TGLBaseSceneObject;
			FProxyOptions : TGLProxyObjectOptions;

		protected
			{ Protected Declarations }
			procedure Notification(AComponent: TComponent; Operation: TOperation); override;
			procedure SetMasterObject(const val : TGLBaseSceneObject);
			procedure SetProxyOptions(const val : TGLProxyObjectOptions);

		public
			{ Public Declarations }
			constructor Create(AOwner: TComponent); override;
			destructor Destroy; override;

			procedure Assign(Source: TPersistent); override;
         procedure Render(objectsSorting : TGLObjectsSorting;
								  const cameraPosition : TVector;
								  var currentStates : TGLStates); override;

		published
			{ Published Declarations }
         property MasterObject : TGLBaseSceneObject read FMasterObject write SetMasterObject;
         property ProxyOptions : TGLProxyObjectOptions read FProxyOptions write SetProxyOptions default cDefaultProxyOptions;

	      property ObjectsSorting;
         property Direction;
         property PitchAngle;
			property Position;
         property RollAngle;
         property Scale;
			property ShowAxes;
         property TransformationMode;
         property TurnAngle;
         property Up;
			property OnProgress;
			property Behaviours;
   end;

   // TLightStyle
   //
   {: Defines the various styles for lightsources.<p>
      - lsSpot : a spot light, oriented and with a cutoff zone (note that if
         cutoff is 180, the spot is rendered as an omni source)<br>
      - lsOmni : an omnidirectionnal source, punctual and sending light in
         all directions uniformously<br>
      - lsParallel : a parallel light, oriented as the light source is (this
         type of light help speed up rendering on non-T&L accelerated cards) }
   TLightStyle = (lsSpot, lsOmni, lsParallel);

	// TGLLightSource
	//
	{: Standard light source.<p>
		The standard GLScene light source covers spotlights, omnidirectionnal and
      parallel sources (see TLightStyle).<br>
      Lights are colored, have distance attenuation parameters and are turned
      on/off through their Shining property.<p>
		Lightsources are managed in a specific object by the TGLScene for rendering
		purposes. The maximum number of light source in a scene is limited by the
	  	OpenGL implementation (8 lights are supported under most ICD). }
  TGLLightSource = class(TGLBaseSceneObject)
     private
			{ Private Declarations }
         FLightID: TObjectHandle;
         FSpotDirection: TGLCoordinates;
         FSpotExponent, FSpotCutOff: TGLFloat;
         FConstAttenuation, FLinearAttenuation, FQuadraticAttenuation: TGLFloat;
         FShining: Boolean;
         FAmbient, FDiffuse, FSpecular: TGLColor;
         FLightStyle : TLightStyle;

		protected
			{ Protected Declarations }
	      //: light sources have different handle types than normal scene objects
      	function GetHandle: TObjectHandle; override;

         procedure SetAmbient(AValue: TGLColor);
         procedure SetDiffuse(AValue: TGLColor);
         procedure SetSpecular(AValue: TGLColor);
         procedure SetConstAttenuation(AValue: TGLFloat);
         procedure SetLinearAttenuation(AValue: TGLFloat);
         procedure SetQuadraticAttenuation(AValue: TGLFloat);
         procedure SetShining(AValue: Boolean);
         procedure SetSpotDirection(AVector: TGLCoordinates);
         procedure SetSpotExponent(AValue: TGLFloat);
         procedure SetSpotCutOff(AValue: TGLFloat);
         procedure SetLightStyle(const val : TLightStyle);

      public
			{ Public Declarations }
         constructor Create(AOwner: TComponent); override;
         destructor Destroy; override;
         procedure DestroyList; override;
         procedure CoordinateChanged(Sender: TGLCoordinates); override;
//    procedure RenderLensFlares(from, at: TAffineVector; near_clip: TGLFloat);
    {: Calculate own global matrix and let the children do the same with their's.<p>
       TODO: something looks weird... GlobalMatrix might get calc'ed twice ??? }
         procedure ValidateTransformation; override;

      published
			{ Published Declarations }
         property Ambient: TGLColor read FAmbient write SetAmbient;
         property ConstAttenuation: TGLFloat read FConstAttenuation write SetConstAttenuation;
         property Diffuse: TGLColor read FDiffuse write SetDiffuse;
         property LinearAttenuation: TGLFloat read FLinearAttenuation write SetLinearAttenuation;
         property QuadraticAttenuation: TGLFloat read FQuadraticAttenuation write SetQuadraticAttenuation;
         property Position;
         property LightStyle : TLightStyle read FLightStyle write SetLightStyle default lsSpot;
         property Shining: Boolean read FShining write SetShining default True;
         property Specular: TGLColor read FSpecular write SetSpecular;
         property SpotCutOff: TGLFloat read FSpotCutOff write SetSpotCutOff;
         property SpotDirection: TGLCoordinates read FSpotDirection write SetSpotDirection;
         property SpotExponent: TGLFloat read FSpotExponent write SetSpotExponent;
         property OnProgress;
  end;

   // TGLCamera
   //
	TGLCamera = class(TGLBaseSceneObject)
      private
         { Private Declarations }
         FFocalLength: Single;
			FDepthOfView: Single;
         FNearPlane: Single;                  // nearest distance to the camera
         FTargetObject : TGLBaseSceneObject;
         FLastTargetObjectPosition : TVector; // Not persistent

         procedure SetDepthOfView(AValue: Single);
			procedure SetFocalLength(AValue: Single);

		protected
			{ Protected Declarations }
			procedure Notification(AComponent: TComponent; Operation: TOperation); override;
			procedure SetTargetObject(const val : TGLBaseSceneObject);

		public
			{ Public Declarations }
			constructor Create(aOwner : TComponent); override;
			destructor Destroy; override;

         {: Nearest clipping plane for the frustum.<p> }
         property NearPlane : Single read FNearPlane;

			//: Apply camera transformation
			procedure Apply;
			procedure ApplyPerspective(Viewport: TRectangle; Width, Height: Integer; DPI: Integer);
			procedure AutoLeveling(Factor: Single);
			procedure Reset;
			//: Position the camera so that the whole scene can be seen
			procedure ZoomAll;
			{: Change camera's position to make it move around its target.<p>
				If TargetObject is nil, nothing happens. This method helps in quickly
				implementing camera controls. Camera's Up and Direction properties
				are unchanged.<br>
				Angle deltas are in degrees, camera parent's coordinates should be identity.<p>
				Tip : make the camera a child of a "target" dummycube and make
				it a target the dummycube. Now, to pan across the scene, just move
				the dummycube, to change viewing angle, use this method. }
			procedure MoveAroundTarget(pitchDelta, turnDelta : Single);
			{: Adjusts distance from camera to target by applying a ratio.<p>
				If TargetObject is nil, nothing happens. This method helps in quickly
				implementing camera controls. Only the camera's position is changed.<br>
				Camera parent's coordinates should be identity. }
			procedure AdjustDistanceToTarget(distanceRatio : Single);
			{: Returns the distance from camera to target.<p>
				If TargetObject is nil, returns 1. }
			function DistanceToTarget : Single;
			{: Calculate an absolute translation vector from a screen vector.<p>
				Ratio is applied to both screen delta, planeNormal should be the
				translation plane's normal. }
			function ScreenDeltaToVector(deltaX, deltaY : Integer; ratio : Single;
												  const planeNormal : TVector) : TVector;
			{: Same as ScreenDeltaToVector but optimized for XY plane. }
			function ScreenDeltaToVectorXY(deltaX, deltaY : Integer; ratio : Single) : TVector;
         {: Same as ScreenDeltaToVector but optimized for XZ plane. }
         function ScreenDeltaToVectorXZ(deltaX, deltaY : Integer; ratio : Single) : TVector;
         {: Same as ScreenDeltaToVector but optimized for YZ plane. }
         function ScreenDeltaToVectorYZ(deltaX, deltaY : Integer; ratio : Single) : TVector;

		published
			{ Published Declarations }
			property DepthOfView: Single read FDepthOfView write SetDepthOfView;
			property FocalLength: Single read FFocalLength write SetFocalLength;
			{: If set, camera will point to this object.<p>
				When camera is pointing an object, the Direction vector is ignored
				and the Up vector is used as an absolute vector to the up. }
			property TargetObject : TGLBaseSceneObject read FTargetObject write SetTargetObject;

			property Position;
			property Direction;
			property Up;
         property OnProgress;
	end;

	// TGLScene
	//
	TGLScene = class(TGLUpdateAbleComponent)
      private
			{ Private Declarations }
         FUpdateCount: Integer;
         FObjects: TGLSceneRootObject;
         FCameras: TGLBaseSceneObject;
         FBaseContext: HGLRC;
         FLights, FViewers: TList;
         FLasTGLCamera, FCurrenTGLCamera: TGLCamera;
         FCurrentViewer: TGLSceneViewer;
         FObjectsSorting : TGLObjectsSorting;
         FOnProgress : TGLProgressEvent;

		protected
			{ Protected Declarations }
			procedure AddLight(ALight: TGLLightSource);
         procedure SetupLights(Maximum: Integer);
         procedure DoAfterRender;
         procedure GetChildren(AProc: TGetChildProc; Root: TComponent); override;
			procedure Loaded; override;
			procedure RemoveLight(ALight: TGLLightSource);
			procedure SetChildOrder(AChild: TComponent; Order: Integer); override;
			procedure SetObjectsSorting(const val : TGLObjectsSorting);

		public
			{ Public Declarations }
			constructor Create(AOwner: TComponent); override;
			destructor Destroy; override;

{$ifndef DFS_DELPHI_5_UP}
			procedure Notification(AComponent: TComponent; Operation: TOperation); override;
{$endif}

			procedure AddViewer(AViewer: TGLSceneViewer);
         procedure BeginUpdate;
			procedure RenderScene(AViewer: TGLSceneViewer);
         procedure EndUpdate;
         function  IsUpdating: Boolean;
         procedure NotifyChange; override;
         procedure RemoveViewer(AViewer: TGLSceneViewer);
			procedure ValidateTransformation(ACamera: TGLCamera);
         procedure Progress(const deltaTime, newTime : Double);

         {: Saves the scene to a file (recommended extension : .GLS) }
			procedure SaveToFile(const fileName : String);
         {: Load the scene from a file.<p>
            Existing objects/lights/cameras are freed, then the file is loaded.<br>
            Delphi's IDE is not handling this behaviour properly yet, ie. if
            you load a scene in the IDE, objects will be properly loaded, but
            no declare will be placed in the code. }
			procedure LoadFromFile(const fileName : String);

         property Cameras: TGLBaseSceneObject read FCameras;
			property CurrentGLCamera: TGLCamera read FCurrentGLCamera;
         property Lights: TList read FLights;
         property Objects: TGLSceneRootObject read FObjects;
         property CurrentViewer: TGLSceneViewer read FCurrentViewer;

		published
			{ Published Declarations }
			property ObjectsSorting : TGLObjectsSorting read FObjectsSorting write SetObjectsSorting default osRenderBlendedLast;
         property OnProgress : TGLProgressEvent read FOnProgress write FOnProgress;

   end;

  PPickRecord = ^TPickRecord;
  TPickRecord = record
	 AObject: TGLBaseSceneObject;
    ZMin, ZMax: Single;
  end;

  TPickSortType = (psDefault, psName, psMinDepth, psMaxDepth);

  // list class for object picking
  TGLPickList =  class(TList)
  private
    function GetFar(AValue: Integer): Single;
    function GetHit(AValue: Integer): TGLBaseSceneObject;
	 function GetNear(AValue: Integer): Single;
  protected
  public
    constructor Create(SortType: TPickSortType);
    destructor Destroy; override;
	 procedure AddHit(Obj: TGLBaseSceneObject; ZMin, ZMax: Single);
    procedure Clear; override;
    function FindObject(AObject: TGLBaseSceneObject): Integer;
    property FarDistance[Index: Integer]: Single read GetFar;
	 property Hit[Index: Integer]: TGLBaseSceneObject read GetHit; default;
    property NearDistance[Index: Integer]: Single read GetNear;
  end;

  TDrawState = (dsNone, dsRendering, dsPicking, dsPrinting);

  TFogMode = (fmLinear, fmExp, fmExp2);
  TGLFogEnvironment = class (TGLUpdateAbleObject)
  private
    FSceneViewer: TGLSceneViewer;
    FFogColor: TGLColor;       // alpha value means the fog density
    FFogStart,
	 FFogEnd: TGLfloat;
    FFogMode: TFogMode;
    FChanged: Boolean;
    procedure SetFogColor(Value: TGLColor);
    procedure SetFogStart(Value: TGLfloat);
    procedure SetFogEnd(Value: TGLfloat);
    procedure SetFogMode(Value: TFogMode);

  public
    constructor Create(Owner : TPersistent); override;
	 destructor Destroy; override;

    procedure NotifyChange; override;
    procedure ApplyFog;
    procedure Assign(Source: TPersistent); override;
  published

    property FogColor: TGLColor read FFogColor write SetFogColor;
    property FogStart: TGLfloat read FFogStart write SetFogStart;
    property FogEnd: TGLfloat read FFogEnd write SetFogEnd;
    property FogMode: TFogMode read FFogMode write SetFogMode; 
  end;

  TSpecial = (spLensFlares, spLandScape);
  TSpecials = set of TSpecial;

  TGLSceneViewer = class(TWinControl)
  private
	 // handle
	 FRenderingContext: HGLRC;

	 // OpenGL properties
	 FMaxLightSources: Integer;
	 FDoubleBuffered, FDepthTest, FFaceCulling, FFogEnable, FLighting: Boolean;
	 FCurrentStates, FSaveStates: TGLStates;
	 FBackgroundColor: TColor;
	 FBackground: TGLTexture;

	 // private variables
	 FCanvas: TCanvas;
	 FFrames: Longint;              // used to perform monitoring
	 FTicks: TLargeInteger;        // used to perform monitoring
	 FState: TDrawState;
    FMonitor: Boolean;
    FFramesPerSecond: TGLFloat;
    FViewPort: TRectangle;
    FBuffers: TBuffers;
    FDisplayOptions: TDisplayOptions;
	 FContextOptions: TContextOptions;
    FCamera: TGLCamera;
//    FSpecials: TSpecials;
    FFogEnvironment: TGLFogEnvironment;
    invalidated : Boolean;
    afterRenderEffects : TList;

    FBeforeRender: TNotifyEvent;
    FPostRender : TNotifyEvent;
    FAfterRender: TNotifyEvent;
    procedure SetBackgroundColor(AColor: TColor);
    function  GetLimit(Which: TLimitType): Integer;
    procedure SeTGLCamera(ACamera: TGLCamera);
    procedure SetContextOptions(Options: TContextOptions);
    procedure SetDepthTest(AValue: Boolean);
	 procedure SetFaceCulling(AValue: Boolean);
	 procedure SetLighting(AValue: Boolean);
	 procedure SetFogEnable(AValue: Boolean);
	 procedure SeTGLFogEnvironment(AValue: TGLFogEnvironment);
//	 procedure SetSpecials(Value: TSpecials);

	 procedure WMEraseBkgnd(var Message: TWMEraseBkgnd); Message WM_ERASEBKGND;
	 procedure WMPaint(var Message: TWMPaint); Message WM_PAINT;
	 procedure WMSize(var Message: TWMSize); Message WM_SIZE;

  protected
	 procedure CreateParams(var Params: TCreateParams); override;
	 procedure CreateWnd; override;
	 {: Clear all allocated OpenGL buffers.<p>
		 The color buffer is a special case, because transparency must be
		 simulated if required. }
	 procedure ClearBuffers;
    procedure DestroyWnd; override;
	 procedure Loaded; override;
    function ObjectInScene(Obj: TGLBaseSceneObject): Boolean;
	 procedure ReadContextProperties;
	 procedure SetupRenderingContext;

  public
	 constructor Create(AOwner: TComponent); override;
	 destructor  Destroy; override;

	 procedure Notification(AComponent: TComponent; Operation: TOperation); override;

    procedure RequestedState(States: TGLStates);
    procedure UnnecessaryState(States: TGLStates);

	 //: Fills the PickList with objects in Rect area
	 procedure PickObjects(const Rect: TRect; PickList: TGLPickList; ObjectCountGuess: Integer);
	 {: Returns a PickList with objects in Rect area.<p>
		 Returned list should be freed by caller.<br>
		 Objects are sorted by depth (nearest objects first). }
	 function GetPickedObjects(const Rect: TRect; objectCountGuess : Integer = 64) : TGLPickList;
	 //: Returns the nearest object at x, y coordinates or nil if there is none
	 function GetPickedObject(x, y : Integer) : TGLBaseSceneObject;

	 procedure Render;
    {: Render the scene to a bitmap at given DPI.<p>
       DPI = "dots per inch".<p>
       The "magic" DPI of the screen is 96 under Windows. }
    procedure RenderToBitmap(ABitmap: TBitmap; DPI: Integer = 0);
    {: Render the scene to a bitmap at given DPI and saves it to a file.<p>
       DPI = "dots per inch".<p>
       The "magic" DPI of the screen is 96 under Windows. }
    procedure RenderToFile(const AFile: String; DPI: Integer = 0); overload;
    {: Renders to bitmap of given size, then saves it to a file.<p>
       DPI is adjusted to make the bitmap similar to the viewer. }
    procedure RenderToFile(const AFile: String; bmpWidth, bmpHeight : Integer); overload;

    procedure Invalidate; override;
	 procedure SetViewPort(X, Y, W, H: Integer);
    procedure ShowInfo;
    procedure ResetPerformanceMonitor;

//    function GetLandsHeight(X, Y: TGLfloat): TGLfloat;

    property Buffers: TBuffers read FBuffers;
    property Canvas: TCanvas read FCanvas;
	 property CurrentStates: TGLStates read FCurrentStates;
    property DoubleBuffered: Boolean read FDoubleBuffered;
    property FramesPerSecond: TGLFloat read FFramesPerSecond;
    property LimitOf[Which: TLimitType]: Integer read GetLimit;
    property MaxLightSources: Integer read FMaxLightSources;
    property RenderingContext: HGLRC read FRenderingContext;
    property State: TDrawState read FState;

  published
    property FogEnvironment: TGLFogEnvironment read FFogEnvironment write SeTGLFogEnvironment;
    property Align;
    property Anchors;
	 property BackgroundColor: TColor read FBackgroundColor write SetBackgroundColor default clBtnFace;
    property Camera: TGLCamera read FCamera write SeTGLCamera;
    property Constraints;
    property ContextOptions: TContextOptions read FContextOptions write SetContextOptions default [roDoubleBuffer, roRenderToWindow];
	 property DepthTest: Boolean read FDepthTest write SetDepthTest default True;
    property DisplayOptions: TDisplayOptions read FDisplayOptions write FDisplayOptions;
    property DragCursor;
    property DragMode;
    property Enabled;
    property FaceCulling: Boolean read FFaceCulling write SetFaceCulling default True;
    property HelpContext;
    property Hint;
    property FogEnable: Boolean read FFogEnable write SetFogEnable default False;
    property Lighting: Boolean read FLighting write SetLighting default True;
    property Monitor: Boolean read FMonitor write FMonitor default False;
    property PopupMenu;
//    property Specials: TSpecials read FSpecials write SetSpecials default [];
    property Visible;

	 // events
	 property BeforeRender: TNotifyEvent read FBeforeRender write FBeforeRender;
	 property PostRender: TNotifyEvent read FPostRender write FPostRender;
	 property AfterRender: TNotifyEvent read FAfterRender write FAfterRender;
	 property OnClick;
	 property OnDragDrop;
	 property OnDragOver;
	 property OnMouseDown;
	 property OnMouseMove;
	 property OnMouseUp;
  end;

  EOpenGLError = class(Exception);

procedure CheckOpenGLError;
procedure RaiseOpenGLError(const msg : String);

{: Register an event handler triggered by any TGLBaseSceneObject Name change.<p>
	*INCOMPLETE*, currently allows for only 1 (one) event, and is used by
	GLSceneEdit in the IDE. }
procedure RegisterGLBaseSceneObjectNameChangeEvent(notifyEvent : TNotifyEvent);
{: Deregister an event handler triggered by any TGLBaseSceneObject Name change.<p>
	See RegisterGLBaseSceneObjectNameChangeEvent. }
procedure DeRegisterGLBaseSceneObjectNameChangeEvent(notifyEvent : TNotifyEvent);
{: Register an event handler triggered by any TGLBehaviour Name change.<p>
	*INCOMPLETE*, currently allows for only 1 (one) event, and is used by
	FBehavioursEditor in the IDE. }
procedure RegisterGLBehaviourNameChangeEvent(notifyEvent : TNotifyEvent);
{: Deregister an event handler triggered by any TGLBaseSceneObject Name change.<p>
	See RegisterGLBaseSceneObjectNameChangeEvent. }
procedure DeRegisterGLBehaviourNameChangeEvent(notifyEvent : TNotifyEvent);

{: Issues OpenGL calls for drawing X, Y, Z axes in a standard style. }
procedure AxesBuildList(Pattern: TGLushort; AxisLen: TGLFloat);

//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
implementation
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------

uses
  Consts, Dialogs, ExtDlgs, Forms, GLStrings, Info; //GLSpecials

const
  GLAllStates = [stAlphaTest..stStencilTest];

var
  vCounterFrequency : TLargeInteger;

//------------------ external global routines ----------------------------------

procedure CheckOpenGLError;

// Gets the oldest error from OpenGL engine and tries to clear the error queue.
// Because under some circumstances reading the error code creates a new error
// and thus hanging up the thread, we limit the loop to 6 reads.

var GLError: TGLEnum;
	 Count: Word;

begin
	GLError := glGetError;
	if GLError <> GL_NO_ERROR then begin
		Count := 0;
      try
         while (glGetError <> GL_NO_ERROR) and (Count < 6) do Inc(Count);
      except
         // Egg : ignore exceptions here, will perhaps avoid problem expressed before
		end;
		raise EOpenGLError.Create(gluErrorString(GLError));
	end;
end;

procedure RaiseOpenGLError(const msg : String);
begin
	raise EOpenGLError.Create(msg);
end;

// AxesBuildList
//
procedure AxesBuildList(Pattern: TGLushort; AxisLen: TGLFloat);
begin
   glPushAttrib(GL_ENABLE_BIT or GL_LIGHTING_BIT or GL_LINE_BIT);
   glDisable(GL_LIGHTING);
   glEnable(GL_LINE_STIPPLE);
   glEnable(GL_LINE_SMOOTH);
   glEnable(GL_BLEND);
   glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
   glLineWidth(1);
   glLineStipple(1, Pattern);
   glBegin(GL_LINES);
      glColor3f(0.5, 0.0, 0.0); glVertex3f(0, 0, 0); glVertex3f(-AxisLen, 0, 0);
      glColor3f(1.0, 0.0, 0.0); glVertex3f(0, 0, 0); glVertex3f(AxisLen, 0, 0);
      glColor3f(0.0, 0.5, 0.0); glVertex3f(0, 0, 0); glVertex3f(0, -AxisLen, 0);
      glColor3f(0.0, 1.0, 0.0); glVertex3f(0, 0, 0); glVertex3f(0, AxisLen, 0);
      glColor3f(0.0, 0.0, 0.5); glVertex3f(0, 0, 0); glVertex3f(0, 0, -AxisLen);
      glColor3f(0.0, 0.0, 1.0); glVertex3f(0, 0, 0); glVertex3f(0, 0, AxisLen);
   glEnd;
   glPopAttrib;
end;

//------------------ internal global routines ----------------------------------

var
	vGLBaseSceneObjectNameChangeEvent : TNotifyEvent;
	vGLBehaviourNameChangeEvent : TNotifyEvent;

// RegisterGLBaseSceneObjectNameChangeEvent
//
procedure RegisterGLBaseSceneObjectNameChangeEvent(notifyEvent : TNotifyEvent);
begin
	vGLBaseSceneObjectNameChangeEvent:=notifyEvent;
end;

// DeRegisterGLBaseSceneObjectNameChangeEvent
//
procedure DeRegisterGLBaseSceneObjectNameChangeEvent(notifyEvent : TNotifyEvent);
begin
	vGLBaseSceneObjectNameChangeEvent:=nil;
end;

// RegisterGLBehaviourNameChangeEvent
//
procedure RegisterGLBehaviourNameChangeEvent(notifyEvent : TNotifyEvent);
begin
   vGLBehaviourNameChangeEvent:=notifyEvent;
end;

// DeRegisterGLBehaviourNameChangeEvent
//
procedure DeRegisterGLBehaviourNameChangeEvent(notifyEvent : TNotifyEvent);
begin
   vGLBehaviourNameChangeEvent:=nil;
end;

//----------------- TGLPickList -------------------------------------------------

var
  SortFlag: TPickSortType;

constructor TGLPickList.Create(SortType: TPickSortType);

begin
  SortFlag := SortType;
  inherited Create;
end;

//------------------------------------------------------------------------------

destructor TGLPickList.Destroy;

begin
  Clear;
  inherited Destroy;
end;

//------------------------------------------------------------------------------

function CompareFunction(Item1, Item2: Pointer): Integer;

var
  Diff: Single;

begin
  Result := 0;
  case SortFlag of
    psName:
      Result := CompareText(PPickRecord(Item1).AObject.Name, PPickRecord(Item2).AObject.Name);
    psMinDepth:
      begin
        Diff := PPickRecord(Item1).ZMin - PPickRecord(Item2).ZMin;
        if Diff < 0 then Result := -1
                    else
          if Diff > 0 then Result := 1
                      else Result := 0;
      end;
    psMaxDepth:
      begin
        Diff := Round(PPickRecord(Item1).ZMax - PPickRecord(Item2).ZMax);
        if Diff < 0 then Result := -1
                    else
          if Diff > 0 then Result := 1
                      else Result := 0;
      end;
  end;
end;

//------------------------------------------------------------------------------

procedure TGLPickList.AddHit(Obj: TGLBaseSceneObject; ZMin, ZMax: Single);

var
  NewRecord: PPickRecord;

begin
  New(NewRecord);
  try
    NewRecord.AObject := Obj;
    NewRecord.ZMin := ZMin;
    NewRecord.ZMax := ZMax;
    Add(NewRecord);
    if SortFlag <> psDefault then Sort(CompareFunction);
  except
    Dispose(NewRecord);
    raise;
  end;
end;

//------------------------------------------------------------------------------

procedure TGLPickList.Clear;

var
  I: Integer;

begin
  for I := 0 to Count-1 do
    Dispose(PPickRecord(Items[I]));
  inherited Clear;
end;

// FindObject
//
function TGLPickList.FindObject(AObject: TGLBaseSceneObject): Integer;
var
   i : Integer;
begin
   Result:=-1;
   if Assigned(AObject) then for i:=0 to Count-1 do
      if Hit[i]=AObject then begin
         Result:=i;
         Break;
      end;
end;

//------------------------------------------------------------------------------

function TGLPickList.GetFar(AValue: Integer): Single;

begin
  Result := PPickRecord(Items[AValue]).ZMax;
end;

//------------------------------------------------------------------------------

function TGLPickList.GetHit(AValue: Integer): TGLBaseSceneObject;

begin
  Result := PPickRecord(Items[AValue]).AObject;
end;

//------------------------------------------------------------------------------

function TGLPickList.GetNear(AValue: Integer): Single;

begin
  Result := PPickRecord(Items[AValue]).ZMin;
end;

//----------------- TGLCoordinates ---------------------------------------------

// CreateInitialized
//
constructor TGLCoordinates.CreateInitialized(aOwner : TPersistent; const value : TVector);
begin
   Create(aOwner);
   FCoords:=value;
   FDefaultCoords:=value;
end;

// Initialize
//
procedure TGLCoordinates.Initialize(const value : TVector);
begin
   FCoords:=value;
   FDefaultCoords:=value;
end;

// Assign
//
procedure TGLCoordinates.Assign(Source: TPersistent);
begin
   if Source is TGLCoordinates then
      FCoords:=TGLCoordinates(Source).FCoords
   else inherited;
end;

// WriteToFiler
//
procedure TGLCoordinates.WriteToFiler(writer : TWriter);
var
   writeCoords : Boolean;
begin
   with writer do begin
      WriteInteger(0); // Archive Version 0
      writeCoords:=not VectorEquals(FDefaultCoords, FCoords);
      WriteBoolean(writeCoords);
      if writeCoords then
         Write(FCoords, SizeOf(FCoords));
   end;
end;

// ReadFromFiler
//
procedure TGLCoordinates.ReadFromFiler(reader : TReader);
begin
   with reader do begin
      ReadInteger; // Ignore ArchiveVersion
      if ReadBoolean then
         Read(FCoords, SizeOf(FCoords))
      else FCoords:=FDefaultCoords;
   end;
end;

// DefineProperties
//
procedure TGLCoordinates.DefineProperties(Filer: TFiler);
begin
	inherited;
	Filer.DefineBinaryProperty('Coordinates', ReadData, WriteData,
                              not VectorEquals(FDefaultCoords, FCoords));
end;

// ReadData
//
procedure TGLCoordinates.ReadData(Stream: TStream);
begin
	Stream.Read(FCoords, SizeOf(FCoords));
end;

// WriteData
//
procedure TGLCoordinates.WriteData(Stream: TStream);
begin
	Stream.Write(FCoords, SizeOf(FCoords));
end;

// NotifyChange
//
procedure TGLCoordinates.NotifyChange;
begin
	if (Owner is TGLBaseSceneObject) then
		TGLBaseSceneObject(Owner).CoordinateChanged(Self)
	else inherited NotifyChange;
end;

// Translate
//
procedure TGLCoordinates.Translate(const translationVector : TVector);
begin
	FCoords[0]:=FCoords[0]+translationVector[0];
	FCoords[1]:=FCoords[1]+translationVector[1];
	FCoords[2]:=FCoords[2]+translationVector[2];
	NotifyChange;
end;

// AddScaledVector
//
procedure TGLCoordinates.AddScaledVector(const factor : Single; const translationVector : TVector);
begin
	FCoords[0]:=FCoords[0]+factor*translationVector[0];
	FCoords[1]:=FCoords[1]+factor*translationVector[1];
	FCoords[2]:=FCoords[2]+factor*translationVector[2];
	NotifyChange;
end;

// Invert
//
procedure TGLCoordinates.Invert;
begin
   NegateVector(FCoords);
end;

// VectorLength
//
function TGLCoordinates.VectorLength : TGLFloat;
begin
   Result:=Geometry.VectorLength(FCoords);
end;

// SetVector
//
procedure TGLCoordinates.SetVector(const x, y, z, w : Single);
begin
   Geometry.SetVector(FCoords, x, y, z, w);
	NotifyChange;
end;

// SetPoint
//
procedure TGLCoordinates.SetPoint(const x, y, z : Single);
begin
   Geometry.MakePoint(FCoords, x, y, z);
	NotifyChange;
end;

// AsAddress
//
function TGLCoordinates.AsAddress : PGLFloat;
begin
   Result:=@FCoords;
end;

// SetAsVector
//
procedure TGLCoordinates.SetAsVector(const value: TVector);
begin
	FCoords:=Value;
	NotifyChange;
end;

// SetCoordinate
//
procedure TGLCoordinates.SetCoordinate(Index: Integer; AValue: TGLFloat);
begin
	FCoords[Index] := AValue;
	NotifyChange;
end;

//------------------ TGLBaseSceneObject ----------------------------------------

constructor TGLBaseSceneObject.Create(AOwner: TComponent);
begin
	inherited Create(AOwner);
   FObjectStyle:=[];
	FChanges:=[ocTransformation, ocStructure];
	FPosition:=TGLCoordinates.CreateInitialized(Self, NullHmgPoint);
	FDirection:=TGLCoordinates.CreateInitialized(Self, ZHmgVector);
	FUp:=TGLCoordinates.CreateInitialized(Self, YHmgVector);
	FScaling:=TGLCoordinates.CreateInitialized(Self, XYZHmgVector);
	FGlobalMatrix:=IdentityHmgMatrix;
	FLocalMatrix:=IdentityHmgMatrix;
	FChildren:=TList.Create;
	FVisible:=True;
	FMatrixDirty:=True;
	FObjectsSorting:=osInherited;
	FBehaviours:=TGLBehaviours.Create(Self);
   FObjectEffects:=TGLObjectEffects.Create(Self);
end;

// Destroy
//
destructor TGLBaseSceneObject.Destroy;
begin
   FObjectEffects.Free;
	FBehaviours.Free;
	DestroyList;
	FPosition.Free;
	FDirection.Free;
	FUp.Free;
	if Assigned(FParent) then FParent.Remove(Self, False);
	if FChildren.Count>0 then DeleteChildren;
	FChildren.Free;
	if Assigned(FScene) then FScene.NotifyChange;
	inherited Destroy;
end;

// DestroyList
//
procedure TGLBaseSceneObject.DestroyList;
var
   i : Integer;
begin
   Include(FChanges, ocStructure);
   for i:=0 to FChildren.Count-1 do
      TGLBaseSceneObject(FChildren[i]).DestroyList;
   if FHandle > 0 then begin
      glDeleteLists(FHandle, 1);
      FHandle := 0;
   end;
end;

// BeginUpdate
//
procedure TGLBaseSceneObject.BeginUpdate;
begin
   Inc(FUpdateCount);
end;

// BuildList
//
procedure TGLBaseSceneObject.BuildList;
begin
//   glListBase(0);
end;

// DeleteChildren
//
procedure TGLBaseSceneObject.DeleteChildren;
var
	child : TGLBaseSceneObject;
begin
	while FChildren.Count>0 do begin
		child:=TGLBaseSceneObject(FChildren.Items[FChildren.Count-1]);
		child.FParent:=nil;
		FChildren.Delete(FChildren.Count-1);
		child.Free;
	end;
end;

// Loaded
//
procedure TGLBaseSceneObject.Loaded;
begin
   inherited;
   if FPosition.W=0 then FPosition.W:=1;
	FBehaviours.Loaded;
   FObjectEffects.Loaded;
end;

// DefineProperties
//
procedure TGLBaseSceneObject.DefineProperties(Filer: TFiler);
begin
	inherited;
	Filer.DefineBinaryProperty('BehavioursData',
                              ReadBehaviours, WriteBehaviours, 
								      (FBehaviours.Count>0));
end;

// WriteBehaviours
//
procedure TGLBaseSceneObject.WriteBehaviours(stream : TStream);
var
   writer : TWriter;
begin
   writer:=TWriter.Create(stream, 16384);
   try
      FBehaviours.WriteToFiler(writer);
   finally
      writer.Free;
   end;
end;

// ReadBehaviours
//
procedure TGLBaseSceneObject.ReadBehaviours(stream : TStream);
var
   reader : TReader;
begin
   reader:=TReader.Create(stream, 16384);
   try
      FBehaviours.ReadFromFiler(reader);
   finally
      reader.Free;
   end;
end;

// DrawAxes
//
procedure TGLBaseSceneObject.DrawAxes(Pattern: TGLushort);
begin
   AxesBuildList(Pattern, FScene.CurrentViewer.FCamera.FDepthOfView);
end;

// GetChildren
//
procedure TGLBaseSceneObject.GetChildren(AProc: TGetChildProc; Root: TComponent);
var
   i : Integer;
begin
   for i:=0 to Count-1 do
      AProc(FChildren[I]);
end;

// RebuildMatrix
//
procedure TGLBaseSceneObject.RebuildMatrix;
var
   vl : TGLFloat;
begin
	if FMatrixDirty then begin
		FLocalMatrix[0]:=VectorCrossProduct(FUp.FCoords, FDirection.FCoords);
      vl:=VectorLength(FLocalMatrix[0]);
      if vl<>0 then
   		ScaleVector(FLocalMatrix[0], Scale.X/vl);
      FLocalMatrix[1]:=FUp.FCoords;          ScaleVector(FLocalMatrix[1], Scale.Y);
		FLocalMatrix[2]:=FDirection.FCoords;   ScaleVector(FLocalMatrix[2], Scale.Z);
		FLocalMatrix[3]:=FPosition.FCoords;
		FMatrixDirty:=False;
  	end;
end;

// Get
//
function TGLBaseSceneObject.Get(Index: Integer): TGLBaseSceneObject;
begin
	Result := FChildren[Index];
end;

// GetCount
//
function TGLBaseSceneObject.GetCount: Integer;
begin
	Result := FChildren.Count;
end;

// AddChild
//
procedure TGLBaseSceneObject.AddChild(AChild: TGLBaseSceneObject);
begin
	if Assigned(FScene) and (AChild is TGLLightSource) then
		FScene.AddLight(TGLLightSource(AChild));
	FChildren.Add(AChild);
	AChild.FParent:=Self;
	AChild.SetScene(FScene);
	TransformationChanged;
end;

// AddNewChild
//
function TGLBaseSceneObject.AddNewChild(AChild: TGLSceneObjectClass): TGLBaseSceneObject;
begin
	Result:=AChild.Create(Self);
	AddChild(Result);
end;

// RestoreMatrix
//
procedure TGLBaseSceneObject.RestoreMatrix;
begin
	glLoadMatrixf(@FGlobalMatrix);
end;

// AbsoluteMatrix
//
function TGLBaseSceneObject.AbsoluteMatrix : TMatrix;
begin
   RebuildMatrix;
   if Assigned(Parent) and (not (Parent is TGLSceneRootObject)) then
      Result:=MatrixMultiply(TGLBaseSceneObject(Parent).AbsoluteMatrix, Self.FLocalMatrix)
   else Result:=Self.FLocalMatrix;
end;

// InvAbsoluteMatrix
//
function TGLBaseSceneObject.InvAbsoluteMatrix : TMatrix;
begin
   Result:=AbsoluteMatrix;
   TransposeMatrix(Result);
end;

// AbsolutePosition
//
function TGLBaseSceneObject.AbsolutePosition : TVector;
var
	myMatrix : TMatrix;
begin
	if (Parent<>nil) and (Parent is TGLBaseSceneObject) then begin
      with TGLBaseSceneObject(Parent) do begin
      	myMatrix:=AbsoluteMatrix;
	   	myMatrix[3]:=AbsolutePosition;
      end;
		Result:=VectorTransform(Position.AsVector, myMatrix);
	end else Result:=Position.AsVector;
end;

// AbsoluteXVector
//
function TGLBaseSceneObject.AbsoluteXVector : TVector;
var
	myMatrix : TMatrix;
begin
   myMatrix:=AbsoluteMatrix;
   SetVector(Result, myMatrix[0][0], myMatrix[0][1], myMatrix[0][2]);
end;

// AbsoluteYVector
//
function TGLBaseSceneObject.AbsoluteYVector : TVector;
var
	myMatrix : TMatrix;
begin
   myMatrix:=AbsoluteMatrix;
   SetVector(Result, myMatrix[1][0], myMatrix[1][1], myMatrix[1][2]);
end;

// AbsoluteZVector
//
function TGLBaseSceneObject.AbsoluteZVector : TVector;
var
	myMatrix : TMatrix;
begin
   myMatrix:=AbsoluteMatrix;
   SetVector(Result, myMatrix[2][0], myMatrix[2][1], myMatrix[2][2]);
end;

// BarycenterAbsolutePosition
//
function TGLBaseSceneObject.BarycenterAbsolutePosition : TVector;
begin
	Result:=AbsolutePosition;
end;

// SqrDistanceTo
//
function TGLBaseSceneObject.SqrDistanceTo(const pt : TVector) : Single;
var
	d : TVector;
begin
	d:=AbsolutePosition;
	Result:=Sqr(d[0]-pt[0])+Sqr(d[1]-pt[1])+Sqr(d[2]-pt[2]);
end;

// BarycenterSqrDistanceTo
//
function TGLBaseSceneObject.BarycenterSqrDistanceTo(const pt : TVector) : Single;
var
	d : TVector;
begin
	d:=BarycenterAbsolutePosition;
	Result:=Sqr(d[0]-pt[0])+Sqr(d[1]-pt[1])+Sqr(d[2]-pt[2]);
end;

// AxisAlignedDimensions
//
function TGLBaseSceneObject.AxisAlignedDimensions : TVector;
begin
   Result:=Scale.AsVector;
end;

// PointInObject
//
function TGLBaseSceneObject.PointInObject(const point : TVector) : Boolean;
var
   localPt, dim : TVector;
begin
   dim:=AxisAlignedDimensions;
   localPt:=VectorTransform(point, InvAbsoluteMatrix);
   Result:=(Abs(localPt[0])<=dim[0]) and (Abs(localPt[1])<=dim[1])
           and (Abs(localPt[2])<=dim[2]);
end;

// RayCastIntersect
//
function TGLBaseSceneObject.RayCastIntersect(const rayStart, rayVector : TAffineVector;
                                 intersectPoint : PAffineVector = nil;
                                 intersectNormal : PAffineVector = nil) : Boolean;
var
   localStart, localVector : TAffineVector;
   m : TMatrix;
begin
   m:=InvAbsoluteMatrix;
   localStart:=VectorTransform(rayStart, m);
   localVector:=VectorTransform(rayVector, m);
   Result:=RayCastTriangleIntersect(rayStart, rayVector,
                                    NullVector,
                                    XVector,
                                    ZVector,
                                    intersectPoint,
                                    intersectNormal);
end;

// Assign
//
procedure TGLBaseSceneObject.Assign(Source: TPersistent);
var
	i : Integer;
	child, newChild : TGLBaseSceneObject;
begin
	if Source is TGLBaseSceneObject then begin
		FPosition.FCoords:=TGLBaseSceneObject(Source).FPosition.FCoords;
		FChanges:=[ocTransformation, ocStructure];
		FVisible:=TGLBaseSceneObject(Source).FVisible;
		FGlobalmatrix:=TGLBaseSceneObject(Source).FGLobalMatrix;
		SetMatrix(TGLCustomSceneObject(Source).FLocalMatrix);
		FObjectsSorting:=TGLBaseSceneObject(Source).FObjectsSorting;
		DeleteChildren;
		if Assigned(Scene) then Scene.BeginUpdate;
		for i:=0 to TGLBaseSceneObject(Source).FChildren.Count-1 do begin
			child:=TGLBaseSceneObject(Source).FChildren[I];
			newChild:=AddNewChild(TGLSceneObjectClass(child.ClassType));
			newChild.Assign(child);
		end;
		if Assigned(Scene) then Scene.EndUpdate;
		OnProgress:=TGLBaseSceneObject(Source).OnProgress;
		FBehaviours.Assign(TGLBaseSceneObject(Source).FBehaviours);
      FObjectEffects.Assign(TGLBaseSceneObject(Source).FObjectEffects);
   end else inherited Assign(Source);
end;

// Insert
//
procedure TGLBaseSceneObject.Insert(aIndex : Integer; aChild : TGLBaseSceneObject);
begin
	with FChildren do begin
		if Assigned(aChild.FParent) then
			aChild.FParent.Remove(aChild, False);
		Insert(aIndex, aChild);
	end;
	aChild.FParent := Self;
	if AChild.FScene<>FScene then AChild.DestroyList;
	AChild.SetScene(FScene);
	if Assigned(FScene) and (AChild is TGLLightSource) then
		FScene.AddLight(TGLLightSource(AChild));
  	TransformationChanged;
end;

//------------------------------------------------------------------------------

function TGLBaseSceneObject.IsUpdating: Boolean;

begin
  Result := (FUpdateCount <> 0) or (csReading in ComponentState);
end;

//------------------------------------------------------------------------------

function TGLBaseSceneObject.GetIndex: Integer;

begin
  Result := -1;
  if assigned(FParent) then Result := FParent.FChildren.IndexOf(Self);
end;

// GetHandle
//
function TGLBaseSceneObject.GetHandle: TObjectHandle;
begin
   if ocStructure in FChanges then begin
      if FHandle = 0 then begin
         FHandle := glGenLists(1);
         Assert(FHandle<>0, 'Handle=0 for '+ClassName);
      end;
      glNewList(FHandle, GL_COMPILE);
      try
         BuildList;
      finally
         glEndList;
      end;
      Exclude(FChanges, ocStructure);
   end;
   Result := FHandle;
end;

// GetOrientationVectors
//
procedure TGLBaseSceneObject.GetOrientationVectors(var Up, Direction: TAffineVector);
begin
   if (FTransMode <> tmLocal) and assigned(FParent) then begin
      Up:=AffineVectorMake(FParent.FUp.FCoords);
      Direction := AffineVectorMake(FParent.FDirection.FCoords);
   end else begin
      Up := AffineVectorMake(FUp.FCoords);
      Direction := AffineVectorMake(FDirection.FCoords);
   end;
end;

//------------------------------------------------------------------------------

function TGLBaseSceneObject.GetParentComponent: TComponent;

begin
  Result := FParent;
end;

//------------------------------------------------------------------------------

function TGLBaseSceneObject.HasParent: Boolean;

begin
  Result := assigned(FParent);
end;

//------------------------------------------------------------------------------

procedure TGLBaseSceneObject.Lift(ADistance: Single);

// moves object along the Up vector (move up/down)

var
  Up, Dir: TAffineVector;

begin
  if FTransMode = tmParentWithPos then
  begin
    GetOrientationVectors(Up, Dir);
    FPosition.X := FPosition.X + ADistance * Up[0];
    FPosition.Y := FPosition.Y + ADistance * Up[1];
    FPosition.Z := FPosition.Z + ADistance * Up[2];
  end
  else
  begin
    FPosition.X := FPosition.X + ADistance * FUp.FCoords[0];
    FPosition.Y := FPosition.Y + ADistance * FUp.FCoords[1];
    FPosition.Z := FPosition.Z + ADistance * FUp.FCoords[2];
  end;
  TransformationChanged;
end;

// Move
//
procedure TGLBaseSceneObject.Move(ADistance: Single);
var
   Len: Single;
   Up, Dir: TAffineVector;
begin
   if FTransMode = tmParentWithPos then begin
      GetOrientationVectors(Up, Dir);
      Len := 1 / Vectorlength(Dir);
      FPosition.AsVector := VectorMake(FPosition.X + ADistance * Dir[0] * Len,
                                       FPosition.Y + ADistance * Dir[1] * Len,
                                       FPosition.Z + ADistance * Dir[2] * Len, 1);
   end else begin
      Len := 1 / Vectorlength(FDirection.FCoords);
      FPosition.AsVector := VectorMake(FPosition.X + ADistance * FDirection.X * Len,
                                       FPosition.Y + ADistance * FDirection.Y * Len,
                                       FPosition.Z + ADistance * FDirection.Z * Len, 1);
   end;
   TransformationChanged;
end;

//------------------------------------------------------------------------------

procedure TGLBaseSceneObject.Pitch(Angle: Single);

var
  RightVector: TAffineVector;
  Up, Dir: TAffineVector;

begin
  GetOrientationVectors(Up, Dir);
  RightVector := VectorCrossProduct(Dir, Up);
  Angle := DegToRad(Angle);
  VectorRotate(FUp.FCoords, RightVector, -Angle);
  NormalizeVector(FUp.FCoords);
  VectorRotate(FDirection.FCoords, RightVector, -Angle);
  NormalizeVector(FDirection.FCoords);
  if FTransMode = tmParentWithPos then VectorRotate(FPosition.FCoords, RightVector, Angle);
  // Direction is automatically updated

  FPitchAngle := -RadToDeg(arctan2(FDirection.Y, Sqrt(Sqr(FDirection.X) + Sqr(FDirection.Z))));
  if FDirection.X < 0 then
    if FDirection.Y < 0 then FPitchAngle := 180 - FPitchAngle
                        else FPitchAngle := -180 - FPitchAngle;
  TransformationChanged;
end;

// SetPitchAngle
//
procedure TGLBaseSceneObject.SetPitchAngle(AValue: Single);
var
   RightVector: TAffineVector;
   Up, Dir: TAffineVector;
   Diff: Single;
   rotMatrix : TMatrix;
begin
   if AValue<>FPitchAngle then begin
      if not (csLoading in ComponentState) then begin
         GetOrientationVectors(Up, Dir);
         Diff := DegToRad(FPitchAngle - AValue);
         RightVector := VectorCrossProduct(Dir, Up);
         rotMatrix:=CreateRotationMatrix(RightVector, Diff);
         FUp.FCoords:=VectorTransform(FUp.FCoords, rotMatrix);
         NormalizeVector(FUp.FCoords);
         FDirection.FCoords:=VectorTransform(FDirection.FCoords, rotMatrix);
         NormalizeVector(FDirection.FCoords);
         if FTransMode = tmParentWithPos then
            FPosition.FCoords:=VectorTransform(FPosition.FCoords, rotMatrix);
         TransformationChanged;
      end;
      FPitchAngle := NormalizeAngle(AValue);
   end;
end;

//------------------------------------------------------------------------------

procedure TGLBaseSceneObject.Roll(Angle: Single);

var
  RightVector: TVector;
  Up, Dir: TAffineVector;

begin
  GetOrientationVectors(Up, Dir);
  Angle := DegToRad(Angle);
  VectorRotate(FUp.FCoords, Dir, Angle);
  NormalizeVector(FUp.FCoords);
  VectorRotate(FDirection.FCoords, Dir, Angle);
  NormalizeVector(FDirection.FCoords);
  if FTransMode = tmParentWithPos then VectorRotate(FPosition.FCoords, Dir, Angle);

  // calculate new rotation angle from vectors
  RightVector := VectorCrossProduct(FDirection.FCoords, FUp.FCoords);
  FRollAngle := -RadToDeg(arctan2(Rightvector[1], Sqrt(Sqr(RightVector[0]) + Sqr(RightVector[2]))));
  if RightVector[0] < 0 then
    if RightVector[1] < 0 then FRollAngle := 180 - FRollAngle
                          else FRollAngle := -180 - FRollAngle;
  TransformationChanged;
end;

// SetRollAngle
//
procedure TGLBaseSceneObject.SetRollAngle(AValue: Single);
var
   Up, Dir: TAffineVector;
   Diff: Single;
   rotMatrix : TMatrix;
begin
   if AValue <> FRollAngle then begin
      if not (csLoading in ComponentState) then begin
         GetOrientationVectors(Up, Dir);
         Diff := DegToRad(FRollAngle - AValue);
         rotMatrix:=CreateRotationMatrix(Dir, Diff);
         FUp.FCoords:=VectorTransform(FUp.FCoords, rotMatrix);
         NormalizeVector(FUp.FCoords);
         FDirection.FCoords:=VectorTransform(FDirection.FCoords, rotMatrix);
         NormalizeVector(FDirection.FCoords);
         if FTransMode = tmParentWithPos then
            FPosition.FCoords:=VectorTransform(FPosition.FCoords, rotMatrix);
         TransformationChanged;
      end;
      FRollAngle := NormalizeAngle(AValue);
  end;
end;

//------------------------------------------------------------------------------

procedure TGLBaseSceneObject.Slide(ADistance: Single);
var
   RightVector: TAffineVector;
   Up, Dir: TAffineVector;
begin
   if FTransMode = tmParentWithPos then
      GetOrientationVectors(Up, Dir)
   else begin
      SetVector(Up, FUp.FCoords);
      SetVector(Dir, FDirection.FCoords);
   end;
   RightVector := VectorCrossProduct(Dir, Up);
   NormalizeVector(RightVector);
   MakePoint(FPosition.FCoords,
             FPosition.X + ADistance * RightVector[0],
             FPosition.Y + ADistance * RightVector[1],
             FPosition.Z + ADistance * RightVector[2]);
   TransformationChanged;
end;

//------------------------------------------------------------------------------

procedure TGLBaseSceneObject.Turn(Angle: Single);

var
  Up, Dir: TAffineVector;

begin
  GetOrientationVectors(Up, Dir);
  Angle := DegToRad(Angle);
  VectorRotate(FUp.FCoords, Up, Angle);
  NormalizeVector(FUp.FCoords);
  VectorRotate(FDirection.FCoords, Up, Angle);
  NormalizeVector(FDirection.FCoords);
  if FTransMode = tmParentWithPos then VectorRotate(FPosition.FCoords, Up, Angle);
  FTurnAngle := -RadToDeg(arctan2(FDirection.X, Sqrt(Sqr(FDirection.Y) + Sqr(FDirection.Z))));
  if FDirection.X < 0 then
    if FDirection.Y < 0 then FTurnAngle := 180 - FTurnAngle
                        else FTurnAngle := -180 - FTurnAngle;
  TransformationChanged;
end;

// SetTurnAngle
//
procedure TGLBaseSceneObject.SetTurnAngle(AValue: Single);
var
   up, dir : TAffineVector;
   diff : Single;
   rotMatrix : TMatrix;
begin
   if AValue <> FTurnAngle then begin
      if not (csLoading in ComponentState) then begin
         GetOrientationVectors(Up, Dir);
         Diff := DegToRad(FTurnAngle - AValue);
         rotMatrix:=CreateRotationMatrix(Up, Diff);
         FUp.FCoords:=VectorTransform(FUp.FCoords, rotMatrix);
         NormalizeVector(FUp.FCoords);
         FDirection.FCoords:=VectorTransform(FDirection.FCoords, rotMatrix);
         NormalizeVector(FDirection.FCoords);
         if FTransMode = tmParentWithPos then
            FPosition.FCoords:=VectorTransform(FPosition.FCoords, rotMatrix);
         TransformationChanged;
      end;
      FTurnAngle := NormalizeAngle(AValue);
   end;
end;

// SetShowAxes
//
procedure TGLBaseSceneObject.SetShowAxes(AValue: Boolean);
begin
   if FShowAxes <> AValue then begin
      FShowAxes := AValue;
      NotifyChange;
   end;
end;

//------------------------------------------------------------------------------

procedure TGLBaseSceneObject.SetScaling(AValue: TGLCoordinates);
begin
   FScaling.Assign(AValue);
   TransformationChanged;
end;

// SetName
//
procedure TGLBaseSceneObject.SetName(const NewName: TComponentName);
begin
	if Name <> NewName then begin
		inherited SetName(NewName);
		if Assigned(vGLBaseSceneObjectNameChangeEvent) then
			vGLBaseSceneObjectNameChangeEvent(Self);
	end;
end;

// GetMatrix
//
function TGLBaseSceneObject.GetMatrix: TMatrix;
begin
	// update local matrix if necessary
	RebuildMatrix;
  	Result := FLocalMatrix;
end;

// SetParent
//
procedure TGLBaseSceneObject.SetParent(const val : TGLBaseSceneObject);
begin
   MoveTo(val);
end;

// SetIndex
//
procedure TGLBaseSceneObject.SetIndex(AValue: Integer);

var
  Count: Integer;
  AParent: TGLBaseSceneObject;

begin
  if assigned(FParent) then
  begin
    Count := FParent.Count;
    AParent := FParent;
    if AValue < 0 then AValue :=  0;
    if AValue >= Count then AValue :=  Count - 1;
    if AValue <> Index then
    begin
      if assigned(FScene) then FScene.BeginUpdate;
      FParent.Remove(Self, False);
      AParent.Insert(AValue, Self);
      if assigned(FScene) then FScene.EndUpdate;
    end;
  end;
end;

//------------------------------------------------------------------------------

procedure TGLBaseSceneObject.SetParentComponent(Value: TComponent);
begin
	if assigned(FParent) then begin
		FParent.Remove(Self, False);
		FParent := nil;
	end;
	// first level object?
	if Value is TGLScene then
		if Self is TGLCamera then
			TGLScene(Value).Cameras.AddChild(Self)
		else TGLScene(Value).Objects.AddChild(Self)
	else TGLBaseSceneObject(Value).AddChild(Self);  // normal parent-child relation
end;

//------------------------------------------------------------------------------

procedure TGLBaseSceneObject.StructureChanged;
begin
   Include(FChanges, ocStructure);
   NotifyChange;
end;

//------------------------------------------------------------------------------

procedure TGLBaseSceneObject.TransformationChanged;

begin
  FMatrixDirty := True;
  Include(FChanges, ocTransformation);
  NotifyChange;
end;

// MoveTo
//
procedure TGLBaseSceneObject.MoveTo(newParent : TGLBaseSceneObject);
begin
	if Assigned(FParent) then begin
		FParent.Remove(Self, False);
		FParent:=nil;
	end;
	if Assigned(newParent) then
		newParent.AddChild(Self)
	else SetScene(nil);
end;

// MoveUp
//
procedure TGLBaseSceneObject.MoveUp;
begin
   if Assigned(parent) then
      parent.MoveChildUp(parent.IndexOfChild(Self));
end;

// MoveDown
//
procedure TGLBaseSceneObject.MoveDown;
begin
   if Assigned(parent) then
      parent.MoveChildDown(parent.IndexOfChild(Self));
end;

//------------------------------------------------------------------------------

procedure TGLBaseSceneObject.EndUpdate;

begin
   if FUpdateCount > 0 then Dec(FUpdateCount);
   if FUpdateCount = 0 then NotifyChange;
end;

//------------------------------------------------------------------------------

procedure TGLBaseSceneObject.CoordinateChanged(Sender: TGLCoordinates);
var
   rightVector : TVector;
begin
  if FIsCalculating then Exit;
  FIsCalculating := True;
  try
      if Sender = FDirection then begin
         if VectorLength(FUp.FCoords) = 0 then
            FUp.FCoords := YHmgVector;
         NormalizeVector(FDirection.FCoords);
         // adjust up vector
         rightVector := VectorCrossProduct(FDirection.FCoords, FUp.FCoords);
         // Rightvector is zero if direction changed exactly by 90 degrees,
         // in this case assume a default vector
         if VectorLength(rightVector) = 0 then with FDirection do
            Geometry.SetVector(rightVector, X+1, Y+2, Z+3);
         FUp.FCoords:=VectorCrossProduct(RightVector, FDirection.FCoords);
         NormalizeVector(FUp.FCoords);
      end else if Sender = FUp then begin
         if VectorLength(FDirection.FCoords) = 0 then
            FDirection.FCoords := ZHmgVector;
         NormalizeVector(FUp.FCoords);
         // adjust up vector
         rightVector := VectorCrossProduct(FDirection.FCoords, FUp.FCoords);
         // Rightvector is zero if direction changed exactly by 90 degrees,
         // in this case assume a default vector
         if VectorLength(rightVector) = 0 then with FUp do
            Geometry.SetVector(rightVector, X+1, Y+2, Z+3);
         FDirection.FCoords:=VectorCrossProduct(FUp.FCoords, RightVector);
         NormalizeVector(FDirection.FCoords);
      end;
      TransformationChanged;
   finally
      FIsCalculating := False;
   end;
end;

// DoProgress
//
procedure TGLBaseSceneObject.DoProgress(const deltaTime, newTime : Double);
var
   i : Integer;
begin
	for i:=FChildren.Count-1 downto 0 do
		TGLBaseSceneObject(FChildren[i]).DoProgress(deltaTime, newTime);
   if Behaviours.Count>0 then
   	Behaviours.DoProgress(deltaTime, newTime);
   if Effects.Count>0 then
      Effects.DoProgress(deltaTime, newTime);
   if Assigned(FOnProgress) then
		FOnProgress(Self, deltaTime, newTime);
end;

// PrepareObject
//
procedure TGLBaseSceneObject.PrepareObject;
begin
	glLoadMatrixf(@FGLobalMatrix);
	if FScene.FCurrentViewer.State=dsPicking then
		glLoadName(Integer(Self));
end;

// FinishObject
//
procedure TGLBaseSceneObject.FinishObject;
begin
  	Exclude(FChanges, ocTransformation);
end;

// Remove
//
procedure TGLBaseSceneObject.Remove(AChild: TGLBaseSceneObject; KeepChildren: Boolean);
begin
	if Assigned(FScene) and (AChild is TGLLightSource) then
		FScene.RemoveLight(TGLLightSource(AChild));
	FChildren.Remove(AChild);
	AChild.FParent:=nil;
	if KeepChildren then begin
		BeginUpdate;
		with AChild do while Count>0 do
			Children[0].MoveTo(Self);
		EndUpdate;
	end else NotifyChange;
end;

// IndexOfChild
//
function TGLBaseSceneObject.IndexOfChild(AChild: TGLBaseSceneObject) : Integer;
begin
	Result:=FChildren.IndexOf(AChild);
end;

// MoveChildUp
//
procedure TGLBaseSceneObject.MoveChildUp(anIndex : Integer);
begin
   if anIndex>0 then begin
      FChildren.Exchange(anIndex, anIndex-1);
      NotifyChange;
   end;
end;

// MoveChildDown
//
procedure TGLBaseSceneObject.MoveChildDown(anIndex : Integer);
begin
   if anIndex<FChildren.Count-1 then begin
      FChildren.Exchange(anIndex, anIndex+1);
      NotifyChange;
   end;
end;

//------------------------------------------------------------------------------

procedure QuickSortObjectsByDist(startIndex, endIndex : Integer;
											distList, objList : TList);
var
	I, J : Integer;
	P : single;
begin
	if endIndex-startIndex>1 then begin
		repeat
			I:=startIndex; J:=endIndex;
			P:=Single(distList[(I + J) shr 1]);
			repeat
				while Single(distList[I])<P do Inc(I);
				while Single(distList[J])>P do Dec(J);
				if I <= J then begin
					distList.Exchange(i, J);
					objList.Exchange(i, j);
					Inc(I); Dec(J);
				end;
			until I > J;
			if startIndex < J then
				QuickSortObjectsByDist(startIndex, J, distList, objList);
			startIndex := I;
		until I >= endIndex;
	end else if endIndex-startIndex>0 then begin
		p:=Single(distList[startIndex]);
		if Single(distList[endIndex])<p then begin
			distList.Exchange(startIndex, endIndex);
			objList.Exchange(startIndex, endIndex);
		end;
	end;
end;

// Render
//
procedure TGLBaseSceneObject.Render(objectsSorting : TGLObjectsSorting;
												const cameraPosition : TVector;
												var currentStates : TGLStates);
var
   gotEffects : Boolean;
begin
	if FVisible then begin
   	PrepareObject;
		//if FScene.FCurrentViewer.ObjectInScene(Self) then
      gotEffects:=(Effects.Count>0);
      if gotEffects then
         Effects.RenderPreEffects(Scene.CurrentViewer, cameraPosition, currentStates);
      if osDirectDraw in ObjectStyle then
         BuildList
      else glCallList(Handle);
      if osDoesTemperWithColorsOrFaceWinding in ObjectStyle then begin
         ResetGLPolygonMode;
         ResetGLMaterialColors;
      end;
      if Count>0 then
         RenderChildren(0, Count-1, objectsSorting, cameraPosition, currentStates);
      if gotEffects then
         Effects.RenderPostEffects(Scene.CurrentViewer, cameraPosition, currentStates);
   	if FShowAxes then
	   	DrawAxes($CCCC);
      FinishObject;
   end;
end;

// RenderChildren
//
procedure TGLBaseSceneObject.RenderChildren(firstChildIndex, lastChildIndex : Integer;
      objectsSorting : TGLObjectsSorting; const cameraPosition : TVector;
		var currentStates : TGLStates);
var
	i : Integer;
	distList, objList : TList;
	obj : TGLBaseSceneObject;
begin
   if lastChildIndex=firstChildIndex then
      Get(firstChildIndex).Render(objectsSorting, cameraPosition, currentStates)
   else if lastChildIndex>firstChildIndex then begin
	   if Self.ObjectsSorting<>osInherited then
		   objectsSorting:=Self.ObjectsSorting;
	   case objectsSorting of
         osNone : begin
            for i:=firstChildIndex to lastChildIndex do
               Self[i].Render(objectsSorting, cameraPosition, currentStates);
         end;
         osRenderFarthestFirst, osRenderBlendedLast : begin
            distList:=TList.Create;
            objList:=TList.Create;
            try
               if objectsSorting=osRenderBlendedLast then begin
                  // render opaque stuff
                  for i:=firstChildIndex to lastChildIndex do begin
                     obj:=Get(i);
                     if (not (obj is TGLCustomSceneObject)) or (TGLCustomSceneObject(obj).Material.BlendingMode=bmOpaque) then
                        obj.Render(objectsSorting, cameraPosition, currentStates)
                     else begin
                        objList.Add(obj);
                        distList.Add(Pointer(obj.BarycenterSqrDistanceTo(cameraPosition)));
                     end;
                  end;
               end else for i:=firstChildIndex to lastChildIndex do begin
                  obj:=Get(i);
                  objList.Add(obj);
                  distList.Add(Pointer(obj.BarycenterSqrDistanceTo(cameraPosition)));
               end;
               if distList.Count>1 then
                  QuickSortObjectsByDist(0, distList.Count-1, distList, objList);
               for i:=objList.Count-1 downto 0 do
                  TGLBaseSceneObject(objList[i]).Render(objectsSorting, cameraPosition, currentStates);
            finally
               distList.Free;
               objList.Free;
            end;
         end;
      else
         Assert(False);
      end;
   end;
end;

// NotifyChange
//
procedure TGLBaseSceneObject.NotifyChange;
begin
   if Assigned(FScene) and (not IsUpdating) then
      FScene.NotifyChange;
end;

// ValidateTransformation
//
procedure TGLBaseSceneObject.ValidateTransformation;
var
   i : Integer;
begin
   // determine predecessor in transformation pipeline
   if (FParent = nil) then begin
      if (Scene.FLastGLCamera<>Scene.FCurrentGLCamera) or (ocTransformation in Scene.FCurrenTGLCamera.FChanges) then begin
         FGlobalMatrix:=Scene.CurrentGLCamera.FGLobalMatrix;
         Include(FChanges, ocTransformation);
      end;
   end else if ocTransformation in FChanges + FParent.FChanges then begin
      if ocTransformation in FChanges then RebuildMatrix;
      FGlobalMatrix:=MatrixMultiply(FLocalMatrix, FParent.FGlobalMatrix);
      Include(FChanges, ocTransformation);
   end;
   // validate for children
   for i:=0 to Count-1 do Self[i].ValidateTransformation;
   // all done
   Exclude(FChanges, ocTransformation);
end;

//------------------------------------------------------------------------------

procedure TGLBaseSceneObject.SetMatrix(AValue: TMatrix);

var
  Temp: TAffineVector;

begin
  FLocalMatrix := AValue;
  FDirection.FCoords := FLocalMatrix[2];
  FUp.FCoords := FLocalMatrix[1];
  SetVector(Temp, FLocalMatrix[0]);
  Scale.AsVector:=VectorMake(VectorLength(Temp),
                             VectorLength(FUp.FCoords),
                             VectorLength(FDirection.FCoords));
  FPosition.FCoords := FLocalMatrix[3];
  FMatrixDirty := False;
  Include(FChanges, ocTransformation);
  NotifyChange;
end;

procedure TGLBaseSceneObject.SetPosition(APosition: TGLCoordinates);
begin
   FPosition.FCoords := APosition.FCoords;
   TransformationChanged;
end;

procedure TGLBaseSceneObject.SetDirection(AVector: TGLCoordinates);
begin
   FDirection.FCoords := AVector.FCoords;
   TransformationChanged;
end;

procedure TGLBaseSceneObject.SetUp(AVector: TGLCoordinates);
begin
   FUp.FCoords := AVector.FCoords;
   TransformationChanged;
end;

procedure TGLBaseSceneObject.SetVisible(AValue: Boolean);
begin
   if FVisible <> AValue then begin
      FVisible := AValue;
      NotifyChange;
   end;
end;

// SetObjectsSorting
//
procedure TGLBaseSceneObject.SetObjectsSorting(const val : TGLObjectsSorting);
begin
	if FObjectsSorting<>val then begin
		FObjectsSorting:=val;
		NotifyChange;
	end;
end;

// SetBehaviours
//
procedure TGLBaseSceneObject.SetBehaviours(const val : TGLBehaviours);
begin
	FBehaviours.Assign(val);
end;

// SetEffects
//
procedure TGLBaseSceneObject.SetEffects(const val : TGLObjectEffects);
begin
   FObjectEffects.Assign(val);
end;

// SetScene
//
procedure TGLBaseSceneObject.SetScene(const value : TGLScene);
var
  i : Integer;
begin
	if value<>FScene then begin
		FScene:=value;
		for i:=0 to FChildren.Count-1 do
		Children[I].SetScene(FScene);
	end;
end;

// Translate
//
procedure TGLBaseSceneObject.Translate(tx, ty, tz : TGLFloat);
begin
	SetVector(FPosition.FCoords, Tx, Ty, Tz, FPosition.W);
end;

// ------------------
// ------------------ TGLBaseBehaviour ------------------
// ------------------

// Create
//
constructor TGLBaseBehaviour.Create(aOwner : TXCollection);
begin
	inherited Create(aOwner);
   // nothing more, yet
end;

// Destroy
//
destructor TGLBaseBehaviour.Destroy;
begin
   // nothing more, yet
	inherited Destroy;
end;

// SetName
//
procedure TGLBaseBehaviour.SetName(const val : String);
begin
   inherited SetName(val);
   if Assigned(vGLBehaviourNameChangeEvent) then
      vGLBehaviourNameChangeEvent(Self);
end;

// WriteToFiler
//
procedure TGLBaseBehaviour.WriteToFiler(writer : TWriter);
begin
   with writer do begin
      WriteInteger(0); // Archive Version 0
      // nothing more, yet
   end;
end;

// ReadFromFiler
//
procedure TGLBaseBehaviour.ReadFromFiler(reader : TReader);
begin
   with reader do begin
      Assert(ReadInteger=0);
      // nothing more, yet
   end;
end;

// OwnerBaseSceneObject
//
function TGLBaseBehaviour.OwnerBaseSceneObject : TGLBaseSceneObject;
begin
	Result:=TGLBaseSceneObject(Owner.Owner);
end;

// DoProgress
//
procedure TGLBaseBehaviour.DoProgress(const deltaTime, newTime : Double);
begin
	// does nothing
end;

// ------------------
// ------------------ TGLBehaviours ------------------
// ------------------

// Create
//
constructor TGLBehaviours.Create(aOwner : TPersistent);
begin
   Assert(aOwner is TGLBaseSceneObject);
	inherited Create(aOwner);
end;

// ItemsClass
//
class function TGLBehaviours.ItemsClass : TXCollectionItemClass;
begin
   Result:=TGLBehaviour;
end;

// GetBehaviour
//
function TGLBehaviours.GetBehaviour(index : Integer) : TGLBehaviour;
begin
	Result:=TGLBehaviour(Items[index]);
end;

// DoProgress
//
procedure TGLBehaviours.DoProgress(const deltaTime, newTime : Double);
var
	i : Integer;
begin
	for i:=0 to Count-1 do
		TGLBehaviour(Items[i]).DoProgress(deltaTime, newTime);
end;

//------------------ TGLObjectEffect -------------------------------------------

// WriteToFiler
//
procedure TGLObjectEffect.WriteToFiler(writer : TWriter);
begin
   with writer do begin
      WriteInteger(0); // Archive Version 0
      // nothing more, yet
   end;
end;

// ReadFromFiler
//
procedure TGLObjectEffect.ReadFromFiler(reader : TReader);
begin
   with reader do begin
      Assert(ReadInteger=0);
      // nothing more, yet
   end;
end;

// Render
//
procedure TGLObjectEffect.Render(sceneViewer : TGLSceneViewer;
                                 const cameraPosition : TVector;
      								   var currentStates : TGLStates);
begin
   // nothing here, this implem is just to avoid "abstract error"
end;

//------------------ TGLObjectEffects ------------------------------------------

// Create
//
constructor TGLObjectEffects.Create(aOwner : TPersistent);
begin
   Assert(aOwner is TGLBaseSceneObject);
	inherited Create(aOwner);
end;

// ItemsClass
//
class function TGLObjectEffects.ItemsClass : TXCollectionItemClass;
begin
   Result:=TGLObjectEffect;
end;

// GetEffect
//
function TGLObjectEffects.GetEffect(index : Integer) : TGLObjectEffect;
begin
	Result:=TGLObjectEffect(Items[index]);
end;

// DoProgress
//
procedure TGLObjectEffects.DoProgress(const deltaTime, newTime : Double);
var
	i : Integer;
begin
	for i:=0 to Count-1 do
		TGLBehaviour(Items[i]).DoProgress(deltaTime, newTime);
end;

// RenderPreEffects
//
procedure TGLObjectEffects.RenderPreEffects(sceneViewer : TGLSceneViewer;
                  const cameraPosition : TVector; var currentStates : TGLStates);
var
	i : Integer;
   effect : TGLObjectEffect;
begin
	for i:=0 to Count-1 do begin
      effect:=TGLObjectEffect(Items[i]);
      if effect is TGLObjectPreEffect then
   		effect.Render(sceneViewer, cameraPosition, currentStates);
   end;
end;

// RenderPostEffects
//
procedure TGLObjectEffects.RenderPostEffects(sceneViewer : TGLSceneViewer;
                  const cameraPosition : TVector; var currentStates : TGLStates);
var
	i : Integer;
   effect : TGLObjectEffect;
begin
	for i:=0 to Count-1 do begin
      effect:=TGLObjectEffect(Items[i]);
      if effect is TGLObjectPostEffect then
   		effect.Render(sceneViewer, cameraPosition, currentStates)
      else if Assigned(sceneViewer) and (effect is TGLObjectAfterEffect) then
         sceneViewer.afterRenderEffects.Add(effect);
   end;
end;

//------------------ TGLCustomSceneObject --------------------------------------

// Create
//
constructor TGLCustomSceneObject.Create(AOwner: TComponent);
begin
	inherited Create(AOwner);
   FHandle := 0;
   FMaterial := TGLMaterial.Create(Self);
end;

// Destroy
//
destructor TGLCustomSceneObject.Destroy;
begin
   FMaterial.Free;
   inherited Destroy;
end;

// Assign
//
procedure TGLCustomSceneObject.Assign(Source: TPersistent);
begin
	if Source is TGLCustomSceneObject then begin
		FMaterial.Assign(TGLCustomSceneObject(Source).FMaterial);
		FTagFloat:=TGLCustomSceneObject(Source).FTagFloat;
	end;
	inherited Assign(Source);
end;

// SetGLMaterial
//
procedure TGLCustomSceneObject.SetGLMaterial(AValue: TGLMaterial);
begin
   FMaterial.Assign(AValue);
   NotifyChange;
end;

// ReloadTexture
//
procedure TGLCustomSceneObject.ReloadTexture;
var
   i : Integer;
begin
  FMaterial.ReloadTexture;
  for I := 0 to Count - 1 do
    if Self[I] is TGLCustomSceneObject then TGLCustomSceneObject(Self[I]).ReloadTexture;
end;

// Render
//
procedure TGLCustomSceneObject.Render(objectsSorting : TGLObjectsSorting;
												  const cameraPosition : TVector;
												  var currentStates : TGLStates);
begin
   if Visible then begin
   	FMaterial.Apply(currentStates);
      inherited Render(objectsSorting, cameraPosition, currentStates);
   end;
end;

//----------------- TGLCamera ----------------------------------------------------

// Create
//
constructor TGLCamera.Create(aOwner : TComponent);
begin
   inherited Create(aOwner);
   FFocalLength := 50;
   FDepthOfView := 100;
   FDirection.Initialize(VectorMake(0, 0, -1, 0));
end;

// destroy
//
destructor TGLCamera.Destroy;
begin
   inherited;
end;

procedure TGLCamera.Apply;
var
	v, d : TVector;
	absPos : TVector;
begin
	if Assigned(FTargetObject) then begin
		// get our target's absolute coordinates
		v:=TargetObject.AbsolutePosition;
		d:=VectorSubstract(v, Position.FCoords);
		NormalizeVector(d);
		// check if target moved
		if (ocTransformation in FChanges) or (VectorSpacing(d, FLastTargetObjectPosition)>0) then begin
			absPos:=AbsolutePosition;
			FLastTargetObjectPosition:=d;
			// if direction changed, we need to call gluLookAt
			gluLookAt(absPos[0], absPos[1], absPos[2], //Position.X, Position.Y, Position.Z,
						 v[0], v[1], v[2],
						 Up.X, Up.Y, Up.Z);
			glGetFloatv(GL_MODELVIEW_MATRIX, @FGLobalMatrix);
		end;
	end else begin
		if ocTransformation in FChanges then begin
			gluLookAt(Position.X, Position.Y, Position.Z,
						 Position.X + Direction.X,
						 Position.Y + Direction.Y,
						 Position.Z + Direction.Z,
						 FUp.X, FUp.Y, FUp.Z);
			glGetFloatv(GL_MODELVIEW_MATRIX, @FGLobalMatrix);
		end;
	end;
end;

procedure TGLCamera.ApplyPerspective(Viewport: TRectangle; Width, Height: Integer; DPI: Integer);
var
	Left, Right, Top, Bottom, zFar, MaxDim, Ratio: Double;
begin
   // determine biggest dimension and resolution (height or width)
   MaxDim := Width;
   if Height > MaxDim then
      MaxDim := Height;

  // calculate near plane distance and extensions;
  // Scene ratio is determined by the window ratio. The viewport is just a
  // specific part of the entire window and has therefore no influence on the
  // scene ratio. What we need to know, though, is the ratio between the window
  // borders (left, top, right and bottom) and the viewport borders.
  // Note: viewport.top is actually bottom, because the window (and viewport) origin
  // in OGL is the lower left corner

  // calculate window/viewport ratio for right extent
  Ratio := (2 * Viewport.Width + 2 * Viewport.Left - Width) / Width;
  // calculate aspect ratio correct right value of the view frustum and take
  // the window/viewport ratio also into account
  Right := Ratio * Width / (2 * MaxDim);

  // the same goes here for the other three extents
  // left extent:
  Ratio := (Width - 2 * Viewport.Left) / Width;
  Left := -Ratio * Width / (2 * MaxDim);

  // top extent (keep in mind the origin is left lower corner):
  Ratio := (2 * Viewport.Height + 2 * Viewport.Top - Height) / Height;
  Top := Ratio * Height / (2 * MaxDim);

  // bottom extent:
  Ratio := (Height - 2 * Viewport.Top) / Height;
  Bottom := -Ratio * Height / (2 * MaxDim);

  FNearPlane := FFocalLength * 2 * DPI / (25.4 * MaxDim);
  zFar := FNearPlane + FDepthOfView;

  // finally create view frustum
  glFrustum(Left, Right, Bottom, Top, FNearPlane, zFar);
end;

//------------------------------------------------------------------------------

procedure TGLCamera.AutoLeveling(Factor: Single);
var
  rightVector, rotAxis: TVector;
  angle: Single;
begin
   angle := RadToDeg(arccos(VectorDotProduct(FUp.FCoords, YVector)));
   rotAxis := VectorCrossProduct(YHmgVector, FUp.FCoords);
   if (angle > 1) and (VectorLength(rotAxis) > 0) then begin
      rightVector := VectorCrossProduct(FDirection.FCoords, FUp.FCoords);
      VectorRotate(FUp.FCoords, AffineVectorMake(rotAxis), Angle / 10 / Factor);
      NormalizeVector(FUp.FCoords);
      // adjust local coordinates
      FDirection.FCoords := VectorCrossProduct(FUp.FCoords, rightVector);
      FRollAngle := -RadToDeg(arctan2(Rightvector[1], Sqrt(Sqr(RightVector[0]) + Sqr(RightVector[2]))));
      TransformationChanged;
   end;
end;

procedure TGLCamera.Notification(AComponent: TComponent; Operation: TOperation);
begin
   inherited;
   if (Operation = opRemove) and (AComponent = FTargetObject) then
      TargetObject:=nil;
end;

// SetTargetObject
//
procedure TGLCamera.SetTargetObject(const val : TGLBaseSceneObject);
begin
   if (FTargetObject<>val) then begin
      FTargetObject:=val;
      FLastTargetObjectPosition:=Direction.AsVector;
      if not (csLoading in Scene.ComponentState) then
         TransformationChanged;
   end;
end;

procedure TGLCamera.Reset;
var
   Extent: Single;
begin
	FRollAngle := 0;
   FFocalLength := 50;
   with FScene.CurrentViewer do begin
      glMatrixMode(GL_PROJECTION);
      glLoadIdentity;
      ApplyPerspective(FViewport, Width, Height, GetDeviceCaps(Canvas.Handle, LOGPIXELSX));
      FUp.FCoords := YHmgVector;
      if FViewport.Height<FViewport.Width then
         Extent:=FViewport.Height * 0.25
      else Extent:=FViewport.Width * 0.25;
	end;
   MakePoint(FPosition.FCoords, 0, 0, FNearPlane * Extent);
   MakeVector(FDirection.FCoords, 0, 0, -1);
   TransformationChanged;
end;

// ZoomAll
//
procedure TGLCamera.ZoomAll;
var
	extent: Single;
begin
	with Scene.CurrentViewer do begin
      if FViewport.Height<FViewport.Width then
         Extent:=FViewport.Height * 0.25
      else Extent:=FViewport.Width * 0.25;
		FPosition.FCoords := NullHmgPoint;
		Move(-FNearPlane * Extent);
		// let the camera look at the scene center
		MakeVector(FDirection.FCoords, -FPosition.X, -FPosition.Y, -FPosition.Z);
  	end;
end;

// MoveAroundTarget
//
procedure TGLCamera.MoveAroundTarget(pitchDelta, turnDelta : Single);
var
	vectAbsoluteFromTargetToCamera : TVector;
	vectLocalX, vectLocalZ, vectLocalY, vect, temp : TVector;
	mat : TMatrix;
	dist : Single;
begin
	if Assigned(FTargetObject) then begin
		// This stuff can probably be made shorter...
		//
		// calculate vector from target to camera in absolute coordinates
		vectAbsoluteFromTargetToCamera:=VectorSubstract(AbsolutePosition, TargetObject.AbsolutePosition);
		dist:=VectorLength(vectAbsoluteFromTargetToCamera);
		// calculate a local x vector (points right in camera's view)
		vectLocalX:=VectorCrossProduct(Up.AsVector, vectAbsoluteFromTargetToCamera);
		NormalizeVector(vectLocalX);
		// calculate local Z direction (points up)
		vectLocalZ:=VectorCrossProduct(vectAbsoluteFromTargetToCamera, vectLocalX);
		NormalizeVector(vectLocalZ);
		// complete with local Y (points to target)
		vectLocalY:=VectorCrossProduct(vectLocalZ, vectLocalX);
		// express new vector in local coordinates and rotate it
		MakeVector(vect, 0, -dist, 0);
		turnDelta:=turnDelta*PI/180;
		mat:=CreateRotationMatrixZ(Sin(turnDelta), Cos(turnDelta));
		vect:=VectorTransform(vect, mat);
		pitchDelta:=pitchDelta*PI/180;
		mat:=CreateRotationMatrixX(Sin(pitchDelta), Cos(pitchDelta));
		vect:=VectorTransform(vect, mat);
		// express in absolute coordinates
		temp:=vect;
		vect[0]:=vectLocalX[0]*temp[0]+vectLocalY[0]*temp[1]+vectLocalZ[0]*temp[2];
		vect[1]:=vectLocalX[1]*temp[0]+vectLocalY[1]*temp[1]+vectLocalZ[1]*temp[2];
		vect[2]:=vectLocalX[2]*temp[0]+vectLocalY[2]*temp[1]+vectLocalZ[2]*temp[2];
		// camera translation
		temp:=VectorSubstract(vect, vectAbsoluteFromTargetToCamera);
		Position.AsVector:=VectorAdd(Position.AsVector, temp);
	end;
end;

// AdjustDistanceToTarget
//
procedure TGLCamera.AdjustDistanceToTarget(distanceRatio : Single);
var
	vect : TVector;
begin
	if Assigned(FTargetObject) then begin
		// calculate vector from target to camera in absolute coordinates
		vect:=VectorSubstract(AbsolutePosition, TargetObject.AbsolutePosition);
		// ratio -> translation vector
		ScaleVector(vect, -(1-distanceRatio));
		Position.AsVector:=VectorAdd(Position.AsVector, vect);
	end;
end;

// DistanceToTarget
//
function TGLCamera.DistanceToTarget : Single;
var
	vect : TVector;
begin
	if Assigned(FTargetObject) then begin
		vect:=VectorSubstract(AbsolutePosition, TargetObject.AbsolutePosition);
		Result:=VectorLength(vect);
	end else Result:=1;
end;

// ScreenDeltaToVector
//
function TGLCamera.ScreenDeltaToVector(deltaX, deltaY : Integer; ratio : Single;
												 const planeNormal : TVector) : TVector;
var
	screenY, screenX : TVector;
	screenYoutOfPlaneComponent : Single;
begin
	// calculate projection of direction vector on the plane
	if Assigned(FTargetObject) then
		screenY:=VectorSubstract(TargetObject.AbsolutePosition, AbsolutePosition)
	else screenY:=Direction.AsVector;
	screenYoutOfPlaneComponent:=VectorDotProduct(screenY, planeNormal);
	screenY:=VectorCombine(screenY, planeNormal, 1, -screenYoutOfPlaneComponent);
	NormalizeVector(screenY);
	// calc the screenX vector
	screenX:=VectorCrossProduct(screenY, planeNormal);
	// and here, we're done
	Result:=VectorCombine(screenX, screenY, deltaX*ratio, deltaY*ratio);
end;

// ScreenDeltaToVectorXY
//
function TGLCamera.ScreenDeltaToVectorXY(deltaX, deltaY : Integer; ratio : Single) : TVector;
var
	screenY : TVector;
	dxr, dyr, d : Single;
begin
	// calculate projection of direction vector on the plane
	if Assigned(FTargetObject) then
		screenY:=VectorSubstract(TargetObject.AbsolutePosition, AbsolutePosition)
	else screenY:=Direction.AsVector;
	d:=VectorLength(screenY[0], screenY[1]);
	if d<=1e-10 then d:=ratio else d:=ratio/d;
	// and here, we're done
	dxr:=deltaX*d;
	dyr:=deltaY*d;
	Result[0]:=screenY[1]*dxr+screenY[0]*dyr;
	Result[1]:=screenY[1]*dyr-screenY[0]*dxr;
	Result[2]:=0;
	Result[3]:=0;
end;

// ScreenDeltaToVectorXZ
//
function TGLCamera.ScreenDeltaToVectorXZ(deltaX, deltaY : Integer; ratio : Single) : TVector;
var
   screenY : TVector;
   d, dxr, dzr : Single;
begin
   // calculate the projection of direction vector on the plane
   if Assigned(fTargetObject) then
      screenY:=VectorSubstract(TargetObject.AbsolutePosition, AbsolutePosition)
   else screenY:=Direction.AsVector;
	d:=VectorLength(screenY[0], screenY[2]);
   if d<=1e-10 then d:=ratio else d:=ratio/d;
   dxr:=deltaX*d;
   dzr:=deltaY*d;
   Result[0]:=-screenY[2]*dxr+screenY[0]*dzr;
   Result[1]:=0;
   Result[2]:=screenY[2]*dzr+screenY[0]*dxr;
   Result[3]:=0;
end;

// ScreenDeltaToVectorYZ
//
function TGLCamera.ScreenDeltaToVectorYZ(deltaX, deltaY : Integer; ratio : Single) : TVector;
var
   screenY : TVector;
   d, dyr, dzr : single;
begin
   // calculate the projection of direction vector on the plane
   if Assigned(fTargetObject) then
      screenY:=VectorSubstract(TargetObject.AbsolutePosition,AbsolutePosition)
   else screenY:=Direction.AsVector;
   d:=VectorLength(screenY[1], screenY[2]);
   if d<=1e-10 then d:=ratio else d:=ratio/d;
   dyr:=deltaX*d;
   dzr:=deltaY*d;
   Result[0]:=0;
   Result[1]:=screenY[2]*dyr+screenY[1]*dzr;
   Result[2]:=screenY[2]*dzr-screenY[1]*dyr;
   Result[3]:=0;
end;

// SetDepthOfView
//
procedure TGLCamera.SetDepthOfView(AValue: Single);
begin
  if FDepthOfView <> AValue then
  begin
	 FDepthOfView := AValue;
	 if not (csLoading in Scene.ComponentState) then
      TransformationChanged;
  end;
end;

// SetFocalLength
//
procedure TGLCamera.SetFocalLength(AValue: Single);
begin
   if AValue < 1 then AValue := 1;
   if FFocalLength <> AValue  then begin
	   FFocalLength := AValue;
      if not (csLoading in Scene.ComponentState) then
         TransformationChanged;
   end;
end;

// ------------------
// ------------------ TGLProxyObject ------------------
// ------------------

// Create
//
constructor TGLProxyObject.Create(AOwner: TComponent);
begin
   inherited;
   FProxyOptions:=cDefaultProxyOptions;
end;

// Destroy
//
destructor TGLProxyObject.Destroy;
begin
   SetMasterObject(nil);
   inherited;
end;

// Assign
//
procedure TGLProxyObject.Assign(Source: TPersistent);
begin
   if Source is TGLProxyObject then begin
      SetMasterObject(TGLProxyObject(Source).MasterObject);
   end;
   inherited Assign(Source);
end;

// Render
//
procedure TGLProxyObject.Render(objectsSorting : TGLObjectsSorting;
                 const cameraPosition : TVector;
                 var currentStates : TGLStates);
var
   gotMaster, selfGotEffects, masterGotEffects : Boolean;
//   f : Single;
begin
	if FVisible then begin
   	PrepareObject;
      selfGotEffects:=(Effects.Count>0);
      gotMaster:=Assigned(FMasterObject);
      masterGotEffects:=gotMaster and (pooEffects in FProxyOptions)
                        and (FMasterObject.Effects.Count>0);
      if selfGotEffects then
         Effects.RenderPreEffects(Scene.CurrentViewer, cameraPosition, currentStates);
      if gotMaster then begin
         if (FMasterObject is TGLCustomSceneObject) then
            TGLCustomSceneObject(FMasterObject).Material.Apply(currentStates);
         if masterGotEffects then
            FMasterObject.Effects.RenderPreEffects(Scene.CurrentViewer, cameraPosition, currentStates);
         if pooBuildList in FProxyOptions then begin
            if osDirectDraw in FMasterObject.ObjectStyle then
               FMasterObject.BuildList
            else glCallList(FMasterObject.Handle);
            if osDoesTemperWithColorsOrFaceWinding in FMasterObject.ObjectStyle then begin
               ResetGLPolygonMode;
               ResetGLMaterialColors;
            end;
         end;
         // experimental code for pooChildren
{         glMatrixMode(GL_PROJECTION);
         glPushMatrix;
         f:=1;
         glTranslatef(f*GlobalMatrix[3][0], f*GlobalMatrix[3][1], f*GlobalMatrix[3][2]);
//         glMultMatrixf(@FGLobalMatrix);
         glMatrixMode(GL_MODELVIEW);
         if pooChildren in FProxyOptions then
            FMasterObject.RenderChildren(0, Count-1, objectsSorting, cameraPosition, currentStates);
         glMatrixMode(GL_PROJECTION);
         glPopMatrix;
         glMatrixMode(GL_MODELVIEW);}
      end;
      PrepareObject;
		if Count>0 then
			RenderChildren(0, Count-1, objectsSorting, cameraPosition, currentStates);
		if masterGotEffects then
			FMasterObject.Effects.RenderPostEffects(Scene.CurrentViewer, cameraPosition, currentStates);
		if selfGotEffects then
			Effects.RenderPostEffects(Scene.CurrentViewer, cameraPosition, currentStates);
   	if FShowAxes then
	   	DrawAxes($CCCC);
		FinishObject;
	end;
end;

// Notification
//
procedure TGLProxyObject.Notification(AComponent: TComponent; Operation: TOperation);
begin
   if (Operation = opRemove) and (AComponent = FMasterObject) then
      MasterObject := nil;
   inherited;
end;

// SetMasterObject
//
procedure TGLProxyObject.SetMasterObject(const val : TGLBaseSceneObject);
begin
   if FMasterObject<>val then begin
		if Assigned(FMasterObject) then
			FMasterObject.RemoveFreeNotification(Self);
		FMasterObject:=val;
		if Assigned(FMasterObject) then
         FMasterObject.FreeNotification(Self);
      StructureChanged;
   end;
end;

// SetProxyOptions
//
procedure TGLProxyObject.SetProxyOptions(const val : TGLProxyObjectOptions);
begin
   if FProxyOptions<>val then begin
      FProxyOptions:=val;
      StructureChanged;
   end;
end;

// ------------------
// ------------------ TGLLightSource ------------------
// ------------------

// Create
//
constructor TGLLightSource.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FShining := True;
  FSpotDirection := TGLCoordinates.Create(Self);
  MakeVector(FSpotDirection.FCoords, 0, 0, -1);
  FConstAttenuation := 1;
  FLinearAttenuation := 0;
  FQuadraticAttenuation := 0;
  FSpotCutOff := 180;
  FSpotExponent := 0;
  FLightStyle:=lsSpot;
  FAmbient := TGLColor.Create(Self);
  FDiffuse := TGLColor.Create(Self);
  FDiffuse.Initialize(clrWhite);
  FSpecular := TGLColor.Create(Self);
  FVisible := False;
  FChanges := [];
end;

// Destroy
//
destructor TGLLightSource.Destroy;
begin
   FSpotDirection.Free;
   FAmbient.Free;
   FDiffuse.Free;
   FSpecular.Free;
   inherited Destroy;
end;

// DestroyList
//
procedure TGLLightSource.DestroyList;
begin
   if FShining then
      glDisable(FLightID);
end;

// CoordinateChanged
//
procedure TGLLightSource.CoordinateChanged(Sender: TGLCoordinates);
begin
   inherited;
   if Sender = FSpotDirection then
      Include(FChanges, ocSpot);
   TransformationChanged;
end;

// GetHandle
//
function TGLLightSource.GetHandle: TObjectHandle;
begin
   Result:=0;
end;

// SetShining
//
procedure TGLLightSource.SetShining(AValue: Boolean);
begin
   if AValue<>FShining then begin
      FShining := AValue;
      NotifyChange;
   end;
end;

// SetSpotDirection
//
procedure TGLLightSource.SetSpotDirection(AVector: TGLCoordinates);
begin
   FSpotDirection.FCoords := AVector.FCoords;
   Include(FChanges, ocSpot);
   NotifyChange;
end;

// SetSpotExponent
//
procedure TGLLightSource.SetSpotExponent(AValue: TGLFloat);
begin
   if FSpotExponent <> AValue then begin
      FSpotExponent := AValue;
      Include(FChanges, ocSpot);
      NotifyChange;
   end;
end;

// SetSpotCutOff
//
procedure TGLLightSource.SetSpotCutOff(AValue: TGLFloat);
begin
   if FSpotCutOff <> AValue then begin
      FSpotCutOff := AValue;
      Include(FChanges, ocSpot);
      NotifyChange;
   end;
end;

// SetLightStyle
//
procedure TGLLightSource.SetLightStyle(const val : TLightStyle);
begin
   if FLightStyle<>val then begin
      FLightStyle:=val;
      Include(FChanges, ocSpot);
      NotifyChange;
   end;
end;

// SetAmbient
//
procedure TGLLightSource.SetAmbient(AValue: TGLColor);
begin
   FAmbient.Color := AValue.Color;
   NotifyChange;
end;

// SetDiffuse
//
procedure TGLLightSource.SetDiffuse(AValue: TGLColor);
begin
   FDiffuse.Color := AValue.Color;
   NotifyChange;
end;

// SetSpecular
//
procedure TGLLightSource.SetSpecular(AValue: TGLColor);
begin
   FSpecular.Color := AValue.Color;
   NotifyChange;
end;

// SetConstAttenuation
//
procedure TGLLightSource.SetConstAttenuation(AValue: TGLFloat);
begin
   if FConstAttenuation <> AValue then begin
      FConstAttenuation := AValue;
      Include(FChanges, ocAttenuation);
      NotifyChange;
   end;
end;

// SetLinearAttenuation
//
procedure TGLLightSource.SetLinearAttenuation(AValue: TGLFloat);
begin
   if FLinearAttenuation <> AValue then begin
      FLinearAttenuation := AValue;
      Include(FChanges, ocAttenuation);
      NotifyChange;
   end;
end;

// SetQuadraticAttenuation
//
procedure TGLLightSource.SetQuadraticAttenuation(AValue: TGLFloat);
begin
   if FQuadraticAttenuation <> AValue then begin
      FQuadraticAttenuation := AValue;
      Include(FChanges, ocAttenuation);
      NotifyChange;
   end;
end;

//------------------------------------------------------------------------------

{procedure TGLLightSource.RenderLensFlares(from, at: TAffineVector; near_clip: TGLFloat);
const
  global_scale = 0.5;
  MinDot = 1e-20;
var
   view_dir, tmp, light_dir, pos, LightPos : TAffineVector;
   dx, dy, center, axis, sx, sy: TAffineVector;
   dot: Extended;
   I: Integer;
   NewFrom, NewAt: TAffineVector;
   LightColor: TAffineVector;
begin
   // determine current light position
   LightPos := MakeAffineVector([FGLobalMatrix[3, 0], FGLobalMatrix[3, 1], FGLobalMatrix[3, 2]]);
   // take out camera influence
   Newat := VectorAffineSubtract(at, from);
   Newfrom := NullVector;

   // view_dir = normalize(at-from)
   view_dir := VectorAffineSubtract(Newat, NewFrom);
   VectorNormalize(view_dir);

   // center = from + near_clip * view_dir
   tmp := view_dir;
   VectorScale(tmp, near_clip);
   center := VectorAffineAdd(Newfrom, tmp);

   // light_dir = normalize(light-from)
   light_dir := VectorAffineSubtract(LightPos, Newfrom);
   VectorNormalize(light_dir);

   // light = from + dot(light, view_dir) * near_clip * light_dir
   dot := VectorAffineDotProduct(light_dir, view_dir);
   tmp := light_dir;
   if Abs(Dot) < MinDot then
      if Dot < 0 then Dot := -MinDot else Dot := MinDot;
	VectorScale(tmp, near_clip / dot);
	LightPos := VectorAffineAdd(Newfrom, tmp);

	// axis = light - center
	axis := VectorAffineSubtract(LightPos, center);

	// dx = normalize(axis)
	dx := axis;
	VectorNormalize(dx);

	// dy = cross(dx, view_dir)
	dy := VectorCrossProduct(dx, view_dir);

	//glPushAttrib(GL_ENABLE_BIT or GL_CURRENT_BIT or GL_LIGHTING_BIT or GL_LINE_BIT
	//  or GL_COLOR_BUFFER_BIT or GL_TEXTURE_BIT);
	glPushAttrib(GL_ALL_ATTRIB_BITS);
	Scene.CurrentViewer.UnnecessaryState([stDepthTest, stLighting]);
	Scene.CurrentViewer.RequestedState([stBlend, sTGLTexture2D]);
	glBlendFunc(GL_ONE, GL_ONE);
	glLoadIdentity;

	for I := 0 to LensFlares.Count - 1 do with LensFlares do begin
		sx := dx;
		VectorScale(sx, Flare[I].Scale * global_scale*10);
		sy := dy;
		VectorScale(sy, Flare[I].Scale * global_scale*10);
		//sx:=MakeAffineVector(1, 0, 0);
		//sy:=MakeAffineVector(0, 1, 0);

      //glColor3fv(LensFlares.Flare[I].ColorAddr);
      LightColor := MakeAffineVector(Diffuse.Color);
      LightColor := VectorAffineAdd(LightColor, LensFlares.Flare[I].Color);;
      VectorScale(LightColor, 0.5);
      glColor3fv(@LightColor);
       glEnable(GL_TEXTURE_2D);;
      if Flare[I].FlareType < 0 then begin
         glBindTexture(GL_TEXTURE_2D, ShineTexture[FlareTic]);
         FlareTic := (FlareTic + 1) mod 10;
      end else glBindTexture(GL_TEXTURE_2D, FlareTexture[Flare[I].FlareType]);

      // position = center + flare[i].loc * axis
      tmp := axis;
      VectorScale(tmp, Flare[I].Location);
      Pos := VectorAffineAdd(center, tmp);

      glBegin(GL_QUADS);
        glTexCoord2f(0, 0);
        tmp := VectorAffineCombine3(Pos, sx, sy, 1, -1, -1);
        glVertex3fv(@tmp);

        glTexCoord2f(128, 0);
        tmp := VectorAffineCombine3(Pos, sx, sy, 1, 1, -1);
        glVertex3fv(@tmp);

        glTexCoord2f(128, 128);
        tmp := VectorAffineCombine3(Pos, sx, sy, 1, 1, 1);
        glVertex3fv(@tmp);

        glTexCoord2f(0, 128);
        tmp := VectorAffineCombine3(Pos, sx, sy, 1, -1, 1);
        glVertex3fv(@tmp);
      glEnd;
   end;
   Scene.CurrentViewer.RequestedState([stDepthTest, stLighting]);
   Scene.CurrentViewer.UnnecessaryState([stBlend, sTGLTexture2D]);
   glPopAttrib;
end;
}

// ValidateTransformation
//
procedure TGLLightSource.ValidateTransformation;
begin
   // check the predecessor and its transformation state
   if Assigned(FParent) then begin
      // has the object or its parent a new local/global matrix?
      if ocTransformation in FChanges + FParent.FChanges then begin
         FGlobalMatrix := MatrixMultiply(CreateTranslationMatrix(FPosition.FCoords), FParent.FGlobalMatrix);
         Include(FChanges, ocTransformation);
      end
   end else begin
      FGlobalMatrix := CreateTranslationMatrix(FPosition.FCoords);
      Include(FChanges, ocTransformation);
   end;
   // now let the children validate their matrices
   inherited ValidateTransformation;
end;

// ------------------
// ------------------ TGLScene ------------------
// ------------------

// Create
//
constructor TGLScene.Create(AOwner: TComponent);
begin
  inherited;
  // root creation
  FObjects:=TGLSceneRootObject.Create(Self);
  FObjects.Name:='ObjectRoot';
  FObjects.FScene:=Self;
  FCameras:=TGLBaseSceneObject.Create(Self);
  FCameras.Name:='CameraRoot';
  FCameras.FScene:=Self;
  FLights:=TList.Create;
  FObjectsSorting:=osRenderBlendedLast;
  // actual maximum number of lights is stored in TGLSceneViewer
  FLights.Count:=8;
end;

// Destroy
//
destructor TGLScene.Destroy;
begin
	FCameras.Free;
	FLights.Free;
	FObjects.Free;
	inherited Destroy;
end;

{$ifndef DFS_DELPHI_5_UP}
// Notification
//
procedure TGLScene.Notification(AComponent: TComponent; Operation: TOperation);
begin
	// nothing more, here, this is just a workaround the lack of a decent
	// 'RemoveFreeNotification' under Delphi 4
	inherited Notification(AComponent, Operation);
end;
{$endif}

// AddLight
//
procedure TGLScene.AddLight(ALight: TGLLightSource);
var
	i : Integer;
begin
   for i := 0 to FLights.Count - 1 do
		if FLights[i] = nil then begin
			FLights[i] := ALight;
         ALight.FLightID := GL_LIGHT0 + i;
         Break;
      end;
end;

//------------------------------------------------------------------------------

procedure TGLScene.AddViewer(AViewer: TGLSceneViewer);
begin
   if FViewers = nil then FViewers := TList.Create;
   if FViewers.IndexOf(AViewer) < 0 then FViewers.Add(AViewer);
   if FBaseContext = 0 then begin
      FBaseContext := TGLSceneViewer(FViewers[0]).RenderingContext;
      // the following initialization should be conditionally made to avoid adding
      // unnecessary overhead to the application
//      if spLensFlares in TGLSceneViewer(FViewers[0]).Specials then InitLensFlares;
//      if spLandScape in TGLSceneViewer(FViewers[0]).Specials then InitLandScape;
   end;
   if FViewers.Count > 1 then
      OpenGL12.wglShareLists(FBaseContext, AViewer.RenderingContext);
end;

// GetChildren
//
procedure TGLScene.GetChildren(AProc: TGetChildProc; Root: TComponent);
begin
   FObjects.GetChildren(AProc, Root);
   FCameras.GetChildren(AProc, Root);
end;

//------------------------------------------------------------------------------

procedure TGLScene.RemoveLight(ALight: TGLLightSource);

var LIndex: Integer;

begin
  LIndex := FLights.IndexOf(ALight);
  if LIndex > -1 then FLights[LIndex] := nil;
end;

//------------------------------------------------------------------------------

procedure TGLScene.SetChildOrder(AChild: TComponent; Order: Integer);

begin
  (AChild as TGLBaseSceneObject).Index := Order;
end;

//------------------------------------------------------------------------------

procedure TGLScene.Loaded;

begin
  inherited Loaded;
end;

// IsUpdating
//
function TGLScene.IsUpdating: Boolean;
begin
  Result := (FUpdateCount <> 0) or (csLoading in ComponentState) or (csDestroying in ComponentState);
end;

// BeginUpdate
//
procedure TGLScene.BeginUpdate;
begin
  Inc(FUpdateCount);
end;

// EndUpdate
//
procedure TGLScene.EndUpdate;
begin
   Assert(FUpdateCount>0);
   Dec(FUpdateCount);
   if FUpdateCount = 0 then NotifyChange;
end;

// SetObjectsSorting
//
procedure TGLScene.SetObjectsSorting(const val : TGLObjectsSorting);
begin
   if FObjectsSorting<>val then begin
      FObjectsSorting:=val;
      NotifyChange;
   end;
end;

// RenderScene
//
procedure TGLScene.RenderScene(AViewer: TGLSceneViewer);
var
   i : Integer;
   cameraPosition : TVector;
begin
   ResetGLPolygonMode;
   ResetGLMaterialColors;
   AViewer.afterRenderEffects.Clear;
   FCurrentViewer:=AViewer;
   cameraPosition:=AViewer.Camera.AbsolutePosition;
   FObjects.Render(FObjectsSorting, cameraPosition, AViewer.FCurrentStates);
   with AViewer.afterRenderEffects do if Count>0 then
      for i:=0 to Count-1 do
         TGLObjectAfterEffect(Items[i]).Render(AViewer, cameraPosition,
                                               AViewer.FCurrentStates);
end;

procedure TGLScene.RemoveViewer(AViewer: TGLSceneViewer);
begin
  if assigned(FViewers) then
  begin
    // we need a new base context to share with if the previous base context
    // is about to be destroyed
    if (FViewers.IndexOf(AViewer) = 0) and (FViewers.Count > 1) then
       FBaseContext := TGLSceneViewer(FViewers[1]).RenderingContext;
    // if AViewer is the last one in the list then remove other
    // shared stuff before (!) the viewer is deleted
{    if FViewers.Count = 1 then
    begin
      LensFlares.Free;
      LandScape.Free;
    end;}
    FViewers.Remove(AViewer);
    if FViewers.Count = 0 then
    begin
      FViewers.Free;
      FViewers := nil;
      FBaseContext := 0;
      if not (csDestroying in ComponentState) then
      begin
        FObjects.DestroyList;
        FObjects.ReloadTexture;
      end;  
    end;  
  end;
end;

//------------------------------------------------------------------------------

procedure TGLScene.ValidateTransformation(ACamera: TGLCamera);

begin
  FCurrentGLCamera := ACamera;
  ACamera.Apply;
  FObjects.ValidateTransformation;
  Exclude(ACamera.FChanges, ocTransformation);
  FLastGLCamera := FCurrentGLCamera;
end;

// Progress
//
procedure TGLScene.Progress(const deltaTime, newTime : Double);
begin
   FObjects.DoProgress(deltaTime, newTime);
   FCameras.DoProgress(deltaTime, newTime);
end;

// SaveToFile
//
procedure TGLScene.SaveToFile(const fileName : String);
begin
	WriteComponentResFile(fileName, Self);
end;

// LoadFromFile
//
procedure TGLScene.LoadFromFile(const fileName : String);
var
  i : Integer;
begin
   if Assigned(FViewers) then for i:=FViewers.Count-1 downto 0 do
      TGLSceneViewer(FViewers[i]).Camera:=nil;
	Cameras.DeleteChildren;
	Objects.DeleteChildren;
{ TODO : Camera & Targets are lost, some persistence upgraded is needed here }
	ReadComponentResFile(fileName, Self);
end;

//------------------------------------------------------------------------------

procedure TGLScene.NotifyChange;
var
   i : Integer;
begin
   if (not IsUpdating) and assigned(FViewers) then
      for i:=0 to FViewers.Count-1 do
         TGLSceneViewer(FViewers[i]).Invalidate;
end;

//------------------------------------------------------------------------------

procedure TGLScene.SetupLights(Maximum: Integer);
var
   I: Integer;
   LS: TGLLightSource;
   Max: Integer;
   v : TVector;
begin
   // start searching through all light sources
   if Maximum < FLights.Count then
      Max := Maximum
   else Max := FLights.Count;
   for I := 0 to Max - 1 do begin
      LS := TGLLightSource(FLights[I]);
		if Assigned(LS) then with LS do begin
         if Shining then begin
            glEnable(FLightID);
            glLoadMatrixf(@FGlobalMatrix);
            if LightStyle=lsParallel then begin
               v:=FSpotDirection.FCoords;
               v[3]:=0;
               glLightfv(FLightID, GL_POSITION, @v);
            end else glLightfv(FLightID, GL_POSITION, Position.AsAddress);
            with FAmbient  do glLightfv(FLightID, GL_AMBIENT, AsAddress);
            with FDiffuse  do glLightfv(FLightID, GL_DIFFUSE, AsAddress);
            with FSpecular do glLightfv(FLightID, GL_SPECULAR, AsAddress);
            if ocSpot in FChanges then begin
               case LightStyle of
                  lsSpot : begin
                     if FSpotCutOff<>180 then begin
                        glLightfv(FLightID, GL_SPOT_DIRECTION, @FSpotDirection.FCoords);
                        glLightfv(FLightID, GL_SPOT_EXPONENT, @FSpotExponent);
                     end;
                     glLightfv(FLightID, GL_SPOT_CUTOFF, @FSpotCutOff);
                  end;
                  lsOmni :
                     glLightf(FLightID, GL_SPOT_CUTOFF, 180);
               end;
               Exclude(FChanges, ocSpot);
            end;
            if ocAttenuation in FChanges then begin
               glLightfv(FLightID, GL_CONSTANT_ATTENUATION, @FConstAttenuation);
               glLightfv(FLightID, GL_LINEAR_ATTENUATION, @FLinearAttenuation);
               glLightfv(FLightID, GL_QUADRATIC_ATTENUATION, @FQuadraticAttenuation);
               Exclude(FChanges, ocAttenuation);
            end;
         end else glDisable(FLightID);
      end else glDisable(GL_LIGHT0+I);
   end;
end;

//------------------------------------------------------------------------------

procedure TGLScene.DoAfterRender;
{var
   i : Integer;
   light : TGLLightSource;}
begin
{   for I := 0 to FLights.Count-1 do begin
      light:=TGLLightSource(FLights[I]);
      if Assigned(light) and light.Shining then
         light.RenderLensFlares(MakeAffineVector(CurrenTGLCamera.Position.FCoords),
                                MakeAffineVector(CurrenTGLCamera.FDirection.FCoords),
                                CurrentViewer.FCamera.FNearPlane);
   end;}
end;

//------------------ TGLFogEnvironment ------------------------------------------------

// Note: The fog implementation is not conformal with the rest of the scene management
//       because it is viewer bound not scene bound.

constructor TGLFogEnvironment.Create(Owner : TPersistent);
begin
   inherited;
   FSceneViewer := (Owner as TGLSceneViewer);
   FFogColor :=  TGLColor.Create(Self);
	FFogColor.Initialize(clrBlack);
   FFogMode :=  fmLinear;
   FFogStart :=  10;
   FFogEnd :=  1000;
end;

//------------------------------------------------------------------------------

destructor TGLFogEnvironment.Destroy;

begin
  FFogColor.Free;
  inherited Destroy;
end;

//------------------------------------------------------------------------------

procedure TGLFogEnvironment.NotifyChange;
begin
   FChanged := True;
   FSceneViewer.Invalidate;
end;

procedure TGLFogEnvironment.SetFogColor(Value: TGLColor);
begin
   if Assigned(Value) then begin
      FFogColor.Assign(Value);
      NotifyChange;
   end;
end;

procedure TGLFogEnvironment.SetFogStart(Value: TGLfloat);
begin
   if Value <> FFogStart then begin
      FFogStart :=  Value;
      NotifyChange;
   end;
end;

//------------------------------------------------------------------------------

procedure TGLFogEnvironment.SetFogEnd(Value: TGLfloat);
begin
   if Value <> FFogEnd then begin
      FFogEnd :=  Value;
      NotifyChange;
   end;
end;

//------------------------------------------------------------------------------

procedure TGLFogEnvironment.Assign(Source: TPersistent);

begin
  if Source is TGLFogEnvironment then
  begin
    FFogColor.Assign(TGLFogEnvironment(Source).FFogColor);
    FFogStart :=  TGLFogEnvironment(Source).FFogStart;
    FFogEnd :=  TGLFogEnvironment(Source).FFogEnd;
    FFogMode :=  TGLFogEnvironment(Source).FFogMode;
    FChanged := True;
  end
  else
    inherited Assign(Source);
end;

//------------------------------------------------------------------------------

procedure TGLFogEnvironment.SetFogMode(Value: TFogMode);
begin
   if Value <> FFogMode then begin
      FFogMode :=  Value;
      NotifyChange;
   end;
end;

// ApplyFog
//
procedure TGLFogEnvironment.ApplyFog;
begin
   if FChanged then with FSceneViewer do begin
      if FRenderingContext > 0 then begin
         ActivateRenderingContext(FCanvas.Handle, FRenderingContext);
         try
            case FFogMode of
               fmLinear : glFogi(GL_FOG_MODE, GL_LINEAR);
               fmExp : begin
                  glFogi(GL_FOG_MODE, GL_EXP);
                  glFogf(GL_FOG_DENSITY, FFogColor.Alpha);
               end;
               fmExp2 : begin
                  glFogi(GL_FOG_MODE, GL_EXP2);
                  glFogf(GL_FOG_DENSITY, FFogColor.Alpha);
               end;
            end;
            glFogfv(GL_FOG_COLOR, FFogColor.AsAddress);
            glFogf(GL_FOG_START, FFogStart);
            glFogf(GL_FOG_END, FFogEnd);
         finally
            DeactivateRenderingContext;
            FChanged:=False;
         end;
      end;
   end;
end;

//------------------ TGLSceneViewer --------------------------------------------------

constructor TGLSceneViewer.Create(AOwner: TComponent);
begin
  InitOpenGL;
  inherited Create(AOwner);
  ControlStyle := [csClickEvents, csDoubleClicks, csOpaque, csCaptureMouse];
  if csDesigning in ComponentState then ControlStyle := ControlStyle + [csFramed];
  FCanvas := TCanvas.Create;
  Width:=100;
  Height:=100;
  FDisplayOptions := TDisplayOptions.Create;
  FBackground := TGLTexture.Create(nil);

  // initialize private state variables
  FFogEnvironment :=  TGLFogEnvironment.Create(Self);
  FBackgroundColor :=  clBtnFace;
  FDepthTest :=  True;
  FFaceCulling :=  True;
  FLighting :=  True;
  FFogEnable :=  False;
  afterRenderEffects:=TList.Create;

  FContextOptions := [roDoubleBuffer, roRenderToWindow];

  // performance check off
  FMonitor := False;
  ResetPerformanceMonitor;
  FState := dsNone;
end;

//------------------------------------------------------------------------------
 
destructor TGLSceneViewer.Destroy;
begin
   afterRenderEffects.Free;
   // clean up and terminate
   if assigned (FCamera) and assigned (FCamera.FScene) then begin
      FCamera.FScene.RemoveViewer(Self);
      FCamera := nil;
   end;
   FBackground.Free;
   FDisplayOptions.Free;
   DestroyHandle;
   FFogEnvironment.free;
   //FLandScapeOption.free;
   FCanvas.Free;
   inherited Destroy;
end;

//------------------------------------------------------------------------------

procedure TGLSceneViewer.CreateParams(var Params: TCreateParams);

begin
  inherited CreateParams(Params);
  with Params do
  begin
    if (not (csDesigning in ComponentState) and (woDesktop in FDisplayOptions.WindowAttributes)) or
       (not assigned(Parent) and (ParentWindow = 0)) then
    begin
      WndParent := 0;
      Style := WS_POPUP or WS_VISIBLE;
    end;
    //else Style := WS_CHILD;
    Style := Style or WS_CLIPCHILDREN or WS_CLIPSIBLINGS;
    if woTransparent in FDisplayOptions.WindowAttributes then
         ExStyle := ExStyle or WS_EX_TRANSPARENT;
    WindowClass.Style :=  CS_VREDRAW or CS_HREDRAW; // or CS_OWNDC;
  end;
end;

//------------------------------------------------------------------------------

function TGLSceneViewer.ObjectInScene(Obj: TGLBaseSceneObject): Boolean;

var
  ModelMatrix: THomogeneousDblMatrix;
  ProjectMatrix: THomogeneousDblMatrix;
  VP: THomogeneousIntVector;
  WinX, WinY, WinZ: Double;
  R: TRect;
  P: TPoint;
  
begin
  Result :=  True;
  glGetDoublev(GL_MODELVIEW_MATRIX, @ModelMatrix);
  glGetDoublev(GL_PROJECTION_MATRIX, @ProjectMatrix);
  glGetIntegerv(GL_VIEWPORT, @VP);
  gluProject(Obj.Position.X, Obj.Position.Y, Obj.Position.Z, ModelMatrix, ProjectMatrix, VP, @WinX, @WinY, @WinZ);
  R :=  Rect(Vp[0], Vp[1], Vp[2], Vp[3]);
  P.x :=  Round(WinX);
  P.y :=  Round(WinY);
  if (not PtInRect(R, P)) then
    Result :=  False;
end;

//------------------------------------------------------------------------------

procedure TGLSceneViewer.ReadContextProperties;
begin
  FMaxLightSources := LimitOf[limLights];
  FDoubleBuffered := LimitOf[limDoubleBuffer] > 0;
  if glIsEnabled(GL_DEPTH_TEST) > 0 then Include(FCurrentStates, stDepthTest);
  if glIsEnabled(GL_CULL_FACE) > 0 then Include(FCurrentStates, stCullFace);
  if glIsEnabled(GL_LIGHTING) > 0 then Include(FCurrentStates, stLighting);
  if glIsEnabled(GL_FOG) > 0 then Include(FCurrentStates, stFog);
end;

// SetupRenderingContext
//
procedure TGLSceneViewer.SetupRenderingContext;
var
   ColorDepth: Cardinal;
   NewStates: TGLStates;
begin
   ColorDepth := GetDeviceCaps(Canvas.Handle, BITSPIXEL) * GetDeviceCaps(Canvas.Handle, PLANES);
   if roTwoSideLighting in FContextOptions then
      glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE)
   else glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_FALSE);
   NewStates := [stNormalize];
   if DepthTest then Include(NewStates, stDepthTest);
   if FaceCulling then Include(NewStates, stCullFace);
   if Lighting then Include(NewStates, stLighting);
   if FogEnable then Include(NewStates, stFog);
   if ColorDepth < 24 then
      Include(NewStates, stDither)
   else Exclude(NewStates, stDither);
   RequestedState(NewStates);
   glDepthFunc(GL_LESS);
   glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);
end;

//------------------------------------------------------------------------------

function TGLSceneViewer.GetLimit(Which: TLimitType): Integer;

// determine the limit of the given kind from OpenGL implementation

var
  VP: array[0..1] of Double;

begin
  case Which of
    limClipPlanes:
      glGetIntegerv(GL_MAX_CLIP_PLANES, @Result);
    limEvalOrder:
      glGetIntegerv(GL_MAX_EVAL_ORDER, @Result);
    limLights:
      glGetIntegerv(GL_MAX_LIGHTS, @Result);
    limListNesting:
      glGetIntegerv(GL_MAX_LIST_NESTING, @Result);
    limModelViewStack:
      glGetIntegerv(GL_MAX_MODELVIEW_STACK_DEPTH, @Result);
    limNameStack:
      glGetIntegerv(GL_MAX_NAME_STACK_DEPTH, @Result);
    limPixelMapTable:
      glGetIntegerv(GL_MAX_PIXEL_MAP_TABLE, @Result);
    limProjectionStack:
      glGetIntegerv(GL_MAX_PROJECTION_STACK_DEPTH, @Result);
    limTextureSize:
      glGetIntegerv(GL_MAX_TEXTURE_SIZE, @Result);
    limTextureStack:
      glGetIntegerv(GL_MAX_TEXTURE_STACK_DEPTH, @Result);
    limViewportDims:
      begin
        glGetDoublev(GL_MAX_VIEWPORT_DIMS, @VP);
        if VP[0]>VP[1] then Result:=Round(VP[0]) else Result:=Round(VP[1]);
      end;
    limAccumAlphaBits:
      glGetIntegerv(GL_ACCUM_ALPHA_BITS, @Result);
    limAccumBlueBits:
      glGetIntegerv(GL_ACCUM_BLUE_BITS, @Result);
    limAccumGreenBits:
      glGetIntegerv(GL_ACCUM_GREEN_BITS, @Result);
    limAccumRedBits:
      glGetIntegerv(GL_ACCUM_RED_BITS, @Result);
    limAlphaBits:
      glGetIntegerv(GL_ALPHA_BITS, @Result);
    limAuxBuffers:
      glGetIntegerv(GL_AUX_BUFFERS, @Result);
    limDepthBits:
      glGetIntegerv(GL_DEPTH_BITS, @Result);
    limStencilBits:
      glGetIntegerv(GL_STENCIL_BITS, @Result);
    limBlueBits:
      glGetIntegerv(GL_BLUE_BITS, @Result);
    limGreenBits:
      glGetIntegerv(GL_GREEN_BITS, @Result);
    limRedBits:
      glGetIntegerv(GL_RED_BITS, @Result);
    limIndexBits:
      glGetIntegerv(GL_INDEX_BITS, @Result);
    limStereo:
      glGetIntegerv(GL_STEREO, @Result);
    limDoubleBuffer:
      glGetIntegerv(GL_DOUBLEBUFFER, @Result);
    limSubpixelBits:
      glGetIntegerv(GL_SUBPIXEL_BITS, @Result);
  else
    Result := 0;
  end;
end;

//------------------------------------------------------------------------------

procedure TGLSceneViewer.Loaded;

var
  NewMode: Integer;

begin
  inherited Loaded;
  if not (csDesigning in ComponentState) then
    // set display mode depending on the different screen options
    // full screen requested?
    if DisplayOptions.FullScreen then
    begin
      // full screen mode, so check for window fitting
      case DisplayOptions.WindowFitting of
        wfFitWindowToScreen: // set screen to the specified size
          begin
            NewMode := DisplayOptions.ScreenResolution;
            if NewMode <> 0 then SetFullScreenMode(NewMode);
          end;
        wfFitScreenToWindow: // adjust screen size to window size
          begin
            NewMode := GetIndexFromResolution(Width, Height, VideoModes[0].ColorDepth);
            SetFullScreenMode(NewMode);
          end;
      end;
      Left := 0;
      Top := 0;
      ShowWindow(Handle, SW_SHOWMAXIMIZED);
    end
    else
      // no full screen mode for the application, but perhaps
      // a specific resolution or color depth?
      if DisplayOptions.ScreenResolution <> 0 then
        SetFullScreenMode(DisplayOptions.ScreenResolution);
  // initiate window creation
  HandleNeeded;
  if assigned(FCamera) and assigned(FCamera.FScene) then FCamera.FScene.AddViewer(Self);
end;

//------------------------------------------------------------------------------

procedure TGLSceneViewer.WMEraseBkgnd(var Message: TWMEraseBkgnd);
begin
  Message.Result := 1;
end;

//------------------------------------------------------------------------------

procedure TGLSceneViewer.WMSize(var Message: TWMSize);
//var
//   aPoint : TPoint;
begin
	inherited;
	{ !!!!!!!!!!!!!! Disabled !!!!!!!!!!!!!!!
	if (woTransparent in DisplayOptions.WindowAttributes) then begin
		aPoint.X := Left;
		aPoint.Y := Top;
		if (not (woDesktop in DisplayOptions.WindowAttributes))
				or (csDesigning in ComponentState) then
			aPoint := Parent.ClientToScreen(aPoint);
		with FBackground.Image do begin
			TGLCaptureImage(FBackground.Image).Left := aPoint.X;
			TGLCaptureImage(FBackground.Image).Top := aPoint.Y;
			Width := RoundUpToPowerOf2(Message.Width);
			Height := RoundUpToPowerOf2(Message.Height);
		end;
	end; }
	// define viewport
	if FRenderingContext <> 0 then begin
		ActivateRenderingContext(FCanvas.Handle, FRenderingContext);
      try
         with FViewPort do begin
            Width := Message.Width;
            Height := Message.Height;
				if Height = 0 then Height := 1;
				glViewport(0, 0, Width, Height);
			end;
		finally
			DeactivateRenderingContext;
		end;
	end;
end;

// WMPaint
//
procedure TGLSceneViewer.WMPaint(var Message: TWMPaint);
var
   PS : TPaintStruct;
begin
   invalidated:=False;
	BeginPaint(Handle, PS);
	try
		Render;
	finally
		EndPaint(Handle, PS);
		Message.Result:=0;
   end;
end;

// RenderToFile
//
procedure TGLSceneViewer.RenderToFile(const AFile: String; DPI: Integer);
var
  ABitmap: TBitmap;
  SaveDialog: TSavePictureDialog;
  SaveAllowed: Boolean;
  FName: String;
begin
	Assert((FState = dsNone), glsAlreadyRendering);
   SaveDialog := nil;
   ABitmap := TBitmap.Create;
   try
      ABitmap.Width := Width;
      ABitmap.Height := Height;
      ABitmap.PixelFormat := pf24Bit;
      RenderToBitmap(ABitmap, DPI);
      FName := AFile;
      SaveAllowed := True;
      if FName = '' then begin
         SaveDialog := TSavePictureDialog.Create(Application);
         with SaveDialog do begin
            Options := [ofHideReadOnly, ofNoReadOnlyReturn];
            SaveAllowed := Execute;
         end;
      end;
      if SaveAllowed then begin
         if FName = '' then begin
            FName := SaveDialog.FileName;
            if (FileExists(SaveDialog.FileName)) then
               SaveAllowed := MessageDlg(Format('Overwrite file %s?', [SaveDialog.FileName]), mtConfirmation, [mbYes, mbNo], 0) = mrYes;
         end;
         if SaveAllowed then ABitmap.SaveToFile(FName);
      end;
   finally
      SaveDialog.Free;
      ABitmap.Free;
   end;
end;

// RenderToFile
//
procedure TGLSceneViewer.RenderToFile(const AFile: String; bmpWidth, bmpHeight : Integer);
var
   ABitmap: TBitmap;
   SaveDialog: TSavePictureDialog;
   SaveAllowed: Boolean;
   FName: String;
begin
	Assert((FState = dsNone), glsAlreadyRendering);
   SaveDialog := nil;
   ABitmap := TBitmap.Create;
   try
      ABitmap.Width := bmpWidth;
      ABitmap.Height := bmpHeight;
      ABitmap.PixelFormat := pf24Bit;
      RenderToBitmap(ABitmap, (GetDeviceCaps(ABitmap.Canvas.Handle, LOGPIXELSX)*bmpWidth) div width);
      FName := AFile;
      SaveAllowed := True;
      if FName = '' then begin
         SaveDialog := TSavePictureDialog.Create(Application);
         with SaveDialog do begin
            Options := [ofHideReadOnly, ofNoReadOnlyReturn];
            SaveAllowed := Execute;
         end;
      end;
      if SaveAllowed then begin
         if FName = '' then begin
            FName := SaveDialog.FileName;
            if (FileExists(SaveDialog.FileName)) then
               SaveAllowed := MessageDlg(Format('Overwrite file %s?', [SaveDialog.FileName]), mtConfirmation, [mbYes, mbNo], 0) = mrYes;
         end;
         if SaveAllowed then ABitmap.SaveToFile(FName);
      end;
   finally
      SaveDialog.Free;
      ABitmap.Free;
   end;
end;

// Invalidate
//
procedure TGLSceneViewer.Invalidate;
begin
   if not invalidated then begin
	   inherited Invalidate;
      invalidated:=True;
   end;
end;

// SetViewPort
//
procedure TGLSceneViewer.SetViewPort(X, Y, W, H: Integer);
begin
   with FViewPort do begin
      Left := X;
      Top := Y;
      Width := W;
      Height := H;
   end;
   Perform(WM_SIZE, SIZE_RESTORED, MakeLong(Width, Height));
   if not (csReading in ComponentState) then Invalidate;
end;

// SetupStates
//
procedure SetupStates(NeededStates: TGLStates);
begin
   { TODO : better and faster version }
   if stAlphaTest      in NeededStates then
     glEnable(GL_ALPHA_TEST);
   if stAutoNormal     in NeededStates then
     glEnable(GL_AUTO_NORMAL);
   if stBlend          in NeededStates then
     glEnable(GL_BLEND);
   if stColorMaterial  in NeededStates then
     glEnable(GL_COLOR_MATERIAL);
   if stCullFace       in NeededStates then
     glEnable(GL_CULL_FACE);
   if stDepthTest      in NeededStates then
     glEnable(GL_DEPTH_TEST);
   if stDither         in NeededStates then
     glEnable(GL_DITHER);
   if stFog            in NeededStates then
     glEnable(GL_FOG);
   if stLighting       in NeededStates then
     glEnable(GL_LIGHTING);
   if stLineSmooth     in NeededStates then
     glEnable(GL_LINE_SMOOTH);
   if stLineStipple    in NeededStates then
     glEnable(GL_LINE_STIPPLE);
   if stLogicOp        in NeededStates then
     glEnable(GL_LOGIC_OP);
   if stNormalize      in NeededStates then
     glEnable(GL_NORMALIZE);
   if stPointSmooth    in NeededStates then
     glEnable(GL_POINT_SMOOTH);
   if stPolygonSmooth  in NeededStates then
     glEnable(GL_POLYGON_SMOOTH);
   if stPolygonStipple in NeededStates then
     glEnable(GL_POLYGON_STIPPLE);
   if stScissorTest    in NeededStates then
     glEnable(GL_SCISSOR_TEST);
   if stStencilTest    in NeededStates then
     glEnable(GL_STENCIL_TEST);
   if stTexture1D      in NeededStates then
     glEnable(GL_TEXTURE_1D);
   if stTexture2D      in NeededStates then
     glEnable(GL_TEXTURE_2D);
end;

// RenderToBitmap
//
procedure TGLSceneViewer.RenderToBitmap(ABitmap: TBitmap; DPI: Integer);
var
   BitmapContext: HGLRC;
   BackColor: TColorVector;
   ColorBits: Integer;
   Viewport: TRectangle;
   LastStates: TGLStates;
   Resolution: Integer;
begin
	Assert((FState = dsNone), glsAlreadyRendering);
   FState := dsPrinting;
   try
      case ABitmap.PixelFormat of
         pfCustom, pfDevice :  // use current color depth
            ColorBits := VideoModes[CurrentVideoMode].ColorDepth;
         pf1bit, pf4bit : // OpenGL needs at least 4 bits
            ColorBits := 4;
         pf8bit : ColorBits := 8;
         pf15bit : ColorBits := 15;
         pf16bit : ColorBits := 16;
         pf24bit : ColorBits := 24;
         pf32bit : ColorBits := 32;
      else
         ColorBits := 24;
      end;
      BitmapContext := CreateRenderingContext(ABitmap.Canvas.Handle, [], ColorBits, 0, 0, 0, 0);
      Assert(BitmapContext<>0);
      try
         // save current window context states
         lastStates := FCurrentStates;
         ActivateRenderingContext(ABitmap.Canvas.Handle, BitmapContext);
         try
            SetupStates(FCurrentStates);
            BackColor := ConvertWinColor(FBackgroundColor);
            glClearColor(BackColor[0], BackColor[1], BackColor[2], BackColor[3]);
//            if (woTransparent in DisplayOptions.WindowAttributes) then
//               FBackground.ReloadImage;
            // set the desired viewport and limit output to this rectangle
            with Viewport do begin
               Left := 0;
               Top := 0;
               Width := ABitmap.Width;
               Height := ABitmap.Height;
               glViewport(Left, Top, Width, Height);
            end;
            ClearBuffers;
            glMatrixMode(GL_PROJECTION);
            glLoadIdentity;
            ResetGLPolygonMode;
            ResetGLMaterialColors;
            Resolution:=DPI;
            if Resolution=0 then
               Resolution:=GetDeviceCaps(ABitmap.Canvas.Handle, LOGPIXELSX);
            FCamera.ApplyPerspective(Viewport, ABitmap.Width, ABitmap.Height, Resolution);
            glMatrixMode(GL_MODELVIEW);
            glLoadIdentity;
            // start rendering
            if Assigned(FBeforeRender) then FBeforeRender(Self);
            if Assigned(FCamera) and Assigned(FCamera.FScene) then with FCamera.FScene do begin
               ValidateTransformation(Camera);
               Objects.ReloadTexture;
               Objects.DestroyList;
               SetupLights(FMaxLightSources);
               FogEnvironment.ApplyFog;
               RenderScene(Self);
               if Assigned(FPostRender) then FPostRender(Self);
               Objects.DestroyList;
            end;
            glFinish;
         finally
            DeactivateRenderingContext;
            FCurrentStates := LastStates;
         end;
      finally
         DestroyRenderingContext(BitmapContext);
      end;
   finally
      FState := dsNone;
   end;
   if Assigned(FAfterRender) then FAfterRender(Self);
end;

// RequestedState
//
procedure TGLSceneViewer.RequestedState(States: TGLStates);
var
   neededStates: TGLStates;
begin
   // create window and rendering context if not yet done
   HandleNeeded;
   // get all states, which are requested but not yet set
   NeededStates := States - FCurrentStates;
   if NeededStates <> [] then begin
      SetupStates(NeededStates);
      FCurrentStates := FCurrentStates + NeededStates;
   end;
end;

// ShowInfo
//
procedure TGLSceneViewer.ShowInfo;
var
   infoForm: TInfoForm;
begin
   Application.CreateForm(TInfoForm, infoForm);
   try
      ActivateRenderingContext(Canvas.Handle, FRenderingContext);
      // most info is available with active context only
      try
         infoForm.GetInfoFrom(Self);
      finally
         DeactivateRenderingContext;
      end;
      infoForm.ShowModal;
   finally
      infoForm.Free;
   end;
end;

// UnnecessaryState
//
procedure TGLSceneViewer.UnnecessaryState(States: TGLStates);
var
   takeOutStates: TGLStates;
begin
   { TODO : Better and faster version }
  // create window and rendering context if not yet done
  HandleNeeded;
  // get all states, which are to be taken out, but still set
  TakeOutStates := States * FCurrentStates;
  if TakeOutStates <> [] then begin
    // now reset all these states
    if stAlphaTest      in TakeOutStates then
      glDisable(GL_ALPHA_TEST);
    if stAutoNormal     in TakeOutStates then
      glDisable(GL_AUTO_NORMAL);
    if stBlend          in TakeOutStates then
      glDisable(GL_BLEND);
    if stColorMaterial  in TakeOutStates then
      glDisable(GL_COLOR_MATERIAL);
    if stCullFace       in TakeOutStates then
      glDisable(GL_CULL_FACE);
    if stDepthTest      in TakeOutStates then
      glDisable(GL_DEPTH_TEST);
    if stDither         in TakeOutStates then
      glDisable(GL_DITHER);
    if stFog            in TakeOutStates then
      glDisable(GL_FOG);
    if stLighting       in TakeOutStates then
      glDisable(GL_LIGHTING);
    if stLineSmooth     in TakeOutStates then
      glDisable(GL_LINE_SMOOTH);
    if stLineStipple    in TakeOutStates then
      glDisable(GL_LINE_STIPPLE);
    if stLogicOp        in TakeOutStates then
      glDisable(GL_LOGIC_OP);
    if stNormalize      in TakeOutStates then
      glDisable(GL_NORMALIZE);
    if stPointSmooth    in TakeOutStates then
      glDisable(GL_POINT_SMOOTH);
    if stPolygonSmooth  in TakeOutStates then
      glDisable(GL_POLYGON_SMOOTH);
    if stPolygonStipple in TakeOutStates then
      glDisable(GL_POLYGON_STIPPLE);
    if stScissorTest    in TakeOutStates then
      glDisable(GL_SCISSOR_TEST);
    if stStencilTest    in TakeOutStates then
      glDisable(GL_STENCIL_TEST);
	 if stTexture1D      in TakeOutStates then
		glDisable(GL_TEXTURE_1D);
    if stTexture2D      in TakeOutStates then
      glDisable(GL_TEXTURE_2D);
    FCurrentStates := FCurrentStates - TakeOutStates;
  end;
end;

// ResetPerformanceMonitor
//
procedure TGLSceneViewer.ResetPerformanceMonitor;
begin
   FFramesPerSecond := 0;
   FFrames := 0;
   FTicks := 0;
end;

// ClearBuffers
//
procedure TGLSceneViewer.ClearBuffers;
type
   PPixelArray  = ^TByteVector;
var
   bufferBits : TGLBitfield;
begin
   // handle transparency simulation
{   if (woTransparent in DisplayOptions.WindowAttributes) then begin
     glPushAttrib(GL_ENABLE_BIT);
     glEnable(GL_TEXTURE_2D);
     glDisable(GL_LIGHTING);
     glDisable(GL_DITHER);
     glDisable(GL_DEPTH_TEST);
     glDisable(GL_BLEND);
     // Invalidate initiated by the scene itself?
     FBackground.DisableAutoTexture;
     FBackground.Apply;
     glMatrixMode(GL_MODELVIEW);
     glPushMatrix;
     glLoadIdentity;
     glMatrixMode(GL_PROJECTION);
     glPushMatrix;
     glLoadIdentity;
     glOrtho(0, Width - 1, 0, Height - 1, 0, 100);
     glFrontFace(GL_CCW);
     glBegin(GL_QUADS);
       glTexCoord2f(0, 1 - Height / FBackground.Image.Height);
       glVertex3f(0, 0, 0);

       glTexCoord2f(Width / FBackground.Image.Width, 1 - Height / FBackground.Image.Height);
       glVertex3f(Width - 1, 0, 0);

       glTexCoord2f(Width/FBackground.Image.Width, 1);
       glVertex3f(Width - 1, Height - 1, 0);

       glTexCoord2f(0, 1);
       glVertex3f(0, Height - 1, 0);
     glEnd;
     glMatrixMode(GL_MODELVIEW);
     glPopMatrix;
     glMatrixMode(GL_PROJECTION);
     glPopMatrix;
     glPopAttrib;
   end;}

   // Convert our buXXX in Buffers to the GL boolean set
   if (buColor in Buffers) then //and not (woTransparent in DisplayOptions.WindowAttributes) then
      bufferBits:=GL_COLOR_BUFFER_BIT
   else bufferBits:=0;
   if buDepth in Buffers then
      bufferBits:=bufferBits or GL_DEPTH_BUFFER_BIT;
   if buStencil in Buffers then
      bufferBits:=bufferBits or GL_STENCIL_BUFFER_BIT;
   if buAccum in Buffers then
      bufferBits:=bufferBits or GL_ACCUM_BUFFER_BIT;
   if bufferBits<>0 then glClear(BufferBits);
end;

// Notification
//
procedure TGLSceneViewer.Notification(AComponent: TComponent; Operation: TOperation);
begin
   inherited;
   if (Operation = opRemove) and (AComponent = FCamera) then
      Camera := nil;
end;

// PickObjects
//
procedure TGLSceneViewer.PickObjects(const Rect: TRect; PickList: TGLPickList;
												 objectCountGuess: Integer);
var
   buffer : PCardinalVector;
   hits : Integer;
   i : Integer;
   current, next : Cardinal;
   szmin, szmax : Single;
begin
	Assert((FState = dsNone), glsAlreadyRendering);
   Assert(Assigned(PickList));
   ActivateRenderingContext(FCanvas.Handle, FRenderingContext);
   FState := dsPicking;
   try
      glMatrixMode(GL_PROJECTION);
      glPushMatrix;
      buffer := nil;
      try
			glLoadIdentity;
			gluPickMatrix(Rect.Left, Height - Rect.Top,
							  Abs(Rect.Right - Rect.Left), Abs(Rect.Bottom - Rect.Top),
							  TVector4i(FViewport));
			FCamera.ApplyPerspective(FViewport, Width, Height,
											 GetDeviceCaps(Canvas.Handle, LOGPIXELSX));
			// check countguess, memory waste is not an issue here
         if objectCountGuess<8 then objectCountGuess:=8;
         hits:=-1;
         repeat
				if hits < 0 then begin
               // Allocate 4 integers per row (Egg : dunno why 4)
               // Add 32 integers of slop (an extra cache line) to end for buggy
					// hardware that uses DMA to return select results but that sometimes
               // overrun the buffer.  Yuck.
               ReallocMem(buffer, objectCountGuess * 4 * SizeOf(Integer) + 32 * 4);
               // increase buffer by 50% if we get nothing
               Inc(objectCountGuess, objectCountGuess shr 1);
            end;
            // pass buffer to opengl and prepare render
            glSelectBuffer(objectCountGuess*4, @Buffer^);
            glRenderMode(GL_SELECT);
				glInitNames;
            glPushName(0);
            glMatrixMode(GL_MODELVIEW);
            glLoadIdentity;
            // render the scene (in select mode, nothing is drawn)
            if Assigned(FCamera) and Assigned(FCamera.FScene) then
               with FCamera.FScene do begin
                  ValidateTransformation(Camera);
                  RenderScene(Self);
               end;
            glFlush;
            Hits := glRenderMode(GL_RENDER);
			until Hits > -1; // try again with larger selection buffer
         Next := 0;
			PickList.Clear;
         PickList.Capacity := Hits;
         for I := 0 to Hits-1 do begin
            Current := Next;
            Next := Current + Buffer[Current] + 3;
            szmin := (Buffer[current + 1] shr 1) / MaxInt;
            szmax := (Buffer[current + 2] shr 1) / MaxInt;
				PickList.AddHit(TGLCustomSceneObject(Buffer[Current + 3]), szmin, szmax);
         end;
      finally
         FreeMem(Buffer);
         glMatrixMode(GL_PROJECTION);
         glPopMatrix;
      end;
   finally
      FState := dsNone;
      DeactivateRenderingContext;
   end;
end;

// GetPickedObjects
//
function TGLSceneViewer.GetPickedObjects(const Rect: TRect; objectCountGuess : Integer = 64) : TGLPickList;
begin
	Result:=TGLPickList.Create(psMinDepth);
	PickObjects(Rect, Result, objectCountGuess);
end;

// GetPickedObject
//
function TGLSceneViewer.GetPickedObject(x, y : Integer) : TGLBaseSceneObject;
var
	pkList : TGLPickList;
begin
	pkList:=GetPickedObjects(Rect(x-1, y-1, x+1, y+1));
	try
		if pkList.Count>0 then
			Result:=pkList.Hit[0]
		else Result:=nil;
	finally
		pkList.Free;
	end;
end;

// Render
//
procedure TGLSceneViewer.Render;
var
   Counter1, Counter2: TLargeInteger;
begin
	if (not visible) or (FState<>dsNone) then Exit;
   ActivateRenderingContext(FCanvas.Handle, FRenderingContext);
   FState := dsRendering;
   try
      // performance data demanded?
      if FMonitor then QueryPerformanceCounter(Counter1);
      // clear the buffers
      ClearBuffers;
      // setup rendering stuff
      glMatrixMode(GL_PROJECTION);
      glLoadIdentity;
      if assigned(FCamera) then
         FCamera.ApplyPerspective(FViewport, Width, Height,
                                  GetDeviceCaps(Canvas.Handle, LOGPIXELSX));
      glMatrixMode(GL_MODELVIEW);
      glLoadIdentity;
      ResetGLPolygonMode;
      ResetGLMaterialColors;
      // rendering
      if Assigned(FBeforeRender) then FBeforeRender(Self);
      if Assigned(FCamera) and Assigned(FCamera.FScene) then with FCamera.FScene do begin
         ValidateTransformation(FCamera);
         SetupLights(FMaxLightSources);
         FogEnvironment.ApplyFog;
         RenderScene(Self);
         if Assigned(FPostRender) then FPostRender(Self);
      end;
      glFlush;
      if FDoubleBuffered then SwapBuffers(Canvas.Handle);
      {$ifdef DEBUG} CheckOpenGLError; {$endif}

      // performance data demanded?
      if FMonitor then begin
         // yes, calculate average frames per second...
         Inc(FFrames);
         if FFrames > 1 then begin // ...but leave out the very first frame
            QueryPerformanceCounter(Counter2);
            // in second run take an 'average' value for the first run into account
            // by simply using twice the time from this run
            if FFrames = 2 then
               FTicks := FTicks + 2 * (Counter2 - Counter1)
            else FTicks := FTicks + Counter2 - Counter1;
            if FTicks > 0 then
					FFramesPerSecond := FFrames * vCounterFrequency / FTicks;
         end;
      end;
   finally
      FState := dsNone;
      DeactivateRenderingContext;
   end;
   if Assigned(FAfterRender) then FAfterRender(Self);
end;

// CreateWnd
//
procedure TGLSceneViewer.CreateWnd;
var
   BackColor: TColorVector;
   Options: TRCOptions;
begin
	inherited CreateWnd;
	FCanvas.Handle := GetDC(Handle);
	// initialize and activate the OpenGL rendering context
	// need to do this only once per window creation as we have a private DC
	FState := dsRendering;
	try
		Options := [];
		if roDoubleBuffer in ContextOptions then Include(Options, opDoubleBuffered);
		// will be freed in DestroyWnd
		FRenderingContext := CreateRenderingContext(Canvas.Handle, Options, 32, 0, 0, 0, 0);
		Assert(FRenderingContext<>0);
		ActivateRenderingContext(Canvas.Handle, FRenderingContext);
		try
			// this one should not be replaced with an assert
			if not GL_VERSION_1_1 then
				raise EOpenGLError.Create(glsWrongVersion);
			FBuffers := [buColor, buDepth];
			// define viewport, this is necessary because the first WM_SIZE message
			// is posted before the rendering context has been created
			with FViewPort do begin
				Left := 0;
				Top := 0;
				Width := Self.Width;
				Height := Self.Height;
            glViewport(0, 0, Width, Height);
         end;
         // set up initial context states
         if FSaveStates <> [] then begin
            // might be a recreated window, so reset all states which where
            // activated when the window was destroyed
            SetupStates(FSaveStates);
            FCurrentStates := FSaveStates;
            FSaveStates := [];
         end else begin
            ReadContextProperties;
            SetupRenderingContext;
         end;
         BackColor := ConvertWinColor(FBackgroundColor);
         glClearColor(BackColor[0], BackColor[1], BackColor[2], BackColor[3]);
      finally
         DeactivateRenderingContext;
         if woStayOnTop in DisplayOptions.WindowAttributes then
				SetWindowPos(Handle, HWND_TOPMOST, 0, 0, 0, 0,
                         SWP_NOCOPYBITS or SWP_NOMOVE or SWP_NOSIZE);
      end;
   finally
      FState := dsNone;
   end;
   invalidated:=False;
end;

// DestroyWnd
//
procedure TGLSceneViewer.DestroyWnd;
begin
   FSaveStates := FCurrentStates;
   FBackground.DestroyHandle;
   DestroyRenderingContext(FRenderingContext);
   FRenderingContext := 0;
   inherited DestroyWnd;
end;

// SetBackgroundColor
//
procedure TGLSceneViewer.SetBackgroundColor(AColor: TColor);
var
   backColor: TColorVector;
begin
   if FBackgroundColor <> AColor then begin
      FBackgroundColor := AColor;
      if not (csReading in ComponentState) then begin
         backColor := ConvertWinColor(FBackgroundColor);
         ActivateRenderingContext(FCanvas.Handle, FRenderingContext);
         try
            glClearColor(BackColor[0], BackColor[1], BackColor[2], BackColor[3]);
         finally
            DeactivateRenderingContext;
         end;
         Invalidate;
      end;
   end;
end;

// SetGLCamera
//
procedure TGLSceneViewer.SetGLCamera(ACamera: TGLCamera);
begin
   if FCamera <> ACamera then begin
      if Assigned(FCamera) then begin
         if Assigned(FCamera.FScene) then
            FCamera.FScene.RemoveViewer(Self);
         FCamera := nil;
      end;
      if Assigned(ACamera) and Assigned(ACamera.FScene) then begin
         FCamera := ACamera;
         Include(FCamera.FChanges, ocTransformation);
         if not (csLoading in ComponentState) then begin
            RecreateWnd;
            HandleNeeded;
            ACamera.FScene.AddViewer(Self);
         end;
      end;
      Invalidate;
   end;
end;

// SetContextOptions
//
procedure TGLSceneViewer.SetContextOptions(Options: TContextOptions);
begin
   if FContextOptions<>Options then begin
      FContextOptions:=Options;
      Invalidate;
   end;
end;

// SetDepthTest
//
procedure TGLSceneViewer.SetDepthTest(AValue: Boolean);
begin
   if FDepthTest<>AValue then begin
      FDepthTest:=AValue;
      if not (csReading in ComponentState) then begin
         ActivateRenderingContext(FCanvas.Handle, FRenderingContext);
         try
            if AValue then
               RequestedState([stDepthTest])
            else UnnecessaryState([stDepthTest]);
         finally
            DeactivateRenderingContext;
         end;
         Invalidate;
      end;
  end;
end;

//------------------------------------------------------------------------------

procedure TGLSceneViewer.SetFaceCulling(AValue: Boolean);
begin
   if FFaceCulling <> AValue then begin
      FFaceCulling := AValue;
      if not (csReading in ComponentState) then begin
         ActivateRenderingContext(FCanvas.Handle, FRenderingContext);
         try
            if AValue then
               RequestedState([stCullFace])
            else UnnecessaryState([stCullFace]);
         finally
            DeactivateRenderingContext;
         end;
         Invalidate;
      end;
   end;
end;

//------------------------------------------------------------------------------

procedure TGLSceneViewer.SetLighting(AValue: Boolean);
begin
   if FLighting <> AValue then begin
      FLighting := AValue;
      if not (csReading in ComponentState) then begin
         ActivateRenderingContext(FCanvas.Handle, FRenderingContext);
         try
            if AValue then
               RequestedState([stLighting])
            else UnnecessaryState([stLighting]);
         finally
            DeactivateRenderingContext;
         end;
         Invalidate;
      end;
   end;
end;

//------------------------------------------------------------------------------

procedure TGLSceneViewer.SetFogEnable(AValue: Boolean);
begin
   if FFogEnable <> AValue then begin
      FFogEnable := AValue;
      if not (csReading in ComponentState) then begin
         ActivateRenderingContext(FCanvas.Handle, FRenderingContext);
         try
            FFogEnvironment.NotifyChange;
            if AValue then
               RequestedState([stFog])
            else UnnecessaryState([stFog]);
         finally
            DeactivateRenderingContext;
         end;
         Invalidate;
      end;
   end;
end;

//------------------------------------------------------------------------------

{procedure TGLSceneViewer.SetSpecials(Value: TSpecials);
begin
   if FSpecials <> Value then begin
      FSpecials :=  Value;
      Invalidate;
   end;
end; }

// SetGLFogEnvironment
//
procedure TGLSceneViewer.SetGLFogEnvironment(AValue: TGLFogEnvironment);
begin
   ActivateRenderingContext(FCanvas.Handle, FRenderingContext);
   try
      FFogEnvironment.Assign(AValue);
      FFogEnvironment.NotifyChange;
   finally
      DeactivateRenderingContext;
   end;
   Invalidate;
end;

//------------------------------------------------------------------------------

initialization

	RegisterClasses([TGLLightSource, TGLCamera, TGLProxyObject]);

	// preparation for high resolution timer
	if not QueryPerformanceFrequency(vCounterFrequency) then
		vCounterFrequency := 0;

end.
