// GLMesh
{: Core classes for Mesh support in GLScene<p>

	<b>Historique : </b><font size=-1><ul>
	   <li>18/06/00 - Egg - Creation from split of GLObjects,
                           TVertexList now uses TVertexData,
                           Rewrite of TMesh.CalcNormals (smaller & faster)
	</ul></font>
}
unit GLMesh;

interface

uses Classes, GLMisc, GLScene, Geometry, OpenGL12, GLTexture;

type

   TVertexData = packed record
      textCoord : TTexPoint;
      color : TVector;
      normal : TAffineVector;
      coord : TVertex;
   end;
   PVertexData = ^TVertexData;
   TVertexDataArray = array [0..(MAXINT shr 6)] of TVertexData;
   PVertexDataArray = ^TVertexDataArray;

	// TVertexList
	//
	TVertexList = class(TGLUpdateAbleObject)
		private
			{ Private Declarations }
			FValues : PVertexDataArray;
			FCount  : Integer;
			FOwnerScene : TGLSceneObject;
         FCapacity, FGrowth : Integer;

      protected
			{ Protected Declarations }
         procedure DefineProperties(Filer: TFiler); override;

//         procedure ReadItems(Reader: TReader);
//         procedure WriteItems(Writer: TWriter);

         procedure SetCapacity(const val : Integer);
         procedure SetGrowth(const val : Integer);
         procedure Grow;

         function GetFirstEntry: PGLFloat;
         function GetFirstColor: PGLFLoat;
         function GetFirstNormal: PGLFLoat;
         function GetFirstVertex: PGLFLoat;

      public
			{ Public Declarations }
         constructor Create(AOwner: TPersistent); override;
         destructor Destroy; override;

         {: Adds a vertex to the list, fastest method. }
         procedure AddVertex(const vertexData : TVertexData); overload;
         {: Adds a vertex to the list, fastest method for adding a triangle. }
         procedure AddVertex3(const vd1, vd2, vd3 : TVertexData); overload;
         {: Adds a vertex to the list.<p>
            Use the NullVector, NullHmgVector or NullTexPoint constants for
            params you don't want to set. }
         procedure AddVertex(const aVertex : TVertex; const aNormal : TAffineVector;
                             const aColor: TColorVector; const aTexPoint: TTexPoint); overload;
         {: Adds a vertex to the list, no texturing ccord version.<p> }
         procedure AddVertex(const vertex: TVertex; const normal: TAffineVector;
                             const color: TColorVector); overload;
         procedure Assign(Source: TPersistent); override;
         procedure Clear;

         property Count : Integer read FCount;
         {: Capacity of the list (nb of vertex).<p>
            Use this to allocate memory quickly before calling AddVertex. }
         property Capacity : Integer read FCapacity write SetCapacity;
         {: Vertex capacity that will be added each time the list needs to grow.<p>
            default value is 256 (chunks of approx 13 kb). }
         property Growth : Integer read FGrowth write SetGrowth;

         property FirstColor: PGLFloat read GetFirstColor;
         property FirstEntry: PGLFLoat read GetFirstEntry;
         property FirstNormal: PGLFloat read GetFirstNormal;
         property FirstVertex: PGLFloat read GetFirstVertex;
   end;

   // TMesh
   //
   {: Basic mesh object.<p>
      Each mesh holds a set of vertices and a Mode value defines how they make
      up the mesh (triangles, strips...) }
   TMesh = class(TGLSceneObject)
      private
			{ Private Declarations }
         FVertices   : TVertexList;
         FMode       : TMeshMode;
         FVertexMode : TVertexMode;

      protected
			{ Protected Declarations }
         procedure SetMode(AValue: TMeshMode);
         procedure SetVertices(AValue: TVertexList);
         procedure SetVertexMode(AValue: TVertexMode);

      public
			{ Public Declarations }
         constructor Create(AOwner: TComponent); override;
         destructor Destroy; override;
         procedure Assign(Source: TPersistent); override;
         procedure BuildList; override;
         procedure CalcNormals(Frontface: TFaceWinding);
         property Vertices: TVertexList read FVertices write SetVertices;

      published
			{ Published Declarations }
         property Mode: TMeshMode read FMode write SetMode;
         property VertexMode: TVertexMode read FVertexMode write SetVertexMode default vmVNCT;
   end;

// ------------------------------------------------------------------
// ------------------------------------------------------------------
// ------------------------------------------------------------------
implementation
// ------------------------------------------------------------------
// ------------------------------------------------------------------
// ------------------------------------------------------------------

uses SysUtils, GLStrings;

//----------------- TVertexList ------------------------------------------------

constructor TVertexList.Create(AOwner: TPersistent);
begin
   inherited;
   FOwnerScene := (AOwner as TGLSceneObject);
   FValues := nil;
   FCount := 0;
   FCapacity := 0;
   FGrowth := 256;
end;

// Destroy
//
destructor TVertexList.Destroy;
begin
   FreeMem(FValues);
   inherited;
end;

// SetCapacity
//
procedure TVertexList.SetCapacity(const val : Integer);
begin
   FCapacity:=val;
   if FCapacity<FCount then
      FCapacity:=FCount;
   ReallocMem(FValues, FCapacity*SizeOf(TVertexData));
end;

// SetGrowth
//
procedure TVertexList.SetGrowth(const val : Integer);
begin
   if val>16 then
      FGrowth:=val
   else FGrowth:=16;
end;

// Grow
//
procedure TVertexList.Grow;
begin
   FCapacity:=FCapacity+FGrowth;
   ReallocMem(FValues, FCapacity*SizeOf(TVertexData));
end;

// GetFirstColor
//
function TVertexList.GetFirstColor: PGLFLoat;
begin
   Result:=@(FValues[0].color);
end;

// GetFirstEntry
//
function TVertexList.GetFirstEntry: PGLFloat;
begin
   Result:=Pointer(FValues);
end;

// GetFirstNormal
//
function TVertexList.GetFirstNormal: PGLFLoat;
begin
   Result:=@(FValues[0].normal);
end;

// GetFirstVertex
//
function TVertexList.GetFirstVertex: PGLFLoat;
begin
   Result:=@(FValues[0].coord);
end;

// DefineProperties
//
procedure TVertexList.DefineProperties(Filer:TFiler);
begin
//   Filer.DefineProperty('Items', ReadItems, WriteItems, Count>0);
end;

{// ReadItems
//
procedure TVertexList.ReadItems(Reader: TReader);
begin
   Clear;
   Reader.ReadListBegin;
   //while not Reader.EndOfValues do FValues.Add(Pointer(Reader.ReadInteger));
   Reader.ReadListEnd;
end;

// TVertexList
//
procedure  TVertexList.WriteItems(Writer: TWriter);
//var I : Integer;
begin
  Writer.WriteListBegin;
  //for i := 0 to FValues.Count-1 do Writer.WriteInteger(LongInt(FValues[i]));
  Writer.WriteListEnd;
end;}

// AddVertex (direct)
//
procedure TVertexList.AddVertex(const vertexData : TVertexData);
begin
   if FCount=FCapacity then Grow;
   FValues[FCount]:=vertexData;
   Inc(FCount);
   NotifyChange;
end;

// AddVertex3
//
procedure TVertexList.AddVertex3(const vd1, vd2, vd3 : TVertexData);
begin
   // extend memory space
   if FCount+2>=FCapacity then Grow;
   // calculate destination address for new vertex data
   FValues[FCount]:=vd1;
   FValues[FCount+1]:=vd2;
   FValues[FCount+2]:=vd3;
   Inc(FCount, 3);
   NotifyChange;
end;

// AddVertex (texturing)
//
procedure TVertexList.AddVertex(const aVertex: TVertex; const aNormal: TAffineVector;
                                const aColor: TColorVector; const aTexPoint: TTexPoint);
begin
   if FCount=FCapacity then Grow;
   // calculate destination address for new vertex data
   with FValues[FCount] do begin
      textCoord:=aTexPoint;
      color:=aColor;
      normal:=aNormal;
      coord:=aVertex;
   end;
   Inc(FCount);
   NotifyChange;
end;

// AddVertex (no texturing)
//
procedure TVertexList.AddVertex(const vertex: TVertex; const normal: TAffineVector;
                                const color: TColorVector);
begin
   AddVertex(vertex, normal, color, NullTexPoint);
end;

// Clear
//
procedure TVertexList.Clear;
begin
   FreeMem(FValues);
   FCount := 0;
   FCapacity := 0;
   FValues := nil;
   NotifyChange;
end;

// Assign
//
procedure TVertexList.Assign(Source: TPersistent);
begin
   if Assigned(Source) and (Source is TVertexList) then begin
      FCount := TVertexList(Source).FCount;
      FCapacity := FCount;
      ReallocMem(FValues, FCount*SizeOf(TVertexData));
      Move(TVertexList(Source).FValues^, FValues^, FCount*SizeOf(TVertexData));
   end else inherited Assign(Source);
end;

//----------------- TMesh ------------------------------------------------------

// Create
//
constructor TMesh.Create(AOwner:TComponent);
begin
  inherited Create(AOwner);
//  ObjectStyle:=ObjectStyle+[osDirectDraw];
  FVertices := TVertexList.Create(Self);
  FVertices.AddVertex(XVector, ZVector, NullHmgVector, NullTexPoint);
  FVertices.AddVertex(YVector, ZVector, NullHmgVector, NullTexPoint);
  FVertices.AddVertex(ZVector, ZVector, NullHmgVector, NullTexPoint);
  FVertexmode := vmVNCT; //should change this later to default to vmVN. But need to
end;                     //change GLMeshPropform so that it greys out unused vertex info

// Destroy
//
destructor TMesh.Destroy;
begin
   FVertices.Free;
   inherited Destroy;
end;

// BuildList
//
procedure TMesh.BuildList;
var
   VertexCount : Longint;
begin
   inherited BuildList;
   glPushAttrib(GL_POLYGON_BIT);
   glFrontFace(GL_CCW);
   case FVertexMode of
      vmV    : glInterleavedArrays(GL_V3F, SizeOf(TVertexData), FVertices.FirstVertex);
      vmVN   : glInterleavedArrays(GL_N3F_V3F, SizeOf(TVertexData), FVertices.FirstNormal);
      vmVNC  : glInterleavedArrays(GL_C4F_N3F_V3F, SizeOf(TVertexData), FVertices.FirstColor);
      vmVNCT : glInterleavedArrays(GL_T2F_C4F_N3F_V3F, 0, FVertices.FirstEntry);
   else
      Assert(False, glsInterleaveNotSupported);
   end;
   VertexCount := FVertices.Count;
   case FMode of
      mmTriangleStrip : glDrawArrays(GL_TRIANGLE_STRIP, 0, VertexCount);
      mmTriangleFan   : glDrawArrays(GL_TRIANGLE_FAN, 0, VertexCount);
      mmTriangles     : glDrawArrays(GL_TRIANGLES, 0, VertexCount);
      mmQuadStrip     : glDrawArrays(GL_QUAD_STRIP, 0, VertexCount);
      mmQuads         : glDrawArrays(GL_QUADS, 0, VertexCount);
      mmPolygon       : glDrawArrays(GL_POLYGON, 0, VertexCount);
   else
      Assert(False);
   end;
   glPopAttrib;
end;

// SetMode
//
procedure TMesh.SetMode(AValue: TMeshMode);
begin
   if AValue <> FMode then begin
      FMode := AValue;
      StructureChanged;
   end;
end;

// SetVertices
//
procedure TMesh.SetVertices(AValue: TVertexList);
begin
   if AValue <> FVertices then begin
      FVertices.Assign(AValue);
      StructureChanged;
   end;
end;

// SetVertexMode
//
procedure TMesh.SetVertexMode(AValue: TVertexMode);
begin
   if AValue<>FVertexMode then begin
      FVertexMode := AValue;
      StructureChanged;
   end;
end;

// CalcNormals
//
procedure TMesh.CalcNormals(Frontface: TFaceWinding);
var
   vn  : TAffineFltVector;
   i, j : Integer;
begin
   case FMode of
      mmTriangleStrip:
         with Vertices do for i:=0 to Count-3 do begin
            if (FrontFace=fwCounterClockWise) xor ((i and 1)=0) then
               vn:=CalcPlaneNormal(FValues[i+0].coord, FValues[i+1].coord, FValues[i+2].coord)
            else vn:=CalcPlaneNormal(FValues[i+2].coord, FValues[i+1].coord, FValues[i+0].coord);
            FValues[i].Normal:=vn;
         end;
      mmTriangles:
         with Vertices do for i:=0 to ((Count-3) div 3) do begin
            j:=i*3;
            if FrontFace=fwCounterClockWise then
               vn:=CalcPlaneNormal(FValues[j+0].coord, FValues[j+1].coord, FValues[j+2].coord)
            else vn:=CalcPlaneNormal(FValues[j+2].coord, FValues[j+1].coord, FValues[j+0].coord);
            FValues[j+0].normal:=vn;
            FValues[j+1].normal:=vn;
            FValues[j+2].normal:=vn;
         end;
      mmQuads:
         with Vertices do for I := 0 to ((Count-4) div 4) do begin
            j:=i*4;
            if FrontFace=fwCounterClockWise then
               vn:=CalcPlaneNormal(FValues[j+0].coord, FValues[j+1].coord, FValues[j+2].coord)
            else vn:=CalcPlaneNormal(FValues[j+2].coord, FValues[j+1].coord, FValues[j+0].coord);
            FValues[j+0].normal:=vn;
            FValues[j+1].normal:=vn;
            FValues[j+2].normal:=vn;
            FValues[j+3].normal:=vn;
         end;
      mmPolygon:
         with Vertices do if Count>2 then begin
            if FrontFace=fwCounterClockWise then
               vn:=CalcPlaneNormal(FValues[0].coord, FValues[1].coord, FValues[2].coord)
            else vn:=CalcPlaneNormal(FValues[2].coord, FValues[1].coord, FValues[0].coord);
            for i:=0 to Count-1 do
              FValues[i].normal:=vn;
         end;
   else
      Assert(False);
   end;
   StructureChanged;
end;

// Assign
//
procedure TMesh.Assign(Source:TPersistent);
begin
   if Assigned(Source) and (Source is TMesh) then begin
      FVertices.Assign(TMesh(Source).Vertices);
      FMode := TMesh(Source).FMode;
      FVertexMode := TMesh(Source).FVertexMode;
   end else inherited Assign(Source);
end;

// ------------------------------------------------------------------
// ------------------------------------------------------------------
// ------------------------------------------------------------------
initialization
// ------------------------------------------------------------------
// ------------------------------------------------------------------
// ------------------------------------------------------------------

	// class registrations
   RegisterClasses([TMesh]);

end.

