unit FileVersionInfo;
{
///////////////////////////////////////////////////////////////////////
// TFileVersionInfo component
// Author: Helmut Schottmueller
// Contact: hschottm@informatik.uni-bremen.de
// Copyright: (C) 1998,1999 H. Schottmueller
// Version: 1.0
//
// This component is freeware. You are free to use it in any of your
// applications.
///////////////////////////////////////////////////////////////////////
}
interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls;

type
  TFileVersionInfo = class(TComponent)
  private
    { private declarations }
    fCompanyName : String;
    fFileDescription : String;
    fFileVersion : String;
    fInternalName : String;
    fLegalCopyright : String;
    fLegalTradeMarks : String;
    fOriginalFilename : String;
    fProductName : String;
    fProductVersion : String;
    fComments : String;
    fExecutableFile : String;
    fCountryCode    : String;
    procedure setExecutableFile(FileName : String);
    procedure clearInfo;
  protected
    { protected declarations }
  public
    { public declarations }
    constructor Create(AOwner : TComponent); override;
    procedure FindFileVersionInfo;
  published
    { published declarations }
    property CompanyName : String read fCompanyName write fCompanyName;
    property FileDescription : String read fFileDescription write fFileDescription;
    property FileVersion : String read fFileVersion write fFileVersion;
    property InternalName : String read fInternalName write fInternalName;
    property LegalCopyright : String read fLegalCopyright write fLegalCopyright;
    property LegalTradeMarks : String read fLegalTradeMarks write fLegalTradeMarks;
    property OriginalFilename : String read fOriginalFilename write fOriginalFilename;
    property ProductName : String read fProductName write fProductName;
    property ProductVersion : String read fProductVersion write fProductVersion;
    property Comments : String read fComments write fComments;
    property ExecutableFile : String read fExecutableFile write setExecutableFile;
    property CountryCode : String read fCountryCode write fCountryCode;
  end;

	TTranslation = record
  	b1 : byte;
    b2 : byte;
    b3 : byte;
    b4 : byte;
  end;

  PTranslation = ^TTranslation;

procedure Register;

implementation

constructor TFileVersionInfo.Create(AOwner : TComponent);
// creates a TFileVersionInfo object
begin
  inherited Create(AOwner);
  setExecutableFile(ParamStr(0));
end;

procedure TFileVersionInfo.setExecutableFile(FileName : String);
// sets the executable filename and calls the fileversioninfo API
begin
  fExecutableFile := FileName;
  FindFileVersionInfo;
end;

procedure TFileVersionInfo.FindFileVersionInfo;
// calls the fileversioninfo API and sets the component properties
var
  StringBuffer       : PChar;
  VersionPointer     : PChar;
  Size               : DWord;
  VersionValueLength : DWord;
begin
	// clear all properties
	clearInfo;

	// get size of fileversioninfo buffer
  Size := GetFileVersionInfoSize(PChar(fExecutableFile), Size);

  if Size > 0 then begin
		// fileversioninfo exists

		// allocate memory for fileversioninfo strings
    StringBuffer := AllocMem(Size);
		// copy fileversioninfo into buffer
    GetFileVersionInfo(PChar(fExecutableFile), 0, Size, StringBuffer);

		// convert Translation to correct countrycode
    VerQueryValue(StringBuffer, PChar('\VarFileInfo\Translation'),
      Pointer(VersionPointer), VersionValueLength);

  	fCountryCode :=
    	IntToHex(PTranslation(VersionPointer)^.b2, 2) +
    	IntToHex(PTranslation(VersionPointer)^.b1, 2) +
      IntToHex(PTranslation(VersionPointer)^.b4, 2) +
      IntToHex(PTranslation(VersionPointer)^.b3, 2);

		// get company name
    if VerQueryValue(StringBuffer, PChar('\StringFileInfo\' + fCountryCode + '\' +
      'CompanyName'), Pointer(VersionPointer), VersionValueLength) then
      fCompanyName := StrPas(VersionPointer);
		// get file description
    if VerQueryValue(StringBuffer, PChar('\StringFileInfo\' + fCountryCode + '\' +
      'FileDescription'), Pointer(VersionPointer), VersionValueLength) then
      fFileDescription := StrPas(VersionPointer);
		// get file version
    if VerQueryValue(StringBuffer, PChar('\StringFileInfo\' + fCountryCode + '\' +
      'FileVersion'), Pointer(VersionPointer), VersionValueLength) then
      fFileVersion := StrPas(VersionPointer);
		// get internal name
    if VerQueryValue(StringBuffer, PChar('\StringFileInfo\' + fCountryCode + '\' +
      'InternalName'), Pointer(VersionPointer), VersionValueLength) then
      fInternalName := StrPas(VersionPointer);
		// get legal copyright
    if VerQueryValue(StringBuffer, PChar('\StringFileInfo\' + fCountryCode + '\' +
      'LegalCopyright'), Pointer(VersionPointer), VersionValueLength) then
      fLegalCopyright := StrPas(VersionPointer);
		// get legal trademarks
    if VerQueryValue(StringBuffer, PChar('\StringFileInfo\' + fCountryCode + '\' +
      'LegalTradeMarks'), Pointer(VersionPointer), VersionValueLength) then
      fLegalTradeMarks := StrPas(VersionPointer);
		// get original filename
    if VerQueryValue(StringBuffer, PChar('\StringFileInfo\' + fCountryCode + '\' +
      'OriginalFilename'), Pointer(VersionPointer), VersionValueLength) then
      fOriginalFilename := StrPas(VersionPointer);
		// get product name
    if VerQueryValue(StringBuffer, PChar('\StringFileInfo\' + fCountryCode + '\' +
      'ProductName'), Pointer(VersionPointer), VersionValueLength) then
      fProductName := StrPas(VersionPointer);
		// get product version
    if VerQueryValue(StringBuffer, PChar('\StringFileInfo\' + fCountryCode + '\' +
      'ProductVersion'), Pointer(VersionPointer), VersionValueLength) then
      fProductVersion := StrPas(VersionPointer);
		// get comments
    if VerQueryValue(StringBuffer, PChar('\StringFileInfo\' + fCountryCode + '\' +
      'Comments'), Pointer(VersionPointer), VersionValueLength) then
      fComments := StrPas(VersionPointer);

		// free previously allocated memory
    FreeMem(StringBuffer, Size);
  end;
end;

procedure TFileVersionInfo.clearInfo;
// clear all properties
begin
  fCompanyName := '';
  fFileDescription := '';
  fFileVersion := '';
  fInternalName := '';
  fLegalCopyright := '';
  fLegalTradeMarks := '';
  fOriginalFilename := '';
  fProductName := '';
  fProductVersion := '';
  fComments := '';
  fCountryCode := '';
end;

procedure Register;
begin
	// register the component in the 3rdParty tab
  RegisterComponents('3rdParty', [TFileVersionInfo]);
end;

end.
