{ ****************************************************************** }
{   TFireCracker                                                     }
{   Component to Control the FireCracker Module for X10 Devices      }
{   Copyright  2005 by Lee Christensen                              }
{ ****************************************************************** }

unit X10Unit;

interface

uses WinTypes, WinProcs, Messages, SysUtils, Classes, Controls,
     Windows, Math;


type
  TPortType    = (ptCom1,ptCom2,ptCom3,ptCom4,ptCom5,ptCom6,ptCom7,ptCom8);
  TCommandType = (ctOn,ctOff,ctBright,ctDim);

  EFireCracker = class(Exception);

  TFireCracker = class(TComponent)
    private
        FCommand : TCommandType;
        FConnected : Boolean;
        FPort : TPortType;
        ReadBufSize : DWORD;
        WriteBufSize : DWORD;
        FX10Address : String;
        FHandle : THandle;

        function GetCommand : TCommandType;
        procedure SetCommand(Value : TCommandType);
        function GetConnected : Boolean;
        function GetPort : TPortType;
        procedure SetPort(Value : TPortType);
        function GetX10Address : String;
        procedure SetX10Address(Value : String);
        function ByteToBin(t:byte): string;
        procedure CreateHandle;
        procedure DestroyHandle;
        procedure SetDCB;
        procedure SetupComPort;
        function  ComString: String;
        procedure Standby;
    protected
        procedure Loaded; override;
        function SendCommand(Addr : String;Command : TCommandType) : Boolean;
        procedure SetDTR(State: Boolean);
        procedure SetRTS(State: Boolean);
    public
        constructor Create(AOwner: TComponent); override;
        destructor Destroy; override;
        function Execute : Boolean;
        function LastErr :String;
        procedure Open;
        procedure SetComm;
        procedure Close;
    published
        property Command    : TCommandType read GetCommand write SetCommand default ctOff;
        property Connected  : Boolean read GetConnected write FConnected default False;
        property Port       : TPortType read GetPort write SetPort default ptCom1;
        property X10Address : String read GetX10Address write SetX10Address;
  end;

var
  x10dim   : array [1..2]  of string;
  DeviceIDs: TStringList;
  OffCmds  : TStringList;
  OnCmds   : TStringList;
  DimCmds  : TStringList;

const
  Footer        = '10101101';
  Header        = '1101010110101010';
  ReadBuffSize  = 4096;
  WriteBuffSize = 4096;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('X10', [TFireCracker]);
end;
(******************************************************************************)

function TFireCracker.GetCommand : TCommandType;
begin
  Result := FCommand
end;
(******************************************************************************)

procedure TFireCracker.SetCommand(Value : TCommandType);
begin
  Case Value of
    ctOn     : FCommand := ctOn;
    ctOff    : FCommand := ctOff;
    ctBright : FCommand := ctBright;
    ctDim    : FCommand := ctDim;
  end;
end;
(******************************************************************************)

function TFireCracker.GetConnected : Boolean;
begin
  Result := FConnected;
end;
(******************************************************************************)

function TFireCracker.GetPort : TPortType;
begin
  Result := FPort;
end;
(******************************************************************************)

procedure TFireCracker.SetPort(Value : TPortType);
begin
  FPort := Value;
end;
(******************************************************************************)

function TFireCracker.GetX10Address : String;
begin
  Result := FX10Address;
end;
(******************************************************************************)

procedure TFireCracker.SetX10Address(Value : String);
begin
  FX10Address := Value;
end;
(******************************************************************************)

function TFireCracker.ByteToBin(t:byte): string;
var
  shold : string;
  iCounter : Integer;
begin
  shold := EmptyStr;

  for iCounter := 7 downto 0 do begin
    if (t and Trunc(Power(2, iCounter))) >= Trunc(Power(2, iCounter)) then begin
      sHold := sHold + '1';
      t := t - Trunc(Power(2, iCounter));
    end else
      sHold := sHold + '0'
  end;

  Result := shold;
end;
(******************************************************************************)

constructor TFireCracker.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

   FCommand := ctOff;
   FConnected := False;
   FPort := ptCom1;
   FX10Address := 'A1';

   //Associates Housecodes with a specific Output number
   DeviceIDs := TStringList.Create;
   DeviceIds.CommaText := 'A=96,B=112,C=64,D=72,E=128,F=136,G=160,H=96,I=224,J=240,K=192,L=208,M=0,N=16,O=32,P=48';

   //power On commands
   OnCmds := TStringList.Create;
   OnCmds.CommaText := '1=0,2=16,3=8,4=24,5=64,6=80,7=72,8=88';

   //Power Off Commands
   OffCmds := TStringList.Create;
   OffCmds.CommaText := '1=32,2=48,3=40,4=56,5=96,6=112,7=104,8=120';

   //Dim or Brighten Commands
   DimCmds := TStringList.Create;
   DimCmds.CommaText := '1=136,2=152';

end;
(******************************************************************************)

{ Gets the handle to the comm port }
procedure TFireCracker.CreateHandle;
begin
  FHandle := CreateFile(PChar(ComString),GENERIC_READ or GENERIC_WRITE,
    0,nil,OPEN_EXISTING,FILE_FLAG_OVERLAPPED,0);

  if FHandle = INVALID_HANDLE_VALUE then
    raise EFireCracker.Create('Unable to open com port: ' + LastErr);
end;
(******************************************************************************)

destructor TFireCracker.Destroy;
begin
  DeviceIDs.Free;
  OffCmds.Free;
  OnCmds.Free;
  inherited Destroy;
end;
(******************************************************************************)

//free the handle to the modem
procedure TFireCracker.DestroyHandle;
begin
  if FHandle <> INVALID_HANDLE_VALUE then CloseHandle(FHandle);
end;
(******************************************************************************)

function TFireCracker.Execute : Boolean;
begin
  Result := SendCommand(FX10Address,FCommand);
end;
(******************************************************************************)

function TFireCracker.LastErr :String;
begin
  Result := IntToStr(GetLastError);
end;
(******************************************************************************)

procedure TFireCracker.Loaded;
begin
  inherited Loaded;
end;
(******************************************************************************)
//opens a connection to the comm port
procedure TFireCracker.Open;
begin
  Close;
  CreateHandle;
  FConnected := True;
  try
    SetupComPort;
  except
    DestroyHandle;
    FConnected := False;
    raise;
  end;
end;
(******************************************************************************)
//this is the function that does all the dirty work!!
function TFireCracker.SendCommand(Addr : String; Command : TCommandType) : Boolean;
var
  HouseCode : Byte;
  DeviceIDX : Integer;
  iCounter  : Integer;
  OutPut    : String;
begin
  Output := Header;

  //get the housecode number for the letter entered
  HouseCode := StrToInt(DeviceIds.Values[UpperCase(Addr[1])]);
  //get the device Index number from the address string
  DeviceIDX  := StrToInt(Copy(Addr,2,Length(Addr)));

  if (DeviceIDX > 8) and (Ord(Command) < 2) then begin
    HouseCode := HouseCode + 4;
    DeviceIDX := DeviceIDX - 8;
  end;

  //format the Output into a "Binary" string to send to the Firecracker Module
  Output := Output + ByteToBin(HouseCode);
  Case Ord(Command) of
    0   : Output := Output + ByteToBin(StrToInt(OnCmds.Values[IntToStr(DeviceIDX)])); //On
    1   : OutPut := OutPut + ByteToBin(StrToInt(OffCmds.Values[IntToStr(DeviceIDX)]));//Off
    2,3 : Output := Output + ByteToBin(StrToInt(DimCmds.Values[IntToStr(Ord(Command) - 1)])); //Bright & Dim
  end;
  Output := Output + Footer;

  //Go to Standby mode to get ready to send a command
  StandBy;

  //send the "Binary" command to the Firecracker by toggling the RTS and DTR states of the comm port
  //when a character in the OutPut Variable is a "1" then turn on DTR wait then turn off DTR
  //when a caracter in the OutPut variable is a "0" then turn on RTS wait then turn off RTS
  //Output example: "10111000101001"
  try
    for iCounter := 1 to Length(Output) do begin

      Case StrToInt(Output[iCounter]) of
        0 : begin
          SetRTS(False);
          Sleep(1);
          SetRTS(True);
        end;
        1 : begin
          SetDTR(False);
          Sleep(1);
          SetDTR(True);
        end;
      end;

    end;//for loop

    Result := True;
  except
    Result := False;
  end;
end;
(******************************************************************************)

procedure TFireCracker.SetComm;
begin
  if FConnected then begin
    if not SetupComm(FHandle, ReadBufSize, WriteBufSize) then
      raise EFireCracker.Create('Unable to set com state: ' + LastErr);
  end;
end;
(******************************************************************************)

{ Sets the Parameters for the comm port }
procedure TFireCracker.SetDCB;
var
  DCB: TDCB;
  Temp: DWORD;
begin
  if FConnected then begin
    FillChar(DCB, SizeOf(DCB), 0);

    DCB.DCBlength := SizeOf(DCB);
    DCB.XonChar   := #17;
    DCB.XoffChar  := #19;
    DCB.XonLim    := 4096 div 4;
    DCB.XoffLim   := DCB.XonLim;
    DCB.EvtChar   := #9;

    DCB.Flags    := DCB.Flags or $00000001;
    DCB.Flags    := DCB.Flags or $00000004;
    Temp         := DTR_CONTROL_ENABLE;
    DCB.Flags    := DCB.Flags or Integer($00000030 and (Temp shl 4));
    Temp         := RTS_CONTROL_ENABLE;
    DCB.Flags    := DCB.Flags or Integer($00003000 and (Temp shl 12));
    DCB.Flags    := DCB.Flags or $00000100;
    DCB.Parity   := NOPARITY;
    DCB.StopBits := ONESTOPBIT;
    DCB.BaudRate := CBR_9600;
    DCB.ByteSize := 8;

    if not SetCommState(FHandle, DCB) then
      raise EFireCracker.Create('Unable to set com Parameters: ' + LastErr);
  end;
end;
(******************************************************************************)

{ Initializes the port }
procedure TFireCracker.SetupComPort;
begin
  SetComm;
  SetDCB;
end;
(******************************************************************************)

function TFireCracker.ComString: String;
begin
  case FPort of
    ptCOM1: Result := 'COM1';
    ptCOM2: Result := 'COM2';
    ptCOM3: Result := 'COM3';
    ptCOM4: Result := 'COM4';
    ptCOM5: Result := 'COM5';
    ptCOM6: Result := 'COM6';
    ptCOM7: Result := 'COM7';
    ptCOM8: Result := 'COM8';
  end;
end;
(******************************************************************************)

procedure TFireCracker.Standby;
begin
  SetDtr(True);
  SetRTS(True);
end;
(******************************************************************************)

procedure TFireCracker.SetDTR(State: Boolean);
var
  Act: DWORD;
begin
  if State then
    Act := Windows.SETDTR
  else
    Act := Windows.CLRDTR;

  //Send the command to windows to set the DTR state
  if not EscapeCommFunction(FHandle, Act) then
    raise EFireCracker.Create('Unable to set signal: ' + LastErr);
end;
(******************************************************************************)

procedure TFireCracker.SetRTS(State: Boolean);
var
  Act: DWORD;
begin
  if State then
    Act := Windows.SETRTS
  else
    Act := Windows.CLRRTS;

  //Send the command to windows to set the RTS state
  if not EscapeCommFunction(FHandle, Act) then
    raise EFireCracker.Create('Unable to set signal: ' + LastErr);
end;
(******************************************************************************)

procedure TFireCracker.Close;
begin
  DestroyHandle;
  FConnected := False;
end;
(******************************************************************************)

end.
