unit MFExtenso;
{
This component is freeware. You may use it, distribute it and modify it, but
you may not charge for it.
In case of modifications you must mail me (jair@microflex.com.br) a copy of
the modifications. The reason are simple: Any changes that improve this free-
ware component should be to benefit for everybody, not only you. That way you
can be pretty sure, that this component has few errors and much functionality.
In case of modifications, you will be on the credits list beneath:

To install, from Delphi main menu:
- Select "Component";
- Select "Install component...";
- Set "Unit file name" to the MFExtenso.pas unpacked file, using, if you prefer, the Browse button to find it;
- Click "OK".
}
{
Version 1.0 by Jair Roberto Nunes da Silva:
  This version is the base version:
    Use:
    with MFExtenso1 do
    begin
      NumberValue := 123.45;
      Memo1.Text := NumberText;
      QRLabel1.Caption := NumberText;
    end;

    PUBLIC:
      property NumberText: text calculated based in the NumberValue property (texto calculado com base na propriedade NumberValue)
    PUBLISHED:
      property CurrencyTitle: titles of your currency in singular and plural (Default = 'real' / 'reais') (nome da moeda utilizada no singular e no plural)
      property ErrorOutOfBounds: error showed when de number limits are exceeded (erro apresentado quando os limites de valor so superados)
      property NameCent: title of your currency cents in singular and plural (Default = 'centavo' / 'centavos') (nome dos centavos no singular e no plural)
      property NumberValue: this property must be set for the component calculates de NumberText public property (defina esta propriedade para que o clculo seja realizado e o resultado colocado em NumberText)
      property OnCalculateUnit: you can use this event to make changes when the Unit Names are being set (use este evento para fazer alteraes no texto definido para a unidade calculada)
      property OnCalculateTen: you can use this event to make changes when the Tenth Names are being set (use este evento para fazer alteraes no texto definido para a dezena calculada)
      property OnCalculateHundred: you can use this event to make changes when the Hundred Names are being set (use este evento para fazer alteraes no texto definido para a centena calculada)
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs;

type
  TCalculateNumber = procedure (const Value: LongWord; var Text: string) of object;

  TNumberNames = class(TPersistent)
  private
    FControl: TComponent;
    FOnChange: TNotifyEvent;
    FSingular: string;
    FPlural: string;
    procedure SetText(Index: Integer; Value: string);
    function GetValues(IndexNo: Integer): string;
  protected
    procedure Change; dynamic;
    procedure AssignTo(Dest: TPersistent); override;
    property Control: TComponent read FControl;
  public
    constructor Create(Control: TComponent); virtual;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
    property Values[IndexNo: Integer]: string read GetValues; default;
  published
    property Singular: string index 0 read FSingular write SetText;
    property Plural: string index 1 read FPlural write SetText;
  end;

  TMFExtenso = class(TComponent)
  private
    FCurrencyTitle: TNumberNames;
    FErrorOutOfBounds: string;
    FNameCent: TNumberNames;
    FNumberText: string;
    FNumberValue: Double;
    FOnCalculateHundred: TCalculateNumber;
    FOnCalculateTen: TCalculateNumber;
    FOnCalculateUnit: TCalculateNumber;
    procedure DoCalculateUnit(const Value: LongWord; var Text: string);
    procedure DoCalculateTen(const Value: LongWord; var Text: string);
    procedure DoCalculateHundred(const Value: LongWord; var Text: string);
    procedure DoNamesChange(Sender: TObject);
    procedure SetErrorOutOfBounds(const Value: string);
    procedure SetNumberValue(const Value: Double);
    procedure SetNumberText(const Value: string);
    procedure UpdateNumberText;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property NumberText: string read FNumberText;
  published
    property CurrencyTitle: TNumberNames read FCurrencyTitle write FCurrencyTitle;
    property ErrorOutOfBounds: string read FErrorOutOfBounds write SetErrorOutOfBounds;
    property NameCent: TNumberNames read FNameCent write FNameCent;
    property NumberValue: Double read FNumberValue write SetNumberValue;
    property OnCalculateUnit: TCalculateNumber read FOnCalculateUnit write FOnCalculateUnit;
    property OnCalculateTen: TCalculateNumber read FOnCalculateTen write FOnCalculateTen;
    property OnCalculateHundred: TCalculateNumber read FOnCalculateHundred write FOnCalculateHundred;
  end;

  procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('MicroFlex', [TMFExtenso]);
end;

{ TNumberNames }

procedure TNumberNames.AssignTo(Dest: TPersistent);
begin
  if Dest is TNumberNames then
    with TNumberNames(Dest) do
    begin
      FSingular := Self.FSingular;
      FPlural := Self.FPlural;
      Change;
    end
  else
    inherited AssignTo(Dest);
end;

procedure TNumberNames.Change;
begin
  if Assigned(FOnChange) then
    FOnChange(Self);
end;

constructor TNumberNames.Create(Control: TComponent);
begin
  inherited Create;
  FControl := Control;
end;

function TNumberNames.GetValues(IndexNo: Integer): string;
begin
  case IndexNo of
  0: Result := FSingular;
  1: Result := FPlural;
  else
    Result := '';
  end;
end;

procedure TNumberNames.SetText(Index: Integer; Value: string);
begin
  case Index of
  0: FSingular := Value;
  1: FPlural := Value;
  end;
  Change;
end;

{ TMFExtenso }

constructor TMFExtenso.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FCurrencyTitle := TNumberNames.Create(Self);
  FCurrencyTitle.OnChange := DoNamesChange;
  FNameCent := TNumberNames.Create(Self);
  FNameCent.OnChange := DoNamesChange;

  FCurrencyTitle.Singular := 'real';
  FCurrencyTitle.Plural := 'reais';
  FNameCent.Singular := 'centavo';
  FNameCent.Plural := 'centavos';
  FErrorOutOfBounds := 'Valor fora dos limites!';
  FNumberValue := 0;
end;

destructor TMFExtenso.Destroy;
begin
  FCurrencyTitle.Free;
  FNameCent.Free;
  inherited Destroy;
end;

procedure TMFExtenso.DoNamesChange(Sender: TObject);
begin
  UpdateNumberText;
end;

procedure TMFExtenso.SetErrorOutOfBounds(const Value: string);
begin
  if Value = FErrorOutOfBounds then
    Exit;

  FErrorOutOfBounds := Value;
  UpdateNumberText;
end;

procedure TMFExtenso.SetNumberValue(const Value: Double);
begin
  if Value = FNumberValue then
    Exit;

  FNumberValue := Value;
  UpdateNumberText;
end;

procedure TMFExtenso.SetNumberText(const Value: string);
begin
  FNumberText := Value;
end;

procedure TMFExtenso.DoCalculateHundred(const Value: LongWord;
  var Text: string);
begin
  if (Value < 200) and (Value mod 100 > 0) then
    Text := 'cento';

  if Assigned(OnCalculateHundred) then
    OnCalculateHundred(Value, Text);
end;

procedure TMFExtenso.DoCalculateTen(const Value: LongWord;
  var Text: string);
begin
  if Assigned(OnCalculateTen) then
    OnCalculateTen(Value, Text);
end;

procedure TMFExtenso.DoCalculateUnit(const Value: LongWord;
  var Text: string);
begin
  if Assigned(OnCalculateUnit) then
    OnCalculateUnit(Value, Text);
end;

procedure TMFExtenso.UpdateNumberText;
const
  NameNumbers: array[1..19] of string = ('um', 'dois', 'trs', 'quatro', 'cinco', 'seis', 'sete', 'oito', 'nove', 'dez',
                                         'onze', 'doze', 'treze', 'quatorze', 'quinze', 'dezesseis', 'dezessete', 'dezoito', 'dezenove');
  NameTenths: array[2..9] of string = ('vinte', 'trinta', 'quarenta', 'cinqenta', 'sessenta', 'setenta', 'oitenta', 'noventa');
  NameHundreds: array[1..9] of string = ('cem', 'duzentos', 'trezentos', 'quatrocentos', 'quinhentos', 'seiscentos', 'setecentos', 'oitocentos', 'novecentos');
  NameThousand = 'mil';
  NameMillion: array[Boolean] of string = ('milho', 'milhes');
  NameBillion: array[Boolean] of string = ('bilho', 'bilhes');
var
  Number: Double;
  s: string;

  function RecurseNumber(N: LongWord): string;
  begin
    case N of
    1..19: // Unit
      begin
        Result := NameNumbers[N];
        DoCalculateUnit(N, Result);
      end;
    20..99: // Ten
      begin
        Result := NameTenths[N div 10];
        DoCalculateTen(N, Result);
        if N mod 10 > 0 then
          Result := Result + ' e ' + RecurseNumber(N mod 10);
      end;
    100..999: // Hundred
      begin
        Result := NameHundreds[N div 100];
        DoCalculateHundred(N, Result);
        if N mod 100 > 0 then
          Result := Result + ' e ' + RecurseNumber(N mod 100);
      end;
    1000..999999: // Thousand
      begin
        Result := RecurseNumber(N div 1000) + ' ' + NameThousand;

        if N mod 1000 > 0 then
          if N mod 100 = 0 then
            Result := Result + ' e ' + RecurseNumber(N mod 1000)
          else
            Result := Result + ', ' + RecurseNumber(N mod 1000);
      end;
    1000000..999999999: // Million
      begin
        Result := RecurseNumber(N div 1000000) + ' ' + NameMillion[N div 1000000 <> 1];

        if N mod 1000000 = 0 then
          Result := Result + ' de'
        else
        if N mod 100000 = 0 then
          Result := Result + ' e ' + RecurseNumber(N mod 1000000)
        else
          Result := Result + ', ' + RecurseNumber(N mod 1000000);
      end;
    1000000000..4294967295: // Billion
      begin
        Result := RecurseNumber(N div 1000000000) + ' ' + NameBillion[N div 1000000000 <> 1];

        if N mod 1000000000 = 0 then
          Result := Result + ' de'
        else
        if (N mod 1000000000) mod 100000000 = 0 then
          Result := Result + ' e ' + RecurseNumber(N mod 1000000000)
        else
          Result := Result + ', ' + RecurseNumber(N mod 1000000000);
      end;
    end; // case N of
  end; // RecurseNumber

begin // UpdateNumberText
  if NumberValue = 0 then
  begin
    SetNumberText('');
    Exit;
  end
  else
  if (NumberValue < 0.01) or (NumberValue > 4294967295) then
  begin
    SetNumberText(ErrorOutOfBounds);
    Exit;
  end;

  Number := NumberValue;

  // Integer part
  s := RecurseNumber(Round(Int(Number)));
  if Round(Int(Number)) = 1 then
    s := s + ' ' + CurrencyTitle.Singular
  else
    s := s + ' ' + CurrencyTitle.Plural;

  // Cents
  if not (Frac(Number) = 0.00) then
  begin
    Number := Frac(Number) * 100;
    if Number = 1 then
      s := s + ' e ' + RecurseNumber(Round(Number)) + ' ' + NameCent.Singular
    else
      s := s + ' e ' + RecurseNumber(Round(Number)) + ' ' + NameCent.Plural;
  end;

  // Delete double spaces
  while Pos('  ', s) > 0 do
    Delete(s, Pos('  ', s), 1);

  SetNumberText(s);
end;

end.
