unit EuXMLToolbox;

interface

uses
  SysUtils;

function CreateXMLFile(AFilename: string): boolean;
function OpenXMLFile(AFilename: string): boolean;
procedure CloseXMLFile;

function ReadXMLLine: string;
function WriteXMLLine(AValue: string): boolean;

function IsXMLStartTag(ATag: string): boolean;
function IsXMLEndTag(ATag: string): boolean;
function WriteXMLStartTag(ATag: string): boolean;
function WriteXMLEndTag(ATag: string): boolean;

function ReadXMLStringValue(ATag: string): string;
function ReadXMLIntValue(ATag: string): integer;
function ReadXMLBoolValue(ATag: string): boolean;
function WriteXMLStringValue(ATag, AValue: string): boolean;
function WriteXMLIntValue(ATag: string; AValue: integer): boolean;
function WriteXMLBoolValue(ATag: string; AValue: boolean): boolean;

var
  FXMLFileOpen: boolean;
  FXMLFile: TextFile;
const
  XML_FILE_ID = '<?xml version="1.0" standalone="yes"?>';

implementation

{ **************************************************************************** }

function CreateXMLFile(AFilename: string): boolean;
begin
  Result:=false;
  if Length(AFilename)>0 then
  begin
    AssignFile(FXMLFile,AFilename);
    {$I-}
    Rewrite(FXMLFile);
    {$I+}
    Result:=IOResult=0;
    FXMLFileOpen:=IOResult=0;
    if Result then Writeln(FXMLFile,XML_FILE_ID);
  end;
  if not Result then CloseFile(FXMLFile);
end;

function OpenXMLFile(AFilename: string): boolean;
begin
  Result:=false;
  if (Length(AFilename)>0) and (FileExists(AFilename)) then
  begin
    AssignFile(FXMLFile,AFilename);
    {$I-}
    Reset(FXMLFile);
    {$I+}
    Result:=IOResult=0;
    FXMLFileOpen:=IOResult=0;
    if Result then Result:=ReadXMLLine=XML_FILE_ID;
  end;
  if not Result then CloseFile(FXMLFile);
end;

procedure CloseXMLFile;
begin
  CloseFile(FXMLFile);
  FXMLFileOpen:=false;
end;

{ **************************************************************************** }

function ReadXMLLine: string;
begin
  Result:='';
  if FXMLFileOpen then
  begin
    {$I-}
    Readln(FXMLFile,Result);
    {$I+}
    if IOResult<>0 then Result:='';
  end;
end;

function WriteXMLLine(AValue: string): boolean;
begin
  Result:=false;
  if FXMLFileOpen then
  begin
    {$I-}
    Writeln(FXMLFile,AValue);
    {$I+}
    Result:=IOResult=0;
  end;
end;

{ **************************************************************************** }

function IsXMLStartTag(ATag: string): boolean;
begin
  if ATag[1]<>'<' then ATag:='<'+ATag;
  if ATag[Length(ATag)]<>'>' then ATag:=ATag+'>';
  Result:=(ReadXMLLine=ATag);
end;

function IsXMLEndTag(ATag: string): boolean;
begin
  if ATag[1]<>'<' then ATag:='<'+ATag;
  if ATag[2]<>'/' then ATag:=ATag[1]+'/'+Copy(ATag,2,Length(ATag));
  if ATag[Length(ATag)]<>'>' then ATag:=ATag+'>';
  Result:=(ReadXMLLine=ATag);
end;

{ **************************************************************************** }

function WriteXMLStartTag(ATag: string): boolean;
begin
  Result:=false;
  if Length(ATag)>0 then
  begin
    if ATag[1]<>'<' then ATag:='<'+ATag;
    if ATag[Length(ATag)]<>'>' then ATag:=ATag+'>';
    Result:=WriteXMLLine(ATag);
  end;
end;

function WriteXMLEndTag(ATag: string): boolean;
begin
  Result:=false;
  if Length(ATag)>0 then
  begin
    if ATag[1]<>'<' then ATag:='<'+ATag;
    if ATag[2]<>'/' then ATag:=ATag[1]+'/'+Copy(ATag,2,Length(ATag));
    if ATag[Length(ATag)]<>'>' then ATag:=ATag+'>';
    Result:=WriteXMLLine(ATag);
  end;
end;

{ **************************************************************************** }

function ReadXMLStringValue(ATag: string): string;
var
  FValue: string;
  FStart: integer;
  FEnd: integer;
begin
  Result:='';
  if Length(ATag)>0 then
  begin
    FValue:=ReadXMLLine;
    if Length(FValue)>0 then
    begin
      if ATag[1]<>'<' then ATag:='<'+ATag;
      if ATag[Length(ATag)]<>'>' then ATag:=ATag+'>';
      FStart:=Pos(ATag,FValue)+Length(ATag);
      if FStart>Length(ATag) then
      begin
        if ATag[2]<>'/' then ATag:=ATag[1]+'/'+Copy(ATag,2,Length(ATag));
        FEnd:=Pos(ATag,FValue);
        if (FEnd>0) and (FEnd-FStart>0) then
        begin
          Result:=Copy(FValue,FStart,FEnd-FStart);
        end;
      end;
    end;
  end;
end;

function ReadXMLIntValue(ATag: string): integer;
var
  FValue: string;
begin
  Result:=0;
  FValue:=ReadXMLStringValue(ATag);
  if Length(FValue)>0 then
  begin
    Result:=StrToInt(FValue);
  end;
end;

function ReadXMLBoolValue(ATag: string): boolean;
var
  FValue: string;
begin
  Result:=false;
  FValue:=ReadXMLStringValue(ATag);
  if Length(FValue)>0 then
  begin
    Result:=(FValue='1');
  end;
end;

{ **************************************************************************** }

function WriteXMLStringValue(ATag, AValue: string): boolean;
begin
  Result:=false;
  if Length(ATag)>0 then
  begin
    if ATag[1]<>'<' then ATag:='<'+ATag;
    if ATag[Length(ATag)]<>'>' then ATag:=ATag+'>';
    AValue:=ATag+AValue;
    if ATag[2]<>'/' then ATag:=ATag[1]+'/'+Copy(ATag,2,Length(ATag));
    AValue:=AValue+ATag;
    Result:=WriteXMLLine(AValue);
  end;
end;

function WriteXMLIntValue(ATag: string; AValue: integer): boolean;
begin
  Result:=WriteXMLStringValue(ATag,IntToStr(AValue));
end;

function WriteXMLBoolValue(ATag: string; AValue: boolean): boolean;
begin
  Result:=WriteXMLStringValue(ATag,IntToStr(ord(AValue)));
end;

end.
