{ Version 1.0.0; Build date: Mar-24-2001 }
unit EuMessageDialog;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls, MMSystem, Registry, EuXMLToolbox;

type

{ *** Component data type definitions **************************************** }

  TEuMessageDialogIconType = (itNone, itWarning, itInformation, itQuestion,
                              itError, itKeyboard, itConnection, itPrinter,
                              itHelp, itDisk, itNewspaper, itClock, itChart,
                              itPhone, itNeutral, itHappy, itSad, itCDRom,
                              itNetwork, itComputer, itWinLogo, itCalculator,
                              itLightbulb, itKey);

  TEuMessageDialogBevelRange = (brNone, brText, brAll);

  TEuMessageDialogButtons = set of (dbCustom1, dbCustom2, dbCustom3, dbCustom4,
                                    dbYes, dbYesToAll, dbNo, dbNoToAll, dbOk,
                                    dbCancel, dbAbort, dbRetry, dbIgnore, dbAll,
                                    dbClose, dbBack, dbNext, dbHelp);

  TEuMessageDialogDefaultButton = (dbDefaultButton1, dbDefaultButton2,
                                   dbDefaultButton3, dbDefaultButton4);

  TEuMessageDialogOptions = set of (doCenterText, doRightAlignText, doRTLReading,
                                    doPlaySystemSound, doCenterIcon,
                                    doCreateCustomButtonsLast, doRightAlignButtons);

{ *** Component TPersistent type definitions ********************************* }

  TEuMessageDialogCustomCaptions = class(TPersistent)
  private
    FCustomButton1Caption: string;
    FCustomButton2Caption: string;
    FCustomButton3Caption: string;
    FCustomButton4Caption: string;
  published
    property CustomButton1Caption: string read FCustomButton1Caption write FCustomButton1Caption;
    property CustomButton2Caption: string read FCustomButton2Caption write FCustomButton2Caption;
    property CustomButton3Caption: string read FCustomButton3Caption write FCustomButton3Caption;
    property CustomButton4Caption: string read FCustomButton4Caption write FCustomButton4Caption;
  end;

  TEuMessageDialogButton = class(TPersistent)
  private
    FButtonList: TEuMessageDialogButtons;
    FCustomCaptions: TEuMessageDialogCustomCaptions;
    FDefaultButton: TEuMessageDialogDefaultButton;
  published
    property ButtonList: TEuMessageDialogButtons read FButtonList write FButtonList;
    property CustomCaptions: TEuMessageDialogCustomCaptions read FCustomCaptions write FCustomCaptions;
    property DefaultButton: TEuMessageDialogDefaultButton read FDefaultButton write FDefaultButton;
  end;

  TEuMessageDialogBevel = class(TPersistent)
  private
    FRange: TEuMessageDialogBevelRange;
    FShape: TBevelShape;
    FStyle: TBevelStyle;
  published
    property Range: TEuMessageDialogBevelRange read FRange write FRange;
    property Shape: TBevelShape read FShape write FShape;
    property Style: TBevelStyle read FStyle write FStyle;
  end;

  TEuMessageDialogPosition = class(TPersistent)
  private
    FLeft: integer;
    FTop: integer;
  published
    property Left: integer read FLeft write FLeft;
    property Top: integer read FTop write FTop;
  end;

{ *** Component data structure definitions *********************************** }

  PEuMessageDialogData = ^TEuMessageDialogData;
  TEuMessageDialogData = record
    BevelRange: TEuMessageDialogBevelRange;
    BevelShape: TBevelShape;
    BevelStyle: TBevelStyle;
    ButtonList: TEuMessageDialogButtons;
    Caption: string;
    CustomButtonCaption: array [1..4] of string;
    DefaultButton: TEuMessageDialogDefaultButton;
    HelpContext: integer;
    Icon: TEuMessageDialogIconType;
    Left: integer;
    Options: TEuMessageDialogOptions;
    Text: string;
    Top: integer;
  end;

{ *** Component class definitions ******************************************** }

  TEuMessageDialog = class(TComponent)
  private
    { Private declarations }
    FDialogList: TList;
    FBevel: TEuMessageDialogBevel;
    FButtons : TEuMessageDialogButton;
    FCaption : string;
    FData: TEuMessageDialogData;                                                //data structure for dialog
    FOptions : TEuMessageDialogOptions;
    FHelpContext : THelpContext;
    FIcon : TEuMessageDialogIconType;
    FPosition: TEuMessageDialogPosition;
    FResult: word;
    FText : string;
    { Dialog variables }
    FMsgForm: TForm;
    FMsgImage: TImage;
    FMsgBevel: TBevel;
    FMsgScrollBox: TScrollBox;
    FMsgPaintBox: TPaintBox;
    procedure CreateButtons;
    function GetButtonDim: integer;
    function GetOptimumPaintBoxWidth(AText: string; AIncludeScrollbar: boolean): integer;
    function GetOptimumPaintBoxHeight(AText: string; AUseDisplayLimit: boolean): integer;
    function GetDialogWidth: integer;
    function GetDialogHeight: integer;
    procedure PlaySystemSound;
    function SetControlDimensions: boolean;
    { Dialog event handlers }
    procedure FormKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
    procedure PaintBoxPaint(Sender: TObject);
    procedure ButtonClick(Sender: TObject);
  protected
    { Protected declarations }
    function GetDialogListCount: integer;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function Execute: word;
    function ExecuteDialogItem(AIndex: integer): word;
    function ExecuteA: word;
    function SaveToFile(const AFilename: string): boolean;
    function LoadFromFile(const AFilename: string): boolean;
    function Add: integer;
    procedure Clear;
    procedure Delete(AIndex: integer);
    function Insert(AIndex: integer): integer;
    property Count: integer read GetDialogListCount;
    property Result: word read FResult;
  published
    { Published declarations }
    property Bevel: TEuMessageDialogBevel read FBevel write FBevel;
    property Buttons: TEuMessageDialogButton read FButtons write FButtons;
    property Caption: string read FCaption write FCaption;
    property Options: TEuMessageDialogOptions read FOptions write FOptions;
    property HelpContext: THelpContext read FHelpContext write FHelpContext;
    property Icon: TEuMessageDialogIconType read FIcon write FIcon;
    property Position: TEuMessageDialogPosition read FPosition write FPosition;
    property Text: string read FText write FText;
  end;

var
  { common variables }
  FString: string;
  FResourceLocale: integer;
  FDialogMinWidth: integer;
  FDialogMaxWidth: integer;
  FDialogMinHeight: integer;
  FDialogMaxHeight: integer;
  FDialogLeftSpace: integer;
  FDialogRightSpace: integer;
  FDialogTopSpace: integer;
  { form format variables }
  FPaintBoxMinWidth: integer;
  FPaintBoxMinHeight: integer;
  FPaintBoxLeft: integer;
  FPaintBoxTop: integer;
  FPaintBoxWidth: integer;
  FPaintBoxHeight: integer;
  FPaintBoxSpace: integer;
  FPaintBoxMaxLines: integer;
  FPaintBoxScrollbarWidth: integer;
  { icon format variables }
  FIconRightSpace: integer;
  { frame format variables }
  FBevelLeftSpace: integer;
  FBevelRightSpace: integer;
  FBevelTopSpace: integer;
  FBevelBottomSpace: integer;
  { button format variables }
  FStandardButtonWidth: integer;
  FCustomButtonWidth: integer;
  FButtonHeight: integer;
  FButtonLeftSpace: integer;
  FButtonRightSpace: integer;
  FButtonTopSpace: integer;
  FButtonBottomSpace: integer;
  FButtonLeading: integer;
  FButtonsLeft: integer;
  FButtonsTop: integer;
  FButtonsWidth: integer;
  FFontLeading: integer;

const
  { MessageDialog return values }
  emrNone = 0;
  emrCustom1 = 1;
  emrCustom2 = 2;
  emrCustom3 = 3;
  emrCustom4 = 4;
  emrYes = 5;
  emrYesToAll = 6;
  emrNo = 7;
  emrNoToAll = 8;
  emrOk = 9;
  emrCancel = 10;
  emrAbort = 11;
  emrRetry = 12;
  emrIgnore = 13;
  emrAll = 14;
  emrClose = 15;
  emrBack = 16;
  emrNext = 17;
  emrHelp = 18;
  { dialog text line break }
  CRLF = '/CR';
  TAB = '/TAB';

  function RTLConvertString(AString: string): string;
  function CountSubString(ASubString, AString: string): integer;
  function ExtractSubString(var AString: string; ASeparator: string): string;
  function HexToInt(AHexString: string): integer;

  procedure Register;

implementation

{$R EuMessageDialog1.res}

{ *** Component Constructor / Destructor ************************************* }

constructor TEUMessageDialog.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FBevel:=TEuMessageDialogBevel.Create;
  FBevel.Shape:=bsFrame;
  FBevel.Range:=brNone;
  FBevel.Style:=bsLowered;
  FButtons:=TEuMessageDialogButton.Create;
  FButtons.ButtonList:=[dbOK,dbCancel];
  FButtons.DefaultButton:=dbDefaultButton1;
  FButtons.CustomCaptions:=TEuMessageDialogCustomCaptions.Create;
  FButtons.CustomCaptions.FCustomButton1Caption:='';
  FButtons.CustomCaptions.FCustomButton2Caption:='';
  FButtons.CustomCaptions.FCustomButton3Caption:='';
  FButtons.CustomCaptions.FCustomButton4Caption:='';
  FCaption:='';
  FDialogList:=TList.Create;
  FOptions:=[];
  FHelpContext:=0;
  FIcon:=itNone;
  FPosition:=TEuMessageDialogPosition.Create;
  FPosition.Left:=-1;
  FPosition.Top:=-1;
  FResult:=0;
  FText:='';
end;

destructor TEUMessageDialog.Destroy;
begin
  Clear;
  FPosition.free;
  FDialogList.free;
  FButtons.FCustomCaptions.free;
  FButtons.free;
  FBevel.free;
  inherited Destroy;
end;

{ *** Dialog Execute Routines ************************************************ }

function TEUMessageDialog.Execute: word;
begin
  { copy property to dialog data structure }
  FData.BevelRange:=FBevel.Range;
  FData.BevelShape:=FBevel.Shape;
  FData.BevelStyle:=FBevel.Style;
  FData.ButtonList:=FButtons.ButtonList;
  FData.DefaultButton:=FButtons.DefaultButton;
  FData.Caption:=FCaption;
  FData.CustomButtonCaption[1]:=FButtons.CustomCaptions.CustomButton1Caption;
  FData.CustomButtonCaption[2]:=FButtons.CustomCaptions.CustomButton2Caption;
  FData.CustomButtonCaption[3]:=FButtons.CustomCaptions.CustomButton3Caption;
  FData.CustomButtonCaption[4]:=FButtons.CustomCaptions.CustomButton4Caption;
  FData.Options:=FOptions;
  FData.HelpContext:=FHelpContext;
  FData.Icon:=FIcon;
  FData.Left:=FPosition.Left;
  FData.Text:=FText;
  FData.Top:=FPosition.Top;
  { create and display dialog }
  FResult:=ExecuteA;
  Result:=FResult;
end;

function TEUMessageDialog.ExecuteDialogItem(AIndex: integer): word;
begin
  Result:=0;
  { copy property to dialog data structure }
  if (AIndex>=0) and (AIndex<FDialogList.Count) then
  begin
    FData.BevelRange:=TEuMessageDialogData(FDialogList[AIndex]^).BevelRange;
    FData.BevelShape:=TEuMessageDialogData(FDialogList[AIndex]^).BevelShape;
    FData.BevelStyle:=TEuMessageDialogData(FDialogList[AIndex]^).BevelStyle;
    FData.ButtonList:=TEuMessageDialogData(FDialogList[AIndex]^).ButtonList;
    FData.DefaultButton:=TEuMessageDialogData(FDialogList[AIndex]^).DefaultButton;
    FData.Caption:=TEuMessageDialogData(FDialogList[AIndex]^).Caption;
    FData.CustomButtonCaption[1]:=TEuMessageDialogData(FDialogList[AIndex]^).CustomButtonCaption[1];
    FData.CustomButtonCaption[2]:=TEuMessageDialogData(FDialogList[AIndex]^).CustomButtonCaption[2];
    FData.CustomButtonCaption[3]:=TEuMessageDialogData(FDialogList[AIndex]^).CustomButtonCaption[3];
    FData.CustomButtonCaption[4]:=TEuMessageDialogData(FDialogList[AIndex]^).CustomButtonCaption[4];
    FData.Options:=TEuMessageDialogData(FDialogList[AIndex]^).Options;
    FData.HelpContext:=TEuMessageDialogData(FDialogList[AIndex]^).HelpContext;
    FData.Icon:=TEuMessageDialogData(FDialogList[AIndex]^).Icon;
    FData.Left:=TEuMessageDialogData(FDialogList[AIndex]^).Left;
    FData.Text:=TEuMessageDialogData(FDialogList[AIndex]^).Text;
    FData.Top:=TEuMessageDialogData(FDialogList[AIndex]^).Top;
    { create and display dialog }
    FResult:=ExecuteA;
    Result:=FResult;
  end;
end;

function TEUMessageDialog.ExecuteA: word;
var
  FSystemMenu: HMENU;
begin
  Result:=0;
  if (Length(FData.Text)>0) and (FData.ButtonList<>[]) and (FData.ButtonList<>[dbHelp]) then
  begin
    { create dialog form incl. controls }
    if FData.HelpContext=0 then FData.HelpContext:=Application.MainForm.HelpContext;
    FMsgForm:=TForm.Create(Application.MainForm);
    try
      FMsgForm.Parent:=nil;
      FMsgForm.Name:='DialogForm';
      FMsgForm.BorderStyle:=bsDialog;
      FMsgForm.Caption:=FData.Caption;
      FMsgForm.KeyPreview:=true;
      FMsgForm.OnKeyDown:=FormKeyDown;
      FSystemMenu:=GetSystemMenu(FMsgForm.Handle,false);
      if FSystemMenu<>0 then
      begin
        EnableMenuItem(FSystemMenu,SC_CLOSE,MF_BYCOMMAND or MF_GRAYED);
        DrawMenuBar(FMsgForm.Handle);
      end;
      FMsgImage:=TImage.Create(FMsgForm);
      FMsgImage.Parent:=FMsgForm;
      FMsgImage.Name:='DialogImage';
      FMsgBevel:=TBevel.Create(FMsgForm);
      FMsgBevel.Parent:=FMsgForm;
      FMsgBevel.Name:='DialogBevel';
      FMsgScrollBox:=TScrollBox.Create(FMsgForm);
      FMsgScrollBox.Parent:=FMsgForm;
      FMsgScrollBox.Name:='DialogScrollBox';
      FMsgScrollBox.BorderStyle:=bsNone;
      FMsgPaintBox:=TPaintBox.Create(FMsgScrollBox);
      FMsgPaintBox.Parent:=FMsgScrollBox;
      FMsgPaintBox.Name:='DialogPaintBox';
      FMsgPaintBox.OnPaint:=PaintBoxPaint;
      { prepare form layout }
      FMsgImage.Picture.Icon.Handle:=LoadIcon(hInstance,pchar('ICON_'+IntToStr(ord(FData.Icon))));
      FMsgImage.Width:=FMsgImage.Picture.Width;
      FMsgImage.Height:=FMsgImage.Picture.Height;
      if SetControlDimensions then
      begin
        CreateButtons;
        PlaySystemSound;
        Result:=FMsgForm.ShowModal;
      end;
    finally
      FMsgForm.Release;
    end;
  end;
end;

{ *** Dialog Array Load/Save Routines **************************************** }

function TEUMessageDialog.SaveToFile(const AFilename: string): boolean;
var
  FCount: integer;
  FString: string;
begin
  Result:=false;
  if (FDialogList.Count>0) then
  begin
    if CreateXMLFile(AFilename) then
    begin
      try
        if not WriteXMLStartTag('Items') then Exit;
        if not WriteXMLIntValue('Count',FDialogList.Count) then Exit;
        for FCount:=0 to FDialogList.Count-1 do
        begin
          if not WriteXMLStartTag('Item'+inttostr(FCount)) then Exit;
          with TEuMessageDialogData(FDialogList[FCount]^) do
          begin
            if not WriteXMLIntValue('Range',ord(BevelRange)) then Exit;
            if not WriteXMLIntValue('Shape',ord(BevelShape)) then Exit;
            if not WriteXMLIntValue('Style',ord(BevelStyle)) then Exit;
            FString:='';
            FString:=FString+IntToStr(ord(dbCustom1 in ButtonList));
            FString:=FString+IntToStr(ord(dbCustom2 in ButtonList));
            FString:=FString+IntToStr(ord(dbCustom3 in ButtonList));
            FString:=FString+IntToStr(ord(dbCustom4 in ButtonList));
            FString:=FString+IntToStr(ord(dbYes in ButtonList));
            FString:=FString+IntToStr(ord(dbYesToAll in ButtonList));
            FString:=FString+IntToStr(ord(dbNo in ButtonList));
            FString:=FString+IntToStr(ord(dbNoToAll in ButtonList));
            FString:=FString+IntToStr(ord(dbOk in ButtonList));
            FString:=FString+IntToStr(ord(dbCancel in ButtonList));
            FString:=FString+IntToStr(ord(dbAbort in ButtonList));
            FString:=FString+IntToStr(ord(dbRetry in ButtonList));
            FString:=FString+IntToStr(ord(dbIgnore in ButtonList));
            FString:=FString+IntToStr(ord(dbAll in ButtonList));
            FString:=FString+IntToStr(ord(dbClose in ButtonList));
            FString:=FString+IntToStr(ord(dbBack in ButtonList));
            FString:=FString+IntToStr(ord(dbNext in ButtonList));
            FString:=FString+IntToStr(ord(dbHelp in ButtonList));
            if not WriteXMLStringValue('Buttons',FString) then Exit;
            if not WriteXMLIntValue('DefaultButton',ord(DefaultButton)) then Exit;
            if not WriteXMLStringValue('Caption',Caption) then Exit;
            if not WriteXMLStringValue('CustomButtonCaption1',CustomButtonCaption[1]) then Exit;
            if not WriteXMLStringValue('CustomButtonCaption2',CustomButtonCaption[2]) then Exit;
            if not WriteXMLStringValue('CustomButtonCaption3',CustomButtonCaption[3]) then Exit;
            if not WriteXMLStringValue('CustomButtonCaption4',CustomButtonCaption[4]) then Exit;
            FString:='';
            FString:=FString+IntToStr(ord(doCenterText in Options));
            FString:=FString+IntToStr(ord(doRightAlignText in Options));
            FString:=FString+IntToStr(ord(doRTLReading in Options));
            FString:=FString+IntToStr(ord(doPlaySystemSound in Options));
            FString:=FString+IntToStr(ord(doCenterIcon in Options));
            FString:=FString+IntToStr(ord(doCreateCustomButtonsLast in Options));
            FString:=FString+IntToStr(ord(doRightAlignButtons in Options));
            if not WriteXMLStringValue('Options',FString) then Exit;
            if not WriteXMLIntValue('HelpContext',HelpContext) then Exit;
            if not WriteXMLIntValue('Icon',ord(Icon)) then Exit;
            if not WriteXMLIntValue('Left',Left) then Exit;
            if not WriteXMLStringValue('Text',Text) then Exit;
            if not WriteXMLIntValue('Top',Top) then Exit;
          end;
          if not WriteXMLEndTag('Item'+inttostr(FCount)) then Exit;
        end;
        if not WriteXMLEndTag('Items') then Exit;
        Result:=true;
      finally
        CloseXMLFile;
      end;
    end;
    if not Result then DeleteFile(AFilename);
  end;
end;

function TEUMessageDialog.LoadFromFile(const AFilename: string): boolean;
var
  FItemCount: integer;
  FCount: integer;
  FPointer: PEuMessageDialogData;
  FString: string;
begin
  Result:=false;
  if OpenXMLFile(AFilename) then
  begin
    try
      if not IsXMLStartTag('Items') then Exit;
      FItemCount:=ReadXMLIntValue('Count');
      if FItemCount=0 then Exit;
      Clear;
      for FCount:=0 to FItemCount-1 do
      begin
        if not IsXMLStartTag('Item'+IntToStr(FCount)) then Exit;
        New(FPointer);
        FDialogList.Add(FPointer);
        with TEuMessageDialogData(FDialogList[FCount]^) do
        begin
          BevelRange:=TEuMessageDialogBevelRange(ReadXMLIntValue('Range'));
          BevelShape:=TBevelShape(ReadXMLIntValue('Shape'));
          BevelStyle:=TBevelStyle(ReadXMLIntValue('Style'));
          FString:=ReadXMLStringValue('Buttons');
          if Length(FString)=0 then Exit;
          ButtonList:=[];
          if FString[01]='1' then ButtonList:=ButtonList+[dbCustom1];
          if FString[02]='1' then ButtonList:=ButtonList+[dbCustom2];
          if FString[03]='1' then ButtonList:=ButtonList+[dbCustom3];
          if FString[04]='1' then ButtonList:=ButtonList+[dbCustom4];
          if FString[05]='1' then ButtonList:=ButtonList+[dbYes];
          if FString[06]='1' then ButtonList:=ButtonList+[dbYesToAll];
          if FString[07]='1' then ButtonList:=ButtonList+[dbNo];
          if FString[08]='1' then ButtonList:=ButtonList+[dbNoToAll];
          if FString[09]='1' then ButtonList:=ButtonList+[dbOk];
          if FString[10]='1' then ButtonList:=ButtonList+[dbCancel];
          if FString[11]='1' then ButtonList:=ButtonList+[dbAbort];
          if FString[12]='1' then ButtonList:=ButtonList+[dbRetry];
          if FString[13]='1' then ButtonList:=ButtonList+[dbIgnore];
          if FString[14]='1' then ButtonList:=ButtonList+[dbAll];
          if FString[15]='1' then ButtonList:=ButtonList+[dbClose];
          if FString[16]='1' then ButtonList:=ButtonList+[dbBack];
          if FString[17]='1' then ButtonList:=ButtonList+[dbNext];
          if FString[18]='1' then ButtonList:=ButtonList+[dbHelp];
          DefaultButton:=TEuMessageDialogDefaultButton(ReadXMLIntValue('DefaultButton'));
          Caption:=ReadXMLStringValue('Caption');
          CustomButtonCaption[1]:=ReadXMLStringValue('CustomButtonCaption1');
          CustomButtonCaption[2]:=ReadXMLStringValue('CustomButtonCaption2');
          CustomButtonCaption[3]:=ReadXMLStringValue('CustomButtonCaption3');
          CustomButtonCaption[4]:=ReadXMLStringValue('CustomButtonCaption4');
          FString:=ReadXMLStringValue('Options');
          if Length(FString)=0 then Exit;
          Options:=[];
          if FString[1]='1' then Options:=Options+[doCenterText];
          if FString[2]='1' then Options:=Options+[doRightAlignText];
          if FString[3]='1' then Options:=Options+[doRTLReading];
          if FString[4]='1' then Options:=Options+[doPlaySystemSound];
          if FString[5]='1' then Options:=Options+[doCenterIcon];
          if FString[6]='1' then Options:=Options+[doCreateCustomButtonsLast];
          if FString[7]='1' then Options:=Options+[doRightAlignButtons];
          HelpContext:=ReadXMLIntValue('HelpContext');
          Icon:=TEuMessageDialogIconType(ReadXMLIntValue('Icon'));
          Left:=ReadXMLIntValue('Left');
          Text:=ReadXMLStringValue('Text');
          Top:=ReadXMLIntValue('Top');
        end;
        if not IsXMLEndTag('Item'+IntToStr(FCount)) then Exit;
      end;
      if not IsXMLEndTag('Items') then Exit;
      Result:=true;
    finally
      CloseXMLFile;
    end;
  end;
  if not Result then Clear;
end;

{ *** Dialog Format Routines ************************************************* }

function TEUMessageDialog.GetOptimumPaintBoxWidth(AText: string; AIncludeScrollbar: boolean): integer;
var
  FFlag: boolean;
  FCount: integer;
  FWidth: integer;
begin
  Result:=0;
  if Length(AText)>0 then
  begin
    FFlag:=(CountSubString(CRLF,AText)+1)>FPaintBoxMaxLines;
    for FCount:=0 to CountSubString(CRLF,AText) do
    begin
      FWidth:=FMsgPaintBox.Canvas.TextWidth(ExtractSubString(AText,CRLF));
      if Result<FWidth then Result:=FWidth;
    end;
    inc(Result,(ord(AIncludeScrollbar and FFlag)*(FPaintBoxScrollbarWidth+2)));
  end;
end;

function TEUMessageDialog.GetOptimumPaintBoxHeight(AText: string; AUseDisplayLimit: boolean): integer;
begin
  Result:=0;
  if Length(AText)>0 then
  begin
    Result:=CountSubString(CRLF,AText)+1;
    if (AUseDisplayLimit) and (Result>FPaintBoxMaxLines) then Result:=FPaintBoxMaxLines;
    Result:=Result*(FMsgPaintBox.Canvas.Font.Size+FFontLeading);
  end;
end;

function TEUMessageDialog.GetDialogWidth: integer;
begin
  Result:=0;
  if GetButtonDim<=FDialogMaxWidth then
  begin
    { compute required paintbox width but not below minimum paintbox width }
    Result:=GetOptimumPaintBoxWidth(FData.Text,true);
    if Result<FPaintBoxMinWidth then Result:=FPaintBoxMinWidth;
    inc(Result,FDialogLeftSpace+FDialogRightSpace);
    { add icon related dimensions }
    if FData.Icon<>itNone then
    begin
      inc(Result,FMsgImage.Width+FIconRightSpace);
    end;
    { add bevel related dimensions }
    if FData.BevelRange<>brNone then
    begin
      if FData.BevelShape in [bsLeftLine] then inc(Result,FBevelLeftSpace);
      if FData.BevelShape in [bsRightLine] then inc(Result,FBevelRightSpace);
      if FData.BevelShape in [bsBox, bsFrame,bsSpacer] then inc(Result,FBevelLeftSpace+FBevelRightSpace);
    end;
    { report back required width }
    if Result>FDialogMaxWidth then Result:=FDialogMaxWidth;                     //limit to maximum width
    if Result<GetButtonDim then Result:=GetButtonDim;                           //buttons limit width here
  end;
end;

function TEUMessageDialog.GetDialogHeight: integer;
begin
  Result:=GetOptimumPaintBoxHeight(FData.Text,true);
  if Result<FPaintBoxMinHeight then Result:=FPaintBoxMinHeight;
  inc(Result,FDialogTopSpace);
  { add icon related dimensions }
  if FData.Icon<>itNone then
  begin
    if Result<(FDialogTopSpace+FMsgImage.Height) then
    begin
      Result:=FDialogTopSpace+FMsgImage.Height;
    end;
  end;
  { add bevel related dimensions }
  if FData.BevelRange<>brNone then
  begin
    if FData.BevelShape in [bsTopLine] then inc(Result,FBevelTopSpace);
    if FData.BevelShape in [bsBottomLine] then inc(Result,FBevelBottomSpace);
    if FData.BevelShape in [bsBox, bsFrame,bsSpacer] then inc(Result,FBevelTopSpace+FBevelBottomSpace);
  end;
  { add button related dimensions }
  inc(Result,FButtonTopSpace+FButtonHeight+FButtonBottomSpace);
  { report back required width }
  if Result>FDialogMaxHeight then Result:=0;                                    //dimension error; should not happen
end;

function TEUMessageDialog.SetControlDimensions: boolean;
begin
  Result:=false;
  FMsgForm.ClientWidth:=GetDialogWidth;
  FMsgForm.ClientHeight:=GetDialogHeight;
  if (FMsgForm.ClientWidth>0) and (FMsgForm.ClientHeight>0) then
  begin
    { set image dimensions }
    FMsgImage.Visible:=FData.Icon<>itNone;
    if FMsgImage.Visible then
    begin
      FMsgImage.Left:=FDialogLeftSpace+(FBevelLeftSpace*ord(FData.BevelRange=brAll)*
                     ord(FData.BevelShape in [bsLeftLine,bsBox,bsFrame,bsSpacer]));
      FMsgImage.Top:=FDialogTopSpace+(FBevelTopSpace*ord(FData.BevelRange=brAll)*
                     ord(not(doCenterIcon in FData.Options))*
                     ord(FData.BevelShape in [bsTopLine,bsBox,bsFrame,bsSpacer]))+
                     (((FMsgForm.ClientHeight-FDialogTopSpace-FMsgImage.Height-FButtonTopSpace-
                     FButtonHeight-FButtonBottomSpace) div 2)*ord(doCenterIcon in FData.Options));
    end;
    { set bevel dimensions }
    FMsgBevel.Visible:=FData.BevelRange<>brNone;
    if FMsgBevel.Visible then
    begin
      FMsgBevel.Shape:=FData.BevelShape;
      FMsgBevel.Style:=FData.BevelStyle;
      FMsgBevel.Left:=FDialogLeftSpace+((FMsgImage.Width+FIconRightSpace)*
                      ord(FData.Icon<>itNone)*ord(FData.BevelRange=brText));
      FMsgBevel.Top:=FDialogTopSpace;
      FMsgBevel.Width:=FMsgForm.ClientWidth-FMsgBevel.Left-FDialogRightSpace;
      FMsgBevel.Height:=FMsgForm.ClientHeight-FMsgBevel.Top-FButtonTopSpace-FButtonHeight-FButtonBottomSpace;
    end;
    { set scrollbox dimensions }
    FMsgScrollBox.Left:=FDialogLeftSpace+
                        ((FMsgImage.Width+FIconRightSpace)*ord(FData.Icon<>itNone))+
                        (FBevelLeftSpace*ord((FData.BevelRange<>brNone) and
                        (FData.BevelShape in [bsLeftLine,bsBox,bsFrame,bsSpacer])));
    FMsgScrollbox.Top:=FDialogTopSpace+(FBevelTopSpace*ord((FData.BevelRange<>brNone)
                       and (FData.BevelShape in [bsTopLine,bsBox,bsFrame,bsSpacer])));
    FMsgScrollbox.Width:=FMsgForm.ClientWidth-FMsgScrollBox.Left-FDialogRightSpace-
                         (FBevelRightSpace*ord((FData.BevelRange<>brNone) and
                         (FData.BevelShape in [bsRightLine,bsBox,bsFrame,bsSpacer])));
    FMsgScrollbox.Height:=FMsgForm.ClientHeight-FMsgScrollbox.Top-FButtonTopSpace-
                          FButtonHeight-FButtonBottomSpace-(FBevelBottomSpace*
                          ord((FData.BevelRange<>brNone) and
                          (FData.BevelShape in [bsBottomLine,bsBox,bsFrame,bsSpacer])));
    { set paintbox object }
    FMsgPaintBox.Left:=0;
    FMsgPaintBox.Top:=0;
    FMsgPaintBox.Width:=GetOptimumPaintBoxWidth(FData.Text,false);
    FMsgPaintBox.Height:=GetOptimumPaintBoxHeight(FData.Text,false);
    { set dialog horizontal position }
    if FData.Left<0 then
    begin
      FMsgForm.Left:=(Screen.Width-FMsgForm.Width) div 2;
    end
    else
    begin
      if FData.Left+FMsgForm.Width>Screen.Width then
      begin
        FMsgForm.Left:=Screen.Width-FMsgForm.Width;
      end
      else
      begin
        FMsgForm.Left:=FData.Left;
      end;
    end;
    if FMsgForm.Left<0 then FMsgForm.Left:=0;                                   //always start at left
    { set vertical horizontal position }
    if FData.Top<0 then
    begin
      FMsgForm.Top:=(Screen.Height-FMsgForm.Height) div 2;
    end
    else
    begin
      if FData.Top+FMsgForm.Height>Screen.Height then
      begin
        FMsgForm.Top:=Screen.Height-FMsgForm.Height;
      end
      else
      begin
        FMsgForm.Top:=FData.Top;
      end;
    end;
    if FMsgForm.Top<0 then FMsgForm.Top:=0;                                     //always start at top
    { set function result }
    Result:=true;
  end;
end;

{ *** DialogButton(s) Routines *********************************************** }

function TEUMessageDialog.GetButtonDim: integer;
var
  FStandardButtons : integer;
  FCustomButtons   : integer;
begin
  Result:=0;
  FCustomButtons:=0;
  inc(FCustomButtons,ord(dbCustom1 in FData.ButtonList));
  inc(FCustomButtons,ord(dbCustom2 in FData.ButtonList));
  inc(FCustomButtons,ord(dbCustom3 in FData.ButtonList));
  inc(FCustomButtons,ord(dbCustom4 in FData.ButtonList));
  FStandardButtons:=0;
  inc(FStandardButtons,ord(dbYes in FData.ButtonList));
  inc(FStandardButtons,ord(dbYesToAll in FData.ButtonList));
  inc(FStandardButtons,ord(dbNo in FData.ButtonList));
  inc(FStandardButtons,ord(dbNoToAll in FData.ButtonList));
  inc(FStandardButtons,ord(dbOk in FData.ButtonList));
  inc(FStandardButtons,ord(dbCancel in FData.ButtonList));
  inc(FStandardButtons,ord(dbAbort in FData.ButtonList));
  inc(FStandardButtons,ord(dbRetry in FData.ButtonList));
  inc(FStandardButtons,ord(dbIgnore in FData.ButtonList));
  inc(FStandardButtons,ord(dbAll in FData.ButtonList));
  inc(FStandardButtons,ord(dbClose in FData.ButtonList));
  inc(FStandardButtons,ord(dbBack in FData.ButtonList));
  inc(FStandardButtons,ord(dbNext in FData.ButtonList));
  inc(FStandardButtons,ord(dbHelp in FData.ButtonList));
  if FStandardButtons+FCustomButtons>0 then
  begin
    Result:=FButtonLeftSpace+FButtonRightSpace-FButtonLeading;
    Result:=Result+(FStandardButtons*(FStandardButtonWidth+FButtonLeading));
    Result:=Result+(FCustomButtons*(FCustomButtonWidth+FButtonLeading));
  end;
end;

procedure TEUMessageDialog.CreateButtons;
var
  FIndex : integer;
  FResourceStringBase: integer;

  procedure CreateButton(var AIndex: integer; AButton: TEuMessageDialogButtons);
  begin
    inc(AIndex);
    { create and set button }
    with TButton.Create(FMsgForm) do
    begin
      { set button properties }
      Parent:=FMsgForm;
      Name:='frmMessageButton'+InttoStr(AIndex);
      Left:=FButtonsLeft;
      Top:=FButtonsTop;
      Height:=FButtonHeight;
      { set button tag value }
      Tag:=emrNone;
      if AButton=[dbCustom1] then Tag:=emrCustom1;
      if AButton=[dbCustom2] then Tag:=emrCustom2;
      if AButton=[dbCustom3] then Tag:=emrCustom3;
      if AButton=[dbCustom4] then Tag:=emrCustom4;
      if AButton=[dbYes] then Tag:=emrYes;
      if AButton=[dbYesToAll] then Tag:=emrYesToAll;
      if AButton=[dbNo] then Tag:=emrNo;
      if AButton=[dbNoToAll] then Tag:=emrNoToAll;
      if AButton=[dbOk] then Tag:=emrOk;
      if AButton=[dbCancel] then Tag:=emrCancel;
      if AButton=[dbAbort] then Tag:=emrAbort;
      if AButton=[dbRetry] then Tag:=emrRetry;
      if AButton=[dbIgnore] then Tag:=emrIgnore;
      if AButton=[dbAll] then Tag:=emrAll;
      if AButton=[dbClose] then Tag:=emrClose;;
      if AButton=[dbBack] then Tag:=emrBack;;
      if AButton=[dbNext] then Tag:=emrNext;
      if AButton=[dbHelp] then Tag:=emrHelp;
      { set button caption and width }
      if (AButton=[dbCustom1]) or (AButton=[dbCustom2]) or
         (AButton=[dbCustom3]) or (AButton=[dbCustom4]) then
      begin
        Width:=FCustomButtonWidth;
        Caption:=FData.CustomButtonCaption[Tag];
      end
      else
      begin
        Width:=FStandardButtonWidth;
        case FResourceLocale mod 256 of
          $07  : FResourceStringBase:=2200;                                     {german captions}
          $09  : FResourceStringBase:=2000;                                     {english captions}
          //$0C  : FResourceStringBase:=2100;                                     {french captions}
          //$10  : FResourceStringBase:=2300;                                     {italian captions}
          //$16  : FResourceStringBase:=2400;                                     {portuguese captions}
          //$0A  : FResourceStringBase:=2500;                                     {spanish captions}
          //$1D  : FResourceStringBase:=2600;                                     {swedish captions}
        else
          FResourceStringBase:=2000;
        end;
        Caption:=LoadStr(FResourceStringBase+Tag);
      end;
      FButtonsLeft:=FButtonsLeft+Width+FButtonLeading;
      OnClick:=ButtonClick;
      { set default button if not Help button }
      if AIndex=1 then Default:=(FData.DefaultButton=dbDefaultButton1);
      if AIndex=2 then Default:=(FData.DefaultButton=dbDefaultButton2);
      if AIndex=3 then Default:=(FData.DefaultButton=dbDefaultButton3);
      if AIndex=4 then Default:=(FData.DefaultButton=dbDefaultButton4);
      { set Cancel property if possible }
      Cancel:=((AButton=[dbClose]) or (AButton=[dbAbort]) or (AButton=[dbCancel]));
      { set special Help button properties }
      if AButton=[dbHelp] then
      begin
        Default:=false;
        HelpContext:=FData.HelpContext;
      end;
    end;
  end;

begin
  FButtonsLeft:=FButtonLeftSpace+((FMsgForm.ClientWidth-GetButtonDim) div (2-ord(doRightAlignButtons in FData.Options)));
  FButtonsTop:=FMsgForm.ClientHeight-FButtonHeight-FButtonBottomSpace;
  FIndex:=0;
  { create custom buttons at first if requested }
  if not(doCreateCustomButtonsLast in FData.Options) then
  begin
    if dbCustom1 in FData.ButtonList then CreateButton(FIndex,[dbCustom1]);
    if dbCustom2 in FData.ButtonList then CreateButton(FIndex,[dbCustom2]);
    if dbCustom3 in FData.ButtonList then CreateButton(FIndex,[dbCustom3]);
    if dbCustom4 in FData.ButtonList then CreateButton(FIndex,[dbCustom4]);
  end;
  { create standard buttons }
  if dbYes in FData.ButtonList then CreateButton(FIndex,[dbYes]);
  if dbYesToAll in FData.ButtonList then CreateButton(FIndex,[dbYesToAll]);
  if dbNo in FData.ButtonList then CreateButton(FIndex,[dbNo]);
  if dbNoToAll in FData.ButtonList then CreateButton(FIndex,[dbNoToAll]);
  if dbOk in FData.ButtonList then CreateButton(FIndex,[dbOk]);
  if dbCancel in FData.ButtonList then CreateButton(FIndex,[dbCancel]);
  if dbAbort in FData.ButtonList then CreateButton(FIndex,[dbAbort]);
  if dbRetry in FData.ButtonList then CreateButton(FIndex,[dbRetry]);
  if dbIgnore in FData.ButtonList then CreateButton(FIndex,[dbIgnore]);
  if dbAll in FData.ButtonList then CreateButton(FIndex,[dbAll]);
  if dbClose in FData.ButtonList then CreateButton(FIndex,[dbClose]);
  if dbBack in FData.ButtonList then CreateButton(FIndex,[dbBack]);
  if dbNext in FData.ButtonList then CreateButton(FIndex,[dbNext]);
  if dbHelp in FData.ButtonList then CreateButton(FIndex,[dbHelp]);
  { create custom buttons at last [default] }
  if doCreateCustomButtonsLast in FData.Options then
  begin
    if dbCustom1 in FData.ButtonList then CreateButton(FIndex,[dbCustom1]);
    if dbCustom2 in FData.ButtonList then CreateButton(FIndex,[dbCustom2]);
    if dbCustom3 in FData.ButtonList then CreateButton(FIndex,[dbCustom3]);
    if dbCustom4 in FData.ButtonList then CreateButton(FIndex,[dbCustom4]);
  end;
end;

{ --- Form Event Handlers ---------------------------------------------------- }

procedure TEUMessageDialog.FormKeyDown(Sender: TObject; var Key: Word; Shift: TShiftState);
begin
  if (Key = VK_F4) and (ssAlt in Shift) then Key := 0;
end;

procedure TEUMessageDialog.PaintBoxPaint(Sender: TObject);
var
  FLeft   : integer;
  FTop    : integer;
  FCount  : integer;
  FText   : string;
  FString : string;
begin
  FText:=FData.Text;
  FTop:=0;
  for FCount:=1 to CountSubString(CRLF,FText)+1 do
  begin
    FString:=ExtractSubString(FText,CRLF);
    if (doRTLReading in FData.Options) then FString:=RTLConvertString(FString);
    FLeft:=0;
    if (doCenterText in FData.Options) then
    begin
      FLeft:=(TPaintbox(Sender).Width-TPaintbox(Sender).Canvas.TextWidth(FString)) div 2;
    end;
    if (doRightAlignText in FData.Options) then
    begin
      FLeft:=TPaintbox(Sender).Width-TPaintbox(Sender).Canvas.TextWidth(FString);
    end;
    TPaintbox(Sender).Canvas.TextOut(FLeft,FTop,FString);
    FTop:=FTop+TPaintbox(Sender).Canvas.Font.Size+FFontLeading;
  end;
end;

procedure TEUMessageDialog.ButtonClick(Sender: TObject);
begin
  FMsgForm.ModalResult:=TButton(Sender).Tag;
  if FMsgForm.ModalResult=emrHelp then
  begin
    Application.HelpContext(TButton(Sender).HelpContext);
  end;
end;

{ *** Dialog List Routines *************************************************** }

function TEUMessageDialog.Add: integer;
var
  FPointer: PEuMessageDialogData;
begin
  New(FPointer);
  FPointer^.BevelRange:=FBevel.Range;
  FPointer^.BevelShape:=FBevel.Shape;
  FPointer^.BevelStyle:=FBevel.Style;
  FPointer^.ButtonList:=FButtons.ButtonList;
  FPointer^.DefaultButton:=FButtons.FDefaultButton;
  FPointer^.Caption:=FCaption;
  FPointer^.CustomButtonCaption[1]:=FButtons.FCustomCaptions.FCustomButton1Caption;
  FPointer^.CustomButtonCaption[2]:=FButtons.FCustomCaptions.FCustomButton2Caption;
  FPointer^.CustomButtonCaption[3]:=FButtons.FCustomCaptions.FCustomButton3Caption;
  FPointer^.CustomButtonCaption[4]:=FButtons.FCustomCaptions.FCustomButton4Caption;
  FPointer^.Options:=Options;
  FPointer^.HelpContext:=FHelpContext;
  FPointer^.Icon:=FIcon;
  FPointer^.Left:=FPosition.Left;
  FPointer^.Top:=FPosition.Top;
  FPointer^.Text:=FText;
  Result:=FDialogList.Add(FPointer);
end;

procedure TEUMessageDialog.Clear;
var
  FCount: integer;
begin
  if FDialogList.Count>0 then
  begin
    For FCount:=0 to FDialogList.Count-1 do
    begin
      Dispose(FDialogList[FCount]);
    end;
    FDialogList.Clear;
  end;
end;

procedure TEUMessageDialog.Delete(AIndex: integer);
begin
  if (AIndex>=0) and (AIndex<=FDialogList.Count) then
  begin
    Dispose(FDialogList[AIndex]);
    FDialogList.Delete(AIndex);
  end;
end;

function TEUMessageDialog.Insert(AIndex: integer): integer;
var
  FPointer: PEuMessageDialogData;
begin
  if (FDialogList.Count=0) or (AIndex>=FDialogList.Count) then
  begin
    Result:=Add;
  end
  else
  begin
    if (AIndex<=0) then AIndex:=0;
    New(FPointer);
    FPointer^.BevelRange:=FBevel.Range;
    FPointer^.BevelShape:=FBevel.Shape;
    FPointer^.BevelStyle:=FBevel.Style;
    FPointer^.ButtonList:=FButtons.ButtonList;
    FPointer^.DefaultButton:=FButtons.FDefaultButton;
    FPointer^.Caption:=FCaption;
    FPointer^.CustomButtonCaption[1]:=FButtons.FCustomCaptions.FCustomButton1Caption;
    FPointer^.CustomButtonCaption[2]:=FButtons.FCustomCaptions.FCustomButton2Caption;
    FPointer^.CustomButtonCaption[3]:=FButtons.FCustomCaptions.FCustomButton3Caption;
    FPointer^.CustomButtonCaption[4]:=FButtons.FCustomCaptions.FCustomButton4Caption;
    FPointer^.Options:=Options;
    FPointer^.HelpContext:=FHelpContext;
    FPointer^.Icon:=FIcon;
    FPointer^.Left:=FPosition.Left;
    FPointer^.Top:=FPosition.Top;
    FPointer^.Text:=FText;
    FDialogList.Insert(AIndex,FPointer);
    Result:=AIndex;
  end;
end;

{ *** System Sound Routines ************************************************** }

procedure TEUMessageDialog.PlaySystemSound;
begin
  if (doPlaySystemSound in Options) then
  begin
    case FData.Icon of
      itError       :  PlaySound(pChar('SYSTEMHAND'),0,SND_ASYNC);
      itWarning     :  PlaySound(pChar('SYSTEMEXCLAMATION'),0,SND_ASYNC);
      itInformation :  PlaySound(pChar('SYSTEMASTERISK'),0,SND_ASYNC);
      itQuestion    :  PlaySound(pChar('SYSTEMQUESTION'),0,SND_ASYNC);
    else
      PlaySound(pChar('SYSTEMWELCOME'),0,SND_ASYNC);
    end;
  end;
end;

{ *** Property Access Routines *********************************************** }

function TEUMessageDialog.GetDialogListCount: Integer;
begin
  Result:=FDialogList.Count;
end;

{ *** Suppoer Routines ******************************************************* }

function RTLConvertString(AString: string): string;
var
  FCount : integer;
begin
  Result:='';
  if Length(AString)>0 then
  begin
    for FCount:=Length(AString) downto 1 do
    begin
      case ord(AString[FCount]) of
        040: Result:=Result+#041;
        041: Result:=Result+#040;
        091: Result:=Result+#093;
        093: Result:=Result+#091;
        123: Result:=Result+#125;
        125: Result:=Result+#123;
      else
        Result:=Result+AString[FCount];
      end;
    end;
  end;
end;

function CountSubString(ASubString, AString: string): integer;
var
  FPosition: integer;
begin
  Result:=0;
  if (Length(AString)>0) and (Length(ASubString)>0) and (Length(AString)>=Length(ASubString)) then
  begin
    for FPosition:=1 to Length(AString)-Length(ASubString)+1 do
    begin
      if AString[FPosition]=ASubString[1] then
      begin
        inc(Result,ord(Copy(AString,FPosition,Length(ASubString))=ASubString));
      end;
    end;
  end;
end;

function ExtractSubString(var AString: string; ASeparator: string): string;
var
  FPosition: integer;
begin
  Result:=AString;
  FPosition:=Pos(ASeparator,AString);
  if FPosition=0 then AString:='';
  if FPosition=1 then Result:='';
  if FPosition>1 then
  begin
    Result:=Copy(AString,1,FPosition-1);
    if FPosition+Length(ASeparator)>Length(AString) then
      AString:=''
    else
      AString:=Copy(AString,FPosition+Length(ASeparator),Length(AString));
  end;
end;

function HexToInt(AHexString: string): integer;
var
  FCount: integer;
  FDigit: byte;
  FBoolean: boolean;
begin
  Result:=-1;
  FCount:=0;
  if (Length(AHexString)>0) or (Length(AHexString)<9) then
  begin
    AHexString:=Uppercase(AHexString);
    repeat
      inc(FCount);
      FDigit:=Byte(AHexString[FCount]);
      FBoolean:=((FDigit<48) or ((FDigit>57) and (FDigit<64)) or (FDigit>69));
    until (FCount=Length(AHexString)) or (FBoolean);
    if not FBoolean then
    begin
      Result:=0;
      for FCount:=1 to Length(AHexString) do
      begin
        FDigit:=Byte(AHexString[FCount])-48;
        if FDigit>9 then FDigit:=FDigit-7;
        Result:=(Result*16)+FDigit;
      end;
    end;
  end;
end;

{ *** Component Registering Routines ***************************************** }

procedure Register;
begin
  RegisterComponents('Dialogs',[TEuMessageDialog]);
end;

{ *** Component Initialization Routine *************************************** }

initialization
  { dialog dimensions }
  FDialogMinWidth:=130;
  FDialogMaxWidth:=trunc(Screen.Width*0.75);
  FDialogMinHeight:=73;
  FDialogMaxHeight:=trunc(Screen.Height*0.75);
  FDialogLeftSpace:=11;
  FDialogRightSpace:=11;
  FDialogTopSpace:=11;
  { icon/bitmap dimensions }
  FIconRightSpace:=11;
  { icon/bitmap dimensions }
  FPaintBoxMaxLines:=10;
  FPaintBoxMinWidth:=108;                                                       //also valid for scrollbox
  FPaintBoxScrollbarWidth:=GetSystemMetrics(SM_CXVSCROLL);
  { bevel dimensions }
  FBevelLeftSpace:=8;
  FBevelRightSpace:=8;
  FBevelTopSpace:=8;
  FBevelBottomSpace:=8;
  { button dimensions }
  FStandardButtonWidth:=75;
  FCustomButtonWidth:=100;
  FButtonHeight:=23;
  FButtonLeading:=8;
  FButtonLeftSpace:=26;
  FButtonRightSpace:=26;
  FButtonTopSpace:=11;
  FButtonBottomSpace:=11;
  { font dimensions }
  FFontLeading:=5;
  {read locale constant from system}
  with TRegistry.Create do
  begin
    if OpenKey('\Control Panel\Desktop\ResourceLocale',false) then
    begin
      FString:=ReadString('');
      FResourceLocale:=HexToInt(FString);
    end
    else
    begin
      RootKey:=HKEY_USERS;
      if OpenKey('\.DEFAULT\Control Panel\International',false) then
      begin
        FString:=ReadString('Locale');
        FString:=UpperCase(Copy(FString,Length(FString)-3,4));
        FResourceLocale:=(HexToInt(FString));
      end;
    end;
    Free;
  end;
end.

