{# Copyright Notice ====================================================*
   This file contains proprietary information of
   Emkari, LLC. / Michael T.Carey.
   Copying or reproduction without prior written approval is prohibited.
   Copyright (c) 1996 - 1998
=================================================== Copyright Notice ===}
unit Emkari724Grid;

interface

uses
  Windows, Messages, SysUtils, Graphics, Classes, Controls, StdCtrls, Grids;

type
  TDayOfWeek = (Sunday, Monday, Tuesday, Wednesday, Thursday, Friday, Saturday);
  THourOfDay = 0..23;
  T724ByteArray = array[0..20] of byte;

  TOnScheduleChangedEvent = procedure(Sender: TObject; HexBits: string) of object;

  TEmkari724Grid = class(TDrawGrid)
  private
    { Private declarations }
    FMouseDown, FBoundsSet: boolean;
    FBits: TBits;
    FOnScheduleChanged: TOnScheduleChangedEvent;
    procedure CMFontChanged(var Message: TMessage); message CM_FONTCHANGED;
    procedure WMSize(var Msg: TWMSize); message WM_SIZE;
    function GetHourOfDaySet(Day: TDayOfWeek; Hour: THourOfDay): boolean;
    procedure SetHourOfDaySet(Day: TDayOfWeek; Hour: THourOfDay; Value: boolean);
    function GetBitsAsHexString: string;
    procedure SetBitsFromHexString(Value: string);
    function GetBitsAsBytes: T724ByteArray;
    procedure SetBitsFromBytes(Value: T724ByteArray);
  protected
    { Protected declarations }
    procedure CreateParams(var Params: TCreateParams); override;
    procedure DrawCell(ACol, ARow: Longint; ARect: TRect; AState:
                       TGridDrawState); override;
    procedure KeyUp(var Key: Word; Shift: TShiftState); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState;
                      X, Y: Integer); override;
    function SelectCell(ACol, ARow: Longint): Boolean; override;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function DateTimeIsScheduled(ADateTime: TDateTime): boolean;
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
    property ByteBits: T724ByteArray read GetBitsAsBytes write SetBitsFromBytes;
    property HexBits: string read GetBitsAsHexString write SetBitsFromHexString;
    property HourOfDaySet[Day: TDayOfWeek; Hour: THourOfDay]: boolean
               read GetHourOfDaySet write SetHourOfDaySet;
  published
    { Published declarations }
    property OnScheduleChanged: TOnScheduleChangedEvent
               read FOnScheduleChanged write FOnScheduleChanged;
  end;

  procedure Register;
  
implementation
{$R *.DCR}

procedure Register;
begin
  RegisterComponents('Emkari', [TEmkari724Grid]);
end;

function TEmkari724Grid.GetHourOfDaySet(Day: TDayOfWeek; Hour: THourOfDay): boolean;
begin
  result := false;
  if csLoading in ComponentState then exit;
  result := FBits[(ord(Day) * 24) + Hour];
end;

procedure TEmkari724Grid.SetHourOfDaySet(Day: TDayOfWeek; Hour: THourOfDay;
                                      Value: boolean);
begin
  if csLoading in ComponentState then exit;
  FBits[(ord(Day) * 24) + Hour] := Value;
end;

procedure TEmkari724Grid.CMFontChanged(var Message: TMessage);
begin
  FBoundsSet := false;
  inherited;
end;

procedure TEmkari724Grid.WMSize(var Msg: TWMSize);
begin
  inherited;
  if not FBoundsSet then
    begin
      FBoundsSet := true;
      Canvas.Font := Font;
      DefaultColWidth := Canvas.TextWidth('12') + 5;
      DefaultRowHeight := DefaultColWidth;
      ColWidths[0] := Canvas.TextWidth(ShortDayNames[4]) + 5;
      ColCount := 25;
      RowCount := 9;
      Width := (DefaultColWidth * 24) + ColWidths[0] + 4;
      Height := (DefaultRowHeight * 9) + 3;
    end;
end;

procedure TEmkari724Grid.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
begin
  if fBoundsSet then
    begin
      AWidth := (DefaultColWidth * 24) + ColWidths[0] + 4;
      AHeight := (DefaultRowHeight * 9) + 3;
    end;
  inherited SetBounds(ALeft, ATop, AWidth, AHeight);
end;

constructor TEmkari724Grid.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FBoundsSet := false;
  FBits := TBits.Create;
  FBits.Size := 7 * 24;
  FMouseDown := false;
  ScrollBars := ssNone;
  FixedCols := 0;
  FixedRows := 0;
  DefaultDrawing := false;
  Options := [];
  FOnScheduleChanged := nil;
end;

procedure TEmkari724Grid.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
end;

destructor TEmkari724Grid.Destroy;
begin
  if FBits <> nil then FBits.Free;
  inherited Destroy;
end;

function TEmkari724Grid.DateTimeIsScheduled(ADateTime: TDateTime): boolean;
var
  Day, Hour, Min, Sec, MSec: Word;
begin
  result := false;
  if FBits = nil then exit;
  Day := SysUtils.DayOfWeek(ADateTime);
  dec(Day);
  DecodeTime(ADateTime, Hour, Min, Sec, MSec);
  result := FBits[(Day * 24) + Hour];
end;

procedure Frame3D(ACanvas: TCanvas; var Rect: TRect;
                  TopColor, BottomColor: TColor; Width: Integer);
var
  OldColor: TColor;
  OldWidth: integer;
  procedure DoRect;
  var
    TopRight, BottomLeft: TPoint;
  begin
    with ACanvas, Rect do
      begin
        TopRight.X := Right;
        TopRight.Y := Top;
        BottomLeft.X := Left;
        BottomLeft.Y := Bottom;
        Pen.Color := TopColor;
        PolyLine([BottomLeft, TopLeft, TopRight]);
        Pen.Color := BottomColor;
        Dec(BottomLeft.X);
        PolyLine([TopRight, BottomRight, BottomLeft]);
      end;
  end;
begin
  OldWidth := ACanvas.Pen.Width;
  ACanvas.Pen.Width := 1;
  OldColor := ACanvas.Pen.Color;
  Dec(Rect.Bottom); Dec(Rect.Right);
  while Width > 0 do
  begin
    Dec(Width);
    DoRect;
    InflateRect(Rect, -1, -1);
  end;
  Inc(Rect.Bottom); Inc(Rect.Right);
  ACanvas.Pen.Color := OldColor;
  ACanvas.Pen.Width := OldWidth;
end;

procedure TEmkari724Grid.DrawCell(ACol, ARow: LongInt; ARect: TRect;
                               AState: TGridDrawState);
var
  CheckD, OffLeft, OffTop: integer;
  R: TRect;
  S: string;
begin
  CheckD := Canvas.TextHeight('M');
  OffTop := ((ARect.Bottom - ARect.Top) - CheckD) div 2;
  OffLeft := ((ARect.Right - ARect.Left) - CheckD) div 2;
  if (ARow in [0,8]) or (ACol = 0) then
    begin
      R := ARect;
      Canvas.Brush.Color := clSilver;
      Canvas.FillRect(ARect);
      if ARow = 8 then
        begin
          Canvas.Pen.Color := clGray;
          Canvas.MoveTo(R.Left, R.Top+1);
          Canvas.LineTo(R.Right, R.Top+1);
          Canvas.MoveTo(R.Left, R.Bottom-2);
          Canvas.LineTo(R.Right, R.Bottom-2);
          Canvas.Pen.Color := clWhite;
          Canvas.MoveTo(R.Left, R.Top);
          Canvas.LineTo(R.Right, R.Top);
          Canvas.MoveTo(R.Left, R.Bottom-1);
          Canvas.LineTo(R.Right, R.Bottom-1);
          Canvas.Pen.Color := clBlack;
          Canvas.MoveTo(R.Left, R.Bottom);
          Canvas.LineTo(R.Right, R.Bottom);
          inc(R.Top, 3);
          dec(R.Bottom, 3);
          case ACol of
            1..12:
              begin
                Canvas.Brush.Color := clBlue;
                Canvas.FillRect(R);
              end;
            else // 13..24
              begin
                Canvas.Brush.Color := clBlack;
                Canvas.FillRect(R);
              end;
          end;
          if ACol in [7,19] then
            begin
              dec(R.Left, DefaultColWidth div 2);
              Canvas.Brush.Style := bsClear;
              Canvas.Font.Color := clWhite;
              if ACol = 7 then
                S := TimeAMString
              else
                S := TimePMString;
              Canvas.TextOut(R.Left,R.Top + OffTop - 3,S);
              Canvas.Brush.Style := bsSolid;
            end;
        end
      else
      if fMouseDown and (gdFocused in AState) then
        Frame3D(Canvas, R, clBtnShadow, clBtnHighlight, 1)
      else
        Frame3D(Canvas, R, clBtnHighlight, clBtnShadow, 1);
      if (ARow = 0) then
        case ACol of
          0:      begin
                    dec(R.Top);
                    InflateRect(R, -OffLeft, -OffTop);
                    if HourOfDaySet[TDayOfWeek(0), 0] then
                      Canvas.Brush.Color := clSilver
                    else
                      Canvas.Brush.Color := clLime;
                    Canvas.Polygon([Point(R.Left, R.Top), Point(R.Right, R.Top),
                      Point(R.Right, R.Bottom), Point(R.Left, R.Bottom)]);
                  end;
          1..12:  begin
                    if ACol <> 1 then
                      S := IntToStr(ACol-1)
                    else
                      S := IntToStr(12);
                    R.Left := R.Left + (((R.Right - R.Left) -
                                         Canvas.TextWidth(S)) div 2);
                    Canvas.Font.Color := clBlue;
                    Canvas.TextRect(R,R.Left,R.Top + OffTop - 1,S);

                  end;
          13..24: begin
                    if ACol <> 13 then
                      S := IntToStr(ACol-13)
                    else
                      S := '12';
                    R.Left := R.Left +
                      (((R.Right - R.Left) - Canvas.TextWidth(S)) div 2);
                    Canvas.Font.Color := clBlack;
                    Canvas.TextRect(R,R.Left, R.Top + OffTop - 1, S);
                  end;
        end
      else
      if (ACol = 0) and (ARow <> 8) then
        begin
          S := ShortDayNames[ARow];
          case ARow of
            1, 7: Canvas.Font.Color := clBlue;
            else Canvas.Font.Color := clBlack;
          end;
          Canvas.TextRect(R,R.Left + 1, R.Top + OffTop - 1, S);
        end;
    end
  else
    begin
      Canvas.Brush.Color := clSilver;
      Canvas.FillRect(ARect);
      R.Left := ARect.Left + OffLeft;
      R.Right := R.Left + CheckD;
      R.Top := ARect.Top + OffTop;
      R.Bottom := R.Top + CheckD;
      inflateRect(R,1,1);
      dec(R.Bottom);
      dec(R.Right);
      if HourOfDaySet[TDayOfWeek(ARow - 1), ACol-1] then
        Canvas.Brush.Color := clLime;
      Canvas.FillRect(R);
      Canvas.PolyGon([Point(R.Left, R.Top),Point(R.Right, R.Top),
                      Point(R.Right, R.Bottom),Point(R.Left, R.Bottom)]);
    end;
  inherited;
end;

function TEmkari724Grid.SelectCell(ACol, ARow: LongInt): boolean;
var
  i,j: integer;
  SelState: boolean;
begin
  if ARow = 8 then
    begin
      result := false;
      if ACol in [6,18] then
        InvalidateCell(ACol+1, ARow);
      exit;
    end;
  if (ARow = 0) or (ACol = 0) then
    begin
      FMouseDown := true;
      InvalidateCell(ACol, ARow);
    end;
  result := inherited SelectCell(ACol, ARow);
  if result and (ARow > 0) and (ACol > 0) then
    begin
      HourOfDaySet[TDayOfWeek(ARow - 1), ACol-1] :=
        not HourOfDaySet[TDayOfWeek(ARow - 1), ACol-1];
      if (ACol = 1) and (ARow = 1) then
        InvalidateCell(0, 0);
    end;
  if (ARow + ACol = 0) then
    begin
      SelState := not HourOfDaySet[TDayOfWeek(0), 0];
      for i := 1 to 7 do
        begin
          for j := 1 to 24 do
            HourOfDaySet[TDayOfWeek(i - 1), j-1] := SelState;
          InvalidateRow(i);
        end;
    end
  else
  if (ARow = 0) or (ACol = 0) then
    begin
      if (ARow = 0) and (ACol > 0) then
        begin
          SelState := not HourOfDaySet[TDayOfWeek(0), ACol-1];
          for i := 1 to 7 do
            HourOfDaySet[TDayOfWeek(i - 1), ACol-1] := SelState;
          InvalidateCol(ACol);
          if ACol in [6,18] then
            InvalidateCell(ACol+1, 8);
        end
      else
      if (ACol = 0) and (ARow > 0) then
        begin
          SelState := not HourOfDaySet[TDayOfWeek(ARow - 1), 0];
          for i := 1 to 24 do
            HourOfDaySet[TDayOfWeek(ARow - 1), i-1] := SelState;
          InvalidateRow(ARow);
        end;
    end;
  if Assigned(FOnScheduleChanged) then
    FOnScheduleChanged(Self, HexBits);
end;

procedure TEmkari724Grid.KeyUp(var Key: Word; Shift: TShiftState);
var
  GC: TGridRect;
begin
  FMouseDown := false;
  GC := Selection;
  inherited KeyUp(Key, Shift);
  InvalidateCell(GC.TopLeft.x, GC.TopLeft.y);
end;

procedure TEmkari724Grid.MouseUp(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
var
  GC: TGridCoord;
begin
  FMouseDown := false;
  GC := MouseCoord(X, Y);
  inherited MouseUp(Button, Shift, X, Y);
  InvalidateCell(GC.x, GC.y);
end;

function TEmkari724Grid.GetBitsAsHexString: string;
var
  i,j,k: integer;
  b: byte;
begin
  result := '';
  if fBits = nil then exit;
  for i := 0 to 6 do
    for j := 0 to 2 do
      begin
        b := 0;
        for k := 0 to 7 do
          if HourOfDaySet[TDayOfWeek(i), (j * 8) + k] then
            b := b or (1 shl k);
        result := result + IntToHex(b, 2);
      end;
end;

procedure TEmkari724Grid.SetBitsFromHexString(Value: string);
var
  i,j,k: integer;
  b: byte;
  S: string;
begin
  if (FBits = nil) or (length(Value) < 42) then exit;
  S := Value;
  for i := 0 to 6 do
    for j := 0 to 2 do
      begin
        b := byte(StrToInt('$' + S[1] + S[2]));
        system.Delete(S, 1, 2);
        for k := 0 to 7 do
          HourOfDaySet[TDayOfWeek(i), (j * 8) + k] := (b and (1 shl k) <> 0);
      end;
  Refresh;
end;

function TEmkari724Grid.GetBitsAsBytes: T724ByteArray;
var
  i,j: integer;
  S: string;
begin
  if FBits = nil then exit;
  fillchar(result, sizeof(result), 0);
  S := GetBitsAsHexString;
  i := 1; j := 0;
  while i < 43 do
    begin
      result[j] := byte(StrToInt('$' + S[i] + S[i + 1]));
      inc(i, 2);
      inc(j);
    end;
end;

procedure TEmkari724Grid.SetBitsFromBytes(Value: T724ByteArray);
var
  i: integer;
  S: string;
begin
  if (FBits = nil) then exit;
  S:= '';
  for i := 0 to 20 do
    S := S + IntToHex(Value[i], 2);
  SetBitsFromHexString(S);
end;


end.
