unit ScriptDemoWin;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Buttons, StdCtrls, dwsComp, dwsTokenizer, Menus, dwsUtilsLib, dwsDemoLib,
  dwsErrors, ComCtrls, ExtCtrls;

type
  TfrmScriptDemo = class(TForm)
    MainMenu1: TMainMenu;
    FileMenu: TMenuItem;
    NewItem: TMenuItem;
    OpenItem: TMenuItem;
    SaveItem: TMenuItem;
    ExitItem: TMenuItem;
    N1: TMenuItem;
    ScriptMenu: TMenuItem;
    CompileItem: TMenuItem;
    ExecuteItem: TMenuItem;
    N2: TMenuItem;
    ShowTreeItem: TMenuItem;
    DemosMenu: TMenuItem;
    OpenDialog1: TOpenDialog;
    SaveDialog1: TSaveDialog;
    N3: TMenuItem;
    VariablesItem: TMenuItem;
    HelpMenu: TMenuItem;
    StopItem: TMenuItem;
    BlockItem: TMenuItem;
    SaveAsItem: TMenuItem;
    dwsUtilsLib1: TdwsUtilsLib;
    fncPrint: TdwsFunc;
    fncGetCapitals: TdwsFunc;
    script: TDelphiWebScript;
    CommentModeItem: TMenuItem;
    dwsDemoLib1: TdwsDemoLib;
    Panel1: TPanel;
    memoResult: TRichEdit;
    Splitter1: TSplitter;
    lbLog: TListBox;
    Splitter2: TSplitter;
    N4: TMenuItem;
    NormalModeItem: TMenuItem;
    AspModeItem: TMenuItem;
    Panel2: TPanel;
    memoSource: TMemo;
    StatusBar1: TStatusBar;
    DebugItem: TMenuItem;
    VarChangeItem: TMenuItem;
    DocuItem: TMenuItem;
    N5: TMenuItem;
    AboutItem: TMenuItem;
    HomepageItem: TMenuItem;
    procedure NewItemClick(Sender: TObject);
    procedure CompileItemClick(Sender: TObject);
    procedure ExecuteItemClick(Sender: TObject);
    procedure ExitItemClick(Sender: TObject);
    procedure ShowTreeItemClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure VariablesItemClick(Sender: TObject);
    procedure StopItemClick(Sender: TObject);
    procedure SaveItemClick(Sender: TObject);
    procedure OpenItemClick(Sender: TObject);
    procedure BlockItemClick(Sender: TObject);
    procedure SaveAsItemClick(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure fncGetCapitalsEval(Sender: TObject; scr: TDelphiWebScript; pat: Integer; args: TArgList; var Result: Variant);
    procedure fncPrintEval(Sender: TObject; scr: TDelphiWebScript; pat: Integer; args: TArgList; var Result: Variant);
    procedure ModeItemClick(Sender: TObject);
    procedure AboutItemClick(Sender: TObject);
    procedure memoSourceChange(Sender: TObject);
    procedure DemosMenuClick(Sender: TObject);
    procedure scriptVarChange(scr: TDelphiWebScript; item: TDataItem;
      oldval, newval: Variant);
    procedure DebugItemClick(Sender: TObject);
    procedure scriptDebug(scr: TDelphiWebScript; expr: TExpr);
    procedure VarChangeItemClick(Sender: TObject);
    procedure DocuItemClick(Sender: TObject);
    procedure HomepageItemClick(Sender: TObject);
    procedure dwsISAPILib1CookieEval(Sender: TObject;
      scr: TDelphiWebScript; pat: Integer; args: TArgList;
      var Result: Variant);
    procedure dwsISAPILib1FormVarEval(Sender: TObject;
      scr: TDelphiWebScript; pat: Integer; args: TArgList;
      var Result: Variant);
  private
    ScriptChanged: Boolean;
    scrfile: string;
    DebugLine: Integer;
    procedure OpenScript(sname: string);
    procedure CreateDemosMenu;
  public
    progpath: string;
    scriptpath: string;
    docupath: string
  end;

var
  frmScriptDemo: TfrmScriptDemo;

implementation

uses ScriptDemoTree, ScriptDemoVar, ShellApi, ScriptDemoDebug;

{$R *.DFM}

procedure TfrmScriptDemo.FormCreate(Sender: TObject);
begin
  progpath := ExtractFilePath(Application.Exename);
  scriptpath := progpath + '..\scripts\';
  docupath := progpath + '..\help\';

  ScriptChanged := True;
  DebugLine := 0;

  BlockItem.Checked := script.Blocking;
  NormalModeItem.Click;
  DemosMenu.OnClick := nil;

  CreateDemosMenu;
end;


procedure TfrmScriptDemo.CreateDemosMenu;
var
  mi: TMenuItem;
  sr: TSearchRec;
  res: Integer;
begin
  res := FindFirst(scriptpath + '*.dws', faAnyFile, sr);

  while res = 0 do
  begin

    mi := TMenuItem.Create(MainMenu1);
    mi.Caption := sr.Name;
    mi.OnClick := DemosMenuClick;

    DemosMenu.Add(mi);

    res := FindNext(sr);
  end;

  FindClose(sr);
end;

procedure TfrmScriptDemo.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
begin
  NewItemClick(Sender);
  CanClose := not memoSource.Modified;
end;

procedure TfrmScriptDemo.NewItemClick(Sender: TObject);
var
  res: Integer;
begin
  if memoSource.Modified then
  begin
    res := MessageDlg('There are unsaved changes! Do you want to save now?',
      mtInformation, [mbYes, mbNo, mbCancel], 0);
    if res = mrCancel then exit;
    if res = mrYes then SaveItemClick(self);
  end;
  memoSource.Clear;
  memoSource.Modified := false;
  ScriptChanged := True;
  scrfile := '';
  Caption := 'DWS Demo';

  frmScriptDemo.ActiveControl := memoSource;
end;

procedure TfrmScriptDemo.OpenItemClick(Sender: TObject);
begin
  OpenDialog1.InitialDir := scriptpath;
  if OpenDialog1.Execute then
  begin
    NewItemClick(sender);
    if memoSource.Modified then exit;
    scrfile := OpenDialog1.FileName;
    Caption := 'DWS Demo - ' + scrfile;
    memoSource.Lines.LoadFromFile(OpenDialog1.FileName);
  end;
end;

procedure TfrmScriptDemo.SaveItemClick(Sender: TObject);
begin
  if scrfile = '' then
    SaveAsItemClick(Sender)
  else
  begin
    memoSource.Lines.SaveToFile(scrfile);
    memoSource.Modified := false;
  end;
end;

procedure TfrmScriptDemo.SaveAsItemClick(Sender: TObject);
begin
  SaveDialog1.InitialDir := scriptpath;
  if SaveDialog1.Execute then
  begin
    scrfile := SaveDialog1.FileName;
    Caption := 'DWS Demo - ' + scrfile;
    memoSource.Lines.SaveToFile(scrfile);
  end;
end;

procedure TfrmScriptDemo.ExitItemClick(Sender: TObject);
begin
  Close
end;

procedure TfrmScriptDemo.CompileItemClick(Sender: TObject);
begin
  try
    // Assigns the script code in the memo to the script component
    script.Text := memoSource.Lines.Text;
    ScriptChanged := False;

    // Compiles the script
    if script.Compile then
    begin
      lbLog.Clear;
      lbLog.Items.Add('*** Compiled.')
    end
    else
      lbLog.Items.Add('*** Not compiled (Script is running)');
  except
    // Errors in the script syntax will raise an EScriptError-exception
    on e: EScriptError do
    begin
      // The position of the token causing the error is stored in e.ePos
      memoSource.SelStart := e.FPos.Pos - 1;
      memoSource.SelLength := 1;
      StatusBar1.SimpleText := 'Error: ' + GetErrorString(e.FPos);
      StatusBar1.Color := clRed;
      ShowMessage(e.Message);
      StatusBar1.SimpleText := '';
      StatusBar1.Color := clBtnFace;
    end;
    on e: Exception do
      ShowMessage(e.Message);
  end;
end;

procedure TfrmScriptDemo.ExecuteItemClick(Sender: TObject);
var
  t: LongInt;
begin
  try
    if ScriptChanged then script.Text := memoSource.Lines.Text;
    if not script.Compiled then CompileItemClick(Sender);
    if not script.Compiled then exit;

    lbLog.Items.Add('==========================================');
    lbLog.Items.Add('*** Executing...  press [Ctrl+Del] to stop');
    t := GetTickCount;

    if script.Execute then
    begin
      // Execution successful!
      lbLog.Items.Add(Format('*** Executed. [%.2f s]', [(LongInt(GetTickCount) - t) / 1000]));
      // Shows the value of the script variable "result"
      memoResult.Text := script.Result;
    end
    else
      lbLog.Items.Add('*** Execution failed');
  except
    // Runtime errors in the script will raise an EScriptError-exception
    on e: EScriptError do
    begin
      // Highlight the position of the runtime error
      memoResult.SelStart := e.FPos.Pos - 1;
      memoResult.SelLength := 1;
      StatusBar1.SimpleText := 'Error: ' + GetErrorString(e.FPos);
      StatusBar1.Color := clRed;
      ShowMessage(e.Message);
      StatusBar1.SimpleText := '';
      StatusBar1.Color := clBtnFace;
    end;
    on e: Exception do
      ShowMessage(e.Message);
  end;
end;

procedure TfrmScriptDemo.StopItemClick(Sender: TObject);
begin
  // If a script is executing it will be stopped now.
  script.Stop;
end;

procedure TfrmScriptDemo.BlockItemClick(Sender: TObject);
begin
  // Switch blocking mode
  script.Blocking := not script.Blocking;
  BlockItem.Checked := script.Blocking;
end;

procedure TfrmScriptDemo.DebugItemClick(Sender: TObject);
begin
  DebugItem.Checked := not DebugItem.Checked;
  if DebugItem.Checked then
    script.OnDebug := scriptDebug
  else
    script.OnDebug := nil;
  ScriptChanged := true;
end;

procedure TfrmScriptDemo.VarChangeItemClick(Sender: TObject);
begin
  VarChangeItem.Checked := not VarChangeItem.Checked;
  if VarChangeItem.Checked then
    script.OnVarChange := scriptVarChange
  else
    script.OnVarChange := nil;
  ScriptChanged := True;
end;

procedure TfrmScriptDemo.ModeItemClick(Sender: TObject);
begin
  // Switch html mode
  TMenuItem(sender).Checked := True;
  if NormalModeItem.Checked then
    script.Mode := tmNormal
  else if AspModeItem.Checked then
    script.Mode := tmHtmlAsp
  else if CommentModeItem.Checked then
    script.Mode := tmHtmlComment;
end;

procedure TfrmScriptDemo.VariablesItemClick(Sender: TObject);
begin
  frmScriptDemoVar.ShowModal;
end;

procedure OpenFile(const FileName: string);
begin
  if FileExists(FileName) then
    ShellExecute(Application.MainForm.Handle, 'open', PChar(FileName), nil, nil, SW_SHOW)
  else
    ShowMessage(Filename + ' not found');
end;

procedure TfrmScriptDemo.DocuItemClick(Sender: TObject);
begin
  OpenFile(docupath + 'scrip_en.html');
end;

procedure TfrmScriptDemo.HomepageItemClick(Sender: TObject);
begin
  OpenFile('http://www.dwscript.com');
end;

procedure TfrmScriptDemo.ShowTreeItemClick(Sender: TObject);
begin
  // Shows the code tree, generated by the script parser in a TTreeView component
  script.ShowTree(frmScriptTree.TV);
  frmScriptTree.ShowModal;
end;

procedure TfrmScriptDemo.OpenScript;
begin
  NewItemClick(nil);
  memoSource.Lines.LoadFromFile(scriptpath + sname);
  scrfile := scriptpath + sname;
  Caption := 'DWS Demo - ' + scrfile;
end;

procedure TfrmScriptDemo.DemosMenuClick(Sender: TObject);
begin
  script.SetGlobal('opentime', now);
  script.SetGlobal('exectime', now);
  OpenScript(TMenuItem(Sender).Caption);
end;

procedure TfrmScriptDemo.AboutItemClick(Sender: TObject);
begin
  ShowMessage('Demonstration program for DelphiWebScript. Version 1.0');
end;

procedure TfrmScriptDemo.memoSourceChange(Sender: TObject);
begin
  ScriptChanged := True;
end;

procedure TfrmScriptDemo.scriptVarChange(scr: TDelphiWebScript; item: TDataItem; oldval, newval: Variant);
begin
  lbLog.Items.Add(format('Variable %s changes from %s to %s', [item.Name, vartostr(oldval), vartostr(newval)]));
end;

procedure TfrmScriptDemo.scriptDebug(scr: TDelphiWebScript; expr: TExpr);
var
  res: Integer;
begin
  if expr.FPos.Line = DebugLine then exit;

  DebugLine := expr.FPos.Line;

  memoSource.SelStart := GetErrorLineStart(expr.FPos);
  memoSource.SelLength := GetErrorLineEnd(expr.FPos) - memoSource.SelStart;

  res := frmScriptDemoDebug.ShowModal;

  memoSource.SelLength := 0;

  if res = mrAbort then
  begin
    script.Stop;
  end
  else if res = mrAll then
  begin
    script.OnVarChange := nil;
    script.OnDebug := nil;
  end;

end;


procedure TfrmScriptDemo.fncPrintEval;
var
  x: Integer;
  s: string;
begin
  {
  fncPrint.Declaration: Print (all open)
  -> The number and type of the arguments varies
  }
  s := '';
  for x := 0 to args.count - 1 do
    s := s + VarToStr(args[x].Eval);
  lbLog.Items.Add(s);
end;

procedure TfrmScriptDemo.fncGetCapitalsEval;
var
  v: Variant;
begin
  v := VarArrayCreate([0, 4, 0, 1], varOleStr);

  v[0, 0] := 'SWITZERLAND';
  v[1, 0] := 'GERMANY';
  v[2, 0] := 'AUSTRIA';
  v[3, 0] := 'ITALIA';
  v[4, 0] := 'FRANCE';

  v[0, 1] := 'BERN';
  v[1, 1] := 'BERLIN';
  v[2, 1] := 'WIEN';
  v[3, 1] := 'ROM';
  v[4, 1] := 'PARIS';

  result := v;
end;

procedure TfrmScriptDemo.dwsISAPILib1CookieEval(Sender: TObject;
  scr: TDelphiWebScript; pat: Integer; args: TArgList;
  var Result: Variant);
begin
  result := 'test';
end;

procedure TfrmScriptDemo.dwsISAPILib1FormVarEval(Sender: TObject;
  scr: TDelphiWebScript; pat: Integer; args: TArgList;
  var Result: Variant);
begin
  result := formatdatetime('c', now);
end;

end.
