{*******************************************************}
{                                                       }
{                     Disk Utility                      }
{                                                       }
{         Copyright (c) 1999 Luis Cruz Reverte          }
{                  e-mail: lcruz@nigsun.es              }
{                                                       }
{*******************************************************}
unit uDiskVisor;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, TeEngine, Series, ExtCtrls, TeeProcs, Chart, FileCtrl, ColorGrd,
  Buttons, ComCtrls, ActnList, StdActns, Menus, DirTree, ToolWin, ImgList,
  Registry, uAbout;

type
  TFileType = (ftAll, ftFiles);
  TSizeType = (stBytes, stKb, stMb);

  { TInfoStructure }
  
  TInfoStructure = class(TObject)
  private
  	FPathName: string;
     FPathFileSize: Extended;
     FPathAllSize: Extended;
  	FListInfoName: TStringList;
  	FListInfoSize: TStringList;
  public
  	constructor Create;
     destructor Destroy; override;
     procedure ClearInfo;
     property PathName: string read FPathName write FPathName;
     property PathFileSize: Extended read FPathFileSize write FPathFileSize;
     property PathAllSize: Extended read FPathAllSize write FPathAllSize;
     property ListInfoName: TStringList read FListInfoName write FListInfoName;
  	property ListInfoSize: TStringList read FListInfoSize write FListInfoSize;
  end;

  { TFormDiskVisor }
  
  TFormDiskVisor = class(TForm)
    PageControl1: TPageControl;
    TabSheet1: TTabSheet;
    TabSheet2: TTabSheet;
    Chart1: TChart;
    DiscoSerie: TPieSeries;
    Chart2: TChart;
    BarSerie: TBarSeries;
    MainMenu1: TMainMenu;
    File1: TMenuItem;
    View1: TMenuItem;
    Showfiles1: TMenuItem;
    ActionList1: TActionList;
    ShowFiles: TAction;
    Close1: TMenuItem;
    Exit: TAction;
    Timer1: TTimer;
    RotatePie1: TMenuItem;
    RotatePie: TAction;
    View3D: TAction;
    ExploreBigest: TAction;
    rrr1: TMenuItem;
    ee1: TMenuItem;
    Separator1: TMenuItem;
    DirTree1: TDirTree;
    Splitter1: TSplitter;
    ImageList1: TImageList;
    Panel1: TPanel;
    ToolBar1: TToolBar;
    ToolButton2: TToolButton;
    ImageList2: TImageList;
    N1: TMenuItem;
    SizeinKb1: TMenuItem;
    SizeinMb1: TMenuItem;
    SizeinBytes1: TMenuItem;
    SizeInBytes: TAction;
    SizeInKb: TAction;
    SizeInMb: TAction;
    N2: TMenuItem;
    Refresh1: TMenuItem;
    Refresh: TAction;
    Help1: TMenuItem;
    Content1: TMenuItem;
    N3: TMenuItem;
    About1: TMenuItem;
    Contents: TAction;
    About: TAction;
    procedure Chart1ClickSeries(Sender: TCustomChart; Series: TChartSeries;
      ValueIndex: Integer; Button: TMouseButton; Shift: TShiftState; X,
      Y: Integer);
    procedure FormShow(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure Chart2ClickSeries(Sender: TCustomChart; Series: TChartSeries;
      ValueIndex: Integer; Button: TMouseButton; Shift: TShiftState; X,
      Y: Integer);
    procedure ShowFilesExecute(Sender: TObject);
    procedure ExitExecute(Sender: TObject);
    procedure Timer1Timer(Sender: TObject);
    procedure RotatePieExecute(Sender: TObject);
    procedure PageControl1Change(Sender: TObject);
    procedure View3DExecute(Sender: TObject);
    procedure ExploreBigestExecute(Sender: TObject);
    procedure DirTree1Change(Sender: TObject; Node: TTreeNode);
    procedure ToolButton2Click(Sender: TObject);
    procedure SizeInBytesExecute(Sender: TObject);
    procedure SizeInKbExecute(Sender: TObject);
    procedure SizeInMbExecute(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure RefreshExecute(Sender: TObject);
    procedure ContentsExecute(Sender: TObject);
    procedure AboutExecute(Sender: TObject);
  private
    { Private declarations }
    FShowFiles: Boolean;
    colores: array[0..16] of TColor;
    RotateAngle: LongInt;
    SizeType: TSizeType;
    cadSizeType: string;
    ListInfoStructure: TStringList;
    procedure ProcessPath(Path: string; Refresh, ShowSizeFilesInDirectory: Boolean);
    procedure FreeAllInfo;
    function GetFileSize(FileName: string; SearchRec: TSearchRec; FileType: TFileType): Extended;
    function GetFilePathSize(FileName: string; FileType: TFileType): Extended;
    function IsValidFile(SearchRec: TSearchRec): Boolean;
    function IsValidDirectory(SearchRec: TSearchRec): Boolean;
    procedure getDirectoryList(FileName: string; InfoStructure: TInfoStructure);
    function getSizeRatio(Size: Extended): Extended;
    function Round2Decimal(Value: Extended): Extended;
    function GetShortPath(LongPath: string; MaxSize: LongInt; AlwaysShowDrive: Boolean): string;
  public
    { Public declarations }
  end;

var
  FormDiskVisor: TFormDiskVisor;

implementation

{$R *.DFM}

{ TInfoStructure }

constructor TInfoStructure.Create;
begin
	inherited;
	FListInfoName:=TStringList.Create;
  	FListInfoSize:=TStringList.Create;
end;

destructor TInfoStructure.Destroy;
begin
	ClearInfo;
	FListInfoName.Free;
  	FListInfoSize.Free;
	inherited;
end;

procedure TInfoStructure.ClearInfo;
begin
	FPathName     := '';
     FPathFileSize := 0;
     FPathAllSize  := 0;
	FListInfoName.Clear;
     FListInfoSize.Clear;
end;

{ TFormDiskVisor }

procedure TFormDiskVisor.FormCreate(Sender: TObject);
begin
	FShowFiles  := True;
     RotateAngle := 0;
     SizeType    := stKb;
     cadSizeType := 'Kb';
     /////////////////
     ListInfoStructure:=TStringList.Create;
     /////////////////
	colores[0]:=clBlack;
     colores[1]:=clBlue;
	colores[2]:=clDkGray;
	colores[3]:=clFuchsia;
	colores[4]:=clGray;
	colores[5]:=clGreen;
	colores[6]:=clLime;
	colores[7]:=clLtGray;
     colores[8]:=clMaroon;
	colores[9]:=clNavy;
	colores[10]:=clOlive;
	colores[11]:=clPurple;
	colores[12]:=clRed;
     colores[13]:=clSilver;
	colores[14]:=clTeal;
	colores[15]:=clYellow;
	colores[16]:=clWhite;
end;

procedure TFormDiskVisor.FormDestroy(Sender: TObject);
var Registry: TRegistry;
begin
	Registry:=TRegistry.Create;
     try
     	if Registry.OpenKey('Software', False) then begin
	     	if Registry.OpenKey('DiskVisor', True) then begin
     	     	Registry.WriteString('Path', Chart1.Title.Text[0]);
          	end;
          end;
     finally
     	Registry.Free;
     end;
     ////////////// Free's
     FreeAllInfo;
	ListInfoStructure.Free;
end;

procedure TFormDiskVisor.FreeAllInfo;
var i: LongInt;
begin
	for i:=0 to ListInfoStructure.Count-1 do begin
     	(ListInfoStructure.Objects[i] as TInfoStructure).Free;
     end;
     ListInfoStructure.Clear;
end;

procedure TFormDiskVisor.FormShow(Sender: TObject);
var
  Registry: TRegistry;
  Path: string;
begin
	Registry:=TRegistry.Create;
     try
     	Path:='C:\';
     	if Registry.OpenKey('Software', False) then begin
	     	if Registry.OpenKey('DiskVisor', False) then begin
     	     	Path:=Registry.ReadString('Path');
          	end;
          end;
     	ProcessPath(Path, False, FShowFiles);
     finally
     	Registry.Free;
     end;
end;

function TFormDiskVisor.GetFileSize(FileName: string; SearchRec: TSearchRec; FileType: TFileType): Extended;
begin
     if ((SearchRec.Attr and faDirectory) = faDirectory) then begin
     	if FileType = ftAll then Result:=GetFilePathSize(ExtractFilePath(FileName)+SearchRec.Name+'\*.*', ftAll)
          else Result:=0;
     end
     else Result:=SearchRec.Size;
end;

function TFormDiskVisor.GetFilePathSize(FileName: string; FileType: TFileType): Extended;
var SearchRec: TSearchRec;
begin
	Result:=0;
	if FindFirst(FileName, faAnyFile, SearchRec) = 0 then begin
     	if IsValidFile(SearchRec) then Result:=GetFileSize(FileName, SearchRec, FileType);
     	while FindNext(SearchRec) = 0 do begin
     		if IsValidFile(SearchRec) then Result:=Result+GetFileSize(FileName, SearchRec, FileType);
	     end;
     	FindClose(SearchRec);
     end;
end;

function TFormDiskVisor.IsValidFile(SearchRec: TSearchRec): Boolean;
begin
	if (SearchRec.Name<>'.') and (SearchRec.Name<>'..') then Result:=True
	else Result:=False;
end;

function TFormDiskVisor.IsValidDirectory(SearchRec: TSearchRec): Boolean;
begin
	if ((SearchRec.Attr and faDirectory) = faDirectory) and
        (SearchRec.Name<>'.') and (SearchRec.Name<>'..') then Result:=True
     else Result:=False;
end;

procedure TFormDiskVisor.getDirectoryList(FileName: string; InfoStructure: TInfoStructure);
var SearchRec: TSearchRec;
begin
	if FindFirst(FileName, faAnyFile, SearchRec) = 0 then begin
     	if IsValidDirectory(SearchRec) then begin
          	InfoStructure.ListInfoName.Add(SearchRec.Name);
               InfoStructure.ListInfoSize.Add(Floattostr(GetFilePathSize(ExtractFilePath(FileName)+SearchRec.Name+'\*.*', ftAll)));
          end;
     	while FindNext(SearchRec) = 0 do begin
          	if IsValidDirectory(SearchRec) then begin
               	InfoStructure.ListInfoName.Add(SearchRec.Name);
                    InfoStructure.ListInfoSize.Add(FloatToStr(GetFilePathSize(ExtractFilePath(FileName)+SearchRec.Name+'\*.*', ftAll)));
               end;
	     end;
          FindClose(SearchRec);
     end;
end;

function TFormDiskVisor.getSizeRatio(Size: Extended): Extended;
begin
	case SizeType of
     	stBytes: Result:=Size;

          stKb: begin
          	Result:=(Size / 1024);
               Result:=Round2Decimal(Result);
          end;

     	stMb:begin
          	Result:=((Size / 1024) / 1024);
               Result:=Round2Decimal(Result);
          end;

          else Result:=Size;
     end;
end;

function TFormDiskVisor.Round2Decimal(Value: Extended): Extended;
var
  aux: Extended;
begin
     aux:=Frac(Value);
     aux:=aux*100;
     aux:=Round(aux);
     aux:=aux/100;
     Result:=Int(Value)+aux;
end;

procedure TFormDiskVisor.ProcessPath(Path: string; Refresh, ShowSizeFilesInDirectory: Boolean);
var
  InfoStructure: TInfoStructure;
  i: LongInt;
  recCursor: TCursor;

  procedure _Initialize(Path: string);
  begin
  	if CompareText(DirTree1.Directory, Path) <> 0 then DirTree1.Directory:=Path;
     DiscoSerie.Clear;
     BarSerie.Clear;
  end;

  procedure _Set_Title(InfoS: TInfoStructure; Path: string; ShowSizeFilesInDirectory: Boolean);
  var
    lab: string;
    val: Extended;
  begin
  	// Set chart title
     Chart1.Title.Text.Clear;
     Chart2.Title.Text.Clear;
     Chart1.Title.Text.Add(Path);
     Chart2.Title.Text.Add(Path);
     /////////////
  	val:= getSizeRatio(InfoS.PathFileSize);
     if ShowSizeFilesInDirectory then begin
     	lab:= '<Files in '+GetShortPath(Path, 15, True)+'>';
          DiscoSerie.AddPie(val, lab, clWhite);
          BarSerie.AddBar(val, lab, clWhite);
     end;
     lab:='Files Size = ' + FormatFloat('###,###,###,###,###,###,###,##0.##', val) + ' ' + cadSizeType;
     Chart1.Title.Text.Add(lab);
     Chart2.Title.Text.Add(lab);
     val:=getSizeRatio(InfoS.PathAllSize);
     lab:='Total Size = ' + FormatFloat('###,###,###,###,###,###,###,##0.##', val) + ' ' + cadSizeType;
     Chart1.Title.Text.Add(lab);
     Chart2.Title.Text.Add(lab);
  end;

  procedure _Generate_Pies_Bars(InfoS: TInfoStructure);
  var
    i: LongInt;
    inColor: LongInt;
    colindex: LongInt;
    lab: string;
    val: Extended;
  begin
  	colindex:=0;
     inColor:=0;
     for i:=0 to InfoS.ListInfoName.Count-1 do begin
     	lab:=InfoS.ListInfoName.Strings[i];
          val:=getSizeRatio(strtofloat(InfoS.ListInfoSize.Strings[i]));
          DiscoSerie.AddPie(val, lab, colores[colindex]+inColor);
          BarSerie.AddBar(val, lab, colores[colindex]+inColor);
          inc(colindex);
          if colindex>15 then begin
          	colindex:=0;
               inc(inColor, 100);
          end;
     end;
  end;

  procedure _Generate_InfoStructure(InfoS: TInfoStructure; Path: string);
  begin
     InfoS.ClearInfo;
  	InfoS.PathFileSize := GetFilePathSize(Path+'*.*', ftFiles);
     InfoS.PathAllSize  := GetFilePathSize(Path+'*.*', ftAll);
     getDirectoryList(Path+'*.*', InfoS);
  end;

  procedure _SetNewCursor;
  begin
  	recCursor       := Cursor;
	Cursor          := crHourGlass;
     Chart1.Cursor   := crHourGlass;
     DirTree1.Cursor := crHourGlass;
  end;

  procedure _SetOldCursor;
  begin
  	Cursor          := recCursor;
     Chart1.Cursor   := recCursor;
     DirTree1.Cursor := recCursor;
  end;

begin
	_SetNewCursor;
	if not DirectoryExists(Path) then Path:='C:\';
	_Initialize(Path);
	i:=ListInfoStructure.IndexOf(Path);
	if i < 0 then begin
		InfoStructure:=TInfoStructure.Create;
     	ListInfoStructure.AddObject(Path, InfoStructure);
          _Generate_InfoStructure(InfoStructure, Path);
          _Set_Title(InfoStructure, Path, ShowSizeFilesInDirectory);
          _Generate_Pies_Bars(InfoStructure);
     end
     else begin
     	InfoStructure:=TInfoStructure(ListInfoStructure.Objects[i]);
          if Refresh then _Generate_InfoStructure(InfoStructure, Path);
          _Set_Title(InfoStructure, Path, ShowSizeFilesInDirectory);
          _Generate_Pies_Bars(InfoStructure);
     end;
     _SetOldCursor;
end;

procedure TFormDiskVisor.Chart1ClickSeries(Sender: TCustomChart;
  Series: TChartSeries; ValueIndex: Integer; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  i: LongInt;
begin
  	i:=(Series as TPieSeries).CalcClickedPie(x, y);
     if Series.XLabel[i][1] <> '<' then
     	ProcessPath(Sender.Title.Text[0]+Series.XLabel[i]+'\', False, FShowFiles);
end;

procedure TFormDiskVisor.Chart2ClickSeries(Sender: TCustomChart;
  Series: TChartSeries; ValueIndex: Integer; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var
  i: LongInt;
begin
  	i:=(Series as TChartSeries).Clicked(x, y);
     if Series.XLabel[i][1] <> '<' then
     	ProcessPath(Sender.Title.Text[0]+Series.XLabel[i]+'\', False, FShowFiles);
end;

procedure TFormDiskVisor.ShowFilesExecute(Sender: TObject);
begin
	if (Sender as TAction).Checked then begin
     	(Sender as TAction).Checked:=False;
		FShowFiles:=False;
     end
     else begin
     	(Sender as TAction).Checked:=True;
          FShowFiles:=True;
     end;
     ProcessPath(Chart1.Title.Text[0], False, FShowFiles);
end;

procedure TFormDiskVisor.ExitExecute(Sender: TObject);
begin
	Close;
end;

procedure TFormDiskVisor.Timer1Timer(Sender: TObject);
begin
	DiscoSerie.Rotate(RotateAngle);
end;

procedure TFormDiskVisor.RotatePieExecute(Sender: TObject);
begin
	if RotatePie.Checked then begin
     	RotateAngle       := 0;
          RotatePie.Checked := False;
     end
     else begin
     	RotateAngle       := 1;
          RotatePie.Checked := True;
     end;
end;

procedure TFormDiskVisor.PageControl1Change(Sender: TObject);
begin
	if PageControl1.ActivePage = TabSheet1 then begin
     	Separator1.Visible    := True;
     	RotatePie.Visible     := True;
     	RotatePie.Enabled     := True;
          ExploreBigest.Visible := True;
          ExploreBigest.Enabled := True;
     end
     else begin
     	Separator1.Visible    := False;
     	RotatePie.Visible     := False;
     	RotatePie.Enabled     := False;
          ExploreBigest.Visible := False;
          ExploreBigest.Enabled := False;
     end;
end;

procedure TFormDiskVisor.View3DExecute(Sender: TObject);
begin
	if View3D.Checked then begin
		Chart1.View3D  := False;
          Chart2.View3D  := False;
          View3D.Checked := False;
     end
     else
     begin
     	Chart1.View3D  := True;
          Chart2.View3D  := True;
          View3D.Checked := True;
     end;
end;

procedure TFormDiskVisor.ExploreBigestExecute(Sender: TObject);
begin
	if ExploreBigest.Checked then begin
		DiscoSerie.ExplodeBiggest := 0;
          ExploreBigest.Checked     := False;
     end
     else begin
     	DiscoSerie.ExplodeBiggest := 5;
          ExploreBigest.Checked     := True;
     end;
end;

procedure TFormDiskVisor.DirTree1Change(Sender: TObject; Node: TTreeNode);
begin
	if Assigned(ListInfoStructure) then ProcessPath(DirTree1.Directory, False, FShowFiles);
end;

procedure TFormDiskVisor.ToolButton2Click(Sender: TObject);
var
  cad: string;
  len: LongInt;
begin
	cad:=Chart1.Title.Text[0];
     len:=Length(cad);
     dec(len);
     while (cad[len] <> '\') and (cad[len] <> ':') do dec(len);
     if cad[len]<>':' then begin
		cad:=Copy(cad, 1, len);
     	ProcessPath(cad, False, FShowFiles);
     end;
end;

procedure TFormDiskVisor.SizeInBytesExecute(Sender: TObject);
begin
     SizeInBytes.Checked:=True;
	SizeInKb.Checked:=False;
     SizeInMb.Checked:=False;
     SizeType    := stBytes;
     cadSizeType := 'Bytes';
end;

procedure TFormDiskVisor.SizeInKbExecute(Sender: TObject);
begin
	SizeInBytes.Checked:=False;
     SizeInKb.Checked:=True;
     SizeInMb.Checked:=False;
     SizeType    := stKb;
     cadSizeType := 'Kb';
end;

procedure TFormDiskVisor.SizeInMbExecute(Sender: TObject);
begin
	SizeInBytes.Checked:=False;
     SizeInKb.Checked:=False;
     SizeInMb.Checked:=True;
     SizeType    := stMb;
     cadSizeType := 'Mb';
end;

procedure TFormDiskVisor.RefreshExecute(Sender: TObject);
var cad: string;
begin
	cad:=Chart1.Title.Text[0];
     ProcessPath(cad, True, FShowFiles);
end;

function TFormDiskVisor.GetShortPath(LongPath: string; MaxSize: LongInt; AlwaysShowDrive: Boolean): string;
var
  pos: LongInt;
  auxCad: string;

  function _GetPosSlash: LongInt;
  var
    len: LongInt;
    lastPos: LongInt;
    loop: Boolean;
    CompareSize: Boolean;
  begin
  	len         := Length(LongPath);
     Result      := len;
     loop        := True;
     CompareSize := False;
     lastPos     := 0;
     dec(len);
  	while Loop do begin
     	if (CompareSize) then begin
          	if Result-len+6 >= MaxSize then begin
               	if lastPos <> 0 then len:=lastPos;
               	Loop:=False
               end
               else begin
               	lastPos:=len;
               	dec(len);
                    CompareSize:=False;
               end;
          end
          else begin
	     	if LongPath[len] <> '\' then dec(len)
               else CompareSize:=True;
          end;
     end;
     Result:=len;
  end;

begin
	if (LongPath = '') or (Length(LongPath) <= MaxSize) then Result:=LongPath
	else begin
     	Result := Copy(LongPath, 1, 3);
          pos    := _GetPosSlash;
          auxCad := Copy(LongPath, pos, Length(LongPath));
          Result := Result+'...'+auxCad;
          if (not AlwaysShowDrive) and (Length(Result) > MaxSize) then begin
          	Result:=Copy(auxCad, 2, Length(auxCad));
          end;
	end;
end;

procedure TFormDiskVisor.ContentsExecute(Sender: TObject);
begin
	/////
     showmessage(' Sorry !!');
end;

procedure TFormDiskVisor.AboutExecute(Sender: TObject);
var FormAbout: TFormDVAbout;
begin
	FormAbout:=TFormDVAbout.Create(nil);
     FormAbout.ShowModal;
     FormAbout.Free;
end;

end.

