{ ##
  @PROJECT_NAME             ShellFolderDemo
  @PROJECT_DESC             Demonstrates shell folders unit.
  @FILE                     ShellFolderDemoForm.pas
  @COMMENTS                 Main form and code for demo program.
  @DEPENDENCIES             Requires TPJSpecialFolderInfo and TPJBrowseDialog.
  @LICENSE                  The demo is released under the Mozilla public
                            license (see below).
  @COPYRIGHT                Copyright (c) 2003-2005, Peter D Johnson.
  @HISTORY(
    @REVISION(
      @VERSION              1.0
      @DATE                 15/06/2003
      @COMMENTS             Original version.
    )
    @REVISION(
      @VERSION              1.1
      @DATE                 29/07/2003
      @COMMENTS             Updated - detail not known.
    )
    @REVISION(
      @VERSION              1.2
      @DATE                 22/08/2004
      @COMMENTS             Added code to demonstrate new style dialog.
    )
    @REVISION(
      @VERSION              1.3
      @DATE                 22/12/2005
      @COMMENTS             Fixed range check bug.
    )
  )
}


{
 * ***** BEGIN LICENSE BLOCK *****
 *
 * Version: MPL 1.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with the
 * License. You may obtain a copy of the License at http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
 * the specific language governing rights and limitations under the License.
 *
 * The Original Code is ShellFolderDemoForm.pas.
 *
 * The Initial Developer of the Original Code is Peter Johnson
 * (http://www.delphidabbler.com/).
 *
 * Portions created by the Initial Developer are Copyright (C) 2003-2005 Peter
 * Johnson. All Rights Reserved.
 *
 * ***** END LICENSE BLOCK *****
}


unit ShellFolderDemoForm;


interface


uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ComCtrls, StdCtrls, ShlObj, ExtCtrls, ImgList,

  PJShellFolders;

type
  TForm1 = class(TForm)
    BrowseBtn: TButton;
    BrowseDlg: TPJBrowseDialog;
    FolderImage: TImage;
    ImageList: TImageList;
    LV: TListView;
    OnSelChangeExRadio: TRadioButton;
    OnSelChangeRadio: TRadioButton;
    SpecialFolderInfo: TPJSpecialFolderInfo;
    VirtualChk: TCheckBox;
    NewStyleChk: TCheckBox;
    procedure BrowseBtnClick(Sender: TObject);
    procedure BrowseDlgClose(Sender: TObject);
    procedure BrowseDlgInitialise(Sender: TObject);
    procedure BrowseDlgSelChange(Sender: TObject; FolderName,
      DisplayName: String; var StatusText: String; var OKEnabled: Boolean);
    procedure BrowseDlgSelChangeEx(Sender: TObject; PIDL: PItemIDList;
      var StatusText: String; var OKEnabled: Boolean);
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure LVDblClick(Sender: TObject);
    procedure OnSelChangeExRadioClick(Sender: TObject);
    procedure OnSelChangeRadioClick(Sender: TObject);
    procedure VirtualChkClick(Sender: TObject);
    procedure LVSelectItem(Sender: TObject; Item: TListItem;
      Selected: Boolean);
    procedure NewStyleChkClick(Sender: TObject);
  private
    fPJFont: TFont;
    procedure DisplaySpecialFolders;
    function IsSelectionSupported: Boolean;
  end;

var
  Form1: TForm1;


implementation


uses
  ShellAPI;


{$R *.DFM}


procedure TForm1.BrowseBtnClick(Sender: TObject);
  {Display browse for folder dlg box for selected folder}
var
  Item: TListItem;
begin
  Item := LV.Selected;
  Assert(Item <> nil);
  BrowseDlg.RootFolderID := StrToSpecialFolderID(Item.Caption);
  SpecialFolderInfo.FolderID := BrowseDlg.RootFolderID;
  BrowseDlg.Headline := 'Browsing for folders under: '
    + SpecialFolderInfo.DisplayName;
  BrowseDlg.FolderName := '';   // don't remember last time's folder
  if BrowseDlg.Execute then
  begin
    // display info about selected folder
    MessageBox(Handle, PChar('Folder chosen:'#13
      + '    Folder Name: ' + BrowseDlg.FolderName + #13
      + '    Display Name: ' + BrowseDlg.DisplayName),
      'Shell Folder Demo',
      MB_OK or MB_ICONINFORMATION);
  end;
end;

procedure TForm1.BrowseDlgClose(Sender: TObject);
  {Clear any displayed icon when browse dialog closes}
begin
  FolderImage.Picture.Bitmap.Assign(nil);
end;

procedure TForm1.BrowseDlgInitialise(Sender: TObject);
  {Add a new text control that displays id of special folder we're editing}
  // ---------------------------------------------------------------------------
  procedure GetBounds(HWnd, HParentWnd: THandle;
    PLeft, PTop, PWidth, PHeight: PInteger);
    {Gets left, top, width and height of the given window HWnd which has parent
    HParentWnd and returns in given parameters. If any of PLeft, PTop, PWidth
    and PHeight are nil then these values are not set}
  var
    ScreenRect: TRect;  // boundign rectangle of window in screen coordinates
    TopLeft: TPoint;    // coordinates of top left corner of widnow
  begin
    // Get bounding rectangle of window (in screen coords)
    GetWindowRect(Hwnd, ScreenRect);
    // Get top left corner of window in (parent's coordinates)
    TopLeft := ScreenRect.TopLeft;
    Windows.ScreenToClient(HParentWnd, TopLeft);
    // Calculate the required value for Left, Top, Width and Height
    if Assigned(PTop) then PTop^ := TopLeft.Y;
    if Assigned(PLeft) then PLeft^ := TopLeft.X;
    if Assigned(PWidth) then PWidth^ := ScreenRect.Right - ScreenRect.Left;
    if Assigned(PHeight) then PHeight^ := ScreenRect.Bottom - ScreenRect.Top;
  end;
  // ---------------------------------------------------------------------------
var
  HText: THandle; // handle of new text control
  TVLeft, TVTop, TVWidth, TVHeight: THandle;  // dimensions of dlg's tree view
begin
  // This customisatio doesn't work for new style dialog: so don't do it!!
  if boNewDlgStyle in BrowseDlg.Options then
    Exit;
  // Get dimensions of tree view
  // (tree view has ID $3741 per constants in PJShellFolders.pas)
  GetBounds(GetDlgItem(BrowseDlg.Handle, $3741),
    BrowseDlg.Handle,
    @TVLeft, @TVTop, @TVWidth, @TVHeight);
  // Create a static text control
  HText := CreateWindowEx(
    0,            // no extended style
    'static',     // use static
    PChar(SpecialFolderIDToStr(SpecialFolderInfo.FolderID)),  // the text
    WS_VISIBLE or WS_CHILD or SS_CENTER or SS_SUNKEN,  // window style
    TVLeft,       // align left with tree view
    TVTop + TVHeight,   // vertically centre with buttons
    TVWidth,           // width of control
    17,           // height of control
    BrowseDlg.Handle,       // handle of parent window
    $1000,        // id of window
    GetWindowLong(Handle, GWL_HINSTANCE), // instance of owning app
    nil);         // no CREATESTRUCT data
  // Set the font the help control
  SendMessage(HText, WM_SETFONT, Integer(fPJFont.Handle), MakeLParam(1, 0));
end;

procedure TForm1.BrowseDlgSelChange(Sender: TObject; FolderName,
  DisplayName: String; var StatusText: String; var OKEnabled: Boolean);
  {Event triggered when selection changes in browse dialog. This event provides
  access to selected folder's path and display name. We show display name of
  selected item in dlg box's status text. Nothing happens if dialog is new style
  because status text is not supported for this style}
begin
  // Simply show display name in status text
  StatusText := DisplayName;
  // and always enable button
  OKEnabled := True;
end;

procedure TForm1.BrowseDlgSelChangeEx(Sender: TObject; PIDL: PItemIDList;
  var StatusText: String; var OKEnabled: Boolean);
  {Event triggered when selection changes in browse dialog. This event provides
  access to selected folder's PIDL. We display if folder is virtual, and folder
  or path name in dlg's status text. We also display large icon associated with
  folder in main form. Status text is not updated when dialog box is new style
  since status text not supported in that style. Icon update code works
  regardless of what style dialog box is}

  function IconIndex(PIDL: PItemIDList): Integer;
    {Return index of icon associated with given PIDL}
  var
    FI: TSHFileInfo;
  begin
    SHGetFileInfo(PChar(PIDL), 0, FI, SizeOf(FI), SHGFI_ICON + SHGFI_PIDL);
    Result := FI.iIcon;
  end;

var
  Path: string;
  IsVirtual: Boolean;
  DisplayName: string;
  IconIdx: Integer;
begin
  // Get index of icon for selected folder in system image list
  IconIdx := IconIndex(PIDL);
  // Get display name and path for selected folder and decide if virtual
  DisplayName := PIDLToFolderDisplayName(PIDL);
  Path := PIDLToFolderPath(PIDL);
  IsVirtual := Path = '';
  // Display folder info in status area of dlg (+ image list index)
  if not IsVirtual then
  begin
    if Path <> '' then
      if Length(Path) > 40 then
        StatusText := Format('File System: ... %s (%d)', [DisplayName, IconIdx])
      else
        StatusText := Format('File System: %s (%d)', [Path, IconIdx]);
  end
  else
    StatusText := Format('Virtual: %s (%d)', [DisplayName, IconIdx]);
  // Display large icon associated with selected folder on main form
  ImageList.GetBitmap(IconIdx, FolderImage.Picture.Bitmap);
  FolderImage.Refresh;  // lets image list refresh
end;

procedure TForm1.DisplaySpecialFolders;
  {Displays info about special folders}
var
  DisplayName, FolderPath: string;
  Enum: IPJSpecialFolderEnum;
  Item: TListItem;
begin
  // iterate special folders using enumerator
  Enum := TPJSpecialFolderEnum.Create;
  while not Enum.AtEnd do
  begin
    SpecialFolderInfo.FolderID := Enum.Next;
    if SpecialFolderInfo.IsSupported then
    begin
      // folder is supported: gather into
      if SpecialFolderInfo.IsVirtual then
        FolderPath := '<virtual folder>'
      else
        FolderPath := SpecialFolderInfo.Path;
      DisplayName := SpecialFolderInfo.DisplayName;
    end
    else
    begin
      // folder is not supported
      DisplayName := '<not supported>';
      FolderPath := '';
    end;
    // add collected info about folder to list view
    Item := LV.Items.Add;
    Item.Caption := SpecialFolderIdToStr(SpecialFolderInfo.FolderID);
    Item.SubItems.Add(DisplayName);
    Item.SubItems.Add(FolderPath);
  end;
end;

procedure TForm1.FormCreate(Sender: TObject);
  {Set up images, special dlg box font and display list of all supported special
  folders}

  function SysImgListHandle: THandle;
    {Return handle to system image list containing large icons}
  var
    FI: TSHFileInfo;
  begin
    Result := SHGetFileInfo(
      'C:\',
      0,
      FI,
      SizeOf(FI),
      SHGFI_SYSICONINDEX or SHGFI_ICON
    );
  end;

begin
  // make our image list contain large system images
  ImageList.Handle := SysImgListHandle;
  // display the special folders
  DisplaySpecialFolders;
  // create a font to use in browse for folder dlg
  fPJFont := TFont.Create;
  fPJFont.Name := 'Comic Sans MS';
  fPJFont.Size := 7;
  fPJFont.Style := [];
  // update check boxes
  VirtualChk.Checked := not (boDirsOnly in BrowseDlg.Options);
  NewStyleChk.Checked := boNewDlgStyle in BrowseDlg.Options;
end;

procedure TForm1.FormDestroy(Sender: TObject);
  {Tidy up - free special font}
begin
  fPJFont.Free;
end;

function TForm1.IsSelectionSupported: Boolean;
  {Return true if selected folder is supported on this system. Returns false if
  not supported or nothing selected}
var
  Item: TListItem;
begin
  Item := LV.Selected;
  if Assigned(Item) then
  begin
    SpecialFolderInfo.FolderID := StrToSpecialFolderID(Item.Caption);
    Result := SpecialFolderInfo.IsSupported;
  end
  else
    Result := False;
end;

procedure TForm1.LVDblClick(Sender: TObject);
  {Display browse dlg box for selected folder, if supported}
begin
  if IsSelectionSupported then
    BrowseBtnClick(Self)
  else
    MessageBox(Handle, 'Shell folder not supported', 'Shell Folder Demo',
      MB_ICONERROR or MB_OK);
end;

procedure TForm1.LVSelectItem(Sender: TObject; Item: TListItem;
  Selected: Boolean);
  {Enable / disable browse button according to if selected folder is supported}
begin
  if Assigned(Item) and Selected then
    BrowseBtn.Enabled := IsSelectionSupported;
end;

procedure TForm1.NewStyleChkClick(Sender: TObject);
  {Set browse for folder dialog options depending on if new style dialog
  required}
begin
  if NewStyleChk.Checked then
    BrowseDlg.Options := BrowseDlg.Options + [boNewDlgStyle]
  else
    // need to reinstate since setting [boNewDlgStyle] removes these
    BrowseDlg.Options := BrowseDlg.Options + [boShowHelp, boStatusText];
end;

procedure TForm1.OnSelChangeExRadioClick(Sender: TObject);
  {We use OnSelChangeEx event handler with browse dlg}
begin
  BrowseDlg.OnSelChangeEx := BrowseDlgSelChangeEx;
  BrowseDlg.OnSelChange := nil;
end;

procedure TForm1.OnSelChangeRadioClick(Sender: TObject);
  {We use OnSelChange event handler with browse dlg}
begin
  BrowseDlg.OnSelChangeEx := nil;
  BrowseDlg.OnSelChange := BrowseDlgSelChange;
end;

procedure TForm1.VirtualChkClick(Sender: TObject);
  {Decide if we permit virtual folders to be displayed and accepted in browse
  dlg and configure dlg accordingly}
begin
  if not VirtualChk.Checked then
    BrowseDlg.Options := BrowseDlg.Options + [boDirsOnly]
  else
    BrowseDlg.Options := BrowseDlg.Options - [boDirsOnly];
end;

end.
