{
 BUSINESS CONSULTING
 s a i n t - p e t e r s b u r g

         Components Library for Borland Delphi 4.x - 6.x
         Copyright (c) 1998-2001 Alex'EM

}
unit DCSyntaxData;

interface

uses
  Windows, SysUtils, Classes, Graphics, Controls;

type
  TLexemType = (lxWhitespace, lxIdentifier, lxString, lxNumber, lxComment,
    lxSymbol, lxKeyWord, lxHighlight);

  PLexemItem = ^TLexemItem;
  TLexemItem = packed record
    Item: TLexemType;
    ItemIndex: byte;
    Length: WORD;
  end;

  PLexemItems = ^TLexemItems;
  TLexemItems = packed array[0..0] of TLexemItem;

  THighlights = packed record
    Enabled: boolean;
    Count: byte;
    case Integer of
      0: (Value: DWORD);
      1: (Items: array[1..4] of byte);
  end;

  PLineDataItem  = ^TLineDataItem;
  TLineDataItem = packed record
    FString: string;
    FObject: TObject;
    LComment: byte;                    {Current line comment}
    HComment: byte;                    {Previous line comment}
    Capacity: WORD;
    Count: WORD;
    Highlights: THighlights;
    Lexems: PLexemItems;
  end;

  PLineDataItems = ^TLineDataItems;
  TLineDataItems = packed array[0..0] of TLineDataItem;

  TDCCustomSyntaxData = class;

  TSyntaxDataClass = class of TDCCustomSyntaxData;

  TDCSyntaxMemoColor = class(TPersistent)
  private
    FBackground: TColor;
    FFontStyle: TFontStyles;
    FForeground: TColor;
    FOnChange: TNotifyEvent;
    FSyntaxData: TDCCustomSyntaxData;
    procedure SetBackground(const Value: TColor);
    procedure SetFontStyle(const Value: TFontStyles);
    procedure SetForeground(const Value: TColor);
  protected
    procedure AssignTo(Dest: TPersistent); override;
    procedure Change; virtual;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
  public
    constructor Create(SyntaxData: TDCCustomSyntaxData);
  published
    property FontStyle: TFontStyles read FFontStyle write SetFontStyle;
    property Background: TColor read FBackground write SetBackground;
    property Foreground: TColor read FForeground write SetForeground;
  end;

  TCharArray = array[Char] of boolean;

  TSyntaxLexemColors = class;
  TLexemColorItem = class(TCollectionItem)
  private
    FColor: TDCSyntaxMemoColor;
    FLexemType: TLexemType;
    procedure SetColor(const Value: TDCSyntaxMemoColor);
    procedure SetLexemType(const Value: TLexemType);
  public
    constructor Create(Collection: TCollection); override;
    destructor Destroy; override;
  published
    property Color: TDCSyntaxMemoColor read FColor write SetColor;
    property LexemType: TLexemType read FLexemType write SetLexemType;
  end;

  TSyntaxLexemColors = class(TCollection)
  private
    FSyntaxData: TDCCustomSyntaxData;
    function GetItem(Index: TLexemType): TLexemColorItem;
  public
    constructor Create(SyntaxData: TDCCustomSyntaxData);
    function Add(ALexemType: TLexemType): TLexemColorItem;
    property Items[Index: TLexemType]: TLexemColorItem read GetItem;
  end;

  THighlighOptions = set of (hoIncludeBounds, hoUnbreakable);

  TDataHighlight = class;
  TDataHighlightItem = class(TCollectionItem)
  private
    FCloseLexemPos: integer;
    FCloseLexem: string;
    FColor: TDCSyntaxMemoColor;
    FKeyIndex: integer;
    FLexemIndex: integer;
    FLexemType: TLexemType;
    FOpenLexemPos: integer;
    FOpenLexem: string;
    FOptions: THighlighOptions;
    FValues: string;
    procedure SetColor(const Value: TDCSyntaxMemoColor);
    procedure SetOptions(const Value: THighlighOptions);
  protected
    property OpenLexemPos: integer read FOpenLexemPos;
    property CloseLexemPos: integer read FCloseLexemPos;
  public
    constructor Create(Collection: TCollection); override;
    destructor Destroy; override;
  published
    property Color: TDCSyntaxMemoColor read FColor write SetColor;
    property Options: THighlighOptions read FOptions write SetOptions;
  end;

  TDataHighlight = class(TCollection)
  private
    FSyntaxData: TDCCustomSyntaxData;
    function GetItem(Index: Integer): TDataHighlightItem;
    procedure SetItem(Index: Integer; const Value: TDataHighlightItem);
    function StrLPos(const Str1: string; const pStr2: PChar; Len: integer): integer;
  protected
    function CheckHighlight(const ValuePos: integer; Value: PChar;
      var Highlights: THighlights; var Lexem: TLexemType; var LexemIndex: byte;
      var LexemLength: WORD): boolean;
    function HightlightOpen(LexemPos, KeyIndex: integer; Value: PChar;
      var Lexem: TLexemType; var Highlights: THighlights;
      var LexemLength: WORD): boolean;
  public
    constructor Create(SyntaxData: TDCCustomSyntaxData);
    destructor Destroy; override;
    function AddItem(AOpenLexem, ACloseLexem: string;
     AKeyIndex: integer; ALexemType: TLexemType; ALexemIndex: integer;
     AValues: string): TDataHighlightItem;
    property Items[Index: Integer]: TDataHighlightItem read GetItem
      write SetItem;
  end;

  TSyntaxHashItem = packed record
    PosB, Length: DWORD;
  end;

  TSyntaxHashData = packed record
    Size: integer;
    Data: Pointer;
    CharArray: array[Char] of TSyntaxHashItem;
  end;

  TSyntaxKeyWords = class;

  TSyntaxKeyWordItem = class(TCollectionItem)
  private
    FColor: TDCSyntaxMemoColor;
    FHashData: TSyntaxHashData;
    FValues: string;
    procedure InitHashData;
    procedure SortKeyWords;
    procedure SetColor(const Value: TDCSyntaxMemoColor);
  public
    constructor Create(Collection: TCollection); override;
    destructor Destroy; override;
  published
    property Color: TDCSyntaxMemoColor read FColor write SetColor;
  end;

  TSyntaxKeyWords = class(TCollection)
  private
    FSyntaxData: TDCCustomSyntaxData;
    function GetItem(Index: Integer): TSyntaxKeyWordItem;
    procedure SetItem(Index: Integer; const Value: TSyntaxKeyWordItem);
  public
    constructor Create(SyntaxData: TDCCustomSyntaxData );
    function Add(const Values: string): TSyntaxKeyWordItem;
    property Items[Index: Integer]: TSyntaxKeyWordItem read GetItem
      write SetItem;
  end;

  TDCCustomSyntaxData = class(TPersistent)
  private
    FDataHighlight: TDataHighlight;
    FHighlights: THighlights;
    FKeyWords: TSyntaxKeyWords;
    FLexemColors: TSyntaxLexemColors;
    FOpenComment: DWORD;
    FOwner: TComponent;
    function GetCommentLen(Comment: integer): integer;
    function Comment2ID(Value: DWORD): byte;
    function ID2Comment(Value: byte): DWORD;
    function CanOpenHighlight: boolean;
  protected
    FSymbols: string;
    FQuotes: string;
    FNumbers: string;
    FOpenComment1: DWORD;
    FCloseComment1: DWORD;
    FOpenComment2: DWORD;
    FCloseComment2: DWORD;
    FEOLComment1: DWORD;
    FEOLComment2: DWORD;
    FAIdents, FANumbers, FASymbols: TCharArray;
    function GetCloseComment(OpenComment: DWORD): DWORD;
    function IsKeyWord(Value: PChar; Len: WORD; var Lexem: TLexemType;
      var KeyIndex: byte): boolean;
    function IsIdentChar(Value: Char; lHeading: boolean): boolean; virtual;
    function GetBlockComment(Source: PChar; OpenComment: DWORD; var LexemItem: TLexemItem;
           AInc: boolean = True): boolean;
    function GetEOLComment(Source: PChar; var LexemItem: TLexemItem; Comment: DWORD): boolean;
    function GetIdent(Source: PChar; var LexemItem: TLexemItem): boolean;
    function GetLex(Source: PChar; var LexemItem: TLexemItem): boolean;
    function GetDecNumber(Source: PChar; var LexemItem: TLexemItem): boolean;
    function GetHexNumber(Source: PChar; var LexemItem: TLexemItem): boolean;
    function GetKeyWordPos(Value: PChar; Index: integer): integer;
    function GetNumber(Source: PChar; var LexemItem: TLexemItem): boolean; virtual;
    function GetString(Source: PChar; var LexemItem: TLexemItem): boolean; virtual;
    procedure InitSyntaxColor; virtual;
    procedure InitSyntaxData; virtual;
    procedure InitHash; virtual;
  public
    constructor Create(AOwner: TComponent); virtual;
    destructor Destroy; override;
    function BuildComment(Value: PChar): DWORD;
    procedure ParseLine(pLineItems: PLineDataItem);
    function IsDelimiter(Value: Char): boolean;
    property DataHighlight: TDataHighlight read FDataHighlight;
    property KeyWords: TSyntaxKeyWords read FKeyWords;
    property LexemColors: TSyntaxLexemColors read FLexemColors;
  end;

  TDCDelphiSyntaxData = class(TDCCustomSyntaxData)
  protected
    procedure InitSyntaxColor; override;
    procedure InitSyntaxData; override;
  public
    function GetNumber(Source: PChar; var LexemItem: TLexemItem): boolean; override;
    function GetString(Source: PChar; var LexemItem: TLexemItem): boolean; override;
  end;

  TDCSQLSyntaxData = class(TDCCustomSyntaxData)
  protected
    procedure InitSyntaxColor; override;
    procedure InitSyntaxData; override;
  public
    function GetNumber(Source: PChar; var LexemItem: TLexemItem): boolean; override;
    function GetString(Source: PChar; var LexemItem: TLexemItem): boolean; override;
  end;

function LISetCapacity(var LexemItems: PLexemItems; Capacity: WORD): PLexemItems;

implementation

{ TDCCustomSyntaxData }

function LISetCapacity(var LexemItems: PLexemItems; Capacity: WORD): PLexemItems;
begin
  ReallocMem(LexemItems, Capacity * SizeOf(TLexemItem));
  Result := LexemItems;
end;

function TDCCustomSyntaxData.BuildComment(Value: PChar): DWORD;
begin
  Result := PDWORD(Value)^;
end;

function TDCCustomSyntaxData.CanOpenHighlight: boolean;
begin
  with FHighlights do
  begin
    Result := not(Enabled and
      (hoUnbreakable in FDataHighlight.Items[Items[Count]].Options));
  end;
end;

function TDCCustomSyntaxData.Comment2ID(Value: DWORD): byte;
begin
  if Value = FOpenComment1 then
    Result := 1
  else
    Result := 2
end;

constructor TDCCustomSyntaxData.Create(AOwner: TComponent);
begin
  inherited Create;

  FOwner := AOwner;
  FLexemColors := TSyntaxLexemColors.Create(Self);
  FKeyWords := TSyntaxKeyWords.Create(Self);

  FDataHighlight := TDataHighlight.Create(Self);

  FHighlights.Enabled := False;
  FHighlights.Count := 0;

  with FLexemColors do
  begin
    Add(lxWhitespace);
    Add(lxIdentifier);
    Add(lxString);
    Add(lxNumber);
    Add(lxComment);
    Add(lxSymbol);
  end;
  InitSyntaxData;
  InitSyntaxColor;
  InitHash;
end;

destructor TDCCustomSyntaxData.Destroy;
begin
  FDataHighlight.Free;
  FLexemColors.Free;
  FKeyWords.Free;
  inherited;
end;

function TDCCustomSyntaxData.GetBlockComment(Source: PChar;
  OpenComment: DWORD; var LexemItem: TLexemItem; AInc: boolean = True): boolean;
 var
  CloseComment: DWORD;
  pValue: PChar;
  i, j: DWORD;
begin
  LexemItem.Item   := lxComment;

  LexemItem.Length := GetCommentLen(OpenComment);
  CloseComment   := GetCloseComment(OpenComment);

  i := GetCommentLen(CloseComment);
  j := 8 * (Sizeof(DWORD) - i);

  pValue := Source;
  Result := True;

  if AInc then Inc(pValue, LexemItem.Length);

  while (pValue^ <> #0) and
        ((PDWORD(pValue)^ shl j) shr j <> CloseComment) do Inc(pValue);

  LexemItem.Length := pValue - Source;

  if pValue^ = #0 then
    FOpenComment := OpenComment
  else begin
    LexemItem.Length := LexemItem.Length + i;
    FOpenComment := 0;
  end;
end;

function TDCCustomSyntaxData.GetCloseComment(OpenComment: DWORD): DWORD;
begin
  if OpenComment = FOpenComment1 then
    Result := FCloseComment1
  else
    if OpenComment = FOpenComment2 then
      Result := FCloseComment2
    else
      Result := 0;
end;

function TDCCustomSyntaxData.GetCommentLen(Comment: integer): integer;
begin
  Result := 0;
  while Comment > 0 do
  begin
    Comment := Comment shr 8;
    inc(Result);
  end;
end;

function TDCCustomSyntaxData.GetDecNumber(Source: PChar;
  var LexemItem: TLexemItem): boolean;

 type
  TNumericPart = (npIntegral, npDecimal, npExponent);

 var
  pValue: PChar;
  NumericPart:  TNumericPart;
  Values: array[TNumericPart] of string[30];
  ESigns: array[TNumericPart] of ShortInt;
begin
  if FHighlights.Enabled and
    (FDataHighlight.Items[FHighlights.Items[FHighlights.Count]].FValues = '') then
  begin
    LexemItem.Item := lxHighlight;
    LexemItem.ItemIndex := FHighlights.Count;
  end
  else
    LexemItem.Item := lxNumber;

  Result   := True;
  pValue   := Source;

  ESigns[npIntegral] := 0;
  ESigns[npDecimal ] := -1;
  ESigns[npExponent] := 0;

  for NumericPart := npIntegral to npExponent do Values[NumericPart] := '';
  NumericPart := npIntegral;

  while (Source^ <> #0) do
  begin
    case Source^ of
      '+', '-':
        if (ESigns[NumericPart] = 0) and (Values[NumericPart] = '') then
          ESigns[NumericPart] := ESigns[NumericPart] + 1
        else
          Break;
      'E', 'e':
        if (NumericPart <> npExponent) and
           ((NumericPart = npIntegral) and (Values[NumericPart] <> '') or
            (NumericPart = npDecimal ) and (Values[NumericPart] <> '')) then
          NumericPart := npExponent
        else
          Break;
      '0'..'9':
        Values[NumericPart] := Values[NumericPart] + Source^;
      else
        if (Source^ = {DecimalSeparator}'.') and
           (NumericPart = npIntegral)
        then
          NumericPart := npDecimal
        else
          Break;
    end;
    Inc(Source);
  end;

  LexemItem.Length := Source - pValue;
end;


function TDCCustomSyntaxData.GetEOLComment(Source: PChar;
  var LexemItem: TLexemItem; Comment: DWORD): boolean;
begin
  Result   := True;
  LexemItem.Item   := lxComment;
  LexemItem.Length := StrLen(Source);
end;

function TDCCustomSyntaxData.GetHexNumber(Source: PChar;
  var LexemItem: TLexemItem): boolean;
 var
  pValue: PChar;
begin
  if FHighlights.Enabled and
    (FDataHighlight.Items[FHighlights.Items[FHighlights.Count]].FValues = '') then
  begin
    LexemItem.Item := lxHighlight;
    LexemItem.ItemIndex := FHighlights.Count;
  end
  else
    LexemItem.Item := lxNumber;

  Result   := True;
  pValue   := Source;

  Inc(Source);
  while (Source^ >= '0') and (Source^ <= '9') or
        (Source^ >= 'A') and (Source^ <= 'F') or
        (Source^ >= 'a') and (Source^ <= 'f') do  Inc(Source);

  LexemItem.Length := Source - pValue;
end;

function TDCCustomSyntaxData.GetIdent(Source: PChar;
  var LexemItem: TLexemItem): boolean;
 var
  pValue: PChar;
begin
  Result  := True;
  pValue  := Source;
  Inc(pValue);

  while (pValue^ <> #0) and IsIdentChar(pValue^, False) do Inc(pValue);

  LexemItem.Length := pValue - Source;
  LexemItem.Item :=  lxIdentifier;

  with LexemItem  do IsKeyWord(Source, Length, Item, ItemIndex);
end;

function TDCCustomSyntaxData.GetKeyWordPos(Value: PChar; Index: integer): integer;
 var
  pValue: PChar;
  c: Char;
begin
  c := (Value + 1)^;

  pValue := nil;
  Result := -1;
  with FKeyWords.Items[Index] do
  begin
    if FHashData.CharArray[c].Length > 0 then
    begin
      Move(PChar(PChar(FValues) + FHashData.CharArray[c].PosB)^,
        FHashData.Data^, FHashData.CharArray[c].Length);
      pValue := StrPos(FHashData.Data, Value)
    end;
    if pValue <> nil then
      Result := FHashData.CharArray[c].PosB + pValue - PChar(FHashData.Data)
  end;
end;

function TDCCustomSyntaxData.GetLex(Source: PChar;
  var LexemItem: TLexemItem): boolean;
 var
  pValue: PChar;
  C: Char;
  I, W: DWORD;
  h: boolean;

 function BeginOpenComment(AComment, W: DWORD): boolean;
  var
   i, j: DWORD;
 begin
   i := GetCommentLen(AComment);
   j := 8 * (Sizeof(DWORD) - i);
   Result := (W shl j) shr j = AComment;
 end;

begin
  FOpenComment := 0;

  if Source^ = #0
  then begin
    Result := False;
    Exit;
  end;

  Result := True;
  pValue := Source;

  while Byte(pValue^) = VK_SPACE do Inc(pValue);

  if pValue - Source > 0 then
  begin
     LexemItem.Item   := lxWhiteSpace;
     LexemItem.Length := pValue - Source;
     Exit;
  end;

  while pValue^ <> #0 do
  begin
    C := pValue^;
    I := Byte(C);
    W := PDWORD(pValue)^;

    if (I = FOpenComment1) or BeginOpenComment(FOpenComment1, W)
    then begin
      Result := GetBlockComment(pValue, FOpenComment1, LexemItem);
      Exit;
    end;
    if (I = FOpenComment2) or BeginOpenComment(FOpenComment2, W)
    then begin
      Result := GetBlockComment(pValue, FOpenComment2, LexemItem);
      Exit;
    end;

    if (I = FEOLComment1) or BeginOpenComment(FEOLComment1, W)
    then begin
      Result := GetEOLComment(pValue, LexemItem, FEOLComment1);
      Exit;
    end;
    if (I = FEOLComment2) or BeginOpenComment(FEOLComment2, W)
    then begin
      Result := GetEOLComment(pValue, LexemItem, FEOLComment2);
      Exit;
    end;

    if StrScan(PChar(FQuotes), C) <> nil then
    begin
      Result := GetString(pValue, LexemItem);
      Exit;
    end;

    if FANumbers[C] then
    begin
      Result := GetNumber(pValue, LexemItem);
      Exit;
    end;

    if FASymbols[C] then
    begin
      Inc(pValue);
      while (pValue^ <> #0) and FASymbols[pValue^] do
      begin
        I := Byte(pValue^);
        W := PWORD(pValue)^;
        if (I = FOpenComment1) or (I = FOpenComment2) or
           (I = FEOLComment1 ) or (I = FEOLComment2 ) or
           (W = FOpenComment1) or (W = FOpenComment2) or
           (W = FEOLComment1 ) or (W = fEOLComment2 ) or
           (StrScan(PChar(FQuotes), pValue^) <> nil) then
          Break;
        Inc(pValue);
      end;
      LexemItem.Item   := lxSymbol;
      LexemItem.Length := pValue - Source;
      LexemItem.ItemIndex := 0;

      h := False;
      with FDataHighlight do
      begin
        if CanOpenHighlight then
          h := HightlightOpen(0, LexemItem.ItemIndex, Source, LexemItem.Item,
             FHighlights, LexemItem.Length);
        if FHighlights.Enabled and not h then
          CheckHighlight(0, Source, FHighlights, LexemItem.Item,
            LexemItem.ItemIndex, LexemItem.Length)
      end;
      Exit;
    end;

    if IsIdentChar(C, True) then
    begin
      Result := GetIdent(pValue, LexemItem);
      Exit;
    end;

    Inc(pValue);
  end;

  if pValue - Source > 0 then
  begin
     LexemItem.Item   := lxWhiteSpace;
     LexemItem.Length := pValue - Source;
  end;

end;

function TDCCustomSyntaxData.GetNumber(Source: PChar;
  var LexemItem: TLexemItem): boolean;
 var
  pValue: PChar;
begin
  if FHighlights.Enabled and
    (FDataHighlight.Items[FHighlights.Items[FHighlights.Count]].FValues = '') then
  begin
    LexemItem.Item := lxHighlight;
    LexemItem.ItemIndex := FHighlights.Count;
  end
  else
    LexemItem.Item := lxNumber;

  Result   := True;
  pValue   := Source;

  Inc(Source);
  while (Source^ >= '0') and (Source^ <= '9') do  Inc(Source);

  LexemItem.Length := Source - pValue;
end;

function TDCCustomSyntaxData.GetString(Source: PChar;
  var LexemItem: TLexemItem): boolean;
 var
  Quote : Char;
  pValue: PChar;
begin
  if FHighlights.Enabled and
    (FDataHighlight.Items[FHighlights.Items[FHighlights.Count]].FValues = '') then
  begin
    LexemItem.Item := lxHighlight;
    LexemItem.ItemIndex := FHighlights.Count;
  end
  else
    LexemItem.Item := lxString;

  Result   := True;
  Quote    := Source^;
  pValue   := Source;

  Inc(Source);

  while (Source^ <> #0) and (Source^ <> Quote) do Inc(Source);

  LexemItem.Length := Source - pValue;

  if Source^ <> #0 then Inc(LexemItem.Length);
end;

function TDCCustomSyntaxData.ID2Comment(Value: byte): DWORD;
begin
  case Value of
    1: Result := FOpenComment1;
    2: Result := FOpenComment2;
    else
      Result := 0
  end;
end;

procedure TDCCustomSyntaxData.InitHash;
 var
  c: Char;
begin
  for c:= #0 to #255 do
    FAIdents[c] := (c in ['a'..'z', 'A'..'Z', '_']) or
     ((c >= '') and (c <= '')) or ((c >= '') and (c <= '')) or (c in ['0'..'9']);

  for c:= #0 to #255 do
    FANumbers[c] := (c in ['0'..'9']) or (StrScan(PChar(FNumbers), c) <> nil);

  for c:= #0 to #255 do
    FASymbols[c] := (StrScan(PChar(FSymbols), c) <> nil);
end;

procedure TDCCustomSyntaxData.InitSyntaxColor;
begin
  {}
end;

procedure TDCCustomSyntaxData.InitSyntaxData;
begin
  {}
end;

function TDCCustomSyntaxData.IsDelimiter(Value: Char): boolean;
begin
  Result := (Value = #32) or (StrScan(PChar(FSymbols), Value) <> nil);
end;

function TDCCustomSyntaxData.IsIdentChar(Value: char;
  lHeading: boolean): boolean;
begin
  Result := FAIdents[Value]
end;

function TDCCustomSyntaxData.IsKeyWord(Value: PChar; Len: WORD;
  var Lexem: TLexemType; var KeyIndex: byte): boolean;
 var
  pUpperValue: PChar;
  ValuePos, i: integer;
  h: boolean;
begin
  pUpperValue := AllocMem(Len + 3);
  try
    pUpperValue^ := ' ';
    StrLCopy(PChar(pUpperValue + 1), Value, Len);
    (pUpperValue + Len + 1)^ := ' ';

    CharUpperBuff(Pointer(pUpperValue + 1), Len);

    ValuePos := -1;
    for i := 0 to FKeyWords.Count - 1 do
    begin
      ValuePos := GetKeyWordPos(pUpperValue, i);
      if ValuePos <> -1 then
      begin
        KeyIndex := i;
        Break;
      end;
    end;

    Result := ValuePos <> -1;

    h := False;
    if Result then
    begin
      {  DataHighlight}
      Lexem := lxKeyWord;
      if CanOpenHighlight then
        h := FDataHighlight.HightlightOpen(ValuePos, KeyIndex, pUpperValue,
          Lexem, FHighlights, Len);
    end;
    if FHighlights.Enabled and not h then
    begin
      if not(FDataHighlight.CheckHighlight(ValuePos, pUpperValue, FHighlights,
        Lexem, KeyIndex, Len) or Result) then
      begin
        Lexem := lxIdentifier
      end;
    end

  finally
    FreeMem(pUpperValue, Len + 3)
  end;
end;

procedure TDCCustomSyntaxData.ParseLine(pLineItems: PLineDataItem);
 var
  LexemItem: TLexemItem;
  Source: PChar;

  procedure AddItem;
  begin
    with pLineItems^ do
    begin
      if (Count > 0) and (Lexems^[Count-1].Item = LexemItem.Item)  then
        Inc(Lexems^[Count-1].Length, LexemItem.Length)
      else begin
        if Count = Capacity then
        begin
          if Capacity > 16 then
            Inc(Capacity, 16)
          else
            if Capacity > 8 then
              Inc(Capacity, 8)
            else
              Inc(Capacity, 4);
          Lexems := LISetCapacity(Lexems, Capacity);
        end;
        with Lexems^[Count] do
        begin
          Item := LexemItem.Item;
          ItemIndex := LexemItem.ItemIndex;
          Length := LexemItem.Length;
        end;
        Inc(Count);
      end;
      Highlights := FHighlights;
    end;

    Inc(Source, LexemItem.Length);
    if (LexemItem.Item = lxComment) and (FOpenComment <> 0) then
      pLineItems^.LComment := Comment2ID(FOpenComment);
  end;

begin
  with pLineItems^ do
  begin
    Count   := 0;
    LComment := 0;
    Source  := PChar(FString);
    FHighlights := Highlights;
  end;

  if pLineItems^.HComment <> 0
  then begin
     if GetBlockComment(Source, ID2Comment(pLineItems^.HComment), LexemItem, False) then
     begin
       AddItem;
       if FOpenComment <> 0 then
       begin
         pLineItems^.HComment := Comment2ID(FOpenComment);
         Exit;
       end;
     end;
  end;

  while GetLex(Source, LexemItem) do AddItem;

end;

{ TDCSyntaxMemoColor }

procedure TDCSyntaxMemoColor.AssignTo(Dest: TPersistent);
begin
  if Dest is TDCSyntaxMemoColor then
    with TDCSyntaxMemoColor(Dest) do
    begin
      FBackground := Self.FBackground;
      FFontStyle := Self.FFontStyle;
      FForeground := Self.FForeground;
      Change;
    end
  else
    inherited AssignTo(Dest);
end;

procedure TDCSyntaxMemoColor.Change;
begin
  if Assigned(OnChange) then FOnChange(Self);
end;

constructor TDCSyntaxMemoColor.Create(SyntaxData: TDCCustomSyntaxData);
begin
  inherited Create;
  FBackground := clWindow;
  FFontStyle := [];
  FForeground := clBlack;
  FSyntaxData := SyntaxData;
end;

procedure TDCSyntaxMemoColor.SetBackground(const Value: TColor);
begin
  if FBackground <> Value then
  begin
    FBackground := Value;
    Change;
  end;
end;

procedure TDCSyntaxMemoColor.SetFontStyle(const Value: TFontStyles);
begin
  if FFontStyle <> Value then
  begin
    FFontStyle := Value;
    Change;
  end;
end;

procedure TDCSyntaxMemoColor.SetForeground(const Value: TColor);
begin
  if FForeground <> Value then
  begin
    FForeground := Value;
    Change;
  end;
end;

{ TDCDelphiSyntaxData }

function TDCDelphiSyntaxData.GetNumber(Source: PChar;
  var LexemItem: TLexemItem): boolean;
begin
  if Source^ = '$' then
    Result := GetHexNumber(Source, LexemItem)
  else
    Result := GetDecNumber(Source, LexemItem)
end;

function TDCDelphiSyntaxData.GetString(Source: PChar;
  var LexemItem: TLexemItem): boolean;
 var
  Quote : Char;
  pValue: PChar;
  W, EscapeValue: WORD;
begin
   LexemItem.Item := lxString;

   Result   := True;
   Quote    := Source^;
   pValue   := Source;

   if Source^ = '#' then
   begin
     Inc(Source);
     while (Source^ <> #0) and (Source^ >= '0') and (Source^ <= '9') do Inc(Source);
   end
   else begin
     Inc(Source);
     EscapeValue := Ord('''') shl 8 or Ord('''');
     while (Source^ <> #0) do begin
       W := PWORD(Source)^;
       if W = EscapeValue then
         Inc(Source)
       else
         if Source^ = Quote then Break;
       Inc(Source)
     end;
     if Source^ <> #0 then Inc(Source);
   end;

   LexemItem.Length := Source - pValue;
end;

procedure TDCDelphiSyntaxData.InitSyntaxColor;
begin
  inherited;
  with FLexemColors do
  begin
    Items[lxNumber ].FColor.FForeground := clRed;
    Items[lxString ].FColor.FForeground := clBlue;
    Items[lxComment].FColor.FForeground := clNavy;

    Items[lxNumber ].FColor.FontStyle := [fsBold];
    Items[lxString ].FColor.FontStyle := [fsBold];
    Items[lxComment].FColor.FontStyle := [fsItalic];
  end;

  FKeyWords.Items[0].FColor.FontStyle := [fsBold];

  FDataHighlight.Items[0].FColor.FForeground := clGray;
  FDataHighlight.Items[1].FColor.FontStyle := [fsBold];
end;

procedure TDCDelphiSyntaxData.InitSyntaxData;
 var
  Value: string;
  Item: TSyntaxKeyWordItem;
begin
  inherited;
  Value := ' AND ARRAY AS ASM BEGIN CASE CLASS CONST CONSTRUCTOR' +
           ' DESTRUCTOR DISPINTERFACE DIV DO DOWNTO ELSE END EXCEPT' +
           ' EXPORTS FILE FINALIZATION FINNALY FOR FUNCTION GOTO IF' +
           ' IMPLEMENTATION IN INHERITED INITIALIZATION INLINE INTERFACE' +
           ' IS LABEL LIBRARY MOD NIL NOT OBJECT OF OR OUT PACKED'  +
           ' PROCEDURE PROGRAM PROPERTY RAISE RECORD REPEAT RESOURCESTRING' +
           ' SET SHL SHR STRING THEN THREADVAR TO TRY TYPE UNIT UNTIL USES' +
           ' VAR WHILE WITH XOR MESSAGE' +
           {property keywords}
           ' PRIVATE PROTECTED PUBLIC PUBLISHED' +
           {special}
           ' ON ';

  Item := FKeyWords.Add(Value);

  with FDataHighlight.AddItem(' ASM ', ' END ', Item.Index, lxHighlight, -1, '') do
  begin
    Options := [hoUnbreakable];
  end;

  FDataHighlight.AddItem(' PROPERTY ', ';', Item.Index, lxKeyWord,
    Item.Index, ' READ WRITE DEFAULT STORED ');

  FSymbols := '`~!@#$%^&*()-+=|\{[}]:;<,>.?/"''';
  FQuotes   := '#''';
  FNumbers  := '$';

  FOpenComment1  := BuildComment('{');
  FCloseComment1 := BuildComment('}') ;
  FOpenComment2  := BuildComment('(*');
  FCloseComment2 := BuildComment('*)');

  FEOLComment1  := BuildComment('//');
  FEOLComment2  := 0;
end;

{ TDCSQLSyntaxData }

function TDCSQLSyntaxData.GetNumber(Source: PChar;
  var LexemItem: TLexemItem): boolean;
begin
  if Source^ = '$' then
    Result := GetHexNumber(Source, LexemItem)
  else
    Result := GetDecNumber(Source, LexemItem)
end;

function TDCSQLSyntaxData.GetString(Source: PChar;
  var LexemItem: TLexemItem): boolean;
 var
  Quote : Char;
  pValue: PChar;
  W, EscapeValue: WORD;
begin
   LexemItem.Item := lxString;

   Result   := True;
   Quote    := Source^;
   pValue   := Source;

   Inc(Source);
   EscapeValue := Ord('''') shl 8 or Ord('''');
   while (Source^ <> #0) do begin
     W := PWORD(Source)^;
     if W = EscapeValue then
       Inc(Source)
     else
       if Source^ = Quote then Break;
     Inc(Source)
   end;
   if Source^ <> #0 then Inc(Source);

   LexemItem.Length := Source - pValue;
end;

procedure TDCSQLSyntaxData.InitSyntaxColor;
begin
  inherited;
  with FLexemColors do
  begin
    Items[lxNumber ].FColor.FForeground := clNavy;
    Items[lxString ].FColor.FForeground := clred;
    Items[lxComment].FColor.FForeground := clDkGray;
  end;
  FKeyWords.Items[0].FColor.FForeground := clBlue;
  FKeyWords.Items[1].FColor.FForeground := clGray;
  FKeyWords.Items[2].FColor.FForeground := clGreen;
  FKeyWords.Items[3].FColor.FForeground := clFuchsia;
end;

procedure TDCSQLSyntaxData.InitSyntaxData;
 var
  Value: string;
begin
  inherited;
  Value := ' ADD ALL ALTER ANY AS ASC AUTHORIZATION AVG BACKUP BEGIN' +
           ' BETWEEN BREAK BROWSE BULK BY CASCADE CHECKPOINT' +
           ' CLOSE CLUSTERED COLESCE COLUMN COMMIT COMMITED COMPUTE' +
           ' CONFIRM CONSTRAINT CONTAINS CONTAINSTABLE CONTINUE' +
           ' CONTROLROW CREATE CROSS CURRENT CURRENT_DATE' +
           ' CURRENT_TIME CURRENT_TIMESTAMP CURRENT_USER CURSOR DATABASE' +
           ' DBCC DEALLOCATE DECLARE DEFAULT DELETE DENY DESC DISK' +
           ' DISTINCT DISTRIBUTED DOUBLE DROP DUMMY DUMP ELSE END ERRLVL' +
           ' ERROREXIT EXCEPT EXEC EXECUTE' +

           ' EXIT FETCH FILE FILLFACTOR FLOPPY FOR FOREIGN FREETEXT' +
           ' FREETEXTTABLE FROM FULL GOTO GRANT GROUP HAVING HOLDLOCK' +
           ' IDENTITY IDENTITY_INSERT IDENTITYCOL IF INDEX INNER' +
           ' INSERT INTERSECT INTO IS ISOLATION JOIN KEY KILL LEFT' +
           ' LEVEL LIKE LINENO LOAD MAX MIN MIRROREXIT NATIONAL NOCHECK' +
           ' NONCLUSTERED OF OFF OFFSETS ON ONCE ONLY' +
           ' OPEN OPENDATASOURCE OPENQUERY OPENROWSET OPTION ORDER' +
           ' OUTER OVER PERCENT PERM PERMANENT PIP PLAN PRECISION' +

           ' PRIMARY PRINT PRIVILEGES PROC PROCEDURE PROCESSEXIT PUBLIC' +
           ' RAISERROR READ READTEXT RECONFIGURE REFERENCES REPEATABLE' +
           ' REPLICATION RESTORE RESTRICT RETURN REVOKE RIGHT ROLLBACK' +
           ' ROWCOUNT ROWGUIDCOL RULE SAVE SCHEMA SELECT SERIALIZABLE' +
           ' SESSION_USER SET SETUSER SHUTDOWN STATISTICS SUM' +
           ' SYSTEM_USER TABLE TAPE TEMP TEMPORARY TEXTSIZE THEN TO TOP' +
           ' TRAN TRANSACTION TRIGGER TRUNCATE TSEQUAL UNCOMMITTED UNION' +
           ' UNIQUE UPDATE UPDATETEXT USE USER VALUES VARYING VIEW WAITFOR' +
           ' WHEN WHERE WHILE WITH WORK WRITETEXT PREPARE ';

  FKeyWords.Add(Value);

  Value := ' IN OR AND NOT NULL EXISTS SOME ';
  FKeyWords.Add(Value);

  Value := ' SYSLOGINS SYSOBJECTS SYSCOLUMNS SYSINDEXES SYSUSERS SYSMEMBERS' +
           ' SYSTEM SYSDEVICES SYSDATABASES ';
  FKeyWords.Add(Value);

  Value := ' CASE CONVERT COUNT NULLIF PI OBJECT_ID SUBSTRING CHARINDEX LOWER' +
           ' GETDATE ';
  FKeyWords.Add(Value);

  FSymbols  := '`~!@#$%^&*()-+=|\{[}]:;<,>.?/"''';
  FQuotes    := '#''"';
  FNumbers   := '$';

  FOpenComment1  := BuildComment('/*');
  FCloseComment1 := BuildComment('*/');
  FOpenComment2  := 0;
  FCloseComment2 := 0;

  FEOLComment1  := BuildComment('--');
  FEOLComment2  := 0;
end;

{ TDataHighligh }

function TDataHighlight.AddItem(AOpenLexem, ACloseLexem: string;
     AKeyIndex: integer; ALexemType: TLexemType; ALexemIndex: integer;
     AValues: string): TDataHighlightItem;
begin
  Result := TDataHighlightItem(inherited Add);
  with Result do
  begin
    if AOpenLexem[1] =' ' then
      FOpenLexemPos := FSyntaxData.GetKeyWordPos(PChar(AOpenLexem), AKeyIndex)
    else begin
      FOpenLexem := AOpenLexem;
      FOpenLexemPos := -1;
    end;

    if ACloseLexem[1] = ' ' then
      FCloseLexemPos := FSyntaxData.GetKeyWordPos(PChar(ACloseLexem), AKeyIndex)
    else begin
      FCloseLexem := ACloseLexem;
      FCloseLexemPos := -1;
    end;

    FKeyIndex := AKeyIndex;
    FValues := AValues;
    FLexemType := ALexemType;
    if ALexemIndex <> -1 then
      FLexemIndex := ALexemIndex
    else
      FLexemIndex := Result.Index;
  end;
end;

function TDataHighlight.CheckHighlight(const ValuePos: integer; Value: PChar;
  var Highlights: THighlights; var Lexem: TLexemType; var LexemIndex: byte;
  var LexemLength: WORD): boolean;
 var
  LexemOffset: integer;

 procedure DeleteHighlightItem;
 begin
   with Highlights do
   begin
     Dec(Count);
     if Count = 0 then Enabled := False;
   end;
 end;

begin
  with Items[Highlights.Items[Highlights.Count]] do
  begin
    if (CloseLexemPos = -1) and (ValuePos = 0) then
    begin
      LexemOffset := StrLPos(FCloseLexem, Value, LexemLength);
      if LexemOffset = 0 then
      begin
        Result := True;
        if hoIncludeBounds in Options then
        begin
          LexemIndex := FLexemIndex;
          Lexem := FLexemType;
        end;
        DeleteHighlightItem;
      end
      else begin
        if LexemOffset > 0 then LexemLength := LexemOffset;
        Result := False;
      end;
      Exit;
    end;

    if (CloseLexemPos <> -1)  and (CloseLexemPos = ValuePos) then
    begin
      Result := True;
      if hoIncludeBounds in Options then LexemIndex := FLexemIndex;
      DeleteHighlightItem
    end
    else begin
      if (FValues = '') or (StrPos(PChar(FValues), Value) <> nil) then
      begin
        Lexem := FLexemType;
        LexemIndex := FLexemIndex;
        Result := True;
      end
      else
        Result := False;
    end;
  end;
end;

constructor TDataHighlight.Create(SyntaxData: TDCCustomSyntaxData);
begin
  inherited Create(TDataHighlightItem);
  FSyntaxData := SyntaxData;
end;

destructor TDataHighlight.Destroy;
begin
  inherited;
end;

function TDataHighlight.GetItem(Index: Integer): TDataHighlightItem;
begin
  Result := TDataHighlightItem(inherited GetItem(Index))
end;

function TDataHighlight.HightlightOpen(LexemPos, KeyIndex: integer;
  Value: PChar; var Lexem: TLexemType; var Highlights: THighlights;
  var LexemLength: WORD): boolean;
 var
  i, LexemOffset, c: integer;

 procedure AddHighlightItem(Index: byte);
 begin
   with Highlights do if Count < 4 then
   begin
     Inc(Count);
     Items[Count] := Index;
     Enabled := True;
   end
 end;

begin
  c := Highlights.Count;
  for i := 0 to Count - 1 do
    with Items[i] do
    begin
      if FKeyIndex = KeyIndex then
      begin
        if (OpenLexemPos = -1) and (LexemPos = 0) then
        begin
          LexemOffset := StrLPos(FCloseLexem, Value, LexemLength);
          if LexemOffset = 0 then
          begin
            AddHighlightItem(i);
            if hoIncludeBounds in FOptions then Lexem := FLexemType;
            Break;
          end
          else begin
            if LexemOffset > 0 then
            begin
              LexemLength := LexemOffset;
              Break;
            end;
          end;
        end
        else begin
          if OpenLexemPos = LexemPos then
          begin
            AddHighlightItem(i);
            if hoIncludeBounds in FOptions then Lexem := FLexemType;
            Break;
          end;
        end;
      end;
    end;
  Result := c <> Highlights.Count;
end;

procedure TDataHighlight.SetItem(Index: Integer;
  const Value: TDataHighlightItem);
begin
  inherited SetItem(Index, Value);
end;

function TDataHighlight.StrLPos(const Str1: string; const pStr2: PChar;
  Len: integer): integer;
 var
  l: integer;
  pStr1, pStr3: PChar;
  StrEquals: boolean;
begin
  l := Length(Str1);
  pStr1 := PChar(Str1);
  pStr3 := pStr2;
  StrEquals := False;

  while (Len >= l) and not StrEquals do
  begin
    StrEquals := StrLComp(pStr3, pStr1, l) = 0;
    if not StrEquals then
    begin
      Inc(pStr3);
      Dec(Len);
    end;
  end;
  if not StrEquals then
    Result := -1
  else
    Result := pStr3 - pStr2;
end;

{ TSyntaxKeyWords }

function TSyntaxKeyWords.Add(const Values: string): TSyntaxKeyWordItem;
begin
  Result := TSyntaxKeyWordItem(inherited Add);
  Result.FValues := Values;
  Result.InitHashData;
end;

constructor TSyntaxKeyWords.Create(SyntaxData: TDCCustomSyntaxData);
begin
  inherited Create(TSyntaxKeyWordItem);
  FSyntaxData := SyntaxData;
end;

function TSyntaxKeyWords.GetItem(Index: Integer): TSyntaxKeyWordItem;
begin
  Result := TSyntaxKeyWordItem(inherited GetItem(Index));
end;

procedure TSyntaxKeyWords.SetItem(Index: Integer;
  const Value: TSyntaxKeyWordItem);
begin
  inherited SetItem(Index, Value);
end;

{ TSyntaxKeyWordItem }

constructor TSyntaxKeyWordItem.Create(Collection: TCollection);
begin
  inherited;
  FColor := TDCSyntaxMemoColor.Create(TSyntaxKeyWords(Collection).FSyntaxData);
end;

destructor TSyntaxKeyWordItem.Destroy;
begin
  if FHashData.Data <> nil then FreeMem(FHashData.Data, FHashData.Size);
  FColor.Free;
  inherited;
end;

procedure TSyntaxKeyWordItem.InitHashData;
 var
  c: Char;
  p, pValue: PChar;
  s: string;
  j: integer;
begin
  FHashData.Size := Length(FValues) + 1;
  if FHashData.Data = nil then
    FHashData.Data := AllocMem(FHashData.Size)
  else
    ReallocMem(FHashData.Data, FHashData.Size);

  FillChar(FHashData.CharArray, SizeOf(FHashData.CharArray), 0);

  for c:= #1 to #255 do with FHashData do
  begin
    CharArray[c].Length := 1;
  end;

  SortKeyWords;

  pValue := PChar(FValues);
  j := Length(FValues);
  for c:= #1 to #255 do with FHashData do
  begin
    if c <> ' ' then
    begin
      s := ' '  + c;
      p := StrPos(pValue, PChar(s));
      if p <> nil then
      begin
        CharArray[c].Length := 1;
        CharArray[c].PosB := p - PChar(FValues);
        pValue := p;
      end
      else
        CharArray[c].Length := 0;
    end
    else
       CharArray[c].Length := 0;
  end;

  pValue := PChar(FValues);
  p := pValue + j;
  for c:= #255 downto #1 do with FHashData do
  begin
    if CharArray[c].Length <> 0 then
    begin
      CharArray[c].Length := p - PChar(pValue + CharArray[c].PosB);
      if p^ <> #0 then Inc(CharArray[c].Length);
      p := PChar(pValue + CharArray[c].PosB);
    end;
  end;
end;

procedure TSyntaxKeyWordItem.SetColor(const Value: TDCSyntaxMemoColor);
begin
  FColor.Assign(Value);
end;

procedure TSyntaxKeyWordItem.SortKeyWords;
 var
  Value: string;
  c: Char;
  p, p1, p2: PChar;
  i: integer;
  FindKeyWord: boolean;
begin
  Value := '';
  for c:= #1 to #255 do
  begin
    if FHashData.CharArray[c].Length <> 0 then
    begin
      p  := PChar(FValues);
      p1 := nil;
      p2 := nil;
      i := 0;
      FindKeyWord := False;
      while p^ <> #0 do
      begin
        if p^ = ' ' then
          if FindKeyWord then
          begin
           Value := Value + Copy(FValues, i, p - p2 + 1);
            Delete(FValues, i, p - p2 + 1);
            p := PChar(FValues);
            FindKeyWord := False;
          end
          else
        else
          if (p^ = c) and (p1^ = ' ') then
          begin
            i := p1 - PChar(FValues) + 1;
            FindKeyWord := True;
            p2 := p;
          end;
        p1 := p;
        Inc(p);
      end;
    end;
  end;
  FValues := Value + ' ' + FValues;
end;

{ TDataHighlightItem }

constructor TDataHighlightItem.Create(Collection: TCollection);
begin
  inherited;
  FColor := TDCSyntaxMemoColor.Create(TSyntaxKeyWords(Collection).FSyntaxData);
  FValues := '';
  FOptions := [];
end;

destructor TDataHighlightItem.Destroy;
begin
  FColor.Free;
  inherited;
end;

procedure TDataHighlightItem.SetColor(const Value: TDCSyntaxMemoColor);
begin
  FColor.Assign(Value);
  Changed(True);
end;

procedure TDataHighlightItem.SetOptions(const Value: THighlighOptions);
begin
  FOptions := Value;
  Changed(True);
end;

{ TLexemColorItem }

constructor TLexemColorItem.Create(Collection: TCollection);
begin
  inherited;
  FColor := TDCSyntaxMemoColor.Create(TSyntaxKeyWords(Collection).FSyntaxData);
end;

destructor TLexemColorItem.Destroy;
begin
  FColor.Free;
  inherited;
end;

procedure TLexemColorItem.SetColor(const Value: TDCSyntaxMemoColor);
begin
  FColor.Assign(Value);
end;

procedure TLexemColorItem.SetLexemType(const Value: TLexemType);
begin
  {}
end;

{ TSyntaxLexemColors }

function TSyntaxLexemColors.Add(ALexemType: TLexemType): TLexemColorItem;
begin
  Result := TLexemColorItem(inherited Add);
  Result.FLexemType := ALexemType;
end;

constructor TSyntaxLexemColors.Create(SyntaxData: TDCCustomSyntaxData);
begin
  inherited Create(TLexemColorItem);
  FSyntaxData := SyntaxData;
end;

function TSyntaxLexemColors.GetItem(Index: TLexemType): TLexemColorItem;
 var
  i: integer;
begin
  Result := nil;
  for i := 0 to Count - 1 do
  begin
    if TLexemColorItem(inherited GetItem(i)).FLexemType = Index then
    begin
      Result := TLexemColorItem(inherited GetItem(i));
      Break;
    end
  end;
end;

end.


