{
 BUSINESS CONSULTING
 s a i n t - p e t e r s b u r g

         Components Library for Borland Delphi 4.x - 6.x
         Copyright (c) 1998-2001 Alex'EM

}
unit DCPopupMenu;

{$I DCConst.inc}
interface

uses
  SysUtils, Classes, ImgList, DCEditTools;

type
  TDCPopupMenu = class;
  TDCPopupMenuItem = class;
  TDCPopupMenuItemClass = class of TDCPopupMenuItem;

  TDCPopupMenuChangeEvent = procedure (Sender: TObject; Source: TDCPopupMenuItem;
    Rebuild: Boolean) of object;

  TDCPopupMenuItem = class(TComponent)
  private
    FCaption: string;
    FChecked: boolean;
    FEnabled: boolean;
    FGroupIndex: Byte;
    FImageIndex: integer;
    FItems: TList;
    FMenu: TDCPopupMenu;
    FOnChange: TDCPopupMenuChangeEvent;
    FParent: TDCPopupMenuItem;
    FRadioItem: Boolean;
    FVisible: Boolean;
    function GetMenuIndex: Integer;
    procedure RebuildHandle;
    procedure SubItemChanged(Sender: TObject; Source: TDCPopupMenuItem;
      Rebuild: Boolean);
    procedure TurnSiblingsOff;
    procedure VerifyGroupIndex(Position: Integer; Value: Byte);
  protected
    function GetCount: Integer;
    function GetItem(Index: Integer): TDCPopupMenuItem;
    function GetParentMenu: TDCPopupMenu;
    procedure MenuChanged(Rebuild: Boolean); virtual;
    procedure SetCaption(const Value: string);
    procedure SetChecked(Value: Boolean);
    procedure SetEnabled(const Value: Boolean);
    procedure SetGroupIndex(const Value: Byte);
    procedure SetImageIndex(const Value: integer);
    procedure SetMenuIndex(Value: Integer);
    procedure SetRadioItem(const Value: Boolean);
    procedure SetVisible(const Value: Boolean);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function Add(AName: string;
      ItemClass: TDCPopupMenuItemClass = nil): TDCPopupMenuItem;
    procedure Clear;
    procedure Click; virtual;
    procedure Delete(Index: Integer);
    function IndexOf(Item: TDCPopupMenuItem): Integer;
    procedure Insert(Index: Integer; Item: TDCPopupMenuItem);
    procedure Remove(Item: TDCPopupMenuItem);
    property Count: integer read GetCount;
    property Items[Index: Integer]: TDCPopupMenuItem read GetItem; default;
    property MenuIndex: Integer read GetMenuIndex write SetMenuIndex;
    property Parent: TDCPopupMenuItem read FParent;
  published
    property Caption: string read FCaption write SetCaption;
    property Enabled: Boolean read FEnabled write SetEnabled default True;
    property GroupIndex: Byte read FGroupIndex write SetGroupIndex default 0;
    property ImageIndex: integer read FImageIndex write SetImageIndex default -1;
    property RadioItem: Boolean read FRadioItem write SetRadioItem default False;
    property Visible: Boolean read FVisible write SetVisible;
  end;

  TDCPopupMenuInitialize = packed record
    MenuItemClass: TDCPopupMenuItemClass;
  end;

  TDCPopupMenu = class(TComponent)
  private
    FItems: TDCPopupMenuItem;
    FImageChangeLink: TChangeLink;
    FImages: TCustomImageList;
    FItemClass: TDCPopupMenuItemClass;
    FOnChange: TDCPopupMenuChangeEvent;
    procedure ImageListChange(Sender: TObject);
    procedure SetImages(const Value: TCustomImageList);
  protected
    function Initialize: TDCPopupMenuInitialize; virtual;
    function GetItems: TDCPopupMenuItem;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    procedure UpdateItems;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property Images: TCustomImageList read FImages write SetImages;
    property Items: TDCPopupMenuItem read GetItems;
    property OnChange: TDCPopupMenuChangeEvent read FOnChange write FOnChange;
  end;

implementation

{ TDCCustomPopupMenu }

constructor TDCPopupMenu.Create(AOwner: TComponent);
 var
  MenuCreateItems: TDCPopupMenuInitialize;
begin
  inherited;
  MenuCreateItems := Initialize;
  FItemClass := MenuCreateItems.MenuItemClass;

  FItems := FItemClass.Create(Self);
  FItems.FMenu := Self;

  FImageChangeLink := TChangeLink.Create;
  FImageChangeLink.OnChange := ImageListChange;
end;

destructor TDCPopupMenu.Destroy;
begin
  FItems.Destroy;
  inherited;
end;

function TDCPopupMenu.GetItems: TDCPopupMenuItem;
begin
  Result := FItems;
end;

procedure TDCPopupMenu.ImageListChange(Sender: TObject);
begin
  UpdateItems;
end;

function TDCPopupMenu.Initialize: TDCPopupMenuInitialize;
begin
  Result.MenuItemClass := TDCPopupMenuItem;
end;

procedure TDCPopupMenu.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (AComponent = Images) and (Operation = opRemove) then Images := nil;
end;

procedure TDCPopupMenu.SetImages(const Value: TCustomImageList);
begin
  if FImages <> nil then FImages.UnRegisterChanges(FImageChangeLink);
  FImages := Value;
  if FImages <> nil then
  begin
    FImages.RegisterChanges(FImageChangeLink);
    FImages.FreeNotification(Self);
  end;
  UpdateItems;
end;

procedure TDCPopupMenu.UpdateItems;
begin
  {}
end;

{ TDCCustomPopupMenuItem }

function TDCPopupMenuItem.Add(AName: string;
  ItemClass: TDCPopupMenuItemClass): TDCPopupMenuItem;
 var
  ParentMenu: TDCPopupMenu;

  function DoAddItem: TDCPopupMenuItem;
  begin
    Result := ItemClass.Create(nil);
    Result.Caption := AName;
    Insert(Count, Result)
  end;
begin
  if ItemClass = nil then
  begin
    ParentMenu := GetParentMenu;
    if Assigned(ParentMenu) then
    begin
      ItemClass := ParentMenu.FItemClass;
      Result := DoAddItem
    end
    else
      Result := nil
  end
  else
    Result := DoAddItem;
end;

procedure TDCPopupMenuItem.Clear;
 var
  i: Integer;
begin
  for i := Count - 1 downto 0 do Items[i].Free;
end;

procedure TDCPopupMenuItem.Click;
begin
  {}
end;

constructor TDCPopupMenuItem.Create(AOwner: TComponent);
begin
  inherited;
  FVisible := True;
  FEnabled := True;
  FImageIndex := -1;
end;

procedure TDCPopupMenuItem.Delete(Index: Integer);
 var
  Cur: TDCPopupMenuItem;
begin
  if (Index < 0) or (FItems = nil) or (Index >= GetCount) then Exit;
  Cur := FItems[Index];
  FItems.Delete(Index);
  Cur.FParent := nil;
  Cur.FOnChange := nil;
  MenuChanged(Count = 0);
end;

destructor TDCPopupMenuItem.Destroy;
begin
  if Assigned(FItems) then FreeAndNil(FItems);
  inherited;
end;

function TDCPopupMenuItem.GetCount: Integer;
begin
  if FItems = nil then Result := 0 else Result := FItems.Count;
end;

function TDCPopupMenuItem.GetItem(Index: Integer): TDCPopupMenuItem;
begin
  if FItems <> nil then Result := FItems[Index] else Result := nil
end;

function TDCPopupMenuItem.GetMenuIndex: Integer;
begin
  Result := -1;
  if FParent <> nil then Result := FParent.IndexOf(Self);
end;

function TDCPopupMenuItem.GetParentMenu: TDCPopupMenu;
 var
  MenuItem: TDCPopupMenuItem;
begin
  MenuItem := Self;
  while Assigned(MenuItem.FParent) do MenuItem := MenuItem.FParent;
  Result := MenuItem.FMenu
end;

function TDCPopupMenuItem.IndexOf(Item: TDCPopupMenuItem): Integer;
begin
  Result := -1;
  if FItems <> nil then Result := FItems.IndexOf(Item);
end;

procedure TDCPopupMenuItem.Insert(Index: Integer; Item: TDCPopupMenuItem);
begin
  if Item.FParent <> nil then Exit;
  if FItems = nil then FItems := TList.Create;

  if (Index - 1 >= 0) and (Index - 1 < FItems.Count) then
    if Item.GroupIndex < TDCPopupMenuItem(FItems[Index - 1]).GroupIndex then
      Item.GroupIndex := TDCPopupMenuItem(FItems[Index - 1]).GroupIndex;

  VerifyGroupIndex(Index, Item.GroupIndex);
  FItems.Insert(Index, Item);
  Item.FParent := Self;
  Item.FOnChange := SubItemChanged;
  MenuChanged(Count = 1);
end;

procedure TDCPopupMenuItem.MenuChanged(Rebuild: Boolean);
 var
  Source: TDCPopupMenuItem;
begin
  if (Parent = nil) and (Owner is TDCPopupMenu) then
    Source := nil
  else
    Source := Self;
    
  if Assigned(FOnChange) then FOnChange(Self, Source, Rebuild);
end;

procedure TDCPopupMenuItem.RebuildHandle;
begin
  {}
end;

procedure TDCPopupMenuItem.Remove(Item: TDCPopupMenuItem);
 var
  i: Integer;
begin
  i := IndexOf(Item);
  if i <> -1 then Delete(I);
end;

procedure TDCPopupMenuItem.SetCaption(const Value: string);
begin
  if FCaption <> Value then
  begin
    FCaption := Value;
    MenuChanged(True);
  end;
end;

procedure TDCPopupMenuItem.SetChecked(Value: Boolean);
begin
  if FChecked <> Value then
  begin
    FChecked := Value;
    if Value and FRadioItem then TurnSiblingsOff;
  end;
end;

procedure TDCPopupMenuItem.SetEnabled(const Value: Boolean);
begin
  if FEnabled <> Value then
  begin
    FEnabled := Value;
    MenuChanged(True)
  end;
end;

procedure TDCPopupMenuItem.SetGroupIndex(const Value: Byte);
begin
  if FGroupIndex <> Value then
  begin
    if Parent <> nil then Parent.VerifyGroupIndex(Parent.IndexOf(Self), Value);
    FGroupIndex := Value;
    if FChecked and FRadioItem then TurnSiblingsOff;
  end;
end;

procedure TDCPopupMenuItem.SetImageIndex(const Value: integer);
begin
  if Value <> FImageIndex then
  begin
    FImageIndex := Value;
    MenuChanged(True);
  end;
end;

procedure TDCPopupMenuItem.SetMenuIndex(Value: Integer);
 var
  Parent: TDCPopupMenuItem;
  Count: Integer;
begin
  if FParent <> nil then
  begin
    Count := FParent.Count;
    if Value < 0 then Value := 0;
    if Value >= Count then Value := Count - 1;
    if Value <> MenuIndex then
    begin
      Parent := FParent;
      Parent.Remove(Self);
      Parent.Insert(Value, Self);
    end;
  end;
end;

procedure TDCPopupMenuItem.SetRadioItem(const Value: Boolean);
begin
  if FRadioItem <> Value then
  begin
    FRadioItem := Value;
    if FChecked and FRadioItem then
      TurnSiblingsOff;
    MenuChanged(True);
  end;
end;

procedure TDCPopupMenuItem.SetVisible(const Value: Boolean);
begin
  if Value <> FVisible then
  begin
    FVisible := Value;
    MenuChanged(True);
  end;
end;

procedure TDCPopupMenuItem.SubItemChanged(Sender: TObject;
  Source: TDCPopupMenuItem; Rebuild: Boolean);
begin
  if Rebuild then RebuildHandle;
  if Parent <> nil then Parent.SubItemChanged(Self, Source, False);
end;

procedure TDCPopupMenuItem.TurnSiblingsOff;
 var
  i: Integer;
  Item: TDCPopupMenuItem;
begin
  if FParent <> nil then
    for i := 0 to FParent.Count - 1 do
    begin
      Item := FParent[i];
      if (Item <> Self) and Item.FRadioItem and (Item.GroupIndex = GroupIndex) then
        Item.SetChecked(False);
    end;
end;

procedure TDCPopupMenuItem.VerifyGroupIndex(Position: Integer;
  Value: Byte);
 var
  i: Integer;
begin
  for i := 0 to GetCount - 1 do
    if i < Position then
      if Items[i].GroupIndex > Value then Exit else
    else
      if Items[i].GroupIndex < Value then Items[i].FGroupIndex := Value;
end;

end.
