{
 BUSINESS CONSULTING
 s a i n t - p e t e r s b u r g

         Components Library for Borland Delphi 4.x, 5.x
         Copyright (c) 1998-2000 Alex'EM

}
unit DCExtCtrls;

interface

uses Messages, Windows, SysUtils, Classes, Controls, Graphics,
     StdCtrls, DCConst;

type
  TDCGradientProgress = class;
  TProgressStyle = (psSingleDirect, psMultiDirect);

  TProgressThread = class(TThread)
  private
    FGradient: TDCGradientProgress;
  protected
    procedure Execute; override;
  public
    constructor Create(AGradient: TDCGradientProgress; CreateSuspended: boolean);
    destructor Destroy; override;
  end;

  TDCGradientProgress = class(TCustomControl)
  private
    FBrushColor: TColor;
    FColor: TColor;
    FDirection: integer;
    FGradientBitmap: TBitmap;
    FInterval: DWORD;
    FLock: TRTLCriticalSection;
    FPosition: integer;
    FStyle: TProgressStyle;
    FStopedEvent: THandle;
    FSuspendEvent: THandle;
    FStep: byte;
    FSyncThread: boolean;
    FTimer: boolean;
    FTimerHandle: THandle;
    procedure CreateGradientBitmap;
    procedure SetPosition(AValue: integer);
    procedure SetBrushColor(const Value: TColor);
    procedure SetColor(const Value: TColor);
    procedure SetInterval(const Value: DWORD);
    function GetActive: boolean;
    procedure SetStyle(const Value: TProgressStyle);
  protected
    procedure CreateParams(var Params: TCreateParams); override;
    procedure DoStep; virtual;
    procedure WMEraseBkGnd(var Message: TWMEraseBkGnd); message WM_ERASEBKGND;
    procedure WMSize(var Message: TWMSize); message WM_SIZE;
  public
    procedure Paint; override;
    constructor Create(AComponent: TComponent); override;
    destructor Destroy; override;
    procedure Resume;
    procedure Suspend;
    property Active: boolean read GetActive;
  published
    property Align;
    property Color: TColor read FColor write SetColor;
    property BrushColor: TColor read FBrushColor write SetBrushColor;
    property Interval: DWORD read FInterval write SetInterval default 10;
    property Position: integer read FPosition write SetPosition;
    property Step: byte read FStep write FStep default 2;
    property Style: TProgressStyle read FStyle write SetStyle;
    property SyncThread: boolean read FSyncThread write FSyncThread default True;
  end;

implementation

uses
  Forms, DCVerInfo;

type

  TCreateWaitableTimerProc = function(lpTimerAttributes: PSecurityAttributes;
    bManualReset: BOOL; lpTimerName: PAnsiChar): THandle; stdcall;
  TSetWaitableTimerProc = function (hTimer: THandle; var lpDueTime: TLargeInteger;
    lPeriod: Longint; pfnCompletionRoutine: TFNTimerAPCRoutine;
    lpArgToCompletionRoutine: Pointer; fResume: BOOL): BOOL; stdcall;
  TCancelWaitableTimerProc = function (hTimer: THandle): BOOL; stdcall;

var
  dllHandle: THandle;
  CancelWaitableTimer: TCancelWaitableTimerProc;
  CreateWaitableTimer: TCreateWaitableTimerProc;
  SetWaitableTimer: TSetWaitableTimerProc;

function WaitableTimerSupported: boolean;
begin
  Result := not (GetWindowsVersion in [pvUnknown, pvWin32s, pvWindows95,
    pvWindows95OSR2]);
end;

{ TDCGradientProgress }

constructor TDCGradientProgress.Create(AComponent: TComponent);
 var
  sa: TSecurityAttributes;
  sd: TSecurityDescriptor;
begin
  inherited;
  InitializeCriticalSection(FLock);
  ControlStyle := [csNoDesignVisible];

  DoubleBuffered := True;
  Height := 5;
  FGradientBitmap := TBitmap.Create;

  FColor := clSelectedLight;
  FBrushColor := clNavy;
  FPosition := 0;
  FDirection := 0;
  FTimer := False;
  FInterval := 10;
  FSyncThread := True;
  FStep := 2;

//  FStyle := psMultiDirect;
  FStyle := psSingleDirect;

  InitializeSecurityDescriptor(@sd, SECURITY_DESCRIPTOR_REVISION);
  SetSecurityDescriptorDacl(@sd, True, nil, False);

  sa.nLength := SizeOf(sa);
  sa.bInheritHandle := True;
  sa.lpSecurityDescriptor := @sd;

  FSuspendEvent := CreateEvent(@sa, True, False, nil);
  FStopedEvent  := CreateEvent(@sa, True, False, nil);

  if Assigned(CreateWaitableTimer) then
    FTimerHandle  := CreateWaitableTimer(@sa, False, nil)
  else
    FTimerHandle  := 0;
end;

procedure TDCGradientProgress.CreateGradientBitmap;
 var
  rv1, bv1, gv1, rv2, bv2, gv2: byte;
  Color1, Color2: longint;
  i, r: integer;
  hWidth: integer;
  gPos: extended;
  lActive: boolean;
begin
  if (Width > 0) and (Height > 0) and (FLock.RecursionCount = 0) then
  with FGradientBitmap do
  begin
    lActive := Active;
    Suspend;
    EnterCriticalSection(FLock);
    try
      case FStyle of
        psMultiDirect:
          Width  := Self.Width shl 1;
        psSingleDirect:
          Width  := Self.Width shl 1;
      end;
      Height := Self.Height;

      Color1 := ColorToRGB(FBrushColor);
      Color2 := ColorToRGB(FColor);

      rv1 := GetRValue(Color1);
      gv1 := GetGValue(Color1);
      bv1 := GetBValue(Color1);
      rv2 := GetRValue(Color2);
      gv2 := GetGValue(Color2);
      bv2 := GetBValue(Color2);

      case FStyle of
        psSingleDirect:
          begin
            hWidth := Width shr 1;
            for i := 1 to hWidth do
            begin
              gPos := SQR(i / hWidth);
              Canvas.Pen.Color := RGB(Trunc(rv2 + (rv1 - rv2) * gPos),
                                      Trunc(gv2 + (gv1 - gv2) * gPos),
                                      Trunc(bv2 + (bv1 - bv2) * gPos));
              Canvas.MoveTo(i - 1, 0);
              Canvas.LineTo(i - 1, Height);
            end;

            for i := 0 to hWidth - 1 do
            begin
              gPos := SQR(1 - i / hWidth);
              Canvas.Pen.Color := RGB(Trunc(rv1 + (rv2 - rv1) * (1 - gPos)),
                                      Trunc(gv1 + (gv2 - gv1) * (1 - gPos)),
                                      Trunc(bv1 + (bv2 - bv1) * (1 - gPos)));
              Canvas.MoveTo(i + hWidth, 0);
              Canvas.LineTo(i + hWidth, Height);
            end;
          end;
        psMultiDirect:
          begin
            r := 0;
            hWidth := Self.Width shr 1;

            for i := 0 to hWidth do
            begin
              gPos := {SQR}(i / hWidth);
              Canvas.Pen.Color := RGB(Trunc(rv2 + (rv1 - rv2) * gPos),
                                      Trunc(gv2 + (gv1 - gv2) * gPos),
                                      Trunc(bv2 + (bv1 - bv2) * gPos));
              Canvas.MoveTo(i, 0);
              Canvas.LineTo(i, Height);
            end;
            Inc(r, hWidth);

            Canvas.Brush.Color := Color2;
            Canvas.FillRect(Rect(r, 0, Self.Width + r, Height));
            Inc(r, Self.Width);

            for i := 0 to hWidth do
            begin
              gPos := {SQRT}(i / hWidth);
              Canvas.Pen.Color := RGB(Trunc(rv1 + (rv2 - rv1) * gPos),
                                      Trunc(gv1 + (gv2 - gv1) * gPos),
                                      Trunc(bv1 + (bv2 - bv1) * gPos));
              Canvas.MoveTo(r + i, 0);
              Canvas.LineTo(r + i, Height);
            end;
         end;
      end;
    finally
      LeaveCriticalSection(FLock);
    end;
    if lActive then Resume;
  end;
end;

procedure TDCGradientProgress.CreateParams(var Params: TCreateParams);
begin
  inherited;
end;

destructor TDCGradientProgress.Destroy;
begin
  Suspend;
  CloseHandle(FSuspendEvent);
  CloseHandle(FStopedEvent);
  if FTimerHandle <> 0 then CloseHandle(FTimerHandle);
  DeleteCriticalSection(FLock);
  FGradientBitmap.Free;
  inherited;
end;

procedure TDCGradientProgress.DoStep;
begin
  SetPosition(Position + FStep)
end;

function TDCGradientProgress.GetActive: boolean;
begin
  Result := FTimer;
end;

procedure TDCGradientProgress.Paint;
 var
  i, r: integer;
  ARect, BRect, CRect: TRect;
  BMPWidth, BMPHeight: integer;
begin
  if HandleAllocated and (FLock.RecursionCount = 0) then
  begin
    EnterCriticalSection(FLock);
    try
      BRect := BoundsRect;
      OffsetRect(BRect, -BRect.Left, -BRect.Top);

      BMPWidth  := FGradientBitmap.Width;
      BMPHeight := FGradientBitmap.Height;

      case FStyle of
        psSingleDirect:
          begin
            if FDirection < 0 then
            begin
              if FPosition = BMPWidth then
              begin
                FPosition := 0;
              end;
              i := 0;
              ARect := Rect(FPosition, 0, BMPWidth, BMPHeight);

              while (ARect.Right - ARect.Left + i) <= BRect.Right do
              begin
                CRect := ARect;
                OffsetRect(CRect, -CRect.Left + i, -CRect.Top);

                Canvas.CopyRect(CRect, FGradientBitmap.Canvas, ARect);

                Inc(i, ARect.Right - ARect.Left);
                ARect := Rect(0, 0, BMPWidth, BMPHeight);
              end;

              if (ARect.Right + i) > BRect.Right then ARect.Right := BRect.Right - i;

              CRect := ARect;
              OffsetRect(CRect, -CRect.Left + i, -CRect.Top);

              Canvas.CopyRect(CRect,  FGradientBitmap.Canvas, ARect);
            end
            else begin
              if FPosition = BMPWidth then
              begin
                FPosition := 0;
              end;
              ARect := Rect(0, 0, BMPWidth - FPosition, BMPHeight);
              i := BRect.Right;
              while i >= 0 do
              begin
                Dec(i, ARect.Right - ARect.Left);

                CRect := ARect;
                OffsetRect(CRect, -CRect.Left + i, -CRect.Top);

                Canvas.CopyRect(CRect, FGradientBitmap.Canvas, ARect);
                ARect := Rect(0, 0, BMPWidth, BMPHeight);
              end;

              if i < 0 then ARect.Left := -i;
              CRect := ARect;
              OffsetRect(CRect, -CRect.Left, -CRect.Top);

              Canvas.CopyRect(CRect,  FGradientBitmap.Canvas, ARect);
            end;
          end;
        psMultiDirect:
          begin
            r := Self.Width shr 1;
            if FPosition = Self.Width + r then
            begin
              FPosition := 0;
              if FDirection < 0 then
                FDirection := 0
              else
                FDirection := -1
            end;
            if FDirection <  0 then
            begin
              i := FPosition - r;
              if i < 0 then
              begin
                ARect := Rect(r, 0, r + Width - FPosition, BMPHeight);
                CRect := ARect;
                OffsetRect(CRect, -CRect.Left, -CRect.Top);
                Canvas.CopyRect(CRect, FGradientBitmap.Canvas, ARect);

                ARect := Rect(r + Width, 0, r + Width + FPosition, BMPHeight);
                CRect := ARect;
                OffsetRect(CRect, -CRect.Left + Width - FPosition, -CRect.Top);
                Canvas.CopyRect(CRect, FGradientBitmap.Canvas, ARect);
              end
              else begin
                ARect := Rect(r, 0, Self.Width - i, BMPHeight);
                CRect := ARect;
                OffsetRect(CRect, -CRect.Left, -CRect.Top);
                Canvas.CopyRect(CRect, FGradientBitmap.Canvas, ARect);

                ARect := Rect(Width + r, 0, BMPWidth, BMPHeight);
                CRect := ARect;
                OffsetRect(CRect, -CRect.Left + Width - FPosition, -CRect.Top);
                Canvas.CopyRect(CRect, FGradientBitmap.Canvas, ARect);

                ARect := Rect(r, 0, FPosition, BMPHeight);
                CRect := ARect;
                OffsetRect(CRect, -CRect.Left + Width - i, -CRect.Top);
                Canvas.CopyRect(CRect, FGradientBitmap.Canvas, ARect);
              end;
            end
            else begin
              i := r - FPosition;
              if i > 0 then
              begin
                ARect := Rect(i, 0, i + Self.Width, FGradientBitmap.Height);
                CRect := ARect;
                OffsetRect(CRect, -CRect.Left, -CRect.Top);
                Canvas.CopyRect(CRect, FGradientBitmap.Canvas, ARect);
              end
              else begin
                ARect := Rect(r, 0, r - i, FGradientBitmap.Height);
                CRect := ARect;
                OffsetRect(CRect, -CRect.Left, -CRect.Top);
                Canvas.CopyRect(CRect, FGradientBitmap.Canvas, ARect);

                ARect := Rect(0, 0, i + Self.Width, FGradientBitmap.Height);
                CRect := ARect;
                OffsetRect(CRect, -CRect.Left - i, -CRect.Top);
                Canvas.CopyRect(CRect, FGradientBitmap.Canvas, ARect);
              end;
            end
          end;
      end;
    finally
      LeaveCriticalSection(FLock);
    end;
  end;
end;

procedure TDCGradientProgress.Resume;
 var
  lpDueTime: TLargeInteger;
begin
  if not FTimer and (Width > 0) and (Height > 0) then
  begin
    ResetEvent(FStopedEvent);
    ResetEvent(FSuspendEvent);
    TProgressThread.Create(Self, False);

    lpDueTime := -1;

    if (FTimerHandle <> 0) and Assigned(SetWaitableTimer) then
      SetWaitableTimer(FTimerHandle, lpDueTime, FInterval, nil, nil, False);

    FTimer := True;
  end;
end;

procedure TDCGradientProgress.SetBrushColor(const Value: TColor);
begin
  if FBrushColor <> Value then
  begin
    FBrushColor := Value;
    CreateGradientBitmap;
    invalidate;
  end;
end;

procedure TDCGradientProgress.SetColor(const Value: TColor);
begin
 if FColor <> Value then
  begin
    FColor := Value;
    CreateGradientBitmap;
    invalidate;
  end;
end;

procedure TDCGradientProgress.SetInterval(const Value: DWORD);
begin
  if FInterval <> Value then
  begin
    if Active then
    begin
      Suspend;
      FInterval := Value;
      Resume;
    end
    else
      FInterval := Value;
  end;
end;

procedure TDCGradientProgress.SetPosition(AValue: integer);
begin
  FPosition := AValue;
  Paint;
end;

procedure TDCGradientProgress.SetStyle(const Value: TProgressStyle);
begin
  if FStyle <> Value then
  begin
    FStyle := Value;
    CreateGradientBitmap;
    invalidate;
  end;
end;

procedure TDCGradientProgress.Suspend;
begin
  if FTimer then
  begin
    SetEvent(FSuspendEvent);
    while WaitForSingleObject(FStopedEvent, FInterval) = WAIT_TIMEOUT do
      Application.ProcessMessages;
    FTimer := False;
    if (FTimerHandle <> 0) and Assigned(CancelWaitableTimer) then
      CancelWaitableTimer(FTimerHandle);
  end;
end;

procedure TDCGradientProgress.WMEraseBkGnd(var Message: TWMEraseBkGnd);
begin
  Message.Result := 0;
end;

procedure TDCGradientProgress.WMSize(var Message: TWMSize);
begin
  CreateGradientBitmap;
  inherited;
end;

{ TProgressThread }

constructor TProgressThread.Create(AGradient: TDCGradientProgress;
 CreateSuspended: boolean);
begin
  inherited Create(CreateSuspended);
  FreeOnTerminate := True;
  FGradient := AGradient;
  Priority := tpHighest;
end;

destructor TProgressThread.Destroy;
begin
  inherited;
end;

procedure TProgressThread.Execute;
 var
  lResult, ObjectCouns, AInterval: DWORD;
  lpHandles: array[0..1] of THandle;

 function ThreadClosed: Boolean;
 begin
   Result := Terminated or (Application = nil) or Application.Terminated;
 end;

 procedure TryStepGradient;
 begin
   if not ThreadClosed then
     with FGradient do
     begin
       if FSyncThread then Synchronize(DoStep) else DoStep;
     end;
 end;

begin
  lpHandles[0] := FGradient.FSuspendEvent;
  lpHandles[1] := FGradient.FTimerHandle;

  if WaitableTimerSupported then
  begin
    ObjectCouns := Length(lpHandles);
    AInterval := INFINITE;
  end
  else begin
    ObjectCouns := 1;
    AInterval := FGradient.Interval
  end;

  while not ThreadClosed do with FGradient do
  begin
    lResult := WaitForMultipleObjects(ObjectCouns, @lpHandles, False, AInterval);
    case lResult of
      WAIT_OBJECT_0 + 0:             // FSuspendEvent
        Terminate;
      WAIT_OBJECT_0 + 1:             // FTimerHandle
        TryStepGradient;
      WAIT_TIMEOUT:
        if AInterval = INFINITE then Terminate else TryStepGradient
      else
        Terminate;
    end
  end;
  SetEvent(FGradient.FStopedEvent);
end;

procedure LoadExtProcs;
begin
  dllHandle := LoadLibrary(PChar(kernel32));
  if dllHandle > HINSTANCE_ERROR then
  begin
    @SetWaitableTimer := GetProcAddress(dllHandle, 'SetWaitableTimer');
    @CreateWaitableTimer := GetProcAddress(dllHandle, 'CreateWaitableTimerA');
    @CancelWaitableTimer := GetProcAddress(dllHandle, 'CancelWaitableTimer');
  end;
end;

initialization
  LoadExtProcs;

finalization
  if dllHandle > HINSTANCE_ERROR then FreeLibrary(dllHandle);

end.
