{
 BUSINESS CONSULTING
 s a i n t - p e t e r s b u r g

         Components Library for Borland Delphi 4.x - 6.x
         Copyright (c) 1998-2001 Alex'EM

}
unit DCDataGrid;

interface
{$I DCConst.inc}

uses
  Windows, SysUtils, Classes, Dialogs, Graphics, Grids, Controls, ImgList,
  {$IFDEF DELPHI_V6}
    Variants,
  {$ENDIF}
  DCKnots, DCGrids, DCConst;

type
  TDataContainer = class;
  TDataFields = class;
  TDataField = class;
  TDCCustomDataGrid = class;
  TDataFieldClass = class of TDataField;

  TDataFieldType = (dtUnknown, dtString, dtInteger, dtWord, dtBoolean, dtFloat,
    dtDate, dtTime, dtDateTime, dtBytes, dtAutoInc);

  TDataFieldGetText = procedure(Sender: TDataField; var Text: string) of object;

  TDataField = class(TComponent)
  private
    FDataType: TDataFieldType;
    FDisplayFormat: string;
    FFieldName: string;
    FFields: TDataFields;
    FIndex: integer;
    FOnGetText: TDataFieldGetText;
    FOptions: TEditOptions;
    FSize: integer;
    procedure SetFieldName(const Value: string);
    function GetIndex: integer;
    procedure SetDataSize(const Value: integer);
    procedure SetDataType(Value: TDataFieldType);
    function GetDataContainer: TDataContainer;
    procedure SetDataContainer(const Value: TDataContainer);
    procedure SetOptions(const Value: TEditOptions);
  protected
    function DataAssigned: boolean;
    function GetAsBoolean: Boolean; virtual;
    function GetAsDateTime: TDateTime; virtual;
    function GetAsFloat: Extended; virtual;
    function GetAsInteger: Longint; virtual;
    function GetAsString: string; virtual;
    function GetAsVariant: Variant; virtual;
    function GetData(Buffer: Pointer): boolean;
    function GetDataSize: integer; virtual;
    function GetDisplayText: string;
    procedure GetText(var Text: string); virtual;
    procedure ReadState(Reader: TReader); override;
    procedure SetAsBoolean(const Value: Boolean); virtual;
    procedure SetAsDateTime(const Value: TDateTime); virtual;
    procedure SetAsFloat(const Value: Extended); virtual;
    procedure SetAsInteger(const Value: Longint); virtual;
    procedure SetAsString(const Value: string); virtual;
    procedure SetAsVariant(const Value: Variant); virtual;
    procedure SetData(Buffer: Pointer); virtual;
    procedure SetDisplayFormat(const Value: string);
    procedure SetParentComponent(AParent: TComponent); override;
    procedure PropertyChanged;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Clear;
    function GetParentComponent: TComponent; override;
    function HasParent: Boolean; override;
    property AsBoolean: Boolean read GetAsBoolean write SetAsBoolean;
    property AsDateTime: TDateTime read GetAsDateTime write SetAsDateTime;
    property AsFloat: Extended read GetAsFloat write SetAsFloat;
    property AsInteger: Longint read GetAsInteger write SetAsInteger;
    property AsString: string read GetAsString write SetAsString;
    property AsVariant: Variant read GetAsVariant write SetAsVariant;
    property DataContainer: TDataContainer read GetDataContainer write SetDataContainer;
    property DataType: TDataFieldType read FDataType;
    property DisplayText: string read GetDisplayText;
    property Fields: TDataFields read FFields;
    property Size: integer read GetDataSize write SetDataSize;
  published
    property DisplayFormat: string read FDisplayFormat write SetDisplayFormat;
    property FieldName: string read FFieldName write SetFieldName;
    property Index: integer read GetIndex;
    property OnGetText: TDataFieldGetText read FOnGetText write FOnGetText;
    property Options: TEditOptions read FOptions write SetOptions;
  end;

  TDataFields = class(TObject)
  private
    FList: TList;
    FDataContainer: TDataContainer;
    FUpdateCount: integer;
    FChanged: boolean;
    FIndex: integer;
    FDeleted: boolean;
  protected
    procedure Changed; virtual;
    function GetCount: Integer;
    function GetField(Index: Integer): TDataField;
    procedure SetField(Index: Integer; Value: TDataField);
  public
    procedure Add(Field: TDataField);
    procedure BeginUpdate;
    procedure Clear;
    constructor Create(ADataContainer: TDataContainer);
    destructor Destroy; override;
    procedure EndUpdate;
    function FindField(const FieldName: string): TDataField;
    function FieldByName(const FieldName: string): TDataField;
    procedure GetFieldNames(List: TStrings);
    function IndexOf(Field: TDataField): Integer;
    procedure Remove(Field: TDataField);
    property Count: Integer read GetCount;
    property Fields[Index: Integer]: TDataField read GetField write SetField; default;
    property DataContainer: TDataContainer read FDataContainer;
  end;

  TStringDataField = class(TDataField)
  protected
    function GetAsString: string; override;
    procedure GetText(var Text: string); override;
    procedure SetAsString(const Value: string); override;
  public
    constructor Create(AOwner: TComponent); override;
  end;

  TNumericDataField = class(TDataField)
  private
    FDigitsCount: integer;
    procedure SetDigitsCount(const Value: integer);
  public
    constructor Create(AOwner: TComponent); override;
  published
    property DigitsCount: integer read FDigitsCount write SetDigitsCount default -1;
  end;

  TIntegerDataField = class(TNumericDataField)
  protected
    function GetAsFloat: Extended; override;
    function GetAsInteger: Longint; override;
    function GetAsString: string; override;
    function GetAsVariant: Variant; override;
    function GetDataSize: Integer; override;
    procedure GetText(var Text: string); override;
    function GetValue(var Value: Integer): Boolean;
    procedure SetAsFloat(const Value: Extended); override;
    procedure SetAsInteger(const Value: Longint); override;
    procedure SetAsString(const Value: string); override;
    procedure SetAsVariant(const Value: Variant); override;
  public
    constructor Create(AOwner: TComponent); override;
  end;

  TFloatDataField = class(TNumericDataField)
  private
    FCurrency: boolean;
    FPrecision: integer;
    procedure SetCurrency(const Value: Boolean);
    procedure SetPrecision(const Value: Integer);
  protected
    function GetAsFloat: Extended; override;
    function GetAsInteger: Longint; override;
    function GetAsString: string; override;
    function GetAsVariant: Variant; override;
    function GetDataSize: Integer; override;
    procedure GetText(var Text: string); override;
    procedure SetAsFloat(const Value: Extended); override;
    procedure SetAsInteger(const Value: Longint); override;
    procedure SetAsString(const Value: string); override;
    procedure SetAsVariant(const Value: Variant); override;
  public
    constructor Create(AOwner: TComponent); override;
  published
    property Currency: Boolean read FCurrency write SetCurrency default False;
    property Precision: Integer read FPrecision write SetPrecision default -1;
  end;

  TDisplayItem = class(TCollectionItem)
  private
    FText: string;
    FImageIndex: integer;
    procedure SetImageIndex(const Value: integer);
    procedure SetText(const Value: string);
  protected
    property ImageIndex: integer read FImageIndex write SetImageIndex default -1;
    property Text: string read FText write SetText;
  public
    constructor Create(Collection: TCollection); override;
  end;

  TBooleanDisplayItem = class(TDisplayItem)
  published
    property ImageIndex;
    property Text;
  end;

  TDisplayOption = (dvDropDown);
  TDisplayOptions = set of TDisplayOption;
  TDisplayStyle = (dsText, dsImage, dsValue);

  TDisplayValues = class(TCollection)
  private
    FDataField: TDataField;
    FImages: TImageList;
    FImageChangeLink: TChangeLink;
    FOptions: TDisplayOptions;
    FStyle: TDisplayStyle;
    procedure ImageListChange(Sender: TObject);
    procedure SetOptions(const Value: TDisplayOptions);
    procedure SetStyle(const Value: TDisplayStyle);
    procedure SetImages(const Value: TImageList);
  protected
    function  GetOwner: TPersistent; override;
    property Images: TImageList read FImages write SetImages;
    property Options: TDisplayOptions read FOptions write SetOptions
      default [];
    property Style: TDisplayStyle read FStyle write SetStyle default dsValue;
  public
    function CanAdd: boolean; virtual;
    constructor Create(ItemClass: TCollectionItemClass; ADataField: TDataField);
    destructor Destroy; override;
    procedure Changed;
  end;

  TBooleanValues = class(TDisplayValues)
  private
    function GetItem(Index: Boolean): TBooleanDisplayItem;
  public
    function CanAdd: boolean; override;
    constructor Create(ADataField: TDataField);
    property Items[Index: Boolean]: TBooleanDisplayItem read GetItem;
  published
    property Images;
    property Options;
    property Style default dsImage;
  end;

  TBooleanDataField = class(TDataField)
  private
    FValues: TBooleanValues;
    procedure SetValues(const Value: TBooleanValues);
  protected
    function GetAsBoolean: Boolean; override;
    function GetAsString: string; override;
    function GetAsVariant: Variant; override;
    function GetDataSize: Integer; override;
    procedure GetText(var Text: string); override;
    procedure SetAsBoolean(const Value: Boolean); override;
    procedure SetAsString(const Value: string); override;
    procedure SetAsVariant(const Value: Variant); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    property Values: TBooleanValues read FValues write SetValues;
  end;

  TWordDataField = class(TIntegerDataField)
  protected
    function GetDataSize: Integer; override;
  public
    constructor Create(AOwner: TComponent); override;
  end;

  TDateTimeDataField = class(TDataField)
  protected
    function GetAsDateTime: TDateTime; override;
    function GetAsFloat: Extended; override;
    function GetAsString: string; override;
    function GetAsVariant: Variant; override;
    function GetDataSize: Integer; override;
    procedure GetText(var Text: string); override;
    function GetValue(var Value: TDateTime): boolean;
    procedure SetAsDateTime(const Value: TDateTime); override;
    procedure SetAsFloat(const Value: Extended); override;
    procedure SetAsString(const Value: string); override;
    procedure SetAsVariant(const Value: Variant); override;
  public
    constructor Create(AOwner: TComponent); override;
  end;

  TDateDataField = class(TDateTimeDataField)
  protected
    procedure GetText(var Text: string); override;
    procedure SetAsString(const Value: string); override;
  public
    constructor Create(AOwner: TComponent); override;
  end;

  TTimeDataField = class(TDateTimeDataField)
  protected
    procedure GetText(var Text: string); override;
    procedure SetAsString(const Value: string); override;
  public
    constructor Create(AOwner: TComponent); override;
  end;

  TBytesDataField = class(TStringDataField)
    {}
  end;

  TAutoIncDataField = class(TIntegerDataField)
    {}
  end;

  PDataValues_tag = ^TDataValues;
  TDataValues = packed array[0..0] of Pointer;

  TDataContainer = class(TObject)
  private
    FCapacity: integer;
    FDataGrid: TDCCustomDataGrid;
    FActiveItem: TKnotItem;
    FFields: TDataFields;
    FLockEvent: THandle;
    FLockCount: integer;
    procedure SetCapacity(const Value: integer);
    procedure SetActiveItem(const Value: TKnotItem);
  protected
    procedure SetDataSize(const Value: integer; var pData: Pointer);
    procedure UpdateDataSize(const Value: integer); virtual;
    procedure UpdateGridColumns;
    procedure ClearFieldData(Field: TDataField);
  public
    constructor Create(ADataGrid: TDCCustomDataGrid);
    function CreateData: Pointer;
    procedure ClearKnotData(KnotItem: TKnotItem);
    function DataAssigned: boolean;
    destructor Destroy; override;
    procedure Lock(AItem: TKnotItem);
    procedure Unlock;
    property Capacity: integer read FCapacity write SetCapacity;
    property ActiveItem: TKnotItem read FActiveItem write SetActiveItem;
    property Fields: TDataFields read FFields;
    property DataGrid: TDCCustomDataGrid read FDataGrid;
  end;

  TDataGridColumnsState = (dgColumnsDefault, dgColumnsCustomized);

  TDataColumn = class(TKnotColumn)
  private
    FDataField: TDataField;
    FFieldName: string;
    procedure SetFieldName(const Value: string);
    procedure SetDataField(const Value: TDataField);
    function GetGrid: TDCCustomDataGrid;
  protected
    function UpdateDataField: boolean;
  public
    property DataField: TDataField read FDataField write SetDataField;
    property Grid: TDCCustomDataGrid read GetGrid;
  published
    property FieldName: string read FFieldName write SetFieldName;
  end;

  TDataColumns = class(TKnotColumns)
  private
    function GetItem(Index: Integer): TDataColumn;
    procedure SetItem(Index: Integer; const Value: TDataColumn);
  public
    function Add: TDataColumn;
    property Items[Index: Integer]: TDataColumn read GetItem write SetItem; default;
  end;

  TDataKnotItems = class;

  TDataKnotItem = class(TKnotItem)
  private
    function GetGrid: TDCCustomDataGrid;
    function GetField(Index: Integer): TDataField;
    procedure SetField(Index: Integer; const Value: TDataField);
  public
    constructor Create(AOwner: TKnotItems; AParent: TKnotItem; AName: string); override;
    destructor Destroy; override;
    function FieldByName(const FieldName: string): TDataField;
    property Fields[Index: Integer]: TDataField read GetField write SetField; default;
    property Grid: TDCCustomDataGrid read GetGrid;
  end;

  TDataKnotItems = class(TKnotItems)
  private
    function GetItem(Index: integer): TDataKnotItem;
    procedure SetItem(Index: integer; const Value: TDataKnotItem);
  public
    function Add(Name: string; Position: integer = KN_ITEMONBOTTOM): TDataKnotItem;
    property Items[Index: integer]: TDataKnotItem read GetItem write SetItem;
  end;

  TDrawDataCellEvent = procedure (Sender: TObject; const Rect: TRect;
    Canvas: TCanvas; Column: TDataColumn; KnotItem: TDataKnotItem;
    State: TGridCellState; var Default: boolean) of object;

  TDCCustomDataGrid = class(TDCCustomTreeGrid)
  private
    FDataContainer: TDataContainer;
    FOnDrawDataCell: TDrawDataCellEvent;
    function GetFiedls: TDataFields;
    function GetColumns: TDataColumns;
    procedure SetColumns(const Value: TDataColumns);
    function GetKnots: TDataKnotItems;
    procedure SetKnots(const Value: TDataKnotItems);
  protected
    function CreateDataContainer: TDataContainer; virtual;
    procedure DefaultDrawDataCell(Canvas: TCanvas; ARect: TRect; ACol: integer;
      AColumn: TKnotColumn; AKnot: TDataKnotItem; AState: TGridCellState); virtual;
    procedure DoDrawColumnCell(Canvas: TCanvas; ARect: TRect; ACol: integer;
      AColumn: TKnotColumn; AKnot: TKnotItem; AState: TGridCellState); override;
    procedure GetChildren(Proc: TGetChildProc; Root: TComponent); override;
    function Initialize: TDCTreeGridInitialize; override;
    property OnDrawDataCell: TDrawDataCellEvent read FOnDrawDataCell write FOnDrawDataCell;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure DrawTextEx(Canvas: TCanvas; AColumn: TKnotColumn; R: TRect;
      Text: string; Images: TImageList; Alignment: TAlignment); virtual;
    procedure GetFieldNames(AList: TStrings);
    property Columns: TDataColumns read GetColumns write SetColumns;
    property DataContainer: TDataContainer read FDataContainer;
    property DataFields: TDataFields read GetFiedls;
    property Knots: TDataKnotItems read GetKnots write SetKnots;
  end;

  TDCDataGrid = class(TDCCustomDataGrid)
  public
    property Knots;
    property SelectedArea;
  published
    property Columns;
    property DataFields;
  end;

const
  DataFieldClasses: array[TDataFieldType] of TDataFieldClass = (
    nil,                    { dtUnknown }
    TStringDataField,       { dtString }
    TIntegerDataField,      { dtInteger }
    TWordDataField,         { dtWord }
    TBooleanDataField,      { dtBoolean }
    TFloatDataField,        { dtFloat }
    TDateDataField,         { dtDate }
    TTimeDataField,         { dtTime }
    TDateTimeDataField,     { dtDateTime }
    TBytesDataField,        { dtBytes }
    TAutoIncDataField       { dtAutoInc }
  );

  DataFieldTypeNames: array[TDataFieldType] of string = (
    'Unknown', 'String', 'Integer', 'Word', 'Boolean', 'Float',  'Date',
    'Time', 'DateTime', 'Bytes', 'AutoInc');

implementation
uses
  DCEditTools;

{ TDataFields }

procedure TDataFields.Add(Field: TDataField);
begin
  FList.Add(Field);
  Field.FFields := Self;
  Field.Findex  := FIndex;
  if FUpdateCount = 0 then
    Changed
  else
    FChanged := True;
end;

procedure TDataFields.BeginUpdate;
begin
  Inc(FUpdateCount);
end;

procedure TDataFields.Changed;
begin
  DataContainer.SetCapacity(Count);
  DataContainer.UpdateGridColumns;
  FIndex := Count;
  FDeleted := False;
  FChanged := False;
end;

procedure TDataFields.Clear;
 var
  KnotItem: TKnotItem;
  F: TDataField;
begin
  BeginUpdate;
  with DataContainer do
  begin
    if not(csDestroying in DataGrid.ComponentState) then
    begin
      KnotItem := DataGrid.Knots.First;
      while KnotItem <> nil do
      begin
        ClearKnotData(KnotItem);
        KnotItem := KnotItem.GetNext;
      end;
      FCapacity := 0;
    end;
  end;
  while FList.Count > 0 do
  begin
    F := FList.Last;
    F.FFields := nil;
    F.Free;
    FList.Delete(FList.Count-1);
  end;
  FChanged := True;
  FDeleted := True;
  EndUpdate;
end;

constructor TDataFields.Create(ADataContainer: TDataContainer);
begin
  inherited Create;
  FList := TList.Create;
  FDataContainer := ADataContainer;
  FUpdateCount := 0;
  FChanged := False;
  FDeleted := False;
end;

destructor TDataFields.Destroy;
begin
  if FList <> nil then Clear;
  FList.Free;
  inherited;
end;

procedure TDataFields.EndUpdate;
begin
  if FUpdateCount > 0 then
  begin
    Dec(FUpdateCount);
    if (FUpdateCount = 0) and FChanged then
    begin
      Changed;
    end
  end;
end;

function TDataFields.FieldByName(const FieldName: string): TDataField;
begin
  Result := FindField(FieldName);
end;

function TDataFields.FindField(const FieldName: string): TDataField;
var
  I: Integer;
begin
  for I := 0 to FList.Count - 1 do
  begin
    Result := FList.Items[I];
    if AnsiCompareText(Result.FFieldName, FieldName) = 0 then Exit;
  end;
  Result := nil;
end;

function TDataFields.GetCount: Integer;
begin
  Result := FList.Count;
end;

function TDataFields.GetField(Index: Integer): TDataField;
begin
  Result := FList[Index];
end;

procedure TDataFields.GetFieldNames(List: TStrings);
var
  I: Integer;
begin
  List.BeginUpdate;
  try
    List.Clear;
    for I := 0 to FList.Count - 1 do
      List.Add(TDataField(FList.Items[I]).FieldName)
  finally
    List.EndUpdate;
  end;
end;

function TDataFields.IndexOf(Field: TDataField): Integer;
begin
  Result := FList.IndexOf(Field);
end;

procedure TDataFields.Remove(Field: TDataField);
begin
  FList.Remove(Field);
  Field.FFields := nil;
  if FUpdateCount = 0 then
    Changed
  else begin
    FChanged := True;
    FDeleted := True;
  end;
  DataContainer.ClearFieldData(Field);
end;

procedure TDataFields.SetField(Index: Integer; Value: TDataField);
begin
  Fields[Index].Assign(Value);
end;

{ TDataField }

constructor TDataField.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  SetDataType(dtUnknown);
  FIndex := -1;
  FSize  := 0;
end;

function TDataField.GetIndex: integer;
begin
  Result := FIndex;
end;

function TDataField.GetDataSize: integer;
begin
  Result := FSize;
end;

procedure TDataField.SetFieldName(const Value: string);
begin
  FFieldName := Value;
end;

function TDataField.GetData(Buffer: Pointer): Boolean;
 var
  pValue: Pointer;
begin
  if DataAssigned and (Buffer <> nil) then
  begin
    pValue := TDataValues(DataContainer.ActiveItem.Data^)[Index];
    if pValue <> nil then
    begin
      Move(pValue^, Buffer^, Size);
      Result := True;
    end
    else
      Result := False;
  end
  else
    Result := False;
end;

procedure TDataField.SetData(Buffer: Pointer);
 var
  pValue, Data:  Pointer;
  ASize: integer;
begin
  if DataAssigned then
  begin
    ASize := Size;
    Data := DataContainer.ActiveItem.Data;
    if Buffer <> nil then
    begin
      pValue := TDataValues(Data^)[Index];
      ReallocMem(pValue, ASize);
      TDataValues(Data^)[Index] := pValue;
      Move(Buffer^, pValue^, ASize);
    end
    else begin
      pValue := TDataValues(Data^)[Index];
      if Assigned(pValue) then
      begin
        FreeMem(pValue, ASize);
        FillChar(TDataValues(Data^)[Index], SizeOf(Pointer), 0);
      end;
    end;
  end
end;

function TDataField.DataAssigned: boolean;
begin
  Result := (Index > -1) and DataContainer.DataAssigned;
end;

procedure TDataField.Clear;
begin
  SetData(nil);
end;

procedure TDataField.PropertyChanged;
 var
  DataContainer: TDataContainer;
begin
  if Fields <> nil then
  begin
    DataContainer := Fields.DataContainer;
    if (DataContainer <> nil) then DataContainer.DataGrid.Invalidate;
  end;
end;

procedure TDataField.GetText(var Text: string);
begin
  Text := GetAsString;
end;

function TDataField.GetAsBoolean: Boolean;
var
  S: string;
begin
  S := GetAsString;
  Result := (Length(S) > 0) and (S[1] in ['T', 't', 'Y', 'y']);
end;

function TDataField.GetAsDateTime: TDateTime;
begin
  Result := StrToDateTime(GetAsString);
end;

function TDataField.GetAsFloat: Extended;
begin
  Result := StrToFloat(GetAsString);
end;

function TDataField.GetAsInteger: Longint;
begin
  Result := StrToInt(GetAsString);
end;

function TDataField.GetAsString: string;
begin
  SetLength(Result, Size);
  GetData(PChar(Result));
  SetLength(Result, StrLen(PChar(Result)));
end;

function TDataField.GetAsVariant: Variant;
begin
  Result := GetAsString;
end;

procedure TDataField.SetAsBoolean(const Value: Boolean);
const
  Values: array[Boolean] of string[1] = ('F', 'T');
begin
  SetAsString(Values[Value]);
end;

procedure TDataField.SetAsDateTime(const Value: TDateTime);
begin
  SetAsString(DateTimeToStr(Value));
end;

procedure TDataField.SetAsFloat(const Value: Extended);
begin
  SetAsString(FloatToStr(Value));
end;

procedure TDataField.SetAsInteger(const Value: Longint);
begin
  SetAsBoolean(Boolean(Value));
end;

procedure TDataField.SetAsString(const Value: string);
begin
  if Value = '' then
    Clear
  else
    SetData(PChar(Value));
end;

procedure TDataField.SetAsVariant(const Value: Variant);
begin
  SetAsString(Value);
end;

procedure TDataField.SetDataSize(const Value: integer);
begin
  FSize := Value;
end;

function TDataField.GetDisplayText: string;
begin
  Result := '';
  if Assigned(FOnGetText) then
    FOnGetText(Self, Result)
  else
    GetText(Result)
end;

procedure TDataField.SetDataType(Value: TDataFieldType);
begin
  FDataType := Value; 
end;

function TDataField.GetDataContainer: TDataContainer;
begin
  if FFields <> nil then
    Result := FFields.DataContainer
  else
    Result := nil;
end;

procedure TDataField.SetDataContainer(const Value: TDataContainer);
begin
  if Value <> DataContainer then
  begin
    if DataContainer <> nil then DataContainer.FFields.Remove(Self);
    if Value <> nil then Value.FFields.Add(Self);
  end;
end;

function TDataField.GetParentComponent: TComponent;
begin
  if DataContainer <> nil then
    Result := DataContainer.DataGrid
  else
    Result := nil;
end;

function TDataField.HasParent: Boolean;
begin
  HasParent := True;
end;

procedure TDataField.SetParentComponent(AParent: TComponent);
begin
  if not (csLoading in ComponentState) then
    DataContainer := TDCCustomDataGrid(AParent).DataContainer;
end;

procedure TDataField.ReadState(Reader: TReader);
begin
  inherited ReadState(Reader);
  if Reader.Parent is TDCCustomDataGrid then
    DataContainer := TDCCustomDataGrid(Reader.Parent).DataContainer;
end;

destructor TDataField.Destroy;
begin
  if DataContainer <> nil then FFields.Remove(Self);
  inherited;
end;

procedure TDataField.SetDisplayFormat(const Value: string);
begin
  FDisplayFormat := Value;
  PropertyChanged;
end;

procedure TDataField.SetOptions(const Value: TEditOptions);
 var
  ChangedOptions: TEditOptions;
begin
  if FOptions <> Value then
  begin
    ChangedOptions := (FOptions + Value) - (FOptions * Value);
    FOptions := Value;
    if eoReadOnly in ChangedOptions then PropertyChanged;
  end;
end;

{ TDataContainer }

procedure TDataContainer.ClearFieldData(Field: TDataField);
 var
  KnotItem: TKnotItem;
begin
  KnotItem := DataGrid.Knots.First;
  while KnotItem <> nil do
  begin
    ActiveItem := KnotItem;
    Field.Clear;
    ActiveItem := nil;
    KnotItem := KnotItem.GetNext;
  end;
end;

procedure TDataContainer.ClearKnotData(KnotItem: TKnotItem);
 var
  Field: TDataField;
  i: integer;
begin
  if KnotItem.Data <> nil then
  begin
    for i := 0 to Fields.FList.Count - 1 do
    begin
      Field := Fields.FList[i];
      ActiveItem := KnotItem;
      Field.Clear;
      ActiveItem := nil;
    end;
    FreeMem(KnotItem.Data, Capacity * SizeOf(Pointer));
    KnotItem.Data := nil;
  end;
end;

constructor TDataContainer.Create(ADataGrid: TDCCustomDataGrid);
begin
  inherited Create;
  FFields := TDataFields.Create(Self);
  FDataGrid := ADataGrid;
  FCapacity := 0;

  FLockEvent := CreateEvent(nil, False, False, nil);
  FLockCount := 0;
end;

function TDataContainer.CreateData: Pointer;
 var
  DataSize: integer;
begin
  DataSize := Capacity * SizeOf(Pointer);
  GetMem(Result, DataSize);
  FillChar(Result^, DataSize, 0)
end;

function TDataContainer.DataAssigned: boolean;
begin
  Result := (FActiveItem <> nil) and (FActiveItem.Data <> nil);
end;

destructor TDataContainer.Destroy;
begin
  FFields.Free;
  CloseHandle(FLockEvent);
  inherited;
end;

procedure TDataContainer.Lock(AItem: TKnotItem);
begin
  if FActiveItem <> AItem then
  begin
    if FLockCount > 0 then
    begin
      while WaitForSingleObject(FLockEvent, 50) <> WAIT_OBJECT_0 do
        Sleep(50);
    end;
    InterlockedIncrement(FLockCount);
    FActiveItem := AItem;
  end;
end;

procedure TDataContainer.SetActiveItem(const Value: TKnotItem);
begin
  if Value = nil then
    Unlock
  else
    Lock(Value)
end;

procedure TDataContainer.SetCapacity(const Value: integer);
begin
  if Value <> FCapacity then
  begin
    UpdateDataSize(Value);
    FCapacity := Value;
  end;
end;

procedure TDataContainer.SetDataSize(const Value: integer;
  var pData: Pointer);
 var
  DataSize, i, j: integer;
begin
  DataSize := Value * SizeOf(Pointer);

  if FFields.FDeleted then
  begin
    j := 0;
    for i := 0 to Fields.Count - 1 do
    begin
      if Fields.Fields[i].FIndex <> j then
        Move(TDataValues(pData)[i], TDataValues(pData)[j], (i - j)*SizeOf(Pointer));
      j := i + 1;
    end;
    i := Fields.Count;
    FillChar((PChar(pData) + i*SizeOf(Pointer))^, (Capacity - i)*SizeOf(Pointer), 0);
  end;

  if Capacity <> Value then
  begin
    ReallocMem(pData, DataSize);
    if Value > Capacity then
      FillChar((PChar(pData) + Capacity*SizeOf(Pointer))^,
        (Value - Capacity)*SizeOf(Pointer), 0);
  end;
end;

procedure TDataContainer.Unlock;
begin
  FActiveItem := nil;
  InterlockedDecrement(FLockCount);
  if FLockEvent > 0 then SetEvent(FLockEvent);
end;

procedure TDataContainer.UpdateDataSize(const Value: integer);
 var
  KnotItem: TKnotItem;
  DataSize: integer;

begin
  KnotItem := DataGrid.Knots.First;
  while KnotItem <> nil do
  begin
    if Capacity = 0 then
    begin
      DataSize := Value * SizeOf(Pointer);
      KnotItem.Data := AllocMem(DataSize);
      FillChar(KnotItem.Data^, DataSize, 0);
    end;
    KnotItem := KnotItem.GetNext;
  end;
end;

procedure TDataContainer.UpdateGridColumns;
 var
  i: integer;
  NeedLayout: boolean;
begin
  if (DataGrid <> nil) and
    not(csDestroying in DataGrid.ComponentState) then with DataGrid do
  begin
    NeedLayout := False;
    BeginLayout;
    try
      for i := 0 to Columns.Count - 1 do
        if TDataColumn(Columns[i]).UpdateDataField then NeedLayout := True;
    finally
      if NeedLayout then
        EndLayout
      else
        DeferLayout;
    end;
  end;
end;

{ TNumericDataField }

constructor TNumericDataField.Create(AOwner: TComponent);
begin
  inherited;
  FDigitsCount := -1
end;

procedure TNumericDataField.SetDigitsCount(const Value: integer);
begin
  FDigitsCount := Value;
end;

{ TIntegerField }

constructor TIntegerDataField.Create(AOwner: TComponent);
begin
  inherited;
  SetDataType(dtInteger);
end;

function TIntegerDataField.GetAsFloat: Extended;
begin
  Result := GetAsInteger;
end;

function TIntegerDataField.GetAsInteger: Longint;
begin
  if not GetValue(Result) then Result := 0;
end;

function TIntegerDataField.GetAsString: string;
 var
  L: Longint;
begin
  if GetValue(L) then Str(L, Result) else Result := '';
end;

function TIntegerDataField.GetAsVariant: Variant;
 var
  L: Longint;
begin
  if GetValue(L) then Result := L else Result := Null;
end;

function TIntegerDataField.GetDataSize: Integer;
begin
  Result := SizeOf(Integer);
end;

procedure TIntegerDataField.GetText(var Text: string);
var
  L: Longint;
begin
  if GetValue(L) then
  begin
    if DisplayFormat = '' then
      Str(L, Text)
    else
      Text := FormatFloat(DisplayFormat, L);
  end
  else
    Text := '';
end;

{ TFloatDataField }

constructor TFloatDataField.Create(AOwner: TComponent);
begin
  inherited;
  SetDataType(dtFloat);
  FPrecision := -1
end;

function TFloatDataField.GetAsFloat: Extended;
begin
  if not GetData(@Result) then Result := 0;
end;

function TFloatDataField.GetAsInteger: Longint;
begin
  Result := Longint(Round(GetAsFloat));
end;

function TFloatDataField.GetAsString: string;
 var
  F: Extended;
begin
  if GetData(@F) then Result := FloatToStr(F) else Result := '';
end;

function TFloatDataField.GetAsVariant: Variant;
 var
  F: Extended;
begin
  if GetData(@F) then Result := F else Result := Null;
end;

function TFloatDataField.GetDataSize: Integer;
begin
  Result := SizeOf(Extended);
end;

procedure TFloatDataField.GetText(var Text: string);
var
  Format: TFloatFormat;
  Digits: Integer;
  F: Extended;
begin
  if GetData(@F) then
  begin
    if DisplayFormat = '' then
    begin
      if FCurrency then
      begin
        Format := ffCurrency;
        Digits := CurrencyDecimals;
      end
      else begin
        Format := ffGeneral;
        Digits := 0;
      end;
      Text := FloatToStrF(F, Format, FPrecision, Digits);
    end else
      Text := FormatFloat(DisplayFormat, F);
  end else
    Text := '';
end;

procedure TFloatDataField.SetAsFloat(const Value: Extended);
begin
  SetData(@Value);
end;

procedure TFloatDataField.SetAsInteger(const Value: Integer);
begin
  SetAsFloat(Value);
end;

procedure TFloatDataField.SetAsString(const Value: string);
 var
  F: Extended;
begin
  if Value = '' then
    Clear
  else begin
    if TextToFloat(PChar(Value), F, fvExtended) then SetAsFloat(F);
  end;
end;

procedure TFloatDataField.SetAsVariant(const Value: Variant);
begin
  SetAsFloat(Value);
end;

procedure TFloatDataField.SetCurrency(const Value: Boolean);
begin
  FCurrency := Value;
end;

procedure TFloatDataField.SetPrecision(const Value: Integer);
begin
  FPrecision := Value;
end;

function TIntegerDataField.GetValue(var Value: Integer): Boolean;
begin
  Result := GetData(@Value);
end;

procedure TIntegerDataField.SetAsFloat(const Value: Extended);
begin
  SetAsInteger(Integer(Round(Value)));
end;

procedure TIntegerDataField.SetAsInteger(const Value: Integer);
begin
  SetData(@Value)
end;

procedure TIntegerDataField.SetAsString(const Value: string);
 var
  E: Integer;
  L: Longint;
begin
  if Value = '' then
    Clear
  else begin
    Val(Value, L, E);
    if E = 0 then SetAsInteger(L);
  end;
end;

procedure TIntegerDataField.SetAsVariant(const Value: Variant);
begin
  SetAsInteger(Value);
end;

{ TBooleanDataField }

constructor TBooleanDataField.Create(AOwner: TComponent);
begin
  inherited;
  SetDataType(dtBoolean);
  FValues := TBooleanValues.Create(Self);
end;

destructor TBooleanDataField.Destroy;
begin
  FValues.Free;
  inherited;
end;

function TBooleanDataField.GetAsBoolean: Boolean;
 var
  B: WordBool;
begin
  if GetData(@B) then Result := B else Result := False;
end;

function TBooleanDataField.GetAsString: string;
 var
  B: WordBool;
begin
  if GetData(@B) then Result := FValues.Items[B].Text else Result := '';
end;

function TBooleanDataField.GetAsVariant: Variant;
 var
  B: WordBool;
begin
  if GetData(@B) then Result := B else Result := Null;
end;

function TBooleanDataField.GetDataSize: Integer;
begin
  Result := SizeOf(WordBool)
end;

procedure TBooleanDataField.GetText(var Text: string);
begin
  case Values.Style of
    dsText:
      Text := GetAsString;
    dsValue:
      Text := IntToStr(Integer(GetAsBoolean));
    dsImage:
      {Noting}
  end;
end;

procedure TBooleanDataField.SetAsBoolean(const Value: Boolean);
 var
  B: WordBool;
begin
  if Value then Word(B) := 1 else Word(B) := 0;
  SetData(@B);
end;

procedure TBooleanDataField.SetAsString(const Value: string);
begin
 if AnsiCompareText(Value, Values.Items[False].Text) = 0 then
   SetAsBoolean(False)
 else
   if AnsiCompareText(Value, Values.Items[False].Text) = 0 then SetAsBoolean(True)
end;

procedure TBooleanDataField.SetAsVariant(const Value: Variant);
begin
  SetAsBoolean(Value);
end;

procedure TBooleanDataField.SetValues(const Value: TBooleanValues);
begin
  FValues.Assign(Value);
end;

{ TWordDataField }

constructor TWordDataField.Create(AOwner: TComponent);
begin
  inherited;
  SetDataType(dtWord)
end;

function TWordDataField.GetDataSize: Integer;
begin
  Result := SizeOf(Byte);
end;

{ TDataDateTimeField }

constructor TDateTimeDataField.Create(AOwner: TComponent);
begin
  inherited;
  SetDataType(dtDateTime);
end;

function TDateTimeDataField.GetAsDateTime: TDateTime;
begin
  if not GetValue(Result) then Result := 0;
end;

function TDateTimeDataField.GetAsFloat: Extended;
begin
  Result := GetAsDateTime;
end;

function TDateTimeDataField.GetAsString: string;
begin
  GetText(Result);
end;

function TDateTimeDataField.GetAsVariant: Variant;
 var
  D: TDateTime;
begin
  if GetValue(D) then Result := VarFromDateTime(D) else Result := Null;
end;

function TDateTimeDataField.GetDataSize: Integer;
begin
  Result := SizeOf(TDateTime);
end;

procedure TDateTimeDataField.GetText(var Text: string);
 var
  D: TDateTime;
begin
  if GetValue(D) then
    DateToStrY2K(D, Text, dkDateTime)
  else
    Text := '';
end;

function TDateTimeDataField.GetValue(var Value: TDateTime): boolean;
begin
  Result := GetData(@Value);
end;

procedure TDateTimeDataField.SetAsDateTime(const Value: TDateTime);
begin
  SetData(@Value);
end;

procedure TDateTimeDataField.SetAsFloat(const Value: Extended);
begin
  SetAsDateTime(Value);
end;

procedure TDateTimeDataField.SetAsString(const Value: string);
begin
  if Value = '' then
    Clear
  else
    SetAsDateTime(StrToDateTime(Value));
end;

procedure TDateTimeDataField.SetAsVariant(const Value: Variant);
begin
  SetAsDateTime(VarToDateTime(Value));
end;

{ TDataDateField }

constructor TDateDataField.Create(AOwner: TComponent);
begin
  inherited;
  SetDataType(dtDate);
end;

procedure TDateDataField.GetText(var Text: string);
 var
  D: TDateTime;
begin
  if GetValue(D) then
    DateToStrY2K(D, Text, dkDate)
  else
    Text := '';
end;

procedure TDateDataField.SetAsString(const Value: string);
begin
  if Value = '' then
    Clear
  else
    SetAsDateTime(StrToDate(Value));
end;

{ TDataTimeField }

constructor TTimeDataField.Create(AOwner: TComponent);
begin
  inherited;
  SetDataType(dtTime);
end;

procedure TTimeDataField.GetText(var Text: string);
var
  D: TDateTime;
begin
  if GetValue(D) then
    DateTimeToString(Text, LongTimeFormat, D)
  else
    Text := '';
end;

procedure TTimeDataField.SetAsString(const Value: string);
begin
  if Value = '' then
    Clear
  else
    SetAsDateTime(StrToTime(Value));
end;

{ TStringDataField }

constructor TStringDataField.Create(AOwner: TComponent);
begin
  inherited;
  SetDataType(dtString);
end;

function TStringDataField.GetAsString: string;
 var
  pValue: Pointer;
  ASize1, ASize2, ALen: integer;
begin
  if DataAssigned then
  begin
    pValue := TDataValues(DataContainer.ActiveItem.Data^)[Index];
    if pValue <> nil then
    begin
      ASize1 := SizeOf(Integer);
      Move(pValue^, ALen, ASize1);
      ASize2 := ALen * SizeOf(Char);
      SetLength(Result, ALen);
      Inc(PChar(pValue), ASize1);
      Move(pValue^, Pointer(Result)^, ASize2);
    end
  end
end;

procedure TStringDataField.GetText(var Text: string);
begin
  if DisplayFormat <> '' then
    Text := Format(DisplayFormat, [GetAsString])
  else
    Text := GetAsString;  
end;

procedure TStringDataField.SetAsString(const Value: string);
 var
  pValue, Data:  Pointer;
  ASize1, ASize2, ALen: integer;
begin
  if DataAssigned then
  begin
    ASize1 := SizeOf(Integer);
    Data := DataContainer.ActiveItem.Data;
    if Value <> '' then
    begin
      ALen := Length(Value);
      ASize2 := ALen * SizeOf(Char);

      pValue := TDataValues(Data^)[Index];
      ReallocMem(pValue, ASize1 + ASize2);
      TDataValues(Data^)[Index] := pValue;

      Move(ALen, pValue^, ASize1);
      Inc(PChar(pValue), ASize1);
      Move(Pointer(Value)^, pValue^, ASize2);
    end
    else begin
      pValue := TDataValues(Data^)[Index];
      if Assigned(pValue) then
      begin
        Move(pValue^, ALen, ASize1);
        ASize2 := ALen * SizeOf(Char);
        FreeMem(pValue, ASize1 + ASize2);
      end;
      FillChar(TDataValues(Data^)[Index], SizeOf(Pointer), 0);
    end;
  end
end;

{ TDataColumn }

function TDataColumn.GetGrid: TDCCustomDataGrid;
begin
  Result := TDCCustomDataGrid(inherited GetGrid);
end;

procedure TDataColumn.SetDataField(const Value: TDataField);
begin
  FDataField := Value;
  FFieldName := Value.Name;
end;

procedure TDataColumn.SetFieldName(const Value: string);
begin
  if FFieldName <> Value then
  begin
    if Title.Caption = FFieldName then Title.Caption := Value;
    FFieldName := Value;
    UpdateDataField;
  end;
end;

function TDataColumn.UpdateDataField: boolean;
 var
  ADataField: TDataField;
begin
  Result := False;
  if Grid <> nil then
  begin
    ADataField := FDataField;
    FDataField := Grid.DataContainer.Fields.FindField(FFieldName);
    Result := FDataField <> ADataField;
  end;
end;

{ TDCCustomDataGrid }

constructor TDCCustomDataGrid.Create(AOwner: TComponent);
begin
  inherited;
  FDataContainer := CreateDataContainer;
end;

function TDCCustomDataGrid.CreateDataContainer: TDataContainer;
begin
  Result := TDataContainer.Create(Self);
end;

procedure TDCCustomDataGrid.DefaultDrawDataCell(Canvas: TCanvas;
  ARect: TRect; ACol: integer; AColumn: TKnotColumn; AKnot: TDataKnotItem;
  AState: TGridCellState);
 const
  AlignFlags : array [TAlignment] of Integer =
    ( DT_LEFT   or DT_NOPREFIX or DT_END_ELLIPSIS or DT_EXPANDTABS,
      DT_RIGHT  or DT_NOPREFIX or DT_END_ELLIPSIS or DT_EXPANDTABS,
      DT_CENTER or DT_NOPREFIX or DT_END_ELLIPSIS or DT_EXPANDTABS );
  StandartChecks : array[Boolean] of integer =
    (nsiNormalCheck0, nsiNormalCheck1);
 var
  AText: string;
  F: TDataField;
  B: Boolean;

 function GetImageDrawText(ImageIndex: integer): string;
 begin
   if Focused and (gsSelected in AState) then
     Result := Format('/is{%d}', [ImageIndex])
   else
     Result := Format('/ip{%d}', [ImageIndex]);
 end;

 procedure WriteText(Canvas: TCanvas; Text: string; R: TRect;
   Alignment: TAlignment);
  var
   I: DWORD;
   Left: integer;
 begin
   I := ColorToRGB(Canvas.Brush.Color);
   if not(kcEllipsis in AColumn.Options) and
     (GetNearestColor(Canvas.Handle, I) = I) then
   begin
     if (Canvas.CanvasOrientation = coRightToLeft) then
       ChangeBiDiModeAlignment(Alignment);
     case Alignment of
       taLeftJustify:
         Left := R.Left;
       taRightJustify:
         Left := _intMax(2, R.Right - Canvas.TextWidth(AText) - 3);
     else { taCenter }
       Left := _intMax(2, R.Left + (R.Right - R.Left) shr 1 -
         (Canvas.TextWidth(Text) shr 1));
     end;
     Canvas.TextRect(R, Left, R.Top, Text);
   end
   else
     DrawText(Canvas.Handle, PChar(Text), Length(AText), R,
       AlignFlags[AColumn.Alignment]);
 end;

begin
  F := TDataColumn(AColumn).DataField;
  if F = nil then Exit;
  case F.DataType of
    dtBoolean:
      with TBooleanDataField(F) do
      begin
        case Values.Style of
          dsImage:
            begin
              begin
                B := GetAsBoolean;
                if (Values.Images <> nil) and (Values.Items[B].ImageIndex <> -1) then
                begin
                  AText := GetImageDrawText(Values.Items[B].ImageIndex);
                  DrawTextEx(Canvas, AColumn, ARect, AText, Images,
                    AColumn.Alignment);
                end
                else begin
                  AText := GetImageDrawText(StandartChecks[B]);
                  DrawTextEx(Canvas, AColumn, ARect, AText,
                    ETGetSystemImages(DCGIM_SMALLICON), AColumn.Alignment);
                end;

              end
            end;
          else begin
            AText := F.DisplayText;
            WriteText(Canvas, AText, ARect, AColumn.Alignment);
          end;
        end;
      end;
    else begin
      AText := F.DisplayText;
      InflateRect(ARect, -1, -1);
      WriteText(Canvas, AText, ARect, AColumn.Alignment);
    end;
  end;
end;

destructor TDCCustomDataGrid.Destroy;
begin
  Knots.Clear;
  FDataContainer.Free;
  FDataContainer := nil;
  inherited;
end;

procedure TDCCustomDataGrid.DoDrawColumnCell(Canvas: TCanvas; ARect: TRect;
  ACol: integer; AColumn: TKnotColumn; AKnot: TKnotItem;
  AState: TGridCellState);
 var
  DefaultDraw: boolean;
begin
  DataContainer.Lock(AKnot);
  DefaultDraw := True;
  if Assigned(FOnDrawDataCell) then
  begin
    DefaultDraw := False;
    FOnDrawDataCell(Self, ARect, Canvas, TDataColumn(AColumn),
      TDataKnotItem(AKnot), AState, DefaultDraw)
  end;
  if DefaultDraw then
    DefaultDrawDataCell(Canvas, ARect, ACol, AColumn, TDataKnotItem(AKnot), AState);
  DataContainer.Unlock;
end;

procedure TDCCustomDataGrid.DrawTextEx(Canvas: TCanvas; AColumn: TKnotColumn;
  R: TRect; Text: string; Images: TImageList; Alignment: TAlignment);
 var
  Flags: DWORD;
begin
  if kcWordBreak in AColumn.Options then
    Flags := DT_WORDBREAK
  else
    Flags := 0;
  case Alignment of
    taLeftJustify:
      DrawHighLightText(Canvas, PChar(Text), R, 1, DT_NOPREFIX or DT_CENTER or
        Flags, Images);
    taCenter, taRightJustify:
      DrawTitleRect(Canvas, R, Text, Alignment, True, Flags, Images)
  end;
end;

procedure TDCCustomDataGrid.GetChildren(Proc: TGetChildProc;
  Root: TComponent);
var
  i: Integer;
  Field: TDataField;
begin
  inherited;
  for i := 0 to DataFields.Count - 1 do
  begin
    Field := DataFields[i];
    if Field.Owner = Root then Proc(Field);
  end;
end;

function TDCCustomDataGrid.GetColumns: TDataColumns;
begin
  Result := TDataColumns(inherited GetColumns);
end;

function TDCCustomDataGrid.GetFiedls: TDataFields;
begin
  Result := FDataContainer.Fields;
end;

procedure TDCCustomDataGrid.GetFieldNames(AList: TStrings);
 var
  i: integer;
begin
  with AList do
  begin
    BeginUpdate;
    Clear;
    for i := 0 to DataFields.Count - 1 do
      Add(DataFields.Fields[i].FieldName);
    EndUpdate;
  end;
end;

function TDCCustomDataGrid.GetKnots: TDataKnotItems;
begin
  Result := TDataKnotItems(inherited GetKnots);
end;

function TDCCustomDataGrid.Initialize: TDCTreeGridInitialize;
begin
  Result := inherited Initialize;
  Result.ColumnClass := TDataColumn;
  Result.ItemsClass := TDataKnotItems;
  Result.ItemClass := TDataKnotItem;
end;

procedure TDCCustomDataGrid.SetColumns(const Value: TDataColumns);
begin
  inherited SetColumns(Value);
end;

procedure TDCCustomDataGrid.SetKnots(const Value: TDataKnotItems);
begin
  inherited SetKnots(Value);
end;

{ TDataKnotItem }

constructor TDataKnotItem.Create(AOwner: TKnotItems; AParent: TKnotItem;
  AName: string);
begin
  inherited;
  if Assigned(Grid) and Assigned(Grid.DataContainer) then
    Data := Grid.DataContainer.CreateData;
end;

destructor TDataKnotItem.Destroy;
begin
  if Assigned(Grid.DataContainer) then Grid.DataContainer.ClearKnotData(Self);
  inherited;
end;

function TDataKnotItem.FieldByName(const FieldName: string): TDataField;
begin
  Result := Grid.DataContainer.Fields.FindField(FieldName);
end;

function TDataKnotItem.GetField(Index: Integer): TDataField;
begin
  Result := Grid.DataContainer.Fields.FList[Index];
end;

function TDataKnotItem.GetGrid: TDCCustomDataGrid;
begin
  Result := TDCCustomDataGrid(inherited GetGrid);
end;

procedure TDataKnotItem.SetField(Index: Integer; const Value: TDataField);
begin
  Grid.DataContainer.Fields[Index].Assign(Value);
end;

{ TDisplayValues }

function TDisplayValues.CanAdd: boolean;
begin
  Result :=True;
end;

constructor TDisplayValues.Create(ItemClass: TCollectionItemClass;
  ADataField: TDataField);
begin
  inherited Create(ItemClass);
  FDataField := ADataField;
  FOptions := [];
  FStyle := dsValue;
  FImageChangeLink :=  TChangeLink.Create;
  FImageChangeLink.OnChange := ImageListChange;
end;

destructor TDisplayValues.Destroy;
begin
  FImageChangeLink.Free;
  inherited;
end;

procedure TDisplayValues.ImageListChange(Sender: TObject);
begin
  Changed;
end;

procedure TDisplayValues.Changed;
begin
  if FDataField <> nil then FDataField.PropertyChanged;
end;

procedure TDisplayValues.SetImages(const Value: TImageList);
begin
  if Images <> nil then
    Images.UnRegisterChanges(FImageChangeLink);
  FImages := Value;
  if Images <> nil then
  begin
    Images.RegisterChanges(FImageChangeLink);
    Images.FreeNotification(FDataField);
  end;
  Changed;
end;

procedure TDisplayValues.SetOptions(const Value: TDisplayOptions);
begin
  FOptions := Value;
end;

procedure TDisplayValues.SetStyle(const Value: TDisplayStyle);
begin
  if FStyle <> Value then
  begin
    FStyle := Value;
    Changed;
  end;
end;

function TDisplayValues.GetOwner: TPersistent;
begin
  if FDataField <> nil then
    Result := FDataField.GetParentComponent
  else
    Result := nil;
end;

{ TBooleanValues }

function TBooleanValues.CanAdd: boolean;
begin
  Result := False;
end;

constructor TBooleanValues.Create(ADataField: TDataField);
begin
  inherited Create(TBooleanDisplayItem, ADataField);
  FStyle := dsImage;

  with TBooleanDisplayItem(Add) do FText := 'False';
  with TBooleanDisplayItem(Add) do FText := 'True';
end;

function TBooleanValues.GetItem(Index: Boolean): TBooleanDisplayItem;
begin
  Result := TBooleanDisplayItem(inherited GetItem(Integer(Index)));
end;

{ TDisplayItem }

constructor TDisplayItem.Create(Collection: TCollection);
begin
  inherited;
  FImageIndex := -1;
end;

procedure TDisplayItem.SetImageIndex(const Value: integer);
begin
  if FImageIndex <> Value then
  begin
    FImageIndex := Value;
    with TDisplayValues(Collection) do
    begin
      if (Images <> nil) and (Style = dsImage) then Changed;
    end;
  end;
end;

procedure TDisplayItem.SetText(const Value: string);
begin
  if FText <> Value then
  begin
    FText := Value;
    TDisplayValues(Collection).Changed;
  end;
end;

{ TDataColumns }

function TDataColumns.Add: TDataColumn;
begin
  Result := TDataColumn(inherited Add);
end;

function TDataColumns.GetItem(Index: Integer): TDataColumn;
begin
  Result := TDataColumn(inherited GetItem(Index))
end;

procedure TDataColumns.SetItem(Index: Integer; const Value: TDataColumn);
begin
  inherited SetItem(Index, Value);
end;

{ TDataKnotItems }

function TDataKnotItems.Add(Name: string;
  Position: integer): TDataKnotItem;
begin
  Result := TDataKnotItem(inherited Add(Name, Position));
end;

function TDataKnotItems.GetItem(Index: integer): TDataKnotItem;
begin
  Result := TDataKnotItem(inherited GetItem(Index))
end;

procedure TDataKnotItems.SetItem(Index: integer; const Value: TDataKnotItem);
begin
  inherited SetItem(Index, Value);
end;

end.
