{
 BUSINESS CONSULTING
 s a i n t - p e t e r s b u r g

         Components Library for Borland Delphi 4.x - 6.x
         Copyright (c) 2001 decosp

}
unit DCDBDataGrid;

{$R-}
{$G+}

interface
{$I DCConst.inc}

uses
  Windows, SysUtils, Messages, Classes, Dialogs, Graphics, Grids, Controls,
  ImgList, StdCtrls, 
  {$IFDEF DELPHI_V6}
    Variants,
  {$ENDIF}
  DB, DCKnots, DCDataGrid, DCConst, DCEditTools;

type
  TGridLoadState = (gsDataView, gsLoadStruct, gsFullLoad);

  TDBDataKnotItems = class;
  TDCDBCustomDataGrid = class;

  TFieldMapping = packed record
    Count: integer;
    Capacity: integer;
    Fields: PIntegerValues;
  end;

  TDCTreeGridDataLink = class(TDataLink)
  private
    FGrid: TDCDBCustomDataGrid;
    FFieldMapping: TFieldMapping;
    function GetFields(I: Integer): TField;
  protected
    procedure ActiveChanged; override;
    procedure DataSetChanged; override;
    procedure DataSetScrolled(Distance: Integer); override;
    procedure LayoutChanged; override;
    procedure RecordChanged(Field: TField); override;
  public
    function AddMapping(const FieldName: string): Boolean;
    procedure ClearMapping;
    constructor Create(AGrid: TDCDBCustomDataGrid);
    property Fields[I: Integer]: TField read GetFields;
  end;

  TDBDataKnotItem = class(TDataKnotItem)
  end;

  TDBDataKnotItems = class(TDataKnotItems)
  private
    function GetGrid: TDCDBCustomDataGrid;
  public
    function Delete(Knot: TKnotItem): boolean; override;
    property Grid: TDCDBCustomDataGrid read GetGrid;
  end;

  TDCDBCustomDataGrid = class(TDCCustomDataGrid)
  private
    FDataLink: TDCTreeGridDataLink;
    FLoadState: TGridLoadState;
    procedure DataChanged;
    function GetDataSource: TDataSource;
    procedure SetDataSource(const Value: TDataSource);
    procedure RecordChanged(Field: TField);
  protected
    procedure ClearBookmarkData(Bookmark: TKnotBookmark); override;
    function CompareBookmarks(Item1, Item2: Pointer): integer; override;
    procedure DefineFieldMap; virtual;
    procedure DoEndDrawCell(Data: Pointer); override;
    function DoBeginDrawCell: Pointer; override;
    function DoBeginInternalLayout: Pointer; override;
    function DoMouseWheelDown(Shift: TShiftState; MousePos: TPoint): Boolean; override;
    function DoMouseWheelUp(Shift: TShiftState; MousePos: TPoint): Boolean; override;
    function Eof: boolean; override;
    procedure First; override;
    function GetBookmark(KnotItem: TKnotItem): TKnotBookmark; override;
    function GetRowCount: integer; override;
    function GetRecordCount: integer; override;
    procedure GetVertScrollInfo(var ScrollInfo: TScrollInfo); override;
    procedure GotoBookmark(BookmarkInfo: TBookmarkInfo); override;
    function HideVertScrollBar: boolean; override;
    function Initialize: TDCTreeGridInitialize; override;
    procedure InsertKnot(lChild: boolean; Shift: TShiftState); override;
    procedure Last; override;
    procedure LinkActive(Value: Boolean); virtual;
    procedure LoadDataFields;
    procedure MoveBy(Offset: integer); override;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    procedure SelectAll; override;
    procedure Scroll(Distance: Integer); virtual;
    procedure UpdateActive; override;
    function ValidBookmark(Bookmark: TKnotBookmark): boolean; override;
    function VertScrollBarVisible: boolean; override;
    procedure WMVScroll(var Message: TWMVScroll); message WM_VSCROLL;
    property DataLink: TDCTreeGridDataLink read FDataLink;
    property DataSource: TDataSource read GetDataSource write SetDataSource;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function GetKnotByCell(ARow: integer): TKnotItem; override;
    procedure SavePosition; override;
  end;

  TDCDBDataGrid = class(TDCDBCustomDataGrid)
  published
    property DataSource;
    property Options;
  end;

implementation

type
  TPrivateDataSet = class(TDataSet)
    {}
  end;

{ TDCTreeGridDataLink }

procedure TDCTreeGridDataLink.ActiveChanged;
begin
  FGrid.LinkActive(Active);
end;

function TDCTreeGridDataLink.AddMapping(const FieldName: string): Boolean;
var
  Field: TField;
begin
  Result := True;
  Field := DataSet.FindField(FieldName);

  with FFieldMapping do
  begin
    if Count >= Capacity then
    begin
      Capacity := _intMax(Count + 4, 8);
      ReallocMem(Fields, Capacity * SizeOf(Integer));
    end;
    if Assigned(Field) then
    begin
      Fields[Count] := Field.Index;
      Field.FreeNotification(FGrid);
    end
    else
      Fields[Count] := -1;
    Inc(Count);
  end;
end;

procedure TDCTreeGridDataLink.ClearMapping;
begin
  with FFieldMapping do
  begin
    FreeMem(Fields, Capacity * SizeOf(Integer));
    Fields := nil;
    Count := 0;
    Capacity := 0;
  end;
end;

constructor TDCTreeGridDataLink.Create(AGrid: TDCDBCustomDataGrid);
begin
  inherited Create;
  FGrid := AGrid;
  with FFieldMapping do
  begin
    Count := 0;
    Capacity := 0;
    Fields := nil;
  end;
  VisualControl := True;
end;

procedure TDCTreeGridDataLink.DataSetChanged;
begin
  with FGrid do
  begin
    DataChanged;
    SetModified(False);
  end;  
end;

procedure TDCTreeGridDataLink.DataSetScrolled(Distance: Integer);
begin
  FGrid.Scroll(Distance);
end;

function TDCTreeGridDataLink.GetFields(I: Integer): TField;
begin
  with FFieldMapping do
  begin
    if (0 <= I) and (I < Count) and (Fields[I] >= 0) then
      Result := DataSet.FieldList[Fields[I]]
    else
      Result := nil;
  end;
end;

procedure TDCTreeGridDataLink.LayoutChanged;
begin
  if FGrid.FLoadState in [gsDataView, gsLoadStruct] then FGrid.LayoutChanged;
  inherited;
end;

procedure TDCTreeGridDataLink.RecordChanged(Field: TField);
begin
  FGrid.RecordChanged(Field);
end;

{ TDCDBCustomDataGrid }

constructor TDCDBCustomDataGrid.Create(AOwner: TComponent);
begin
  inherited;
  FDataLink := TDCTreeGridDataLink.Create(Self);
  ScrollBars := ssHorizontal;
end;

procedure TDCDBCustomDataGrid.DoEndDrawCell(Data: Pointer);
begin
  if (FLoadState = gsDataView) and DataLink.Active then
  begin
    DataLink.ActiveRecord := Integer(Data^);
    FreeMem(Data, SizeOf(Integer));
  end;
end;

function TDCDBCustomDataGrid.DoBeginDrawCell: Pointer;
begin
  if (FLoadState = gsDataView) and DataLink.Active then
  begin
    GetMem(Result, SizeOf(Integer));
    Integer(Result^) := DataLink.ActiveRecord;
  end
  else
    Result := nil;
end;

function TDCDBCustomDataGrid.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource;
end;

function TDCDBCustomDataGrid.GetRowCount: integer;
 var
  DrawInfo: TGridDrawInfo;
begin
  if FLoadState = gsDataView then
  begin
    with FDataLink do
      if not Active or (RecordCount = 0) or not HandleAllocated then
        Result := 1
      else
      begin
        CalcDrawInfoEx(DrawInfo, ColCount, 250);
        FDataLink.BufferCount := DrawInfo.Vert.LastFullVisibleCell - TopRow + 1;
        Result := RecordCount;
      end;
  end
  else
    Result := inherited GetRowCount;
end;

procedure TDCDBCustomDataGrid.LinkActive(Value: Boolean);
begin
  if FLoadState in [gsDataView, gsLoadStruct] then
  begin
    SetModified(False);
    if not Value then HideEditor;
    try
      LayoutChanged;
      UpdateRowCount;

      Knots.Add('$$');
      
    finally
      if Value and (tgAlwaysShowEditor in Options) then ShowEditor;
    end;
  end;
end;

procedure TDCDBCustomDataGrid.LoadDataFields;
 var
  i: integer;
  Field: TField;
  DataField: TDataField;
begin
  if (FLoadState = gsDataView) and (DataLink.DataSet <> nil) then
  begin
    Knots.BeginUpdate;
    DataContainer.Fields.Clear;
    Knots.Clear;
    Knots.EndUpdate;
    with Datalink.DataSet do
      for i := 0 to FieldCount - 1 do
      begin
        Field := Fields[i];
        DataField := TStringDataField.Create(Owner);
        with DataField do
        begin
          Name := Format('%s%s', [Self.Name, Field.FieldName]);
          FieldName := Field.FieldName;
          DataContainer := Self.DataContainer;
        end;
      end;
  end;
end;

procedure TDCDBCustomDataGrid.MoveBy(Offset: integer);
begin
  if FLoadState = gsDataView then
    FDataLink.MoveBy(Offset)
  else
    inherited;
end;

procedure TDCDBCustomDataGrid.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited;
  if (Operation = opRemove) then
  begin
    if (FDataLink <> nil) then
    begin
      if (AComponent = DataSource) then
        DataSource := nil
      else if (AComponent is TField) then
      begin
        {!!}
      end;
    end;
  end;
end;

procedure TDCDBCustomDataGrid.RecordChanged(Field: TField);
 var
  i: Integer;
begin
  if not(HandleAllocated and (FLoadState in [gsDataView, gsLoadStruct])) then Exit;
  if Field = nil then
    Invalidate
  else begin
    for i := 0 to Columns.Count - 1 do
      if (Columns[I].DataField.FieldName = Field.FieldName) then
        InvalidateCol(DataToRawColumn(I));
  end;
end;

procedure TDCDBCustomDataGrid.Scroll(Distance: Integer);
 var
  OldRect, NewRect: TRect;
  RowHeight: Integer;
begin
  if not(HandleAllocated and (FLoadState in [gsDataView, gsLoadStruct])) then Exit;
  OldRect := BoxRectEx(0, Row, ColCount - 1, Row);
  if (FDataLink.ActiveRecord >=  RawToDataRow(RowCount)) then UpdateRowCount;
  UpdateVertScrollBar;
  UpdateActive;
  NewRect := BoxRectEx(0, Row, ColCount - 1, Row);
  ValidateRect(Handle, @OldRect);
  InvalidateRect(Handle, @OldRect, False);
  InvalidateRect(Handle, @NewRect, False);
  if Distance <> 0 then
  begin
    HideEditor;
    try
      if Abs(Distance) > VisibleRowCount then
      begin
        Invalidate;
        Exit;
      end
      else
      begin
        RowHeight := DefaultRowHeight;
        if tgRowLines in Options then Inc(RowHeight, GridLineWidth);
        NewRect := BoxRectEx(0, FixedRows, ColCount - 1, RowCount);
        ScrollWindowEx(Handle, 0, -RowHeight * Distance, @NewRect, @NewRect,
          0, nil, SW_Invalidate);
      end;
    finally
      if tgAlwaysShowEditor in Options then ShowEditor;
    end;
  end;
  if UpdateLock = 0 then Update;
end;

procedure TDCDBCustomDataGrid.SetDataSource(const Value: TDataSource);
begin
  if Value = FDatalink.DataSource then Exit;
  SelectedRows.Clear;
  FDataLink.DataSource := Value;
  if Value <> nil then Value.FreeNotification(Self);
  LinkActive(FDataLink.Active);
end;

procedure TDCDBCustomDataGrid.WMVScroll(var Message: TWMVScroll);
 var
  ScrolInfo: TScrollInfo;
begin
  if not AcquireFocus and not DataVisible then Exit;
  if FLoadState = gsDataView then
  begin
    if FDatalink.Active then
      with Message, FDataLink.DataSet do
      begin
        case ScrollCode of
          SB_LINEUP:
            FDataLink.MoveBy(-FDatalink.ActiveRecord - 1);
          SB_LINEDOWN:
            FDataLink.MoveBy(FDatalink.RecordCount - FDatalink.ActiveRecord);
          SB_PAGEUP:
            FDataLink.MoveBy(-VisibleRowCount);
          SB_PAGEDOWN:
            FDataLink.MoveBy(VisibleRowCount);
          SB_THUMBPOSITION:
            begin
              if IsSequenced then with ScrolInfo do
              begin
                cbSize := sizeof(ScrolInfo);
                fMask := SIF_ALL;
                GetScrollInfo(Self.Handle, SB_VERT, ScrolInfo);
                if nTrackPos <= 1 then
                  First
                else if nTrackPos >= RecordCount then
                  Last
                else
                  RecNo := nTrackPos;
              end
              else
                case Pos of
                  0: First;
                  1: FDataLink.MoveBy(-VisibleRowCount);
                  2: Exit;
                  3: FDataLink.MoveBy(VisibleRowCount);
                  4: Last;
                end;
            end;
          SB_BOTTOM:
            Last;
          SB_TOP:
            First;
        end;
      end;
  end
  else
    inherited;
end;

function TDCDBCustomDataGrid.DoBeginInternalLayout: Pointer;
begin
  Result := nil;
  if DataLink <> nil then
  begin
    LoadDataFields;
    Datalink.ClearMapping;
    if Datalink.Active then DefineFieldMap;
  end;
end;

procedure TDCDBCustomDataGrid.DefineFieldMap;
 var
  i: integer;
begin
  for i := 0 to Columns.Count-1 do
    FDataLink.AddMapping(Columns[I].FieldName);
end;

procedure TDCDBCustomDataGrid.GetVertScrollInfo(
  var ScrollInfo: TScrollInfo);
 var
  SINew: TScrollInfo;
begin
  with FDatalink.DataSet, ScrollInfo do
  begin
    if IsSequenced then
    begin
      nMin := 1;
      nPage := VisibleRowCount;
      nMax := Integer(DWORD(RecordCount) + nPage - 1);
      if State in [dsInactive, dsBrowse, dsEdit] then nPos := RecNo;
    end
    else begin
      nMin := 0;
      if VisibleRowCount < FDataLink.BufferCount then
        SINew.nPage := VisibleRowCount
      else
        SINew.nPage := 0;
      nMax := 4;
      if BOF then 
        nPos := 0
      else 
        if EOF then nPos := 4 else nPos := 2;
    end;
  end;  
end;

function TDCDBCustomDataGrid.VertScrollBarVisible: boolean;
begin
  Result := FDatalink.Active;
end;

procedure TDCDBCustomDataGrid.DataChanged;
begin
  if not HandleAllocated then Exit;
  UpdateRowCount;
  UpdateVertScrollBar;
end;

procedure TDCDBCustomDataGrid.UpdateActive;
 var
  NewRow: integer;
begin
  if FLoadState = gsDataView then
  begin
    if FDatalink.Active and HandleAllocated and not
      (csLoading in ComponentState) then
    begin
      if FDatalink.DataSet.Active and (FDatalink.ActiveRecord = FDatalink.RecordCount)
      then begin
        FDatalink.DataSet.Prior;
        FDatalink.DataSet.Next;
      end;
      NewRow := FDatalink.ActiveRecord + FixedRows;
      if Row <> NewRow then
      begin
        if not (tgAlwaysShowEditor in Options) then HideEditor;
        MoveColRow(Col, NewRow, False, False);
      end;
    end;
  end
  else
    inherited;
end;

function TDCDBCustomDataGrid.Eof: boolean;
begin
  if FLoadState = gsDataView then
    Result := FDataLink.Eof
  else
    Result := inherited Eof;
end;

procedure TDCDBCustomDataGrid.First;
begin
  if FLoadState = gsDataView then
    DataLink.DataSet.First
  else
    inherited;
end;

procedure TDCDBCustomDataGrid.Last;
begin
  if FLoadState = gsDataView then
    DataLink.DataSet.Last
  else
    inherited;
end;

destructor TDCDBCustomDataGrid.Destroy;
begin
  FDataLink.Free;
  FDataLink := nil;
  inherited;
end;

function TDCDBCustomDataGrid.DoMouseWheelDown(Shift: TShiftState;
  MousePos: TPoint): Boolean;
begin
  if (FLoadState = gsDataView) and not(ssShift in Shift) then
    Result := inherited DoMouseWheelDown(Shift + [ssShift], MousePos)
  else
    Result := inherited DoMouseWheelDown(Shift, MousePos)
end;

function TDCDBCustomDataGrid.DoMouseWheelUp(Shift: TShiftState;
  MousePos: TPoint): Boolean;
begin
  if (FLoadState = gsDataView) and not(ssShift in Shift) then
    Result := inherited DoMouseWheelUp(Shift + [ssShift], MousePos)
  else
    Result := inherited DoMouseWheelUp(Shift, MousePos)
end;

function TDCDBCustomDataGrid.HideVertScrollBar: boolean;
begin
  Result := False;
end;

function TDCDBCustomDataGrid.GetRecordCount: integer;
begin
  if FLoadState = gsDataView then
  begin
    Result := FDataLink.DataSet.RecordCount
  end
  else
    Result := inherited GetRecordCount;
end;

procedure TDCDBCustomDataGrid.SelectAll;
 var
  AList: TList;
begin
  if FLoadState = gsDataView then
  begin
    if DataLink.DataSet <> nil then
      with DataLink.DataSet do
      begin
        SavePosition;
        AList := TList.Create;
        DisableControls;
        try
          First;
          while not Eof do
          begin
            AList.Add(Self.GetBookmark(nil));
            Next;
          end;
          SelectedRows.Load(AList);
        finally
          AList.Free;
          RestPosition;
          EnableControls;
        end;
      end;
  end
  else
    inherited;
end;

function TDCDBCustomDataGrid.GetBookmark(KnotItem: TKnotItem): TKnotBookmark;
begin
  if FLoadState = gsDataView then
    if Assigned(DataLink) and Assigned(DataLink.DataSet) then
    begin
      with TPrivateDataSet(DataLink.Dataset) do
      begin
        if BookmarkAvailable then
        begin
          GetMem(Result, BookmarkSize);
          GetBookmarkData(ActiveBuffer, Result);
        end
        else
          Result := nil
      end
      end
    else
        Result := nil
  else
    Result := inherited GetBookmark(KnotItem);
end;

procedure TDCDBCustomDataGrid.ClearBookmarkData(Bookmark: TKnotBookmark);
begin
  inherited;
end;

function TDCDBCustomDataGrid.CompareBookmarks(Item1,
  Item2: Pointer): integer;
begin
  if FLoadState = gsDataView then
  begin
    with DataLink.DataSource.Dataset do
      Result := CompareBookmarks(TBookmark(Item1), TBookmark(Item2));
  end
  else
    Result := inherited CompareBookmarks(Item1, Item2);
end;

function TDCDBCustomDataGrid.GetKnotByCell(ARow: integer): TKnotItem;
begin
  if FLoadState = gsDataView then
  begin
    DataLink.ActiveRecord := ARow;
    Result := Knots.GetFirstVisibleNode;
  end
  else
    Result := inherited GetKnotByCell(ARow);
end;

procedure TDCDBCustomDataGrid.SavePosition;
begin
  if FLoadState = gsDataView then
  begin
    if Assigned(DataLink) and Assigned(DataLink.DataSet) and DataLink.Active and
       (TPrivateDataSet(DataSource.DataSet).BookmarkSize > 0)
    then begin
      with DataLink.DataSet do
      begin
        ClearBookmarkData(FCurrentPos[1].Bookmark);
        ClearBookmarkData(FCurrentPos[2].Bookmark);

        DisableControls;
        Prior;
        if not Bof then
        begin
          FCurrentPos[1].Row      := Row;
          FCurrentPos[1].Bookmark := GetBookmark;
          FCurrentPos[1].ActiveRecord := DataLink.ActiveRecord;
          Next;
        end
        else
          FCurrentPos[1].Bookmark := nil;

        FCurrentPos[2].Row      := Row;
        FCurrentPos[2].Bookmark := GetBookmark;
        FCurrentPos[2].ActiveRecord := DataLink.ActiveRecord;
        EnableControls;
      end;
    end;
  end
  else
    inherited SavePosition;
end;

procedure TDCDBCustomDataGrid.GotoBookmark(BookmarkInfo: TBookmarkInfo);
 var
  ARow, BRow, i, j: integer;
begin
  if FLoadState = gsDataView then
  begin
    if ValidBookmark(BookmarkInfo.Bookmark) then
    begin
      with TPrivateDataSet(DataSource.DataSet) do
      begin
        DisableControls;
        try
          CheckBrowseMode;
          DoBeforeScroll;
          DataLink.ActiveRecord := BookmarkInfo.ActiveRecord;
          InternalGotoBookmark(BookmarkInfo.Bookmark);
          Resync([rmExact]);

          ARow := Row;
          BRow := BookmarkInfo.Row;

          if BRow > ARow then
          begin
            i := RawToDataRow(BRow); j := 0;
            while (i>0) and not FDatalink.DataSet.Bof do
            begin
              FDatalink.MoveBy(-1); inc(j); dec(i);
            end;
            if FDatalink.DataSet.Bof then Dec(j);
            FDatalink.MoveBy(j);
          end
          else begin
            if BRow < ARow then
            begin
              i := RowCount - BRow - 1; j := 0;
              while (i>0) and not FDatalink.DataSet.Eof do
              begin
                FDatalink.MoveBy(1); inc(j); dec(i);
              end;
              if FDatalink.DataSet.Eof then Dec(j);
              FDatalink.MoveBy(-j);
            end
            else begin
              FDatalink.MoveBy(-ARow + 1);
              FDatalink.MoveBy(ARow - 1);
            end;
          end;
          DoAfterScroll;
        finally
          EnableControls;
        end;
      end;
    end;
  end
  else
    inherited GotoBookmark(BookmarkInfo);
end;

function TDCDBCustomDataGrid.ValidBookmark(Bookmark: TKnotBookmark): boolean;
begin
  if FLoadState = gsDataView then
  begin
     try
       with TPrivateDataSet(DataSource.DataSet) do
         Result := not(Eof and Bof) and (BookmarkSize > 0);
       Result := Result and DataSource.Dataset.BookmarkValid(Bookmark)
     except
       Result := False;
     end;
  end
  else
    Result := inherited ValidBookmark(Bookmark)
end;

procedure TDCDBCustomDataGrid.InsertKnot(lChild: boolean;
  Shift: TShiftState);
begin
  if FLoadState = gsDataView then
  begin
    with FDataLink.DataSet do
    begin
      if FDataLink.EOF and CanModify and not ReadOnly and
        (tgEditing in Options) then
      begin
        Append;
        Knots.SetState(ksInsert);
      end;
    end;
  end
  else
    inherited;
end;

function TDCDBCustomDataGrid.Initialize: TDCTreeGridInitialize;
begin
  Result := inherited Initialize;
  Result.ItemsClass := TDBDataKnotItems;
  Result.ItemClass := TDBDataKnotItem;
end;

{ TDBDataKnotItems }

function TDBDataKnotItems.Delete(Knot: TKnotItem): boolean;
begin
  if Grid.FLoadState = gsDataView then
  begin
    if State = ksInsert then Grid.FDataLink.DataSet.Cancel;
    Result := True;
  end
  else
    Result := inherited Delete(Knot);
end;

function TDBDataKnotItems.GetGrid: TDCDBCustomDataGrid;
begin
  Result := TDCDBCustomDataGrid(inherited GetGrid);
end;

end.

