unit SimpleJournalTemplate;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Menus, ActnList, ComCtrls, Grids, Buttons, ToolWin, ExtCtrls, CommCtrl,
  DBTables, DCChoice, StdCtrls,
  DCPopupWindow, DCStringGrid, DCEditTools, DCConst, DCKnots, DCStdCtrls,
  BasedbLib, D2KConst, DCData;

type
  TLoadThread = class(TThread)
  private
    FGrid: TDCStringGrid;
    FOwner: TComponent;
  protected
    procedure Execute; override;
  public
    class procedure CreateThread(AGrid: TDCStringGrid; AOwner: TComponent);
    destructor Destroy; override;
  end;

  TSimpleJournalForm = class(TForm)
    pnMain: TPanel;
    tbMenu: TToolBar;
    tbFile: TToolButton;
    tbFunctions: TToolButton;
    tbView: TToolButton;
    tbButtons: TToolBar;
    tbSep_1: TToolButton;
    tbApply: TToolButton;
    pnGrid: TPanel;
    pnGridInfo: TDCHeaderPanel;
    pnSep_1: TPanel;
    pnGridBorder: TPanel;
    pnHints: TDCHeaderPanel;
    pnSep_2: TPanel;
    sbInfo: TStatusBar;
    ActionList: TActionList;
    acSaveExit: TAction;
    acNew: TAction;
    acEdit: TAction;
    acDelete: TAction;
    acSelectAll: TAction;
    acSaveAs: TAction;
    acPrint: TAction;
    acExport: TAction;
    acPagesetup: TAction;
    acApply: TAction;
    acHints: TAction;
    pmMain: TPopupMenu;
    mnMain_File: TMenuItem;
    mnSaveAs: TMenuItem;
    mnExport: TMenuItem;
    mnSep_1: TMenuItem;
    mnPagesetup: TMenuItem;
    mnPrint: TMenuItem;
    mnSep_2: TMenuItem;
    mnExitSave: TMenuItem;
    mnMain_Func: TMenuItem;
    mnMew: TMenuItem;
    mnEdit: TMenuItem;
    mnSep_5: TMenuItem;
    mnApply: TMenuItem;
    mnSep_3: TMenuItem;
    mnSelectAll: TMenuItem;
    mnSep_4: TMenuItem;
    mnDelete: TMenuItem;
    mnMain_View: TMenuItem;
    mnViewHints: TMenuItem;
    pmGrid: TPopupMenu;
    pmNew: TMenuItem;
    pmEdit: TMenuItem;
    pmSep_1: TMenuItem;
    pmSelectAll: TMenuItem;
    pmSep_2: TMenuItem;
    pmDelete: TMenuItem;
    Grid: TDCStringGrid;
    acExit: TAction;
    mnExit: TMenuItem;
    acAscending: TAction;
    acDescending: TAction;
    acBestFit: TAction;
    pmTitle: TPopupMenu;
    mnAscending: TMenuItem;
    mnDescending: TMenuItem;
    mnSep_6: TMenuItem;
    mnBestFit: TMenuItem;
    tbExit: TToolButton;
    procedure GridClipClick(Sender: TObject; X, Y: Integer;
      var Show: Boolean);
    procedure sbInfoDrawPanel(StatusBar: TStatusBar; Panel: TStatusPanel;
      const Rect: TRect);
    procedure acHintsExecute(Sender: TObject);
    procedure GridLoadData(Sender: TObject);
    procedure GridSaveData(Sender: TObject);
    procedure acApplyExecute(Sender: TObject);
    procedure GridError(ErrorCode: Integer; P: Pointer);
    procedure acSaveExitExecute(Sender: TObject);
    procedure acDeleteExecute(Sender: TObject);
    procedure acNewExecute(Sender: TObject);
    procedure acEditExecute(Sender: TObject);
    procedure acSelectAllExecute(Sender: TObject);
    procedure GridInitData(Sender: TObject; ColumnData: TColumnData;
      var RecordItem: TRecordItemData);
    procedure GridGetRecordCode(Sender: TObject; var Code: String);
    procedure FormShow(Sender: TObject);
    procedure acExitExecute(Sender: TObject);
    procedure GridColumnComment(Sender: TObject; Mode: Integer;
      Column: TKnotColumn);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure Button1Click(Sender: TObject);
  private
    FComment: string;
    FJournalName: string;
    FJournalTitle: string;
    FJournalIndex: integer;
    FFilterTitle: string;
    FDatabase: TDatabase;
    FQuery: TQuery;
    FTableName: string;
    FErrorHint: string;
    FErrorCode: integer;
    FSQLQuery: string;
    FMainForm: TCustomForm;
    FBreakThread: boolean;
    FApplied: boolean;
    FSaveData: boolean;
    FIdent: TUIDType;
    procedure SetJournalIndex(const Value: integer);
    procedure SetJournalName(const Value: string);
    function GetHintsState: boolean;
    procedure SetHintsState(const Value: boolean);
    procedure SetDatabase(const Value: TDatabase);
    procedure SetJournalTitle(const Value: string);
    procedure SetFilterTitle(const Value: string);
    function InitDatabase(Mode: TJournalState): integer;
    function FormatError(Text: PChar): string;
    function GetTableName: string;
    procedure DatabaseAction(Action: TDatabaseAction);
    procedure SetComment(const Value: string);
  protected
    procedure InitJournal; virtual;
    procedure DoShow; override;
    procedure DoClose(var Action: TCloseAction); override;
    procedure GSUpdateState(var Message: TMessage); message GS_UPDATE_STATE;
    procedure GSUpdateRecordCount(var Message: TMessage); message GS_UPDATE_RECORDCOUNT;
    procedure GSUserFormFree(var Message: TMessage); message SG_USERFORM_FREE;
    procedure Load; dynamic;
    procedure Save; dynamic;
  public
    constructor CreateJournal(AOwner: TComponent; ADatabase: TDatabase;
      ATitle: string; AInitGrid: boolean = True; AVisible: boolean = True);
    constructor Create(AOwner: TComponent); override;
    procedure InitGrid; virtual;
    procedure ShowJournal; virtual;
    procedure UpdatePanelInfo(AIndex: integer);
    procedure UpdateFilterInfo;
    procedure RefreshGrid;
    procedure SetErrorHint(ErrorHint: string);
    procedure DefineIdent(AFieldName: string; ADataType: TDetailDataType);
    property JournalName: string read FJournalName write SetJournalName;
    property JournalTitle: string read FJournalTitle write SetJournalTitle;
    property JournalIndex: integer read FJournalIndex write SetJournalIndex;
    property FilterTitle: string read FFilterTitle write SetFilterTitle;
    property Database: TDatabase read FDatabase write SetDatabase;
    property TableName: string read GetTableName write FTableName;
    property HintsPanelVisible: boolean read GetHintsState write SetHintsState;
    property Query: TQuery read FQuery;
    property ErrorCode: integer read FErrorCode;
    property SQLQuery: string read FSQLQuery write FSQLQuery;
    property Ident: TUIDType read FIdent;
    property Comment: string read FComment write SetComment;
  end;

var
  SimpleJournalForm: TSimpleJournalForm;

implementation

uses
  Main, DMResource, DCEditButton, DCResource;

const
  SGrid_KnotName     = 'K';
  SGrid_RecordCode   = 'C';

  { }
  SGrid_pnInfo_Count   = 0;
  SGrid_pnInfo_State   = 1;
  SGrid_pnInfo_Comment = 2;

  SGrid_InfoFmt_Count    = '/f{%s}%d %s';
  SGrid_InfoFmt_StLoad   = '/f{%s}/im{56}/ow{3} ';
  SGrid_InfoFmt_StSave   = '/f{%s}/im{57}/ow{3} ';
  SGrid_InfoFmt_StBrowse = '';
  SGrid_InfoFmt_StEdit   = '/f{%s}/ow{2}';
  SGrid_InfoFmt_StView   = '/f{%s}/im{54}/ow{1}';
  SGrid_InfoFmt_Comment  = '/f{%s}%s';

  { }
  SGrid_Fmt_ErrEdit = '/ip{27}/ow{4}';
  SGrid_Fmt_ErrInit = '/ip{55}/ow{4}';
  SGrid_Fmt_ErrData = '/ip{58}/ow{4}';

  SGrid_Fmt_Err_DB  = SGrid_Fmt_ErrInit+'  ';
  SGrid_Fmt_Err_QE  = SGrid_Fmt_ErrData+'   ';
  SGrid_Fmt_Err_OI  = SGrid_Fmt_ErrData+'   ';
  SGrid_Fmt_Err_OU  = SGrid_Fmt_ErrData+'   ';
  SGrid_Fmt_Err_OD  = SGrid_Fmt_ErrData+'   ';

  SGrid_Fmt_Dat_IN  = ' ';
  SGrid_Fmt_Dat_UN  = '  ';

{$R *.DFM}

{ TSimpleJournalForm }

function TSimpleJournalForm.GetHintsState: boolean;
begin
  Result := acHints.Checked;
end;

procedure TSimpleJournalForm.SetHintsState(const Value: boolean);
begin
  acHints.Checked   := Value;
  pnHints.Visible := Value;
end;

procedure TSimpleJournalForm.SetJournalIndex(const Value: integer);
 var
  Button: TDCEditButton;
begin
  if Value <> FJournalIndex then
  begin
    FJournalIndex := Value;
    Button := pnGridInfo.Buttons.FindButton('$Image$');
    if Assigned(Button) then
    begin
      Button.Text := Format('/ip{%d}',[FJournalIndex]);
      Button.Paint;
    end;
  end;
end;

procedure TSimpleJournalForm.SetJournalName(const Value: string);
begin
  FJournalName := Value;
  Self.Caption := Format('%s', [FJournalName]);
end;

procedure TSimpleJournalForm.SetDatabase(const Value: TDatabase);
begin
  FDatabase := Value;
end;

procedure TSimpleJournalForm.GridClipClick(Sender: TObject; X, Y: Integer;
  var Show: Boolean);
begin
  with (Sender as TKNClipPopup) do
  begin
    Clear;
    AddButton('#Query'   , 'DC_DBQUERY'   , LoadStr(RES_STRN_VAL_QUERY), 0, 0);
    AddButton('#Property', 'DC_DBPROPERTY', LoadStr(RES_STRN_VAL_PROP) , 0, 1);
    AddButton('#Find'    , 'DC_DBFIND'    , LoadStr(RES_STRN_VAL_FIND) , 0, 2);
    AddButton('#Print'   , 'DC_PRINT'     , LoadStr(RES_STRN_VAL_PRINT), 0, 3);
  end;
end;

procedure TSimpleJournalForm.UpdatePanelInfo(AIndex: integer);
 var
  PanelRect: TRect;
begin
  with sbInfo.Panels[AIndex] do
  begin
    SendMessage(sbInfo.Handle, SB_GETRECT, Index, Integer(@PanelRect));
    ValidateRect(sbInfo.Handle, @PanelRect);
    InvalidateRect(sbInfo.Handle, @PanelRect, True);
  end;
end;

procedure TSimpleJournalForm.GSUpdateRecordCount(var Message: TMessage);
begin
  UpdatePanelInfo(SGrid_pnInfo_Count);
end;

procedure TSimpleJournalForm.GSUpdateState(var Message: TMessage);
begin
  UpdatePanelInfo(SGrid_pnInfo_State);
end;

procedure TSimpleJournalForm.sbInfoDrawPanel(StatusBar: TStatusBar;
  Panel: TStatusPanel; const Rect: TRect);
 var
  R: TRect;
  nCount: integer;
  sText: string;
begin
  R     := Rect;
  R.Top := R.Top  + 1;
  case Panel.Index of
    SGrid_pnInfo_Count:
      begin
        R.Left := R.Left + 4;
        if (Grid.State = jsBrowse) or (Grid.State = jsView) then
        begin
          nCount := Grid.Knots.Count;
          DrawHighLightText(StatusBar.Canvas,
            PChar(Format(SGrid_InfoFmt_Count, [Font.Name, nCount,
                  RecordCount2Str(nCount)])),
            R, 1, DT_END_ELLIPSIS, DMResourceForm.imSmallImages);
        end
        else begin
          DrawHighLightText(StatusBar.Canvas,
            PChar('Loading...'), R, 1, DT_END_ELLIPSIS, DMResourceForm.imSmallImages);
        end;
      end;
    SGrid_pnInfo_State:
      begin
        R.Left := R.Left + 2;
        case Grid.State of
          jsLoad  : sText := SGrid_InfoFmt_StLoad;
          jsSave  : sText := SGrid_InfoFmt_StSave;
          jsBrowse: sText := SGrid_InfoFmt_StBrowse;
          jsEdit  : sText := SGrid_InfoFmt_StEdit;
          jsView  : sText := SGrid_InfoFmt_StView;
        end;
        DrawHighLightText(StatusBar.Canvas,
          PChar(Format(sText,[Font.Name])),
          R, 1,DT_END_ELLIPSIS, DMResourceForm.imSmallImages);
      end;
    SGrid_pnInfo_Comment:
      begin
        R.Left := R.Left + 2;
        DrawHighLightText(StatusBar.Canvas,
          PChar(Format(SGrid_InfoFmt_Comment,[Font.Name, FComment])),
          R, 1, DT_END_ELLIPSIS, DMResourceForm.imSmallImages);
      end;
  end;
end;

procedure TSimpleJournalForm.GridLoadData(Sender: TObject);
 var
  DataLoaded: boolean;
  i: integer;
begin
  DataLoaded := False;
  repeat
    with Grid, Grid.Knots do
    begin
      SavePosition;
      BeginUpdate(True);
      try
        Clear;
        Perform(GS_ERRORCODE, 0, ERR_EDIT_NONE);
        if InitDatabase(Self.Grid.State) = ERR_EDIT_NONE then
        begin
          FQuery := TQuery.Create(nil);
          FQuery.DatabaseName := FDatabase.DatabaseName;
          FQuery.SQL.Text := FSQLQuery;
          try
            FQuery.Open;
            with FQuery do
            begin
              I := 0;
              while not(Eof or FBreakThread) do
              begin
                Application.ProcessMessages;
                Add(SGrid_KnotName);
                Next;
                //if i = 10 then Break;
                Inc(i);
              end;
            end;
            DataLoaded := True;
          except
            on E: Exception do
            begin
              Perform(GS_ERRORCODE, Integer(E.Message), SGRid_ErrData_QueryExec);
              FBreakThread    := True;
              Self.Grid.State := jsView;
            end;
          end;
          FQuery.Free;
        end
        else
          FBreakThread := True;
      finally
        EndUpdate;
        RestPosition;
      end;
    end;
  until DataLoaded or FBreakThread;
  FBreakThread := False;
  Perform(GS_UPDATE_RECORDCOUNT, 0, 0);
end;

procedure TSimpleJournalForm.InitJournal;
begin
  JournalIndex   := 9;
  JournalName    := 'JournalName';
end;

constructor TSimpleJournalForm.CreateJournal(AOwner: TComponent;
  ADatabase: TDatabase; ATitle: string; AInitGrid: boolean = True;
  AVisible: boolean = True);
begin
  Create(AOwner);
  FMainForm     := TCustomForm(AOwner);
  FDatabase     := ADatabase;
  JournalTitle  := ATitle;
  InitJournal;
  if AInitGrid then InitGrid;
  if AVisible  then ShowJournal;
end;

constructor TSimpleJournalForm.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  sbInfo.DoubleBuffered := True;
  with pnGridInfo.Buttons do
  begin
    with AddButton do
    begin
      Name  := '$Image$';
      Images:= pnGridInfo.Images;
      Font  := Self.Font;
      SetBounds(Rect(pnGridInfo.Width-22, 1, 18, pnGridInfo.Height-2));
      Text  := '/c{clBlack}Image';
      Style := stShadowFlat;
      AbsolutePos  := False;
      DisableStyle := deNone;
      BrushColor   := pnGridInfo.Color;
      AnchorStyle  := asTR;
      Allignment   := abImageTop;
      EventStyle   := esDropDown;
      Font         := pnGridInfo.Font;
      Enabled      := False;
    end;
    UpdateFilterInfo;
  end;

  HintsPanelVisible:= True;
  Grid.Perform(GS_ERRORCODE, 0, ERR_EDIT_NONE);

  FBreakThread := False;
  FApplied     := False;
  FTableName   := '';
  FSaveData    := True;
  FComment     := '';

end;

procedure TSimpleJournalForm.UpdateFilterInfo;
 var
  Button: TDCEditButton;
  P: TPoint;
  sText: string;
begin
  with pnGridInfo.Buttons do
  begin
    Button := FindButton('$Filter$');
    if not Assigned(Button) then
    begin
      Button := AddButton;
      with Button do begin
        Name  := '$Filter$';
        Images:= pnGridInfo.Images;
        Font  := Self.Font;
        Style := stShadowFlat;
        AbsolutePos  := False;
        DisableStyle := deNone;
        BrushColor   := pnGridInfo.Color;
        AnchorStyle  := asTR;
        Font         := pnGridInfo.Font;
        Enabled      := False;
      end;
    end;
    with Button do
    begin
      if FFilterTitle <> '' then
      begin
        sText := Format('/b0%s',[FFilterTitle]);
        Visible  := True;
        P := DrawHighLightText(Canvas, PChar(sText),
              Rect(0,0,ClientWidth, ClientHeight), 0,  DT_END_ELLIPSIS, Images);
        P.X := P.X + 6;
        SetBounds(Rect(FindButton('$Image$').Left-P.X, 1, P.X, 18));
        Text := sText;
      end
      else
        Visible  := False;
    end;
    if Button.Visible then
      pnGridInfo.SetMargins(-1, -1, pnGridInfo.Width-FindButton('$Filter$').Left, -1)
    else
      pnGridInfo.SetMargins(-1, -1, pnGridInfo.Width-FindButton('$Image$').Left, -1)
  end;
end;

procedure TSimpleJournalForm.SetJournalTitle(const Value: string);
begin
  FJournalTitle := Value;
  pnGridInfo.Caption := FJournalTitle;
end;

procedure TSimpleJournalForm.acHintsExecute(Sender: TObject);
begin
   HintsPanelVisible := not acHints.Checked;
end;

procedure TSimpleJournalForm.SetFilterTitle(const Value: string);
begin
  FFilterTitle := Value;
  UpdateFilterInfo;
end;

procedure TSimpleJournalForm.InitGrid;
begin
  {  Grid'}
  with Grid do
  begin
    UpdateGridColumns;
  end;
  DefineIdent('Ident', ddInteger);
end;

procedure TSimpleJournalForm.ShowJournal;
begin
  { }
  Show;
  //SetWindowPos(Handle, HWND_TOPMOST, 0, 0, 0, 0, SWP_NOMOVE or SWP_NOSIZE);
  ProcessPaintMessages;
  Load;
end;

procedure TSimpleJournalForm.RefreshGrid;
begin
  { }
  if FApplied then Exit;
  FApplied := True;
  Grid.Perform(GS_ERRORCODE, 0, ERR_EDIT_NONE);
  with Grid do
  begin
    if not(State in [jsBrowse, jsView]) then
    begin
      FApplied := False;
      Exit;
    end;
    SelectedRows.Clear;
    Self.Save;
    if FErrorCode = ERR_EDIT_NONE then Load;
  end;
  FApplied := False;
end;

procedure TSimpleJournalForm.GridSaveData(Sender: TObject);
 var
  DataSaved: boolean;

  procedure ProcessDeleteRecords;
   var
    i: integer;
  begin
    i := 0;
    while not(FBreakThread or (Grid.Deleted.Count <= i) or
              (FErrorCode <> ERR_EDIT_NONE)) do
    begin
      with FQuery do
      begin
        SQL.Clear;
        SQL.Add(Format('delete from %s where %s = %s',
          [TableName, FIdent.Name, Grid.Deleted.Strings[i]]));
        Prepare;
        try
          ExecSQL;
        except
          on E: Exception do
          begin
            Grid.Perform(GS_ERRORCODE, Integer(E.Message), SGRid_ErrData_OnDelete);
            FBreakThread    := True;
          end;
        end;
      end;
      inc(i);
    end;
  end;

  function SetDelimiter(APos, ACount: integer): string;
  begin
    if APos < ACount-1 then
      Result := ', '
    else
      Result := '';
  end;

  procedure ProcessUpdateRecords;
   var
    i: integer;
    KnotItem: TKnotItem;
    pRecordData: PRecordData_tag;

    procedure UpdateRecord;
     var
      j: integer;
      pColumnData: PColumnData_tag;
      FirstValue: boolean;
      ASetFormat: string;
    begin
      with FQuery do
      begin
        SQL.Clear;
        SQL.Add(Format('update %s set',[TableName]));
        FirstValue := True;
        for j:= 0 to Grid.ColumnsData.Count-1 do
        begin
          pColumnData := Grid.ColumnsData.Items[j];
          if DIGetFlag(PRecordData_tag(KnotItem.Data)^.Data[pColumnData^.DataIndex], DFLAG_CHANGED) <> 0 then
          begin
            if FirstValue then
            begin
              ASetFormat := '   %s = %s';
              FirstValue := False;
            end
            else
              ASetFormat := '  ,%s = %s';

            SQL.Add(Format(ASetFormat,
              [pColumnData^.FieldName,
               Grid.GetDataItem(KnotItem, pColumnData^, nil, True)]));
          end;
        end;
        SQL.Add(Format('where  %s = %s' , [FIdent.Name, pRecordData^.Code]));
        try
          Prepare;
          ExecSQL;
        except
          on E: Exception do
          begin
            Grid.Perform(GS_ERRORCODE, Integer(E.Message), SGRid_ErrData_OnUpdate);
            FBreakThread    := True;
          end;
        end;
      end;
    end;

    procedure InsertRecord;
     var
      j: integer;
      pColumnData: PColumnData_tag;
    begin
      with FQuery do
      begin
        SQL.Clear;
        SQL.Add(Format('insert into %s',[TableName]));
        SQL.Add('       (');
        SQL.Add('values (');
        for j:= 0 to Grid.ColumnsData.Count-1 do
        begin
          pColumnData := Grid.ColumnsData.Items[j];
          SQL.Strings[1] := SQL.Strings[1] +
            Format('%s%s',[pColumnData^.FieldName,
               SetDelimiter(j, Grid.ColumnsData.Count)]);
          SQL.Strings[2] := SQL.Strings[2] +
            Format('%s%s',[Grid.GetDataItem(KnotItem, pColumnData^, nil, True),
               SetDelimiter(j, Grid.ColumnsData.Count)]);
        end;
        SQL.Strings[1] := SQL.Strings[1] + ')';
        SQL.Strings[2] := SQL.Strings[2] + ')';
        try
          Prepare;
          ExecSQL;
        except
          on E: Exception do
          begin
            Grid.Perform(GS_ERRORCODE, Integer(E.Message), SGRid_ErrData_OnInsert);
            FBreakThread    := True;
          end;
        end;
      end;
    end;

  begin
    i := 0;
    while not(FBreakThread or (Grid.Knots.Count <= i) or
              (FErrorCode <> ERR_EDIT_NONE)) do
    begin
      KnotItem := Grid.Knots.Items[i];
      pRecordData := KnotItem.Data;
      case pRecordData.State of
        rsInserted  : InsertRecord;
        rsUpdated   : UpdateRecord;
        rsNotChanged:;
      end;
      inc(i);
    end;
  end;

begin
  DataSaved := False;
  with Grid do
    if (Knots.State = ksInsert) then
       if RowModified then
       begin
         if not ValidRecord(SelectedKnot) then  Exit;
       end
       else begin
         Knots.Delete(SelectedKnot);
         Knots.SetState(ksBrowse);
       end;
  repeat
    Grid.Perform(GS_ERRORCODE, 0, ERR_EDIT_NONE);
    if InitDatabase(Grid.State) = ERR_EDIT_NONE then
    begin
      FQuery := TQuery.Create(nil);
      FQuery.DatabaseName := FDatabase.DatabaseName;

      ProcessDeleteRecords;
      ProcessUpdateRecords;

      if (FErrorCode = ERR_EDIT_NONE) and not FBreakThread then
        DatabaseAction(daCommit)
      else
        DatabaseAction(daRollback);

      DataSaved := True;
      FQuery.Free;
    end;
  until DataSaved or FBreakThread or (FErrorCode <> ERR_EDIT_NONE);
  FBreakThread := False;
  Perform(GS_UPDATE_RECORDCOUNT, 0, 0);
end;

procedure TSimpleJournalForm.acApplyExecute(Sender: TObject);
begin
  RefreshGrid;
end;

procedure TSimpleJournalForm.GridError(ErrorCode: Integer; P: Pointer);
  function GetErrorMessageFmt(const Error: string): string;
  begin
    if  Assigned(P) then
      Result := Format('%s'#10'%s',[Error, FormatError(PChar(P))])
    else
      Result := Error;
  end;
begin
  FErrorCode := ErrorCode;
  case FErrorCode of
    ERR_EDIT_NONE          :
      FErrorHint := '';
    ERR_EDIT_EMPTYVALUE    :
      FErrorHint := Format(LoadStr(RES_GRID_ERR_EMPTY),
        [PColumnData_tag(P)^.Caption]);
    ERR_EDIT_INCORRECTDEC  :
      FErrorHint := Format(LoadStr(RES_GRID_ERR_DEC),
        [PColumnData_tag(P)^.Caption]);
    ERR_EDIT_INCORRECTCURR :
      FErrorHint := Format(LoadStr(RES_GRID_ERR_CURR),
        [PColumnData_tag(P)^.Caption]);
    ERR_EDIT_INCORRECTFLOAT:
      FErrorHint := Format(LoadStr(RES_GRID_ERR_FLOAT),
        [PColumnData_tag(P)^.Caption]);
    ERR_EDIT_NEEDUNIQ      :
      FErrorHint := Format(LoadStr(RES_GRID_ERR_UNIQ),
        [PColumnData_tag(P)^.Caption]);
    SGRid_ErrData_Database :
      FErrorHint := GetErrorMessageFmt(SGrid_Fmt_Err_DB);
    SGRid_ErrData_QueryExec:
      FErrorHint := GetErrorMessageFmt(SGrid_Fmt_Err_QE);
    SGRid_ErrData_OnInsert :
      FErrorHint := GetErrorMessageFmt(SGrid_Fmt_Err_OI);
    SGRid_ErrData_OnUpdate :
      FErrorHint := GetErrorMessageFmt(SGrid_Fmt_Err_OU);
    SGRid_ErrData_OnDelete :
      FErrorHint := GetErrorMessageFmt(SGrid_Fmt_Err_OD);
  end;
  SetErrorHint(FErrorHint);
end;

procedure TSimpleJournalForm.SetErrorHint(ErrorHint: string);
 var
  P: TPoint;
begin
  if not(csDestroying in ComponentState) then
  begin
    if ErrorHint = '' then
      P := DrawHighLightText(Canvas, 'Wg', Rect(0, 0, Height, Width),
             0, DT_END_ELLIPSIS, DMResourceForm.imSmallImages)
    else begin
      P := DrawHighLightText(Canvas, PChar(ErrorHint), Rect(0, 0, Height, Width),
             0, DT_END_ELLIPSIS, DMResourceForm.imSmallImages);
    end;

    pnHints.Height := P.Y + 4;
    pnHints.Caption := ErrorHint;
  end;
end;

procedure TSimpleJournalForm.acSaveExitExecute(Sender: TObject);
begin
  Close;
end;

procedure TSimpleJournalForm.DoClose(var Action: TCloseAction);
begin
  inherited DoClose(Action);
  if FSaveData then Save;
  if FErrorCode <> ERR_EDIT_NONE then
  begin
    if Action = caFree then
      FMainForm.Perform(SG_USERFORM_FREE, Integer(Self), MODE_SHOWWINDOW);
    Action := caNone;
  end
  else
    PostMessage(FMainForm.Handle, SG_USERFORM_FREE, Integer(Self), MODE_FREEWINDOW);
end;

procedure TSimpleJournalForm.Load;
begin
//  Grid.Load;
   TLoadThread.CreateThread(Grid, Self);
end;

procedure TSimpleJournalForm.Save;
begin
  with Grid do
  begin
    case State of
      jsBrowse:
        begin
          Grid.Perform(GS_ERRORCODE, 0, ERR_EDIT_NONE);
          Save;
        end;
      jsView  : Perform(GS_ERRORCODE, 0, ERR_EDIT_NONE);
      jsEdit  :
        begin
          if ValidEditValue and ValidRecord(SelectedKnot) then
          begin
            Knots.SetState(ksBrowse);
            Save;
          end;
        end;
    end;
  end;
end;

procedure TSimpleJournalForm.DoShow;
begin
  inherited;
  FMainForm.Perform(SG_USERFORM_SHOW, Integer(Self), 0);
end;

procedure TSimpleJournalForm.acDeleteExecute(Sender: TObject);
begin
  Grid.DeleteDataItem;
end;

procedure TSimpleJournalForm.acNewExecute(Sender: TObject);
begin
  Grid.InsertDataItem;
end;

procedure TSimpleJournalForm.acEditExecute(Sender: TObject);
begin
  if Grid.State = jsBrowse then Grid.ShowEditor;
end;

procedure TSimpleJournalForm.acSelectAllExecute(Sender: TObject);
begin
  Grid.SelectAll;
end;

function TSimpleJournalForm.InitDatabase(Mode: TJournalState): integer;
begin
  if Assigned(FDatabase) then
  begin
    if not FDatabase.Connected then DatabaseAction(daOpen);
    if FErrorCode = ERR_EDIT_NONE then
    begin
      case Mode of
        jsLoad:;
        jsSave:
          begin
            if FDatabase.InTransaction then
              GridError(SGRid_ErrData_Database,PChar(SGrid_Fmt_Dat_IN))
            else
              DatabaseAction(daStartTransaction);
          end;
      end;
    end;
  end
  else
    GridError(SGRid_ErrData_Database, PChar(SGrid_Fmt_Dat_UN));

  Result := FErrorCode;
end;

function TSimpleJournalForm.FormatError(Text: PChar): string;
 var
  FirstChar: boolean;
begin
  Result := '';
  FirstChar := True;
  while Text^ <> #0 do
  begin
    if FirstChar then
    begin
      Result    := Result + '/ow{20}/{';
      FirstChar := False;
    end;
    case Text^ of
      #13:
        begin
          FirstChar := True;
          Result := Result + '/}'#10;
        end;
      #10:;
      else
        Result := Result + Text^;
    end;
    Inc(Text);
  end;
end;

procedure TSimpleJournalForm.GridInitData(Sender: TObject;
  ColumnData: TColumnData; var RecordItem: TRecordItemData);
begin
  if Grid.State = jsLoad then
    with ColumnData do
    begin
      case DataType of
        ddInteger:
          DISetValue(RecordItem, daInteger, FQuery.FieldByName(FieldName).AsString);
        ddDate, ddFloat, ddCurrency:
          DISetValue(RecordItem, daFLoat, FQuery.FieldByName(FieldName).AsString);
        ddString:
          DISetValue(RecordItem, daString, FQuery.FieldByName(FieldName).AsString);
      end;
    end
  else begin
    {}
  end;
end;

function TSimpleJournalForm.GetTableName: string;
 const
  FromSubstring = 'from ';
 var
  FromPos: integer;
  pSQLQuery: PChar;
begin
  if FTableName <> '' then
    Result := FTableName
  else begin
    FromPos := Pos(AnsiUpperCase(FromSubstring), AnsiUpperCase(FSQLQuery));
    if FromPos > 0 then
    begin
      pSQLQuery := PChar(FSQLQuery);
      Result    := '';
      Inc(pSQLQuery, FromPos+Length(FromSubstring)-1);
      while not(pSQLQuery^ in [#0,' ',#10,#13]) do
      begin
        Result := Result + pSQLQuery^;
        Inc(pSQLQuery);
      end;
      FTableName := Result;
    end
    else
      Result := '';
  end;
end;

procedure TSimpleJournalForm.GridGetRecordCode(Sender: TObject;
  var Code: String);
begin
  if Grid.State = jsLoad then
  begin
    case FIdent.DataType of
      ddString:
        Code := Format('"%s"',[FQuery.FieldByName(FIdent.Name).AsString ]);
      else
        Code := FQuery.FieldByName(FIdent.Name).AsString
    end;
  end
  else
    Code := SGrid_RecordCode;
end;

procedure TSimpleJournalForm.DatabaseAction(Action: TDatabaseAction);
begin
  try
    case Action of
      daCommit:
        FDatabase.Commit;
      daRollback:
        FDatabase.Rollback;
      daStartTransaction:
        FDatabase.StartTransaction;
      daOpen:
        FDatabase.Open;
     end;
  except
    on E: Exception do
    begin
      Grid.Perform(GS_ERRORCODE, Integer(E.Message), SGRid_ErrData_Database);
      Grid.State := jsView;
    end
  end;
end;

procedure TSimpleJournalForm.DefineIdent(AFieldName: string;
  ADataType: TDetailDataType);
begin
  with FIdent do
  begin
    Name := AFieldName;
    DataType := ADataType;
  end;
end;

procedure TSimpleJournalForm.FormShow(Sender: TObject);
begin
  FSaveData := True;
end;

procedure TSimpleJournalForm.acExitExecute(Sender: TObject);
begin
  Grid.Perform(GS_ERRORCODE, 0, ERR_EDIT_NONE);
  FSaveData := False;
  Close;
end;

procedure TSimpleJournalForm.GSUserFormFree(var Message: TMessage);
begin
  acSaveExit.Execute;
end;

procedure TSimpleJournalForm.GridColumnComment(Sender: TObject;
  Mode: Integer; Column: TKnotColumn);
begin
  if Column <> nil then
    FComment := Column.Comment
  else
    FComment := '';
  UpdatePanelInfo(SGrid_pnInfo_Comment);
end;

procedure TSimpleJournalForm.SetComment(const Value: string);
begin
  FComment := Value;
  UpdatePanelInfo(SGrid_pnInfo_Comment);
end;

{ TLoadThread }

class procedure TLoadThread.CreateThread(AGrid: TDCStringGrid;
  AOwner: TComponent);
begin
  with TLoadThread.Create(True) do
  begin
    FOwner := AOwner;
    FGrid  := AGrid;
    Resume;
  end;
end;

destructor TLoadThread.Destroy;
begin
  inherited;
end;

procedure TLoadThread.Execute;
begin
  try
    Synchronize(FGrid.Load);
  finally
  end;
end;

procedure TSimpleJournalForm.FormCloseQuery(Sender: TObject;
  var CanClose: Boolean);
begin
  if Grid.Knots.Updating then CanClose := False;
end;

procedure TSimpleJournalForm.Button1Click(Sender: TObject);
begin
  SetScrollRange(Grid.Handle, SB_BOTH, 0, 0, True);
  Grid.Invalidate;
end;

end.
