unit Main;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  Menus, ImgList, ActnList, ToolWin, ComCtrls, Buttons, ExtCtrls,
  DCIniStream, DCRecordStream, CommCtrl, StdCtrls, DCStdCtrls, DCChoice,
  RegConsts, Clipbrd;

type
  TPrivateRegKeyFile = class(TRegKeyFile)
  end;

  PKeyDataInfo_tag = ^TKeyDataInfo;
  TKeyDataInfo = packed record
    Changed   : boolean;
    Loaded    : boolean;
    HDataKey  : HIniKey;
    HRootKey  : HIniRootKey;
    KeyPath   : string;
    Flags     : WORD;
    Name      : TKeyName;
    Data      : Longint;
    NumSubKeys: WORD;
    NumValues : WORD;
  end;

  TMainForm = class(TForm)
    sbInfo: TStatusBar;
    tbMainMenu: TToolBar;
    ActionList: TActionList;
    imImages: TImageList;
    pmMain: TPopupMenu;
    mnMain_Reestr: TMenuItem;
    mnMain_Edit: TMenuItem;
    mnMain_Help: TMenuItem;
    acImport: TAction;
    acExport: TAction;
    acPrint: TAction;
    acExit: TAction;
    acAddBranch: TAction;
    acAddString: TAction;
    acAddBinary: TAction;
    acEdit: TAction;
    acDelete: TAction;
    acRename: TAction;
    acExpand: TAction;
    acFind: TAction;
    acRefresh: TAction;
    acAbout: TAction;
    acCopy2Clip: TAction;
    mnExport: TMenuItem;
    mnImport: TMenuItem;
    mnSep_1: TMenuItem;
    mnPrint: TMenuItem;
    mnSep_2: TMenuItem;
    mnExit: TMenuItem;
    mnEdit: TMenuItem;
    mnCreate: TMenuItem;
    mnSep_3: TMenuItem;
    mnDelete: TMenuItem;
    mnRename: TMenuItem;
    mnSep_4: TMenuItem;
    mnCopy2Clip: TMenuItem;
    mnSep_5: TMenuItem;
    mnFind: TMenuItem;
    mnRefresh: TMenuItem;
    mnSep_6: TMenuItem;
    mnAbout: TMenuItem;
    mnAddBranch: TMenuItem;
    mnSep_7: TMenuItem;
    mnAddString: TMenuItem;
    mnAddBinary: TMenuItem;
    acAddInteger: TAction;
    mnAddInteger: TMenuItem;
    tbMain_Reestr: TToolButton;
    tbMain_Edit: TToolButton;
    tbMain_Help: TToolButton;
    pmPopup: TPopupMenu;
    tbButtons: TToolBar;
    tbSep_1: TToolButton;
    acOpen: TAction;
    mnOpen: TMenuItem;
    mnSep_8: TMenuItem;
    tbOpen: TToolButton;
    tbSep_2: TToolButton;
    tbDropAdd: TToolButton;
    pnLeft: TPanel;
    Splitter: TSplitter;
    pnClient: TPanel;
    pnTree: TPanel;
    pnValues: TPanel;
    ListView: TListView;
    pmFind: TMenuItem;
    pmSep_2: TMenuItem;
    pmDelete: TMenuItem;
    pmRename: TMenuItem;
    pmSep_3: TMenuItem;
    pmCopy2Clip: TMenuItem;
    acLink: TAction;
    mnLink: TMenuItem;
    pmEdit: TMenuItem;
    pmSep_4: TMenuItem;
    pmCreate: TMenuItem;
    pmAddBranch: TMenuItem;
    pmDropAdd: TPopupMenu;
    pmExpand: TMenuItem;
    pmAddString: TMenuItem;
    pmAddBinary: TMenuItem;
    pmAddInteger: TMenuItem;
    TreeView: TTreeView;
    rbExit: TToolButton;
    tbAdd: TToolBar;
    procedure acExitExecute(Sender: TObject);
    procedure acAboutExecute(Sender: TObject);
    procedure acLinkExecute(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure acOpenExecute(Sender: TObject);
    procedure TreeViewExpanding(Sender: TObject; Node: TTreeNode;
      var AllowExpansion: Boolean);
    procedure acAddBranchExecute(Sender: TObject);
    procedure pmPopupPopup(Sender: TObject);
    procedure pmAddBranchClick(Sender: TObject);
    procedure TreeViewChange(Sender: TObject; Node: TTreeNode);
    procedure ListViewSelectItem(Sender: TObject; Item: TListItem;
      Selected: Boolean);
    procedure ListViewExit(Sender: TObject);
    procedure acExpandExecute(Sender: TObject);
    procedure sbInfoDrawPanel(StatusBar: TStatusBar; Panel: TStatusPanel;
      const Rect: TRect);
    procedure acDeleteExecute(Sender: TObject);
    procedure pmDeleteClick(Sender: TObject);
    procedure TreeViewKeyUp(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure FormDestroy(Sender: TObject);
    procedure TreeViewChanging(Sender: TObject; Node: TTreeNode;
      var AllowChange: Boolean);
    procedure acAddStringExecute(Sender: TObject);
    procedure acAddBinaryExecute(Sender: TObject);
    procedure acAddIntegerExecute(Sender: TObject);
    procedure pmAddStringClick(Sender: TObject);
    procedure acEditExecute(Sender: TObject);
    procedure pmAddIntegerClick(Sender: TObject);
    procedure ListViewDblClick(Sender: TObject);
    procedure pmCopy2ClipClick(Sender: TObject);
    procedure acCopy2ClipExecute(Sender: TObject);
    procedure ToolButton1Click(Sender: TObject);
  private
    { Private declarations }
    FRunMode: WORD;
    FTitle: string;
    FActive: boolean;
    FIniStream: TPrivateRegKeyFile;
    FValueList: TValueList;
    FFocusedNode: TTreeNode;
    procedure SetTitle(const Value: string);
    procedure CloseKeyStream;
    procedure LoadSubKeys(KeyInfo: TIniKeyType; TreeNode: TTreeNode);
    function LoadKeyInfo(DataKey: HIniKey; TreeNode: TTreeNode): TTreeNode;
    function GetKeyPath(KeyPath: string; Value: string): string;
    function AddBranch(AName: string; TreeNode: TTreeNode): DWORD;
    procedure DelBranch(TreeNode: TTreeNode);
    procedure DelValue(ListItem: TListItem);
    procedure SetActive(const Value: boolean);
    procedure LoadKeyValues;
    procedure LoadKeyValue(Index: integer);
    procedure UpdateNodesAction(Flags: integer);
    procedure UpdateItemsAction(Flags: integer);
    procedure ClearValuesData;
    function ValueData2Str(Value: TIniKeyDataType): string;
    procedure Copy2Clipboard(TreeNode: TTreeNode);
    function GetFullPath(KeyData: PKeyDataInfo_tag): string;
  protected
    procedure CMRegFileOpen(var Message: TMessage); message CM_REGFILE_OPEN;
    procedure CMRegFileCreateKey(var Message: TMessage); message CM_REGFILE_CREATEKEY;
    procedure CMRegFileCreateValue(var Message: TMessage); message CM_REGFILE_CREATEVALUE;
  public
    { Public declarations }
    procedure ReadParams;
    procedure UpdatePanelInfo(AIndex: integer);
    property RunMode: WORD read FRunMode write FRunMode;
    property Title: string read FTitle write SetTitle;
    property Active: boolean read FActive write SetActive;
  end;

var
  MainForm: TMainForm;
  ApplicationTitle: string;

implementation

uses About, ShellAPI, DCPopupWindow, DCEditTools, NewKey, StringValue,
  IntegerValue;

{$R *.DFM}
{$R RGImages.res}

const

  { }
  Reg_pnInfo_KeyPath   = 0;

  Reg_InfoFmt_Comment  = '/f{%s}%s';


function BinaryToStr(Data: Pointer; Size: Longint): string;
 var
  i: Longint;
begin
  Result := '';
  i := 0;
  while i < Size do
  begin
    if (i > 0) then Result := Result + ' ';
    Result :=  Result + IntToHex(Integer((PChar(Data)+i)^), 2);
    inc(i, 1);
  end;
end;

function CustomSortProc(Node1, Node2: TTreeNode; Data: Longint): integer; stdcall;
begin
  Result :=  AnsiStrIComp(Pchar(Node1.Text), PChar(Node2.Text))
end;

procedure TMainForm.acExitExecute(Sender: TObject);
begin
  Close;
end;

procedure TMainForm.acAboutExecute(Sender: TObject);
 var
  AboutForm: TAboutForm;
begin
  Application.CreateForm(TAboutForm, AboutForm);
  AboutForm.ShowModal;
  AboutForm.Free;
end;

procedure TMainForm.acLinkExecute(Sender: TObject);
begin
  ShellExecute(Self.Handle, 'open', 'mailto:alexem@decosp.spb.ru?subject=Registry',
    '', '', SW_SHOWNORMAL);
end;

procedure TMainForm.ReadParams;
 var
  ParamValue: string;
  FileName: PChar;
begin
  FRunMode := APPLICATION_MODE_SHOW;
  if ParamCount > 0 then
  begin
    ParamValue := AnsiUpperCase(ParamStr(1));

    if (ParamValue[1] = '-') or (ParamValue[1] = '/') then
    begin
      System.Delete(ParamValue, 1, 1);
      case ParamValue[1] of
        'O': {Open}
          begin

            ParamValue := ParamStr(2);
            GetMem(FileName, Length(ParamValue)+1);
            StrPCopy(FileName, ParamValue);

            PostMessage(Self.Handle, CM_REGFILE_OPEN, Integer(FileName), 0);
          end;
        'E': {Export}
          begin
            FRunMode := APPLICATION_MODE_HIDE;
          end;
        'I': {Import}
          begin
            FRunMode := APPLICATION_MODE_HIDE;
          end;
      end;
    end;

  end;
end;

procedure TMainForm.CMRegFileOpen(var Message: TMessage);
 var
  FileName: string;
  i: integer;
  TreeNode: TTreeNode;
  PKeyData: PKeyDataInfo_tag;
  SKeyData: PKeyDataInfo_tag;
  pKeyInfo: PIniKeyType_tag;
begin
  ProcessPaintMessages;
  Cursor := crHourGlass;

  {  }

  with Message do
  begin
    SetString(FileName, PChar(WParam), StrLen(PChar(WParam)));
    FreeMem(PChar(WParam));
  end;

  i := LastDelimiter('.', FileName);
  if i <> 0 then FileName := Copy(FileName, 1, i-1);

  Title    := ExtractFileName(FileName);
  SetCurrentDirectory(PChar(ExtractFilePath(FileName)));

  CloseKeyStream;

  FIniStream := TPrivateRegKeyFile.Create(FileName);
  Active     := True;

  {  }
  with FIniStream do
  begin
    TreeNode := TreeView.Items.Add(nil, Title);

    New(PKeyData);

    with PKeyData^ do
    begin
      Changed   := False;
      Loaded    := True;
      HDataKey  := 0;
      HRootKey  := 0;
      KeyPath   := '';
      Flags     := PIniKeyType_tag(RootData)^.Flags;
      Name      := '';
      Data      := PIniKeyType_tag(RootData)^.Data;
      NumSubKeys:= PIniKeyType_tag(RootData)^.NumSubKeys;
      NumValues := PIniKeyType_tag(RootData)^.NumValues;
    end;

    with TreeNode do
    begin
      ImageIndex    := IMAGEINDEX_ROOT;
      SelectedIndex := IMAGEINDEX_ROOT;
      Data          := PKeyData;
    end;

    //     
    GetMem(PKeyInfo, SizeOf(TIniKeyType));

    with PIniKeyType_tag(RootData)^ do
      for i := Low(HashTable) to High(HashTable) do
        if HashTable[i] <> 0 then
        begin
          New(SKeyData);
          FIniStream.RecNo := HashTable[i];
          FIniStream.GetKeyInfo(PKeyInfo);

          with SKeyData^ do
          begin
            Changed   := False;
            Loaded    := False;
            HDataKey  := HashTable[i];
            HRootKey  := HashTable[i];
            KeyPath   := '';
            Flags     := PKeyInfo^.Flags;
            Name      := PKeyInfo^.Name;
            Data      := PKeyInfo^.Data;
            NumSubKeys:= PKeyInfo^.NumSubKeys;
            NumValues := PKeyInfo^.NumValues;
          end;

          with TreeView.Items.AddChild(TreeNode, PKeyInfo^.Name) do
          begin
            Data := SKeyData;
            ImageIndex    := IMAGEINDEX_KEYCLOSE;
            SelectedIndex := IMAGEINDEX_KEY0PEN;
            if SKeyData^.NumSubKeys <> 0 then HasChildren := True;
          end;

      end;

    FreeMem(PKeyInfo);

    TreeNode.CustomSort(@CustomSortProc, 0);
    TreeNode.Expand(False);
  end;

  TreeView.Selected := TreeNode;
  Cursor := crDefault;
end;

procedure TMainForm.FormCreate(Sender: TObject);
begin
  Title  := '';
  Active := False;

  FFocusedNode   := nil;
  FValueList     := TValueList.Create;
  acEdit.Enabled := False;
end;

procedure TMainForm.SetTitle(const Value: string);
begin
  FTitle := Value;
  if FTitle <> '' then
  begin
    Application.Title := Format('%s - %s',[ApplicationTitle, FTitle]);
    Application.Icon.Handle := LoadIcon(hInstance, '_FILEICONX16');
  end
  else begin
    Application.Title := ApplicationTitle;
    Application.Icon.Handle := LoadIcon(hInstance, 'MAINICON');
  end;
  Caption := Application.Title;
end;

procedure TMainForm.CloseKeyStream;
 var
  AllowChange: boolean;
begin
  if Active then
  begin
    AllowChange := True;

    TreeViewChanging(Self, TreeView.Selected, AllowChange);

    FValueList.Clear;

    ListView.Items.BeginUpdate;
    ListView.Items.Clear;
    ListView.Items.EndUpdate;

    TreeView.Items.BeginUpdate;
    TreeView.Items.Clear;
    TreeView.Items.EndUpdate;

    FIniStream.Free;
    
    Active := False;
  end;
end;

procedure TMainForm.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  CloseKeyStream;
end;

procedure TMainForm.acOpenExecute(Sender: TObject);
 var
  pFileName: PChar;
  sFileName: string;
begin
  with TOpenDialog.Create(nil) do
  begin
    FileEditStyle := fsEdit;
    Options    := Options - [ofEnableSizing, ofHideReadOnly];
    Filter     := '  (*.key)|*.key';
    DefaultExt := 'key';
    if Execute then
      sFileName := FileName
    else
      sFileName := '';
    Free;
  end;
  if sFileName <> '' then
  begin
    GetMem(pFileName, Length(sFileName)+1);
    StrPCopy(pFileName, sFileName);
    Perform(CM_REGFILE_OPEN, integer(pFileName), 0);
  end;
end;

procedure TMainForm.LoadSubKeys(KeyInfo: TIniKeyType; TreeNode: TTreeNode);
 var
  i: integer;
begin
  with KeyInfo do begin
    for i := Low(HashTable) to High(HashTable) do
    begin
      if HashTable[i] <> 0 then
      begin
        LoadKeyInfo(HashTable[i], TreeNode);
      end;
    end;
  end;
  TreeNode.CustomSort(@CustomSortProc, 0);
  PKeyDataInfo_tag(TreeNode.Data)^.Loaded := True;
end;

function TMainForm.LoadKeyInfo(DataKey: HIniKey; TreeNode: TTreeNode): TTreeNode;
 var
  PKeyInfo: PIniKeyType_tag;
  SKeyData: PKeyDataInfo_tag;
begin
  GetMem(PKeyInfo, SizeOf(TIniKeyType));
  New(SKeyData);
  try
    FIniStream.RecNo := DataKey;
    FIniStream.GetKeyInfo(pKeyInfo);

    with SKeyData^ do
    begin
      Changed   := False;
      Loaded    := False;
      HDataKey  := DataKey;
      HRootKey  := PKeyDataInfo_tag(TreeNode.Data)^.HRootKey;
      KeyPath   := GetKeyPath(PKeyDataInfo_tag(TreeNode.Data)^.KeyPath, PKeyInfo^.Name);
      Flags     := pKeyInfo^.Flags;
      Name      := pKeyInfo^.Name;
      Data      := pKeyInfo^.Data;
      NumSubKeys:= pKeyInfo^.NumSubKeys;
      NumValues := pKeyInfo^.NumValues;
    end;

    Result := TreeView.Items.AddChild(TreeNode, pKeyInfo^.Name);
    with Result do
    begin
      Data := SKeyData;
      ImageIndex    := IMAGEINDEX_KEYCLOSE;
      SelectedIndex := IMAGEINDEX_KEY0PEN;
      if SKeyData^.NumSubKeys <> 0 then HasChildren := True;
    end;

  finally
    FreeMem(PKeyInfo);
  end;
end;

procedure TMainForm.TreeViewExpanding(Sender: TObject; Node: TTreeNode;
  var AllowExpansion: Boolean);
 var
  KeyData: PKeyDataInfo_tag;
  pKeyInfo: PIniKeyType_tag;
begin
  KeyData := Node.Data;
  if not KeyData^.Loaded then
  begin
    GetMem(pKeyInfo, SizeOf(TIniKeyType));
    try
      FIniStream.RecNo := KeyData^.HDataKey;
      FIniStream.GetKeyInfo(pKeyInfo);
      LoadSubKeys(pKeyInfo^, Node);
    finally
      FreeMem(pKeyInfo);
    end;
  end;
end;

function TMainForm.AddBranch(AName: string; TreeNode: TTreeNode): DWORD;
 var
  KeyData: PKeyDataInfo_tag;
  TempKey: HIniKey;
  AllowExpansion: boolean;
begin
  if not Assigned(TreeNode) then
  begin
    Result := ERROR_INVALID_DATA;
    Exit;
  end;

  KeyData := TreeNode.Data;

  if not KeyData^. Loaded then
    TreeViewExpanding(Self, TreeNode, AllowExpansion)
  else
    TreeNode.Expand(False);

  Result := FIniStream.CreateKeyEx(KeyData.HRootKey, GetKeyPath(KeyData^.KeyPath, AName), TempKey);

  if Result = ERROR_SUCCESS then
  begin
    PostMessage(TreeView.Handle, WM_SETFOCUS, ActiveControl.Handle, 0);
    TreeView.Selected := LoadKeyInfo(TempKey, TreeNode);
  end

end;

procedure TMainForm.acAddBranchExecute(Sender: TObject);
begin
  CreateNewKey(Self.Handle, TreeView.Selected);
end;

procedure TMainForm.pmAddBranchClick(Sender: TObject);
begin
  CreateNewKey(Self.Handle, FFocusedNode);
end;

function TMainForm.GetKeyPath(KeyPath, Value: string): string;
begin
  if KeyPath <> '' then
    Result := Format('%s\%s', [KeyPath, Value])
  else
    Result := Value;
end;

procedure TMainForm.pmPopupPopup(Sender: TObject);
 var
  MousePos: TPoint;
  KeyData: PKeyDataInfo_tag;
begin
  GetCursorPos(MousePos);
  if pmPopup.PopupComponent is TTreeView then
  begin
    pmExpand.Visible := True;
    pmEdit.Visible   := False;
    pmSep_4.Visible  := False;

    MousePos := TreeView.ScreenToClient(MousePos);
    with MousePos do
      FFocusedNode := TreeView.GetNodeAt(X, Y);

    if (FFocusedNode <> nil) and FFocusedNode.HasChildren then
    begin
      acExpand.Enabled := True;
      if FFocusedNode.Expanded then
        pmExpand.Caption := POPUP_COLLAPSE_CAPTION
      else
        pmExpand.Caption := POPUP_EXPAND_CAPTION;
     end
     else begin
      pmExpand.Visible := False;
      acExpand.Enabled := False;
    end;

    if FFocusedNode <> nil then
    begin
      KeyData := FFocusedNode.Data;

      if KeyData.Flags and INIKEY_FLAG_READONLY <> 0 then
      begin
        pmDelete.Enabled := False;
        pmRename.Enabled := False;
      end
      else begin
        pmDelete.Enabled := True;
        pmRename.Enabled := True;
      end;
      if KeyData.Flags and INIKEY_FLAG_NOTSKEYS <> 0 then
        pmAddBranch.Enabled := False
      else
        pmAddBranch.Enabled := True;
      if KeyData.Flags and INIKEY_FLAG_NOTVALUE <> 0 then
      begin
        pmAddString.Enabled  := False;
        pmAddBinary.Enabled  := False;
        pmAddInteger.Enabled := False;
      end
      else begin
        pmAddString.Enabled  := True;
        pmAddBinary.Enabled  := True;
        pmAddInteger.Enabled := True;
      end;
    end
    else
      pmDelete.Enabled := False;

    if (FFocusedNode <> nil) then
      UpdateNodesAction(PKeyDataInfo_tag(FFocusedNode.Data)^.Flags)
    else
      UpdateNodesAction(INIKEY_FLAG_READONLY or INIKEY_FLAG_NOTSKEYS or INIKEY_FLAG_NOTVALUE);
  end;

  if pmPopup.PopupComponent is TListView then
  begin
    pmExpand.Visible := False;
    pmEdit.Visible   := True;
    pmSep_4.Visible  := True;
    FFocusedNode := TreeView.Selected;

    if FFocusedNode <> nil then
    begin
      KeyData := FFocusedNode.Data;

      if KeyData.Flags and INIKEY_FLAG_NOTSKEYS <> 0 then
        pmAddBranch.Enabled := False
      else
        pmAddBranch.Enabled := True;

      if KeyData.Flags and INIKEY_FLAG_NOTVALUE <> 0 then
      begin
        pmAddString.Enabled  := False;
        pmAddBinary.Enabled  := False;
        pmAddInteger.Enabled := False;
      end
      else begin
        pmAddString.Enabled  := True;
        pmAddBinary.Enabled  := True;
        pmAddInteger.Enabled := True;
      end;

    end;
    if Assigned(ListView.Selected) then
      UpdateItemsAction(PIniKeyData_tag(ListView.Selected.Data).Flags)
    else
      UpdateItemsAction(INIDAT_FLAG_READONLY or INIDAT_FLAG_NOTEDIT)
  end;

end;

procedure TMainForm.TreeViewChange(Sender: TObject; Node: TTreeNode);
 var
  KeyData: PKeyDataInfo_tag;
begin
  KeyData := Node.Data;
  UpdateNodesAction(KeyData^.Flags);
  UpdatePanelInfo(Reg_pnInfo_KeyPath);

  ListView.Items.BeginUpdate;
  ClearValuesData;
  if KeyData^.HDataKey <> 0 then
  begin
    KeyData^.Changed := False;
    FIniStream.GetKeyValuesEx(KeyData^.HDataKey, FValueList);
    LoadKeyValues;
  end;
  ListView.Items.EndUpdate;

end;

procedure TMainForm.ListViewSelectItem(Sender: TObject; Item: TListItem;
  Selected: Boolean);
begin
  acEdit.Enabled := True;
end;

procedure TMainForm.ListViewExit(Sender: TObject);
begin
  acEdit.Enabled := False;
end;

procedure TMainForm.SetActive(const Value: boolean);
begin
  FActive := Value;

  acImport.Enabled    := FActive;
  acExport.Enabled    := FActive;
  acPrint.Enabled     := FActive;
  acAddBranch.Enabled := FActive;
  acAddString.Enabled := FActive;
  acAddBinary.Enabled := FActive;
  acAddInteger.Enabled:= FActive;
  acEdit.Enabled      := FActive;
  acDelete.Enabled    := FActive;
  acRename.Enabled    := FActive;
  acExpand.Enabled    := FActive;
  acFind.Enabled      := FActive;
  acRefresh.Enabled   := FActive;
  acCopy2Clip.Enabled := FActive;

  mnCreate.Enabled  := FActive;
  pmCreate.Enabled  := FActive;
  tbDropAdd.Enabled := FActive;

end;

procedure TMainForm.acExpandExecute(Sender: TObject);
begin
  if FFocusedNode <> nil then
  begin
    if FFocusedNode.Expanded then
      FFocusedNode.Collapse(False)
    else
      FFocusedNode.Expand(False);
  end;
end;

procedure TMainForm.UpdatePanelInfo(AIndex: integer);
 var
  PanelRect: TRect;
begin
  with sbInfo.Panels[AIndex] do
  begin
    SendMessage(sbInfo.Handle, SB_GETRECT, Index, Integer(@PanelRect));
    ValidateRect(sbInfo.Handle, @PanelRect);
    InvalidateRect(sbInfo.Handle, @PanelRect, True);
  end;
end;

procedure TMainForm.sbInfoDrawPanel(StatusBar: TStatusBar;
  Panel: TStatusPanel; const Rect: TRect);
 var
  R: TRect;
  KeyData: PKeyDataInfo_tag;
  sText: string;
begin
  R     := Rect;
  R.Top := R.Top  + 1;
  case Panel.Index of
    Reg_pnInfo_KeyPath:
      begin
        if TreeView.Selected <> nil then
        begin
          KeyData := TreeView.Selected.Data;
          sText   := GetFullPath(KeyData);

          R.Left := R.Left + 4;
          DrawHighLightText(StatusBar.Canvas,
            PChar(Format(Reg_InfoFmt_Comment,[Font.Name, sText])),
            R, 1, DT_END_ELLIPSIS, imImages);
        end
      end;
  end;
end;

procedure TMainForm.DelBranch(TreeNode: TTreeNode);
 var
  KeyData: PKeyDataInfo_tag;
begin
  if not Assigned(TreeNode) then Exit;

  KeyData := TreeNode.Data;

  if (KeyData^.Flags and INIKEY_FLAG_READONLY = 0) then
  begin
    if FIniStream.DeleteKeyEx(KeyData.HDataKey) = ERROR_SUCCESS
    then begin
      TreeView.Selected := TreeNode.Parent;
      TreeNode.Delete;
    end;
  end;

end;

procedure TMainForm.acDeleteExecute(Sender: TObject);
begin
  if ActiveControl = TreeView then DelBranch(TreeView.Selected);
  if ActiveControl = ListView then DelValue(ListView.Selected);
end;

procedure TMainForm.pmDeleteClick(Sender: TObject);
begin
  if ActiveControl = TreeView then DelBranch(FFocusedNode);
  if ActiveControl = ListView then DelValue(ListView.Selected);
end;

procedure TMainForm.TreeViewKeyUp(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  case Key of
    VK_DELETE: acDelete.Execute;
  end;
end;

procedure TMainForm.FormDestroy(Sender: TObject);
begin
  FValueList.Free;
end;

procedure TMainForm.TreeViewChanging(Sender: TObject; Node: TTreeNode;
  var AllowChange: Boolean);
 var
  KeyData: PKeyDataInfo_tag;
begin
  if (AllowChange) and Assigned(TreeView.Selected) then
  begin
    KeyData := TreeView.Selected.Data;
    if KeyData^.Changed then
    begin
      FIniStream.CloseKeyEx(KeyData.HDataKey, FValueList);
    end
  end;
end;

procedure TMainForm.LoadKeyValues;
 var
  i: integer;
begin
  for i:= 0 to FValueList.Count-1 do  LoadKeyValue(i)
end;

procedure TMainForm.UpdateNodesAction(Flags: integer);
begin
  if Flags and INIKEY_FLAG_READONLY <> 0 then
  begin
    acDelete.Enabled := False;
    acRename.Enabled := False;
  end
  else begin
    acDelete.Enabled := True;
    acRename.Enabled := True;
  end;

  if Flags and INIKEY_FLAG_NOTSKEYS <> 0 then
    acAddBranch.Enabled := False
  else
    acAddBranch.Enabled := True;

  if Flags and INIKEY_FLAG_NOTVALUE <> 0 then
  begin
    acAddString.Enabled  := False;
    acAddBinary.Enabled  := False;
    acAddInteger.Enabled := False;
  end
  else begin
    acAddString.Enabled  := True;
    acAddBinary.Enabled  := True;
    acAddInteger.Enabled := True;
  end;
end;

procedure TMainForm.UpdateItemsAction(Flags: integer);
begin
  if Flags and INIDAT_FLAG_READONLY <> 0 then
  begin
    acDelete.Enabled := False;
    pmDelete.Enabled := False;
    acRename.Enabled := False;
    pmRename.Enabled := False;
  end
  else begin
    acDelete.Enabled := True;
    pmDelete.Enabled := True;
    acRename.Enabled := True;
    acRename.Enabled := True;
  end;

  if Flags and INIDAT_FLAG_NOTEDIT <> 0 then
    acEdit.Enabled := False
  else
    acEdit.Enabled := True;

end;

procedure TMainForm.acAddStringExecute(Sender: TObject);
begin
  CreateStringValue(Self.Handle);
end;

procedure TMainForm.pmAddStringClick(Sender: TObject);
begin
  if FFocusedNode <> TreeView.Selected then TreeView.Selected := FFocusedNode;
  acAddString.Execute;
end;

procedure TMainForm.acAddBinaryExecute(Sender: TObject);
begin
  {}
end;

procedure TMainForm.acAddIntegerExecute(Sender: TObject);
begin
  CreateIntegerValue(Self.Handle);
end;

procedure TMainForm.pmAddIntegerClick(Sender: TObject);
begin
  if FFocusedNode <> TreeView.Selected then TreeView.Selected := FFocusedNode;
  acAddInteger.Execute;
end;

procedure TMainForm.ClearValuesData;
begin
  FValueList.Clear;
  ListView.Items.Clear;
end;

function TMainForm.ValueData2Str(Value: TIniKeyDataType): string;
 var
  sTempValue: string;
  iTempValue: Longint;
begin
  with Value do
  begin
    case TIniDataType(DataType) of
      idUnknown:;
      idString :
        begin
          SetString(sTempValue, PChar(DataValue), DataLen-1);
          if (sTempValue = '') and (Flags and INIDAT_FLAG_EMPTY  <> 0) then
            Result := VALUE_DEFAULT_DATA
          else
            Result := Format('"%s"',[sTempValue]);
        end;
      idInteger:
        begin
          System.Move(DataValue^, iTempValue, DataLen);
          Result := Format('0x%s(%d)',[IntToHex(iTempValue, 8), iTempValue]);
        end;
      idBinary :
        begin
          Result := BinaryToStr(DataValue,  DataLen);
        end;
    end;
  end;
end;

procedure TMainForm.CMRegFileCreateKey(var Message: TMessage);
begin
  with Message, PWPCreateData(WParam)^ do
    Result := AddBranch(DataName, TTreeNode(DataObject));
end;

procedure TMainForm.CMRegFileCreateValue(var Message: TMessage);
 var
  AppendData: boolean;
  i: integer;
  pValue: PIniKeyData_tag;
  KeyData: PKeyDataInfo_tag;
  AName: string;
begin
  with Message, PWPCreateData(WParam)^ do
  begin
    AppendData := DataType > -1;
    if (AppendData) and ( FValueList.IndexOf(DataName) > -1) then
    begin
      Result := ERROR_DUP_NAME;
      Exit
    end;
    if not AppendData then
    begin
       DataType := PIniKeyData_tag(TListItem(DataObject).Data)^.DataType;
       AName    := PIniKeyData_tag(TListItem(DataObject).Data)^.NameValue;
    end
    else
      AName    := DataName;

    case DataType of
      Ord(idString) :
        FValueList.WriteString(AName, PChar(DataBuffer^));
      Ord(idInteger):
        FValueList.WriteInteger(AName, Integer(DataBuffer^));
      Ord(idBinary) :;
    end;
    
    i := FValueList.IndexOf(AName);
    if i > -1 then
    begin
      if AppendData then
        LoadKeyValue(i)
      else
      with TListItem(DataObject) do
      begin
        pValue  := FValueList.KeyValue[i];
        Data    := pValue;
        SubItems.Strings[0] := ValueData2Str(pValue^);
      end;
      KeyData := TreeView.Selected.Data;
      KeyData^.Changed := True;
      Result := ERROR_SUCCESS;
    end
    else
      Result := ERROR_WRITE_FAULT;
  end;
end;

procedure TMainForm.LoadKeyValue(Index: integer);
 var
  Item: TListItem;
  pValue: PIniKeyData_tag;
begin
  Item := ListView.Items.Add;
  with Item do
  begin
    pValue := FValueList.KeyValue[Index];
    if pValue^.Flags and INIDAT_FLAG_DEFAULT <> 0 then
      Caption := VALUE_DEFAULT_CAPTION
    else
      Caption := FValueList.Strings[Index];
    Data       := pValue;
    ImageIndex := DataTypeImageIndexes[TIniDataType(pValue^.DataType)];
    SubItems.Add(ValueData2Str(pValue^));
 end;
end;

procedure TMainForm.acEditExecute(Sender: TObject);
begin
  if Assigned(ListView.Selected) then
  with ListView.Selected, PIniKeyData_tag(Data)^ do
  begin
    case DataType of
      Ord(idString) :
        ModifyStringValue(Caption, PChar(DataValue), Self.Handle, ListView.Selected);
      Ord(idInteger):
        ModifyIntegerValue(Caption, Integer(DataValue^), Self.Handle, ListView.Selected);
      Ord(idBinary) :;
    end;
  end;
end;

procedure TMainForm.ListViewDblClick(Sender: TObject);
begin
  acEdit.Execute;
end;

procedure TMainForm.DelValue(ListItem: TListItem);
 var
  KeyData: PKeyDataInfo_tag;
begin
  if Assigned(ListItem) and
     FValueList.DeleteValue(PIniKeyData_tag(ListItem.Data)^.NameValue) then
  begin
    ListItem.Delete;
    KeyData := TreeView.Selected.Data;
    KeyData^.Changed := True;
  end;
end;

procedure TMainForm.pmCopy2ClipClick(Sender: TObject);
begin
  Copy2Clipboard(FFocusedNode);
end;

procedure TMainForm.acCopy2ClipExecute(Sender: TObject);
begin
  Copy2Clipboard(TreeView.Selected);
end;

procedure TMainForm.Copy2Clipboard(TreeNode: TTreeNode);
 var
  KeyData: PKeyDataInfo_tag;
begin
  if Assigned(TreeNode) then
  begin
    KeyData := TreeNode.Data;
    ClipBoard.SetTextBuf(PChar(GetFullPath(KeyData)));
  end;
end;

function TMainForm.GetFullPath(KeyData: PKeyDataInfo_tag): string;
begin
  if KeyData^.HRootKey = KeyData^.HDataKey then
  begin
    if KeyData^.HRootKey = 0 then
      Result := Title
    else
      Result := Format('%s\%s', [Title,  SystemIniKeyNames[KeyData^.HRootKey]])
  end
  else
    Result := Format('%s\%s\%s', [Title, SystemIniKeyNames[KeyData^.HRootKey],
      KeyData^.KeyPath]);
end;

procedure TMainForm.ToolButton1Click(Sender: TObject);
 var
  Strings: TStringList;
begin
  Strings := TStringList.Create;
  FIniStream.OpenKey(PKeyDataInfo_tag(TreeView.Selected.Data)^.Name, False);
  FIniStream.GetKeyNames(Strings);
  Strings.Sort;
end;

end.
