{
 BUSINESS CONSULTING
 s a i n t - p e t e r s b u r g

         Components Library for Borland Delphi 4.x, 5.x
         Copyright (c) 1998-2000 Alex'EM

}
unit DCTray;

interface

uses Windows, Messages,
     Classes, Graphics, SysUtils, Forms, Controls, Menus, ShellAPI, DCConst;

type

{
  PNotifyIconData = ^TNotifyIconData;
  TNotifyIconData = record
    cbSize: Longint;
    Wnd: Longint;
    uID: Longint;
    uFlags: Longint;
    uCallbackMessage: Longint;
    hIcon: Longint;
    szTip: array [0..63] of Char;
  end;
}

  TMouseButtons = set of TMouseButton;

  TDCTrayIcon = class(TComponent)
  private
    FHandle: HWnd;
    FActive: Boolean;
    FAdded: Boolean;
    FClicked: TMouseButtons;
    FIconData: TNotifyIconData;
    FIcon: TIcon;
    FDestroying: Boolean;
    FHint: string;
    FShowDesign: Boolean;
    FPopupMenu: TPopupMenu;
    FOnClick: TMouseEvent;
    FOnDblClick: TNotifyEvent;
    FOnMouseMove: TMouseMoveEvent;
    FOnMouseDown: TMouseEvent;
    FOnMouseUp: TMouseEvent;
    FStartMinimized: boolean;
    procedure ChangeIcon;
    procedure SendCancelMode;
    function CheckMenuPopup(X, Y: Integer): Boolean;
    function CheckDefaultMenuItem: Boolean;
    procedure SetHint(const Value: string);
    procedure SetIcon(Value: TIcon);
    procedure SetPopupMenu(Value: TPopupMenu);
    procedure Activate;
    procedure Deactivate;
    procedure SetActive(Value: Boolean);
    procedure SetShowDesign(Value: Boolean);
    procedure IconChanged(Sender: TObject);
    procedure WndProc(var Message: TMessage);
    function GetActiveIcon: TIcon;
    procedure LoadDefaultIcon;
  protected
    procedure DblClick; dynamic;
    procedure DoClick(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); dynamic;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); dynamic;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); dynamic;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); dynamic;
    procedure Loaded; override;
    procedure Notification(AComponent: TComponent; Operation: TOperation); override;
    procedure UpdateNotifyData; virtual;
    property Handle: HWnd read FHandle;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Hide;
    procedure Show;
  published
    property Active: Boolean read FActive write SetActive default True;
    property Hint: string read FHint write SetHint;
    property Icon: TIcon read FIcon write SetIcon;
    property PopupMenu: TPopupMenu read FPopupMenu write SetPopupMenu;
    property ShowDesign: Boolean read FShowDesign write SetShowDesign stored False;
    property OnClick: TMouseEvent read FOnClick write FOnClick;
    property OnDblClick: TNotifyEvent read FOnDblClick write FOnDblClick;
    property OnMouseMove: TMouseMoveEvent read FOnMouseMove write FOnMouseMove;
    property OnMouseDown: TMouseEvent read FOnMouseDown write FOnMouseDown;
    property OnMouseUp: TMouseEvent read FOnMouseUp write FOnMouseUp;
    property StartMinimized: boolean read FStartMinimized write FStartMinimized;
  end;

type
  TExecState = (esNormal, esMinimized, esMaximized, esHidden);

type
  TPreviousInstance = class(TObject)
  private
    FMessageID: DWORD;
    FMutexHandle: THandle;
    FhPrevInst: boolean;
    FNewWndProc: Pointer;
    FDefWndProc: Pointer;
  protected
    procedure NewWndProc(var Message: TMessage);
  public
    destructor Destroy; override;
    procedure SethPrevInst;
    property MutexHandle: THandle read FMutexHandle;
    property hPrevInst: boolean read FhPrevInst write FhPrevInst;
    property MessageID: DWORD read FMessageID;
  end;

function CheckToMultyInstance: boolean;

function FileExecute(const FileName, Params, StartDir: string;
  InitialState: TExecState): THandle;
function FileExecuteWait(const FileName, Params, StartDir: string;
  InitialState: TExecState): Integer;

var
  PreviousInstance: TPreviousInstance;

implementation

const
  ShowCommands: array[TExecState] of Integer =
    (SW_SHOWNORMAL, SW_MINIMIZE, SW_SHOWMAXIMIZED, SW_HIDE);

function FileExecute(const FileName, Params, StartDir: string;
  InitialState: TExecState): THandle;
begin
  Result := ShellExecute(Application.Handle, nil, PChar(FileName),
    PChar(Params), PChar(StartDir), ShowCommands[InitialState]);
end;

function FileExecuteWait(const FileName, Params, StartDir: string;
  InitialState: TExecState): Integer;
var
  Info: TShellExecuteInfo;
  ExitCode: DWORD;
begin
  FillChar(Info, SizeOf(Info), 0);
  Info.cbSize := SizeOf(TShellExecuteInfo);
  with Info do begin
    fMask := SEE_MASK_NOCLOSEPROCESS;
    Wnd := Application.Handle;
    lpFile := PChar(FileName);
    lpParameters := PChar(Params);
    lpDirectory := PChar(StartDir);
    nShow := ShowCommands[InitialState];
  end;
  if ShellExecuteEx(@Info) then begin
    repeat
      Application.ProcessMessages;
      GetExitCodeProcess(Info.hProcess, ExitCode);
    until (ExitCode <> STILL_ACTIVE) or Application.Terminated;
    Result := ExitCode;
  end
  else Result := -1;
end;

procedure SwitchToWindow(Wnd: HWnd; Restore: Boolean);
begin
  if IsWindowEnabled(Wnd) then begin
    SetForegroundWindow(Wnd);
    if Restore and IsWindowVisible(Wnd) then begin
      if not IsZoomed(Wnd) then
        SendMessage(Wnd, WM_SYSCOMMAND, SC_RESTORE, 0);
      SetFocus(Wnd);
    end;
  end;
end;

function GetShiftState: TShiftState;
begin
  Result := [];
  if GetKeyState(VK_SHIFT  ) < 0 then Include(Result, ssShift);
  if GetKeyState(VK_CONTROL) < 0 then Include(Result, ssCtrl);
  if GetKeyState(Vk_MENU   ) < 0 then Include(Result, ssAlt);
end;

constructor TDCTrayIcon.Create(AOwner: Tcomponent);
begin
  inherited Create(AOwner);
  FHandle := AllocateHWnd(WndProc);
  FIcon := TIcon.Create;
  FIcon.OnChange := IconChanged;
  FActive        := True;
  StartMinimized := False;
  LoadDefaultIcon;
end;

destructor TDCTrayIcon.Destroy;
begin
  FDestroying := True;
  FIcon.OnChange := nil;
  Deactivate;
  DeallocateHWnd(FHandle);
  FIcon.Free;
  FIcon := nil;
  inherited Destroy;
end;

procedure TDCTrayIcon.Loaded;
begin
  inherited Loaded;
  if FActive and not (csDesigning in ComponentState) then Activate;

  if FStartMinimized then
  begin
    Application.ShowMainForm := False;
    ShowWindow(Application.Handle, SW_HIDE);
  end;

end;

procedure TDCTrayIcon.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (AComponent = PopupMenu) and (Operation = opRemove) then
    PopupMenu := nil;
end;

procedure TDCTrayIcon.SetPopupMenu(Value: TPopupMenu);
begin
  FPopupMenu := Value;
  if Value <> nil then Value.FreeNotification(Self);
end;

procedure TDCTrayIcon.SendCancelMode;
var
  F: TForm;
begin
  if not ((csDestroying in ComponentState) or FDestroying) then begin
    F := Screen.ActiveForm;
    if F = nil then F := Application.MainForm;
    if F <> nil then F.SendCancelMode(nil);
  end;
end;

function TDCTrayIcon.CheckMenuPopup(X, Y: Integer): Boolean;
begin
  Result := False;
  if not (csDesigning in ComponentState) and Active and
    (PopupMenu <> nil) and PopupMenu.AutoPopup then
  begin
    PopupMenu.PopupComponent := Self;
    SendCancelMode;
    SwitchToWindow(FHandle, False);
    Application.ProcessMessages;
    try
      PopupMenu.Popup(X, Y);
    finally
      SwitchToWindow(FHandle, False);
    end;
    Result := True;
  end;
end;

function TDCTrayIcon.CheckDefaultMenuItem: Boolean;
var
  Item: TMenuItem;
  I: Integer;
begin
  Result := False;
  if not (csDesigning in ComponentState) and Active and
    (PopupMenu <> nil) and (PopupMenu.Items <> nil) then
  begin
    I := 0;
    while (I < PopupMenu.Items.Count) do begin
      Item := PopupMenu.Items[I];
      if Item.Default and Item.Enabled then begin
        Item.Click;
        Result := True;
        Break;
      end;
      Inc(I);
    end;
  end;
end;

procedure TDCTrayIcon.SetIcon(Value: TIcon);
begin
  FIcon.Assign(Value);
end;

function TDCTrayIcon.GetActiveIcon: TIcon;
begin
  Result := FIcon;
end;

procedure TDCTrayIcon.SetActive(Value: Boolean);
begin
  if (Value <> FActive) then begin
    FActive := Value;
    if not (csDesigning in ComponentState) then
      if Value then Activate else Deactivate;
  end;
end;

procedure TDCTrayIcon.Show;
begin
  Active := True;
end;

procedure TDCTrayIcon.Hide;
begin
  Active := False;
end;

procedure TDCTrayIcon.SetShowDesign(Value: Boolean);
begin
  if (csDesigning in ComponentState) then begin
    if Value then Activate else Deactivate;
    FShowDesign := FAdded;
  end;
end;

procedure TDCTrayIcon.IconChanged(Sender: TObject);
begin
  ChangeIcon;
end;

procedure TDCTrayIcon.SetHint(const Value: string);
begin
  if FHint <> Value then begin
    FHint := Value;
    ChangeIcon;
  end;
end;

procedure TDCTrayIcon.UpdateNotifyData;
var
  Ico: TIcon;
begin
  with FIconData do begin
    cbSize := SizeOf(TNotifyIconData);
    Wnd := FHandle;
    uFlags := NIF_MESSAGE or NIF_ICON or NIF_TIP;
    Ico := GetActiveIcon;
    if Ico <> nil 
       then hIcon := Ico.Handle
       else hIcon := INVALID_HANDLE_VALUE;
    StrPLCopy(szTip, GetShortHint(FHint), SizeOf(szTip) - 1);
    uCallbackMessage := CM_TRAYICON;
    uID := 0;
  end;
end;

procedure TDCTrayIcon.Activate;
var
  Ico: TIcon;
begin
  Deactivate;
  Ico := GetActiveIcon;
  if (Ico <> nil) and not Ico.Empty then begin
    FClicked := [];
    UpdateNotifyData;
    FAdded := Shell_NotifyIcon(NIM_ADD, @FIconData);
    if (GetShortHint(FHint) = '') and FAdded then
      Shell_NotifyIcon(NIM_MODIFY, @FIconData);
  end;
end;

procedure TDCTrayIcon.Deactivate;
begin
  Shell_NotifyIcon(NIM_DELETE, @FIconData);
  FAdded := False;
  FClicked := [];
end;

procedure TDCTrayIcon.ChangeIcon;
var
  Ico: TIcon;
begin
  if FAdded then begin
    Ico := GetActiveIcon;
    if (Ico <> nil) and not Ico.Empty then begin
      UpdateNotifyData;
      Shell_NotifyIcon(NIM_MODIFY, @FIconData);
    end
    else Deactivate;
  end
  else begin
    if ((csDesigning in ComponentState) and FShowDesign) or
      (not (csDesigning in ComponentState) and FActive) then Activate;
  end;
end;

procedure TDCTrayIcon.MouseMove(Shift: TShiftState; X, Y: Integer);
begin
  if Assigned(FOnMouseMove) then FOnMouseMove(Self, Shift, X, Y);
end;

procedure TDCTrayIcon.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  if Assigned(FOnMouseDown) then FOnMouseDown(Self, Button, Shift, X, Y);
end;

procedure TDCTrayIcon.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  if Assigned(FOnMouseUp) then FOnMouseUp(Self, Button, Shift, X, Y);
end;

procedure TDCTrayIcon.DblClick;
begin
  if not CheckDefaultMenuItem and Assigned(FOnDblClick) then
    FOnDblClick(Self);
end;

procedure TDCTrayIcon.DoClick(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
begin
  if (Button = mbRight) and CheckMenuPopup(X, Y) then Exit;
  if Assigned(FOnClick) then FOnClick(Self, Button, Shift, X, Y);
end;

procedure TDCTrayIcon.WndProc(var Message: TMessage);
 var
  P: TPoint;
  Shift: TShiftState;
begin
  try
    with Message do
    begin
      if Msg = CM_TRAYICON then begin
        case lParam of
          WM_LBUTTONDBLCLK:
            begin
              GetCursorPos(P);
              MouseDown(mbLeft, GetShiftState + [ssDouble], P.X, P.Y);
              DblClick;
            end;
          WM_RBUTTONDBLCLK:
            begin
              GetCursorPos(P);
              MouseDown(mbRight, GetShiftState + [ssDouble], P.X, P.Y);
            end;
          WM_MBUTTONDBLCLK:
            begin
              GetCursorPos(P);
              MouseDown(mbMiddle, GetShiftState + [ssDouble], P.X, P.Y);
            end;
          WM_MOUSEMOVE:
            begin
              GetCursorPos(P);
              MouseMove(GetShiftState, P.X, P.Y);
            end;
          WM_LBUTTONDOWN:
            begin
              GetCursorPos(P);
              MouseDown(mbLeft, GetShiftState + [ssLeft], P.X, P.Y);
              Include(FClicked, mbLeft);
            end;
          WM_LBUTTONUP:
            begin
              Shift := GetShiftState + [ssLeft];
              GetCursorPos(P);
              if (mbLeft in FClicked) then begin
                Exclude(FClicked, mbLeft);
                DoClick(mbLeft, Shift, P.X, P.Y);
              end;
              MouseUp(mbLeft, Shift, P.X, P.Y);
            end;
          WM_RBUTTONDOWN:
            begin
              GetCursorPos(P);
              MouseDown(mbRight, GetShiftState + [ssRight], P.X, P.Y);
              Include(FClicked, mbRight);
            end;
          WM_RBUTTONUP:
            begin
              Shift := GetShiftState + [ssRight];
              GetCursorPos(P);
              if (mbRight in FClicked) then begin
                Exclude(FClicked, mbRight);
                DoClick(mbRight, Shift, P.X, P.Y);
              end;
              MouseUp(mbRight, Shift, P.X, P.Y);
            end;
          WM_MBUTTONDOWN:
            begin
              GetCursorPos(P);
              MouseDown(mbMiddle, GetShiftState + [ssMiddle], P.X, P.Y);
            end;
          WM_MBUTTONUP:
            begin
              GetCursorPos(P);
              MouseUp(mbMiddle, GetShiftState + [ssMiddle], P.X, P.Y);
            end;
        end;
      end
      else Result := DefWindowProc(FHandle, Msg, wParam, lParam);
    end
  except
    Application.HandleException(Self);
  end;
end;

destructor TPreviousInstance.Destroy;
begin
  CloseHandle(PreviousInstance.MutexHandle);
  if FDefWndProc <> nil then
    SetWindowLong(Application.Handle, GWL_WNDPROC, LongInt(FDefWndProc));
  FreeObjectInstance(FNewWndProc);
  inherited;
end;

procedure TPreviousInstance.NewWndProc(var Message: TMessage);
begin
  with Message do
  begin
    if Msg = FMessageID then
    begin
      if IsIconic(Application.Handle) then
      begin
        Application.MainForm.WindowState := wsNormal;
        Application.Restore;
      end;
      SetForegroundWindow(Application.Handle);
    end
    else
      Result := CallWindowProc(FDefWndProc, Application.Handle, Msg, WParam, LParam);
  end;
end;

procedure TPreviousInstance.SethPrevInst;
begin
  FMessageID   := RegisterWindowMessage(PChar(Application.Title));
  FMutexHandle := CreateMutex(nil, TRUE, PChar(Application.Title));
  if MutexHandle <> 0 then
  begin
    if GetLastError = ERROR_ALREADY_EXISTS then
    begin
      hPrevInst := TRUE;
      CloseHandle(MutexHandle);
    end
    else begin
      hPrevInst := FALSE;
      FNewWndProc := MakeObjectInstance(NewWndProc);
      FDefWndProc := Pointer(SetWindowLong(Application.Handle, GWL_WNDPROC,
        LongInt(FNewWndProc)));
    end;
  end
  else
    hPrevInst := FALSE;
end;

function CheckToMultyInstance: boolean;
 var
  BSMReceptions: DWORD;
begin
  if PreviousInstance.hPrevInst then
  begin
    Application.ShowMainForm := False;
    BSMReceptions := BSM_APPLICATIONS;
    BroadCastSystemMessage(BSF_IGNORECURRENTTASK or BSF_POSTMESSAGE,
      @BSMReceptions, PreviousInstance.MessageID, 0 ,0);
    Result := True
  end
  else
    Result := False;
end;

procedure TDCTrayIcon.LoadDefaultIcon;
begin
  FIcon.Handle := LoadIcon(hInstance, 'MAINICONX16');
  if FIcon.Handle = 0 then
    FIcon.Handle := LoadIcon(0, IDI_WINLOGO);
end;

initialization
  PreviousInstance := TPreviousInstance.Create;
  PreviousInstance.SethPrevInst;

finalization
  PreviousInstance.Free;

end.
