{
 BUSINESS CONSULTING
 s a i n t - p e t e r s b u r g

         Components Library for Borland Delphi 4.x, 5.x
         Copyright (c) 1998-2000 Alex'EM

}
unit DCPopupWindow;

interface
{$I DCConst.inc}

uses
  Windows, Messages, SysUtils, Classes, Controls, Forms, Graphics, StdCtrls,
  CommCtrl, FlatSb, DCdbGrids, DB, Grids, Dialogs, ComCtrls,
  DCEditTools, DCEditButton, DCConst;

type

  TClipFormOptions = set of TClipFormValue;

  TDCPopupWindow = class(TCustomControl)
  private
    FVisible: boolean;
    FOwner: TControl;
    FWindowRect: TRect;
    FAlwaysVisible: boolean;
    FPopupAlignment: TPopupAlignment;
    FSrcImage: TBitmap;
    FSaveSource: boolean;
    FOrientation: integer;
    procedure SetPopupAlignment(Value: TPopupAlignment);
  protected
    procedure CreateParams(var Params: TCreateParams); override;
    procedure CreateWnd; override;
    procedure WMMouseActivate(var Message: TWMActivate); message WM_MOUSEACTIVATE;
    procedure WMNCPaint(var Message: TWMNCPaint); message WM_NCPAINT;
    procedure WMNCCalcSize(var Message: TWMNCCalcSize); message WM_NCCALCSIZE;
    procedure CMShowingChanged (var Message: TMessage); message CM_SHOWINGCHANGED;
  public
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
    procedure SetBoundsEx(ALeft, ATop, AWidth, AHeight: Integer);
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    function GetTextHeight(Value: string): integer;
    function GetTextWidth(Value: string): integer;
    procedure Show; virtual;
    procedure Hide; virtual;
    property AlwaysVisible: boolean read FAlwaysVisible write FAlwaysVisible;
    property PopupAlignment: TPopupAlignment read FPopupAlignment
             write SetPopupAlignment;
    property Owner: TControl read FOwner write FOwner;
    property Canvas;
    property Parent;
    property SaveSource: boolean read FSaveSource write FSaveSource;
    property Orientation: integer read FOrientation write FOrientation;
  end;

  TDCMessageWindow = class(TDCPopupWindow)
  private
    FButtons: TDCEditButtons;
    FBitmap: TBitmap;
    FBitmapVisible: boolean;
    FDialogStyle: TDialogStyle;
    FMargins: TRect;
    FAutoHide: boolean;
    FTimerHandle: Word;
    FTimeOut: integer;
    FMessageStyle: TMessageStyle;
    FImage: TBitmap;
    FRoundValue: integer;
    FTailValue: integer;
    FBitmapOffset: integer;
    procedure AdjustWindowSize;
    procedure SetDialogStyle(Value: TDialogStyle);
    procedure SetBitmap(Value: TBitmap);
    procedure SetBitmapVisible(Value: boolean);
    procedure SetAutoHide(const Value: boolean);
    procedure SetTimeOut(const Value: integer);
    procedure StartTimer(Value: Integer);
    procedure StopTimer;
    function GetRegion: HRGN;
    procedure SetMessageStyle(const Value: TMessageStyle);
  protected
    procedure CreateWnd; override;
    procedure CreateParams(var Params: TCreateParams); override;
    procedure WMNCHitTest(var Message: TWMNCHitTest); message WM_NCHITTEST;
    procedure WMEraseBkgnd(var Message: TWmEraseBkgnd); message WM_ERASEBKGND;
    procedure WMTimer(var Message: TWMTimer); message WM_TIMER;
    procedure CMTextChanged(var Message: TMessage); message CM_TEXTCHANGED;
    procedure CMMouseLeave(var Message: TMessage); message CM_MOUSELEAVE;
    procedure SetMargins;
    procedure Paint; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Show; override;
    function AddButton(AName, AResource, ACaption: string;
      AClick: TNotifyEvent): TDCEditButton;
    property Caption;
    property Color;
    property Canvas;
    property DialogStyle: TDialogStyle read FDialogStyle write SetDialogStyle;
    property Bitmap: TBitmap read FBitmap write SetBitmap;
    property BitmapVisible: boolean read FBitmapVisible write SetBitmapVisible;
    property Buttons: TDCEditButtons read FButtons;
    property AutoHide: boolean read FAutoHide write SetAutoHide;
    property TimeOut: integer read FTimeOut write SetTimeOut;
    property MessageStyle: TMessageStyle read FMessageStyle write SetMessageStyle;
  end;

  TDCPopupListBox = class(TCustomListBox)
  private
    FVisible: boolean;
    FOwner: TControl;
    FWindowRect: TRect;
    FAlwaysVisible: boolean;
    FPopupAlignment: TPopupAlignment;
    FPopupBorderStyle: TPopupBorderStyle;
    FBorderSize: integer;
    FDropDownRows: integer;
    procedure RedrawBorder;
    procedure SetPopupAlignment(Value: TPopupAlignment);
    procedure SetPopupBorderStyle(Value: TPopupBorderStyle);
  protected
    procedure CreateParams(var Params: TCreateParams); override;
    procedure CreateWnd; override;
    function DoMouseWheel(Shift: TShiftState; WheelDelta: Integer;
      MousePos: TPoint): Boolean; override;
    procedure WMMouseActivate(var Message: TWMActivate); message WM_MOUSEACTIVATE;
    procedure WMNCCalcSize(var Message: TWMNCCalcSize); message WM_NCCALCSIZE;
    procedure WMNCPaint(var Message: TWMNCPaint); message WM_NCPAINT;
    procedure CNDrawItem(var Message: TWMDrawItem); message CN_DRAWITEM;
    procedure WMMouseMove(var Message: TWMMouseMove); message WM_MOUSEMOVE;
    procedure WMFontChange(var Message: TWMFontChange); message WM_FONTCHANGE;
  public
    procedure AdjustNewHeight;
    procedure SetListHeight(Increment: integer);
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
    procedure SetBoundsEx(ALeft, ATop, AWidth, AHeight: Integer);

    constructor Create(AOwner: TComponent); override;
    procedure Show;
    procedure Hide;
    property AlwaysVisible: boolean read FAlwaysVisible write FAlwaysVisible;
    property PopupAlignment: TPopupAlignment read FPopupAlignment
             write SetPopupAlignment;
    property Owner: TControl read FOwner write FOwner;
    property PopupBorderStyle: TPopupBorderStyle read FPopupBorderStyle write SetPopupBorderStyle;
    property DropDownRows: integer read FDropDownRows write FDropDownRows;

    property Style;
    property Font;
    property OnDrawItem;
    property OnMeasureItem;
    property OnMouseUp;
    property ItemHeight;
    property Color;
    property ListVisible: boolean read FVisible write FVisible;
  end;

  TDCPopupDBGrid = class(TDCCustomDBGrid)
  private
    FButtons: TDCEditButtons;
    FVisible: boolean;
    FOwner: TControl;
    FWindowRect: TRect;
    FAlwaysVisible: boolean;
    FPopupAlignment: TPopupAlignment;
    FPopupBorderStyle: TPopupBorderStyle;
    FBorderSize: integer;
    FDataSet: TDataSet;
    FDataSource: TDataSource;
    FDropDownRows: integer;
    FItemHeight: integer;
    FMargins: TRect;
    FCursorMode: TCursorMode;
    FShowHeader: boolean;
    FOnButtonClick: TNotifyEvent;
    FFindButton: TDCEditButton;
    FCanAppend: boolean;
    FPosition: TBookmarkStr;
    procedure RedrawBorder;
    procedure SetPopupAlignment(Value: TPopupAlignment);
    procedure SetPopupBorderStyle(Value: TPopupBorderStyle);
    procedure SetDataSet(const Value: TDataSet);
    procedure DrawHeader;
    procedure DrawFooter;
    procedure SetMargins;
    procedure BeginMoving(XCursor, YCursor: integer);
    procedure DoButtonClick(Sender: TObject);
    procedure InvalidateButtons;
    procedure SetShowHeader(const Value: boolean);
    procedure SetCanAppend(const Value: boolean);
    procedure DoDrawHint(Sender: TObject; Mode: Integer);
    procedure CheckRefreshButton;
    procedure PaintEmptyMessage(Sender: TObject; Canvas: TCanvas; ARect: TRect;
    UpdateMessage: string);
  protected
    procedure CreateParams(var Params: TCreateParams); override;
    procedure CreateWnd; override;
    procedure WMMouseActivate(var Message: TWMActivate); message WM_MOUSEACTIVATE;
    procedure WMNCCalcSize(var Message: TWMNCCalcSize); message WM_NCCALCSIZE;
    procedure WMNCPaint(var Message: TWMNCPaint); message WM_NCPAINT;
    function  HighlightCell(DataCol, DataRow: Integer; const Value: string;
      AState: TGridDrawState): Boolean; override;
    procedure WMFontChange(var Message: TWMFontChange); message WM_FONTCHANGE;
    procedure WMLButtonDblClk(var Message: TWMLButtonDown); message WM_LBUTTONDBLCLK;
    procedure WMNCHitTest(var Message: TWMNCHitTest); message WM_NCHITTEST;
    procedure WMSetCursor(var Message: TWMSetCursor); message WM_SETCURSOR;
    procedure WMNCLButtonDown(var Message: TWMNCLButtonDown); message WM_NCLBUTTONDOWN;
    procedure CMMouseEnter(var Message: TMessage); message CM_MOUSEENTER;
    procedure CMMouseLeave(var Message: TMessage); message CM_MOUSELEAVE;
    procedure CMHintShow(var Message: TCMHintShow); message CM_HINTSHOW;
    procedure CMSetAlignment(var Message: TMessage); message CM_SETALIGNMENT;
    procedure WMPaint(var Message: TWMPaint); message WM_PAINT;
    procedure WMSize(var Message: TWMSize); message WM_SIZE;
    procedure WMTimer(var Message: TWMTimer); message WM_TIMER;
    procedure WMNCMouseMove(var Message: TWMNCMouseMove); message WM_NCMOUSEMOVE;
    function  MouseUpBeforeDblClk: boolean; override;
  public
    procedure AdjustNewHeight;
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
    procedure SetBoundsEx(ALeft, ATop, AWidth, AHeight: Integer);
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure KeyPress(var Key: Char);override;

    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure SetParent(AParent: TWinControl); override;
    procedure Show;
    procedure Hide;
    procedure StartSearch(Key: Char; AValue: string = '');
    procedure StopSearch;
    property AlwaysVisible: boolean read FAlwaysVisible write FAlwaysVisible;
    property PopupAlignment: TPopupAlignment read FPopupAlignment
             write SetPopupAlignment;
    property Owner: TControl read FOwner write FOwner;
    property PopupBorderStyle: TPopupBorderStyle read FPopupBorderStyle write SetPopupBorderStyle;
    property DataSet: TDataSet read FDataSet write SetDataSet;
    property DropDownRows: integer read FDropDownRows write FDropDownRows;
    property Columns;
    property OnCellClick;
    property OnDblClick;
    property BorderStyle;
    property Buttons: TDCEditButtons read FButtons;
    property ShowHeader: boolean read FShowHeader write SetShowHeader;
    property OnButtonClick: TNotifyEvent read FOnButtonClick write FOnButtonClick;
    property OnTitleClick;
    property CanAppend: boolean read FCanAppend write SetCanAppend;
    property Position: TBookmarkStr read FPosition;
    property DataSource;
  end;

  TDCPopupTreeView = class(TCustomTreeView)
  private
    FVisible: boolean;
    FOwner: TControl;
    FWindowRect: TRect;
    FAlwaysVisible: boolean;
    FPopupAlignment: TPopupAlignment;
    FPopupBorderStyle: TPopupBorderStyle;
    FItemHeight: integer;
    FBorderSize: integer;
    FDropDownRows: integer;
    FMargins: TRect;
    FCursorMode: TCursorMode;
    FButtons: TDCEditButtons;
    FShowHeader: boolean;
    procedure RedrawBorder;
    procedure SetPopupAlignment(Value: TPopupAlignment);
    procedure SetPopupBorderStyle(Value: TPopupBorderStyle);
    procedure DrawHeader;
    procedure DrawFooter;
    procedure SetMargins;
    procedure BeginMoving(XCursor, YCursor: integer);
    procedure InvalidateButtons;
    procedure SetShowHeader(const Value: boolean);
  protected
    procedure CreateParams(var Params: TCreateParams); override;
    procedure CreateWnd; override;
    procedure WMMouseActivate(var Message: TWMActivate); message WM_MOUSEACTIVATE;
    procedure WMNCCalcSize(var Message: TWMNCCalcSize); message WM_NCCALCSIZE;
    procedure WMNCPaint(var Message: TWMNCPaint); message WM_NCPAINT;
    procedure WMLButtonDblClk(var Message: TWMLButtonDown); message WM_LBUTTONDBLCLK;
    procedure WMFontChange(var Message: TWMFontChange); message WM_FONTCHANGE;
    procedure WMNCHitTest(var Message: TWMNCHitTest); message WM_NCHITTEST;
    procedure WMSetCursor(var Message: TWMSetCursor); message WM_SETCURSOR;
    procedure WMNCLButtonDown(var Message: TWMNCLButtonDown); message WM_NCLBUTTONDOWN;
    procedure CMMouseEnter(var Message: TMessage); message CM_MOUSEENTER;
    procedure CMMouseLeave(var Message: TMessage); message CM_MOUSELEAVE;
    procedure CMSetAlignment(var Message: TMessage); message CM_SETALIGNMENT;
    procedure WMPaint(var Message: TWMPaint); message WM_PAINT;
    procedure WMSize(var Message: TWMSize); message WM_SIZE;
    procedure WMActivate (var Message: TWMActivate); message WM_ACTIVATE;
    procedure WMHScroll(var Message: TWMHScroll); message WM_HSCROLL;
    procedure WMShowWindow(var Message: TWMShowWindow); message WM_SHOWWINDOW;
  public
    procedure AdjustNewHeight;
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override;
    procedure SetBoundsEx(ALeft, ATop, AWidth, AHeight: Integer);
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure KeyPress(var Key: Char);override;

    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure SetParent(AParent: TWinControl); override;
    procedure Show;
    procedure Hide;
    property AlwaysVisible: boolean read FAlwaysVisible write FAlwaysVisible;
    property PopupAlignment: TPopupAlignment read FPopupAlignment
             write SetPopupAlignment;
    property Owner: TControl read FOwner write FOwner;
    property PopupBorderStyle: TPopupBorderStyle read FPopupBorderStyle write SetPopupBorderStyle;
    property DropDownRows: integer read FDropDownRows write FDropDownRows;
    property OnDblClick;
    property BorderStyle;
    property Buttons: TDCEditButtons read FButtons;
    property Color;
    property Items;
    property Images;
    property OnChange;
    property OnCollapsed;
    property OnExpanded;
    property OnCollapsing;
    property OnExpanding;
    property OnKeyPress;
    property Caption;
    property ShowHeader: boolean read FShowHeader write SetShowHeader;
  end;

  TDCClipPopup = class(TDCPopupWindow)
  private
    FButtons: TDCEditButtons;
    FCursorMode: TCursorMode;
    FOptions: TClipFormOptions;
    FPopupBorderStyle: TPopupBorderStyle;
    FBorderSize: integer;
    FMargins: TRect;
    procedure SetPopupBorderStyle(Value: TPopupBorderStyle);
    procedure BeginMoving(XCursor, YCursor, Delta: integer);
    procedure SetOptions(const Value: TClipFormOptions);
  protected
    procedure RedrawBorder; virtual;
    procedure DrawHeader; virtual;
    procedure DrawFooter; virtual;
    procedure SetMargins; virtual;
    procedure InvalidateButtons; virtual;
    procedure WMNCCalcSize(var Message: TWMNCCalcSize); message WM_NCCALCSIZE;
    procedure WMNCPaint(var Message: TWMNCPaint); message WM_NCPAINT;
    procedure WMNCHitTest(var Message: TWMNCHitTest); message WM_NCHITTEST;
    procedure WMSetCursor(var Message: TWMSetCursor); message WM_SETCURSOR;
    procedure WMNCLButtonDown(var Message: TWMNCLButtonDown); message WM_NCLBUTTONDOWN;
    procedure WMPaint(var Message: TWMPaint); message WM_PAINT;
    procedure WMActivate (var Message: TWMActivate); message WM_ACTIVATE;
    procedure WMSize(var Message: TWMSize); message WM_SIZE;
    procedure CMSetAlignment(var Message: TMessage); message CM_SETALIGNMENT;
    procedure CreateWnd; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure Show; override;
    procedure Hide; override;
    property Buttons: TDCEditButtons read FButtons;
    property Options: TClipFormOptions read FOptions write SetOptions;
    property PopupBorderStyle: TPopupBorderStyle read FPopupBorderStyle write SetPopupBorderStyle;
    property Margins: TRect read FMargins;
    property BorderSize: integer read FBorderSize;
    property Caption;
    property Color;
    property Font;
  end;

  TDBClipPopup = class(TDCClipPopup)
  private
    FButtonWidth: integer;
    FHintHeight: integer;
    FLinesCount: integer;
    FMaxPos: integer;
    FOnButtonClick: TNotifyEvent;
    function GetActiveButton: TDCEditButton;
  protected
    procedure AddButtons; virtual;
    procedure DrawButtonHint(Sender: TObject; Mode: integer); virtual;
    procedure RedrawBorder; override;
    procedure ButtonClick(Sender: TObject); virtual;
  public
    constructor Create(AOwner: TComponent); override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure Clear;
    function AddButton(AName, AResource, AHint: string; Line,
      Pos: integer): TDCEditButton;
    property OnButtonClick: TNotifyEvent read FOnButtonClick write FOnButtonClick;
    property ActiveButton: TDCEditButton read GetActiveButton;
  end;

  TKNClipPopup = class(TDBClipPopup)
  protected
    procedure AddButtons; override;
  end;

procedure ProcessPaintMessages;
procedure ProcessMovingWindow(Sender: TWinControl; XCursor, YCursor: integer;
  CursorMode: TCursorMode; ItemHeight: integer);

function ShowWindow(Handle: HWND; Alignment: TPopUpAlignment;
   WindowBounds: TRect; AlwaysVisible: boolean; SourceImage: TBitmap = nil; Parent: TControl = nil): integer;
procedure HideWindow(Handle: HWND);

implementation
uses DCResource, DCChoice, Math;

const
  br_SizerWidth    = 14;
  br_FooterHeight  = 13;

  SRCTIMER_IDEVENT = $AE;

type

  TPrivateControl = class(TControl)
  end;
  TPrivateDBGrid = class(TDCDBGrid)
  end;

{ TDCPopupWindow }

function ShowWindow(Handle: HWND; Alignment: TPopUpAlignment;
   WindowBounds: TRect; AlwaysVisible: boolean; SourceImage: TBitmap = nil; Parent: TControl = nil): integer;
 var
  P: TPoint;
  AHeight, AWidth: integer;
  ScreenDC: HDC;
begin
  Result  := 0;
  AHeight :=  WindowBounds.Bottom - WindowBounds.Top;
  AWidth  :=  WindowBounds.Right  - WindowBounds.Left;
  if Parent <> nil then
  begin
    case Alignment of
      wpNone       :
        begin
          Result  := -1;
          P := Point(WindowBounds.Left, WindowBounds.Top);
          //if AlwaysVisible then SetRectInDesktop(P, AWidth, AHeight, Point(0,0));
        end;
      wpBottomLeft :
        begin
          P := Point((Parent.ClientWidth-Parent.Width) div 2,
                     Parent.ClientHeight + (Parent.Height-Parent.ClientHeight) shr 1);
          P := Parent.ClientToScreen(P);
          if AlwaysVisible then
            Result := SetRectInDesktop(P, AWidth, AHeight,
              Point(0,(Screen.DesktopTop+Screen.DesktopHeight)-P.Y+Parent.Height));
        end;
      wpBottomRight:
        begin
          P := Point(Parent.ClientWidth  + (Parent.Width-Parent.ClientWidth) div 2,
                     Parent.ClientHeight + (Parent.Height-Parent.ClientHeight) shr 1);
          P := Parent.ClientToScreen(P);
          P.X := P.X - AWidth;
          if AlwaysVisible then
            Result := SetRectInDesktop(P, AWidth, AHeight,
              Point(0,(Screen.DesktopTop+Screen.DesktopHeight)-P.Y+Parent.Height));
        end;
      wpTopRight   :
        begin
          P := Point(Parent.ClientWidth, -((Parent.Height-Parent.ClientHeight) shr 1));
          P := Parent.ClientToScreen(P);
          if AlwaysVisible then
            Result := SetRectInDesktop(P, AWidth, AHeight,
              Point((Screen.DesktopLeft+Screen.DesktopWidth)-P.X+Parent.ClientWidth+
              (Parent.Width-Parent.ClientWidth) shr 1,0));
        end;
      wpOffset     :
        begin
          P := Point(WindowBounds.Left, WindowBounds.Top);
          P := Parent.ClientToScreen(P);
          if AlwaysVisible then
            Result := SetRectInDesktop(P, AWidth, AHeight, Point(0,0));
        end;
    end;
  end
  else begin
    Result  := -1;
    P := Point(WindowBounds.Left, WindowBounds.Top);
    if AlwaysVisible then
    begin
      if P.Y < 0 then P.Y := 0;
      if (P.Y + AHeight) > Screen.Height then
      begin
        P.Y := Screen.Height - AHeight;
      end;
      if P.X < 0 then P.X := 0;
      if (P.X + AWidth) > Screen.Width then
      begin
        P.X := Screen.Width - AWidth;
      end;
    end;
  end;

  if SourceImage <> nil then
  begin
    ProcessPaintMessages;
    ScreenDC := GetDC(0);
    try
      with SourceImage do begin
        Width  := WindowBounds.Right  - WindowBounds.Left;
        Height := WindowBounds.Bottom - WindowBounds.Top;
        BitBlt(Canvas.Handle, 0, 0, Width, Height, ScreenDC, P.X+1, P.Y+1, SRCCOPY);
      end;
    finally
      ReleaseDC(0, ScreenDC);
    end;
  end;

  SetWindowPos(Handle, HWND_TOPMOST, P.X, P.Y, 0, 0,
    SWP_NOSIZE + SWP_NOZORDER + SWP_NOACTIVATE + SWP_SHOWWINDOW);
  BringWindowToTop(Handle);
end;

procedure ProcessPaintMessages;
 var
  Msg: TMsg;
begin
  while PeekMessage(Msg, 0, WM_PAINT, WM_PAINT, PM_NOREMOVE) do begin
    case Integer(GetMessage(Msg, 0, WM_PAINT, WM_PAINT)) of
      -1: Break;
      0 :
        begin
          PostQuitMessage(Msg.WParam);
          Break;
        end;
    end;
    DispatchMessage(Msg);
  end;
end;


procedure HideWindow(Handle: HWND);
begin
  SetWindowPos(Handle, 0, 0, 0, 0, 0,
    SWP_NOZORDER + SWP_NOMOVE + SWP_NOSIZE + SWP_NOACTIVATE + SWP_HIDEWINDOW);
end;

procedure ProcessMovingWindow(Sender: TWinControl; XCursor, YCursor: integer;
  CursorMode: TCursorMode; ItemHeight: integer);
 var
  ScreenDC: HDC;
  Accept: boolean;
  Msg: TMsg;
  MousePoint: TPoint;
  NextRect, LastRect: TRect;
  AItemHeight,ADelta: integer;
  InfoWindow: TDCMessageWindow;

 procedure UpdateInfoWindow;
 begin
   with InfoWindow do
   begin
     Left := LastRect.Left + 4;
     Top  := LastRect.Top  + 4;
     case CursorMode of
       cmMove:
         Caption := Format('%d, %d', [LastRect.Left, LastRect.Top]);
       cmResize:
         Caption := Format('%d x %d',
           [(LastRect.Right - LastRect.Left), (LastRect.Bottom - LastRect.Top)]);
     end;
     Show;
   end
 end;

 procedure MouseMoved;
  var
   Value,Pos: TPoint;
 begin

   GetCursorPos(Pos);
   Value := Pos;

   { }
   Pos.X := Pos.X - MousePoint.X;
   Pos.Y := Pos.Y - MousePoint.Y;
   NextRect   := LastRect;

   case CursorMode of
     cmResize: {Resize Window}
      begin
        NextRect.Right  := NextRect.Right  + Pos.X;
        if Abs(ADelta) >= (ItemHeight shr 1) then
        begin
          if Abs(ADelta) > ItemHeight then
            AItemHeight := (ADelta div ItemHeight)*ItemHeight
          else
            AItemHeight := (ADelta div Abs(ADelta))*ItemHeight;
          NextRect.Bottom := NextRect.Bottom + AItemHeight;
          Value.Y := NextRect.Bottom;
          ADelta  := 0;
          SetCursorPos(Value.X, Value.Y);
        end
        else
          ADelta := ADelta + Pos.Y;
      end;
     cmMove  : {Move Window}
      OffsetRect(NextRect, Pos.X, Pos.Y);
   end;

   if not EqualRect(NextRect, LastRect) then
   begin
     InfoWindow.Hide;
     DrawFocusedRect(ScreenDC, @LastRect, @NextRect, 2);
     LastRect := NextRect;
     UpdateInfoWindow;
   end;

   MousePoint := Value;

 end;

 procedure Dropped;
  var
   Value: integer;
 begin
   Sender.Perform(CM_SETALIGNMENT, Integer(wpNone), 0);
   if LastRect.Left > LastRect.Right then
   begin
     Value := LastRect.Left;
     LastRect.Left  := LastRect.Right;
     LastRect.Right := Value;
   end;
   if LastRect.Top > LastRect.Bottom then
   begin
     Value := LastRect.Top;
     LastRect.Top   := LastRect.Bottom;
     LastRect.Bottom:= Value;
   end;

   Sender.SetBounds(LastRect.Left, LastRect.Top,
      LastRect.Right-LastRect.Left, LastRect.Bottom-LastRect.Top);

 end;
begin
  Accept := False;
  ADelta := 0;
  MousePoint := Point(XCursor, YCursor);
  with Sender do
  begin
    LastRect   := Rect(Left, Top, Left+Width, Top+Height);

    ProcessPaintMessages;

    {   ,    }
    //LockWindowUpdate(GetDesktopWindow);
    LockWindowUpdate(Parent.Handle);

    ScreenDC := GetDCEx(GetDesktopWindow, 0,
      DCX_LOCKWINDOWUPDATE or DCX_CACHE or DCX_WINDOW);

    InfoWindow  := TDCMessageWindow.Create(nil);

    with InfoWindow do
    begin
      AutoHide := False;
      Parent   := Sender;
      PopupAlignment := wpOffset;
      SaveSource := True;
    end;

    try
      SetCapture(Handle);
      DrawFocusedRect(ScreenDC, nil, @LastRect, 2);
      UpdateInfoWindow;

      while GetCapture = Handle do begin
        case Integer(GetMessage(Msg, 0, 0, 0)) of
          -1: Break;
          0 :
           begin
             PostQuitMessage(Msg.WParam);
             Break;
           end;
        end;

        case Msg.Message of
          WM_KEYDOWN, WM_KEYUP:
            case Msg.WParam of
              VK_ESCAPE:Break;
            end;
          WM_MOUSEMOVE:
            MouseMoved;
          WM_LBUTTONDOWN, WM_LBUTTONDBLCLK:
            Break;
          WM_LBUTTONUP:
            begin
              Accept := True;
              Break;
            end;
          WM_RBUTTONDOWN..WM_MBUTTONDBLCLK:;
          else begin
            TranslateMessage(Msg);
            DispatchMessage(Msg);
          end;
        end;
      end;

    finally
      if GetCapture = Handle then ReleaseCapture;

      { Hide dragging outline and release the DC }
      DrawFocusedRect(ScreenDC, nil, @LastRect, 2);
      ReleaseDC(GetDesktopWindow, ScreenDC);

      LockWindowUpdate(0);
      InfoWindow.Free;
    end;
  end;

  if Accept then Dropped;
end;

procedure TDCPopupWindow.CMShowingChanged(var Message: TMessage);
 var
  AOrientation: integer;
begin
  AOrientation := ShowWindow(Handle, FPopupAlignment, FWindowRect,
    FAlwaysVisible, FSrcImage, Owner);
  if AOrientation <> -1 then FOrientation := AOrientation;
end;

constructor TDCPopupWindow.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FVisible := False;
  ControlStyle := ControlStyle + [csNoDesignVisible, csReplicatable,
    csAcceptsControls, csOpaque];

  Visible := False;
  Canvas.Brush.Style := bsClear;
  FAlwaysVisible := True;
  FOwner := TControl(AOwner);
  SetRectEmpty(FWindowRect);
  FSrcImage := TBitmap.Create;
  FSaveSource := False;
end;


procedure TDCPopupWindow.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  with Params do
  begin
    ExStyle := WS_EX_TOOLWINDOW or WS_EX_TOPMOST ;
    WindowClass.Style := CS_SAVEBITS;
    AddBiDiModeExStyle(ExStyle);
  end;
end;

procedure TDCPopupWindow.CreateWnd;
begin
  inherited CreateWnd;
  if HandleAllocated then
  begin
    Windows.SetParent(Handle, 0);
    CallWindowProc(DefWndProc, Handle, WM_SETFOCUS, 0, 0);
  end;
end;

destructor TDCPopupWindow.Destroy;
begin
  FSrcImage.Free;
  inherited;
end;

function TDCPopupWindow.GetTextHeight(Value: string): integer;
 var
  R: TSize;
begin
  Windows.GetTextExtentPoint(Canvas.Handle, PChar(Value), Length(Value), R);
  Result := R.CY;
end;

function TDCPopupWindow.GetTextWidth(Value: string): integer;
 var
  R: TSize;
begin
  Windows.GetTextExtentPoint(Canvas.Handle, PChar(Value), Length(Value), R);
  Result := R.CX;
end;

procedure TDCPopupWindow.Hide;
begin
  HideWindow(Handle);
  if FSaveSource and (FSrcImage <> nil) then
    BitBlt(GetDC(0), Left+1, Top+1, Width, Height, FSrcImage.Canvas.Handle, 0, 0, SRCCOPY);
  FVisible := False;
end;

procedure TDCPopupWindow.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
begin
  inherited;
  FWindowRect := Rect(Left, Top, Left + Width, Top + Height);
end;

procedure TDCPopupWindow.SetBoundsEx(ALeft, ATop, AWidth,
  AHeight: Integer);
begin
  FWindowRect := Rect(ALeft,ATop,ALeft+AWidth,aTop+AHeight);
  if FVisible then Show;
end;

procedure TDCPopupWindow.SetPopupAlignment(Value: TPopupAlignment);
begin
  if Value <> FPopupAlignment then
  begin
    FPopupAlignment := Value;
    if Visible then Show;
  end;
end;

procedure TDCPopupWindow.Show;
 var
  AOrientation: integer;
begin
  AOrientation := ShowWindow(Handle, FPopupAlignment, FWindowRect,
    FAlwaysVisible, FSrcImage, Owner);
  if AOrientation <> -1 then FOrientation := AOrientation;
  FVisible :=  True;
end;

procedure TDCPopupWindow.WMMouseActivate(var Message: TWMActivate);
begin
  inherited;
  Message.Result := MA_NOACTIVATE;
  SetWindowPos (Handle, HWND_TOP, 0, 0, 0, 0,
     SWP_NOACTIVATE or SWP_NOMOVE or SWP_NOSIZE);
end;

procedure TDCPopupWindow.WMNCCalcSize(var Message: TWMNCCalcSize);
begin
  inherited;
end;

procedure TDCPopupWindow.WMNCPaint(var Message: TWMNCPaint);
begin
  {}
end;

{ TDCPopupListBox }

constructor TDCPopupListBox.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FVisible    := False;
  ControlStyle := ControlStyle + [csNoDesignVisible, csReplicatable,
                                  csAcceptsControls];

  Visible := False;
  
  Canvas.Brush.Style := bsClear;
  FAlwaysVisible := True;
  FOwner := TControl(AOwner);
  SetRectEmpty(FWindowRect);
  Style := lbOwnerDrawVariable;
  FDropDownRows := 7;
  AdjustNewHeight;
end;

procedure TDCPopupListBox.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  with Params do
  begin
    ExStyle := WS_EX_TOOLWINDOW or WS_EX_TOPMOST;
    AddBiDiModeExStyle(ExStyle);
    WindowClass.Style := CS_SAVEBITS;
  end;
end;

procedure TDCPopupListBox.CreateWnd;
begin
  inherited CreateWnd;
  if Parent <> nil then
  begin
    Windows.SetParent(Handle, 0);
    CallWindowProc(DefWndProc, Handle, WM_SETFOCUS, 0, 0);
  end;  
end;

procedure TDCPopupListBox.Hide;
begin
  HideWindow(Handle);
  FVisible := False;
end;

procedure TDCPopupListBox.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
begin
  inherited;
  FWindowRect := Rect(Left,Top,Left+Width,Top+Height);
end;

procedure TDCPopupListBox.SetBoundsEx(ALeft, ATop, AWidth,
  AHeight: Integer);
begin
  FWindowRect := Rect(ALeft,ATop,ALeft+AWidth,aTop+AHeight);
  if FVisible then Show;
end;

procedure TDCPopupListBox.SetPopupAlignment(Value: TPopupAlignment);
begin
  if Value <> FPopupAlignment then
  begin
    FPopupAlignment := Value;
    if Visible then Show;
  end;
end;

procedure TDCPopupListBox.Show;
begin
  SetListHeight(0);
  ShowWindow(Handle, FPopupAlignment, FWindowRect, FAlwaysVisible, nil, Owner);
  FVisible :=  True;
end;

procedure TDCPopupListBox.WMMouseActivate(var Message: TWMActivate);
begin
  inherited;
  Message.Result := MA_NOACTIVATE;
  SetWindowPos (Handle, HWND_TOP, 0, 0, 0, 0,
     SWP_NOACTIVATE or SWP_NOMOVE or SWP_NOSIZE);
end;

procedure TDCPopupListBox.WMNCCalcSize(var Message: TWMNCCalcSize);
begin
  inherited;
  case FPopupBorderStyle of
    brNone  :;
    brSingle:
      begin
        InflateRect(Message.CalcSize_Params^.rgrc[0], -1, -1);
      end;
    brRaised:
      begin
        InflateRect(Message.CalcSize_Params^.rgrc[0], -2, -2);
      end;
  end;
end;

procedure TDCPopupListBox.SetPopupBorderStyle(Value: TPopupBorderStyle);
begin
  if FPopupBorderStyle <> Value then
  begin
    FPopupBorderStyle := Value;
    case FPopupBorderStyle of
      brNone  :FBorderSize := 0;
      brSingle:FBorderSize := 1;
      brRaised:FBorderSize := 2;
    end;
    RecreateWnd;
  end;
end;

procedure TDCPopupListBox.WMNCPaint(var Message: TWMNCPaint);
begin
  inherited;
  RedrawBorder;
end;

procedure TDCPopupListBox.RedrawBorder;
 var
  DC: HDC;
  R: TRect;
  ABrush: HBRUSH;
begin
  DC := GetWindowDC(Handle);
  try
    GetWindowRect(Handle, R);  OffsetRect(R, -R.Left, -R.Top);
    case FPopupBorderStyle of
      brNone:;
      brSingle:
        begin 
          ABrush :=  CreateSolidBrush(clBlack);
          FrameRect( DC, R, ABrush);
          DeleteObject(ABrush);
        end;
      brRaised:
        begin
          DrawEdge(DC, R, BDR_RAISEDOUTER, BF_RECT);
          InflateRect(R, -1, -1);
          DrawEdge(DC, R, BDR_RAISEDINNER, BF_RECT);
        end;
    end;
  finally
    ReleaseDC(Handle, DC);
    DeleteDC(DC);
  end;
end;

procedure TDCPopupListBox.CNDrawItem(var Message: TWMDrawItem);
var
  State: TOwnerDrawState;
begin
  with Message.DrawItemStruct^ do
  begin
    {$IFDEF DELPHI_V5}
       State := TOwnerDrawState(LongRec(itemState).Lo);
    {$ELSE}
       State := TOwnerDrawState(WordRec(LongRec(itemState).Lo).Lo);
    {$ENDIF}
    Canvas.Handle := hDC;
    Canvas.Font := Font;
    Canvas.Brush := Brush;
    if (Integer(itemID) >= 0) and (odSelected in State) then
    begin
      Canvas.Brush.Color := clHighlight;
      Canvas.Font.Color := clHighlightText
    end;
    if Integer(itemID) >= 0 then
      DrawItem(itemID, rcItem, State) else
      Canvas.FillRect(rcItem);
    Canvas.Handle := 0;
  end;

end;

procedure TDCPopupListBox.WMMouseMove(var Message: TWMMouseMove);
 var
  ItemPos: integer;
begin
  inherited;
  with Message do
    ItemPos := ItemAtPos(Point(XPos,YPos), True);
  if ItemPos <> -1 then begin
    ItemIndex := ItemPos;
  end;
end;

procedure TDCPopupListBox.WMFontChange(var Message: TWMFontChange);
begin
  inherited;
  AdjustNewHeight;
end;

procedure TDCPopupListBox.AdjustNewHeight;
var
  DC: HDC;
  SaveFont: HFONT;
  Metrics: TTextMetric;
begin
  DC := GetDC(0);
  SaveFont := SelectObject(DC, Font.Handle);
  try
    GetTextMetrics (DC, Metrics);
    SelectObject(DC, SaveFont);
    ReleaseDC(0, DC);

    ItemHeight := Metrics.tmHeight + 3;
    SelectObject(DC, SaveFont);
  finally
    DeleteObject(DC);
    DeleteObject(SaveFont);
  end;
end;

procedure TDCPopupListBox.SetListHeight(Increment: integer);
 var
  ItemsCount: integer;
  AWindowRect: TRect;
begin
  AWindowRect := FWindowRect;
  if Items.Count < FDropDownRows then
   ItemsCount := Items.Count+Increment
  else
   ItemsCount := Items.Count;

  if ItemsCount > 0 then
  begin
    if ItemsCount > FDropDownRows then
       Height := ItemHeight*FDropDownRows + 2*FBorderSize
    else
       Height := ItemHeight*ItemsCount + 2*FBorderSize
  end
  else
    Height := ItemHeight + 2*FBorderSize;

  AWindowRect.Bottom := FWindowRect.Bottom - FWindowRect.Top + AWindowRect.Top;
  FWindowRect := AWindowRect;
end;

function TDCPopupListBox.DoMouseWheel(Shift: TShiftState;
  WheelDelta: Integer; MousePos: TPoint): Boolean;
 var
  ATopIndex: integer;
begin
  Result := inherited DoMouseWheel(Shift, WheelDelta, MousePos);
  if not Result then
  begin
    ATopIndex := TopIndex - (WheelDelta div WHEEL_DELTA);
    if (ATopIndex >= 0) and (ATopIndex + DropDownRows <= Items.Count) then
      TopIndex := ATopIndex;
    Result := True
  end;
end;

{ TDCMessageWindow }

procedure TDCMessageWindow.CreateParams(var Params: TCreateParams);
begin
  inherited;
  with Params do
    Style := Style or WS_BORDER
end;

procedure TDCMessageWindow.WMNCHitTest(var Message: TWMNCHitTest);
begin
  inherited;
  if FButtons.Count = 0 then Message.Result := HTTRANSPARENT;
end;

procedure TDCMessageWindow.Paint;
 var
  R: TRect;
  OffsetX, OffsetY: integer;
  FImageRgn: HRGN;
begin

  R := ClientRect;
  FImage.Handle := CreateCompatibleBitmap(Canvas.Handle, R.Right - R.Left,
    R.Bottom - R.Top);

  FImageRgn := GetRegion;

  with FImage do
  begin
    Canvas.Brush.Color := Color;
    case FMessageStyle of
      msNormal:
        begin
          Canvas.FillRect(R);
          DrawEdge(Canvas.Handle, R, BDR_RAISEDOUTER, BF_RECT);
        end;
      msRoundRect, msTail:
        begin
          FImage.Canvas.Draw(0, 0, FSrcImage);
          Canvas.Brush.Color := Color;
          PaintRgn(FImage.Canvas.Handle, FImageRgn);
          Canvas.Brush.Color := clBtnShadow;
          FrameRgn(FImage.Canvas.Handle, FImageRgn, FImage.Canvas.Brush.Handle, 1, 1);
        end;
    end;

    Canvas.Font := Font;
    Canvas.Font.Color  := clInfoText;

    OffsetX := 0;
    OffsetY := 0;
    case FMessageStyle of
      msNormal:
        begin
          OffsetX := FMargins.Left;
          OffsetY := FMargins.Top;
        end;
      msRoundRect:
        begin
         OffsetX := FRoundValue-3 + FMargins.Left div 2;
         OffsetY := FRoundValue-3 + FMargins.Top div 2;
        end;
      msTail:
        begin
         OffsetX := FRoundValue-3 + FMargins.Left div 2;
         OffsetY := FRoundValue-3 + FMargins.Top div 2;
         case FOrientation of
           0: OffsetY := OffsetY + FTailValue;
           2: OffsetY := OffsetY + FTailValue;
           4: OffsetY := OffsetY + FTailValue;
         end;
        end;
    end;

    R.Left := R.Left + OffsetX;
    R.Top  := R.Top  + OffsetY;
    if FBitmapVisible and Assigned(FBitmap) then
    begin
       DrawBitmap(Canvas, FBitmap, R, False);
       R.Left := R.Left + FBitmap.Width + FBitmapOffset;
       OffsetX := OffsetX + FBitmap.Width + FBitmapOffset;
    end;

    DrawHighLightText(Canvas, PChar(Text), Rect(OffsetX,OffsetY,Width,Height), 1);
  end;
  Canvas.Draw(0, 0, FImage);
end;

procedure TDCMessageWindow.SetDialogStyle(Value: TDialogStyle);
begin
  if Value <> FDialogStyle then
  begin
    FDialogStyle   := Value;
    FBitmapVisible := True;
    AdjustWindowSize;
  end;
end;

procedure TDCMessageWindow.AdjustWindowSize;
 var
  P: TPoint;
  R: TRect;
begin
  Canvas.Font := Font;
  P := DrawHighLightText(Canvas, PChar(Text), Rect(0, 0, Width, Height), 0);
  if FButtons.Count > 0 then
  begin
    R := FButtons.GetButtonsRect;
    OffsetRect(R, -R.Left, -R.Top);
    P.Y := P.Y + R.Bottom + 5;
    if P.X < R.Right then P.X := R.Right;
  end;

  case FDialogStyle of
    dsSimple      :
       begin
         Color := clMessageWindow;
         FBitmapVisible := False;
         Width  := P.X + FMargins.Left + FMargins.Right;
         Height := P.Y + FMargins.Top  + FMargins.Bottom;
       end;
    dsInvalidValue:
       begin
         Color := clMessageWindow;
         FBitmap.LoadFromResourceName(HInstance, 'DC_MV_INVALIDVALUE');
         if FBitmapVisible then
         begin
           Width  := P.X + FMargins.Left + FMargins.Right + FBitmap.Width + FBitmapOffset;
           Height := Max(P.Y, FBitmap.Height) + FMargins.Top + FMargins.Bottom;
         end
         else begin
           Width  := P.X + FMargins.Left + FMargins.Right;
           Height := P.Y + FMargins.Top  + FMargins.Bottom;
         end;
       end;
    dsCustom      :
       begin
         if FBitmapVisible then
         begin
           Width  := P.X + FMargins.Left + FMargins.Right + FBitmap.Width + FBitmapOffset;
           Height := Max(P.Y, FBitmap.Height) + FMargins.Top + FMargins.Bottom;
         end
         else begin
           Width  := P.X + FMargins.Left + FMargins.Right;
           Height := P.Y + FMargins.Top + FMargins.Bottom;
         end;
       end;
  end;
  case FMessageStyle of
    msNormal:
      begin
        Width  := Width  + 2;
        Height := Height + 2;
      end;  
    msRoundRect:
      begin
        Width  := Width  + 2*(FRoundValue-3);
        Height := Height + 2*(FRoundValue-3);
      end;
    msTail:
      begin
        Width  := Width  + 2*(FRoundValue-3);
        Height := Height + 2*(FRoundValue-3) + FTailValue;
      end;
  end;
end;

constructor TDCMessageWindow.Create(AOwner: TComponent);
begin
  inherited;
  FImage  := TBitmap.Create;
  FBitmap := TBitmap.Create;
  Canvas.Brush.Style := bsClear;
  FDialogStyle := dsSimple;
  FBitmapVisible := True;

  FButtons := TDCEditButtons.Create(Self);
  FButtons.AnchorStyle := asNone;
  FButtons.Color := clBtnFace;
  SetMargins;

  FAutoHide := False;
  FTimeOut  := 2000;

  FImage  := TBitmap.Create;
  FBitmapOffset := 6;
  FRoundValue   := 7;
  FTailValue    := 8;

  FMessageStyle := msNormal;
end;

destructor TDCMessageWindow.Destroy;
begin
  FImage.Free;
  FButtons.Free;
  FButtons := nil;
  FBitmap.Free;
  inherited;
end;

procedure TDCMessageWindow.CreateWnd;
begin
  inherited;
  if Parent <> nil then FButtons.SetWndProc;
end;

procedure TDCMessageWindow.SetBitmap(Value: TBitmap);
begin
  if FBitmap <> Value then begin
     FBitmap.Assign(Value);
     BitmapVisible := True;
  end;
end;

procedure TDCMessageWindow.SetBitmapVisible(Value: boolean);
begin
  if FBitmapVisible <> Value then
  begin
    FBitmapVisible := Value;
    AdjustWindowSize;
  end;
end;

procedure TDCMessageWindow.CMTextChanged(var Message: TMessage);
begin
  inherited;
  case FDialogStyle of
    dsSimple,
    dsInvalidValue : AdjustWindowSize;
    dsCustom       : AdjustWindowSize;
  end;
end;

procedure TDCMessageWindow.Show;
begin
  inherited;
  if FAutoHide then StartTimer(FTimeOut);
end;

function TDCMessageWindow.AddButton(AName, AResource, ACaption: string;
  AClick: TNotifyEvent): TDCEditButton;
 var
  P: TPoint;
  Pos: integer;
  ABounds: TRect;
begin
  if ACaption = '' then ACaption := 'null';

  if FButtons.Count > 0 then
    with FButtons do Pos := Buttons[Count-1].Left - 3
  else
    Pos := Self.Width - 3;

  Result := FButtons.AddButton;
  SetMargins;
  with Result do
  begin
    Name := AName;
    BrushColor   := clBtnFace;
    if AResource <> '' then
      Glyph.LoadFromResourceName(hInstance, PChar(AResource));
    BrushColor   := $E7FFFF;
    Style        := stHint;
    Allignment   := abCentr;
    AnchorStyle  := asBR;
    Font         := Self.Font;
    Caption      := ACaption;
    OnClick      := AClick;
    Self.Canvas.Font  := Font;

    P := DrawHighLightText(Self.Canvas, PChar(ACaption), Rect(0, 0, 0, 0), 0);

    if Assigned(Glyph) then
    begin
      P.X := P.X + Glyph.Width + ButtonOffset + TextBtnOffset;
    end;
    P.X := P.X + 4;
    P.Y := P.Y + 8;
    if FMessageStyle <> msNormal then
      ABounds := Rect(Pos - P.X - FRoundValue + 2,
        Self.Height - P.Y - FMargins.Bottom - FRoundValue + 2, P.X, P.Y)
    else
      ABounds := Rect(Pos-P.X, Self.Height - P.Y - FMargins.Bottom, P.X, P.Y);
    SetBounds(ABounds);
  end;
  AdjustWindowSize;
end;

procedure TDCMessageWindow.CMMouseLeave(var Message: TMessage);
begin
  inherited;
  if Assigned(FButtons) then
    FButtons.UpdateButtons(-1, -1, FButtons.MouseDown, True);
end;

procedure TDCMessageWindow.SetMargins;
begin
  with FMargins do
  begin
    Left  := 3;
    Top   := 2;
    Right := 3;
    Bottom:= 2
  end;
end;

procedure TDCMessageWindow.SetAutoHide(const Value: boolean);
begin
  FAutoHide := Value;
end;

procedure TDCMessageWindow.WMTimer(var Message: TWMTimer);
begin
  inherited;
  StopTimer;
end;

procedure TDCMessageWindow.SetTimeOut(const Value: integer);
begin
  FTimeOut := Value;
end;

procedure TDCMessageWindow.StartTimer(Value: Integer);
begin
  StopTimer;
  if FButtons.Count = 0 then FTimerHandle := SetTimer(Handle, 0, Value, nil);
end;

procedure TDCMessageWindow.StopTimer;
begin
  if FTimerHandle <> 0 then
  begin
    KillTimer(0, FTimerHandle);
    FTimerHandle := 0;
    if Assigned(FOwner) and (FOwner is TWinControl) then
      PostMessage(TWinControl(FOwner).Handle, CM_ERRORMESSAGE, 0, 0)
    else
      Hide;
  end;
end;

procedure TDCMessageWindow.WMEraseBkgnd(var Message: TWmEraseBkgnd);
begin
  Message.Result := 0;
end;

function TDCMessageWindow.GetRegion: HRGN;
 var
  FRect: TRect;
  Tail: HRGN;

  function CreatePolyRgn(const Points: array of TPoint): HRgn;
  type
    PPoints = ^TPoints;
    TPoints = array[0..0] of TPoint;
  begin
    Result := CreatePolygonRgn(PPoints(@Points)^, High(Points) + 1, WINDING);
  end;

begin
  FRect  := ClientRect;
  Result :=  0;
  case FMessageStyle of
    msNormal:
      Result := CreateRectRgnIndirect(FRect);
    msRoundRect:
      Result := CreateRoundRectRgn(FRect.Left, FRect.Top, FRect.Right,
        FRect.Bottom, FRoundValue, FRoundValue);
    msTail:
      begin
        Tail :=  0;
        case FOrientation of
          0:
            begin
              Tail := CreatePolyRgn([Point(8, FTailValue), Point(8, 0),
                Point(8+FTailValue, FTailValue)]);
              FRect.Top := FRect.Top + FTailValue;
            end;
          1:
            begin
              Tail := 0;
              FRect.Bottom := FRect.Bottom - FTailValue;
            end;
          2:
            begin
              Tail := CreatePolyRgn([Point(FRect.Right-8, FTailValue),
                Point(FRect.Right-8, 0), Point(FRect.Right-8-FTailValue, FTailValue)]);
              FRect.Top := FRect.Top + FTailValue;
            end;
          3:
            begin
              Tail := 0;
              FRect.Bottom := FRect.Bottom - FTailValue;
            end;
          4:
            begin
              Tail := CreatePolyRgn([Point((FRect.Left + FRect.Right) div 2 - 4, FTailValue),
                Point((FRect.Left + FRect.Right) div 2, 0),
                Point((FRect.Left + FRect.Right) div 2 + 4, FTailValue)]);
              FRect.Top := FRect.Top + FTailValue;
            end;
        end;
        try
         Result := CreateRoundRectRgn(FRect.Left, FRect.Top, FRect.Right,
           FRect.Bottom, FRoundValue, FRoundValue);
         CombineRgn(Result, Result, Tail, RGN_OR);
        finally
          DeleteObject(Tail);
        end;
        FRect.Top :=  FRect.Top;
      end;
  end;
end;

procedure TDCMessageWindow.SetMessageStyle(const Value: TMessageStyle);
begin
  Hide;
  AdjustWindowSize;  
  FMessageStyle := Value;
end;

{ TDCPopupDBGrid }

procedure TDCPopupDBGrid.AdjustNewHeight;
var
  DC: HDC;
  SaveFont: HFONT;
  Metrics: TTextMetric;
begin
  DC := GetDC(0);
  SaveFont := SelectObject(DC, Font.Handle);
  try
    GetTextMetrics (DC, Metrics);
    SelectObject(DC, SaveFont);
    ReleaseDC (0, DC);

    FItemHeight := Metrics.tmHeight;
    if dgRowLines in Options then
      FItemHeight := FItemHeight + 5
    else
      FItemHeight := FItemHeight + 3;
    SelectObject(DC, SaveFont);
  finally
    DeleteObject(DC);
    DeleteObject(SaveFont);
  end;
end;

constructor TDCPopupDBGrid.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FVisible    := False;
  ControlStyle := ControlStyle + [csNoDesignVisible, csReplicatable,
                                  csAcceptsControls];

  Visible := False;

  Canvas.Brush.Style := bsClear;
  FAlwaysVisible := True;
  FOwner := TControl(AOwner);
  Font   := TPrivateControl(AOwner).Font;
  TitleFont := TPrivateControl(AOwner).Font;

  SetRectEmpty(FWindowRect);
  SetRectEmpty(FMargins);
  FDropDownRows := 8;

  FDataSource := TDataSource.Create(Self);
  FDataSource.DataSet := FDataSet;
  DataSource  := FDataSource;
  AdjustNewHeight;

  {Special Grid properies}

  Options := Options + [dgRowSelect, dgAlwaysShowSelection, dgHighlightRow,
    dgTitleClicked, dgCompleteLines];
  Options := Options - [dgIndicator, dgRowLines];
  ScrollBars  := ssNone;
  FCursorMode := cmNone;

  FButtons := TDCEditButtons.Create(Self);
  FButtons.AnchorStyle := asBL;
  FButtons.Color := clBtnFace;

  FShowHeader := True;

  OnPaintEmptyMessage := PaintEmptyMessage;
end;

procedure TDCPopupDBGrid.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  with Params do
  begin
    ExStyle := WS_EX_TOOLWINDOW or WS_EX_TOPMOST ;
    AddBiDiModeExStyle(ExStyle);
  end;
end;

procedure TDCPopupDBGrid.CreateWnd;
 var
  LeftPos: integer;
  AButton: TDCEditButton;
begin
  inherited CreateWnd;

  if Parent <> nil then
  begin
    Windows.SetParent(Handle, 0);
    CallWindowProc(DefWndProc, Handle, WM_SETFOCUS, 0, 0);
    SetMargins;

    FButtons.SetWndProc;

    if FShowHeader then
    begin
      LeftPos := FMargins.Left+3;
      FButtons.Clear;

      if FCanAppend then begin
        AButton := FButtons.AddButton;
        with AButton do
        begin
          Name := '#Append';
          Allignment  := abLeft;
          AnchorStyle := asBL;
          Font     := Self.Font;
          Color   := Self.Color;
          DrawText := False;
          Glyph.LoadFromResourceName(HInstance, 'DC_SBTNNEW');
          Caption := 'Append';

          SetBounds(Rect(LeftPos, Self.Height-FMargins.Bottom-1,
            Glyph.Width + 5, FMargins.Bottom-1));

          DisableStyle := deNormal;
          Style   := stShadowFlat;
          Enabled := True;
          Visible := False;
          Tag     := 3;
          OnClick := DoButtonClick;
          OnDrawHint := DoDrawHint;
        end;
        LeftPos := LeftPos + AButton.Width + 1;
      end;

      AButton := FButtons.AddButton;
      with AButton do
      begin
        Name := '#Refresh';
        Allignment  := abLeft;
        AnchorStyle := asBL;
        Font     := Self.Font;
        Color   := Self.Color;
        DrawText := False;
        Glyph.LoadFromResourceName(HInstance, 'DC_SBTNREFRESH');
        Caption := 'Refresh';

        SetBounds(Rect(LeftPos, Self.Height-FMargins.Bottom-1,
          Glyph.Width + 5, FMargins.Bottom-1));

        DisableStyle := deNormal;
        Style   := stShadowFlat;
        Enabled := True;
        Visible := False;
        Tag     := 1;
        OnClick := DoButtonClick;
        OnDrawHint := DoDrawHint;
      end;
      LeftPos := LeftPos + AButton.Width + 1;

      FFindButton := FButtons.AddButton;
      with FFindButton do
      begin
        Name := '#SearchText';
        Allignment := abLeft;
        AnchorStyle := asBLR;
        Font    := Self.Font;
        Color   := Self.Color;
        Glyph.LoadFromResourceName(HInstance, 'DC_SBTNFILTER');
        Caption := '';

        SetBounds(Rect(LeftPos, Self.Height-FMargins.Bottom-1,
          Self.Width - LeftPos - 2*FBorderSize - FMargins.Left- FMargins.Right- 15,
          FMargins.Bottom-1));

        DisableStyle := deNormal;
        Style   := stNone;
        Enabled := False;
        Visible := False;
        Tag     := 2;
        OnClick := DoButtonClick;
      end;
    end;
  end;
end;

procedure TDCPopupDBGrid.Hide;
begin
  HideWindow(Handle);
  FVisible := False;
end;

procedure TDCPopupDBGrid.RedrawBorder;
 var
  DC: HDC;
  R: TRect;
  ABrush: HBRUSH;
begin
  DC := GetWindowDC(Handle);
  try
    GetWindowRect (Handle, R);  OffsetRect (R, -R.Left, -R.Top);
    case FPopupBorderStyle of
      brNone:;
      brSingle:
        begin 
          ABrush := CreateSolidBrush(clBlack);
          FrameRect( DC, R, ABrush);
          DeleteObject(ABrush);
        end;
      brRaised:
        begin
          DrawEdge(DC, R, BDR_RAISEDOUTER, BF_RECT);
          InflateRect(R, -1, -1);
          DrawEdge(DC, R, BDR_RAISEDINNER, BF_RECT);

          DrawHeader;
          DrawFooter;
        end;
    end;
  finally
    ReleaseDC(Handle, DC);
    DeleteDC(DC);
  end;
end;

procedure TDCPopupDBGrid.SetPopupBorderStyle(Value: TPopupBorderStyle);
begin
  if FPopupBorderStyle <> Value then
  begin
    FPopupBorderStyle := Value;
    case FPopupBorderStyle of
      brNone  :FBorderSize := 0;
      brSingle:FBorderSize := 1;
      brRaised:FBorderSize := 2;
    end;
    RecreateWnd;
  end;
end;

procedure TDCPopupDBGrid.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
begin
  if AHeight < FItemHeight * 5 then AHeight := FItemHeight * 5;
  if AWidth  < 80 then AWidth  := 80;
  inherited;
  FWindowRect := Rect(Left,Top,Left+Width,Top+Height);
end;

procedure TDCPopupDBGrid.SetBoundsEx(ALeft, ATop, AWidth,
  AHeight: Integer);
begin
  FWindowRect := Rect(ALeft,ATop,ALeft+AWidth,aTop+AHeight);
  if FVisible then Show;
end;

procedure TDCPopupDBGrid.SetPopupAlignment(Value: TPopupAlignment);
begin
  if Value <> FPopupAlignment then
  begin
    FPopupAlignment := Value;
    if Visible then Show;
  end;
end;

procedure TDCPopupDBGrid.Show;
begin
  SetMargins;
  Height := FItemHeight*FDropDownRows + RowHeights[0] + 2*FBorderSize + FMargins.Top + FMargins.Bottom;
  ShowWindow(Handle, FPopupAlignment, FWindowRect, FAlwaysVisible, nil, Owner);
  FVisible  := True;
  if (DataSource <> nil) and (DataSource.DataSet <> nil) then
   FPosition := DataSource.DataSet.Bookmark;

  CheckRefreshButton;
end;

procedure TDCPopupDBGrid.WMMouseActivate(var Message: TWMActivate);
begin
  inherited;
  Message.Result := MA_NOACTIVATE;
  SetWindowPos (Handle, HWND_TOP, 0, 0, 0, 0,
     SWP_NOACTIVATE or SWP_NOMOVE or SWP_NOSIZE);
end;

procedure TDCPopupDBGrid.WMNCCalcSize(var Message: TWMNCCalcSize);
begin
  case FPopupBorderStyle of
    brNone  :FBorderSize := 0;
    brSingle:
      begin
        FBorderSize := 2;
        InflateRect(Message.CalcSize_Params^.rgrc[0], -2, -2);
      end;
    brRaised:
      begin
        FBorderSize := 2;
        InflateRect(Message.CalcSize_Params^.rgrc[0], -2, -2);
      end;
  end;
  with Message.CalcSize_Params^.rgrc[0] do
  begin
    Top    := Top    + FMargins.Top;
    Left   := Left   + FMargins.Left;
    Bottom := Bottom - FMargins.Bottom;
    Right  := Right  - FMargins.Right;
  end;
  inherited;
end;

procedure TDCPopupDBGrid.WMNCPaint(var Message: TWMNCPaint);
begin
  inherited;
  RedrawBorder;
end;

procedure TDCPopupDBGrid.SetDataSet(const Value: TDataSet);
begin
  FDataSet := Value;
  FDataSource.DataSet := FDataSet;
  CheckRefreshButton;
end;

function TDCPopupDBGrid.HighlightCell(DataCol, DataRow: Integer;
  const Value: string; AState: TGridDrawState): Boolean;
begin
  Result :=  inherited HighlightCell(DataCol, DataRow, Value, AState);
end;

procedure TDCPopupDBGrid.WMFontChange(var Message: TWMFontChange);
 var
  i: integer;
begin
  inherited;
  AdjustNewHeight;
  for i := 0 to FButtons.Count-1 do
    FButtons.Buttons[i].Font := Font;
end;

procedure TDCPopupDBGrid.WMLButtonDblClk(var Message: TWMLButtonDown);
begin
  inherited;
end;

procedure TDCPopupDBGrid.WMNCHitTest(var Message: TWMNCHitTest);
 var
  R, WindowR: TRect;
  BS: Integer;
  Button: TDCEditButton;
 function InCaptArea(XPos, YPos: integer): boolean;
 begin
   R := WindowR;
   InflateRect(R, -BS, -BS);
   R.Bottom := R.Top + FMargins.Top;
   Result := PtInRect(R, Point(XPos, YPos));
 end;
 function InSizeArea(XPos, YPos: integer): boolean;
 begin
   R := WindowR;
   InflateRect(R, -BS, -BS);
   R.Top  := R.Bottom - br_FooterHeight;
   R.Left := R.Right  - br_SizerWidth;
   Result := PtInRect(R, Point(XPos, YPos));
 end;
 function InGridArea(XPos, YPos: integer): boolean;
 begin
   R := WindowR;
   InflateRect(R, -BS, -BS);
   R.Left   := R.Left   + FMargins.Left;
   R.Top    := R.Top    + FMargins.Top;
   R.Right  := R.Right  - FMargins.Right;
   R.Bottom := R.Bottom - FMargins.Bottom;
   Result := PtInRect(R, Point(XPos, YPos));
 end;
 function InButtonsArea(XPos, YPos: integer): boolean;
  var
   P: TPoint;
 begin
   P.X := XPos - Left;
   P.Y := YPos - Top;
   Result := FButtons.MouseInButtonArea(P.X, P.Y, Button);
   R := WindowR;
   InflateRect(R, -BS, -BS);
 end;
 function InFooterArea(XPos, YPos: integer): boolean;
 begin
   R := WindowR;
   InflateRect(R, -BS, -BS);
   R.Top  := R.Bottom - br_FooterHeight;
   Result := PtInRect(R, Point(XPos, YPos));
 end;
begin
  inherited;
  if not FShowHeader then begin
    FCursorMode := cmGrid;
    Exit;
  end;

  FCursorMode := cmNone;
  BS := FBorderSize;
  GetWindowRect(Handle, WindowR);
  with Message do
  begin
    if InCaptArea(XPos, YPos) then
    begin
      FCursorMode := cmMove;
      Result := HTBORDER;
    end;

    if InFooterArea(XPos, YPos) then
    begin
      FCursorMode := cmFooter;
      Result := HTBORDER;
    end;

    if InSizeArea(XPos, YPos) then
    begin
      FCursorMode := cmResize;
      Result := HTSIZE;
    end;

    if InGridArea(XPos, YPos) then FCursorMode := cmGrid;

    if InButtonsArea(XPos, YPos) then
    begin
      FCursorMode := cmButtons;
      Result := HTBORDER;
    end;
  end;
end;

procedure TDCPopupDBGrid.SetParent(AParent: TWinControl);
begin
  inherited;
  if (AParent <> nil) and (AParent.Parent <> nil) and
     (AParent is TDCCustomChoiceEdit)
  then begin
    Caption := TDCCustomChoiceEdit(AParent).DBObject.Caption;
  end;
end;

procedure TDCPopupDBGrid.DrawFooter;
 var
  DC: HDC;
  R: TRect;
  Bitmap: TBitmap;
begin
  if not FShowHeader then Exit;
  DC := GetWindowDC(Handle);
  Bitmap := TBitmap.Create;
  try
    GetWindowRect (Handle, R); OffsetRect (R, -R.Left, -R.Top);
    InflateRect(R, -2, -2);
    Bitmap.LoadFromResourceName(HInstance, 'DC_BTNSIZE');
    R.Top := R.Bottom - FMargins.Bottom;
    FillRect(DC, R,  GetSysColorBrush(COLOR_BTNFACE));
    R.Left := R.Right-Bitmap.Width-2;
    R.Top  := R.Bottom-Bitmap.Height-2;
    DrawTransparentBitmap(DC, Bitmap, R, False, Bitmap.Canvas.Pixels[0,0]);
  finally
    Bitmap.Free;
    ReleaseDC(Handle, DC);
    DeleteObject(DC);
  end;
end;

procedure TDCPopupDBGrid.DrawHeader;
 var
  DC: HDC;
  R: TRect;
begin
  if not FShowHeader then Exit;
  DC := GetWindowDC(Handle);
  try
    GetWindowRect (Handle, R);  OffsetRect (R, -R.Left, -R.Top);
    InflateRect(R, -2, -2);
    R.Bottom := R.Top + FMargins.Top;
    FillRect(DC, R,  GetSysColorBrush(COLOR_BTNFACE));
    R.Bottom := R.Bottom - 1;
    DrawCaption(Handle, DC, R, DC_TEXT or DC_SMALLCAP or DC_ACTIVE or DC_GRADIENT);
  finally
    ReleaseDC(Handle, DC);
    DeleteObject(DC);
  end;
end;

procedure TDCPopupDBGrid.WMSetCursor(var Message: TWMSetCursor);
begin
  case FCursorMode of
    cmNone   : SetCursor(Screen.Cursors[crArrow]);
    cmResize : SetCursor(Screen.Cursors[crSizeNWSE]);
    cmMove   : SetCursor(Screen.Cursors[crArrow]);
    cmButtons: SetCursor(Screen.Cursors[crArrow]);
    cmFooter : SetCursor(Screen.Cursors[crArrow]);
    cmGrid   : inherited;
  end;
end;

procedure TDCPopupDBGrid.WMNCLButtonDown(var Message: TWMNCLButtonDown);
begin
  inherited;
  with Message do
  begin
    case FCursorMode of
      cmResize, cmMove: BeginMoving(XCursor, YCursor);
    end;
  end;
end;

procedure TDCPopupDBGrid.BeginMoving(XCursor, YCursor: integer);
begin
  ProcessMovingWindow(Self, XCursor, YCursor, FCursorMode, FItemHeight);
end;

procedure TDCPopupDBGrid.SetMargins;
begin
  FMargins := Rect(0,0,0,0);
  if not FShowHeader then Exit;
  case FPopupBorderStyle of
    brNone  :;
    brSingle:;
    brRaised:
      begin
        // Margins.Properties
        FMargins.Top    := 14;
        FMargins.Bottom := br_FooterHeight+4;
      end;
  end;
end;

destructor TDCPopupDBGrid.Destroy;
begin
  FDataSource.Free;
  FButtons.Free;
  FButtons := nil;
  inherited;
end;

procedure TDCPopupDBGrid.CMMouseEnter(var Message: TMessage);
begin
  inherited;
  if Assigned(FButtons) then
    FButtons.MouseDown := GetAsyncKeyState(VK_LBUTTON) < 0;
end;

procedure TDCPopupDBGrid.CMMouseLeave(var Message: TMessage);
begin
  inherited;
  if Assigned(FButtons) then
    FButtons.UpdateButtons( -1, -1, False, True);
end;

procedure TDCPopupDBGrid.WMPaint(var Message: TWMPaint);
begin
  if Assigned(FButtons) then  FButtons.UpdateDeviceRegion(Message.DC);
  inherited;
  if Assigned(FButtons) then InvalidateButtons;
end;

procedure TDCPopupDBGrid.KeyDown(var Key: Word; Shift: TShiftState);
begin
  inherited;
  case Key of
    VK_LEFT :
      begin
        if ssCtrl in Shift then
          SetBounds(Left-POPUP_MOVE_STEPX, Top, Width, Height);
      end;
    VK_RIGHT:
      begin
        if ssCtrl in Shift then
          SetBounds(Left+POPUP_MOVE_STEPX, Top, Width, Height);
      end;
    VK_UP   :
      begin
        if ssCtrl in Shift then
          SetBounds(Left, Top-POPUP_MOVE_STEPY, Width, Height);
      end;
    VK_DOWN :
      begin
        if ssCtrl in Shift then
          SetBounds(Left, Top+POPUP_MOVE_STEPY, Width, Height);
      end;
  end;
end;

procedure TDCPopupDBGrid.DoButtonClick(Sender: TObject);
 var
  ACursor: TCursor;
begin
  if Assigned(FOnButtonClick) then FOnButtonClick(Sender);
  case TDCEditButton(Sender).Tag of
    1{Refresh}:
      begin
        ACursor := Screen.Cursor;
        Screen.Cursor := crHourGlass;
        try
          try
            if FDataSet <> nil  then
            begin
              FDataSet.Close;
              FDataSet.Open;
            end;  
          except
            on E: Exception do
            begin
              if Assigned(FOwner) and (FOwner is TDCCustomEdit) then
              begin
                TDCCustomEdit(FOwner).ErrorCode := ERR_GRID_EXCEPTONREFRESH;
                TDCCustomEdit(FOwner).ErrorHint := E.Message;
                TDCCustomEdit(FOwner).ShowErrorMessage;
              end;
            end;
          end;
        finally
          Screen.Cursor := ACursor;
        end;
      end;
    3{New}:
      if Assigned(FOwner) and (FOwner is TDCCustomGridEdit) then
        PostMessage(TWinControl(FOwner).Handle, CM_APPENDRECORD, 0, 0)
  end;
end;

procedure TDCPopupDBGrid.WMSize(var Message: TWMSize);
begin
  inherited;
  if Assigned(FButtons) then InvalidateButtons;
end;

procedure TDCPopupDBGrid.InvalidateButtons;
 var
  i, RightPos: integer;
  Button: TDCEditButton;
  Changed: boolean;
begin
  RightPos := Width - br_SizerWidth - FBorderSize - FMargins.Left - 3;
  Changed  := False;
  for i := 0 to FButtons.Count-1 do
  begin
    Button := FButtons.Buttons[i];
    if (Button.Left + Button.Width) > RightPos then
    begin
      if Button.Visible then
      begin
        Button.Visible := False;
        Changed := True;
      end
    end
    else
      if not Button.Visible then
      begin
        Button.Visible := True;
        Changed := True;
      end;
  end;

  if Changed then SendMessage(Self.Handle, WM_NCPAINT, 0, 0);
end;

procedure TDCPopupDBGrid.SetShowHeader(const Value: boolean);
begin
  FShowHeader := Value;
  RecreateWnd;
end;

procedure TDCPopupDBGrid.StartSearch(Key: Char; AValue: string = '');
 var
  ASearch, ADate: string;
  VLocate: variant;
  ACursor:TCursor;
begin
  KillTimer(Handle, SRCTIMER_IDEVENT);
  FFindButton.Enabled := True;

  ACursor := Screen.Cursor;
  Screen.Cursor := crHourGlass;
  if AValue = '' then
  begin
    case Key of
      #8:
        begin
          ASearch := FFindButton.Caption;
          ASearch := Copy(ASearch, 1, Length(ASearch)-1);
        end;
      else
        ASearch := FFindButton.Caption + Key;
    end;
  end
  else
    ASearch := AValue;
  VLocate := null;

  try
    if Assigned(SelectedField) then
    begin
      if SelectedField.FieldKind = fkLookup then
      begin
        if SelectedField.LookUpDataSet.Locate(
             SelectedField.LookUpResultField, ASearch,[loCaseInsensitive,loPartialKey]) then
           DataSet.Locate(SelectedField.KeyFields,
             SelectedField.LookUpDataSet.FieldByName(SelectedField.LookUpKeyFields).AsString,
             [loCaseInsensitive, loPartialKey]);
      end
      else begin
        case SelectedField.DataType of
         ftString, ftWideString:
           VLocate := ASearch;
         ftAutoInc, ftInteger, ftSmallInt, ftWord, ftLargeInt:
           if IsValidInteger(ASearch) then VLocate := StrToInt(ASearch);
         ftFloat:
           if IsValidFloat(ASearch) then VLocate := StrToFloat(ASearch);
         ftCurrency:
           if IsValidCurrency(ASearch, CurrencyDecimals) then VLocate := StrToCurr(ASearch);
         ftDate, ftTime, ftDateTime:
           if DateToStrY2K(ASearch, ADate) then VLocate := StrToDateTime(ADate);
        end;

        if VLocate <> null then
           DataSource.DataSet.Locate(SelectedField.FieldName, VLocate,
             [loCaseInsensitive, loPartialKey]);
      end;
    end;

  except
    ASearch := FFindButton.Caption;
  end;

  FFindButton.Caption := ASearch;
  FFindButton.Invalidate;
  Screen.Cursor := ACursor;
  SetTimer(Handle, SRCTIMER_IDEVENT, 2000, nil);
end;

procedure TDCPopupDBGrid.StopSearch;
begin
  FFindButton.Enabled := False;
  FFindButton.Caption := '';
  FFindButton.Invalidate;
  KillTimer(Handle, SRCTIMER_IDEVENT)
end;

procedure TDCPopupDBGrid.KeyPress(var Key: Char);
begin
  inherited;
  StartSearch(Key);
end;

procedure TDCPopupDBGrid.WMTimer(var Message: TWMTimer);
begin
  inherited;
  if Message.TimerID = SRCTIMER_IDEVENT then StopSearch;
end;

procedure TDCPopupDBGrid.SetCanAppend(const Value: boolean);
begin
  if FCanAppend <> Value then
  begin
    FCanAppend := Value;
    RecreateWnd;
  end;
end;

procedure TDCPopupDBGrid.DoDrawHint(Sender: TObject; Mode: Integer);
begin
  {}
end;

procedure TDCPopupDBGrid.CheckRefreshButton;
 var
  Button: TDCEditButton;
begin
  Button := FButtons.FindButton('#Refresh');
  if Button <> nil then
  begin
    Button.Visible := FDataSet <> nil;
  end;
end;

procedure TDCPopupDBGrid.CMSetAlignment(var Message: TMessage);
begin
  PopupAlignment := TPopupAlignment(Message.WParam);
end;

procedure TDCPopupDBGrid.CMHintShow(var Message: TCMHintShow);
begin
  inherited;
end;

procedure TDCPopupDBGrid.WMNCMouseMove(var Message: TWMNCMouseMove);
begin
  inherited;
end;

function TDCPopupDBGrid.MouseUpBeforeDblClk: boolean;
begin
  Result := True;
end;

procedure TDCPopupDBGrid.PaintEmptyMessage(Sender: TObject;
  Canvas: TCanvas; ARect: TRect; UpdateMessage: string);
begin
  ARect.Top := ARect.Top + FMargins.Top;
  ARect.Bottom := ARect.Bottom - FMargins.Bottom;
  Canvas.FillRect(ARect);
  InflateRect(ARect, -5, -5);
  DrawHighLightText(Canvas, Pchar(UpdateMessage), ARect, 1, DT_END_ELLIPSIS);
end;

{ TDCPopupTreeView }

procedure TDCPopupTreeView.AdjustNewHeight;
var
  DC: HDC;
  SaveFont: HFONT;
  Metrics: TTextMetric;
begin
  DC := GetDC(0);
  SaveFont := SelectObject(DC, Font.Handle);
  try
    GetTextMetrics (DC, Metrics);
    SelectObject(DC, SaveFont);
    ReleaseDC (0, DC);

    FItemHeight := Metrics.tmHeight;
    FItemHeight := FItemHeight + 3;
    SelectObject(DC, SaveFont);
  finally
    DeleteObject(DC);
    DeleteObject(SaveFont);
  end;
end;

procedure TDCPopupTreeView.BeginMoving(XCursor, YCursor: integer);
begin
  ProcessMovingWindow(Self, XCursor, YCursor, FCursorMode, FItemHeight);
end;

procedure TDCPopupTreeView.CMMouseEnter(var Message: TMessage);
begin
  inherited;
  if Assigned(FButtons) then
    FButtons.MouseDown := GetAsyncKeyState(VK_LBUTTON)<0;
end;

procedure TDCPopupTreeView.CMMouseLeave(var Message: TMessage);
begin
  inherited;
  if Assigned(FButtons) then
    FButtons.UpdateButtons( -1, -1, False, True);
end;

procedure TDCPopupTreeView.CMSetAlignment(var Message: TMessage);
begin
  PopupAlignment := TPopupAlignment(Message.WParam);
end;

constructor TDCPopupTreeView.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FVisible    := False;
  
  ControlStyle := ControlStyle + [csNoDesignVisible, csReplicatable,
                                  csAcceptsControls, csOpaque];
  Visible := False;

  Canvas.Brush.Style := bsClear;
  FAlwaysVisible := True;
  FOwner := TControl(AOwner);

  SetRectEmpty(FWindowRect);
  SetRectEmpty(FMargins);
  FDropDownRows := 10;

  AdjustNewHeight;

  BorderStyle := bsNone;

  FCursorMode := cmNone;
  FButtons := TDCEditButtons.Create(Self);
  FButtons.AnchorStyle := asBL;
  FButtons.Color := clBtnFace;

  ReadOnly    := True;
  FShowHeader := True;
end;

procedure TDCPopupTreeView.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  with Params do
  begin
    ExStyle := WS_EX_TOOLWINDOW or WS_EX_TOPMOST;
    Style   := Style and not(WS_HSCROLL or WS_VSCROLL);
  end;
end;

procedure TDCPopupTreeView.CreateWnd;
begin
  if Parent <> nil then
  begin
    inherited CreateWnd;
    Windows.SetParent(Handle, 0);
    CallWindowProc(DefWndProc, Handle, WM_SETFOCUS, 0, 0);
    SetMargins;
  end;
end;

destructor TDCPopupTreeView.Destroy;
begin
  FButtons.Free;
  FButtons := nil;
  inherited;
end;

procedure TDCPopupTreeView.DrawFooter;
 var
  DC: HDC;
  R: TRect;
  Bitmap: TBitmap;
begin
  if not FShowHeader then Exit;
  DC := GetWindowDC(Handle);
  Bitmap := TBitmap.Create;
  try
    GetWindowRect (Handle, R); OffsetRect (R, -R.Left, -R.Top);
    InflateRect(R, -2, -2);
    Bitmap.Canvas.Brush.Color := COLOR_BTNFACE;
    Bitmap.LoadFromResourceName(HInstance, 'DC_BTNSIZE');
    R.Top := R.Bottom - FMargins.Bottom;
    FillRect(DC, R,  GetSysColorBrush(COLOR_BTNFACE));
    R.Left := R.Right-Bitmap.Width-2;
    R.Top  := R.Bottom-Bitmap.Height-2;
    DrawTransparentBitmap(DC, Bitmap, R, False, Bitmap.Canvas.Pixels[0,0]);
  finally
    Bitmap.Free;
    ReleaseDC(Handle, DC);
    DeleteObject(DC);
  end;
end;

procedure TDCPopupTreeView.DrawHeader;
 var
  DC: HDC;
  R: TRect;
begin
  if not FShowHeader then Exit;
  DC := GetWindowDC(Handle);
  try
    GetWindowRect (Handle, R);  OffsetRect (R, -R.Left, -R.Top);
    InflateRect(R, -2, -2);
    R.Bottom := R.Top + FMargins.Top;
    FillRect(DC, R,  GetSysColorBrush(COLOR_BTNFACE));
    R.Bottom := R.Bottom - 1;
    DrawCaption(Handle, DC, R, DC_TEXT or DC_SMALLCAP or DC_ACTIVE or DC_GRADIENT);
  finally
    ReleaseDC(Handle, DC);
    DeleteObject(DC);
  end;
end;

procedure TDCPopupTreeView.Hide;
begin
  FButtons.ClrWndProc;
  HideWindow(Handle);
  FVisible := False;
end;

procedure TDCPopupTreeView.InvalidateButtons;
 var
  i, RightPos: integer;
  Button: TDCEditButton;
  Changed: boolean;
begin
  RightPos := Width-br_SizerWidth-FBorderSize-FMargins.Left-3;
  Changed  := False;
  for i := 0 to FButtons.Count-1 do
  begin
    Button := FButtons.Buttons[i];
    if (Button.Left+Button.Width) > RightPos then
    begin
      if Button.Visible then
      begin
        Button.Visible := False;
        Changed := True;
      end
    end
    else
      if not Button.Visible then
      begin
        Button.Visible := True;
        Changed := True;
      end;
  end;
 if Changed then
     SendMessage(Self.Handle, WM_NCPAINT, 0, 0);
end;

procedure TDCPopupTreeView.KeyDown(var Key: Word; Shift: TShiftState);
begin
  inherited;
  case Key of
    VK_LEFT :
      begin
        if ssCtrl in Shift then
          SetBounds(Left-POPUP_MOVE_STEPX, Top, Width, Height);
      end;
    VK_RIGHT:
      begin
        if ssCtrl in Shift then
          SetBounds(Left+POPUP_MOVE_STEPX, Top, Width, Height);
      end;
    VK_UP   :
      begin
        if ssCtrl in Shift then
          SetBounds(Left, Top-POPUP_MOVE_STEPY, Width, Height);
      end;
    VK_DOWN :
      begin
        if ssCtrl in Shift then
          SetBounds(Left, Top+POPUP_MOVE_STEPY, Width, Height);
      end;
  end;
end;

procedure TDCPopupTreeView.KeyPress(var Key: Char);
begin
  case Key of
    '+': if Selected <> nil then Selected.Expand(False);
    '-': if Selected <> nil then Selected.Collapse(False);
    '*': FullExpand;
  end;
end;

procedure TDCPopupTreeView.RedrawBorder;
 var
  DC: HDC;
  R: TRect;
  ABrush: HBRUSH;
begin
  if not ShowScrollBar(Handle, SB_HORZ, False) then Exit;
  DC := GetWindowDC(Handle);
  try
    GetWindowRect(Handle, R);  OffsetRect(R, -R.Left, -R.Top);

    case FPopupBorderStyle of
      brNone:;
      brSingle:
        begin 
          ABrush := CreateSolidBrush(clBlack);
          FrameRect( DC, R, ABrush);
          DeleteObject(ABrush);
        end;
      brRaised:
        begin
          DrawEdge(DC, R, BDR_RAISEDOUTER, BF_RECT);
          InflateRect(R, -1, -1);
          DrawEdge(DC, R, BDR_RAISEDINNER, BF_RECT);

          DrawHeader;
          DrawFooter;
        end;
    end;
  finally
    ReleaseDC(Handle, DC);
    DeleteDC(DC);
  end;
end;

procedure TDCPopupTreeView.SetBounds(ALeft, ATop, AWidth,
  AHeight: Integer);
begin
  if AHeight < FItemHeight*5 then AHeight := FItemHeight*5;
  if AWidth  < 80 then AWidth  := 80;
  inherited;
  FWindowRect := Rect(Left,Top,Left+Width,Top+Height);
end;

procedure TDCPopupTreeView.SetBoundsEx(ALeft, ATop, AWidth,
  AHeight: Integer);
begin
  FWindowRect := Rect(ALeft,ATop,ALeft+AWidth,aTop+AHeight);
  if FVisible then Show;
end;

procedure TDCPopupTreeView.SetMargins;
begin
  FMargins := Rect(0,0,0,0);
  if not FShowHeader then Exit;
  case FPopupBorderStyle of
    brNone  :;
    brSingle:;
    brRaised:
      begin
        // Margins.Properties
        FMargins.Top    := 14;
        FMargins.Bottom := br_FooterHeight+4;
      end;
  end;
end;

procedure TDCPopupTreeView.SetParent(AParent: TWinControl);
begin
  inherited;
end;

procedure TDCPopupTreeView.SetPopupAlignment(Value: TPopupAlignment);
begin
  if Value <> FPopupAlignment then
  begin
    FPopupAlignment := Value;
    if Visible then Show;
  end;
end;

procedure TDCPopupTreeView.SetPopupBorderStyle(Value: TPopupBorderStyle);
begin
  if FPopupBorderStyle <> Value then
  begin
    FPopupBorderStyle := Value;
    case FPopupBorderStyle of
      brNone  :FBorderSize := 0;
      brSingle:FBorderSize := 1;
      brRaised:FBorderSize := 2;
    end;
    RecreateWnd;
  end;
end;

procedure TDCPopupTreeView.SetShowHeader(const Value: boolean);
begin
  FShowHeader := Value;
  RecreateWnd;
end;

procedure TDCPopupTreeView.Show;
begin
  SetMargins;
  Height := FItemHeight*FDropDownRows + 2*FBorderSize + FMargins.Top + FMargins.Bottom;
  ShowWindow(Handle, FPopupAlignment, FWindowRect, FAlwaysVisible, nil, Owner);
  FVisible :=  True;
end;

procedure TDCPopupTreeView.WMActivate(var Message: TWMActivate);
 var
  ParentForm: TCustomForm;
begin
  inherited;
  ParentForm := GetParentForm(Self);
  if Assigned(ParentForm) and ParentForm.HandleAllocated then
  begin
    SendMessage (ParentForm.Handle, WM_NCACTIVATE, Ord(Message.Active <> WA_INACTIVE), 0);
    if Message.ActiveWindow <> ParentForm.Handle then
      SetActiveWindow (ParentForm.Handle);
  end;
end;

procedure TDCPopupTreeView.WMFontChange(var Message: TWMFontChange);
 var
  i: integer;
begin
  inherited;
  AdjustNewHeight;
  for i := 0 to FButtons.Count-1 do
    FButtons.Buttons[i].Font := Font;
end;

procedure TDCPopupTreeView.WMHScroll(var Message: TWMHScroll);
begin
  inherited;
end;

procedure TDCPopupTreeView.WMLButtonDblClk(var Message: TWMLButtonDown);
 var
  HitTest: THitTests;
begin
  HitTest := GetHitTestInfoAt(Message.XPos, Message.YPos);
  if not(htOnButton in HitTest) then
  begin
    SendMessage(TWinControl(FOwner).Handle, Message.Msg, $AE, TMessage(Message).LParam);
    Message.Msg    := 0;
    inherited;
  end
  else
    Perform(WM_LBUTTONDOWN, TMessage(Message).WParam, TMessage(Message).LParam);
end;

procedure TDCPopupTreeView.WMMouseActivate(var Message: TWMActivate);
begin
  inherited;
  Message.Result := MA_NOACTIVATE;
  SetWindowPos (Handle, HWND_TOP, 0, 0, 0, 0,
     SWP_NOACTIVATE or SWP_NOMOVE or SWP_NOSIZE);
end;

procedure TDCPopupTreeView.WMNCCalcSize(var Message: TWMNCCalcSize);
begin
  case FPopupBorderStyle of
    brNone  :FBorderSize := 0;
    brSingle:
      begin
        FBorderSize := 2;
        InflateRect(Message.CalcSize_Params^.rgrc[0], -2, -2);
      end;
    brRaised:
      begin
        FBorderSize := 2;
        InflateRect(Message.CalcSize_Params^.rgrc[0], -2, -2);
      end;
  end;
  with Message.CalcSize_Params^.rgrc[0] do
  begin
    Top    := Top    + FMargins.Top;
    Left   := Left   + FMargins.Left;
    Bottom := Bottom - FMargins.Bottom;
    Right  := Right  - FMargins.Right;
  end;

  inherited;
end;

procedure TDCPopupTreeView.WMNCHitTest(var Message: TWMNCHitTest);
 var
  R, WindowR: TRect;
  BS: Integer;
  Button: TDCEditButton;
 function InCaptArea(XPos, YPos: integer): boolean;
 begin
   R := WindowR;
   InflateRect(R, -BS, -BS);
   R.Bottom := R.Top + FMargins.Top;
   Result := PtInRect(R, Point(XPos, YPos));
 end;
 function InSizeArea(XPos, YPos: integer): boolean;
 begin
   R := WindowR;
   InflateRect(R, -BS, -BS);
   R.Top  := R.Bottom - br_FooterHeight;
   R.Left := R.Right  - br_SizerWidth;
   Result := PtInRect(R, Point(XPos, YPos));
 end;
 function InGridArea(XPos, YPos: integer): boolean;
 begin
   R := WindowR;
   InflateRect(R, -BS, -BS);
   R.Left   := R.Left   + FMargins.Left;
   R.Top    := R.Top    + FMargins.Top;
   R.Right  := R.Right  - FMargins.Right;
   R.Bottom := R.Bottom - FMargins.Bottom;
   Result := PtInRect(R, Point(XPos, YPos));
 end;
 function InButtonsArea(XPos, YPos: integer): boolean;
  var
   P: TPoint;
 begin
   P.X := XPos - Left;
   P.Y := YPos - Top;
   Result := FButtons.MouseInButtonArea(P.X, P.Y, Button);
   R := WindowR;
   InflateRect(R, -BS, -BS);
 end;
 function InFooterArea(XPos, YPos: integer): boolean;
 begin
   R := WindowR;
   InflateRect(R, -BS, -BS);
   R.Top  := R.Bottom - br_FooterHeight;
   Result := PtInRect(R, Point(XPos, YPos));
 end;
begin
  inherited;
  if not FShowHeader then begin
    FCursorMode := cmGrid;
    Exit;
  end;
  FCursorMode := cmNone;
  BS := FBorderSize;
  GetWindowRect(Handle, WindowR);
  with Message do
  begin
    if InCaptArea(XPos, YPos) then
    begin
      FCursorMode := cmMove;
      Result := HTBORDER;
    end;

    if InFooterArea(XPos, YPos) then
    begin
      FCursorMode := cmFooter;
      Result := HTBORDER;
    end;

    if InSizeArea(XPos, YPos) then
    begin
      FCursorMode := cmResize;
      Result := HTSIZE;
    end;
    if InGridArea(XPos, YPos) then FCursorMode := cmGrid;

    if InButtonsArea(XPos, YPos) then
    begin
      FCursorMode := cmButtons;
      Result := HTBORDER;
    end;
  end;
end;

procedure TDCPopupTreeView.WMNCLButtonDown(var Message: TWMNCLButtonDown);
begin
  inherited;
  with Message do
  begin
    case FCursorMode of
      cmResize: BeginMoving(XCursor, YCursor);
      cmMove  : BeginMoving(XCursor, YCursor);
    end;
  end;
end;

procedure TDCPopupTreeView.WMNCPaint(var Message: TWMNCPaint);
begin
  inherited;
  RedrawBorder;
end;

procedure TDCPopupTreeView.WMPaint(var Message: TWMPaint);
begin
  if Assigned(FButtons) then FButtons.UpdateDeviceRegion(Message.DC);
  inherited;
  if Assigned(FButtons) then InvalidateButtons;
end;

procedure TDCPopupTreeView.WMSetCursor(var Message: TWMSetCursor);
begin
  case FCursorMode of
    cmNone   : SetCursor(Screen.Cursors[crArrow]);
    cmResize : SetCursor(Screen.Cursors[crSizeNWSE]);
    cmMove   : SetCursor(Screen.Cursors[crArrow]);
    cmButtons: SetCursor(Screen.Cursors[crArrow]);
    cmFooter : SetCursor(Screen.Cursors[crArrow]);
    cmGrid   : inherited;
  end;
end;

procedure TDCPopupTreeView.WMShowWindow(var Message: TWMShowWindow);
begin
  inherited;
end;

procedure TDCPopupTreeView.WMSize(var Message: TWMSize);
begin
  inherited;
  if Assigned(FButtons) then InvalidateButtons;
end;

{ TDCClipPopup }

procedure TDCClipPopup.BeginMoving(XCursor, YCursor, Delta: integer);
begin
  ProcessMovingWindow(Self, XCursor, YCursor, FCursorMode, Delta);
end;

procedure TDCClipPopup.CMSetAlignment(var Message: TMessage);
begin
  PopupAlignment := TPopupAlignment(Message.WParam);
end;

constructor TDCClipPopup.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  SetRectEmpty(FMargins);
  FCursorMode := cmNone;
  FButtons := TDCEditButtons.Create(Self);
  FButtons.AnchorStyle := asNone;
  FButtons.Color := clBtnFace;

  FOptions := [];
end;

procedure TDCClipPopup.CreateWnd;
begin
  inherited;
  if Parent <> nil then
  begin
    FButtons.ClrWndProc;
    FButtons.SetWndProc;
  end;
end;

destructor TDCClipPopup.Destroy;
begin
  FButtons.Free;
  FButtons := nil;
  inherited;
end;

procedure TDCClipPopup.DrawFooter;
 var
  DC: HDC;
  R: TRect;
  Bitmap: TBitmap;
begin
  if not(coFooter in FOptions) then Exit;
  DC := GetWindowDC(Handle);
  Bitmap := TBitmap.Create;
  try
    GetWindowRect (Handle, R); OffsetRect (R, -R.Left, -R.Top);
    InflateRect(R, -2, -2);
    Bitmap.LoadFromResourceName(HInstance, 'DC_BTNSIZE');
    R.Top := R.Bottom - FMargins.Bottom;
    FillRect(DC, R,  GetSysColorBrush(COLOR_BTNFACE));
    R.Left := R.Right-Bitmap.Width-2;
    R.Top  := R.Bottom-Bitmap.Height-2;
    DrawTransparentBitmap(DC, Bitmap, R, False, Bitmap.Canvas.Pixels[0,0]);
  finally
    Bitmap.Free;
    ReleaseDC(Handle, DC);
    DeleteObject(DC);
  end;
end;

procedure TDCClipPopup.DrawHeader;
 var
  DC: HDC;
  R: TRect;
begin
  if not(coHeader in FOptions) then Exit;
  DC := GetWindowDC(Handle);
  try
    GetWindowRect (Handle, R);  OffsetRect (R, -R.Left, -R.Top);
    InflateRect(R, -2, -2);
    R.Bottom := R.Top + FMargins.Top;
    FillRect(DC, R,  GetSysColorBrush(COLOR_BTNFACE));
    R.Bottom := R.Bottom - 1;
    DrawCaption(Handle, DC, R, DC_TEXT or DC_SMALLCAP or DC_ACTIVE or DC_GRADIENT);
  finally
    ReleaseDC(Handle, DC);
    DeleteObject(DC);
  end;
end;

procedure TDCClipPopup.Hide;
begin
  inherited;
end;

procedure TDCClipPopup.InvalidateButtons;
begin
  {}
end;

procedure TDCClipPopup.KeyDown(var Key: Word; Shift: TShiftState);
begin
  inherited;
  if coHeader in FOptions then
  begin
    case Key of
      VK_LEFT :
        begin
          if ssCtrl in Shift then
            SetBounds(Left-POPUP_MOVE_STEPX, Top, Width, Height);
        end;
      VK_RIGHT:
        begin
          if ssCtrl in Shift then
            SetBounds(Left+POPUP_MOVE_STEPX, Top, Width, Height);
        end;
      VK_UP   :
        begin
          if ssCtrl in Shift then
            SetBounds(Left, Top-POPUP_MOVE_STEPY, Width, Height);
        end;
      VK_DOWN :
        begin
          if ssCtrl in Shift then
            SetBounds(Left, Top+POPUP_MOVE_STEPY, Width, Height);
        end;
    end;
  end;
end;

procedure TDCClipPopup.RedrawBorder;
 var
  DC: HDC;
  R: TRect;
  ABrush: HBRUSH;
begin
  DC := GetWindowDC(Handle);
  try
    GetWindowRect (Handle, R);  OffsetRect (R, -R.Left, -R.Top);
    Canvas.FillRect(R);
    case FPopupBorderStyle of
      brNone:;
      brSingle:
        begin 
          ABrush := CreateSolidBrush(clBlack);
          FrameRect( DC, R, ABrush);
          DeleteObject(ABrush);
        end;
      brRaised:
        begin
          DrawEdge(DC, R, BDR_RAISEDOUTER, BF_BOTTOMRIGHT);
          DrawEdge(DC, R, BDR_SUNKENOUTER, BF_TOPLEFT);
          InflateRect(R, -1, -1);
          DrawEdge(DC, R, BDR_RAISEDINNER, BF_RECT);

          DrawHeader;
          DrawFooter;
        end;
    end;
  finally
    ReleaseDC(Handle, DC);
    DeleteDC(DC);
  end;
end;

procedure TDCClipPopup.SetMargins;
begin
  FMargins := Rect(0,0,0,0);
  case FPopupBorderStyle of
    brNone  :;
    brSingle:;
    brRaised:
      begin
        // Margins.Properties
        if coHeader in FOptions then FMargins.Top    := 14;
        if coFooter in FOptions then FMargins.Bottom := br_FooterHeight+4;
      end;
  end;
end;

procedure TDCClipPopup.SetOptions(const Value: TClipFormOptions);
begin
  FOptions := Value;
  SetMargins;
  RecreateWnd;
end;

procedure TDCClipPopup.SetPopupBorderStyle(Value: TPopupBorderStyle);
begin
  if FPopupBorderStyle <> Value then
  begin
    FPopupBorderStyle := Value;
    case FPopupBorderStyle of
      brNone  :FBorderSize := 0;
      brSingle:FBorderSize := 1;
      brRaised:FBorderSize := 2;
    end;
    SetMargins;
    RecreateWnd;
  end;
end;

procedure TDCClipPopup.Show;
begin
  FButtons.ResetProperties;
  SetCapture(Self.Handle);
  inherited;
end;

procedure TDCClipPopup.WMActivate(var Message: TWMActivate);
 var
  ParentForm: TCustomForm;
begin
  inherited;
  ParentForm := GetParentForm(Self);
  if Assigned(ParentForm) and ParentForm.HandleAllocated then
  begin
    SendMessage (ParentForm.Handle, WM_NCACTIVATE, Ord(Message.Active <> WA_INACTIVE), 0);
    if Message.ActiveWindow <> ParentForm.Handle then
      SetActiveWindow (ParentForm.Handle);
  end;
end;

procedure TDCClipPopup.WMNCCalcSize(var Message: TWMNCCalcSize);
begin
  case FPopupBorderStyle of
    brNone  :FBorderSize := 0;
    brSingle:
      begin
        FBorderSize := 2;
        InflateRect(Message.CalcSize_Params^.rgrc[0], -2, -2);
      end;
    brRaised:
      begin
        FBorderSize := 2;
        InflateRect(Message.CalcSize_Params^.rgrc[0], -2, -2);
      end;
  end;
  with Message.CalcSize_Params^.rgrc[0] do
  begin
    Top    := Top    + FMargins.Top;
    Left   := Left   + FMargins.Left;
    Bottom := Bottom - FMargins.Bottom;
    Right  := Right  - FMargins.Right;
  end;
  inherited;
end;

procedure TDCClipPopup.WMNCHitTest(var Message: TWMNCHitTest);
 var
  R, WindowR: TRect;
  BS: Integer;
  Button: TDCEditButton;
 function InCaptArea(XPos, YPos: integer): boolean;
 begin
   if not(coHeader in FOptions) then
   begin
     Result := False;
     Exit;
   end;
   R := WindowR;
   InflateRect(R, -BS, -BS);
   R.Bottom := R.Top + FMargins.Top;
   Result := PtInRect(R, Point(XPos, YPos));
 end;
 function InSizeArea(XPos, YPos: integer): boolean;
 begin
   if not(coFooter in FOptions) then
   begin
     Result := False;
     Exit;
   end;
   R := WindowR;
   InflateRect(R, -BS, -BS);
   R.Top  := R.Bottom - br_FooterHeight;
   R.Left := R.Right  - br_SizerWidth;
   Result := PtInRect(R, Point(XPos, YPos));
 end;
 function InGridArea(XPos, YPos: integer): boolean;
 begin
   R := WindowR;
   InflateRect(R, -BS, -BS);
   R.Left   := R.Left   + FMargins.Left;
   R.Top    := R.Top    + FMargins.Top;
   R.Right  := R.Right  - FMargins.Right;
   R.Bottom := R.Bottom - FMargins.Bottom;
   Result := PtInRect(R, Point(XPos, YPos));
 end;
 function InButtonsArea(XPos, YPos: integer): boolean;
  var
   P: TPoint;
 begin
   P.X := XPos - Left;
   P.Y := YPos - Top;
   Result := FButtons.MouseInButtonArea(P.X, P.Y, Button);
   R := WindowR;
   InflateRect(R, -BS, -BS);
 end;
 function InFooterArea(XPos, YPos: integer): boolean;
 begin
   if not(coFooter in FOptions) then
   begin
     Result := False;
     Exit;
   end;
   R := WindowR;
   InflateRect(R, -BS, -BS);
   R.Top  := R.Bottom - br_FooterHeight;
   Result := PtInRect(R, Point(XPos, YPos));
 end;
begin
  inherited;
  FCursorMode := cmNone;
  BS := FBorderSize;
  GetWindowRect(Handle, WindowR);
  with Message do
  begin
    if InCaptArea(XPos, YPos) then
    begin
      FCursorMode := cmMove;
      Result := HTBORDER;
    end;

    if InFooterArea(XPos, YPos) then
    begin
      FCursorMode := cmFooter;
      Result := HTBORDER;
    end;

    if InSizeArea(XPos, YPos) then
    begin
      FCursorMode := cmResize;
      Result := HTSIZE;
    end;
    if InGridArea(XPos, YPos) then FCursorMode := cmGrid;

    if InButtonsArea(XPos, YPos) then
    begin
      FCursorMode := cmButtons;
      Result := HTBORDER;
    end;
  end;
end;

procedure TDCClipPopup.WMNCLButtonDown(var Message: TWMNCLButtonDown);
begin
  inherited;
  with Message do
  begin
    case FCursorMode of
      cmResize: BeginMoving(XCursor, YCursor, 1);
      cmMove  : BeginMoving(XCursor, YCursor, 1);
    end;
  end;
end;

procedure TDCClipPopup.WMNCPaint(var Message: TWMNCPaint);
begin
  inherited;
  RedrawBorder;
end;

procedure TDCClipPopup.WMPaint(var Message: TWMPaint);
begin
  inherited;
  InvalidateButtons;
  RedrawBorder;
end;

procedure TDCClipPopup.WMSetCursor(var Message: TWMSetCursor);
begin
  case FCursorMode of
    cmNone   : SetCursor(Screen.Cursors[crArrow]);
    cmResize : SetCursor(Screen.Cursors[crSizeNWSE]);
    cmMove   : SetCursor(Screen.Cursors[crArrow]);
    cmButtons: SetCursor(Screen.Cursors[crArrow]);
    cmFooter : SetCursor(Screen.Cursors[crArrow]);
    cmGrid   : inherited;
  end;
end;

procedure TDCClipPopup.WMSize(var Message: TWMSize);
begin
  inherited;
  InvalidateButtons;
end;

{ TDBClipPopup }

function TDBClipPopup.AddButton(AName, AResource, AHint: string;
  Line, Pos: integer): TDCEditButton;
begin
  Result := Buttons.AddButton;
  with Result do
  begin
    Name := AName;
    SetBounds(Rect(3+Pos*FButtonWidth,Margins.Top+3+Line*FButtonWidth,FButtonWidth,FButtonWidth));
    BrushColor := clBtnFace;
    Enabled := Enabled;
    Glyph.LoadFromResourceName(HInstance, PChar(AResource));
    BrushColor := clWhite;
    Style := stAssist;
    Allignment := abCentr;
    Font := Self.Font;
    Hint := AHint;
    OnDrawHint := DrawButtonHint;
    OnClick    := ButtonClick;
    if Line > FLinesCount then FLinesCount := Line;
    if Pos  > FMaxPos then FMaxPos := Pos;
  end;
  SetBounds(FWindowRect.Left, FWindowRect.Top, 3*2+(FMaxPos+1)*FButtonWidth,
     Margins.Top+3*2+(FLinesCount+1)*FButtonWidth+2+FHintHeight);
end;

procedure TDBClipPopup.AddButtons;
begin
  AddButton('#Query'   , 'DC_DBQUERY'   , LoadStr(RES_STRN_VAL_QUERY), 0, 0);
  AddButton('#Property', 'DC_DBPROPERTY', LoadStr(RES_STRN_VAL_PROP) , 0, 1);
  AddButton('#Find'    , 'DC_DBFIND'    , LoadStr(RES_STRN_VAL_FIND) , 0, 2);
  AddButton('#Print'   , 'DC_PRINT'     , LoadStr(RES_STRN_VAL_PRINT), 0, 3);
end;

procedure TDBClipPopup.ButtonClick(Sender: TObject);
begin
  inherited;
  if Assigned(FOnButtonClick) then FOnButtonClick(Sender);
end;

procedure TDBClipPopup.Clear;
begin
  FButtons.Clear;
  FLinesCount  := 0;
  FMaxPos      := 0;
end;

constructor TDBClipPopup.Create(AOwner: TComponent);
 var
  P: TPoint;
  R: TRect;

begin
  inherited;
  Parent := TWinControl(AOwner);
  R := TPrivateDBGrid(AOwner).CellRect(0,0);
  P := Point(R.Left, R.Top);

  FPopupAlignment := wpOffset;
  FButtonWidth := 26;
  FHintHeight  := 18;
  FLinesCount  := 0;
  FMaxPos      := 0;

  Color := clBtnFace;

  PopupBorderStyle := brRaised;
  SetBounds(P.X, P.Y+R.Bottom-R.Top, 6, 8+FHintHeight);
  AddButtons;
end;

procedure TDBClipPopup.DrawButtonHint(Sender: TObject; Mode: integer);
 var
  sHint: string;
  R: TRect;
  DC: HDC;
begin
  if Mode = 0 then
    sHint := (Sender as TDCEditButton).Hint
  else
    sHint := '';

  R := Rect(2,2+(FLinesCount+1)*FButtonWidth+3,
         Self.Width-6,2+(FLinesCount+1)*FButtonWidth+FHintHeight);


  Canvas.Brush.Color := Color;
  Canvas.Brush.Style := bsSolid;
  Canvas.FillRect(R);
  R.Left:= R.Left+ 2;

  if sHint <> '' then
  begin
    DC := GetWindowDC(Handle);
    SelectObject(DC, Font.Handle);
    OffsetRect(R, 0, Margins.Top);
    if TDCEditButton(Sender).Enabled then
      SetTextColor(DC, ColorToRGB(Font.Color))
    else
      SetTextColor(DC, $535353);

    SetBkMode(DC, TRANSPARENT);

    DrawText( DC, PChar(sHint), Length(sHint), R,
              DT_CENTER or DT_VCENTER or DT_SINGLELINE or DT_END_ELLIPSIS);

    ReleaseDC(DC,Handle);
  end;
end;

function TDBClipPopup.GetActiveButton: TDCEditButton;
begin
  Result := Buttons.ActiveButton;
end;

procedure TDBClipPopup.KeyDown(var Key: Word; Shift: TShiftState);
begin
  case Key of
    VK_LEFT :;
    VK_RIGHT:;
  end
end;

procedure TDBClipPopup.RedrawBorder;
 var
  DC: HDC;
  R: TRect;
begin
  DC := GetWindowDC(Handle);
  try
    GetWindowRect (Handle, R);  OffsetRect (R, -R.Left, -R.Top);
    DrawEdge(DC, R, BDR_RAISEDOUTER, BF_BOTTOMRIGHT);
    DrawEdge(DC, R, BDR_SUNKENOUTER, BF_TOPLEFT);
    InflateRect(R, -1, -1);
    DrawEdge(DC, R, BDR_RAISEDINNER, BF_RECT);

    DrawHeader;
    DrawFooter;

    InflateRect(R, -2, -2);
    R.Top    := R.Bottom - FHintHeight;
    DrawEdge(DC, R, BDR_SUNKENOUTER, BF_RECT);
  finally
    ReleaseDC(Handle, DC);
    DeleteDC(DC);
  end;
end;

{ TKNClipPopup }

procedure TKNClipPopup.AddButtons;
begin
  AddButton('#Property', 'DC_DBPROPERTY', LoadStr(RES_STRN_VAL_PROP) , 0, 0);
  AddButton('#Find'    , 'DC_DBFIND'    , LoadStr(RES_STRN_VAL_FIND) , 0, 1);
  AddButton('#Print'   , 'DC_PRINT'     , LoadStr(RES_STRN_VAL_PRINT), 0, 2);
end;

end.

