unit DatasetList;

{ TDatasetList (C)opyright 1998 Version 1.0
  Author : Jeff Swim
  eMail : doodler@doodler.com
  Internet : http://www.doodler.com }

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs, DB, dsgnintf;

type
  TOnCloseOperation = (ocPost,ocCancel);

  TdslItem = class(TCollectionItem)
  private
    FDataSet: TDataSet;
    FBeforeOpen: TNotifyEvent;
    FAfterOpen: TNotifyEvent;
    FBeforeClose: TNotifyEvent;
    FAfterClose: TNotifyEvent;
    FBeforeCancel: TNotifyEvent;
    FAfterCancel: TNotifyEvent;
    FBeforePost: TNotifyEvent;
    FAfterPost: TNotifyEvent;
    FBeforeRefresh: TNotifyEvent;
    FAfterRefresh: TNotifyEvent;
    FIncludeInOpen: boolean;
    FIncludeInClose: boolean;
    FIncludeInCancel: boolean;
    FIncludeInPost: boolean;
    FIncludeInRefresh: boolean;
    FOnClose: TOnCloseOperation;
  public
    constructor Create(Collection: TCollection); override;
    procedure Assign(Source: TPersistent); override;
 published
    property DataSet: TDataSet read FDataSet write FDataSet;
    property IncludeInOpen: boolean read FIncludeInOpen write FIncludeInOpen default True;
    property IncludeInClose: boolean read FIncludeInClose write FIncludeInClose default True;
    property IncludeInPost: boolean read FIncludeInPost write FIncludeInPost default True;
    property IncludeInCancel: boolean read FIncludeInCancel write FIncludeInCancel default True;
    property IncludeInRefresh: boolean read FIncludeInRefresh write FIncludeInRefresh default True;
    property OnClose: TOnCloseOperation read FOnClose write FOnClose;
    property BeforeOpen: TNotifyEvent read FBeforeOpen write FBeforeOpen;
    property AfterOpen: TNotifyEvent read FAfterOpen write FAfterOpen;
    property BeforeClose: TNotifyEvent read FBeforeClose write FBeforeClose;
    property AfterClose: TNotifyEvent read FAfterClose write FAfterClose;
    property BeforeCancel: TNotifyEvent read FBeforeCancel write FBeforeCancel;
    property AfterCancel: TNotifyEvent read FAfterCancel write FAfterCancel;
    property BeforePost: TNotifyEvent read FBeforePost write FBeforePost;
    property AfterPost: TNotifyEvent read FAfterPost write FAfterPost;
    property BeforeRefresh: TNotifyEvent read FBeforeRefresh write FBeforeRefresh;
    property AfterRefresh: TNotifyEvent read FAfterRefresh write FAfterRefresh;
  end;

  TdslCollection = class(TCollection)
  protected
    FOwner: TPersistent;
    function GetOwner:TPersistent; override;
    function GetItem(Index: Integer): TdslItem;
    procedure SetItem(Index: Integer; Value: TdslItem);
  public
    property Items[Index: Integer]: TdslItem read GetItem write SetItem; default;
  end;

  TDatasetList = class(TComponent)
  private
    FDataSets: TdslCollection;
    FOpenInDesigner: boolean;
    FAutoOpen: boolean;
    FAutoClose: boolean;
  protected
    procedure SetCollection(const Value: TdslCollection);
    procedure SetOpenInDesigner(Value: boolean);
    procedure Loaded; override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure OpenAll;
    procedure CloseAll;
    procedure CancelAll;
    procedure PostAll;
    procedure RefreshAll;
  published
    property DataSets: TdslCollection read FDataSets write SetCollection;
    property OpenInDesigner: boolean read FOpenInDesigner write SetOpenInDesigner;
    property AutoOpen: boolean read FAutoOpen write FAutoOpen default True;
    property AutoClose: boolean read FAutoClose write FAutoClose default True;
  end;

  TDatasetListEditor = class(TComponentEditor)
    function GetVerbCount: integer; override;
    function GetVerb(Index: integer): string; override;
    procedure ExecuteVerb(Index: integer); override;
  end;

procedure Register;

implementation

constructor TdslItem.Create(Collection: TCollection);
begin
   inherited Create(Collection);
    FIncludeInOpen := True;
    FIncludeInClose := True;
    FIncludeInCancel := True;
    FIncludeInPost := True;
    FIncludeInRefresh := True;
end;

procedure TdslItem.Assign(Source: TPersistent);
begin
   if Source is TCheckConstraint then
      begin
         DataSet := TdslItem(Source).DataSet;
         IncludeInOpen := TdslItem(Source).IncludeInOpen;
         IncludeInClose := TdslItem(Source).IncludeInClose;
         IncludeInPost := TdslItem(Source).IncludeInPost;
         IncludeInCancel := TdslItem(Source).IncludeInCancel;
         IncludeInRefresh := TdslItem(Source).IncludeInRefresh;
         OnClose := TdslItem(Source).OnClose;
         BeforeOpen := TdslItem(Source).BeforeOpen;
         AfterOpen := TdslItem(Source).AfterOpen;
         BeforeClose := TdslItem(Source).BeforeClose;
         AfterClose := TdslItem(Source).AfterClose;
         BeforeCancel := TdslItem(Source).BeforeCancel;
         AfterCancel := TdslItem(Source).AfterCancel;
         BeforePost := TdslItem(Source).BeforePost;
         AfterPost := TdslItem(Source).AfterPost;
         BeforeRefresh := TdslItem(Source).BeforeRefresh;
         AfterRefresh := TdslItem(Source).FAfterRefresh;
      end;
   inherited Assign(Source);
end;

function TdslCollection.GetOwner: TPersistent;
begin
   Result := FOwner;
end;

function TdslCollection.GetItem(Index: Integer): TdslItem;
begin
   Result := TdslItem(inherited GetItem(Index));
end;

procedure TdslCollection.SetItem(Index: Integer; Value: TdslItem);
begin
   inherited SetItem(Index, Value);
end;

constructor TDatasetList.Create(AOwner: TComponent);
begin
   inherited Create(AOwner);
   FDataSets := TdslCollection.Create(TdslItem);
   FDataSets.FOwner := Self;
   FAutoOpen := True;
   FAutoClose := True;
   FOpenInDesigner := False;
end;

destructor TDatasetList.Destroy;
begin
   if not (csDesigning in ComponentState) and (AutoClose) then
      CloseAll;
   inherited;
   FDataSets.Free;
end;

procedure TDatasetList.OpenAll;
var
   idx: integer;
   item: TdslItem;
begin
   for idx := 0 to FDataSets.Count - 1 do begin
      item := TdslItem(FDataSets.Items[idx]);
      if not Assigned(item.DataSet) then
         continue;
      if item.IncludeInOpen then
         begin
            if Assigned(item.BeforeOpen) then item.BeforeOpen(item);
            item.DataSet.Open;
            if Assigned(item.AfterOpen) then item.AfterOpen(item);
         end;
   end;
end;

procedure TDatasetList.CloseAll;
var
   idx: integer;
   item: TdslItem;
begin
   for idx := 0 to FDataSets.Count - 1 do begin
      item := TdslItem(FDataSets.Items[idx]);
      if not Assigned(item.DataSet) then
         continue;
      if (item.IncludeInClose) and (item.DataSet.Active) then
         begin
            if Assigned(item.BeforeClose) then item.BeforeClose(item);
            if item.DataSet.State in [dsEdit,dsInsert] then
               case item.OnClose of
                  ocPost: item.DataSet.Post;
                ocCancel: item.DataSet.Cancel;
               end;
            item.DataSet.Close;
            if Assigned(item.AfterClose) then item.AfterClose(item);
         end;
   end;
end;

procedure TDatasetList.CancelAll;
var
   idx: integer;
   item: TdslItem;
begin
   for idx := 0 to FDataSets.Count - 1 do begin
      item := TdslItem(FDataSets.Items[idx]);
      if not Assigned(item.DataSet) then
         continue;
      if (item.IncludeInCancel) and (item.DataSet.Active)
     and (item.DataSet.State in [dsEdit,dsInsert]) then
         begin
            if Assigned(item.BeforeCancel) then item.BeforeCancel(item);
            item.DataSet.Cancel;
            if Assigned(item.AfterCancel) then item.AfterCancel(item);
         end;
   end;
end;

procedure TDatasetList.PostAll;
var
   idx: integer;
   item: TdslItem;
begin
   for idx := 0 to FDataSets.Count - 1 do begin
      item := TdslItem(FDataSets.Items[idx]);
      if not Assigned(item.DataSet) then
         continue;
      if (item.IncludeInPost) and (item.DataSet.Active)
     and (item.DataSet.State in [dsEdit, dsInsert]) then
         begin
            if Assigned(item.BeforePost) then item.BeforePost(item);
            item.DataSet.Post;
            if Assigned(item.AfterPost) then item.AfterPost(item);
         end;
   end;
end;

procedure TDatasetList.RefreshAll;
var
   idx: integer;
   item: TdslItem;
begin
   for idx := 0 to FDataSets.Count - 1 do begin
      item := TdslItem(FDataSets.Items[idx]);
      if not Assigned(item.DataSet) then
         continue;
      if (item.IncludeInRefresh) and (item.DataSet.Active) then
         begin
            if Assigned(item.BeforeRefresh) then item.BeforeRefresh(item);
            item.DataSet.Refresh;
            if Assigned(item.AfterRefresh) then item.AfterRefresh(item);
         end;
   end;
end;

procedure TDatasetList.SetOpenInDesigner(Value: boolean);
begin
   if Value <> FOpenInDesigner then
      begin
         FOpenInDesigner := Value;
         if csDesigning in ComponentState then
            if FOpenInDesigner then
               OpenAll
            else
               CloseAll;
      end;
end;

procedure TDatasetList.SetCollection(const Value: TdslCollection);
begin
   FDataSets.Assign(Value);
end;

procedure TDatasetList.Loaded;
begin
   inherited;
   if ((csDesigning in ComponentState) and (OpenInDesigner))
   or (not (csDesigning in ComponentState) and (AutoOpen)) then
      OpenAll;
end;

function TDatasetListEditor.GetVerbCount: integer;
begin
   Result := 1;
end;

function TDatasetListEditor.GetVerb(Index: integer): string;
begin
   Result := 'Open/Close Datasets';
end;

procedure TDatasetListEditor.ExecuteVerb(Index: integer);
begin
   TDatasetList(Component).OpenInDesigner := not TDatasetList(Component).OpenInDesigner;
   Designer.Modified;
end;


procedure Register;
begin
  RegisterComponents('Data Access', [TDatasetList]);
  RegisterComponentEditor(TDatasetList, TDatasetListEditor);
end;

end.
