// DbHdrCtrlGrid. Copyright (c) 2000 Altive Software Ltd. http://www.dbaltgrid.com/freeware
// DbAltGrid provides functionality of the DBCtrlGrid and hierarchical columns structure in a DBGrid descendant. http://www.dbaltgrid.com
unit DbHCGrid;
{$R-}
interface
uses Windows, Messages, SysUtils, Classes, Controls, ComCtrls, Forms, Graphics, DB;
type

  TDbHdrCtrlGrid = class;

  TdhcgSection = class(THeaderSection) // fo. New type.
  private
    FControl: TControl;
    FLeftMargin: Integer;
    FRightMargin: Integer;
    procedure SetControl(const Value: TControl);
    procedure SetLeftMargin(const Value: Integer);
    procedure SetRightMargin(const Value: Integer);
  protected
    function GetDisplayName: string; override;
    procedure MoveControl;
  public
    constructor Create(Collection: TCollection); override;
    procedure Assign(Source: TPersistent); override;
    function DataFieldName: string; // fo. If control assigned to the Control property is data-aware, then function returns its DataField property value, otherwise null string
  published
    property Control: TControl read FControl write SetControl; // fo. Any control supported by DBCtrlGrid can be attached to the section.
    property LeftMargin: Integer read FLeftMargin write SetLeftMargin default -1; // fo. Specifies the horizontal coordinate of the left edge of the Control relative to the left edge of the HeaderSection in pixels.
    property RightMargin: Integer read FRightMargin write SetRightMargin default 0; // fo. Specifies the horizontal coordinate of the right edge of the Control relative to the right edge of the HeaderSection in pixels.
  end;

  TdhcgSectionClass = class of TdhcgSection;
  TdhcgHeader = class;

  TdhcgSections = class(TCollection) // fo. New type.
  private
    FHeader: TdhcgHeader;
    function GetItem(Index: Integer): TdhcgSection;
    procedure SetItem(Index: Integer; Value: TdhcgSection);
  protected
    function GetOwner: TPersistent; override;
    procedure Update(Item: TCollectionItem); override;
  public
    constructor Create(AHeader: TdhcgHeader; AItemClass: TdhcgSectionClass);
    function Add: TdhcgSection;
    procedure MoveControls(Item: TCollectionItem); virtual;
    property Header: TdhcgHeader read FHeader;
    property Items[Index: Integer]: TdhcgSection read GetItem write SetItem; default; // fo. You can refer any header section in the grid as a bidimensional array item (i.e. Headers[I].Sections[J]). See TdhcgHeaders.Items property.
  end;

  TdhcgHeader = class(TCollectionItem) // fo. New type.
  private
    FHeaderControl: THeaderControl;
    FSections: TdhcgSections;
    FOnDrawSection: TDrawSectionEvent;
    procedure HeaderControlDrawSectionEvent(HeaderControl: THeaderControl;
      Section: THeaderSection; const Rect: TRect; Pressed: Boolean);
    procedure SetSections(const Value: TdhcgSections);
  protected
  public
    constructor Create(Collection: TCollection); override;
    destructor Destroy; override;
    procedure Assign(Source: TPersistent); override;
    property HeaderControl: THeaderControl read FHeaderControl write FHeaderControl;
  published
    property Sections: TdhcgSections read FSections write SetSections;
    property OnDrawSection: TDrawSectionEvent read FOnDrawSection write FOnDrawSection;
  end;

  TdhcgHeaderClass = class of TdhcgHeader;

  TdhcgHeaders = class(TCollection) // fo. New type.
  private
    DbHdrCtrlGrid: TDbHdrCtrlGrid;
    FDefaultLeftMargin: Integer;
    FDefaultRightMargin: Integer;
    function GetItem(Index: Integer): TdhcgHeader;
    procedure SetItem(Index: Integer; Value: TdhcgHeader);
    procedure SetDefaultLeftMargin(const Value: Integer);
    procedure SetDefaultRightMargin(const Value: Integer);
  protected
    function FindSection(AHeaderControl: THeaderControl; AHeaderSection: THeaderSection;
      var AHeader: TdhcgHeader; var ASection: TdhcgSection): Boolean;
    function GetOwner: TPersistent; override;
    procedure Update(Item: TCollectionItem); override;
  public
    constructor Create(AaDbCtrlGrid: TDbHdrCtrlGrid; AItemClass: TdhcgHeaderClass);
    function Add: TdhcgHeader;
    function ControlToSection(AControl: TControl): TdhcgSection; // fo. Returns section item attached to the control plased on the TDbHdrCtrlGrid's panel.
    property Grid: TDbHdrCtrlGrid read DbHdrCtrlGrid;
    property Items[Index: Integer]: TdhcgHeader read GetItem write SetItem; default; // fo. You can refer any header section in the grid as a bidimensional array item (i.e. Headers[I].Sections[J]). See TdhcgSections.Items property.
  published
    property DefaultLeftMargin: Integer read FDefaultLeftMargin write SetDefaultLeftMargin default -1; // fo. Determines default LeftMargin value for new header section.
    property DefaultRightMargin: Integer read FDefaultRightMargin write SetDefaultRightMargin default 0; // fo. Determines default RightMargin value for new header section.
  end;

  TDbHdrCtrlGridLink = class(TDataLink)
  private
    DbHdrCtrlGrid: TDbHdrCtrlGrid; // fo. Grid type.
  protected
    procedure ActiveChanged; override;
    procedure DataSetChanged; override;
  public
    constructor Create(DBCtrlGrid: TDbHdrCtrlGrid);
  end;

  TDbHdrCtrlPanel = class(TWinControl)
  private
    DbHdrCtrlGrid: TDbHdrCtrlGrid; // fo. Grid type.
    procedure CMControlListChange(var Message: TCMControlListChange); message CM_CONTROLLISTCHANGE;
    procedure WMEraseBkgnd(var Message: TMessage); message WM_ERASEBKGND;
    procedure WMPaint(var Message: TWMPaint); message WM_PAINT;
    procedure WMNCHitTest(var Message: TWMNCHitTest); message WM_NCHITTEST;
  protected
    procedure CreateParams(var Params: TCreateParams); override;
    procedure PaintWindow(DC: HDC); override; // fo. Changed.
  public
    constructor CreateLinked(DBCtrlGrid: TDbHdrCtrlGrid); // fo. Grid type.
  end;

  TDbHdrCtrlGridBorder = (gbNone, gbRaised, gbSingle); // fo. gbSingle added.
  TDBCtrlGridKey = (gkNull, gkEditMode, gkPriorTab, gkNextTab, gkLeft,
    gkRight, gkUp, gkDown, gkScrollUp, gkScrollDown, gkPageUp, gkPageDown,
    gkHome, gkEnd, gkInsert, gkAppend, gkDelete, gkCancel);

  TdhcgPaintPanelEvent = procedure(DBCtrlGrid: TDbHdrCtrlGrid; Index: Integer) of object; // fo. Grid type.
  TdhcgHeaderEvent = procedure(DBCtrlGrid: TDbHdrCtrlGrid; Header: TdhcgHeader;
    Section: TdhcgSection) of object; // fo. New event type.

  TDbHdrCtrlGrid = class(TWinControl)
  private
    FDataLink: TDbHdrCtrlGridLink;
    FPanel: TDbHdrCtrlPanel;
    FCanvas: TCanvas;
    FRowCount: Integer;
    FPanelWidth: Integer;
    FPanelHeight: Integer;
    FPanelIndex: Integer;
    FPanelCount: Integer;
    FBitmapCount: Integer;
    FPanelBitmap: HBitmap;
    FSaveBitmap: HBitmap;
    FPanelDC: HDC;
//    FOrientation: TDBCtrlGridOrientation;  // fo. Obsolete.
    FPanelBorder: TDbHdrCtrlGridBorder;
    FAllowInsert: Boolean;
    FAllowDelete: Boolean;
    FShowFocus: Boolean;
    FFocused: Boolean;
    FClicking: Boolean;
    FSelColorChanged: Boolean;
//    FScrollBarKind: Integer;          // fo. Obsolete.
    FSelectedColor: TColor;
    FOnPaintPanel: TdhcgPaintPanelEvent;
    FTotalHeadersHeight: Integer; // fo. New field.
    FHeaders: TdhcgHeaders; // fo. New field.
    FOnHeaderResize: TdhcgHeaderEvent; // fo. New field.
    FOnHeaderClick: TdhcgHeaderEvent; // fo. New field.
    FHorzShift: Integer; // fo. New field.
    FBorderStyle: TBorderStyle; // fo. New field.
    function AcquireFocus: Boolean;
    procedure AdjustPanel; // fo. New procedure.
    procedure AdjustRowCount; // fo. New procedure.
    procedure AdjustSize; reintroduce; // fo. Changed.
    procedure CreatePanelBitmap;
    procedure DataSetChanged(Reset: Boolean); // fo. Changed.
    procedure DestroyPanelBitmap;
    procedure DrawPanel(DC: HDC; Index: Integer);
    procedure DrawPanelBackground(DC: HDC; const R: TRect; Erase, Selected: Boolean); // fo. Changed.
    function FindNext(StartControl: TWinControl; GoForward: Boolean; var WrapFlag: Integer): TWinControl;
    function GetDataSource: TDataSource;
    function GetEditMode: Boolean;
    function GetPanelBounds(Index: Integer): TRect; // fo. Changed.
    procedure HeaderResize(AHeaderControl: THeaderControl; AHeaderSection: THeaderSection); // fo. TSectionNotifyEvent
    procedure HeaderClick(AHeaderControl: THeaderControl; AHeaderSection: THeaderSection); // fo. TSectionNotifyEvent
    function PointInPanel(const P: TSmallPoint): Boolean;
    procedure Reset; // fo. Changed.
    procedure Scroll(Inc: Integer; ScrollLock: Boolean); // fo. Changed.
    procedure ScrollMessage(var Message: TWMScroll);
    procedure SelectNext(GoForward: Boolean);
    procedure SetBorderStyle(Value: TBorderStyle); // fo. New method.
    procedure SetDataSource(Value: TDataSource);
    procedure SetEditMode(Value: Boolean);
    procedure SetHeaders(const Value: TdhcgHeaders); // fo. New method.
//    procedure SetOrientation(Value: TDBCtrlGridOrientation);   // fo. Obsolete.
    procedure SetPanelBorder(Value: TDbHdrCtrlGridBorder);
    procedure SetPanelCount(Value: Integer); // fo. New method.
    procedure SetPanelHeight(Value: Integer);
    procedure SetPanelIndex(Value: Integer);
    procedure SetPanelWidth(Value: Integer);
    procedure SetRowCount(Value: Integer);
    procedure SetSelectedColor(Value: TColor);
    procedure SetTotalHeadersHeight(Value: Integer); // fo. New method.
    procedure UpdateDataLinks(Control: TControl; Inserting: Boolean); // fo. Changed.
    procedure UpdateHorzScrollBar; // fo. New procedure.
    procedure UpdateScrollBar; // fo. Changed.
    function VisibleHeight: Integer; // fo. New function.
    function VisibleWidth: Integer; // fo. New function.
    procedure WMLButtonDown(var Message: TWMLButtonDown); message WM_LBUTTONDOWN;
    procedure WMLButtonDblClk(var Message: TWMLButtonDblClk); message WM_LBUTTONDBLCLK;
    procedure WMHScroll(var Message: TWMHScroll); message WM_HSCROLL; // fo. Changed.
    procedure WMVScroll(var Message: TWMVScroll); message WM_VSCROLL;
    procedure WMEraseBkgnd(var Message: TMessage); message WM_ERASEBKGND;
    procedure WMPaint(var Message: TWMPaint); message WM_PAINT;
    procedure WMSetFocus(var Message: TWMSetFocus); message WM_SETFOCUS;
    procedure WMKillFocus(var Message: TWMKillFocus); message WM_KILLFOCUS;
    procedure WMGetDlgCode(var Message: TWMGetDlgCode); message WM_GETDLGCODE;
    procedure WMSize(var Message: TMessage); message WM_SIZE;
    procedure CMChildKey(var Message: TCMChildKey); message CM_CHILDKEY;
    procedure CMColorChanged(var Message: TMessage); message CM_COLORCHANGED;
  protected
    procedure CreateParams(var Params: TCreateParams); override; // fo. Changed.
    procedure CreateWnd; override; // fo. Changed.
    function GetChildParent: TComponent; override;
    procedure GetChildren(Proc: TGetChildProc; Root: TComponent); override;
    procedure PaintPanel(Index: Integer); virtual;
    procedure PaintWindow(DC: HDC); override; // fo. Changed.
    procedure ReadState(Reader: TReader); override;
    property PanelWidth: Integer read FPanelWidth write SetPanelWidth; // fo. By Borland published.
    property RowCount: Integer read FRowCount write SetRowCount;
  public
    constructor Create(AOwner: TComponent); override; // fo. Changed.
    destructor Destroy; override; // fo. Changed.
    procedure DoKey(Key: TDBCtrlGridKey); // fo. Changed.
    function ExecuteAction(Action: TBasicAction): Boolean; override;
    procedure GetTabOrderList(List: TList); override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure SetBounds(ALeft, ATop, AWidth, AHeight: Integer); override; // fo. Changed.
    function UpdateAction(Action: TBasicAction): Boolean; override;
    property Canvas: TCanvas read FCanvas;
    property EditMode: Boolean read GetEditMode write SetEditMode;
    property Panel: TDbHdrCtrlPanel read FPanel;   // fo. By Borland protected.
    property PanelCount: Integer read FPanelCount write SetPanelCount; // fo. In TDBCtrlGrid is assigned directly without any metod.
    property PanelIndex: Integer read FPanelIndex write SetPanelIndex;
    property TotalHeadersHeight: Integer read FTotalHeadersHeight write SetTotalHeadersHeight; // fo. New property.
  published
    property Align;
    property AllowDelete: Boolean read FAllowDelete write FAllowDelete default True;
    property AllowInsert: Boolean read FAllowInsert write FAllowInsert default True;
    property Anchors;
    property BorderStyle: TBorderStyle read FBorderStyle write SetBorderStyle default bsSingle; // fo. New property.
    property Color;
    property Constraints;
    property DataSource: TDataSource read GetDataSource write SetDataSource;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
//    property Orientation: TDBCtrlGridOrientation read FOrientation write SetOrientation default goVertical; // fo. Obsolete.
    property Headers: TdhcgHeaders read FHeaders write SetHeaders; // fo. New property. Default property editor, launched when the component is double-clicked.
    property PanelBorder: TDbHdrCtrlGridBorder read FPanelBorder write SetPanelBorder default gbSingle; // fo. Changed.
    property PanelHeight: Integer read FPanelHeight write SetPanelHeight default 24; // fo. In TDBCtrlGrid there is no default value.
//    property PanelWidth: Integer read FPanelWidth write SetPanelWidth;  // fo. Made protected.
    property ParentColor;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property TabOrder;
    property TabStop default True;
    property SelectedColor: TColor read FSelectedColor write SetSelectedColor stored FSelColorChanged default clWindow;
    property ShowFocus: Boolean read FShowFocus write FShowFocus default True;
    property ShowHint;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnHeaderResize: TdhcgHeaderEvent read FOnHeaderResize write FOnHeaderResize; // fo. New event.
    property OnHeaderClick: TdhcgHeaderEvent read FOnHeaderClick write FOnHeaderClick; // fo. New event.
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnPaintPanel: TdhcgPaintPanelEvent read FOnPaintPanel write FOnPaintPanel;
    property OnStartDrag;
  end;

implementation
uses DBConsts, Math, TypInfo;
{--------------------------------------------------------------------------}
{ TdhcgSection }
{--------------------------------------------------------------------------}
constructor TdhcgSection.Create(Collection: TCollection);
begin
  inherited Create(Collection);
  LeftMargin := TdhcgHeaders(TdhcgSections(Collection).Header.Collection).DefaultLeftMargin;
  RightMargin := TdhcgHeaders(TdhcgSections(Collection).Header.Collection).DefaultRightMargin;
end;
{--------------------------------------------------------------------------}
function TdhcgSection.GetDisplayName: string;
begin
  if FControl <> nil then
    Result := FControl.Name + ' (' + Text + ')'
  else Result := Text;
  if Result = '' then Result := inherited GetDisplayName;
end;
{--------------------------------------------------------------------------}
procedure TdhcgSection.SetControl(const Value: TControl);
var
  lOldSection: TdhcgSection;
begin
  if FControl <> Value then begin
    lOldSection := TdhcgHeaders(TdhcgSections(Collection).Header.Collection).ControlToSection(Value);
    if lOldSection <> nil then lOldSection.Control := nil;
    FControl := Value;
    if FControl = nil then
      Text := ''
    else if Text = '' then Text := DataFieldName;
    TdhcgSections(Collection).Update(Self);
  end;
end;
{--------------------------------------------------------------------------}
procedure TdhcgSection.SetLeftMargin(const Value: Integer);
begin
  if (FLeftMargin <> Value) and (Value >= -1) then begin
    FLeftMargin := Value;
    MoveControl;
  end;
end;
{--------------------------------------------------------------------------}
procedure TdhcgSection.SetRightMargin(const Value: Integer);
begin
  if (FRightMargin <> Value) and (Value >= -1) then begin
    FRightMargin := Value;
    MoveControl;
  end;
end;
{--------------------------------------------------------------------------}
procedure TdhcgSection.MoveControl;
begin
  if FControl <> nil then begin
    FControl.Left := Left + FLeftMargin;
    FControl.Width := Width - FLeftMargin - FRightMargin;
  end;
end;
{--------------------------------------------------------------------------}
procedure TdhcgSection.Assign(Source: TPersistent);
begin
  inherited Assign(Source);
  if Source is TdhcgSection then begin
    Control := TdhcgSection(Source).Control;
  end;
end;
{--------------------------------------------------------------------------}
function TdhcgSection.DataFieldName: string;
var
  lPropInfo: PPropInfo;
begin
  Result := '';
  if FControl <> nil then begin
    lPropInfo := GetPropInfo(FControl.ClassInfo, 'DataField');
    if lPropInfo <> nil then Result := GetStrProp(FControl, lPropInfo);
  end;
end;
{--------------------------------------------------------------------------}
{ TdhcgSections }
{--------------------------------------------------------------------------}
constructor TdhcgSections.Create(AHeader: TdhcgHeader; AItemClass: TdhcgSectionClass);
begin
  inherited Create(AItemClass);
  FHeader := AHeader;
end;
{--------------------------------------------------------------------------}
function TdhcgSections.GetItem(Index: Integer): TdhcgSection;
begin
  Result := TdhcgSection(inherited Items[Index]);
end;
{--------------------------------------------------------------------------}
procedure TdhcgSections.SetItem(Index: Integer; Value: TdhcgSection);
begin
  inherited SetItem(Index, Value);
end;
{--------------------------------------------------------------------------}
function TdhcgSections.GetOwner: TPersistent;
begin
  Result := FHeader;
end;
{--------------------------------------------------------------------------}

procedure TdhcgSections.Update(Item: TCollectionItem);
begin
  if Item = nil
    then FHeader.HeaderControl.Sections.Assign(Self)
  else FHeader.HeaderControl.Sections[Item.Index].Assign(Item);
  MoveControls(Item);
  TdhcgHeaders(FHeader.Collection).Grid.AdjustPanel;
end;
{--------------------------------------------------------------------------}
function TdhcgSections.Add: TdhcgSection;
begin
  Result := TdhcgSection(inherited Add);
end;
{--------------------------------------------------------------------------}
procedure TdhcgSections.MoveControls(Item: TCollectionItem);
var
  I: Integer;
begin
  if (Item <> nil) and (Item.Index < Count)
    then for I := Item.Index to Count - 1 do TdhcgSection(Items[I]).MoveControl
  else for I := 0 to Count - 1 do TdhcgSection(Items[I]).MoveControl
end;
{--------------------------------------------------------------------------}
{ TdhcgHeader }
{--------------------------------------------------------------------------}
constructor TdhcgHeader.Create(Collection: TCollection);
begin
  inherited Create(Collection);
  FHeaderControl := THeaderControl.Create(TdhcgHeaders(Collection).DbHdrCtrlGrid);
  with FHeaderControl do begin
    Align := alNone;
    Parent := TdhcgHeaders(Collection).Grid;
    Width := Parent.Width;
    OnSectionResize := TdhcgHeaders(Collection).Grid.HeaderResize;
    OnSectionClick := TdhcgHeaders(Collection).Grid.HeaderClick;
    OnDrawSection := HeaderControlDrawSectionEvent;
  end;
  FSections := TdhcgSections.Create(Self, TdhcgSection);
  FSections.Add;
end;
{--------------------------------------------------------------------------}
destructor TdhcgHeader.Destroy;
begin
  FHeaderControl.Free;
  FSections.Free;
  inherited Destroy;
end;
{--------------------------------------------------------------------------}
procedure TdhcgHeader.HeaderControlDrawSectionEvent(HeaderControl: THeaderControl;
      Section: THeaderSection; const Rect: TRect; Pressed: Boolean);
begin
  if Assigned(FOnDrawSection) then
    FOnDrawSection(HeaderControl, Section, Rect, Pressed)
  else HeaderControl.Canvas.FillRect(Rect);
end;
{--------------------------------------------------------------------------}
procedure TdhcgHeader.SetSections(const Value: TdhcgSections);
begin
  FSections.Assign(Value);
end;
{--------------------------------------------------------------------------}
procedure TdhcgHeader.Assign(Source: TPersistent);
begin
  inherited Assign(Source);
  if Source is TdhcgHeader then begin
    HeaderControl := TdhcgHeader(Source).HeaderControl;
  end;
end;
{--------------------------------------------------------------------------}
{ TdhcgHeaders }
{--------------------------------------------------------------------------}
constructor TdhcgHeaders.Create(AaDbCtrlGrid: TDbHdrCtrlGrid; AItemClass: TdhcgHeaderClass);
begin
  inherited Create(AItemClass);
  DbHdrCtrlGrid := AaDbCtrlGrid;
  FDefaultLeftMargin := -1;
  FDefaultRightMargin := 0;
end;
{--------------------------------------------------------------------------}
function TdhcgHeaders.GetItem(Index: Integer): TdhcgHeader;
begin
  Result := TdhcgHeader(inherited Items[Index]);
end;
{--------------------------------------------------------------------------}
procedure TdhcgHeaders.SetItem(Index: Integer; Value: TdhcgHeader);
begin
  inherited SetItem(Index, Value);
end;
{--------------------------------------------------------------------------}
procedure TdhcgHeaders.SetDefaultLeftMargin(const Value: Integer);
var
  I, J: Integer;
begin
  if (FDefaultLeftMargin <> Value) and (Value >= 0) then begin
    FDefaultLeftMargin := Value;
    for I := 0 to Count - 1 do with TdhcgHeader(Items[I]).Sections do
        for J := 0 to Count - 1 do TdhcgSection(Items[I]).LeftMargin := FDefaultLeftMargin;
  end;
end;
{--------------------------------------------------------------------------}
procedure TdhcgHeaders.SetDefaultRightMargin(const Value: Integer);
var
  I, J: Integer;
begin
  if (FDefaultRightMargin <> Value) and (Value >= 0) then begin
    FDefaultRightMargin := Value;
    for I := 0 to Count - 1 do with TdhcgHeader(Items[I]).Sections do
        for J := 0 to Count - 1 do TdhcgSection(Items[I]).RightMargin := FDefaultRightMargin;
  end;
end;
{--------------------------------------------------------------------------}
function TdhcgHeaders.FindSection(AHeaderControl: THeaderControl; AHeaderSection: THeaderSection;
  var AHeader: TdhcgHeader; var ASection: TdhcgSection): Boolean;
var
  I: Integer;
begin
  Result := False;
  AHeader := nil;
  ASection := nil;
  I := Count - 1;
  while (I >= 0) and (AHeader = nil) do begin
    if Items[I].HeaderControl = AHeaderControl then AHeader := Items[I];
    Dec(I);
  end;
  if (AHeader <> nil) and (AHeaderSection.Index < AHeader.Sections.Count) then begin
    ASection := AHeader.Sections[AHeaderSection.Index];
    Result := True;
  end;
end;
{--------------------------------------------------------------------------}
function TdhcgHeaders.GetOwner: TPersistent;
begin
  Result := DbHdrCtrlGrid;
end;
{--------------------------------------------------------------------------}
procedure TdhcgHeaders.Update(Item: TCollectionItem);
var
  I, lTop: Integer;
begin
  lTop := 0;
  for I := 0 to Count - 1 do with TdhcgHeader(Items[I]).HeaderControl do begin
      Left := Grid.FHorzShift;
      Top := lTop;
      Inc(lTop, Height);
    end;
  if csDesigning in Grid.ComponentState then Grid.AdjustSize;
end;
{--------------------------------------------------------------------------}
function TdhcgHeaders.Add: TdhcgHeader;
begin
  Result := TdhcgHeader(inherited Add);
end;
{--------------------------------------------------------------------------}
function TdhcgHeaders.ControlToSection(AControl: TControl): TdhcgSection;
var
  I, J: Integer;
begin
  Result := nil;
  if AControl <> nil then
    for I := 0 to Count - 1 do with TdhcgHeader(Items[I]) do
        for J := 0 to Sections.Count - 1 do
          if Sections[J].Control = AControl then begin
            Result := Sections[J];
            exit;
          end
end;
{--------------------------------------------------------------------------}
{ TDbHdrCtrlGridLink }
{--------------------------------------------------------------------------}
constructor TDbHdrCtrlGridLink.Create(DBCtrlGrid: TDbHdrCtrlGrid);
begin
  inherited Create;
  DbHdrCtrlGrid := DBCtrlGrid;
  VisualControl := True;
  RPR;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGridLink.ActiveChanged;
begin
  DbHdrCtrlGrid.DataSetChanged(False);
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGridLink.DataSetChanged;
begin
  DbHdrCtrlGrid.DataSetChanged(False);
end;
{--------------------------------------------------------------------------}
{ TDbHdrCtrlPanel }
{--------------------------------------------------------------------------}
constructor TDbHdrCtrlPanel.CreateLinked(DBCtrlGrid: TDbHdrCtrlGrid);
begin
  inherited Create(DBCtrlGrid);
  ControlStyle := [csAcceptsControls, csCaptureMouse, csClickEvents,
    csDoubleClicks, csOpaque, csReplicatable];
  DbHdrCtrlGrid := DBCtrlGrid;
  Parent := DBCtrlGrid;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlPanel.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  with Params.WindowClass do
    Style := Style and not (CS_HREDRAW or CS_VREDRAW);
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlPanel.PaintWindow(DC: HDC);
var
  R: TRect;
  Selected: Boolean;
begin
  with DbHdrCtrlGrid do begin
    if FDataLink.Active then begin
      Selected := (FDataLink.ActiveRecord = FPanelIndex);
      DrawPanelBackground(DC, Self.ClientRect, True, Selected);
      FCanvas.Handle := DC;
      try
        FCanvas.Font := Font;
        FCanvas.Brush.Style := bsSolid;
        FCanvas.Brush.Color := Color;
        PaintPanel(FDataLink.ActiveRecord);
        if FShowFocus and FFocused and Selected then begin
          R := Self.ClientRect;
          if FPanelBorder = gbRaised then InflateRect(R, -2, -2);
          if FPanelBorder = gbSingle then begin // fo.
            InflateRect(R, -2, -1);
            if FBorderStyle <> bsNone then R.Left := R.Left - 1;
            R.Bottom := R.Bottom - 1;
          end;
          FCanvas.Brush.Color := Color;
          FCanvas.DrawFocusRect(R);
        end;
      finally
        FCanvas.Handle := 0;
      end;
    end
    else DrawPanelBackground(DC, Self.ClientRect, True, csDesigning in ComponentState);
  end;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlPanel.CMControlListChange(var Message: TCMControlListChange);
begin
  DbHdrCtrlGrid.UpdateDataLinks(Message.Control, Message.Inserting);
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlPanel.WMPaint(var Message: TWMPaint);
var
  DC: HDC;
  PS: TPaintStruct;
begin
  if Message.DC = 0 then begin
    DbHdrCtrlGrid.CreatePanelBitmap;
    try
      Message.DC := DbHdrCtrlGrid.FPanelDC;
      PaintHandler(Message);
      Message.DC := 0;
      DC := BeginPaint(Handle, PS);
      BitBlt(DC, 0, 0, Width, Height, DbHdrCtrlGrid.FPanelDC, 0, 0, SRCCOPY);
      EndPaint(Handle, PS);
    finally
      DbHdrCtrlGrid.DestroyPanelBitmap;
    end;
  end
  else PaintHandler(Message);
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlPanel.WMNCHitTest(var Message: TWMNCHitTest);
begin
  if csDesigning in ComponentState then
    Message.Result := HTCLIENT else
    Message.Result := HTTRANSPARENT;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlPanel.WMEraseBkgnd(var Message: TMessage);
begin
  Message.Result := 1;
end;
{--------------------------------------------------------------------------}
{ TDbHdrCtrlGrid }
{--------------------------------------------------------------------------}
constructor TDbHdrCtrlGrid.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  ControlStyle := [csOpaque, csDoubleClicks];
  TabStop := True;
  FDataLink := TDbHdrCtrlGridLink.Create(Self);
  FCanvas := TCanvas.Create;
  FPanel := TDbHdrCtrlPanel.CreateLinked(Self);
  FHeaders := TdhcgHeaders.Create(Self, TdhcgHeader);
  FPanelHeight := 25;
  FPanelWidth := 300;
  Width := 320;
  AdjustSize;
  Height := 120;
  FPanelBorder := gbSingle;
  FAllowInsert := True;
  FAllowDelete := True;
  FShowFocus := True;
  ParentColor := False;
  Color := clWindow;
  FSelectedColor := Color;
  FBorderStyle := bsSingle;
end;
{--------------------------------------------------------------------------}
destructor TDbHdrCtrlGrid.Destroy;
begin
  FCanvas.Free;
  FDataLink.Free;
  FDataLink := nil;
  FHeaders.Free;
  FHeaders := nil;
  inherited Destroy;
end;
{--------------------------------------------------------------------------}
function TDbHdrCtrlGrid.AcquireFocus: Boolean;
begin
  Result := True;
  if not (Focused or EditMode) then begin
    SetFocus;
    Result := Focused;
  end;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.AdjustSize;
begin
  AdjustPanel; // fo.
  SetBounds(Left, Top, Width, Height);
  Reset;
  {if csDesigning in ComponentState then} Repaint;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.AdjustPanel;
var
  I, lTotal, lRight: Integer;
begin
  lTotal := 0;
  lRight := 0;
  with Headers do begin
    for I := 0 to Count - 1 do with TdhcgHeader(Items[I]) do begin
        Inc(lTotal, HeaderControl.Height);
        if Sections.Count > 0 then lRight := Max(lRight, Sections[Sections.Count - 1].Right);
      end;
    TotalHeadersHeight := lTotal;
    if lRight > 0 then begin
      for I := 0 to Count - 1 do TdhcgHeader(Items[I]).HeaderControl.Width := lRight;
      PanelWidth := lRight;
    end;
  end;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.AdjustRowCount;
var
  lNewRowCount: Integer;
begin
  lNewRowCount := VisibleHeight div FPanelHeight;
  if lNewRowCount < 1 then lNewRowCount := 1;
{  if (lNewRowCount > FPanelCount) and (FPanelCount > 0) and not (csDesigning in ComponentState)
    then lNewRowCount := FPanelCount; }
  if FRowCount <> lNewRowCount then begin
    FRowCount := lNewRowCount;
    Reset;
  end;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.CreatePanelBitmap;
var
  DC: HDC;
begin
  if FBitmapCount = 0 then begin
    DC := GetDC(0);
    FPanelBitmap := CreateCompatibleBitmap(DC, FPanel.Width, FPanel.Height);
    ReleaseDC(0, DC);
    FPanelDC := CreateCompatibleDC(0);
    FSaveBitmap := SelectObject(FPanelDC, FPanelBitmap);
  end;
  Inc(FBitmapCount);
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  with Params do begin
    Style := Style or WS_CLIPCHILDREN;
    WindowClass.style := WindowClass.style and not (CS_HREDRAW or CS_VREDRAW);
    if FBorderStyle = bsSingle then // fo.
      if NewStyleControls and Ctl3D then begin
        Style := Style and not WS_BORDER;
        ExStyle := ExStyle or WS_EX_CLIENTEDGE;
      end
      else Style := Style or WS_BORDER;
  end;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.CreateWnd;
begin
  inherited CreateWnd;
  if not FDataLink.Active then SetScrollRange(Handle, SB_VERT, 0, 4, False); // fo.
  UpdateScrollBar;
  UpdateHorzScrollBar;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.DataSetChanged(Reset: Boolean);
var
  NewPanelIndex, NewPanelCount: Integer;
  FocusedControl: TWinControl;
  R: TRect;
begin
  if csDesigning in ComponentState then begin
    NewPanelIndex := 0;
    NewPanelCount := 1;
  end
  else
    if FDataLink.Active then begin
      NewPanelIndex := FDataLink.ActiveRecord;
      NewPanelCount := FDataLink.RecordCount;
      if NewPanelCount = 0 then NewPanelCount := 1;
      PanelCount := NewPanelCount; // fo. here is duplicated, original place below.
    end
    else begin
      NewPanelIndex := 0;
      NewPanelCount := 0;
    end;
  FocusedControl := nil;
  R := GetPanelBounds(NewPanelIndex);
  if Reset or not HandleAllocated then
    FPanel.BoundsRect := R
  else begin
    FocusedControl := FindControl(GetFocus);
    if (FocusedControl <> FPanel) and FPanel.ContainsControl(FocusedControl) then
      FPanel.SetFocus
    else FocusedControl := nil;
    if NewPanelIndex <> FPanelIndex then begin
      SetWindowPos(FPanel.Handle, 0, R.Left, R.Top, R.Right - R.Left,
        R.Bottom - R.Top, SWP_NOZORDER or SWP_NOREDRAW);
      RedrawWindow(FPanel.Handle, nil, 0, RDW_INVALIDATE or RDW_ALLCHILDREN);
    end;
  end;
  FPanelIndex := NewPanelIndex;
  PanelCount := NewPanelCount; // fo. here original place, duplicate above
  FPanel.Visible := FPanelCount > 0;
  FPanel.Invalidate;
  if not Reset then begin
    Invalidate;
    Update;
  end;
  UpdateScrollBar;
  if (FocusedControl <> nil) and not FClicking and FocusedControl.CanFocus then
    FocusedControl.SetFocus;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.DestroyPanelBitmap;
begin
  Dec(FBitmapCount);
  if FBitmapCount = 0 then begin
    SelectObject(FPanelDC, FSaveBitmap);
    DeleteDC(FPanelDC);
    DeleteObject(FPanelBitmap);
  end;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.DoKey(Key: TDBCtrlGridKey);
begin
  if FDataLink.Active then begin // fo.
    with FDataLink.DataSet do
      case Key of
        gkEditMode: EditMode := not EditMode;
        gkPriorTab: SelectNext(False);
        gkNextTab: SelectNext(True);
        gkLeft: Scroll(-1, False);
        gkRight: Scroll(1, False);
        gkUp: Scroll(-1, False);
        gkDown: Scroll(1, False);
        gkScrollUp: Scroll(-1, True);
        gkScrollDown: Scroll(1, True);
        gkPageUp: Scroll(-FDataLink.BufferCount, True);
        gkPageDown: Scroll(FDataLink.BufferCount, True);
        gkHome: First;
        gkEnd: Last;
        gkInsert: if FAllowInsert and CanModify then begin
            Insert;
            EditMode := True;
          end;
        gkAppend: if FAllowInsert and CanModify then begin
            Append;
            EditMode := True;
          end;
        gkDelete: if FAllowDelete and CanModify then begin
            Delete;
            EditMode := False;
          end;
        gkCancel: begin
            Cancel;
            EditMode := False;
          end;
      end;
  end;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.DrawPanel(DC: HDC; Index: Integer);
var
  SaveActive: Integer;
  R: TRect;
begin
  R := GetPanelBounds(Index);
  if Index < FPanelCount then begin
    SaveActive := FDataLink.ActiveRecord;
    FDataLink.ActiveRecord := Index;
    FPanel.PaintTo(FPanelDC, 0, 0);
    FDataLink.ActiveRecord := SaveActive;
  end
  else DrawPanelBackground(FPanelDC, FPanel.ClientRect, True, False);
  BitBlt(DC, R.Left, R.Top, R.Right - R.Left, R.Bottom - R.Top, FPanelDC, 0, 0, SRCCOPY);
end;
{--------------------------------------------------------------------------}
type
  PdhcgPoints = ^TdhcgPoints;
  TdhcgPoints = array[0..0] of TPoint;

procedure dhcgPolyline(const ADC: HDC; const APoints: array of TPoint);
begin
  Windows.Polyline(ADC, PdhcgPoints(@APoints)^, High(APoints) + 1);
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.DrawPanelBackground(DC: HDC; const R: TRect;
  Erase, Selected: Boolean);
var
  Brush: HBrush;
  Pen, OldPen: HPen;
begin
  if Erase then begin
    if Selected then
      FPanel.Color := FSelectedColor
    else FPanel.Color := Color;
    Brush := CreateSolidBrush(ColorToRGB(FPanel.Color));
    FillRect(DC, R, Brush);
    DeleteObject(Brush);
  end;
  if FPanelBorder = gbRaised then
    DrawEdge(DC, PRect(@R)^, BDR_RAISEDINNER, BF_RECT);
  if FPanelBorder = gbSingle then begin // fo.
    Pen := CreatePen(PS_SOLID, 0, ColorToRGB(Font.Color));
    OldPen := SelectObject(DC, Pen);
{    if FBorderStyle <> bsNone
      then MoveToEx(DC, R.Left, R.Bottom - 1, nil)
    else begin
      MoveToEx(DC, R.Left, R.Top, nil);
      LineTo(DC, R.Left, R.Bottom - 1);
    end;
    LineTo(DC, R.Right - 1, R.Bottom - 1);
    LineTo(DC, R.Right - 1, R.Top - 1); }// fo. NT4SP5 draws LineTo not correctly.
    with R do
      if FBorderStyle <> bsNone then
        dhcgPolyline(DC,
          [Point(Left, Bottom - 1), Point(Right - 1, Bottom - 1), Point(Right - 1, Top - 1)])
      else dhcgPolyline(DC,
          [Point(Left, Top), Point(Left, Bottom - 1), Point(Right - 1, Bottom - 1), Point(Right - 1, Top - 1)]);
    SelectObject(DC, OldPen);
    DeleteObject(Pen);
  end;
end;
{--------------------------------------------------------------------------}
function TDbHdrCtrlGrid.GetChildParent: TComponent;
begin
  Result := FPanel;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.GetChildren(Proc: TGetChildProc; Root: TComponent);
begin
  FPanel.GetChildren(Proc, Root);
end;
{--------------------------------------------------------------------------}
function TDbHdrCtrlGrid.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource;
end;
{--------------------------------------------------------------------------}
function TDbHdrCtrlGrid.GetEditMode: Boolean;
begin
  Result := not Focused and ContainsControl(FindControl(GetFocus));
end;
{--------------------------------------------------------------------------}
function TDbHdrCtrlGrid.GetPanelBounds(Index: Integer): TRect;
begin
  Result.Left := FHorzShift; // fo.
  Result.Top := FTotalHeadersHeight + FPanelHeight * Index;
  Result.Right := Result.Left + FPanelWidth;
  Result.Bottom := Result.Top + FPanelHeight;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.HeaderResize(AHeaderControl: THeaderControl;
  AHeaderSection: THeaderSection);
var
  lHeader: TdhcgHeader;
  lSection: TdhcgSection;
begin
  if Headers.FindSection(AHeaderControl, AHeaderSection, lHeader, lSection) then begin
    lSection.Width := AHeaderSection.Width;
    if Assigned(FOnHeaderResize) then FOnHeaderResize(Self, lHeader, lSection);
    Invalidate;
  end;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.HeaderClick(AHeaderControl: THeaderControl;
  AHeaderSection: THeaderSection);
var
  lHeader: TdhcgHeader;
  lSection: TdhcgSection;
begin
  if (Headers.FindSection(AHeaderControl, AHeaderSection, lHeader, lSection))
    and Assigned(FonHeaderClick) then
    FonHeaderClick(Self, lHeader, lSection);
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.GetTabOrderList(List: TList);
begin
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.KeyDown(var Key: Word; Shift: TShiftState);
var
  GridKey: TDBCtrlGridKey;
begin
  inherited KeyDown(Key, Shift);
  GridKey := gkNull;
  case Key of
    VK_LEFT: GridKey := gkLeft;
    VK_RIGHT: GridKey := gkRight;
    VK_UP: GridKey := gkUp;
    VK_DOWN: GridKey := gkDown;
    VK_PRIOR: GridKey := gkPageUp;
    VK_NEXT: GridKey := gkPageDown;
    VK_HOME: GridKey := gkHome;
    VK_END: GridKey := gkEnd;
    VK_RETURN, VK_F2: GridKey := gkEditMode;
    VK_INSERT:
      if GetKeyState(VK_CONTROL) >= 0 then
        GridKey := gkInsert
      else GridKey := gkAppend;
    VK_DELETE: if GetKeyState(VK_CONTROL) < 0 then GridKey := gkDelete;
    VK_ESCAPE: GridKey := gkCancel;
  end;
  DoKey(GridKey);
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.PaintWindow(DC: HDC);
var
  I: Integer;
  Brush: HBrush;
begin
  AdjustRowCount; // fo.
  if csDesigning in ComponentState then begin
    FPanel.Update;
    Brush := CreateHatchBrush(HS_BDIAGONAL, ColorToRGB(clBtnShadow));
    SetBkColor(DC, ColorToRGB(Color));
    FillRect(DC, ClientRect, Brush);
    DeleteObject(Brush);
    for I := 1 to FRowCount - 1 do
      DrawPanelBackground(DC, GetPanelBounds(I), False, False);
  end
  else begin
    CreatePanelBitmap;
    try
      for I := 0 to FRowCount - 1 do
        if (FPanelCount <> 0) and (I = FPanelIndex) then
          FPanel.Update
        else DrawPanel(DC, I);
    finally
      DestroyPanelBitmap;
    end;
  end;
  { When width or height are not evenly divisible by panel size, fill the gaps }
  if HandleAllocated then begin
    if (Height <> FTotalHeadersHeight + FPanel.Height * FRowCount) then begin // fo.
      Brush := CreateSolidBrush(ColorToRGB(Color));
      FillRect(DC, Rect(0, FTotalHeadersHeight + FPanel.Height * FRowCount, Width, Height), Brush); // fo.
      DeleteObject(Brush);
    end;
    if (Width <> FPanel.Width) then begin
      Brush := CreateSolidBrush(ColorToRGB(Color));
      FillRect(DC, Rect(FPanelWidth + FHorzShift, 0, Width, Height), Brush); // fo.
      DeleteObject(Brush);
    end;
  end;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.PaintPanel(Index: Integer);
begin
  if Assigned(FOnPaintPanel) then FOnPaintPanel(Self, Index);
end;
{--------------------------------------------------------------------------}
function TDbHdrCtrlGrid.PointInPanel(const P: TSmallPoint): Boolean;
begin
  Result := (FPanelCount > 0)
    and PtInRect(GetPanelBounds(FPanelIndex), SmallPointToPoint(P));
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.ReadState(Reader: TReader);
begin
  inherited ReadState(Reader);
  FPanel.FixupTabList;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.Reset;
begin
  if csDesigning in ComponentState then
    FDataLink.BufferCount := 1
  else
    if (FRowCount + 1) * FPanelHeight < VisibleHeight then
      FDataLink.BufferCount := FRowCount + 1 // fo. Add 1 for better behavior.
    else FDataLink.BufferCount := FRowCount;
  DataSetChanged(True);
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.Scroll(Inc: Integer; ScrollLock: Boolean);
var
  NewIndex: Integer;
begin
  if FDataLink.Active and (Inc <> 0) then
    with FDataLink.DataSet do
      if State = dsInsert then begin
        UpdateRecord;
        if Modified then
          Post
        else if (Inc < 0) or not EOF then Cancel;
      end
      else begin
        CheckBrowseMode;
        DisableControls;
        try
          if ScrollLock then
            if Inc > 0 then
              MoveBy(Inc - MoveBy(Inc + FDataLink.BufferCount - FPanelIndex - 1))
            else MoveBy(Inc - MoveBy(Inc - FPanelIndex))
          else begin // fo.
            NewIndex := FPanelIndex + Inc;
            if (NewIndex >= 0) and (NewIndex < FDataLink.BufferCount) then
              MoveBy(Inc)
            else if MoveBy(Inc) = Inc then MoveBy(-MoveBy(0));
          end;
          if (Inc = 1) and EOF and FAllowInsert and CanModify then Append;
        finally
          EnableControls;
        end;
      end;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.ScrollMessage(var Message: TWMScroll);
var
  Key: TDBCtrlGridKey;
  SI: TScrollInfo;
begin
  if AcquireFocus then begin
    Key := gkNull;
    case Message.ScrollCode of
      SB_LINEUP: Key := gkScrollUp;
      SB_LINEDOWN: Key := gkScrollDown;
      SB_PAGEUP: Key := gkPageUp;
      SB_PAGEDOWN: Key := gkPageDown;
      SB_TOP: Key := gkHome;
      SB_BOTTOM: Key := gkEnd;
      SB_THUMBPOSITION:
        if FDataLink.Active and FDataLink.DataSet.IsSequenced then begin
          SI.cbSize := sizeof(SI);
          SI.fMask := SIF_ALL;
          GetScrollInfo(Self.Handle, SB_VERT, SI);
          if SI.nTrackPos <= 1 then
            Key := gkHome
          else
            if SI.nTrackPos >= FDataLink.DataSet.RecordCount then
              Key := gkEnd
            else begin
              FDataLink.DataSet.RecNo := SI.nTrackPos;
              Exit;
            end;
        end
        else begin
          case Message.Pos of
            0: Key := gkHome;
            1: Key := gkPageUp;
            3: Key := gkPageDown;
            4: Key := gkEnd;
          end;
        end;
    end;
    DoKey(Key);
  end;
end;
{--------------------------------------------------------------------------}
function TDbHdrCtrlGrid.FindNext(StartControl: TWinControl; GoForward: Boolean;
  var WrapFlag: Integer): TWinControl;
var
  I, StartIndex: Integer;
  List: TList;
begin
  List := TList.Create;
  try
    StartIndex := 0;
    I := 0;
    Result := StartControl;
    FPanel.GetTabOrderList(List);
    if List.Count > 0 then begin
      StartIndex := List.IndexOf(StartControl);
      if StartIndex = -1 then
        if GoForward then
          StartIndex := List.Count - 1
        else StartIndex := 0;
      I := StartIndex;
      repeat
        if GoForward then begin
          Inc(I);
          if I = List.Count then I := 0;
        end
        else begin
          if I = 0 then I := List.Count;
          Dec(I);
        end;
        Result := List[I];
      until (Result.CanFocus and Result.TabStop) or (I = StartIndex);
    end;
    WrapFlag := 0;
    if GoForward then begin
      if I <= StartIndex then WrapFlag := 1;
    end
    else if I >= StartIndex then WrapFlag := -1;
  finally
    List.Free;
  end;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.SelectNext(GoForward: Boolean);
var
  WrapFlag: Integer;
  ParentForm: TCustomForm;
  ActiveControl, Control: TWinControl;
begin
  ParentForm := GetParentForm(Self);
  if ParentForm <> nil then begin
    ActiveControl := ParentForm.ActiveControl;
    if ContainsControl(ActiveControl) then begin
      Control := FindNext(ActiveControl, GoForward, WrapFlag);
      if not (FDataLink.DataSet.State in dsEditModes) then
        FPanel.SetFocus;
      try
        if WrapFlag <> 0 then Scroll(WrapFlag, False);
      except
        ActiveControl.SetFocus;
        raise;
      end;
      if not Control.CanFocus then
        Control := FindNext(Control, GoForward, WrapFlag);
      Control.SetFocus;
    end;
  end;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.SetBounds(ALeft, ATop, AWidth, AHeight: Integer);
begin
  inherited SetBounds(ALeft, ATop, AWidth, AHeight);
  if (PanelWidth <= VisibleWidth) and (FHorzShift < 0) then begin
    ScrollBy(-FHorzShift, 0);
    FHorzShift := 0;
  end;
  UpdateHorzScrollBar;
  AdjustRowCount;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.SetBorderStyle(Value: TBorderStyle);
begin
  if FBorderStyle <> Value then begin
    FBorderStyle := Value;
    RecreateWnd;
  end;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.SetDataSource(Value: TDataSource);
begin
  FDataLink.DataSource := Value;
  UpdateDataLinks(FPanel, True);
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.SetEditMode(Value: Boolean);
var
  Control: TWinControl;
begin
  if GetEditMode <> Value then
    if Value then begin
      Control := FPanel.FindNextControl(nil, True, True, False);
      if Control <> nil then Control.SetFocus;
    end
    else SetFocus;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.SetHeaders(const Value: TdhcgHeaders);
begin
  FHeaders.Assign(Value);
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.SetPanelBorder(Value: TDbHdrCtrlGridBorder);
begin
  if FPanelBorder <> Value then begin
    FPanelBorder := Value;
    Invalidate;
    FPanel.Invalidate;
  end;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.SetPanelCount(Value: Integer);
begin
  if FPanelCount <> Value then begin
    FPanelCount := Value;
    SetBounds(Left, Top, Width, Height);
  end;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.SetPanelHeight(Value: Integer);
begin
  if Value < 1 then Value := 1;
  if Value > 65535 then Value := 65535;
  if FPanelHeight <> Value then begin
    FPanelHeight := Value;
    AdjustSize;
  end;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.SetPanelIndex(Value: Integer);
begin
  if FDataLink.Active and (Value < FPanelCount) then
    FDataLink.DataSet.MoveBy(Value - FPanelIndex);
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.SetPanelWidth(Value: Integer);
begin
  if Value < 1 then Value := 1;
  if Value > 65535 then Value := 65535;
  if FPanelWidth <> Value then begin
    FPanelWidth := Value;
    AdjustSize;
  end;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.SetRowCount(Value: Integer);
begin
  if Value < 1 then Value := 1;
  if Value > 100 then Value := 100;
  if FRowCount <> Value then begin
    FRowCount := Value;
    AdjustSize;
  end;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.SetSelectedColor(Value: TColor);
begin
  if Value <> FSelectedColor then begin
    FSelectedColor := Value;
    FSelColorChanged := Value <> Color;
    Invalidate;
    FPanel.Invalidate;
  end;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.SetTotalHeadersHeight(Value: Integer);
begin
  if FTotalHeadersHeight <> Value then begin
    FTotalHeadersHeight := Value;
    AdjustSize;
  end;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.UpdateDataLinks(Control: TControl; Inserting: Boolean);
var
  I: Integer;
  DataLink: TDataLink;
  lSection: TdhcgSection;
begin
  if Inserting and not (csReplicatable in Control.ControlStyle) then
    DatabaseError(SNotReplicatable);
  DataLink := TDataLink(Control.Perform(CM_GETDATALINK, 0, 0));
  if DataLink <> nil then begin
    DataLink.DataSourceFixed := False;
    if Inserting then begin
      DataLink.DataSource := DataSource;
      DataLink.DataSourceFixed := True;
    end;
  end;
  if Control is TWinControl then
    with TWinControl(Control) do
      for I := 0 to ControlCount - 1 do UpdateDataLinks(Controls[I], Inserting);
  if not Inserting then begin // fo.
    lSection := Headers.ControlToSection(Control);
    if lSection <> nil then lSection.Control := nil;
  end;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.UpdateHorzScrollBar;
var
  lScrollInfo: TScrollInfo;
begin
  if HandleAllocated then begin
    lScrollInfo.cbSize := SizeOf(lScrollInfo);
    lScrollInfo.fMask := SIF_ALL;
    lScrollInfo.nMin := 0;
    lScrollInfo.nPage := VisibleWidth;
    lScrollInfo.nMax := PanelWidth - 1;
    lScrollInfo.nPos := -FHorzShift;
    SetScrollInfo(Handle, SB_HORZ, lScrollInfo, True);
    if not (csDesigning in ComponentState) then FHeaders.Update(nil);
  end;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.UpdateScrollBar;
var
  SIOld, SINew: TScrollInfo;
begin
  if FDatalink.Active and HandleAllocated then
    with FDatalink.DataSet do begin
      SIOld.cbSize := sizeof(SIOld);
      SIOld.fMask := SIF_ALL;
      GetScrollInfo(Self.Handle, SB_VERT, SIOld);
      SINew := SIOld;
      if IsSequenced then begin
        SINew.nMin := 1;
        SINew.nPage := Self.RowCount;
        SINew.nMax := DWORD(RecordCount) + SINew.nPage - 1;
        if State in [dsInactive, dsBrowse, dsEdit] then
          SINew.nPos := RecNo;
      end
      else begin
        SINew.nMin := 0;
        SINew.nPage := 0;
        SINew.nMax := 4;
        if BOF then
          SINew.nPos := 0
        else if EOF then
          SINew.nPos := 4
        else SINew.nPos := 2;
      end;
      if (SINew.nMin <> SIOld.nMin) or (SINew.nMax <> SIOld.nMax) or
        (SINew.nPage <> SIOld.nPage) or (SINew.nPos <> SIOld.nPos) then
        SetScrollInfo(Self.Handle, SB_VERT, SINew, True);
    end;
end;
{--------------------------------------------------------------------------}
function TDbHdrCtrlGrid.VisibleWidth: Integer;
begin
  if HandleAllocated then
    Result := GetClientRect.Right
  else begin
    Result := Width - GetSystemMetrics(SM_CXVSCROLL);
    if FBorderStyle <> bsNone then Dec(Result, 2 * GetSystemMetrics(SM_CXBORDER));
  end;
end;
{--------------------------------------------------------------------------}
function TDbHdrCtrlGrid.VisibleHeight: Integer;
begin
  if HandleAllocated then
    Result := GetClientRect.Bottom - FTotalHeadersHeight
  else begin
    Result := Height - FTotalHeadersHeight;
    if PanelWidth > VisibleWidth then Dec(Result, GetSystemMetrics(SM_CYHSCROLL));
    if FBorderStyle <> bsNone then Dec(Result, 2 * GetSystemMetrics(SM_CYBORDER));
  end;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.WMLButtonDown(var Message: TWMLButtonDown);
var
  I: Integer;
  P: TPoint;
  Window: HWnd;
begin
  if FDataLink.Active then begin
    P := SmallPointToPoint(Message.Pos);
    for I := 0 to FPanelCount - 1 do
      if (I <> FPanelIndex) and PtInRect(GetPanelBounds(I), P) then begin
        FClicking := True;
        try
          SetPanelIndex(I);
        finally
          FClicking := False;
        end;
        P := ClientToScreen(P);
        Window := WindowFromPoint(P);
        if IsChild(FPanel.Handle, Window) then begin
          Windows.ScreenToClient(Window, P);
          Message.Pos := PointToSmallPoint(P);
          with TMessage(Message) do SendMessage(Window, Msg, WParam, LParam);
          Exit;
        end;
        Break;
      end;
  end;
  if AcquireFocus then begin
    if PointInPanel(Message.Pos) then begin
      EditMode := False;
      Click;
    end;
    inherited;
  end;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.WMLButtonDblClk(var Message: TWMLButtonDblClk);
begin
  if PointInPanel(Message.Pos) then DblClick;
  inherited;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.WMHScroll(var Message: TWMHScroll);
var
  lScrollInfo: TScrollInfo;
  lShift, lMaxShift: Integer;
begin
  lShift := 0;
  lMaxShift := VisibleWidth - PanelWidth;
  case Message.ScrollCode of
    SB_LINELEFT: lShift := 1;
    SB_LINERIGHT: lShift := -1;
    SB_PAGELEFT: lShift := VisibleWidth;
    SB_PAGERIGHT: lShift := -VisibleWidth;
    SB_LEFT: lShift := -lMaxShift;
    SB_RIGHT: lShift := lMaxShift;
    SB_THUMBPOSITION: begin
        lScrollInfo.cbSize := SizeOf(lScrollInfo);
        lScrollInfo.fMask := SIF_ALL;
        GetScrollInfo(Handle, SB_HORZ, lScrollInfo);
        lShift := -lScrollInfo.nTrackPos - FHorzShift;
      end;
    SB_ENDSCROLL: ;
  end;
  lShift := Max(lShift, lMaxShift - FHorzShift);
  lShift := Min(lShift, -FHorzShift);
  if lShift <> 0 then begin
    ScrollBy(lShift, 0);
    FHorzShift := FHorzShift + lShift;
  end;
  UpdateHorzScrollBar;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.WMVScroll(var Message: TWMVScroll);
begin
  ScrollMessage(Message);
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.WMEraseBkgnd(var Message: TMessage);
begin
  Message.Result := 1;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.WMPaint(var Message: TWMPaint);
begin
  PaintHandler(Message);
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.WMSetFocus(var Message: TWMSetFocus);
begin
  FFocused := True;
  FPanel.Repaint;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.WMKillFocus(var Message: TWMKillFocus);
begin
  FFocused := False;
  FPanel.Repaint;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.WMGetDlgCode(var Message: TWMGetDlgCode);
begin
  Message.Result := DLGC_WANTARROWS or DLGC_WANTCHARS;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.WMSize(var Message: TMessage);
begin
  inherited;
  Invalidate;
end;
{--------------------------------------------------------------------------}
function GetShiftState: TShiftState;
begin
  Result := [];
  if GetKeyState(VK_SHIFT) < 0 then Include(Result, ssShift);
  if GetKeyState(VK_CONTROL) < 0 then Include(Result, ssCtrl);
  if GetKeyState(VK_MENU) < 0 then Include(Result, ssAlt);
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.CMChildKey(var Message: TCMChildKey);
var
  ShiftState: TShiftState;
  GridKey: TDBCtrlGridKey;
begin
  with Message do
    if Sender <> Self then begin
      ShiftState := GetShiftState;
      if Assigned(OnKeyDown) then OnKeyDown(Sender, CharCode, ShiftState);
      GridKey := gkNull;
      case CharCode of
        VK_TAB:
          if not (ssCtrl in ShiftState) and
            (Sender.Perform(WM_GETDLGCODE, 0, 0) and DLGC_WANTTAB = 0) then
            if ssShift in ShiftState then
              GridKey := gkPriorTab
            else GridKey := gkNextTab;
        VK_RETURN:
          if (Sender.Perform(WM_GETDLGCODE, 0, 0) and DLGC_WANTALLKEYS = 0) then
            GridKey := gkEditMode;
        VK_F2: GridKey := gkEditMode;
        VK_ESCAPE: GridKey := gkCancel;
      end;
      if GridKey <> gkNull then begin
        DoKey(GridKey);
        Result := 1;
        Exit;
      end;
    end;
  inherited;
end;
{--------------------------------------------------------------------------}
procedure TDbHdrCtrlGrid.CMColorChanged(var Message: TMessage);
begin
  inherited;
  if not FSelColorChanged then
    FSelectedColor := Color;
end;
{--------------------------------------------------------------------------}
{ Defer action processing to datalink }
function TDbHdrCtrlGrid.ExecuteAction(Action: TBasicAction): Boolean;
begin
  Result := inherited ExecuteAction(Action) or (FDataLink <> nil)
   and FDataLink.ExecuteAction(Action);
end;
{--------------------------------------------------------------------------}
function TDbHdrCtrlGrid.UpdateAction(Action: TBasicAction): Boolean;
begin
  Result := inherited UpdateAction(Action) or (FDataLink <> nil) and
    FDataLink.UpdateAction(Action);
end;
{--------------------------------------------------------------------------}
end.

