{
This file Copyright 2000 (c) CDF, Inc.
Written By: Edward Flick (Directrix1@yahoo.com)
Derived from: TListDataset, written by DrBob
Use at your own risk!
}

unit TextFileDataset;

interface

uses
  SysUtils, Classes, Db, ListDataset;

type  // A row of data
  TTextFileType = (tftFixedWidths, tftDelimited);
  TTextData = class
  public
    data: array of String;
  end;

// A Read-Only dataset for importing text files
type
  TTextFileDataset = class(TListDataSet)
  private
    FColumnSeperator: String;
    FColumnWidths: String;
    FExtractColumnNamesFromFirstLine: boolean;
    FRowSeperator: String;
    FSkipRows: Integer;
    FStringQualifier: String;
    FTableName: String;
    FTextFileType: TTextFileType;
    Fwidths: array of integer;
    Frowwidth: integer;
    FBadRecords: TStrings;
    buf: array[0..4095] of char;
  protected
    curfile: TextFile;
    procedure InternalAddRecord(Buffer: Pointer; Append: Boolean); override;
    procedure InternalInitFieldDefs; override;
    procedure InternalInsert; override;
    procedure InternalPost; override;
    procedure SetFieldData(Field: TField; Buffer: Pointer); override;
    procedure ReadListData; override;
    function GetCanModify: Boolean; override;
    //Formatting methods
    function GetTextData: TTextData;
    procedure OpenFile;
    procedure ensureTableClosed(index: integer; t1: String);
    procedure setExtractColumnNamesFromFirstLine(t1: boolean);
    procedure setSkipRows(t1: integer);
    procedure setTextFileType(t1: TTextFileType);
    procedure setBadRecords(ti: TStrings);
  public
    constructor Create(Owner: TComponent); override;
    destructor Destroy; override;
    function GetFieldData(Field: TField; Buffer: Pointer): Boolean; override;
  published
    property ColumnSeperator: String index 0 read FColumnSeperator write ensureTableClosed;
    property ColumnWidths: String index 1 read FColumnWidths write ensureTableClosed;
    property ExtractColumnNamesFromFirstLine: boolean read FExtractColumnNamesFromFirstLine write setExtractColumnNamesFromFirstLine;
    property RowSeperator: String index 2 read FRowSeperator write ensureTableClosed;
    property SkipRows: Integer read FSkipRows write setSkipRows;
    property StringQualifier: String index 3 read FStringQualifier write ensureTableClosed;
    property TableName: String index 4 read FTableName write ensureTableClosed;
    property TextFileType: TTextFileType read FTextFileType write setTextFileType;
    property BadRecords: TStrings read FBadRecords write setBadRecords;
  end;

procedure Register;

implementation

uses
  TypInfo, Dialogs, Windows, Forms, Controls, fileCtrl;

procedure Register;
begin
  RegisterComponents('Data Access', [TTextFileDataset]);
end;

//Translation table for characters which are non-visible
function ParseSpecial(instr: String): String;
begin
result:=instr;
if result<>'' then
  begin
  result:=StringReplace(result,'<null>',chr(0),[rfReplaceAll,rfIgnoreCase]);
  result:=StringReplace(result,'<cr>',chr(13),[rfReplaceAll,rfIgnoreCase]);
  result:=StringReplace(result,'<lf>',chr(10),[rfReplaceAll,rfIgnoreCase]);
  result:=StringReplace(result,'<tab>',chr(9),[rfReplaceAll,rfIgnoreCase]);
  result:=StringReplace(result,'<>','',[rfReplaceAll,rfIgnoreCase]);
  end;
end;

//Checks if instr ends with seq
function EndsWith(instr, seq: String): boolean;
begin
if seq='' then
  result:=false
else
if length(seq)<=length(instr) then
  begin
  if copy(instr,length(instr)-length(seq)+1,length(seq))=seq then
    result:=true
  else
    result:=false;
  end
else
  result:=false;
end;

procedure TTextFileDataset.setBadRecords(ti: TStrings);
begin
//Do nothing READ-ONLY
end;

//If a property is changed, this validates it and ensures the table is closed
procedure TTextFileDataset.ensureTableClosed(index: integer; t1: String);
var
  j: integer;
  t: boolean;
begin
case index of
  0:  FColumnSeperator:=t1;
  1:
    begin
    t:=true;
    if length(t1)>0 then
      for j:=1 to length(t1) do
        t:=t and (((ord(t1[j])>=ord('0')) and (ord(t1[j])<=ord('9'))) or (t1[j]=';'));
    if t then
      FColumnWidths:=t1
    else
      raise Exception.create('Only valid chars for ColumnWidths are ''0''..''9'' and '';''.');
    end;
  2:  FRowSeperator:=t1;
  3:  if length(ParseSpecial(t1))<=1 then
        FStringQualifier:=t1
      else
        raise Exception.create('StringQualifier should only evaluate to one character.');
  4:  FTableName:=t1;
  end;
if not (csLoading in ComponentState) then
  active:=false;
end;

//Ditto
procedure TTextFileDataset.setExtractColumnNamesFromFirstLine(t1: boolean);
begin
if not (csLoading in ComponentState) then
  active:=false;
FExtractColumnNamesFromFirstLine:=t1;
end;

//Ditto
procedure TTextFileDataset.setSkipRows(t1: integer);
begin
if t1>=0 then
  begin
  if not (csLoading in ComponentState) then
    active:=false;
  FSkipRows:=t1;
  end
else
  raise Exception.create('Skip Rows cannot be negative');
end;

//Ditto
procedure TTextFileDataset.setTextFileType(t1: TTextFileType);
begin
if not (csLoading in ComponentState) then
  active:=false;
FTextFileType:=t1;
end;

//This function parses the text file, and returns a row of data
function TTextFileDataset.GetTextData: TTextData;
var
  done, done2: boolean;
  Buffer,rs,cs,sq: String;
  j, pos: integer;
  temp: char;
begin
  result:=TTextData.create();
  setLength(result.data,0);
  done:=false;
  rs:=ParseSpecial(frowseperator);
  cs:=ParseSpecial(fcolumnseperator);
  sq:=ParseSpecial(fstringqualifier);
  read(curfile,temp);
  Buffer:=temp;
  if TextFileType=tftFixedWidths then   //For fixed widths text files
    begin
    setLength(result.data,length(fwidths));
    while (length(Buffer)<(frowwidth+length(rs))) and not system.eof(curfile) do
      begin
      read(curfile,temp);
      Buffer:=Buffer+temp;
      end;
    pos:=1;
    for j:=0 to length(fwidths)-1 do
      if length(Buffer)>pos then
        begin
        result.data[j]:=trim(copy(Buffer,pos,fwidths[j]));
        inc(pos,fwidths[j]);
        end;
    end
  else
    repeat                            //For delimited text files
      if not system.eof(curfile) then
        begin
        if endswith(buffer, rs) then  //Row Seperator encountered
          begin
          buffer:=copy(buffer, 1, length(buffer)-length(rs));
          setLength(result.data, length(result.data)+1);
          result.data[length(result.data)-1]:=buffer;
          done:=true;
          end
        else
          if endswith(buffer, cs) then  //Column Seperator encountered
            begin
            buffer:=copy(buffer, 1, length(buffer)-length(cs));
            setLength(result.data, length(result.data)+1);
            result.data[length(result.data)-1]:=buffer;
            buffer:='';
            end
          else
            if endswith(buffer, sq) then  //Quote Character encountered
              begin
              buffer:=copy(buffer, 1, length(buffer)-length(sq));
              done2:=false;
              repeat
              read(curfile,temp);
              if temp=sq then
                begin
                if not system.eof(curfile) then
                  begin
                  read(curfile, temp);
                  buffer:=buffer+temp;
                  if temp<>sq then
                    done2:=true;
                  end
                else
                  done2:=true;
                end
              else
                buffer:=buffer+temp;
              until done2 or system.eof(curfile);
              end
            else
              begin
              read(curfile,temp);
              Buffer:=buffer+temp;
              end;
        end
      else
        begin                           //Its the end of file, flush whats left and move on
        if endswith(buffer, rs) then buffer:=copy(buffer, 1, length(buffer)-length(rs));
        if length(buffer)>0 then
          begin
          setLength(result.data, length(result.data)+1);
          result.data[length(result.data)-1]:=buffer;
          end;
        done:=true;
        end;
    until Done;
end;

procedure TTextFileDataset.OpenFile;   //Allocates the file cursor, skips rows,
var                                    //and initializes fixed widths fields
  j: integer;
  ts: string;
begin
  setlength(fwidths, 0);
  frowwidth:=0;
  try                                 //Fixed widths parser
    if fTextFileType=tftFixedWidths then
      begin
      if length(fcolumnwidths)>0 then
        begin
        ts:='';
        for j:=1 to length(fcolumnwidths) do
          begin
          if fcolumnwidths[j]=';' then
            begin
            if length(ts)>0 then
              begin
              setLength(fwidths, length(fwidths)+1);
              fwidths[length(fwidths)-1]:=strtoint(ts);
              inc(frowwidth,strtoint(ts));
              ts:='';
              end;
            end
          else
            ts:=ts+fcolumnwidths[j];
          end;
        if length(ts)>0 then
          begin
          setLength(fwidths, length(fwidths)+1);
          fwidths[length(fwidths)-1]:=strtoint(ts);
          inc(frowwidth,strtoint(ts));
          ts:='';
          end;
        end;
      end;
  except
    raise Exception.create('Invalid column widths');
    end;
  if not fileexists(ftablename) then                   //Table cursor allocator
    raise Exception.create('Table does not exist');
  try
    AssignFile(curfile, ftablename);
    SetTextBuf(curfile,buf);
    reset(curfile);
  except
    raise Exception.create('Table does not exist');
    end;
  try
    if FSkipRows>0 then                               //Row Skipper
      for j:=1 to FSkipRows do
        readln(curfile, ts);
  except
    try
      CloseFile(curfile);
    except
      end;
    raise Exception.create('Hit end of file while skipping rows');
    end;
end;

//Gets all the data in the table, and puts it in FList
procedure TTextFileDataset.ReadListData;
var
  j:integer;
  ttd:TTextData;
begin
  FList.Clear;
  try
    OpenFile;
    try
      if FExtractColumnNamesFromFirstLine then          //Skip the colnames row
        GetTextData;
    except
      raise Exception.create('Error while retrieving column names');
      end;
    j:=0;
    FBadRecords.Clear;
    while not system.eof(curFile) do
      begin
      inc(j);
      try
        ttd:=GetTextData;                            //Get a row
        if length(ttd.data)<>fields.Count then  //If the # of cols <> # of fields then
          FBadRecords.Add(inttostr(j))          // the row is bad
        else
          FList.Add (ttd);                      //It's Good!
      except
        FBadRecords.Add(inttostr(j));           //It's Bad!
        end;
      end;
  finally                                       //Clean up
    try
      CloseFile(curFile);
    except
      end;
    end;
end;

//Either takes the first row of data as the fieldnames, or makes fields in the
//form of FieldN where N=1..numcols in first row of data
procedure TTextFileDataset.InternalInitFieldDefs;
var
  theLine: TTextData;
  j: integer;
begin
  try
    OpenFile;
    theLine:=GetTextData;
    FieldDefs.Clear;
    for j:=0 to length(theLine.data)-1 do
      if FExtractColumnNamesFromFirstLine then
        begin
        FieldDefs.Add (theLine.data[j], ftString, 30, True); //Yippee, col names
        end
      else
        begin
        FieldDefs.Add ('Field'+inttostr(j+1), ftString, 30, True);//Oh boy generic
        end;
  finally
    try                                                       //Clean up time
      CloseFile(curFile);
    except
      end;
    end;
end;

procedure TTextFileDataset.InternalPost;
begin
  // TODO: support editing
end;

procedure TTextFileDataset.InternalAddRecord(Buffer: Pointer; Append: Boolean);
begin
  // TODO: support adding
end;

function DateTimeToNative(DataType: TFieldType; Data: TDateTime): TDateTimeRec;
var
  TimeStamp: TTimeStamp;
begin
  TimeStamp := DateTimeToTimeStamp(Data);
  case DataType of
    ftDate: Result.Date := TimeStamp.Date;
    ftTime: Result.Time := TimeStamp.Time;
  else
    Result.DateTime := TimeStampToMSecs(TimeStamp);
  end;
end;

//Called when the dataset wants to grab a fields worth of data
function TTextFileDataset.GetFieldData (
  Field: TField; Buffer: Pointer): Boolean;
var
  theLine: TTextData;
begin
  if (PRecInfo(ActiveBuffer).Index>=0) and (PRecInfo(ActiveBuffer).Index<flist.count) then
    begin
    try
      theLine := fList [PRecInfo(ActiveBuffer).Index] as TTextData;
      StrCopy (Buffer, pchar(theLine.data[Field.FieldNo-1]));
      Result := True;
    except
      raise Exception.create('Could not fetch field data for: '+Field.FieldName);
      end;
    end
  else
    raise Exception.create('Could not fetch field data for: '+Field.FieldName);
end;

// III: Move data from field to record buffer
procedure TTextFileDataset.SetFieldData(Field: TField; Buffer: Pointer);
begin
  // todo: support changes
end;

procedure TTextFileDataset.InternalInsert;
begin
  // todo: support inserting
end;

function TTextFileDataset.GetCanModify: Boolean;
begin
  Result := False; // read-only
end;

constructor TTextFileDataSet.Create(Owner: TComponent);
begin
  inherited;
  FBadRecords := TStringList.Create ();
end;

destructor TTextFileDataSet.Destroy;
begin
  inherited;
  FBadRecords.Free;
end;

end.

