(*------------------------------------------------------------------------------
** GaugeEdit Delphi Control:  Combines the functionality of a TEdit and TGauge
** Copyright (C) 2000  Brendan Rempel (rempelb@cybersurf.net)
**
** This library is free software; you can redistribute it and/or
** modify it under the terms of the GNU Lesser General Public
** License as published by the Free Software Foundation; either
** version 2.1 of the License.
**
** This library is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** Lesser General Public License for more details.
**
** You should have received a copy of the GNU Lesser General Public
** License along with this library; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
------------------------------------------------------------------------------*)

(*------------------------------------------------------------------------------
** The TGaugeEdit component is a copy of functionality of a TEdit control
** with a progress gauge under the editable text.
** To use this control, simply place it on your form and use it like you would
** use a TEdit box.  This also provides Position and Max properties
** to adjust the gauge under the text and the EndColor property to adjust the
** gauge color.
**------------------------------------------------------------------------------
** For an official copy of the LGPL license, go to
** http://www.gnu.org/copyleft/lesser.html
** In short, you may:
**   - use this component for free in any commercial or freeware
**     product, royalty free
**   - modify this component for use within your own application.
** you cannot
**   - distribute modified versions of this source code.
**     You can distribute compiled applications with use this library.
**   - charge money for this source other than for cost of media or warranty
**     services.  You can charge money for applications which use this library.
**
** If you have any suggestions, bugs or bug fixes, rave reviews, you can email
** me at:
**   rempelb@cybersurf.net
**
** You can also check my web page for updates, screen shots, and other Delphi
** components at:
**   http://brendan.cybersurf.net/delphi
**
**------------------------------------------------------------------------------
** Change Log:
**-|date       |developer      |function and reason for change
**-+-----------+---------------+------------------------------------------------
** |Sep-13-2000|Brendan Rempel |Library Created
------------------------------------------------------------------------------*)
unit GaugeEdit;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls, Menus, clipbrd;

{$ifdef VER130}
 {$define Delphi5} { Delphi 5 added context handling to TComponent }
{$endif}

type
  TSelType = (stNone, stLeft, stRight);

{$ifndef Delphi5}
  TWMContextMenu = packed record
    Msg: Cardinal;
    hWnd: HWND;
    case Integer of
      0: (
        XPos: Smallint;
        YPos: Smallint);
      1: (
        Pos: TSmallPoint;
        Result: Longint);
  end;
{$endif}

  TGaugeEdit = class(TCustomControl)
  private
    { Private declarations }
    FPosition: integer;
    FMax: integer;
    FEndColor: TColor;
    FBitmapCache: TBitmap;
    FFirstLoad: boolean;

    FSelType: TSelType;
    FReadOnly: boolean;

    FBC: array[0..255] of Longint;
    FOriginalText: string;

    FCursorTimer: TTimer;

    FCurrShift: TShiftState;

    FCursorVisible: boolean;
    FSelStart, FSelEnd: integer;
    FCursorPos: integer;
    FTextOffset: integer;
    FDragCursorPos: integer;

    FDragOn: boolean;

    FBorderStyle: TBorderStyle;
    FCharCase: TEditCharCase;
    FColor: TColor;
    FAutoSelect: boolean;
    FAvoidSelectAll: boolean;

    FOnChange: TNotifyEvent;

    procedure wmLButtonDown(var Msg: TwmLButtonDown); message WM_LBUTTONDOWN;
    procedure wmLButtonUp(var Msg: TwmLButtonUp); message WM_LBUTTONUP;
    procedure wmMouseMove(var Msg: TwmMouseMove); message WM_MOUSEMOVE;
    procedure wmLButtonDblClk(var Msg: TwmLButtonDblClk); message WM_LBUTTONDBLCLK;
{$ifndef Delphi5}
    procedure WMContextMenu(var Message: TWMContextMenu); message WM_CONTEXTMENU;
{$endif}
    procedure wmGetDlgCode(var M: TWmGetDlgCode); message WM_GETDLGCODE;
    procedure wmCmTextChanged(var M: TMessage); message CM_TEXTCHANGED;

    procedure FlashCursor(Sender: TObject);
    procedure FontChange(Sender: TObject);

    procedure SetMax(const Value: integer);
    procedure SetPosition(const Value: integer);
    procedure SetEndColor(const Value: TColor);

    procedure MenuUndoClick(Sender: TObject);
    procedure MenuCutClick(Sender: TObject);
    procedure MenuCopyClick(Sender: TObject);
    procedure MenuPasteClick(Sender: TObject);
    procedure MenuDeleteClick(Sender: TObject);
    procedure MenuSelectAllClick(Sender: TObject);
    procedure SetBorderStyle(const Value: TBorderStyle);
    procedure SetCharCase(const Value: TEditCharCase);
    procedure SetColor(const Value: TColor);

    procedure UpdateText(const Value: TCaption);
  protected
    { Protected declarations }
    procedure LoadColors; dynamic;

    procedure PaintBorders(Canvas: TCanvas); virtual;
    procedure DrawGradient(Canvas: TCanvas); virtual;
    procedure DrawBackground(Canvas: TCanvas); virtual;
    procedure DrawText(Canvas: TCanvas); virtual;
    procedure DrawCursor(Canvas: TCanvas); virtual;

    procedure Paint; override;

    procedure DoChange; dynamic;
    procedure DoEnter; override;
    procedure DoExit; override;

{$ifdef Delphi5}
    procedure DoContextPopup(MousePos: TPoint; var Handled: Boolean); override;
{$else}
    procedure DoContextPopup(MousePos: TPoint; var Handled: Boolean); dynamic;
{$endif}
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure KeyPress(var Key: Char); override;
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  published
    { Published declarations }
    property AutoSelect: boolean read FAutoSelect write FAutoSelect default True;
    property BorderStyle: TBorderStyle read FBorderStyle write SetBorderStyle default bsSingle;
    property CharCase: TEditCharCase read FCharCase write SetCharCase default ecNormal;

    property Color: TColor read FColor write SetColor default clWindow;
    property EndColor: TColor read FEndColor write SetEndColor;

    property Max: integer read FMax write SetMax;
    property Position: integer read FPosition write SetPosition;

    property ReadOnly: boolean read FReadOnly write FReadOnly;

    property OnChange: TNotifyEvent read FOnChange write FOnChange;

    property Anchors;
    property Constraints;
    property Cursor default crIBeam;
    property DragCursor;
    property DragKind;
    property DragMode;
    property Enabled;
    property Font;
    property ParentColor;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Text;
    property Visible;

    property OnClick;
{$ifdef Delphi5}
    property OnContextPopup;
{$endif}
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDock;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
    property OnStartDock;
    property OnStartDrag;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Cool Stuff', [TGaugeEdit]);
end;

(*------------------------------------------------------------------------------
** Popup menu for text copy/paste
------------------------------------------------------------------------------*)
var
  EditMenu: TPopupMenu;
  MenuUndo, MenuCut, MenuCopy, MenuPaste, MenuDelete, MenuSelectAll: TMenuItem;

{ TGaugeEdit }

(*------------------------------------------------------------------------------
** Constructor - property initialization
------------------------------------------------------------------------------*)
constructor TGaugeEdit.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);

  ControlStyle:= [csCaptureMouse, csClickEvents, csOpaque, csDoubleClicks, csReflector];

  BorderStyle:= bsSingle;
  CharCase:= ecNormal;
  FColor:= clWindow;
  FAutoSelect:= True;
  FFirstLoad:= True;

  Width:= 121;
  Height:= 21;

  FMax:= 100;
  FPosition:= 50;

  Cursor:= crIBeam;

  FEndColor:= $0099CCFF;

  FSelStart:= 0;
  FSelEnd:= 0;
  FCursorPos:= 0;

  FTextOffset:= 0;

  if not (csSetCaption in ControlStyle) then
    ControlStyle:= ControlStyle + [csSetCaption];

  FDragOn:= False;

  if csDesigning in ComponentState then
    Font.OnChange:= FontChange;

  LoadColors;
  TabStop:= True;

  FCursorVisible:= False;

  FBitmapCache:= TBitmap.Create;
end;

(*------------------------------------------------------------------------------
** Destructor
------------------------------------------------------------------------------*)
destructor TGaugeEdit.Destroy;
begin
  if FCursorTimer <> Nil then
    FCursorTimer.Free;

  FBitmapCache.Free;

  inherited Destroy;
end;

(*------------------------------------------------------------------------------
** PaintBorders - Override this procedure to change how borders are painted
------------------------------------------------------------------------------*)
procedure TGaugeEdit.PaintBorders(Canvas: TCanvas);
begin
  if FBorderStyle = bsSingle then
    with Canvas do
    begin
      Pen.Color:= clBtnShadow;
      MoveTo(0,Height-1); LineTo(0,0); LineTo(Width-1,0);

      Pen.Color:= clBtnHighlight;
      LineTo(Width-1,Height-1); LineTo(-1,Height-1);

      Pen.Color:= clWindowFrame;
      MoveTo(1,Height-2); LineTo(1,1); LineTo(Width-2,1);

      Pen.Color:= clBtnFace;
      LineTo(Width-2,Height-2); LineTo(0,Height-2);
    end;
end;

(*------------------------------------------------------------------------------
** DrawGradient - Override this procedure to change how the gradient is painted
------------------------------------------------------------------------------*)
procedure TGaugeEdit.DrawGradient(Canvas: TCanvas);
var
  TempRect: TRect;
  TempStepV: single;
  ColorCode: integer;
  TempTop: integer;
  TempHeight: integer;
  x: integer;
  Slant: array[0..3] of TPoint;
begin
  if FPosition > 0 then
  begin
    x:= FPosition * Width div FMax;      { width of the gauge }

    TempStepV:= x / 255;                 { Width of each color band }
    TempHeight:= Trunc(TempStepV + 1);   { adjust up to ensure overlap }

    with Canvas do
    begin
      TempTop:= 0;
      TempRect.Top:= 0;
      TempRect.Bottom:= Height;

      for ColorCode:= 0 to 255 do        { draw bands }
      begin
        Brush.Color:= FBC[ColorCode];

        TempRect.Left:= TempTop;
        TempRect.Right:= TempTop + TempHeight;

        FillRect(TempRect);
        TempTop:= Trunc(TempStepV * ColorCode);
      end;

      Slant[0].x:= x; Slant[0].y:= Height; { calculate slant edge polygon }
      Slant[1].x:= x; Slant[1].y:= 0;
      Slant[2].x:= x + 14; Slant[2].y:= 0;
      Slant[3]:= Slant[0];

      Pen.Color:= Brush.Color;
      Polygon(Slant);                      { draw slanted edge }
    end;
  end;
end;

(*------------------------------------------------------------------------------
** DrawBackground - Override this procedure to modify how the background is
** painted
------------------------------------------------------------------------------*)
procedure TGaugeEdit.DrawBackground(Canvas: TCanvas);
begin
  with Canvas do
  begin
    Pen.Color:= FColor;
    Brush.Color:= FColor;

    Rectangle(0,0,Width,Height);
  end;
end;

(*------------------------------------------------------------------------------
** DrawText - Override this procedure to modify how text is painted
------------------------------------------------------------------------------*)
procedure TGaugeEdit.DrawText(Canvas: TCanvas);
var
  x, y: integer;
  s1, s2: integer;
  TestString: string;
  t: string;
begin
  if Text <> '' then
    with Canvas do
    begin
      Font.Color:= Self.Font.Color;
      Brush.Style:= bsClear;

      y:= Height shr 1 - TextHeight('Wy') shr 1;
      if Focused and (FSelType <> stNone) then
      begin
        if FSelEnd < FSelStart then            { correct backwards start/stop }
        begin
          s1:= FSelEnd; s2:= FSelStart;
        end
        else
        begin
          s1:= FSelStart; s2:= FSelEnd;
        end;

        Dec(s1,FTextOffset);                   { adjust selection with offsets }
        Dec(s2,FTextOffset);

        TestString:= Copy(Text,FTextOffset+1,Length(Text)); { get visible text }

        x:= 4;
        t:= Copy(TestString, 1, s1 - 1); { get first unselect section of the text }
        TextOut(x, y, t);                      { draw it }
        Inc(x,TextWidth(t));

        Brush.Color:= clHighlight;
        Brush.Style:= bsSolid;
        Font.Color:= clHighlightText;

        if s1 < 0 then              { reset the selection's offset if beyond left margin }
          s1:= 1;

        t:= Copy(TestString,s1, s2 - s1 + 1);  { get selected text }
        TextOut(x, y, t);                      { draw it }
        Inc(x,TextWidth(t));

        Brush.Style:= bsClear;
        Font.Color:= Self.Font.Color;
        t:= Copy(TestString,s2 + 1,Length(Text)); { get second unselected text }
        TextOut(x, y, t);                      { and draw it }
      end
      else
        TextOut(4, y, Copy(Text,FTextOffset + 1, Length(Text))); { draw all text as unselected }
    end;
end;

(*------------------------------------------------------------------------------
** DrawCursor - Override this procedure to modify how the cursor is painted
------------------------------------------------------------------------------*)
procedure TGaugeEdit.DrawCursor(Canvas: TCanvas);
begin
  if FCursorVisible and Focused then
    with Canvas do
    begin
      Pen.Color:= clWindowFrame;
      MoveTo(TextWidth(Copy(Text,FTextOffset + 1,FCursorPos-FTextOffset))+4,4);
      LineTo(TextWidth(Copy(Text,FTextOffset + 1,FCursorPos-FTextOffset))+4,Height-4);
    end;
end;

(*------------------------------------------------------------------------------
** Paint - Overrides the TCustomControl Paint - called when the component
** has been invalidated
------------------------------------------------------------------------------*)
procedure TGaugeEdit.Paint;
begin
  FBitmapCache.Width:= Width;
  FBitmapCache.Height:= Height;

  FBitmapCache.Canvas.Font.Assign(Font);

  DrawBackground(FBitmapCache.Canvas);

  if FPosition > 0 then
    DrawGradient(FBitmapCache.Canvas);

  if Text <> '' then
    DrawText(FBitmapCache.Canvas);

  if FCursorVisible and Focused then
    DrawCursor(FBitmapCache.Canvas);

  if FBorderStyle = bsSingle then
    PaintBorders(FBitmapCache.Canvas);

  Canvas.Draw(0,0,FBitmapCache);
end;

(*------------------------------------------------------------------------------
** SetMax - sets the maximum position of the gauge
------------------------------------------------------------------------------*)
procedure TGaugeEdit.SetMax(const Value: integer);
begin
  if Value > 0 then
  begin
    FMax:= Value;
    Invalidate;
  end;
end;

(*------------------------------------------------------------------------------
** SetPosition - sets the position of the gauge
------------------------------------------------------------------------------*)
procedure TGaugeEdit.SetPosition(const Value: integer);
begin
  if (Value >= 0) and (Value <= FMax) then
  begin
    FPosition:= Value;
    Invalidate;
  end;
end;

(*------------------------------------------------------------------------------
** wmCmTextChanged - called by TComponent when the published Text property
** changes
------------------------------------------------------------------------------*)
procedure TGaugeEdit.wmCmTextChanged(var M: TMessage);
begin
  inherited;

  if FFirstLoad then
  begin
    FFirstLoad:= False;
    FOriginalText:= Text;
  end;

  if FCursorPos > Length(Text)+1 then
    FCursorPos:= Length(Text)+1;

  Invalidate;

  DoChange;

  FTextOffset:= 0;
end;

(*------------------------------------------------------------------------------
** UpdateText - sets the value of the edit box
------------------------------------------------------------------------------*)
procedure TGaugeEdit.UpdateText(const Value: TCaption);
begin
  if Not FReadOnly and (Text <> Value) then
  begin
    Text:= Value;
    if FCursorPos > Length(Text)+1 then
      FCursorPos:= Length(Text)+1;

    Invalidate;

    DoChange;
  end;
end;

(*------------------------------------------------------------------------------
** DoChange - calls the OnChange event
------------------------------------------------------------------------------*)
procedure TGaugeEdit.DoChange;
begin
  if Assigned(FOnChange) then
    FOnChange(Self);
end;

(*------------------------------------------------------------------------------
** LoadColors - initializes the colors used in the gradient
------------------------------------------------------------------------------*)
procedure TGaugeEdit.LoadColors;
var
  X: Integer;
  SColor, EColor : TColor;
  DiffR, DiffG, DiffB : Currency;
  LastR, LastG, LastB : Currency;
  MySR, MySG, MySB : Integer;
  MyER, MyEG, MyEB : Integer;

  procedure SortOut(SelCol : TColor; var CRed : Integer;
                                     var CGreen : Integer;
                                     var CBlue : Integer);
  begin
    CRed := GetRValue(ColorToRGB(SelCol));
    CGreen := GetGValue(ColorToRGB(SelCol));
    CBlue := GetBValue(ColorToRGB(SelCol));
  end;
begin
  SortOut(FEndColor, MySR, MySG, MySB);
  SortOut(FColor, MyER, MyEG, MyEB);

  EColor := RGB(MySR, MySG, MySB); { set the colors normally }
  SColor := RGB(MyER, MyEG, MyEB);

  DiffR := (GetRValue(SColor) - GetRValue(EColor)) / 255.0; { work out the difference between }
  DiffG := (GetGValue(SColor) - GetGValue(EColor)) / 255.0; { each R G and B pair and divide }
  DiffB := (GetBValue(SColor) - GetBValue(EColor)) / 255.0; { into 255 increments }

  FBC[0] := SColor; { fix the start color }
  LastR := GetRValue(SColor); { remember the last colors used }
  LastG := GetGValue(SColor); { i.e. the start color }
  LastB := GetBValue(SColor);

  for X := 1 to 254 do
  begin
    LastR := LastR - DiffR; { keep incrementing the color stored }
    LastG := LastG - DiffG;
    LastB := LastB - DiffB;
    FBC[x] := RGB( Trunc(LastR), Trunc(LastG), Trunc(LastB) );
  end;
  FBC[255] := EColor; { fix the end color }
end;

(*------------------------------------------------------------------------------
** SetEndColor - Sets the final color in the gradient
------------------------------------------------------------------------------*)
procedure TGaugeEdit.SetEndColor(const Value: TColor);
begin
  FEndColor:= Value;
  LoadColors;
  Invalidate;
end;

(*------------------------------------------------------------------------------
** DoEnter - Overrides TWinControl to detect when this control gains focus
------------------------------------------------------------------------------*)
procedure TGaugeEdit.DoEnter;
begin
  FCursorVisible:= True;

  if FCursorTimer = Nil then { create timer object (this cuts down on window handles }
  begin
    FCursorTimer:= TTimer.Create(Nil);
    FCursorTimer.Interval:= 300;
    FCursorTimer.Enabled:= False;
    FCursorTimer.OnTimer:= FlashCursor;
  end;

  if FAutoSelect and Not FAvoidSelectAll then
    MenuSelectAllClick(Self);

  FCursorTimer.Enabled:= True;
  Invalidate;
end;

(*------------------------------------------------------------------------------
** DoExit - Overrides TWinControl to detect when this control loses focus
------------------------------------------------------------------------------*)
procedure TGaugeEdit.DoExit;
begin
  FCursorVisible:= False;

  if FCursorTimer <> Nil then { free and nil timer }
  begin
    FCursorTimer.Enabled:= False;
    FCursorTimer.Free;
    FCursorTimer:= Nil;
  end;

  FSelType:= stNone;
  FSelStart:= 0;
  FSelEnd:= 0;
  FTextOffset:= 0;
  FCursorPos:= 0;

  Invalidate;
end;

(*------------------------------------------------------------------------------
** FlashCursor - Inverts the visiblility of the cursor
------------------------------------------------------------------------------*)
procedure TGaugeEdit.FlashCursor(Sender: TObject);
begin
  FCursorVisible:= Not FCursorVisible;
  Invalidate;
end;

(*------------------------------------------------------------------------------
** wmLButtonDown - called by Windows when the left mouse button is depressed.
** This is used to set focus to the control if not focused and positions the
** cursor.
------------------------------------------------------------------------------*)
procedure TGaugeEdit.wmLButtonDown(var Msg: TwmLButtonDown);
var
  i: integer;
  TestString: string;
  CharWidth: integer;
begin
  inherited;

  try
    FSelType:= stNone;
    FSelStart:= 0;
    FSelEnd:= 0;

    FBitmapCache.Canvas.Font.Assign(Font);  { set cache font for text sizing }

    TestString:= Copy(Text,FTextOffset+1,Length(Text));
    if TestString <> '' then
    begin
      for i:= 0 to Length(TestString) do { locate character position of cursor }
      begin
        CharWidth:= FBitmapCache.Canvas.TextWidth(Copy(TestString, i+1, 1)) shr 1;

        if Msg.XPos < FBitmapCache.Canvas.TextWidth(Copy(TestString, 1, i)) + CharWidth + 4 then
        begin
          FCursorPos:= i + FTextOffset;
          Exit; { skip to finally section }
        end;
      end;
    end;
    FCursorPos:= Length(Text);
  finally
    if Not Focused then
    begin
      FAvoidSelectAll:= True;
      SetFocus;
      FAvoidSelectAll:= False;
    end;
    FDragOn:= True;
    FDragCursorPos:= FCursorPos;
    Invalidate;
  end;
end;

(*------------------------------------------------------------------------------
** wmLButtonUp - called by Windows when the Left mouse button is released.
** This is used to stop cursor drag
------------------------------------------------------------------------------*)
procedure TGaugeEdit.wmLButtonUp(var Msg: TwmLButtonUp);
var
  x: integer;
begin
  inherited;

  FDragOn:= False;

  if FSelEnd < FSelStart then
  begin
    x:= FSelEnd;
    FSelEnd:= FSelStart;
    FSelStart:= x;
    Invalidate;
  end;
end;

(*------------------------------------------------------------------------------
** wmMouseMove - called by windows when the mouse moves within this control.
** Using the wmLButtonDown and wmLButtonUp, this us used to detect when text
** is dragged within this control to make text selections
------------------------------------------------------------------------------*)
procedure TGaugeEdit.wmMouseMove(var Msg: TwmMouseMove);
var
  i: integer;
  TestString: string;
  CharWidth: integer;

  NewCurPos: integer;
  NewStart, NewEnd: integer;
  Found: boolean;
begin
  inherited;

  FBitmapCache.Canvas.Font.Assign(Font);

  if FDragOn then { execute only if mouse button is down }
  begin
    if (Msg.XPos < 0) and (FTextOffset > 0) then { scroll text (offset) if mouse is outside window to the left }
      Dec(FTextOffset,4);

    if (Msg.XPos > Width) and                    { scroll text (offset) if mouse is outside window to the right }
       (FBitmapCache.Canvas.TextWidth(Copy(Text, FTextOffset + 1, FCursorPos-FTextOffset)) + 8 > Width) then
      Inc(FTextOffset,4);

    if FSelType = stNone then
      FSelType:= stLeft;

    NewCurPos:= Length(Text);;

    TestString:= Copy(Text,FTextOffset+1,Length(Text));
    if TestString <> '' then
    begin
      i:= 0;
      Found:= False;
      while Not Found and (i <= Length(TestString)) do { locate new cursor position }
      begin
        CharWidth:= FBitmapCache.Canvas.TextWidth(Copy(TestString, i+1, 1)) shr 1;

        if Msg.XPos < FBitmapCache.Canvas.TextWidth(Copy(TestString, 1, i)) + CharWidth + 4 then
        begin
          NewCurPos:= i + FTextOffset;
          Found:= True;
        end
        else
          Inc(i);
      end;
    end;

    if NewCurPos = FDragCursorPos then
      FSelType:= stNone;

    NewEnd:= NewCurPos + 1;             { calculate selection area }
    NewStart:= FDragCursorPos + 1;

    if NewStart > NewEnd then
      Dec(NewStart)
    else
      Dec(NewEnd);

    if (NewStart <> FSelStart) or (NewEnd <> FSelEnd) or (NewCurPos <> FCursorPos) then
    begin                               { update selection only if changed }
      if FSelType = stNone then
      begin
        FSelStart:= 0;
        FSelEnd:= 0;
      end
      else
      begin
        FSelStart:= NewStart;
        FSelEnd:= NewEnd;
        FCursorPos:= NewCurPos;
        Invalidate;
      end;
    end;
  end;
end;

(*------------------------------------------------------------------------------
** wmLButtonDblClk - This is called by Windows when the user double clicks
** this control.  This is interpreted as a Select All.
------------------------------------------------------------------------------*)
procedure TGaugeEdit.wmLButtonDblClk(var Msg: TwmLButtonDblClk);
begin
  inherited;

  MenuSelectAllClick(Self);
end;

(*------------------------------------------------------------------------------
** wmGetDlgCode - This is called by Windows to retrieve this controls options.
** This is required to tell windows that this control wishes to receive cursor
** movements and dialog characters.
------------------------------------------------------------------------------*)
procedure TGaugeEdit.wmGetDlgCode(var M: TWmGetDlgCode);
begin
  inherited;

  M.Result:= DLGC_WANTARROWS or DLGC_WANTCHARS;
end;

(*------------------------------------------------------------------------------
** KeyDown - This overrides the TWinControl to detect when any key is pressed.
** This is used to detect cursor movement
------------------------------------------------------------------------------*)
procedure TGaugeEdit.KeyDown(var Key: Word; Shift: TShiftState);
  (*----------------------------------------------------------------------------
  ** LeftCursor - moves the cursor to the left
  ----------------------------------------------------------------------------*)
  procedure LeftCursor;
  var
    Found: boolean;
    StartPos: integer;
  begin
    if FCursorPos > 0 then
    begin
      if ssCtrl in FCurrShift then { move cursor the the next word }
      begin
        StartPos:= FCursorPos;
        Dec(FCursorPos);

        Found:= False;
        while Not Found and (FCursorPos > 0) do { find start of word }
          case Text[FCursorPos] of
            'A'..'Z', 'a'..'z':
              Found:= True;
            else
              Dec(FCursorPos);
          end;

        Found:= False;
        while Not Found and (FCursorPos > 0) do { find end of next word }
          case Text[FCursorPos] of
            'A'..'Z', 'a'..'z':
              Dec(FCursorPos);
            else
              Found:= True;
          end;
      end
      else
      begin
        Dec(FCursorPos);
        StartPos:= 0; { redundant but kills warning }
      end;

      if ssShift in FCurrShift then { select text }
      begin
        case FSelType of
          stNone:                       { new selection }
          begin
            FSelStart:= FCursorPos + 1;
            FSelType:= stLeft;

            if ssCtrl in FCurrShift then { also select current word }
            begin
              Found:= False;
              FSelEnd:= StartPos + 1;

              while Not Found and (FSelEnd <= Length(Text)) do { find start of this word }
              begin
                case Text[FSelEnd] of
                  'A'..'Z','a'..'z':
                    Found:= True;
                  else
                    Inc(FSelEnd);
                end;
              end;

              Found:= False;
              while Not Found and (FSelEnd <= Length(Text)) do { find end of next word }
              begin
                case Text[FSelEnd] of
                  'A'..'Z','a'..'z':
                    Inc(FSelEnd);
                  else
                    Found:= True;
                end;
              end;
              Dec(FSelEnd);
            end
            else
              FSelEnd:= FCursorPos + 1;
          end;
          stLeft:
            FSelStart:= FCursorPos+1; { set selection start }
          stRight:
          begin
            FSelEnd:= FCursorPos;
            if FSelEnd+1 = FSelStart then
              FSelType:= stNone;
          end;
        end;
      end
      else
        FSelType:= stNone;

      if FSelType = stNone then { clear selection if none }
      begin
        FSelStart:= 0;
        FSelEnd:= 0;
      end;

      if FCursorPos -1 < FTextOffset then { adjust offsets if cursor goes outside }
      begin
        Dec(FTextOffset,10);
        if FTextOffset < 0 then
          FTextOffset:= 0;
      end;

      Invalidate;
    end
    else if (FSelType <> stNone) and not (ssShift in FCurrShift) then
    begin
      FSelType:= stNone;
      Invalidate;
    end;
  end;

  (*----------------------------------------------------------------------------
  ** RightCursor - moves the cursor to the right
  ----------------------------------------------------------------------------*)
  procedure RightCursor;
  var
    Found: boolean;
  begin
    if FCursorPos < Length(Text) then { make sure cursor doesn't exit text }
    begin
      if ssCtrl in FCurrShift then { if word select }
      begin
        Inc(FCursorPos);
        Found:= False;
        while Not Found and (FCursorPos <= Length(Text)) do { find end of current word }
          case Text[FCursorPos] of
            'A'..'Z', 'a'..'z':
              Inc(FCursorPos);
            else
              Found:= True;
          end;

        Found:= False;
        while Not Found and (FCursorPos <= Length(Text)) do { find start of next word }
          case Text[FCursorPos] of
            'A'..'Z', 'a'..'z':
              Found:= True;
            else
              Inc(FCursorPos);
          end;
        Dec(FCursorPos);
      end
      else
        Inc(FCursorPos);

      if ssShift in FCurrShift then { if marking a selection }
      begin
        case FSelType of
          stNone:
          begin
            FSelEnd:= FCursorPos;   { start a new selection }
            FSelType:= stRight;

            if ssCtrl in FCurrShift then
            begin
              Found:= False;
              FSelStart:= FCursorPos - 1;

              while Not Found and (FSelStart > 0) do { find start of current word }
              begin
                case Text[FSelStart] of
                  'A'..'Z','a'..'z':
                    Found:= True;
                  else
                    Dec(FSelStart);
                end;
              end;

              Found:= False;
              while Not Found and (FSelStart > 0) do { find end of current word }
              begin
                case Text[FSelStart] of
                  'A'..'Z','a'..'z':
                    Dec(FSelStart);
                  else
                    Found:= True;
                end;
              end;
              Inc(FSelStart);
            end
            else
              FSelStart:= FCursorPos;
          end;
          stRight:
            FSelEnd:= FCursorPos; { move end of selection to the cursor }
          stLeft:
          begin
            FSelStart:= FCursorPos+1;  { move start of selection to the cursor }
            if FSelEnd+1 = FSelStart then
              FSelType:= stNone;
          end;
        end;
      end
      else
        FSelType:= stNone;

      if FSelType = stNone then { clear empty selection }
      begin
        FSelStart:= 0;
        FSelEnd:= 0;
      end;

      if FBitmapCache.Canvas.TextWidth(Copy(Text, FTextOffset + 1, FCursorPos-FTextOffset)) + 8 > Width then
        Inc(FTextOffset,10); { adjust for cursors outside visible area }

      Invalidate;
    end
    else if (FSelType <> stNone) and not (ssShift in FCurrShift) then
    begin
      FSelType:= stNone;
      Invalidate;
    end;
  end;

  (*----------------------------------------------------------------------------
  ** HomeCursor - moves the cursor to the beginning of the line
  ----------------------------------------------------------------------------*)
  procedure HomeCursor;
  begin
    if FCursorPos > 0 then
    begin
      if FCurrShift = [ssShift] then { if making a selection }
      begin
        case FSelType of
          stNone:
          begin
            FSelStart:= 1;           { start selection at the first character }
            FSelEnd:= FCursorPos;    { and end it at the cursor position }
            FSelType:= stLeft;
          end;
          stLeft:
            FSelStart:= 1;           { readjust selection start }
          stRight:
          begin
            FSelEnd:= FSelStart;     { start selection at the first character }
            FSelStart:= 1;
            FSelType:= stLeft;
          end;
        end;
      end
      else
        FSelType:= stNone;

      FTextOffset:= 0;
      FCursorPos:= 0;
      Invalidate;
    end
    else if (FSelType <> stNone) and (FCurrShift <> [ssShift]) then
    begin
      FSelType:= stNone;
      Invalidate;
    end;
  end;

  (*------------------------------------------------------------------------------
  ** EndCursor - moves the cursor to the end of the line
  ------------------------------------------------------------------------------*)
  procedure EndCursor;
  begin
    if FCursorPos <> Length(Text) then
    begin
      if FCurrShift = [ssShift] then { if making a selection }
      begin
        case FSelType of
          stNone:
          begin                       { make a new selection }
            FSelStart:= FCursorPos+1;
            FSelEnd:= Length(Text);
            FSelType:= stRight;
          end;
          stRight:
            FSelEnd:= Length(Text);  { adjust the current selection }
          stLeft:
          begin
            FSelStart:= FSelEnd;
            FSelend:= Length(Text);
            FSelType:= stRight;
          end;
        end;
      end
      else
        FSelType:= stNone;

      FCursorPos:= Length(Text);     { move cursor to the end of the text }
      if FBitmapCache.Canvas.TextWidth(Text) > Width then
      begin                           { move text to the left to move cursor into }
        FTextOffset:= Length(Text);  { visible area }
        while (FTextOffset > 0) and (FBitmapCache.Canvas.TextWidth(Copy(Text,FTextOffset-2,Length(Text))) < Width) do
          Dec(FTextOffset);
      end;
      Invalidate;
    end
    else if (FSelType <> stNone) and (FCurrShift <> [ssShift]) then
    begin
      FSelType:= stNone;
      Invalidate;
    end;
  end;
  (*----------------------------------------------------------------------------
  ** DeleteChar - delete the character to the right of the cursor
  ----------------------------------------------------------------------------*)
  procedure DeleteChar;
  begin
    if FSelType = stNone then { is text selected? }
    begin
      if FCursorPos < Length(Text) then { delete selected text }
        UpdateText(Copy(Text,1,FCursorPos)+Copy(Text,FCursorPos+2,Length(Text)));
    end
    else
    begin
      if FCurrShift = [ssShift] then { cut if shift is pressed }
        MenuCutClick(Self)           { recycle cut event }
      else
        MenuDeleteClick(Self);       { recycle delete event }
    end;
  end;
  (*----------------------------------------------------------------------------
  ** InsertClipboard - inserts the text in the clipboard to the right of the
  ** cursor.
  ----------------------------------------------------------------------------*)
  procedure InsertClipboard;
  begin
    if FCurrShift = [ssShift] then   { shift toggles insert }
      MenuPasteClick(Self)           { recycle paste }
    else if FCurrShift = [ssCtrl] then { ctrl toggles copy }
      MenuCopyClick(Self);           { recycle copy }
  end;
  (*----------------------------------------------------------------------------
  ** Undo - overwrites the text with the value of the undo buffer
  ----------------------------------------------------------------------------*)
  procedure Undo;
  begin
    if FCurrShift = [ssAlt] then     { alt+backspace = Undo }
      Text:= FOriginalText;
  end;
begin
  inherited KeyDown(Key,Shift);

  FCurrShift:= Shift;

  case Key of                        { cursor movement }
    VK_LEFT:
      LeftCursor;
    VK_RIGHT:
      RightCursor;
    VK_HOME:
      HomeCursor;
    VK_END:
      EndCursor;
    VK_DELETE:
      DeleteChar;
    VK_INSERT:
      InsertClipboard;
    VK_BACK:
      Undo;
    else
      case Char(Key) of             { map ^C, ^X, and ^V to Copy Delete and Paste }
        'C','c':
          if FCurrShift = [ssCtrl] then
            MenuCopyClick(Self);
        'X','x':
          if FCurrShift = [ssCtrl] then
            MenuCutClick(Self);
        'V','v':
          if FCurrShift = [ssCtrl] then
            MenuPasteClick(Self);
      end;
  end;
end;

(*------------------------------------------------------------------------------
** KeyPress - overrides TWinControl to retrieve character and backspace
** characters
------------------------------------------------------------------------------*)
procedure TGaugeEdit.KeyPress(var Key: Char);
begin
  inherited KeyPress(Key);

  case word(Key) of
    VK_BACK:                       { backspace deletes previous character }
    begin
      if FCursorPos > 0 then
      begin
        if FSelType = stNone then
        begin
          UpdateText(Copy(Text,1,FCursorPos-1)+Copy(Text,FCursorPos+1,Length(Text)));
          Dec(FCursorPos);

          if FCursorPos -1 < FTextOffset then
          begin
            Dec(FTextOffset,10);
            if FTextOffset < 0 then
              FTextOffset:= 0;
          end;
        end
        else
          MenuDeleteClick(Self);
      end;
    end;
    VK_ESCAPE: { must be intercepted and ignored }
      ;
    else
    begin                           { character keys, shift optional }
      if (FCurrShift = []) or (FcurrShift = [ssShift]) then
      begin
        if FSelType <> stNone then
          MenuDeleteClick(Self);

        case CharCase of
          ecNormal:    ;            { map upper/lower case keys }
          ecUpperCase: if (Key >= 'a') and (Key <= 'z') then Dec(Key, 32);
          ecLowerCase: if (Key >= 'A') and (Key <= 'Z') then Inc(Key, 32);
        end;

        UpdateText(Copy(Text,1,FCursorPos)+Key+Copy(Text,FCursorPos+1,Length(Text)));
        Inc(FCursorPos);
                                    { keep cursor in viewable area }
        if FBitmapCache.Canvas.TextWidth(Copy(Text, FTextOffset + 1, FCursorPos-FTextOffset)) + 8 > Width then
          Inc(FTextOffset,10);
      end;
    end;
  end;
end;

(*------------------------------------------------------------------------------
** MenuUndoClick (event) - replace the text with the undo buffer
------------------------------------------------------------------------------*)
procedure TGaugeEdit.MenuUndoClick(Sender: TObject);
begin
  Text:= FOriginalText;
end;

(*------------------------------------------------------------------------------
** MenuCutClick (event) - move selected text into the clipboard
------------------------------------------------------------------------------*)
procedure TGaugeEdit.MenuCutClick(Sender: TObject);
begin
  MenuCopyClick(Self);
  MenuDeleteClick(Self);
end;

(*------------------------------------------------------------------------------
** MenuCopyClick (event) - copy selected text into the clipboard
------------------------------------------------------------------------------*)
procedure TGaugeEdit.MenuCopyClick(Sender: TObject);
begin
  if FSelType <> stNone then
    Clipboard.AsText:= Copy(Text,FSelStart,FSelEnd-FSelStart+1);
end;

(*------------------------------------------------------------------------------
** MenuPasteClick (event) - insert text in the clipboard to the right of the
** cursor
------------------------------------------------------------------------------*)
procedure TGaugeEdit.MenuPasteClick(Sender: TObject);
var
  NewText: string;
begin
  if Clipboard.HasFormat(CF_TEXT) then { text in the clipboard? }
  begin
    if FSelType = stNone then          { text selected? }
    begin                              { no, insert text at cursor }
      NewText:= Copy(Text,1,FCursorPos) + Clipboard.AsText + Copy(Text,FCursorPos+1,Length(Text));
      Inc(FCursorPos,Length(Clipboard.AsText));
    end
    else
    begin                              { yes, clear selected text and insert clipboard text }
      NewText:= Copy(Text,1,FSelStart-1) + Clipboard.AsText + Copy(Text,FSelEnd+1,Length(Text));
      Inc(FCursorPos,FSelStart - FSelEnd + 2);
    end;

    FSelStart:= 0;
    FSelEnd:= 0;
    FSelType:= stNone;

    if FTextOffset > 0 then
      Inc(FTextOffset,Length(Clipboard.AsText)); { adjust cursor }

    UpdateText(NewText);
  end;
end;

(*------------------------------------------------------------------------------
** MenuDeleteClick (event) - delete selected text
------------------------------------------------------------------------------*)
procedure TGaugeEdit.MenuDeleteClick(Sender: TObject);
var
  NewText: string;
begin
  if FSelType <> stNone then { only if text is selected }
  begin
    FCursorPos:= FSelStart-1;

    NewText:= Copy(Text,1,FSelStart-1) + Copy(Text,FSelEnd+1,Length(Text));
    FSelStart:= 0;
    FSelEnd:= 0;
    FSelType:= stNone;

    UpdateText(NewText);
  end;
end;

(*------------------------------------------------------------------------------
** MenuSelectAllClick (event) - selects all text within the edit box
------------------------------------------------------------------------------*)
procedure TGaugeEdit.MenuSelectAllClick(Sender: TObject);
begin
  FSelType:= stLeft;
  FSelStart:= 1;
  FSelEnd:= Length(Text);
  FCursorPos:= Length(Text);

  if FBitmapCache.Canvas.TextWidth(Text) > Width then { move cursor to the end, scroll text if necessary }
  begin
    FTextOffset:= Length(Text);
    while (FTextOffset > 0) and (FBitmapCache.Canvas.TextWidth(Copy(Text,FTextOffset-2,Length(Text))) < Width) do
      Dec(FTextOffset);
  end;

  Invalidate;
end;

(*------------------------------------------------------------------------------
** SetBorderStyle - sets the type of border drawn around the edit box
------------------------------------------------------------------------------*)
procedure TGaugeEdit.SetBorderStyle(const Value: TBorderStyle);
begin
  FBorderStyle:= Value;
  Invalidate;
end;

(*------------------------------------------------------------------------------
** SetCharCase - sets the case of characters to be entered into thee edit box
------------------------------------------------------------------------------*)
procedure TGaugeEdit.SetCharCase(const Value: TEditCharCase);
begin
  FCharCase:= Value;
  case FCharCase of
    ecNormal:    ;
    ecUpperCase: UpdateText(UpperCase(Text));
    ecLowerCase: UpdateText(LowerCase(Text));
  end;
end;

(*------------------------------------------------------------------------------
** SetColor - sets the background color
------------------------------------------------------------------------------*)
procedure TGaugeEdit.SetColor(const Value: TColor);
begin
  FColor:= Value;
  LoadColors;
  Invalidate;
end;

(*------------------------------------------------------------------------------
** FontChange - called when any item within the font object is modified
------------------------------------------------------------------------------*)
procedure TGaugeEdit.FontChange(Sender: TObject);
begin
  Canvas.Font.Assign(Font);
  Height:= Canvas.TextHeight('Wy') + 8; { adjust component height to fit font }
  ParentFont:= False;
  Invalidate;
end;

(*------------------------------------------------------------------------------
** DoContextPopup - called by TWinControl when the user right-clicks this
** control or presses the popup button on the keyboard
------------------------------------------------------------------------------*)
procedure TGaugeEdit.DoContextPopup(MousePos: TPoint; var Handled: Boolean);
begin
{$ifdef Delphi5}
  inherited DoContextPopup(MousePos, Handled); { must call inherited }
{$endif}

  if Not Focused then
  begin
    SetFocus;
    MenuSelectAllClick(Self);           { right clicking an unfocused edit box selects all }
  end;

  if (MousePos.x = -1) and (MousePos.y = -1) then { this happens when pressing the "Windows Popup" key }
  begin                                 { popup menu in the middle }
    MousePos.x:= Width shr 1;
    MousePos.y:= Height shr 1;
  end;

  MousePos:= ClientToScreen(MousePos);  { find actual location }

  if Not Handled then                   { handled by the application? }
  begin
    if (PopupMenu <> nil) and PopupMenu.AutoPopup then
      PopupMenu.Popup(MousePos.X, MousePos.Y)  { show form specified menu }
    else
    begin
      MenuUndo.OnClick:=      MenuUndoClick; { show our menu }
      MenuCut.OnClick:=       MenuCutClick;
      MenuCopy.OnClick:=      MenuCopyClick;
      MenuPaste.OnClick:=     MenuPasteClick;
      MenuDelete.OnClick:=    MenuDeleteClick;
      MenuSelectAll.OnClick:= MenuSelectAllClick;

      MenuUndo.Enabled:= FOriginalText <> Text;
      MenuCut.Enabled:= (FSelType <> stNone) and Not FReadOnly;
      MenuCopy.Enabled:= FSelType <> stNone;
      MenuDelete.Enabled:= (FSelType <> stNone) and Not FReadOnly;
      MenuSelectAll.Enabled:= (FSelStart <> 1) and (FSelEnd <> Length(Text));
      MenuPaste.Enabled:= Clipboard.HasFormat(CF_TEXT) and Not FReadOnly;

      EditMenu.Popup(MousePos.X, MousePos.Y);
    end;
  end;
end;

{$ifndef Delphi5}
(*------------------------------------------------------------------------------
** WMContextMenu - This windows event was not handled until Delphi 5.0 was
** released
------------------------------------------------------------------------------*)
procedure TGaugeEdit.WMContextMenu(var Message: TWMContextMenu);
var
  Pt, Temp: TPoint;
  Handled: Boolean;
  PopupMenu: TPopupMenu;
begin
  if Message.Result <> 0 then Exit;
  if csDesigning in ComponentState then Exit;

  Pt := SmallPointToPoint(Message.Pos);
  if Pt.X < 0 then
    Temp := Pt
  else
  begin
    Temp := ScreenToClient(Pt);
    if not PtInRect(ClientRect, Temp) then
    begin
      inherited;
      Exit;
    end;
  end;

  Handled := False;
  DoContextPopup(Temp, Handled);
  Message.Result := Ord(Handled);
  if Handled then Exit;

  PopupMenu := GetPopupMenu;
  if (PopupMenu <> nil) and PopupMenu.AutoPopup then
  begin
    SendCancelMode(nil);
    PopupMenu.PopupComponent:= Self;
    if Pt.X < 0 then
      Pt := ClientToScreen(Point(0,0));
    PopupMenu.Popup(Pt.X, Pt.Y);
    Message.Result := 1;
  end;

  if Message.Result = 0 then
    inherited;
end;
{$endif}

(*------------------------------------------------------------------------------
** AddmenuItem - called by initialization to add a menu item to the popup menu
------------------------------------------------------------------------------*)
function AddMenuItem(Caption: string): TMenuItem;
begin
  Result:= TMenuItem.Create(EditMenu);
  Result.Caption:= Caption;

  EditMenu.Items.Add(Result);
end;

(*------------------------------------------------------------------------------
** initialization - creates the shared popup menu
------------------------------------------------------------------------------*)
initialization
  EditMenu:= TPopupMenu.Create(Nil);

  MenuUndo:= AddMenuItem('&Undo');
  AddMenuItem('-');
  MenuCut:= AddMenuItem('Cu&t');
  MenuCopy:= AddMenuItem('&Copy');
  MenuPaste:= AddMenuItem('&Paste');
  MenuDelete:= AddMenuItem('&Delete');
  AddMenuItem('-');
  MenuSelectAll:= AddMenuItem('Select &All');

(*------------------------------------------------------------------------------
** finalization - frees the shared popup menu
------------------------------------------------------------------------------*)
finalization
  EditMenu.Free;
end.
