{--------------------------------------------------------
TBkGround component version 1.0
Written by BaThanh Nguyen <26-Apr-2000>

You are free to use, distribute and edit this unit.
Just a small request: When you copy parts of this unit
into your programs, I'm very please if you put a
comment in your programs to show that the codes are
taken from my unit. Thank you!
---------------------- DOCUMENTATION --------------------
PROPERTIES:
Name              Type       Feature
Align             TAlign     Determines how the control aligns within
                             its container.
AutoSize          Boolean    Determines whether the control resizes to
                             accommodate the image it displays.
                             (This takes effect only when Picture is assigned!)
Bevel             TBevelCut  Determines the style of the border.
Center            Boolean    Indicates whether the image is centered in
                             the control.
                             (This takes effect only when Picture is assigned!)
colorTopLeft      TColor     --+
colorTopRight     TColor     --+
colorBottomLeft   TColor     --+ Colours at 4 conners
colorBottomRight  TColor     --+
Picture           TPicture   Specifies the image that appears on the control.
Stretch           Boolean    Indicates whether the image should be changed
                             so that it exactly fits the bounds of the control.
                             (This takes effect only when Picture is assigned!)
Transparent       Boolean    Specifies whether the background of the image
                             obscures objects below the image object.
                             (This takes effect only when Picture is assigned!)
--------------------------------------------------------}
unit BkGround;
{$R BkGround.res}
interface

uses
  Windows, Classes, Graphics, Controls;

type
  TBkGround = class(TGraphicControl)
  private
    { Private declarations }
    FAutoSize: Boolean;
    FBevel   : TBevelCut;
    FCenter  : Boolean;
    FcolorTL,           //Top-Left color
    FcolorTR,           //Top-Right color
    FcolorBL,           //Bottom-Left color
    FcolorBR: TColor;   //Bottom-Right color
    FPicture: TPicture;
    FStretch: Boolean;
    FTransparent: Boolean;
    function DestRect: TRect;
    procedure DrawPicture;
  protected
    { Protected declarations }
    procedure Paint; override;
    procedure PictureChanged(Sender: TObject);
    procedure SetAutoSize(Value: Boolean);
    procedure SetBevel(value: TBevelCut);
    procedure SetCenter(Value: Boolean);
    procedure SetColorBL(Value: TColor);
    procedure SetColorBR(Value: TColor);
    procedure SetColorTL(Value: TColor);
    procedure SetColorTR(Value: TColor);
    procedure SetPicture(Value: TPicture);
    procedure SetStretch(Value: boolean);
    procedure SetTransparent(Value: boolean);
  public
    { Public declarations }
    constructor Create(AOwner: TComponent); override;
  published
    { Published declarations }
    property Align;
    property AutoSize: Boolean read FAutoSize
                               write SetAutoSize default false;
    property Bevel: TBevelCut read FBevel
                              write SetBevel default bvNone;
    property Center: boolean read FCenter
                             write SetCenter default false;
    property colorTopLeft: TColor read FcolorTL
                                  write SetColorTL default clBlue;
    property colorTopRight: TColor read FcolorTR
                                  write SetColorTR default clBlue;
    property colorBottomLeft: TColor read FcolorBL
                                  write SetColorBL default clBlack;
    property colorBottomRight: TColor read FcolorBR
                                  write SetColorBR default clBlack;
    property Picture: TPicture read FPicture
                               write SetPicture;
    property Stretch: Boolean read FStretch
                              write SetStretch default false;
    property Transparent: Boolean read FTransparent
                                  write SetTransparent default false;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('BaThanh', [TBkGround]);
end;

//Private declarations
function TBkGround.DestRect: TRect;
begin
  if Stretch then
    Result := ClientRect
  else if Center then
    Result := Bounds((Width - Picture.Width) div 2, (Height - Picture.Height) div 2,
      Picture.Width, Picture.Height)
  else
    Result := Rect(0, 0, Picture.Width, Picture.Height);
end;

procedure TBkGround.DrawPicture;
{var
  bmp: TPicture;}
begin
{  bmp := TPicture.Create;
  bmp.Assign(Picture);}
  Canvas.StretchDraw(DestRect, Picture.Graphic);

{  bmp.Free}
end;

// Protected declarations
procedure TBkGround.Paint;
type
  TColorRec = record
    b,g,r: byte;
    unused: byte;
  end;

  function CreateColor(c1, c2: TColorRec; first, last, current: integer): TColorRec;
  var
    C: TColorRec;
    Percent: single;
  begin
    if (current < first) or (current > last) then
      begin Result := c1; Exit; end;
    if current = first then
      begin Result := c1; Exit; end;
    if current = last then
      begin Result := c2; Exit; end;
    Percent := (current - first)/(last - first);
    C.b := c1.b + round((c2.b-c1.b)*Percent);
    C.g := c1.g + round((c2.g-c1.g)*Percent);
    C.r := c1.r + round((c2.r-c1.r)*Percent);
    Result := C;
  end;
type
  PLongInt = ^LongInt;
var
  bmp: TBitmap;
  i,j,n: integer;
  c1,c2,c: TColorRec;
  h,w: integer;
  P: Pointer;
  L: PChar;
  t: byte;
  Color1, Color2 : TColor;
begin
  try with Canvas do
    begin
      bmp := TBitmap.Create;
      bmp.Width := Width;
      bmp.Height := Height;
      n := (Width*Height) shl 2;
      h := height - 1;
      w := width - 1;
      GetMem(P, n);
      L := P;
      for i := 0 to bmp.Height - 1 do
        begin
          c1 := CreateColor(TColorRec(FcolorTL), TColorRec(FColorBL), 0, h, i);
          c2 := CreateColor(TColorRec(FcolorTR), TColorRec(FColorBR), 0, h, i);
          for j := 0 to bmp.Width - 1 do
            begin
              C := CreateColor(c1, c2, 0, w, j);
              {Color components are stored in TColor as followed:
                 BLUE, GREEN, RED
               While they are stored in Bitmap in this order:
                 RED, GREEN, BLUE
               So we need to swap RED and BLUE components before
               transferring to Bitmap}
              t := C.b; C.b := C.r; C.r := t;
              PLongInt(L)^ := LongInt(C);
              L := L + 4;
            end;
        end; {for}
      SetBitmapBits(bmp.Handle, n, P);
      Draw(0, 0, bmp);
      FreeMem(P, n);
      bmp.Free;
    end; //with
  finally
  end; //try

  if Picture.Graphic <> nil then DrawPicture;

  if Bevel <> bvNone then
    begin
      if Bevel = bvRaised then
        begin Color1 := clBtnHighlight; Color2 := clBtnShadow; end
      else begin Color2 := clBtnHighlight; Color1 := clBtnShadow; end;
      with Canvas do
      begin
        MoveTo(0, 0);
        Pen.Color := Color1;
        LineTo(Width-1, 0);
        Pen.Color := Color2;
        LineTo(Width-1, Height-1);
        LineTo(0, Height-1);
        Pen.Color := Color1;
        LineTo(0, 0);
      end; //with
    end; //if
end; //paint

procedure TBkGround.PictureChanged(Sender: TObject);
var
  G: TGraphic;
begin
  if Picture.Graphic = nil then Exit;
  if AutoSize and (Picture.Width > 0) and (Picture.Height > 0) then
    SetBounds(Left, Top, Picture.Width, Picture.Height);
  G := Picture.Graphic;
  if G <> nil then
    if not ((G is TMetaFile) or (G is TIcon)) then
      G.Transparent := FTransparent;
  Invalidate;
end;

procedure TBkGround.SetAutoSize(Value: Boolean);
begin
  if Value <> FAutoSize then
    begin
      FAutoSize := Value;
      PictureChanged(Self);
    end;
end;

procedure TBkGround.SetBevel(Value: TBevelCut);
begin
  if FBevel <> Value then
    begin
      FBevel := Value;
      Invalidate;
    end;
end;

procedure TBkGround.SetCenter(Value: Boolean);
begin
  if Value <> FCenter then
    begin
      FCenter := Value;
      if Picture.Graphic <> nil then Invalidate;
    end;
end;

procedure TBkGround.SetColorBL(Value: TColor);
begin
  if FcolorBL <> Value then
    begin
      FcolorBL := Value;
      Invalidate;
    end;
end;

procedure TBkGround.SetColorBR(Value: TColor);
begin
  if FcolorBR <> Value then
    begin
      FcolorBR := Value;
      Invalidate;
    end;
end;

procedure TBkGround.SetColorTL(Value: TColor);
begin
  if FcolorTL <> Value then
    begin
      FcolorTL := Value;
      Invalidate;
    end;
end;

procedure TBkGround.SetColorTR(Value: TColor);
begin
  if FcolorTR <> Value then
    begin
      FcolorTR := Value;
      Invalidate;
    end;
end;

procedure TBkGround.SetPicture(Value: TPicture);
begin
  FPicture.Assign(Value);
end;

procedure TBkGround.SetStretch(Value: Boolean);
begin
  if Value <> FStretch then
    begin
      FStretch := Value;
      if Picture.Graphic <> nil then Invalidate;
    end;
end;

procedure TBkGround.SetTransparent(Value: boolean);
begin
  if Value <> FTransparent then
    begin
      FTransparent := value;
      PictureChanged(Self);
    end;
end;

// Public declarations
constructor TBkGround.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FAutoSize := false;
  FBevel := bvNone;
  FCenter := false;
  FcolorTL := clBlue; FcolorTR := clBlue;
  FcolorBL := clBlack; FcolorBR := clBlack;
  FPicture := TPicture.Create;
  FPIcture.OnChange := PictureChanged;
  FStretch := false;
  FTransparent := false;
end;

end.
