UNIT UPie;

INTERFACE

USES
  Windows, SysUtils, ExtCtrls, Graphics, Math, Registry, Forms, Globals;

TYPE
  TPieParas = RECORD
    ppBrushColor,
    ppBorderColor 	: TColor;
    ppBorderWidth   	: INTEGER;
    ppdStartAngle,		// bezogen auf 2 x PI
    ppdStopAngle,
    ppdRadiusX,
    ppdRadiusY,
    ppdOriginX,			// wg. MappingModes und Bitmap
    ppdOriginY,
    ppdMulX,			// wg. MappingModes und Bitmap
    ppdMulY		: DOUBLE;
    END;

  TPieRect = RECORD
    X1,
    Y1,
    X2,
    Y2 			: LONGINT;
    END;

TYPE
  TPie = CLASS( TObject )
   PRIVATE
    { Private declarations }
    myPieParas		: TPieParas;
    myBoundingRect	: TPieRect;
    myPieRect		: TPieRect;
    PROCEDURE SetStartAngle( dValue: DOUBLE );
    PROCEDURE SetStopAngle( dValue: DOUBLE );
    PROCEDURE SetRadiusX( dValue: DOUBLE );
    PROCEDURE SetRadiusY( dValue: DOUBLE );
    PROCEDURE SetOriginX( dValue: DOUBLE );
    PROCEDURE SetOriginY( dValue: DOUBLE );
    FUNCTION  ReCalculate: BOOLEAN;
   PROTECTED
    { Protected declarations }
   PUBLIC
    { Public declarations }
   PUBLISHED
    { Published declarations }
    CONSTRUCTOR Create;
    CONSTRUCTOR CreateWithParas( aTPieParas: TPieParas );			// wg. Copy
    DESTRUCTOR Destroy; OVERRIDE;
    PROPERTY BrushColor: TColor READ myPieParas.ppBrushColor WRITE myPieParas.ppBrushColor;
    PROPERTY BorderColor: TColor READ myPieParas.ppBorderColor WRITE myPieParas.ppBorderColor;
    PROPERTY BorderWidth: INTEGER READ myPieParas.ppBorderWidth WRITE myPieParas.ppBorderWidth;
    PROPERTY StartAngle: DOUBLE READ myPieParas.ppdStartAngle WRITE SetStartAngle;
    PROPERTY StopAngle: DOUBLE READ myPieParas.ppdStopAngle WRITE SetStopAngle;
    PROPERTY RadiusX: DOUBLE READ myPieParas.ppdRadiusX WRITE SetRadiusX;
    PROPERTY RadiusY: DOUBLE READ myPieParas.ppdRadiusY WRITE SetRadiusY;
    PROPERTY OriginX: DOUBLE READ myPieParas.ppdOriginX WRITE SetOriginX;
    PROPERTY OriginY: DOUBLE READ myPieParas.ppdOriginY WRITE SetOriginY;

    PROPERTY PieParas: TPieParas READ myPieParas; 				// read only
    PROPERTY BoundingRect: TPieRect READ myBoundingRect;
    PROPERTY PieRect: TPieRect READ myPieRect;
    END;


  TState	= (stON, stOff);
  TExtra	= RECORD
    xSecondsDec	: DWORD;
    xSecondsRef	: DWORD;
    xOnColor	: TColor;
    xDimColor	: TColor;
    xOffColor	: TColor;
    xBrdColor	: TColor;
    xState	: TState;
    xDimmed	: BOOLEAN;
    END;

  TBerlinPie	= CLASS(TPie)
   PRIVATE
    myExtra	: TExtra;
    PROCEDURE SetState( aState: TState );
    FUNCTION  GetDimmedColor( aColor: TColor ): TColor;
    { Private declarations }
   PROTECTED
    { Protected declarations }
   PUBLIC
    { Public declarations }
   PUBLISHED
    { Published declarations }
    CONSTRUCTOR Create;
    CONSTRUCTOR CreateWithParas( aTPieParas: TPieParas; aExtra: TExtra );	// wg. Copy
    FUNCTION  GetStateChanged( VAR aValue: DWORD ): BOOLEAN;
    PROCEDURE SetDimmed( aValue: BOOLEAN );

    PROPERTY SecondsDecrement: DWORD READ myExtra.xSecondsDec WRITE myExtra.xSecondsDec;
    PROPERTY SecondsReference: DWORD READ myExtra.xSecondsRef WRITE myExtra.xSecondsRef;
    PROPERTY State: TState READ myExtra.xState WRITE SetState;
    PROPERTY ColorOn: TColor READ myExtra.xOnColor WRITE myExtra.xOnColor;
    PROPERTY ColorDim: TColor READ myExtra.xDimColor WRITE myExtra.xDimColor;
    PROPERTY ColorOff: TColor READ myExtra.xOffColor WRITE myExtra.xOffColor;
    PROPERTY ColorBorder: TColor READ myExtra.xBrdColor WRITE myExtra.xBrdColor;
    PROPERTY Dimmed: BOOLEAN READ myExtra.xDimmed WRITE SetDimmed;
    END;


  TBerlinUhrColors = RECORD
    bucBorderColor	: TColor;
    bucHColorOn  	: TColor;
    bucHColorDim 	: TColor;
    bucHColorOff 	: TColor;
    bucMColorOn  	: TColor;
    bucMColorDim 	: TColor;
    bucMColorOff 	: TColor;
    bucSColorOn  	: TColor;
    bucSColorDim 	: TColor;
    bucSColorOff 	: TColor;
    END;

  TBerlinUhr	= CLASS( TObject )
   PRIVATE
    { Private declarations }
    myColors	: TBerlinUhrColors;
    myBitmap	: TBitmap;
    bUpdateAll	: BOOLEAN;
    bIsPainting	: BOOLEAN;
    PROCEDURE SetColors( aColors: TBerlinUhrColors );
   PROTECTED
    allPies	: ARRAY[1 .. 26] OF TBerlinPie;
    { Protected declarations }
   PUBLIC
    { Public declarations }
   PUBLISHED
    { Published declarations }
    CONSTRUCTOR Create( aColors: TBerlinUhrColors; aWidth, aHeight, aBorderWidth: INTEGER );
    DESTRUCTOR	Destroy; OVERRIDE;
    FUNCTION	GetBitmap( aDateTime: TDateTime ): TBitmap;
    PROCEDURE	DimmAll( bDimmed: BOOLEAN );
    PROCEDURE	Refresh;
    PROPERTY	Colors: TBerlinUhrColors READ myColors WRITE SetColors;
    PROPERTY	IsProcessing: BOOLEAN READ bIsPainting;
    END;

  TBerlinUhrRegEntries	= CLASS( TObject )
   PRIVATE
    myRegistry	: TRegistry;
    myDefColors	: TBerlinUhrColors;
    strKey	: STRING;
    { Private declarations }
    FUNCTION	GetBerlinUhrColors: TBerlinUhrColors;
    PROCEDURE	SetBerlinUhrColors( aColors: TBerlinUhrColors );
   PROTECTED
    { Protected declarations }
   PUBLIC
    { Public declarations }
   PUBLISHED
    { Published declarations }
    CONSTRUCTOR Create;
    DESTRUCTOR	Destroy; OVERRIDE;
    PROPERTY	BerlinUhrColors: TBerlinUhrColors READ GetBerlinUhrColors WRITE SetBerlinUhrColors;
    END;


IMPLEMENTATION

(*---------------------------------*)
CONSTRUCTOR TPie.Create;

BEGIN
INHERITED Create;
WITH myPieParas DO	// Standardwerte
  BEGIN
  ppBrushColor		:=  clYellow;
  ppBorderColor		:=  clLime;
  ppBorderWidth		:=  1;
  ppdStartAngle		:=  0.0;
  ppdStopAngle		:=  PI;
  ppdRadiusX		:=  100.0;
  ppdRadiusY		:=  50.0;
  ppdOriginX		:=  0.0;
  ppdOriginY		:=  0.0;
  ppdMulX   		:=  1.0;
  ppdMulY   		:= -1.0
  END;

ReCalculate;
END;

(*---------------------------------*)
CONSTRUCTOR TPie.CreateWithParas( aTPieParas: TPieParas );

BEGIN
INHERITED Create;
myPieParas := aTPieParas;
ReCalculate;
END;

(*---------------------------------*)
DESTRUCTOR TPie.Destroy;

BEGIN
INHERITED Destroy;
END;

(*---------------------------------*)
FUNCTION TPie.ReCalculate: BOOLEAN;

VAR
  eSin,
  eCos : EXTENDED;

BEGIN
WITH myBoundingRect, myPieParas DO
  BEGIN
  X1		:= Round( ppdOriginX-ppdRadiusX*ppdMulX );
  Y1		:= Round( ppdOriginY+ppdRadiusY*ppdMulY );
  X2		:= Round( ppdOriginX+ppdRadiusX*ppdMulX );
  Y2		:= Round( ppdOriginY-ppdRadiusY*ppdMulY );
  END;

WITH myPieRect, myPieParas DO
  BEGIN
  SinCos( ppdStartAngle, eSin, eCos );
  X1		:= Round( ppdOriginX+ppdMulX*(ppdRadiusX*eCos) );
  Y1		:= Round( ppdOriginY+ppdMulY*(ppdRadiusY*eSin) );
  SinCos( ppdStopAngle, eSin, eCos );
  X2		:= Round( ppdOriginX+ppdMulX*(ppdRadiusX*eCos) );
  Y2		:= Round( ppdOriginY+ppdMulY*(ppdRadiusY*eSin) );
  END;

Result := true;
END;

(*---------------------------------*)
PROCEDURE TPie.SetStartAngle( dValue: DOUBLE );

BEGIN
myPieParas.ppdStartAngle := dValue;
ReCalculate;
END;

(*---------------------------------*)
PROCEDURE TPie.SetStopAngle( dValue: DOUBLE );

BEGIN
myPieParas.ppdStopAngle := dValue;
ReCalculate;
END;

(*---------------------------------*)
PROCEDURE TPie.SetRadiusX( dValue: DOUBLE );

BEGIN
myPieParas.ppdRadiusX := dValue;
ReCalculate;
END;

(*---------------------------------*)
PROCEDURE TPie.SetRadiusY( dValue: DOUBLE );

BEGIN
myPieParas.ppdRadiusY := dValue;
ReCalculate;
END;

(*---------------------------------*)
PROCEDURE TPie.SetOriginX( dValue: DOUBLE );

BEGIN
myPieParas.ppdOriginX := dValue;
ReCalculate;
END;

(*---------------------------------*)
PROCEDURE TPie.SetOriginY( dValue: DOUBLE );

BEGIN
myPieParas.ppdOriginY := dValue;
ReCalculate;
END;

(*=================================*)
CONSTRUCTOR TBerlinPie.Create;

BEGIN
INHERITED Create;
WITH myExtra DO
  BEGIN
  xSecondsDec	:= 43200;		// 12Std
  xSecondsRef	:= 43200;               // 12Std
  xOnColor	:= clLime;
  xDimColor	:= clBlue;
  xOffColor	:= clMaroon;
  xState  	:= stOn;
  END;
State := State; //!!
END;

(*---------------------------------*)
CONSTRUCTOR TBerlinPie.CreateWithParas( aTPieParas: TPieParas; aExtra: TExtra );

BEGIN
INHERITED CreateWithParas( aTPieParas );
myExtra := aExtra;
END;


(*---------------------------------*)
FUNCTION  TBerlinPie.GetDimmedColor( aColor: TColor ): TColor;

BEGIN
Result := (aColor AND $FF000000)
	 +(GetBValue(aColor) SHR 1) SHL 16+
	 +(GetGValue(aColor) SHR 1) SHL  8+
	 +(GetRValue(aColor) SHR 1);
END;

(*---------------------------------*)
PROCEDURE TBerlinPie.SetState( aState: TState );

BEGIN
myExtra.xState := aState;
CASE myExtra.xState OF
  stOn	: IF myExtra.xDimmed THEN
	    myPieParas.ppBrushColor := ColorDim
	   ELSE
	    myPieParas.ppBrushColor := ColorOn;
  stOff	: IF myExtra.xDimmed THEN
	    myPieParas.ppBrushColor := GetDimmedColor(ColorOff)
	   ELSE
	    myPieParas.ppBrushColor := ColorOff;
  END;
IF myExtra.xDimmed THEN
  myPieParas.ppBorderColor := GetDimmedColor(ColorBorder)
 ELSE
  myPieParas.ppBorderColor := ColorBorder;
END;

(*---------------------------------*)
PROCEDURE TBerlinPie.SetDimmed( aValue: BOOLEAN );

BEGIN
myExtra.xDimmed := aValue;
SetState( myExtra.xState );
END;

(*---------------------------------*)
FUNCTION  TBerlinPie.GetStateChanged( VAR aValue: DWORD ): BOOLEAN;

BEGIN
IF aValue >= myExtra.xSecondsRef THEN
  BEGIN
  Dec( aValue, myExtra.xSecondsDec );
  IF myExtra.xState = stOff THEN
    BEGIN
    Result := true;
    SetState( stOn );
    END
   ELSE
    BEGIN
    Result := false;
    END;
  END
 ELSE
  BEGIN
  IF myExtra.xState = stOn THEN
    BEGIN
    Result := true;
    SetState( stOff );
    END
   ELSE
    BEGIN
    Result := false;
    END;
  END;
END;

(*=================================*)
CONSTRUCTOR TBerlinUhr.Create( aColors: TBerlinUhrColors; aWidth, aHeight, aBorderWidth: INTEGER );

VAR
  tmpExtra	: TExtra;
  tmpPieParas	: TPieParas;


BEGIN
bIsPainting	:= false;
bUpdateAll	:= true;
FillChar(tmpPieParas, SizeOf(tmpPieParas), 0);		//!!
FillChar(tmpExtra, SizeOf(tmpExtra), 0);		//!!
myBitmap       	:= TBitmap.Create;
myBitmap.Width 	:= aWidth;
myBitmap.Height	:= aHeight;

// Stunden ----------------
WITH tmpPieParas DO
  BEGIN
  ppBorderWidth		:= aBorderWidth;
  ppdStartAngle		:= 0.0;
  ppdStopAngle		:= Pi;
  ppdRadiusX		:= 0.75*Int(myBitmap.Width);
  ppdRadiusY		:= 0.75*Int(myBitmap.Height);
  ppdOriginX		:= 0.5*Int(myBitmap.Width);
  ppdOriginY		:= 0.5*Int(myBitmap.Height);
  ppdMulX  		:=  1.0;
  ppdMulY  		:= -1.0
  END;
WITH tmpExtra DO
  BEGIN
  xSecondsDec		:= 43200;			// 12Std
  xSecondsRef		:= 43200;
  xDimmed		:= false;
  END;
allPies[1]		:= TBerlinPie.CreateWithParas( tmpPieParas, tmpExtra );

WITH tmpPieParas DO
  BEGIN
  ppdStartAngle		:= 0.0;
  ppdStopAngle		:= Pi;
  ppdRadiusX		:= 0.45*Int(myBitmap.Width);
  ppdRadiusY		:= 0.45*Int(myBitmap.Height);
  END;
WITH tmpExtra DO
  BEGIN
  xSecondsDec		:= 21600;			// 6Std
  xSecondsRef		:= 21600;
  END;
allPies[2]		:= TBerlinPie.CreateWithParas( tmpPieParas, tmpExtra );

WITH tmpPieParas DO
  BEGIN
  ppdStartAngle		:= 2.0/5.0*Pi;
  ppdStopAngle		:= 3.0/5.0*Pi;
  ppdRadiusX		:= 0.35*Int(myBitmap.Width);
  ppdRadiusY		:= 0.35*Int(myBitmap.Height);
  END;
WITH tmpExtra DO
  BEGIN
  xSecondsDec		:=  3600;			// 1Std
  xSecondsRef		:= 18000;
  END;
allPies[3]		:= TBerlinPie.CreateWithParas( tmpPieParas, tmpExtra );

WITH tmpPieParas DO
  BEGIN
  ppdStartAngle		:= 1.0/5.0*Pi;
  ppdStopAngle		:= 2.0/5.0*Pi;
  END;
WITH tmpExtra DO
  BEGIN
  xSecondsRef		:= 14400;
  END;
allPies[4]		:= TBerlinPie.CreateWithParas( tmpPieParas, tmpExtra );

WITH tmpPieParas DO
  BEGIN
  ppdStartAngle		:= 3.0/5.0*Pi;
  ppdStopAngle		:= 4.0/5.0*Pi;
  END;
WITH tmpExtra DO
  BEGIN
  xSecondsRef		:= 10800;
  END;
allPies[5]		:= TBerlinPie.CreateWithParas( tmpPieParas, tmpExtra );

WITH tmpPieParas DO
  BEGIN
  ppdStartAngle		:= 0.0/5.0*Pi;
  ppdStopAngle		:= 1.0/5.0*Pi;
  END;
WITH tmpExtra DO
  BEGIN
  xSecondsRef		:= 7200;
  END;
allPies[6]		:= TBerlinPie.CreateWithParas( tmpPieParas, tmpExtra );

WITH tmpPieParas DO
  BEGIN
  ppdStartAngle		:= 4.0/5.0*Pi;
  ppdStopAngle		:= 5.0/5.0*Pi;
  END;
WITH tmpExtra DO
  BEGIN
  xSecondsRef		:= 3600;
  END;
allPies[7]		:= TBerlinPie.CreateWithParas( tmpPieParas, tmpExtra );

// Minuten ----------------
WITH tmpPieParas DO
  BEGIN
  ppdStartAngle		:= 1.0/3.0*Pi;
  ppdStopAngle		:= 2.0/3.0*Pi;
  ppdRadiusX		:= 0.2*Int(myBitmap.Width);
  ppdRadiusY		:= 0.2*Int(myBitmap.Height);
  END;
WITH tmpExtra DO
  BEGIN
  xSecondsDec		:= 900;				// 15Min
  xSecondsRef		:= 2700;
  xDimmed		:= false;
  END;
allPies[8]		:= TBerlinPie.CreateWithParas( tmpPieParas, tmpExtra );

WITH tmpPieParas DO
  BEGIN
  ppdStartAngle		:= 0.0/3.0*Pi;
  ppdStopAngle		:= 1.0/3.0*Pi;
  END;
WITH tmpExtra DO
  BEGIN
  xSecondsRef		:= 1800;
  END;
allPies[9]		:= TBerlinPie.CreateWithParas( tmpPieParas, tmpExtra );

WITH tmpPieParas DO
  BEGIN
  ppdStartAngle		:= 2.0/3.0*Pi;
  ppdStopAngle		:= 3.0/3.0*Pi;
  END;
WITH tmpExtra DO
  BEGIN
  xSecondsRef		:= 900;
  END;
allPies[10]		:= TBerlinPie.CreateWithParas( tmpPieParas, tmpExtra );

WITH tmpPieParas DO
  BEGIN
  ppdStartAngle		:= 3.0/2.0*Pi;
  ppdStopAngle		:= 4.0/2.0*Pi;
  ppdRadiusX		:= 0.75*Int(myBitmap.Width);
  ppdRadiusY		:= 0.75*Int(myBitmap.Height);
  END;
WITH tmpExtra DO
  BEGIN
  xSecondsDec		:= 300;				// 5Min
  xSecondsRef		:= 600;
  END;
allPies[11]		:= TBerlinPie.CreateWithParas( tmpPieParas, tmpExtra );

WITH tmpPieParas DO
  BEGIN
  ppdStartAngle		:= 2.0/2.0*Pi;
  ppdStopAngle		:= 3.0/2.0*Pi;
  END;
WITH tmpExtra DO
  BEGIN
  xSecondsRef		:= 300;
  END;
allPies[12]		:= TBerlinPie.CreateWithParas( tmpPieParas, tmpExtra );

WITH tmpPieParas DO
  BEGIN
  ppdStartAngle		:= 6.0/4.0*Pi;
  ppdStopAngle		:= 7.0/4.0*Pi;
  ppdRadiusX		:= 0.45*Int(myBitmap.Width);
  ppdRadiusY		:= 0.45*Int(myBitmap.Height);
  END;
WITH tmpExtra DO
  BEGIN
  xSecondsDec		:= 60;				// 1Min
  xSecondsRef		:= 240;
  END;
allPies[13]		:= TBerlinPie.CreateWithParas( tmpPieParas, tmpExtra );

WITH tmpPieParas DO
  BEGIN
  ppdStartAngle		:= 5.0/4.0*Pi;
  ppdStopAngle		:= 6.0/4.0*Pi;
  END;
WITH tmpExtra DO
  BEGIN
  xSecondsRef		:= 180;
  END;
allPies[14]		:= TBerlinPie.CreateWithParas( tmpPieParas, tmpExtra );

WITH tmpPieParas DO
  BEGIN
  ppdStartAngle		:= 7.0/4.0*Pi;
  ppdStopAngle		:= 8.0/4.0*Pi;
  END;
WITH tmpExtra DO
  BEGIN
  xSecondsRef		:= 120;
  END;
allPies[15]		:= TBerlinPie.CreateWithParas( tmpPieParas, tmpExtra );

WITH tmpPieParas DO
  BEGIN
  ppdStartAngle		:= 4.0/4.0*Pi;
  ppdStopAngle		:= 5.0/4.0*Pi;
  END;
WITH tmpExtra DO
  BEGIN
  xSecondsRef		:= 60;
  END;
allPies[16]		:= TBerlinPie.CreateWithParas( tmpPieParas, tmpExtra );

// Sekunden ----------------
WITH tmpPieParas DO
  BEGIN
  ppdStartAngle		:= 7.0/5.0*Pi;
  ppdStopAngle		:= 8.0/5.0*Pi;
  ppdRadiusX		:= 0.35*Int(myBitmap.Width);
  ppdRadiusY		:= 0.35*Int(myBitmap.Height);
  END;
WITH tmpExtra DO
  BEGIN
  xSecondsDec		:= 10;					// 10Sec
  xSecondsRef		:= 50;
  xDimmed		:= false;
  END;
allPies[17]		:= TBerlinPie.CreateWithParas( tmpPieParas, tmpExtra );

WITH tmpPieParas DO
  BEGIN
  ppdStartAngle		:= 8.0/5.0*Pi;
  ppdStopAngle		:= 9.0/5.0*Pi;
  END;
WITH tmpExtra DO
  BEGIN
  xSecondsRef		:= 40;
  END;
allPies[18]		:= TBerlinPie.CreateWithParas( tmpPieParas, tmpExtra );

WITH tmpPieParas DO
  BEGIN
  ppdStartAngle		:= 6.0/5.0*Pi;
  ppdStopAngle		:= 7.0/5.0*Pi;
  END;
WITH tmpExtra DO
  BEGIN
  xSecondsRef		:= 30;
  END;
allPies[19]		:= TBerlinPie.CreateWithParas( tmpPieParas, tmpExtra );

WITH tmpPieParas DO
  BEGIN
  ppdStartAngle		:= 9.0/5.0*Pi;
  ppdStopAngle		:= 10.0/5.0*Pi;
  END;
WITH tmpExtra DO
  BEGIN
  xSecondsRef		:= 20;
  END;
allPies[20]		:= TBerlinPie.CreateWithParas( tmpPieParas, tmpExtra );

WITH tmpPieParas DO
  BEGIN
  ppdStartAngle		:= 5.0/5.0*Pi;
  ppdStopAngle		:= 6.0/5.0*Pi;
  END;
WITH tmpExtra DO
  BEGIN
  xSecondsRef		:= 10;
  END;
allPies[21]		:= TBerlinPie.CreateWithParas( tmpPieParas, tmpExtra );

WITH tmpPieParas DO
  BEGIN
  ppdStartAngle		:= 5.001/5.0*Pi;
  ppdStopAngle		:= 10.0/5.0*Pi;
  ppdRadiusX		:= 0.275*Int(myBitmap.Width);
  ppdRadiusY		:= 0.275*Int(myBitmap.Height);
  END;
WITH tmpExtra DO
  BEGIN
  xSecondsDec		:= 5;					// 5Sec
  xSecondsRef		:= 5;
  END;
allPies[22]		:= TBerlinPie.CreateWithParas( tmpPieParas, tmpExtra );

WITH tmpPieParas DO
  BEGIN
  ppdStartAngle		:= 5.995/4.0*Pi;
  ppdStopAngle		:= 7.0/4.0*Pi;
  ppdRadiusX		:= 0.2*Int(myBitmap.Width);
  ppdRadiusY		:= 0.2*Int(myBitmap.Height);
  END;
WITH tmpExtra DO
  BEGIN
  xSecondsDec		:= 1;					// 1Sec
  xSecondsRef		:= 4;
  END;
allPies[23]		:= TBerlinPie.CreateWithParas( tmpPieParas, tmpExtra );

WITH tmpPieParas DO
  BEGIN
  ppdStartAngle		:= 5.0/4.0*Pi;
  ppdStopAngle		:= 5.995/4.0*Pi;
  END;
WITH tmpExtra DO
  BEGIN
  xSecondsDec		:= 1;					// 1Sec
  xSecondsRef		:= 3;
  END;
allPies[24]		:= TBerlinPie.CreateWithParas( tmpPieParas, tmpExtra );

WITH tmpPieParas DO
  BEGIN
  ppdStartAngle		:= 7.0/4.0*Pi;
  ppdStopAngle		:= 8.0/4.0*Pi;
  END;
WITH tmpExtra DO
  BEGIN
  xSecondsDec		:= 1;					// 1Sec
  xSecondsRef		:= 2;
  END;
allPies[25]		:= TBerlinPie.CreateWithParas( tmpPieParas, tmpExtra );

WITH tmpPieParas DO
  BEGIN
  ppdStartAngle		:= 4.0/4.0*Pi;
  ppdStopAngle		:= 5.0/4.0*Pi;
  END;
WITH tmpExtra DO
  BEGIN
  xSecondsDec		:= 1;					// 1Sec
  xSecondsRef		:= 1;
  END;
allPies[26]		:= TBerlinPie.CreateWithParas( tmpPieParas, tmpExtra );

SetColors( aColors );
END;

(*---------------------------------*)
DESTRUCTOR TBerlinUhr.Destroy;

VAR
  iCount : INTEGER;

BEGIN
WHILE bIsPainting DO
  BEGIN
  AppendText(' was painting' );
  Sleep(100);
  Application.ProcessMessages;
  END;

myBitmap.Free;
FOR iCount := Low(allPies) TO High(allPies) DO
  allPies[iCount].Free;
END;

(*---------------------------------*)
FUNCTION TBerlinUhr.GetBitmap( aDateTime: TDateTime ): TBitmap;

VAR
  wHr, wM, wS, wD	: WORD;
  dwSeconds		: DWORD;
  iCount		: INTEGER;

BEGIN
IF NOT( bIsPainting ) THEN
  BEGIN
  bIsPainting	:= true;
  DecodeTime( aDateTime, wHr, wM, wS, wD );
  dwSeconds	:= wHr*3600+wM*60+wS;
  FOR iCount := Low(allPies) TO High(allPies) DO
    BEGIN
    IF Assigned(allPies[iCount]) THEN
      BEGIN
      WITH allPies[iCount] DO
	BEGIN
	IF GetStateChanged( dwSeconds ) OR bUpdateAll THEN
	  BEGIN
	  myBitmap.Canvas.Brush.Color	:= BrushColor;
	  myBitmap.Canvas.Pen.Color 	:= BorderColor;
	  myBitmap.Canvas.Pen.Width	:= BorderWidth;
	  myBitmap.Canvas.Pie( 	BoundingRect.X1, BoundingRect.Y1, BoundingRect.X2,BoundingRect.Y2,
				PieRect.X1, PieRect.Y1, PieRect.X2, PieRect.Y2
			     );
	  END; {IF GetStateChanged}
	END; {WITH allPies}
      END; {IF Assigned}
    END; {FOR iCount}
  bUpdateAll	:= false;
  bIsPainting	:= false;
  END; {IF NOT( bIsPainting )}

Result 		:= myBitmap;
END;

(*---------------------------------*)
PROCEDURE TBerlinUhr.DimmAll( bDimmed: BOOLEAN );

VAR
  iCount		: INTEGER;

BEGIN
FOR iCount := Low(allPies) TO High(allPies) DO
  IF Assigned(allPies[iCount]) THEN
    allPies[iCount].Dimmed := bDimmed;

Refresh;
END;

(*---------------------------------*)
PROCEDURE TBerlinUhr.Refresh;

BEGIN
bUpdateAll	:= true;
END;

(*---------------------------------*)
PROCEDURE TBerlinUhr.SetColors( aColors: TBerlinUhrColors );

VAR
  iCount		: INTEGER;

BEGIN
myColors := aColors;
FOR iCount := 1 TO 7 DO				//Stunden
  BEGIN
  IF Assigned(allPies[iCount]) THEN
    BEGIN
    WITH allPies[iCount] DO
      BEGIN
      ColorOn		:= aColors.bucHColorOn;
      ColorDim		:= aColors.bucHColorDim;
      ColorOff		:= aColors.bucHColorOff;
      ColorBorder	:= aColors.bucBorderColor;
      State		:= State;     		//!!!! RaRuRick Babatrick !!!
      END;
    END;
  END;

FOR iCount := 8 TO 16 DO			//Minuten
  BEGIN
  IF Assigned(allPies[iCount]) THEN
    BEGIN
    WITH allPies[iCount] DO
      BEGIN
      ColorOn		:= aColors.bucMColorOn;
      ColorDim		:= aColors.bucMColorDim;
      ColorOff		:= aColors.bucMColorOff;
      ColorBorder	:= aColors.bucBorderColor;
      State		:= State;     		//!!!! RaRuRick Babatrick !!!
      END;
    END;
  END;

FOR iCount := 17 TO 26 DO			//Sekunden
  BEGIN
  IF Assigned(allPies[iCount]) THEN
    BEGIN
    WITH allPies[iCount] DO
      BEGIN
      ColorOn		:= aColors.bucSColorOn;
      ColorDim		:= aColors.bucSColorDim;
      ColorOff		:= aColors.bucSColorOff;
      ColorBorder	:= aColors.bucBorderColor;
      State		:= State;     		//!!!! RaRuRick Babatrick !!!
      END;
    END;
  END;

Refresh;
END;

(*=================================*)
CONSTRUCTOR TBerlinUhrRegEntries.Create;

BEGIN
WITH myDefColors DO
  BEGIN
  bucBorderColor	:= clSilver;
  bucHColorOn		:= $000000FF;
  bucHColorDim		:= $0000007F;
  bucHColorOff		:= clGray;
  bucMColorOn		:= $0000FF00;
  bucMColorDim 		:= $00007F00;
  bucMColorOff		:= clGray;
  bucSColorOn		:= $00FF0000;
  bucSColorDim		:= $007F0000;;
  bucSColorOff		:= clGray;
  END;

strKey	:= '\Software\UweC\'+Application.Title;
myRegistry := TRegistry.Create;
WITH myRegistry DO
  BEGIN
  RootKey := HKEY_CURRENT_USER;
  IF NOT(OpenKey(strKey, false)) THEN
    BEGIN
    SetBerlinUhrColors( myDefColors );
    CloseKey;
    END;
  END;
END;

(*---------------------------------*)
DESTRUCTOR  TBerlinUhrRegEntries.Destroy;

BEGIN
myRegistry.Free;
END;

(*---------------------------------*)
FUNCTION TBerlinUhrRegEntries.GetBerlinUhrColors: TBerlinUhrColors;

BEGIN
WITH myRegistry DO
  BEGIN
  IF OpenKey( strKey, false ) THEN
    BEGIN
    TRY
      WITH Result DO
	BEGIN
	bucBorderColor	:= ReadInteger( 'BorderColor' );
	bucHColorOn   	:= ReadInteger( 'hColorOn' );
	bucHColorDim  	:= ReadInteger( 'hColorDim' );
	bucHColorOff  	:= ReadInteger( 'hColorOff' );
	bucMColorOn   	:= ReadInteger( 'mColorOn' );
	bucMColorDim  	:= ReadInteger( 'mColorDim' );
	bucMColorOff  	:= ReadInteger( 'mColorOff' );
	bucSColorOn   	:= ReadInteger( 'sColorOn' );
	bucSColorDim  	:= ReadInteger( 'sColorDim' );
	bucSColorOff  	:= ReadInteger( 'sColorOff' );
	END;
     FINALLY
      CloseKey;
      END;
    END
   ELSE
    BEGIN
    Result := myDefColors;
    END; {IF}
  END; {WITH myRegistry}
END;

(*---------------------------------*)
PROCEDURE TBerlinUhrRegEntries.SetBerlinUhrColors( aColors: TBerlinUhrColors );

BEGIN
WITH myRegistry DO
  BEGIN
  IF OpenKey( strKey, true ) THEN
    BEGIN
    TRY
      WITH aColors DO
	BEGIN
	WriteInteger( 'BorderColor', 	bucBorderColor );
	WriteInteger( 'hColorOn',   	bucHColorOn );
	WriteInteger( 'hColorDim',  	bucHColorDim );
	WriteInteger( 'hColorOff',  	bucHColorOff );
	WriteInteger( 'mColorOn',   	bucMColorOn );
	WriteInteger( 'mColorDim',  	bucMColorDim );
	WriteInteger( 'mColorOff',  	bucMColorOff );
	WriteInteger( 'sColorOn',   	bucSColorOn );
	WriteInteger( 'sColorDim',  	bucSColorDim );
	WriteInteger( 'sColorOff',  	bucSColorOff );
	END;
     FINALLY
      CloseKey;
      END;
    END;
  END;
END;

(*+++++++++++++++++++++++++++++++++*)
END.
