unit AppUpdateTst;

{*******************************************************************************
          AppUpdateTst - Main Form for the TCynthesisAppUpdate Sample

                    (c) Copyright 1999, 2000 Cynthesis Software Inc.

  Author:

    James Waletzky (cynthesis@home.com)
                   (http://www.npsnet.com/waletzky/)

  Description:

    Main form for the AppUpdateTest sample application that demonstrates the
    use of the TCynthesisAppUpdate component.

  Limitations of Use:

    The TCynthesisAppUpdate component (and its various pieces) is free (no
    strings attached) for NON-COMMERICAL use. For use in a commercial
    application, please contact James Waletzky via e-mail at cynthesis@home.com
    for information about licensing fees.

    The source code for this component (for NON-COMMERCIAL use) is available
    from Cynthesis Software for a cost of $10. Please remit a cheque or money
    order payable to Cynthesis Software Inc. to the following address:

             Cynthesis Software Inc.
             408 - 260 Newport Drive
             Port Moody, BC
             Canada
             V3H 5C6

    This source code may NOT be distributed to other sources, and is for the
    exclusive use of the licensee. The source code may be used in any
    application developed by the licensee on a royalty-free basis, as long as
    that application is distributed as freeware. For use in a commercial
    application, see the terms of the licensing agreement.

  Request:

    Please send your comments about the TCynthesisAppUpdate component, or
    suggestions for improvement, to cynthesis@home.com. The comoponent can only
    get better with your help!

*******************************************************************************}

interface

uses
  Classes, Forms, Dialogs, StdCtrls, ComCtrls, Controls, ExtCtrls,
  CynthesisAppUpdate, CynthesisAppUpdateVersionInfo, CynthesisAppUpdateTypes;

type
  { simple state variables for the form object. The various states are
    as follows:
         ausIdle -            - application is idle waiting for user request
         ausCheckingForUpdate - in the middle of checking for an update info
                                file
         ausUpdateFound       - an update is available - the app is now waiting
                                for the user to initiate the download
         ausDownloadingUpdate - in the middle of downloading the actual update
         ausDownloadComplete  - the update has been successfully downloaded
  }
  TAppUpdateStates = (ausIdle, ausCheckingForUpdate, ausUpdateFound,
                      ausDownloadingUpdate, ausDownloadComplete);

  TAppUpdateForm = class(TForm)
    pcPages: TPageControl;
    tsSample: TTabSheet;
    tsComponentUpdate: TTabSheet;
    btnSampleCheckForUpdate: TButton;
    pbSampleProgress: TProgressBar;
    AppUpdateSample: TCynthesisAppUpdate;
    lblSampleMsg: TLabel;
    lblComponentMsg: TLabel;
    AppUpdateComponent: TCynthesisAppUpdate;
    pbComponentProgress: TProgressBar;
    btnComponentCheckForUpdate: TButton;
    pnlNewVersion: TPanel;
    lblUpdateAvailable: TLabel;
    lblChooseDirectory: TLabel;
    editDirectory: TEdit;
    btnBrowse: TButton;
    dlgBrowse: TOpenDialog;
    memoSampleProgress: TMemo;
    memoComponentProgress: TMemo;
    cmbSampleProtocol: TComboBox;
    lblSampleProtocol: TLabel;
    editSampleUserName: TEdit;
    lblSampleUserName: TLabel;
    lblSamplePassword: TLabel;
    editSamplePassword: TEdit;
    lblSampleHost: TLabel;
    editSampleHost: TEdit;
    lblSampleProxyPort: TLabel;
    editSampleProxyPort: TEdit;
    lblSamplePort: TLabel;
    editSamplePort: TEdit;
    lblSampleProxyServer: TLabel;
    editSampleProxyServer: TEdit;

    procedure FormCreate(Sender: TObject);
    procedure SampleDownloadProgress(Sender: TObject;
      Progress: Integer; Msg: String);
    procedure SampleUpdateAvailable(Sender: TObject;
      versionInfo: TCynthesisAppUpdateVersionInfo);
    procedure btnSampleCheckForUpdateClick(Sender: TObject);
    procedure SampleDownloadComplete(Sender: TObject);
    procedure SampleError(Sender: TObject;
      ErrorInfo: TCynthesisAppUpdateError);
    procedure ComponentDownloadComplete(Sender: TObject);
    procedure ComponentDownloadProgress(Sender: TObject;
      Progress: Integer; Msg: String);
    procedure ComponentError(Sender: TObject;
      ErrorInfo: TCynthesisAppUpdateError);
    procedure ComponentUpdateAvailable(Sender: TObject;
      versionInfo: TCynthesisAppUpdateVersionInfo);
    procedure btnComponentCheckForUpdateClick(Sender: TObject);
    procedure editDirectoryChange(Sender: TObject);
    procedure pcPagesChanging(Sender: TObject; var AllowChange: Boolean);
    procedure ComponentNoUpdateAvailable(Sender: TObject);
    procedure SampleNoUpdateAvailable(Sender: TObject);
    procedure btnBrowseClick(Sender: TObject);
    procedure SampleUpdateAborted(Sender: TObject;
      eReason: TUpdateAbortReasons);
    procedure ComponentUpdateAborted(Sender: TObject;
      eReason: TUpdateAbortReasons);
    procedure cmbSampleProtocolChange(Sender: TObject);

  protected
    procedure Initialize;
    procedure ChangeState(eState: TAppUpdateStates);
    procedure ChangeProtocol(eProtocol: TCynthesisAppUpdateOnlineModes);

  private
    m_eState: TAppUpdateStates;

    procedure AddSampleStatusMessage(strMsg: string);

  end;  { TAppUpdateForm }

var
  AppUpdateForm: TAppUpdateForm;


implementation


uses
  SysUtils, FileCtrl, frmUpdateInfo;

{$R *.DFM}


procedure TAppUpdateForm.FormCreate(Sender: TObject);
begin
  { not much to do on creation - just initialize }
  Self.Initialize;

  with AppUpdateSample do
  begin
    editSampleHost.Text := UpdateInfoURL;
    editSampleUserName.Text := UserName;
    editSamplePassword.Text := UserPassword;
    editSampleProxyServer.Text := ProxyServer;
    editSamplePort.Text := IntToStr(Port);
    editSampleProxyPort.Text := IntToStr(ProxyPort);
  end;  { with AppUpdateSample }

  { setup the initial protocol to be the design-time value }
  Self.ChangeProtocol(AppUpdateSample.OnlineMode);

end;  { FormCreate }


procedure TAppUpdateForm.Initialize;
begin
  { start off in the "check for update" state }
  Self.ChangeState(ausIdle);

  memoSampleProgress.Lines.Clear;
  pbSampleProgress.Position := 0;

  memoComponentProgress.Lines.Add(' ');
  pbComponentProgress.Position := 0;

  { the default directory for installation of the component update is
    the directory in which this application was run - that's as good a
    guess as any, right? }
  editDirectory.Text := ExtractFilePath(Application.ExeName);

  { make sure the app update component is initialized - this will set up
    the component's state, get the user's temp directory, among other
    things }
  AppUpdateSample.Initialize;
  AppUpdateComponent.Initialize;

end;  { Initialize }


procedure TAppUpdateForm.ChangeState(eState: TAppUpdateStates);
begin
  m_eState := eState;

  { depending on what the state is, the button's text on the UI should be
    adjusted according to the action that may be performed }
  case m_eState of

    ausIdle,
    ausDownloadComplete:
    begin
      btnSampleCheckForUpdate.Caption := 'Check for Update';
      btnComponentCheckForUpdate.Caption := 'Check for Update';
    end;  { check for update }

    ausUpdateFound:
    begin
      btnSampleCheckForUpdate.Caption := 'Download Update';
      btnComponentCheckForUpdate.Caption := 'Download Update';
    end;  { download update }

    ausCheckingForUpdate,
    ausDownloadingUpdate:
    begin
      btnSampleCheckForUpdate.Caption := 'Cancel';
      btnComponentCheckForUpdate.Caption := 'Cancel';
    end;  { Cancel }

    else
      ASSERT(FALSE, 'Invalid state');

  end;  { case state }

end;  { ChangeState }


procedure TAppUpdateForm.ChangeProtocol(eProtocol: TCynthesisAppUpdateOnlineModes);
begin
  { select the protocol on the AppUpdate component }
  AppUpdateSample.OnLineMode := eProtocol;

  { set the appropriate state in the protocol combo box }
  cmbSampleProtocol.ItemIndex := Ord(eProtocol);

  editSampleUserName.Enabled := eProtocol <> olHTTP;
  lblSampleUserName.Enabled := editSampleUserName.Enabled;

  editSamplePassword.Enabled := editSampleUserName.Enabled;
  lblSamplePassword.Enabled := editSampleUserName.Enabled;

  editSampleProxyServer.Enabled := eProtocol <> olLANWAN;
  lblSampleProxyServer.Enabled := editSampleProxyServer.Enabled;

  editSampleProxyPort.Enabled := eProtocol <> olLANWAN;
  lblSampleProxyPort.Enabled := editSampleProxyPort.Enabled;

  editSamplePort.Enabled := eProtocol <> olLANWAN;
  lblSamplePort.Enabled := editSamplePort.Enabled;

end;  { ChangeProtocol }


procedure TAppUpdateForm.SampleDownloadProgress(
  Sender: TObject; Progress: Integer; Msg: String);
begin
  { this event is fired by the update component periodically to indicate the
    current progress and status of the update download. Simply update the
    progress bar based on the value, and display the message }

  { only accept messages from the update component if we are not in the idle
    state and some transaction is in progress - otherwise ignore the message }
  if (m_eState = ausCheckingForUpdate) OR (m_eState = ausDownloadingUpdate) then
  begin
    Self.AddSampleStatusMessage(Msg);

    if (Progress >= 0) then
    begin
      pbSampleProgress.Position := Progress;
    end;  { if positive progress }
  end;  { if state not idle }

end;  { SampleDownloadProgress }


procedure TAppUpdateForm.ComponentDownloadProgress(
  Sender: TObject; Progress: Integer; Msg: String);
begin
  { this event is fired by the update component periodically to indicate the
    current progress and status of the update download. Simply update the
    progress bar based on the value, and display the message }

  { only accept messages from the update component if we are not in the idle
    state and some transaction is in progress - otherwise ignore the message }
  if (m_eState = ausCheckingForUpdate) OR (m_eState = ausDownloadingUpdate) then
  begin
    memoComponentProgress.Lines.Add(Msg);
    pbComponentProgress.Position := Progress;
  end;  { if state not idle }

end;  { ComponentDownloadProgress }


procedure TAppUpdateForm.SampleUpdateAvailable(Sender: TObject;
  versionInfo: TCynthesisAppUpdateVersionInfo);
var
  frmUpdateInfoDlg: TFormUpdateInfo;
begin
  { this event is fired by the update component when an update has been found.
    This allows the caller to prompt the user to download the update, or dismiss
    the update automatically until some later time }

  frmUpdateInfoDlg := nil;
  
  { only execute the contents of this method if the component is not being
    automatically updated. Normally your code would not contain this hook }
  if (AppUpdateSample.UpdateMode = umManual) then
  begin

    try
      frmUpdateInfoDlg := TFormUpdateInfo.Create(Self);
      frmUpdateInfoDlg.Initialize(versionInfo);

      if frmUpdateInfoDlg.ShowModal = mrOk then
      begin
        pbSampleProgress.Position := 0;
        Self.AddSampleStatusMessage('Downloading Update...');

        Self.ChangeState(ausDownloadingUpdate);

        { instruct the update component to download the update }
        AppUpdateSample.DownloadUpdate;
      end  { if user wants to download update }
      else
      begin
        Self.SampleUpdateAborted(Sender, arUpdateRefused);
      end;  { else user refused update }

    finally
      frmUpdateInfoDlg.Release;
    end;  { try }

  end;  { if not auto update }

end;  { SampleUpdateAvailable }


procedure TAppUpdateForm.ComponentUpdateAvailable(Sender: TObject;
  versionInfo: TCynthesisAppUpdateVersionInfo);
begin
  { this event is fired by the update component when an update has been found.
    This allows us to inform the user that an update is available }

  { change the state to indicate that an update is available }
  Self.ChangeState(ausUpdateFound);

  memoComponentProgress.Lines.Add('A new update is available.');

  { show the new version panel so that the user is aware that a
    new update is available }
  pnlNewVersion.Visible := TRUE;

end;  { ComponentUpdateAvailable }


procedure TAppUpdateForm.SampleDownloadComplete(Sender: TObject);
var
  ErrorInfo: TCynthesisAppUpdateError;
begin
  { this event is called when the update component has finished downloading
    the update file. This allows you to warn the user that the update is to
    be installed and that the application will now close }

  Self.ChangeState(ausIdle);

  { only execute the contents of this method if the component is not being
    automatically updated. Normally your code would not contain this hook }
  if (AppUpdateSample.UpdateMode = umManual) then
  begin
    pbSampleProgress.Position := 0;

    if MessageDlg('This application will now close so that the update can ' +
                  'be installed.' + #13 + #13 + 'Press Ok to continue and ' +
                  'Cancel to abort the update process. To install the update ' +
                  'manually later, click Cancel and execute the file ' +
                  AppUpdateSample.UpdateFilename,
                  mtConfirmation, mbOKCancel, 0) = mrOk then
    begin
      Self.AddSampleStatusMessage('Closing application and installing update...');

      { start the install process. Depending on the install action chosen,
        the downloaded update file will be launched if it is an executable,
        or unzipped if it is a zip file }
      ErrorInfo := AppUpdateSample.InstallUpdate;

      if (ErrorInfo.ErrorCode <> rcOk) then
      begin
        MessageDlg(ErrorInfo.ErrorMsg + #10 + #10 + ErrorInfo.ErrorAction,
                   mtError, [mbOk], 0);

        Self.AddSampleStatusMessage(' ');
      end;

    end;  { if run update }
  end;  { if not auto update }
end;  { SampleDownloadComplete }


procedure TAppUpdateForm.ComponentDownloadComplete(Sender: TObject);
var
  ErrorInfo: TCynthesisAppUpdateError;
begin
  { this event is called when the update component has finished downloading
    the update file. This allows you to warn the user that the update is to
    be installed and that the application will now close }

  Self.ChangeState(ausIdle);

  pbComponentProgress.Position := 0;

  if MessageDlg('This application will now close so that the update can ' +
                'be installed.' + #13 + #13 + 'Press Ok to continue and ' +
                'Cancel to abort the update process. To install the update ' +
                'manually later, click Cancel and unzip the file ' +
                AppUpdateComponent.UpdateFilename,
                mtConfirmation, mbOKCancel, 0) = mrOk then
  begin
    memoComponentProgress.Lines.Add('Closing application and installing update...');

    { start the install process. Depending on the install action chosen,
      the downloaded update file will be launched if it is an executable,
      or unzipped if it is a zip file }
    AppUpdateComponent.DestDirectory := editDirectory.Text;

    ErrorInfo := AppUpdateComponent.InstallUpdate;

    if (ErrorInfo.ErrorCode <> rcOk) then
    begin
      MessageDlg(ErrorInfo.ErrorMsg + #10 + #10 + ErrorInfo.ErrorAction,
                 mtError, [mbOk], 0);

      Self.AddSampleStatusMessage(' ');
    end;
  end;  { if run update }

end;  { ComponentDownloadComplete }


procedure TAppUpdateForm.SampleError(Sender: TObject;
  ErrorInfo: TCynthesisAppUpdateError);
begin
  { this event is called when an error occurs when attempting to download
    the update or the update file. The error message will indicate the
    problem }

  { note that an error of auNoUpdateAvailable is a special error in this
    case - it indicates that no update is available and is not really a
    a valid error }

  pbSampleProgress.Position := 0;
  Self.AddSampleStatusMessage('Error during update.');

  MessageDlg('The update of the application failed for the following reason: ''' +
             ErrorInfo.ErrorMsg + '''.' + #10 + #10 + ErrorInfo.ErrorAction,
             mtError, [mbOk], 0);

end;  { SampleError }


procedure TAppUpdateForm.ComponentError(Sender: TObject;
  ErrorInfo: TCynthesisAppUpdateError);
begin
  { this event is called when an error occurs when attempting to download
    the update or the update file. The error message will indicate the
    problem }

  pbComponentProgress.Position := 0;
  memoComponentProgress.Lines.Add('Error during download.');

  MessageDlg('The update of the TCynthesisAppUpdate component failed for ' +
             'the following reason: ''' + ErrorInfo.ErrorMsg + '''.',
             mtError, [mbOk], 0);

  memoComponentProgress.Lines.Add(' ');

  Self.ChangeState(ausIdle);

end;  { ComponentError }


procedure TAppUpdateForm.btnSampleCheckForUpdateClick(Sender: TObject);
begin

{$ifdef _TEST}
  AppUpdateSample.InstallUpdate;
  Exit;
{$endif}

  case (m_eState) of

    ausIdle:
    begin
      { the user pressed the Check for Update button so start the process }
      Self.Initialize;

      Self.AddSampleStatusMessage('Checking for update...');

      { initialize the app update component with all the values from the
        various edit boxes }
      with AppUpdateSample do
      begin
        UpdateInfoURL := editSampleHost.Text;
        ProxyServer := editSampleProxyServer.Text;

        { FIXED: 04/25/2000 }
        { make sure we catch the exception that is thrown if the port
          in the edit box is an illegal value }
        try
          ProxyPort := StrToInt(editSampleProxyPort.Text);
        except
          ProxyPort := 0;
          editSampleProxyPort.Text := IntToStr(ProxyPort);
        end;  { try }

        { FIXED: 04/25/2000 }
        { make sure we catch the exception that is thrown if the port
          in the edit box is an illegal value }
        try
          Port := StrToInt(editSamplePort.Text);
        except
          Port := 0;
          editSamplePort.Text := IntToStr(Port);
        end;  { try }

        UserName := editSampleUserName.Text;
        UserPassword := editSamplePassword.Text;
      end;  { with AppUpdateSample }

      { now that a transaction is in progress, set the state to Cancel so that
        the button takes on the appropriate text }
      Self.ChangeState(ausCheckingForUpdate);

      { the CheckForUpdate call starts the whole process in motion }
      AppUpdateSample.CheckForUpdate;

    end;  { check for update }

    ausCheckingForUpdate,
    ausDownloadingUpdate:
    begin
      { the user pressed the Cancel button so cancel the update }
      AppUpdateSample.CancelUpdate;

    end;  { checking, downloading update }

  end;  { case state }

end;  { btnSampleCheckForUpdateClick }


procedure TAppUpdateForm.btnComponentCheckForUpdateClick(Sender: TObject);
begin
  case (m_eState) of

    ausIdle:
    begin
      { the user pressed the Check for Update button so start the process }
      Self.Initialize;

      memoComponentProgress.Lines.Add('Checking for update...');

      { now that a transaction is in progress, set the state to Cancel so that
        the button takes on the appropriate text }
      Self.ChangeState(ausCheckingForUpdate);

      { the CheckForUpdate call starts the whole process in motion }
      AppUpdateComponent.CheckForUpdate;

    end;  { check for update }

    ausUpdateFound:
    begin
      pbComponentProgress.Position := 0;
      memoComponentProgress.Lines.Add('Downloading Update...');

      Self.ChangeState(ausDownloadingUpdate);

      { hide the directory panel }
      pnlNewVersion.Visible := FALSE;

      { instruct the update component to download the update }
      AppUpdateComponent.DownloadUpdate;

    end;  { download update }

    ausCheckingForUpdate,
    ausDownloadingUpdate:
    begin
      { the user pressed the Cancel button so cancel the update }
      memoComponentProgress.Lines.Add(' ');
      pbComponentProgress.Position := 0;

      { reset the state back to the original }
      Self.ChangeState(ausIdle);

      AppUpdateComponent.CancelUpdate;

    end;  { cancel }

  end;  { case state }

end;  { CheckForUpdateClick }


procedure TAppUpdateForm.editDirectoryChange(Sender: TObject);
begin
  { only enabled the download button if a valid directory has been
    entered into the edit box }
  btnComponentCheckForUpdate.Enabled := DirectoryExists(editDirectory.Text);
end;  { editDirectoryChange }


procedure TAppUpdateForm.pcPagesChanging(Sender: TObject;
  var AllowChange: Boolean);
begin
  { if an update is in progress, don't allow the user to change tabs }
  AllowChange := (m_eState = ausIdle);

end;  { pcPagesChanging }


procedure TAppUpdateForm.ComponentNoUpdateAvailable(
  Sender: TObject);
begin
  Self.ChangeState(ausIdle);
end;  { AppUpdateComponentNoUpdateAvailable }


procedure TAppUpdateForm.SampleNoUpdateAvailable(Sender: TObject);
begin
  Self.ChangeState(ausIdle);
end;  { AppupdateSampleNoUpdateAvailable }


procedure TAppUpdateForm.btnBrowseClick(Sender: TObject);
begin
  dlgBrowse.FileName := editDirectory.Text;
  dlgBrowse.InitialDir := ExtractFilePath(dlgBrowse.FileName);

  if (dlgBrowse.Execute) then
  begin
    editDirectory.Text := ExtractFilePath(dlgBrowse.FileName);
  end;  { if user pressed Ok }
end;  { btnBrowseClick }


procedure TAppUpdateForm.SampleUpdateAborted(Sender: TObject;
  eReason: TUpdateAbortReasons);
begin
  pbSampleProgress.Position := 0;

  case (eReason) of
    arUpdateRefused:
    begin
      Self.AddSampleStatusMessage('Update refused.');
    end;  { update refused }

    arCancelled:
    begin
      Self.AddSampleStatusMessage('Update cancelled.');
    end;  { update cancelled }

    arError:
    begin
      { assume that the update error has already been handled, so don't
        display a message }
      { Self.AddSampleStatusMessage('Error during update.'); }
    end;  { error }

    else
    begin
      Self.AddSampleStatusMessage(' ');
    end;  { default }

  end;  { case abort reason }

  Self.ChangeState(ausIdle);

end;  { SampleUpdateAborted }


procedure TAppUpdateForm.ComponentUpdateAborted(Sender: TObject;
  eReason: TUpdateAbortReasons);
begin
  pbComponentProgress.Position := 0;

  case (eReason) of
    arUpdateRefused:  memoComponentProgress.Lines.Add('Update refused.');
    arCancelled:      memoComponentProgress.Lines.Add('Update cancelled.');
    arError:          memoComponentProgress.Lines.Add('Error during update.');
    else              memoComponentProgress.Lines.Add(' ');
  end;  { case abort reason }

  Self.ChangeState(ausIdle);

end;  { ComponentUpdateAborted }


procedure TAppUpdateForm.cmbSampleProtocolChange(Sender: TObject);
begin
  Self.ChangeProtocol(TCynthesisAppUpdateOnlineModes(cmbSampleProtocol.ItemIndex));
end;  { cmbSampleProtocolChange }


procedure TAppUpdateForm.AddSampleStatusMessage(strMsg: string);
begin
  memoSampleProgress.Lines.Add(strMsg);
end;  { AddSampleStatusMessage }

end.
