unit AppUpdateTst;

{*******************************************************************************
          AppUpdateTst - Main Form for the TCynthesisAppUpdate Sample

                    (c) Copyright 1999 Cynthesis Software Inc.

  Written by:

    James Waletzky (waletzky@npsnet.com)
                   (http://www.npsnet.com/waletzky/)

  Description:

    Main form for the AppUpdateTest sample application that demonstrates the
    use of the TCynthesisAppUpdate component.

  Limitations of Use:

    The TCynthesisAppUpdate component (and its various pieces) is free (no
    strings attached) for non-commercial use. For use in a commercial
    application, please contact James Waletzky via e-mail at waletzky@npsnet.com
    for permission and licensing.

    The source code for this component is available from Cynthesis Software
    for a cost of $10. Please remit a cheque or money order payable to Cynthesis
    Software Inc. to the following address:

             Cynthesis Software Inc.
             408 - 260 Newport Drive
             Port Moody, BC
             Canada
             V3H 5C6

  Request:

    If you decide to make use of this component, I would like to hear from you
    as to your comments about the component, or suggestions for improvement.

*******************************************************************************}


interface

uses
  Classes, Forms, Dialogs, StdCtrls, ComCtrls, Controls, NMHttp, ExtCtrls,
  isp3, CynthesisAppUpdate;

type
  { simple state variables for the form object. The various states are
    as follows:
         ausIdle -            - application is idle waiting for user request
         ausCheckingForUpdate - in the middle of checking for an update info
                                file
         ausUpdateFound       - an update is available - the app is now waiting
                                for the user to initiate the download
         ausDownloadingUpdate - in the middle of downloading the actual update
         ausDownloadComplete  - the update has been successfully downloaded
  }
  TAppUpdateStates = (ausIdle, ausCheckingForUpdate, ausUpdateFound,
                      ausDownloadingUpdate, ausDownloadComplete);

  TAppUpdateForm = class(TForm)
    pcPages: TPageControl;
    tsSample: TTabSheet;
    tsComponentUpdate: TTabSheet;
    lblSampleProgress: TLabel;
    btnSampleCheckForUpdate: TButton;
    pbSampleProgress: TProgressBar;
    AppUpdateSample: TCynthesisAppUpdate;
    lblSampleMsg: TLabel;
    Label1: TLabel;
    AppUpdateComponent: TCynthesisAppUpdate;
    pbComponentProgress: TProgressBar;
    lblComponentProgress: TLabel;
    btnComponentCheckForUpdate: TButton;
    pnlNewVersion: TPanel;
    lblUpdateAvailable: TLabel;
    Label2: TLabel;
    editDirectory: TEdit;
    btnBrowse: TButton;
    dlgBrowse: TOpenDialog;

    procedure FormCreate(Sender: TObject);
    procedure SampleDownloadProgress(Sender: TObject;
      Progress: Integer; Msg: String);
    procedure SampleUpdateAvailable(Sender: TObject;
      Version: String);
    procedure btnSampleCheckForUpdateClick(Sender: TObject);
    procedure SampleDownloadComplete(Sender: TObject);
    procedure SampleError(Sender: TObject;
      ErrorInfo: TCynthesisAppUpdateError);
    procedure ComponentDownloadComplete(Sender: TObject);
    procedure ComponentDownloadProgress(Sender: TObject;
      Progress: Integer; Msg: String);
    procedure ComponentError(Sender: TObject;
      ErrorInfo: TCynthesisAppUpdateError);
    procedure ComponentUpdateAvailable(Sender: TObject;
      Version: String);
    procedure btnComponentCheckForUpdateClick(Sender: TObject);
    procedure editDirectoryChange(Sender: TObject);
    procedure pcPagesChanging(Sender: TObject; var AllowChange: Boolean);
    procedure ComponentNoUpdateAvailable(Sender: TObject);
    procedure SampleNoUpdateAvailable(Sender: TObject);
    procedure btnBrowseClick(Sender: TObject);
    procedure SampleUpdateAborted(Sender: TObject;
      eReason: TUpdateAbortReasons);
    procedure ComponentUpdateAborted(Sender: TObject;
      eReason: TUpdateAbortReasons);

  protected
    procedure Initialize;
    procedure ChangeState(eState: TAppUpdateStates);

  private
    m_eState: TAppUpdateStates;

  end;  { TAppUpdateForm }

var
  AppUpdateForm: TAppUpdateForm;


implementation


uses
  SysUtils, FileCtrl;

{$R *.DFM}


procedure TAppUpdateForm.FormCreate(Sender: TObject);
begin
  { not much to do on creation - just initialize }
  Self.Initialize;

end;  { FormCreate }


procedure TAppUpdateForm.Initialize;
begin
  { start off in the "check for update" state }
  Self.ChangeState(ausIdle);

  lblSampleProgress.Caption := '';
  pbSampleProgress.Position := 0;

  lblComponentProgress.Caption := '';
  pbComponentProgress.Position := 0;

  { the default directory for installation of the component update is
    the directory in which this application was run - that's as good a
    guess as any, right? }
  editDirectory.Text := ExtractFilePath(Application.ExeName);

  { make sure the app update component is initialized - this will set up
    the component's state, get the user's temp directory, among other
    things }
  AppUpdateSample.Initialize;
  AppUpdateComponent.Initialize;

end;  { Initialize }


procedure TAppUpdateForm.ChangeState(eState: TAppUpdateStates);
begin
  m_eState := eState;

  { depending on what the state is, the button's text on the UI should be
    adjusted according to the action that may be performed }
  case m_eState of

    ausIdle,
    ausDownloadComplete:
    begin
      btnSampleCheckForUpdate.Caption := 'Check for Update';
      btnComponentCheckForUpdate.Caption := 'Check for Update';
    end;  { check for update }

    ausUpdateFound:
    begin
      btnSampleCheckForUpdate.Caption := 'Download Update';
      btnComponentCheckForUpdate.Caption := 'Download Update';
    end;  { download update }

    ausCheckingForUpdate,
    ausDownloadingUpdate:
    begin
      btnSampleCheckForUpdate.Caption := 'Cancel';
      btnComponentCheckForUpdate.Caption := 'Cancel';
    end;  { Cancel }

    else
      ASSERT(FALSE, 'Invalid state');

  end;  { case state }

end;  { ChangeState }


procedure TAppUpdateForm.SampleDownloadProgress(
  Sender: TObject; Progress: Integer; Msg: String);
begin
  { this event is fired by the update component periodically to indicate the
    current progress and status of the update download. Simply update the
    progress bar based on the value, and display the message }

  { only accept messages from the update component if we are not in the idle
    state and some transaction is in progress - otherwise ignore the message }
  if (m_eState = ausCheckingForUpdate) OR (m_eState = ausDownloadingUpdate) then
  begin
    lblSampleProgress.Caption := Msg;
    pbSampleProgress.Position := Progress;
  end;  { if state not idle }

end;  { SampleDownloadProgress }


procedure TAppUpdateForm.ComponentDownloadProgress(
  Sender: TObject; Progress: Integer; Msg: String);
begin
  { this event is fired by the update component periodically to indicate the
    current progress and status of the update download. Simply update the
    progress bar based on the value, and display the message }

  { only accept messages from the update component if we are not in the idle
    state and some transaction is in progress - otherwise ignore the message }
  if (m_eState = ausCheckingForUpdate) OR (m_eState = ausDownloadingUpdate) then
  begin
    lblComponentProgress.Caption := Msg;
    pbComponentProgress.Position := Progress;
  end;  { if state not idle }

end;  { ComponentDownloadProgress }


procedure TAppUpdateForm.SampleUpdateAvailable(Sender: TObject;
  Version: String);
begin
  { this event is fired by the update component when an update has been found.
    This allows the caller to prompt the user to download the update, or dismiss
    the update automatically until some later time }

  { only execute the contents of this method if the component is not being
    automatically updated. Normally your code would not contain this hook }
  if (AppUpdateSample.UpdateMode = mManual) then
  begin
    { ask the user if they want to download the update }
    if MessageDlg('A new update is available! (Version: ' + Version + ') ' + #10 +
                 'Would you like to download this version?',
                 mtInformation, [mbYes, mbNo], 0) = mrYes then
    begin
      pbSampleProgress.Position := 0;
      lblSampleProgress.Caption := 'Downloading Update...';

      Self.ChangeState(ausDownloadingUpdate);

      { instruct the update component to download the update }
      AppUpdateSample.DownloadUpdate;
    end  { if user wants to download update }
    else
    begin
      Self.SampleUpdateAborted(Sender, arUpdateRefused);
    end;  { else user refused update }
  end;  { if not auto update }

end;  { SampleUpdateAvailable }


procedure TAppUpdateForm.ComponentUpdateAvailable(Sender: TObject;
  Version: String);
begin
  { this event is fired by the update component when an update has been found.
    This allows us to inform the user that an update is available }

  { change the state to indicate that an update is available }
  Self.ChangeState(ausUpdateFound);

  lblComponentProgress.Caption := 'A new update is available.';

  { show the new version panel so that the user is aware that a
    new update is available }
  pnlNewVersion.Visible := TRUE;

end;  { ComponentUpdateAvailable }


procedure TAppUpdateForm.SampleDownloadComplete(Sender: TObject);
var
  ErrorInfo: TCynthesisAppUpdateError;
begin
  { this event is called when the update component has finished downloading
    the update file. This allows you to warn the user that the update is to
    be installed and that the application will now close }

  Self.ChangeState(ausIdle);

  { only execute the contents of this method if the component is not being
    automatically updated. Normally your code would not contain this hook }
  if (AppUpdateSample.UpdateMode = mManual) then
  begin
    pbSampleProgress.Position := 0;

    if MessageDlg('This application will now close so that the update can ' +
                  'be installed.' + #13 + #13 + 'Press Ok to continue and ' +
                  'Cancel to abort the update process. To install the update ' +
                  'manually later, click Cancel and execute the file ' +
                  AppUpdateSample.UpdateFilename,
                  mtConfirmation, mbOKCancel, 0) = mrOk then
    begin
      lblSampleProgress.Caption := 'Closing application and installing update...';

      { start the install process. Depending on the install action chosen,
        the downloaded update file will be launched if it is an executable,
        or unzipped if it is a zip file }
      ErrorInfo := AppUpdateSample.InstallUpdate;

      if (ErrorInfo.ErrorCode <> rcOk) then
      begin
        MessageDlg(ErrorInfo.ErrorMsg + #10 + #10 + ErrorInfo.ErrorAction,
                   mtError, [mbOk], 0);

        lblSampleProgress.Caption := '';
      end;

    end;  { if run update }
  end;  { if not auto update }
end;  { SampleDownloadComplete }


procedure TAppUpdateForm.ComponentDownloadComplete(Sender: TObject);
var
  ErrorInfo: TCynthesisAppUpdateError;
begin
  { this event is called when the update component has finished downloading
    the update file. This allows you to warn the user that the update is to
    be installed and that the application will now close }

  Self.ChangeState(ausIdle);

  pbComponentProgress.Position := 0;

  if MessageDlg('This application will now close so that the update can ' +
                'be installed.' + #13 + #13 + 'Press Ok to continue and ' +
                'Cancel to abort the update process. To install the update ' +
                'manually later, click Cancel and unzip the file ' +
                AppUpdateComponent.UpdateFilename,
                mtConfirmation, mbOKCancel, 0) = mrOk then
  begin
    lblComponentProgress.Caption := 'Closing application and installing update...';

    { start the install process. Depending on the install action chosen,
      the downloaded update file will be launched if it is an executable,
      or unzipped if it is a zip file }
    AppUpdateComponent.DestDirectory := editDirectory.Text;

    ErrorInfo := AppUpdateComponent.InstallUpdate;

    if (ErrorInfo.ErrorCode <> rcOk) then
    begin
      MessageDlg(ErrorInfo.ErrorMsg + #10 + #10 + ErrorInfo.ErrorAction,
                 mtError, [mbOk], 0);

      lblSampleProgress.Caption := '';
    end;
  end;  { if run update }

end;  { ComponentDownloadComplete }


procedure TAppUpdateForm.SampleError(Sender: TObject;
  ErrorInfo: TCynthesisAppUpdateError);
begin
  { this event is called when an error occurs when attempting to download
    the update or the update file. The error message will indicate the
    problem }

  { note that an error of auNoUpdateAvailable is a special error in this
    case - it indicates that no update is available and is not really a
    a valid error }

  pbSampleProgress.Position := 0;
  lblSampleProgress.Caption := 'Error during update.';

  MessageDlg('The update of the application failed for the following reason: ''' +
             ErrorInfo.ErrorMsg + '''.', mtError, [mbOk], 0);

end;  { SampleError }


procedure TAppUpdateForm.ComponentError(Sender: TObject;
  ErrorInfo: TCynthesisAppUpdateError);
begin
  { this event is called when an error occurs when attempting to download
    the update or the update file. The error message will indicate the
    problem }

  pbComponentProgress.Position := 0;
  lblComponentProgress.Caption := 'Error during download.';

  MessageDlg('The update of the TCynthesisAppUpdate component failed for ' +
             'the following reason: ''' + ErrorInfo.ErrorMsg + '''.',
             mtError, [mbOk], 0);

  lblComponentProgress.Caption := '';

  Self.ChangeState(ausIdle);

end;  { ComponentError }


procedure TAppUpdateForm.btnSampleCheckForUpdateClick(Sender: TObject);
begin

{$ifdef _TEST}
  AppUpdateSample.InstallUpdate;
  Exit;
{$endif}

  case (m_eState) of

    ausIdle:
    begin
      { the user pressed the Check for Update button so start the process }
      Self.Initialize;

      lblSampleProgress.Caption := 'Checking for update...';

      { now that a transaction is in progress, set the state to Cancel so that
        the button takes on the appropriate text }
      Self.ChangeState(ausCheckingForUpdate);

      { the CheckForUpdate call starts the whole process in motion }
      AppUpdateSample.CheckForUpdate;

    end;  { check for update }

    ausCheckingForUpdate,
    ausDownloadingUpdate:
    begin
      { the user pressed the Cancel button so cancel the update }
      AppUpdateSample.CancelUpdate;

    end;  { checking, downloading update }

  end;  { case state }

end;  { btnSampleCheckForUpdateClick }


procedure TAppUpdateForm.btnComponentCheckForUpdateClick(Sender: TObject);
begin
  case (m_eState) of

    ausIdle:
    begin
      { the user pressed the Check for Update button so start the process }
      Self.Initialize;

      lblComponentProgress.Caption := 'Checking for update...';

      { now that a transaction is in progress, set the state to Cancel so that
        the button takes on the appropriate text }
      Self.ChangeState(ausCheckingForUpdate);

      { the CheckForUpdate call starts the whole process in motion }
      AppUpdateComponent.CheckForUpdate;

    end;  { check for update }

    ausUpdateFound:
    begin
      pbComponentProgress.Position := 0;
      lblComponentProgress.Caption := 'Downloading Update...';

      Self.ChangeState(ausDownloadingUpdate);

      { hide the directory panel }
      pnlNewVersion.Visible := FALSE;

      { instruct the update component to download the update }
      AppUpdateComponent.DownloadUpdate;

    end;  { download update }

    ausCheckingForUpdate,
    ausDownloadingUpdate:
    begin
      { the user pressed the Cancel button so cancel the update }
      lblComponentProgress.Caption := '';
      pbComponentProgress.Position := 0;

      { reset the state back to the original }
      Self.ChangeState(ausIdle);

      AppUpdateComponent.CancelUpdate;

    end;  { cancel }

  end;  { case state }

end;  { CheckForUpdateClick }


procedure TAppUpdateForm.editDirectoryChange(Sender: TObject);
begin
  { only enabled the download button if a valid directory has been
    entered into the edit box }
  btnComponentCheckForUpdate.Enabled := DirectoryExists(editDirectory.Text);
end;  { editDirectoryChange }


procedure TAppUpdateForm.pcPagesChanging(Sender: TObject;
  var AllowChange: Boolean);
begin
  { if an update is in progress, don't allow the user to change tabs }
  AllowChange := (m_eState = ausIdle);

end;  { pcPagesChanging }


procedure TAppUpdateForm.ComponentNoUpdateAvailable(
  Sender: TObject);
begin
  Self.ChangeState(ausIdle);
end;  { AppUpdateComponentNoUpdateAvailable }


procedure TAppUpdateForm.SampleNoUpdateAvailable(Sender: TObject);
begin
  Self.ChangeState(ausIdle);
end;  { AppupdateSampleNoUpdateAvailable }


procedure TAppUpdateForm.btnBrowseClick(Sender: TObject);
begin
  dlgBrowse.FileName := editDirectory.Text;
  dlgBrowse.InitialDir := ExtractFilePath(dlgBrowse.FileName);

  if (dlgBrowse.Execute) then
  begin
    editDirectory.Text := ExtractFilePath(dlgBrowse.FileName);
  end;  { if user pressed Ok }
end;  { btnBrowseClick }


procedure TAppUpdateForm.SampleUpdateAborted(Sender: TObject;
  eReason: TUpdateAbortReasons);
begin
  pbSampleProgress.Position := 0;

  case (eReason) of
    arUpdateRefused:  lblSampleProgress.Caption := 'Update refused.';
    arCancelled:      lblSampleProgress.Caption := 'Update cancelled.';
    arError:          lblSampleProgress.Caption := 'Error during update.';
    else              lblSampleProgress.Caption := '';
  end;  { case abort reason }

  Self.ChangeState(ausIdle);

end;  { SampleUpdateAborted }


procedure TAppUpdateForm.ComponentUpdateAborted(Sender: TObject;
  eReason: TUpdateAbortReasons);
begin
  pbComponentProgress.Position := 0;

  case (eReason) of
    arUpdateRefused:  lblComponentProgress.Caption := 'Update refused.';
    arCancelled:      lblComponentProgress.Caption := 'Update cancelled.';
    arError:          lblComponentProgress.Caption := 'Error during update.';
    else              lblComponentProgress.Caption := '';
  end;  { case abort reason }

  Self.ChangeState(ausIdle);

end;  { ComponentUpdateAborted }

end.
