// UniPrn v1.1

// Modu wydrukw dla drukarki igowej i laserowej z funkcj podgldu

// Non-visual component providing full support for dot-matrix printers
// epson-compatible.
// It allows print in text-mode (very fast), print in graphic mode
// (for supporting other printer types), print preview.
// One-command table print, page header, page footer,
// blocks, page numbers and much more.
// Easy to use. Gently check public methods and published properties.

// Author: Jerzy Jaskiewicz mailto:jurek@puch.px.pl

unit UniPrn;

interface

uses
    Windows, SysUtils, Printers, StrUtils, DB, Classes, Controls,
    Graphics, Dialogs, Forms, ExtCtrls;

const Fdel = #255;
      FontFace: String='Arial';

      mazconvstr='';
      latconvstr='䢘'#$97'';
      chrconvstr='acelnosxzACELNOSXZ';
      winconvstr='󜟿ʣӌ';

type
  TPrintColumn = record
    Left: integer;
    Width: integer;
    Alignment: TAlignment;
    LineR: Boolean;
    LineL: Boolean;
    Text: string;
    end;

  TPrintColumns = array of TPrintColumn;
  TScriptMode = (smNormal, smSubscript, smSuperscript);
  TTextPrinterCharset = (pcMazowia, pcLatin, pcWindows, pcNone);

  TPrinterDefinition = class(TStringList)
    public
    Charset: TTextPrinterCharset;
    constructor Create;
    function ControlString(AName: string): string;
    function ControlValue(AName: string): integer;
    procedure LoadFromFile(const FileName: string); override;
    property ControlStr[Aname: string]: string read ControlString; default;
    end;

  TUniPrinter = class(TComponent)
  private
    FBold: boolean;
    FUnderline: boolean;
    FCondensed: boolean;
    FDoubleWidth: boolean;
    FDoubleHeight: boolean;
    FProportional: boolean;
    FItalic: boolean;
    FScriptMode: TScriptMode;
    FAlignment: TAlignment;
    FCursorPos: TPoint;
    FInBlock: boolean;
    FPrintHorLineBy: integer;
    FBlockBuffer: string;
    FFooterText: string;
    FTitle: string;
    FPageSummaryLinesCount: integer;
    FPages: TStringList;
    FOnPageSummary: TNotifyEvent;
    FOnNewPage: TNotifyEvent;
    FPrintingSummary: boolean;
    FOnTableScroll: TDataSetNotifyEvent;
    FPrinterDefinition: TPrinterDefinition;
    FPrinterDefinitionFilename: TFilename;
    FPrintColumns: TPrintColumns;
    FLinesPerPage: integer;
    FMariginLeft: integer;
    FMariginTop: integer;
    FMariginBottom: integer;
    FMariginRight: integer;
    procedure Add(s: string);
    procedure SetX(X: integer);
    procedure SetY(Y: integer);
    procedure SetBold(Value: boolean);
    procedure SetItalic(Value: boolean);
    procedure SetUnderline(Value: boolean);
    procedure SetCondensed(Value: boolean);
    procedure SetDoubleWidth(Value: boolean);
    procedure SetDoubleHeight(Value: boolean);
    procedure SetProportional(Value: boolean);
    procedure SetScriptMode(Value: TScriptMode);
    procedure SetAlignment(Value: TAlignment);
    procedure SetCurrentLine(LineNumber: integer);
    procedure SetCursorPos(Position: TPoint);
    procedure SetPrinterDefinition(PrinterDefinitionFilename: TFilename);
    procedure PrintFooter;
    function GetCurrentLine:integer;
    function GetPagesCount: integer;
    procedure GetTableDefinition(Table: TDataSet; VerticalLines: Boolean);
    procedure AddTableDefinition;
    function TextDraw(Page: integer): string;
    function Translate(s: string): string;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure BeginDoc;
    procedure EndDoc;
    procedure LineFeed;
    procedure LineBack;
    procedure StartBlock;
    procedure EndBlock;
    procedure FormFeed;
    procedure WriteText(s: string);
    procedure PrintTable(Table: TDataSet; HorizontalLines, VerticalLines: Boolean);
    procedure PrintHeader(Table: TDataSet; HorizontalLines, VerticalLines: Boolean);
    procedure PreviewPage(Page: integer);
    procedure Print(Copies: integer);
    procedure TextPrint(Copies: integer);
    procedure Draw(Page: integer; Canvas: TCanvas);
    procedure DrawPreview(Sender: TObject);
    procedure PrintFromPreview(Sender: TObject);
    property X: integer read FcursorPos.X write SetX;
    property Y: integer read FcursorPos.Y write SetY;
    property CurrentLine: integer read GetCurrentLine write SetCurrentLine;
    property CursorPos: TPoint read FCursorPos write SetCursorPos;
    property PagesCount: integer read GetPagesCount;
    property Metatext: TStringList read FPages;
  published
    property MariginLeft: integer read FMariginLeft write FMariginLeft default 0;
    property MariginTop: integer read FMariginTop write FMariginTop default 0;
    property MariginRight: integer read FMariginRight write FMariginRight default 0;
    property MariginBottom: integer read FMariginBottom write FMariginBottom default 0;
    property Bold: boolean read FBold write SetBold default False;
    property Italic: boolean read FItalic write SetItalic default False;
    property Underline: boolean read FUnderline write SetUnderline default False;
    property Condensed: boolean read FCondensed write SetCondensed default False;
    property DoubleWidth: boolean read FDoubleWidth write SetDoubleWidth default False;
    property DoubleHeight: boolean read FDoubleHeight write SetDoubleHeight default False;
    property Proportional: boolean read FProportional write SetProportional default False;
    property ScriptMode: TScriptMode read FScriptMode write SetScriptMode default smNormal;
    property Alignment: TAlignment read FAlignment write SetAlignment default taLeftJustify;
    property FooterText: string read FFooterText write FFooterText;
    property PrinterDefinition: TFilename read FPrinterDefinitionFilename write SetPrinterDefinition;
    property Title: string read FTitle write FTitle;
    property LinesPerPage: integer read FLinesPerPage write FLinesPerPage default 60;
    property HorizontalLineBy: integer read FPrintHorLineBy write FPrintHorLineBy default 1;
    property PageSummaryLinesCount: integer read FPageSummaryLinesCount write FPageSummaryLinesCount default 0;
    property OnPageSummary: TNotifyEvent read FOnPageSummary write FOnPageSummary;
    property OnNewPage: TNotifyEvent read FOnNewPage write FOnNewPage;
    property OnTableScroll: TDataSetNotifyEvent read FOnTableScroll write FOnTableScroll;
  end;

procedure Register;

implementation

uses UniPrv;

// codepage converting routines
procedure konw(var s:string ; const fs,ts:string);
var
  k,p:integer;
begin
  for k:=1 to length(s) do
    begin
      p:=pos(s[k],fs);
      if p<>0 then
        s[k]:=ts[p]
    end;
end;

function win2Asc(s:string):string;
begin
  konw(s,winconvstr,chrconvstr);
  Result:=s;
end;{win2Asc}

function Win2Lat2(s:string):string;
begin
  konw(s,winconvstr,latconvstr);
  Win2Lat2:=s;
end;{Win2Lat2}

function Win2Maz(s:string):string;
begin
  konw(s,winconvstr,Mazconvstr);
  Win2Maz:=s;
end;{Win2Maz}

// general purpose functions

function ischar( ch: char ):Boolean;
begin
	Result := ch in ['A'..'Z', 'a'..'z'];
end;

function isdigit( ch: char ):Boolean;
begin
	Result := ch in ['0'..'9'];
end;

function AlterFont(AFnt: TFont; ADW, ADH, ACondensed: boolean; AStyle: TFontStyles):integer;
var
  l_Fnt:TLOGFONT;
begin
  fillchar(l_Fnt,sizeof(l_Fnt),0);
   with l_Fnt do
   begin
    StrPCopy(lfFaceName,FontFace);
    if ADH then lfHeight:=-50 else lfHeight:=-40;
    if ADW then lfWidth:=50 else lfWidth:=25;
    if ACondensed then lfWidth:=Trunc(lfWidth/1.65);
    if fsUnderline in AStyle then lfUnderline:=1;
    if fsItalic in AStyle then lfItalic:=1;
    if fsBold in AStyle then lfWeight:= FW_EXTRABOLD else lfWeight:=FW_NORMAL;
    lfCharSet:=EASTEUROPE_CHARSET;
    lfQuality:=DEFAULT_QUALITY;
    Result:=Trunc(lfWidth*1.25);
  end;
  if AFnt.Handle<>0 then
    with AFnt do
    begin
//      DeleteObject(Handle); {Kasujemy czcionk utworzon przez Create}
      Height:=l_Fnt.lfHeight;
      Charset:=l_Fnt.lfCharSet;
      Style:=AStyle;
      Color:=clBlack;
      Handle:=CreateFontIndirect(l_Fnt) {Tworzymy wasn czcionk};
    end;
end;

function IntValue(n: integer; s: string):integer;
var i, c: integer;
    FirstChar: Boolean;
begin
c:=0;
FirstChar:=True; Result:=0;
for i:=1 to Length(s) do
    begin
    if s[i] in ['0'..'9'] then
       begin
       if FirstChar then
          begin
          FirstChar:=False;
          inc(c);
          Result:=0;
          end;
       Result:=(Result*10)+Ord(s[i])-Ord('0');
       end else
       begin
       if n=c then Exit;
       FirstChar:=True;
       end;
    end;
end;

function StrValue(n: integer; s: string): string;
var i: integer;
begin
i:=NPos(FDel, s, n+2);
if i=0 then i:=maxint;
Result:=Copy(s, NPos(FDel, s, n+1)+1, i-NPos(FDel, s, n+1)-1);
end;

function CopyFirstN(n: integer; var s: string):string;
var i: integer;
label found;
begin
if Length(s)>n then
   begin
   for i:=1 to n do
       if s[i]=#13 then
          goto found;
   for i:=n downto 1 do
       if s[i] in [' ', ',', '/', '.', '!'] then
          goto found;
   i:=n;
   end else
   begin
   for i:=1 to Length(s) do
       if s[i]=#13 then
          goto found;
   i:=Length(s);
   end;
found:
Result:=DelChars(DelChars(Copy(s, 1, i), #13), #10);
Delete(s, 1, i);
end;

constructor TPrinterDefinition.Create;
begin
inherited Create;
Charset:=pcWindows;
Add('FLinesCount=60');
Add('FPixelsPerLine=36');
Add('FInterlineSpace=12');
end;

procedure TPrinterDefinition.LoadFromFile(const FileName: string);
begin
if FileExists(FileName) then inherited LoadFromFile(FileName);
if Values['Charset']='Mazowia' then Charset:=pcMazowia;
if Values['Charset']='Latin' then Charset:=pcLatin;
if Values['Charset']='Windows' then Charset:=pcWindows;
if Values['Charset']='None' then Charset:=pcNone;
end;

function TPrinterDefinition.ControlString(AName: string): string;
var s, number: string;
    i, l: integer;
begin
if IndexOfName(AName)=-1 then
   begin
   ShowMessage('Parameter missing: '+AName);
   Abort;
   end;
if AName[1]<>'c' then Exit;
s:=Values[AName];
Result:='';
i:=1; l:=Length(s);
while i<=l do
      begin
      if isChar(s[i]) then
         begin
         Result:=Result+s[i];
         Inc(i);
         Continue;
         end;
      if Copy(s, i, 1)='#' then
         begin
         number:='';
         if Copy(s, i+1, 1)='$' then
            begin
            Inc(i, 2);
            while (Copy(s, i, 1)<>'') and  (Copy(s, i, 1)[1] in ['0'..'9', 'A'..'F', 'a'..'f']) do
                  begin
                  number:=number+Copy(s, i, 1);
                  Inc(i);
                  end;
            Result:=Result+chr(Hex2Dec(number));
            end else
            begin
            Inc(i);
            while (Copy(s, i, 1)<>'') and isdigit(Copy(s, i, 1)[1])do
                  begin
                  number:=number+Copy(s, i, 1);
                  Inc(i);
                  end;
            Result:=Result+chr(StrToInt(number));
            end;
         Continue;
         end;
      Inc(i);
      end;
end;

function TPrinterDefinition.ControlValue(AName: string): integer;
begin
if IndexOfName(AName)=-1 then
   begin
   ShowMessage('Parameter missing: '+AName);
   Abort;
   end;
Result:=StrToInt(Values[AName]);
end;

constructor TUniPrinter.Create(AOwner: TComponent);
begin
inherited Create(AOwner);
FPages:=TStringList.Create;
FPrinterDefinition:=TPrinterDefinition.Create;
FLinesPerPage:=60;
FCursorPos.x:=0;
FCursorPos.y:=0;
FInBlock:=False;
FBlockBuffer:='';
FFooterText:='';
FOnNewPage:=nil;
FOnTableScroll:=nil;
FPrinterDefinitionFilename:='';
FBold:=False;
FUnderline:=False;
FCondensed:=False;
FDoubleWidth:=False;
FDoubleHeight:=False;
FProportional:=False;
FItalic:=False;
FScriptMode:=smNormal;
FInBlock:=False;
FAlignment:=taLeftJustify;
FPrintHorLineBy:=1;
FPageSummaryLinesCount:=0;
FPrintingSummary:=False;
if Printer.Fonts.IndexOf('Tahoma')>=0 then FontFace:='Tahoma';
end;

destructor TUniPrinter.Destroy;
begin
FPages.Free;
FPrinterDefinition.Free;
inherited Destroy;
end;

procedure TUniPrinter.SetPrinterDefinition(PrinterDefinitionFilename: TFilename);
begin
FPrinterDefinition.LoadFromFile(PrinterDefinitionFilename);
FLinesPerPage:=FLinesPerPage;
end;

procedure TUniPrinter.SetX(X: integer);
begin
SetCursorPos(Point(x, FCursorPos.y));
end;

procedure TUniPrinter.SetY(Y: integer);
begin
SetCursorPos(Point(FCursorPos.x, y));
end;

function TUniPrinter.Translate(s: string): string;
begin
case FPrinterDefinition.Charset of
     pcNone: Result:=Win2Asc(s);
     pcMazowia: Result:=Win2Maz(s);
     pcLatin: Result:=Win2lat2(s);
     else
     Result:=s;
     end;
end;


procedure TUniPrinter.BeginDoc;
begin
FCursorPos.x:=0;
FCursorPos.y:=0;
FPages.Clear;
FTitle:='';
FPages.Add('');
if Assigned(FOnNewPage) then FOnNewPage(Self);
x:=x; y:=y;
Condensed:=Condensed;
Bold:=Bold;
DoubleHeight:=DoubleHeight;
DoubleWidth:=DoubleWidth;
Italic:=Italic;
ScriptMode:=ScriptMode;
end;

procedure TUniPrinter.EndDoc;
var i: integer;
begin
PrintFooter;
for i:=0 to FPages.Count-1 do
    begin
    FPages[i]:=ReplaceStr(FPages[i], '%p%', IntToStr(FPages.Count)); // pages count
    FPages[i]:=ReplaceStr(FPages[i], '%n%', IntToStr(i+1));  // current page number
    end;
end;

procedure TUniPrinter.StartBlock;
begin
FInBlock:=True;
FBlockBuffer:='';
end;

procedure TUniPrinter.EndBlock;
begin
if not FInBlock then Exit;
FInBlock:=False;
Add(FBlockBuffer);
FBlockBuffer:='';
end;

procedure TUniPrinter.LineFeed;
begin
if GetCurrentLine+FPageSummaryLinesCount>=FLinesPerPage-1
   then FormFeed
   else begin
   Add('<LF><CR>');
   Inc(FCursorPos.y, 1);
   FCursorPos.x:=0;
   end;
end;

procedure TUniPrinter.LineBack;
begin
if GetCurrentLine<=0
   then Exit
   else begin
   Add('<LB><CR>');
   Dec(FCursorPos.y, 1);
   FCursorPos.x:=0;
   if FCursorPos.y<0 then FCursorPos.y:=0;
   end;
end;

procedure TUniPrinter.FormFeed;
var TmpInBlock: boolean;
    tmpPrintColumns: TPrintColumns;
    procedure BackupColumns;
              var i: integer;
              begin
              SetLength(tmpPrintColumns, Length(FPrintColumns));
              for i:=0 to Length(FPrintColumns)-1 do
                  begin
                  tmpPrintColumns[i].Left:=FPrintColumns[i].Left;
                  tmpPrintColumns[i].Width:=FPrintColumns[i].Width;
                  tmpPrintColumns[i].Alignment:=FPrintColumns[i].Alignment;
                  tmpPrintColumns[i].LineR:=FPrintColumns[i].LineR;
                  tmpPrintColumns[i].LineL:=FPrintColumns[i].LineL;
                  tmpPrintColumns[i].Text:=FPrintColumns[i].Text;
                  end;
              end;
    procedure RestoreColumns;
              var i: integer;
              begin
              SetLength(FPrintColumns, Length(tmpPrintColumns));
              for i:=0 to Length(FPrintColumns)-1 do
                  begin
                  FPrintColumns[i].Left:=tmpPrintColumns[i].Left;
                  FPrintColumns[i].Width:=tmpPrintColumns[i].Width;
                  FPrintColumns[i].Alignment:=tmpPrintColumns[i].Alignment;
                  FPrintColumns[i].LineR:=tmpPrintColumns[i].LineR;
                  FPrintColumns[i].LineL:=tmpPrintColumns[i].LineL;
                  FPrintColumns[i].Text:=tmpPrintColumns[i].Text;
                  end;
              tmpPrintColumns:=nil;
              end;
begin
if FPrintingSummary then Exit; // nie ma tu FF
TmpInBlock:=FInBlock;
FInBlock:=False;
BackupColumns;
if Assigned(FOnPageSummary) then
   begin
   FPrintingSummary:=True;
   FOnPageSummary(Self);
   FPrintingSummary:=False;
   end;
PrintFooter;
FPages.Add('');
FCursorPos.y:=0;
x:=x;
Condensed:=Condensed;
Bold:=Bold;
DoubleHeight:=DoubleHeight;
DoubleWidth:=DoubleWidth;
Italic:=Italic;
ScriptMode:=ScriptMode;
if Assigned(FOnNewPage) then FOnNewPage(Self);
RestoreColumns;
AddTableDefinition;
FInBlock:=TmpInBlock;
end;

procedure TUniPrinter.PrintFooter;
var TmpInBlock: boolean;
    OldBold: boolean;
    OldUnderline: boolean;
    OldCondensed: boolean;
    OldDoubleWidth: boolean;
    OldDoubleHeight: boolean;
    OldProportional: boolean;
    OldItalic: boolean;
    OldScriptMode: TScriptMode;
    OldAlignment: TAlignment;
begin
TmpInBlock:=FInBlock;
FInBlock:=False;
OldBold:=Bold; OldUnderline:=Underline; OldCondensed:=Condensed; OldDoubleWidth:=DoubleWidth;
OldDoubleHeight:=DoubleHeight; OldProportional:=Proportional; OldItalic:=Italic;
OldScriptMode:=ScriptMode;     OldAlignment:=Alignment;
Bold:=False; Underline:=False; Condensed:=False; DoubleWidth:=False; DoubleHeight:=False;
Proportional:=False; Italic:=False; ScriptMode:=smNormal; Alignment:=taLeftJustify;

SetCurrentLine(FLinesPerPage);
WriteText(FFooterText);

Bold:=OldBold; Underline:=OldUnderline; Condensed:=OldCondensed; DoubleWidth:=OldDoubleWidth;
DoubleHeight:=OldDoubleHeight; Proportional:=OldProportional; Italic:=OldItalic;
ScriptMode:=OldScriptMode; Alignment:=OldAlignment;
FInBlock:=TmpInBlock;
Add('<PAGE>');
end;

procedure TUniPrinter.Add(s: string);
begin
if FPages.Count=0 then Exit;
   if FInBlock then
      begin
      FBlockBuffer:=FBlockBuffer+s;
      Exit;
      end;
   FPages.Strings[Fpages.Count-1]:=FPages.Strings[Fpages.Count-1]+s;
end;

procedure TUniPrinter.Print(Copies: integer);
var Page: integer;
    MetaFile: TMetaFile;
    Canvas: TMetaFileCanvas;
begin
Printer.Title:=FTitle;
Printer.Copies:=Copies;
Printer.BeginDoc;
for Page:=1 to FPages.Count do
    begin
    if Page>1 then Printer.NewPage;
    MetaFile:=TMetaFile.Create;
    MetaFile.Width:=(Printer.PageWidth*300) div GetDeviceCaps(Printer.Handle,LogPixelSX);
    MetaFile.Height:=(Printer.PageHeight*300) div GetDeviceCaps(Printer.Handle,LogPixelSY);
    Canvas:=TMetaFileCanvas.Create(MetaFile, Printer.Handle);
    Draw(Page, Canvas);
    Canvas.Free;
    Printer.Canvas.StretchDraw(Rect((MariginLeft*GetDeviceCaps(Printer.Handle,LogPixelSX)) div 100, (MariginTop*GetDeviceCaps(Printer.Handle,LogPixelSY)) div 100, Printer.Canvas.ClipRect.Right-(MariginRight*GetDeviceCaps(Printer.Handle,LogPixelSX)) div 100, Printer.Canvas.ClipRect.Bottom-(MariginTop*GetDeviceCaps(Printer.Handle,LogPixelSY)) div 100), MetaFile);
    MetaFile.Free;
    end;
Printer.EndDoc;
end;


procedure TUniPrinter.TextPrint(Copies: integer);
const MaxBytes=255;
type TPrnBuffRec = record
  BuffLength : word;
  Buffer : array [0..255] of char;
  end;
var
  Buff      : TPrnBuffRec;
  TestInt   : integer;
  i, Page   : Integer;
  TmpStr,s  : string;
begin
Printer.Title:=FTitle;
TestInt := PASSTHROUGH;
if not Escape(Printer.Handle,
            QUERYESCSUPPORT,
            sizeof(TestInt),
            @TestInt,
            nil) > 0 then Exit;
while Copies>0 do
      begin
      for Page:=1 to FPages.Count do
          begin
          if not Printer.Printing
                    then Printer.BeginDoc
                    else Printer.NewPage;
          s:=TextDraw(Page);
          while length(s) >0 do
            begin
            TmpStr:=Copy(s, 1, MaxBytes);
            Buff.BuffLength := Length(TmpStr);
            for i:=0 to Buff.BuffLength-1 do Buff.Buffer[i]:=TmpStr[i+1];
            Escape(Printer.Canvas.Handle, PASSTHROUGH, 0, @Buff, nil);
            Delete(s, 1, MaxBytes);
            end;
          end;
      Dec(Copies);
      end;
if Printer.Printing then Printer.EndDoc;
end;

procedure TUniPrinter.SetBold(Value: boolean);
begin
FBold:=Value;
if Value then Add('<B>')
         else Add('<\B>');
end;

procedure TUniPrinter.SetItalic(Value: boolean);
begin
FItalic:=Value;
if Value then Add('<I>')
         else Add('<\I>');
end;

procedure TUniPrinter.SetDoubleWidth(Value: boolean);
begin
FDoubleWidth:=Value;
if Value then Add('<DW>')
         else Add('<\DW>');
end;

procedure TUniPrinter.SetUnderline(Value: boolean);
begin
FUnderline:=Value;
if Value then Add('<U>')
         else Add('<\U>');
end;

procedure TUniPrinter.SetCondensed(Value: boolean);
begin
FCondensed:=Value;
if Value then Add('<C>')
         else Add('<\C>');
end;

procedure TUniPrinter.SetDoubleHeight(Value: boolean);
begin
FDoubleHeight:=Value;
if Value then Add('<DH>')
         else Add('<\DH>');
end;

procedure TUniPrinter.SetProportional(Value: boolean);
begin
FProportional:=Value;
if Value then Add('<PROP>')
         else Add('<\PROP>');
end;

procedure TUniPrinter.SetScriptMode(Value: TScriptMode);
begin
FScriptMode:=Value;
case Value of
     smSubscript: Add('<SUBSCRIPT>');
     smSuperscript: Add('<SUPERSCRIPT>');
     smNormal: Add('<NORMALSCRIPT>');
     end;
end;

procedure TUniPrinter.SetAlignment(Value: TAlignment);
begin
case Value of
     taLeftJustify    : Add('<LEFT>');
     taRightJustify   : Add('<RIGHT>');
     taCenter         : Add('<CENTER>');
     end;
end;

procedure TUniPrinter.SetCurrentLine(LineNumber: integer);
begin
if LineNumber>FLinesPerPage then
   raise Exception.Create('Invalid line number');
Add('<SETPOS 0, '+IntToStr(LineNumber)+'>');
end;

procedure TUniPrinter.SetCursorPos(Position: TPoint);
begin
Add('<SETPOS '+IntToStr(Position.X)+', '+IntToStr(Position.Y)+'>');
FCursorPos:=Position;
end;


function TUniPrinter.GetCurrentLine: integer;
begin
Result:=FCursorPos.y;
end;

function TUniPrinter.GetPagesCount: integer;
begin
Result:=FPages.Count;
end;

procedure TUniPrinter.WriteText(s: string);
var i: integer;
begin
i:=1;
while i<=Length(s) do
      if Ord(s[i])<32
         then Delete(s, i, 1)
         else Inc(i);
Add(s);
end;

procedure TUniPrinter.GetTableDefinition(Table: TDataSet; VerticalLines: Boolean);
var i   : integer;
    Left: integer;
begin
FPrintColumns:=nil;
Left:=0;
for i:=0 to Table.FieldCount-1 do
  if Table.Fields[i].Visible then
      begin
      SetLength(FPrintColumns, Length(FPrintColumns)+1);
      FPrintColumns[Length(FPrintColumns)-1].Left:=Left;
      FPrintColumns[Length(FPrintColumns)-1].Width:=Table.Fields[i].DisplayWidth;
      FPrintColumns[Length(FPrintColumns)-1].Alignment:=Table.Fields[i].Alignment;
      FPrintColumns[Length(FPrintColumns)-1].LineL:=VerticalLines and (Left=0);
      FPrintColumns[Length(FPrintColumns)-1].LineR:=VerticalLines;
      Left:=Left+Table.Fields[i].DisplayWidth+1;
      end;
end;

procedure TUniPrinter.AddTableDefinition;
var i: integer;
begin
if Length(FPrintColumns)=0 then Exit;
Add('<TABLE>'); //<COLUMN Left, Width, Alignment, LineL, LineR>
for i:=0 to Length(FPrintColumns)-1 do
    Add('<COLUMN '+IntToStr(FPrintColumns[i].Left)+' ,'+
        IntToStr(FPrintColumns[i].Width)+' ,'+
        IntToStr(Ord(FPrintColumns[i].Alignment))+' ,'+
        IntToStr(Ord(FPrintColumns[i].LineL))+' ,'+
        IntToStr(Ord(FPrintColumns[i].LineR))+'>');
end;

procedure TUniPrinter.PrintTable(Table: TDataSet; HorizontalLines, VerticalLines: Boolean);
var b: TBookmark;
    s, ts: string;
    i, Lines, tLines, CurrentHorLine: integer;
    LinePrinted: boolean;
begin
LinePrinted:=True;
b:=Table.GetBookmark;
Table.DisableControls;
try
GetTableDefinition(Table, VerticalLines);
AddTableDefinition;
Table.First;
CurrentHorLine:=0;
if HorizontalLines then Add('<LINE>');
while not Table.Eof do
      begin
      Lines:=0;
      s:='<ROW ';
      for i:=0 to Table.FieldCount-1 do
          if Table.Fields[i].Visible then
             begin
             if Table.Fields[i].DataType=ftMemo then
                ts:=Table.Fields[i].AsString else
                ts:=Table.Fields[i].DisplayText;
             s:=s+Fdel+ts;
             tLines:=0;
             while ts<>'' do
                   begin
                   CopyFirstN(Table.Fields[i].DisplayWidth, ts);
                   Inc(tLines);
                   end;
             if tLines>Lines then Lines:=tLines;
             end;
      s:=s+'>';
      if FCursorPos.y+Lines+FPageSummaryLinesCount>=FLinesPerPage then
         begin
         if HorizontalLines then Add('<LINE>');
         i:=FCursorPos.x;
         FormFeed;
         x:=i; //aby na nowej stronie zacz od tego samego miejsca
         CurrentHorLine:=0;
         if HorizontalLines then Add('<LINE>');
         end;
      Inc(FCursorPos.y, Lines);
      Add(s);
      Inc(CurrentHorLine);
      LinePrinted:=False;
      if (HorizontalLines  and ((CurrentHorLine mod FPrintHorLineBy)=0)) then
         begin
         Add('<LINE>');
         LinePrinted:=True;
         end;
      if Assigned(FOnTableScroll) then FOnTableScroll(Table);
      Table.Next;
      end;
if HorizontalLines and (not LinePrinted) then Add('<LINE>');
Add('<\TABLE>');
finally
Table.EnableControls;
Table.GotoBookmark(b);
Table.FreeBookmark(b);
end;  //try
end;


procedure TUniPrinter.PrintHeader(Table: TDataSet; HorizontalLines, VerticalLines: Boolean);
var s, ts: string;
    i: integer;
    Lines, tLines: integer;
begin
GetTableDefinition(Table, VerticalLines);
AddTableDefinition;
Lines:=0;
      if HorizontalLines then s:=('<LINE>')
                         else s:='';
      s:=s+'<ROW ';
      for i:=0 to Table.FieldCount-1 do
          if Table.Fields[i].Visible then
             begin
             ts:=Table.Fields[i].DisplayLabel;
             s:=s+Fdel+ts;
             tLines:=0;
             while ts<>'' do
                   begin
                   CopyFirstN(Table.Fields[i].DisplayWidth, ts);
                   Inc(tLines);
                   end;
             if tLines>Lines then Lines:=tLines;
             end;
      s:=s+'>';
      Inc(FCursorPos.y, Lines);
      if GetCurrentLine+FPageSummaryLinesCount>=FLinesPerPage then
         begin
         i:=FCursorPos.x;
         FormFeed;
         x:=i;  //aby na nowej stronie zacz od tego samego miejsca
         Inc(FCursorPos.y, Lines);
         end;
      Add(s);
end;

procedure TUniPrinter.Draw(Page: integer; Canvas: TCanvas);
var i, col, intscale: word;
    ppi: integer;
    ScaleCalc: double;
    s, command: string;
    cx, cy, Xoffset, Yoffset: integer;
    Style: TFontStyles;
    DW, DH, Condensed: Boolean;
    CharLen: integer;
    Alignment: TAlignment;

    procedure HorLine;
    begin
    if cx=0 then Canvas.PenPos:=Point(0, (cy * intscale))
            else Canvas.PenPos:=Point((CharLen*cx)-(CharLen div 2), (cy * intscale));
    Canvas.LineTo((cx+FPrintColumns[Length(FPrintColumns)-1].Left+FPrintColumns[Length(FPrintColumns)-1].Width+1)*CharLen-(CharLen div 2), (cy * intscale));
    end;

    procedure PrintRow;
    var j: integer;
        AllPrinted: boolean;
        ts: string;
    begin
    XOffset:=0;
    AllPrinted:=False;
    while not AllPrinted do
     begin
     AllPrinted:=True;
     for j:=0 to Length(FPrintColumns)-1 do
        begin
        ts:=CopyFirstN(FPrintColumns[j].Width, FPrintColumns[j].Text);
        case FPrintColumns[j].Alignment of
             taLeftJustify: Canvas.TextOut((CharLen*(FPrintColumns[j].Left+cx)), (cy * intscale), ts);
             taRightJustify: Canvas.TextOut((CharLen*(FPrintColumns[j].Left+cx))+(CharLen*FPrintColumns[j].Width)-Canvas.TextWidth(ts), (cy * intscale), ts);
             taCenter: Canvas.TextOut((CharLen*(FPrintColumns[j].Left+cx))+((CharLen*FPrintColumns[j].Width)-(Canvas.TextWidth(ts))) div 2, (cy * intscale), ts);
             end;
        if FPrintColumns[j].LineL then
           begin
           Canvas.PenPos:=Point((CharLen*(FPrintColumns[j].Left+cx))-(CharLen div 2), (cy * intscale));
           if Canvas.PenPos.x<0 then Canvas.PenPos:=Point(0, Canvas.PenPos.y);
           Canvas.LineTo(Canvas.PenPos.x, (cy+1) * intscale);
           end;
        if FPrintColumns[j].LineR then
           begin
           Canvas.PenPos:=Point(((CharLen*(FPrintColumns[j].Left+FPrintColumns[j].Width+cx))+(CharLen div 2)), (cy * intscale));
           Canvas.LineTo(Canvas.PenPos.x, (cy+1) * intscale);
           end;
        if FPrintColumns[j].Text<>'' then AllPrinted:=False;
        end;
     Inc(cy);
     end;
    end;
begin
if Page>FPages.Count then Exit;

cx:=0; cy:=0; Xoffset:=0; Yoffset:=0;
s:=Fpages[Page-1];
Style:=[]; DW:=False; DH:=False; Condensed:=False;
Alignment:=taLeftJustify;
CharLen:=AlterFont(Canvas.Font, DW, DH, Condensed, Style);
ScaleCalc:=GetDeviceCaps(Printer.Handle,LogPixelSY)/(Screen.PixelsPerInch * 3);
intscale:=Round(Printer.PageHeight/(FLinesPerPage*ScaleCalc));

ppi:=Canvas.Font.PixelsPerInch;
Canvas.Brush.Color:=clWhite;
Canvas.Brush.Style:=bsClear;
i:=1;
while i<>0 do
      begin
      i:=Pos('<', s);
      if i<>1 then
         begin
         CharLen:=AlterFont(Canvas.Font, DW, DH, Condensed, Style);
         case Alignment of
              taLeftJustify : Canvas.TextOut((cx+Xoffset)*CharLen, (cy * intscale)+Yoffset, Copy(s, 1, i-1));
              taRightJustify: Canvas.TextOut((cx+Xoffset)*CharLen-Canvas.TextWidth(Copy(s, 1, i-1)), (cy * intscale)+Yoffset, Copy(s, 1, i-1));
              taCenter      : Canvas.TextOut((cx+Xoffset)*CharLen-(Canvas.TextWidth(Copy(s, 1, i-1)) div 2), (cy * intscale)+Yoffset, Copy(s, 1, i-1));
         end; //case
         Xoffset:=Xoffset+Length(Copy(s, 1, i-1));
         Delete(s, 1, i-1);
         end else
         begin
         i:=Pos('>', s);
         command:=Copy(s, 2, i-2);
         if Command='LF' then                    //pooenie
            Inc(cy);
         if command='CR' then
            begin cx:=0; Xoffset:=0; end;
         if (command='TABLE') or (command='\TABLE') then
            FPrintColumns:=nil;
         if Copy(command, 1, 6)='COLUMN' then
            begin
            SetLength(FPrintColumns, Length(FPrintColumns)+1);
            FPrintColumns[Length(FPrintColumns)-1].Left:=IntValue(1, command);
            FPrintColumns[Length(FPrintColumns)-1].Width:=IntValue(2, command);
            FPrintColumns[Length(FPrintColumns)-1].Alignment:=TAlignment(IntValue(3, command));
            FPrintColumns[Length(FPrintColumns)-1].LineL:=Boolean(IntValue(4, command));
            FPrintColumns[Length(FPrintColumns)-1].LineR:=Boolean(IntValue(5, command));
            end;
         if Copy(Command, 1, 3)='ROW' then
            begin
            CharLen:=AlterFont(Canvas.Font, DW, DH, Condensed, Style);
            for col:=0 to Length(FPrintColumns)-1 do
                FPrintColumns[col].Text:=StrValue(col, command);
            PrintRow;
            end;
         if command='LINE' then
            begin
            CharLen:=AlterFont(Canvas.Font, DW, DH, Condensed, Style);
            Horline;
            end;
         if Copy(command, 1, 6)='SETPOS' then
            begin
            Delete(command, 1, 6);
            cx:=StrToInt(copy(command, 1, Pos(',', command)-1));
            cy:=StrToInt(copy(command, Pos(',', command)+1, maxint));
            Xoffset:=0;
            end;
         if command='LB' then
            Dec(cy);
         if Command='B' then                     //style
            Style:=Style+[fsBold];
         if Command='\B' then
            Style:=Style-[fsBold];
         if Command='I' then
            Style:=Style+[fsItalic];
         if Command='\I' then
            Style:=Style-[fsItalic];
         if Command='U' then
            Style:=Style+[fsUnderline];
         if command='\U' then
            Style:=Style-[fsUnderline];
         if Command='C' then
            Condensed:=True;
         if Command='\C' then
            Condensed:=False;
         if Command='DW' then
            DW:=True;
         if Command='\DW' then
            DW:=False;
         if Command='DH' then
            DH:=True;
         if Command='\DH' then
            DH:=False;
         if command='SUPERSCRIPT' then
            begin
            Canvas.Font.PixelsPerInch:=2*ppi;
            YOffset:=0;
            end;
         if command='SUBSCRIPT' then
            begin
            Canvas.Font.PixelsPerInch:=2*ppi;
            Yoffset:=Round(Abs(Canvas.Font.Height/2));
            end;
         if command='NORMALSCRIPT' then
            begin
            Canvas.Font.PixelsPerInch:=ppi;
            YOffset:=0;
            end;
         if Command='LEFT' then
            Alignment:=taLeftJustify;
         if Command='RIGHT' then
            Alignment:=taRightJustify;
         if Command='CENTER' then
            Alignment:=taCenter;
         Delete(s, 1, i);
         end;
      end;
end;

function TUniPrinter.TextDraw(Page: integer): string;
var i, col: word;
    s, command: string;
    cx, cy: integer;
    Alignment: TAlignment;
    tUnderline: boolean;

    procedure TextOut(Text: string);
    begin
    Result:=Result+Text;
    end;

    procedure SetXPos(APos: integer);
    begin
    if tUnderline then
       TextOut(FPrinterDefinition['cEndUnderline']);
    TextOut(FPrinterDefinition['cCarriageReturn']);
    TextOut(MakeStr(' ', cx));
    if tUnderline then
       TextOut(FPrinterDefinition['cStartUnderline']);
{          TextOut(Chr(Lo(Round(APos/15.5))));
         TextOut(Chr(Hi(Round(APos/15.5))));}
//         cx:=APos;
    end;

    procedure HorLine;
    begin
    SetXPos(cx);
    TextOut(FPrinterDefinition['cBackPixelsByN']);
    TextOut(Chr(FPrinterDefinition.ControlValue('FPixelsPerLine')-FPrinterDefinition.ControlValue('FInterlineSpace')));
    TextOut(MakeStr(FPrinterDefinition['cHorLine'][1], FPrintColumns[Length(FPrintColumns)-1].Left+FPrintColumns[Length(FPrintColumns)-1].Width+2));
    TextOut(FPrinterDefinition['cFeedPixelsByN']);
    TextOut(Chr(FPrinterDefinition.ControlValue('FPixelsPerLine')-FPrinterDefinition.ControlValue('FInterlineSpace')));
    end;

    procedure PrintRow;
    var j: integer;
        AllPrinted: boolean;
        ts: string;
    begin
    SetXPos(cx);
    AllPrinted:=False;
    while not AllPrinted do
     begin
     AllPrinted:=True;
     for j:=0 to Length(FPrintColumns)-1 do
        begin
        ts:=CopyFirstN(FPrintColumns[j].Width, FPrintColumns[j].Text);
        if FPrintColumns[j].LineL
           then TextOut(FPrinterDefinition['cVertLine'])
           else if not FPrintColumns[j].LineR then TextOut(' ');
        case FPrintColumns[j].Alignment of
             taLeftJustify:  TextOut(LeftStr(Translate(ts), FPrintColumns[j].Width));
             taRightJustify: TextOut(RightStr(Translate(ts), FPrintColumns[j].Width));
             taCenter:       TextOut(CenterStr(Translate(ts), FPrintColumns[j].Width));
             end;
        if FPrintColumns[j].LineR then
           TextOut(FPrinterDefinition['cVertLine']);
        if FPrintColumns[j].Text<>'' then AllPrinted:=False;
        end;
     TextOut(FPrinterDefinition['cLineFeed']);
     Inc(cy);
     end;
     end;
begin
if Page>FPages.Count then Exit;
s:=Fpages[Page-1];
Result:=FPrinterDefinition['cInit'];
Alignment:=taLeftJustify; cy:=1; cx:=0;
tUnderline:=False;
i:=1;
while i<>0 do
      begin
      i:=Pos('<', s);
      if i<>1 then
         begin
         SetXPos(cx);
         case Alignment of
              taLeftJustify : TextOut(Translate(Copy(s, 1, i-1)));
              taRightJustify: begin
                              TextOut(StringOfChar(FPrinterDefinition['cBackSpace'][1], i-1));
                              TextOut(Translate(Copy(s, 1, i-1)));
                              end;
              taCenter      : begin
                              TextOut(StringOfChar(FPrinterDefinition['cBackSpace'][1], (i-1) div 2));
                              TextOut(Translate(Copy(s, 1, i-1)));
                              end;
              end; //case
         Inc(cx, i-1);
         Delete(s, 1, i-1);
         end else
         begin
         i:=Pos('>', s);
         command:=Copy(s, 2, i-2);
         if Command='LF' then
            begin                   //pooenie
            TextOut(FPrinterDefinition['cLineFeed']);
            Inc(cy);
            end;
         if command='CR' then
            begin
            TextOut(FPrinterDefinition['cCarriageReturn']);
            cx:=0;
            end;
         if (command='TABLE') or (command='\TABLE') then
            FPrintColumns:=nil;
         if Copy(command, 1, 6)='COLUMN' then
            begin
            SetLength(FPrintColumns, Length(FPrintColumns)+1);
            FPrintColumns[Length(FPrintColumns)-1].Left:=IntValue(1, command);
            FPrintColumns[Length(FPrintColumns)-1].Width:=IntValue(2, command);
            FPrintColumns[Length(FPrintColumns)-1].Alignment:=TAlignment(IntValue(3, command));
            FPrintColumns[Length(FPrintColumns)-1].LineL:=Boolean(IntValue(4, command));
            FPrintColumns[Length(FPrintColumns)-1].LineR:=Boolean(IntValue(5, command));
            end;
         if Copy(Command, 1, 3)='ROW' then
            begin
            for col:=0 to Length(FPrintColumns)-1 do
                FPrintColumns[col].Text:=StrValue(col, command);
            PrintRow;
            end;
         if command='LINE' then
            Horline;
         if Copy(command, 1, 6)='SETPOS' then
            begin
            Delete(command, 1, 6);
            cx:=StrToInt(copy(command, 1, Pos(',', command)-1));
//            SetXPos(cx);
            while cy-1<StrToInt(copy(command, Pos(',', command)+1, maxint))
                  do begin
                  TextOut(FPrinterDefinition['cLineFeed']);
                  Inc(cy);
                  end;
            while cy-1>StrToInt(copy(command, Pos(',', command)+1, maxint))
                  do begin
                  TextOut(FPrinterDefinition['cBackPixelsByN']);
                  TextOut(Chr(FPrinterDefinition.ControlValue('FPixelsPerLine')));
                  Dec(cy);
                  end;
            end;
         if command='LB' then
            begin
            TextOut(FPrinterDefinition['cCarriageReturn']);
            TextOut(FPrinterDefinition['cBackPixelsByN']);
            TextOut(Chr(FPrinterDefinition.ControlValue('FPixelsPerLine')));
            Dec(cy);
            end;
         if Command='B' then                     //style
            TextOut(FPrinterDefinition['cStartBold']);
         if Command='\B' then
            TextOut(FPrinterDefinition['cEndBold']);
         if Command='I' then
            TextOut(FPrinterDefinition['cStartItalic']);
         if Command='\I' then
            TextOut(FPrinterDefinition['cEndItalic']);
         if Command='U' then
            begin
            TextOut(FPrinterDefinition['cStartUnderline']);
            tUnderline:=True;
            end;
         if command='\U' then
            begin
            TextOut(FPrinterDefinition['cEndUnderline']);
            tUnderline:=False;
            end;
         if Command='C' then
            TextOut(FPrinterDefinition['cStartCondensed']);
         if Command='\C' then
            TextOut(FPrinterDefinition['cEndCondensed']);
         if Command='DW' then
            TextOut(FPrinterDefinition['cStartDoubleWidth']);
         if Command='\DW' then
            TextOut(FPrinterDefinition['cEndDoubleWidth']);
         if Command='DH' then
            TextOut(FPrinterDefinition['cStartDoubleHeight']);
         if Command='\DH' then
            TextOut(FPrinterDefinition['cEndDoubleHeight']);
         if command='SUPERSCRIPT' then
            TextOut(FPrinterDefinition['cStartSuperscript']);
         if command='SUBSCRIPT' then
            TextOut(FPrinterDefinition['cStartSubscript']);
         if command='NORMALSCRIPT' then
            TextOut(FPrinterDefinition['cNormalScript']);
         if Command='LEFT' then
            Alignment:=taLeftJustify;
         if Command='RIGHT' then
            Alignment:=taRightJustify;
         if Command='CENTER' then
            Alignment:=taCenter;
         Delete(s, 1, i);
         end;
      end;
end;

procedure TUniPrinter.DrawPreview(Sender: TObject);
var MetaFile: TMetaFile;
    ACanvas: TMetaFileCanvas;
begin
MetaFile:=TMetaFile.Create;
MetaFile.Width:=(Printer.PageWidth*300) div GetDeviceCaps(Printer.Handle,LogPixelSX);
MetaFile.Height:=(Printer.PageHeight*300) div GetDeviceCaps(Printer.Handle,LogPixelSY);
ACanvas:=TMetaFileCanvas.Create(MetaFile, Printer.Handle);
Draw(TPreview(TComponent(Sender).Owner).CurrentPage, ACanvas);
ACanvas.Free;
with TPaintBox(Sender) do
     begin
     Canvas.Brush.Color:=clWhite;
     Canvas.FillRect(Rect(0, 0, Width, Height));
     Canvas.StretchDraw(Rect(0, 0, Width, Height), MetaFile);
     end;
MetaFile.Free;
end;

procedure TUniPrinter.PrintFromPreview(Sender: TObject);
begin
Print(1);
end;

procedure TUniPrinter.PreviewPage(Page: integer);
begin
Preview:=TPreview.Create(Self);
Preview.MaxPages:=PagesCount;
Preview.PaintSurface.OnPaint:=DrawPreview;
Preview.PrintProcedure:=PrintFromPreview;
Preview.ShowModal;
end;

procedure Register;
begin
RegisterComponents('System', [TUniPrinter]);
end;

end.
