{=============================================================================
 Program    : LsSystemImages - is a utility program. It access to the Win32
                               system image list that consists of all the
                               images used by the operating system, then
                               displays images in an ImageListview for ease of
                               viewing.  It also allows you to save the selected
                               image to an icon file (in 16- or 256-color) or
                               a bitmap file (in 16-, 256- or Hi-color) for
                               use as application icons or button glyphs.

 Units      : SysImagMain, About

 Version    : 1.00.0  for D4 & D5 under Windows 95, 98 & Me

 Written by : Leo D. Shih  e-mail ldshih@ecn.ab.ca

 Copyright  : (C)2001 Leo D. Shih, All rights reserved.

 =============================================================================}

unit SysImagMain;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ImgList, Buttons, StdCtrls, Spin, ComCtrls, ExtCtrls, ShellAPI, About;

type

  TIconHeader = packed record
    Reserved: Word;
    wType: Word;
    Count: Word;
  end;

  PIconRec = ^TIconRec;
  TIconRec = packed record
    Width: Byte;
    Height: Byte;
    Colors: Word;
    Reserved1: Word;
    Reserved2: Word;
    DIBSize: Longint;
    DIBOffset: Longint;
  end;

  TForm1 = class(TForm)
    ListView1: TListView;
    Panel1: TPanel;
    StatusBar1: TStatusBar;
    SBtnSaveBmp: TSpeedButton;
    SaveDialog1: TSaveDialog;
    Label1: TLabel;
    SBtnReload: TSpeedButton;
    GroupBox1: TGroupBox;
    RBtnLargeIcon: TRadioButton;
    RBtnSmallIcon: TRadioButton;
    GroupBox2: TGroupBox;
    RBtn4bit: TRadioButton;
    RBtn8bit: TRadioButton;
    SBtnSaveIcon: TSpeedButton;
    Panel2: TPanel;
    Image1: TImage;
    RBtn24bit: TRadioButton;
    procedure CreateImages;
    procedure FormCreate(Sender: TObject);
    procedure LoadImages;
    procedure SBtnReloadClick(Sender: TObject);
    procedure SBtnSaveIconClick(Sender: TObject);
    procedure SBtnSaveBmpClick(Sender: TObject);
    function GetImage(ImgIndex: integer; Img: TBitmap): TBitmap;
    procedure InitializeBitmapInfoHeader(Bitmap: HBITMAP;
      var bi: TBitmapInfoHeader; Colors: Integer);
    procedure GetDIBSizes(Bitmap: HBITMAP; var InfoHeaderSize: DWORD;
      var ImageSize: DWORD; Colors: Integer);
    function GetDIB(Bitmap: HBITMAP; Palette: HPALETTE; var BitmapInfo;
      var Bits; Colors: Integer): Boolean;
    procedure SaveToStream(Stream: TStream; Icon: HIcon; SaveLength: Boolean;
      Colors: Integer);
    procedure GroupBox2Click(Sender: TObject);
    procedure ListView1MouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);

  private
    { Private declarations }
    TypeName: string;
    ImgX, ImgY: integer;
  public
    { Public declarations }
  end;

var
  Form1: TForm1;

implementation


{$R *.DFM}

procedure TForm1.FormCreate(Sender: TObject);
begin
  ImgX := GetSystemMetrics(SM_CXICON);
  ImgY := GetSystemMetrics(SM_CYICON);
  Panel2.Left := 56;
  Panel2.Height := ImgY + 4;
  Panel2.Width := ImgX + 4;
  Panel2.Top := (Panel1.Height - Panel2.Height) div 2;
  RBtnLargeIcon.Checked := True;
  RBtn8bit.Checked := True;
  CreateImages;
  LoadImages;
end;  //FormCreate

procedure TForm1.CreateImages;
var
  SysIL: UINT;
  Sfi: TSHFileInfo;
  LImgList, SImgList: TImageList;
begin
  LImgList := TImageList.Create(Form1);
  SysIL := SHGetFileInfo('', 0, Sfi, SizeOf(Sfi), SHGFI_SYSICONINDEX
    or SHGFI_TYPENAME or SHGFI_LARGEICON);
  if SysIL <> 0 then
  begin
    LImgList.Handle := SysIL;
    LImgList.ShareImages := True;
  end;
  SImgList := TImageList.Create(Form1);
  SysIL := SHGetFileInfo('', 0, Sfi, SizeOf(Sfi), SHGFI_SYSICONINDEX
    or SHGFI_TYPENAME or SHGFI_SMALLICON);
  if SysIL <> 0 then
  begin
    SImgList.Handle := SysIL;
    SImgList.ShareImages := True;
  end;
  ListView1.LargeImages := LImgList;
  ListView1.SmallImages := SImgList;
end;  //CreateImages

procedure TForm1.LoadImages;
var
  New: TListItem;
  cnt,
    i: integer;
  sfi: TSHFileInfo;
begin
  cnt := 0;
  if RBtnLargeIcon.Checked then
    cnt := ListView1.LargeImages.Count
  else if RBtnSmallIcon.Checked then
    cnt := ListView1.SmallImages.Count;
  ListView1.Items.BeginUpdate;
  ListView1.Items.Clear;
  try
    for i := 0 to cnt do
    begin
      sfi.iIcon := i;
      SHGetFileInfo('', 0, sfi, SizeOf(sfi), i or SHGFI_TYPENAME or SHGFI_DISPLAYNAME);
      TypeName := sfi.szTypeName;

      New := ListView1.Items.Add;
      if RBtnLargeIcon.Checked then
        new.Caption := IntToStr(i) + #13 + TypeName
      else
        new.Caption := IntToStr(i);
      New.ImageIndex := i;
    end;
  finally
    ListView1.Items.EndUpdate;
  end;
end;  //LoadImages

function TForm1.GetImage(ImgIndex: integer; Img: TBitmap): TBitmap;
begin
  if RBtnLargeIcon.Checked then
    ListView1.LargeImages.GetBitmap(ImgIndex, Img)
  else if RBtnSmallIcon.Checked then
    ListView1.SmallImages.GetBitmap(ImgIndex, Img);
  Result := Img;
end;  //GetImage

procedure TForm1.SBtnReloadClick(Sender: TObject);
begin
  ListView1.Items.Clear;
  if RBtnLargeIcon.Checked then
    ListView1.ViewStyle := vsIcon
  else if RBtnSmallIcon.Checked then
    ListView1.ViewStyle := vsList;
  LoadImages;
  Image1.Picture := nil;
end;  //SBtnReloadClick

procedure TForm1.ListView1MouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  if Button = mbRight then
    AboutForm.ShowModal
  else if Button = mbLeft then
  begin
    if ListView1.Selected = nil then
      exit;
    Image1.Picture := nil;
    with Image1.Picture do
    begin
      Bitmap := GetImage(Listview1.Selected.ImageIndex, Bitmap);
      if Bitmap.Width > ImgX then
        Panel2.Width := Bitmap.Width + 4;
      if Bitmap.Height > ImgY then
      begin
        Panel2.Height := Bitmap.Height + 4;
        Panel2.Top := (Panel1.Height - Panel2.Height) div 2;
      end;
    end;
  end;
end;  //ListView1MouseDown

procedure TForm1.GroupBox2Click(Sender: TObject);
begin
  Image1.Picture := nil;
  ListView1.Selected := nil;
end;  //GroupBox2Click

procedure TForm1.SBtnSaveBmpClick(Sender: TObject);
var
  Bmp2: TBitMap;
  Image2: TImage;
begin
  if ListView1.Selected = nil then
  begin
    MessageDlg('No image selected !', mtError, [mbOK], 0);
    exit;
  end;

  Bmp2 := TBitmap.Create;
  Image2 := TImage.Create(Self);
  Image2.Visible := False;
  if RBtn4Bit.Checked then
    Bmp2.PixelFormat := pf4bit
  else if RBtn8Bit.Checked then
    Bmp2.PixelFormat := pf8bit
  else if RBtn24Bit.Checked then
    Bmp2.PixelFormat := pf24bit;
  Bmp2 := GetImage(ListView1.Selected.ImageIndex, bmp2);
  Image2.Picture.Assign(Bmp2);

//  Image1.Picture.Bitmap.IgnorePalette := True;
  SaveDialog1.Title := 'Save slected image to Bitmap File';
  SaveDialog1.DefaultExt := '.bmp';
  SaveDialog1.Filter := 'Bitmap Files(*.bmp)|*.bmp';

  if SaveDialog1.Execute then
    Image2.Picture.SaveToFile(SaveDialog1.FileName);
end;  //SBtnSaveBmpClick

procedure TForm1.SBtnSaveIconClick(Sender: TObject);
var
  Icon1: TIcon;
  Colors: integer;
  FStream: TFileStream;
begin
  if ListView1.Selected = nil then
  begin
    MessageDlg('No image selected !', mtError, [mbOK], 0);
    exit;
  end;
  Image1.Picture := nil;
  Colors := 0;
  Icon1 := TIcon.Create;
  Icon1.Width := ImgX;
  Icon1.Height := ImgY;

  ListView1.LargeImages.GetIcon(ListView1.Selected.ImageIndex, Icon1);
  Image1.Picture.Assign(Icon1);

  SaveDialog1.Title := 'Save slected image to Icon File';
  SaveDialog1.DefaultExt := '.ico';
  SaveDialog1.Filter := 'Icon Files(*.ico)|*.ico';
  if SaveDialog1.Execute then
  begin
    if RBtn4Bit.Checked then
      Colors := 16
    else if (RBtn8Bit.Checked) or (RBtn24Bit.Checked) then
      Colors := 256;
    FStream := TFileStream.Create(SaveDialog1.FileName, fmCreate or fmOpenWrite);
    try
      SaveToStream(FStream, Icon1.Handle, False, Colors);
    finally
      FStream.Free;
    end;
  end;
end;  //SBtnSaveIconClick

procedure TForm1.InitializeBitmapInfoHeader(Bitmap: HBITMAP;
  var bi: TBitmapInfoHeader; Colors: Integer);
var
  ds: TDIBSection;
  Bytes: Integer;

  function BytesPerScanline(PixelsPerScanline, BitsPerPixel,
    Alignment: Longint): Longint;
  begin
    Dec(Alignment);
    Result := ((PixelsPerScanline * BitsPerPixel) + Alignment) and not Alignment;
    Result := Result div 8;
  end;

begin
  ds.dsbmih.biSize := 0;
  Bytes := GetObject(Bitmap, SizeOf(ds), @ds);
  if Bytes = 0 then
    ShowMessage('Invalid Bitmap')
  else if (Bytes >= (sizeof(ds.dsbm) + sizeof(ds.dsbmih))) and
    (ds.dsbmih.biSize >= DWORD(sizeof(ds.dsbmih))) then
    bi := ds.dsbmih
  else
  begin
    FillChar(bi, sizeof(bi), 0);
    with bi, ds.dsbm do
    begin
      biSize := SizeOf(bi);
      biWidth := bmWidth;
      biHeight := bmHeight;
    end;
  end;
  if Colors <> 0 then
    case Colors of
      2: bi.biBitCount := 1;
      16: bi.biBitCount := 4;
      256: bi.biBitCount := 8;
    end
  else bi.biBitCount := ds.dsbm.bmBitsPixel * ds.dsbm.bmPlanes;
  bi.biPlanes := 1;
  if bi.biSizeImage = 0 then
    bi.biSizeImage := BytesPerScanLine(bi.biWidth, bi.biBitCount, 32) *
      Abs(bi.biHeight);
end; //InitializeBitmapInfoHeader

procedure TForm1.GetDIBSizes(Bitmap: HBITMAP;
  var InfoHeaderSize: DWORD; var ImageSize: DWORD; Colors: Integer);
var
  bi: TBitmapInfoHeader;
begin
  InitializeBitmapInfoHeader(Bitmap, bi, Colors);
  if bi.biBitCount > 8 then
  begin
    InfoHeaderSize := SizeOf(TBitmapInfoHeader);
    if (bi.biCompression and BI_BITFIELDS) <> 0 then
      Inc(InfoHeaderSize, 12);
  end
  else
    InfoHeaderSize := SizeOf(TBitmapInfoHeader) + SizeOf(TRGBQuad) *
      (1 shl bi.biBitCount);
  ImageSize := bi.biSizeImage;
end; //GetDIBSizes

function TForm1.GetDIB(Bitmap: HBITMAP; Palette: HPALETTE;
  var BitmapInfo; var Bits; Colors: Integer): Boolean;
var
  OldPal: hPalette;
  DC: HDC;
begin
  InitializeBitmapInfoHeader(Bitmap, TBitmapInfoHeader(BitmapInfo), Colors);
  OldPal := 0;
  DC := CreateCompatibleDC(0);
  try
    if Palette <> 0 then
    begin
      OldPal := SelectPalette(DC, Palette, False);
      RealizePalette(DC);
    end;
    Result := GetDIBits(DC, Bitmap, 0, TBitmapInfoHeader(BitmapInfo).biHeight, @Bits,
      TBitmapInfo(BitmapInfo), DIB_RGB_COLORS) <> 0;
  finally
    if OldPal <> 0 then SelectPalette(DC, OldPal, False);
    DeleteDC(DC);
  end;
end; //GetDIB

procedure TForm1.SaveToStream(Stream: TStream; Icon: HIcon;
  SaveLength: Boolean; Colors: Integer);
var
  ii: TIconInfo;
  MonoInfoSize, ColorInfoSize: DWORD;
  MonoBitsSize, ColorBitsSize: DWORD;
  MonoInfo, MonoBits, ColorInfo, ColorBits: Pointer;
  ci: TIconHeader;
  List: TIconRec;
  Length: Longint;
begin
  FillChar(ci, SizeOf(ci), 0);
  FillChar(List, SizeOf(List), 0);
  if not GetIconInfo(Icon, ii) then
    ShowMessage('ERROR - unable to retrive IconInfo');
  try
    GetDIBSizes(ii.hbmMask, MonoInfoSize, MonoBitsSize, 2);
    GetDIBSizes(ii.hbmColor, ColorInfoSize, ColorBitsSize, Colors);
    MonoInfo := nil;
    MonoBits := nil;
    ColorInfo := nil;
    ColorBits := nil;
    try
      MonoInfo := AllocMem(MonoInfoSize);
      MonoBits := AllocMem(MonoBitsSize);
      ColorInfo := AllocMem(ColorInfoSize);
      ColorBits := AllocMem(ColorBitsSize);
      GetDIB(ii.hbmMask, 0, MonoInfo^, MonoBits^, 2);
      GetDIB(ii.hbmColor, 0, ColorInfo^, ColorBits^, Colors);
      if SaveLength then
      begin
        Length := SizeOf(ci) + SizeOf(List) + ColorInfoSize +
          ColorBitsSize + MonoBitsSize;
        Stream.Write(Length, SizeOf(Length));
      end;
      with ci do
      begin
        ci.wType := 1; //RC3_ICON;
        ci.Count := 1;
      end;
      Stream.Write(ci, SizeOf(ci));
      with List, PBitmapInfoHeader(ColorInfo)^ do
      begin
        Width := biWidth;
        Height := biHeight;
        Colors := biPlanes * biBitCount;
        DIBSize := ColorInfoSize + ColorBitsSize + MonoBitsSize;
        DIBOffset := SizeOf(ci) + SizeOf(List);
      end;
      Stream.Write(List, SizeOf(List));
      with PBitmapInfoHeader(ColorInfo)^ do
        Inc(biHeight, biHeight); { color height includes mono bits }
      Stream.Write(ColorInfo^, ColorInfoSize);
      Stream.Write(ColorBits^, ColorBitsSize);
      Stream.Write(MonoBits^, MonoBitsSize);
    finally
      FreeMem(ColorInfo, ColorInfoSize);
      FreeMem(ColorBits, ColorBitsSize);
      FreeMem(MonoInfo, MonoInfoSize);
      FreeMem(MonoBits, MonoBitsSize);
    end;
  finally
    DeleteObject(ii.hbmColor);
    DeleteObject(ii.hbmMask);
  end;
end; //SaveToStream


end.

