{*******************************************************}
{                                                       }
{       Component TProgramTracker                       }
{                                                       }
{       Copyright (c) 2001 Fredrik hman                }
{                                                       }
{*******************************************************}



unit ProgramTracker;

interface

uses
  Windows, ShellAPI, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  stdctrls;

type
  TTrackingState = (tsTrackingStarted, tsTracking, tsTrackingAborted, tsTrackingFinished, tsTrackingError);
  { See "TCustomProgramTracker.CallCreateProcess(.." concerning this type. }
  TExecuteProgramWay = ({This means that file with path and commands
                        (switches) will be passed as separate parameters
                        in the call to "CreateProcess".}
                       ewFileName,
                       {Same as epFileName except that path will be
                        changed to the current directory of this process.}
                       ewCurDir,
                       {Same as epFileName except that path will be
                        changed to systemdirectory.}
                       ewSysDir,
                       {Same as epFileName except that path will be
                        changed to windowsdirectory.}
                       ewWinDir,
                       {Means that file with path and commands
                        (switches) will be concatenated and passed as
                        one command in the call to "CreateProcess"}
                       ewParameter,
                       {Same as epParameter except that path will be
                        changed to the current dir of this process.}
                       ewParameterCurDir,
                       {Same as epParameter except that path will be
                        changed to systemdirectory.}
                       ewParameterSysDir,
                       {Same as epParameter except that path will be
                        changed to windowsdirectory.}
                       ewParameterWinDir);
  TExecuteProgramWays = set of TExecuteProgramWay;

  TTrackingProgress = procedure(Sender: TObject; TrackingMessage: string; TrackingState: TTrackingState; TrackingCode: Integer) of object;

  TTrackingProgressMessage = procedure(TrackingMessage: string; TrackingState: TTrackingState; TrackingCode: Integer) of object;

  EExecuteProgram = class(Exception);


  TAwaitProgramResult = class(TThread)
  private
    { Private declarations }
   {ID of process to await.}
    FProcID: DWORD;
   {Pointer to be able to send messages about progress of process. Points
    to a procedure for sending messages from this thread about start of,
    progress of and end of remote process.}
    FFeedBack: TTrackingProgressMessage;
   {For containig messages.}
    FMessage: string;
   {Will get errorcode if calls to external program fails.}
   {Else container of process termination code.}
    FResult: DWORD;
   {For flagging the current state of tracking.}
    FTrackState: TTrackingState;
   {Determines with wich interval this thread will make inquiries about state
    of remote program executed in milliseconds.}
    FInterval: Integer;
   {If an error occurs this function is called in an attempt to translate
    cryptic errorcodes from system to a hopefully more informative meaning.}
    function GetErrorMessage(ErrorCode: Integer): string;
    procedure SendMessage;
    procedure SetMessage(const Mess: string; const TrackState: TTrackingState; ResultCode: Integer);
    procedure SetInterval(Value: Integer);
    function GetInterval: Integer;
    function GetProcID: DWORD;
  protected
    procedure Execute; override;
   {Stops this thread and prevents it from sending any more messages. This is
    useful for stopping and freeing this thread in a polite way without any
    further operations.}
    procedure StopTrackingAndNoMoreMessages;
   {Stops and frees this thread in a polite way. But before destruction a
    message is sent back via the "FeedBack"-reference passed in constructor.
    The same thing will happen if derived procedure Terminate is called.}
    procedure StopTracking;
   {This constructor executes thread immidiately after construction.
    Params: "FeedBack", see member "FFeedBack". "ProcID" property "ProcessId".
    "TrackInterval", see property "TrackingInterval".}
    constructor Create(FeedBack: TTrackingProgressMessage; ProcID : DWORD; TrackInterval: Integer); reintroduce;
   {Read or set this property to determin at wich speed this thread makes inquiries
    about state of remote program executed. No call to Suspend nescessary.}
    property TrackingInterval: Integer read GetInterval write SetInterval;
   {Returns system-ID of process this thread is tracking.}
    property ProcessID: DWORD read GetProcID;
  end;

  TCustomProgramTracker = class(TComponent)
  private
    //Contains optional settings for path and commands of executable.
    FVariousExecutingWays: TExecuteProgramWays;

    //Pointer for recieving messages from waiting thread..
    FTrackingMessage: TTrackingProgressMessage;
    //Reference to witing thread.
    FWaitThread: TAwaitProgramResult;
    //Containers for keeping handles to remote program..
    FProcessId: DWORD;
    FThreadId: DWORD;
    FProcessHandle: THandle;
    FThreadHandle: THandle;
    FProgramExecuted: Boolean;

    FErrorMessage: string;
    FLastMessage: string;

    FProgramFinished: TTrackingProgress;
    FProgramStarted: TTrackingProgress;
    FProgramProgress: TTrackingProgress;
    FExitCode: string;
    FTrack: Boolean;
    FTrackInterval: Integer;
    FExecutable: string;
    FParams: TStringList;
    FProgramShowState: TWindowState;
    procedure SetTrack(Value: Boolean);
    procedure SetTrackInterval(Value: Integer);
    procedure SetExecutable(Value: string);
    procedure SetParams(Value: TStrings);
    function GetParams: TStrings;
   {FTrackingMessage is initiated to point to this.}
    procedure OnMessageFromThread(TrackingMessage: string; TrackingState: TTrackingState; TrackingCode: Integer);
   {Fires the executable with it's params}
    function ExecuteFile(const FileName: string; Params: TStrings;
      var MessageString, ErrorString: String; var ProcessId, ThreadId: DWORD;
      var ProcessHandle, ThreadHandle: THandle; ShowState: TWindowState): Boolean;
    function CallCreateProcess(const ExecuteWay: TExecuteProgramWay;
      var FileName, Params: string;
      var Startinfo: TStartupInfo; var Procinfo: TProcessInformation;
      var ErrorCode: DWORD): Boolean;
  protected
   {Call this to fire remote program when properties "Executable",
    "Params" and so on is set. If property "Track" is set to true this
    component will await progress and finishing of remote program.
    Throws EExecuteProgram exception.}
    function ExecuteProgram: Boolean; virtual;//Throws exceptions
    //Plan to build support for this.
    procedure CloseProgram;
   {These properties is for making it easy for programmer to retrieve id's and
    handles to remote program and it's main thread if programmer
    wants to make further Windows-calls on the program. When read, they
    will return nothing useful unless remote program has been executed.}
    property ProcessId: DWORD read FProcessId;
    property ThreadId: DWORD read FThreadId;
    property ProcessHandle: THandle read FProcessHandle;
    property ThreadHandle: THandle read FThreadHandle;
    property ExecutionAlternatives: TExecuteProgramWays read FVariousExecutingWays write FVariousExecutingWays default [ewFileName];
   {When program is executed, this determines in wich state
    program runned will be shown.}
    property ProgramShowState: TWindowState read FProgramShowState write FProgramShowState default wsNormal;
   {Name and path of executable of remote program to run.}
    property Executable: string read FExecutable write SetExecutable;
   {For providing switches/commands to executable when runned.}
    property Params: TStrings read GetParams write SetParams;
   {Will contain code returned from remote program that was runned if any.}
    property ExitCode: string read FExitCode;
   {Will contain any errormessage.}
    property ErrorMessage: string read FErrorMessage;
   {Will contain last message gotten when tracking remote program. Same messages
    as gotten from parameter TrackingMessage in the events of this class.
    Except for messages occurring on failures wich is contained in property "ErrorMessage".}
    property LastMessage: string read FLastMessage;
   {Determines if this component will await the end of remote program runned.
    If this is set to false after a remote program is executed the handle to
    that program will be released and therefore tracking will be discontinued.}
    property Track: Boolean read FTrack write SetTrack default true;
   {This determines in tenth of seconds how often OnProgramProgress event will occur.
    A value of 100 = 10 seconds. Default value = 10.
    To save system performance, 1 second interval and longer is recommended.
    Possible to adjust even during tracking of a program in progress.
    A value of 0 means no OnProgramProgress will occur.}
    property TrackingInterval: Integer read FTrackInterval write SetTrackInterval default 10;
   {Occurs when remote program runned is finished and property Track is true.
    Examine property ExitCode if an exitcode is expected from remote program.
    Parameter "TrackingMessage" will contain the same code.}
    property OnProgramFinished: TTrackingProgress read FProgramFinished write FProgramFinished;
   {Occurs right after a successfull start of remote program. Examine property
    "LastMessage" or parameter "TrackingMessage" for a possible message from startup.}
    property OnProgramStarted: TTrackingProgress read FProgramStarted write FProgramStarted;
   {This will occur if property Track is set to true and value of
    property TrackingInterval is above 0.
    Will occur repeatedly with an interval set by property TrackingInterval.
    Parameter "TrackingMessage" may contain a message.}
    property OnProgramProgress: TTrackingProgress read FProgramProgress write FProgramProgress;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
  end;

  TProgramTracker = class(TCustomProgramTracker)
  public
    function ExecuteProgram: Boolean; override;
    property ExitCode;
    property ErrorMessage;
    property LastMessage;
    property ProcessId;
    property ThreadId;
    property ProcessHandle;
    property ThreadHandle;
  published
    //Published properties...
    property ExecutionAlternatives;
    property Params;
    property Executable;
    property Track;
    property TrackingInterval;
    property ProgramShowState;
    //Events
    property OnProgramFinished;
    property OnProgramStarted;
    property OnProgramProgress;
  end;

function GetMeaningOfErrorCode(const ErrorCode: DWORD): string;
function TranslateError(Err: DWORD): String;



implementation

{ TAwaitProgramResult }

constructor TAwaitProgramResult.Create(FeedBack: TTrackingProgressMessage;
              ProcID : DWORD; TrackInterval: Integer);
begin
  //Hold execution of this thread until after creation.
  inherited Create(true);
  FFeedBack := FeedBack;
  FProcID := ProcID;
  FMessage := '';
  FResult := 0;
  FInterval := TrackInterval;
  FTrackState := tsTrackingStarted;
  //This thread will free it self when execution is finished.
  FreeOnTerminate := true;
  Suspended := false; //Activate execution thread..
end;


procedure TAwaitProgramResult.Execute;
var PoinResult : ^DWORD;
    Hand : THandle;
    ResultCode, Wait : DWORD;
    SysMess: string;
    ErrorString: string;
    ErrorCode: DWORD;
begin
  //Get a synchronized handle to process (=FProcID) that we should await.
  //PROCESS_ALL_ACCESS = (STANDARD_RIGHTS_REQUIRED or SYNCHRONIZE or $03)
  Hand := OpenProcess(PROCESS_ALL_ACCESS, False, FProcID);
  if Hand <> NULL then //Did we get a valid handle to use?
  begin
    //Waiting sequence..
    SetMessage('ppnat ett handtag fr vervakning av process.', tsTrackingStarted, 0);
    Wait := WAIT_TIMEOUT;
    while Wait = WAIT_TIMEOUT do
    begin
      SetMessage('Awaiting progress of process', tsTracking, 0);
      if Terminated then
      begin
        CloseHandle(Hand);
        SetMessage('Cancelled tracking of process.', tsTrackingAborted, 0);
        Exit;
      end;
      Wait := WaitForSingleObject(Hand, FInterval);
    end;
    if Wait = WAIT_FAILED then //Did awaiting program fail?
    begin
      ErrorCode := GetLastError;
      SysMess := GetMeaningOfErrorCode(ErrorCode);
      if SysMess <> '' then//Did we get any errormessage from system?
      begin
        ErrorString := 'Failed to track process:' +
                       IntToStr(FProcID) + #13#10 +
                       'Errormessage:' + #13#10 +
                       Sysmess;
      end
      else
      begin
        ErrorString := 'Failed to track process: ' +
                       IntToStr(FProcID);
      end;
      SetMessage('Resultat oknt.' + #13#10 + ErrorString, tsTrackingError, ErrorCode)
    end
    else //If awaiting program succeeded.
    begin
      PoinResult := @ResultCode; //Assign a pointer to resultcontainer..
      //Get result.
      if GetExitCodeProcess(Hand, PoinResult^) then
        SetMessage('Processen avslutad, avslutningskod framgngsrikt hmtad.', tsTrackingFinished, ResultCode)
      else
      begin
        ErrorCode := GetLastError;
        SysMess := GetMeaningOfErrorCode(ErrorCode);
        SetMessage('Process finished, but failed to fetch exitcode.' + #13#10 +
                   'Errormessage: ' + SysMess, tsTrackingError, ErrorCode);
      end;
    end;
    CloseHandle(Hand);
  end
  else //If not Hand <> NULL ..
    SetMessage('Unknown result from start of process, failed to open a handle to process.', tsTrackingError, GetLastError);
end;


function TAwaitProgramResult.GetErrorMessage(ErrorCode: Integer): string;
var Buffer : PChar;
    SysMess : String;
begin
  Buffer := nil;
  SysMess := '';
  //Try to get meaning of errorcode recieved in "ErrorCode".
  if FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER or FORMAT_MESSAGE_FROM_SYSTEM,
       nil, DWORD(ErrorCode), GetUserDefaultLangID, Buffer, 0, nil) > 0 then
    SysMess := Buffer^;
  LocalFree(DWORD(Buffer^));
  if SysMess <> '' then
    Result := SysMess + #13#10 + ' Errorcode: ' + IntToStr(ErrorCode)
  else
    Result := 'Errorcode: ' + IntToStr(ErrorCode);
end;

procedure TAwaitProgramResult.SetMessage(const Mess: string;
            const TrackState: TTrackingState; ResultCode: Integer);
begin
  FMessage := Mess;
  FTrackState := TrackState;
  FResult := ResultCode;
  if FTrackState = tsTrackingError then
  begin
    FMessage := FMessage + #13#10 + GetErrorMessage(FResult);
  end;
  Synchronize(SendMessage);
end;

//This procedure must only be called via a Synchronize call.
procedure TAwaitProgramResult.SendMessage;
begin
  if Assigned(FFeedBack) then
    FFeedBack(FMessage, FTrackState, FResult);
end;

//Call this procedure to let this thread die politely without any collisions.
//Also look at the StopTrackingAndNoMoreMessages procedure.
procedure TAwaitProgramResult.StopTracking;
begin
  Terminate;
end;

//Call this to stop this thread and prevent it from sending messages
procedure TAwaitProgramResult.StopTrackingAndNoMoreMessages;
begin
  FFeedBack := nil;
  Terminate;
end;

function TAwaitProgramResult.GetInterval: Integer;
begin
  Suspended := true;
  Result := FInterval;
  Suspended := false;
end;

procedure TAwaitProgramResult.SetInterval(Value: Integer);
begin
  if Value > 0 then
  begin
    Suspended := true;
    FInterval := Value;
    Suspended := false;
  end;
end;

function TAwaitProgramResult.GetProcID: DWORD;
begin
  Suspended := true;
  Result := FProcID;
  Suspended := false;
end;

{ END TAwaitProgramResult }


{ TProgramTracker }

function TProgramTracker.ExecuteProgram: Boolean;
begin
  Result := inherited ExecuteProgram;
end;

{ TCustomProgramTracker }

//This function executes the file with searchpath "FileName" with
//parameters in given "Params". Internally called by function "fireSendings"
//If function succeeds (returnvalue true), code is an ID of process
//executed and referenceparam "errorstring" will not be changed.
//If not (returnvalue false), code is an errorcode and
//referenceparam "errorstring" will contain an errormessage.
procedure TCustomProgramTracker.CloseProgram;
begin
//
end;

constructor TCustomProgramTracker.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FVariousExecutingWays := [ewFileName, ewCurDir, ewSysDir, ewWinDir,
                            ewParameter, ewParameterCurDir,
                            ewParameterSysDir, ewParameterWinDir];
  FParams := TStringList.Create;
  FTrack := true;
  FTrackInterval := 10;
  FProgramShowState := wsNormal;
  FProgramExecuted := false;
  FTrackingMessage := OnMessageFromThread;
  //ewFileName, ewCurDir, ewSysDir, ewWinDir, ewParameter, ewParameterCurDir, ewParameterSysDir, ewParameterWinDir
  FVariousExecutingWays := [ewFileName];
end;

destructor TCustomProgramTracker.Destroy;
begin
  FParams.Free;
  //Check if the reomte program still runs and perform cleanups according to that.
  if FProgramExecuted then
  begin
    if FTrack then
      FWaitThread.StopTrackingAndNoMoreMessages;
    //Free handles to program executed.
    CloseHandle(FProcessHandle);
    CloseHandle(FThreadHandle);
  end;
//  MessageDlg('CustomProgTrack destroy', mtInformation, [mbOk], 0);
  inherited Destroy;
end;

function TCustomProgramTracker.ExecuteFile(const FileName: string; Params: TStrings;
  var MessageString, ErrorString: String; var ProcessId, ThreadId: DWORD;
  var ProcessHandle, ThreadHandle: THandle; ShowState: TWindowState): Boolean;
var
  Startinfo : TStartupInfo;
  Procinfo : TProcessInformation;
  Sysmess: string;
  I: Integer;
  TmpParams, TmpFileName: string;
  ErrorCode: DWORD;
begin
  Result := false;
  //Some nescessary initialization..
  ZeroMemory(@Startinfo, sizeof(Startinfo));//Makes sure everything that we have not specified in this structure will be zero.
  ZeroMemory(@Procinfo, sizeof(Procinfo));//Makes sure everything that we have not specified in this structure will be zero.
  Startinfo.cb := sizeof(Startinfo);
  Startinfo.dwFlags := STARTF_USESHOWWINDOW;
  case ShowState of
    wsNormal: Startinfo.wShowWindow := SW_SHOWNORMAL;
    wsMaximized: Startinfo.wShowWindow := SW_SHOWMAXIMIZED;
    wsMinimized: Startinfo.wShowWindow := SW_SHOWMINIMIZED;
  end;

  TmpFileName := FileName;
  TmpParams := '';
  for I:=0 to Params.Count-1 do
    TmpParams := TmpParams + Params[I] + ' ';
  TmpParams := Copy(TmpParams, 1, Length(TmpParams)-1);

 {Try various ways of executing remote program.}
  ErrorCode := 0;
  MessageString := '';
  //FileName and Commands separate
  if ewFileName in FVariousExecutingWays then
  begin
    if CallCreateProcess(ewFileName, TmpFileName, TmpParams,
         Startinfo, Procinfo, ErrorCode) then
      MessageString := 'File: ' + TmpFileName + #13#10 + 'Command: ' + TmpParams;
  end;
  //Execute from systemdirectory
  if ((ewSysDir in FVariousExecutingWays) and (MessageString = '')) then
  begin
    if CallCreateProcess(ewSysDir, TmpFileName, TmpParams,
       Startinfo, Procinfo, ErrorCode) then
      MessageString := 'File: ' + TmpFileName + #13#10 + 'Command: ' + TmpParams;
  end;
  //Try windowsdirectory
  if ((ewWinDir in FVariousExecutingWays) and (MessageString = '')) then
  begin
    if CallCreateProcess(ewWinDir, TmpFileName, TmpParams,
       Startinfo, Procinfo, ErrorCode) then
      MessageString := 'File: ' + TmpFileName + #13#10 + 'Command: ' + TmpParams;
  end;
  //Try the current dir of this process.
  if ((ewCurDir in FVariousExecutingWays) and (MessageString = '')) then
  begin
    if CallCreateProcess(ewCurDir, TmpFileName, TmpParams,
       Startinfo, Procinfo, ErrorCode) then
      MessageString := 'Command: ' + TmpFileName + ' ' + TmpParams;
  end;
  //As Command only
  if ((ewParameter in FVariousExecutingWays) and (MessageString = '')) then
  begin
    if CallCreateProcess(ewParameter, TmpFileName, TmpParams,
       Startinfo, Procinfo, ErrorCode) then
      MessageString := 'Command: ' + TmpParams;
  end;
  //As command only with path to systemdirectory
  if ((ewParameterSysDir in FVariousExecutingWays) and (MessageString = '')) then
  begin
    if CallCreateProcess(ewParameterSysDir, TmpFileName, TmpParams,
       Startinfo, Procinfo, ErrorCode) then
      MessageString := 'Command: ' + TmpParams;
  end;
  //As command only with path to windowsdirectory
  if ((ewParameterWinDir in FVariousExecutingWays) and (MessageString = '')) then
  begin//epParameterCurDir
    if CallCreateProcess(ewParameterWinDir, TmpFileName, TmpParams,
       Startinfo, Procinfo, ErrorCode) then
      MessageString := 'Command: ' + TmpParams;
  end;
  //As command only with path to the current directory of this application.
  if ((ewParameterCurDir in FVariousExecutingWays) and (MessageString = '')) then
  begin
    if CallCreateProcess(ewParameterCurDir, TmpFileName, TmpParams,
       Startinfo, Procinfo, ErrorCode) then
      MessageString := 'Command: ' + TmpParams;
  end;
  //Actions taken depending on success or not...
  if MessageString <> '' then//CreateProcess successful..
  begin
    ProcessId := Procinfo.dwProcessId;//This functon will return ID of process created.
    ProcessHandle := Procinfo.hProcess;
    ThreadId := Procinfo.dwThreadId;
    ThreadHandle := Procinfo.hThread;
    MessageString := MessageString + #13#10 + 'Program started successfully.';
    Result := true;
  end
  else//CreateProcess failed..
  begin
    SysMess := GetMeaningOfErrorCode(ErrorCode);
    if SysMess <> '' then//Did we get any errormessage from system?
    begin
      ErrorString := 'Failed to start:' + #13#10 +
                     Filename + ' ' + TmpParams + #13#10 +
                     'Errorcode: ' + IntToStr(ErrorCode) + #13#10 +
                     'Errormessage:' + #13#10 +
                     Sysmess;
    end
    else
    begin
      ErrorString := 'Failed to start:' + #13#10 +
                     Filename + ' ' + TmpParams + #13#10 +
                     'Errorcode: ' + IntToStr(ErrorCode);
    end;
  end;//End else CreateProcess
end;//End  function TCustomProgramTracker.ExecuteFile(..

function TCustomProgramTracker.ExecuteProgram: Boolean;
begin
  Result := false;
  //Control before firing program..
  if FProgramExecuted then
  begin
    FErrorMessage := 'The program is allready running.';
    raise EExecuteProgram.Create(FErrorMessage);
  end;
  //Fire
  if ExecuteFile(FExecutable, FParams, FLastMessage,
                 FErrorMessage, FProcessId, FThreadId,
                 FProcessHandle, FThreadHandle, FProgramShowState) then
  begin
    FProgramExecuted := true;
    if Track then
      FWaitThread := TAwaitProgramResult.Create(FTrackingMessage, FProcessId, TrackingInterval * 100); //The last param is for adjusting from milliseconds to 10th of seconds.
    Result := true;
  end
  else
    raise EExecuteProgram.Create(FErrorMessage);
end;

procedure TCustomProgramTracker.OnMessageFromThread(TrackingMessage: string;
            TrackingState: TTrackingState; TrackingCode: Integer);
begin
  if TrackingState <> tsTrackingError then
  begin
    case TrackingState of
      tsTrackingStarted:
        begin
          if Assigned(FProgramStarted) then
            FProgramStarted(Self, TrackingMessage, TrackingState, TrackingCode);
        end;
      tsTracking:
        begin
          if Assigned(FProgramProgress) then
            FProgramProgress(Self, TrackingMessage, TrackingState, TrackingCode);
        end;
      tsTrackingAborted:
        begin
          if Assigned(FProgramProgress) then
            FProgramProgress(Self, TrackingMessage, TrackingState, TrackingCode);
        end;
      tsTrackingFinished:
        begin
          FProgramExecuted := false;
          if Assigned(FProgramFinished) then
            FProgramFinished(Self, TrackingMessage, TrackingState, TrackingCode);
        end;
    end;
  end
  else
  begin
    FProgramExecuted := false;
    raise EExecuteProgram.Create(TrackingMessage);
  end;
end;

procedure TCustomProgramTracker.SetExecutable(Value: string);
begin
  FExecutable := Value;
end;

procedure TCustomProgramTracker.SetParams(Value: TStrings);
var i: Integer;
begin
  FParams.Clear;
  for i:=0 to Value.Count-1 do
    FParams.Add(Value[i]);
end;

procedure TCustomProgramTracker.SetTrack(Value: Boolean);
begin
  if Value then
    FTrack := true
  else
  begin
    if FProgramExecuted then
    begin
      FWaitThread.StopTracking;
      FProcessId := 0;
      FThreadId := 0;
      CloseHandle(FProcessHandle);
      CloseHandle(FThreadHandle);
      FProgramExecuted := false;
    end;
    FTrack := false;
  end;
end;

procedure TCustomProgramTracker.SetTrackInterval(Value: Integer);
begin
  FTrackInterval := Value;
  if (FProgramExecuted and FTrack) then
  begin
    FWaitThread.TrackingInterval := Value * 100;
  end;
end;

//Call this function to start another application.
//Parameters:
//ExecuteAs: Set this to determine how the call to "CreateProcess" will be
//done. For detailed descriptions of the meaning and values of this type,
//see declaration at the beginning of this unit.
//
//FileName: Is the file including path to execute. Except when "ExecuteAS"
//is set to certain values that makes only the name nescessary. Depending
//on "ExecuteAs" this parameter will look different when this function
//returns. If for an example param "ExecuteAs" is set to epWinDir then
//"FileName" will contain that searchpath providing ofcourse that
//CreateProcess whas a success. "Params" will not be changed.
//
//Params: Is the commands(switches) to the file to execute if any. Depending
//on "ExecuteAs" this parameter will look different when this function
//returns. If "ExecuteAs" is set to epParameter then Params will contain
//the file executed including path and all commands passed. "FileName"
//will be empty.
//
//Startinfo: construct this to determine how the application is started.
//
//Procinfo: this structure will contain id's and handles to application
//started on success.
//
//ErrorCode: Will contain any errorcode from system if errors occur.
//
//If the call to "CreateProcess" is successful returnvalue will be true.
//On failure returnvalue will be false and param "Code" will contain an
//errorcode from the system, no change will be done on any other parameter.
function TCustomProgramTracker.CallCreateProcess(const ExecuteWay: TExecuteProgramWay;
           var FileName, Params: string;
           var Startinfo: TStartupInfo; var Procinfo: TProcessInformation;
           var ErrorCode: DWORD): Boolean;
var
  zFileName, zParams, zSysWinDir : array[0..500] of Char;
  PDir, PFileName, PParams: PChar;
  BufferCount: Integer;
begin
  Result := false;
 {There will be various ways of assigning PFileName and PParams
  depending on the ExecuteAs parameter.}
  BufferCount := 1;
  PDir := zSysWinDir;
  case ExecuteWay of
    ewSysDir, ewParameterSysDir: //Get system dir.
      BufferCount := GetSystemDirectory(PDir, 500);
    ewWinDir, ewParameterWinDir: //Get windows dir.
      BufferCount := GetWindowsDirectory(PDir, 500);
    ewCurDir, ewParameterCurDir: //Get the current dir of this process.
      BufferCount := GetCurrentDirectory(500, PDir);
  end;
  if BufferCount = 0 then
  begin
    ErrorCode := GetLastError;
    Exit;
  end;
  StrCat(zSysWinDir, '\');
  case ExecuteWay of
    ewFileName:
      begin
        PFileName := StrPCopy(zFileName, FileName);
        PParams := StrPCopy(zParams, Params);
      end;
    ewCurDir, ewSysDir, ewWinDir:
      begin
        PFileName := StrPCopy(zFileName, string(PDir) + ExtractFileName(FileName));
        PParams := StrPCopy(zParams, Params);
      end;
    ewParameter:
      begin
        PFileName := nil;
        PParams := StrPCopy(zParams, FileName + ' ' + Params);
      end;
    ewParameterCurDir, ewParameterSysDir, ewParameterWinDir:
      begin
        PFileName := nil;
        PParams := StrPCopy(zParams, string(PDir) + ExtractFileName(FileName) + ' ' + Params);
      end;
  end;
  //FIRE...
  if CreateProcess(PFileName,
                   PParams,
                   nil,
                   nil,
                   False,
                   NORMAL_PRIORITY_CLASS,
                   nil,
                   nil,
                   Startinfo,
                   Procinfo) then
  begin
    {This shows how "CreateProcess" was called.}
    FileName := PFileName;
    Params := PParams;
    Result := true
  end
  else
    ErrorCode := GetLastError;
end;//End  function TCustomProgramTracker.CallCreateProcess(..

function TCustomProgramTracker.GetParams: TStrings;
begin
  Result := FParams;
end;

{ END TCustomProgramTracker }



{ Freestanding procedures }

function GetMeaningOfErrorCode(const ErrorCode: DWORD): string;
var Buffer: PChar;
    SysMess: array [0..254] of Char;
    BufferBytes: Integer;
begin
  Result := '';
  Buffer := SysMess;
  SysMess := #0;
  //Try to get meaning of errorcode recieved in param "ErrorCode".
  BufferBytes := FormatMessage(
        {FORMAT_MESSAGE_ALLOCATE_BUFFER or} FORMAT_MESSAGE_FROM_SYSTEM,
        nil,
        ErrorCode,
        GetUserDefaultLangID, // Default language
        Buffer,//LPTSTR()
        255,
        nil);
  if BufferBytes > 0 then//Did we get any errormessage from system?
    Result := SysMess
  else if ErrorCode <= 32 then
    Result := TranslateError(ErrorCode)
  else
    Result := 'Oknt fel har intrffat.';
end;

{ END Freestanding procedures }



{function FindFileInDir(Path, FileName: String): string;
var F : TSearchRec;
begin
   if Path[Length(Path)] <> '\' then
      Path := Path + '\';//Makes sure we have a path
   FindFirst(Path + '*.*', faAnyFile, F);//Search path for any possible dir or file
   repeat
     if  (((F.Attr and faDirectory) > 0) and (F.Name <> '.') and (F.Name <> '..')) then//We ignore dos-system dirs
     begin//If file found is a dir we have to..
        eraseDirAndAllInIt(Path + F.Name);//..search that dir for underlying files/dirs and erase them..
        RemoveDir(Path + F.Name);//..before we remove the dir it self.
     end
     else if ((F.Name <> '.') and (F.Name <> '..')) then//Ignore System dirs
        DeleteFile(Path + F.Name);
   until FindNext(F) <> 0;
   FindClose(F);
   RemoveDir(Path);//Finally remove the dir that was given in path.
end;}

//uses ShellAPI
function TranslateError(Err: DWORD): String;
begin
  Result := '';
  case Err of
    0: Result := 'The operating system is out of memory or resources'; //'Operativsystemets minne eller resurser r fulla.';
    ERROR_FILE_NOT_FOUND: Result := 'The specified file was not found.'; //'Angiven fil hittades inte.';
    ERROR_PATH_NOT_FOUND: Result := 'The specified path was not found.'; //'Angiven skvg hittades inte.';
    ERROR_BAD_FORMAT:	Result := 'The .EXE file is invalid (non-Win32 .EXE or error in .EXE image).'; //'Exe filen ogiltig (ej Windows applikation eller korrupt fil).';
    SE_ERR_ACCESSDENIED: Result := 'The operating system denied access to the specified file.'; //'Operativ systemet nekade tkomst till denna fil.';
    SE_ERR_ASSOCINCOMPLETE: Result := 'The filename association is incomplete or invalid.'; //'Filnamns associationen	r inte komplett eller ogiltig.';
    SE_ERR_DDEBUSY: Result := 'The DDE transaction could not be completed because other DDE transactions were being processed.'; //'DDE transaktionen kunde inte verkstllas eftersom andra transaktioner pgick.';
    SE_ERR_DDEFAIL: Result := 'The DDE transaction failed.'; //'DDE transaktionen misslyckades.';
    SE_ERR_DDETIMEOUT: Result := 'The DDE transaction could not be completed because the request timed out.'; //'DDE transaktionen kunde inte genomfras pga fr lng timeout p begran.';//
    SE_ERR_DLLNOTFOUND: Result := 'The specified dynamic-link library was not found.'; //'Den angivna DLL:en hittades inte.';//
    SE_ERR_NOASSOC: Result := 'There is no application associated with the given filename extension.'; //'Ingen applikation r associerad med angiven filextension.';
    SE_ERR_OOM: Result := 'There was not enough memory to complete the operation.'; //'Minnet rkte inte till fr att utfra operationen.';
    SE_ERR_SHARE: Result := 'A sharing violation occurred.'; //'Ett fildelningsfel intrffade.';
  else
    Result := 'Error unknown'; //'Oknt fel har intrffat.';
  end;
end;




end.
