UNIT PieVersionInfo;

INTERFACE

USES
  Windows, classes, sysutils;

TYPE
  (* Datensatz fr Versionsinformationen *)
  TPieVersionInfo = class (TComponent)
  private
    FVersionInfoAvailable: Boolean; (* True, wenn Versionsinformationen enthalten sind *)
    FMainVersion      : WORD;       (* Hauptversion *)
    FSubVersion       : WORD;       (* Nebenversion *)
    FRelease          : WORD;       (* Ausgabe      *)
    FBuilt            : WORD;       (* Compilierung *)
    FStringFileInfo   : String;
    FCompanyName      : String;
    FFileDescription  : String;
    FFileVersion      : String;
    FInternalName     : String;
    FLegalCopyright   : String;
    FLegalTrademarks  : String;
    FOriginalFilename : String;
    FProductName      : String;
    FProductVersion   : String;
    FComments         : String;
    FFileName         : String;
    FFileDate         : TDateTime;
    FOS               : String;
    PROCEDURE GetVersionInfo;
    procedure GetOSInfo;
    procedure SetFileName(Value: string);
  published
    property VersionInfoAvailable : Boolean read FVersionInfoAvailable write FVersionInfoAvailable;
    property MainVersion      : WORD read FMainVersion write FMainVersion default 1;         (* Hauptversion *)
    property SubVersion       : WORD read FSubVersion write FSubVersion default 0;          (* Nebenversion *)
    property Release          : WORD read FRelease write FRelease default 0;             (* Ausgabe      *)
    property Built            : WORD read FBuilt write FBuilt default 0;               (* Compilierung *)
    property StringFileInfo   : String read FStringFileInfo write FStringFileInfo;
    property CompanyName      : String read FCompanyName write FCompanyName;
    property FileDescription  : String read FFileDescription write FFileDescription;
    property FileVersion      : String read FFileVersion write FFileVersion;
    property InternalName     : String read FInternalName write FInternalName;
    property LegalCopyright   : String read FLegalCopyright write FLegalCopyright;
    property LegalTrademarks  : String read FLegalTrademarks write FLegalTrademarks;
    property OriginalFilename : String read FOriginalFilename write FOriginalFilename;
    property ProductName      : String read FProductName write FProductName;
    property ProductVersion   : String read FProductVersion write FProductVersion;
    property Comments         : String read FComments write FComments;
    property FileDate         : TDateTime read FFileDate write FFileDate;
    property FileName         : String read FFileName write SetFileName;
    property OS               : String read FOS write FOS;
  protected
    procedure Loaded; override;
  public
    constructor Create(AOwner: TComponent); override;
  END;

IMPLEMENTATION

PROCEDURE TPieVersionInfo.GetVersionInfo;

TYPE
  (* Abbild der Ressource der Versionsinformationen *)
  TResVerInfo = RECORD
    ResSize     : WORD;                        (* Gre der Ressource *)
    Head        : ARRAY[2..$2F] OF BYTE;       (* Header              *)
    SubVersion  : WORD;                        (* Nebenversion        *)
    MainVersion : WORD;                        (* Hauptversion        *)
    Built       : WORD;                        (* Compilierung        *)
    Release     : WORD;                        (* Ausgabe             *)
    Unknown     : ARRAY[$38..$61] OF BYTE;     (* ?                   *)
    Contents    : ARRAY[0..1999] OF WideChar;  (* Informationen als   *)
                                               (*  Widechar-Strings   *)
END;
PResVerInfo = ^TResVerInfo;
(* Versionsinformationen aus Ressource auslesen *)
VAR
  rh : THandle;
  mh : THandle;
  p  : PResVerInfo;
  i  : INTEGER;

  FUNCTION GetEntry(rvi      : PResVerInfo;
               VAR Index     : INTEGER;
                   inc1, inc2: INTEGER) : String;
  VAR
    c  : PWideChar;
    s  : String;
  BEGIN
    c     := @rvi^.Contents[Index];
    s     := String(c);
    Index := Index + Length(s) + inc1;
    c     := @p^.Contents[Index];
    s     := String(c);
    GetEntry := s;
    Index := Index + Length(s) + inc2;
  END;

BEGIN
  FVersionInfoAvailable := FALSE;
  rh:=FindResource(hInstance, Pointer(1), rt_Version);
  IF rh > 0 THEN BEGIN  (* Ressource wurde gefunden *)
    mh := LoadResource(hInstance, rh);
    IF mh > 0 THEN BEGIN  (* Ressource konnte geladen werden *)
      p := LockResource(mh);
      (* Informationen aus dem Header *)
      FSubVersion  := p^.SubVersion;
      FMainVersion := p^.MainVersion;
      FBuilt       := p^.Built;
      FRelease     := p^.Release;
      (* Textinformationen *)
      (* "StringFileInfo" *)
      i := 0;
      FStringFileInfo   := GetEntry(p, i, 4, 4);
      (* "CompanyName" *)
      FCompanyName      := GetEntry(p, i, 2, 4);
      (* "FileDescription" *)
      FFileDescription  := GetEntry(p, i, 2, 5);
      (* "FileVersion" *)
      FFileVersion      := GetEntry(p, i, 2, 4);
      (* "Internal Name" *)
      FInternalName     := GetEntry(p, i, 1, 4);
      (* "LegalCopyright" *)
      FLegalCopyright   := GetEntry(p, i, 1, 5);
      (* "LegalTrademarks" *)
      FLegalTrademarks  := GetEntry(p, i, 2, 6);
      (* "OriginalFilename" *)
      FOriginalFilename := GetEntry(p, i,1 , 4);
      (* "ProductName" *)
      FProductName      := GetEntry(p, i, 2, 5);
      (* "ProductVersion" *)
      FProductVersion   := GetEntry(p, i, 1, 4);
      (* "Comments" *)
      FComments         := GetEntry(p, i, 1, 0);
      (* Ressource freigeben *)
      UnlockResource(mh);
      FreeResource(mh);
      FVersionInfoAvailable := TRUE;
    END;
  END;
END;

constructor TPieVersionInfo.Create(AOwner: TComponent);
BEGIN
  (* Initialisierung *)
  inherited Create(AOwner);
  FVersionInfoAvailable := FALSE;
  FMainVersion      := 0;
  FSubVersion       := 0;
  FRelease          := 0;
  FBuilt            := 0;
  FStringFileInfo   := '';
  FCompanyName      := '';
  FFileDescription  := '';
  FFileVersion      := '';
  FInternalName     := '';
  FLegalCopyright   := '';
  FLegalTrademarks  := '';
  FOriginalFilename := '';
  FProductName      := '';
  FProductVersion   := '';
  FComments         := '';
  FOS               := '';
END;

procedure TPieVersionInfo.Loaded;
begin
  inherited Loaded;
  GetVersionInfo;
  GetOSInfo;
end;

procedure TPieVersionInfo.SetFileName(Value: string);
begin
  if (Value <> FFileName) THEN BEGIN
    FFileName := Value;
    {Dateidatum auslesen}
    if FileExists(FFileName)
      THEN FFileDate := FileDateToDateTime(FileAge(FFileName))
      ELSE FFileDate := 0;
  END;
end;

procedure TPieVersionInfo.GetOSInfo;
VAR
  OSVI: TOSVersionInfo;
begin
  OSVI.dwOSVersionInfoSize := SizeOf(OSVI);
  GetVersionEx(OSVI);
  CASE OSVI.dwPlatformId OF
  VER_PLATFORM_WIN32s: FOS := 'Windows 3.1';
  VER_PLATFORM_WIN32_WINDOWS: FOS := 'Windows 95';
  VER_PLATFORM_WIN32_NT: FOS := 'Windows NT';
  ELSE FOS := 'unknown OS';
  END;
end;

END.
