{$I PIETOOLS.INC}
unit PieTacho;
{Copyright 1997 by I. Pietschmann}

interface

uses
  {$IFDEF WIN32}
  Windows,
  {$ELSE}
  WinTypes, WinProcs,
  {$ENDIF}
  Classes, Controls, Graphics;

const
  TWerteproZeile = 1000;

type
  {TPieTachoSkala: Skalen-Typen}
  TPieTachoSkala = (
    tsNone,             { keine                 }
    tsScale,            { Skalierung            }
    tsLabels,           { Texte                 }
    tsScaleLabels);     { Skalierung mit Texten }

  TLabelposition = (lpAussen, lpInnen);
  TZeigerform = (zfNadel, zfPfeil, zfKreis, zfRing);

  {TPieTacho: Tachometer-Komponente}
  TPieTacho = class(TGraphicControl)
  private
    FSkala: TPieTachoSkala;
    FSkalaNK: Byte;
    FLabelposition: TLabelposition;
    FZeigerfarbe: TColor;
    FZeigerbreite: Integer;
    FZeigerform: TZeigerform;
    FPinfarbe: TColor;
    FPinRadius: Integer;
    FSkalafarbe: TColor;
    FAnzahlIntervalle: Word;
    FMinWert: Double;
    FMaxWert: Double;
    FPosition: Double;
    FTransparenz: Boolean;
    Intervallwinkel: Double; {in Grad}
    procedure SetSkala(Value: TPieTachoSkala);
    procedure SetSkalaNK(Value: Byte);
    procedure SetLabelposition(Value: TLabelposition);
    procedure SetZeigerfarbe(Value: TColor);
    procedure SetZeigerbreite(Value: Integer);
    procedure SetZeigerform(Value: TZeigerform);
    procedure SetPinfarbe(Value: TColor);
    procedure SetPinRadius(Value: Integer);
    procedure SetSkalafarbe(Value: TColor);
    procedure SetAnzahlIntervalle(Value: Word);
    procedure SetMinWert(Value: Double);
    procedure SetMaxWert(Value: Double);
    procedure SetPosition(Value: Double);
    procedure SetTransparenz(Value: Boolean);
    procedure Tacho_malen;
  protected
    procedure Paint; override;
  public
    constructor Create(AOwner: TComponent); override;
  published
    property Scale: TPieTachoSkala read FSkala write SetSkala default tsScaleLabels;
    property ScaleDecimals: Byte read FSkalaNK write SetSkalaNK default 0;
    property LabelPosition: TLabelposition read FLabelposition write SetLabelposition default lpAussen;
    property PointerColor: TColor read FZeigerfarbe write SetZeigerfarbe default clRed;
    property PointerWidth: Integer read FZeigerbreite write SetZeigerbreite default 4;
    property PointerType: TZeigerform read FZeigerform write SetZeigerform default zfPfeil;
    property PinColor: TColor read FPinfarbe write SetPinfarbe default clGreen;
    property PinRadius: Integer read FPinRadius write SetPinRadius default 10;
    property ScaleColor: TColor read FSkalafarbe write SetSkalafarbe default clBlack;
    property NumberOfIntervals: Word read FAnzahlIntervalle write SetAnzahlIntervalle default 10;
    property MinValue: Double read FMinWert write SetMinWert;
    property MaxValue: Double read FMaxWert write SetMaxWert;
    property Position: Double read FPosition write SetPosition;
    property Transparent: Boolean read FTransparenz write SetTransparenz default FALSE;
    property Align;
    property Color;
    property DragCursor;
    property DragMode;
    property Enabled;
    property Font;
    property ParentColor;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

(*procedure Register;*)

implementation

uses
  Dialogs, SysUtils, PieHerk;

{ TPieTacho }

procedure TPieTacho.SetSkala(Value: TPieTachoSkala);
begin
  if Value <> FSkala then begin
    FSkala := Value;
    Invalidate;
  end;
end;

procedure TPieTacho.SetSkalaNK(Value: Byte);
begin
  if Value <> FSkalaNK then begin
    FSkalaNK := Value;
    Invalidate;
  end;
end;

procedure TPieTacho.SetLabelposition(Value: TLabelposition);
begin
  if Value <> FLabelposition then begin
    FLabelposition := Value;
    Invalidate;
  end;
end;

procedure TPieTacho.SetZeigerfarbe(Value: TColor);
begin
  if Value <> FZeigerfarbe then begin
    FZeigerfarbe := Value;
    Invalidate;
  end;
end;

procedure TPieTacho.SetZeigerbreite(Value: Integer);
begin
  if Value <> FZeigerbreite then begin
    FZeigerbreite := Value;
    Invalidate;
  end;
end;

procedure TPieTacho.SetZeigerform(Value: TZeigerform);
begin
  if Value <> FZeigerform then begin
    FZeigerform := Value;
    Invalidate;
  end;
end;

procedure TPieTacho.SetPinfarbe(Value: TColor);
begin
  if Value <> FPinfarbe then begin
    FPinfarbe := Value;
    Invalidate;
  end;
end;

procedure TPieTacho.SetPinRadius(Value: Integer);
begin
  if Value <> FPinRadius then begin
    FPinRadius := Value;
    Invalidate;
  end;
end;

procedure TPieTacho.SetSkalafarbe(Value: TColor);
begin
  if Value <> FSkalafarbe then begin
    FSkalafarbe := Value;
    Invalidate;
  end;
end;

procedure TPieTacho.SetAnzahlIntervalle(Value: Word);
begin
  if (Value < 2) or (Value > 200) then
    MessageDlg('Wert sollte zwischen 2 und 200 liegen.', mtInformation, [mbOK], 0)
  else if Value <> FAnzahlIntervalle then begin
    FAnzahlIntervalle := Value;
    Invalidate;
  end;
end;

procedure TPieTacho.SetMinWert(Value: Double);
begin
  if Value <> FMinWert then begin
    FMinWert := Value;
    IF FMaxWert < FMinWert THEN FMaxWert := FMinWert+1;
    IF FPosition < FMinWert THEN FPosition := FMinWert;
    Invalidate;
  end;
end;

procedure TPieTacho.SetMaxWert(Value: Double);
begin
  if Value <> FMaxWert then begin
    FMaxWert := Value;
    IF FMaxWert < FMinWert THEN FMinWert := FMaxWert-1;
    IF FPosition > FMaxWert THEN FPosition := FMaxWert;
    Invalidate;
  end;
end;

procedure TPieTacho.SetPosition(Value: Double);
begin
  if Value <> FPosition then begin
    FPosition := Value;
    IF FPosition < FMinWert THEN FPosition := FMinWert;
    IF FPosition > FMaxWert THEN FPosition := FMaxWert;
    Invalidate;
  end;
end;

procedure TPieTacho.SetTransparenz(Value: Boolean);
begin
  if Value <> FTransparenz then begin
    FTransparenz := Value;
    Invalidate;
  end;
end;

procedure TPieTacho.Tacho_malen;
VAR
  I              : Integer;
  Xf, Yf, Radius : Double;
  UrX, UrY, W    : Integer;
  Wert, Delta    : Double;
  Pfeil          : Array[1..3] of TPoint;
begin
  WITH Canvas DO BEGIN
    {Intervallwinkel in Pixel ermitteln}
    Intervallwinkel := Pi / (FAnzahlIntervalle);
    {Radius des Tachos}
    IF (Width DIV 2)-10 < Height-FPinRadius-20
      THEN Radius := (Width DIV 2)-10
      ELSE Radius := Height-FPinRadius-20;
    {Ursprungspunkt des Tachos}
    UrX := Width DIV 2;
    UrY := Height-10-FPinRadius;
    {Zeiger malen - Teil 1}
    Pen.Color := FZeigerfarbe;
    Brush.Color := FZeigerfarbe;
    Brush.Style := bsSolid;
    Pen.Width := FZeigerbreite;
    Wert := (FPosition - FMinWert) / (FMaxWert - FMinWert) * Pi; {in Grad}
    IF Wert = 0 THEN Wert := 0.5/180*Pi; {mind. 0,5 Grad anzeigen}
    Xf := -cos(Wert) * Radius;
    Yf := -sin(Wert) * Radius;
    CASE FZeigerform OF
    zfKreis : BEGIN
              CASE FLabelposition OF
              lpAussen: BEGIN
                        W := round(Radius * 0.6);
                        Pie(UrX-W, UrY-W, UrX+W, UrY+W,
                            round(Xf*0.6+UrX), round(Yf*0.6+UrY),
                            UrX-W, UrY);
                        END;
              lpInnen:  BEGIN
                        W := round(Radius * 0.9);
                        Pie(UrX-W, UrY-W, UrX+W, UrY+W,
                            round(Xf*0.9+UrX), round(Yf*0.9+UrY),
                            UrX-W, UrY);
                        END;
              END;
              END;
    zfRing  : BEGIN
              Brush.Style := bsClear;
              CASE FLabelposition OF
              lpAussen: BEGIN
                        W := round(Radius * 0.6);
                        Pie(UrX-W, UrY-W, UrX+W, UrY+W,
                            round(Xf*0.6+UrX), round(Yf*0.6+UrY),
                            UrX-W, UrY);
                        W := round(Radius * 0.5);
                        Pie(UrX-W, UrY-W, UrX+W, UrY+W,
                            round(Xf*0.5+UrX), round(Yf*0.5+UrY),
                            UrX-W, UrY);
                        Brush.Color := FZeigerfarbe;
                        Brush.Style := bsSolid;
                        FloodFill(UrX-W-3, UrY-3, Pen.Color, fsBorder);
                        END;
              lpInnen:  BEGIN
                        W := round(Radius * 0.9);
                        Pie(UrX-W, UrY-W, UrX+W, UrY+W,
                            round(Xf*0.9+UrX), round(Yf*0.9+UrY),
                            UrX-W, UrY);
                        W := round(Radius * 0.7);
                        Pie(UrX-W, UrY-W, UrX+W, UrY+W,
                            round(Xf*0.7+UrX), round(Yf*0.7+UrY),
                            UrX-W, UrY);
                        Brush.Color := FZeigerfarbe;
                        Brush.Style := bsSolid;
                        FloodFill(UrX-W-3, UrY-3, Pen.Color, fsBorder);
                        END;
              END;
              END;
    END;
    Brush.Color := Color;
    {Skalenstriche malen}
    IF FSkala IN [tsScale, tsScaleLabels] THEN BEGIN
      Pen.Color := FSkalafarbe;
      Pen.Width := 2;
      FOR I:=0 TO FAnzahlIntervalle DO BEGIN
        Xf := -cos(I*Intervallwinkel) * Radius;
        Yf := -sin(I*Intervallwinkel) * Radius;
        MoveTo(round(Xf*0.9+UrX), round(Yf*0.9+UrY));
        LineTo(round(Xf*0.7+UrX), round(Yf*0.7+UrY));
      END;
    END;
    {Skalentexte malen}
    IF FSkala IN [tsLabels, tsScaleLabels] THEN BEGIN
      Font := Self.Font;
      SetBKMode(Handle, windows.TRANSPARENT);
      FOR I:=0 TO FAnzahlIntervalle DO BEGIN
        Wert := (FMaxWert - FMinWert) / FAnzahlIntervalle * I + FMinWert;
        Xf := -cos(I*Intervallwinkel) * Radius;
        Yf := -sin(I*Intervallwinkel) * Radius;
        IF FLabelposition = lpAussen THEN BEGIN {Werte auerhalb}
          IF I < FAnzahlIntervalle DIV 2
            THEN SetTextAlign(Handle, ta_Right OR ta_Bottom OR ta_NoUpdateCP);
          IF I >= FAnzahlIntervalle DIV 2
            THEN SetTextAlign(Handle, ta_Left OR ta_Bottom OR ta_NoUpdateCP);
          IF (I = FAnzahlIntervalle DIV 2) AND (FAnzahlIntervalle MOD 2 = 0)
            THEN SetTextAlign(Handle, ta_Center OR ta_Bottom OR ta_NoUpdateCP);
          TextOut(round(Xf*0.9+UrX), round(Yf*0.9+UrY), S_N(Wert, 1, FSkalaNK));
        END;
        IF FLabelposition = lpInnen THEN BEGIN {Werte innerhalb}
          IF I < FAnzahlIntervalle DIV 2
            THEN SetTextAlign(Handle, ta_Left OR ta_Top OR ta_NoUpdateCP);
          IF I >= FAnzahlIntervalle DIV 2
            THEN SetTextAlign(Handle, ta_Right OR ta_Top OR ta_NoUpdateCP);
          IF (I = FAnzahlIntervalle DIV 2) AND (FAnzahlIntervalle MOD 2 = 0)
            THEN SetTextAlign(Handle, ta_Center OR ta_Top OR ta_NoUpdateCP);
          TextOut(round(Xf*0.65+UrX), round(Yf*0.65+UrY), S_N(Wert, 1, FSkalaNK));
        END;
      END;
    END;
    SetBKMode(Handle, OPAQUE);
    {Zeiger malen - Teil 2}
    Pen.Color := FZeigerfarbe;
    Brush.Color := FZeigerfarbe;
    Pen.Width := FZeigerbreite;
    Wert := (FPosition - FMinWert) / (FMaxWert - FMinWert) * Pi; {in Grad}
    IF Wert = 0 THEN Wert := 0.5/180*Pi; {mind. 0,5 Grad anzeigen}
    Delta := Pi/180*2*FZeigerbreite;
    Xf := -cos(Wert) * Radius;
    Yf := -sin(Wert) * Radius;
    MoveTo(round(Xf*0.9+UrX), round(Yf*0.9+UrY)); {Nadel malen}
    LineTo(UrX, UrY);
    IF FZeigerform = zfPfeil THEN BEGIN
      Pfeil[1] := Point(round(Xf*0.9+UrX), round(Yf*0.9+UrY));
      Pfeil[2] := Point(round(-cos(Wert-Delta)*Radius*0.5+UrX),
                        round(-sin(Wert-Delta)*Radius*0.5+UrY));
      Pfeil[3] := Point(round(-cos(Wert+Delta)*Radius*0.5+UrX),
                        round(-sin(Wert+Delta)*Radius*0.5+UrY));
      Polygon(Pfeil);
    END;
    {Pin malen}
    Pen.Color := FPinfarbe;
    Brush.Color := FPinfarbe;
    Ellipse(UrX-FPinRadius, UrY-FPinRadius, UrX+FPinRadius, UrY+FPinRadius);
  END;
end;

procedure TPieTacho.Paint;
var
  Rect: TRect;
begin
  Rect := GetClientRect;
  with Canvas do begin
    {Transparenz?}
    IF FTransparenz THEN SetBKMode(Handle, windows.TRANSPARENT)
                    ELSE SetBKMode(Handle, OPAQUE);
    {Hintergrund malen}
    IF NOT(FTransparenz) THEN BEGIN
      Brush.Color := Color;
      FillRect(Rect);
      Brush.Style := bsClear;
    END;
    {Tacho malen}
    Tacho_malen;
  end;  {WITH Canvas DO ...}
end;

constructor TPieTacho.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FSkala := tsScaleLabels;
  FSkalaNK := 0;
  FLabelposition := lpAussen;
  FZeigerfarbe := clRed;
  FZeigerbreite := 4;
  FZeigerform := zfPfeil;
  FPinfarbe := clGreen;
  FPinRadius := 10;
  FSkalafarbe := clBlack;
  FAnzahlIntervalle := 10;
  FMinWert := 0.0;
  FMaxWert := 10.0;
  FPosition := 0.0;
  FTransparenz := FALSE;
  Width := 250;
  Height := 200;
end;


{ Register }

(*procedure Register;

begin
  RegisterComponents('Pietschmann', [TPieTacho]);
end;*)

end.
