{$I PIETOOLS.INC}
unit PieStat;
{Copyright 1997 by I. Pietschmann}

interface

uses
  Windows, Classes, Controls, Graphics, Menus,
  ExtCtrls, Forms, Messages;

type
  {TTSkalierungsart : Skalierungsart des Histogramms}
  TTSkalierungsart = (
    saAutoXY,           {Automatische Skalierung der X&Y-Achse}
    saAutoX,            {Automatische Skalierung der X-Achse}
    saAutoY,            {Automatische Skalierung der Y-Achse}
    saManually);        {Manuelle Skalierung der X&Y-Achse}

  TPieHistogrampaar = RECORD
                 X, Y: Single;
               END;
  { TTPieHistogramwerte : Diagramm-Werte-Array }
  TTPieHistogramwerte =
    array of TPieHistogrampaar;

  { TTPieHistogramTyp : Diagramm-Typen }
  TTPieHistogramType = (
    ctBars,             { Balken                }
    ctFramedBars,       { Balken mit Rahmen     }
    ctPyramids,         { Pyramiden             }
    ctFramedPyramids,   { Pyramiden mit Rahmen  }
    ctCircles,          { Kreise (gefllt)      }
    ctRectangles,       { Rechtecke (gefllt)   }
    ctLines,            { Linien                }
    ctLinesWithCircles, { Linien mit Kreisen    }
    ctLinesWithRect,    { Linien mit Rechtecken }
    ctXYPlot);          { X-Y-Plot              }

  { TPieHistogram : Diagramm-Komponente }
  TPieHistogram = class(TGraphicControl)
  private
    Values             : TTPieHistogramwerte;
    Intervallwerte     : TTPieHistogramwerte; {umgerechnete Intervallwerte}
    FHistogrammTyp     : TTPieHistogramType;
    FBalkenfarbe       : TColor;
    FFehlerfarbe       : TColor;
    FGitterfarbe       : TColor;
    FHorizGitter       : Boolean;
    FVertGitter        : Boolean;
    FAnzahlHorizIntervalle: Word;
    FAnzahlVertIntervalle: Word;
    FZeigeHorizLabels  : Boolean;
    FZeigeVertLabels   : Boolean;
    FXAchsenPosition   : Integer;
    FYAchsenPosition   : Integer;
    FSkalierungsart    : TTSkalierungsart;
    FBezXAchse         : string;
    FBezYAchse         : string;
    FMinXWert          : Double;
    FMaxXWert          : Double;
    FMinYWert          : Double;
    FMaxYWert          : Double;
    FStellen           : Word;
    FTransparent       : Boolean;
    MaxValue_O,
    MinValue_O         : TPieHistogrampaar;
    MaxValue,
    MinValue           : TPieHistogrampaar;
    Intervallbreite,
    Intervallhoehe     : Integer; {in Pixel}
    Schriftart         : TLogFont;
    function GetXAchsenPosition: Integer;
    procedure SeTPieHistogramTyp(Value: TTPieHistogramType);
    procedure SetXAchsenPosition(Value: Integer);
    procedure SetYAchsenPosition(Value: Integer);
    procedure SetSkalierungsart(Value: TTSkalierungsart);
    procedure SetBalkenfarbe(Value: TColor);
    procedure SetFehlerfarbe(Value: TColor);
    procedure SetGitterfarbe(Value: TColor);
    procedure SetHorizGitter(Value: Boolean);
    procedure SetVertGitter(Value: Boolean);
    procedure SetAnzahlHorizIntervalle(Value: Word);
    procedure SetAnzahlVertIntervalle(Value: Word);
    procedure SetZeigeHorizLabels(Value: Boolean);
    procedure SetZeigeVertLabels(Value: Boolean);
    procedure SetBezXAchse(Value: string);
    procedure SetBezYAchse(Value: string);
    procedure SetMinXWert(Value: Double);
    procedure SetMaxXWert(Value: Double);
    procedure SetMinYWert(Value: Double);
    procedure SetMaxYWert(Value: Double);
    procedure SetStellen(Value: Word);
    procedure SetTransparent(Value: Boolean);
    procedure MyResize(Sender: TObject);
    FUNCTION BezugX(X: Double): Integer; {liefert Pixelkoordianten}
    FUNCTION BezugY(Y: Double): Integer; {liefert Pixelkoordianten}
    PROCEDURE Intervalle_sortieren;
    PROCEDURE Gitter_malen;
    PROCEDURE Achsen_malen;
    PROCEDURE Achsen_bezeichnen;
    PROCEDURE Intervalle_malen;
    PROCEDURE XYPlot_malen;
  protected
    procedure Paint; override;
  public    {PieStat}
    constructor Create(AOwner: TComponent); override;
    function GetMaxValue(Lang: Boolean): TPieHistogrampaar;
    function GetMinValue(Lang: Boolean): TPieHistogrampaar;
    procedure InitValues;
    procedure AddValue(X, Y: Single);
  published
    property Align;
    property Anchors;
    property HistogramType: TTPieHistogramType read FHistogrammTyp write SeTPieHistogramTyp default ctBars;
    property XAxisPosition: Integer read GetXAchsenPosition write SetXAchsenPosition;
    property YAxisPosition: Integer read FYAchsenPosition write SetYAchsenPosition;
    property ScaleType: TTSkalierungsart read FSkalierungsart write SetSkalierungsart default saAutoXY;
    property Color;
    property BarColor: TColor read FBalkenfarbe write SetBalkenfarbe default clYellow;
    property ErrorColor: TColor read FFehlerfarbe write SetFehlerfarbe default clRed;
    property DragCursor;
    property DragMode;
    property Enabled;
    property GridColor: TColor read FGitterfarbe write SetGitterfarbe default clGray;
    property HorizontalGrid: Boolean read FHorizGitter write SetHorizGitter default False;
    property NumberOfHorizontalIntervals: Word read FAnzahlHorizIntervalle write SetAnzahlHorizIntervalle;
    property NumberOfVerticalIntervals: Word read FAnzahlVertIntervalle write SetAnzahlVertIntervalle default 10;
    property ParentColor;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property ShowHorizontalLabels: Boolean read FZeigeHorizLabels write SetZeigeHorizLabels default True;
    property ShowVerticalLabels: Boolean read FZeigeVertLabels write SetZeigeVertLabels default True;
    property VerticalGrid: Boolean read FVertGitter write SetVertGitter default False;
    property NameXAxis: string read FBezXAchse write SetBezXAchse;
    property NameYAxis: string read FBezYAchse write SetBezYAchse;
    property MinXValue: Double read FMinXWert write SetMinXWert;
    property MaxXValue: Double read FMaxXWert write SetMaxXWert;
    property MinYValue: Double read FMinYWert write SetMinYWert;
    property MaxYValue: Double read FMaxYWert write SetMaxYWert;
    property DigitCount: Word read FStellen write SetStellen default 4;
    property Transparent: Boolean read FTransparent write SetTransparent default FALSE;
    property Visible;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

(*procedure Register;*)

implementation

uses
  Dialogs, SysUtils, PieHERK;

{ TPieHistogram }
procedure TPieHistogram.MyResize(Sender: TObject);
begin
{  XAchsenPosition := Height - Alte_Hoehe + FXAchsenPosition;
  Alte_Hoehe := Height;}
END;

procedure TPieHistogram.SeTPieHistogramTyp(Value: TTPieHistogramType);
begin
  if Value <> FHistogrammTyp then begin
    FHistogrammTyp := Value;
    Invalidate;
  end;
end;

function TPieHistogram.GetXAchsenPosition: Integer;
begin
  Result := Height - FXAchsenPosition;
end;

procedure TPieHistogram.SetXAchsenPosition(Value: Integer);
begin
  if Value <> Height-FXAchsenPosition then begin
    FXAchsenPosition := Height-Value;
    Invalidate;
  end;
end;

procedure TPieHistogram.SetYAchsenPosition(Value: Integer);
begin
  if Value <> FYAchsenPosition then begin
    FYAchsenPosition := Value;
    Invalidate;
  end;
end;

procedure TPieHistogram.SetSkalierungsart(Value: TTSkalierungsart);
begin
  if Value <> FSkalierungsart then begin
    FSkalierungsart := Value;
    Invalidate;
  end;
end;

procedure TPieHistogram.SetBalkenfarbe(Value: TColor);
begin
  if Value <> FBalkenfarbe then begin
    FBalkenfarbe := Value;
    Invalidate;
  end;
end;

procedure TPieHistogram.SetFehlerfarbe(Value: TColor);
begin
  if Value <> FFehlerfarbe then begin
    FFehlerfarbe := Value;
    Invalidate;
  end;
end;

procedure TPieHistogram.SetGitterfarbe(Value: TColor);

begin
  if Value <> FGitterfarbe then
    begin
      FGitterfarbe := Value;
      if FHorizGitter or FVertGitter then
        Invalidate;
    end;
end;

procedure TPieHistogram.SetHorizGitter(Value: Boolean);
begin
  if Value <> FHorizGitter then begin
    FHorizGitter := Value;
    Invalidate;
  end;
end;

procedure TPieHistogram.SetAnzahlHorizIntervalle(Value: Word);
begin
  if (Value < 2) or (Value > 200) then MessageDlg('Wert sollte zwischen 2 und 200 liegen.', mtInformation, [mbOK], 0)
  else if Value <> FAnzahlHorizIntervalle then begin
    FAnzahlHorizIntervalle := Value;
    Invalidate;
  end;
end;

procedure TPieHistogram.SetAnzahlVertIntervalle(Value: Word);
begin
  if (Value < 2) or (Value > 200) then MessageDlg('Wert sollte zwischen 2 und 200 liegen.', mtInformation, [mbOK], 0)
  else if Value <> FAnzahlVertIntervalle then begin
    FAnzahlVertIntervalle := Value;
    Invalidate;
  end;
end;

procedure TPieHistogram.SetZeigeHorizLabels(Value: Boolean);
begin
  if Value <> FZeigeHorizLabels then begin
    FZeigeHorizLabels := Value;
    Invalidate;
  end;
end;

procedure TPieHistogram.SetZeigeVertLabels(Value: Boolean);
begin
  if Value <> FZeigeVertLabels then begin
    FZeigeVertLabels := Value;
    Invalidate;
  end;
end;

procedure TPieHistogram.SetVertGitter(Value: Boolean);
begin
  if Value <> FVertGitter then begin
    FVertGitter := Value;
    Invalidate;
  end;
end;

procedure TPieHistogram.SetBezXAchse(Value: string);
begin
  if Value <> FBezXAchse then begin
    FBezXAchse := Value;
    Invalidate;
  end;
end;

procedure TPieHistogram.SetBezYAchse(Value: string);
begin
  if Value <> FBezYAchse then begin
    FBezYAchse := Value;
    Invalidate;
  end;
end;

procedure TPieHistogram.SetMinXWert(Value: Double);
begin
  if Value <> FMinXWert then begin
    FMinXWert := Value;
    IF FSkalierungsart IN [saAutoY, saManually] THEN Invalidate;
  end;
end;

procedure TPieHistogram.SetMaxXWert(Value: Double);
begin
  if Value <> FMaxXWert then begin
    FMaxXWert := Value;
    IF FSkalierungsart IN [saAutoY, saManually] THEN Invalidate;
  end;
end;

procedure TPieHistogram.SetMinYWert(Value: Double);
begin
  if Value <> FMinYWert then begin
    FMinYWert := Value;
    IF FSkalierungsart IN [saAutoX, saManually] THEN Invalidate;
  end;
end;

procedure TPieHistogram.SetMaxYWert(Value: Double);
begin
  if Value <> FMaxYWert then begin
    FMaxYWert := Value;
    IF FSkalierungsart IN [saAutoX, saManually] THEN Invalidate;
  end;
end;

procedure TPieHistogram.SetStellen(Value: Word);
begin
  if (Value <> FStellen) AND (Value > 0) AND (Value < 11) then begin
    FStellen := Value;
    IF FZeigeHorizLabels OR FZeigeVertLabels THEN Invalidate;
  end;
end;

procedure TPieHistogram.SetTransparent(Value: Boolean);
BEGIN
  IF (Value <> FTransparent) THEN BEGIN
    FTransparent := Value;
    Invalidate;
  END;
END;

FUNCTION TPieHistogram.BezugX(X: Double): Integer; {liefert Pixelkoordianten}
VAR
  R: Integer;
BEGIN
  R := Round((FAnzahlHorizIntervalle * Intervallbreite) /
             (MaxValue.X - MinValue.X) *
             (X - MinValue.X));
  BezugX := R+FYAchsenPosition;
END;

FUNCTION TPieHistogram.BezugY(Y: Double): Integer; {liefert Pixelkoordianten}
BEGIN
  Result := Height-FXAchsenposition-
                  Round((FAnzahlVertIntervalle * Intervallhoehe) /
                  (MaxValue.Y - MinValue.Y) *
                  (Y - MinValue.Y));
END;

PROCEDURE TPieHistogram.Intervalle_sortieren;
VAR
  Intvl: Integer;
  Anzahl: Double;
  UG, OG: Double;
  I: Integer;
BEGIN
  SetLength(Intervallwerte, FAnzahlHorizIntervalle+2);
  FOR Intvl:=0 TO FAnzahlHorizIntervalle+1 DO BEGIN
    Anzahl := 0;
    UG := ((MaxValue_O.X - MinValue_O.X) / FAnzahlHorizIntervalle * (Intvl-1)) + MinValue_O.X;
    OG := ((MaxValue_O.X - MinValue_O.X) / FAnzahlHorizIntervalle * Intvl) + MinValue_O.X;
    {alle Werte, die kleiner als MinValue_O.X sind}
    IF Intvl = 0 THEN FOR I := Low(Values) TO High(Values) DO
      IF Values[I].X < OG then Anzahl := Anzahl + Values[I].Y;
    {alle Werte, die kleiner als OG und grer als UG sind}
    IF (Intvl > 0) AND (Intvl < FAnzahlHorizIntervalle) THEN FOR I := Low(Values) TO High(Values) DO
      IF (Values[I].X >= UG) AND (Values[I].X < OG) then Anzahl := Anzahl + Values[I].Y;
    {alle Werte, die kleiner/gleich als OG und grer als UG sind}
    IF (Intvl = FAnzahlHorizIntervalle) THEN FOR I := Low(Values) TO High(Values) DO
      IF (Values[I].X >= UG) AND (Values[I].X <= OG) then Anzahl := Anzahl + Values[I].Y;
    {alle Werte, die grer als MaxValue_O.X sind}
    IF Intvl = FAnzahlHorizIntervalle+1 THEN FOR I := Low(Values) TO High(Values) DO
      IF Values[I].X > UG then Anzahl := Anzahl + Values[I].Y;
    Intervallwerte[Intvl].X := ((MaxValue_O.X - MinValue_O.X) / FAnzahlHorizIntervalle * (Intvl-0.5)) + MinValue_O.X;
    Intervallwerte[Intvl].Y := Anzahl;
  END;
END;

PROCEDURE TPieHistogram.Gitter_malen;
VAR
  I: Integer;
BEGIN
  WITH Canvas DO BEGIN
    {Gitter malen}
    Pen.Color := clGray;
    IF FHorizGitter THEN FOR I:=0 TO FAnzahlVertIntervalle DO BEGIN
      MoveTo(0, BezugY((MaxValue.Y-MinValue.Y)/FAnzahlVertIntervalle*I+MinValue.Y));
      LineTo(Width, BezugY((MaxValue.Y-MinValue.Y)/FAnzahlVertIntervalle*I+MinValue.Y));
    END;
    IF FVertGitter THEN FOR I:=0 TO FAnzahlHorizIntervalle DO BEGIN
      MoveTo(BezugX((MaxValue_O.X-MinValue_O.X)/FAnzahlHorizIntervalle*I+MinValue_O.X), 0);
      LineTo(BezugX((MaxValue_O.X-MinValue_O.X)/FAnzahlHorizIntervalle*I+MinValue_O.X), Height);
    END;
  END; {WITH Canvas ...}
END;

PROCEDURE TPieHistogram.Achsen_malen;
VAR
  I: Integer;
BEGIN
  WITH Canvas DO BEGIN
    {Achsen malen}
    Pen.Color := clBlack;
    MoveTo(0, XAxisPosition);
    LineTo(Width, XAxisPosition);
    MoveTo(FYAchsenPosition, 0);
    LineTo(FYAchsenPosition, Height);
    {Achsenstriche malen}
    FOR I:=0 TO FAnzahlHorizIntervalle DO BEGIN
      MoveTo(BezugX((MaxValue_O.X-MinValue_O.X)/FAnzahlHorizIntervalle*I+MinValue_O.X),
             XAxisPosition-2);
      LineTo(BezugX((MaxValue_O.X-MinValue_O.X)/FAnzahlHorizIntervalle*I+MinValue_O.X),
             XAxisPosition+2);
    END;
    FOR I:=0 TO FAnzahlVertIntervalle DO BEGIN
      MoveTo(FYAchsenPosition -2,
             BezugY((MaxValue.Y-MinValue.Y)/FAnzahlVertIntervalle*I+MinValue.Y));
      LineTo(FYAchsenPosition +2,
             BezugY((MaxValue.Y-MinValue.Y)/FAnzahlVertIntervalle*I+MinValue.Y));
    END;
  END; {WITH Canvas ...}
END;

PROCEDURE TPieHistogram.Achsen_bezeichnen;
VAR
  I: Integer;
  Wert: Double;
  Alte_Schrift: THandle;
BEGIN
  WITH Canvas DO BEGIN
    Alte_Schrift := SelectObject(Handle, CreateFontIndirect(Schriftart));
    {X-Achse}
    IF FZeigeHorizLabels THEN BEGIN
      SetTextAlign(Handle, ta_Right OR ta_Bottom OR ta_NoUpdateCP);
      Schriftart.lfEscapement:= 900;
      DeleteObject(SelectObject(Handle, CreateFontIndirect(Schriftart)));
      {-------------------}
      FOR I:=0 TO FAnzahlHorizIntervalle DO BEGIN
        Wert := ((MaxValue_O.X - MinValue_O.X) / FAnzahlHorizIntervalle * I) + MinValue_O.X;
        TextOut(BezugX(Wert), XAxisPosition+4, Optimal_S_N(Wert, FStellen));
      END;
      {-------------------}
      Schriftart.lfEscapement:= 0;
      DeleteObject(SelectObject(Handle, CreateFontIndirect(Schriftart)));
      SetTextAlign(Handle, ta_Left OR ta_Top OR ta_NoUpdateCP);
      TextOut(1, XAxisPosition+4, FBezXAchse);
    END;
    {Y-Achse}
    IF FZeigeVertLabels THEN BEGIN
      DeleteObject(SelectObject(Handle, CreateFontIndirect(Schriftart)));
      SetTextAlign(Handle, ta_Right OR ta_Bottom OR ta_NoUpdateCP);
      {-------------------}
      FOR I:=0 TO FAnzahlVertIntervalle DO BEGIN
        Wert := ((MaxValue.Y - MinValue.Y) / FAnzahlVertIntervalle * I) + MinValue.Y;
        TextOut(FYAchsenPosition-4, BezugY(Wert), Optimal_S_N(Wert, FStellen));
      END;
      {-------------------}
      SetTextAlign(Handle, ta_Right OR ta_Top OR ta_NoUpdateCP);
      TextOut(FYAchsenPosition-4, 1, FBezYAchse);
    END;
    {-----------------------------------------------}
    SetTextAlign(Handle, ta_Left OR ta_Top OR ta_NoUpdateCP);
    DeleteObject(SelectObject(Handle, Alte_Schrift));
  END;
END;

PROCEDURE TPieHistogram.Intervalle_malen;
VAR
  Intvl: Integer;
  X, Y: Double;
  R: TRect;
BEGIN
  WITH Canvas DO FOR Intvl:=0 TO FAnzahlHorizIntervalle+1 DO BEGIN
    X := Intervallwerte[Intvl].X;
    Y := Intervallwerte[Intvl].Y;
    Brush.Style := bsSolid;
    IF (Intvl > 0) AND (Intvl < FAnzahlHorizIntervalle+1) THEN BEGIN
      Brush.Color := FBalkenfarbe;
      Pen.Color := FBalkenfarbe;
    END
    ELSE BEGIN
      Brush.Color := FFehlerfarbe;
      Pen.Color := FFehlerfarbe;
    END;
{---------------------------------------------}
    CASE FHistogrammTyp OF
    {---------------------}
    ctBars, ctFramedBars: IF Y <> 0 THEN
    BEGIN
      R.Left   := BezugX(X)-Intervallbreite DIV 2+2;
      R.Right  := BezugX(X)+Intervallbreite DIV 2-2;
      R.Top    := BezugY(0);
      R.Bottom := BezugY(Y);
      FillRect(R);
      if FHistogrammTyp = ctFramedBars then begin
        Pen.Color := clBlack;
        Rectangle(R.Left, R.Top, R.Right, R.Bottom);
      end;
    END;
    {---------------------}
    ctPyramids, ctFramedPyramids: IF Y <> 0 THEN
    BEGIN
      IF FHistogrammtyp = ctFramedPyramids THEN Pen.Color := clBlack;
      Polygon([
              Point((BezugX(X)-Intervallbreite DIV 2+2),
                    BezugY(0)),
              Point((BezugX(X)),
                    (BezugY(Y))),
              Point((BezugX(X)+Intervallbreite DIV 2-2),
                    BezugY(0))
              ]);
    END;
    {---------------------}
    ctLines, ctLinesWithCircles, ctLinesWithRect:
    BEGIN
      IF Intvl = 0 THEN MoveTo(BezugX(X), BezugY(Y));
      IF Intvl = 1 THEN Pen.Color := FFehlerfarbe;
      LineTo(BezugX(X), BezugY(Y));
      IF Intvl = 1 THEN Pen.Color := FBalkenfarbe;
    END;
    END; {CASE FHistogrammtyp OF ...}
{---------------------------------------------}
    CASE FHistogrammTyp OF
    {---------------------}
    ctLinesWithCircles, ctCircles:
    BEGIN
      Pen.Color := clBlack;
      Ellipse(BezugX(X)-4, BezugY(Y)-4, BezugX(X)+4, BezugY(Y)+4);
    END;
    {---------------------}
    ctLinesWithRect, ctRectangles:
    BEGIN
      Pen.Color := clBlack;
      Rectangle(BezugX(X)-4, BezugY(Y)-4, BezugX(X)+4, BezugY(Y)+4);
    END;
    END; {CASE FHistogrammtyp OF ...}
{---------------------------------------------}
    Brush.Style := bsClear;
  END; {WITH Canvas DO FOR Intvl:=0 TO ...}
END;

PROCEDURE TPieHistogram.XYPlot_malen;
VAR
  I: Integer;
  X, Y: Double;
  IO: Boolean;
  XSumme, YSumme: Double;
  Alte_Schrift: THandle;
BEGIN
  WITH Canvas DO BEGIN
    XSumme := 0; YSumme := 0;
    FOR I := Low(Values) TO High(Values) DO BEGIN
      X := Values[I].X;
      Y := Values[I].Y;
      XSumme := XSumme + X;
      YSumme := YSumme + Y;
      IO := (X >= FMinXWert) AND (X <= FMaxXWert) AND
            (Y >= FMinYWert) AND (Y <= FMaxYWert);
      Brush.Style := bsSolid;
      IF IO THEN BEGIN
        Brush.Color := FBalkenfarbe;
        Pen.Color := FBalkenfarbe;
      END
      ELSE BEGIN
        Brush.Color := FFehlerfarbe;
        Pen.Color := FFehlerfarbe;
      END;
  {---------------------------------------------}
      Pen.Color := clBlack;
      Rectangle(BezugX(X)-2, BezugY(Y)-2, BezugX(X)+2, BezugY(Y)+2);
  {---------------------------------------------}
      Brush.Style := bsClear;
    END; {FOR I:=Low(Values) TO ...}
  {---------------------------------------------}
    {IO-Ellipse malen}
    Ellipse(BezugX(FMinXWert), BezugY(FMinYWert),
            BezugX(FMaxXwert), BezugY(FMaxYWert));
    {Mittelwerte malen}
    IF High(Values) > 0 THEN BEGIN
      XSumme := XSumme / (High(Values) + 1);
      YSumme := YSumme / (High(Values) + 1);
      Pen.Color := clBlue;
      MoveTo(0, BezugY(YSumme));
      LineTo(Width, BezugY(YSumme));
      MoveTo(BezugX(XSumme), 0);
      LineTo(BezugX(XSumme), Height);
    END;

    Alte_Schrift := SelectObject(Handle, CreateFontIndirect(Schriftart));
    Schriftart.lfEscapement:= 0;
    DeleteObject(SelectObject(Handle, CreateFontIndirect(Schriftart)));
    SetTextAlign(Handle, ta_Right OR ta_Top OR ta_NoUpdateCP);
    TextOut(Width-2, BezugY(YSumme)+4, '');
    TextOut(Width-2, BezugY(YSumme)+4, 'x');
    SetTextAlign(Handle, ta_Left OR ta_Top OR ta_NoUpdateCP);
    TextOut(BezugX(XSumme)+4, 4, '');
    TextOut(BezugX(XSumme)+4, 4, 'x');
    DeleteObject(SelectObject(Handle, Alte_Schrift));
  END;  {WITH Canvas DO ...}
END;

procedure TPieHistogram.Paint;
var
  Rect: TRect;
begin
  Rect := GetClientRect;
  with Canvas do begin
    {Intervallbreite in Pixel ermitteln}
      {Automatisch}
    Intervallbreite := Trunc((Width-FYAchsenPosition) / (FAnzahlHorizIntervalle+2));
    Intervallhoehe  := Trunc((Height-FXAchsenPosition) / (FAnzahlVertIntervalle+2));
    {Min. und Max. Werte ermitteln}
    MinValue_O := GetMinValue(TRUE);
    MaxValue_O := GetMaxValue(TRUE);
    {keine Werte --> Histogrammbereich gewhrleisten!}
    IF MaxValue_O.X = MinValue_O.X THEN MaxValue_O.X := MaxValue_O.X + 1;
    IF MaxValue_O.Y = MinValue_O.Y THEN MaxValue_O.Y := MaxValue_O.Y + 1;
    {Werte sortieren, da alle Werte in einem Intervall zusammengefat werden}
    Intervalle_sortieren;
    {Min. und Max. Werte ermitteln}
    MinValue := GetMinValue(FALSE);
    MaxValue := GetMaxValue(FALSE);
    {keine Werte --> Histogrammbereich gewhrleisten!}
    IF MaxValue.X = MinValue.X THEN MaxValue.X := MaxValue.X + 1;
    IF MaxValue.Y = MinValue.Y THEN MaxValue.Y := MaxValue.Y + 1;
    {Hintergrund malen}
    Brush.Color := Color;
    IF NOT(FTransparent) THEN FillRect(Rect);
    Brush.Style := bsClear;
    {Gitter malen}
    Gitter_malen;
    {Werte malen}
    IF FHistogrammTyp = ctXYPlot
      THEN XYPlot_malen
      ELSE Intervalle_malen;
    {Achsen malen}
    Achsen_malen;
    {Achsen bezeichnen}
    Achsen_bezeichnen;
  end;  {WITH Canvas DO ...}
end;

constructor TPieHistogram.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FHistogrammTyp := ctBars;
  FBalkenfarbe := clYellow;
  FFehlerfarbe := clRed;
  FGitterfarbe := clGray;
  FHorizGitter := False;
  FAnzahlHorizIntervalle := 10;
  FAnzahlVertIntervalle := 10;
  FZeigeHorizLabels := True;
  FZeigeVertLabels := True;
  FVertGitter := False;
  Width := 250;
  Height := 200;
  FXAchsenPosition := 50;
  FYAchsenPosition := 50;
  FSkalierungsart := saAutoXY;
  FBezXAchse := 'x';
  FBezYAchse := 'y';
  FMinXWert := 0.0;
  FMaxXWert := 10.0;
  FMinYWert := 0.0;
  FMaxYWert := 10.0;
  FStellen := 4;
  FTransparent := FALSE;
  OnResize := MyResize;
  InitValues;
  with Schriftart, Font do begin
    lfHeight        := 13;
    lfWidth         := 5;
    lfEscapement    := 0;
    lfOrientation   := 0;
    lfWeight        := fw_Normal;
    lfItalic        := 0;
    lfUnderline     := 0;
    lfStrikeOut     := 0;
    lfCharSet       := ANSI_CharSet;
    lfOutPrecision  := Out_Default_Precis;
    lfClipPrecision := Clip_Default_Precis;
    lfQuality       := Proof_Quality;
    lfPitchAndFamily:= Variable_Pitch or FF_Roman;
    StrCopy(lfFaceName,'Times New Roman');
  end;
end;

function TPieHistogram.GetMaxValue(Lang: Boolean): TPieHistogrampaar;
var
  I: Integer;
begin
  IF FHistogrammtyp = ctXYPlot THEN BEGIN
    Result.X := FMaxXWert; Result.Y := FMaxYWert;
  END
  ELSE BEGIN
    IF Lang THEN BEGIN
      IF High(Values) < 0 THEN BEGIN
        Result.X := 1; Result.Y := 1;
      END
      ELSE BEGIN
        Result.X := Values[0].X; Result.Y := Values[0].Y;
        IF FSkalierungsart IN [saAutoXY, saAutoX] THEN BEGIN
          I:=0;
          REPEAT
            if Values[I].X > Result.X then Result.X := Values[I].X;
            inc(I);
          UNTIL I=High(Values);
        END
        ELSE Result.X := FMaxXWert;
        IF FSkalierungsart IN [saAutoXY, saAutoY] THEN BEGIN
          I:=0;
          REPEAT
            if Values[I].Y > Result.Y then Result.Y := Values[I].Y;
            IF I < High(Values) THEN inc(I);
          UNTIL I=High(Values);
        END
        ELSE Result.Y := FMaxYWert;
      END;
    END
    {-------------------------------------------------------}
    ELSE BEGIN
      Result.X := Intervallwerte[0].X; Result.Y := Intervallwerte[0].Y;
      IF FSkalierungsart IN [saAutoXY, saAutoX] THEN BEGIN
        FOR I:=0 TO FAnzahlHorizIntervalle DO
          if Intervallwerte[I].X > Result.X then Result.X := Intervallwerte[I].X;
      END
      ELSE Result.X := FMaxXWert;
      IF FSkalierungsart IN [saAutoXY, saAutoY] THEN BEGIN
        FOR I:=0 TO FAnzahlHorizIntervalle+1 DO
          if Intervallwerte[I].Y > Result.Y then Result.Y := Intervallwerte[I].Y;
      END
      ELSE Result.Y := FMaxYWert;
    END;
  END;
end;

function TPieHistogram.GetMinValue(Lang: Boolean): TPieHistogrampaar;
var
  I: Integer;
begin
  IF FHistogrammtyp = ctXYPlot THEN BEGIN
    Result.X := FMinXWert; Result.Y := FMinYWert;
  END
  ELSE BEGIN
    IF Lang THEN BEGIN
      IF High(Values) < 0 THEN BEGIN
        Result.X := 1; Result.Y := 1;
      END
      ELSE BEGIN
        Result.X := Values[0].X; Result.Y := Values[0].Y;
        IF FSkalierungsart IN [saAutoXY, saAutoX] THEN BEGIN
          I:=0;
          REPEAT
            if Values[I].X < Result.X then Result.X := Values[I].X;
            inc(I);
          UNTIL I=High(Values);
        END
        ELSE Result.X := FMinXWert;
        IF FSkalierungsart IN [saAutoXY, saAutoY] THEN BEGIN
          I:=0;
          REPEAT
            if Values[I].Y < Result.Y then Result.Y := Values[I].Y;
            IF I < High(Values) THEN inc(I);
          UNTIL I=High(Values);
        END
        ELSE Result.Y := FMinYWert;
      END;
    END
    {-------------------------------------------------------}
    ELSE BEGIN
      Result.X := Intervallwerte[0].X; Result.Y := Intervallwerte[0].Y;
      IF FSkalierungsart IN [saAutoXY, saAutoX] THEN BEGIN
        FOR I:=0 TO FAnzahlHorizIntervalle DO
        if Intervallwerte[I].X < Result.X then Result.X := Intervallwerte[I].X;
      END
      ELSE Result.X := FMinXWert;
      IF FSkalierungsart IN [saAutoXY, saAutoY] THEN BEGIN
        FOR I:=0 TO FAnzahlHorizIntervalle+1 DO
        if Intervallwerte[I].Y < Result.Y then Result.Y := Intervallwerte[I].Y;
      END
      ELSE Result.Y := FMinYWert;
    END;
  END;
end;

procedure TPieHistogram.InitValues;
begin
  SetLength(Values, 0);
end;

procedure TPieHistogram.AddValue(X, Y: Single);
BEGIN
  SetLength(Values, High(Values) + 2);
  Values[High(Values)].X := X;
  Values[High(Values)].Y := Y;
END;


{ Register }

end.
