{$I PIETOOLS.INC}
unit PieRegistry;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls,
  Forms, Dialogs, ComCtrls,
  Registry, IniFiles;

type
  TPieRegistry = class(TRegistry)
  private
    { Private-Deklarationen }
  protected
    { Protected-Deklarationen }
  public
    { Public-Deklarationen }
    function PieReadBinaryData(const Name: string; var Buffer; BufSize: Integer): Integer;
    function PieReadBool(const Name: string; const DefaultBool: Boolean): Boolean;
    function PieReadCurrency(const Name: string; const DefaultCurrency: Currency): Currency;
    function PieReadDate(const Name: string; const DefaultDate: TDateTime): TDateTime;
    function PieReadDateTime(const Name: string; const DefaultDateTime: TDateTime): TDateTime;
    function PieReadFloat(const Name: string; const DefaultFloat: Double): Double;
    function PieReadInteger(const Name: string; const DefaultInteger: Integer): Integer;
    function PieReadString(const Name: string; const DefaultString: string): string;
    function PieReadTime(const Name: string; const DefaultTime: TDateTime): TDateTime;
  published
    { Published-Deklarationen }
  end;

  TPieRegistryFile = class(TObject)
  private
    { Private-Deklarationen }
    Lines: TStringList;
    FFileName: string;
    FCurrentPath: string;
    FGeaendert: Boolean;
    PROCEDURE Daten_lesen;
    PROCEDURE Daten_schreiben;
  protected
    { Protected-Deklarationen }
  public
    { Public-Deklarationen }
    constructor Create(FileName: string);
    destructor Destroy; override;
    property CurrentPath: string read FCurrentPath;
    function SectionExists(const Section: string): Boolean;
    procedure ChangeValue(Name, NewValue: string; Names, Values: TStrings);
    procedure DeleteKey(const Section, Ident: String);
    function CreateKey(const Key: String): Boolean;
    function CreateKeyUnder(const Key: String; const Under: String): Boolean;
    function KeyLayer(Key: string): Integer;
    procedure MoveKey(const OldName, NewName: String; Delete: Boolean); overload;
    procedure MoveKey(const OldName, NewName: String; Delete_Old, Overwrite_New: Boolean); overload;
    function OpenKey(const Key: String; CanCreate: Boolean): Boolean;
    function PieDeleteKey(const Key: String): Boolean;
    function ReadString(const Section, Ident, Default: string): string;
    function PieReadString(const Name, Default: string): string;
    function PieReadInteger(const Name: string; Default: Longint): Longint; virtual;
    function PieReadBool(const Name: string; Default: Boolean): Boolean; virtual;
    function PieReadDate(const Name: string; Default: TDateTime): TDateTime; virtual;
    function PieReadDateTime(const Name: string; Default: TDateTime): TDateTime; virtual;
    function PieReadFloat(const Name: string; Default: Double): Double; virtual;
    function PieReadTime(const Name: string; Default: TDateTime): TDateTime; virtual;
    procedure WriteString(const Section, Ident, Value: string);
    procedure PieWriteString(const Name, Value: String);
    procedure PieWriteInteger(const Name: string; Value: Longint); virtual;
    procedure PieWriteBool(const Name: string; Value: Boolean); virtual;
    procedure PieWriteDate(const Name: string; Value: TDateTime); virtual;
    procedure PieWriteDateTime(const Name: string; Value: TDateTime); virtual;
    procedure PieWriteFloat(const Name: string; Value: Double); virtual;
    procedure PieWriteTime(const Name: string; Value: TDateTime); virtual;
    procedure ReadSections(Strings: TStrings);
    procedure ReadSectionValues(const Section: string; Strings: TStrings);
    procedure PieReadSection(Section: string; Names, Values: TStrings);
    procedure PieWriteSection(Section: string; Names, Values: TStrings; Alte_Keys_loeschen: Boolean);
    procedure EraseSection(const Section: string);
    procedure ReadKeyLayers(const Key: String; from_Layer, to_Layer: Integer; Strings: TStrings);  overload;
    procedure ReadKeyLayers(const Key: String; from_Layer, to_Layer: Integer; Nodes: TTreeNodes); overload;
    procedure ReadKeyLayers(const Key: String; from_Layer, to_Layer: Integer; to_ST: Integer; Nodes: TTreeNodes); overload;
    procedure Zerlege_Key(CONST Key: string; VAR Name, Value: string);
    procedure ClearAll;
    function GetSectionIndex(const Section: string): Integer;
    procedure MoveSection(const Section: string; NewIndex: Integer);
    function GetTNPfad(TN: TTreeNode; mit_TN: Boolean): string;
    function GetTNLayer(TN: TTreeNode): Integer;
    procedure FreeTreeNodes(Nodes: TTreeNodes);
  published
    { Published-Deklarationen }
  end;

  PTreeNodeData = ^TTreeNodeData;
  TTreeNodeData = RECORD
    Path         : string;  {"Gert\SNr\Kanal\Gruppe\Auftrag"}
    Filename     : string;  {"34521"}
    StrukturType : Integer; {1=Gert ... 5 = Auftrag}
    Layer        : Integer; {1 ... 99}
  END;

implementation

function TPieRegistry.PieReadBinaryData(const Name: string; var Buffer; BufSize: Integer): Integer;
begin
  TRY
    Result := ReadBinaryData(Name, Buffer, BufSize);
  EXCEPT
    on ERegistryException DO BEGIN
      WriteBinaryData(Name, Buffer, BufSize);
      Result := BufSize;
    END;
  END;
end;

function TPieRegistry.PieReadBool(const Name: string; const DefaultBool: Boolean): Boolean;
begin
  TRY
    Result := ReadBool(Name);
  EXCEPT
    on ERegistryException DO BEGIN
      WriteBool(Name, DefaultBool);
      Result := DefaultBool;
    END;
  END;
end;

function TPieRegistry.PieReadCurrency(const Name: string; const DefaultCurrency: Currency): Currency;
begin
  TRY
    Result := ReadCurrency(Name);
  EXCEPT
    on ERegistryException DO BEGIN
      WriteCurrency(Name, DefaultCurrency);
      Result := DefaultCurrency;
    END;
  END;
end;

function TPieRegistry.PieReadDate(const Name: string; const DefaultDate: TDateTime): TDateTime;
begin
  TRY
    Result := ReadDate(Name);
  EXCEPT
    on ERegistryException DO BEGIN
      WriteDate(Name, DefaultDate);
      Result := DefaultDate;
    END;
  END;
end;

function TPieRegistry.PieReadDateTime(const Name: string; const DefaultDateTime: TDateTime): TDateTime;
begin
  TRY
    Result := ReadDateTime(Name);
  EXCEPT
    on ERegistryException DO BEGIN
      WriteDateTime(Name, DefaultDateTime);
      Result := DefaultDateTime;
    END;
  END;
end;

function TPieRegistry.PieReadFloat(const Name: string; const DefaultFloat: Double): Double;
begin
  TRY
    Result := ReadFloat(Name);
  EXCEPT
    on ERegistryException DO BEGIN
      WriteFloat(Name, DefaultFloat);
      Result := DefaultFloat;
    END;
  END;
end;

function TPieRegistry.PieReadInteger(const Name: string; const DefaultInteger: Integer): Integer;
begin
  TRY
    Result := ReadInteger(Name);
  EXCEPT
    on ERegistryException DO BEGIN
      WriteInteger(Name, DefaultInteger);
      Result := DefaultInteger;
    END;
  END;
end;

function TPieRegistry.PieReadString(const Name: string; const DefaultString: string): string;
VAR
  S: String;
begin
  TRY
    S := ReadString(Name);
  EXCEPT
    on ERegistryException DO BEGIN
      WriteString(Name, DefaultString);
      S := DefaultString;
    END;
  END;
  IF S = '' THEN BEGIN
    WriteString(Name, DefaultString);
    S := DefaultString;
  END;
  Result := S;
end;

function TPieRegistry.PieReadTime(const Name: string; const DefaultTime: TDateTime): TDateTime;
begin
  TRY
    Result := ReadTime(Name);
  EXCEPT
    on ERegistryException DO BEGIN
      WriteTime(Name, DefaultTime);
      Result := DefaultTime;
    END;
  END;
end;

{*********************************************************}
{*********************************************************}
{*********************************************************}
{TPieRegistryFile}
constructor TPieRegistryFile.Create(FileName: string);
BEGIN
  inherited Create;
  FFileName := FileName;
  Lines := TStringList.Create;
  Lines.Duplicates := dupAccept;
  Lines.Sorted := FALSE;
  Daten_lesen;
END;

destructor TPieRegistryFile.Destroy;
VAR
  I: Integer;
BEGIN
  Daten_schreiben;
  FOR I:=0 TO Lines.Count-1 DO Lines.Objects[I].Free;
  Lines.Free;
  inherited Destroy;
END;

PROCEDURE TPieRegistryFile.Daten_lesen;
VAR
  F: Text;
  S: string;
  Keys: TStringList;
BEGIN
  {Alle Daten lesen}
  Lines.BeginUpdate;
  Keys := NIL;
  AssignFile(F, FFileName);
  {$I- }system.reset(F); {$I+}
  IF IOResult <> 0 THEN exit;
  REPEAT
    readln(F, S);
    IF (length(S) > 0) AND (pos('Dummy', S) = 0) THEN BEGIN
      {Section gefunden}
      IF (S[1] = '[') THEN BEGIN
        Keys := TStringList.Create;
        Keys.Duplicates := dupAccept;
        Keys.Sorted := FALSE;
        Lines.AddObject(S, Keys);
      END
      {Eintrag gefunden}
      ELSE BEGIN
        Keys.Add(S);
      END;
    END;
  UNTIL EOF(F);
  CloseFile(F);
  Lines.EndUpdate;
  FGeaendert := FALSE;
END;

PROCEDURE TPieRegistryFile.Daten_schreiben;
VAR
  F: Text;
  I, J: Integer;
  Keys: TStringList;
BEGIN
  IF NOT(FGeaendert) THEN exit;
  IF Lines.Count = 0 THEN exit;
  {Daten speichern}
  AssignFile(F, FFileName);
  system.rewrite(F);
  FOR I:=0 TO Lines.Count-1 DO BEGIN
    Keys := Lines.Objects[I] as TStringList;
    writeln(F, Lines.Strings[I]);
    FOR J:=0 TO Keys.Count-1 DO writeln(F, Keys.Strings[J]);
  END;
  CloseFile(F);
END;

function TPieRegistryFile.SectionExists(const Section: string): Boolean;
begin
  Result := Lines.IndexOf('[' + Section + ']') > -1;
end;

function Teile_bis_BS(const OldString: string; VAR TeilString, RestString: string): Boolean;
VAR
  P: Integer;
begin
  IF OldString = '' THEN Result := FALSE
  ELSE BEGIN
    P := pos('\', OldString);
    IF P > 0 THEN BEGIN
      TeilString := copy(OldString, 1, P-1);
      RestString := copy(OldString, P+1, length(OldString)-P);
    END
    ELSE BEGIN
      TeilString := OldString;
      RestString := '';
    END;
    Result := TRUE;
  END;
end;

procedure TPieRegistryFile.Zerlege_Key(CONST Key: string; VAR Name, Value: string);
VAR
  PV: Integer;
begin
  PV := pos('=', Key);
  Name := copy(Key, 1, PV-1);
  Value := copy(Key, PV+1, length(Key) - PV);
end;

function TPieRegistryFile.KeyLayer(Key: string): Integer;
VAR
  TS, RS: string;
BEGIN
  RS := Key;
  Result := 0;
  WHILE Teile_bis_BS(RS, TS, RS) DO inc(Result);
END;

function TPieRegistryFile.OpenKey(const Key: String; Cancreate: boolean): Boolean;
var
  S: string;
begin
  S := Key;

  {evt. Section erzeugen}
  IF NOT(SectionExists(S)) AND CanCreate THEN CreateKey(Key);

  IF SectionExists(S) THEN BEGIN
    FCurrentPath := S;
    Result := TRUE;
  END ELSE Result := FALSE;
end;

function TPieRegistryFile.ReadString(const Section, Ident, Default: string): string;
VAR
  S: string;
  I, SectionIndex: Integer;
  Gefunden : Boolean;
  Keys: TStringList;
begin
  {Daten suchen}
  SectionIndex := Lines.IndexOf('[' + Section + ']');

  IF SectionIndex > -1 THEN BEGIN
    Keys := Lines.Objects[SectionIndex] as TStringList;
    Gefunden := FALSE;
    I := 0;
    WHILE (I < Keys.Count) AND NOT(Gefunden) DO BEGIN
      S := Keys.Strings[I];
      Gefunden := (pos(Ident, S) = 1) AND (pos('=',S) = pos(Ident, S) + length(Ident));
      inc(I);
    END;

    IF Gefunden THEN Result := copy(S, pos('=', S)+1, length(S)-pos('=', S))
    ELSE Result := Default;
  END
  ELSE Result := Default;
END;

function TPieRegistryFile.PieReadString(const Name, Default: string): string;
begin
  Result := ReadString(FCurrentPath, Name, Default);
end;

function TPieRegistryFile.PieReadInteger(const Name: string; Default: Longint): Longint;
begin
  TRY
    Result := StrToInt(ReadString(FCurrentPath, Name, IntToStr(Default)));
  EXCEPT
    on EConvertError DO BEGIN
      Result := Default;
    END;
  END;  
end;

function TPieRegistryFile.PieReadBool(const Name: string; Default: Boolean): Boolean;
VAR
  S: String;
begin
  IF Default THEN S := '1' ELSE S := '0';
  Result := ReadString(FCurrentPath, Name, S) = '1';
end;

function TPieRegistryFile.PieReadDate(const Name: string; Default: TDateTime): TDateTime;
VAR
  S: String;
  DS: char;
begin
  S := ReadString(FCurrentPath, Name, '');
  DS := DateSeparator;
  TRY
    DateSeparator := '.';
    TRY
      IF S='' THEN Result := Default ELSE Result := StrToDate(S);
    EXCEPT
      on EConvertError DO Result := Default;
    END;
  FINALLY
    DateSeparator := DS;
  END;
end;

function TPieRegistryFile.PieReadDateTime(const Name: string; Default: TDateTime): TDateTime;
VAR
  S: String;
  DS, TS: char;
begin
  S := ReadString(FCurrentPath, Name, '');
  DS := DateSeparator;
  TS := TimeSeparator;
  TRY
    DateSeparator := '.';
    TimeSeparator := ':';
    TRY
      IF S='' THEN Result := Default ELSE Result := StrToDateTime(S);
    EXCEPT
      on EConvertError DO Result := Default;
    END;
  FINALLY
    DateSeparator := DS;
    TimeSeparator := TS;
  END;
end;

function TPieRegistryFile.PieReadFloat(const Name: string; Default: Double): Double;
begin
  Result := StrToFloat(ReadString(FCurrentPath, Name, FloatToStr(Default)));
end;

function TPieRegistryFile.PieReadTime(const Name: string; Default: TDateTime): TDateTime;
VAR
  S: String;
  TS: char;
begin
  S := ReadString(FCurrentPath, Name, '');
  TS := TimeSeparator;
  TRY
    TimeSeparator := ':';
    TRY
      IF S='' THEN Result := Default ELSE Result := StrToTime(S);
    EXCEPT
      on EConvertError DO Result := Default;
    END;
  FINALLY
    TimeSeparator := TS;
  END;
end;

procedure TPieRegistryFile.WriteString(const Section, Ident, Value: string);
VAR
  S: string;
  I, SectionIndex: Integer;
  Gefunden : Boolean;
  Keys: TStringList;
begin
  {Daten modifizieren}
  SectionIndex := Lines.IndexOf('[' + Section + ']');

  IF SectionIndex > -1 THEN BEGIN
    Keys := Lines.Objects[SectionIndex] as TStringList;
    Gefunden := FALSE;
    I := 0;
    WHILE (I < Keys.Count) AND NOT(Gefunden) DO BEGIN
      S := Keys.Strings[I];
      Gefunden := (pos(Ident, S) = 1) AND (pos('=',S) = pos(Ident, S) + length(Ident));
      inc(I);
    END;

    IF Gefunden THEN Keys.Strings[I-1] := Ident + '=' + Value
    ELSE Keys.Add(Ident + '=' + Value);
  END
  ELSE BEGIN
    Keys := TStringList.Create;
    Keys.Duplicates := dupAccept;
    Keys.Sorted := FALSE;
    Lines.AddObject('[' + Section + ']', Keys);
    Keys.Add(Ident + '=' + Value);
  END;
  FGeaendert := TRUE;
end;

procedure TPieRegistryFile.PieWriteString(const Name, Value: String);
begin
  WriteString(FCurrentPath, Name, Value);
end;

procedure TPieRegistryFile.PieWriteInteger(const Name: string; Value: Longint);
begin
  WriteString(FCurrentPath, Name, IntToStr(Value));
end;

procedure TPieRegistryFile.PieWriteBool(const Name: string; Value: Boolean);
begin
  IF Value THEN WriteString(FCurrentPath, Name, '1')
           ELSE WriteString(FCurrentPath, Name, '0');
end;

procedure TPieRegistryFile.PieWriteDate(const Name: string; Value: TDateTime);
VAR
  S: string;
begin
  DateTimeToString(S, 'dd.mm.yyyy', Value);
  WriteString(FCurrentPath, Name, S);
end;

procedure TPieRegistryFile.PieWriteDateTime(const Name: string; Value: TDateTime);
VAR
  S: string;
begin
  DateTimeToString(S, 'dd.mm.yyyy hh:nn:ss', Value);
  WriteString(FCurrentPath, Name, S);
end;

procedure TPieRegistryFile.PieWriteFloat(const Name: string; Value: Double);
begin
  WriteString(FCurrentPath, Name, FloatToStr(Value));
end;

procedure TPieRegistryFile.PieWriteTime(const Name: string; Value: TDateTime);
VAR
  S: string;
begin
  DateTimeToString(S, 'hh:nn:ss', Value);
  WriteString(FCurrentPath, Name, S);
end;

procedure TPieRegistryFile.DeleteKey(const Section, Ident: String);
VAR
  SectionIndex: Integer;
  Keys: TStringList;
begin
  SectionIndex := Lines.IndexOf('[' + Section + ']');
  IF SectionIndex > -1 THEN BEGIN
    Keys := Lines.Objects[SectionIndex] as TStringList;
    IF Keys.IndexOf(Ident) > -1 THEN BEGIN
      Keys.Delete(Keys.IndexOf(Ident));
      FGeaendert := TRUE;
    END;
  END;
end;

function TPieRegistryFile.CreateKey(const Key: String): Boolean;
VAR
  S, TS, RS: string;
  Keys: TStringList;
begin
  Result := TRUE;
  TRY
    {existiert Key schon, dann exit!}
    IF (Key = '') OR SectionExists(Key) THEN exit;
    {alle Unterschlssel anlegen}
    S := '';
    RS := Key;
    WHILE Teile_bis_BS(RS, TS, RS) DO BEGIN
      IF S <> '' THEN S := S + '\';
      S := S + TS;

      {evt. Section erzeugen}
      IF NOT(SectionExists(S)) THEN BEGIN
        Keys := TStringList.Create;
        Lines.AddObject('[' + S + ']', Keys);
        FGeaendert := TRUE;
      END;
    END;
    FCurrentPath := Key;
  EXCEPT
    Result := FALSE;
  END;
end;

function TPieRegistryFile.CreateKeyUnder(const Key: String; const Under: String): Boolean;
VAR
  S, TS, RS: string;
  Keys: TStringList;
  Index: Integer;
begin
  Result := TRUE;
  TRY
    {existiert Key schon, dann exit!}
    IF (Key = '') OR SectionExists(Key) THEN exit;
    IF (Under <> '') AND NOT(SectionExists(Under)) THEN BEGIN Result := FALSE; exit; END;
    {Index ermitteln}
    IF (Under = '') THEN Index := Lines.Count
                    ELSE Index := Lines.IndexOf('[' + Under + ']') + 1;
    {alle Unterschlssel anlegen}
    S := '';
    RS := Key;
    WHILE Teile_bis_BS(RS, TS, RS) DO BEGIN
      IF S <> '' THEN S := S + '\';
      S := S + TS;

      {evt. Section erzeugen}
      IF NOT(SectionExists(S)) THEN BEGIN
        Keys := TStringList.Create;
        Lines.InsertObject(Index, '[' + S + ']', Keys);
        FGeaendert := TRUE;
      END;
    END;
    FCurrentPath := Key;
  EXCEPT
    Result := FALSE;
  END;
end;

procedure TPieRegistryFile.ReadSections(Strings: TStrings);
VAR
  I: Integer;
  S: String;
BEGIN
  // evt. Objekte lschen
  FOR I:=0 TO Strings.Count-1 DO BEGIN
    Strings.Objects[I].Free;
    Strings.Objects[I] := NIL;
  END;
  Strings.Clear;
  FOR I:=0 TO Lines.Count-1 DO BEGIN
    S := Lines.Strings[I];
    IF S[1] = '[' THEN delete(S, 1, 1);
    IF S[length(S)] = ']' THEN delete(S, length(S), 1);
    Strings.Add(S);
  END;
END;

function TPieRegistryFile.PieDeleteKey(const Key: String): Boolean;
VAR
  S: string;
  I, P: Integer;
begin
  Result := TRUE;
  TRY
    FOR I:=Lines.Count -1 DOWNTO 0 DO BEGIN
      {wenn Key in Sectionname, dann Section lschen}
      S := Lines.Strings[I];
      P := Pos('[' + Key + ']', S);
      IF P = 0 THEN P := Pos('[' + Key + '\', S);
      IF P > 0 THEN BEGIN
        Lines.Objects[I].Free;
        Lines.Delete(I);
        FGeaendert := TRUE;
      END;
    END;
  EXCEPT
    Result := FALSE;
  END;
end;

procedure TPieRegistryFile.MoveKey(const OldName, NewName: String; Delete: Boolean);
begin
  MoveKey(OldName, NewName, Delete, FALSE);
END;

procedure TPieRegistryFile.MoveKey(const OldName, NewName: String; Delete_Old, Overwrite_New: Boolean);
VAR
  I, P: Integer;
  S: string;
  Keys, OldKeys: TStringList;
begin
  {existiert NewName schon, dann exit!}
  IF SectionExists(NewName) THEN BEGIN
    IF NOT(Overwrite_New) THEN exit
    ELSE PieDeleteKey(NewName);
  END;

  FOR I:=0 TO Lines.Count -1 DO BEGIN
    {wenn OldName in Sectionname, dann Section umbenennen}
    S := Lines.Strings[I];
    P := Pos(OldName+'\', S);    {S enthlt OldName}
    IF '[' + OldName + ']' = S THEN P := 2;  {S gleich OldName}
    IF (P > 0) OR (OldName=S) THEN BEGIN
      {Section umbenennen}
      system.Delete(S, P, length(OldName));
      system.Insert(NewName, S, P);
      IF Delete_Old THEN Lines.Strings[I] := S  {umbenennen}
      ELSE BEGIN                            {kopieren}
        OldKeys := Lines.Objects[I] as TStringList;
        Keys := TStringList.Create;
        Keys.Assign(OldKeys);
        Lines.AddObject(S, Keys);
      END;
    END;
  END;
  FGeaendert := TRUE;
end;

procedure TPieRegistryFile.ReadKeyLayers(const Key: String; from_Layer, to_Layer: Integer; Strings: TStrings);
VAR
  I: Integer;
  S: string;
  Loeschen: Boolean;
begin
  ReadSections(Strings);
  FOR I:=Strings.Count-1 DOWNTO 0 DO BEGIN
    S := Strings.Strings[I];
    Loeschen := FALSE;
    {Alle Keys auerhalb der Layers ausblenden}
    IF (((Key <> '') AND (pos(Key, S) > 0)) OR (Key = '')) AND
      (KeyLayer(S) >= from_Layer) AND (KeyLayer(S) <= to_Layer)
      THEN
      ELSE BEGIN
        Loeschen := TRUE;
      END;
    {Allgemein-Key ausblenden}
    IF S = 'Allgemein' THEN Loeschen := TRUE;
    IF S = 'General' THEN Loeschen := TRUE;
    IF Loeschen THEN Strings.Delete(I);
  END;
end;

procedure TPieRegistryFile.ReadKeyLayers(const Key: String; from_Layer, to_Layer: Integer; Nodes: TTreeNodes);
VAR
  I, Z: Integer;
  S, RS, TS, FS: string;
  GStrings: TStringList;
  TN, TNS: TTreeNode;
  TD: PTreeNodeData;
begin
  GStrings := TStringList.Create;
  TRY
    // evt. Objekte lschen
    FreeTreeNodes(Nodes);
    {nur schlssige Keys zulassen}
    IF KeyLayer(Key) < from_Layer - 1 THEN exit;
    {Keys in normales TStrings einlesen}
    ReadKeyLayers(Key, from_Layer, to_Layer, GStrings);
    {Keys in TTreeNodes kopieren}
    FOR I:=0 TO GStrings.Count-1 DO BEGIN
      S := GStrings.Strings[I];
      TN := NIL;
      TNS := Nodes.GetFirstNode;
      Z := 0;
      RS := S;
      FS := '';
      WHILE Teile_bis_BS(RS, TS, RS) DO BEGIN
        inc(Z);
        IF FS = '' THEN FS := TS ELSE FS := FS + '\' + TS;
        IF (Z >= from_Layer) AND (Z <= to_Layer) THEN BEGIN
          {Aufpassen, da keine doppelten Key bestehen!}
          WHILE NOT((TNS = NIL) OR ((PTreeNodeData(TNS.Data)^.Path = FS) AND
                (PTreeNodeData(TNS.Data)^.Layer = Z))) DO
                TNS := TNS.GetNext;
          IF (TNS <> NIL) AND (PTreeNodeData(TNS.Data)^.Path = FS) AND (PTreeNodeData(TNS.Data)^.Layer = Z)
            THEN TN := TNS
            ELSE BEGIN
              TD := New(PTreeNodeData);
              TD.Path := FS;
              TD.Layer := Z;
              OpenKey(FS, FALSE);
              TD.StrukturType := PieReadInteger('StrukturType', 0);
              TD.Filename := PieReadString('Filename', '');
              TN := Nodes.AddChildObject(TN, TS, TD);
              CASE TD.StrukturType OF
              1: TN.StateIndex := -1;  {Gertetyp}
              2: TN.StateIndex := -2;  {Gert (SNr)}
              3: TN.StateIndex := -3;  {Kanal}
              4: TN.StateIndex := -4;  {Gruppe}
              5: TN.StateIndex := -5;  {Auftrag}
              END;
              IF TD.StrukturType IN [1..4] THEN TN.ImageIndex := 6;  {Gruppenimage}
              IF TD.StrukturType IN [5] THEN TN.ImageIndex := 5;  {Auftragsimage}
            END;
        END;
      END;
    END;
  FINALLY
    GStrings.Clear;
    GStrings.Free;
  END;
end;

procedure TPieRegistryFile.ReadKeyLayers(const Key: String; from_Layer, to_Layer: Integer;
          to_ST: Integer; Nodes: TTreeNodes);
VAR
  I, Z, ST: Integer;
  S, RS, TS, FS: string;
  GStrings: TStringList;
  TN, TNS: TTreeNode;
  TD: PTreeNodeData;
begin
  GStrings := TStringList.Create;
  TRY
    // evt. Objekte lschen
    FreeTreeNodes(Nodes);
    {nur schlssige Keys zulassen}
    IF KeyLayer(Key) < from_Layer - 1 THEN exit;
    {Keys in normales TStrings einlesen}
    ReadKeyLayers(Key, from_Layer, to_Layer, GStrings);
    {Keys in TTreeNodes kopieren}
    FOR I:=0 TO GStrings.Count-1 DO BEGIN
      S := GStrings.Strings[I];
      TN := NIL;
      TNS := Nodes.GetFirstNode;
      Z := 0;
      RS := S;
      FS := '';
      WHILE Teile_bis_BS(RS, TS, RS) DO BEGIN
        inc(Z);
        IF FS = '' THEN FS := TS ELSE FS := FS + '\' + TS;
        IF (Z >= from_Layer) AND (Z <= to_Layer) THEN BEGIN
          {Aufpassen, da keine doppelten Key bestehen!}
          WHILE NOT((TNS = NIL) OR ((PTreeNodeData(TNS.Data)^.Path = FS) AND
                (PTreeNodeData(TNS.Data)^.Layer = Z))) DO
                TNS := TNS.GetNext;
          IF (TNS <> NIL) AND (PTreeNodeData(TNS.Data)^.Path = FS) AND (PTreeNodeData(TNS.Data)^.Layer = Z)
            THEN TN := TNS
            ELSE BEGIN
              OpenKey(FS, FALSE);
              ST := PieReadInteger('StrukturType', 0);
              IF ST <= to_ST THEN BEGIN
                TD := New(PTreeNodeData);
                TD.StrukturType := ST;
                TD.Path := FS;
                TD.Layer := Z;
                TD.Filename := PieReadString('Filename', '');
                TN := Nodes.AddChildObject(TN, TS, TD);
                CASE TD.StrukturType OF
                1: TN.StateIndex := -1;  {Gertetyp}
                2: TN.StateIndex := -2;  {Gert (SNr)}
                3: TN.StateIndex := -3;  {Kanal}
                4: TN.StateIndex := -4;  {Gruppe}
                5: TN.StateIndex := -5;  {Auftrag}
                END;
                IF TD.StrukturType IN [1..4] THEN TN.ImageIndex := 6;  {Gruppenimage}
                IF TD.StrukturType IN [5] THEN TN.ImageIndex := 5;  {Auftragsimage}
              END;
            END;
        END;
      END;
    END;
  FINALLY
    GStrings.Clear;
    GStrings.Free;
  END;
end;

procedure TPieRegistryFile.ReadSectionValues(const Section: string; Strings: TStrings);
VAR
  SectionIndex: Integer;
begin
  SectionIndex := Lines.IndexOf('[' + Section + ']');
  IF SectionIndex > -1 
    THEN Strings.Assign(Lines.Objects[SectionIndex] as TStringList)
    ELSE Strings.Clear;
end;

procedure TPieRegistryFile.PieReadSection(Section: string; Names, Values: TStrings);
VAR
  I: Integer;
  N, V: string;
begin
  Values.Clear;
  Names.Clear;
  ReadSectionValues(Section, Names);
  FOR I:=0 TO Names.Count-1 DO BEGIN
    Zerlege_Key(Names.Strings[I], N, V);
    Names.Strings[I] := N;
    Values.Add(V);
  END;
end;

procedure TPieRegistryFile.EraseSection(const Section: string);
VAR
  SectionIndex: Integer;
begin
  SectionIndex := Lines.IndexOf('[' + Section + ']');
  IF SectionIndex > -1 THEN BEGIN
    Lines.Objects[SectionIndex].Free;
    Lines.Delete(SectionIndex);
    FGeaendert := TRUE;
  END;
end;

procedure TPieRegistryFile.PieWriteSection(Section: string; Names, Values: TStrings; Alte_Keys_loeschen: Boolean);
VAR
  I: Integer;
begin
  IF Alte_Keys_loeschen THEN EraseSection(Section);
  OpenKey(Section, TRUE);
  FOR I:=0 TO Names.Count-1 DO
    PieWriteString(Names.Strings[I], Values.Strings[I]);
end;

procedure TPieRegistryFile.ChangeValue(Name, NewValue: string; Names, Values: TStrings);
VAR
  I: Integer;
begin
  I := Names.IndexOfName(Name);
  IF (I>-1) AND (I<Values.Count-1) THEN Values.Strings[I] := NewValue;
end;

procedure TPieRegistryFile.ClearAll;
VAR
  I: Integer;
begin
  FOR I:=0 TO Lines.Count-1 DO BEGIN
    Lines.Objects[I].Free;
    Lines.Objects[I] := NIL;
  END;  
  Lines.Clear;
end;

function TPieRegistryFile.GetSectionIndex(const Section: string): Integer;
begin
  Result := Lines.IndexOf('[' + Section + ']');
end;

procedure TPieRegistryFile.MoveSection(const Section: string; NewIndex: Integer);
begin
  Lines.Move(GetSectionIndex(Section), NewIndex);
end;

function TPieRegistryFile.GetTNPfad(TN: TTreeNode; mit_TN: Boolean): string;
VAR
  G: string;
  I: Integer;
BEGIN
  IF NOT(Assigned(TN)) OR NOT(Assigned(TN.Data)) THEN BEGIN Result := ''; exit; END;
  G := PTreeNodeData(TN.Data)^.Path;
  IF mit_TN THEN G := G + '\'
  ELSE BEGIN
    I := Length(G);
    WHILE ((I > 0) AND (G[I] <> '\')) DO dec(I);
    G := copy(G, 1, I-1);
  END;
  Result := G;
END;

function TPieRegistryFile.GetTNLayer(TN: TTreeNode): Integer;
BEGIN
  IF NOT(Assigned(TN)) OR NOT(Assigned(TN.Data)) THEN BEGIN Result := 0; exit; END;
  Result := KeyLayer(PTreeNodeData(TN.Data)^.Path);
END;

procedure TPieRegistryFile.FreeTreeNodes(Nodes: TTreeNodes);
VAR
  TN: TTreeNode;
  TD: PTreeNodeData;
BEGIN
  TN := Nodes.GetFirstNode;
  while assigned(TN) DO BEGIN
    TD := TN.Data;
    TD.Path := '';
    TD.Filename := '';
    dispose(TN.Data);
    TN.Data := NIL;
    TN := TN.GetNext;
  END;
  Nodes.Clear;
END;

end.

