{$I PIETOOLS.INC}
{ Autor: Ingolf Pietschmann.
  Dieser Quelltext ist Freeware. Die Verwendung und Weitergabe dieser Sourcen zu
  privaten nicht kommerziellen Zwecken ist ausdrcklich erwnscht.
  Die Verwendung zu kommerziellen Zwecken ist nur mit Erlaubnis des Autors
  gestattet. Den Autor knnen Sie unter "Support@Pie-Tools.de" erreichen.

  These sources are freeware. The usage and distribution of these sources for
  private, not commercial purposes is explicit desired.
  The usage for commercial purposes is only permitted in agreement of the author.
  The author can be reached by "Support@Pie-Tools.de".
}
unit PieProgressBar;

interface

uses
  Messages, WinTypes, Classes, Graphics, Controls,
  Menus, ExtCtrls, Comctrls;

type
  TFrameStyle = (fsNone, fsLowered, fsRaised);
  TSide = (sdLeft, sdTop, sdRight, sdBottom);
  TSides = set of TSide;

  TSegmentRange = 1..100;
  TProgressChangeEvent = procedure(Sender: TObject; Position: Double) of object;

  TPieCustomProgressBar = class( TGraphicControl )
  private
    FBackColor: TColor;
    FBarColor: TColor;
    FDecimals: Byte;
    FSmooth: Boolean;
    FBevelWidth: TBevelWidth;
    FBorderColor: TColor;
    FBorderInner: TFrameStyle;
    FBorderOuter: TFrameStyle;
    FBorderWidth: TBorderWidth;
    FNumSegments: TSegmentRange;
    FOrientation: TProgressBarOrientation;
    FPosition: Double;
    FMin: Double;
    FMax: Double;
    FShowText: Boolean;
    FTextUnit: string;
    FStep: Double;
    FOnChange: TProgressChangeEvent;
    procedure SetBackColor( Value: TColor );
    procedure SetBarColor( Value: TColor );
    procedure SetDecimals(Value: Byte);
    procedure SetSmooth(Value: Boolean);
    procedure SetBevelWidth( Value: TBevelWidth );
    procedure SetBorderColor( Value: TColor );
    procedure SetBorderInner( Value: TFrameStyle );
    procedure SetBorderOuter( Value: TFrameStyle );
    procedure SetBorderWidth( Value: TBorderWidth );
    procedure SetNumSegments( Value: TSegmentRange );
    procedure SetOrientation(Value: TProgressBarOrientation);
    procedure SetPosition(Value: Double);
    procedure SetMin(Value: Double);
    procedure SetMax(Value: Double);
    procedure SetShowText(Value: Boolean);
    procedure SetTextUnit(Value: string);
    procedure CMHitTest( var Msg: TMessage ); message cm_HitTest;
    { Special Drawing Procedures }
    function DrawSides(Canvas: TCanvas; Bounds: TRect; ULColor, LRColor: TColor; Sides: TSides ): TRect;
    function DrawBevel(Canvas: TCanvas; Bounds: TRect; ULColor, LRColor: TColor; Width: Integer; Sides: TSides ): TRect;
    function DrawCtl3DBorderSides(Canvas: TCanvas; Bounds: TRect; Lowered: Boolean; Sides: TSides ): TRect;
    function DrawBorderSides(Canvas: TCanvas; Bounds: TRect; Style: TFrameStyle): TRect;
    procedure DrawLEDBar(Canvas: TCanvas; DrawRct: TRect; PercentStr: string);
    procedure DrawPercentBar(Canvas: TCanvas; DrawRct: TRect; PercentStr: string);
  protected
    procedure Paint; override;
    procedure PositionChanged; dynamic;
    property BackColor: TColor          read FBackColor     write SetBackColor   default clWhite;
    property BarColor: TColor           read FBarColor      write SetBarColor    default clHighlight;
    property Decimals: Byte             read FDecimals      write SetDecimals    default 2;
    property Smooth: Boolean            read FSmooth        write SetSmooth      default FALSE;
    property BevelWidth: TBevelWidth    read FBevelWidth    write SetBevelWidth  default 1;
    property BorderColor: TColor        read FBorderColor   write SetBorderColor default clBtnFace;
    property BorderInner: TFrameStyle   read FBorderInner   write SetBorderInner default fsNone;
    property BorderOuter: TFrameStyle   read FBorderOuter   write SetBorderOuter default fsLowered;
    property BorderWidth: TBorderWidth  read FBorderWidth   write SetBorderWidth;
    property NumSegments: TSegmentRange read FNumSegments   write SetNumSegments default 20;
    property Orientation: TProgressBarOrientation read FOrientation   write SetOrientation default pbHorizontal;
    property Position: Double           read FPosition      write SetPosition;
    property Min: Double                read FMin           write SetMin;
    property Max: Double                read FMax           write SetMax;
    property ShowText: Boolean          read FShowText      write SetShowText    default True;
    property TextUnit: string           read FTextUnit      write SetTextUnit;
    property Step: Double               read FStep          write FStep;
    property OnChange: TProgressChangeEvent read FOnChange  write FOnChange;
  public
    constructor Create( AOwner: TComponent ); override;
    procedure SetBounds( ALeft, ATop, AWidth, AHeight: Integer ); override;
    { Inherited Properties & Events }
    property Align;
    property Font;
    property Height default 24;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ShowHint;
    property Visible;
    property Width default 200;
    property OnClick;
    property OnDblClick;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;


  TPieProgressBar = class( TPieCustomProgressBar )
  private
  public
    procedure StepIt;
    procedure StepBy(Delta: Double);
  published
    property Align;
    property BackColor;
    property BarColor;
    property Smooth;
    property BevelWidth;
    property BorderColor;
    property BorderInner;
    property BorderOuter;
    property BorderWidth;
    property Decimals;
    property Font;
    property Height;
    property Max;
    property Min;
    property NumSegments;
    property Orientation;
    property ParentFont;
    property ParentShowHint;
    property Position;
    property PopupMenu;
    property ShowHint;
    property ShowText;
    property TextUnit;
    property Step;
    property Visible;
    property Width;
    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;



implementation

uses
  SysUtils, WinProcs, PieHerk;

const
  sdAllSides = [ sdLeft, sdTop, sdRight, sdBottom ];

function TPieCustomProgressBar.DrawSides(Canvas: TCanvas; Bounds: TRect; ULColor, LRColor: TColor;
                   Sides: TSides): TRect;
begin
  with Canvas, Bounds do begin
    Pen.Color := ULColor;
    if sdLeft in Sides then begin
      MoveTo( Left, Top );
      LineTo( Left, Bottom );
    end;
    if sdTop in Sides then begin
      MoveTo( Left, Top );
      LineTo( Right, Top );
    end;
    Pen.Color := LRColor;
    if sdRight in Sides then begin
      MoveTo( Right - 1, Top );
      LineTo( Right - 1, Bottom );
    end;
    if sdBottom in Sides then begin
      MoveTo( Left, Bottom - 1 );
      LineTo( Right, Bottom - 1 );
    end;
  end;

  if sdLeft   in Sides then Inc(Bounds.Left);
  if sdTop    in Sides then Inc(Bounds.Top);
  if sdRight  in Sides then Dec(Bounds.Right);
  if sdBottom in Sides then Dec(Bounds.Bottom);

  Result := Bounds;
end; {= DrawSides =}

function TPieCustomProgressBar.DrawBevel(Canvas: TCanvas; Bounds: TRect;
                   ULColor, LRColor: TColor;
                   Width: Integer; Sides: TSides): TRect;
var
  I: Integer;
begin
  Canvas.Pen.Width := 1;
  for I := 1 to Width do                         { Loop through width of bevel }
    Bounds := DrawSides(Canvas, Bounds, ULColor, LRColor, Sides);
  Result := Bounds;
end;

function TPieCustomProgressBar.DrawCtl3DBorderSides(Canvas: TCanvas; Bounds: TRect;
                              Lowered: Boolean; Sides: TSides): TRect;
const
  Colors: array[1..4, Boolean] of TColor = ((clBtnFace, clBtnShadow),
                                            (clBtnText, clBtnHighlight),
                                            (clBtnHighlight, clBtnText),
                                            (clBtnShadow, clBtnFace));
begin
  Bounds := DrawBevel(Canvas, Bounds, Colors[1, Lowered],
                      Colors[2, Lowered], 1, Sides);
  Result := DrawBevel(Canvas, Bounds, Colors[3, Lowered],
                      Colors[4, Lowered], 1, Sides);
end;

function TPieCustomProgressBar.DrawBorderSides(Canvas: TCanvas; Bounds: TRect; Style: TFrameStyle): TRect;
begin
  { Draw the Frame }
  if Style <> fsNone then begin
    if Style in [fsLowered, fsRaised] then
      Bounds := DrawCtl3DBorderSides(Canvas, Bounds, Style = fsLowered, sdAllSides);
  end;
  Result := Bounds;
end;

procedure TPieCustomProgressBar.DrawLEDBar(Canvas: TCanvas; DrawRct: TRect; PercentStr: string);
var
  X, I, W, D, M, BoxWidth: Integer;
  BoxRct: TRect;
  Offset: Integer;
  SegmentsOn: Integer;
  Percent: Double;
  TopOffset: Integer;
begin
  Percent := (FPosition-FMin)/(FMax-FMin)*100;
  with Canvas do begin
    if Orientation = pbHorizontal
      then W := DrawRct.Right - DrawRct.Left
      else W := DrawRct.Bottom - DrawRct.Top;

    BoxWidth := W div FNumSegments;

    if ( W <= 80 ) and ( FNumSegments > 10 ) then  begin
      FNumSegments := 10;
      BoxWidth := W div FNumSegments;
    end;

    D := W - (FNumSegments * BoxWidth );
    M := D div BoxWidth;
    if M > 0 then Inc(FNumSegments, M );

    Offset := ( W - (FNumSegments * BoxWidth ) ) div 2;

    { Erase Sides of LED }
    Pen.Style := psClear;
    Brush.Color := BackColor;
    with DrawRct do begin
      if Orientation = pbHorizontal then begin
        Rectangle( Left - 1, Top, Left + Offset + 1, Bottom + 1 );
        Rectangle( Right - Offset - 1, Top, Right + 1, Bottom + 1 );
      end
      else begin
        Rectangle( Left, Top, Right + 1, Top + Offset + 1 );
        Rectangle( Left, Bottom - Offset - 1, Right + 1, Bottom + 1 );
      end;
    end;

    Pen.Color := BackColor;
    Pen.Style := psSolid;
    Brush.Color := BarColor;

    SegmentsOn := Trunc( Percent * FNumSegments / 100 );
    for I := 1 to SegmentsOn do begin
      if Orientation = pbHorizontal then begin
        X := ( I - 1 ) * BoxWidth + Offset;
        BoxRct := Rect( DrawRct.Left + X, DrawRct.Top,
                        DrawRct.Left + X + BoxWidth, DrawRct.Bottom );
      end
      else begin
        X := (FNumSegments - I  ) * BoxWidth + Offset;
        BoxRct := Rect( DrawRct.Left, DrawRct.Top + X,
                        DrawRct.Right, DrawRct.Top + X + BoxWidth );
      end;

      with BoxRct do begin
        if Orientation = pbHorizontal
          then Rectangle( Left, Top, Right, Bottom )
          else Rectangle( Left, Top, Right, Bottom );
      end;
    end;

    Brush.Color := BackColor;

    for I := SegmentsOn + 1 to FNumSegments do begin
      if Orientation = pbHorizontal then begin
        X := ( I - 1 ) * BoxWidth + Offset;
        BoxRct := Rect( DrawRct.Left + X, DrawRct.Top,
                        DrawRct.Left + X + BoxWidth, DrawRct.Bottom );
      end
      else begin
        X := (FNumSegments - I  ) * BoxWidth + Offset;
        BoxRct := Rect( DrawRct.Left, DrawRct.Top + X,
                        DrawRct.Right, DrawRct.Top + X + BoxWidth );
      end;

      with BoxRct do begin
        if Orientation = pbHorizontal
          then Rectangle( Left, Top, Right, Bottom )
          else Rectangle( Left, Top, Right, Bottom );
      end;
    end;

    IF FShowText THEN WITH DrawRct DO BEGIN
      SetBKMode(Handle, TRANSPARENT);
      SetTextAlign(Handle, ta_Center or ta_baseline);
      TopOffset := (Bottom - Top + TextHeight(PercentStr)) DIV 2 + Top;
      TextOut((Right-Left) DIV 2, TopOffset, PercentStr);
    END;
  end;
end; {= DrawLEDBar =}

procedure TPieCustomProgressBar.DrawPercentBar(Canvas: TCanvas; DrawRct: TRect; PercentStr: string);
var
  PctRct: TRect;
  TopOffset: Integer;
  Percent: Double;
begin
  Percent := (FPosition-FMin)/(FMax-FMin)*100;
  with Canvas, DrawRct do begin
    Font.Color := BackColor;
    Brush.Color := BarColor;

    { Calculate the Size of the Left/Bottom portion of the Percentage Bar }

    if Percent >= 100
      then PctRct := Rect( Left, Top, Right, Bottom )
      else if Orientation = pbVertical then begin
        PctRct := Rect(Left,
                       Bottom - Round((Longint(Bottom - Top) * Percent) / 100),
                       Right,
                       Bottom);
      end
      else
      begin
        PctRct := Rect(Left,
                       Top,
                       Left + Round((Longint(Right - Left) * Percent) / 100),
                       Bottom);
      end;

    { Display the Left/Bottom portion of the Percentage Bar }

    SetTextAlign( Handle, ta_Center or ta_Top );
    TopOffset := ( Bottom - Top - TextHeight( 'X' ) ) div 2;
    TextRect(PctRct, Right div 2, Top + TopOffset, PercentStr);

    { Calculate the Size of the Right/Top portion of the Percentage Bar }
    if Orientation = pbVertical then begin
      PctRct.Bottom := PctRct.Top;
      PctRct.Top := Top;
    end
    else begin
      PctRct.Left := PctRct.Right;
      PctRct.Right := Right;
    end;

    { Display the Right/Top portion of the Percentage Bar }
    Font.Color := BarColor;
    Brush.Color := BackColor;
    TextRect(PctRct, Right div 2, Top + TopOffset, PercentStr);
  end; { with }
end; {= DrawPercentBar =}

constructor TPieCustomProgressBar.Create( AOwner: TComponent );
begin
  inherited Create(AOwner);
  ControlStyle := ControlStyle + [csOpaque];
  FPosition := 0;
  FMin := 0;
  FMax := 100;
  FStep := 1;
  FDecimals := 2;
  FShowText := True;
  FTextUnit := '%';
  FNumSegments := 20;
  FOrientation := pbHorizontal;
  FBackColor := clWhite;
  FBarColor := clHighlight;
  FBorderColor := clBtnFace;
  FBorderInner := fsNone;
  FBorderOuter := fsLowered;
  FBevelWidth := 1;
  FBorderWidth := 0;
  Width := 200;
  Height := 24;
end;

procedure TPieCustomProgressBar.SetBackColor( Value: TColor );
begin
  if FBackColor <> Value then
  begin
    FBackColor := Value;
    Invalidate;
  end;
end;

procedure TPieCustomProgressBar.SetBarColor( Value: TColor );
begin
  if FBarColor <> Value then
  begin
    FBarColor := Value;
    Invalidate;
  end;
end;

procedure TPieCustomProgressBar.SetSmooth(Value: Boolean);
begin
  if FSmooth <> Value then
  begin
    FSmooth := Value;
    Invalidate;
  end;
end;

procedure TPieCustomProgressBar.SetBevelWidth( Value: TBevelWidth );
begin
  if FBevelWidth <> Value then
  begin
    FBevelWidth := Value;
    Invalidate;
  end;
end;

procedure TPieCustomProgressBar.SetBorderColor( Value: TColor );
begin
  if FBorderColor <> Value then
  begin
    FBorderColor := Value;
    Invalidate;
  end;
end;

procedure TPieCustomProgressBar.SetBorderInner( Value: TFrameStyle );
begin
  if FBorderInner <> Value then
  begin
    FBorderInner := Value;
    Invalidate;
  end;
end;

procedure TPieCustomProgressBar.SetBorderOuter( Value: TFrameStyle );
begin
  if FBorderOuter <> Value then
  begin
    FBorderOuter := Value;
    Invalidate;
  end;
end;

procedure TPieCustomProgressBar.SetBorderWidth( Value: TBorderWidth );
begin
  if FBorderWidth <> Value then
  begin
    FBorderWidth := Value;
    Invalidate;
  end;
end;

procedure TPieCustomProgressBar.SetNumSegments( Value: TSegmentRange );
begin
  if FNumSegments <> Value then begin
    FNumSegments := Value;
    Invalidate;
  end;
end;

procedure TPieCustomProgressBar.SetOrientation(Value: TProgressBarOrientation);
begin
  if FOrientation <> Value then begin
    FOrientation := Value;
    Invalidate;
  end;
end;

procedure TPieCustomProgressBar.SetMin(Value: Double);
begin
  if FMin <> Value then begin
    FMin := Value;
    PositionChanged;
    Repaint;
  end;
end;

procedure TPieCustomProgressBar.SetMax(Value: Double);
begin
  if FMax <> Value then begin
    FMax := Value;
    Repaint;
  end;
end;

procedure TPieCustomProgressBar.SetPosition(Value: Double);
begin
  if FPosition <> Value then begin
    FPosition := Value;
    Repaint;
  end;
end;

procedure TPieCustomProgressBar.SetShowText( Value: Boolean );
begin
  if FShowText <> Value then begin
    FShowText := Value;
    Invalidate;
  end;
end;

procedure TPieCustomProgressBar.SetTextUnit(Value: string);
begin
  if FTextUnit <> Value then begin
    FTextUnit := Value;
    Invalidate;
  end;
end;

procedure TPieCustomProgressBar.SetDecimals(Value: Byte);
begin
  if FDecimals <> Value then begin
    FDecimals := Value;
    Invalidate;
  end;
end;

procedure TPieCustomProgressBar.PositionChanged;
begin
  if Assigned(FOnChange) then FOnChange(Self, FPosition);
end;

procedure TPieCustomProgressBar.Paint;
var
  DrawRct: TRect;
  PercentStr: string;
begin
  Canvas.Font := Font;

  with Canvas do begin
    DrawRct := ClientRect;
    DrawRct := DrawBorderSides(Canvas, DrawRct, FBorderOuter);
    if BorderWidth > 0 then                                      { Fill Border }
      DrawRct := DrawBevel( Canvas, DrawRct, FBorderColor, FBorderColor,
                            BorderWidth, sdAllSides );
    DrawRct := DrawBorderSides(Canvas, DrawRct, FBorderInner);

    if FShowText
      then PercentStr := S_N(FPosition, FDecimals, FDecimals) + ' ' + FTextUnit
      else PercentStr := '';

    if FSmooth
      then DrawLEDBar(Canvas, DrawRct, PercentStr)
      else DrawPercentBar(Canvas, DrawRct, PercentStr);

  end; { with }
end; {= TPieCustomProgressBar.Paint =}

procedure TPieCustomProgressBar.SetBounds( ALeft, ATop, AWidth, AHeight: Integer );
begin
  inherited SetBounds( ALeft, ATop, AWidth, AHeight );
  if Height > Width
    then Orientation := pbVertical
    else Orientation := pbHorizontal;
end;

{===============================================================================
  TPieCustomProgressBar.CMHitTest

  Description
    This message is handled in this component so that when progress bars are
    used in an PieStatusBar component, the size grip can force the mouse message
    all the way down to the form.
===============================================================================}

procedure TPieCustomProgressBar.CMHitTest( var Msg: TMessage );
var
  R: TRect;
begin
  inherited;

  R := ClientRect;
  R.Left := R.Right - 12;

  if PtInRect( R, Point( Msg.LParamLo, Msg.LParamHi ) ) then
    Msg.Result := Parent.Perform( wm_NCHitTest, Msg.WParam, Msg.LParam );
end;

procedure TPieProgressBar.StepIt;
begin
  FPosition := FPosition + FStep;
  IF FPosition > FMax THEN FPosition := FMax;
  Repaint;
end;

procedure TPieProgressBar.StepBy(Delta: Double);
begin
  Position := Position + Delta;
  IF FPosition > FMax THEN FPosition := FMax;
  Repaint;
end;



end.
