{$I PIETOOLS.INC}
{ Autor: Ingolf Pietschmann.
  Dieser Quelltext ist Freeware. Die Verwendung und Weitergabe dieser Sourcen zu
  privaten nicht kommerziellen Zwecken ist ausdrcklich erwnscht.
  Die Verwendung zu kommerziellen Zwecken ist nur mit Erlaubnis des Autors
  gestattet. Den Autor knnen Sie unter "Support@Pie-Tools.de" erreichen.

  These sources are freeware. The usage and distribution of these sources for
  private, not commercial purposes is explicit desired.
  The usage for commercial purposes is only permitted in agreement of the author.
  The author can be reached by "Support@Pie-Tools.de".
}
unit PiePrinter;

interface

uses SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
     Forms, Dialogs, StdCtrls, ExtCtrls, Printers;

const max_Lines = 5;       {Number of allowable header / footer lines}
      Columns = 20;                             { Number of allowable columns }

type TInfoRecord = Record
       Text: String[240];                   { Header text }
       XPosition: Single;                   { % from the left }
       YPosition: Single;                   { % from the top }
       Alignment: TAlignment;               { Alignment }
     End;

     TRectS = RECORD Left, Top, Right, Bottom: Single; END;
     THeaderFooterStyle = (hfsNone, hfsLineUp, hfsLineDown, hfsLineUpDown, hfsRectangle, hfsEllipse);

     TInfoCoordinates = Record
       Pos: TRectS;
       APen: TPen;
       Style: THeaderFooterStyle;
       Shading: Word;
       Lines: Array[1..max_Lines] of TInfoRecord;
     End;

     TColumnInformationRecord = Record
       XPosition: Single;
       Length: Single;
     End;

     TDokumentStatus = (dsBeginDoc, dsEndDoc, dsAbort);
     TPrinterMode = (pmPrinter, pmImage);
     TTextBase = (tbTop, tbBaseLine, tbBottom);

     TPiePrinter = class(TComponent)
     private
       PixelsPerProzentVertical: Integer;{Number of pixels per Prozent along Y axis}
       PixelsPerProzentHorizontal: Integer; {Number of pixels per Prozent along X axis}
       TotalPageWidthPixels: Integer;    {Full width of page in pixels - includes gutters}
       TotalPageHeightPixels: Integer;   {Full height of page in pixels - includes gutters}
       TotalPageHeightProzent: Single;   {Height of page in Prozent}
       TotalPageWidthProzent: Single;    {Width of page in Prozent}
       GutterLeft: Integer;              {Unprintable area on left}
       GutterRight: Integer;             {Unprintable area on right}
       GutterTop: Integer;               {Unprintable area on top}
       GutterBottom: Integer;            {Unprintable area on bottom}
       DetailTop: Single;                {Prozent from the top where the detail section starts}
       DetailBottom: Single;             {Prozent from the top where the detail section ends}
       ColumnInformation: Array[1..Columns] of TColumnInformationRecord; {Spaltenposition und -breite}
       Header: array[1..2] of TInfoCoordinates; {beinhaltet Header und Footer!}
       {++++++++++++++++++++++++++++++++++++}
       FFont              : TFont;
       FHeaderFont        : TFont;
       FFooterFont        : TFont;
       FPageNumberFont    : TFont;
       FCurrentFont       : TFont;
       FAlignment         : TAlignment;
       FCurrentAlignment  : TAlignment;
       FTab               : Single;
       FTabWriteRect      : Single;
       FCurrentTab        : Single;
       FAutoPaging        : Boolean;
       FCurrentAutoPaging : Boolean;
       FDokumentStatus    : TDokumentStatus;
       FPageNumber        : TInfoRecord;
       FLastYPosition     : Single;   {The Y position where the last write occurred}
       FOrientation       : TPrinterOrientation;
       FTitle             : string;
       FFonts_in_Promille : Boolean;
       FPrinterMode       : TPrinterMode;
       FCanvas            : TCanvas;
       FImage             : TImage;
       FTextBase          : TTextBase;
       FColumnLines       : Boolean;
       function CalculateLineHeight: Integer;
       function ProzentToPixelsHorizontal(Prozent: Single ): Integer;
       function ProzentToPixelsVertical(Prozent: Single ): Integer;
       function PixelsToProzentHorizontal( Pixels: Integer ): Single;
       function PixelsToProzentVertical( Pixels: Integer ): Single;
       procedure CalculateMeasurements;
       procedure WriteHeader(Art: Byte);
       procedure WritePageNumber;
       procedure SaveCurrentFont;
       procedure RestoreCurrentFont;
       {+++++++++++++++++++++++++++++++++}
       function GetPen: TPen;
       function GetBrush: TBrush;
       function GetPageNumber:Integer;
       procedure SetAlignment(Value: TAlignment);
       procedure SetFont(Value: TFont);
       procedure SetHeaderFont(Value: TFont);
       procedure SetFooterFont(Value: TFont);
       procedure SetPageNumberFont(Value: TFont);
       procedure SetAutoPaging(Value: Boolean);
       procedure SetPen(Value: TPen);
       procedure SetBrush(Value: TBrush);
       procedure SetDokumentStatus(Value: TDokumentStatus);
       procedure SetFontInformation(Value: TFont);
       procedure SetOrientation(Value: TPrinterOrientation);
       procedure SetFonts_in_Promille(Value: Boolean);
       procedure SetPrinterMode(Value: TPrinterMode);
       procedure SetImage(Value: TImage);
       procedure SetTextBase(Value: TTextBase);
     protected
     published
       constructor Create(AOwner: TComponent); override;
       destructor Destroy; override;
       property Alignment: TAlignment read FAlignment write SetAlignment;
       property Font: TFont read FFont write SetFont;
       property HeaderFont: TFont read FHeaderFont write SetHeaderFont;
       property FooterFont: TFont read FFooterFont write SetFooterFont;
       property PageNumberFont: TFont read FPageNumberFont write SetPageNumberFont;
       property Tab: Single read FTab write FTab;
       property TabWriteRect: Single read FTabWriteRect write FTabWriteRect;
       property AutoPaging: Boolean read FAutoPaging write SetAutoPaging;
       property Pen: TPen read GetPen write SetPen;
       property Brush: TBrush read GetBrush write SetBrush;
       property DokumentStatus: TDokumentStatus read FDokumentStatus write SetDokumentStatus;
       property PageNumber: Integer read GetPageNumber;
       property YPosition: Single read FLastYPosition write FLastYPosition;
       property Orientation: TPrinterOrientation read FOrientation write SetOrientation;
       property FontHeight_Promille: Boolean read FFonts_in_Promille write SetFonts_in_Promille;
       property Title: string read FTitle write FTitle;
       property PrinterMode: TPrinterMode read FPrinterMode write SetPrinterMode;
       property Image: TImage read FImage write SetImage;
       property TextBase: TTextBase read FTextBase write SetTextBase;
       property ColumnLines: Boolean read FColumnLines write FColumnLines;
     public  {TPiePrinter}
       {THeaderFooterStyle = hfsNone, hfsLineUp, hfsLineDown, hfsLineUpDown, hfsRectangle, hfsEllipse}
       procedure SetDetailTopBottom(Top: Single; Bottom: Single);
       procedure SetPageNumberInformation(XPosition, YPosition:Single; Text:String);
       procedure SetHeaderInformation(Line:Integer; XPosition, YPosition: Single; Text:String);
       procedure SetFooterInformation(Line:Integer; XPosition, YPosition: Single; Text:String);
       procedure SetHeaderDimensions(ALeft, ATop, ARight, ABottom:Single; AStyle: THeaderFooterStyle; Shading:Word);
       procedure SetFooterDimensions(ALeft, ATop, ARight, ABottom:Single; AStyle: THeaderFooterStyle; Shading:Word);
       procedure CreateColumn(Number:Word; XPosition:Single; Length:Single);
       procedure WriteLineColumn(ColumnNumber:Word; Y:Single; Text:String);
       procedure WriteLine(X, Y:Single; Text:String);
       procedure WriteRect(X1, Y1, X2, Y2:Single; Text:String);
       procedure DrawLine(X1, Y1, X2, Y2:Single);
       procedure FillRect(X1, Y1, X2, Y2: Single);
       procedure Rectangle(X1, Y1, X2, Y2: Single);
       procedure FrameRect(X1, Y1, X2, Y2: Single);
       procedure Ellipse(X1, Y1, X2, Y2: Single);
       procedure StretchDraw(X1, Y1, X2, Y2: Single; Bmp: TBitmap; Optimal: Boolean);
       procedure NewPage;
       procedure NextLine;
       procedure NewLines(Number:Word);
       procedure SetTopOfPage;
       function  GetTextWidth(Text: String): Single;
       function  GetLineHeightProzent: Single;
       function  GetCharPerLine: Integer;
       function  GetLinesRest: Word;
       function  GetLinesInDetailArea: Word;
       procedure Font_aendern;
       procedure Schrift_waehlen(ASize: Integer; AStyle: TFontStyles; AAlignment: TAlignment);
   End;

{procedure Register;}

implementation

constructor TPiePrinter.Create(AOwner: TComponent);
var  I, J   : Integer;
BEGIN
  inherited Create(AOwner);
  FPrinterMode := pmPrinter;
  FImage := NIL;
  FCanvas := Printer.Canvas;
  FPageNumber.Text := '';
  FOR J := 1 TO 2 DO WITH Header[J] DO BEGIN
    For I := 1 To max_Lines Do Lines[I].Text := '';
    Style := hfsRectangle;
    Shading := 0;
    APen := TPen.Create;
    APen.Assign(Pen);
  END;
  PixelsperProzentVertical := 2;
  PixelsperProzentHorizontal := 2;
  {+++++++++++++++++++}
  FAlignment := taLeftJustify;
  FFont := TFont.Create;
  FFont.Assign((AOwner as TCustomForm).Font);
  FHeaderFont := TFont.Create;
  FHeaderFont.Assign(FFont);
  FFooterFont := TFont.Create;
  FFooterFont.Assign(FFont);
  FPageNumberFont := TFont.Create;
  FPageNumberFont.Assign(FFont);
  FCurrentFont := TFont.Create;
  FTab := 0.0;
  FCurrentTab := 0.0;
  FAutoPaging := TRUE;
  FDokumentStatus := dsEndDoc;
  FLastYPosition := 0.0;
  FOrientation := poPortrait;
  FFonts_in_Promille := FALSE;
  FTextBase := tbTop;
  FColumnLines := FALSE;
  DetailTop := 1;
  DetailBottom := 99;
  {++++++++++++++++++++}
END;

destructor TPiePrinter.Destroy;
BEGIN
  FFont.Free;
  FHeaderFont.Free;
  FFooterFont.Free;
  FPageNumberFont.Free;
  FCurrentFont.Free;
  Header[1].APen.Free;
  Header[2].APen.Free;
  inherited Destroy;
END;

procedure TPiePrinter.WriteLine( X:Single; Y:Single; Text:String );
   { Write some text.  The parameters represent Prozent from the left ('X')
     and top ('Y')}
var XPixels: Integer;
    YPixels: Integer;
    PixelLength: Word;
    StartPixel: Word;

Begin
  If ( X >= 0.0 ) Then BEGIN
    {How many pixels does the text in 'Text' require?}
    PixelLength := FCanvas.TextWidth(Text);

    StartPixel := ProzentToPixelsHorizontal(X);
    CASE Alignment OF
    taLeftJustify : ;
    taRightJustify: StartPixel := StartPixel - PixelLength;
    taCenter      : StartPixel := StartPixel - PixelLength DIV 2;
    END;
  END
  Else StartPixel := GutterLeft;

  If StartPixel < GutterLeft Then StartPixel := GutterLeft;
  XPixels := StartPixel;
  YPixels := 0;

   {If there is a tab set, increase 'XPixels' by the amount of the tab}
  If (FTab > 0.0) Then Inc(XPixels,ProzentToPixelsHorizontal(FTab));

   { How many pixels are there in the Prozent represented by 'Y'? }
  If (Y > -0.01) Then    { Printing will occur at an absolute location from }
  Begin                                                { the top of the page. }
    YPixels := ProzentToPixelsVertical( Y );
    If ( YPixels < GutterTop ) Then YPixels := GutterTop;
    If ( YPixels > TotalPageHeightPixels ) Then
      YPixels := TotalPageHeightPixels - GutterBottom;
    FLastYPosition := Y;
  End;
  If (Y = -1.0) Then BEGIN  {Write the text at the next line}
    If (FAutoPaging = True) Then Begin
      {If the next line we're going to write to exceeds beyond the
       bottom of the detail section, issue a new page }
      If (FLastYPosition + GetLineHeightProzent > DetailBottom) Then NewPage;
    End;
    YPixels := ProzentToPixelsVertical(FLastYPosition + GetLineHeightProzent);
    FLastYPosition := FLastYPosition + GetLineHeightProzent;
  End;
  If ( Y = -2.0 ) Then     {Write the text on the current line}
    YPixels := ProzentToPixelsVertical(FLastYPosition);

  FCanvas.TextOut( XPixels-GutterLeft,YPixels-GutterTop,Text );
End;

procedure TPiePrinter.WriteLineColumn(ColumnNumber:Word; Y:Single; Text:String);
   {Write text, left aligned against the column represented by
    'ColumnInformation[ColumnNumber]'}
VAR
  X: Single;
  P: Single;
Begin
  X := ColumnInformation[ColumnNumber].XPosition;
  CASE Alignment OF
  taLeftJustify  : ;
  taRightJustify : X := X + ColumnInformation[ColumnNumber].Length;
  taCenter       : X := X + ColumnInformation[ColumnNumber].Length / 2;
  END;
  WriteLine(X,Y,Text);
  IF ColumnLines THEN BEGIN
    P := ColumnInformation[ColumnNumber].XPosition;
    IF ColumnNumber = 1 THEN DrawLine(P-0.2, YPosition, P-0.2, YPosition + GetLineHeightProzent);
    P := P + ColumnInformation[ColumnNumber].Length;
    DrawLine(P+0.2, YPosition, P+0.2, YPosition + GetLineHeightProzent);
  END;
End;

procedure TPiePrinter.WriteRect(X1, Y1, X2, Y2:Single; Text: string);
VAR
  R: TRect;
  P: PChar;
  I: Integer;
Begin
  R.Left := ProzentToPixelsHorizontal(X1)-GutterLeft;
  R.Top := ProzentToPixelsVertical(Y1)-GutterTop;
  R.Right := ProzentToPixelsHorizontal(X2)-GutterLeft;
  R.Bottom := ProzentToPixelsVertical(Y2)-GutterTop;
  P := StrAlloc(length(Text)+1);
  FOR I:=1 TO length(Text) DO P[I-1] := Text[I];
  P[length(Text)] := #0;
  DrawText(FCanvas.Handle, P, StrLen(P), R,
    DT_WORDBREAK OR DT_EXPANDTABS OR DT_TABSTOP OR
    (round(FTabWriteRect * FCanvas.Font.Size) SHL 8));
  StrDispose(P);
End;

procedure TPiePrinter.DrawLine(X1, Y1, X2, Y2:Single);
Begin
  WITH FCanvas DO BEGIN
    MoveTo(ProzentToPixelsHorizontal(X1)-GutterLeft,ProzentToPixelsVertical(Y1)-GutterTop);
    LineTo(ProzentToPixelsHorizontal(X2)-GutterLeft,ProzentToPixelsVertical(Y2)-GutterTop);
  END;
End;

procedure TPiePrinter.FillRect(X1, Y1, X2, Y2: Single);
Begin
  FCanvas.FillRect(Rect(
    ProzentToPixelsHorizontal(X1)-GutterLeft,ProzentToPixelsVertical(Y1)-GutterTop,
    ProzentToPixelsHorizontal(X2)-GutterLeft,ProzentToPixelsVertical(Y2)-GutterTop));
End;

procedure TPiePrinter.Rectangle(X1, Y1, X2, Y2: Single);
Begin
  FCanvas.Rectangle(
    ProzentToPixelsHorizontal(X1)-GutterLeft,ProzentToPixelsVertical(Y1)-GutterTop,
    ProzentToPixelsHorizontal(X2)-GutterLeft,ProzentToPixelsVertical(Y2)-GutterTop);
End;

procedure TPiePrinter.FrameRect(X1, Y1, X2, Y2: Single);
Begin
  FCanvas.FrameRect(Rect(
    ProzentToPixelsHorizontal(X1)-GutterLeft,ProzentToPixelsVertical(Y1)-GutterTop,
    ProzentToPixelsHorizontal(X2)-GutterLeft,ProzentToPixelsVertical(Y2)-GutterTop));
End;

procedure TPiePrinter.Ellipse(X1, Y1, X2, Y2: Single);
Begin
  FCanvas.Ellipse(
    ProzentToPixelsHorizontal(X1)-GutterLeft,ProzentToPixelsVertical(Y1)-GutterTop,
    ProzentToPixelsHorizontal(X2)-GutterLeft,ProzentToPixelsVertical(Y2)-GutterTop);
End;

procedure TPiePrinter.StretchDraw(X1, Y1, X2, Y2: Single; Bmp: TBitmap; Optimal: Boolean);
VAR
  R: TRect;
  V1, V2: Single;
BEGIN
  R.Left := ProzentToPixelsHorizontal(X1)-GutterLeft;
  R.Top := ProzentToPixelsVertical(Y1)-GutterTop;
  R.Right := ProzentToPixelsHorizontal(X2)-GutterLeft;
  R.Bottom := ProzentToPixelsVertical(Y2)-GutterTop;
  IF Optimal THEN BEGIN
    V1 := (R.Right-R.Left) / Bmp.Width;
    V2 := (R.Bottom-R.Top) / Bmp.Height;
    IF V1 > V2
      THEN R.Right := round(Bmp.Width * V2) + R.Left {Breite anpassen}
      ELSE R.Bottom := round(Bmp.Height * V1) + R.Top; {Hhe anpassen}
  END;
  FCanvas.StretchDraw(R, Bmp);
END;

procedure TPiePrinter.SetFontInformation(Value: TFont);
   { Change the current font information }
Begin
  IF NOT(csDesigning in ComponentState) THEN BEGIN
    FCanvas.Font.Assign(Value);
    IF FFonts_in_Promille THEN
      FCanvas.Font.Height := round(Value.Height * PixelsperProzentVertical / 10);
    CalculateMeasurements;
  END;  
End;

function TPiePrinter.CalculateLineHeight: Integer;
   { Calculate the height of a line plus the normal amount of space between
     each line }
VAR
  TextMetrics: TTextMetric;
Begin
 {Call the function GetTextMetrics() to get the specifics of the particular font.}
  GetTextMetrics(FCanvas.Handle,TextMetrics);
  Result := TextMetrics.tmHeight + TextMetrics.tmExternalLeading;
End;

procedure TPiePrinter.NewPage;                            { Issue a new page }
Begin
  WriteHeader(1);
  WriteHeader(2);
  WritePageNumber;
  FLastYPosition := DetailTop;
  IF PrinterMode = pmPrinter THEN Printer.NewPage;
End;

function TPiePrinter.GetPageNumber;
Begin
  IF (DokumentStatus = dsBeginDoc) AND
     (PrinterMode = pmPrinter) THEN Result := Printer.PageNumber ELSE Result := 0;
End;

function TPiePrinter.GetTextWidth(Text:String): Single;
   { Return the width of the text contained in 'Text' in procent }
Begin
  Result := PixelsToProzentHorizontal(FCanvas.TextWidth(Text));
End;

function TPiePrinter.GetLineHeightProzent: Single;
Begin
  Result := PixelsToProzentVertical(CalculateLineHeight);
End;

function TPiePrinter.GetLinesInDetailArea: Word;
                              { Return the number of lines in the detail area }
Begin
  Result := ProzentToPixelsVertical( DetailBottom - DetailTop ) Div
  CalculateLineHeight;
End;

function TPiePrinter.GetCharPerLine: Integer;
            { How many chars are there in a Line? }
var Pixels: Integer;
Begin
  Pixels := TotalPageWidthPixels - GutterLeft - GutterRight;
  Result := Pixels Div FCanvas.TextWidth('B');
End;

function TPiePrinter.ProzentToPixelsHorizontal( Prozent: Single ): Integer;
            { Convert the horizontal Prozent represented in 'Prozent' to pixels }
var Value: Single;
Begin
  Value := Prozent * PixelsPerProzentHorizontal;
  Result := trunc(Value);
End;

function TPiePrinter.ProzentToPixelsVertical( Prozent: Single ): Integer;
              { Convert the vertical Prozent represented in 'Prozent' to pixels }
var Value: Single;
Begin
  Value := Prozent * PixelsPerProzentVertical;
  Result := trunc(Value);
End;

function TPiePrinter.PixelsToProzentHorizontal(Pixels: Integer): Single;
Begin
  Result := Pixels / PixelsPerProzentHorizontal;
End;

function TPiePrinter.PixelsToProzentVertical( Pixels: Integer ): Single;
Begin
  Result := Pixels / PixelsPerProzentVertical;
End;

procedure TPiePrinter.CalculateMeasurements;
   { Calculate some necessary measurements.  Thanks to Robert Fabiszak
     CompuServe: 70304,2047 for the Escape() Windows API calls. }
var pt: TPoint;
Begin

   { Calculate the number of pixels per Prozent vertical and horizontal.
     'GetDeviceCaps' is a Windows API call. }
  IF PrinterMode = pmPrinter
    THEN { Get the gutter on the left and top.  'Escape' is a Windows API call. }
      Escape(FCanvas.Handle,GETPRINTINGOFFSET,0,Nil,@pt )
    ELSE Pt := Point(0,0);
  GutterLeft := pt.X;
  GutterTop := pt.Y;

  IF PrinterMode = pmPrinter
    THEN Escape(FCanvas.Handle,GETPHYSPAGESIZE,0,Nil,@pt )
    ELSE Pt := Point(FImage.Width, FImage.Height);
  TotalPageWidthPixels := pt.X;
  TotalPageHeightPixels := pt.Y;
  TotalPageWidthProzent := 100 {%};
  TotalPageHeightProzent := 100 {%};
  PixelsPerProzentVertical := round(pt.Y / TotalPageHeightProzent);
  PixelsPerProzentHorizontal := round(pt.X / TotalPageWidthProzent);

  IF PrinterMode = pmPrinter THEN BEGIN
    GutterRight := TotalPageWidthPixels - GutterLeft - Printer.PageWidth;
    GutterBottom := TotalPageHeightPixels - GutterTop - Printer.PageHeight;
  END
  ELSE BEGIN
    GutterRight := GutterLeft;
    GutterBottom := GutterTop;
  END;
End;

procedure TPiePrinter.SetHeaderInformation(Line:Integer; XPosition, YPosition: Single; Text:String);
Begin
  If (Line > max_Lines) Then Exit;
  Header[1].Lines[Line].Text := Text;
  Header[1].Lines[Line].XPosition := XPosition;
  Header[1].Lines[Line].YPosition := YPosition;
  Header[1].Lines[Line].Alignment := FAlignment;
End;

procedure TPiePrinter.SetFooterInformation(Line:Integer; XPosition, YPosition: Single; Text:String);
Begin
  If (Line > max_Lines) Then Exit;
  Header[2].Lines[Line].Text := Text;
  Header[2].Lines[Line].XPosition := XPosition;
  Header[2].Lines[Line].YPosition := YPosition;
  Header[2].Lines[Line].Alignment := FAlignment;
End;

procedure TPiePrinter.WriteHeader(Art: Byte);  { If any headers are defined, write them }
var I: Integer;
   TempPen: TPen;
   TempBrush: TBrush;
   Schatten: Byte;
Begin
  SaveCurrentFont;
  TempPen := TPen.Create;
  TempPen.Assign(Pen);
  TempBrush := TBrush.Create;
  TempBrush.Assign(Brush);
  TRY
    WITH Header[Art] DO BEGIN
       { Does the user desire a box around the header? }
      If Style <> hfsNone Then Begin
        Pen := APen;
        IF Shading > 0 THEN Schatten := Shading ELSE Schatten := 255;
        WITH FCanvas DO BEGIN
          Brush.Bitmap := NIL;
          Brush.Style := bsSolid;
          Brush.Color := RGB(Schatten, Schatten, Schatten);
        END;
        WITH Pos DO BEGIN
          CASE Style OF
          hfsLineUp:      DrawLine(Left,Top,Right,Top);
          hfsLineDown:    DrawLine(Left,Bottom,Right,Bottom);
          hfsLineUpDown:  BEGIN DrawLine(Left,Top,Right,Top); DrawLine(Left,Bottom,Right,Bottom); END;
          hfsRectangle:   Rectangle(Left,Top,Right,Bottom);
          hfsEllipse:     Ellipse(Left,Top,Right,Bottom);
          END;
        END;
      END;  {IF Style <> hfsNone...}
      CASE Art OF
      1: SetFontInformation(FHeaderFont);
      2: SetFontInformation(FFooterFont);
      END;
      FTab := 0.0;
      For I := 1 To max_Lines Do With Lines[I], Pos DO BEGIN
        FAlignment := Header[Art].Lines[I].Alignment;
        If Length(Text) > 0 Then WriteLine(XPosition, YPosition, Text);
      END;
    END;

  FINALLY
    RestoreCurrentFont;
    Pen := TempPen;
    Brush.Assign(TempBrush);
    TempPen.Free;
    TempBrush.Free;
  END;
End;

procedure TPiePrinter.SaveCurrentFont;
Begin
  FCurrentFont.Assign(FCanvas.Font);
  FCurrentAlignment := FAlignment;
  FCurrentTab := FTab;
  FCurrentAutoPaging := FAutoPaging;
End;

procedure TPiePrinter.RestoreCurrentFont;
Begin
  FAlignment := FCurrentAlignment;
  FTab := FCurrentTab;
  FAutoPaging := FCurrentAutoPaging;
  FCanvas.Font.Assign(FCurrentFont);
  CalculateMeasurements;
End;

procedure TPiePrinter.SetDetailTopBottom(Top: Single; Bottom: Single);
Begin
  DetailTop := Top;
  DetailBottom := Bottom;
  FLastYPosition := Top - GetLineHeightProzent;
End;

procedure TPiePrinter.SetAutoPaging( Value: Boolean );
Begin
  IF Value <> FAutoPaging THEN FAutoPaging := Value;
End;

procedure TPiePrinter.SetPageNumberInformation(XPosition, YPosition:Single; Text:String);
Begin
  FPageNumber.Text := Text;
  FPageNumber.XPosition := XPosition;
  FPageNumber.YPosition := YPosition;
  FPageNumber.Alignment := FAlignment;
End;

procedure TPiePrinter.WritePageNumber;
var Buffer: String;
Begin
  IF PrinterMode = pmPrinter
    THEN Buffer := Format(FPageNumber.Text,[Printer.PageNumber])
    ELSE Buffer := Format(FPageNumber.Text,[1]);
  SaveCurrentFont;
  FAlignment := FPageNumber.Alignment;
  FTab := 0.0;
  SetFontInformation(FPageNumberFont);
  AutoPaging := False;
  WriteLine(FPageNumber.XPosition, FPageNumber.YPosition, Buffer);
  RestoreCurrentFont;
End;

procedure TPiePrinter.SetHeaderDimensions(ALeft, ATop, ARight, ABottom:Single;
                                          AStyle: THeaderFooterStyle; Shading:Word);
Begin
  WITH Header[1].Pos DO BEGIN
  Left := ALeft;
  Right := ARight;
  Top := ATop;
  Bottom := ABottom;
  END;
  Header[1].Style := AStyle;
  Header[1].APen.Assign(Pen);
  Header[1].Shading := Shading;
End;

procedure TPiePrinter.SetFooterDimensions(ALeft, ATop, ARight, ABottom:Single;
                                          AStyle: THeaderFooterStyle; Shading:Word);
Begin
  WITH Header[2].Pos DO BEGIN
  Left := ALeft;
  Right := ARight;
  Top := ATop;
  Bottom := ABottom;
  END;
  Header[2].Style := AStyle;
  Header[2].APen.Assign(Pen);
  Header[2].Shading := Shading;
End;

procedure TPiePrinter.CreateColumn( Number:Word; XPosition:Single;
                                     Length:Single );
Begin
  ColumnInformation[Number].XPosition := XPosition;
  ColumnInformation[Number].Length := Length;
End;

procedure TPiePrinter.NextLine;
Begin
  FLastYPosition := FLastYPosition + GetLineHeightProzent;
End;

function TPiePrinter.GetLinesRest: Word;
                         { Return the number of lines left in the detail area }
Begin
  Result := trunc((DetailBottom - FLastYPosition) / GetLineHeightProzent);
End;

procedure TPiePrinter.SetTopOfPage;
Begin
  FLastYPosition := DetailTop;
End;

procedure TPiePrinter.NewLines(Number:Word);
                  { Generate the number of line feeds represented in 'Number' }
var I: Word;
Begin
  For I := 1 To Number Do NextLine;
End;

procedure TPiePrinter.SetAlignment(Value: TAlignment);
BEGIN
  IF Value <> FAlignment THEN FAlignment := Value;
END;
procedure TPiePrinter.SetFont(Value: TFont);
BEGIN
  IF Value <> NIL THEN BEGIN
    FFont.Assign(Value);
    SetFontInformation(FFont);
  END;
END;
procedure TPiePrinter.SetHeaderFont(Value: TFont);
BEGIN
  IF Value <> NIL THEN FHeaderFont.Assign(Value);
END;
procedure TPiePrinter.SetFooterFont(Value: TFont);
BEGIN
  IF Value <> NIL THEN FFooterFont.Assign(Value);
END;
procedure TPiePrinter.SetPageNumberFont(Value: TFont);
BEGIN
  IF Value <> NIL THEN FPageNumberFont.Assign(Value);
END;
function TPiePrinter.GetPen: TPen;
BEGIN
  Result := FCanvas.Pen;
END;
procedure TPiePrinter.SetPen(Value: TPen);
BEGIN
  IF Value <> NIL THEN FCanvas.Pen.Assign(Value);
END;
function TPiePrinter.GetBrush: TBrush;
BEGIN
  Result := FCanvas.Brush;
END;
procedure TPiePrinter.SetBrush(Value: TBrush);
BEGIN
  IF Value <> NIL THEN FCanvas.Brush.Assign(Value);
END;
procedure TPiePrinter.SetDokumentStatus(Value: TDokumentStatus);
BEGIN
  IF Value <> FDokumentStatus THEN BEGIN
    FDokumentStatus := Value;
    CASE FDokumentStatus OF
    dsBeginDoc:IF NOT(Printer.Printing) OR
                  NOT(Printermode = pmPrinter) THEN BEGIN
               IF Printermode = pmPrinter THEN BEGIN
                 Printer.Title := FTitle;
                 Printer.Orientation := FOrientation;  {Orientation darf nur gewechselt werden, solange das Drucken nicht im Gange ist.}
                 Printer.BeginDoc;
               END;
               SetTextBase(FTextBase);
               CalculateMeasurements;
               SetFontInformation(FFont);
               IF (Printermode = pmImage) THEN BEGIN {Image lschen}
                 Brush.Style := bsSolid;
                 Rectangle(0,0,100,100);
                 Brush.Style := bsClear;
               END;
               END;
    dsEndDoc:  IF Printer.Printing OR
                  NOT(Printermode = pmPrinter) THEN BEGIN
               WriteHeader(1);
               WriteHeader(2);
               WritePageNumber;
               IF Printermode = pmPrinter THEN Printer.EndDoc;
               END;
    dsAbort:   IF Printer.Printing AND
                 (Printermode = pmPrinter) THEN Printer.Abort;
    END;
  END;
END;
procedure TPiePrinter.SetOrientation(Value: TPrinterOrientation);
Begin
  IF Value <> FOrientation THEN BEGIN
    {Orientation darf nur gewechselt werden, solange das Drucken nicht im Gange ist.}
    {Orientation bringt Exception, wenn noch kein Drucker installiert ist.}
    try
      IF FDokumentStatus <> dsBeginDoc THEN Printer.Orientation := Value;
      FOrientation := Value;
    except
      on Exception do; {nothing}
    end;
    {CalculateMeasurements;}
  END;
End;
procedure TPiePrinter.SetPrinterMode(Value: TPrinterMode);
Begin
  IF Dokumentstatus = dsBeginDoc THEN
    ShowMessage('Wechsel des Ausgabekontextes ist whrend der Ausgabe nicht mglich!')
  ELSE IF Value <> FPrinterMode THEN BEGIN
    FPrinterMode := Value;
    IF FPrinterMode = pmPrinter
      THEN FCanvas := Printer.Canvas
      ELSE IF assigned(FImage)THEN FCanvas := FImage.Canvas;
  END;
End;
procedure TPiePrinter.SetImage(Value: TImage);
Begin
  IF Dokumentstatus = dsBeginDoc THEN
    ShowMessage('Wechsel des Ausgabekontextes ist whrend der Ausgabe nicht mglich!')
  ELSE IF Value <> FImage THEN BEGIN
    FImage := Value;
    IF assigned(FImage) AND (Printermode = pmImage) THEN BEGIN
      FCanvas := FImage.Canvas;
      CalculateMeasurements;
    END;
  END;
End;
procedure TPiePrinter.SetFonts_in_Promille(Value: Boolean);
Begin
  IF Value <> FFonts_in_Promille THEN FFonts_in_Promille := Value;
End;
procedure TPiePrinter.SetTextBase(Value: TTextBase);
BEGIN
  FTextBase := Value;
  IF FDokumentStatus = dsBeginDoc THEN
  CASE FTextBase OF
  tbTop:      SetTextAlign(FCanvas.Handle, TA_Top      OR TA_Left OR TA_NOUPDATECP);
  tbBaseLine: SetTextAlign(FCanvas.Handle, TA_BaseLine OR TA_Left OR TA_NOUPDATECP);
  tbBottom:   SetTextAlign(FCanvas.Handle, TA_Bottom   OR TA_Left OR TA_NOUPDATECP);
  END;
END;
procedure TPiePrinter.Font_aendern;
BEGIN
  SetFontInformation(FFont);
END;
procedure TPiePrinter.Schrift_waehlen(ASize: Integer; AStyle: TFontStyles; AAlignment: TAlignment);
BEGIN
  FFont.Size := ASize;
  FFont.Style := AStyle;
  SetAlignment(AAlignment);
  SetFontInformation(FFont);
END;


(*{FR JOLEI}
procedure Register;
begin
  RegisterComponents('PieTools', [TPiePrinter]);
  RegisterPropertyEditor(TypeInfo(TFont), TPiePrinter,
                         '', TFontProperty );
  RegisterPropertyEditor(TypeInfo(TPen), TPiePrinter,
                         '', TClassProperty );
END;*)

end.


