{$I PIETOOLS.INC}
{ Autor: Ingolf Pietschmann.
  Dieser Quelltext ist Freeware. Die Verwendung und Weitergabe dieser Sourcen zu
  privaten nicht kommerziellen Zwecken ist ausdrcklich erwnscht.
  Die Verwendung zu kommerziellen Zwecken ist nur mit Erlaubnis des Autors
  gestattet. Den Autor knnen Sie unter "Support@Pie-Tools.de" erreichen.

  These sources are freeware. The usage and distribution of these sources for
  private, not commercial purposes is explicit desired.
  The usage for commercial purposes is only permitted in agreement of the author.
  The author can be reached by "Support@Pie-Tools.de".
}
unit PieLED;

interface

uses
  Windows, Controls, Classes, Forms, StdCtrls, Graphics, Messages;

TYPE
  TPieLED = class(TGraphicControl)
  private
    FLichteffekt: Boolean;
    FColor : TColor;
    FRandfarbe: TColor;
    FRandbreite: Byte;
    procedure SetColor(Value: TColor);
    procedure SetRandfarbe(Value: TColor);
    procedure SetRandbreite(Value: Byte);
    procedure SetLichteffekt(Value : Boolean);
  protected
    procedure Paint; override;
  public
    constructor Create(AOwner: TComponent); override;
  published
    property Height default 25;
    property Width default 25;
    property Color: TColor read FColor write SetColor;
    property FrameColor: TColor read FRandfarbe write SetRandfarbe;
    property FrameWidth: Byte read FRandbreite write SetRandbreite Default 1;
    property LightEffect: Boolean read FLichteffekt write SetLichteffekt default FALSE;
    property OnClick;            { Make OnClick event visible }
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

  TPieLEDSegmente = set of Byte;
  TPieLEDStyle = (pls7Segment, pls13Segment);

  TPieLEDLabel = class(TGraphicControl)
  private
    FActiveLEDColor: TColor;
    FColor : TColor;
    FCharWidth: Integer;
    FCharHeight: Integer;
    FCharSpace: Integer;
    FInActiveLEDColor: TColor;
    FLEDStyle: TPieLEDStyle;
    FSegmentSpace: Integer;
    FSegmentWidth: Integer;
    FShowInActiveSegments: Boolean;
    FTransparent: Boolean;
    procedure CMTextChanged(var Message: TMessage); message CM_TEXTCHANGED;
    function GetLED7Segment(C: Char): TPieLEDSegmente;
    function GetLED13Segment(C: Char): TPieLEDSegmente;
    procedure SetColor(Index: Integer; Value: TColor);
    procedure SetIntegerValues(Index: Integer; Value: Integer);
    procedure SetLEDStyle(Value: TPieLEDStyle);
    procedure SetBooleanValues(Index: Integer; Value: Boolean);
    procedure Paint7Segment;
    procedure Paint13Segment;
  protected
    procedure Paint; override;
  public
    constructor Create(AOwner: TComponent); override;
  published
    property Height default 33;
    property Width default 280;
    property Align;
    property Anchors;
    property Caption;
    property CharHeight: Integer index 2 read FCharHeight write SetIntegerValues default 30;
    property CharSpace: Integer index 4 read FCharSpace write SetIntegerValues default 0;
    property CharWidth: Integer index 1 read FCharWidth write SetIntegerValues default 20;
    property Color: TColor index 1 read FColor write SetColor default clBlack;
    property ActiveLEDColor: TColor index 2 read FActiveLEDColor write SetColor default clLime;
    property InActiveLEDColor: TColor index 3 read FInActiveLEDColor write SetColor default clGreen;
    property LEDStyle: TPieLEDStyle read FLEDStyle write SetLEDStyle default pls13Segment;
    property SegmentSpace: Integer index 5 read FSegmentSpace write SetIntegerValues default 0;
    property SegmentWidth: Integer index 3 read FSegmentWidth write SetIntegerValues default 2;
    property ShowInActiveSegments: Boolean index 2 read FShowInActiveSegments write SetBooleanValues default FALSE;
    property Transparent: Boolean index 1 read FTransparent write SetBooleanValues default FALSE;
    property OnClick;            { Make OnClick event visible }
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnMouseDown;
    property OnMouseMove;
    property OnMouseUp;
  end;

implementation

{===========================}
{== TPieLED Methods ==}
{===========================}

constructor TPieLED.Create( AOwner : TComponent );
begin
  inherited Create( AOwner );
  Width := 25;                       { Set Default Width }
  Height := 25;                      { Set Default Height }
  FLichteffekt := FALSE;
  FColor := clGreen;
  FRandfarbe := clBlack;
  FRandbreite := 1;
end;

procedure TPieLED.Paint;
begin
  WITH Canvas DO BEGIN
    {Kreis in Grundfarbe zeichnen}
    Brush.Style := bsSolid;
    Brush.Color := FColor;
    Pen.Style := psSolid;
    Pen.Color := FRandfarbe;
    Pen.Width := FRandbreite;
    Ellipse(1,1,Width, Height);
    IF FLichteffekt THEN BEGIN
      {helle Ecke links oben zeichnen}
      Brush.Color := clWhite;
      Pen.Color := clWhite;
      Pie(round(Width / 8), round(Height / 8),
          Width - round(Width / 8), Height - round(Height / 8),
          round(Width / 2) - 1, 1,
          1, round(Height / 2) - 1);
      Brush.Color := FColor;
      Pen.Color := FColor;
      Pie(round(Width / 4), round(Height / 4),
          Width - round(Width / 4), Height - round(Height / 4),
          round(Width / 2), 1,
          1, round(Height / 2));
      {dunkle Ecke rechts unten zeichnen}
      Brush.Color := clGray;
      Pen.Color := clGray;
      Pie(round(Width / 8), round(Height / 8),
          Width - round(Width / 8), Height - round(Height / 8),
          round(Width / 2) + 1, Height,
          Width, round(Height / 2) + 1);
      Brush.Color := FColor;
      Pen.Color := FColor;
      Pie(round(Width / 4), round(Height / 4),
          Width - round(Width / 4), Height - round(Height / 4),
          round(Width / 2), Height,
          Width, round(Height / 2));
    END;
  END;
end;

procedure TPieLED.SetLichteffekt(Value: Boolean);
begin
  if Value <> FLichteffekt then
  begin
    FLichteffekt := Value;
    Invalidate;
  end;
end;

procedure TPieLED.SetColor(Value: TColor);
begin
  if Value <> FColor then begin
    FColor := Value;
    Invalidate;
  end;
end;

procedure TPieLED.SetRandfarbe(Value: TColor);
begin
  if Value <> FRandfarbe then begin
    FRandfarbe := Value;
    Invalidate;
  end;
end;

procedure TPieLED.SetRandbreite(Value: Byte);
begin
  if Value <> FRandbreite then begin
    FRandbreite := Value;
    Invalidate;
  end;
end;

{===========================}
{== TPieLEDLabel Methods ==}
{===========================}

constructor TPieLEDLabel.Create(AOwner : TComponent);
begin
  inherited Create(AOwner);
  Height := 33;                      { Set Default Height }
  Width := 280;                      { Set Default Width }
  FCharHeight := 30;
  FCharWidth := 20;
  FCharSpace := 0;
  FActiveLEDColor := clLime;
  FInActiveLEDColor := clGreen;
  FColor := clBlack;
  FLEDStyle := pls13Segment;
  FSegmentSpace := 0;
  FSegmentWidth := 2;
  FTransparent := FALSE;
  FShowInActiveSegments := FALSE;
end;

procedure TPieLEDLabel.CMTextChanged(var Message: TMessage);
begin
  Invalidate;
end;

function TPieLEDLabel.GetLED7Segment(C: Char): TPieLEDSegmente;
begin
  CASE C OF
       '0': Result := [1..3,5..7];
       '1': Result := [3,6];
       '2': Result := [1,3..5,7];
       '3': Result := [1,3,4,6,7];
       '4': Result := [2..4,6];
       '5': Result := [1,2,4,6,7];
       '6': Result := [1,2,4..7];
       '7': Result := [1,3,6];
       '8': Result := [1..7];
       '9': Result := [1..4,6,7];
  'A', 'a': Result := [1..6];
  'B', 'b': Result := [2,4..7];
  'C', 'c': Result := [1,2,5,7];
  'D', 'd': Result := [3..7];
  'E', 'e': Result := [1,2,4,5,7];
  'F', 'f': Result := [1,2,4,5];
  'G', 'g': Result := [1,2,4..7];
  'H', 'h': Result := [2,3,4,5,6];
  'I', 'i': Result := [3,6];
  'J', 'j': Result := [3,6,7];
  'K', 'k': Result := [];
  'L', 'l': Result := [2,5,7];
  'M', 'm': Result := [];
  'N', 'n': Result := [];
  'O', 'o': Result := [1..3,5..7];
  'P', 'p': Result := [1..5];
  'Q', 'q': Result := [];
  'R', 'r': Result := [];
  'S', 's': Result := [1,2,4,6,7];
  'T', 't': Result := [];
  'U', 'u': Result := [2,3,5..7];
  'V', 'v': Result := [];
  'W', 'w': Result := [];
  'X', 'x': Result := [];
  'Y', 'y': Result := [];
  'Z', 'z': Result := [];
  ',', '.': Result := [8];
  ELSE Result := [];
  END;
{  '', '': Result := [];}
end;

function TPieLEDLabel.GetLED13Segment(C: Char): TPieLEDSegmente;
begin
  CASE C OF
      '0': Result := [1,2,6,8,12,13];
      '1': Result := [5,6,12];
      '2': Result := [1,6,7,8,13];
      '3': Result := [1,6,7,12,13];
      '4': Result := [2,6,7,12];
      '5': Result := [1,2,7,12,13];
      '6': Result := [1,2,7,8,12,13];
      '7': Result := [1,6,12];
      '8': Result := [1,2,6,7,8,12,13];
      '9': Result := [1,2,6,7,12,13];
  'a','A': Result := [1,2,6,7,8,12];
  'b','B': Result := [1,4,6,7,10,12,13];
  'c','C': Result := [1,2,8,13];
  'd','D': Result := [1,4,6,10,12,13,14];
  'e','E': Result := [1,2,7,8,13];
  'f','F': Result := [1,2,7,8];
  'g','G': Result := [1,2,7,8,12,13];
  'h','H': Result := [2,6,7,8,12];
  'i','I': Result := [4,10,14];
  'j','J': Result := [6,8,12,13];
  'k','K': Result := [4,5,10,11,14];
  'l','L': Result := [2,8,13];
  'm','M': Result := [2,3,5,6,8,12,14];
  'n','N': Result := [2,3,6,8,11,12,14];
  'o','O': Result := [1,2,6,8,12,13];
  'p','P': Result := [1,2,6,7,8];
  'q','Q': Result := [1,2,6,8,11,12,13];
  'r','R': Result := [1,4,6,7,10,11];
  's','S': Result := [1,2,7,12,13];
  't','T': Result := [1,4,10,14];
  'u','U': Result := [2,6,8,12,13];
  'v','V': Result := [2,5,8,9,14];
  'w','W': Result := [2,6,8,9,11,12,14];
  'x','X': Result := [3,5,9,11,14];
  'y','Y': Result := [3,5,10,14];
  'z','Z': Result := [1,5,9,13,14];
  '','': Result := [1,2,6,7,8,12,16,17];
  '','': Result := [1,2,6,8,12,13,16,17];
  '','': Result := [2,6,8,12,13,16,17];
  ','    : Result := [9];
  '.'    : Result := [18];
  ELSE Result := [];
  END;
  {
  '': Result := [];
  }
end;

procedure TPieLEDLabel.Paint;
begin
  WITH Canvas DO BEGIN
    Brush.Style := bsSolid;
    Pen.Style := psSolid;

    {Hintergrund zeichnen}
    IF NOT(FTransparent) THEN BEGIN
      Brush.Color := FColor;
      FillRect(Rect(0, 0, Width, Height));
    END;

    CASE FLEDStyle OF
    pls7Segment: Paint7Segment;
    pls13Segment: Paint13Segment;
    END;
  END;
end;

procedure TPieLEDLabel.Paint7Segment;
VAR
  I, D, _, MitteH: Integer;
  Segmente: TPieLEDSegmente;
  Seg: Byte;
  CharRect, SegRect: TRect;
  PointArray: array[1..6] of TPoint;
begin
  WITH Canvas DO BEGIN
    D := FSegmentWidth;
    _ := FSegmentSpace;
    {Chars anzeigen}
    FOR I:=1 TO length(Caption) DO BEGIN
      Segmente := GetLED7Segment(Caption[I]);
      CharRect := Rect((I-1)*(FCharWidth+FCharSpace), 0, I*(FCharWidth+FCharSpace)-FCharSpace-1, FCharHeight-1);
      {Position der Segment-Mittellinie ermitteln}
      MitteH := CharRect.Top + (CharRect.Bottom-CharRect.Top) DIV 2;
      FOR Seg := 1 TO 8 DO IF (Seg IN Segmente) OR ((Seg < 8) AND FShowInActiveSegments) THEN BEGIN
        CASE Seg OF
        1: SegRect := Rect(CharRect.Left+D+_+1, CharRect.Top+D+1,    CharRect.Right-D-_,  CharRect.Top+D+1);
        2: SegRect := Rect(CharRect.Left+D+1,   CharRect.Top+D+_+1,  CharRect.Left+D+1,   MitteH-_);
        3: SegRect := Rect(CharRect.Right-D,    CharRect.Top+D+_+1,  CharRect.Right-D,    MitteH-_);
        4: SegRect := Rect(CharRect.Left+D+_+1, MitteH,              CharRect.Right-D-_,  MitteH);
        5: SegRect := Rect(CharRect.Left+D+1,   MitteH+_,            CharRect.Left+D+1,   CharRect.Bottom-D-_);
        6: SegRect := Rect(CharRect.Right-D,    MitteH+_,            CharRect.Right-D,    CharRect.Bottom-D-_);
        7: SegRect := Rect(CharRect.Left+D+_+1, CharRect.Bottom-D,   CharRect.Right-D-_,  CharRect.Bottom-D);
        8: SegRect := Rect(CharRect.Left+1,     CharRect.Bottom-2*D, CharRect.Left+1+2*D, CharRect.Bottom);
        END;
        {Polygon-Punkte ermitteln}
        CASE Seg OF
        1,4,7: BEGIN
               PointArray[1] := Point(SegRect.Left,    SegRect.Top);
               PointArray[2] := Point(SegRect.Left+D,  SegRect.Top-D);
               PointArray[3] := Point(SegRect.Right-D, SegRect.Bottom-D);
               PointArray[4] := Point(SegRect.Right,   SegRect.Bottom);
               PointArray[5] := Point(SegRect.Right-D, SegRect.Bottom+D);
               PointArray[6] := Point(SegRect.Left+D,  SegRect.Top+D);
               END;
        2,3,5,6: BEGIN
               PointArray[1] := Point(SegRect.Left,    SegRect.Top);
               PointArray[2] := Point(SegRect.Left+D,  SegRect.Top+D);
               PointArray[3] := Point(SegRect.Right+D, SegRect.Bottom-D);
               PointArray[4] := Point(SegRect.Right,   SegRect.Bottom);
               PointArray[5] := Point(SegRect.Right-D, SegRect.Bottom-D);
               PointArray[6] := Point(SegRect.Left-D,  SegRect.Top+D);
               END;
        8: BEGIN
               PointArray[1] := Point(SegRect.Left,    SegRect.Top);
               PointArray[2] := Point(SegRect.Right,   SegRect.Top);
               PointArray[3] := Point(SegRect.Right,   SegRect.Bottom);
               PointArray[4] := Point(SegRect.Left,    SegRect.Bottom);
               PointArray[5] := PointArray[4];
               PointArray[6] := PointArray[4];
               END;
        END;
        {Segment zeichnen}
        IF Seg IN Segmente THEN Brush.Color := FActiveLEDColor
                           ELSE Brush.Color := FInActiveLEDColor;
        Polygon(PointArray);
      END;
    END;

  END;
end;

procedure TPieLEDLabel.Paint13Segment;
VAR
  I, D, _, MitteH, MitteB: Integer;
  Segmente: TPieLEDSegmente;
  Seg: Byte;
  CharRect, SegRect: TRect;
  PointArray: array[1..6] of TPoint;
begin
  WITH Canvas DO BEGIN
    D := FSegmentWidth;
    _ := FSegmentSpace;
    {Chars anzeigen}
    FOR I:=1 TO length(Caption) DO BEGIN
      Segmente := GetLED13Segment(Caption[I]);
{      CharRect := Rect((I-1)*FCharWidth, 0, I*FCharWidth-1, FCharHeight-1);}
      CharRect := Rect((I-1)*(FCharWidth+FCharSpace), 0, I*(FCharWidth+FCharSpace)-FCharSpace-1, FCharHeight-1);
      {Position der Segment-Mittellinien ermitteln}
      MitteH := CharRect.Top + (CharRect.Bottom-CharRect.Top) DIV 2;
      MitteB := CharRect.Left + (CharRect.Right-CharRect.Left) DIV 2;
      FOR Seg := 1 TO 18 DO IF (Seg IN Segmente) OR ((Seg < 14) AND FShowInActiveSegments) THEN BEGIN
        CASE Seg OF
         1: SegRect := Rect(CharRect.Left+D+_+1,   CharRect.Top+D+1,      CharRect.Right-D-_,    CharRect.Top+D+1);
         2: SegRect := Rect(CharRect.Left+D+1,     CharRect.Top+D+_+1,    CharRect.Left+D+1,     MitteH-_);
         3: SegRect := Rect(CharRect.Left+2*D+_+1, CharRect.Top+2*D+_+1,  MitteB-D-_,            MitteH-D-_);
         4: SegRect := Rect(MitteB,                CharRect.Top+2*D+_+1,  MitteB,                MitteH-D-_);
         5: SegRect := Rect(CharRect.Right-2*D-_,  CharRect.Top+2*D+_+1,  MitteB+D+_,            MitteH-D-_);
         6: SegRect := Rect(CharRect.Right-D,      CharRect.Top+D+_+1,    CharRect.Right-D,      MitteH-_);
         7: SegRect := Rect(CharRect.Left+D+_+1,   MitteH,                CharRect.Right-D-_,    MitteH);
         8: SegRect := Rect(CharRect.Left+D+1,     MitteH+_,              CharRect.Left+D+1,     CharRect.Bottom-D-_);
         9: SegRect := Rect(MitteB-D-_,            MitteH+D+_,            CharRect.Left+2*D+_+1, CharRect.Bottom-2*D-_);
        10: SegRect := Rect(MitteB,                MitteH+D+_,            MitteB,                CharRect.Bottom-2*D-_);
        11: SegRect := Rect(MitteB+D+_,            MitteH+D+_,            CharRect.Right-2*D-_,  CharRect.Bottom-2*D-_);
        12: SegRect := Rect(CharRect.Right-D,      MitteH+_,              CharRect.Right-D,      CharRect.Bottom-D-_);
        13: SegRect := Rect(CharRect.Left+D+_+1,   CharRect.Bottom-D,     CharRect.Right-D-_,    CharRect.Bottom-D);
        14: SegRect := Rect(MitteB,                MitteH-D-_,            MitteB,                MitteH+D+_);
        15: SegRect := Rect(MitteB,                CharRect.Top+1,        MitteB,                CharRect.Top+2*D+1);
        16: SegRect := Rect(CharRect.Left+D+1,     CharRect.Top,          CharRect.Left+D+1,     CharRect.Top+2*D);
        17: SegRect := Rect(CharRect.Right-D,      CharRect.Top,          CharRect.Right-D,      CharRect.Top+2*D);
        18: SegRect := Rect(MitteB-D,              CharRect.Bottom-2*D,   MitteB+D,              CharRect.Bottom);
        END;
        {Polygon-Punkte ermitteln}
        CASE Seg OF
        1,7,13: BEGIN
               PointArray[1] := Point(SegRect.Left,    SegRect.Top);
               PointArray[2] := Point(SegRect.Left+D,  SegRect.Top-D);
               PointArray[3] := Point(SegRect.Right-D, SegRect.Bottom-D);
               PointArray[4] := Point(SegRect.Right,   SegRect.Bottom);
               PointArray[5] := Point(SegRect.Right-D, SegRect.Bottom+D);
               PointArray[6] := Point(SegRect.Left+D,  SegRect.Top+D);
               END;
        2,4,6,8,10,12,14..17: BEGIN
               PointArray[1] := Point(SegRect.Left,    SegRect.Top);
               PointArray[2] := Point(SegRect.Left+D,  SegRect.Top+D);
               PointArray[3] := Point(SegRect.Right+D, SegRect.Bottom-D);
               PointArray[4] := Point(SegRect.Right,   SegRect.Bottom);
               PointArray[5] := Point(SegRect.Right-D, SegRect.Bottom-D);
               PointArray[6] := Point(SegRect.Left-D,  SegRect.Top+D);
               END;
        3,11: BEGIN
               PointArray[1] := Point(SegRect.Left,    SegRect.Top);
               PointArray[2] := Point(SegRect.Left+D,  SegRect.Top);
               PointArray[3] := Point(SegRect.Right,   SegRect.Bottom-D);
               PointArray[4] := Point(SegRect.Right,   SegRect.Bottom);
               PointArray[5] := Point(SegRect.Right-D, SegRect.Bottom);
               PointArray[6] := Point(SegRect.Left,    SegRect.Top+D);
               END;
        5,9: BEGIN
               PointArray[1] := Point(SegRect.Left,    SegRect.Top);
               PointArray[2] := Point(SegRect.Left-D,  SegRect.Top);
               PointArray[3] := Point(SegRect.Right,   SegRect.Bottom-D);
               PointArray[4] := Point(SegRect.Right,   SegRect.Bottom);
               PointArray[5] := Point(SegRect.Right+D, SegRect.Bottom);
               PointArray[6] := Point(SegRect.Left,    SegRect.Top+D);
               END;
        18: BEGIN
               PointArray[1] := Point(SegRect.Left,    SegRect.Top);
               PointArray[2] := Point(SegRect.Right,   SegRect.Top);
               PointArray[3] := Point(SegRect.Right,   SegRect.Bottom);
               PointArray[4] := Point(SegRect.Left,    SegRect.Bottom);
               PointArray[5] := PointArray[4];
               PointArray[6] := PointArray[4];
               END;
        END;
        {Segment zeichnen}
        IF Seg IN Segmente THEN Brush.Color := FActiveLEDColor
                           ELSE Brush.Color := FInActiveLEDColor;
        Polygon(PointArray);
      END;
    END;

  END;
end;

procedure TPieLEDLabel.SetIntegerValues(Index: Integer; Value: Integer);
begin
  CASE Index OF
  1: IF (Value=FCharWidth) THEN exit ELSE FCharWidth:=Value;
  2: IF (Value=FCharHeight) THEN exit ELSE FCharHeight:=Value;
  3: IF (Value=FSegmentWidth) THEN exit ELSE FSegmentWidth:=Value;
  4: IF (Value=FCharSpace) OR (Value < 0) THEN exit ELSE FCharSpace:=Value;
  5: IF (Value=FSegmentSpace) OR (Value < 0) THEN exit ELSE FSegmentSpace:=Value;
  END;
  Invalidate;
end;

procedure TPieLEDLabel.SetColor(Index: Integer; Value: TColor);
begin
  CASE Index OF
  1: IF Value = FColor THEN exit ELSE FColor := Value;
  2: IF Value = FActiveLEDColor THEN exit ELSE FActiveLEDColor := Value;
  3: IF Value = FInActiveLEDColor THEN exit ELSE FInActiveLEDColor := Value;
  END;
  Invalidate;
end;

procedure TPieLEDLabel.SetBooleanValues(Index: Integer; Value: Boolean);
begin
  CASE Index OF
  1: IF Value = FTransparent THEN exit ELSE FTransparent := Value;
  2: IF Value = FShowInActiveSegments THEN exit ELSE FShowInActiveSegments := Value;
  END;
  Invalidate;
end;

procedure TPieLEDLabel.SetLEDStyle(Value: TPieLEDStyle);
begin
  IF Value <> FLEDStyle THEN BEGIN
    FLEDStyle := Value;
    Invalidate;
  END;
end;

end.

