{$I PIETOOLS.INC}
unit PieJahresKalender;

interface

uses Classes, Controls, Messages, WinTypes, Forms, Graphics, WinProcs, StdCtrls,
  Grids, SysUtils;

type
  TDayOfWeek = 0..6;

  TPieYearCalendar = class(TCustomGrid)
  private
    FDate: TDateTime;
    FOnChange: TNotifyEvent;
    FReadOnly: Boolean;
    FStartOfWeek: TDayOfWeek;
    FUpdating: Boolean;
    FUseCurrentDate: Boolean;
    FTagesliste: TList;  {Speichert eine Liste von Tagen, die markiert werden mssen (TList als LongInt-Pointer)}
    FLongMonths: Boolean;
    FYBLeft, FYBRight: TRect;
    FYBLDown, FYBRDown: Boolean;
    function GetCellText(ACol, ARow: Integer): string;
    function GetDateElement(Index: Integer): Integer;
    function GetMonthOffset(Index: Word): Integer;
    procedure SetCalendarDate(Value: TDateTime);
    procedure SetDateElement(Index: Integer; Value: Integer);
    procedure SetStartOfWeek(Value: TDayOfWeek);
    procedure SetLongMonths(Value: Boolean);
    function StoreCalendarDate: Boolean;
  protected
    procedure Change; dynamic;
    procedure Click; override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    function DaysPerMonth(AYear, AMonth: Integer): Integer; virtual;
    procedure DrawCell(ACol, ARow: Longint; ARect: TRect; AState: TGridDrawState); override;
    function IsLeapYear(AYear: Integer): Boolean; virtual;
    function SelectCell(ACol, ARow: Longint): Boolean; override;
    procedure WMSize(var Message: TWMSize); message WM_SIZE;
  public        {TPieYearCalendar}
    constructor Create(AOwner: TComponent); override;
    destructor destroy; override;
    property CalendarDate: TDateTime  read FDate write SetCalendarDate stored StoreCalendarDate;
    property CellText[ACol, ARow: Integer]: string read GetCellText;
    procedure NextYear;
    procedure PrevYear;
    procedure UpdateCalendar; virtual;
    procedure Tag_hinzu(T, M, J: LongInt);
    Procedure Tagesliste_loeschen;
  published
    property Align;
    property BorderStyle;
    property Color;
    property Ctl3D;
    property Day: Integer index 3  read GetDateElement write SetDateElement stored False;
    property DragMode;
    property Enabled;
    property Font;
    property GridLineWidth;
    property LongMonths: Boolean read FLongMonths write SetLongMonths default FALSE;
    property Month: Integer index 2  read GetDateElement write SetDateElement stored False;
    property ParentColor;
    property ParentFont;
    property ParentShowHint;
    property ReadOnly: Boolean read FReadOnly write FReadOnly default False;
    property ShowHint;
    property StartOfWeek: TDayOfWeek read FStartOfWeek write SetStartOfWeek;
    property TabOrder;
    property TabStop;
    property UseCurrentDate: Boolean read FUseCurrentDate write FUseCurrentDate default True;
    property Visible;
    property Year: Integer index 1  read GetDateElement write SetDateElement stored False;
    property OnClick;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
  end;

(*procedure Register;*)

implementation

USES PieHerk;

constructor TPieYearCalendar.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  { defaults }
  FUseCurrentDate := True;
  FixedCols := 1;
  FixedRows := 1;
  ColCount := 38;
  RowCount := 13;
  ScrollBars := ssNone;
  Options := Options - [goRangeSelect] + [goDrawFocusSelected];
  FDate := Date;
  FTagesListe := TList.Create;
  FLongMonths := FALSE;
  FYBLeft := Rect(0,0,0,0);
  FYBRight := Rect(0,0,0,0);
  FYBLDown := FALSE;
  FYBRDown := FALSE;
  GridLineWidth := 0;
  UpdateCalendar;
end;

destructor TPieYearCalendar.Destroy;
begin
  FTagesListe.Free;
  inherited Destroy;
end;

procedure TPieYearCalendar.Change;
begin
  if Assigned(FOnChange) then FOnChange(Self);
end;

procedure TPieYearCalendar.Click;
var
  TheCellText: string;
begin
  inherited Click;
  TheCellText := CellText[Col, Row];
  if (row > 0) and (col > 0) and (TheCellText <> '') then
  try
    CalendarDate := EnCodeDate(Year,Row,StrToInt(TheCellText));
  except
    on EConvertError do CalendarDate := EnCodeDate(Year, Row, 28);
  end;
end;

function TPieYearCalendar.IsLeapYear(AYear: Integer): Boolean;
begin
  Result := (AYear mod 4 = 0) and ((AYear mod 100 <> 0) or (AYear mod 400 = 0));
end;

function TPieYearCalendar.DaysPerMonth(AYear, AMonth: Integer): Integer;
const
  DaysInMonth: array[1..12] of Integer = (31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31);
begin
  Result := DaysInMonth[AMonth];
  if (AMonth = 2) and IsLeapYear(AYear) then Inc(Result); { leap-year Feb is special }
end;

procedure TPieYearCalendar.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
BEGIN
  IF PointInRect(Point(X,Y), FYBLeft) AND NOT(FYBLDown) THEN BEGIN
    FYBLDown := TRUE;
    DrawEdge(Canvas.Handle, FYBLeft, EDGE_SUNKEN, BF_RECT);
  END;
  IF PointInRect(Point(X,Y), FYBRight) AND NOT(FYBRDown) THEN BEGIN
    FYBRDown := TRUE;
    DrawEdge(Canvas.Handle, FYBRight, EDGE_SUNKEN, BF_RECT);
  END;
  inherited MouseDown(Button, Shift, X, Y);
END;

procedure TPieYearCalendar.MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
BEGIN
  IF PointInRect(Point(X,Y), FYBLeft) AND FYBLDown THEN BEGIN
    FYBLDown := FALSE;
    PrevYear;
  END;
  IF PointInRect(Point(X,Y), FYBRight) AND FYBRDown THEN BEGIN
    FYBRDown := FALSE;
    NextYear;
  END;
  inherited MouseUp(Button, Shift, X, Y);
END;

procedure TPieYearCalendar.MouseMove(Shift: TShiftState; X, Y: Integer);
BEGIN
  IF NOT(PointInRect(Point(X,Y), FYBLeft)) AND FYBLDown THEN BEGIN
    FYBLDown := FALSE;
    DrawEdge(Canvas.Handle, FYBLeft, EDGE_RAISED, BF_RECT);
  END;
  IF NOT(PointInRect(Point(X,Y), FYBRight)) AND FYBRDown THEN BEGIN
    FYBRDown := FALSE;
    DrawEdge(Canvas.Handle, FYBRight, EDGE_RAISED, BF_RECT);
  END;
  inherited MouseMove(Shift, X, Y);
END;

procedure TPieYearCalendar.DrawCell(ACol, ARow: Longint; ARect: TRect; AState: TGridDrawState);
var
  TheText: string;
  MalDatum, D: LongInt;
  Gefunden: Boolean;
  I: Integer;
  P : Pointer;
  MitteVertikal: Integer;
begin
  TheText := CellText[ACol, ARow];
  with ARect, Canvas do BEGIN
    IF (FTagesListe <> NIL) AND (ARow > 0) AND (ACol > 0) AND
       (TheText <> '') THEN BEGIN
      try
        MalDatum := (StrToInt(TheText) SHL 24) {Tag} + (ARow SHL 16) {Monat} + Year;
      except
        on EConvertError do Maldatum := 0;
      end;
      Gefunden := FALSE;
      WITH FTagesListe DO FOR I:=0 TO Count-1 DO IF NOT(Gefunden) THEN BEGIN
        P := Items[I];
        D := LongInt(P^);
        IF D = MalDatum THEN Gefunden := TRUE;
      END;
      IF Gefunden THEN BEGIN
        Brush.Color := clLime;
        FillRect(ARect);
      END;
    END;
    IF (ARow = 0) AND (ACol = 0) THEN BEGIN
      OffsetRect(ARect, -1, 0);
      MitteVertikal := (ARect.Bottom - ARect.Top) DIV 2;
      Brush.Style := bsSolid;
      Brush.Color := Font.Color;
      Pen.Color := Font.Color;
      FYBLeft := Rect(ARect.Left, ARect.Top, ARect.Left+10, ARect.Bottom);
      FYBRight := Rect(ARect.Right-10, ARect.Top, ARect.Right, ARect.Bottom);
      DrawEdge(Handle, FYBLeft, EDGE_RAISED, BF_RECT);
      DrawEdge(Handle, FYBRight, EDGE_RAISED, BF_RECT);
      Polygon([Point(ARect.Left+3, MitteVertikal),
               Point(ARect.Left+6, MitteVertikal-3),
               Point(ARect.Left+6, MitteVertikal+3)]);
      Polygon([Point(ARect.Right-4, MitteVertikal),
               Point(ARect.Right-7, MitteVertikal-3),
               Point(ARect.Right-7, MitteVertikal+3)]);
      Brush.Style := bsClear;
      InflateRect(ARect, -7, 0);
    END;
    TextRect(ARect, Left + (Right - Left - TextWidth(TheText)) div 2,
      Top + (Bottom - Top - TextHeight(TheText)) div 2, TheText);
  END;
end;

function TPieYearCalendar.GetMonthOffset(Index: Word): integer;
begin
     Result := DayOfWeek(EncodeDate(Year,Index,1)) - (StartOfWeek+1);
     if Result < 0 then Result := 7 + Result;
end;

function TPieYearCalendar.GetCellText(ACol, ARow: Integer): string;
var
  DayNum: Integer;
begin
  Result := '';
  if (ARow = 0) or (ACol = 0) then  { day names at tops of columns }
  begin
    Result := IntToStr(Year);
    if aCol > 0 then
      Result := copy(ShortDayNames[(StartOfWeek + ACol - 1) mod 7 + 1],1,1);
    IF ARow > 0 THEN BEGIN
      IF FLongMonths
        THEN Result := LongMonthNames[ARow]
        ELSE Result := ShortMonthNames[ARow];
    END;
  end
  else
  begin
    DayNum := ACol - GetMonthOffset(aRow);
    if (DayNum < 1) or (DayNum > DaysPerMonth(Year, aRow)) then Result := ''
    else Result := IntToStr(DayNum);
  end;
end;

function TPieYearCalendar.SelectCell(ACol, ARow: Longint): Boolean;
begin
  if ((not FUpdating) and FReadOnly) or (CellText[ACol, ARow] = '') or
     (ACol = 0) or (ARow = 0) then
    Result := False
  else Result := inherited SelectCell(ACol, ARow);
end;

procedure TPieYearCalendar.SetCalendarDate(Value: TDateTime);
var
  AYear, AMonth, ADay, OldYear: Word;
begin
  DecodeDate(FDate, OldYear, AMonth, ADay);
  FDate := Value;
  DecodeDate(FDate, AYear, AMonth, ADay);
  if AYear <> OldYear then Invalidate;
  UpdateCalendar;
  Change;
end;

function TPieYearCalendar.StoreCalendarDate: Boolean;
begin
  Result := not FUseCurrentDate;
end;

function TPieYearCalendar.GetDateElement(Index: Integer): Integer;
var
  AYear, AMonth, ADay: Word;
begin
  DecodeDate(FDate, AYear, AMonth, ADay);
  case Index of
    1: Result := AYear;
    2: Result := AMonth;
    3: Result := ADay;
    else Result := -1;
  end;
end;

procedure TPieYearCalendar.SetDateElement(Index: Integer; Value: Integer);
var
  AYear, AMonth, ADay: Word;
begin
  if Value > 0 then
  begin
    DecodeDate(FDate, AYear, AMonth, ADay);
    case Index of
      1: if AYear <> Value then
         begin
              AYear := Value;
              Invalidate;
         end
         else Exit;
      2: if (Value <= 12) and (Value <> AMonth) then AMonth := Value else Exit;
      3: if (Value <= DaysPerMonth(AYear, AMonth)) and (Value <> ADay) then ADay := Value else Exit;
      else Exit;
    end;
    FDate := EncodeDate(AYear, AMonth, ADay);
    FUseCurrentDate := False;
    UpdateCalendar;
    Change;
  end;
end;

procedure TPieYearCalendar.SetStartOfWeek(Value: TDayOfWeek);
begin
  if Value <> FStartOfWeek then
  begin
    FStartOfWeek := Value;
    Invalidate;
    UpdateCalendar;
  end;
end;

procedure TPieYearCalendar.NextYear;
begin
  if NOT(IsLeapYear(Year+1)) and (Month = 2) and (Day = 29) then Day := 28;
  Year := Year + 1;
  Invalidate;
end;

procedure TPieYearCalendar.PrevYear;
begin
  if NOT(IsLeapYear(Year-1)) and (Month = 2) and (Day = 29) then Day := 28;
  Year := Year - 1;
  Invalidate;  
end;

procedure TPieYearCalendar.UpdateCalendar;
var
  AYear, AMonth, ADay: Word;
begin
  FUpdating := True;
  try
    DecodeDate(FDate, AYear, AMonth, ADay);
    MoveColRow((ADay + GetMonthOffset(aMonth)), AMonth,False, False);
  finally
    FUpdating := False;
  end;
end;

procedure TPieYearCalendar.WMSize(var Message: TWMSize);
var
   aMonth : Word;
   sText  : string;
   iWidth : integer;
begin
  iWidth := Canvas.TextWidth(IntToStr(Year)) + 2*7; {JahresButton links & rechts}
  for aMonth := 1 to 12 do
  begin
    IF FLongMonths
      THEN sText := LongMonthNames[AMonth]
      ELSE sText := ShortMonthNames[AMonth];
    if Canvas.TextWidth(sText) > iWidth then
      iWidth := Canvas.TextWidth(sText);
  end;
  iWidth := iWidth + 2;
  DefaultColWidth := ((Message.Width-iWidth) - 37 * GridLineWidth) div 37;
  DefaultRowHeight := (Message.Height - 12 * GridLineWidth) div 13;
  ColWidths[0] := Message.Width - (DefaultColWidth+GridLineWidth) * 37;
  RowHeights[0] := Message.Height - (DefaultRowHeight+GridLineWidth) * 12;
end;

procedure TPieYearCalendar.SetLongMonths(Value: Boolean);
VAR
  M: TWMSize;
BEGIN
  IF Value <> FLongMonths THEN BEGIN
    FLongMonths := Value;
    M.Width := Width;
    M.Height := Height;
    WMSize(M);
    Invalidate;
  END;
END;

procedure TPieYearCalendar.Tag_hinzu(T, M, J: LongInt);
TYPE PLongInt = ^LongInt;
VAR
  MalDatum: LongInt;         {TList}
  PMalDatum: PLongInt;
BEGIN
  MalDatum := (T SHL 24) {Tag} + (M SHL 16) {Monat} + J;
  PMaldatum := New(PLongInt);
  PMalDatum^ := MalDatum;
  FTagesListe.Add(PMalDatum);
  Invalidate;
END;

procedure TPieYearCalendar.Tagesliste_loeschen;
BEGIN
  FTagesListe.Clear;
  Invalidate;
END;

end.
