{$I PIETOOLS.INC}
{ Autor: Ingolf Pietschmann.
  Dieser Quelltext ist Freeware. Die Verwendung und Weitergabe dieser Sourcen zu
  privaten nicht kommerziellen Zwecken ist ausdrcklich erwnscht.
  Die Verwendung zu kommerziellen Zwecken ist nur mit Erlaubnis des Autors
  gestattet. Den Autor knnen Sie unter "Support@Pie-Tools.de" erreichen.
  Homepage unter: http://www.Pie-Tools.de/

  These sources are freeware. The usage and distribution of these sources for
  private, not commercial purposes is explicit desired.
  The usage for commercial purposes is only permitted in agreement of the author.
  The author can be reached by "Support@Pie-Tools.de".
  Homepage: http://www.Pie-Tools.de/
}

{Pie-Tools,  2001                                      }
{derive from:  EC Software 1997-1999                   }
{Thanks also to: Richard B. Winston                     }

unit PieDataGrid;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs, ExtCtrls,
  Grids, StdCtrls, Menus, DBCtrls, DB, PieHerk
  {$IFDEF D6_OR_HIGHER}
  , Variants;
  {$ELSE}
  ;
  {$ENDIF}

type
  TColumnValue = (cvColor, cvWidth, cvFont, cvAlignment, cvReadOnly, cvTitleColor, cvTitleCaption, cvTitleAlignment, cvTitleFont);
  TColumnValues = set of TColumnValue;

const
  ColumnTitleValues = [cvTitleColor..cvTitleFont];
  cm_DeferLayout = WM_USER + 100;

type
  TColStyle = (colsEdit, colsCombo, colsButton, colsCheck, colsComboButton);
  TColumnFormat = (cfString, cfNumber, cfDate, cfInteger);

  TPieDataGridOption = (dgoAppendRow, dgoInsertRow, dgoDeleteRow);
  TPieDataGridOptions = set of TPieDataGridOption;
  TColumnSortEvent = procedure(GridControl: TCustomGrid; Column: Integer; ColumnUp: Boolean; VAR DefaultSort: Boolean) of object;
  TSortProgressEvent = procedure (Sender: TObject; Stage: TProgressStage; PercentDone: Byte) of object;
  TPieGridClickEvent = procedure (Sender: TObject; ACol, ARow: Integer) of object;

  TDGColumn = class;
  TPieCustomDataGrid = class;
  TPieDBDataGrid = class;

  TColumnTitle = class(TPersistent)
  private
    FColumn: TDGColumn;
    FCaption: string;
    FFont: TFont;
    FAlignment: TAlignment;
    procedure FontChanged(Sender: TObject);
    function GetAlignment: TAlignment;
    function GetCaption: string;
    function GetFont: TFont;
    function IsAlignmentStored: Boolean;
    function IsFontStored: Boolean;
    function IsCaptionStored: Boolean;
    procedure SetAlignment(Value: TAlignment);
    procedure SetFont(Value: TFont);
    procedure SetCaption(const Value: string); virtual;
  protected
    procedure RefreshDefaultFont;
  public
    constructor Create(Column: TDGColumn);
    destructor Destroy; override;
    procedure Assign(Source: TPersistent); override;
    function DefaultAlignment: TAlignment;
    function DefaultColor: TColor;
    function DefaultFont: TFont;
    function DefaultCaption: string;
    procedure RestoreDefaults; virtual;
  published
    property Alignment: TAlignment read GetAlignment write SetAlignment stored IsAlignmentStored;
    property Caption: string read GetCaption write SetCaption stored IsCaptionStored;
    property Font: TFont read GetFont write SetFont stored IsFontStored;
  end;

  TDGColumn = class(TCollectionItem)
  private
    FColor: TColor;
    FLimitToList: Boolean;
    FEditMask: String;
    FMaxLength: Integer;
    FFormat: TColumnFormat;
    FDisplayMask: String;
    FTitle: TColumnTitle;
    FFont: TFont;
    FPickList: TStrings;
    FPopupMenu: TPopupMenu;
    FDropDownRows: Integer;
    FColStyle: TColStyle;
    FAlignment: TAlignment;
    FReadonly: Boolean;
    FEditButtonEnabled: Boolean;
    FAssignedValues: TColumnValues;
    FInternalCol: Longint;
    FCheckBoxSize: TPieCheckBoxSize;
    FCheckBoxStyle: TPieCheckBoxStyle;
    FHookStyle: TPieHookStyle;
    FHookColor: TColor;
    FTag: Integer;
    procedure FontChanged(Sender: TObject);
    function  GetAlignment: TAlignment;
    function  GetCheckBoxSize: Integer;
    function  GetColor: TColor;
    function  GetFont: TFont;
    function  GetPickList: TStrings;
    function  GetReadOnly: Boolean;
    function  GetDisplayMask: String;
    function  GetMaxLength: Integer;
    function  IsAlignmentStored: Boolean;
    function  IsColorStored: Boolean;
    function  IsFontStored: Boolean;
    function  IsReadOnlyStored: Boolean;
    function  IsDisplayMaskStored: Boolean;
    procedure SetAlignment(Value: TAlignment); virtual;
    procedure SetCheckBoxSize(Value: TPieCheckBoxSize);
    procedure SetCheckBoxStyle(Value: TPieCheckBoxStyle);
    procedure SetColor(Value: TColor);
    procedure SetColStyle(Value: TColStyle);
    procedure SetDisplayMask(Value: String); virtual;
    procedure SetEditButtonEnabled(Value: Boolean);
    procedure SetFormat(Value: TColumnFormat);
    procedure SetFont(Value: TFont);
    procedure SetHookStyle(Value: TPieHookStyle);
    procedure SetHookColor(Value: TColor);
    procedure SetMaxLength(Value: Integer);
    procedure SetPickList(Value: TStrings);
    procedure SetPopupMenu(Value: TPopupMenu);
    procedure SetReadOnly(Value: Boolean); virtual;
    procedure SetTitle(Value: TColumnTitle);
  protected
    function  CreateTitle: TColumnTitle; virtual;
    function  GetGrid: TPieCustomDataGrid;
    procedure RefreshDefaultFont;
  public
    constructor Create(Collection: TCollection); override;
    destructor Destroy; override;
    procedure Assign(Source: TPersistent); override;
    function  DefaultAlignment: TAlignment;
    function  DefaultColor: TColor;
    function  DefaultFont: TFont;
    function  DefaultReadOnly: Boolean;
    function  DefaultWidth: Integer;
    procedure RestoreDefaults; virtual;
    property  Grid: TPieCustomDataGrid read GetGrid;
    property  AssignedValues: TColumnValues read FAssignedValues;
  published
    property Alignment: TAlignment read GetAlignment write SetAlignment stored IsAlignmentStored;
    property CheckBoxSize: TPieCheckBoxSize read FCheckBoxSize write SetCheckBoxSize default pcbsNormal;
    property CheckBoxStyle: TPieCheckBoxStyle read FCheckBoxStyle write SetCheckBoxStyle default pcbsSunken;
    property Color: TColor read GetColor write SetColor stored IsColorStored;
    property ColStyle: TColStyle read FColStyle write SetColStyle default colsEdit;
    property DisplayMask: String read GetDisplayMask write SetDisplayMask stored IsDisplayMaskStored;
    property DropDownRows: Integer read FDropDownRows write FDropDownRows default 7;
    property EditButtonEnabled: Boolean read FEditButtonEnabled write SetEditButtonEnabled default TRUE;
    property EditMask: String read FEditMask write FEditMask;
    property Font: TFont read GetFont write SetFont stored IsFontStored;
    property Format: TColumnFormat read FFormat write SetFormat default cfString;
    property HookColor: TColor read FHookColor write SetHookColor default clBtnText;
    property HookStyle: TPieHookStyle read FHookStyle write SetHookStyle default phsHook;
    property InternalCol: LongInt read FInternalCol;
    property LimitToList: Boolean read FLimitToList write FLimitToList default false;
    property MaxLength: Integer read GetMaxLength write SetMaxLength default 0;
    property PickList: TStrings read GetPickList write SetPickList;
    property PopupMenu: TPopupMenu read FPopupMenu write SetPopupMenu;
    property ReadOnly: Boolean read GetReadOnly write SetReadOnly stored IsReadOnlyStored;
    property Tag: Integer read FTag write FTag default 0;
    property Title: TColumnTitle read FTitle write SetTitle;
  end;

  TDBDGColumn = class(TDGColumn)
  private
    function GetDataField: string;
    procedure SetDataField(Value: string);
  public
  published
    property DataField: string read GetDataField write SetDataField;
  end;

  TColumnClass = class of TDGColumn;

  TPieDataGridColumns = class(TCollection)
  private
    FGrid: TPieCustomDataGrid;
    function GetColumn(Index: Integer): TDGColumn;
    procedure SetColumn(Index: Integer; Value: TDGColumn);
  protected
    function GetOwner: TPersistent; override; {D3}
    procedure Update(Item: TCollectionItem); override;
  public
    constructor Create(Grid: TPieCustomDataGrid; ColumnClass: TColumnClass); virtual;
    function  Add: TDGColumn;
    procedure LoadFromFile(const Filename: string);
    procedure LoadFromStream(S: TStream);
    procedure RestoreDefaults;
    procedure SaveToFile(const Filename: string);
    procedure SaveToStream(S: TStream);
    property Grid: TPieCustomDataGrid read FGrid;
    property Items[Index: Integer]: TDGColumn read GetColumn write SetColumn; default;
  end;

  TPieDBDataGridColumns = class(TPieDataGridColumns)
  private
    function GetColumn(Index: Integer): TDBDGColumn;
    procedure SetColumn(Index: Integer; Value: TDBDGColumn);
  public
    property Items[Index: Integer]: TDBDGColumn read GetColumn write SetColumn; default;
  end;

{ TPieCustomDataGrid }
  TPieCustomDataGrid = class(TStringGrid)
  private
    FColumns: TPieDataGridColumns;
    FRowCountMin: LongInt;
    FDataGridOptions: TPieDataGridOptions;
    FLayoutFlag: Integer;
    FBackground: TBitmap;
    FEditorColor: TColor;
    FSelectedCellColor: TColor;
    FSelectedCellFontColor: TColor;
    FSortColumn: Integer;
    FSortUp: Boolean;
    FSorted: Boolean;
    FOnBeforeInsert: TNotifyEvent;
    FOnBeforeDelete: TNotifyEvent;
    FOnChange: TNotifyEvent;
    FOnColumnSort: TColumnSortEvent;
    FOnEditButtonClick: TNotifyEvent;
    FOnGetEditMask: TGetEditEvent;
    FOnSortProgress: TSortProgressEvent;
    FOnExitCell: TPieGridClickEvent;
    FOnFixedCellClick: TPieGridClickEvent;
    FColMerker: Integer;
    FRowMerker: Integer;
    FVersion: string;
    function  GetIntValue(ACol, ARow: Integer): Integer;
    procedure SetIntValue(ACol, ARow: Integer; const Value: Integer);
    function  GetNumValue(ACol, ARow: Integer): Real;
    procedure SetNumValue(ACol, ARow: Integer; const Value: Real);
    procedure SetBackground(newImg: TBitmap);
    procedure SetColumnCount(NewCount: LongInt);
    procedure SetColumns(Value: TPieDataGridColumns);
    procedure SetEditorColor(Value: TColor);
    function  GetEditorSelLength: Integer;
    procedure SetEditorSelLength(Value: Integer);
    function  GetEditorSelStart: Integer;
    procedure SetEditorSelStart(Value: Integer);
    function  GetEditorSelText: string;
    procedure SetEditorSelText(Value: string);
    procedure SetRowCountMin(Value: LongInt);
    procedure SetSelectedCellColor(Value: TColor);
    procedure SetSelectedCellFontColor(Value: TColor);
    function  GetSelectedIndex: Integer;
    procedure SetSelectedIndex(Value: Integer);
    procedure SetSorted(Value: Boolean);
    procedure SetSortColumn(Value: Integer);
    procedure SetSortUp(Value: Boolean);
    function  GetVersion: string;
    function  CheckDataGridKey(var Key: Word; Shift: TShiftState): Boolean;
    procedure DrawBackground(rect: TRect; AState: TGridDrawState);
    procedure DrawHeaderCell(ACol, ARow: Longint; ARect: TRect; AState: TGridDrawState);
    procedure DrawDataCell(ACol, ARow: Longint; ARect: TRect; AState: TGridDrawState);
    procedure GridSortieren;
  protected
    function CanEditAcceptKey(Key: Char): Boolean; override;
    procedure DataChanged(Col, Row: Integer; Value: string); virtual;
    procedure SizeChanged(OldColCount, OldRowCount: Longint); override;
    procedure RowCountMinChanged; dynamic;
    function  CreateEditor: TInplaceEdit; override;
    function  CreateColumns: TPieDataGridColumns; dynamic;
    function  GetEditMask(ACol, ARow: Longint): string; override;
    procedure EditButtonClick; dynamic;
    procedure DrawCell(ACol, ARow: Longint; ARect: TRect; AState: TGridDrawState); override;
    procedure ColumnMoved(FromIndex, ToIndex: Longint); override;
    procedure DoEnter; override;
    procedure DoExit; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure Loaded; override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure TopLeftChanged; override;
    procedure Paint; override;
    function SelectCell(ACol, ARow: Longint): Boolean; override;
    {Properties}
    property Background: TBitmap read FBackground write SetBackground;
    property Columns: TPieDataGridColumns read FColumns write SetColumns;
    property DataGridOptions: TPieDataGridOptions read FDataGridOptions write FDataGridOptions default [];
    property EditorColor: TColor read FEditorColor write SetEditorColor default clWindow;
    property EditorSelLength: Integer read GetEditorSelLength write SetEditorSelLength;
    property EditorSelStart: Integer read GetEditorSelStart write SetEditorSelStart;
    property EditorSelText: string read GetEditorSelText write SetEditorSelText;
    property SelectedCellColor: TColor read FSelectedCellColor write SetSelectedCellColor default clWindow;
    property SelectedCellFontColor: TColor read FSelectedCellFontColor write SetSelectedCellFontColor default clBlue;
    property RowCountMin: LongInt read FRowCountMin write SetRowCountMin;
    property SelectedIndex: Integer read GetSelectedIndex write SetSelectedIndex;
    property Sorted: Boolean read FSorted write SetSorted default FALSE;
    property SortColumn: Integer read FSortColumn write SetSortColumn default 0;
    property SortUp: Boolean read FSortUp write SetSortUp default TRUE;
    property OnBeforeDelete: TNotifyEvent read FOnBeforeDelete write FOnBeforeDelete;
    property OnBeforeInsert: TNotifyEvent read FOnBeforeInsert write FOnBeforeInsert;
    property OnChange: TNotifyEvent read FOnChange write FOnChange;
    property OnColumnSort: TColumnSortEvent read FOnColumnSort write FOnColumnSort;
    property OnEditButtonClick: TNotifyEvent read FOnEditButtonClick write FOnEditButtonClick;
    property OnExitCell: TPieGridClickEvent read FOnExitCell write FOnExitCell;
    property OnFixedCellClick: TPieGridClickEvent read FOnFixedCellClick write FOnFixedCellClick;
    property OnGetEditMask: TGetEditEvent read FOnGetEditMask write FOnGetEditMask;
    property OnSortProgress: TSortProgressEvent read FOnSortProgress write FOnSortProgress;
    property Version: string read GetVersion write FVersion;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure MoveRow(FromIndex, ToIndex: Longint); virtual;
    procedure AppendRow; virtual;
    procedure DeleteRow(ARow: LongInt); override;
    procedure InsertRow(ARow: LongInt); virtual;
    procedure DefaultHandler(var Msg); override;
    procedure EditorSelectAll;
    procedure SortGrid;
    property  IntValue[ACol, ARow: Integer]: Integer read GetIntValue write SetIntValue;
    property  NumValue[ACol, ARow: Integer]: Real    read GetNumValue write SetNumValue;
  published
  end;

{ TPieDataGrid }
  TPieDataGrid = class(TPieCustomDataGrid)
  private
  public
    property EditorSelLength;
    property EditorSelStart;
    property EditorSelText;
  published
    property Background;
    property Columns;
    property DataGridOptions;
    property EditorColor;
    property SelectedCellColor;
    property SelectedCellFontColor;
    property RowCountMin;
    property SortColumn;
    property Sorted;
    property SortUp;
    property OnBeforeDelete;
    property OnBeforeInsert;
    property OnChange;
    property OnColumnSort;
    property OnEditButtonClick;
    property OnExitCell;
    property OnFixedCellClick;
    property OnGetEditMask;
    property OnSortProgress;
    property Version;
  end;

{ TPieDBDataGrid }

  TPieDBDataGrid = class(TPieCustomDataGrid)
  private
    FDataLink: array of TFieldDataLink;
    FDataSource: TDataSource;
    procedure ActiveChange(Sender: TObject);
    procedure DataChange(Sender: TObject);
    function GetDataField(Index: Integer): string;
    function GetDataSource: TDataSource;
    procedure SetDataField(Index: Integer; const Value: string);
    procedure SetDataSource(Value: TDataSource);
    function GetColumns: TPieDBDataGridColumns;
    procedure SetColumns(Value: TPieDBDataGridColumns);
  protected
    function  CreateColumns: TPieDataGridColumns; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure Click; override;
    procedure DataChanged(Col, Row: Integer; Value: string); override;
    procedure SizeChanged(OldColCount, OldRowCount: Longint); override;
    procedure RowCountMinChanged; override;
    function GetDataLink(Index: Integer): TFieldDataLink;
    property DataField[Index: Integer]: string read GetDataField write SetDataField;
  public
    property EditorSelLength;
    property EditorSelStart;
    property EditorSelText;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure AppendRow; override;
    procedure DeleteRow(ARow: LongInt); override;
    procedure InsertRow(ARow: LongInt); override;
  published
    property Background;
    property Columns: TPieDBDataGridColumns read GetColumns write SetColumns;
    property DataGridOptions;
    property DataSource: TDataSource read GetDataSource write SetDataSource;
    property EditorColor;
    property SelectedCellColor;
    property SelectedCellFontColor;
    property RowCountMin;
    property OnBeforeDelete;
    property OnBeforeInsert;
    property OnChange;
    property OnEditButtonClick;
    property OnExitCell;
    property OnFixedCellClick;
    property OnGetEditMask;
    property Version;
  end;


{ TPieDataGridInplaceEdit }

type
  TPopupListbox = class;

  TPieDataGridInplaceEdit = class(TInplaceEdit)
  private
    FPickList: TPopupListbox;
    FActiveList: TWinControl;
    FColStyle: TColStyle;
    FListVisible: Boolean;
    FTracking: Boolean;
    FPressedButton: Boolean;
    FPressedCombo: Boolean;
    FLastText: String;
    FColumn: TDGColumn;
    function GetButtonWidth(AColStyle: TColStyle): Integer;
    function GetCheckBoxWidth(VAR CBRect: TRect): Integer;
    procedure ListMouseUp(Sender: TObject; Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
    procedure SetColStyle(Value: TColStyle);
    procedure StopTracking;
    procedure TrackButton(X,Y: Integer);
    procedure ToggleChecked;
    procedure CMCancelMode(var Message: TCMCancelMode); message CM_CancelMode;
    procedure WMCancelMode(var Message: TMessage); message WM_CancelMode;
    procedure WMKillFocus(var Message: TMessage); message WM_KillFocus;
    procedure WMLButtonDblClk(var Message: TWMLButtonDblClk); message wm_LButtonDblClk;
    procedure WMPaint(var Message: TWMPaint); message wm_Paint;
    procedure WMSetCursor(var Message: TWMSetCursor); message WM_SetCursor;
  protected
    procedure SetReadOnly;
    procedure ValidateContent; dynamic;
    procedure BoundsChanged; override;
    procedure CloseUp(Accept: Boolean);
    procedure DoDropDownKeys(var Key: Word; Shift: TShiftState);
    procedure DropDown;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure KeyPress(var Key: Char); override;
    procedure MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure MouseMove(Shift: TShiftState; X, Y: Integer); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
    procedure PaintWindow(DC: HDC); override;
    procedure UpdateContents; override;
    procedure WndProc(var Message: TMessage); override;
    property  ActiveList: TWinControl read FActiveList write FActiveList;
    property  PickList: TPopupListbox read FPickList;
  public
    constructor Create(Owner: TComponent); override;
    destructor Destroy; override;
  end;

{ TPopupListbox }

  TPopupListbox = class(TCustomListbox)
  private
    FSearchText: String;
    FSearchTickCount: Longint;
  protected
    procedure CreateParams(var Params: TCreateParams); override;
    procedure CreateWnd; override;
    procedure KeyPress(var Key: Char); override;
    procedure MouseUp(Button: TMouseButton; Shift: TShiftState; X, Y: Integer); override;
  end;


type
  TWinControlCracker = class(TWinControl) end;


function  ReplaceChar(C1: Char; S: String; C2: Char): String;
procedure KillMessage(Wnd: HWnd; Msg: Integer);

implementation

{ TColumnTitle }
constructor TColumnTitle.Create(Column: TDGColumn);
begin
  inherited Create;
  FColumn := Column;
  FFont := TFont.Create;
  FFont.Assign(DefaultFont);
  FFont.OnChange := FontChanged;
end;

destructor TColumnTitle.Destroy;
begin
  FFont.Free;
  inherited Destroy;
end;

procedure TColumnTitle.Assign(Source: TPersistent);
begin
  if Source is TColumnTitle then
  begin
    if cvTitleAlignment in TColumnTitle(Source).FColumn.FAssignedValues then
      Alignment := TColumnTitle(Source).Alignment;
{    if cvTitleColor in TColumnTitle(Source).FColumn.FAssignedValues then
      Color := TColumnTitle(Source).Color;}
    if cvTitleCaption in TColumnTitle(Source).FColumn.FAssignedValues then
      Caption := TColumnTitle(Source).Caption;
    if cvTitleFont in TColumnTitle(Source).FColumn.FAssignedValues then
      Font := TColumnTitle(Source).Font;
  end
  else
    inherited Assign(Source);
end;

function TColumnTitle.DefaultAlignment: TAlignment;
begin
  Result := taLeftJustify;
end;

function TColumnTitle.DefaultColor: TColor;
var
  Grid: TPieCustomDataGrid;
begin
  Grid := FColumn.GetGrid;
  if Assigned(Grid) then
    Result := Grid.FixedColor
  else
    Result := clBtnFace;
end;

function TColumnTitle.DefaultFont: TFont;
var
  Grid: TPieCustomDataGrid;
begin
  Grid := FColumn.GetGrid;
  if Assigned(Grid) then
    Result := Grid.font //Grid.TitleFont
  else
    Result := FColumn.Font;
end;

function TColumnTitle.DefaultCaption: string;
begin
  Result := '';
end;

procedure TColumnTitle.FontChanged(Sender: TObject);
begin
  Include(FColumn.FAssignedValues, cvTitleFont);
  FColumn.Changed(True);
end;

function TColumnTitle.GetAlignment: TAlignment;
begin
  if cvTitleAlignment in FColumn.FAssignedValues then
    Result := FAlignment
  else
    Result := DefaultAlignment;
end;

function TColumnTitle.GetCaption: string;
begin
  if cvTitleCaption in FColumn.FAssignedValues then
    Result := FCaption
  else
    Result := DefaultCaption;
end;

function TColumnTitle.GetFont: TFont;
var
  Save: TNotifyEvent;
  Def: TFont;
begin
  if not (cvTitleFont in FColumn.FAssignedValues) then
  begin
    Def := DefaultFont;
    if (FFont.Handle <> Def.Handle) or (FFont.Color <> Def.Color) then
    begin
      Save := FFont.OnChange;
      FFont.OnChange := nil;
      FFont.Assign(DefaultFont);
      FFont.OnChange := Save;
    end;
  end;
  Result := FFont;
end;

function TColumnTitle.IsAlignmentStored: Boolean;
begin
  Result := (cvTitleAlignment in FColumn.FAssignedValues) and
    (FAlignment <> DefaultAlignment);
end;

function TColumnTitle.IsFontStored: Boolean;
begin
  Result := (cvTitleFont in FColumn.FAssignedValues);
end;

function TColumnTitle.IsCaptionStored: Boolean;
begin
  Result := (cvTitleCaption in FColumn.FAssignedValues) and
    (FCaption <> DefaultCaption);
end;

procedure TColumnTitle.RefreshDefaultFont;
var
  Save: TNotifyEvent;
begin
  if (cvTitleFont in FColumn.FAssignedValues) then Exit;
  Save := FFont.OnChange;
  FFont.OnChange := nil;
  try
    FFont.Assign(DefaultFont);
  finally
    FFont.OnChange := Save;
  end;
end;

procedure TColumnTitle.RestoreDefaults;
var
  FontAssigned: Boolean;
begin
  FontAssigned := cvTitleFont in FColumn.FAssignedValues;
  FColumn.FAssignedValues := FColumn.FAssignedValues - ColumnTitleValues;
  FCaption := '';
  RefreshDefaultFont;
  FColumn.Changed(FontAssigned);
end;

procedure TColumnTitle.SetAlignment(Value: TAlignment);
begin
  if (cvTitleAlignment in FColumn.FAssignedValues) and (Value = FAlignment) then Exit;
  FAlignment := Value;
  Include(FColumn.FAssignedValues, cvTitleAlignment);
  FColumn.Changed(False);
end;

procedure TColumnTitle.SetFont(Value: TFont);
begin
  FFont.Assign(Value);
end;

procedure TColumnTitle.SetCaption(const Value: string);
begin
  if (cvTitleCaption in FColumn.FAssignedValues) and (Value = FCaption) then Exit;
  FCaption := Value;
  Include(FColumn.FAssignedValues, cvTitleCaption);
  FColumn.Changed(False);
end;


{ TDGColumn }

constructor TDGColumn.Create(Collection: TCollection);
var
  Grid: TPieCustomDataGrid;
begin
  Grid := nil;
  if Assigned(Collection) and (Collection is TPieDataGridColumns) then Grid := TPieDataGridColumns(Collection).Grid;
  try
    inherited Create(Collection);
    FDropDownRows := 7;
    FColStyle := colsEdit;
    FFont := TFont.Create;
    FFont.Assign(DefaultFont);
    FFont.OnChange := FontChanged;
    FEditButtonEnabled := TRUE;
    FReadOnly := false;
    FTitle := CreateTitle;
    FCheckBoxSize := pcbsNormal;
    FCheckBoxStyle := pcbsSunken;
    FHookStyle := phsHook;
    FHookColor := clBtnText;
  finally
    if (Grid <> nil) then
    begin
         grid.setcolumncount(Grid.columns.count);
         {if not (csDesigning in Grid.ComponentState) then} FInternalCol := Grid.columns.count-1;
    end;
  end;
end;

destructor TDGColumn.Destroy;
begin
  FTitle.Free;
  FFont.Free;
  FPickList.Free;
  with TPieDataGridColumns(Collection).Grid do if FLayoutFlag = 0 then setcolumncount(Grid.colcount-1);
  inherited Destroy;
end;

procedure TDGColumn.Assign(Source: TPersistent);
VAR
  Src: TDGColumn;
begin
  if Source is TDGColumn then begin
    Src := Source as TDGColumn;
    if Assigned(Collection) then Collection.BeginUpdate;
    try
      RestoreDefaults;
      if cvColor in Src.AssignedValues then Color := Src.Color;
      if cvFont in Src.AssignedValues then Font := Src.Font;
      if cvAlignment in Src.AssignedValues then Alignment := Src.Alignment;
      if cvReadOnly in Src.AssignedValues then ReadOnly := Src.ReadOnly;
      CheckBoxSize := Src.CheckBoxSize;
      CheckBoxStyle := Src.CheckBoxStyle;
      ColStyle := Src.ColStyle;
      DisplayMask := Src.DisplayMask;
      DropDownRows := Src.DropDownRows;
      EditButtonEnabled := Src.EditButtonEnabled;
      EditMask := Src.EditMask;
      Format := Src.Format;
      HookColor := Src.HookColor;
      HookStyle := Src.HookStyle;
      LimitToList := Src.LimitToList;
      MaxLength := Src.MaxLength;
      PickList.Assign(Src.PickList);
      PopupMenu := Src.PopupMenu;
      Title.Assign(Src.Title);
    finally
      if Assigned(Collection) then Collection.EndUpdate;
    end;
  end
  else
    inherited Assign(Source);
end;

function TDGColumn.CreateTitle: TColumnTitle;
begin
  Result := TColumnTitle.Create(Self);
end;

function TDGColumn.DefaultAlignment: TAlignment;
begin
    Result := taLeftJustify;
end;

function TDGColumn.DefaultColor: TColor;
var
  Grid: TPieCustomDataGrid;
begin
  Grid := GetGrid;
  if Assigned(Grid) then
    Result := Grid.Color
  else
    Result := clWindow;
end;

function TDGColumn.DefaultFont: TFont;
var
  Grid: TPieCustomDataGrid;
begin
  Grid := GetGrid;
  if Assigned(Grid) then
    Result := Grid.Font
  else
    Result := FFont;
end;

function TDGColumn.DefaultReadOnly: Boolean;
var
  Grid: TPieCustomDataGrid;
begin
  Grid := GetGrid;
  if Assigned(Grid) then Result := not (goEditing in Grid.options) else Result := true;
end;

function TDGColumn.DefaultWidth: Integer;
begin
  if GetGrid = nil then
  begin
    Result := 64;
    Exit;
  end;
  with GetGrid do
  begin
      Result := DefaultColWidth;
  end;
end;

procedure TDGColumn.FontChanged;
begin
  Include(FAssignedValues, cvFont);
  Title.RefreshDefaultFont;
  Changed(False);
end;

function TDGColumn.GetAlignment: TAlignment;
begin
  if cvAlignment in FAssignedValues then
    Result := FAlignment
  else
    Result := DefaultAlignment;
end;

function TDGColumn.GetCheckBoxSize: Integer;
begin
  Result := 0;
  IF (FColStyle = colsCheck) THEN CASE FCheckBoxSize OF {Breite der Box}
    pcbsSmall:   Result := 11;
    pcbsNormal:  Result := 13;
    pcbsLarge:   Result := 15;
    pcbsLargest: Result := 17;
    ELSE Result := 13;
  END;
end;

function TDGColumn.GetColor: TColor;
begin
  if cvColor in FAssignedValues then
    Result := FColor
  else
    Result := DefaultColor;
end;

function TDGColumn.GetFont: TFont;
var
  Save: TNotifyEvent;
begin
  if not (cvFont in FAssignedValues) and (FFont.Handle <> DefaultFont.Handle) then
  begin
    Save := FFont.OnChange;
    FFont.OnChange := nil;
    FFont.Assign(DefaultFont);
    FFont.OnChange := Save;
  end;
  Result := FFont;
end;

function TDGColumn.GetGrid: TPieCustomDataGrid;
begin
  if Assigned(Collection) and (Collection is TPieDataGridColumns) then
    Result := TPieDataGridColumns(Collection).Grid
  else
    Result := nil;
end;

function TDGColumn.GetPickList: TStrings;
begin
  if FPickList = nil then
    FPickList := TStringList.Create;
  Result := FPickList;
end;

function TDGColumn.GetReadOnly: Boolean;
begin
  if cvReadOnly in FAssignedValues then
    Result := FReadOnly
  else
    Result := DefaultReadOnly;
end;

function TDGColumn.GetDisplayMask: String;
begin
    Result := FDisplayMask;
end;

function TDGColumn.IsAlignmentStored: Boolean;
begin
  Result := (cvAlignment in FAssignedValues) and (FAlignment <> DefaultAlignment);
end;

function TDGColumn.IsColorStored: Boolean;
begin
  Result := (cvColor in FAssignedValues) and (FColor <> DefaultColor);
end;

function TDGColumn.IsFontStored: Boolean;
begin
  Result := (cvFont in FAssignedValues);
end;

function TDGColumn.IsReadOnlyStored: Boolean;
begin
  Result := (cvReadOnly in FAssignedValues) and (FReadOnly <> DefaultReadOnly);
end;

function TDGColumn.IsDisplayMaskStored: Boolean;
begin
  Result := true;
end;

procedure TDGColumn.RefreshDefaultFont;
var
  Save: TNotifyEvent;
begin
  if cvFont in FAssignedValues then Exit;
  Save := FFont.OnChange;
  FFont.OnChange := nil;
  try
    FFont.Assign(DefaultFont);
  finally
    FFont.OnChange := Save;
  end;
end;

procedure TDGColumn.RestoreDefaults;
var
  FontAssigned: Boolean;
begin
  FontAssigned := cvFont in FAssignedValues;
  FTitle.RestoreDefaults;
  FAssignedValues := [];
  RefreshDefaultFont;
  FPickList.Free;
  FPickList := nil;
  ColStyle := colsEdit;
  Changed(FontAssigned);
end;

procedure TDGColumn.SetAlignment(Value: TAlignment);
begin
  if (cvAlignment in FAssignedValues) and (Value = FAlignment) then Exit;
  FAlignment := Value;
  Include(FAssignedValues, cvAlignment);
  Changed(False);
end;

procedure TDGColumn.SetCheckBoxSize(Value: TPieCheckBoxSize);
BEGIN
  IF Value <> FCheckBoxSize THEN BEGIN
    FCheckBoxSize := Value;
    IF FColStyle = colsCheck THEN Grid.InvalidateCol(FInternalCol);
  END;
END;

procedure TDGColumn.SetCheckBoxStyle(Value: TPieCheckBoxStyle);
BEGIN
  IF Value <> FCheckBoxStyle THEN BEGIN
    FCheckBoxStyle := Value;
    IF FColStyle = colsCheck THEN Grid.InvalidateCol(FInternalCol);
  END;
END;

procedure TDGColumn.SetHookStyle(Value: TPieHookStyle);
BEGIN
  IF Value <> FHookStyle THEN BEGIN
    FHookStyle := Value;
    IF FColStyle = colsCheck THEN Grid.InvalidateCol(FInternalCol);
  END;
END;

procedure TDGColumn.SetHookColor(Value: TColor);
BEGIN
  IF Value <> FHookColor THEN BEGIN
    FHookColor := Value;
    IF FColStyle = colsCheck THEN Grid.InvalidateCol(FInternalCol);
  END;
END;

procedure TDGColumn.SetColStyle(Value: TColStyle);
begin
  if Value = FColStyle then Exit;
  FColStyle := Value;
  Changed(False);
end;

procedure TDGColumn.SetFormat(Value: TColumnFormat);
begin
  if Value = FFormat then Exit;
  FFormat := Value;
  FDisplayMask := '';
  Changed(False);
end;

procedure TDGColumn.SetColor(Value: TColor);
begin
  if (cvColor in FAssignedValues) and (Value = FColor) then Exit;
  FColor := Value;
  Include(FAssignedValues, cvColor);
  Changed(False);
end;

procedure TDGColumn.SetEditButtonEnabled(Value: Boolean);
begin
  if Value = FEditButtonEnabled then Exit;
  FEditButtonEnabled := Value;
  Changed(False);
end;

procedure TDGColumn.SetFont(Value: TFont);
begin
  FFont.Assign(Value);
  Include(FAssignedValues, cvFont);
  Changed(False);
end;

procedure TDGColumn.SetPickList(Value: TStrings);
begin
  if Value = nil then
  begin
    FPickList.Free;
    FPickList := nil;
    Exit;
  end;
  PickList.Assign(Value);
end;

procedure TDGColumn.SetPopupMenu(Value: TPopupMenu);
begin
  FPopupMenu := Value;
  if Value <> nil then Value.FreeNotification(GetGrid);
end;

procedure TDGColumn.SetReadOnly(Value: Boolean);
begin
  if (cvReadOnly in FAssignedValues) and (Value = FReadOnly) then Exit;
  FReadOnly := Value;
  Include(FAssignedValues, cvReadOnly);
  Changed(False);
  IF assigned(GetGrid.InplaceEditor) THEN (GetGrid.InplaceEditor as TPieDataGridInplaceEdit).SetReadOnly;
end;

procedure TDGColumn.SetTitle(Value: TColumnTitle);
begin
  FTitle.Assign(Value);
end;

function TDGColumn.GetMaxLength: Integer;
begin
  Result := FMaxLength;
end;

procedure TDGColumn.SetMaxLength(Value: Integer);
begin
  FMaxLength := Value;
end;

procedure TDGColumn.SetDisplayMask(Value: String);
begin
  FDisplayMask := value;
  Changed(False);
end;

{ TPieDataGridColumns }

constructor TPieDataGridColumns.Create(Grid: TPieCustomDataGrid; ColumnClass: TColumnClass);
begin
  inherited Create(ColumnClass);
  FGrid := Grid;
end;

function TPieDataGridColumns.Add: TDGColumn;
begin
  Result := TDGColumn(inherited Add);
end;

function TPieDataGridColumns.GetColumn(Index: Integer): TDGColumn;
begin
  Result := TDGColumn(inherited Items[Index]);
end;

function TPieDataGridColumns.GetOwner: TPersistent;
begin
  Result := FGrid;
end;

procedure TPieDataGridColumns.LoadFromFile(const Filename: string);
var
  S: TFileStream;
begin
  S := TFileStream.Create(Filename, fmOpenRead);
  try
    LoadFromStream(S);
  finally
    S.Free;
  end;
end;

type
  TColumnsWrapper = class(TComponent)
  private
    FColumns: TPieDataGridColumns;
  published
    property Columns: TPieDataGridColumns read FColumns write FColumns;
  end;

procedure TPieDataGridColumns.LoadFromStream(S: TStream);
var
  Wrapper: TColumnsWrapper;
begin
  Wrapper := TColumnsWrapper.Create(nil);
  try
    Wrapper.Columns := FGrid.CreateColumns;
    S.ReadComponent(Wrapper);
    Assign(Wrapper.Columns);
  finally
    Wrapper.Columns.Free;
    Wrapper.Free;
  end;
end;

procedure TPieDataGridColumns.RestoreDefaults;
var
  I: Integer;
begin
  BeginUpdate;
  try
    for I := 0 to Count-1 do
      Items[I].RestoreDefaults;
  finally
    EndUpdate;
  end;
end;

procedure TPieDataGridColumns.SaveToFile(const Filename: string);
var
  S: TStream;
begin
  S := TFileStream.Create(Filename, fmCreate);
  try
    SaveToStream(S);
  finally
    S.Free;
  end;
end;

procedure TPieDataGridColumns.SaveToStream(S: TStream);
var
  Wrapper: TColumnsWrapper;
begin
  Wrapper := TColumnsWrapper.Create(nil);
  try
    Wrapper.Columns := Self;
    S.WriteComponent(Wrapper);
  finally
    Wrapper.Free;
  end;
end;

procedure TPieDataGridColumns.SetColumn(Index: Integer; Value: TDGColumn);
begin
  Items[Index].Assign(Value);
end;

procedure TPieDataGridColumns.Update(Item: TCollectionItem);
VAR
  IPE: TPieDataGridInplaceEdit;
begin
     if (FGrid = nil) or (csLoading in FGrid.ComponentState) then Exit;

     if (csDesigning in FGrid.ComponentState) then FGrid.invalidate
     else if Assigned(Item) then begin
       FGrid.invalidatecol(Item.Index);
       IPE := FGrid.InplaceEditor as TPieDataGridInplaceEdit;
       if Assigned(IPE) then IPE.UpdateContents;
     end;
end;

{ DataGridInplaceEdit }

type
  TSelection = record
    StartPos, EndPos: Integer;
  end;

procedure TPieDataGridInplaceEdit.ToggleChecked;
begin
  IF FColumn.ReadOnly THEN exit;
  IF Text = '1' THEN Text := '0' ELSE Text := '1';
  TPieCustomDataGrid(Grid).Cells[TPieCustomDataGrid(Grid).Col, TPieCustomDataGrid(Grid).Row] := Text;
  Modified := TRUE;
  Invalidate;
end;

procedure TPieDataGridInplaceEdit.KeyDown(var Key: Word; Shift: TShiftState);
VAR
  Navigieren: Boolean;
begin
  {------- Button gedrckt? -----------------------------------}
  if (FColStyle IN [colsButton, colsComboButton]) and (Key = VK_RETURN) and (Shift = [ssCtrl]) then begin
    IF FColumn.EditButtonEnabled THEN TPieCustomDataGrid(Grid).EditButtonClick;
    {irgendwie kommt das CRLF in die Textzeile}
    IF copy(Text, Length(Text)-1, 2) = #$D#$A THEN Text := copy(Text, 1, Length(Text)-2);
    KillMessage(Handle, WM_CHAR);
  end;
  {------- Checkbox getoggelt? --------------------------------}
  IF (FColStyle = colsCheck) AND
       ((Key = VK_RETURN) AND (Shift = [ssCtrl]) OR
       (Key = VK_SPACE)) THEN ToggleChecked;
  {------ jetzt noch Navigationstasten untersttzen -----------}
  Navigieren := key IN [VK_UP, VK_DOWN, VK_LEFT, VK_RIGHT, VK_PRIOR, VK_NEXT, VK_TAB, VK_HOME, VK_END];
  IF Navigieren THEN validateContent;
  {------ Insert/Append/Delete-Untersttzung des Grids --------}
  with TPieCustomDataGrid(Grid) do if CheckDataGridkey(key, shift) then begin
    keydown(key, Shift);
    key := 0;
  end;

  {------ alle anderen Tastendrcke durchlassen ---------------}
  inherited KeyDown(Key, Shift);
end;

procedure TPieDataGridInplaceEdit.KeyPress(var Key: Char);
var
  Selection: TSelection;
  I: Integer;
begin
  {------ Wenn ReadOnly, Return sperren -----------------------}
  IF (Key = #13) AND FColumn.ReadOnly THEN Key := #0;

  TPieCustomDataGrid(Grid).KeyPress(Key);
  if (Key in [#32..#255]) and not TPieCustomDataGrid(Grid).CanEditAcceptKey(Key) then
  begin
    Key := #0;
    MessageBeep(0);
  end;
  case Key of
    #9, #27: Key := #0;
    #13:
      begin
        SendMessage(Handle, EM_GETSEL, Longint(@Selection.StartPos), Longint(@Selection.EndPos));
        if (Selection.StartPos = 0) and (Selection.EndPos = GetTextLen) then
          Deselect else
          SelectAll;
        Key := #0;
      end;
    ^H, ^V, ^X, #32..#255:
      if not TPieCustomDataGrid(Grid).CanEditModify then Key := #0;
  end;
  if (Key <> #0) then
  begin
       if ((FColStyle = colsCombo) OR (FColStyle = colsComboButton)) and FColumn.Limittolist then
       begin
       //check if picklist was visible...(items are assigned in dropdown proc)
            if not FListVisible and Assigned(FActiveList) then
            with TPieCustomDataGrid(Grid) do FPickList.items := Columns[SelectedIndex].Picklist;

            for I := 0 to FPicklist.items.count-1 do if uppercase(copy(FPickList.items[i],1,1)) = uppercase(Key) then
            begin
                 Text := FPickList.items[i];
                 with TPieCustomDataGrid(Grid) do SetEditText(col, row, Text);
                 modified := true;
                 Key := #0;
                 break;
            end;
       end;
       inherited KeyPress(Key);
  end;
end;

procedure TPieDataGridInplaceEdit.ListMouseUp(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
begin
  if Button = mbLeft then
    CloseUp(PtInRect(FActiveList.ClientRect, Point(X, Y)));
end;

procedure TPieDataGridInplaceEdit.MouseDown(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
VAR
  BoxRect: TRect;
begin
  if (Button = mbLeft) and (FColStyle <> colsEdit) THEN BEGIN
    {------- Button geklickt? -----------------------}
    IF FColStyle IN [colsButton, colsComboButton] THEN BEGIN
      IF PtInRect(Rect(Width - GetButtonWidth(colsButton), 0, Width, Height), Point(X,Y)) OR
        (ssDouble in Shift) THEN BEGIN
        FTracking := True;
        TrackButton(X, Y);
      END;
    END;
    {------- CheckBox geklickt? ---------------------}
    IF FColStyle = colsCheck THEN BEGIN
      GetCheckBoxWidth(BoxRect);
      IF PtInRect(BoxRect, Point(X, Y)) OR (ssDouble in Shift) THEN ToggleChecked;
      Invalidate;
    END;
    {------ ComboBox geklickt? ---------------------}
    IF FColStyle = colsCombo THEN BEGIN
      IF PtInRect(Rect(Width - GetButtonWidth(colsCombo), 0, Width, Height), Point(X,Y)) or
      (ssDouble in Shift) and ReadOnly THEN BEGIN
        if FListVisible then CloseUp(False)
        else begin
          MouseCapture := True;
          FTracking := True;
          TrackButton(X, Y);
          if Assigned(FActiveList) then DropDown;
        end;
      END;
    END;
    {------ ComboBox geklickt? ---------------------}
    IF FColStyle = colsComboButton THEN BEGIN
      IF PtInRect(Rect(Width - GetButtonWidth(colsComboButton), 0, Width - GetButtonWidth(colsButton), Height), Point(X,Y)) or
      (ssDouble in Shift) and ReadOnly THEN BEGIN
        if FListVisible then CloseUp(False)
        else begin
          MouseCapture := True;
          FTracking := True;
          TrackButton(X, Y);
          if Assigned(FActiveList) then DropDown;
        end;
      END;
    END;
  end;
  inherited MouseDown(Button, Shift, X, Y);
end;

procedure TPieDataGridInplaceEdit.MouseMove(Shift: TShiftState; X, Y: Integer);
var
  ListPos: TPoint;
  MousePos: TSmallPoint;
begin
  if FTracking then
  begin
    TrackButton(X, Y);
    if FListVisible then
    begin
      ListPos := FActiveList.ScreenToClient(ClientToScreen(Point(X, Y)));
      if PtInRect(FActiveList.ClientRect, ListPos) then
      begin
        StopTracking;
        MousePos := PointToSmallPoint(ListPos);
        SendMessage(FActiveList.Handle, WM_LBUTTONDOWN, 0, Integer(MousePos));
        Exit;
      end;
    end;
  end;
  inherited MouseMove(Shift, X, Y);
end;

procedure TPieDataGridInplaceEdit.MouseUp(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
var
  WasPressed: Boolean;
begin
  WasPressed := FPressedButton;
  StopTracking;
  if (Button = mbLeft) and (FColStyle IN [colsButton, colsComboButton]) and WasPressed then
    IF FColumn.EditButtonEnabled THEN TPieCustomDataGrid(Grid).EditButtonClick;
  inherited MouseUp(Button, Shift, X, Y);
end;

function TPieDataGridInplaceEdit.GetCheckBoxWidth(VAR CBRect: TRect): Integer;
VAR
  B: Integer;
  CR: TRect;
begin
  B := FColumn.GetCheckBoxSize;
  CR := GetClientRect;
  SetRect(CBRect,
         (CR.Right - CR.Left - B) DIV 2,
          CR.Top + (CR.Bottom - CR.Top - B) DIV 2,
         (CR.Right - CR.Left + B) DIV 2,
          CR.Top + (CR.Bottom - CR.Top + B) DIV 2);
  Result := B;
end;

procedure TPieDataGridInplaceEdit.PaintWindow(DC: HDC);
var  {TInplaceEdit}
  R, CR, BoxRect: TRect;
  B, I: Integer;
  Flags: Integer;
  W: Integer;
  OldColor: TColor;
  MyPen: HPen;
  MyBrush: HBrush;
  ColorRef: Integer;
  P: TPoint;
begin
  {--- Button ganz rechts malen ----------------------------------------}
  IF FColStyle IN [colsButton, colsCombo, colsComboButton] THEN BEGIN
    SetRect(R, Width - GetButtonWidth(colsButton), 0, Width, Height);
    Flags := 0;
    {--- ComboBox ----------------------------------------}
    if FColStyle in [colsCombo] then begin
      if FActiveList = nil then
        Flags := DFCS_INACTIVE
      else if FPressedCombo then
        Flags := DFCS_FLAT or DFCS_PUSHED;
      DrawFrameControl(DC, R, DFC_SCROLL, Flags or DFCS_SCROLLCOMBOBOX);
    end;
    {--- Button ----------------------------------------}
    IF FColStyle in [colsButton, colsComboButton] THEN BEGIN
      if FPressedButton then Flags := BF_FLAT;
      IF NOT FColumn.EditButtonEnabled THEN Flags := Flags OR DFCS_INACTIVE;
      DrawFrameControl(DC, R, DFC_BUTTON, Flags or DFCS_BUTTONPUSH);
      Flags := ((R.Right - R.Left) shr 1) - 1 + Ord(FPressedButton);

      W := Height shr 3;
      if W = 0 then W := 1;

      ColorRef := clBlack;
      IF NOT FColumn.EditButtonEnabled THEN ColorRef := clWhite;
      MyBrush := CreateSolidBrush(ColorRef);
      MyBrush := SelectObject(DC, MyBrush);
      PatBlt(DC, R.Left + Flags, R.Top + Flags + 3, W, W, PATCOPY);
      PatBlt(DC, R.Left + Flags - (W * 2), R.Top + Flags + 3, W, W, PATCOPY);
      PatBlt(DC, R.Left + Flags + (W * 2), R.Top + Flags + 3, W, W, PATCOPY);
      DeleteObject(SelectObject(DC, MyBrush));
      IF NOT FColumn.EditButtonEnabled THEN BEGIN
        MyBrush := CreateSolidBrush(clGray);
        MyBrush := SelectObject(DC, MyBrush);
        PatBlt(DC, R.Left + Flags - 1, R.Top + Flags + 3 - 1, W, W, PATCOPY);
        PatBlt(DC, R.Left + Flags - (W * 2) - 1, R.Top + Flags + 3 - 1, W, W, PATCOPY);
        PatBlt(DC, R.Left + Flags + (W * 2) - 1, R.Top + Flags + 3 - 1, W, W, PATCOPY);
        DeleteObject(SelectObject(DC, MyBrush));
      END;
    end;
    ExcludeClipRect(DC, R.Left, R.Top, R.Right, R.Bottom);
  END;
  {--- 2. Button von rechts malen ----------------------------------------}
  IF FColStyle IN [colsComboButton] THEN BEGIN
    SetRect(R, Width - GetButtonWidth(colsComboButton), 0, Width - GetButtonWidth(colsButton), Height);
    Flags := 0;
    if FActiveList = nil then Flags := DFCS_INACTIVE
                         else if FPressedCombo then Flags := DFCS_FLAT or DFCS_PUSHED;
    DrawFrameControl(DC, R, DFC_SCROLL, Flags or DFCS_SCROLLCOMBOBOX);
    ExcludeClipRect(DC, R.Left, R.Top, R.Right, R.Bottom);
  END;
  {--- normales restliches Edit-Feld malen -------------------------------}
  IF NOT(FColStyle IN [colsCheck]) THEN BEGIN
    inherited PaintWindow(DC);
    IF FColumn.ReadOnly THEN Deselect;
  END;
  {--- Checkbox ----------------------------------------}
  IF FColStyle = colsCheck THEN BEGIN

    OldColor := Brush.Color;
    {*********** Kstchen malen ************}
    CR := GetClientRect;
    B := GetCheckBoxWidth(BoxRect);
    IF Enabled AND NOT FColumn.ReadOnly THEN Brush.Color := clWindow ELSE Brush.Color := clInActiveBorder;
    FillRect(DC, BoxRect, Brush.Handle);
    CASE FColumn.FCheckBoxStyle OF
    pcbsSunken: DrawEdge(DC, BoxRect, EDGE_SUNKEN, BF_RECT);
    pcbsRaised: DrawEdge(DC, BoxRect, EDGE_RAISED, BF_RECT);
    pcbsBump: DrawEdge(DC, BoxRect, EDGE_BUMP, BF_RECT);
    pcbsEtched: DrawEdge(DC, BoxRect, EDGE_ETCHED, BF_RECT);
    END;
    {*********** Hkchen malen ************}
    IF Text <> '0' THEN BEGIN
      IF Enabled AND NOT FColumn.ReadOnly
        THEN MyPen := CreatePen(PS_SOLID, 1, FColumn.FHookColor)
        ELSE MyPen := CreatePen(PS_SOLID, 1, clGray);
      MyPen := SelectObject(DC, MyPen);
      Brush.Color := FColumn.FHookColor; {fr phsFill}
      CASE FColumn.FHookStyle OF
      phsHook:      FOR I:=0 TO 2 DO BEGIN
                    MoveToEx(DC, BoxRect.Left + 3,       BoxRect.Top - I + 7, @P);
                    LineTo(DC, BoxRect.Left + B DIV 2-1, BoxRect.Top - I + (B DIV 2) + 3);
                    LineTo(DC, BoxRect.Right - 3,        BoxRect.Top - I + 4);
                    END;
      phsCross:     FOR I:=0 TO 0 DO BEGIN
                    MoveToEx(DC, BoxRect.Left + 3,   BoxRect.Top - I + 3, @P);
                    LineTo(DC, BoxRect.Left + B - 3, BoxRect.Top - I + B - 3);
                    MoveToEx(DC, BoxRect.Left + 3,   BoxRect.Top - I + B - 4, @P);
                    LineTo(DC, BoxRect.Left + B - 3, BoxRect.Top - I + 2);
                    END;
      phsFill:      FillRect(DC, Rect(BoxRect.Left + 2, BoxRect.Top + 2, BoxRect.Right - 2, BoxRect.Bottom - 2), Brush.Handle);
      END; {CASE FHookStyle ...}

      DeleteObject(SelectObject(DC, MyPen));
    END;
    Brush.Color := OldColor;

    ExcludeClipRect(DC, 0, 0, Width, Height);
  END;
end;

procedure TPieDataGridInplaceEdit.SetColStyle(Value: TColStyle);
begin
  FColStyle := Value;
  case Value of
    colsCombo, colsComboButton:
      begin
        if FPickList = nil then begin
          FPickList := TPopupListbox.Create(Self);
          FPickList.Visible := False;
          FPickList.Parent := Self;
          FPickList.OnMouseUp := ListMouseUp;
          FPickList.IntegralHeight := True;
          FPickList.ItemHeight := 11;
        end;
        FActiveList := FPickList;
      end;
  else  { colsEdit, colsCheck, colsButton }
    FActiveList := nil;
  end;
  SetReadOnly;
  Repaint;
end;

procedure TPieDataGridInplaceEdit.SetReadOnly;
begin
  with TPieCustomDataGrid(Grid) do if Columns.count > SelectedIndex then begin
     Self.ReadOnly := FColumn.ReadOnly OR
     (((FColStyle = colsCombo) OR (FColStyle = colsComboButton)) AND FColumn.Limittolist) OR
     (FColStyle = colsCheck);
  end else Self.ReadOnly := false;
end;

procedure TPieDataGridInplaceEdit.StopTracking;
begin
  if FTracking then begin
    TrackButton(-1, -1);
    FTracking := False;
    MouseCapture := False;
  end;
end;

procedure TPieDataGridInplaceEdit.TrackButton(X,Y: Integer);
var
  NewState: Boolean;
  R: TRect;
begin
  IF FColStyle IN [colsButton, colsComboButton] THEN BEGIN
    SetRect(R, ClientWidth - GetButtonWidth(colsButton), 0, ClientWidth, ClientHeight);
    NewState := PtInRect(R, Point(X, Y));
    if FPressedButton <> NewState then begin
      FPressedButton := NewState;
      InvalidateRect(Handle, @R, False);
    end;
  END;
  IF FColStyle IN [colsCombo, colsComboButton] THEN BEGIN
    CASE FColStyle OF
    colsCombo:       SetRect(R, ClientWidth - GetButtonWidth(colsCombo), 0, ClientWidth, ClientHeight);
    colsComboButton: SetRect(R, ClientWidth - GetButtonWidth(colsComboButton), 0, ClientWidth - GetButtonWidth(colsButton), ClientHeight);
    END;
    NewState := PtInRect(R, Point(X, Y));
    if FPressedCombo <> NewState then begin
      FPressedCombo := NewState;
      InvalidateRect(Handle, @R, False);
    end;
  END;
end;

procedure TPieDataGridInplaceEdit.ValidateContent;
var
   value: string;
   InListe: Boolean;
begin
  if not modified then exit;
  with TPieCustomDataGrid(Grid) do if columns.count > SelectedIndex then
  try
    value := Trim(cells[col, row]);
    if (value <> '') then begin
      InListe := Assigned(FPickList) AND (FPickList.Items.IndexOf(value) >= 0);
      Case Columns[SelectedIndex].Format of
      cfString:  value := cells[col, row];
      cfNumber,
      cfInteger: IF (value <> '') AND (value <> '-') AND (value <> '+') AND NOT(InListe)
                 THEN value := floattostr(strtofloat(value));
      cfDate:    begin
                 if pos(uppercase(copy(value,1,1)), 'NTHJD') > 0 then value := datetostr(now)
                   else value := datetostr(strtodate(value));
                 end;
      end;
      cells[col, row] := value;
      TPieCustomDataGrid(Grid).DataChanged(col, row, value);
    end;
  except
    beep;
    SetEditText(col, row, FLastText);
    cells[col, row] := FLastText;
    Modified := false;
  end;
  with TPieCustomDataGrid(Grid) do if Assigned(FOnChange) then FOnChange(self);
end;

procedure TPieDataGridInplaceEdit.UpdateContents;
VAR
  NewStyle: TColStyle;
begin
  with TPieCustomDataGrid(Grid) do if columns.count > SelectedIndex then begin
    FColumn := Columns[SelectedIndex];
    NewStyle := FColumn.ColStyle;
    IF FColumn.ReadOnly AND ((NewStyle = colsCombo) OR (NewStyle = colsComboButton)) THEN NewStyle := colsEdit;
    SetColStyle(NewStyle);
  end;
  inherited UpdateContents;
  with TPieCustomDataGrid(Grid) do if columns.count > SelectedIndex then MaxLength := columns[SelectedIndex].MaxLength;
  FLastText := EditText;
end;

procedure TPieDataGridInplaceEdit.CMCancelMode(var Message: TCMCancelMode);
begin
  if (Message.Sender <> Self) and (Message.Sender <> FActiveList) then
    CloseUp(False);
end;

procedure TPieDataGridInplaceEdit.WMCancelMode(var Message: TMessage);
begin
  StopTracking;
  inherited;
end;

procedure TPieDataGridInplaceEdit.WMKillFocus(var Message: TMessage);
begin
  inherited;
  CloseUp(False);
  ValidateContent;
end;

procedure TPieDataGridInplaceEdit.WMLButtonDblClk(var Message: TWMLButtonDblClk);
begin
  with Message do begin
    CASE FColStyle OF
    colsButton,
    colsCombo,
    colsComboButton: IF PtInRect(Rect(Width - GetButtonWidth(FColStyle), 0, Width, Height), Point(XPos, YPos)) then Exit;
    END;
  end;
  inherited;
end;

procedure TPieDataGridInplaceEdit.WMPaint(var Message: TWMPaint);
begin
  PaintHandler(Message);
end;

procedure TPieDataGridInplaceEdit.WMSetCursor(var Message: TWMSetCursor);
var
  P: TPoint;
begin
  GetCursorPos(P);
  CASE FColStyle OF
  colsButton,
  colsCombo,
  colsComboButton: IF PtInRect(Rect(Width - GetButtonWidth(FColStyle), 0, Width, Height), ScreenToClient(P))
                   then Windows.SetCursor(LoadCursor(0, idc_Arrow)) ELSE inherited;
  colsCheck: Windows.SetCursor(LoadCursor(0, idc_Arrow));
  ELSE inherited;
  END;
end;

procedure TPieDataGridInplaceEdit.WndProc(var Message: TMessage);
begin
  case Message.Msg of
  wm_KeyDown, wm_SysKeyDown, wm_Char:
    if FColStyle in [colsCombo, colsComboButton] then with TWMKey(Message) do begin
      DoDropDownKeys(CharCode, KeyDataToShiftState(KeyData));
      if (CharCode <> 0) and FListVisible then begin
        SendMessage(FActiveList.Handle, Message.Msg, Message.WParam, Message.LParam);
        Exit;
      end;
    end;
  end;
  inherited;
end;

procedure TPopupListBox.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  with Params do
  begin
    Style := Style or WS_BORDER;
    ExStyle := WS_EX_TOOLWINDOW or WS_EX_TOPMOST;
    WindowClass.Style := CS_SAVEBITS;
  end;
end;

procedure TPopupListbox.CreateWnd;
begin
  inherited CreateWnd;
  Windows.SetParent(Handle, 0);
  CallWindowProc(DefWndProc, Handle, wm_SetFocus, 0, 0);
end;

procedure TPopupListbox.Keypress(var Key: Char);
var
  TickCount: Integer;
begin
  case Key of
    #8, #27: FSearchText := '';
    #32..#255:
      begin
        TickCount := GetTickCount;
        if TickCount - FSearchTickCount > 2000 then FSearchText := '';
        FSearchTickCount := TickCount;
        if Length(FSearchText) < 32 then FSearchText := FSearchText + Key;
        SendMessage(Handle, LB_SelectString, WORD(-1), Longint(PChar(FSearchText)));
        Key := #0;
      end;
  end;
  inherited Keypress(Key);
end;

procedure TPopupListbox.MouseUp(Button: TMouseButton; Shift: TShiftState;
  X, Y: Integer);
begin
  inherited MouseUp(Button, Shift, X, Y);
  TPieDataGridInPlaceEdit(Owner).CloseUp((X >= 0) and (Y >= 0) and
      (X < Width) and (Y < Height));
end;

constructor TPieDataGridInplaceEdit.Create(Owner: TComponent);
begin
  inherited Create(Owner);
  FColumn := NIL;
  FColStyle := colsEdit;
end;

destructor TPieDataGridInplaceEdit.Destroy();
begin
  inherited Destroy;
end;

function TPieDataGridInplaceEdit.GetButtonWidth(AColStyle: TColStyle): Integer;
begin
  case AColStyle of
  colsCombo:  result := GetSystemMetrics(SM_CXVSCROLL);
  colsButton: result := GetSystemMetrics(SM_CXSIZE);
  colsComboButton:  result := GetSystemMetrics(SM_CXVSCROLL) + GetSystemMetrics(SM_CXSIZE);
  else result := 0;
  end;
end;

procedure TPieDataGridInplaceEdit.BoundsChanged;
var
  R: TRect;
begin
  SetRect(R, 2, 2, Width - 2, Height);
  CASE FColStyle OF
  colsButton,
  colsCombo,
  colsComboButton: Dec(R.Right, GetButtonWidth(FColStyle));
  colsCheck: SetRect(R, 0, 0, 0, 0);
  END;
  SendMessage(Handle, EM_SETRECTNP, 0, LongInt(@R));
  SendMessage(Handle, EM_SCROLLCARET, 0, 0);
end;

procedure TPieDataGridInplaceEdit.CloseUp(Accept: Boolean);
var
  ListValue: Variant;
begin
  if FListVisible then
  begin
    if GetCapture <> 0 then SendMessage(GetCapture, WM_CANCELMODE, 0, 0);

    if FPickList.ItemIndex <> -1 then ListValue := FPickList.Items[FPicklist.ItemIndex];

    SetWindowPos(FActiveList.Handle, 0, 0, 0, 0, 0, SWP_NOZORDER or
      SWP_NOMOVE or SWP_NOSIZE or SWP_NOACTIVATE or SWP_HIDEWINDOW);
    FListVisible := False;
    Invalidate;
    if Accept then if (not VarIsNull(ListValue)) then
    begin
         if (EditCanModify or (not EditCanModify and not Self.Readonly)) then
         begin
//              Text := ListValue;
              with TPieCustomDataGrid(Grid) do cells[col, row] := ListValue;
              modified := true;
         end;
    end;
  end;
end;

procedure TPieDataGridInplaceEdit.DoDropDownKeys(var Key: Word; Shift: TShiftState);
begin
  case Key of
    VK_UP, VK_DOWN:
      if ssAlt in Shift then
      begin
        if FListVisible then CloseUp(True) else DropDown;
        Key := 0;
      end;
    VK_RETURN, VK_ESCAPE:
      if FListVisible and not (ssAlt in Shift) then
      begin
        CloseUp(Key = VK_RETURN);
        Key := 0;
      end;
  end;
end;

procedure TPieDataGridInplaceEdit.DropDown;
var
  P: TPoint;
  I,J,Y: Integer;
  Column: TDGColumn;
begin
  if not FListVisible and Assigned(FActiveList) then
  begin
    FActiveList.Width := Width;
    with TPieCustomDataGrid(Grid) do Column := Columns[SelectedIndex];

      FPickList.Color := Color;
      FPickList.Font := Font;
      FPickList.Items := Column.Picklist;
      if FPickList.Items.Count >= Integer(Column.DropDownRows) then
        FPickList.Height := Integer(Column.DropDownRows) * FPickList.ItemHeight + 4
      else
        FPickList.Height := FPickList.Items.Count * FPickList.ItemHeight + 4;
//setpicklist...
      FPickList.itemindex := FPickList.items.indexof(Text);
      if FPickList.itemindex = -1 then FPickList.itemindex := 0;

      J := FPickList.ClientWidth;
      for I := 0 to FPickList.Items.Count - 1 do
      begin
        Y := FPickList.Canvas.TextWidth(FPickList.Items[I]);
        if Y > J then J := Y;
      end;
      FPickList.ClientWidth := J;

    P := Parent.ClientToScreen(Point(Left, Top));
    Y := P.Y + Height;
    if Y + FActiveList.Height > Screen.Height then Y := P.Y - FActiveList.Height;
    SetWindowPos(FActiveList.Handle, HWND_TOP, P.X, Y, 0, 0,
      SWP_NOSIZE or SWP_NOACTIVATE or SWP_SHOWWINDOW);
    FListVisible := True;

    Invalidate;
    Windows.SetFocus(Handle);
  end;
end;

{******************************************************************}
{*************** TPieCustomDataGrid *************************************}
{******************************************************************}
constructor TPieCustomDataGrid.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  inherited DefaultRowHeight := 20;
  FBackground := TBitmap.create;
  FLayoutFlag := 2;
  options := [goFixedVertLine, goFixedHorzLine, goVertLine, goHorzLine, goEditing, goColSizing, goAlwaysShowEditor];
  FColumns := CreateColumns;
  HideEditor;
  FLayoutFlag := 0;
  FEditorColor := clWindow;
  FSelectedCellColor := clWindow;
  FSelectedCellFontColor := clBlue;
  FSorted := FALSE;
  FSortColumn := 0;
  FSortUp := TRUE;
  FColMerker := -1;
  FRowMerker := -1;
  sizechanged(ColCount, RowCount);
end;

destructor TPieCustomDataGrid.Destroy;
begin
  FLayoutFlag := 2;
  FColumns.Free;
  FColumns := nil;
  FBackground.free;
  inherited Destroy;
end;

PROCEDURE TPieCustomDataGrid.Loaded;
BEGIN
  inherited Loaded;
  {Auch bei Default-Werten sortieren!}
  IF FSorted AND FSortUp AND (FSortColumn = 0) THEN SortGrid;
END;

procedure TPieCustomDataGrid.DataChanged(Col, Row: Integer; Value: string);
begin
end;

procedure TPieCustomDataGrid.SizeChanged(OldColCount, OldRowCount: Longint);
begin
     if not (csLoading in ComponentState) and (FLayoutFlag = 0) then
     begin
          inc(FLayoutFlag);
          while Columns.count > ColCount do Columns[ColCount].destroy;
          while Columns.count < ColCount do Columns.add;
          dec(FLayoutFlag);
     end;
end;

procedure TPieCustomDataGrid.SetColumnCount(NewCount: LongInt);
begin
     if (FLayoutFlag > 0) or (csLoading in ComponentState) then exit;
     inc(FLayoutFlag);
     ColCount := NewCount;
     dec(FLayoutFlag);
end;

procedure TPieCustomDataGrid.SetRowCountMin(Value: LongInt);
begin
  if value > -1 then FRowCountMin := value else FRowCountMin := 0;
  if (FRowCountMin > 0) and (RowCount < FRowCountMin) then RowCount := FRowCountMin;
  RowCountMinChanged;
end;

procedure TPieCustomDataGrid.RowCountMinChanged;
begin
// TPieDBDataGrid overrides this procedure
end;

function TPieCustomDataGrid.CanEditAcceptKey(Key: Char): Boolean;
begin
  case Columns[SelectedIndex].Format of
  cfNumber:  Result := Key IN ['0'..'9', '-', '+', DecimalSeparator];
  cfInteger: Result := Key IN ['0'..'9', '-', '+'];
  else       Result := True;
  end;
end;

procedure TPieCustomDataGrid.DoEnter;
begin
  if (goediting in options) and (goAlwaysShowEditor in options) then ShowEditor;
  inherited DoEnter;
end;

procedure TPieCustomDataGrid.DoExit;
begin
  if (goediting in options) and (goAlwaysShowEditor in options) then HideEditor;
  inherited DoExit;
end;

procedure TPieCustomDataGrid.DrawCell(ACol, ARow: Longint; ARect: TRect; AState: TGridDrawState);
begin
  {---------- User-DrawCell-Routine? ----------------}
  IF NOT(DefaultDrawing) THEN BEGIN
    inherited DrawCell(ACol, ARow, ARect, AState);
    if Assigned(OnDrawCell) then OnDrawCell(self,ACol, ARow, ARect, AState);
    exit;
  END;

  IF ARow < FixedRows THEN DrawHeaderCell(ACol, ARow, ARect, AState)
                      ELSE DrawDataCell(ACol, ARow, ARect, AState);
end;

procedure TPieCustomDataGrid.DrawHeaderCell(ACol, ARow: Longint; ARect: TRect; AState: TGridDrawState);
var
  XPos, YPos, Breite{, Bitmapbreite, DisIndex}: Integer;
  R: TRect;
  Points: array[0..3] of TPoint;
{  Bitmap: TBitmap;}
  IndexCol: TDGColumn;
  Alignment: TAlignment;
begin
  {Indizierte Spalte ermitteln}
  if ACol < Columns.count then IndexCol := Columns[ACol] ELSE IndexCol := NIL;

  {normale Einstellungen}
  Canvas.Brush.Color := FixedColor;
  Canvas.Font := IndexCol.Title.Font;
  Alignment := IndexCol.Title.Alignment;
  XPos := ARect.Left + 2;
  YPos := ARect.Top + 2;
  R.Top := ARect.Top; R.Bottom := ARect.Bottom;

  {Besonderheiten, wenn gedrckt}
{  IF Pressed THEN BEGIN
    inc(XPos);
    inc(YPos);
  END;}

  {Besonderheiten, wenn Sortierspalte}
  IF FSorted AND (ACol = FSortColumn) THEN XPos := XPos + 14;

  {Besonderheiten, wenn Images}
(*  Bitmapbreite := 0;
  IF Assigned(FHeader.Images) AND (Section.ImageIndex >= 0) AND
                                  (Section.ImageIndex < FHeader.Images.Count) THEN BEGIN
    Bitmap := TBitmap.Create;
    TRY
      FHeader.Images.GetBitmap(Section.ImageIndex, Bitmap);
      Bitmap.Transparent := TRUE;
      IF NOT FHeader.Enabled THEN BEGIN
        DisIndex := DisableGlyph(Bitmap, 0);
        ImageList_DrawEx(FMaskedImageList.Handle, DisIndex, FHeader.Canvas.Handle, XPos, R.Top, 0, 0,
          ColorToRGB(clBtnFace), clNone, ILD_Normal);
      END
      ELSE FHeader.Canvas.StretchDraw(Classes.Rect(XPos, R.Top, XPos+Bitmap.Width, R.Bottom), Bitmap);
      Bitmapbreite := Bitmap.Width + 2;{2 Pixel Abszand zwischen Bitmap und Text}
    FINALLY
      Bitmap.Free;
    END;
    XPos := XPos + Bitmapbreite;
  END;*)

  {Text zeichnen}
  Breite := Canvas.TextWidth(IndexCol.Title.Caption);
{  IF NOT IndexCol.Enabled THEN Canvas.Font.Color := clGrayText;}
  IF (Breite < (ARect.Right-XPos-2)) THEN BEGIN
    R.Left := XPos;
    R.Right := ARect.Right-1;
    CASE Alignment OF
    taLeftJustify:  R.Left := XPos;
    taRightJustify: R.Left := ARect.Right - Breite - 2;
    taCenter:       R.Left := XPos + (ARect.Right - XPos - Breite) shr 1;
    END;
    Canvas.TextRect(R, R.Left, YPos, IndexCol.Title.Caption);
  END
  ELSE BEGIN
    R.Left := XPos;
    R.Right := ARect.Right-1-Canvas.TextWidth('...');
    IF (R.Right > R.Left) THEN Canvas.TextRect(R, R.Left, YPos, IndexCol.Title.Caption);
    R.Left := R.Right+2;
    R.Right := ARect.Right-1;
    IF (R.Left > ARect.Left)
      THEN Canvas.TextRect(R, R.Left, YPos, '...')
      {wenn nicht mal mehr Punkte hinpassen --> grau ausfllen}
      ELSE Canvas.FillRect(ARect);
  END;

  {Besonderheiten, wenn Sortierspalte}
  {blaues Dreieck malen}
  IF FSorted AND (ACol = FSortColumn) {AND (ARect.Right - ARect.Left > Bitmapbreite)} THEN BEGIN
    XPos := ARect.Left + 5;
    YPos := 6;

    {Besonderheiten, wenn gedrckt}
{    IF Pressed THEN BEGIN
      inc(XPos);
      inc(YPos);
    END;}

    IF FSortUp THEN BEGIN
      Points[0] := Point(XPos,YPos);
      Points[1] := Point(XPos+7,YPos);
      Points[2] := Point(XPos+3,YPos+7);
      Points[3] := Point(XPos,YPos);
    END
    ELSE BEGIN
      Points[0] := Point(XPos,YPos+7);
      Points[1] := Point(XPos+7,YPos+7);
      Points[2] := Point(XPos+3,YPos);
      Points[3] := Point(XPos,YPos+7);
    END;

    Canvas.Pen.Color := clBlack;
    Canvas.Polygon(Points);
    Canvas.Brush.Color := clBlue;
    Canvas.FloodFill(XPos+3, YPos+5, clBlack, fsBorder);
  END;

end;

procedure TPieCustomDataGrid.DrawDataCell(ACol, ARow: Longint; ARect: TRect; AState: TGridDrawState);
var
  XPos, YPos, Breite{, Bitmapbreite, DisIndex}: Integer;
  R: TRect;
{  Bitmap: TBitmap;}
  IndexCol: TDGColumn;
  Alignment: TAlignment;
  BrushMerker: TBrush;
  PenMerker: TPen;
  FontMerker: TFont;
  Value: string;
  HasBG: boolean;
  CR, BoxRect: TRect;
  B, I: Integer;
begin
  {Wenn ZeilenHhe gleich 0 --> gar nix tun}
  IF ARect.Bottom - ARect.Top < 1 THEN exit;
  {Indizierte Spalte ermitteln}
  if ACol < Columns.count then IndexCol := Columns[ACol] ELSE IndexCol := NIL;

  {Original-Pinsel, Stift und Schriftart merken}
  BrushMerker := TBrush.Create;
  BrushMerker.Assign(Canvas.Brush);
  PenMerker := TPen.Create;
  PenMerker.Assign(Canvas.Pen);
  FontMerker := TFont.Create;
  FontMerker.Assign(Canvas.Font);

  {normale Einstellungen}
  Canvas.Brush.Color := IndexCol.Color;
  IF (ACol < FixedCols) OR (ARow < FixedRows) THEN Canvas.Brush.Color := FixedColor;
  Canvas.Font := IndexCol.Font;
  Alignment := IndexCol.Alignment;
  XPos := ARect.Left + 2;
  YPos := ARect.Top + 2;
  R.Top := ARect.Top; R.Bottom := ARect.Bottom;
  Value := Cells[ACol, ARow];
  CR := ARect;
  HasBG := (assigned(FBackground) and (FBackground.width > 0) and (FBackground.height > 0));

  {evt. maskieren}
  if IndexCol.Displaymask <> '' then
  try
    IF Value <> '' THEN
    case IndexCol.Format of
    cfNumber:  Value := formatfloat(IndexCol.Displaymask, strtofloat(value));
    cfInteger: Value := formatfloat(IndexCol.Displaymask, strtoint(value));
    cfDate:    Value := formatdateTime(IndexCol.Displaymask, strtodatetime(value));
    end;
  except
  end;

  {Hintergrund malen?}
  {nur Datenzellen mit HintergrundBitmap ohne eigene Farbe}
  IF HasBG AND (ACol >= FixedCols) AND (ARow >= FixedRows) THEN BEGIN
    {Spalte mit HintergrundBitmap}
    IF (IndexCol.Color = clWindow) THEN BEGIN
      DrawBackground(ARect, AState);
      Canvas.Brush.Style := bsClear;
    END
    {Spalte mit eigener Farbe}
    ELSE BEGIN
      Canvas.FillRect(ARect);
      Canvas.Brush.Style := bsSolid;
    END;
  END
  {Alle Zellen ohne HintergrundBitmap}
  ELSE BEGIN
    {nur selektierte Zellen ohne HintergrundBitmap}
    IF (gdSelected in AState) THEN BEGIN
      Canvas.Brush.Color := FSelectedCellColor;
      Canvas.FillRect(ARect);
    END
    {normale Datenzellen und Fixed-Zellen ohne HintergrundBitmap}
    ELSE BEGIN
      Canvas.FillRect(ARect);
      Canvas.Brush.Style := bsSolid;
    END;
  END;
  {nur selektierte Zellen}
  IF (gdSelected in AState) THEN Canvas.Font.Color := FSelectedCellFontColor;

  {--------- CheckBox-Spalte? -----------------------}
  IF (IndexCol.ColStyle = colsCheck) THEN WITH Canvas DO BEGIN
    {*********** evt. Startwert korrigieren ************}
    IF (Value <> '0') AND (Value <> '1') THEN BEGIN
      IF Value = '' THEN Value := '0' ELSE Value := '1';
      Cells[ACol, ARow] := Value;
    END;
    {*********** CheckBox-Gre vorgeben ***********************}
    B := IndexCol.GetCheckBoxSize;
    {*********** Kstchen malen ************}
    CASE IndexCol.Alignment OF
    taLeftJustify : BoxRect.Left := CR.Left + 5;
    taRightJustify: BoxRect.Left := CR.Right - 5 - B;
    taCenter:       BoxRect.Left := CR.Left + (CR.Right - CR.Left - B) SHR 1;
    END;
    BoxRect.Top := CR.Top + (CR.Bottom - CR.Top - B) DIV 2;
    BoxRect.Right := BoxRect.Left + B;
    BoxRect.Bottom := BoxRect.Top + B;
    IF Enabled AND NOT IndexCol.ReadOnly THEN Brush.Color := clWindow ELSE Brush.Color := clInActiveBorder;
    FillRect(BoxRect);
    CASE IndexCol.CheckBoxStyle OF
    pcbsSunken: DrawEdge(Handle, BoxRect, EDGE_SUNKEN, BF_RECT);
    pcbsRaised: DrawEdge(Handle, BoxRect, EDGE_RAISED, BF_RECT);
    pcbsBump: DrawEdge(Handle, BoxRect, EDGE_BUMP, BF_RECT);
    pcbsEtched: DrawEdge(Handle, BoxRect, EDGE_ETCHED, BF_RECT);
    END;
    {*********** Hkchen malen ************}
    IF Value <> '0' THEN BEGIN
      IF Enabled AND NOT IndexCol.ReadOnly THEN BEGIN
        IF (gdSelected in AState) THEN Pen.Color := FSelectedCellFontColor
                                  ELSE Pen.Color := IndexCol.HookColor;
      END
      ELSE Pen.Color := clGrayText;
      Brush.Color := Pen.Color;
      Pen.Width := 1;
      CASE IndexCol.HookStyle OF
      phsHook:      FOR I:=0 TO 2 DO BEGIN
                    MoveTo(BoxRect.Left + 3,         BoxRect.Top - I + 7);
                    LineTo(BoxRect.Left + B DIV 2-1, BoxRect.Top - I + (B DIV 2) + 3);
                    LineTo(BoxRect.Right - 3,        BoxRect.Top - I + 4);
                    END;
      phsCross:     FOR I:=0 TO 0 DO BEGIN
                    MoveTo(BoxRect.Left + 3,     BoxRect.Top - I + 3);
                    LineTo(BoxRect.Left + B - 3, BoxRect.Top - I + B - 3);
                    MoveTo(BoxRect.Left + 3,     BoxRect.Top - I + B - 4);
                    LineTo(BoxRect.Left + B - 3, BoxRect.Top - I + 2);
                    END;
      phsFill:      FillRect(Rect(BoxRect.Left + 2, BoxRect.Top + 2, BoxRect.Right - 2, BoxRect.Bottom - 2));
      END; {CASE FHookStyle ...}
    END;
  END
  ELSE BEGIN
    {Text zeichnen}
    Breite := Canvas.TextWidth(Value);
    IF (Breite < (ARect.Right-XPos-2)) THEN BEGIN
      R.Left := XPos;
      R.Right := ARect.Right-1;
      CASE Alignment OF
      taLeftJustify:  R.Left := XPos;
      taRightJustify: R.Left := ARect.Right - Breite - 2;
      taCenter:       R.Left := XPos + (ARect.Right - XPos - Breite) shr 1;
      END;
      Canvas.TextRect(R, R.Left, YPos, Value);
    END
    ELSE BEGIN
      R.Left := XPos;
      R.Right := ARect.Right-1-Canvas.TextWidth('...');
      IF (R.Right > R.Left) THEN Canvas.TextRect(R, R.Left, YPos, Value);
      R.Left := R.Right+2;
      R.Right := ARect.Right-1;
      IF (R.Left > ARect.Left)
        THEN Canvas.TextRect(R, R.Left, YPos, '...')
        {wenn nicht mal mehr Punkte hinpassen --> grau ausfllen}
        ELSE Canvas.FillRect(ARect);
    END;
  END;

  {Original-Pinsel, Stift und Schriftart wieder herstellen}
  Canvas.Brush.Assign(BrushMerker);
  Canvas.Pen.Assign(PenMerker);
  Canvas.Font.Assign(FontMerker);
  BrushMerker.Free;
  PenMerker.Free;
  FontMerker.Free;
end;

procedure TPieCustomDataGrid.SetIntValue(ACol, ARow: Integer; const Value: Integer);
begin
     cells[ACol,ARow] := inttostr(Value);
end;

function TPieCustomDataGrid.GetIntValue(ACol, ARow: Integer): Integer;
begin
  try
     result := strtoint(cells[ACol,ARow]);
  except;
     result := 0;
  end;
end;

procedure TPieCustomDataGrid.SetNumValue(ACol, ARow: Integer; const Value: Real);
begin
     cells[ACol,ARow] := Floattostr(Value);
end;

function TPieCustomDataGrid.GetNumValue(ACol, ARow: Integer): Real;
begin
  try
     result := strtoFloat(cells[ACol,ARow]);
  except;
     result := 0;
  end;
end;

procedure TPieCustomDataGrid.SetBackground(newImg: TBitmap);
begin
  FBackground.assign(newImg);
  invalidate;
end;

procedure TPieCustomDataGrid.SetSorted(Value: Boolean);
BEGIN
  IF Value <> FSorted THEN BEGIN
    FSorted := Value;
    InvalidateCell(FSortColumn, 0);
    IF FSorted THEN SortGrid;
  END;
END;

procedure TPieCustomDataGrid.SetSortColumn(Value: Integer);
VAR
  I: Integer;
begin
  IF Value <> FSortColumn THEN BEGIN
    I := FSortColumn;
    FSortColumn := Value;
    InvalidateCell(I, 0);
    InvalidateCell(FSortColumn, 0);
    IF FSorted THEN SortGrid;
  end;
end;

procedure TPieCustomDataGrid.SetSortUp(Value: Boolean);
begin
  IF Value <> FSortUp THEN BEGIN
    FSortUp := Value;
    InvalidateCell(FSortColumn, 0);
    IF FSorted THEN SortGrid;
  end;
end;

procedure TPieCustomDataGrid.SortGrid;
VAR
  DefaultSort: Boolean;
  I, R: Integer;
  IndexString, S: string;
begin
  IF RowCount < 2 THEN exit;
  IF assigned(FOnSortProgress) THEN FOnSortProgress(self, psStarting, 0);
  IndexString := '';
  FOR I:=0 TO ColCount-1 DO IndexString := IndexString + Cells[I, Row];
  DefaultSort := TRUE;
  If assigned(FOnColumnSort) THEN FOnColumnSort(Self, FSortColumn, FSortUp, DefaultSort);
  IF DefaultSort THEN GridSortieren;
  {Wieder Fokus setzen}
  R := 0;
  REPEAT
    inc(R);
    S := '';
    FOR I:=0 TO ColCount-1 DO S := S + Cells[I, R];
  UNTIL (R=RowCount-1) OR (S = IndexString);
  IF S = IndexString THEN Row := R;
  IF assigned(FOnSortProgress) THEN FOnSortProgress(self, psEnding, 0);
end;

procedure TPieCustomDataGrid.GridSortieren;
VAR
  TestListe: TStringList;
  Teil: string;
  Index, I: Integer;
  Vis: Boolean;
begin
  {Grid sortieren nur zur Laufzeit! (sonst geht allerhand schief)}
  IF (csDesigning IN ComponentState) OR NOT Sorted THEN exit;

  Vis := Visible;
  If Vis THEN Hide;

  TestListe := TStringList.Create;
  TestListe.Duplicates := dupAccept;
  TestListe.Sorted := TRUE;

  FOR I:=FixedRows TO RowCount-1 DO BEGIN
    IF assigned(FOnSortProgress) THEN FOnSortProgress(self, psRunning, round((I+1)*100/RowCount));

    {Teilstring ermitteln}
    Teil := Cells[FSortColumn, I];

    Index := TestListe.Add(Teil);
    IF NOT FSortUp THEN Index := TestListe.Count - 1 - Index;
    MoveRow(I, Index+FixedRows);
  END;

  {Testliste lschen}
  TestListe.Free;
  IF Vis THEN Show;
end;

function TPieCustomDataGrid.GetEditMask(ACol, ARow: Longint): string;
begin
  Result := '';
  if columns.count > ACol then Result := columns[ACol].EditMask;
  if Assigned(FOnGetEditMask) then FOnGetEditMask(Self, ACol, ARow, Result);
end;

function TPieCustomDataGrid.GetSelectedIndex: Integer;
begin
  Result := Col;
end;

function TPieCustomDataGrid.SelectCell(ACol, ARow: Longint): Boolean;
begin
  IF ((FColMerker >= 0) AND (FRowMerker >= 0)) AND
     ((FColMerker <> ACol) OR (FRowMerker <> ARow)) THEN
    IF Assigned(FOnExitCell) THEN FOnExitCell(Self, FColMerker, FRowMerker);
  Result := inherited SelectCell(ACol, ARow);
  if Result then begin
    FColMerker := ACol;
    FRowMerker := ARow;
  end;
end;

procedure TPieCustomDataGrid.SetSelectedIndex(Value: Integer);
begin
  Col := Value;
end;

procedure TPieCustomDataGrid.SetEditorColor(Value: TColor);
begin
  IF Value <> FEditorColor THEN BEGIN
    FEditorColor := Value;
    IF assigned(InplaceEditor) THEN (InplaceEditor as TPieDataGridInplaceEdit).Color := FEditorColor;
  END;
end;

function TPieCustomDataGrid.GetEditorSelLength: Integer;
begin
  IF assigned(InplaceEditor) THEN Result := (InplaceEditor as TPieDataGridInplaceEdit).SelLength
                             ELSE Result := 0;
end;

procedure TPieCustomDataGrid.SetEditorSelLength(Value: Integer);
begin
  IF assigned(InplaceEditor) THEN (InplaceEditor as TPieDataGridInplaceEdit).SelLength := Value;
end;

function TPieCustomDataGrid.GetEditorSelStart: Integer;
begin
  IF assigned(InplaceEditor) THEN Result := (InplaceEditor as TPieDataGridInplaceEdit).SelStart
                             ELSE Result := 0;
end;

procedure TPieCustomDataGrid.SetEditorSelStart(Value: Integer);
begin
  IF assigned(InplaceEditor) THEN (InplaceEditor as TPieDataGridInplaceEdit).SelStart := Value;
end;

function TPieCustomDataGrid.GetEditorSelText: string;
begin
  IF assigned(InplaceEditor) THEN Result := (InplaceEditor as TPieDataGridInplaceEdit).SelText
                             ELSE Result := '';
end;

procedure TPieCustomDataGrid.SetEditorSelText(Value: string);
begin
  IF assigned(InplaceEditor) THEN (InplaceEditor as TPieDataGridInplaceEdit).SelText := Value;
end;

procedure TPieCustomDataGrid.EditorSelectAll;
begin
  IF assigned(InplaceEditor) THEN (InplaceEditor as TPieDataGridInplaceEdit).SelectAll;
end;

function TPieCustomDataGrid.GetVersion: string;
begin
  Result := '1.3';
end;

procedure TPieCustomDataGrid.SetSelectedCellColor(Value: TColor);
begin
  IF Value <> FSelectedCellColor THEN BEGIN
    FSelectedCellColor := Value;
    InvalidateCell(Col, Row);
  END;
end;

procedure TPieCustomDataGrid.SetSelectedCellFontColor(Value: TColor);
begin
  IF Value <> FSelectedCellFontColor THEN BEGIN
    FSelectedCellFontColor := Value;
    InvalidateCell(Col, Row);
  END;
end;

procedure TPieCustomDataGrid.MouseDown(Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
VAR
  Coord: TGridCoord;
  I: Integer;
begin
  inherited MouseDown(Button, Shift, X, Y);
  Coord := MouseCoord(X, Y);
  IF (Button = mbLeft) AND (Coord.Y = 0) THEN BEGIN
    IF (FSortColumn = Coord.X) THEN FSortUp := NOT FSortUp ELSE FSortUp := TRUE;
    I := FSortColumn;
    FSortColumn := Coord.X;
    InvalidateCell(I, 0);
    InvalidateCell(Coord.X, 0);
    SortGrid;
  END;
  IF (Button = mbLeft) AND ((Coord.Y < FixedRows) OR (Coord.X < FixedCols)) THEN BEGIN
    IF assigned(FOnFixedCellClick) THEN FOnFixedCellClick(self, Coord.X, Coord.Y);
  END;
end;

procedure TPieCustomDataGrid.EditButtonClick;
begin
  if Assigned(FOnEditButtonClick) then begin
    FOnEditButtonClick(Self);
  end;
end;

function TPieCustomDataGrid.CreateEditor: TInplaceEdit;
begin
  Result := TPieDataGridInplaceEdit.Create(Self);
  (Result as TPieDataGridInplaceEdit).Color := FEditorColor;
end;

function TPieCustomDataGrid.CreateColumns: TPieDataGridColumns;
begin
  Result := TPieDataGridColumns.Create(Self, TDGColumn);
end;

procedure TPieCustomDataGrid.SetColumns(Value: TPieDataGridColumns);
begin
  Columns.Assign(Value);
end;

procedure TPieCustomDataGrid.KeyDown(var Key: Word; Shift: TShiftState);
begin
  if CheckDataGridKey(Key, shift) then
  case Key of
    VK_TAB:    AppendRow;
    VK_DOWN:   AppendRow;
    VK_INSERT: InsertRow(row);
    VK_DELETE: DeleteRow(row);
  end;
  inherited keydown(Key, Shift);
end;

function TPieCustomDataGrid.CheckDataGridKey(var Key: Word; Shift: TShiftState): Boolean;
begin
     result := false;
     case key of
          VK_TAB:    if (not (ssShift in Shift) and (Row = Rowcount-1) and (Col = Colcount-1)) and (dgoAppendRow in DataGridOptions) then result := true;
          VK_DOWN:   if (Row = Rowcount-1) and (ssCtrl in Shift) and (dgoAppendRow in DataGridOptions) then result := true;
          VK_INSERT: if (ssCtrl in Shift) and (dgoInsertRow in DataGridOptions) then result := true;
          VK_DELETE: if (ssCtrl in Shift) and (dgoDeleteRow in DataGridOptions) then result := true;
     end;
end;

procedure TPieCustomDataGrid.Paint;
var
   i, endx, endy: integer;
begin
  inherited paint;
  if assigned(FBackground) and (FBackground.width > 0) and (FBackground.height > 0) then
  begin
     endx := 0;
     for I := 0 to FixedCols-1 do inc(endx, (colwidths[i]+1));
     for I := LeftCol to colcount-1 do inc(endx, (colwidths[i]+1));
     endy := 0;
     for I := 0 to FixedRows-1 do inc(endy, (Rowheights[i]+1));
     for I := TopRow to Rowcount-1 do inc(endy, (Rowheights[i]+1));

     if endX < clientwidth  then DrawBackground(rect(endx+1, 0, clientwidth, endy), []);
     if endy < clientheight then DrawBackground(rect(0, endy, clientwidth, clientheight), []);
  end;
end;

procedure TPieCustomDataGrid.TopLeftChanged;
begin
  inherited;
  if assigned(FBackground) and (FBackground.width > 0) and (FBackground.height > 0)
    then invalidatergn(handle, 0, false);
end;

procedure TPieCustomDataGrid.DrawBackground(rect: TRect; AState: TGridDrawState);
var
  srect, drect, irect: trect;
  XCnt, YCnt, X, Y: Integer;
begin
     if (FBackground.width > 0) and (FBackground.height > 0) then
     begin
       XCnt := (Clientwidth) div FBackground.width;
       YCnt := (Clientheight) div FBackground.height;

       for x := 0 to XCnt do
       begin
            for y := 0 to YCnt do
            begin
                 drect.left   := x * FBackground.width;
                 drect.top    := y * FBackground.height;
                 drect.right  := drect.left + FBackground.width;
                 drect.bottom := drect.top  + FBackground.height;

                 if Intersectrect(irect, rect, drect) then
                 begin
                    srect := irect;
                    while srect.left >= FBackground.width  do offsetrect(srect, -FBackground.width, 0);
                    while srect.top  >= FBackground.height do offsetrect(srect, 0, -FBackground.height);
                    canvas.copyrect(irect, FBackground.canvas, srect);
                 end;
            end;
       end;
     end;
end;

procedure TPieCustomDataGrid.DefaultHandler(var Msg);
var
  P: TPopupMenu;
  Cell: TGridCoord;
begin
  inherited DefaultHandler(Msg);
  if TMessage(Msg).Msg = wm_RButtonUp then
    with TWMRButtonUp(Msg) do
    begin
      Cell := MouseCoord(XPos, YPos);
      if (Cell.X < 0) or (Cell.Y < 0) then Exit;
      P := Columns[Cell.X].PopupMenu;
      if (P <> nil) and P.AutoPopup then
      begin
        SendCancelMode(nil);
        P.PopupComponent := Self;
        with ClientToScreen(SmallPointToPoint(Pos)) do
          P.Popup(X, Y);
        Result := 1;
      end;
    end;
end;

procedure TPieCustomDataGrid.ColumnMoved(FromIndex, ToIndex: Longint);
begin
  Columns[FromIndex].Index := ToIndex;
  Inherited;
end;

procedure TPieCustomDataGrid.MoveRow(FromIndex, ToIndex: Longint);
begin
  inherited MoveRow(FromIndex, ToIndex);
end;

procedure TPieCustomDataGrid.AppendRow;
begin
  if assigned(FOnBeforeInsert) then FOnBeforeInsert(self);
  RowCount := RowCount + 1;
  invalidateRow(Rowcount-1);
  SelectCell(Col,Rowcount-1);
end;

procedure TPieCustomDataGrid.InsertRow(ARow: LongInt);
var
   I, L: LongInt;
begin
  if assigned(FOnBeforeInsert) then FOnBeforeInsert(self);
  RowCount := RowCount + 1;
  for I := RowCount-1 downto ARow do for L := 0 to ColCount-1 do begin
    cells[L,I] := cells[L,I-1];
    Objects[L,I] := Objects[L,I-1];
  end;
  for L := 0 to Colcount -1 do begin
    cells[L, ARow] := '';
    Objects[L, ARow] := nil;
  end;
end;

procedure TPieCustomDataGrid.DeleteRow(ARow: LongInt);
var
   I, L: LongInt;
begin
     if assigned(FOnBeforeDelete) then FOnBeforeDelete(self);
     if (RowCount > FixedRows+1) and ((RowCountMin = 0) or (RowCount > RowCountMin)) then
     begin
        for I := ARow to RowCount-1 do for L := 0 to ColCount-1 do begin
          cells[L,I] := cells[L,I+1];
          Objects[L,I] := Objects[L,I+1];
        end;
        for L := 0 to Colcount -1 do cells[L, RowCount-1] := '';
        RowCount := RowCount - 1;
     end else for L := FixedCols to ColCount-1 do cells[L, ARow] := '';
     InvalidateEditor;
end;

{******************************************************************}
{*************** TDBDGColumn **************************************}
{******************************************************************}
function TDBDGColumn.GetDataField: string;
var
  Grid: TPieDBDataGrid;
begin
  Grid := GetGrid as TPieDBDataGrid;
  IF Assigned(Grid) THEN Result := Grid.DataField[InternalCol]
                    ELSE Result := '';
end;

procedure TDBDGColumn.SetDataField(Value: string);
var
  Grid: TPieDBDataGrid;
  I: Integer;
begin
  Grid := GetGrid as TPieDBDataGrid;
  IF Assigned(Grid) THEN BEGIN
    Grid.DataField[InternalCol] := Value;
    {Datenfeld existiert}
    IF (Grid.DataField[InternalCol] <> '') THEN BEGIN
      IF (Title.Caption = '') THEN Title.Caption := Grid.DataField[InternalCol];
      Grid.ActiveChange(Grid);
    END
    {ganze Spalte lschen}
    ELSE BEGIN
      FOR I:=0 TO Grid.RowCount-1 DO Grid.Cells[InternalCol, I] := '';
    END;
  END;
end;

{******************************************************************}
{*************** TPieDBDataGridColumns ****************************}
{******************************************************************}
function TPieDBDataGridColumns.GetColumn(Index: Integer): TDBDGColumn;
begin
  Result := TDBDGColumn(inherited Items[Index]);
end;

procedure TPieDBDataGridColumns.SetColumn(Index: Integer; Value: TDBDGColumn);
begin
  Items[Index].Assign(Value);
end;

{******************************************************************}
{*************** TPieDBDataGrid ***********************************}
{******************************************************************}
constructor TPieDBDataGrid.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  SizeChanged(0, 0);
end;

destructor TPieDBDataGrid.Destroy;
VAR
  I: Integer;
begin
  {DataLink-Objekte fr alle Spalten lschen}
  FOR I:=0 TO High(FDataLink) DO BEGIN
    FDataLink[I].Free;
    FDataLink[I] := NIL;
  END;
  SetLength(FDataLink, 0);
  inherited Destroy;
end;

function TPieDBDataGrid.CreateColumns: TPieDataGridColumns;
begin
  Result := TPieDBDataGridColumns.Create(Self, TDBDGColumn);
end;

function TPieDBDataGrid.GetDataSource: TDataSource;
begin
  Result := FDataSource;
end;

procedure TPieDBDataGrid.SetDataSource(Value: TDataSource);
VAR
  I: Integer;
begin
  FDataSource := Value;
  FOR I:=0 TO High(FDataLink) DO FDataLink[I].DataSource := Value;
  if Value <> nil then Value.FreeNotification(Self);
end;

function TPieDBDataGrid.GetDataLink(Index: Integer): TFieldDataLink;
begin
  IF (0 <= Index) AND (Index <= High(FDataLink)) THEN BEGIN
    Result := FDataLink[Index];
  END
  ELSE Result := NIL;
end;

function TPieDBDataGrid.GetDataField(Index: Integer): string;
begin
  IF (0 <= Index) AND (Index <= High(FDataLink)) THEN BEGIN
    Result := FDataLink[Index].FieldName;
  END
  ELSE Result := '';
end;

procedure TPieDBDataGrid.SetDataField(Index: Integer; const Value: string);
begin
  IF (0 <= Index) AND (Index <= High(FDataLink)) THEN BEGIN
    FDataLink[Index].FieldName := Value;
  END;
end;

function TPieDBDataGrid.GetColumns: TPieDBDataGridColumns;
begin
  Result := FColumns as TPieDBDataGridColumns;
end;

procedure TPieDBDataGrid.SetColumns(Value: TPieDBDataGridColumns);
begin
  inherited SetColumns(Value);
end;

procedure TPieDBDataGrid.Click;
begin
  inherited Click;
  {DataSource zugewiesen und Datenmenge aktiv?}
  IF Assigned(FDataSource) AND FDataSource.DataSet.Active THEN BEGIN
    FDataSource.DataSet.RecNo := Row;
  END;
end;

procedure TPieDBDataGrid.KeyDown(var Key: Word; Shift: TShiftState);
begin
  inherited keydown(key, shift);
  {DataSource zugewiesen und Datenmenge aktiv?}
  IF Assigned(FDataSource) AND FDataSource.DataSet.Active THEN BEGIN
    FDataSource.DataSet.RecNo := Row;
  END;
end;

procedure TPieDBDataGrid.DataChanged(Col, Row: Integer; Value: string);
VAR
  DL: TFieldDataLink;
begin
  {DataSource zugewiesen?}
  IF Assigned(FDataSource) THEN BEGIN
    DL := FDataLink[Col];
    IF Assigned(DL) AND Assigned(DL.Field) THEN BEGIN
      {Datenmenge aktiv?}
      IF FDataSource.DataSet.Active THEN BEGIN
        FDataSource.DataSet.Edit;
        DL.Field.Text := Value;
//        FDataSource.DataSet.Post;   {Post soll User in eigener Verantwortung aufrufen!
      END;
    END;
  END;
end;

procedure TPieDBDataGrid.SizeChanged(OldColCount, OldRowCount: Longint);
VAR
  I: Integer;
begin
  inherited;
  {Anzahl der DataLink-Komponenten ndern}
  IF OldColCount <> ColCount THEN BEGIN
    IF OldColCount > ColCount THEN BEGIN  {Spalten wurden gelscht}
      FOR I:=ColCount TO OldColCount-1 DO BEGIN
        FDataLink[I].Free;
        FDataLink[I] := NIL;
      END;
      SetLength(FDataLink, ColCount);
    END;
    IF OldColCount < ColCount THEN BEGIN  {Spalten wurden hinzugefgt}
      SetLength(FDataLink, ColCount);
      FOR I:=OldColCount TO ColCount-1 DO BEGIN
        FDataLink[I] := TFieldDataLink.Create();
        FDataLink[I].Control := self;
        FDataLink[I].OnActiveChange := ActiveChange;
        FDataLink[I].OnDataChange := DataChange;
        FDataLink[I].DataSource := FDataSource;
      END;
    END;
    if not (csLoading in ComponentState) and (FLayoutFlag = 0) then DataChange(self);
  END;
end;

procedure TPieDBDataGrid.RowCountMinChanged;
begin
  if not (csLoading in ComponentState) and (FLayoutFlag = 0) then DataChange(self);
end;

procedure TPieDBDataGrid.ActiveChange(Sender: TObject);
VAR
  DSNr: Integer;
  I, J: Integer;
begin
  {DataSource zugewiesen und Datenmenge aktiv?}
  IF Assigned(FDataSource) AND FDataSource.DataSet.Active THEN BEGIN
    {Alle Datenstze anzeigen}
    RowCount := FDataSource.DataSet.RecordCount + 1;
    DSNr := FDataSource.DataSet.RecNo;
    TRY
      FDataSource.DataSet.First;
      WHILE NOT FDataSource.DataSet.Eof DO FDataSource.DataSet.Next;
    FINALLY
      FDataSource.DataSet.RecNo := DSNr;
    END;
  END
  ELSE BEGIN
    {Alle Felder lschen}
    RowCount := 2;
    FOR I:=0 TO RowCount-1 DO FOR J:=0 TO ColCount-1 DO Cells[J, I] := '';
  END;
END;

procedure TPieDBDataGrid.DataChange(Sender: TObject);
VAR
  DL: TFieldDataLink;
  I: Integer;
  DSNr: Integer;
begin
  {DataSource zugewiesen?}
  IF Assigned(FDataSource) THEN BEGIN
    FOR I:=0 TO High(FDataLink) DO BEGIN {Fr jede Spalte Daten darstellen}
      DL := FDataLink[I];
      IF Assigned(DL) AND Assigned(DL.Field) THEN BEGIN
        {Datenmenge aktiv?}
        IF FDataSource.DataSet.Active THEN BEGIN
          DSNr := FDataSource.DataSet.RecNo;
          IF (DSNr < RowCount) AND (DSNr >= 0) THEN BEGIN
            Cells[I, DSNr] := DL.Field.Text;
            Row := DSNr;
          END;
        END;
      END;
    END;
  END;
end;

procedure TPieDBDataGrid.AppendRow;
begin
  inherited AppendRow;
  {DataSource zugewiesen und Datenmenge aktiv?}
  IF Assigned(FDataSource) AND FDataSource.DataSet.Active THEN BEGIN
    FDataSource.DataSet.Append;
    FDataSource.DataSet.Post;
  END;
end;

procedure TPieDBDataGrid.InsertRow(ARow: LongInt);
begin
  inherited InsertRow(ARow);
  {DataSource zugewiesen und Datenmenge aktiv?}
  IF Assigned(FDataSource) AND FDataSource.DataSet.Active THEN BEGIN
    FDataSource.DataSet.Insert;
    FDataSource.DataSet.Post;
    ActiveChange(self);
  END;
end;

procedure TPieDBDataGrid.DeleteRow(ARow: LongInt);
begin
  inherited DeleteRow(ARow);
  {DataSource zugewiesen und Datenmenge aktiv?}
  IF Assigned(FDataSource) AND FDataSource.DataSet.Active THEN BEGIN
    FDataSource.DataSet.Delete;
    Paint;
  END;
end;


function ReplaceChar(C1: Char; S: String; C2: Char): String;
var
   I: Integer;
begin
     result := '';
     repeat
           I := pos(C1,S);
           if I > 0 then
           begin
                result := result + copy(S,1,I-1) + C2;
                S := copy(S,I+1,length(S)-I);
           end else result := result + S;
     until I = 0;
end;

procedure KillMessage(Wnd: HWnd; Msg: Integer);
var
  M: TMsg;
begin
  M.Message := 0;
  if PeekMessage(M, Wnd, Msg, Msg, pm_Remove) and (M.Message = WM_QUIT) then PostQuitMessage(M.wparam);
end;

end.
