unit LCDScreenEditor;

////////////////////////////////////////////////////////////////////////////////
//
//  TLCDScreen / TLCDAnimator v3.0 (22/jan/02)
//  written by Jacques VOIRIN
//  E-Mail: jacques.voirin@iname.com
//
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
//
//  This unit contain all the routines for the TLCDScreenLinesEditor:
//   - registering the editor and associate it to TLCDScreen.Lines
//   - the internal code for its form TLCDScreenLinesEditorForm.
//
//  LCDScreenEditor is only use at design-time.
//
////////////////////////////////////////////////////////////////////////////////


interface

uses
  SysUtils, Classes, Controls, Forms, StdCtrls, Messages, ExtCtrls, Buttons,

    {$IFDEF  VER140}
      DesignIntf, DesignEditors,
    {$ELSE}
      DsgnIntf,
    {$ENDIF}

  LCDScreen;



////////////////////////////////////////////////////////////////////////////////
//
// TLCDScreenEditor is the TComponentEditor which generate the custom context
// menu when TLCDScreen is left-clicked and then display the LCDScreenLinesEditorForm.
//
////////////////////////////////////////////////////////////////////////////////
type
  TLCDScreenEditor = class(TComponentEditor)
    procedure ExecuteVerb(Index: Integer); override;
    function  GetVerb(Index: Integer): String; override;
    function  GetVerbCount: Integer; override;
  end;


////////////////////////////////////////////////////////////////////////////////
//
// TLCDScreenLinesEditorProperty is the TPropertyEditor in relation with 'Lines'
// property of TLCDScreen. Clicking on '...' in the Object Inspector display
// the LCDScreenLinesEditorForm.
//
////////////////////////////////////////////////////////////////////////////////

  TLCDScreenLinesEditorProperty = class(TPropertyEditor)
    procedure Edit; override;
    function  GetValue: String; override;
    function  GetAttributes: TPropertyAttributes; override;
  end;


////////////////////////////////////////////////////////////////////////////////
//
// TLCDAnimatorCodeEditorForm is the CodeEditor form, which allow to write
// the code for the animation.
//
////////////////////////////////////////////////////////////////////////////////

  TLCDScreenLinesEditorForm = class(TForm)
    Panel1: TPanel;
    Panel2: TPanel;
    Panel3: TPanel;
    Panel4: TPanel;
    Panel5: TPanel;
    Panel6: TPanel;
    CancelButton: TButton;
    OkButton: TButton;
    LinesMemo: TMemo;
    BrutLabel: TLabel;
    PreviewLabel: TLabel;
    SpBPanel: TPanel;
    BoldSpB: TSpeedButton;
    ItalicSpB: TSpeedButton;
    StrikeSpB: TSpeedButton;
    UnderlineSpB: TSpeedButton;
    InverseSpB: TSpeedButton;
    BlinkSpB: TSpeedButton;
    PreviewSB: TScrollBox;
    PreviewLCD: TLCDScreen;

    procedure FormShow(Sender: TObject);
    procedure LinesMemoChange(Sender: TObject);
    procedure LinesMemoKeyUp(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure LinesMemoMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
    procedure LinesMemoMouseUp(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure SetLCDTag(LCDTag: String);
    procedure BoldSpBClick(Sender: TObject);
    procedure ItalicSpBClick(Sender: TObject);
    procedure StrikeSpBClick(Sender: TObject);
    procedure UnderlineSpBClick(Sender: TObject);
    procedure InverseSpBClick(Sender: TObject);
    procedure BlinkSpBClick(Sender: TObject);
    procedure DeleteLCDTagSpBClick(Sender: TObject);
  end;

procedure Register;

var
  LCDScreenLinesEditorForm: TLCDScreenLinesEditorForm;

implementation

{$R *.DFM}


////////////////////////////////////////////////////////////////////////////////
//
// TLCDScreenEditor
//
// Result := 1; Only one personalized entry in the context menu.
//
////////////////////////////////////////////////////////////////////////////////

function  TLCDScreenEditor.GetVerbCount: Integer;
begin
 Result := 1;
end;


////////////////////////////////////////////////////////////////////////////////
//
// TLCDScreenEditor
//
// Result := 'LCDScreen Lines Editor'; The caption of the personalized entry
//                                     in the context menu.
//
////////////////////////////////////////////////////////////////////////////////

function  TLCDScreenEditor.GetVerb(Index: Integer): String;
begin
  Result := 'LCDScreen Lines Editor';
end;


////////////////////////////////////////////////////////////////////////////////
//
// TLCDScreenEditor
//
// This is te code displaying the editor form when 'LCDScreen Lines Editor' in
// the context menu is clicked or when the component is double-clicked.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreenEditor.ExecuteVerb(Index: Integer);
var F: TLCDScreenLinesEditorForm;
begin
  F := TLCDScreenLinesEditorForm.Create(Application);
  try
    F.Caption := 'TLCDScreen Lines Editor - ' + (Component as TLCDScreen).Name ;

    F.PreviewLCD.Font.Color := (Component as TLCDScreen).Font.Color;
    F.PreviewLCD.PixelOff := (Component as TLCDScreen).PixelOff;
    F.PreviewLCD.Color := (Component as TLCDScreen).Color;
    F.PreviewLCD.Font := (Component as TLCDScreen).Font;

    F.LinesMemo.Lines.Assign((Component as TLCDScreen).Lines);       { Load Code lines from TLCDScreen to LinesStringGrid }

    if F.ShowModal = mrOK  //Display the Editor form
    then (Component as TLCDScreen).Lines.Assign(F.LinesMemo.Lines);  { UnLoad Code lines from LinesStringGrid to TLCDScreen }
         
    try Designer.Modified; except end;
    
  finally
    F.Free;
  end;
end;


////////////////////////////////////////////////////////////////////////////////
//
// TLCDScreenLinesEditorProperty
//
// Result := '(TStringList)'; The 'Lines' caption displayed in the Obj. Inspector
//
////////////////////////////////////////////////////////////////////////////////

function TLCDScreenLinesEditorProperty.GetValue: String;
begin
  Result := '(TStringList)';
end;


////////////////////////////////////////////////////////////////////////////////
//
// TLCDScreenLinesEditorProperty
//
// Result := [paDialog]; Indicates that the '...' button in the Object Inspector
//                       will bring up a dialog form.
//
////////////////////////////////////////////////////////////////////////////////

function TLCDScreenLinesEditorProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paDialog];
end;


////////////////////////////////////////////////////////////////////////////////
//
// TLCDScreenLinesEditorProperty
//
// The code displaying the editor form when the '...' button in the Object
// Inspector is clicked.
//
////////////////////////////////////////////////////////////////////////////////

procedure TLCDScreenLinesEditorProperty.Edit;
var F: TLCDScreenLinesEditorForm;
    C: TPersistent;
begin
  C := GetComponent(0);
  F := TLCDScreenLinesEditorForm.Create(Application);

  try
    F.Caption := 'TLCDScreen Lines Editor - ' + (C as TLCDScreen).Name ;

    F.PreviewLCD.Font.Color := (C as TLCDScreen).Font.Color;
    F.PreviewLCD.PixelOff := (C as TLCDScreen).PixelOff;
    F.PreviewLCD.Color := (C as TLCDScreen).Color;
    F.PreviewLCD.Font := (C as TLCDScreen).Font;

    F.LinesMemo.Lines.Assign((C as TLCDScreen).Lines);       { Load Code lines from TLCDScreen to LinesStringGrid }

    if F.ShowModal = mrOK  //Display the Editor form
    then (C as TLCDScreen).Lines.Assign(F.LinesMemo.Lines);  { UnLoad Code lines from LinesStringGrid to TLCDScreen }

    try Designer.Modified; except end;

  finally
    F.Free;
  end;
end;


////////////////////////////////////////////////////////////////////////////////
//
// Begining of the routines for the Form Editor.
//
////////////////////////////////////////////////////////////////////////////////


procedure TLCDScreenLinesEditorForm.LinesMemoChange(Sender: TObject);
var currentline: Byte;
begin
  currentline := LinesMemo.Perform(EM_LINEFROMCHAR, -1, 0);
  PreviewLCD.Lines.BeginUpdate;
  PreviewLCD.Lines.Clear;
  PreviewLCD.Lines.Append(LinesMemo.Lines[currentline]);
  PreviewLCD.Lines.EndUpdate;
end;

procedure TLCDScreenLinesEditorForm.LinesMemoKeyUp(Sender: TObject;
  var Key: Word; Shift: TShiftState);
begin
  LinesMemoMouseMove(Self, Shift, 0, 0);
end;

procedure TLCDScreenLinesEditorForm.LinesMemoMouseMove(Sender: TObject;
  Shift: TShiftState; X, Y: Integer);
begin
  if LinesMemo.SelLength <> 0
  then begin
         BoldSpB.Enabled := True;
         ItalicSpB.Enabled := True;
         StrikeSpB.Enabled := True;
         UnderlineSpB.Enabled := True;
         InverseSpB.Enabled := True;
         BlinkSpB.Enabled := True;
         end
  else begin
         BoldSpB.Enabled := False;
         ItalicSpB.Enabled := False;
         StrikeSpB.Enabled := False;
         UnderlineSpB.Enabled := False;
         InverseSpB.Enabled := False;
         BlinkSpB.Enabled := False;
         end;
end;

procedure TLCDScreenLinesEditorForm.LinesMemoMouseUp(Sender: TObject;
  Button: TMouseButton; Shift: TShiftState; X, Y: Integer);
begin
  LinesMemoChange(Sender);
end;

procedure TLCDScreenLinesEditorForm.SetLCDTag(LCDTag: String);
var str: Tcaption;
    oldsel: Integer;
begin
  with LinesMemo
  do begin
       //currentline := Perform(EM_LINEFROMCHAR, -1, 0);
       oldSel := SelStart;
       str := Text;
       Insert('<'+LCDTag+'>', str, SelStart + 1);
       Insert('</'+LCDTag+'>', str, SelStart + SelLength + Length(LCDTag) + 3);
       Text := str;
       end;
  LinesMemo.SelStart := oldsel + 2 + Length(LCDTag);
  LinesMemoChange(nil);
end;


procedure TLCDScreenLinesEditorForm.BoldSpBClick(Sender: TObject);
begin
  SetLCDTag('b');
end;

procedure TLCDScreenLinesEditorForm.ItalicSpBClick(Sender: TObject);
begin
  SetLCDTag('i');
end;

procedure TLCDScreenLinesEditorForm.StrikeSpBClick(Sender: TObject);
begin
  SetLCDTag('s');
end;

procedure TLCDScreenLinesEditorForm.UnderlineSpBClick(Sender: TObject);
begin
  SetLCDTag('u');
end;

procedure TLCDScreenLinesEditorForm.InverseSpBClick(Sender: TObject);
begin
  SetLCDTag('inv');
end;

procedure TLCDScreenLinesEditorForm.BlinkSpBClick(Sender: TObject);
begin
  SetLCDTag('bl');
end;

procedure TLCDScreenLinesEditorForm.DeleteLCDTagSpBClick(Sender: TObject);
var currentline: Byte;
begin
  if LinesMemo.SelLength = 0
  then begin
         currentline := LinesMemo.Perform(EM_LINEFROMCHAR, -1, 0);
         if CountLCDTag(LinesMemo.Lines[currentline]) <> 0
         then LinesMemo.Lines[currentline] := RemoveLCDTag(LinesMemo.Lines[currentline])
         else beep;
         end
  else if CountLCDTag(LinesMemo.SelText) <> 0
       then LinesMemo.SelText := RemoveLCDTag(LinesMemo.SelText)
       else beep;
end;
                               
procedure TLCDScreenLinesEditorForm.FormShow(Sender: TObject);
begin
  LinesMemo.SelStart := 0;
  LinesMemoChange(Sender);
end;


////////////////////////////////////////////////////////////////////////////////
//
// End of the routines for the Form Editor.
//
////////////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////////////////////
//
// Registration of the context menu and 'Lines [...]' in the ObjectInspector.
//                                                                            
////////////////////////////////////////////////////////////////////////////////

procedure Register;
begin
  RegisterComponentEditor(TLCDScreen,TLCDScreenEditor);
  RegisterPropertyEditor(TypeInfo(TStringList), TLCDScreen, 'Lines', TLCDScreenLinesEditorProperty);

end;

end.
